// copyright 2001-2002 by The Mind Electric

package electric.util.io;

import java.io.*;
import java.util.*;
import java.net.*;
import electric.util.*;
import electric.util.encoding.*;
import electric.util.log.*;

/**
 * <tt>Streams</tt> defines a set of static methods for manipulating streams.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public final class Streams
  {
  private static final int CHUNK_SIZE = 10000;
  private static final byte[] NO_BYTES = new byte[ 0 ];

  // ********** READERS AND WRITERS *****************************************

  /**
   * @param file
   * @throws IOException
   */
  static public Reader getReader( File file )
    throws IOException
    {
    FileInputStream input = new FileInputStream( file );
    byte[] header = readUpTo( input, 100 );
    input.close();
    String encoding = Encodings.getJavaEncoding( header );
    return new InputStreamReader( new FileInputStream( file ), encoding );
    }

  /**
   * @param input
   * @throws IOException
   */
  static public Reader getReader( InputStream input )
    throws IOException
    {
    if( !(input instanceof BufferedInputStream) )
      input = new BufferedInputStream( input );

    input.mark( 100 );
    byte[] header = readUpTo( input, 100 );
    input.reset();
    String encoding = Encodings.getJavaEncoding( header );
    return new InputStreamReader( input, encoding );
    }

  /**
   * @param file
   * @param encoding
   * @throws IOException
   * @throws UnsupportedEncodingException
   */
  static public Writer getWriter( File file, String encoding )
    throws IOException, UnsupportedEncodingException
    {
    return getWriter( new FileOutputStream( file ), encoding );
    }

  /**
   * @param output
   * @param encoding
   * @throws UnsupportedEncodingException
   */
  static public Writer getWriter( OutputStream output, String encoding )
    throws UnsupportedEncodingException
    {
    encoding = Encodings.getJavaEncoding( encoding );
    return new OutputStreamWriter( output, encoding );
    }

  // ********** READING *****************************************************

  /**
   * @param file
   * @throws IOException
   */
  static public byte[] readFully( File file )
    throws IOException
    {
    RandomAccessFile input = new RandomAccessFile( file, "r" );
    byte[] bytes = new byte[ (int) input.length() ];
    input.readFully( bytes );
    input.close();
    return bytes;
    }

  /**
   * @param input
   * @param buffer
   * @param offset
   * @param length
   * @throws IOException
   */
  static public void readFully( InputStream input, byte buffer[], int offset, int length )
    throws IOException
    {
    readFully( input, buffer, offset, length, -1 );
    }

  /**
   * @param input
   * @param buffer
   * @param offset
   * @param length
   * @param logMask
   * @throws IOException
   */
  static public void readFully( InputStream input, byte buffer[], int offset, int length, long logMask )
    throws IOException
    {
    int total = 0;

    while( total < length )
      {
      int amount = input.read( buffer, offset + total, length - total );

      if( amount < 0 )
        throw new EOFException( "expected " + length + " bytes of content, got " + total );

      if( logMask > -1 && Log.isLogging( logMask ) )
        Log.log( logMask, new String( buffer, offset + total, amount ) );

      total += amount;
      }
    }

  /**
   * @param input
   * @param length
   * @throws IOException
   */
  static public byte[] readFully( InputStream input, int length )
    throws IOException
    {
    return readFully( input, length, (long) -1 );
    }

  /**
   * @param input
   * @param length
   * @param logMask
   * @throws IOException
   */
  static public byte[] readFully( InputStream input, int length, long logMask )
    throws IOException
    {
    if( length == -1 )
      return readFully( input, logMask );

    if( length == 0 )
      return NO_BYTES;

    byte[] buffer = new byte[ length ];
    readFully( input, buffer, 0, length, logMask );
    return buffer;
    }

  /**
   * @param input
   * @throws IOException
   */
  static public byte[] readFully( InputStream input )
    throws IOException
    {
    return readFully( input, (long) -1 );
    }

  /**
   * @param input
   * @param logMask
   * @throws IOException
   */
  static public byte[] readFully( InputStream input, long logMask )
    throws IOException
    {
    byte[] buffer = new byte[ CHUNK_SIZE ];
    ByteArrayOutputStream output = new ByteArrayOutputStream();

    while( true )
      {
      int count = input.read( buffer, 0, buffer.length );

      if( count == -1 )
        break;

      output.write( buffer, 0, count );
      }

    buffer = output.toByteArray();

    if( logMask > -1 && Log.isLogging( logMask ) )
      Log.log( logMask, new String( buffer ) );

    return buffer;
    }

  /**
   * @param input
   * @param buffer
   * @param offset
   * @param length
   * @throws IOException
   */
  static public int readUpTo( InputStream input, byte buffer[], int offset, int length )
    throws IOException
    {
    int total = 0;

    while( total < length )
      {
      int amount = input.read( buffer, offset + total, length - total );

      if( amount < 0 )
        break;

      total += amount;
      }

    return total;
    }

  /**
   * @param input
   * @param length
   * @throws IOException
   */
  static public byte[] readUpTo( InputStream input, int length )
    throws IOException
    {
    if( length <= 0 )
      return NO_BYTES;

    byte[] buffer = new byte[ length ];
    int total = readUpTo( input, buffer, 0, length );

    if( total < length )
      {
      byte[] tmp = buffer;
      buffer = new byte[ total ];
      System.arraycopy( tmp, 0, buffer, 0, total );
      }

    return buffer;
    }

  /**
   * @param input
   * @param delimeter
   * @throws IOException
   */
  static public byte[] readUpTo( InputStream input, byte[] delimeter )
    throws IOException
    {
    ByteArrayOutputStream output = new ByteArrayOutputStream();
    copy( input, output, delimeter, -1 );
    return output.toByteArray();
    }

  /**
   * @param input
   * @throws IOException
   */
  static public String readLine( InputStream input )
    throws IOException
    {
    return readLine( input, Encodings.getSystemEncoding() );
    }

  /**
   * @param input
   * @param encoding
   * @throws IOException
   */
  static public String readLine( InputStream input, String encoding )
    throws IOException
    {
    ByteArrayOutputStream buffer = new ByteArrayOutputStream( 100 );
    int curPos = 0;

    while( true )
      {
      int ch = input.read();

      if( ch == -1 )
        {
        return (curPos == 0 ? null : new String( buffer.toByteArray(), encoding ));
        }
      else if( ch == '\r' )
        {
        if( (ch = input.read()) == '\n' )
          {
          return new String( buffer.toByteArray(), encoding );
          }
        else
          {
          buffer.write( '\r' );
          buffer.write( ch );
          continue;
          }
        }
      else if( ch == '\n' )
        {
        // warning: we're using system default encoding here
        // this may be undesirable under some circumstances
        return new String( buffer.toByteArray(), encoding );
        }

      buffer.write( ch );
      }
    }

  // ********** COPYING *****************************************************

  /**
   * @param input
   * @param output
   * @param length
   * @param buffersize
   * @param logMask
   * @throws IOException
   */
  static public void copy( InputStream input, OutputStream output, int length, int buffersize, long logMask )
    throws IOException
    {
    byte[] buffer = new byte[ length == -1 ? buffersize : Math.min( buffersize, length ) ];
    int total = 0;

    while( true )
      {
      int chunksize = length == -1 ? buffersize : Math.min( buffersize, length - total );
      int readBytes = readUpTo( input, buffer, 0, chunksize );

      if( readBytes == 0 )
        break;

      if( logMask > -1 && Log.isLogging( logMask ) )
        Log.log( logMask, new String( buffer, 0, readBytes ) );

      output.write( buffer, 0, readBytes );
      total += readBytes;

      if( length != - 1 && total >= length )
        break;
      }
    }

  /**
   * @param input
   * @param output
   * @param length
   * @param buffersize
   * @throws IOException
   */
  public static void copy( InputStream input, OutputStream output, int length, int buffersize )
    throws IOException
    {
    copy( input, output, length, buffersize, -1 );
    }

  /**
   * @param input
   * @param ouput
   * @param delimiter
   * @param logMask
   */
  public static void copy( InputStream input, OutputStream output, byte[] delimiter, long logMask )
    throws IOException
    {
    OutputStreamWriter writer = new OutputStreamWriter( output );
    int firstByte = delimiter[ 0 ];

    startOver:
    while( true )
      {
      int ch = input.read();

      if( ch != firstByte )
        {
        writer.write( ch );
        continue;
        }

      for( int i = 1; i < delimiter.length; i++ )
        {
        ch = input.read();

        if( ch != delimiter[ i ] )
          {
          for( int j = 0; j < delimiter.length; j++ )
            writer.write( delimiter[ j ] );

          writer.write( ch );
          continue startOver;
          }
        }

      // if we got here, the delimiter matched
      break;
      }

    writer.flush();
    }
  }