// copyright 2001-2002 by The Mind Electric

package electric.xml;

import java.io.*;
import electric.util.*;

/**
 * <tt>Attribute</tt> represents an XML attribute. Once constructed, you can
 * change its value but not its name. When you add an Attribute to an Element,
 * the attribute's name is automatically resolved. If you add/remove an Attribute
 * that declares a namespace, the Element's namespaces are automatically updated.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

final public class Attribute extends Node implements IQNamed, org.w3c.dom.Attr
  {
  private Element element;
  private String prefix;
  String namespace;
  String name;
  String value;
  boolean isNamespace; // true if i am a namespace declaration
  boolean isId; // true if i am an id
  private boolean raw; // true if subsitution should be disabled when writing

  // ********** CONSTRUCTION ************************************************

  /**
   * Construct an Attribute with the specified name and value.
   * @param name The name.
   * @param value The value.
   */
  public Attribute( String name, String value )
    {
    this.name = name;
    this.value = value;
    initialize();
    }

  /**
   * Construct an Attribute with the specified namespace prefix, name and value.
   * @param namespace The namespace prefix.
   * @param name The name.
   * @param value The value.
   */
  public Attribute( String prefix, String name, String value )
    {
    this.prefix = prefix;
    this.name = name;
    this.value = value;
    initialize();
    }

  /**
   * Construct a copy of the specified attribute.
   * @param attribute The Attribute to copy.
   */
  public Attribute( Attribute attribute )
    {
    this.prefix = attribute.prefix;
    this.namespace = attribute.namespace;
    this.name = attribute.name;
    this.value = attribute.value;
    this.isNamespace = attribute.isNamespace;
    this.isId = attribute.isId;
    this.raw = attribute.raw;
    }

  /**
   * @param name
   * @param value
   * @param isNamespace
   */
  Attribute( String name, String value, boolean isNamespace )
    {
    this.name = name;
    this.value = value;
    this.isNamespace = isNamespace;
    }

  /**
   * Construct an Attribute from the specified lexical analyzer.
   * @param lex The lexical analyzer.
   * @param element My element.
   * @throws IOException If an error occurs during parsing.
   */
  public Attribute( Lex lex, Element element )
    throws IOException
    {
    name = lex.readToken();

    if( lex.peek() == ':' )
      {
      prefix = name;
      lex.read();
      name = lex.readToken();
      }

    lex.readChar( '=' );
    lex.skipWhitespace();
    int ch = lex.read();

    if( ch == '\"' )
      value = lex.readToPattern( "\"", Lex.CONSUME | Lex.HTML );
    else if( ch == '\'' )
      value = lex.readToPattern( "'", Lex.CONSUME | Lex.HTML );
    else
      throw new IOException( "missing quote at start of attribute" );

    initialize();
    }

  // ********** STANDARD METHODS ********************************************

  /**
   * Return true if the object is an Attribute with the same name and value as myself.
   * @param object The object to compare.
   */
  public boolean equals( Object object )
    {
    return object instanceof Attribute
      && ((Attribute) object).name.equals( name )
      && ((Attribute) object).value.equals( value )
      && ArrayUtil.equals( ((Attribute) object).namespace, namespace );
    }

  /**
   * Return my hash code.
   */
  public int hashCode()
    {
    return name.hashCode() + value.hashCode();
    }

  /**
   * Return my description.
   */
  public String toString()
    {
    StringBuffer buffer = new StringBuffer();

    if( isNamespace )
      {
      buffer.append( "xmlns" );

      if( name.length() > 0 )
        buffer.append( ':' ).append( name );
      }
    else
      {
      if( prefix != null )
        buffer.append( prefix ).append( ':' );

      buffer.append( name );
      }

    buffer.append( "=\'" ).append( value ).append( '\'' );
    return buffer.toString();
    }

  // ********** CLONING *****************************************************

  /**
   * Return a clone of this Attribute.
   */
  public Object clone()
    {
    return new Attribute( this );
    }

  // ********** NAME ********************************************************

  /**
   * Return my name.
   */
  public String getName()
    {
    return name;
    }

  /**
   * Return my qualified name.
   */
  public String getQName()
    {
    return (namespace == null ? name : namespace + ":" + name);
    }

  /**
   * Return my prefix, or null if there is none.
   */
  public String getPrefix()
    {
    return prefix;
    }

  /**
   * Return my namespace, or null if there is none.
   */
  public String getNamespace()
    {
    return namespace;
    }

  /**
   * Return true if I am a namespace declaration.
   */
  public boolean isNamespace()
    {
    return isNamespace;
    }

  /**
   * Return true if my namespace and name match the specified values.
   * @param namespace The namespace.
   * @param name The name.
   */
  public boolean hasName( String namespace, String name )
    {
    if( !this.name.equals( name ) )
      return false;
    else if( namespace == null )
      return true;
    else if( this.namespace == null )
      return false;
    else
      return this.namespace.equals( namespace );
    }

  // ********** VALUE *******************************************************

  /**
   * Return my value.
   */
  public String getValue()
    {
    return value;
    }

  /**
   * Set my value.
   * @param value The new value.
   */
  public void setValue( String value )
    {
    this.value = value;
    }

  // ********** INITIALIZATION **********************************************

  /**
   *
   */
  private void initialize()
    {
    if( "xmlns".equals( name ) )
      {
      isNamespace = true;
      name = "";
      namespace = "http://www.w3.org/XML/1998/namespace";
      }
    else if( "xmlns".equals( prefix ) )
      {
      isNamespace = true;
      prefix = null;
      namespace = "http://www.w3.org/XML/1998/namespace";
      }
    else if( "id".equals( name ) )
      {
      isId = true;
      }
    }

  // ********** RESOLVING ***************************************************

  /**
   * If I represent a namespace, resolve my name.
   * @param element My element.
   * @throws NamespaceException If a namespace prefix could not be resolved.
   */
  void resolve( Element element )
    throws NamespaceException
    {
    this.element = element;

    if( prefix != null && !prefix.equals( "xml" ) )
      {
      namespace = element.getNamespace( prefix );

      if( namespace == null )
        throw new NamespaceException( "could not find namespace with prefix " + prefix );
      }
    }

  // ********** ELEMENT *****************************************************

  /**
   * Return the element that I'm an attribute of.
   */
  public Element getElement()
    {
    return element;
    }

  // ********** RAW MODE ****************************************************

  /**
   * Set my raw mode.
   * If true, subsitutions are disabled when I am written.
   * @param flag The new raw value.
   */
  public void setRaw( boolean flag )
    {
    raw = flag;
    }

  /**
   * Return my raw mode.
   */
  public boolean getRaw()
    {
    return raw;
    }

  // ********** WRITING *****************************************************

  /**
   * Write myself to the specified writer.
   * @param writer The nodeWriter.
   * @throws IOException If an I/O exception occurs.
   */
  public void write( NodeWriter writer )
    throws IOException
    {
    if( isNamespace )
      {
      writer.write( "xmlns" );

      if( name.length() > 0 )
        {
        writer.write( ':' );
        writer.write( name );
        }
      }
    else
      {
      if( prefix != null )
        {
        writer.write( prefix );
        writer.write( ':' );
        }

      writer.write( name );
      }

    writer.write( "=\'" );

    if( value == null )
      writer.write( "null" );
    else if( raw )
      writer.write( value );
    else
      Text.writeWithSubstitution( writer, value );

    writer.write( '\'' );
    }

  // ********** DOM *********************************************************

  /**
   * Return ATTRIBUTE_NODE.
   */
  public short getNodeType()
    {
    return ATTRIBUTE_NODE;
    }

  /**
   * Return the name of this attribute.
   */
  public String getNodeName()
    {
    return (isNamespace ? "xmlns:" + name : getQName());
    }

  /**
   * Return the value of this attribute.
   */
  public String getNodeValue()
    {
    return value;
    }

  /**
   * Set the value of this attribute.
   * @param nodeValue The new value.
   */
  public void setNodeValue( String nodeValue )
    {
    value = nodeValue;
    }

  /**
   * Set the namespace prefix of this attribute.
   * @param prefix The new namespace prefix.
   */
  public void setPrefix( String prefix )
    {
    this.prefix = prefix;
    }

  /**
   * Return a clone of this attribute.
   * @param deep If true, return a deep clone.
   */
  public org.w3c.dom.Node cloneNode( boolean deep )
    {
    return new Attribute( this );
    }

  /**
   * Return the Element node this attribute is attached to or null if this
   * attribute is not in use.
   */
  public org.w3c.dom.Document getOwnerDocument()
    {
    return (element != null ? element.getOwnerDocument() : null);
    }

  /**
   *
   */
  public String getLocalName()
    {
    return name;
    }

  /**
   * Return the namespace URI of this node, or null if it is unspecified.
   */
  public String getNamespaceURI()
    {
    return namespace;
    }

  /**
   * Return the Element node this attribute is attached to or null
   * if this attribute is not in use.
   */
  public org.w3c.dom.Element getOwnerElement()
    {
    return element;
    }

  /**
   * If this attribute was explicitly given a value in the original document,
   * this is true; otherwise, it is false
   */
  public boolean getSpecified()
    {
    return true; // for now
    }
  }