// copyright 2001-2002 by The Mind Electric

package electric.xml;

import java.io.*;
import electric.util.*;

/**
 * <tt>Instruction</tt> represents an XML processing instruction.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public class Instruction extends Child implements org.w3c.dom.ProcessingInstruction
  {
  static final String START = "<?";
  static final String STOP = "?>";

  protected String target;
  protected String content;

  // ********** CONSTRUCTION ************************************************

  /**
   * Construct an Instruction with the specified target and content.
   * @param target The target.
   * @param content The content.
   */
  public Instruction( String target, String content )
    {
    this.target = target;
    this.content = content;
    }

  /**
   * Construct a copy of the specified Instruction.
   * @param instruction The Instruction to copy.
   */
  public Instruction( Instruction instruction )
    {
    this.target = instruction.target;
    this.content = instruction.content;
    }

  /**
   *
   */
  protected Instruction()
    {
    }

  /**
   * @param parent
   */
  protected Instruction( Parent parent )
    {
    super( parent );
    }

  /**
   * Construct an Instruction from the specified lexical analyzer.
   * @param lex The lexical analyzer.
   * @param parent The parent of this Instruction.
   * @throws IOException If an error occurs during parsing.
   */
  Instruction( Lex lex, Parent parent )
    throws IOException
    {
    super( parent );
    lex.skip( START.length() );
    target = lex.readToken();
    content = lex.readToPattern( STOP, Lex.SKIP_WS | Lex.CONSUME );
    }

  // ********** CLONING *****************************************************

  /**
   * Return a clone of this Instruction.
   */
  public Object clone()
    {
    return new Instruction( this );
    }

  // ********** INSTRUCTION TARGET ******************************************

  /**
   * Return my target.
   */
  public String getTarget()
    {
    return target;
    }

  /**
   * Set my target.
   * @param target The new target value.
   */
  public void setTarget( String target )
    {
    this.target = target;
    }

  // ********** CONTENT *****************************************************

  /**
   * Return my content.
   */
  public String getContent()
    {
    return content;
    }

  /**
   * Set my content.
   * @param content The new content value.
   */
  public void setContent( String content )
    {
    this.content = content;
    }

  // ********** WRITING *****************************************************

  /**
   * Write myself to the specified writer.
   * @param writer The nodeWriter.
   * @throws IOException If an I/O exception occurs.
   */
  public void write( NodeWriter writer )
    throws IOException
    {
    writer.writeIndent();
    writer.write( START );
    writer.write( target );
    writer.write( ' ' );
    writer.write( getContent() );
    writer.write( STOP );
    }

  // ********** DOM *********************************************************

  /**
   * Return PROCESSING_INSTRUCTION_NODE.
   */
  public short getNodeType()
    {
    return PROCESSING_INSTRUCTION_NODE;
    }

  /**
   * Return the target.
   */
  public String getNodeName()
    {
    return getTarget();
    }

  /**
   * Return the content.
   */
  public String getNodeValue()
    {
    return getContent();
    }

  /**
   * Set the content.
   * @param nodeValue The new content value.
   */
  public void setNodeValue( String nodeValue )
    {
    setContent( nodeValue );
    }

  /**
   * Return the content.
   */
  public String getData()
    {
    return getContent();
    }

  /**
   * Set the content.
   * @param data The new content value.
   */
  public void setData( String data )
    {
    setContent( data );
    }
  }