/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.ide;

import ch.abacus.lib.ui.renderer.common.MetaObject;
import ch.abacus.designcockpit.external.IDEFocusRequester;

import java.awt.*;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

/**
 * Title:        uifactory
 * Description: implements a KeyListener interface for IDEViewDisplayerComponents objects. This interface is used to reposition objects or change the selected object.
 * Copyright:    Copyright (c) 2001
 * Company:      Abacus Research
 * @author Michael Gouker (Cagey Logic)
 * @version 1.0
 */
 public class IDEViewDisplayerComponentKeyListener implements KeyListener {
    MetaObject theDesignObject;
    SuperDesignCockpit theDesignCockpit;

    public IDEViewDisplayerComponentKeyListener(MetaObject objDesignObject) {
        theDesignObject = objDesignObject;
        theDesignCockpit = (SuperDesignCockpit) objDesignObject.theDesignProject.getMetaDataUser();
    }

    /**
     * Handles key presses in the design objects.<br><br>
     * Keys are handled in this order:<br>
     * 1) Control key - the selected objects are moved in the direction of the arrow key pressed. Special provision for the Home, Page Up, Page Down and End keys is enabled to allow diagonal movement.<br>
     * 2) Shift key - the last selected object is moved up/down the object hierarchy<br>
     * 3) If an arrow key is pressed, the focus is switched to either the previous or next object
     *
     * @param evt - KeyEvent object
     */
    public void keyPressed(KeyEvent evt) {
        int keyEventCode = evt.getKeyCode();

        if (evt.isControlDown()) {
           moveSelectedObjects(evt);
        }
        else if (evt.isShiftDown()) {
            changeObjectOrder(keyEventCode);
        }
        else
            navigateToAnotherObject(keyEventCode);

        evt.consume(); // todo: do we really need this? I thought we removed all Key Listeners from classes, anyway?
    }

    public void keyReleased(KeyEvent evt) {
        evt.consume();
    }

    public void keyTyped(KeyEvent evt) {
        evt.consume();
    }

    /**
     * Moves the selected objects in the direction of the arrow being pressed. The objects move faster if the Shift key is pressed. In addition, the following special keys are supported:
     * <UL>
     * <LI>Home moves North-West
     * <LI>Page Up moves North-East
     * <LI>Page Down moves South West
     * <LI>End moves South-East
     * </UL>
     * @param evt a KeyEvent object
     */
    private void moveSelectedObjects(KeyEvent evt) {
        int xDistance = 1;
        int yDistance = 1;
        if (evt.isShiftDown()) { // accelerate positioning by increasing the distance to move by
            xDistance = theDesignCockpit.thePreferences.iViewDisplayerGridX;
            yDistance = theDesignCockpit.thePreferences.iViewDisplayerGridY;
        }
        switch (evt.getKeyCode()) {
            case KeyEvent.VK_UP:
                moveByDistance(0, -yDistance);
                break;
            case KeyEvent.VK_DOWN:
                moveByDistance(0, yDistance);
                break;
            case KeyEvent.VK_LEFT:
                moveByDistance(-xDistance, 0);
                break;
            case KeyEvent.VK_RIGHT:
                moveByDistance(xDistance, 0);
                break;
            case KeyEvent.VK_HOME: // direction = North-West
                moveByDistance(-xDistance, -yDistance);
                break;
            case KeyEvent.VK_PAGE_UP: // direction = North-East
                moveByDistance(xDistance, -yDistance);
                break;
            case KeyEvent.VK_PAGE_DOWN: // direction = South-West
                moveByDistance(-xDistance, yDistance);
                break;
            case KeyEvent.VK_END: // direction = South-East
                moveByDistance(xDistance, yDistance);
                break;
        }
    }

    /**
     * Moves an object up or down within a level in the hierarchy
     * @param keyEventCode a KeyEvent.getKeyCode() value.
     */
    private void changeObjectOrder(int keyEventCode) {
        if (keyEventCode == KeyEvent.VK_UP)
            theDesignCockpit.theObjectTreeViewPane.moveUp();
        else if (keyEventCode == KeyEvent.VK_DOWN)
            theDesignCockpit.theObjectTreeViewPane.moveDown();
    }

    /**
     * navigateToAnotherObject
     *
     * Navigates the selection to a new object.  Need to use request focus to get key messages.
     * @param keyEventCode a KeyEvent.getKeyCode() value.
     */
    private void navigateToAnotherObject(int keyEventCode) {
        MetaObject newObject;
        switch (keyEventCode) {
            case KeyEvent.VK_UP:
                newObject = null;
                if (theDesignObject.thePreviousObject != null)
                    newObject = theDesignObject.thePreviousObject;
                else {
                    if (theDesignObject.theParentObject != null)
                        newObject = theDesignObject.theParentObject.theLastChild;
                }
                if (newObject != null) {
                    theDesignCockpit.deselectObjects();
                    theDesignCockpit.SelectObject(newObject, false);
                    IDEFocusRequester.requestFocus(newObject.theVisualObject);
                }
                break;
            case KeyEvent.VK_DOWN:
                newObject = null;
                if (theDesignObject.theNextObject != null)
                    newObject = theDesignObject.theNextObject;
                else {
                    if (theDesignObject.theParentObject != null)
                        newObject = theDesignObject.theParentObject.theFirstChild;
                }
                if (newObject != null) {
                    theDesignCockpit.deselectObjects();
                    theDesignCockpit.SelectObject(newObject, false);
                    IDEFocusRequester.requestFocus(newObject.theVisualObject);
                }
                break;
            case KeyEvent.VK_LEFT:
                newObject = theDesignObject.theParentObject;
                if (newObject != null) {
                    theDesignCockpit.deselectObjects();
                    theDesignCockpit.SelectObject(newObject, false);
                    IDEFocusRequester.requestFocus(newObject.theVisualObject);
                }
                break;
            case KeyEvent.VK_RIGHT:
                newObject = theDesignObject.theFirstChild;
                if (newObject != null) {
                    theDesignCockpit.deselectObjects();
                    theDesignCockpit.SelectObject(newObject, false);
                    IDEFocusRequester.requestFocus(newObject.theVisualObject);
                }
                break;
        }
    }

    /**
     * Repaints the sizeboxes of an object when it has been moved
     * @param theObject the object to repaint
     */
    private void repaintSizeboxesForMovedObject(MetaObject theObject) {
        Dimension dimSize = theObject.getSize();
        theObject.bAnchoringOff = true;
        if ((theObject.isAbalet() != true) && (theObject.isFrame() != true))
            theDesignCockpit.theSizeboxes.update(theObject, theObject.iXLocation,
                    theObject.iYLocation,
                    (int) dimSize.getWidth(),
                    (int) dimSize.getHeight());
    }

    /**
     * Moves an object by a relative distance
     * @param xDistance the horizontal distance to move
     * @param yDistance the vertical distance to move
     */
    private void moveByDistance(int xDistance, int yDistance){
        MetaObject theObject;
        int iNumSelectedObjects = theDesignCockpit.getDesignProject().getSelectedObjectCount();
        for (int iObject=0; iObject < iNumSelectedObjects; iObject++) {
            theObject = theDesignCockpit.getDesignProject().getSelectedObject(iObject);
            Point ptLocation = theObject.getLocation();
            theObject.setLocation(ptLocation.x + xDistance, ptLocation.y + yDistance);
            repaintSizeboxesForMovedObject(theObject); // don't update the property table
        }
        repaintSizeboxesForMovedObject(theDesignObject); // ensure the property table is updated with the last selected object
        theDesignCockpit.getPropertyInspectorController().refreshPropertyInspector();

    }
}



