/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.metadataEditor;

import ch.abacus.lib.ui.*;
import ch.abacus.lib.ui.renderer.common.MetaClassDetail;
import ch.abacus.lib.ui.renderer.common.MetaPropertyDescriptor;
import ch.abacus.lib.ui.renderer.common.IDEComponentPaletteNode;

import javax.swing.tree.TreeNode;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.event.TreeSelectionListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.*;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;
import javax.swing.table.DefaultTableModel;
import java.awt.*;
import java.awt.event.ActionEvent;

/**
 * Created by IntelliJ IDEA.
 * User: Stuart
 * Date: Nov 4, 2004
 * Time: 5:15:49 PM
 */
public class ClassPropertiesPage extends JAPanel {
    private JAPanel pnlProperties;
    private JAPanel pnlPropertiesButtons;
    private JAPanel pnlPropertyAttributes;
    private PropertiesTree treProperties;
    private PropertyAttributesTable tblPropertyAttributes;
    private JAPanel pnlEditButton;
    private final ManageClassDialog theDialog;
    private final MetaClassDetail theMetadata;

    public ClassPropertiesPage(ManageClassDialog dialog, MetaClassDetail metadata) {
        super(new BorderLayout());
        this.theDialog = dialog;
        theMetadata = metadata;
        createInterface();
        populateInterface();
    }

    private void createInterface() {
        treProperties = new PropertiesTree(theMetadata);
        pnlProperties = new JAPanel(new BorderLayout());

        pnlProperties.add(new JAScrollPane(treProperties), BorderLayout.CENTER);
        pnlPropertiesButtons = new JAPanel(new FlowLayout(FlowLayout.LEFT));
        pnlEditButton = new JAPanel(new FlowLayout(FlowLayout.LEFT));

        JAButton btnAdd = new JAButton("Add");
        JAButton btnDelete = new JAButton("Delete");
        btnAdd.addActionListener(new AddPropertyAction(treProperties));
        btnDelete.addActionListener(new DeletePropertyAction(treProperties));

        JAButton btnEdit = new JAButton("Edit");
        btnEdit.addActionListener(new EditPropertyAction(treProperties));

        pnlPropertiesButtons.add(btnAdd);
        pnlPropertiesButtons.add(btnDelete);
        pnlProperties.add(pnlPropertiesButtons, BorderLayout.SOUTH);
        //add(pnlProperties, BorderLayout.WEST);
        pnlPropertyAttributes = new JAPanel(new BorderLayout());
        tblPropertyAttributes = new PropertyAttributesTable();
        pnlPropertyAttributes.add(new JAScrollPane(tblPropertyAttributes), BorderLayout.CENTER);

        pnlEditButton.add(btnEdit);
        pnlPropertyAttributes.add(pnlEditButton, BorderLayout.SOUTH);
        //add(pnlPropertyAttributes, BorderLayout.CENTER);
        add(new JASplitPane(JSplitPane.HORIZONTAL_SPLIT, pnlProperties, pnlPropertyAttributes));
    }

    private void populateInterface() {
        treProperties.populateInterface();
        tblPropertyAttributes.populateInterface();
    }

    class DeletePropertyAction extends AbstractAction {
        private final PropertiesTree treProperties;

        public DeletePropertyAction(PropertiesTree treProperties) {
            this.treProperties = treProperties;
        }

        public void actionPerformed(ActionEvent e) {
            MetaPropertyDescriptor property = treProperties.getSelectedProperty();
            if (property != null) {
                String sPropertyName = property.getName();
                if (property.theSuperProperty != null) {
                    property.theSuperProperty.removeSubProperty(sPropertyName);
                } else {
                    theMetadata.removeProperty(sPropertyName);
                }
                populateInterface();
            }
        }
    }

    class AddPropertyAction extends AbstractAction {
        private final PropertiesTree treProperties;

        public AddPropertyAction(PropertiesTree treProperties) {
            this.treProperties = treProperties;
        }

        public void actionPerformed(ActionEvent e) {
            final MetaPropertyDescriptor parentProperty = treProperties.getSelectedProperty();

            // create a new property to be edited...
            final MetaPropertyDescriptor property = new MetaPropertyDescriptor(null, null, null, null, null, null, null, null, null, null, 0, theDialog.theEditor.theMetadataDispenser, theMetadata);
            property.bGenerate = false; // otherwise it will be written to the metadata
            final ManageClassPropertyDialog dialog = new ManageClassPropertyDialog(theDialog, property, parentProperty);
            dialog.pack();
            dialog.setLocationRelativeTo(null);
            dialog.show(new Runnable() {
                public void run () {
                    if (dialog.isDialogModified()) {
                        theDialog.theEditor.setMetadataModified(true);

                        if (parentProperty != null) { // sub-property
                            parentProperty.addSubproperty(property);
                        } else { // regular property
                            theMetadata.addProperties(property);
                        }
                        populateInterface();
                    }
                }
            });

        }
    }

    class EditPropertyAction extends AbstractAction {
        private final PropertiesTree treProperties;

        public EditPropertyAction(PropertiesTree treProperties) {
            this.treProperties = treProperties;
        }

        public void actionPerformed(ActionEvent e) {
            MetaPropertyDescriptor property = treProperties.getSelectedProperty();
            if (property == null) {
                JOptionPane.showMessageDialog(theDialog, "You must select a property before you can edit it.", "No Property Selected", JOptionPane.ERROR_MESSAGE);
                return;
            }
            final ManageClassPropertyDialog dialog = new ManageClassPropertyDialog(theDialog, treProperties.getSelectedProperty());

            dialog.pack();
            dialog.setLocationRelativeTo(null);
            dialog.show(new Runnable() {
                public void run () {
                    if (dialog.isDialogModified()) {
                        populateInterface();
                        theDialog.theEditor.setMetadataModified(true);
                    }
                }
            });

        }
    }


    class PropertiesTree extends JATree {
        private final MetaClassDetail theMetadata;

        public PropertiesTree(MetaClassDetail metadata) {
            super();
            theMetadata = metadata;
            addTreeSelectionListener(new PropertySelectionListener());
        }

        void populateInterface() {
            setModel(new DefaultTreeModel(new ClassNode("Class Properties")));
            MetaPropertyDescriptor property = theMetadata.theFirstProperty;
            while (property != null) {
                ((ClassNode)getModel().getRoot()).addPropertyNode(property);
                property = property.getNextProperty();
            }
            expandRow(0);
            setSelectionRow(0);
        }

        MetaPropertyDescriptor getSelectedProperty() {
            ClassNode node = (ClassNode) treProperties.getLastSelectedPathComponent();
            if (node.getNodeType() == ClassNode.PROPERTY_NODE) {
                return node.getMetaProperty();
            }

            return null;
        }

        class ClassNode extends DefaultMutableTreeNode {
            static final int TEXT_NODE = 0;
            static final int PROPERTY_NODE = 1;

            private final int nodeType;
            private final MetaPropertyDescriptor metaProperty;

            public ClassNode(MetaPropertyDescriptor metaProperty) {
                super(metaProperty.getName());
                this.metaProperty = metaProperty;
                nodeType = PROPERTY_NODE;
            }

            public ClassNode(String userObject) {
                super(userObject);
                nodeType = TEXT_NODE;
                metaProperty = null;
            }

            public MetaPropertyDescriptor getMetaProperty() {
                return metaProperty;
            }

            public int getNodeType() {
                return nodeType;
            }

            void addPropertyNode (MetaPropertyDescriptor property) {
                ClassNode propertyNode = new ClassNode(property);

                MetaPropertyDescriptor subProperty = property.theFirstSubproperty;
                while (subProperty != null) {
                    ClassNode subNode = new ClassNode(subProperty);
                    propertyNode.add(subNode);
                    subProperty = subProperty.getNextProperty();
                }

                add(propertyNode);
            }

        }

        class PropertySelectionListener implements TreeSelectionListener {

            public void valueChanged(TreeSelectionEvent e) {
                ClassNode node = ((ClassNode)e.getPath().getLastPathComponent());
                //System.out.println(node + " " + node.nodeType);
                if (node.nodeType == ClassNode.TEXT_NODE) {
                    //System.out.println("Text");
                    tblPropertyAttributes.displayNoAttributes();
                } else {
                    MetaPropertyDescriptor property = node.getMetaProperty();
                    //System.out.println(property.getGetMethod() + " " + property.getSetMethod());
                    tblPropertyAttributes.displayPropertyAttributes(property);
                }
            }
        }
    }

    class PropertyAttributesTable extends JATable {
        public PropertyAttributesTable() {
            super ();
            setModel(new MyTableModel());
            setRowSelectionAllowed(false);
        }

        void populateInterface() {
            tblPropertyAttributes.setValueAt("Name", 0,0);
            tblPropertyAttributes.setValueAt("Class", 1,0);
            tblPropertyAttributes.setValueAt("Get", 2,0);
            tblPropertyAttributes.setValueAt("Set", 3,0);
            tblPropertyAttributes.setValueAt("Constant Group", 4,0);
            tblPropertyAttributes.setValueAt("Default Value", 5,0);
            tblPropertyAttributes.setValueAt("Visible", 6,0);
            tblPropertyAttributes.setValueAt("Array", 7,0);
            tblPropertyAttributes.setValueAt("Languages", 8,0);
            tblPropertyAttributes.setValueAt("Read-Only", 9,0);
            tblPropertyAttributes.setValueAt("Helper", 10,0);
        }

        public void displayPropertyAttributes(MetaPropertyDescriptor property) {
            tblPropertyAttributes.setValueAt(property.getName(), 0, 1);
            tblPropertyAttributes.setValueAt(property.getClassName(), 1, 1);
            tblPropertyAttributes.setValueAt(property.getGetMethod(), 2, 1);
            tblPropertyAttributes.setValueAt(property.getSetMethod(), 3, 1);
            tblPropertyAttributes.setValueAt(property.getIncludeTypeConstants(), 4, 1);
            tblPropertyAttributes.setValueAt(property.getDefaultValue(), 5, 1);
            tblPropertyAttributes.setValueAt(new Boolean(property.bVisible), 6, 1);
            tblPropertyAttributes.setValueAt(new Boolean(property.bAllowMultipleValues), 7, 1);
            tblPropertyAttributes.setValueAt(new Boolean(property.bAllowMultipleLanguages), 8, 1);
            tblPropertyAttributes.setValueAt(new Boolean(property.isReadOnly()), 9, 1);
            tblPropertyAttributes.setValueAt(property.getHelperClass(), 10, 1);
        }

        public void displayNoAttributes() {
            tblPropertyAttributes.setValueAt("", 0, 1);
            tblPropertyAttributes.setValueAt("", 1, 1);
            tblPropertyAttributes.setValueAt("", 2, 1);
            tblPropertyAttributes.setValueAt("", 3, 1);
            tblPropertyAttributes.setValueAt("", 4, 1);
            tblPropertyAttributes.setValueAt("", 5, 1);
            tblPropertyAttributes.setValueAt("", 6, 1);
            tblPropertyAttributes.setValueAt("", 7, 1);
            tblPropertyAttributes.setValueAt("", 8, 1);
            tblPropertyAttributes.setValueAt("", 9, 1);
            tblPropertyAttributes.setValueAt("", 10, 1);
        }

        private class MyTableModel extends DefaultTableModel {
            public int getColumnCount() {
                return 2;
            }

            public int getRowCount() {
                return 11;
            }

            public String getColumnName(int columnIndex) {
                if (columnIndex == 0)
                    return "Attribute Name";
                else
                    return "Attribute Value";
            }

            public boolean isCellEditable(int row, int column) {
                return false;
            }
        }
    }
}
