/*
 * Copyright (c) 1996 IBM Corp. All Rights Reserved.
 */

import java.awt.*;
import java.awt.image.*;
import java.lang.*;
import java.util.*;




/**
** The PMLayout, along with the PMConstraint class, provides functionality
** similar to that of the PM placement. As components are added to
** a Container using the PMLayout manager, a PMConstraint should also
** be built and used to setConstraints on the PMLayout.
**
*/
public class PMLayout extends Object implements LayoutManager {


  /**
  ** Holds the PMConstraint to Component binding for the layout
  */
   private protected Hashtable constraints = new Hashtable();

  // Keep a pointer to the Container we are helping out with the layout
   private Container ltarget = null;

  /**
  ** Store the initial PM window dimension without insets
  */
   protected Dimension d;
   protected Dimension szPrevious=new Dimension(0,0);


  /**
  ** Construct a PMLayout object.
  */
   public PMLayout() {
   }

  /**
  ** Construct a PMLayout object, and set the horizontalSpacing to hspace,
  ** the verticalSpacing to vspace, and the fractionBase to fbase.
  */
   public PMLayout(int cx,int cy) {
      d=new Dimension(cx,cy);

   }

  /**
  ** Add a component to be managed by this PMLayout. This routine is here to
  ** fulfill the interface commitment for the LayoutManager interface, and is
  ** currently ignored.
  */
   public void addLayoutComponent(String name, Component comp) {


   }

  /**
  ** Remove a component to be managed by this PMLayout. This routine is here
  ** to fulfill the interface commitment for the LayoutManager interface
  ** it removes the element also from the constaints.
  */
   public void removeLayoutComponent(Component comp) {
      constraints.remove(comp);
   }

  /**
  ** This method provides the means of associating a Component with its
  ** constraints which will ultimately drive the layout. The PMConstraint
  ** object is not kept by the PMLayout, so the caller can reuse the same
  ** PMConstraint object for all PMLayout settings (checkout the
  ** reset method on PMConstraint).
  **<p>
  ** When specifying constraints, make sure to specify the PM coordinate in
  ** its parent coordinates.
  */
 ////////////////////////
   public void setConstraints(Component comp, PMConstraint c) {
 ////////////////////////

      if (ltarget != null) ltarget.invalidate();

      PMConstraint PMc = new PMConstraint(c);
      PMc.component = comp;
      constraints.put(comp, PMc);
   }

  /**
  ** Get a copy of the PMConstraint registered for the specified Component.
  ** If the specified Component is not registered with the layout, then
  ** a reset constraint object will be returned.
  */
 ////////////////////////
   public PMConstraint getConstraints(Component comp) {
 ////////////////////////

      return new PMConstraint(((PMConstraint)constraints.get(comp)));
   }


  /**
  ** Fulfill the LayoutManager Interface definition by calculating and
  ** returning the preferred size for the current contraints
  */
 ////////////////////////
   public Dimension preferredLayoutSize(Container target) {
 ////////////////////////
      Insets insets=target.insets();
      return new Dimension(d.width+insets.left+insets.right,
                           d.height+insets.top+insets.bottom);
   }

  /**
  ** Fulfill the LayoutManager Interface definition by calculating and
  ** returning the minimum size for the current contraints
  */
 ////////////////////////
   public Dimension minimumLayoutSize(Container target) {
 ////////////////////////
      Insets insets=target.insets();
      return new Dimension(d.width+insets.left+insets.right,
                           d.height+insets.top+insets.bottom);
   }

  /**
  ** Fulfill the LayoutManager Interface definition by applying the
  ** constraints to the target container using the containers current size
  ** for layout.
  */
 ////////////////////////
   public void layoutContainer(Container target) {
 ////////////////////////
      PMConstraint PMc;

      Dimension sz=target.size();
      Insets    insets;


      /* do not perform same job twice */
      if ((szPrevious.width !=sz.width)||
          (szPrevious.height!=sz.height)) {
         int cnt= target.countComponents();
         if (target instanceof Frame) {
            insets=target.insets();
            sz.width -=(insets.left+insets.right);
            sz.height-=(insets.top+insets.bottom);
         } else {
            insets=new Insets(0,0,0,0);
         }
         szPrevious.width =sz.width;
         szPrevious.height=sz.height;

         for(int i=0; i < cnt; i++) {
            Component c = target.getComponent(i);
            if (c.isVisible()) {
               PMc = (PMConstraint)constraints.get(c);
               // If constraints are set, reshape object at valid position
               // Insets should already be included in coordinates
               if (PMc != null) {
                  Rectangle rc;
                  rc=PMc.bounds();
                  if (d==null) {
                     c.reshape(rc.x,
                               (sz.height-rc.height-rc.y),
                               rc.width, rc.height);
                  } else {
                     c.reshape(insets.left
                               +(rc.x*sz.width)/d.width,
                               insets.top
                               +((d.height-rc.height-rc.y)*sz.height)/d.height,
                               (rc.width*sz.width)/d.width,
                               (rc.height*sz.height)/d.height);

                  }
               }
            }
         }
      }
   }


  /**
  ** print all the constraints for this PMLayout
  */
 ////////////////////////
   public String toString() {
 ////////////////////////

      return "PMLayout: [" + constraints.toString() + "]";
   }
}
