/******************************************************************************
 * Project JKobo
 * XKobo arcade style game
 *
 * Copyright (C) 1997 John "Jake" Wetzel All Rights Reserved
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 *****************************************************************************/

/***************************** FILE INFORMATION ******************************
 * NAME OF FILE 	  : PlayWin.java
 * CONCEPT & ARCHITECTURE : Akira Higuchi
 * MAIN PROGRAMMER	  :
 * DATE CREATED 	  :
 * DESCRIPTION		  :
 * CHANGE LOG		  :
 *	  WHO		  WHEN			  WHY
 *	  ---		  ----			  ---
 *
 *****************************************************************************/

/******************************* IMPORT FILES ********************************/

import java.awt.Canvas;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.event.InputEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseMotionAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.lang.Math;
import java.text.MessageFormat;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import Manage;
import ScaleParms;
import Tiff;

class PlayWin extends Frame implements GlobalDefines
{

  static final int TITLE   = 0;
  static final int PLAYING = 1;
  static final int PAUSED  = 2;

  static final String CheatShips = "999999999";
  static String CheatMessage;
  static String StartMessage;
  static String StageMessage;
  static String QuitMessage;
  static String ScoreName;
  static String StageName;
  static String ShipName;
  static String PauseMessage;

/******************************** GLOBAL DATA ********************************/

  PlayWinCanvas canvas;
  int radar_x;
  int radar_y;
  int radar_width;
  int radar_height;
  int viewport_x;
  int viewport_y;
  int viewport_width;
  int viewport_height;
  int x_offset;
  int y_offset;
  int mouse_x;
  int mouse_y;
  Color text_info;
  boolean left;
  boolean right;
  boolean up;
  boolean down;
  boolean shot;
  boolean isIcon;
  boolean info_update;
  byte direction;
  String instructions[];
  Image window_bitmap;	 /* bitmap on which viewport area of space_bitmap is   *
			  * blitted and enemies are blitted		       */
  Graphics window_gc;

/**************************** FUNCTION DECLARATIONS  *************************/

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : PlayWin
 * DESCRIPTION	  :
 *****************************************************************************/
  PlayWin()
  {
    int wsize, sl;
    Font f, nf;
    Object objs[];



    setTitle(JKobo.NAME);
    canvas = new PlayWinCanvas();
    add(canvas, "Center");
    pack();
    mouse_x = mouse_y = 0;
    left = right = up = down = shot = isIcon = false;
    direction = 1;
    info_update = true;

    wsize = JKobo.Context.getScaleParms().getWindowWidth();
    window_bitmap = canvas.createImage(wsize, wsize);
    window_gc = window_bitmap.getGraphics();

    text_info = new Color(Color.green.getRed(), Color.green.getGreen(),
			  Color.green.getBlue());

    sl = JKobo.Context.getScaleLog2();
    if(sl > 0)
      {
	f = getFont();
	nf = new Font(f.getName(), f.getStyle(), f.getSize() << sl);
	setFont(nf);
      }

    instructions = new String[3];
    objs = new Object[1];
    objs[0] = (Object) "'s'";
    instructions[0] = MessageFormat.format(StartMessage, objs);
    objs[0] = (Object) "'q'";
    instructions[2] = MessageFormat.format(QuitMessage, objs);
    objs = new Object[2];
    objs[0] = (Object) "'+'";
    objs[1] = (Object) "'-'";
    instructions[1] = MessageFormat.format(StageMessage, objs);

    canvas.setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
    canvas.addKeyListener(new KeyAdapter()
      {
	public void keyPressed(KeyEvent event)
	{

	  if((event.getModifiers() &
	      (InputEvent.ALT_MASK | InputEvent.CTRL_MASK |
	       InputEvent.META_MASK | InputEvent.SHIFT_MASK)) > 0)
	    return;

	  parseKey(event.getKeyCode(), true);
	}
	public void keyReleased(KeyEvent event)
	{
	  int kc;

	  if((event.getModifiers() &
	      (InputEvent.ALT_MASK | InputEvent.CTRL_MASK |
	       InputEvent.META_MASK | InputEvent.SHIFT_MASK)) > 0)
	    return;

	  kc = event.getKeyCode();
	  if(kc == KeyEvent.VK_S)
	    JKobo.Context.getManage().startKey();
	  else if(kc == KeyEvent.VK_Q)
	    JKobo.Context.getManage().exitKey();
	  else if(kc == KeyEvent.VK_SUBTRACT)
	    JKobo.Context.getManage().minusKey();
	  else if(kc == KeyEvent.VK_ADD)
	    JKobo.Context.getManage().plusKey();
	  else
	    parseKey(kc, false);
	}
      });
    canvas.addMouseListener(new MouseAdapter()
      {
	public void mousePressed(MouseEvent event)
	{
	  int mod;

	  mod = event.getModifiers();
	  if((mod & (InputEvent.ALT_MASK | InputEvent.CTRL_MASK |
		     InputEvent.META_MASK | InputEvent.SHIFT_MASK)) > 0)
	    return;

	  if(JKobo.Context.getManage().statePlaying() == false)
	    return;

	  if((mod & InputEvent.BUTTON1_MASK) > 0)
	    shot = true;

	}
	public void mouseReleased(MouseEvent event)
	{
	  int mod;

	  mod = event.getModifiers();
	  if((mod & (InputEvent.ALT_MASK | InputEvent.CTRL_MASK |
		     InputEvent.META_MASK | InputEvent.SHIFT_MASK)) > 0)
	    return;

	  if(JKobo.Context.getManage().statePlaying() == false)
	    return;

	  if((mod & InputEvent.BUTTON1_MASK) > 0)
	    shot = false;
	  else if((mod & InputEvent.BUTTON3_MASK) > 0)
	    JKobo.Context.getManage().startKey();

	}
	public void mouseEntered(MouseEvent event)
	{
	  Manage m;
	  int state;

	  m = JKobo.Context.getManage();
	  state = m.getState();
	  if(state == PAUSED)
	    m.startKey();
	}
	public void mouseExited(MouseEvent event)
	{
	  Manage m;
	  int state;

	  shot = false;
	  m = JKobo.Context.getManage();
	  state = m.getState();
	  if(state == PLAYING)
	    m.startKey();
	}
      });
    canvas.addMouseMotionListener(new MouseMotionAdapter()
      {
	public void mouseDragged(MouseEvent event)
	{

	  if((event.getModifiers() &
	      (InputEvent.ALT_MASK | InputEvent.CTRL_MASK |
	       InputEvent.META_MASK | InputEvent.SHIFT_MASK)) > 0)
	    return;

	  if(JKobo.Context.getManage().statePlaying() == false)
	    return;

	  mousePosition(event.getX(), event.getY());

	}
	public void mouseMoved(MouseEvent event)
	{

	  if((event.getModifiers() &
	      (InputEvent.ALT_MASK | InputEvent.CTRL_MASK |
	       InputEvent.META_MASK | InputEvent.SHIFT_MASK)) > 0)
	    return;

	  if(JKobo.Context.getManage().statePlaying() == false)
	    return;

	  mousePosition(event.getX(), event.getY());

	}
      });
    super.addWindowListener(new WindowAdapter()
      {
	public void windowClosing(WindowEvent event)
	{
	  close();
	}
	public void windowDeiconified(WindowEvent event)
	{
	  isIcon = false;
	}
	public void windowIconified(WindowEvent event)
	{
	  isIcon = info_update = true;
	}
      });
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : init
 * DESCRIPTION	  :
 *****************************************************************************/
  void init()
  {
    int wsize;
    Dimension maxd, d;
    Point p;
    Toolkit tk;
    ScaleParms sp;


    sp = JKobo.Context.getScaleParms();
    wsize = sp.getWindowWidth();
    radar_x = (MARGIN << 1) + wsize;
    radar_width = sp.getRadarWidth();
    radar_height = sp.getRadarHeight();
    radar_y = MARGIN  + (wsize - radar_height);
    viewport_x = viewport_y = MARGIN;
    viewport_width = viewport_height = wsize;
    x_offset = y_offset = 0;

    tk = getToolkit();
    d = getSize();
    maxd = tk.getScreenSize();
    maxd.width >>>= 1;
    p = new Point();
    if(d.width < maxd.width)
      p.x = (maxd.width - d.width) >>> 1;
    else
      p.x = maxd.width - (d.width >>> 1);
    p.y = (maxd.height - d.height) >>> 1;
    setLocation(p);

    if(isVisible() == false)
      setVisible(true);

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : parseKey
 * DESCRIPTION	  :
 *****************************************************************************/
  void parseKey(int keycode, boolean value)
  {

    switch(keycode)
      {
      case KeyEvent.VK_SPACE:
	shot = value;
	break;
      case KeyEvent.VK_PAGE_DOWN:
	right = down = value;
	break;
      case KeyEvent.VK_PAGE_UP:
	up = right = value;
	break;
      case KeyEvent.VK_DOWN:
	down = value;
	break;
      case KeyEvent.VK_LEFT:
	left = value;
	break;
      case KeyEvent.VK_RIGHT:
	right = value;
	break;
      case KeyEvent.VK_UP:
	up = value;
	break;
      case KeyEvent.VK_END:
	left = down = value;
	break;
      case KeyEvent.VK_HOME:
	up = left = value;
	break;
      }

    if(up == true && down == true)
      up = down = false;
    if(left == true && right == true)
      left = right = false;
    if(left == true)
      {
	if(up == true)
	  direction = 8;
	else if(down == true)
	  direction = 6;
	else
	  direction = 7;
      }
    else if(right == true)
      {
	if(up == true)
	  direction = 2;
	else if(down == true)
	  direction = 4;
	else
	  direction = 3;
      }
    else
      {
	if(up == true)
	  direction = 1;
	else if(down == true)
	  direction = 5;
      }
    if(left == true || right == true || up == true || down == true)
      mouse_x = mouse_y = 0;

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : mousePosition
 * DESCRIPTION	  :
 *****************************************************************************/
  void mousePosition(int mx, int my)
  {
     int tx, ty;


     /*
      * Directions:
      *
      * 	1
      *      8	|  2
      *    7 ---*--- 3
      *      6	|  4
      * 	5
      */

     mouse_x = tx = mx - viewport_x - (viewport_width >>> 1);
     mouse_y = ty = my - viewport_y - (viewport_height >>> 1);

     if(tx > 0)
       {
	 if(ty > 0)
	   {
	     if(tx > (ty << 1))
	       direction = 3;
	     else if(ty > (tx << 1))
	       direction = 5;
	     else
	       direction = 4;
	   }
	 else if(ty < 0)
	   {
	     if(tx > ((-ty) << 1))
	       direction = 3;
	     else if((-ty) > (tx << 1))
	       direction = 1;
	     else
	       direction = 2;
	   }
	 else
	   direction = 3;
       }
     else if(tx < 0)
       {
	 if(ty > 0)
	   {
	     if((-tx) > (ty << 1))
	       direction = 7;
	     else if(ty > ((-tx) << 1))
	       direction = 5;
	     else
	       direction = 6;
	   }
	 else if(ty < 0)
	   {
	     if((-tx) > ((-ty) << 1))
	       direction = 7;
	     else if((-ty) > ((-tx) << 1))
	       direction = 1;
	     else
	       direction = 8;
	   }
	 else
	   direction = 7;
       }
     else if(ty > 0)
       direction = 5;
     else
       direction = 1;

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : toggleInfoUpdate
 * DESCRIPTION	  :
 *****************************************************************************/
  void toggleInfoUpdate()
  {
    info_update = true;
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : setFont
 * DESCRIPTION	  :
 *****************************************************************************/
  public synchronized void setFont(Font f)
  {
    super.setFont(f);
    canvas.setFont(f);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : setForeground
 * DESCRIPTION	  :
 *****************************************************************************/
  public void setForeground(Color c)
  {
    super.setForeground(c);
    canvas.setForeground(c);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : setBackground
 * DESCRIPTION	  :
 *****************************************************************************/
  public void setBackground(Color c)
  {
    super.setBackground(c);
    canvas.setBackground(c);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : clear
 * DESCRIPTION	  :
 *****************************************************************************/
  void clear()
  {
    Graphics g;
    Rectangle r;

    g = canvas.getGraphics();
    r = g.getClipBounds();
    g.clearRect(r.x, r.y, r.width, r.height);
    g.dispose();

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : updateAll
 * DESCRIPTION	  :
 *****************************************************************************/
  void updateAll()
  {
    Graphics g;


    if(JKobo.Context.getManage().statePlaying() == true)
      {
	window_gc.drawImage(JKobo.Context.getSpace().getSpaceBitmap(),
			    0, 0, viewport_width, viewport_height,
			    x_offset, y_offset,
			    x_offset + viewport_width,
			    y_offset + viewport_height, null);
	JKobo.Context.getEnemies().put();
	JKobo.Context.getMyShip().put();
      }
    g = canvas.getGraphics();
    canvas.paint(g);
    g.dispose();

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : drawString
 * DESCRIPTION	  :
 *****************************************************************************/
  void drawString(int x, int y, String s)
  {
    Graphics g;

    g = canvas.getGraphics();
    g.drawString(s, x, y);
    g.dispose();

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getXOffset
 * DESCRIPTION	  :
 *****************************************************************************/
  final int getXOffset()
  {
    return(x_offset);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : setXOffset
 * DESCRIPTION	  :
 *****************************************************************************/
  void setXOffset(int x)
  {
    x_offset = x;
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getYOffset
 * DESCRIPTION	  :
 *****************************************************************************/
  final int getYOffset()
  {
    return(y_offset);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : setYOffset
 * DESCRIPTION	  :
 *****************************************************************************/
  void setYOffset(int y)
  {
    y_offset = y;
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getViewportWidth
 * DESCRIPTION	  :
 *****************************************************************************/
  final int getViewportWidth()
  {
    return(viewport_width);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getViewportHeight
 * DESCRIPTION	  :
 *****************************************************************************/
  final int getViewportHeight()
  {
    return(viewport_height);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : close
 * DESCRIPTION	  :
 *****************************************************************************/
  void close()
  {

    if(isVisible() == true)
      setVisible(false);

    isIcon = false;

    JKobo.Context.getRadar().close();
    JKobo.Context.getSpace().close();
    JKobo.Context.getScoreWin().close();
    JKobo.Context.getTimerThread().stop();
    JKobo.Context.getLoopThread().stop();
    window_gc.dispose();

    dispose();

    System.exit(0);

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getWindowBitmap
 * DESCRIPTION	  :
 *****************************************************************************/
  final Image getWindowBitmap()
  {
    return(window_bitmap);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getWindowGraphics
 * DESCRIPTION	  :
 *****************************************************************************/
  final Graphics getWindowGraphics()
  {
    return(window_gc);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getDirection
 * DESCRIPTION	  :
 *****************************************************************************/
  final int getDirection()
  {
    return((int) direction);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getShot
 * DESCRIPTION	  :
 *****************************************************************************/
  final boolean getShot()
  {
    return(shot);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getMouseX
 * DESCRIPTION	  :
 *****************************************************************************/
  final int getMouseX()
  {
    return(mouse_x);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getMouseY
 * DESCRIPTION	  :
 *****************************************************************************/
  final int getMouseY()
  {
    return(mouse_y);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getCanvas
 * DESCRIPTION	  :
 *****************************************************************************/
  final Canvas getCanvas()
  {
    return((Canvas) canvas);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : copySprite
 * DESCRIPTION	  :
 *****************************************************************************/
  boolean copySprite(int sx, int sy, int w, int h, int dx, int dy)
  {
    int tx, ty;


    if(dx > (x_offset + viewport_width))
      return(false);
    if((dx + w) < x_offset)
      return(false);
    if(dy > (y_offset + viewport_height))
      return(false);
    if((dy + h) < y_offset)
      return(false);

    tx = dx - x_offset;
    ty = dy - y_offset;
    return(window_gc.drawImage(JKobo.Context.getEnemies().getEnemyBitmap(),
			       tx, ty, tx + w, ty + h, sx, sy, sx + w, sy + h,
			       null));

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : static
 * DESCRIPTION	  :
 *****************************************************************************/
  static
  {
    ResourceBundle bundle;


    bundle = JKobo.Context.getResourceBundle();

    try {
      CheatMessage = bundle.getString("PlayWinCheatMode.message");
    }
    catch(MissingResourceException e) {
      CheatMessage = "cheat mode";
    }
    try {
      StartMessage = bundle.getString("PlayWinInstructions1.message");
    }
    catch(MissingResourceException e) {
      StartMessage = "push {0} ky to start a game";
    }
    try {
      StageMessage = bundle.getString("PlayWinInstructions2.message");
    }
    catch(MissingResourceException e) {
      StageMessage = "{0}, {1} to choose a stage";
    }
    try {
      QuitMessage = bundle.getString("PlayWinInstructions3.message");
    }
    catch(MissingResourceException e) {
      QuitMessage = "{0} to quit";
    }
    try {
      ScoreName = bundle.getString("PlayWinScore.message");
    }
    catch(MissingResourceException e) {
      ScoreName = "SCORE";
    }
    try {
      StageName = bundle.getString("PlayWinStage.message");
    }
    catch(MissingResourceException e) {
      StageName = "STAGE";
    }
    try {
      ShipName = bundle.getString("PlayWinShips.message");
    }
    catch(MissingResourceException e) {
      ShipName = "SHIPS";
    }
    try {
      PauseMessage = bundle.getString("PlayWinPaused.message");
    }
    catch(MissingResourceException e) {
      PauseMessage = "PAUSED";
    }

  }

  class PlayWinCanvas extends Canvas
  {
/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : update
 * DESCRIPTION	  :
 *****************************************************************************/
    public void update(Graphics g)
    {
      info_update = true;
      super.update(g);
    }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : paint
 * DESCRIPTION	  :
 *****************************************************************************/
    public void paint(Graphics g)
    {
      int s;

/*
      System.out.println("painting ...");
*/
      if(isIcon == true)
	return;

      s = JKobo.Context.getManage().getState();
      if(s == TITLE)
	paintTitle(g);
      else if(s == PLAYING)
	paintPlaying(g);
      else if(s == PAUSED)
	paintPaused(g);
      if(info_update == true)
	{
	  paintInfo(g);
	  info_update = false;
	}
      paintRadar(g);

    }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : paintTitle
 * DESCRIPTION	  :
 *****************************************************************************/
    void paintTitle(Graphics g)
    {
      int x, y, xb, yb, sl;
      String s;
      Rectangle r;
      Color c[];
      Font f, df;
      FontMetrics fm;

      sl = JKobo.Context.getScaleLog2();
      df = g.getFont();
      f = new Font(df.getName(), df.getStyle(), df.getSize() * 3);
      fm = g.getFontMetrics(f);

      s = JKobo.Context.getManage().getHighScore();
      c = JKobo.Context.getSpace().getPixels();
      r = g.getClipBounds();
      g.setClip(viewport_x, viewport_y, viewport_width, viewport_height);
      g.clearRect(viewport_x, viewport_y, viewport_width, viewport_height);
      info_update = true;

      g.setColor(c[0]);
      g.setFont(f);
      xb = viewport_x + (viewport_width >>> 1);
      yb = viewport_y + (viewport_height >>> 2);
      x = xb - (fm.stringWidth(JKobo.NAME) >>> 1);
      y = yb - fm.getMaxDescent();
      g.drawString(JKobo.NAME, x, y);

      g.setColor(c[1]);
      g.setFont(df);
      fm = g.getFontMetrics(df);
      x = xb - (fm.stringWidth(JKobo.VERSION) >>> 1);
      y = yb + fm.getMaxAscent() + fm.getLeading();
      g.drawString(JKobo.VERSION, x, y);
      x = xb - (fm.stringWidth(JKobo.DATE) >>> 1);
      y += fm.getHeight();
      g.drawString(JKobo.DATE, x, y);

      yb = viewport_y + (viewport_height >>> 1) + fm.getHeight() +
	    fm.getMaxAscent();
      y = yb;
      if(JKobo.Context.getCheatMode() == true)
	{
	  x = xb - (fm.stringWidth(CheatMessage) >>> 1);
	  g.drawString(CheatMessage, x, y);
	}
      else
	{
	  x = xb - (fm.stringWidth(s) >>> 1);
	  g.drawString(s, x, y);
	}
      y += (fm.getHeight() << 1);
      g.setColor(c[2]);
      x = xb - (fm.stringWidth(instructions[0]) >>> 1);
      g.drawString(instructions[0], x, y);
      x = xb - (fm.stringWidth(instructions[1]) >>> 1);
      y += fm.getHeight();
      g.drawString(instructions[1], x, y);
      x = xb - (fm.stringWidth(instructions[2]) >>> 1);
      y += fm.getHeight();
      g.drawString(instructions[2], x, y);
      g.setClip(r);

    }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : paintInfo
 * DESCRIPTION	  :
 *****************************************************************************/
    void paintInfo(Graphics g)
    {
      int x, y, l1, l2, l3, l4;
      int h, xm;
      Font f;
      FontMetrics fm;
      Rectangle r;
      String s;
      Manage m;


      r = g.getClipBounds();
      h = viewport_height - radar_height;
      g.setClip(radar_x, viewport_y, radar_width, h);
      g.clearRect(radar_x, viewport_y, radar_width, h);
      g.setColor(text_info);
      m = JKobo.Context.getManage();
      f = g.getFont();
      fm = g.getFontMetrics(f);
      xm = l1 = fm.stringWidth(ScoreName);
      l2 = fm.stringWidth(StageName);
      xm = Math.max(l2, xm);
      l3 = fm.stringWidth(ShipName);
      xm = Math.max(l3, xm);
      l4 = fm.stringWidth(CheatShips);
      xm = Math.max(l4, xm);

      x = radar_x + ((radar_width - xm) >>> 1);
      y = viewport_y + (h >>> 1) - (fm.getHeight() * 3) + fm.getMaxAscent();
      g.drawString(ScoreName, x+xm-l1, y);
      s = String.valueOf(m.getScore());
      l1 = fm.stringWidth(s);
      y += fm.getHeight();
      g.drawString(s, x+xm-l1, y);

      y += fm.getHeight();
      g.drawString(StageName, x+xm-l2, y);
      s = String.valueOf(m.getSceneNum() + 1);
      l2 = fm.stringWidth(s);
      y += fm.getHeight();
      g.drawString(s, x+xm-l2, y);

      y += fm.getHeight();
      g.drawString(ShipName, x+xm-l3, y);
      y += fm.getHeight();
      if(JKobo.Context.getCheatMode() == false)
	{
	  s = String.valueOf(m.getShips());
	  l3 = fm.stringWidth(s);
	  g.drawString(s, x+xm-l3, y);
	}
      else
	{
	  l4 = fm.stringWidth(CheatShips);
	  g.drawString(CheatShips, x+xm-l4, y);
	}
      g.setClip(r);

    }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : paintRadar
 * DESCRIPTION	  :
 *****************************************************************************/
    void paintRadar(Graphics g)
    {

      g.drawImage(JKobo.Context.getRadar().getRadarBitmap(),
		  radar_x, radar_y, radar_width, radar_height,
		  null);

    }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : paintPlaying
 * DESCRIPTION	  :
 *****************************************************************************/
    void paintPlaying(Graphics g)
    {

      g.drawImage(window_bitmap, viewport_x, viewport_y, viewport_width,
		  viewport_height, null);

    }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : paintPaused
 * DESCRIPTION	  :
 *****************************************************************************/
    void paintPaused(Graphics g)
    {
      int x, y;
      Color c[];
      Font f, df;
      FontMetrics fm;
      Rectangle r;


      paintPlaying(g);
      r = g.getClipBounds();
      g.setClip(viewport_x, viewport_y, viewport_width, viewport_height);
      c = JKobo.Context.getSpace().getPixels();
      df = g.getFont();
      f = new Font(df.getName(), df.getStyle(), df.getSize() * 3);
      fm = g.getFontMetrics(f);

      g.setColor(c[0]);
      g.setFont(f);
      x = viewport_x + (viewport_width >>> 1);
      y = viewport_y + (viewport_height >>> 1);
      x -= (fm.stringWidth(PauseMessage) >>> 1);
      y -= (fm.getHeight() >>> 1);
      g.drawString(PauseMessage, x, y);
      g.setFont(df);
      g.setClip(r);

    }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getPreferredSize
 * DESCRIPTION	  :
 *****************************************************************************/
    public Dimension getPreferredSize()
    {
      int w, h;
      ScaleParms sp;

      sp = JKobo.Context.getScaleParms();
      w = (MARGIN * 3) + (sp.getWindowWidth() + sp.getRadarWidth());
      h = (MARGIN << 1) + sp.getWindowHeight();

      return(new Dimension(w, h));

    }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getMinimumSize
 * DESCRIPTION	  :
 *****************************************************************************/
    public Dimension getMinimumSize()
    {
      return(getPreferredSize());
    }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getMaximumSize
 * DESCRIPTION	  :
 *****************************************************************************/
    public Dimension getMaximumSize()
    {
      return(getPreferredSize());
    }

  }  /* END OF CLASS : PlayWinCanvas */

}  /* END OF CLASS : PlayWin */


/* END OF FILE : PlayWin.java */

