/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.Health.DurationType.NORMAL;
import static javoids.ImageMap.Name.PACMAN_DOT;
import static javoids.Shapes.Shape.POINT;

/* Bullet--------------------- */
/**
 * A fired bullet.
 * @author mallette
 */
public final class Bullet extends ShieldedSprite
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private final static long  serialVersionUID = 1L;
  /** The maximum number of these sprites allowed in the game (36 directions * 5 shots = 180 bullets) */
  protected final static int MAX_NUMBER       = 180;
  /** The maximum size for a non-specialized sprite */
  private static int         MAX_SIZE         = 100;
  /** The minimum size for a non-specialized sprite */
  private static int         MIN_SIZE         = 1;
  /** The default size for a non-specialized sprite */
  private static int         DEFAULT_SIZE     = 5;  // This should match the size of the image!
  /** is the bullet shielded */
  private boolean            shielded;
  /** the weapon that shot this bullet */
  private Item.Type          weapon;

  /**
   * Constructor
   * @param _parent this sprite's parent sprite
   * @param _health the health information to use
   * @param _move the movement information to use
   * @param _item the type of item this bullet was created from
   */
  public Bullet(BasicSprite _parent,Health _health,Move _move,Item _item)
  {
    super(_parent,_health,_move,new Shield(_parent,new Health(1,Shield.MAXIMUM_DAMAGE_CAPACITY / 4,Shield.DEFAULT_DURATION,Shield.DEFAULT_AGING_RATE * 2,NORMAL),new Move(_move),((Weapon)_item).getSize() + 1,POINT),((Weapon)_item).getSize(),((Weapon)_item).getBulletImage(),Shapes.getAreas(POINT,((Weapon)_item).getSize(),Bullet.MAX_SIZE),Shapes.getColors(POINT));
    this.getShield().setParent(this);
    this.weapon = ((Ship)_parent).getCurrentItem();
    this.setHoming(((Weapon)_item).getHomings() > 0);
    this.shielded = ((Weapon)_item).getShields() > 0;
    this.getShield().setDamage(this.shielded ? Shield.MAXIMUM_DAMAGE_CAPACITY / 8 : Shield.MAXIMUM_DAMAGE_CAPACITY);
    this.getShield().setDuration(this.shielded ? Shield.DEFAULT_DURATION / 2 : Shield.DEFAULT_DURATION);
    super.accelerate(_move.getMaxVelocity());
    this.setShape(POINT);
    this.setDisplayAreas(false);
  }

  /**
   * @return the default size
   */
  public int getDefaultSize()
  {
    return Bullet.DEFAULT_SIZE;
  }

  /**
   * @return the maximum size
   */
  public int getMaximumSize()
  {
    return Bullet.MAX_SIZE;
  }

  /**
   * @return the minimum size
   */
  public int getMinimumSize()
  {
    return Bullet.MIN_SIZE;
  }

  /**
   * @param _size the size the sprite is desired
   * @return the passed size, or the minimum, or maximum if it exceed either of those bounds.
   */
  public static int getValidSize(int _size)
  {
    return _size >= Bullet.MIN_SIZE ? _size <= Bullet.MAX_SIZE ? _size : Bullet.MAX_SIZE : Bullet.MIN_SIZE;
  }

  /**
   * @return the type of item this bullet is
   */
  public Item.Type getWeapon()
  {
    return this.weapon;
  }

  /**
   * @param _points the number of points to add to this sprite's point total
   * @return the new point total
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#modifyPoints(int)
   */
  @Override
  public int modifyPoints(int _points)
  {
    return this.getParent().modifyPoints(_points);
  }

  /**
   * create multiple bullets from a firing wepon
   * @param parent the parent sprite
   * @param directions the number of directions to shoot in
   * @param initialDirection the initial direction the parent sprite is facing
   * @param health the health infromation
   * @param move the movment information
   * @param item the item that created this(these) sprite(s)
   * @return a SpriteVector<Bullet> containing all of the fired bullets
   */
  public static SpriteVector<Bullet> spawn(BasicSprite parent,int directions,double initialDirection,Health health,Move move,Item item)
  {
    SpriteVector<Bullet> bullets = new SpriteVector<Bullet>(Math.min(Bullet.MAX_NUMBER,directions));

    synchronized (bullets)
    {
      switch (directions)
      {
        case 0 :
          break;
        case 1 :
          bullets.add(new Bullet(parent,new Health(health),new Move(move),item));
          break;
        case 2 :
          move.setDirection(initialDirection - 15.0);
          bullets.add(new Bullet(parent,new Health(health),new Move(move),item));
          move.setDirection(initialDirection + 15.0);
          bullets.add(new Bullet(parent,new Health(health),new Move(move),item));
          break;
        case 3 :
          move.setDirection(initialDirection);
          bullets.add(new Bullet(parent,new Health(health),new Move(move),item));
          move.setDirection(initialDirection - 30.0);
          bullets.add(new Bullet(parent,new Health(health),new Move(move),item));
          move.setDirection(initialDirection + 30.0);
          bullets.add(new Bullet(parent,new Health(health),new Move(move),item));
          break;
        case 5 :
          move.setDirection(initialDirection);
          bullets.add(new Bullet(parent,new Health(health),new Move(move),item));
          move.setDirection(initialDirection - 15.0);
          bullets.add(new Bullet(parent,new Health(health),new Move(move),item));
          move.setDirection(initialDirection + 15.0);
          bullets.add(new Bullet(parent,new Health(health),new Move(move),item));
          move.setDirection(initialDirection - 30.0);
          bullets.add(new Bullet(parent,new Health(health),new Move(move),item));
          move.setDirection(initialDirection + 30.0);
          bullets.add(new Bullet(parent,new Health(health),new Move(move),item));
          break;
        default :
          for (int n = 0;n < directions;n++)
          {
            move.setDirection(initialDirection + n * 360.0 / directions);
            bullets.add(new Bullet(parent,new Health(health),new Move(move),item));
          }
          break;
      }
    }
    return bullets;
  }

  /**
   * @return the image name for this sprite
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#getImageNumber()
   */
  @Override
  public ImageMap.Name getImageNumber()
  {
    return BasicSprite.getPacmanGame() ? PACMAN_DOT : this.getOriginalImage();
  }

  /**
   * @param damage the damage amount of damage this sprite sustained
   * @return the new damage level
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#modifyDamage(int)
   */
  @Override
  public int modifyDamage(int damage)
  {
    this.setPoints(this.getDamage());
    return super.modifyDamage(damage);
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("Bullet.ToString"),super.toString(),this.weapon,Boolean.valueOf(this.shielded)); //$NON-NLS-1$
  }
}
/* Bullet--------------------- */
