/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import java.io.Serializable;

/* Common--------------------- */
/**
 * A class of commonly used static functions that should not apply to any specific class getDistanceXY -calculate the distance between two cartesian coordinates getDistanceM -calculate the distance between two cartesian coordinates w/ Mobeus effect smallest of XY and XY2 getQuadrantXY -calculate the relative quadrant that another sprite is in getQuadrantM -calculate the relative quadrant that another sprite is in w/ Mobeus effect getAngleOrigin -calculate the angle between two cartesian coordinates effect relative to 0 (straight up) 0..359 clockwise getAngleXY -calculate the angle between two cartesian coordinates effect relative to 0 (straight up) 0..359 clockwise getAngleM -calculate the angle between two cartesian coordinates w/ Mobeus effect relative to 0 (straight up) 0..359 clockwise target -have one sprite turn to face another sprite (instantaneous turns)
 * @author mallette
 */
public class Common implements Serializable
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long serialVersionUID = 1L;
  /** constant for separating URLs */
  public static String      URL_SEPARATOR    = Messages.getString("/"); //$NON-NLS-1$
  /** constant for 2*PI used in calculations */
  public static double      PI2              = 2.0 * Math.PI;
  /** constant to quickly convert to radian values (for calling Java's math functions that use radian values) */
  public static double      toRadians        = Math.PI / 180.0d;

  /**
   * @param x the value to square
   * @return the square of x
   */
  public static double pow2(double x)
  {
    return x * x;
  }

  /**
   * @param x1 point1's x coordinate
   * @param y1 point1's y coordinate
   * @param x2 point2's x coordinate
   * @param y2 point2's y coordinate
   * @return the distance between two points
   */
  public static double getDistanceXY(double x1,double y1,double x2,double y2)
  {
    return Math.sqrt(Common.pow2(x2 - x1) + Common.pow2(y2 - y1));
  }

  /**
   * @param x1 point1's x coordinate
   * @param y1 point1's y coordinate
   * @param x2 point2's x coordinate
   * @param y2 point2's y coordinate
   * @param width the width of the screen
   * @param height the height of the screen
   * @return the shortest distance between two points (where Mobeus wrapping occurs around screen sides)
   */
  public static double getDistanceM(double x1,double y1,double x2,double y2,double width,double height)
  {
    return Math.sqrt(Common.pow2(Math.min(Math.min(Math.abs(x2 - x1),Math.abs(x2 + width - x1)),Math.abs(x1 + width - x2))) + Common.pow2(Math.min(Math.min(Math.abs(y2 - y1),Math.abs(y2 + height - y1)),Math.abs(y1 + height - y2))));
  }

  /**
   * @param x1 point1's x coordinate
   * @param y1 point1's y coordinate
   * @param x2 point2's x coordinate
   * @param y2 point2's y coordinate
   * @return the quadrant that the point2 is in relative to point1
   */
  public static int getQuadrantXY(double x1,double y1,double x2,double y2)
  {
    boolean dxgt = x2 - x1 >= 0;
    boolean dygt = y2 - y1 >= 0;
    int quadrant = 0;
    if (dxgt && dygt)
      quadrant = 1; // bottom right screen
    else if (!dxgt && dygt)
      quadrant = 2; // bottom left screen
    else if (!dxgt && !dygt)
      quadrant = 3; // top left screen
    else if (dxgt && !dygt)
      quadrant = 4; // top right screen
    return quadrant;
  }

  /**
   * @param x1 point1's x coordinate
   * @param y1 point1's y coordinate
   * @param x2 point2's x coordinate
   * @param y2 point2's y coordinate
   * @param width the width of the screen
   * @param height the height of the screen
   * @return the quadrant that the point2 is in relative to point1 (taking Mobeus distances around screen sides into account)
   */
  public static int getQuadrantM(double x1,double y1,double x2,double y2,double width,double height)
  {
    boolean xltm = Common.getDistanceXY(x1,0,x2,0) <= Common.getDistanceM(x1,0,x2,0,width,height);
    boolean yltm = Common.getDistanceXY(0,y1,0,y2) <= Common.getDistanceM(0,y1,0,y2,width,height);
    boolean xyltm = Common.getDistanceXY(x1,y1,x2,y2) <= Common.getDistanceM(x1,y1,x2,y2,width,height);
    int quadrant = 0;
    if (xyltm) // within 1/2 a screen, distanceXY is less than Mobeus distance for both directions! (good)
    {
      quadrant = Common.getQuadrantXY(x1,y1,x2,y2);
    }
    else if (xltm && !yltm)
    {
      if (x1 <= x2) // then it's 1 or 4
      {
        quadrant = y1 <= y2 ? 4 : 1;
      }
      else
      // then it's 2 or 3
      {
        quadrant = y1 <= y2 ? 3 : 2;
      }
    }
    else if (!xltm && yltm)
    {
      if (y1 <= y2) // then it's 1 or 2
      {
        quadrant = x1 <= x2 ? 2 : 1;
      }
      else
      // then it's 4 or 3
      {
        quadrant = x1 <= x2 ? 3 : 4;
      }
    }
    else if (!xltm && !yltm) // ok I think, quadrant 1 works here!
    {
      if (x1 <= x2)
      {
        quadrant = y1 <= y2 ? 3 : 2;
      }
      else
      {
        quadrant = y1 <= y2 ? 4 : 1;
      }
    }
    return quadrant;
  }

  /**
   * @param x1 point1's x coordinate
   * @param y1 point1's y coordinate
   * @param x2 point2's x coordinate
   * @param y2 point2's y coordinate
   * @return the angle between the rays formed by point1 and point2 relative to the horizontal
   */
  public static double getAngleOrigin(double x1,double y1,double x2,double y2)
  {
    return x2 - x1 != 0 && y2 - y1 != 0 ? (x2 >= x1 ? 90 : 270) - Math.toDegrees(Math.atan((y2 - y1) / (x2 - x1))) : y2 - y1 == 0 ? x2 >= x1 ? 90 : 270 : x2 - x1 == 0 ? y2 >= y1 ? 0 : 180 : 0;
  }

  /**
   * @param x1 point1's x coordinate
   * @param y1 point1's y coordinate
   * @param x2 point2's x coordinate
   * @param y2 point2's y coordinate
   * @return the angle between the rays formed by point1 and point2 relative to the horizontal
   */
  public static double getAngleXY(double x1,double y1,double x2,double y2)
  {
    return (x2 - x1 >= 0 ? 180 : 540) - Common.getAngleOrigin(x1,y1,x2,y2);
  }

  /**
   * @param x1 point1's x coordinate
   * @param y1 point1's y coordinate
   * @param x2 point2's x coordinate
   * @param y2 point2's y coordinate
   * @param width the width of the screen
   * @param height the height of the screen
   * @return the angle between the rays formed by point1 and point2 relative to the horizontal (taking Mobeus distances around screen sides into account)
   */
  public static double getAngleM(double x1,double y1,double x2,double y2,double width,double height)
  {
    double dx1 = Math.abs(x2 - x1);
    double dx2 = Math.abs(x2 + width - x1);
    double dx3 = Math.abs(x1 + width - x2);
    double dy1 = Math.abs(y2 - y1);
    double dy2 = Math.abs(y2 + height - y1);
    double dy3 = Math.abs(y1 + height - y2);
    double dx = Math.min(Math.min(dx1,dx2),dx3);
    double dy = Math.min(Math.min(dy1,dy2),dy3);
    boolean xlt = dx1 < dx2 && dx1 < dx3;
    boolean ylt = dy1 < dy2 && dy1 < dy3;
    int quadrant = Common.getQuadrantM(x1,y1,x2,y2,width,height);
    double angle = 0;
    if (dx != 0 && dy != 0)
    {
      angle = Math.toDegrees(Math.atan(dy / dx));
      switch (quadrant)
      {
        case 1 :
          angle = 90 + angle;
          break; // bottom right screen
        case 2 :
          angle = 270 - angle;
          break; // bottom left screen
        case 3 :
          angle = 270 + angle;
          break; // top left screen
        case 4 :
          angle = 90 - angle;
          break; // top right screen
        default :
          break;
      }
    }
    else
      angle = dx == 0 ? dy == 0 ? 999.999 : ylt ? 270 : 90 : dy == 0 ? xlt ? 0 : 180 : -999.999;
    return angle;
  }

  /**
   * @param move1 the first location information
   * @param move2 the second location information
   * @return the distance between the two locations (taking Mobeus distances around screen sides into account)
   */
  public static double getDistanceM(Move move1,Move move2)
  {
    return Common.getDistanceM(move1.getX(),move1.getY(),move2.getX(),move2.getY(),move1.getScreen().width,move1.getScreen().height);
  }

  /**
   * change the direction of move1 toward the location in move2 (to target that point)
   * @param move1 the first location information
   * @param move2 the second location information
   */
  public static void target(Move move1,Move move2)
  {
    double angle = 0;
    double absAngle;
    int direction;
    // determine the angle from ship to javoid then put into 0..359 range (absolute angle)
    // determine difference of two angles -360..360
    // determine which way to turn based on the sum of the two angles : range of -360..360
    // adjust angle to be in range -180..180 (360 degrees)
    // determine the amount to turn based on how far to left or right (this seems to give little wobble)
    // turn and accelerate in that direction
    angle = Common.getAngleM(move1.getX(),move1.getY(),move2.getX(),move2.getY(),move1.getScreen().width,move1.getScreen().height);
    angle = angle - move1.getDirection();
    direction = angle > 0 && angle <= 180 || angle < -180 && angle >= -360 ? Move.RIGHT : Move.LEFT;
    if (angle > 180)
      angle = angle - 360;
    else if (angle < -180)
      angle = angle + 360;
    absAngle = Math.abs(angle);
    move1.turn(direction,absAngle);
  }
}
/* Common--------------------- */
