/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.Health.DurationType.AGELESS;
import static javoids.Health.DurationType.IMMORTAL;
import static javoids.Health.DurationType.IMMORTAL_AGELESS;

import java.io.Serializable;

/**
 * Health.java is used to keep track of a sprite's duration and damage level.
 */
public class Health implements Serializable
{
  /** sprite's duration/immortality type */
  public enum DurationType
  {
    /** sprite doesn't expire over time */
    AGELESS,
    /** sprite cannot be destroyed */
    IMMORTAL,
    /** combination of AGELESS and IMMORTAL */
    IMMORTAL_AGELESS,
    /** normal everyday sprite */
    NORMAL;
  }

  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long serialVersionUID        = 1L;
  /** standard rate that sprites decay (take it off duration of sprite after every delay) */
  public static int         DEFAULT_AGING_RATE      = 10;
  /** standard damage capacity for a sprite */
  public static int         DEFAULT_DAMAGE_CAPACITY = 100;
  /** standard amount of "ticks" that a sprite can survive for (if not ageless and an aging rate of -1) */
  public static int         DEFAULT_DURATION        = 200;
  /** standard number of deaths for a sprite */
  public static int         DEFAULT_DEATHS          = 3;
  /** standard maximum number of deaths for any sprite (>= minimum number of deaths and <= 9) */
  public static int         MAX_DEATHS              = 9;
  /** standard minimum number of deaths a sprite can have (>= 1 always) */
  public static int         MIN_DEATHS              = 1;
  /** the amount that the sprite ages after each delay */
  private int               agingRate;
  /** the sprite's damage level */
  private int               damage;
  /** the sprite's duration time */
  private int               duration;
  /** the number of deaths the sprite has */
  private int               deaths;
  /** the sprite's maximum damage level */
  private int               maxDamage;
  /** the sprite's maximum duration */
  private int               maxDuration;
  /** the maxmimum number of deaths the sprite has */
  private int               maxDeaths;
  /** the state of the sprite's duration type */
  private DurationType      state;

  /** default constructor */
  public Health()
  {
    this(Health.DEFAULT_DEATHS,Health.DEFAULT_DAMAGE_CAPACITY,Health.DEFAULT_DURATION,Health.DEFAULT_AGING_RATE,IMMORTAL_AGELESS);
  }

  /**
   * copy constructor
   * @param health the Health object to copy
   */
  public Health(Health health)
  {
    this(health.maxDeaths,health.maxDamage,health.maxDuration,health.agingRate,health.state);
  }

  /**
   * constructor that accepts parameters for every value
   * @param _maxLives the maximum number of deaths
   * @param _maxDamage the maximum damage capactiy of the sprite
   * @param _maxDuration the maximum number of time units "ticks" the object will survive
   * @param _agingRate the rate that the sprite decays (the number of "ticks" used up after every delay)
   * @param _state the state of the object (normal, ageless, immortal, etc)
   */
  public Health(int _maxLives,int _maxDamage,int _maxDuration,int _agingRate,DurationType _state)
  {
    this.setMaxDeaths(_maxLives);
    this.setMaxDamage(_maxDamage);
    this.setMaxDuration(_maxDuration);
    this.setDeaths(0);
    this.setDamage(0);
    this.setDuration(0);
    this.setAgingRate(_agingRate);
    this.setState(_state);
  }

  /** Cause the object to age (reduces duration if it is not ageless) */
  public void age()
  {
    if (!this.isAgeless())
      this.modifyDuration(this.agingRate);
  }

  /**
   * return the aging rate of the sprite
   * @return this.agingRate
   */
  public int getAgingRate()
  {
    return this.agingRate;
  }

  /**
   * return the hit points of the sprite
   * @return the current damage level
   */
  public int getDamage()
  {
    return this.damage;
  }

  /**
   * return the duration left for the sprite
   * @return duration
   */
  public int getDuration()
  {
    return this.duration;
  }

  /**
   * return the number of deaths of the sprite
   * @return this.lives
   */
  public int getDeaths()
  {
    return this.deaths;
  }

  /**
   * return the maximum hit points of the sprite
   * @return the maximum damage level allowable before the object dies
   */
  public int getMaxDamage()
  {
    return this.maxDamage;
  }

  /**
   * return the maximum duration left for the sprite
   * @return this.maxDuration
   */
  public int getMaxDuration()
  {
    return this.maxDuration;
  }

  /**
   * return the maximum number of deaths of the sprite
   * @return this.maxLives
   */
  public int getMaxDeaths()
  {
    return this.maxDeaths;
  }

  /**
   * Is the object ageless (won't die from aging, but will from damage)?
   * @return true/false
   */
  public boolean isAgeless()
  {
    return AGELESS.equals( this.state ) || IMMORTAL_AGELESS.equals( this.state );
  }

  /**
   * Is the object alive?
   * @return true/false
   */
  public boolean isAlive()
  {
    return this.damage < this.maxDamage && this.duration < this.maxDuration && this.deaths < this.maxDeaths;
  }

  /**
   * Is the object immortal (won't die from damage or aging)?
   * @return true/false
   */
  public boolean isImmortal()
  {
    return IMMORTAL.equals( this.state ) || IMMORTAL_AGELESS.equals( this.state );
  }

  /**
   * @return whether or not the object can be restored (i.e. it has lives left)
   */
  public boolean isRestorable()
  {
    return this.deaths < this.maxDeaths;
  }

  /**
   * Modify the number of hitpoints for an object if it is not immortal.
   * @param _damage the number of hitpoints to give an object (positive values grants hitpoints, negative values take hitpoints away).
   * @return The current number of hitpoints an object has.
   */
  public int modifyDamage(int _damage)
  {
    if (!this.isImmortal())
    {
      this.damage += _damage;
      this.damage = this.damage > this.maxDamage ? this.maxDamage : this.damage < 0 ? 0 : this.damage;
    }
    return this.damage;
  }

  /**
   * Modify the amount of duration for an object if it is not ageless.
   * @param _duration the amount of duration to give an object (positive values grants duration, negative values take duration away).
   * @return The current amount of duration an object has.
   */
  public int modifyDuration(int _duration)
  {
    if (!this.isAgeless())
    {
      this.duration += _duration;
      this.duration = this.duration > this.maxDuration ? this.maxDuration : this.duration < 0 ? 0 : this.duration;
    }
    return this.duration;
  }

  /**
   * Modify the number of deaths for an object if it is not immortal.
   * @param _deaths the number of deaths to give an object (positive values grants deaths, negative values take deaths away).
   * @return The current number of deaths an object has.
   */
  public int modifyDeaths(int _deaths)
  {
    if (!this.isImmortal())
    {
      this.deaths += _deaths;
      this.deaths = this.deaths > this.maxDeaths ? this.maxDeaths : this.deaths < 0 ? 0 : this.deaths;
    }
    return this.deaths;
  }

  /**
   * Set the rate that the object decays (time it has left) before it is destroyed.
   * @param _agingRate The rate that the object decays (time it has left) before it is destroyed.
   */
  public void setAgingRate(int _agingRate)
  {
    this.agingRate = _agingRate;
  }

  /**
   * Set the number of hitpoints for an object (the amount of damage it can take before a lfie is taken ).
   * @param _damage The number of hitpointslives the object will have (0 <= __damage <= maxDamage)
   */
  public void setDamage(int _damage)
  {
    this.damage = _damage < 0 ? 0 : _damage > this.maxDamage ? this.maxDamage : _damage;
  }

  /**
   * Set the amount of time that an object will survive.
   * @param _duration The amount of time the object will survive the object will have (0 <= _duration <= maxDuration)
   */
  public void setDuration(int _duration)
  {
    this.duration = _duration < 0 ? 0 : _duration > this.maxDuration ? this.maxDuration : _duration;
  }

  /**
   * Set the number of deaths for an object (the number of times it may die).
   * @param _deaths The number of deaths the object will have (0 <= _lives <= MAX_DEATHS)
   */
  public void setDeaths(int _deaths)
  {
    this.deaths = _deaths < 0 ? 0 : _deaths > this.maxDeaths ? this.maxDeaths : _deaths;
  }

  /**
   * Set the maximum number of hitpoints an object may have (but doesn't necessarily have).
   * @param _maxDamage The maximum number of hitpoints the object will have (_maxDamage >= 1)
   */
  public void setMaxDamage(int _maxDamage)
  {
    this.maxDamage = _maxDamage < 1 ? 1 : _maxDamage;
  }

  /**
   * Set the maximum amount of tie the object will survive.
   * @param _maxDuration The maximum amount of time the object will survive (_maxDuration >= 1)
   */
  public void setMaxDuration(int _maxDuration)
  {
    this.maxDuration = _maxDuration > 0 ? _maxDuration : 1;
  }

  /**
   * Set the maximum number of deaths an object may have (but doesn't necessarily have).
   * @param _maxLives The maximum number of deaths the object will have (_maxLives >= MIN_DEATHS)
   */
  public void setMaxDeaths(int _maxLives)
  {
    this.maxDeaths = _maxLives < Health.MIN_DEATHS ? Health.MIN_DEATHS : _maxLives > Health.MAX_DEATHS ? Health.MAX_DEATHS : _maxLives;
  }

  /**
   * Set the state the object is in (NORMAL,AGELESS,IMMORTAL,IMMORTAL_AGELESS).
   * @param _state The state the object is in.
   */
  public void setState(DurationType _state)
  {
    this.state = _state;
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("Health.ToString"),Integer.valueOf(this.damage),Integer.valueOf(this.maxDamage),Integer.valueOf(this.duration),Integer.valueOf(this.maxDuration),Integer.valueOf(this.agingRate),Integer.valueOf(this.deaths),Integer.valueOf(this.maxDeaths),this.state); //$NON-NLS-1$
  }
}
/* Health---------------------- */
