import java.util.*;

/**
 * FileSystem is the Java Test File System (JTFS) class
 * which handles all the low lever I/O implementation and
 * communication with a block device.
 * @author Dimitris Michelinakis
 */
public class FileSystem {
    /** Maximum file name length. */
    public static final int maxFileNameLength=12;
    /** Maximum number of files. */
    public static final int maxNumberOfFiles=30;
    /** Maximum number of blocks assigned to a file. */
    public static final int maxNumberOfBlocksPerFile=4096;
    /** Maximum concurrent open files. */
    public static final int maxOpenFiles=4;
    /** Format the block device on file system initialization? */
    public static final boolean flagFormat=true;
    /** Is this a case sensitive file system? */
    public static final boolean flagCaseSensitive=false;
    /** Array to store concurrent open files. */
    private static OpenFile of[]=new OpenFile[maxOpenFiles];
    /** Block device initialized during file system creation. */
    private static BlockDevice bDev;
    /** Array to store the free blocks of the file system. */
    private static int freeBlocks[];

    /**
     * FileSystem constructor creates the actual file system if
     * createFS is true, otherwise it initializes an existing
     * file system.
     * @param b BlockDevice argument is the block device to use.
     * @param createFS boolean argument tells the constructor if the file system exists or not.
     * @author Dimitris Michelinakis
     */
    public FileSystem(BlockDevice b, boolean createFS) {
        /* Assign the block device */
        bDev=b;
        /* Initialize the free blocks array */
        freeBlocks=new int[bDev.getNumberOfBlocks()];
        /* Fill the free blocks array */
        Arrays.fill(freeBlocks,-1);
        /* Fill the open files array */
        Arrays.fill(of,null);
        /* Initialize our buffer */
        byte buffer[]=new byte[bDev.getBlockSize()];
        /* Create the file system or use an existing one */
        if(createFS==true) {
            /* Format the block device with 0 */
            if(flagFormat==true) {
                for(int i=0;i<bDev.getBlockSize();i++)
                    buffer[i]=0;
                for(int i=0;i<bDev.getNumberOfBlocks();i++)
                    bDev.writeBlock(buffer,i);
            }

            /* Initialize the file table */
            int currentBlock=0;
            String file=new String(""), spaces="";
            /* 'spaces' is used to represent the fixed
             length of the file name. */
            for(int i=1;i<=maxFileNameLength;i++)
                spaces=spaces.concat(" ");

            /* Create a buffer and write it 'per block' */
            for(int i=1;i<=maxNumberOfFiles;i++) {
                /* If one more entry fits in the block... */
                if(file.length()+5+new Integer(maxNumberOfFiles).toString().length()+spaces.length()+new Integer(bDev.getNumberOfBlocks()).toString().length()+new Integer(bDev.getNumberOfBlocks()-1).toString().length()<=bDev.getBlockSize()) {
                    /* add it to the buffer */
                    file=file.concat(":"+i+";0.00;"+spaces);
                } else {
                    /* If file entry doesn't fit, pad with spaces */
                    if(file.length()<bDev.getBlockSize()) {
                        int j2=file.length();
                        for(int j=1;j<=(bDev.getBlockSize()-j2);j++)
                            file=file.concat(" ");
                    }
                    /* Convert String buffer to byte buffer */
                    buffer=file.getBytes();
                    /* Write byte buffer to the current block */
                    bDev.writeBlock(buffer,currentBlock);
                    /* Remove current block from free blocks array */
                    freeBlocks[currentBlock]=0;
                    /* Move to next block and initialize String buffer */
                    currentBlock++;
                    file=":"+i+";0.00;"+spaces;
                }
            }
            /* Make sure the final block is padded with space */
            if(file.length()<bDev.getBlockSize()) {
                int j2=file.length();
                for(int j=1;j<=(bDev.getBlockSize()-j2);j++)
                    file=file.concat(" ");
            }
            /* Convert the remaining String buffer to byte buffer */
            buffer=file.getBytes();
            /* Write the remaining byte buffer to the final block */
            bDev.writeBlock(buffer,currentBlock);
            /* Remove current block from free blocks array */
            freeBlocks[currentBlock]=0;
        } else {
            /* Fill the freeBlocks[] array */
            String str="";
            int currentBlock=0, indexStart, indexEnd=0;
            /* Go through the file table and gather file information */
            for(int i=1;i<=maxNumberOfFiles;i++) {
                /* First mark the current block as used */
                freeBlocks[currentBlock]=0;
                /* Read the current block */
                bDev.readBlock(buffer,currentBlock);
                /* Convert the byte buffer to the String buffer */
                str=new String(buffer);
                indexStart=str.indexOf(":",indexEnd)+1;
                if(indexStart==0) {
                    currentBlock++; indexStart=0; indexEnd=0; i--;
                    continue;
                }
                /* Skip unneeded options */
                indexEnd=str.indexOf(";",indexStart+1);
                indexStart=indexEnd+1;
                indexEnd=str.indexOf(".",indexStart+1);
                /* Remember the info block */
                int infoBlock=Integer.parseInt(str.substring(indexStart,indexEnd));
                indexStart=indexEnd+1;
                indexEnd=str.indexOf(";",indexStart+1);
                /* Remember the attributes */
                int attr=Integer.parseInt(str.substring(indexStart,indexEnd));
                indexStart=indexEnd+1;
                indexEnd=indexStart+maxFileNameLength-1;
                /* If the slot isn't empty... */
                if(attr!=00) {
                    /* Mark the block that is the info block as used */
                    freeBlocks[infoBlock]=0;
                    /* Read info block from block device */
                    bDev.readBlock(buffer,infoBlock);
                    /* Convert the byte buffer to String */
                    str=new String(buffer);
                    /* Is info block empty? */
                    if(str.indexOf(" ")>0) {
                        /* No, so read the listed block(s) */
                        while(str.indexOf(" ")>0) {
                            freeBlocks[(Integer.parseInt(str.substring(0,str.indexOf(","))))]=0;
                            str=str.substring(str.indexOf(",")+1);
                        }
                    } /* If info block is empty, skip file */
                }
            }
        }
    }

    /**
     * open will open a file in varius ways depending on the mode. It can
     * open a file in Read, Write or Append mode.
     * @param filename String argument is the name of the file to open.
     * @param mode char argument tells in which mode to open the file.
     * @return an <code>int</code> of the file descriptor if the file was opened else return -1 for failure.
     * @author Dimitris Michelinakis
     */
    public int open(String filename, char mode) {
        /* Check that the file name doesn't exceed the length limit */
        if(filename.length()>maxFileNameLength) {
            return -1;
        } else if(filename.length()<maxFileNameLength) {
            /* Resize with spaces to a fixed length by adding spaces */
            while(filename.length()<maxFileNameLength)
                filename=filename.concat(" ");
        }

        /* So good so far, initialize our variables */
        byte buffer[]=new byte[bDev.getBlockSize()];
        String str, fileList[]=this.dir();
        boolean exists=false;

        /* If there are files in the file system,
         check to see if the requested file exists. */
        if(fileList!=null) {
            /* Check for case sensitivity */
            if(flagCaseSensitive==false) {
                for(int i=0;i<fileList.length;i++)
                    /* Case insensitive comparisson */
                    if(fileList[i]!=null)
                        if(fileList[i].equalsIgnoreCase(filename.trim())) {
                            /* File exists! */
                            exists=true;
                            break;
                        }
            } else {
                for(int i=0;i<fileList.length;i++)
                    /* Case sensitive comparisson */
                    if(of[i]!=null)
                        if(fileList[i].equals(filename.trim())) {
                            /* File exists */
                            exists=true;
                            break;
                        }
            }
        }

        /* Check for requested open mode */
        if(mode=='w'|mode=='a') {
            /* Is this file already open? */
            if(this.openFilesCheck(filename.trim()))
                return -1; /* Yes, but w&a modes operate on closed files */
            /* If file exists and requested mode is 'w', delete file */
            if(exists==true&mode=='w')
                this.delete(filename);
            /* Go through the file table */
            int currentBlock=0, indexStart, indexEnd=0;
            for(int i=1;i<=maxNumberOfFiles;i++) {
                /* Read current block from block device */
                bDev.readBlock(buffer,currentBlock);
                /* Convert the byte buffer to String */
                str=new String(buffer);
                /* Scan file entries */
                indexStart=str.indexOf(":",indexEnd)+1;
                if(indexStart==0) {
                    currentBlock++; indexStart=0; indexEnd=0; i--;
                    continue;
                }
                indexEnd=str.indexOf(";",indexStart+1);
                /* Remember the file descriptor */
                int fd=Integer.parseInt(str.substring(indexStart,indexEnd));
                indexStart=indexEnd+1;
                indexEnd=str.indexOf(".",indexStart+1);
                /* Remember the info block */
                int infoBlock=Integer.parseInt(str.substring(indexStart,indexEnd));
                indexStart=indexEnd+1;
                indexEnd=str.indexOf(";",indexStart+1);
                /* Remember the attributes */
                int attr=Integer.parseInt(str.substring(indexStart,indexEnd));
                /* If mode is 'w' and this is an empty slot, then store our
                 current file here. Do the same in mode 'a' if the file
                 doesn't exist. First store the attributes in the buffer */
                if((mode=='w'&attr==00)|(mode=='a'&exists==false))
                    str=str.substring(0,indexStart)+"01"+str.substring(indexEnd);
                indexStart=indexEnd+1;
                indexEnd=indexStart+maxFileNameLength-1;
                /* Remember the file name */
                String name=str.substring(indexStart,indexEnd);
                if((mode=='w'&attr==00)|(mode=='a'&exists==false)) {
                    /* If mode is 'w' then store the rest of the file
                     information in the buffer, do the same in mode 'a'
                     if the file doesn't exist */
                    str=str.substring(0,indexStart)+filename+str.substring(indexEnd);
                    /* Convert our String buffer to byte array */
                    buffer=str.getBytes();
                    /* Write the buffer to the current block... */
                    bDev.writeBlock(buffer,currentBlock);
                    /* ... and open the file */
                    if(this.openFilesOpen(mode,infoBlock,fd,attr,filename.trim()))
                        return fd;
                    return -1;
                } else if(mode=='a'&exists==true) {
                    /* If mode is 'a' and the file exists, then
                     find the file and open it */
                    if(flagCaseSensitive==false) {
                        /* Case insensitive search */
                        if(filename.equalsIgnoreCase(name)) {
                            if(this.openFilesOpen(mode,infoBlock,fd,attr,filename.trim()))
                                return fd;
                            return -1;
                        }
                    } else {
                        /* Case sensitive search */
                        if(filename.equals(name)) {
                            if(this.openFilesOpen(mode,infoBlock,fd,attr,filename.trim()))
                                return fd;
                            return -1;
                        }
                    }
                }
            }
            return -1;
        } else if(mode=='r') {
            /* If file doesn't already exist, exit */
            if(exists==false)
                return -1;
            /* Is this file already open? */
            if(this.openFilesCheck(filename.trim())) {
                /* Yes, so find fd of opened file */
                for(int i=0;i<of.length;i++) {
                    if(of[i]!=null)
                        if(flagCaseSensitive==false) {
                            /* Case insensitive search */
                            if(of[i].filename.trim().equalsIgnoreCase(filename.trim()))
                                return of[i].fd; /* Return fd of opened file */
                        } else {
                            /* Case sensitive search */
                            if(of[i].filename.trim().equals(filename.trim()))
                                return of[i].fd; /* Return fd of opened file */
                        }
                }
            }

            /* File not already open, so open it.
             Go through the file table first to find it */
            int currentBlock=0, indexStart, indexEnd=0;
            for(int i=1;i<=maxNumberOfFiles;i++) {
                /* Read current block from block device */
                bDev.readBlock(buffer,currentBlock);
                /* Convert the byte buffer to String */
                str=new String(buffer);
                /* Scan file entries */
                indexStart=str.indexOf(":",indexEnd)+1;
                if(indexStart==0) {
                    currentBlock++; indexStart=0; indexEnd=0; i--;
                    continue;
                }
                indexEnd=str.indexOf(";",indexStart+1);
                /* Remember the file descriptor */
                int fd=Integer.parseInt(str.substring(indexStart,indexEnd));
                indexStart=indexEnd+1;
                indexEnd=str.indexOf(".",indexStart+1);
                /* Remember the info block */
                int infoBlock=Integer.parseInt(str.substring(indexStart,indexEnd));
                indexStart=indexEnd+1;
                indexEnd=str.indexOf(";",indexStart+1);
                /* Remember the attributes */
                int attr=Integer.parseInt(str.substring(indexStart,indexEnd));
                indexStart=indexEnd+1;
                indexEnd=indexStart+maxFileNameLength-1;
                /* Remember the file name */
                String name=str.substring(indexStart,indexEnd).trim();
                /* Find file based on case sensitivity */
                if(flagCaseSensitive==false) {
                    /* Case insensitive search */
                    if(filename.trim().equalsIgnoreCase(name)) {
                        if(this.openFilesOpen(mode,infoBlock,fd,attr,filename.trim()))
                            return fd;
                        return -1;
                    }
                } else {
                    /* Case sensitive search */
                    if(filename.trim().equals(name)) {
                        if(this.openFilesOpen(mode,infoBlock,fd,attr,filename.trim()))
                            return fd;
                        return -1;
                    }
                }
            }
        }
        /* Invalid mode, max files reached, file not found, error */
        return -1;
    }

    /**
     * close will close an opened file.
     * @param fd int argument is the file descriptor of a file.
     * @author Dimitris Michelinakis
     */
    public void close(int fd) {
        /* Scan through the open files */
        for(int i=0;i<of.length;i++) {
            /* Find matching file descriptor */
            if(of[i]!=null)
                if(of[i].fd==fd)
                    of[i]=null; /* Remove from open files array */
        }
    }

    /**
     * read will read a number of bytes from an opened file (in Read mode only).
     * @param fd int argument is the file descriptor of the file to read from.
     * @param buffer byte[] argument is an array of bytes were we store the read bytes.
     * @param count int argument is the number of bytes to read.
     * @return an <code>int</code> of the number of bytes read.
     * @author Dimitris Michelinakis
     */
    public int read (int fd, byte[] buffer, int count) {
        /* Define byte buffer array, String buffer array etc */
        byte buffer2[]=new byte[bDev.getBlockSize()];
        int filePos=0;
        String str;
        boolean flag=false;
        int currentBlock=0;
        Vector usableBlocks=new Vector(maxNumberOfBlocksPerFile/2,maxNumberOfBlocksPerFile/2);

        /* Check that the requested filename is open and readable */
        for(filePos=0;filePos<of.length;filePos++) {
            if(of[filePos]!=null)
                /* If filename is there and readable then flag and break */
                if((of[filePos].fd==fd)&of[filePos].mode=='r') {
                    flag=true;
                    break;
                }
        }
        /* If file is not found or not readable return with error */
        if(flag==false)
            return -1;

        /* Get blocks used by this file */
        bDev.readBlock(buffer2,of[filePos].infoBlock);
        /* Convert the byte buffer to String */
        str=new String(buffer2);
        /* Add blocks into our Vector */
        while(str.indexOf(" ")>0) {
            usableBlocks.addElement(new Integer((Integer.parseInt(str.substring(0,str.indexOf(","))))));
            str=str.substring(str.indexOf(",")+1);
        }
        /* Find our current possition in those blocks */
        Enumeration enum=usableBlocks.elements();
        if(of[filePos].readPos[0]>1) {
            while(enum.hasMoreElements()) {
                /* Fill Vector with used blocks */
                currentBlock=Integer.parseInt(enum.nextElement().toString());
                if(of[filePos].readPos[0]==currentBlock)
                    break;
            }
        } else { /* Our possition is the start of the file */
            currentBlock=Integer.parseInt(usableBlocks.firstElement().toString());
            of[filePos].readPos[0]=currentBlock;
        }

        /* Get read possition and block */
        int length=0;
        String str2="";

        while(length<count) {
            /* Read current block from block device */
            bDev.readBlock(buffer2,currentBlock);
            /* Convert the byte buffer to String */
            str=new String(buffer2);
            /* Read the block 'as far as possible' */
            if(currentBlock==Integer.parseInt(usableBlocks.lastElement().toString())) {
                /* If this is the last block, check for 'end of block' marker */
                if(count+of[filePos].readPos[1]<str.lastIndexOf("-",str.length())) {
                    str=str.substring(of[filePos].readPos[1],count+of[filePos].readPos[1]-length);
                    /* Move the read possition marker */
                    of[filePos].readPos[1]+=count-length;
                    length+=str.length();
                } else {
                    str=str.substring(of[filePos].readPos[1],str.lastIndexOf("-",str.length()));
                    length+=str.length();
                    /* Move the read possition marker */
                    of[filePos].readPos[1]+=length+1;
                    /* Add remaining bytes */
                    str2=str2.concat(str);
                    /* EOF reached */
                    break;
                }
            } else {
                /* If current block is not the last block of the file */
                if(count-length+of[filePos].readPos[1]<=bDev.getBlockSize()) {
                    /* Count is small enough to read from the block */
                    str=str.substring(of[filePos].readPos[1],count+of[filePos].readPos[1]-length);
                    /* Move the read possition marker */
                    of[filePos].readPos[1]+=count-length;
                    length+=str.length();
                } else {
                    /* Count is big and we need to move to the next block */
                    str=str.substring(of[filePos].readPos[1]);
                    /* Move the read possition marker and block */
                    if(enum.hasMoreElements())
                        /* Move enum pointer */
                        of[filePos].readPos[0]=Integer.parseInt(enum.nextElement().toString());
                    /* Store the new possition for the next block */
                    currentBlock=of[filePos].readPos[0];
                    of[filePos].readPos[1]=0;
                    length+=str.length();
                }
            }
            /* Add to general buffer */
            str2=str2.concat(str);
        }
        /* Convert buffer and store in caller's buffer */
        buffer2=str2.getBytes();
        java.lang.System.arraycopy(buffer2,0,buffer,0,buffer.length);
        return length;
    }

    /**
     * write will write to a file which has been opened in Write mode.
     * @param fd int argument is the file descriptor of the file to write to.
     * @param buffer byte[] argument is an array of bytes to write to the file.
     * @param count int argument is the number of bytes to write from the buffer.
     * @return an <code>int</code> of the number of bytes written to the file.
     * @author Dimitris Michelinakis
     */
    public int write(int fd, byte[] buffer, int count) {
        /* Define byte buffer array, String buffer array etc */
        byte buffer2[]=new byte[bDev.getBlockSize()];
        int filePos=0;
        String str;
        boolean flag=false;
        int currentBlock=0, indexStart, indexEnd=0;
        Vector usableBlocks=new Vector(maxNumberOfBlocksPerFile/2,maxNumberOfBlocksPerFile/2);

        /* Check that the requested filename is open and writable */
        for(filePos=0;filePos<of.length;filePos++) {
            if(of[filePos]!=null)
                /* If filename is there and writable then flag and break */
                if((of[filePos].fd==fd)&(of[filePos].mode=='w'|of[filePos].mode=='a')) {
                    flag=true;
                    break;
                }
        }

        /* If file is open and the size of the bytes to write
         are not bigger than the buffer, proceed to write to file */
        if(flag==true&count<=buffer.length) {
            /* Check to see if this file has any assigned blocks */
            if(of[filePos].infoBlock==0) {
                /* Since it doesnt have any, assign a block table */
                for(int i=freeBlocks.length-1;i>0;i--) {
                    /* If block is free then assign it to current file */
                    if(freeBlocks[i]==-1) {
                        of[filePos].infoBlock=i;
                        /* Remove current block from free blocks array */
                        freeBlocks[of[filePos].infoBlock]=0;
                        /* Go through the file table and change file entry
                         to reflect to the new info block */
                        for(i=1;i<=maxNumberOfFiles;i++) {
                            /* Read current block from block device */
                            bDev.readBlock(buffer2,currentBlock);
                            /* Convert the byte buffer2 to String */
                            str=new String(buffer2);
                            /* Scan file entries */
                            indexStart=str.indexOf(":",indexEnd)+1;
                            if(indexStart==0) {
                                currentBlock++; indexStart=0; indexEnd=0; i--;
                                continue;
                            }
                            indexEnd=str.indexOf(";",indexStart+1);
                            /* Remember the file descriptor */
                            int fd2=Integer.parseInt(str.substring(indexStart,indexEnd));
                            indexStart=indexEnd+1;
                            indexEnd=str.indexOf(".",indexStart+1);
                            /* Store the new infoBlock on the String buffer2 */
                            if(fd==fd2) {
                                int strLength=new Integer(of[filePos].infoBlock).toString().length()-1;
                                str=str.substring(0,indexStart)+of[filePos].infoBlock+str.substring(indexEnd,str.length()-strLength);
                                /* Convert the String buffer2 to the byte array */
                                buffer2=str.getBytes();
                                /* Write the buffer2 to the current block */
                                bDev.writeBlock(buffer2,currentBlock);
                                /* Scan complete so have a break */
                                break;
                            }
                        }
                        /* info block assigned, so have a break */
                        break;
                    }
                }

                /* If no info block was assigned... */
                if(of[filePos].infoBlock==0)
                    return -1; /* Return with error */
                else {
                    /* Otherwise initialize the info block */
                    str="";
                    /* Pad String buffer with spaces */
                    for(int i=1;i<=(bDev.getBlockSize());i++)
                        str=str.concat(" ");
                    /* Convert the String buffer2 to the byte array */
                    buffer2=str.getBytes();
                    /* Write the buffer2 to the current block */
                    bDev.writeBlock(buffer2,of[filePos].infoBlock);
                }
            }

            /* Read info block from block device */
            bDev.readBlock(buffer2,of[filePos].infoBlock);
            /* Convert the byte buffer2 to String */
            str=new String(buffer2);
            /* Is info block empty? */
            if(str.indexOf(" ")>0) {
                /* No, so read the listed block(s) */
                while(str.indexOf(" ")>0) {
                    usableBlocks.addElement(new Integer((Integer.parseInt(str.substring(0,str.indexOf(","))))));
                    str=str.substring(str.indexOf(",")+1);
                }
            } else {
                /* Yes, so find an empty block to write to */
                for(int i=0;i<freeBlocks.length;i++) {
                    /* If block is free then assign it to current file */
                    if(freeBlocks[i]==-1) {
                        usableBlocks.addElement(new Integer(i));
                        /* Remove current block from free blocks array */
                        freeBlocks[i]=0;
                        /* Add block number into info block */
                        str=i+",";
                        /* If too small, pad with spaces */
                        if(str.length()<bDev.getBlockSize()) {
                            int j2=str.length();
                            for(int j=1;j<=(bDev.getBlockSize()-j2);j++)
                                str=str.concat(" ");
                        }
                        /* Convert the String buffer2 to the byte array */
                        buffer2=str.getBytes();
                        /* Write the buffer2 to the current block */
                        bDev.writeBlock(buffer2,of[filePos].infoBlock);
                        /* Finally initialize the new block with
                         our 'null' character */
                        str="-";
                        /* Pad with spaces */
                        if(str.length()<bDev.getBlockSize()) {
                            int j2=str.length();
                            for(int j=1;j<=(bDev.getBlockSize()-j2);j++)
                                str=str.concat(" ");
                        }
                        /* Convert the String buffer2 to the byte array */
                        buffer2=str.getBytes();
                        /* Write the buffer2 to the new block */
                        bDev.writeBlock(buffer2,i);
                        break;
                    }
                }
                /* If no free block was found, return error */
                if(usableBlocks.size()==0)
                    return -1;
            }

            /* Prepare to write data to blocks */
            int size=0;
            currentBlock=Integer.parseInt(usableBlocks.lastElement().toString());
            /* Get the remaining space on the currentBlock */
            bDev.readBlock(buffer2,currentBlock);
            String str2=new String(buffer2);
            /* Find where the append possition */
            int endMarker=str2.lastIndexOf("-",str2.length());
            int totalSize=bDev.getBlockSize();
            /* Find the remaining space (if any) */
            if(endMarker>0) {
                totalSize=totalSize-endMarker;
                /* Remember the data already written in the block */
                str2=str2.substring(0,endMarker);
            } else
                str2="";
            /* Is there enough space in one block? */
            if(count>totalSize) {
                /* Not enough space, so take as much as its possible */
                str=new String(buffer);
                str=str.substring(0,totalSize);
                /* Remember the total size to write */
                size=str.length();
                /* Add existing data */
                str=str2.concat(str);
                /* Convert the String buffer to the byte array */
                buffer2=str.getBytes();
                /* Write the buffer2 to the current block */
                bDev.writeBlock(buffer2,currentBlock);
                /* Because data doesn't fit in one block we assign
                 new block to the file */
                for(int i=0;i<freeBlocks.length;i++) {
                    /* If block is free then assign it to current file */
                    if(freeBlocks[i]==-1) {
                        /* Read existing blocks and append new block */
                        bDev.readBlock(buffer2,of[filePos].infoBlock);
                        str=new String(buffer2);
                        /* Make sure there is enough space in the info block */
                        if(str.trim().length()+new Integer(i).toString().length()>bDev.getBlockSize())
                            return -1;
                        /* Add new block in our usable blocks */
                        usableBlocks.addElement(new Integer(i));
                        /* Remove current block from free blocks array */
                        freeBlocks[i]=0;
                        str=str.substring(0,str.lastIndexOf(",")+1)+i+",";
                        /* If too small, pad with spaces */
                        if(str.length()<bDev.getBlockSize()) {
                            int j2=str.length();
                            for(int j=1;j<=(bDev.getBlockSize()-j2);j++)
                                str=str.concat(" ");
                        }
                        /* Convert the String buffer2 to the byte array */
                        buffer2=str.getBytes();
                        /* Write the buffer2 to the current block */
                        bDev.writeBlock(buffer2,of[filePos].infoBlock);
                        /* Finally initialize the new block with
                         our 'null' character */
                        str="-";
                        /* Pad with spaces */
                        if(str.length()<bDev.getBlockSize()) {
                            int j2=str.length();
                            for(int j=1;j<=(bDev.getBlockSize()-j2);j++)
                                str=str.concat(" ");
                        }
                        /* Convert the String buffer2 to the byte array */
                        buffer2=str.getBytes();
                        /* Write the buffer2 to the new block */
                        bDev.writeBlock(buffer2,i);
                        break;
                    }
                }
                /* If no free block was found, return 0 bytes written */
                if(usableBlocks.size()==0)
                    return 0;
                /* Find un-written data */
                buffer=(new String(buffer)).substring(totalSize).getBytes();
                /* And recurse the un-written data */
                size+=this.write(fd,buffer,count-size);
                return size;
            } else if(count<totalSize) {
                /* There is enough space, so convert
                 byte buffer to String buffer */
                str=new String(buffer);
                /* Get the requested size only */
                str=str.substring(0,count);
                /* Remember the total size to write */
                size=str.length();
                /* Append the existing data and our 'null' character */
                str=str2.concat(str.concat("-"));
                /* Pad with spaces if the buffer is not the correct size */
                if(str.length()<bDev.getBlockSize()) {
                    int j2=str.length();
                    for(int j=1;j<=(bDev.getBlockSize()-j2);j++)
                        str=str.concat(" ");
                }
                /* Convert the String buffer to the byte array */
                buffer2=str.getBytes();
                /* Write the buffer2 to the current block */
                bDev.writeBlock(buffer2,currentBlock);
            } else {
                /* count is the same as our block size */
                str=new String(buffer);
                size=str.length();
                /* Convert the String buffer to the byte array */
                buffer2=str.getBytes();
                /* Write the buffer2 to the current block */
                bDev.writeBlock(buffer2,currentBlock);
            }
            return size;
        }
        /* Failed to write, no free blocks */
        return -1;
    }

    /**
     * seek will change the read possition of an opened file in Read mode.
     * @param fd int argument is the file descriptor of the file to seek.
     * @param position int argument is the number of bytes to change the read possition, which can be negative.
     * @param relative boolean argument is true to seek based on the current possition, or false to seek based on the start of the file.
     * @return an <code>int</code> of the number of bytes written to the file.
     * @author Dimitris Michelinakis
     */
    public boolean seek (int fd, int position, boolean relative) {
        byte buffer[]=new byte[bDev.getBlockSize()];
        Vector usableBlocks=new Vector(maxNumberOfBlocksPerFile/2,maxNumberOfBlocksPerFile/2);
        int currentBlock=0, filePos;
        boolean flag=false;
        String str;
        /* Check that the requested filename is open and readable */
        for(filePos=0;filePos<of.length;filePos++) {
            if(of[filePos]!=null)
                /* If filename is there and readable then flag and break */
                if((of[filePos].fd==fd)&of[filePos].mode=='r') {
                    flag=true;
                    break;
                }
        }
        /* If file is not found or not readable return with error */
        if(flag==true) {
            /* Just in case we are not going anywhere... */
            if(position==0&relative==true)
                return true;

            /* Get blocks used by this file */
            bDev.readBlock(buffer,of[filePos].infoBlock);
            /* Convert the byte buffer to String */
            str=new String(buffer);
            /* If info block is empty, error */
            if(str.indexOf(" ")<=0)
                return false;
            /* Add blocks into our Vector */
            while(str.indexOf(" ")>0) {
                usableBlocks.addElement(new Integer((Integer.parseInt(str.substring(0,str.indexOf(","))))));
                str=str.substring(str.indexOf(",")+1);
            }

            /* Find our current possition in those blocks */
            Enumeration enum=usableBlocks.elements();
            if(of[filePos].readPos[0]>1&relative==true) {
                while(enum.hasMoreElements()) {
                    /* Fill Vector with used blocks */
                    currentBlock=Integer.parseInt(enum.nextElement().toString());
                    if(of[filePos].readPos[0]==currentBlock)
                        break;
                }
            } else { /* Our possition is the start of the file */
                if(relative==false&position<0)
                    /* Can't move out of bounds */
                    return false;
                currentBlock=Integer.parseInt(usableBlocks.firstElement().toString());
                of[filePos].readPos[0]=currentBlock;
                /* SPECIAL CASE, reset seek at start of file */
                if(relative==false&position==0) {
                    of[filePos].readPos[1]=0;
                    return true;
                }
            }

            /* File found and is readable, seek */
            int length=0;
            byte buffer2[]=new byte[bDev.getBlockSize()];

            /* Which way do we 'seek'? */
            if(position>0) {
                while(length<position) {
                    /* Read current block from block device */
                    bDev.readBlock(buffer2,currentBlock);
                    /* Convert the byte buffer to String */
                    str=new String(buffer2);
                    /* Read the block 'as far as possible' */
                    if(currentBlock==Integer.parseInt(usableBlocks.lastElement().toString())) {
                        /* If this is the last block, check for 'end of block' marker */
                        if(position+of[filePos].readPos[1]<str.lastIndexOf("-",str.length())) {
                            /* Position is within our bounds so
                             move the read possition marker */
                            of[filePos].readPos[1]+=position-length;
                            length+=position+of[filePos].readPos[1]-length-of[filePos].readPos[1];
                        } else {
                            /* Position is out of bounds */
                            length+=str.lastIndexOf("-",str.length())-of[filePos].readPos[1];
                            /* Move the read possition marker */
                            of[filePos].readPos[1]+=length+1;
                            /* EOF reached */
                            break;
                        }
                    } else {
                        /* If current block is not the last block of the file */
                        if(position-length+of[filePos].readPos[1]<=bDev.getBlockSize()) {
                            /* Position is small enough so
                             move the read possition marker */
                            of[filePos].readPos[1]+=position-length;
                            length+=position+of[filePos].readPos[1]-length-of[filePos].readPos[1];
                        } else {
                            /* Position is big and we need to move to the next block */
                            if(enum.hasMoreElements())
                                /* Move enum pointer */
                                of[filePos].readPos[0]=Integer.parseInt(enum.nextElement().toString());
                            /* Store the new possition for the next block */
                            currentBlock=of[filePos].readPos[0];
                            of[filePos].readPos[1]=0;
                            length+=bDev.getBlockSize()-of[filePos].readPos[1];
                        }
                    }
                }
            } else {
                /* We 'seek' backwards */
                while(length<(position*-1)) {
                    /* Read current block from block device */
                    bDev.readBlock(buffer2,currentBlock);
                    /* Convert the byte buffer to String */
                    str=new String(buffer2);
                    /* Read the block 'as far as possible' */
                    if(currentBlock==Integer.parseInt(usableBlocks.firstElement().toString())) {
                        /* We are 'seeking' in the first block */
                        if(position+of[filePos].readPos[1]>=0) {
                            /* Position is within our bounds so
                             move the read possition marker */
                            of[filePos].readPos[1]+=position+length;
                            length+=position+of[filePos].readPos[1]+length;
                        } else {
                            /* Position is out of bounds */
                            length+=of[filePos].readPos[1];
                            /* Move the read possition marker
                             to the start of the block*/
                            of[filePos].readPos[1]=0;
                            /* EOF reached */
                            break;
                        }
                    } else {
                        /* If current block is not the last block of the file */
                        if(position-length+of[filePos].readPos[1]<=bDev.getBlockSize()) {
                            /* Position is small enough so
                             move the read possition marker */
                            of[filePos].readPos[1]+=position-length;
                            length+=position+of[filePos].readPos[1]-length-of[filePos].readPos[1];
                        } else {
                            /* Count is big and we need to move to the next block */
                            if(enum.hasMoreElements())
                                /* Move enum pointer */
                                of[filePos].readPos[0]=Integer.parseInt(enum.nextElement().toString());
                            /* Store the new possition for the next block */
                            currentBlock=of[filePos].readPos[0];
                            of[filePos].readPos[1]=0;
                            length+=bDev.getBlockSize()-of[filePos].readPos[1];
                        }
                    }
                }
            }
        }
        return true;
    }

    /**
     * length will find the length of a file.
     * @param fd int argument is the file descriptor of the file to find its length.
     * @return an <code>int</code> of the length of the file in bytes.
     * @author Dimitris Michelinakis
     */
    public int length (int fd) {
        byte buffer[]=new byte[bDev.getBlockSize()];
        int filePos, infoBlock=0, size=0, currentBlock=0;
        boolean flag=false;
        String str;
        /* If file is already open then read information from
         there, which is faster than reading the file entry */
        for(filePos=0;filePos<of.length;filePos++) {
            if(of[filePos]!=null)
                /* If filename is open, flag and break */
                if((of[filePos].fd==fd)&of[filePos].mode=='r') {
                    flag=true;
                    break;
                }
        }
        /* Is the file open? */
        if(flag==true)
            /* Yes so read info block faster */
            infoBlock=of[filePos].infoBlock;
        else {
            /* Reset flag to use it again */
            flag=false;
            /* No so read file entry in file table */
            int indexStart, indexEnd=0;
            /* Go through the file table and gather file information */
            for(int i=1;i<=maxNumberOfFiles;i++) {
                /* Read the current block */
                bDev.readBlock(buffer,currentBlock);
                /* Convert the byte buffer to the String buffer */
                str=new String(buffer);
                indexStart=str.indexOf(":",indexEnd)+1;
                if(indexStart==0) {
                    currentBlock++; indexStart=0; indexEnd=0; i--;
                    continue;
                }
                indexEnd=str.indexOf(";",indexStart+1);
                /* Remember the file descriptor */
                int fd2=Integer.parseInt(str.substring(indexStart,indexEnd));
                indexStart=indexEnd+1;
                indexEnd=str.indexOf(".",indexStart+1);
                /* Remember the info block */
                infoBlock=Integer.parseInt(str.substring(indexStart,indexEnd));
                indexStart=indexEnd+1;
                indexEnd=str.indexOf(";",indexStart+1);
                /* Remember the attributes */
                int attr=Integer.parseInt(str.substring(indexStart,indexEnd));
                indexStart=indexEnd+1;
                indexEnd=indexStart+maxFileNameLength-1;
                /* If the slot isn't empty... */
                if(fd2==fd&attr!=00) {
                    /* The file exists so break */
                    flag=true;
                    break;
                }
            }
            /* If file wasn't found at all, return error */
            if(flag==false)
                return -1;
        }

        /* Now that we know the info block, read
         the file blocks from the info block */
        bDev.readBlock(buffer,infoBlock);
        /* Convert the byte buffer to String */
        str=new String(buffer);
        /* Is info block empty? */
        if(str.indexOf(" ")>0) {
            /* No, so read the listed block(s) */
            while(str.indexOf(" ")>0) {
                currentBlock=Integer.parseInt(str.substring(0,str.indexOf(",")));
                str=str.substring(str.indexOf(",")+1);
                /* Calculate the size of the block */
                if(str.indexOf(" ")>0)
                    size+=bDev.getBlockSize();
                else {
                    /* Read the block */
                    bDev.readBlock(buffer,currentBlock);
                    /* Convert the byte buffer to String */
                    str=new String(buffer);
                    /* Find the ending marker */
                    return size+str.lastIndexOf("-",str.length());
                }
            }
        } else /* If info block is empty, file is empty */
            return 0;
        return -1; /* Make compiler happy :) */
    }

    /**
     * delete will delete a file from the file system.
     * @param filename String argument is the name of the file to delete.
     * @author Dimitris Michelinakis
     */
    public void delete (String filename) {
        /* Check that the file name doesn't exceed the length limit */
        if(filename.length()>maxFileNameLength)
            return;

        /* So good so far, initialize our variables */
        byte buffer[]=new byte[bDev.getBlockSize()];
        String str, fileList[]=this.dir();
        boolean exists=false;

        /* If there are files in the file system,
         check to see if the requested file exists. */
        if(fileList!=null) {
            /* Check for case sensitivity */
            if(flagCaseSensitive==false) {
                for(int i=0;i<fileList.length;i++)
                    /* Case insensitive comparisson */
                    if(fileList[i]!=null)
                        if(fileList[i].equalsIgnoreCase(filename)) {
                            /* File exists! */
                            exists=true;
                            break;
                        }
            } else {
                for(int i=0;i<fileList.length;i++)
                    /* Case sensitive comparisson */
                    if(of[i]!=null)
                        if(fileList[i].equals(filename)) {
                            /* File exists */
                            exists=true;
                            break;
                        }
            }
        }

        /* Is file open? */
        if(this.openFilesCheck(filename))
            return; /* Yes, but 'delete' operates on closed files */

        int currentBlock=0, indexStart, indexEnd=0;
        /* Go through the file table and gather file information */
        for(int i=1;i<=maxNumberOfFiles;i++) {
            /* Read the current block */
            bDev.readBlock(buffer,currentBlock);
            /* Convert the byte buffer to the String buffer */
            str=new String(buffer);
            indexStart=str.indexOf(":",indexEnd)+1;
            if(indexStart==0) {
                currentBlock++; indexStart=0; indexEnd=0; i--;
                continue;
            }
            /* Skip unneeded options */
            indexEnd=str.indexOf(";",indexStart+1);
            indexStart=indexEnd+1;
            indexEnd=str.indexOf(".",indexStart+1);
            /* Remember the info block */
            int infoBlock=Integer.parseInt(str.substring(indexStart,indexEnd));
            indexStart=indexEnd+1;
            indexEnd=str.indexOf(";",indexStart+1);
            /* Remember the attributes */
            int attr=Integer.parseInt(str.substring(indexStart,indexEnd));
            /* Remember the possition of the attributes */
            int attrS=indexStart;
            int attrE=indexEnd;
            indexStart=indexEnd+1;
            indexEnd=indexStart+maxFileNameLength-1;
            /* Remember the file name */
            String name=str.substring(indexStart,indexEnd).trim();
            /* If the slot isn't empty... */
            if(attr!=00) {
                /* Find file based on case sensitivity */
                boolean found=false;
                if(flagCaseSensitive==false) {
                    /* Case insensitive search */
                    if(filename.equalsIgnoreCase(name))
                        found=true;
                } else {
                    /* Case sensitive search */
                    if(filename.equals(name))
                        found=true;
                }
                /* If this is the file we are looking for, delete it */
                if(found==true) {
                    /* Mark file blocks in free blocks array as free */
                    bDev.readBlock(buffer,infoBlock);
                    /* Convert the byte buffer to the String buffer */
                    String str2=new String(buffer);
                    /* Is info block empty? */
                    if(str2.indexOf(" ")>0) {
                        /* No, so unmark the listed block(s) */
                        while(str2.indexOf(" ")>0) {
                            freeBlocks[(Integer.parseInt(str2.substring(0,str2.indexOf(","))))]=-1;
                            str2=str2.substring(str2.indexOf(",")+1);
                        }
                    }
                    /* Unmark info block in free blocks array as free */
                    freeBlocks[infoBlock]=-1;
                    /* Unmark file as deleted in file table */
                    str=str.substring(0,attrS)+"00"+str.substring(attrE);
                    buffer=str.getBytes();
                    /* Store changes */
                    bDev.writeBlock(buffer,currentBlock);
                    return;
                }
            }
        }
    }

    /**
     * dir will find all the file names in the file system.
     * @return a <code>String[]</code> with all the files in the file system.
     * @author Dimitris Michelinakis
     */
    public String[] dir () {
        /* Initialize our byte, String and Vector buffers */
        byte buffer[]=new byte[bDev.getBlockSize()];
        String str="";
        Vector dirdir=new Vector(maxNumberOfFiles/2,maxNumberOfFiles/2);
        int currentBlock=0, indexStart, indexEnd=0;
        /* Go through the file table and gather file information */
        for(int i=1;i<=maxNumberOfFiles;i++) {
            /* Read the current block */
            bDev.readBlock(buffer,currentBlock);
            /* Convert the byte buffer to the String buffer */
            str=new String(buffer);
            indexStart=str.indexOf(":",indexEnd)+1;
            if(indexStart==0) {
                currentBlock++; indexStart=0; indexEnd=0; i--;
                continue;
            }
            /* Skip unneeded options */
            indexEnd=str.indexOf(";",indexStart+1);
            indexStart=indexEnd+1;
            indexEnd=str.indexOf(".",indexStart+1);
            indexStart=indexEnd+1;
            indexEnd=str.indexOf(";",indexStart+1);
            /* Remember the attributes */
            int attr=Integer.parseInt(str.substring(indexStart,indexEnd));
            indexStart=indexEnd+1;
            indexEnd=indexStart+maxFileNameLength-1;
            /* Remember the file name */
            String name=str.substring(indexStart,indexEnd);
            /* If the slot isn't empty... */
            if(attr!=00) {
                /* Add the file name in the Vector buffer */
                dirdir.addElement(name.trim());
            }
        }
        /* If there are files in the file table */
        if(dirdir.size()>0)
            /* Return an array of the file names */
            return (String[])dirdir.toArray(new String[] {});
        /* Otherwise return null */
        return null;
    }

    /**
     * openFilesCheck will check if a file is open or not based on the name of the file.
     * @param filename String argument is the name of the file.
     * @return a <code>boolean</code> of true if the file is open or false otherwise.
     * @author Dimitris Michelinakis
     */
    private boolean openFilesCheck(String filename) {
        if(flagCaseSensitive==false) {
            /* Scan through the open files array */
            for(int i=0;i<of.length;i++) {
                if(of[i]!=null)
                    /* Case insensitive search */
                    if(of[i].filename.equalsIgnoreCase(filename))
                        return true;
            }
        } else {
            /* Scan through the open files array */
            for(int i=0;i<of.length;i++) {
                if(of[i]!=null)
                    /* Case sensitive search */
                    if(of[i].filename.equals(filename))
                        return true;
            }
        }
        return false;
    }

    /**
     * openFilesOpen will open a file.
     * @param mode char argument is the Read/Write/Append mode.
     * @param infoBlock int argument is the location of the info block.
     * @param fd int argument is the file descriptor of this file.
     * @param attr int argument are the attributes of this file.
     * @param filename String argument is the name of this file.
     * @return a <code>boolean</code> with true if the file was opened, false if something went wrong.
     * @author Dimitris Michelinakis
     */
    private boolean openFilesOpen(char mode,int infoBlock,int fd,int attr,String filename) {
        /* Scan through opened filed */
        for(int i=0;i<of.length;i++) {
            /* Find empty slot */
            if(of[i]==null) {
                /* Open file in empty slot */
                of[i]=new OpenFile(mode,infoBlock,fd,attr,filename);
                return true;
            }
        }
        return false;
    }

    /**
     * OpenFile is a private class to represent an open file.
     * @author Dimitris Michelinakis
     */
    private class OpenFile {
        /** File descriptor. */
        public int fd;
        /** Info block, stores blocks used by the file. */
        public int infoBlock;
        /** File attributes. */
        public int attr;
        /** File name. */
        public String filename;
        /** File open mode (r/w/a). */
        public char mode;
        /** Read possition pointer, {block,possition}. */
        public int readPos[]={1,0};

        /**
         * OpenFile is the constructor of the OpenFile class, which
         * initializes the open file variables.
         * @param mode char argument is the Read/Write/Append mode.
         * @param infoBlock int argument is the location of the info block.
         * @param fd int argument is the file descriptor of this file.
         * @param attr int argument are the attributes of this file.
         * @param filename String argument is the name of this file.
         * @author Dimitris Michelinakis
         */
        public OpenFile(char mode,int infoBlock,int fd,int attr,String filename) {
            /* The constructor initializes the variables */
            this.fd=fd;
            this.infoBlock=infoBlock;
            this.attr=attr;
            this.filename=filename;
            this.mode=mode;
        }
    }
}
