/*************************************************************************/
/*																		 */
/*		  File: language.cr 											 */
/*	   Created: 4 Jul 1994												 */
/* Description: File type/extension language support macros 			 */
/*																		 */
/*************************************************************************/

# include "crisp.h"

int 	_package_buf;
string	_curr_active = "";

/*****************************************/
/*	  Register edit & open new hooks	 */
/*****************************************/

void
main()
{
	register_macro(REG_EDIT, "_call_on_packages");
	register_macro(REG_NEW,  "_call_first_packages");
	_package_buf = create_buffer("Packages", NULL, 1);
}

void
_dump_packages()
{
	int		cur_buf = inq_buffer();
	set_buffer(_package_buf);
	top_of_buffer();
	drop_anchor(MK_LINE);
	end_of_buffer();
	copy();
	raise_anchor();
	set_buffer(cur_buf);
	paste();
}


/*****************************************/
/*	  Call "on" event                    */
/*****************************************/

void
_call_on_packages()
{
	_call_packages("_on");
}


/*****************************************/
/*	  Call new if file is new and		 */
/*	  first for either case 			 */
/*****************************************/

void
_call_first_packages()
{
	string pathname;

	inq_names(pathname);
	if (!exist(pathname))
		_call_packages("_new");
	_call_packages("_first");
}


/****************************************************************************/
/*	  This macro is the main event parser and dispatcher for the language	*/
/*	  package system. It calls packages specified in the BPACKAGES			*/
/*	  environment variable in response to events generated by the system or */
/*	  by the user. It then maintains some housekeeping information that 	*/
/*	  allows rapid access to these pacakages in the future. 				*/
/****************************************************************************/

void
_call_packages(string event_name)
{
	string		to_call,
				mac_name,
				file_ext,
				bpackages,
				escaped_ext;
	int 		loc,
				file_buf,
				already_parsed;

	/*****************************************/
	/*	  First, we get the event type		 */
	/*	  and the extension of the file.	 */
	/*	  If there are any "off" macro       */
	/*	  to call (saved in the 			 */
	/*	  _curr_active global string),		 */
	/*	  we reset them before continuing.	 */
	/*****************************************/
	inq_names(NULL, file_ext);
	escaped_ext = escape_re(file_ext);
	_reset_packages();

	/***********************************************************/
	/*    The global _package_buf contains housekeeping		   */
	/*    information that allows the language system to	   */
	/*    keep parsed event information around for the		   */
	/*    duration of the editing session. Each event		   */
	/*    occupies one line in the buffer (per extension),	   */
	/*    and one additional line is taken per extension	   */
	/*    to save the pre-parsed packages and arguments.	   */
	/*    													   */
	/*    Extensions are always found together in the		   */
	/*    package buffer; the events come first, followed	   */
	/*    by the pre-parsed information. This allows the	   */
	/*    search pattern to fall through to the pre-parsed	   */
	/*    information if the event hasn't occurred yet.		   */
	/*    													   */
	/*    The format of the strings is:						   */
	/*    													   */
	/*    extension][event name, or nothing if				   */
	/*    pre_parse];[information]							   */
	/***********************************************************/
	file_buf = inq_buffer();
	set_buffer(_package_buf);
	top_of_buffer();

	already_parsed = search_fwd ("<_" + escaped_ext + "\\c{" + event_name + "}@;");
	if (already_parsed)
	{
		next_char(--already_parsed);
		to_call = trim(read());
	}
	else
	{
		int 		located, pass;
		string		pattern;

		/***********************************************************/
		/*    If no information is present in the buffer for	   */
		/*    the given extension, the BPACKAGES string must	   */
		/*    be retrieved from the environment and parsed.		   */
		/*    Note that this will only happen once per			   */
		/*    extension.										   */
		/*    													   */
		/*    Since the BPACKAGES string is quite complex, and	   */
		/*    a given extension can appear more than once (and	   */
		/*    with a number of different packages), and parser	   */
		/*    loops and search for the extension until it		   */
		/*    can't be located any more.						   */
		/*    													   */
		/*    If the extension isn't found at all, the			   */
		/*    "default" extension is used and a second pass is	   */
		/*    made over the BPACKAGES string.					   */
		/*    													   */
		/*    If neither the extension nor the "default"		   */
		/*    extension is found, no events are generated for	   */
		/*    this extension.									   */
		/***********************************************************/

		top_of_buffer();
		pattern = "<|[,;] @" + escaped_ext + " @[~;]@:\\c[~:\n;]@[\n;]";
		bpackages = compress(getenv("BPACKAGES"), 1) + "\n";
		to_call = "";

		located = 0;
		for (pass = 0; pass++ < 2; )
		{
			int loc = search_string(pattern, bpackages, NULL, NULL, 0);
			while (loc != 0)
			{
				++located;
				bpackages = substr(bpackages, loc); /* Junk the unused part */
				if ((loc = index(bpackages, ";")) == 0)
					loc = strlen(bpackages);

					/************************************/
					/*	All of the pre-expansion		*/
					/*	packages for the current		*/
					/*	extension are combined			*/
					/*	together in the to_call string	*/
					/*	separated by commas. This		*/
					/*	string is used later to expand	*/
					/*	and call the individual 		*/
					/*	package macros, and is			*/
					/*	inserted into the buffer as 	*/
					/*	pre-parse information once the	*/
					/*	parse has been completed.		*/
					/************************************/

				if (strlen(to_call))
					to_call += ",";
				to_call += substr(bpackages, 1, loc - 1);
				bpackages = substr(bpackages, loc + 1);
				loc = search_string(pattern, bpackages, NULL, NULL, 0);
			}

			/*************************************/
			/*	If we can't find the extension   */
			/*	any more and we've located       */
			/*	some packages to call, then 	 */
			/*	the parse is over. Otherwise,	 */
			/*	we use "default" as the          */
			/*	extension and try one more time  */
			/*************************************/

			if (located)
				break;

			pattern = "<|[,;] @default @[~;]@:\\c[~:\n;]+[\n;]";
		}
	}

	file_ext = "_" + file_ext;

	/************************************************************************/
	/* If the package buffer didn't contain any information about the		*/
	/* current file extension, we insert the pre-parse information at		*/
	/* the top of the buffer. This information can be used later to avoid	*/
	/* doing an additional BPACKAGES parse.									*/
	/************************************************************************/
	beginning_of_line();
	if (!already_parsed)
	{
		insert(file_ext + ";" + to_call + "\n");

		/**********************************/
		/* We move up here because the	  */
		/* event information must appear  */
		/* before the pre-parse 		  */
		/* information, and the "\n" that */
		/* was inserted moves us down a   */
		/* line.						  */
		/**********************************/

		up();
	}
	else
	{
		/*********************************************************************/
		/* As you may recall, locating the pre-parse information will set	 */
		/* already_parsed to 2, and locating specific event information will */
		/* set it to some number above that (depending on the length of the  */
		/* event name). Decrementing already_parsed at this point will force */
		/* the insertion of event-specific information if only pre-parse	 */
		/* information had been located in the package buffer search.		 */
		/*********************************************************************/
		--already_parsed;
	}

	set_buffer(file_buf);
	bpackages = "";

	/************************************************************************/
	/* At this point, we've got a list of pre-expanded package names that   */
	/* look something like "s,pvcs,wp". These have to be expanded and       */
	/* converted into the event and extension specific variety. Each		*/
	/* individual package is separated and passed to process_package,		*/
	/* which expands the name (if necessary), handles extension-specific	*/
	/* and generic naming conventions, deals with parameters, and the like.	*/
	/* Once completed, the expanded package name (if any) is returned and	*/
	/* saved in another string that will be inserted as the event-specific	*/
	/* information in the package buffer.									*/
	/************************************************************************/
	while (strlen(to_call))
	{
		if ((loc = index(to_call, ",")) == 0)
			loc = strlen(to_call) + 1;
		mac_name = process_package(event_name,ltrim(trim(substr(to_call,1,loc-1))), already_parsed);
		if (strlen(mac_name))
		{
			if (strlen(bpackages))
				bpackages += ",";
			bpackages += mac_name;
		}
		to_call = substr(to_call, loc+1);
	}

	/*********************************************************/
	/* If the event-specific information wasn't found in the */
	/* buffer, it's inserted here.                           */
	/*********************************************************/
	if (!already_parsed)
	{
		file_buf = inq_buffer();
		set_buffer(_package_buf);
		insert(file_ext + event_name + ";" + bpackages + "\n");
		set_buffer(file_buf);
	}
}

/****************************************************************************/
/*	  escape_re()															*/
/*																			*/
/*	  This function changes a string with regular expression characters in	*/
/*	  it to one without, and returns the result.							*/
/****************************************************************************/

string
escape_re (string original)
{
	int 	curr_pos,
			original_len;
	string	escaped,
			curr_char;

	original_len = strlen(original);
	curr_pos = 0;
	escaped = "";
	while(original_len--)
	{
		curr_char = substr(original, ++curr_pos, 1);
		if (index ("?*@+\\|%${}[]<>", curr_char))
			escaped += "\\";
		escaped += curr_char;
	}
	return escaped;
}



/**************************************************************************/
/*	  process_package:													  */
/*																		  */
/*	  This macro converts generic package names into their generic or	  */
/*	  extension-specific macro name counterparts.						  */
/**************************************************************************/


string
process_package(string event_name, string mac_name, int processed)
{
	string		file_ext,
				initial_mac,
				parms;
	int 		loc,
				length;

	/*****************************************/
	/*	  If the names haven't already       */
	/*	  been parsed and expanded by		 */
	/*	  this routine, we have a lot of	 */
	/*	  work to do. This work is			 */
	/*	  skipped once it has been done 	 */
	/*	  once. 							 */
	/*****************************************/

	if (!processed)
	{
		inq_names(NULL, file_ext);
		file_ext = "_" + file_ext;
		loc = index(mac_name, " ");
		if (loc)
		{
			parms = substr(mac_name, loc + 1);
			mac_name = substr(mac_name, 1, loc - 1);
		}
		else
			parms = "";

		/*****************************************/
		/*	  If part of the package name is	 */
		/*	  found in the abbreviation 		 */
		/*	  list, the full name is			 */
		/*	  retrieved and used instead.		 */
		/*	  Note that the macros in the		 */
		/*	  abbreviation list could be		 */
		/*	  demotable groups (separated by	 */
		/*	  commas).							 */
		/*****************************************/

		initial_mac = _package_abbrev();
		loc = search_string("[,;]" + mac_name + "[~;]+;", initial_mac, length);
		initial_mac = (loc) ? substr(initial_mac, loc + 1, length - 2)
							: mac_name;

		/*****************************************/
		/*	  This loop searches for			 */
		/*	  specific and generic event		 */
		/*	  handler macros. First,			 */
		/*	  specific macros are searched		 */
		/*	  for. If none are found, the		 */
		/*	  last member of the demotable		 */
		/*	  group is searched for 			 */
		/*	  generically. This is because		 */
		/*	  generic higher level demotion 	 */
		/*	  macros don't make any sense        */
		/*	  (they'd override anything          */
		/*	  else, and wouldn't leave           */
		/*	  anything to demote to).			 */
		/*****************************************/

		while (!processed)
		{
			loc = index(initial_mac, ",");
			if (!loc)
				loc = strlen(initial_mac) + 1;
			mac_name = substr(initial_mac, 1, loc - 1);
			initial_mac = substr(initial_mac, loc + 1);
			mac_name = "_" + mac_name + event_name;
			processed = inq_macro(file_ext + mac_name);
			if (processed)
				mac_name = file_ext + mac_name;
			else
			{
				if (!strlen(initial_mac))
				{
					processed = inq_macro(mac_name);
					break;
				}
			}
		}
	}

	/*****************************************/
	/*	  If the macro has been 			 */
	/*	  processed & located, we call		 */
	/*	  it with its parameters.			 */
	/*****************************************/

	if (processed)
	{
		if (strlen(parms))
			mac_name += " " + parms;
		initial_mac = execute_macro(mac_name);
		if (initial_mac != "" && event_name == "_on")
		{
			if (strlen(_curr_active))
				_curr_active += ";";
			_curr_active += initial_mac;
		}
		return mac_name;
	}

	return "";
}


/******************************************************************************/
/*	  reset_packages:														  */
/*																			  */
/*	  This macro processes the "off" event. Any macros that returned off      */
/*	  event macros from their event handlers have those strings kept in the   */
/*	  _curr_active global variable. These macros are called here.			  */
/*																			  */
/*	  Note that "off" macro strings should never contain semicolons.          */
/*																			  */
/******************************************************************************/

void
_reset_packages()
{
	int loc;

	while (strlen(_curr_active))
	{
		loc = index(_curr_active, ";");
		if (!loc)
			loc = strlen(_curr_active) + 1;
		execute_macro (substr(_curr_active, 1, loc - 1));
		_curr_active = substr(_curr_active, loc + 1);
	}
}


/******************************************************************************/
/*	  package_abbrev:														  */
/*																			  */
/*	  This macro is at the terminus of the chain. It returns a string		  */
/*	  consisting of the abbreviations and demotable groups for the built-in   */
/*	  packages (indenting, word processing and pvcs).						  */
/******************************************************************************/

string
_package_abbrev()
{
	return ";template,smart,regular;smart,regular;pvcs;wp;";
}

