#ifndef TAGS_MAIN_H
#define TAGS_MAIN_H

#include <vsapi.h>


extern "C" {

//////////////////////////////////////////////////////////////////////
// "C" style API for creating and accessing BTREE tag database.
//

//////////////////////////////////////////////////////////////////////
// Database version, corresponding to Visual Slickedit, release 3.0
//
#define VS_TAG_USER_VERSION             3000
#define VS_TAG_USER_VERSION_WIDE_FLAGS  3100
#define VS_TAG_USER_VERSION_FILE_TYPES  4000
#define VS_TAG_USER_VERSION_CASE_FIXED  4100
#define VS_TAG_LATEST_VERSION           4100
#define VS_REF_USER_VERSION             4000
#define VS_REF_LATEST_VERSION           4000


//////////////////////////////////////////////////////////////////////
// Database file types that can be opened using this library
//
#define VS_DBTYPE_tags        0
#define VS_DBTYPE_references  1
#define VS_DBTYPE_msbrowse    2


//////////////////////////////////////////////////////////////////////
// Standard file types, used to distinguish between source files
// pulled in naturally by tagging, object files, class files,
// executables, browser databaser, debug databases, and source files
// referenced by various other items.
//
#define VS_FILETYPE_source      0
#define VS_FILETYPE_references  1
#define VS_FILETYPE_LASTID      1


//////////////////////////////////////////////////////////////////////
// Standard tag types, by default, always present in database
// standard type name is always "xxx" for VS_TAGTYPE_xxx,
// for example, the type name for VS_TAGTYPE_proc is "proc".
// ID's 30-127 are reserved for future use by Microedge.
//
#define VS_TAGTYPE_proc         1  // procedure or command
#define VS_TAGTYPE_proto        2  // function prototype
#define VS_TAGTYPE_define       3  // preprocessor macro definition
#define VS_TAGTYPE_typedef      4  // type definition
#define VS_TAGTYPE_gvar         5  // global variable declaration
#define VS_TAGTYPE_struct       6  // structure definition
#define VS_TAGTYPE_enumc        7  // enumeration value
#define VS_TAGTYPE_enum         8  // enumerated type
#define VS_TAGTYPE_class        9  // class definition
#define VS_TAGTYPE_union       10  // structure / union definition
#define VS_TAGTYPE_label       11  // label
#define VS_TAGTYPE_interface   12  // interface, eg, for Java
#define VS_TAGTYPE_constructor 13  // class constructor
#define VS_TAGTYPE_destructor  14  // class destructor
#define VS_TAGTYPE_package     15  // package / module / namespace
#define VS_TAGTYPE_var         16  // member of a class / struct / package
#define VS_TAGTYPE_lvar        17  // local variable declaration
#define VS_TAGTYPE_constant    18  // pascal constant
#define VS_TAGTYPE_function    19  // function
#define VS_TAGTYPE_property    20  // property?
#define VS_TAGTYPE_program     21  // pascal program
#define VS_TAGTYPE_library     22  // pascal library
#define VS_TAGTYPE_parameter   23  // overloaded operator
#define VS_TAGTYPE_import      24  // package import or using
#define VS_TAGTYPE_friend      25  // C++ friend relationship
#define VS_TAGTYPE_database    26  // SQL/OO Database
#define VS_TAGTYPE_table       27  // Database Table
#define VS_TAGTYPE_column      28  // Database Column
#define VS_TAGTYPE_index       29  // Database index
#define VS_TAGTYPE_view        30  // Database view
#define VS_TAGTYPE_trigger     31  // Database trigger
#define VS_TAGTYPE_form        32  // GUI Form or window
#define VS_TAGTYPE_menu        33  // GUI Menu
#define VS_TAGTYPE_control     34  // GUI Control or Widget
#define VS_TAGTYPE_eventtab    35  // GUI Event table
#define VS_TAGTYPE_procproto   36  // Prototype for procedure
#define VS_TAGTYPE_task        37  // Ada task object
#define VS_TAGTYPE_include     38  // C++ include, Ada with
#define VS_TAGTYPE_LASTID      38  // last tag type ID
#define VS_TAGTYPE_FIRSTUSER  128  // first user-defined tag type ID

//////////////////////////////////////////////////////////////////////
// Tag type filtering flags, formerly PUSHTAG_* flags in slick.sh
//
#define VS_TAGFILTER_CASESENSITIVE 0x01
#define VS_TAGFILTER_PROC          0x02
#define VS_TAGFILTER_PROTO         0x04
#define VS_TAGFILTER_DEFINE        0x08
#define VS_TAGFILTER_ENUM          0x10
#define VS_TAGFILTER_GVAR          0x20
#define VS_TAGFILTER_TYPEDEF       0x40
#define VS_TAGFILTER_STRUCT        0x80
#define VS_TAGFILTER_UNION         0x100
#define VS_TAGFILTER_LABEL         0x200
#define VS_TAGFILTER_INTERFACE     0x400
#define VS_TAGFILTER_PACKAGE       0x800
#define VS_TAGFILTER_VAR           0x1000
#define VS_TAGFILTER_CONSTANT      0x2000
#define VS_TAGFILTER_PROPERTY      0x4000
#define VS_TAGFILTER_LVAR          0x8000
#define VS_TAGFILTER_MISCELLANEOUS 0x10000
#define VS_TAGFILTER_DATABASE      0x20000
#define VS_TAGFILTER_GUI           0x40000
#define VS_TAGFILTER_INCLUDE       0x80000
#define VS_TAGFILTER_ANYTHING      0xffffe
#define VS_TAGFILTER_ANYPROC       (VS_TAGFILTER_PROTO|VS_TAGFILTER_PROC)
#define VS_TAGFILTER_ANYDATA       (VS_TAGFILTER_GVAR|VS_TAGFILTER_VAR|VS_TAGFILTER_LVAR|VS_TAGFILTER_PROPERTY)
#define VS_TAGFILTER_ANYSTRUCT     (VS_TAGFILTER_STRUCT|VS_TAGFILTER_UNION|VS_TAGFILTER_INTERFACE)


//////////////////////////////////////////////////////////////////////
// administrative functions

// Specify the amount of memory to use for the database cache.
//    cache_size -- amount of memory in bytes
// Returns 0 on success, <0 on error.  The minimum size cache
// allowed is 512k.
//
int VSAPI tag_set_cache_size(int cache_size);

// Create a tag database, with standard tables, index, and types.
//    file_name -- file path where to create new database
//                 If file_name exists, it will be truncated.
//    db_type   -- (optional) if not given, creates tag database.
//                 if (db_type==VS_DBTYPE_references), then creates
//                 a tag references database.
// Returns database handle >= 0 on success, <0 on error.
//
int VSAPI tag_create_db(VSPSZ file_name /*, int db_type */);

// Open an existing tag database and return a handle to the database.
// This function opens the database for read-write access.
//    file_name -- file name of tag database to open.
// Returns database handle >= 0 on success, <0 on error.
//
int VSAPI tag_open_db(VSPSZ file_name);

// Open an existing tag database and return a handle to the database.
// This function opens the database for read-only access.
//    file_name -- file name of tag database to open.
// Returns database handle >= 0 on success, <0 on error.
//
int VSAPI tag_read_db(VSPSZ file_name);

// Flush all unwritten data to disk for the database.
// Returns 0 on success, <0 on error.
//
int VSAPI tag_flush_db();

// Return the name of the database currently open
// Returns the empty string on error.
VSPSZ VSAPI tag_current_db();

// Close the current tag database.
//    file_name -- (optional) explicite filename of database to close
//                 otherwise the current open database is closed.
// Returns 0 on success, <0 on error.
//
int VSAPI tag_close_db(/*VSPSZ file_name*/);

// Close all open tag databases.
// Returns 0 on success, <0 on error.
//
int VSAPI tag_close_all_db();

// Display the effective version of the tagsdb.dll
void VSAPI tagsdb_version();

// Return the version of the tags database currently open.
// Returns VS_TAG_USER_VERSION or higher
int VSAPI tag_current_version();

// Return the database description/title.
// Returns the empty string on error.
VSPSZ VSAPI tag_get_db_comment();

// Sets the database description/title.
//    comment -- description or title of this database
// Returns 0 on success, <0 on error.
int VSAPI tag_set_db_comment(VSPSZ comment);


//////////////////////////////////////////////////////////////////////
// insertion and removal of tags

// Set up for inserting a series of tags from a single file for
// update.  Doing this allows the tag database engine to detect
// and handle updates more effeciently, even in the presence of
// duplicates.
//    file_name   -- full path of file the tags are located in
// Returns 0 on success, <0 on error.
//
int VSAPI tag_insert_file_start(VSPSZ file_name);

// Clean up after inserting a series of tags from a single file
// for update.  Doing this allows the tag database engine to
// remove any tags from the database that are no longer valid.
// Returns 0 on success, <0 on error.
//
int VSAPI tag_insert_file_end();

// Remove all references from the given references (browse database or
// object) file from the database.  This is an effective, but costly way
// to perform an incremental update of the data imported from a
// references file.  First remove all items associated with that file,
// then insert them again.
//    file_name   -- full path of file the reference info came from
//    remove_file -- if non-zero, the file is removed from the database
// Returns 0 on success, <0 on error.
//
int VSAPI tag_remove_from_file(VSPSZ file_name /*, int remove_file */);


//////////////////////////////////////////////////////////////////////
// file name handling functions

// Modify the date of tagging for the given file.  Since date of tagging
// is not involved in indexing, this is safe to do in the record, in place.
// This method always uses the current date when setting the date of tagging.
//    file_name   -- name of file to update date of tagging for
//    modify_date -- (optional) modification date when tagged, read from disk
//                   if modify_date is NULL.  Format is YYYYMMDDHHMMSSmmm.
// Returns 0 on success, <0 on error.
//
int VSAPI tag_set_date(VSPSZ file_name /*,VSPSZ modify_date*/);

// Retrieve the date of tagging for the given file.
//    file_name   -- name of file to update date of tagging for
//    file_date   -- (reference) returns the file's modification date when tagged
// The string returned by this function is structured such
// that consecutive dates are ordered lexicographically,
// and is reported in local time cooridinates (YYYYMMDDHHMMSSmmm).
// This function has the side effect of finding and position the file iterator
// on the given file name, returns BT_RECORD_NOT_FOUND_RC if file_name is not
// in the database.  Returns 0 on success, <0 on error.
//
int VSAPI tag_get_date(VSPSZ file_name, VSHREFVAR modify_date);

// Retreive the name of the next file included in this tag database.
//    file_id    -- id of file, from tag_get_detail()
//    file_name  -- (reference) full path of file containing tags
// Files are ordered lexicographically, case-sensitive on UNIX platforms,
// case insensitive on DOS/OS2/Windows platforms.
//
int VSAPI tag_get_file(int file_id, VSHREFVAR file_name);

// Retreive the name of the first file included in this tag database.
//    file_name  -- (reference) full path of file containing tags
//    search_for -- (optional) specific file to search for (prefix search)
// Files are ordered lexicographically, case-sensitive on UNIX platforms,
// case insensitive on DOS/OS2/Windows platforms.
//
int VSAPI tag_find_file(VSHREFVAR file_name /*, VSPSZ search_for*/);

// Retreive the name of the next file included in this tag database.
//    file_name  -- (reference) full path of file containing tags
// Files are ordered lexicographically, case-sensitive on UNIX platforms,
// case insensitive on DOS/OS2/Windows platforms.
//
int VSAPI tag_next_file(VSHREFVAR file_name);


//////////////////////////////////////////////////////////////////////
// type name handling functions

// Retreive the name of the next type included in this tag database.
//    type_id    -- id of type, from tag_get_detail()
//    type_name  -- (reference) full path of type containing tags
// types are ordered lexicographically, case-sensitive on UNIX platforms,
// case insensitive on DOS/OS2/Windows platforms.
//
int VSAPI tag_get_type(int type_id, VSHREFVAR type_name);

// Retreive the name of the first type included in this tag database.
//    type_name  -- (reference) full path of type containing tags
//    search_for -- (optional) specific type to search for (prefix search)
// types are ordered lexicographically, case-sensitive on UNIX platforms,
// case insensitive on DOS/OS2/Windows platforms.
//
int VSAPI tag_find_type(VSHREFVAR type_name /*, VSPSZ search_for*/);

// Retreive the name of the next type included in this tag database.
//    type_name  -- (reference) full path of type containing tags
// types are ordered lexicographically, case-sensitive on UNIX platforms,
// case insensitive on DOS/OS2/Windows platforms.
//
int VSAPI tag_next_type(VSHREFVAR type_name);

// Filter the given tag type based on the given filter flags
//    type_id      -- tag type ID
//    filter_flags -- VS_TAGFILTER_*
//    type_name    -- (optional) look up type ID using this name
//    tag_flags    -- (optional) check tag flags for VS_TAGFLAG_maybe_var
// Returns 1 if the type is allowed according to the flags, 0 if not.
//
int VSAPI tag_filter_type(int type_id, int filter_flags /*, VSPSZ type_name, int tag_flags*/);


//////////////////////////////////////////////////////////////////////
// miscellaneous private functions

// Pretty-print function arguments to output buffer
//    string     -- signature, straight from the database
//    length     -- length of signature
//    buffer     -- output buffer
//    bufsize    -- size of output buffer
//
int vsTagPrettyPrintArgs(const char *string, int length,
                         char *buffer, int bufsize);

// Compare two sets of function arguments for weak equality.
// If two argument lists are weakly equivelent, then they may
// or not be matching argument lists.  If two arguments are not
// weakly equivelent, then they are absolutely not equivelent.
//    args1 -- first set of arguments
//    len1  -- length of first set of argument
//    args2 -- second set of arguments
//    len2  -- length of second set of argument
// Returns 0 if they are equal, 1 if not.
//
int vsTagCountArgList(const char *args1, int len1, const char *args2, int len2);

// Compare two sets of function arguments
//    args1     -- first set of arguments
//    len1      -- length of first set of argument
//    args2     -- second set of arguments
//    len2      -- length of second set of argument
//    unqualify -- remove class qualifications and compare again (loose compare)
// Returns 0 if they are equal, 1 if not.
//
int vsTagCompareArgList(const char *args1, int len1, const char *args2, int len2, int unqualify=1);

// Compare two sets of function arguments for equality.
// Handles both pascal and "C" style arguments, allowing for
// default parameters, and differing parameter names.
// Definately does not handle every possible case.
//    sig1 -- first function signature
//    len1 -- length of first function signature
//    sig2 -- second function signature
//    len2 -- length of second function signature
// Returns 0 if it determines that the functions are similar, 1 otherwise.
//
int vsTagArgsCompare(const char *sig1, int len1, const char *sig2, int len2);

// Filter the given tag type based on the given filter flags
//    type_id      -- tag type ID, VS_TAGTYPE_*
//    filter_flags -- VS_TAGFILTER_*
//    type_name    -- tag type name
//    tag_flags    -- tag flags
// Returns 1 if the type is allowed according to the flags, 0 if not.
//
int vsTagFilterType(int type_id, int filter_flags,
                    const char *type_name=0, int tag_flags=0);

// Return the string associated with the standard tag type ID
//    type_id -- tag type ID
// Returns 0 if type_id is not a standard type.
// Returns "" if (type_id == 0).
//
const char *vsTagGetType(int type_id);


// Get the next tag file name from the given array of strings
//    list  --  list of tag files (from tag_filenamea, see tags.e)
//              this is a reference to an array of Slick-C strings
//    i     --  reference to integer index
// Returns the name of the tag file (const char*) on success or
// 0 (null) on error or if there are no more tag files in list.
// This function is implemented in tagscntx.c.
//
const char *vsNextTagFileA(VSHREFVAR list, int &i);


//////////////////////////////////////////////////////////////////////
// map standard tag types to pushtag categories
#if CREATE_TAGTYPE_MAPS
static unsigned int gpushtag_map[] = {
   VS_TAGFILTER_PROC,          // 0 is considered as a proc
   VS_TAGFILTER_PROC,          // #define VS_TAGTYPE_proc         1  // procedure or command
   VS_TAGFILTER_PROTO,         // #define VS_TAGTYPE_proto        2  // function prototype
   VS_TAGFILTER_DEFINE,        // #define VS_TAGTYPE_define       3  // preprocessor macro definition
   VS_TAGFILTER_TYPEDEF,       // #define VS_TAGTYPE_typedef      4  // type definition
   VS_TAGFILTER_GVAR,          // #define VS_TAGTYPE_gvar         5  // global variable declaration
   VS_TAGFILTER_STRUCT,        // #define VS_TAGTYPE_struct       6  // structure definition
   VS_TAGFILTER_ENUM,          // #define VS_TAGTYPE_enumc        7  // enumeration value
   VS_TAGFILTER_ENUM,          // #define VS_TAGTYPE_enum         8  // enumerated type
   VS_TAGFILTER_STRUCT,        // #define VS_TAGTYPE_class        9  // class definition
   VS_TAGFILTER_UNION,         // #define VS_TAGTYPE_union       10  // structure / union definition
   VS_TAGFILTER_LABEL,         // #define VS_TAGTYPE_label       11  // label
   VS_TAGFILTER_INTERFACE,     // #define VS_TAGTYPE_interface   12  // interface, eg, for Java
   VS_TAGFILTER_PROC,          // #define VS_TAGTYPE_constructor 13  // class constructor
   VS_TAGFILTER_PROC,          // #define VS_TAGTYPE_destructor  14  // class destructor
   VS_TAGFILTER_PACKAGE,       // #define VS_TAGTYPE_package     15  // package / module / namespace
   VS_TAGFILTER_VAR,           // #define VS_TAGTYPE_var         16  // member of a class / struct / package
   VS_TAGFILTER_LVAR,          // #define VS_TAGTYPE_lvar        17  // local variable declaration
   VS_TAGFILTER_CONSTANT,      // #define VS_TAGTYPE_constant    18  // pascal constant
   VS_TAGFILTER_PROC,          // #define VS_TAGTYPE_function    19  // function
   VS_TAGFILTER_PROPERTY,      // #define VS_TAGTYPE_property    20  // property
   VS_TAGFILTER_PACKAGE,       // #define VS_TAGTYPE_program     21  // pascal program
   VS_TAGFILTER_PACKAGE,       // #define VS_TAGTYPE_library     22  // pascal library
   VS_TAGFILTER_LVAR,          // #define VS_TAGTYPE_parameter   23  // overloaded operator
   VS_TAGFILTER_MISCELLANEOUS, // #define VS_TAGTYPE_import      24  // package import or using
   VS_TAGFILTER_MISCELLANEOUS, // #define VS_TAGTYPE_friend      25  // C++ friend relationship
   VS_TAGFILTER_DATABASE,      // #define VS_TAGTYPE_database    26  // SQL/OO Database
   VS_TAGFILTER_DATABASE,      // #define VS_TAGTYPE_table       27  // Database Table
   VS_TAGFILTER_DATABASE,      // #define VS_TAGTYPE_column      28  // Database Column
   VS_TAGFILTER_DATABASE,      // #define VS_TAGTYPE_index       29  // Database index
   VS_TAGFILTER_DATABASE,      // #define VS_TAGTYPE_view        30  // Database view
   VS_TAGFILTER_DATABASE,      // #define VS_TAGTYPE_trigger     31  // Database trigger
   VS_TAGFILTER_GUI,           // #define VS_TAGTYPE_form        32  // GUI Form or window
   VS_TAGFILTER_GUI,           // #define VS_TAGTYPE_menu        33  // GUI Menu
   VS_TAGFILTER_GUI,           // #define VS_TAGTYPE_control     34  // GUI Control or Widget
   VS_TAGFILTER_GUI,           // #define VS_TAGTYPE_eventtab    35  // GUI Event table
   VS_TAGFILTER_PROTO,         // #define VS_TAGTYPE_procproto   36  // prototype for procedure
   VS_TAGFILTER_INTERFACE,     // #define VS_TAGTYPE_task        37  // Ada task object
   VS_TAGFILTER_INCLUDE        // #define VS_TAGTYPE_include     38  // C++ include, Ada with
};

// map tag IDs to tag strings
static const char *gtag_map[] = {
   "",           // 0 is interpreted as a proc
   "proc",       // #define VS_TAGTYPE_proc         1  // procedure or command
   "proto",      // #define VS_TAGTYPE_proto        2  // function prototype
   "define",     // #define VS_TAGTYPE_define       3  // preprocessor macro definition
   "typedef",    // #define VS_TAGTYPE_typedef      4  // type definition
   "gvar",       // #define VS_TAGTYPE_gvar         5  // global variable declaration
   "struct",     // #define VS_TAGTYPE_struct       6  // structure definition
   "enumc",      // #define VS_TAGTYPE_enumc        7  // enumeration value
   "enum",       // #define VS_TAGTYPE_enum         8  // enumerated type
   "class",      // #define VS_TAGTYPE_class        9  // class definition
   "union",      // #define VS_TAGTYPE_union       10  // structure / union definition
   "label",      // #define VS_TAGTYPE_label       11  // label
   "interface",  // #define VS_TAGTYPE_interface   12  // interface, eg, for Java
   "constr",     // #define VS_TAGTYPE_constructor 13  // class constructor
   "destr",      // #define VS_TAGTYPE_destructor  14  // class destructor
   "package",    // #define VS_TAGTYPE_package     15  // package / module / namespace
   "var",        // #define VS_TAGTYPE_var         16  // member of a class / struct / package
   "lvar",       // #define VS_TAGTYPE_lvar        17  // local variable declaration
   "const",      // #define VS_TAGTYPE_constant    18  // pascal constant
   "func",       // #define VS_TAGTYPE_function    19  // function
   "prop",       // #define VS_TAGTYPE_property    20  // property
   "prog",       // #define VS_TAGTYPE_program     21  // pascal program
   "lib",        // #define VS_TAGTYPE_library     22  // pascal library
   "param",      // #define VS_TAGTYPE_parameter   23  // overloaded operator
   "import",     // #define VS_TAGTYPE_import      23  // package import or using
   "friend",     // #define VS_TAGTYPE_friend      24  // C++ friend relationship
   "database",   // #define VS_TAGTYPE_database    26  // SQL/OO Database
   "table",      // #define VS_TAGTYPE_table       27  // Database Table
   "column",     // #define VS_TAGTYPE_column      28  // Database Column
   "index",      // #define VS_TAGTYPE_index       29  // Database index
   "view",       // #define VS_TAGTYPE_view        30  // Database view
   "trigger",    // #define VS_TAGTYPE_trigger     31  // Database trigger
   "form",       // #define VS_TAGTYPE_form        32  // GUI Form or window
   "menu",       // #define VS_TAGTYPE_menu        33  // GUI Menu
   "control",    // #define VS_TAGTYPE_control     34  // GUI Control or Widget
   "eventtab",   // #define VS_TAGTYPE_eventtab    35  // GUI Event table
   "procproto",  // #define VS_TAGTYPE_procproto   36  // prototype for procedure
   "task",       // #define VS_TAGTYPE_task        37  // Ada task object
   "include"     // #define VS_TAGTYPE_include     38  // C++ include, Ada with
};
#endif

} // extern "C"


#endif
// TAGS_MAIN_H
