/**************************************************************************/
/*                                                                        */
/* RexxMail is a REXX program that uses the Workplace Shell, the          */
/* graphical user interface of the IBM OS/2 operating system, to          */
/* create an easy to use, object-oriented, highly flexible e-mail         */
/* message processing system.                                             */
/*                                                                        */
/**************************************************************************/
/*                                                                        */
/* This program forms part of the RexxMail package, and may not be        */
/* distributed separately.                                                */
/*                                                                        */
/**************************************************************************/
/*                                                                        */
/* The latest version of RexxMail can be found at                         */
/*          www.degeus.com/rexxmail  */
/*                                                                        */
/**************************************************************************/
/*                                                                        */
/* This program is released under the terms of the GNU license, see       */
/*          www.gnu.org/copyleft/gpl.html                                 */
/*                                                                        */
/**************************************************************************/
/*                                                                        */
/* (c) 2006 Marcus de Geus  */
/*          marcus@degeus.com  */
/*          www.degeus.com  */
/*                                                                        */
/**************************************************************************/
/*                                                                        */
/* Use it if you like it. Don't if you don't. No legalese.                */
/*                                                                        */
/**************************************************************************/

/**************************************************************************/
/* Main                                                                   */
/**************************************************************************/

signal on Error  /* handles general error condition */
signal on Failure  /* handles external program errors */
signal on Halt  /* handles halt condition */
signal on NoValue  /* handles initialization errors */
signal on Syntax  /* handles syntax errors */
signal off NotReady  /* we handle I/O errors */
numeric digits 12  /* recognize up to 12 digits */

/**************************************************************************/
/* initialize some global strings                                         */
/**************************************************************************/

drop Global.  /* start with nothing */

Global.!CurDir = directory()  /* get the current directory */
Global.!Build = '20061224.155329'  /* define the build number */
Global.!BuildMess = 'RexxMail e-mail processor for OS/2 - build number '||Global.!Build  /* a build number message */
Global.!Copyright = '(c) 2006 Marcus de Geus - marcus@degeus.com - www.degeus.com'  /* a copyright line */
Global.!CRLF = d2c(13)||d2c(10)  /* define CRLF */
Global.!EmptyLine = Global.!CRLF||Global.!CRLF  /* define an empty line */
Global.!Warning = d2c(16)  /* the warning string to use in message headers and object titles -- see the CheckCommLine procedure! */
Global.!Errors = ''  /* no error messages yet */
Global.!ExtraCharacters = '(){}[]<>/\-_=+|"*;:,.'  /* extra formatting characters allowed in some settings strings */
Global.!SMTPConnect = ''  /* we have no SMTP connection yet */
Global.!MessBar.0 = 0  /* we have nothing on the message bar yet */
Global.!SilentOnce = 0  /* no need to be silent yet */

Global.!ASCII_PC819 = '________________'||,  /* bytes 80 to 8F of the ASCII => PC819 (ISO-8859-1) conversion table */
                      '________________'||,  /* bytes 90 to 9F */
                      '________________'||,  /* bytes A0 to AF */
                      '________________'||,  /* bytes B0 to BF */
                      '________________'||,  /* bytes C0 to CF */
                      '________________'||,  /* bytes D0 to DF */
                      '________________'||,  /* bytes E0 to EF */
                      '________________'  /* bytes F0 to FF */

Global.!PC437_PC819 = ''||,  /* bytes 80 to 8F of the PC437 => PC819 (ISO-8859-1) conversion table */
                      '_f'||,  /* bytes 90 to 9F */
                      'Ѫ_'||,  /* bytes A0 to AF */
                      '________________'||,  /* bytes B0 to BF */
                      '________________'||,  /* bytes C0 to CF */
                      '________________'||,  /* bytes D0 to DF */
                      'aGpSstFTOd_fe_'||,  /* bytes E0 to EF */
                      '__________'  /* bytes F0 to FF */

Global.!PC850_PC819 = ''||,  /* bytes 80 to 8F of the PC850 => PC819 (ISO-8859-1) conversion table */
                      'f'||,  /* bytes 90 to 9F */
                      'Ѫ'||,  /* bytes A0 to AF */
                      '__________'||,  /* bytes B0 to BF */
                      '_____________'||,  /* bytes C0 to CF */
                      'Ȁ_____'||,  /* bytes D0 to DF */
                      'յݯ'||,  /* bytes E0 to EF */
                      '_*'  /* bytes F0 to FF */

Global.!PC1004_PC819 = '_'||"'"||'_".+*^_S<O_Z_'||,  /* bytes 80 to 8F of the PC1004 => PC819 (ISO-8859-1) conversion table */
                       '_'||"''"||'""*--~Ts>o_zY'||,  /* bytes 90 to 9F */
                       xrange('A0'x,'FF'x)  /* bytes A0 to FF */

Global.!PC819_ASCII = 'E_______________'||,  /* bytes 80 to 8F of the PC819 (ISO-8859-1) => ASCII conversion table */
                      '________________'||,  /* bytes 90 to 9F */
                      ' !cL*Y|p"ca<\-r_'||,  /* bytes A0 to AF */
                      'o_23'||"'"||'us-,1o>___?'||,  /* bytes B0 to BF */
                      'AAAAAAACEEEEIIII'||,  /* bytes C0 to CF */
                      'DNOOOOOxOUUUUYps'||,  /* bytes D0 to DF */
                      'aaaaaaaceeeeiiii'||,  /* bytes E0 to EF */
                      'dnooooo/ouuuuyPy'  /* bytes F0 to FF */

Global.!PC819_PC437 = 'E_______________'||,  /* bytes 80 to 8F of the PC819 (ISO-8859-1) => PC437 conversion table */
                      '________________'||,  /* bytes 90 to 9F */
                      '*Y|p"c-r_'||,  /* bytes A0 to AF */
                      '3'||"'"||'s,1_'||,  /* bytes B0 to BF */
                      'AAAAEEEIIII'||,  /* bytes C0 to CF */
                      'DOOOOxOUUUYp'||,  /* bytes D0 to DF */
                      'a'||,  /* bytes E0 to EF */
                      'dooyP'  /* bytes F0 to FF */

Global.!PC819_PC850 = '_______________'||,  /* bytes 80 to 8F of the PC819 (ISO-8859-1) => PC850 conversion table */
                      '________________'||,  /* bytes 90 to 9F */
                      'Ͼ'||,  /* bytes A0 to AF */
                      ''||,  /* bytes B0 to BF */
                      'ǎԐ'||,  /* bytes C0 to CF */
                      'ѥ噞'||,  /* bytes D0 to DF */
                      'Ƅ'||,  /* bytes E0 to EF */
                      'Ф'  /* bytes F0 to FF */

Global.!PC819_PC1004 = xrange('80'x,'FF'x)  /* bytes 80 to FF of the PC819 (ISO-8859-1) => PC1004 conversion table */

/**************************************************************************/
/* put out an identifying message                                         */
/**************************************************************************/

call ShowLine Global.!BuildMess  /* report to the console */
call ShowLine Global.!Copyright  /* report to the console */

/**************************************************************************/
/* Get the operating system, program directory and program name           */
/**************************************************************************/

parse source OS . Global.!ProgSpec  /* get the OS and ProgSpec */
Global.!ProgDir = strip(filespec('D',Global.!ProgSpec)||filespec('P',Global.!ProgSpec),'T','\')  /* the program directory name */
Global.!ErrorLog = Global.!ProgDir||'\error.log'  /* the error log name for the time being (until we get our configuration sorted out) */
Global.!TempDir = value('TEMP',,'OS2ENVIRONMENT')  /* a temp temp dir */

if (Global.!TempDir == '') then  /* if no TEMP system var was defined */
do
 Global.!TempDir = Global.!ProgDir  /* use the program dir */
end

/**************************************************************************/
/* Make sure this is OS/2                                                 */
/**************************************************************************/

if (OS >< 'OS/2') then  /* if the OS is not OS/2 */
do
 call AddError,'Fatal error: this program requires OS/2 or eCS to run'  /* report */
 call Quit  /* and quit */
end

/**************************************************************************/
/* Change to the program directory to load DLLs                           */
/**************************************************************************/

call directory Global.!ProgDir  /* change to the program dir */

/**************************************************************************/
/* Try to load the REXX utilities                                         */
/**************************************************************************/

if (\LoadRexxUtil()) then  /* if we cannot get the Rexx utilities library */
do
 call Quit  /* and quit */
end

/**************************************************************************/
/* Try to load the REXX socket functions                                  */
/**************************************************************************/

if (\LoadRexxSock()) then  /* if we cannot get the Rexx socket functions library */
do
 call Quit  /* and quit */
end

/**************************************************************************/
/* Try to load the RexxMailUtil functions                                 */
/**************************************************************************/

if (\LoadRexxMailUtil()) then  /* if we cannot get the RexxMail functions library */
do
 call Quit  /* and quit */
end

/**************************************************************************/
/* Change back to the original directory                                  */
/**************************************************************************/

call directory Global.!CurDir  /* change back */

/**************************************************************************/
/* Make sure we have the utilities level we need                          */
/**************************************************************************/

if (rxfuncquery('sysutilversion')) then  /* if we cannot find the SysUtilVersion function */
do
 call AddError 'Fatal error: REXXUTIL.DLL is not at the required level'  /* report */
 call Quit  /* and quit */
end

RexxUtilLevel = sysutilversion()  /* get the REXX utilities lib version */

if (RexxUtilLevel < '2.00') then  /* if it is not the right level */
do
 call AddError 'Fatal error: REXXUTIL.DLL is not at the required level; found level : '||RexxUtilLevel  /* report */
 call Quit  /* and quit */
end

/**************************************************************************/
/* Get the currently active code page and load another if necessary       */
/**************************************************************************/

OriginalCP = sysqueryprocesscodepage()  /* get the current code page for use later on */
OKCodePages = '437 850 1004'  /* the fully supported code pages */
CPIndex = 0  /* start an index  at 0 */

do until (GotCP | (CPIndex == words(OKCodePages)))  /* go on until we get what we want or run out of options */

 Global.!CodePage = sysqueryprocesscodepage()  /* get the current code page */
 GotCP = (pos(Global.!CodePage,OKCodePages) > 0)  /* if we have a valid code page, set a flag */

 if (\GotCP) then  /* if we have not got a valid code page yet */
 do

  CPIndex = CPIndex + 1  /* up the index */
  NextCodePage = word(OKCodePages,CPIndex)  /* get the next option */

  if (syssetprocesscodepage(NextCodePage) == 0) then  /* if we can load it */
  do
   call ShowLine 'Loaded code page '||NextCodePage  /* report */
   Global.!CodePage = NextCodePage  /* the current code page */
   GotCP = 1  /* set a flag */
  end

 end

end

if (GotCP) then  /* if we found or loaded a supported code page */
do
 Global.!FilterIn = value('Global.!PC819_PC'||Global.!CodePage)  /* the incoming filter string */
 Global.!FilterOut = value('Global.!PC'||Global.!CodePage||'_PC819')  /* the outgoing filter string */
end
else  /* if we failed to load a valid code page */
do
 Global.!FilterIn = value('Global.!PC819_PCASCII')  /* the incoming filter string */
 Global.!FilterOut = value('Global.!PCASCII_PC819')  /* the outgoing filter string */
 Global.!CodePage = OriginalCP  /* we are running the original code page */
 call ShowLine 'No fully supported code page found;'  /* report */
 call ShowLine 'using original code page '||OriginalCP||' and ASCII/ISO-9959-1 filters.'  /* report */
end

/**************************************************************************/
/* Set the icon dir                                                       */
/**************************************************************************/

Global.!IconDir = Global.!ProgDir||'\Icons'  /* the icon dir */

/**************************************************************************/
/* Look for the location file                                             */
/**************************************************************************/

Global.!LocationFile = Global.!ProgDir||'\location.txt'  /* the user location text file name */

if (\FileCheck(Global.!LocationFile,1)) then  /* if we cannot find a location file */
do
 call Quit  /* quit */
end

/**************************************************************************/
/* Determine the user name and the main dir location                      */
/**************************************************************************/

Global.!User = value('USER',,'OS2ENVIRONMENT')  /* try to get the user name from the OS/2 environment */

if (Global.!User == '') then  /* if we still have no user name */
do
 Global.!User = 'DEFAULT'  /* use this */
end

Global.!MainDir = GetFileEntry(Global.!LocationFile,Global.!User)  /* look for the user's location in the location file */

if (Global.!MainDir == '') then  /* if we have no directory spec */
do
 call AddError 'No user location defined for user "'||Global.!User||'"'  /* report */
 call Quit  /* and quit */
end

if (\DirCheck(Global.!MainDir,1)) then  /* if the directory does not exist */
do
 call Quit  /* quit */
end

RxMlDirs = '*Addresses *Configuration *In *In_Archive *Out *Out_Archive +Temp -Drafts'  /* the RexxMail user folder var names */
call sysfiletree Global.!MainDir||'\*','Dirs.','DO'  /* get the main dir subfolders */

do Index = 1 to Dirs.0  /* take each one */

 RxMlName = GetObjectEA(Dirs.Index,'RXMLDIRNAME')  /* look for the RexxMail name EA */

 if (RxMlName == '') then  /* if there is none */
 do
  RealName = filespec('N',Dirs.Index)  /* the name part */
  RxMlName = translate(RealName,'_',' ')  /* use the name, changing spaces into underscores */
 end

 if (pos(RxMlName,RxMlDirs) == 0) then  /* if we are not looking at one of the original RexxMail folders */
 do
  RxMlName = ''  /* no name (and clear any name set by previous RexxMail builds) */
 end

 call PutObjectEA Dirs.Index,'RXMLDIRNAME',RxMlName  /* set the EA -- clear it if this is not a RexxMail folder */

 if (RxMlName >< '') then  /* if we have a variable part name to use */
 do
  call value 'Global.!'||RxMlName||'Dir',Dirs.Index  /* set a var */
 end

end

do while (RxMlDirs >< '')  /* go on while we have names left */

 parse var RxMlDirs NextDir RxMlDirs  /* get the next one */
 parse var NextDir DirStatus 2 NextDir  /* get the dir status and name */
 GotDir = (symbol('Global.!'||NextDir||'Dir') = 'VAR')  /* if the var exists, the dir exists */

 select  /* do one of the following */

  when (DirStatus == '*') then  /* if the dir must be present */
  do

   if (\GotDir) then  /* if the dir does not exist */
   do
    call Quit  /* quit */
   end

  end

  when (DirStatus == '+') then  /* if the dir is required */
  do

   if (\GotDir) then  /* if the dir does not exist */
   do

    NewDir = Global.!MainDir||'\'||NextDir  /* the new dir name */

    if (sysmkdir(NewDir) >< 0) then  /* if we cannot create it */
    do
     call AddError 'Cannot create directory "'||NewDir||'"'  /* report */
     call Quit  /* and quit */
    end

    call value 'Global.!'||NextDir||'Dir',NewDir  /* set a var */

   end

  end

  when (DirStatus == '-') then  /* if the dir is not required */
  do

   if (\GotDir) then  /* if the dir does not exist */
   do
    call value 'Global.!'||NextDir||'Dir',''  /* set a var to nothing */
   end

  end

  otherwise  /* if none of the above -- which should not occur */
  do
   call AddError 'Invalid directory status specification : '||DirStatus  /* report */
   call Quit  /* quit */
  end

 end

end

/**************************************************************************/
/* Set a few file names in the Configuration dir                          */
/**************************************************************************/

Global.!ControlColl = Global.!ConfigurationDir||'\collect.txt'  /* the collect control file */
Global.!ControlRegi = Global.!ConfigurationDir||'\register.txt'  /* the register control file */
Global.!ControlSend = Global.!ConfigurationDir||'\send.txt'  /* the send control file */
Global.!ControlView = Global.!ConfigurationDir||'\view.txt'  /* the view control file */
Global.!Addresses = Global.!ConfigurationDir||'\addresses.txt'  /* the default addresses vCard file */
Global.!MIMETypes = Global.!ConfigurationDir||'\mimetype.txt'  /* the list of MIME types */
Global.!Signature = Global.!ConfigurationDir||'\signature.txt'  /* the signature text file */
Global.!DeSpam = Global.!ConfigurationDir||'\despam.txt'  /* the despam file */
Global.!Translations = Global.!ConfigurationDir||'\translations.txt'  /* the keyword translations file */
Global.!ActionLog = Global.!ConfigurationDir||'\action.log'  /* the action log name */
Global.!ErrorLog = Global.!ConfigurationDir||'\error.log'  /* the error log name */
Global.!MailLog = Global.!ConfigurationDir||'\mail.log'  /* the mail log name */

/**************************************************************************/
/* Define the various operational settings                                */
/**************************************************************************/

Settings. = ''  /* empty all settings */
Index = 0  /* start at 0 */

/**************************************************************************/
/* Define the text settings options                                       */
/**************************************************************************/

StartIndex = Index + 1  /* the start index for later */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'ADDRESS'  /* defines the default e-mail address of the sender for outgoing messages */
Settings.!Single.Index = 1  /* this is a single word entry */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'ADDRESSTO'  /* address To: insertion folder title to show in attachments folder when editing */
Settings.!Value.Index = 'Address To'  /* use this title as the default value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'ADDRESSCC'  /* address Cc: insertion folder title to show in attachments folder when editing */
Settings.!Value.Index = 'Address Cc'  /* use this title as the default value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'ADDRESSACC'  /* address Acc: insertion folder title to show in attachments folder when editing */
Settings.!Value.Index = 'Address Acc'  /* use this title as the default value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'ADDRESSBCC'  /* address Bcc: insertion folder title to show in attachments folder when editing */
Settings.!Value.Index = 'Address Bcc'  /* use this title as the default value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'ADDTOHEADER'  /* content to add to the header of outgoing messages */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'COLLECTACTION'  /* the action to take after collecting mail */
Settings.!Options.Index = 'OPENFOLDERS WARNICON'  /* the available options */
Settings.!Single.Index = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'DATETIMEBODY'  /* defines how message times are included in the body text of messages */
Settings.!Value.Index = 'ORIGINAL'  /* use this as the default value */
Settings.!Options.Index = 'ORIGINAL SYSTEM UNIVERSAL UTC ORIGINALISO SYSTEMISO UNIVERSALISO UTCISO'  /* the available options */
Settings.!Fuzzy.Index = 1  /* this entry can contain additional formatting */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'DATETIMEHEADER'  /* defines how message times are shown in view and edit files headers */
Settings.!Value.Index = 'ORIGINAL'  /* use this as the default value */
Settings.!Options.Index = 'ORIGINAL SYSTEM UNIVERSAL UTC ORIGINALISO SYSTEMISO UNIVERSALISO UTCISO'  /* the available options */
Settings.!Fuzzy.Index = 1  /* this entry can contain additional formatting */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'DATETIMETITLE'  /* defines how message times are shown in in object titles */
Settings.!Value.Index = 'ORIGINAL'  /* use this as the default value */
Settings.!Options.Index = 'ORIGINAL SYSTEM UNIVERSAL UTC ORIGINALISO SYSTEMISO UNIVERSALISO UTCISO'  /* the available options */
Settings.!Fuzzy.Index = 1  /* this entry can contain additional formatting */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'FORWARDTEXT'  /* defines the text content of forwarded messages */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'HOSTNAME'  /* defines the host name of the user machine */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'MESSAGETOOLBAR'  /* defines a floating toolbar with buttons for open messages and attachment folders */
Settings.!Options.Index = 'CLOSEDRAWER DRAWERTEXT FLOAT HIDECTLS SMALLICONS TEXT VERTICAL YES'  /* the available options */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'NAME'  /* defines the default "real" name of the sender for outgoing messages */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'OBJECTTITLEIN'  /* defines the parts to be included in incoming mail message object titles */
Settings.!Value.Index = 'DATE FROM TO CC SUBJECT'  /* use this as the default value */
Settings.!Options.Index = 'DATE FROM TO CC ACC SUBJECT'  /* the available options */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'OBJECTTITLEOUT'  /* defines the parts to be included in outgoing mail message object titles */
Settings.!Value.Index = 'DATE FROM TO CC ACC BCC SUBJECT'  /* use this as the default value */
Settings.!Options.Index = 'DATE FROM TO CC ACC BCC SUBJECT'  /* the available options */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'OPENFOLDERS'  /* defines the RexxMail folders to open on the desktop when the /open switch is used */
Settings.!Value.Index = 'IN IN_ARCHIVE OUT OUT_ARCHIVE'  /* use this as the default value */
Settings.!Options.Index = 'ADDRESSES CONFIGURATION IN INARCHIVE IN_ARCHIVE MAIN OUT OUTARCHIVE OUT_ARCHIVE DRAFTS'  /* the available options */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'POP3PASSWORD'  /* defines the POP3 password to use */
Settings.!Single.Index = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'POP3SERVER'  /* defines the POP3 server to contact, if any */
Settings.!Single.Index = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'POP3USER'  /* defines the POP3 user name to use */
Settings.!Single.Index = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'REPLYADDRESS'  /* defines the default reply e-mail address of the sender for outgoing messages */
Settings.!Single.Index = 1  /* this is a single word entry */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'REPLYNAME'  /* defines the default reply name of the sender for outgoing messages */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'REPLYTEXT'  /* defines the intro text for reply messages */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNAFTERADDRESSES'  /* defines the command to run after opening the addresses folder */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNAFTERCLOSE'  /* defines the command to run after closing the mail folders */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNAFTERCOLLECT'  /* defines the command to run after collecting mail */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNAFTEREDIT'  /* defines the command to run after editing an outgoing message */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNAFTEROPEN'  /* defines the command to run after opening the RexxMail user folders */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNAFTERSEND'  /* defines the command to run after sending mail */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNAFTERVIEW'  /* defines the command to run after viewing the text part of an incoming or sent message */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNAFTERVIEWRAW'  /* defines the command to run after viewing the text part of an incoming or sent message */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNATTACHIN'  /* defines a command to run on incoming mail attachments right after they have been unpacked from a mesage file */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNATTACHOUT'  /* defines a command to run on outgoing attachments before they are added to the sendable message file */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNBEFOREADDRESSES'  /* defines the command to run before opening the addresses folder */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNBEFORECLOSE'  /* defines the command to run before closing the mail folders */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNBEFORECOLLECT'  /* defines the command to run before collecting mail */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNBEFOREEDIT'  /* defines the command to run before editing an outgoing message */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNBEFOREOPEN'  /* defines the command to run before opening the RexxMail user folders */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNBEFORESEND'  /* defines the command to run before sending mail */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNBEFOREVIEW'  /* defines the command to run before viewing the text part of an incoming or sent message */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNBEFOREVIEWRAW'  /* defines the command to run before viewing the text part of an incoming or sent message */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNCOLLECT'  /* defines a command to run for collecting mail */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNEDIT'  /* defines the command to run on editing outgoing messages */
Settings.!Value.Index = 'E.EXE "#N"'  /* use the OS/2 system editor as the default value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNRECEIVED'  /* defines a command to run on received message files */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNSEND'  /* defines a command to run on sendable message files */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNVIEW'  /* defines the command to run for viewing the text part of an incoming or sent message */
Settings.!Value.Index = 'E.EXE "#N"'  /* use the OS/2 system editor as the default value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'RUNVIEWRAW'  /* defines the command to run for viewing a raw message */
Settings.!Value.Index = 'E.EXE "#N"'  /* use the OS/2 system editor as the default value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SHADOWREGISTER'  /* defines if and where shadows of incoming messages must be created when redirecting after registering */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SHADOWSEND'  /* defines if and where shadows of outgoing messages must be created when redirecting after sending */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SHADOWVIEW'  /* defines if and where shadows of incoming messages must be created when redirecting after viewing */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SORTADDRESS'  /* address type to use for sorting */
Settings.!Value.Index = 'FULL'  /* use this as the default value */
Settings.!Single.Index = 1  /* this is a single word entry  */
Settings.!Options.Index = 'ADDRESS FULL NAME'  /* the available options */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SMTPPASSWORD'  /* defines the SMTP password to use */
Settings.!Single.Index = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SMTPSERVER'  /* defines the SMTP server to use */
Settings.!Single.Index = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SMTPUSER'  /* defines the SMTP user name to use */
Settings.!Single.Index = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'STATIONERY'  /* defines the name of the default stationery folder */
Settings.!Value.Index = 'BLUE_WHITE'  /* use this as the default value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'TIMESERVERS'  /* the names of time servers to contact for constructing a time zone string */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'TIMEZONE'  /* a static time zone setting to use instead of contacting a time server */
Settings.!Single.Index = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'TITLEADDRESS'  /* Address type to use for message title */
Settings.!Value.Index = 'FULL'  /* use this as the default value */
Settings.!Single.Index = 1  /* this is a single word entry  */
Settings.!Options.Index = 'ADDRESS FULL NAME'  /* the available options */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'VERSION'  /* The build number of the last RexxMail update */
Settings.!Value.Index = 0  /* use this as the default value */
Settings.!Single.Index = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'VIEWHEADERIN'  /* defines which message header entries of incoming messages are to be included in the view file */
Settings.!Value.Index = 'DATE FROM TO CC SUBJECT'  /* use this as the default value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'VIEWHEADEROUT'  /* defines which message header entries of sent messages are to be included in the view file */
Settings.!Value.Index = 'DATE FROM TO CC ACC BCC SUBJECT'  /* use this as the default value */

do NewIndex = StartIndex to Index  /* take each of the above settings */
 Settings.!Type.NewIndex = 'T'  /* and make it a text type */
end

/**************************************************************************/
/* Define the signal settings options                                     */
/**************************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SIGNALERROR'  /* defines the error signal */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SIGNALRECEIVED'  /* defines the signal for incoming mail */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SIGNALSENT'  /* defines the signal for sent mail */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SIGNALSYSTEM'  /* defines the system message signal */

do NewIndex = StartIndex to Index  /* take each of the above settings */
 Settings.!Type.NewIndex = 'S'  /* and make it a signal type */
end

/**************************************************************************/
/* Define the numerical settings options                                  */
/**************************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'BODYLINELENGTH'  /* defines the length of body text lines (0 = unwrapped) */
Settings.!Min.Index = 1  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'HEADERLINELENGTH'  /* defines the length of text lines in View and Edit header lines */
Settings.!Min.Index = 1  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'LOGACTIONLINES'  /* defines the number of action log lines we want to keep */
Settings.!Value.Index = ''  /* use this as the default value (grow unchecked) */
Settings.!Min.Index = 0  /* use this as the minimum value (no logging) */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'LOGERRORLINES'  /* defines the number of error log lines we want to keep */
Settings.!Value.Index = ''  /* use this as the default value (grow unchecked) */
Settings.!Min.Index = 0  /* use this as the minimum value (no logging) */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'LOGMAILLINES'  /* defines the number of mail log lines we want to keep */
Settings.!Value.Index = ''  /* use this as the default value (grow unchecked) */
Settings.!Min.Index = 0  /* use this as the minimum value (no logging) */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'MAXADDRESSES'  /* defines the maximum number of addresses to show in a recpients list in incoming messages */
Settings.!Min.Index = 0  /* use this as the minimum value */
              
Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'MAXCOLLECTNUMBER'  /* defines the maximum number of messages to collect */
Settings.!Value.Index = ''  /* use this as the default value: collect all */
Settings.!Min.Index = 0  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'MAXCOLLECTSIZE'  /* defines the maximum number of bytes of messages to collect automatically from the POP3 server */
Settings.!Value.Index = ''  /* use this as the default value */
Settings.!Min.Index = 0  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'MAXCOLLECTSIZETOTAL'  /* defines the maximum total number of bytes of messages to collect automatically from the POP3 server */
Settings.!Value.Index = ''  /* use this as the default value */
Settings.!Min.Index = 0  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'POP3ATTEMPTS'  /* defines the maximum number of attempts to connect to the POP3 server */
Settings.!Value.Index = 1  /* use this as the default value */
Settings.!Min.Index = 0  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'REPLYLINELENGTH'  /* defines the maximum length of quoted reply message lines */
Settings.!Min.Index = 1  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SMTPATTEMPTS'  /* defines the maximum number of attempts to connect to the SMTP server */
Settings.!Value.Index = 1  /* use this as the default value */
Settings.!Min.Index = 0  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'TIMEATTEMPTS'  /* defines the maximum number of attempts to connect to the time server */
Settings.!Value.Index = 1  /* use this as the default value */
Settings.!Min.Index = 0  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'TITLELINELENGTH'  /* defines the maximum length of message object title lines */
Settings.!Value.Index = 42  /* use this as the default value */
Settings.!Min.Index = 24  /* use this as the minimum value */
Settings.!Max.Index = 240  /* use this as the maximum value */

do NewIndex = StartIndex to Index  /* take each of the above settings */
 Settings.!Type.NewIndex = 'N'  /* and make it a numerical type */
end

/**************************************************************************/
/* Define the Boolean settings options                                    */
/**************************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'CLOSEATTACHAFTEREDIT'  /* CloseAttachAfterEdit = YES: close outgoing attachments folders after editing messages */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'CLOSEATTACHAFTERVIEW'  /* CloseAttachAfterView = YES: close incoming attachments folders after viewing messages */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'EDITCHECK'  /* EditCheck = YES: outgoing messages are O.K. to send only if changed during editing */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'ERRORMAIL'  /* ErrorMail = YES: send error messages to the user as mail */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'HTMLKEEP'  /* HTMLKEEP = YES: keep HTML files in attachments folder after extracting text content */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'HTMLLINES'  /* HTMLLINES = YES: insert separator lines into text extracted from HTML content */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'HTMLSAFE'  /* HTMLSAFE = YES: rewrite HTML content to prevent unwanted net access etc. */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'HTMLURLLIST'  /* HTMLURLLIST = YES: rewrite text extracted from HTML content with URLs at the end of the text content */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'OPENATTACHBEFOREEDIT'  /* OpenAttachBeforeEdit = YES: open outgoing attachments folders when editing messages */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'OPENATTACHBEFOREVIEW'  /* OpenAttachBeforeView = YES: open incoming attachments folders when viewing messages */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'POP3INTERACTIVE'  /* POP3Interactive = YES: prompt the user for POP3 message retrieval and deletion */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'POP3KEEPMESSAGES'  /* POP3KeepMessages = YES: do not delete messages from the POP3 server after retrieval */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'POP3SORTBYSIZE'  /* POP3SortBySize = YES: collect messages from the POP3 server in ascending order of size */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SENDASCII'  /* SendASCII = YES: convert outgoing messages to ASCII, word-wrapped at 76 characters */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SHOWPROGRESS'  /* ShowProgress = YES: show a byte counter and progress bar when sending or collecting mail */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SOCKETMONITOR'  /* SocketMonitor = YES: show socket traffic */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'SILENT'  /* Silent = YES: suppress error signals and messages */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'TITLEKEYWORDS'  /* TitleKeywords = YES: include keywords in message object title lines */

Index = Index + 1  /* up the counter */
Settings.!Name.Index = 'USECURRENTDIR'  /* UseCurrentDir = YES: create new/reply/forward messages in the current or old message dir */

do NewIndex = StartIndex to Index  /* take each of the above settings */
 Settings.!Type.NewIndex = 'B'  /* make it a Boolean type */
 Settings.!Value.NewIndex = 0  /* and make it FALSE */
end

Settings.!Name.0 = Index  /* the total number of text, numerical, and Boolean settings */

/**************************************************************************/
/* Define the various command switches                                    */
/**************************************************************************/

Commands. = ''  /* empty all commands */
Index = 0  /* start at 0 */

/**************************************************************************/
/* Define the valid command entries that require a mail file argument */
/**************************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'ADDADDRESSESTOOPEN'  /* copy (address template) mail file to addresses folder(s) of open RexxMail messages */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'CLOSEATTACH'  /* close an attachments dir */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'EDIT'  /* edit an outgoing message */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'FORWARD'  /* forward a message */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'FORWARDOPEN'  /* forward a message and open it on the desktop */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'OPENATTACH'  /* open an attachments dir */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'VIEWRAW'  /* view the raw contents of an incoming message */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REPLY'  /* reply to "Reply-To:" or "From:" address */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REPLYOPEN'  /* reply to "Reply-To:" or "From:" address and open */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REPLYTOALL'  /* reply to all addresses */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REPLYTOALLOPEN'  /* reply to all addresses and open */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REPLYTOALLASLIST'  /* reply to all addresses through a local list */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REPLYTOALLASLISTOPEN'  /* reply to all addresses through a local list and open */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REPLYTOLIST'  /* reply to list address */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REPLYTOLISTOPEN'  /* reply to list address and open */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REPLYTORECIPIENTS'  /* reply to all recipients */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REPLYTORECIPIENTSOPEN'  /* reply to all recipients and open */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REPLYTORECIPIENTSASLIST'  /* reply to all recipients through a local list */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REPLYTORECIPIENTSASLISTOPEN'  /* reply to all recipients through a local list and open */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'RESEND'  /* resend a message */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SEND'  /* send a message */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SENDBARE'  /* send a message "bare" */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETASCII'  /* set a message to "ASCII" */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETASCIIQP'  /* toggle a message between ASCII and Q-P */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETMESSAGEICON'  /* set the current message icon */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETNEW'  /* set a message to "new" */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETNOTOKTOSEND'  /* set a message to "not OK to send" */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETOKNOTOKTOSEND'  /* toggle a message between "OK to send" and  "not OK to send" */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETOKTOSEND'  /* set a message to "OK to send" */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETQP'  /* set a message to "Q-P" */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETRECEIVED'  /* set a message to "received" */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETSENT'  /* set a message to "sent" */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETSTATIONERY'  /* set message stationery */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETTITLE'  /* set the title of a message file */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETVIEWED'  /* set a message to "viewed" */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'VIEW'  /* view an incomping message */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.!Type.NewIndex = 'C'  /* set the type to indicate that a mail file spec is required */
end

/**************************************************************************/
/* Define the entries that require a parameter and filespec argument      */
/**************************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REDIRECTADD'  /* adds a redirection line */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'GETHEADER'  /* sends one or more lines from the header to the standard output */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'FORWARDTO'  /* forward a message to specified address(es) */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'FORWARDTOOPEN'  /* forward a message to specified address(es) and open it on the desktop */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.!Type.NewIndex = 'P'  /* set the type to indicate that a parameter and mail file spec are required */
end

/**************************************************************************/
/* Define the valid command entries that require a filespec argument      */
/**************************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'ATTACHTOALLINDRAFTS'  /* attach file(s) to all messages in the Drafts dir */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'ATTACHTOALLINOUT'  /* attach file(s) to all messages in the Out dir */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'ATTACHTOOPEN'  /* attach file(s) to all messages open on the desktop */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REGISTER'  /* register an incoming message */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.!Type.NewIndex = 'R'  /* set the type to indicate that a (non-RexxMail) file spec is required */
end

/**************************************************************************/
/* Define the valid command entries that take no argument                 */
/**************************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'CLEANUPTEMP'  /* clean up the temp folder */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'CLOSE'  /* close the user folders */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'COLLECT'  /* collect mail from a POP3 server */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'OPEN'  /* open the user folders */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'RECURSE'  /* recurse all commands into subdirectories */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'REFERENCE'  /* open the RexxMail Reference Guide */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SENDREADY'  /* send all messages in the Out dir marked "Ready to send" */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'TRIMLOGFILES'  /* trim the log files */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'TUTORIAL'  /* open the RexxMail Tutorial */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'WARNICON'  /* create a warning icon */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'WARNICONSHOW'  /* show an existing warning icon */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'WARNICONHIDE'  /* hide an existing warning icon */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'WARNICONDELETE'  /* delete an existing warning icon */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.!Type.NewIndex = 'N'  /* set the type to indicate that no argument is permitted */
end

/**************************************************************************/
/* Define the valid command entries with an optional file argument        */
/**************************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'ADDRESSIMPORT'  /* import addresses */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.!Type.NewIndex = 'A'  /* set the type to indicate that a file spec is acceptable */
end

/**************************************************************************/
/* Define the valid command entries with an optional folder argument      */
/**************************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'SETPROGRAMOBJECTICONS'  /* set the object icons of the RexxMail program objects */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.!Type.NewIndex = 'F'  /* set the type to indicate that a folder is acceptable */
end

/**************************************************************************/
/* Define the valid command entries that accept single files only         */
/**************************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'ADDRESSEXPORT'  /* export address data to an addresses text file */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.!Type.NewIndex = 'S'  /* set the type to indicate that any argument must point to a single file */
end

/**************************************************************************/
/* Define the valid command entries that take a non-filespec argument */
/**************************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'HELP'  /* display a short help message */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'NEWMESSAGE'  /* create a new outgoing message file */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'NEWMESSAGEOPEN'  /* create a new outgoing message file and open it */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'NEWMESSAGESEND'  /* create a new outgoing message file and send it */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'TOOLBARCREATE'  /* create an almost empty toolbar */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'TOOLBARDELETE'  /* delete a toolbar */

Index = Index + 1  /* up the counter */
Commands.!Name.Index = 'TOOLBAROPEN'  /* open a toolbar */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.!Type.NewIndex = 'O'  /* set the type to indicate that any argument need not be a file spec */
end

Commands.!Name.0 = Index  /* the total number of command switch options */

/**************************************************************************/
/* Get the settings from the settings file                                */
/**************************************************************************/

SettingsFile = Global.!ConfigurationDir||'\settings.txt'  /* the settings text file */

if (\GetDefinitions(SettingsFile,'!Settings',1,1)) then  /* if we cannot get the settings file contents into a named array (bare values allowed) */
do
 call Quit  /* quit */
end

/**************************************************************************/
/* Get the command-line arguments                                         */
/**************************************************************************/

parse arg Commands  /* get the command-line arguments */
Commands = translate(Commands,' ',d2c(9))  /* convert any TAB characters to spaces */
Commands = strip(Commands,'B',' ')  /* remove any excess whitespace */
Global.!StoreCommands = Commands  /* save the command-line arguments for later */

if (left(Commands,1) == '/') then  /* if the arguments start with a switch delimiter */
do
 parse var Commands '/' Commands  /* get rid of the first switch delimiter */
end
else  /* if the arguments do not start with a switch delimiter */
do

 parse var Commands PreCommands '/' Commands  /* get the bit before the first switch delimiter and get rid of the delimiter */
 PreCommands = strip(PreCommands,'B',' ')  /* get rid of excess space */

 if (PreCommands >< '') then  /* if we have something */
 do
  call AddError 'Missing command switch before "'||PreCommands||'"'  /* report an error */
 end

end

NewCommands. = ''  /* start with nothing */
CommCount = 0  /* start at 0 */

do while (Commands >< '')  /* as long as we have commands left */

 parse var Commands Command '/' Commands  /* get the next command sequence */
 parse var Command Switch Argument  /* get any bits separated by the first space, if any */
 parse var Switch Switch '=' Rest  /* look for an equal sign in the switch */

 if (Rest >< '') then  /* if we have this, there was an equal sign, and this bit belangs to the argument */
 do
  Argument = strip(Rest)||' '||Argument  /* restore it with the blank space */
 end

 Switch = translate(strip(Switch))  /* remove excess space and make the switch upper case */
 Argument = strip(Argument)  /* remove excess space */
 Index = 0  /* start at 0 */

 do until ((Switch == Settings.!Name.Index) | (Index > Settings.!Name.0))  /* go on until we find the switch in the settings or run out of options */
  Index = Index + 1  /* up the index */
 end

 if (Index <= Settings.!Name.0) then  /* if we found a match, we have a valid setting name */
 do
  call value 'Global.!Settings.'||Switch,Argument  /* set the value */
 end
 else  /* if we did not find a match, see if we have a valid command */
 do

  Index = 0  /* start at 0 */

  do until ((Switch == Commands.!Name.Index) | (Index > Commands.!Name.0))  /* go on until we find the switch in the commands or run out of options */
   Index = Index + 1  /* up the index */
  end

  if (Index <= Commands.!Name.0) then  /* if we found a match */
  do

   if (Switch == 'HELP') then  /* if the switch is this one */
   do

    call ShowLine 'RexxUtil level = '||RexxUtilLevel  /* report */
    call ShowLine 'User name      = '||Global.!User  /* report */

    select  /* do one of the following */

     when (translate(Argument) == 'COMMANDS') then  /* if we have this */
     do

      call sysstemsort 'Commands.!Name.'  /* sort the defined commands */
      call ShowLine 'RexxMail commands:'  /* empty line */
      call ShowLine  /* empty line */

      do Index = 1 to Commands.!Name.0  /* take each defined command */
       call lineout 'STDOUT:',Commands.!Name.Index  /* info */
      end

     end

     when (translate(Argument) == 'SETTINGS') then  /* if we have this */
     do

      call sysstemsort 'Settings.!Name.'  /* sort the defined settings */
      call ShowLine 'RexxMail settings:'  /* empty line */
      call ShowLine  /* empty line */

      do Index = 1 to Settings.!Name.0  /* take each defined command */
       call lineout 'STDOUT:',Settings.!Name.Index  /* info */
      end

     end

     otherwise  /* if none of the above */
     do
      call ShowLine 'Usage: RexxMail /<switch> [=] [<argument>] [...]'  /* show a minimal help message */
      call ShowLine  /* empty line */
      call ShowLine 'The program must be called with at least one command switch, which may be'  /* info */
      call ShowLine 'followed by one or more arguments and/or any combination of additional command'  /* info */
      call ShowLine 'and/or settings switches and arguments.'  /* info */
      call ShowLine  /* empty line */
      call ShowLine 'Switches and arguments must be separated by whitespace and/or a single "=".'  /* info */
      call ShowLine 'Switch/argument combinations must be separated by whitespace.'  /* info */
      call ShowLine  /* empty line */
      call ShowLine 'For a list of RexxMail commands, type:'  /* more info */
      call ShowLine '       RexxMail /help commands'  /* info */
      call ShowLine 'For a list of RexxMail settings, type:'  /* more info */
      call ShowLine '       RexxMail /help settings'  /* info */
      call ShowLine 'For more information, see the RexxMail Reference Guide and RexxMail Tutorial:'  /* info */
      call ShowLine '       RexxMail /reference'  /* info */
      call ShowLine '       RexxMail /tutorial'  /* info */
     end

    end

    Global.!Errors = ''  /* make sure we skip any command-line errors */
    call Quit  /* and quit */

   end
   else  /* if the switch is something else */
   do
    CommCount = CommCount + 1  /* up the command counter */
    NewCommands.CommCount.!Switch = Switch  /* store the switch */
    NewCommands.CommCount.!Argument = Argument  /* store the argument */
    NewCommands.CommCount.!FullArgument = Argument  /* store the argument a second time, for logging */
    NewCommands.CommCount.!Type = Commands.!Type.Index  /* store the command argument type */
   end

  end
  else  /* if we did not find a match */
  do
   call AddError 'Invalid switch: '||Switch  /* report */
  end

 end

end

/**************************************************************************/
/* Check the settings values                                              */
/**************************************************************************/

do Index = 1 to Settings.!Name.0  /* take each possible setting */

 BaseName = Settings.!Name.Index  /* the base variable name */
 FullName = 'Global.!Settings.'||BaseName  /* the full variable name */

 select  /* do one of the following */

  when (symbol(FullName) >< 'VAR') then  /* if this one was not set through the settings file or the command line */
  do
   call value FullName,Settings.!Value.Index  /* set the default value (which may be empty) */
  end

  when (Settings.!Type.Index == 'T') then  /* if it is a text setting */
  do
   call value FullName,CheckValText(BaseName,Settings.!Options.Index,Settings.!Single.Index,Settings.!Fuzzy.Index)  /* check and set the value */
  end

  when (Settings.!Type.Index == 'N') then  /* if it is a numerical setting */
  do
   call value FullName,CheckValNum(BaseName,Settings.!Value.Index,Settings.!Min.Index,Settings.!Max.Index)  /* check and set the value */
  end

  when (Settings.!Type.Index == 'S') then  /* if it is a signal setting */
  do
   call value FullName,CheckValSignal(BaseName)  /* check and set the value */
  end

  when (Settings.!Type.Index == 'B') then  /* if it is a Boolean setting */
  do
   call value FullName,CheckValBool(BaseName)  /* check and set the value */
  end

  otherwise  /* if none of the above (which should not occur) */
  do
   call AddError 'Invalid setting type for '||BaseName  /* report a coding error */
  end

 end

end

Global.!Settings.Name = strip(Global.!Settings.Name,'B','"')  /* remove any " around the mail name */
Global.!Settings.Address = strip(Global.!Settings.Address,'L','<')  /* remove any leading < from the mail address */
Global.!Settings.Address = strip(Global.!Settings.Address,'T','>')  /* remove any trailing > from the mail address */

/**************************************************************************/
/* See if we need to update the configuration file                        */
/**************************************************************************/

if (Global.!Settings.Version == '') then  /* if we found no version number */
do
 Global.!Settings.Version = 0  /* assume 0 */
end

if (Global.!Build >< Global.!Settings.Version) then  /* if we are running a different build of RexxMail */
do

 if (\UpdateSettings(SettingsFile,Global.!ProgDir||'\settupda.txt')) then  /* if we cannot update the configuration file */
 do
  call Adderror 'Cannot update the configuration file'  /* report */
 end

end

/**************************************************************************/
/* Check the command-line commands we collected earlier                   */
/**************************************************************************/

Recurse = ''  /* nothing means recurse = off */

if (CommCount == 0) then  /* if we have no valid commands at all */
do
 call AddError 'Missing command'  /* report */
end
else  /* if we have one or more commands */
do Index = 1 to CommCount  /* run through the commands we collected */

 NewCommands.Index.!Files. = ''  /* start with nothing */
 FileCount = 0  /* start a file counter at 0 */

 if (NewCommands.Index.!Argument == '') then  /* if we have no argument */
 do

  select  /* do one of the following */

   when (NewCommands.Index.!Switch == 'ADDRESSEXPORT') then  /* if the switch is this special case */
   do
    NewCommands.Index.!Argument = Global.!Addresses  /* use the standard address text file */
   end

   when (NewCommands.Index.!Switch == 'RECURSE') then  /* if the switch is this special case */
   do
    NewCommands.Index.!Switch = ''  /* don't run the command later */
    Recurse = 'S'  /* use this in the sysfiletree command later on when we go looking for files */
   end

   when (NewCommands.Index.!Switch == 'SENDREADY') then  /* if the switch is this special case */
   do
    NewCommands.Index.!Argument = '"'||Global.!OutDir||'\*"'  /* use any files that may be in the Out dir */
   end

   when (pos(NewCommands.Index.!Type,'CPR') > 0) then  /* if the switch is a command that requires an argument */
   do

    call AddError NewCommands.Index.!Switch||' requires an argument'  /* start a message */

    if (NewCommands.Inde.!Type == 'CP') then  /* if the switch is a command that requires a mail file spec */
    do
     call AddError 'If you double-clicked a program object, drop a mail file on it instead'  /* finish the message */
    end

   end

   otherwise  /* if none of the above apply */
   do
    nop  /* nothing to do */
   end

  end

 end
 else  /* if we do have an argument */
 do

  if (NewCommands.Index.!Type == 'N') then  /* if the switch is a command that cannot take an argument */
  do
   call AddError NewCommands.Index.!Switch||' cannot take an argument'  /* report */
   NewCommands.Index.!Argument = ''  /* skip the argument */
  end

 end

 if (NewCommands.Index.!Argument >< '') then  /* if we do have an argument now */
 do

  select  /* do one of the following */

   when (NewCommands.Index.!Type == 'O') then  /* if the switch is the "other" type */
   do
    nop  /* nothing */
   end

   when (NewCommands.Index.!Type == 'F') then  /* if the switch is the "optional folder" type */
   do

    NewCommands.Index.!Argument = strip(NewCommands.Index.!Argument,'B','"')  /* remove any double quotation marks */
    NewCommands.Index.!Argument = strip(NewCommands.Index.!Argument,'T','\')  /* remove any trailing backslash */

    if (NewCommands.Index.!Argument >< '') then  /* if we still have something */
    do

     if (directory(NewCommands.Index.!Argument) >< '') then  /* if the folder exists */
     do
      NewCommands.Index.!Argument = directory()  /* use it */
      call directory Global.!CurDir  /* change back to the original dir */
     end
     else  /* if it does not */
     do
      call AddError '"'||NewCommands.Index.!Argument||'" is not an existing folder'  /* report */
     end

    end

   end

   otherwise  /* if none of the above, assume the argument is a file spec */
   do

    if (NewCommands.Index.!Type == 'P') then  /* if the switch requires a parameter */
    do

     parse var NewCommands.Index.!Argument NewCommands.Index.!Parameter NewCommands.Index.!Argument  /* extract it */
     NewCommands.Index.!Parameter = strip(NewCommands.Index.!Parameter)  /* remove excess whitespace */

     if ((left(NewCommands.Index.!Parameter,1) == '"') & (right(NewCommands.Index.!Parameter,1) >< '"')) then  /* if it starts with a " but does not end with a " */
     do
      parse var NewCommands.Index.!Argument Rest '"' NewCommands.Index.!Argument  /* get the rest of it */
      NewCommands.Index.!Parameter = NewCommands.Index.!Parameter||' '||Rest  /* restore it */
     end

     NewCommands.Index.!Parameter = strip(NewCommands.Index.!Parameter,'B','"')  /* remove any double quotation marks */

    end

    if (NewCommands.Index.!Argument == '') then  /* if we have nothing (left) */
    do
     call AddError 'Missing file specification'  /* report */
    end
    else  /* if we have something left, it must be the file spec */
    do

     do while (NewCommands.Index.!Argument >< '')  /* as long as we have something left */

      NewCommands.Index.!Argument = strip(NewCommands.Index.!Argument)  /* get rid of excess blanks */

      if (left(NewCommands.Index.!Argument,1) == '"') then  /* if the specs start with a double quotation mark */
      do
       NewCommands.Index.!Argument = strip(NewCommands.Index.!Argument,'L','"')  /* remove all leading quotation marks (in case we have double pairs) */
       parse var NewCommands.Index.!Argument FileLoc '"' NewCommands.Index.!Argument  /* get the bit before the next double quotation marks */
       NewCommands.Index.!Argument = strip(NewCommands.Index.!Argument,'L','"')  /* remove all leading quotation marks (in case we had double pairs) */
      end
      else  /* if the specs do not start with a double quotation mark */
      do
       parse var NewCommands.Index.!Argument FileLoc NewCommands.Index.!Argument  /* get the next space-delimited bit */
      end

      if (FileLoc >< '') then  /* if we have something */
      do

       if (NewCommands.Index.!Type == 'S') then  /* if the switch accepts a single (output) file spec only (which need not exist) */
       do

        if (verify(FileLoc,'?*','M') > 0) then  /* if the filespec contains wildcards */
        do
         call AddError NewCommands.Index.!Switch||' does not accept wildcards'  /* report an error */
        end
        else  /* if the filespec does not contain wildcards */
        do
         FileCount = FileCount + 1  /* up the file counter */
         NewCommands.Index.!Files.FileCount = FileLoc  /* store the file name */
        end

       end
       else  /* if the switch is another type, taking one or more existing files */
       do

        call sysfiletree FileLoc,'LocFiles.','FO'||Recurse  /* look for files */

        if (LocFiles.0 == 0) then  /* if we find nothing */
        do

         if (NewCommands.Index.!Switch >< 'SENDREADY') then  /* if it is not this special case */
         do
          call AddError 'Cannot find file "'||FileLoc||'"'  /* report an error */
         end

        end
        else  /* if we find files */
        do FileLocCount = 1 to LocFiles.0  /* take each of the files found */
         FileCount = FileCount + 1  /* up the file counter */
         NewCommands.Index.!Files.FileCount = LocFiles.FileLocCount  /* store the file name */
        end

       end

      end

     end

     if (FileCount == 0) then  /* if we found no files at all */
     do

      if (NewCommands.Index.!Switch >< 'SENDREADY') then  /* unless the switch is this special case */
      do
       call AddError NewCommands.Index.!Switch||': no files found'  /* report an error */
      end

     end
     else  /* if we did find files */
     do

      if ((FileCount > 1) & (NewCommands.Index.!Type == 'S')) then  /* if we found more than one file and the command is one that only takes a single file spec */
      do
       call AddError NewCommands.Index.!Switch||' cannot process multiple files'  /* report an error */
      end

     end

    end

   end

  end

 end

 NewCommands.Index.!Files.0 = FileCount  /* store the file counter */
 Global.!FilesToDo = NewCommands.Index.!Files.0  /* copy this to keep a running tab later and to signal in subroutines (SMTPSendMessage) */

end

/**************************************************************************/
/* If we have collected any errors, abort further processing              */
/**************************************************************************/

if (Global.!Errors >< '') then  /* if we have errors */
do
 call Quit  /* report and quit */
end

/**************************************************************************/
/* Run the command-line commands                                          */
/**************************************************************************/

do Index = 1 to CommCount  /* run through the commands we collected */

 call LogAction 'Command = '||NewCommands.Index.!Switch||' '||NewCommands.Index.!FullArgument,1  /* report, quietly */

 select  /* do one of the following */

  when (NewCommands.Index.!Switch == '') then  /* if the switch is empty (e.g. it was "recurse") */
  do
   nop  /* nothing */
  end

  when (NewCommands.Index.!Switch == 'ADDADDRESSESTOOPEN') then  /* if the switch is "addaddressestoopen" */
  do
   call ProcessFiles 'AddressAddToOpen'  /* call the add addresses procedure */
  end

  when (NewCommands.Index.!Switch == 'GETHEADER') then  /* if the switch is "getheader" */
  do
   call ProcessFiles 'GetHeader "'||NewCommands.Index.!Parameter||'"'  /* call the GetHeader procedure with the parameter */
  end

  when (NewCommands.Index.!Switch == 'REDIRECTADD') then  /* if the switch is "redirectadd" */
  do
   call ProcessFiles 'RedirectAdd "'||NewCommands.Index.!Switch||' '||NewCommands.Index.!Parameter||'"'  /* call the RedirectAdd procedure with the parameters */
  end

  when (NewCommands.Index.!Switch == 'ADDRESSEXPORT') then  /* if the switch is "addressexport" */
  do
   call AddressExport NewCommands.Index.!Files.1  /* call the mail address book export procedure with the output file name */
  end

  when (left(NewCommands.Index.!Switch,8) == 'ATTACHTO') then  /* if the switch starts with "AttachTo" */
  do
   call ProcessFiles 'CopyAttachment "'||NewCommands.Index.!Switch||'"',1  /* call the attachment copy procedure with the switch as argument and show progress */
  end

  when (NewCommands.Index.!Switch == 'CLEANUPTEMP') then  /* if the switch is "cleanuptemp" */
  do
   call CleanUpTemp  /* clean up temporary files and folders */
  end

  when (NewCommands.Index.!Switch == 'CLOSE') then  /* if the switch is "close" */
  do
   call FoldersClose  /* call the close folders procedure */
  end

  when (NewCommands.Index.!Switch == 'CLOSEATTACH') then  /* if the switch is "closeattach" */
  do
   call ProcessFiles 'AttDirShut'  /* call the shut attachments folder procedure */
  end

  when (NewCommands.Index.!Switch == 'COLLECT') then  /* if the switch is "collect" */
  do

   if (CollectMail() > 0) then  /* if we collected any messages from the POP3 server */
   do

    if (wordpos('WARNICON',translate(Global.!Settings.CollectAction)) > 0) then  /* if we want a warning icon */
    do
     call WarnIcon  /* create a mail warning icon, or show the existing object */
    end

    if (wordpos('OPENFOLDERS',translate(Global.!Settings.CollectAction)) > 0) then  /* if we want the user folders opened */
    do
     call FoldersOpen  /* open the mail folders on the desktop */
    end

   end

  end

  when (NewCommands.Index.!Switch == 'EDIT') then  /* if the switch is  "edit" */
  do
   call ProcessFiles 'EditMessage'  /* edit the message (or view it if it was sent before) */
  end

  when (left(NewCommands.Index.!Switch,9) == 'FORWARDTO') then  /* if the switch starts with "forwardto" */
  do
   call ProcessFiles 'MakeForwardReplyMessage "'||NewCommands.Index.!Switch||' '||NewCommands.Index.!Parameter||'"'  /* call the forward/reply procedure with the switch and parameter as argument */
  end

  when (left(NewCommands.Index.!Switch,7) == 'FORWARD') then  /* if the switch starts with "forward" */
  do
   call ProcessFiles 'MakeForwardReplyMessage "'||NewCommands.Index.!Switch||'"'  /* call the forward/reply procedure with the switch as argument */
  end

  when (NewCommands.Index.!Switch == 'ADDRESSIMPORT') then  /* if the switch is "addressimport" */
  do

   if (NewCommands.Index.!Files.0 == 0) then  /* if we have no files to process */
   do
    call RunCommand Global.!Settings.RunBeforeAddresses  /* see if we have a command to run */
    call LogAction 'Opening Addresses folder',1  /* report, quietly */
    call sysopenobject Global.!AddressesDir,0,1  /* open the Addresses folder */
    call RunCommand Global.!Settings.RunAfterAddresses  /* see if we have a command to run */
   end
   else  /* if we have an argument */
   do
    call ProcessFiles 'AddressImport'  /* call the address import procedure */
   end

  end

  when (left(NewCommands.Index.!Switch,10) == 'NEWMESSAGE') then  /* if the switch starts with "newmessage" */
  do
   call MakeNewMessage NewCommands.Index.!Argument,NewCommands.Index.!Switch  /* make a new message, passing the original argument and the switch */
  end

  when (NewCommands.Index.!Switch == 'OPEN') then  /* if the switch is "open" */
  do
   call FoldersOpen  /* call the open mail folders procedure */
  end

  when (NewCommands.Index.!Switch == 'OPENATTACH') then  /* if the switch is "openattach" */
  do
   call ProcessFiles 'AttDirShow'  /* call the show attachments folder procedure */
  end

  when (NewCommands.Index.!Switch == 'VIEWRAW') then  /* if the switch is "viewraw" */
  do
   call ProcessFiles 'ViewRawMessage'  /* call the raw message viewing procedure */
  end

  when (NewCommands.Index.!Switch == 'REGISTER') then  /* if the switch is "register" */
  do
   call ProcessFiles 'RegisterMessage',1  /* call the mail register procedure and show progress */
  end

  when (NewCommands.Index.!Switch == 'REFERENCE') then  /* if the switch is "reference" */
  do

   if (\sysopenobject(Global.!ProgDir||'\RexxMail_Reference_Guide.INF',0,1)) then  /* if we cannot open the RexxMail Reference Guide on the Desktop */
   do
    call AddError 'Cannot open the RexxMail Reference Guide : "'||Global.!ProgDir||'\RexxMail Reference Guide.INF"'  /* report, not fatal */
   end

  end

  when (left(NewCommands.Index.!Switch,5) == 'REPLY') then  /* if the switch starts with "reply" */
  do
   call ProcessFiles 'MakeForwardReplyMessage "'||NewCommands.Index.!Switch||'"'  /* call the forward/reply procedure with the switch as argument */
  end

  when (NewCommands.Index.!Switch == 'RESEND') then  /* if the switch is "resend" */
  do
   call ProcessFiles 'ResendMessage'  /* call the resend procedure */
  end

  when (left(NewCommands.Index.!Switch,4) == 'SEND') then  /* if the switch starts with "send" */
  do
   call ProcessFiles 'SendMessage '||(NewCommands.Index.!Switch = 'SENDBARE')  /* send the message, setting a "bare" flag if necessary */
  end

  when (NewCommands.Index.!Switch == 'SETPROGRAMOBJECTICONS') then  /* if the switch is "SetProgramObjectIcons" */
  do
   call SetProgramObjectIcons NewCommands.Index.!Argument  /* set the program object icons from the argument folder, if any */
  end

  when (NewCommands.Index.!Switch == 'SETSTATIONERY') then  /* if the switch is "SetStationery" */
  do
   call ProcessFiles 'SetStationery',1  /* call the stationery set procedure and show progress */
  end

  when (NewCommands.Index.!Switch == 'SETTITLE') then  /* if the switch is "settitle" */
  do
   call ProcessFiles 'SetTitle',1  /* call the title set procedure and show progress */
  end

  when (left(NewCommands.Index.!Switch,3) == 'SET') then  /* if the switch starts with "set" */
  do
   call ProcessFiles 'SetMessageType "'||NewCommands.Index.!Switch||'"',1  /* call the message settings procedure with the switch and show progress */
  end

  when (left(NewCommands.Index.!Switch,7) == 'TOOLBAR') then  /* if the switch starts with "toolbar" */
  do
   call Toolbar substr(NewCommands.Index.!Switch,8),NewCommands.Index.!Argument  /* call the toolbar procedure with the rest of the switch and the argument */
  end

  when (NewCommands.Index.!Switch == 'TRIMLOGFILES') then  /* if the switch is "trimlogfiles" */
  do
   call TrimLogFile Global.!ActionLog,Global.!Settings.LogActionLines  /* trim if necessary */
   call TrimLogFile Global.!ErrorLog,Global.!Settings.LogErrorLines  /* trim if necessary */
   call TrimLogFile Global.!MailLog,Global.!Settings.LogMailLines  /* trim if necessary */
  end

  when (NewCommands.Index.!Switch == 'TUTORIAL') then  /* if the switch is "tutorial" */
  do

   if (\sysopenobject(Global.!ProgDir||'\RexxMail_Tutorial.INF',0,1)) then  /* if we cannot open the RexxMail Tutorial on the Desktop */
   do
    call AddError 'Cannot open the RexxMail Tutorial : "'||Global.!ProgDir||'\RexxMail Tutorial.INF"'  /* report, not fatal */
   end

  end

  when (NewCommands.Index.!Switch == 'VIEW') then  /* if the switch was "view" */
  do
   call ProcessFiles 'ViewMessage'  /* call the message viewing procedure */
  end

  when (left(NewCommands.Index.!Switch,8) == 'WARNICON') then  /* if the switch starts with "warnicon" */
  do
   call WarnIcon substr(NewCommands.Index.!Switch,9)  /* call the warning icon procedure with the rest of the switch as argument */
  end

  otherwise  /* if none of the above applies */
  do
   call AddError 'Command parsing error: "'||NewCommands.Index.!Switch||'"'  /* report */
   call Quit  /* quit -- but this should never arise */
  end

 end

end

call Quit  /* that's all, folks! */

/**************************************************************************/
ProcessFiles:  /* processes a number of files in succession and show progress */
/**************************************************************************/

parse arg ProcLine,Show  /* get the arguments */

Show = (Show = 1)  /* 1 = TRUE */
Global.!Break = 0  /* no need to break off processing yet */
ProcLine = 'call '||ProcLine  /* add the call keyword */

if (Show) then  /* if we are to show progress */
do
 call syscurstate 'OFF'  /* hide the cursor */
 call charout 'CON:','Files to process: '||NewCommands.Index.!Files.0  /* show the total number of files */
end

do LocalIndex = 1 to NewCommands.Index.!Files.0  /* take each of the files we found */

 call LogAction 'Processing file no. '||LocalIndex||' of '||NewCommands.Index.!Files.0||': "'||NewCommands.Index.!Files.LocalIndex||'"',1  /* report, quietly */
 Global.!ProcFile = NewCommands.Index.!Files.LocalIndex  /* copy the file name to a global var (interpret does not work with long file names) */
 Global.!FilesToDo = Global.!FilesToDo - 1  /* so many files left to do */

 signal on syntax name InterpError  /* handles an interpreter error condition */

 interpret ProcLine  /* run the process */

 if ((Result == 0) & (Global.!Break)) then  /* if we did not succeed and we want to break off the process */
 do

  if (Show) then  /* if we are showing progress */
  do
   call ShowLine ' - error.'  /* show this */
   call syscurstate 'ON'  /* show the cursor */
  end

  return 0  /* quit with no result */

 end

 if (Show) then  /* if we are to show progress */
 do
  call charout 'CON:',copies(d2c(8),length(NewCommands.Index.!Files.0))||,  /* back up on the screen */
                      right(Global.!FilesToDo,length(NewCommands.Index.!Files.0),' ')  /* show the number of files left to do */
 end

end

if (Show) then  /* if we are to show progress */
do
 call ShowLine ' - ready.'  /* show this */
 call syscurstate 'ON'  /* show the cursor */
end

return 1  /* end of ProcessFiles */

/**************************************************************************/
InterpError:  /* handles interpreter errors */
/**************************************************************************/

signal on syntax name Syntax  /* use the normal syntax error routine */

call AddError 'Command syntax error : '||subword(ProcLine,3)||'.'  /* report */
call Quit  /* and quit */

/**************************************************************************/
Syntax:  /* handles syntax errors */
/**************************************************************************/

call FatalError 'REXX did not recognize the instruction',Sigl  /* start with this */
call Quit  /* and quit */

/**************************************************************************/
Error:  /* handles general errors */
/**************************************************************************/

call FatalError 'External program error',Sigl  /* start with this */
call Quit  /* and quit */

/**************************************************************************/
Failure:  /* handles external program errors */
/**************************************************************************/

call FatalError 'External program failure',Sigl  /* start with this */
call Quit  /* and quit */

/**************************************************************************/
NoValue:  /* handles initialization errors */
/**************************************************************************/

call FatalError 'Initialization error',Sigl  /* start with this */
call Quit  /* and quit */

/**************************************************************************/
Halt:  /* handles halt condition */
/**************************************************************************/

call FatalError 'Program aborted by user',Sigl  /* report */
call Quit  /* and quit */

/**************************************************************************/
FatalError: procedure expose Global.  /* reports a fatal error */
/**************************************************************************/

parse arg ErrorStart,Sigl  /* get the argument */

call AddError 'Fatal error: '||ErrorStart||' on line '||Sigl||':'||Global.!CRLF,  /* report */
              '    '||sourceline(Sigl)  /* report */

return 1  /* end of FatalError */

/**************************************************************************/
Quit:  /* handles normal and premature exits */
/**************************************************************************/

Errors = LogErrors(Global.!Errors)  /* show and log any waiting error messages */

if (symbol('OriginalCP') == 'VAR') then  /* if the original code page is defined (it may not be if we did not even get that far) */
do
 call syssetprocesscodepage OriginalCP  /* restore the original code page */
end

call directory Global.!CurDir  /* change back to the original directory */

exit Errors  /* quit */

/**************************************************************************/
AddError: procedure expose Global.  /* add an error string */
/**************************************************************************/

parse arg Message  /* get the argument */

Global.!Errors = Global.!Errors||'- '||Message||Global.!CRLF  /* add the message to the existing error string */

return  /* end of AddError */

/**************************************************************************/
AddressAddToOpen: procedure expose Global.  /* copy (address template) mail file to "To:" folder(s) of open RexxMail messages */
/**************************************************************************/

AddrFile = Global.!ProcFile  /* the file to process */
AttDir = AttDirGet(AddrFile)  /* get the corresponding attachments file */

if (sysqueryswitchlist('ListItems.') == 0) then  /* if we can get a switch list */
do

 do Index = 1 to ListItems.0  /* take each of the items found  */

  FileName = filespec('N',ListItems.Index)  /* try to extract a file name from the list item */

  if ((left(FileName,4) == 'RXML') & (right(FileName,5) == '.EDIT') & (datatype(substr(FileName,5,4),'W'))) then  /* if we find a RexxMail edit file name */
  do

   ListAttDir = Global.!TempDir||'\'||left(FileName,8)  /* the corresponding attachments dir */

   if (AttDir >< ListAttDir) then  /* if it is not our own attachments dir */
   do

    MessAddDir = ListAttDir||'\'||Global.!Settings.AddressTo  /* this should be the messages's additional addresses folder */

    if (syscopyobject(AddrFile,MessAddDir)) then  /* if we can copy the file to the additional addresses folder */
    do
     call LogAction 'Address file copied to message addresses folder "'||MessAddDir||'"',1  /* report, quietly */
    end

   end

  end

 end

end

return 1  /* end of AddressAddToOpen */

/**************************************************************************/
AddressCheck: procedure expose Global.  /* check address components */
/**************************************************************************/

parse arg FirstPart,LastPart  /* get the arguments */

Group = (pos(':;',FirstPart) > 0)  /* have we got a group here? */

if (Group && (words(LastPart) == 0)) then  /* if we have a group and the last part is not empty, or vice versa (EXOR) */
do
 return ''  /* return nothing */
end

if (Group) then  /* if we are dealing with a group */
do
 return FirstPart  /* this is the complete address */
end

FirstPart = strip(FirstPart,'L','<')  /* remove any leading < */
LastPart = strip(LastPart,'T','>')  /* remove any trailing > */
FirstPart = strip(FirstPart,'L','[')  /* remove any leading [ */
LastPart = strip(LastPart,'T',']')  /* remove any trailing ] */

if (left(translate(FirstPart),7) == 'MAILTO:') then  /* if the address starts with mailto: in whatever case */
do
 FirstPart = substr(FirstPart,8)  /* discard it */
end

if (words(FirstPart) == 0) then  /* if the first part no longer contains a word */
do
 return ''  /* return nothing */
end

if (words(LastPart) == 0) then  /* if the last part no longer contains a word */
do
 return ''  /* return nothing */
end

if (verify(translate(FirstPart||LastPart),'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789.!#$%&*+-/=?^_`{|}~'||"'",'NOMATCH') > 0) then  /* if the first part and/or the last part contains illegal characters */
do
 return ''  /* return nothing */
end

return FirstPart||'@'||LastPart  /* end of AddressCheck */

/**************************************************************************/
AddressExport: procedure expose Global.  /* exports address book contents and edits address text file */
/**************************************************************************/

signal on halt name HaltAddressExport  /* handles halt locally */

parse arg ExpFile  /* get the file name to process */

BakFile = ExpFile||'.backup'  /* the backup file */
call LogAction 'Exporting address data to "'||ExpFile||'"'  /* report */

if (FileCheck(ExpFile)) then  /* if the address file already exists */
do

 if (FileCheck(BakFile)) then  /* if the backup file already exists */
 do

  if (sysfiledelete(BakFile) > 0) then  /* if we cannot delete the old backup file */
  do
   call AddError 'Cannot delete "'||BakFile||'"'  /* report */
   return 0  /* and quit */
  end

 end

 if (\FileCopy(ExpFile,BakFile)) then  /* if we cannot copy the export file to the backup file */
 do
  call AddError 'Cannot copy "'||ExpFile||'" to "'||BakFile||'"'  /* report */
  return 0  /* and quit */
 end

 if (sysfiledelete(ExpFile) > 0) then  /* if we cannot delete the existing export file */
 do
  call AddError 'Cannot delete "'||ExpFile||'"'  /* report */
  return 0  /* and quit */
 end

end

OutText = ''  /* start with nothing */
call sysfiletree Global.!AddressesDir||'\*','Files.','FOS'  /* get the address book contents */

do Index = 1 to Files.0  /* take each of the objects */

 FileCont = GetFileContents(Files.Index)  /* store the contents of the file in an input buffer */
 parse var Files.Index (Global.!AddressesDir) '\' Name  /* get the bit of the file spec we want */
 OutText = OutText||'<TITLE>'||Global.!CRLF||Name||Global.!CRLF  /* add the file name to the output buffer with a header */

 if (FileCont >< '') then  /* if we have content */
 do
  OutText = OutText||'<TEXT>'||Global.!CRLF||FileCont||Global.!CRLF  /* add them to the output buffer with a header */
 end

 Subject = GetObjectEA(Files.Index,'.SUBJECT')  /* get the subject, if any */

 if (Subject >< '') then  /* if we have something */
 do
  OutText = OutText||'<SUBJECT>'||Global.!CRLF||Subject||Global.!CRLF  /* add it to the output buffer with a header */
 end

 Comments = GetObjectEA(Files.Index,'.COMMENTS')  /* get the file's comments, if any */

 if (Comments >< '') then  /* if we have something */
 do
  OutText = OutText||'<COMMENTS>'||Global.!CRLF||Comments||Global.!CRLF  /* add it to the output buffer with a header */
 end

 Keywords = GetObjectEA(Files.Index,'.KEYPHRASES')  /* get the groups, if any */

 if (Keywords >< '') then  /* if we have something */
 do
  OutText = OutText||'<KEYWORDS>'||Global.!CRLF||Keywords||Global.!CRLF  /* add it to the output buffer with a header */
 end

 OutText = OutText||'<END>'||Global.!EmptyLine  /* add the end tag and an empty line */

end

if (\PutFileContents(ExpFile,OutText)) then  /* if we cannot write the contents of the output buffer to the file */
do
 return 0  /* and quit */
end

return 1  /* end of AddressExport */

/**************************************************************************/
HaltAddressExport:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
AddressFormat: procedure expose Global.  /* formats a string of one or more addresses and optionally checks address syntax */
/**************************************************************************/

parse arg Addresses,Full,Check,Names  /* get the arguments */

Full = (Full == 1)  /* 1 = TRUE */
Check = (Check == 1)  /* 1 = TRUE */
Names = (Names == 1)  /* 1 = TRUE */
NewAddresses = ''  /* start with no new addresses */

if (Full) then  /* if we want a list of full addresses */
do
 Spacer = d2c(0)  /* the spacer is a null byte */
end
else  /* if we want bare addresses */
do
 Spacer = ' '  /* the spacer is a blank space */
end

call GetDefinitions Global.!DeSpam,'!DeSpam',0  /* see if we need to set any despam definitions */

if (Global.!DeSpam.0 > 0) then  /* if we have something */
do

 do Index = 1 to Global.!DeSpam.0  /* for each of the stored despam lines */

  DeSpamAddresses = ''  /* nothing yet */

  do while (Addresses >< '')  /* as long as we have address stuff */

   parse var Addresses NextPart (Global.!DeSpam.Index.!Right) Addresses  /* look for the next part followed by the despam string */
   DeSpamAddresses = DeSpamAddresses||NextPart  /* restore this part */

   if (Addresses >< '') then  /* if there is more */
   do
    DeSpamAddresses = DeSpamAddresses||Global.!DeSpam.Index.!Left  /* add the despammed string */
   end

  end

  Addresses = DeSpamAddresses  /* use this */

 end

end

do while (Addresses >< '')  /* as long as we have addresses */

 Addresses = strip(Addresses,'L',' ')  /* remove leading spaces */

 do while (left(word(Addresses,1),1) == ',')  /* as long as the next address starts with a comma */
  Addresses = strip(Addresses,'L',',')  /* remove leading commas */
  Addresses = strip(Addresses,'L',' ')  /* remove leading spaces */
 end

 if (Addresses >< '') then  /* if we have something left */
 do

  if (left(Addresses,1) == Global.!Warning) then  /* if the next address starts with an error marker */
  do
   Addresses = substr(Addresses,2)  /* use the rest */
  end

  OrgAddresses = Addresses  /* save the original addresses string */
  Comment = ''  /* we have no leading comment yet */
  EndComment = ''  /* we have no trailing comment yet */

  if (left(Addresses,1) == '"') then  /* if it starts with a quotation mark, it must be a comment preceding the actual address */
  do
   parse var Addresses '"' Comment '"' Addresses  /* get the comment part */
   Comment = strip(Comment,'B',"'")  /* get rid of any single quotes around the comment */
   Addresses = strip(Addresses,'L',' ')  /* remove any leading blanks from the addresses string */
  end

  LastPart = ''  /* start with nothing in the last part of the address */

  if (Addresses == '') then  /* if we have nothing left */
  do
   FirstPart = ''  /* there is no first address part either */
  end
  else  /* if we have more */
  do

   GroupPos = pos(':;',Addresses)  /* the first occurrence of a group terminator */
   AttPos = pos('@',Addresses)  /* the first occurrence of a normal address indicator */

   if ((GroupPos > 0) & ((AttPos == 0) | (GroupPos < AttPos))) then  /* if we have a leading group */
   do
    parse var Addresses FirstPart ':;' Addresses  /* get the group part */
    FirstPart = FirstPart||':;'  /* restore the terminator */
   end
   else  /* if we do not have a leading group */
   do
    parse var Addresses FirstPart '@' Addresses  /* look for the first part of the actual address */
   end

   LastLeft = lastpos('<',FirstPart)  /* the position in the first address part of the last left angle bracket, if any */

   if (LastLeft = 0) then  /* if we find none */
   do
    LastLeft = lastpos('[',FirstPart)  /* the position in the first address part of the last left square bracket, if any */
   end

   if (LastLeft > 0) then  /* if we found a left < or [ */
   do

    LastSpace = lastpos(' ',FirstPart)  /* the position of the last space in the first part */

    if (LastLeft > (LastSpace + 1)) then  /* if the last < or [ is more than one character past the last space (which may not exist), we have an address with a comment stuck to it */
    do
     parse var FirstPart LeftPart =(LastLeft) RightPart  /* get the two parts before and starting with the last < or [ */
     FirstPart = LeftPart||' '||RightPart  /* and stick them back together with a space in between */
    end

   end

   if ((Comment == '') & (words(FirstPart) > 1)) then  /* if we have no comment yet, and the first address part contains more than one word, part of it must be a leading comment */
   do
    Comment = subword(FirstPart,1,words(FirstPart)-1)  /* use all but the last word of the first part for the comment */
    Comment = strip(Comment,'T','"')  /* get rid of any trailing quotation mark to correct asymmetry */
    FirstPart = word(FirstPart,words(FirstPart))  /* use the last word for the first address part */
   end

   PrevChar = ''  /* start with no previous character */
   GotLastPart = 0  /* we have no last part of the address yet */

   do until (GotLastPart)  /* go on until we have what we need */

    NextChar = left(Addresses,1)  /* get the next character from the remaining addresses string */

    if (pos(NextChar,', (') > 0) then  /* if it is a comma, this is where a new address starts; if it is a space, an end comment may follow, if it is a left parenthesis, we have an end comment */
    do
     GotLastPart = 1  /* we have all we need */
    end
    else  /* if the next character is not a comma, a space, or a left parenthesis */
    do

     LastPart = LastPart||NextChar  /* add it to what we have */
     Addresses = substr(Addresses,2)  /* remove the first character from the remaining addresses string */

     if ((NextChar == ';') & (PrevChar == ':')) then  /* if we have a complete group terminator */
     do
      GotLastPart = 1  /* we have all we need */
     end
     else  /* if we have something else */
     do
      PrevChar = NextChar  /* store the character */
     end

    end

   end

   if (left(word(Addresses,1),1) == '(') then  /* if the first word of the remaining addresses string starts with a left parenthesis, we have an end comment */
   do
    parse var Addresses '(' EndComment ')' Addresses  /* extract the end comment */
   end

  end

  Address = AddressCheck(FirstPart,LastPart)  /* see if the address components are O.K., i.e. they make up a new address */

  if (Address == '') then  /* if the address was zapped (because it was bad) */
  do

   CutPoint = length(OrgAddresses) - length(Addresses) + 1  /* where the next original address, if any, starts */
   parse var OrgAddresses Address =(CutPoint) OrgAddresses  /* get the original address */

   if (Check) then  /* if we are checking the address */
   do

    if (Full) then  /* if we wanted the full address */
    do
     Address = Global.!Warning||Address  /* add a marker */
    end
    else  /* if we did not want the full address */
    do
     Address = Global.!Warning||word(Address,words(Address))  /* add a marker, but use only the last word (any previous words are comments) */
    end

   end

  end
  else  /* if all is well */
  do

   if (LastPart == '') then  /* if we have no last part, i.e. it is a group */
   do

    if (pos(Spacer||Address,Spacer||NewAddresses) > 0) then  /* if we already have the address in the list of new addresses */
    do
     Address = ''  /* zap it */
    end

   end
   else  /* if it is not a group */
   do

    if (pos('<'||Address||'>',NewAddresses) > 0) then  /* if we already have the address in the list of new addresses */
    do
     Address = ''  /* zap it */
    end

   end

   if (Address >< '') then  /* if we still have something, i.e. if it is a new address */
   do

    if ((Full) | (Names)) then  /* if we want the full address or a list of names */
    do

     if (EndComment >< '') then  /* if we have an end comment */
     do

      if (Comment >< '') then  /* if we have a front comment */
      do
       Comment = Comment||' - '  /* stick on a separator */
      end

      EndComment = translate(EndComment,' ','"')  /* remove any double quotation marks */
      Comment = Comment||EndComment  /* append the end comment to the front comment, if any */

     end

     if (Comment >< '') then  /* if we have a comment */
     do

      if (Comment >< Address) then  /* if it's not the same as the address */
      do

       if (Full) then  /* if we want the full address */
       do
        Comment = '"'||Comment||'" '  /* add double quotes and a space */
       end

      end
      else  /* if it is the same as the address (some brain-dead mailers will do this) */
      do
       Comment = ''  /* use nothing */
      end

     end

     if (Full) then  /* if we want the full address */
     do

      if (LastPart >< '') then  /* if we have a last part, i.e. it is not a group */
      do
        Address = '<'||Address||'>'  /* add angle brackets to the bare address */
      end
    
      Address = Comment||Address  /* stick any comment we have on the front */

     end
     else  /* if we want a list of names */
     do

      if (LastPart >< '') then  /* if there is a last part (i.e. it is not a group) */
      do

       if (Comment >< '') then  /* if we a have a comment (i.e. a real name) */
       do

        if (Global.!NamesList >< '') then  /* if this is not the first one */
        do
         Global.!NamesList = Global.!NamesList||','  /* add a comma */
        end

        Global.!NamesList = Global.!NamesList||Global.!CRLF||' '||Comment  /* add the comment (name) to the list */

       end
       else  /* if there is no comment/name */
       do
        Global.!NoNames = Global.!NoNames + 1  /* up the no names counter */
       end

      end

     end

    end

   end
   else  /* if it is not a new address */
   do
    Address = ''  /* zap it */
   end

  end

  if (Address >< '') then  /* if we have something  */
  do
   NewAddresses = NewAddresses||Spacer||Address  /* add a spacer and the result to what we already have */
  end

 end

end

NewAddresses = strip(NewAddresses,'L',Spacer)  /* remove the leading spacer */

return NewAddresses  /* end of AddressFormat */

/**************************************************************************/
AddressFromFile: procedure expose Global.  /* gets one or more addresses from a RexxMail message file */
/**************************************************************************/

parse arg MessFile,MakeObject  /* get the argument */

MakeObject = (MakeObject == 1)  /* 1 = TRUE */
MessCont = GetFileContents(MessFile)  /* get the file contents */

if (MessageSettings(MessFile,,'CHECK')) then  /* if it is a RexxMail message file */
do

 if (\MessageContents(MessCont,'Global.!MessHead.')) then  /* if we cannot get the header sorted out */
 do
  return ''  /* and quit with nothing */
 end

 if (MessageSettings(MessFile,'1*******','MATCH')) then  /* if it is an incoming message */
 do
  Address = GetHeaderEntry('FROM','Global.!MessHead.')  /* use this address */
 end
 else  /* if it is not an incoming message */
 do
  Address = GetHeaderEntry('TO','Global.!MessHead.')  /* use this address */
 end

 if (Address == '') then  /* if there is no address */
 do
  call AddError 'No address found'  /* report */
  return ''  /* and quit with nothing */
 end

 if (MakeObject) then  /* if we want an address template object */
 do
  return MakeAddressObject(Global.!AddressesDir||'\'||CheckCommLine(AddressIndent(Address)),'To: '||AddressIndent(Address,1,76)||Global.!CRLF||'Subject: ')  /* make an address template object and quit */
 end
 else  /* if we just want the address */
 do
  return AddressFormat(Address,1,0)  /* reformat the address(es): full, no check, and quit */
 end

end

/**************************************************************************/
/* If we get to here, it is not a RexxMail message file                   */
/**************************************************************************/

if (MakeObject) then  /* if we want message template objects */
do
 Success = 1  /* all is well to start with */
end
else  /* if we want addresses */
do
 Addresses = ''  /* start with nothing */
end


if (pos('<TITLE>',translate(MessCont)) == 0) then  /* if it looks like a normal address text file */
do

 do while (MessCont >< '')  /* go on until we run out of content */

  parse var MessCont Address (Global.!CRLF) MessCont  /* get the next address line */

  if (Address >< '') then  /* if we have something */
  do

   if (left(Address,1) >< '#') then  /* if it is not a comment */
   do

    Address = AddressFormat(Address,1,0)  /* reformat the address(es), full, no check */

    if (MakeObject) then  /* if we want message template objects */
    do
     Success = (Success & MakeAddressObject(Global.!AddresseDir||'\'||filespec('N',MessFile)||'\'||CheckCommLine(AddressIndent(Address)),'To: '||Address||Global.!CRLF||'Subject: '))  /* make an address template object */
    end
    else  /* if we want the address */
    do
     Addresses = Addresses||','||Address  /* add it to what we have */
    end

   end

  end

 end

 if (MakeObject) then  /* if we want message template objects */
 do
  return Success  /* quit with the overall result */
 end
 else  /* if we want the addresses */
 do
  return strip(Addresses,'L',',')  /* quit with any addresses we collected, after removing the leading comma */
 end

end

/**************************************************************************/
/* If we get to here, it is a RexxMail address import file                */
/**************************************************************************/

if (\MakeObject) then  /* if we want addresses */
do
 return ''  /* and quit with nothing */
end

MessCont = Global.!CRLF||MessCont  /* add a CRLF to make sure the first block marker is recognized */
Sep_1 = Global.!CRLF||'<TITLE>'  /* define a separator */
Sep_2 = Global.!CRLF||'<TEXT>'  /* define a separator */
Sep_3 = Global.!CRLF||'<SUBJECT>'  /* define a separator */
Sep_4 = Global.!CRLF||'<COMMENTS>'  /* define a separator */
Sep_5 = Global.!CRLF||'<KEYWORDS>'  /* define a separator */
Sep_6 = Global.!CRLF||'<END>'  /* define a separator */
EntryNo = 0  /* start at 0 */

do while (MessCont >< '')  /* go on while we have contents left */

 EntryNo = EntryNo + 1  /* up the entry number */
 parse var MessCont (Sep_1) Block (Sep_6) MessCont  /* get the next block */

 if (Block == '') then  /* if the block is empty */
 do

  if (MessCont >< '') then  /* if it not because we have run out of contents */
  do
   call AddError 'Entry number '||EntryNo||' contains no data'  /* report */
  end

 end
 else  /* if the block is not empty */
 do

  Title = ''  /* start with nothing */
  Text = ''  /* start with nothing */
  Subject = ''  /* start with nothing */
  Comments = ''  /* start with nothing */
  Keywords = ''  /* start with nothing */
  parse var Block Block (Sep_5) Keywords  /* extract any keywords */
  parse var Block Block (Sep_4) Comments  /* extract any comments */
  parse var Block Block (Sep_3) Subject  /* extract any subject */
  parse var Block Title (Sep_2) Text  /* extract any title and text contents */

  if (Title >< '') then  /* if we have a title bit */
  do

   do while (left(Title,2) == Global.!CRLF)  /* as long as the title starts with a CRLF */
    Title = substr(Title,3)  /* get rid of it */
   end

   do while (right(Title,2) == Global.!CRLF)  /* as long as the title ends with a CRLF */
    Title = substr(Title,1,length(Title) - 2)  /* get rid of it */
   end

  end

  if (Title == '') then  /* if we have no title */
  do
   call AddError 'Entry number '||EntryNo||' contains no title'  /* report, not fatal */
  end
  else  /* if we have a title */
  do

   if (Text >< '') then  /* if we have a text bit */
   do

    do while (left(Text,2) == Global.!CRLF)  /* as long as the text starts with a CRLF */
     Text = substr(Text,3)  /* get rid of it */
    end

    do while (right(Text,2) == Global.!CRLF)  /* as long as the text ends with a CRLF */
     Text = substr(Text,1,length(Text) - 2)  /* get rid of it */
    end

   end

   if (Text == '') then  /* if we have no text */
   do
    call AddError 'Entry number '||EntryNo||' contains no text'  /* report */
   end

   NewTitle = ''  /* start with nothing */

   do while (Title >< '')  /* as long as we have bits of title left */
    parse var Title NextBit (Global.!CRLF) Title  /* get the bit up to the next CRLF */
    NewTitle = NewTitle||NextBit||' '  /* store the next bit */
   end

   NewTitle = strip(NewTitle,'T',' ')  /* remove excess whitespace */

   if (substr(NewTitle,2,1) >< ':') then  /* if we do not have a drive spec */
   do
    NewTitle = Global.!AddressesDir||'\'||strip(NewTitle,'L','\')  /* remove any leading backslash, and add the address dir with a backslash */
   end

   Success = (Success & MakeAddressObject(NewTitle,Text,Subject,Comments,Keywords))  /* create the address template */

  end

 end

end

return Success  /* end of AddressFromFile */

/**************************************************************************/
AddressGetAdds: procedure expose Global.  /* looks for additional recipient addresses */
/**************************************************************************/

parse arg AddsDir  /* get the argument */

Addresses = ''  /* start with nothing to add */
ZapAddsDir = 1  /* no need to keep the adds dir afterwards */
call sysfiletree AddsDir||'\*','AddressFiles.','FO'  /* look for files in the address insertion folder */

if (AddressFiles.0 > 0) then  /* if we found any */
do

 do Index = 1 to AddressFiles.0  /* run through the files we found */

  NewAddresses = AddressFromFile(AddressFiles.Index,0)  /* get the relevant address(es) */

  if (NewAddresses >< '') then  /* if all is well */
  do
   Addresses = Addresses||','||NewAddresses  /* add the new stuff to what we already have */
   call sysdestroyobject AddressFiles.Index  /* get rid of the file */
  end
  else  /* if we have one or more invalid addresses */
  do
   ZapAddsDir = 0  /* block zapping */
  end

 end

end

if (ZapAddsDir) then  /* if we are to zap the address insertion folder */
do
 call sysdestroyobject AddsDir  /* get rid of the folder -- no checks, it may have been deleted already by the user */
end

return Addresses  /* end of AddressGetAdds */

/**************************************************************************/
AddressImport: procedure expose Global.  /* imports address book entries */
/**************************************************************************/

signal on halt name HaltAddressImport  /* handles halt locally */

call LogAction 'Importing address data from "'||Global.!ProcFile||'"'  /* report */
call AddressFromFile Global.!ProcFile,1  /* process the file */

return 1  /* end of AddressImport */

/**************************************************************************/
HaltAddressImport:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
AddressIndent: procedure expose Global.  /* indents a list of formatted addresses */
/**************************************************************************/

parse arg Addresses,Indent,TextLength  /* get the arguments */

if (Indent == '') then  /* if we have no indent argument */
do
 Spacer = ', '  /* use this as a spacer */
end
else  /* if we have an indent value */
do
 Spacer = ','||Global.!CRLF||copies(' ',Indent)  /* use this */
end

NewAddresses = ''  /* nothing yet */
Separator = d2c(0)  /* the separator to look for */

do while (Addresses >< '')  /* go on until we run out of steam */

 parse var Addresses NextAddress (Separator) Addresses  /* get the next address */

 if (TextLength >< '') then  /* if we have a maximum text length */
 do
  NextAddress = wordWrap(NextAddress,TextLength,0,copies(' ',Indent))  /* word-wrap the address (no break) */
 end

 NewAddresses = NewAddresses||NextAddress  /* add it to what we have */

 if (Addresses >< '') then  /* if there is more to come */
 do
  NewAddresses = NewAddresses||Spacer  /* add a spacer */
 end

end

return NewAddresses  /* end of AddressIndent */

/**************************************************************************/
AddressMakeAddDir: procedure expose Global.  /* creates address addition folders in attachments folder of messages when editing */
/**************************************************************************/

parse arg AttDir,FolderType  /* get the folder type */

AddrType = 'ADDRESS'||FolderType  /* set the full address folder type variable */

if (Global.!Settings.AddrType == '') then  /* if we have no corresponding global variable value */
do
 return ''  /* nothing to return */
end

FolderName = CheckCommLine(Global.!Settings.AddrType,':')  /* clean up the name */

if (FolderName == '') then  /* if we have nothing left */
do
 call AddError 'No legal characters found in Address'||FolderType||' setting : '||Global.!Settings.AddrType  /* report */
 return ''  /* and quit */
end
 
Settings = 'ICONFILE='||Global.!IconDir||'\Folders\Address_'||FolderType||'_0.ICO;'||,  /* use this closed icon */
           'ICONNFILE=1,'||Global.!IconDir||'\Folders\Address_'||FolderType||'_1.ICO;'||,  /* and this open icon */
           'BACKGROUND='||Global.!IconDir||'\Folders\Mail_Background.BMP,T,,I;'  /* and this background */

if (syscreateobject('WPFolder',FolderName,AttDir,,'FAIL')) then  /* if we can create this folder */
do
 call SetObjectData AttDir||'\'||FolderName,Settings  /* try to adjust its settings */
end

return FolderName  /* end of AddressMakeAddDir */

/**************************************************************************/
AttDirClose: procedure expose Global.  /* closes an attachments dir and returns an attachments indicator */
/**************************************************************************/

parse arg AttDir  /* get the argument */

call RexxMailCloseObject filespec('N',AttDir)  /* close the attachments dir -- EXTERNAL -- RXMLUTIL.DLL */
call sysfiletree AttDir||'\*','Objects.','BO'  /* is there anything in the attachments folder? */

if (Objects.0 == 0) then  /* if we have no contents */
do
 return 0  /* return with nothing */
end

return 1  /* end of AttDirCLose */

/**************************************************************************/
AttDirCreate: procedure expose Global.  /* create an attachment dir and EA */
/**************************************************************************/

parse arg MessFile  /* get the argument, if any */

AttDir = TempFileName()  /* get a unique directory name (in the temp directory) */

if (AttDir == '') then  /* if we cannot get a temp file */
do
 return ''  /* quit with no success */
end

AttDirName = filespec('N',AttDir)  /* the name bit */
Settings = 'BACKGROUND='||Global.!IconDir||'\Folders\Attachments_Background.BMP,T,,I;'  /* use this background */

if (\syscreateobject('WPFolder',AttDirName,Global.!TempDir,,'FAIL')) then  /* if we cannot create an attachments folder in the temp dir */
do
 call AddError 'Cannot create attachments folder'  /* report */
 return ''  /* return with nothing */
end

call SetObjectData AttDir,Settings  /* try to adjust the settings */

if (MessFile >< '') then  /* if we have a file spec */
do

 if (\AttDirLink(MessFile,AttDir)) then  /* if we cannot link the attachments dir to the message file */
 do
  return ''  /* return with nothing */
 end

end

return AttDir  /* end of AttDirCreate */

/**************************************************************************/
AttDirGet: procedure expose Global.  /* get an attachment directory name, optionally creating the dir */
/**************************************************************************/

parse arg MessFile,Create  /* get the arguments */

Create = (Create == 1)  /* 1 = true */
AttDir = GetObjectEA(MessFile,'RXMLATTDIR')  /* get the attachments dir EA */

if (AttDir >< '') then  /* if we have something */
do

 call sysfiletree AttDir,'Dirs.','DO'  /* look for the actual directory */

 if (Dirs.0 == 0) then  /* if no such directory was found (i.e. it was deleted) */
 do
  AttDir = ''  /* we have nothing */
 end

end

if ((AttDir == '') & Create) then  /* if we still have nothing, and we want one created */
do
 AttDir = AttDirCreate(MessFile)  /* create a new one and link it to the message file */
end

return AttDir  /* end of AttDirGet */

/**************************************************************************/
AttDirLink: procedure expose Global.  /* link an attachment dir to a message file */
/**************************************************************************/

parse arg MessFile,AttDir  /* get the arguments */

if (\PutObjectEA(MessFile,'RXMLATTDIR',AttDir)) then  /* if we cannot set the EA */
do
 return 0  /* return with an error */
end

return 1  /* end of AttDirLink */

/**************************************************************************/
AttDirOpen: procedure expose Global.  /* opens an attachments dir on the desktop */
/**************************************************************************/

parse arg Messfile  /* get the argument */

if ((MessageSettings(MessFile,'1*******','MATCH')) | (MessageSettings(MessFile,'0*1*****','MATCH'))) then  /* if it is an incoming or sent message */
do
 AttDir = AttDirGet(MessFile,0)  /* get any existing attachments dir (i.e. only if the message is open on the desktop) */
end
else  /* if it is an outgoing message */
do
 AttDir = AttDirGet(MessFile,1)  /* get the attachments dir and create it if necessary */
end

if (AttDir >< '') then  /* if we have something to show */
do

 if (\sysopenobject(AttDir,0,1)) then  /* if we cannot open the attachments folder */
 do
  call AddError 'Cannot open attachments folder'  /* report */
 end

end

return AttDir  /* end of AttDirOpen */

/**************************************************************************/
AttDirShow: procedure expose Global.  /* shows an attachments dir on the desktop */
/**************************************************************************/

return (AttDirOpen(Global.!ProcFile))  /* end of AttDirShow */

/**************************************************************************/
AttDirShut: procedure expose Global.  /* shuts an attachments dir for the main routine */
/**************************************************************************/

AttDir = AttDirGet(Global.!ProcFile)  /* get the attachments dir */

if (AttDir == '') then  /* if there is none */
do
 return 0  /* return with nothing */
end

return (AttDirClose(AttDir))  /* end of AttDirShut */

/**************************************************************************/
AttDirUnlink: procedure expose Global.  /* delete the attachments directory EA */
/**************************************************************************/

parse arg MessFile  /* get the argument */

if (\PutObjectEA(MessFile,'RXMLATTDIR','')) then  /* if we cannot reset the EA */
do
 return 0  /* and quit with no succes */
end

return 1  /* end of AttDirUnlink */

/**************************************************************************/
CheckCommLine: procedure  /* replaces awkward characters in a string for command-line use and removes double spaces */
/**************************************************************************/

parse arg String,Extras  /* get the arguments */

Source = xrange('00'x,'0F'x)||xrange('11'x,'1F'x)||'<>[]~%^?&*=\|/;,'||'"'||'`'||Extras  /* bad characters -- Global.!Warning =  (16 dec) */
Target = copies(' ',31)||'()()__________..'||"'"||"'"||copies(' ',length(Extras))  /* replacements */
String = translate(String,Target,Source)  /* remove difficult characters from the string */
NewString = ''  /* start with nothing */

do while (words(String) > 0)  /* as long as we have words left */
 parse var String NextWord String  /* get the next word */
 NewString = NewString||' '||NextWord  /* and add to what we've got */
end

NewString = strip(NewString,'L',' ')  /* get rid of the leading blank */

return NewString  /* end of CheckCommLine */

/**************************************************************************/
CheckRecipients: procedure expose Global.  /* checks a list of new recipients */
/**************************************************************************/

parse arg Recipients,GotRecipients,FileName  /* get the arguments */

NewRecipients = ''  /* start with nothing */

do while (Recipients >< '')  /* run through the list of recipients */

 parse var Recipients Recipient Recipients  /* get the next address */

 if (pos(Global.!Warning,Recipient) > 0) then  /* if the address contains a warning marker */
 do

  parse var Recipient (Global.!Warning) Recipient  /* get rid of the marker */

  if (FileName >< '') then  /* if we have a file name */
  do
   Insert = 'found in "'||FileName||'"'  /* insert this later */
  end
  else  /* if we do not have a file name */
  do
   Insert = ''  /* no insert */
  end

  call AddError 'Excluded malformed recipient address '||Insert||': '||Recipient  /* report, non-fatal */

 end
 else  /* if all is well */
 do

  if (wordpos(Recipient,GotRecipients||' '||NewRecipients) == 0) then  /* if it is one we haven't got yet */
  do
   NewRecipients = NewRecipients||' '||Recipient  /* add it to the list */
   call LogAction 'Adding address: '||Recipient,1  /* report, quietly */
  end

 end

end

return NewRecipients  /* end of CheckRecipients */

/**************************************************************************/
CheckValBool: procedure expose Global.  /* checks the value of a Boolean argument */
/**************************************************************************/

parse arg Name  /* get the arguments */

CurVal = value('Global.!Settings.'||Name)  /* get the current value */

if (CurVal == '') then  /* if we have no value */
do
 return 1  /* the setting is TRUE */
end

SetTrue = 'TRUE T YES Y ON 1 +'  /* these all count as true */
SetFalse = 'FALSE F NO N OFF 0 -'  /* these all count as false */
CurVal = translate(CurVal)  /* make the value upper case */

if (wordpos(CurVal,SetTrue) > 0) then  /* if the setting is TRUE */
do
 return 1  /* return this */
end

if (wordpos(CurVal,SetFalse) > 0) then  /* if the setting is FALSE */
do
 return 0  /* return this */
end

call AddError 'The "'||Name||'" setting requires a TRUE or FALSE argument ('||SetTrue||' or '||SetFalse||')'  /* report */

return 0  /* end of CheckValBool */

/**************************************************************************/
CheckValNum: procedure expose Global.  /* checks the value of a parameter and sets a default value if necessary */
/**************************************************************************/

parse arg Name,Default,Low,High  /* get the arguments */

CurVal = value('Global.!Settings.'||Name)  /* get the current value */

if (CurVal == '') then  /* if the value is empty */
do
 return Default  /* return the default value */
end

if (\datatype(CurVal,'W')) then  /* if the value is not a whole number */
do
 call AddError 'The "'||Name||'" setting requires a whole number instead of '||CurVal  /* report */
 return Default  /* return the default value */
end

if ((Low >< '') & (CurVal < Low)) then  /* if it is out of the low range */
do
 call AddError 'The "'||Name||'" setting '||CurVal||' is out of range (< '||Low||')'  /* report */
 return Low  /* return the low value */
end

if ((High >< '') & (CurVal > High)) then  /* if it is out of the high range */
do
 call AddError 'The "'||Name||'" setting '||CurVal||' is out of range (> '||High||')'  /* report */
 return High  /* return the high value */
end

return CurVal  /* end of CheckValNum */

/**************************************************************************/
CheckValSignal: procedure expose Global.  /* check the value of a beep sequence signal string */
/**************************************************************************/

parse arg Name  /* get the argument */

CurVal = value('Global.!Settings.'||Name)  /* get the current value */

if (CurVal = '') then  /* if the value is empty */
do
 return ''  /* return with nothing */
end

ErrorString = ''  /* start with nothing */

if (translate(word(CurVal,1)) = 'BEEP') then  /* if it looks like a beep sequence, check it (if not, it must be an external command) */
do

 CurVal = translate(CurVal)  /* make the signal string upper case for use by the "SoundSignal" procedure */
 BeepSequence = subword(CurVal,2)  /* get the beep sequence */

 do while (BeepSequence >< '')  /* go on until we run out of steam */

  parse var BeepSequence Frequency ',' Duration BeepSequence  /* get the components */

  if (datatype(Frequency,'W')) then  /* if it is a whole number */
  do

   if ((Frequency < 37) | (Frequency > 32767)) then  /* if it is out of range */
   do
    ErrorString = ErrorString||Global.!CRLF||'  Frequency out of range (37-32767): '||Frequency  /* add this */
    CurVal = ''  /* we have an error */
   end

  end
  else  /* if it is not a whole number */
  do
   ErrorString = ErrorString||Global.!CRLF||'  Not a whole number: '||Frequency  /* add this */
   CurVal = ''  /* we have an error */
  end
   
  if (Duration >< '') then  /* if we have a duration value */
  do

   if (datatype(Duration,'W')) then  /* if this is a whole number */
   do
    
    if ((Duration < 1) | (Duration > 60000)) then  /* if it is out of range */
    do
     ErrorString = ErrorString||Global.!CRLF||'  Duration out of range (1-60000): '||Duration  /* add this */
     CurVal = ''  /* we have an error */
    end

   end
   else  /* if it is not a whole number */
   do
    ErrorString = ErrorString||Global.!CRLF||'  Not a whole number: '||Duration  /* add this */
    CurVal = ''  /* we have an error */
   end
   
  end
  else  /* if we have no duration value */
  do
   ErrorString = ErrorString||Global.!CRLF||'  Missing duration value'  /* add this */
   CurVal = ''  /* we have an error */
  end

 end
 
end

if (ErrorString >< '') then  /* if we have one or more errors */
do
 call AddError 'Invalid "'||Name||'" setting: '||ErrorString  /* report */
end

return CurVal  /* end of CheckValSignal */

/**************************************************************************/
CheckValText: procedure expose Global.  /* checks the value of a text parameter */
/**************************************************************************/

parse arg Name,OKValues,Single,Fuzzy  /* get the arguments */

CurVal = value('Global.!Settings.'||Name)  /* get the current value */

if (CurVal == '') then  /* if the value is empty */
do
 return ''  /* return it */
end

Single = (Single == 1)  /* 1 = True */
Fuzzy = (Fuzzy == 1)  /* 1 = True */

if (Fuzzy) then  /* if additional formatting characters are allowed */
do
 CheckValue = translate(CurVal,' ',Global.!ExtraCharacters)  /* turn extra formatting characters into blanks */
end
else  /* if no additional characters are allowed */
do
 CheckValue = CurVal  /* use the original */
end

if ((Single) & (words(CheckValue) > 1)) then  /* if the value should be single, and contains more than one word */
do
 call AddError 'The "'||Name||'" setting requires a single value instead of "'||CurVal||'"'  /* report */
 return ''  /* return with nothing */
end

if (OKValues >< '') then  /* if we have a list of OK values */
do

 do while (CheckValue >< '')  /* go on while we have stuff */

  parse var CheckValue CheckWord CheckValue  /* get the next word */

  if (wordpos(translate(CheckWord),translate(OKValues)) == 0) then  /* if the word is not in the OK values */
  do
   call AddError 'Invalid "'||Name||'" setting: "'||CheckWord||'"'  /* report */
   return ''  /* return with nothing */
  end

 end

end

return CurVal  /* end of CheckValText */

/**************************************************************************/
CleanUpTemp: procedure expose Global.  /* cleans up files and directories in the Temp dir */
/**************************************************************************/

DelByDate = 42  /* the number of days to leave obsolete files and dirs alone */
DoneZap = 0  /* nothing deleted yet */
NewDir = Global.!In_ArchiveDir||'\Obsolete_RexxMail_Attachments_Folder'  /* a new dir name */
call sysfiletree Global.!TempDir||'\RXML*','AttDirs.','DL'  /* look for RexxMail attachments folders in the temp directory */

do Index = 1 to AttDirs.0  /* take each of the objects found */

 parse value AttDirs.Index with DirDT . . . DirName  /* get the bits we want */
 DirName = strip(DirName)  /* remove excess whitespace */
 call SetObjectData DirName,'NODELETE=NO;NOMOVE=NO;NODRAG=NO;NOSETTINGS=NO'  /* make sure we can manipulate it */
 call sysfiletree DirName||'\*','DirCont.','BOS'  /* look for stuff in the temp directory */

 ZapIt = ((DirCont.0 == 0) & (right(AttDirs.Index,4) >< '.ATT'))  /* if the dir is empty and not a send dir, zap it */

 if (\ZapIt) then  /* if it is not empty, or a send dir */
 do

  if ((DateApprox(date('S')) - DateApprox(DirDT)) > DelByDate) then  /* if it is too old */
  do

   ZapIt = 1  /* we will zap the folder */

   if (DirCont.0 > 0) then  /* if we have contents */
   do

    DoneZap = 1  /* we zapped something with contents */
    call sysmkdir NewDir  /* create the new dir if necessary */

    do SubIndex = 1 to DirCont.0  /* take each of the objects we found */
     call SetObjectData DirCont.SubIndex,'NODELETE=NO;NOMOVE=NO;NODRAG=NO;NOSETTINGS=NO'  /* make sure we can manipulate it */
     call sysmoveobject DirCont.SubIndex,NewDir  /* move it */
    end

   end

  end

 end

 if (ZapIt) then  /* if we want the dir zapped */
 do
  call sysdestroyobject DirName  /* get rid of it */
  call LogAction 'Deleted obsolete attachments dir "'||DirName||'"',1  /* report, quietly */
 end

end

if (DoneZap) then  /* if we zapped one or more folders with contents */
do
 Subject = 'Orphaned attachments folder found.'  /* message subject */
 Message = 'RexxMail found one or more orphaned attachments folders in'||Global.!CRLF,  /* message text */
           '    "'||Global.!TempDir||'"'||Global.!CRLF,  /* message text */
           'Their contents have been moved to'||Global.!CRLF,  /* message text */
           '    "'||NewDir||'"'  /* message text */
 call SystemMessage Subject,Message  /* report */
end

call sysfiletree Global.!TempDir||'\*','TempFiles.','FL'  /* look for files in the temp directory */

do Index = 1 to TempFiles.0  /* take each of the objects found */

 parse value TempFiles.Index with FileDT . . . FileName  /* get the bits we want */
 FileName = strip(FileName)  /* remove excess whitespace */
 call SetObjectData FileName,'NODELETE=NO;NOMOVE=NO;NODRAG=NO;NOSETTINGS=NO'  /* make sure we can manipulate it */

 if (((DateApprox(date('S')) - DateApprox(FileDT)) > DelByDate) | (left(filespec('N',FileName),4) >< 'RXML')) then  /* if it is too old, or not a temp file */
 do
  call sysfiledelete FileName  /* get rid of it */
 end

end

return 1  /* end of CleanUpTemp */

/**************************************************************************/
CollectMail: procedure expose Global.  /* Gets waiting mail messages from POP3 server */
/**************************************************************************/

signal on halt name HaltCollectMail  /* handles halt locally */

Socket = ''  /* no socket yet */
SemHandle = ''  /* no semaphore handle yet */

/**************************************************************************/
/* Have we got an external command to run?                                */
/**************************************************************************/

call RunCommand Global.!Settings.RunBeforeCollect  /* see if we can run an external command before collecting mail */

if (RunCommand(Global.!Settings.RunCollect) >< '') then  /* if we can run an external command to collect mail */
do
 call RunCommand Global.!Settings.RunAfterCollect  /* see if we can run an external command after collecting mail */
 return 1  /* quit */
end

/**************************************************************************/
/* See if we have all the arguments we really need                        */
/**************************************************************************/

if (Global.!Settings.POP3Server == '') then  /* if we have no server name */
do
 call AddError 'Missing configuration entry: POP3Server'  /* report */
 return 0  /* return */
end

if (Global.!Settings.POP3User == '') then  /* if we have no user name */
do
 call AddError 'Missing configuration entry: POP3User'  /* report */
 return 0  /* return */
end

if (Global.!Settings.POP3Password == '') then  /* if we have no password */
do
 call AddError 'Missing configuration entry: POP3Password'  /* report */
 return 0  /* return */
end

/**************************************************************************/
/* See if we have a non-standard port number specified                    */
/**************************************************************************/

parse var Global.!Settings.POP3Server Global.!Settings.POP3Server ':' PortNumber  /* look for a port number */

if (PortNumber == '') then  /* if there is none */
do
 PortNumber = 110  /* default to this (POP3) */
end

/**************************************************************************/
/* See if we are the only collect process active at the moment            */
/**************************************************************************/

MutexName = Global.!Settings.POP3Server||Global.!Settings.Pop3User  /* construct the mutex semaphore name */
SemHandle = syscreatemutexsem('\SEM32\'||MutexName)  /* get a semaphore handle */

if (sysrequestmutexsem(SemHandle,1) >< 0) then  /* if we cannot grab the semaphore */
do
 call LogAction 'Another RexxMail process is already collecting mail for "'||,  /* report */
                Global.!Settings.POP3User||'" from "'||Global.!Settings.POP3Server||'"'  /* report */
 return 0  /* and quit */
end

/**************************************************************************/
/* Try to connect to the server                                           */
/**************************************************************************/

Socket = ServerConnect(Global.!Settings.POP3Server,PortNumber,Global.!Settings.POP3Attempts)  /* get the socket number for a server connection through the specified or default port */

if (Socket == '') then  /* if we have no socket */
do
 call sysreleasemutexsem SemHandle  /* release the semaphore */
 return 0  /* return */
end

Global.Socket.SocketBuffer = ''  /* start with an empty socket buffer */

if (\SocketAnswer(Socket,'+OK')) then  /* if we get the wrong return code */
do
 call AddError 'No response from "'||Global.!Settings.POP3Server||'"'  /* report */
 call sysreleasemutexsem SemHandle  /* release the semaphore */
 return 0  /* return */
end

/**************************************************************************/
/* Try to log on to the server                                            */
/**************************************************************************/

call LogAction 'Logging on as user "'||Global.!Settings.POP3User||'"'  /* report */

if (\SocketSendLine(Socket,'USER '||Global.!Settings.POP3User,'+OK')) then  /* if we cannot send this and get the right reply */
do
 call LogAction 'User logon failed'  /* report */
 call AddError 'User not accepted'  /* report */
 call ServerDisconnect Socket,Global.!Settings.POP3Server  /* disconnect from the POP3 server */
 call sysreleasemutexsem SemHandle  /* release the semaphore */
 return 0  /* return */
end

call LogAction 'Sending password'  /* report */

if (\SocketSendLine(Socket,'PASS '||Global.!Settings.POP3Password,'+OK')) then  /* if we cannot send this and get the right reply */
do
 call LogAction 'Password verification failed'  /* report */
 call AddError 'Password not accepted'  /* report */
 call ServerDisconnect Socket,Global.!Settings.POP3Server  /* disconnect from the POP3 server */
 call sysreleasemutexsem SemHandle  /* release the semaphore */
 return 0  /* return */
end

call LogAction 'Logon successful'  /* report */
Messages = POP3Process(Socket)  /* process any waiting mail */

/**************************************************************************/
HaltCollectMail:  /* handles halt locally */
/**************************************************************************/

if (Socket >< '') then  /* if we have a socket */
do
 call ServerDisconnect Socket,Global.!Settings.POP3Server  /* disconnect */
end

if (SemHandle >< '') then  /* if we have a semaphore handle */
do
 call sysreleasemutexsem SemHandle  /* release the semaphore */
end

call RunCommand Global.!Settings.RunAfterCollect  /* see if we can run an external command after collecting mail */

return Messages  /* end of CollectMail */

/**************************************************************************/
CopyAttachment: procedure expose Global.  /* attach file to outgoing RexxMail message(s) */
/**************************************************************************/

parse upper arg Switch  /* get the argument, in upper case */

if (Switch == '') then  /* if we have no switch */
do
 return 0  /* quit */
end

FileName = Global.!ProcFile  /* the file to process */
UseDir = ''  /* start with nothing */

select  /* do one of the following */

 when (left(Switch,12) == 'ATTACHTOOPEN') then  /* if we want to use all the open messages on the desktop */
 do

  if (sysqueryswitchlist('ListItems.') == 0) then  /* if we can get a switch list */
  do Index = 1 to ListItems.0  /* take each of the items found  */

   EditFile = filespec('N',ListItems.Index)  /* try to extract a file name from the list item */

   if ((left(EditFile,4) == 'RXML') & (right(EditFile,5) == '.EDIT') & (datatype(substr(EditFile,5,4),'W'))) then  /* if we find a RexxMail edit file name */
   do
    call MakeAttCopy FileName,Global.!TempDir||'\'||left(EditFile,8)  /* attempt the actual copy */
   end

  end

 end

 when (Switch == 'ATTACHTOALLINDRAFTS') then  /* if we want to use the attachment folder of all messages in the Drafts folder */
 do
  UseDir = Global.!Draftsdir  /* use this dir later on */
 end

 when (Switch == 'ATTACHTOALLINOUT') then  /* if we want to use the attachment folder of all messages in the Out folder */
 do
  UseDir = Global.!Outdir  /* use this dir later on */
 end

 otherwise  /* if none of the above */
 do
  call AddError 'Invalid switch value: "'||Switch||'"'  /* report */
  return 0  /* return with no success */
 end

end

if (UseDir >< '') then  /* if we have a dir name to use */
do

 call sysfiletree UseDir||'\*','Files.','FO'  /* look for RexxMail files in the indicated folder */

 do Index = 1 to Files.0  /* take each of the files found */

  if (MessageSettings(Files.Index,,'CHECK')) then  /* if the file is a RexxMail message */
  do

   if (MessageSettings(Files.Index,'0*******','MATCH')) then  /* if the message is outgoing */
   do

    AttDir = AttDirGet(Files.Index,1)  /* get an attachments dir, creating it if necessary */

    if (AttDir >< '') then  /* if we got one */
    do
     call MakeAttCopy FileName,AttDir,Files.Index  /* attempt the actual copy */
    end

   end

  end

 end

end

return 1  /* end of CopyAttachment */

/**************************************************************************/
CreateCWMFObject: procedure expose Global.  /* creates a CWMF object */
/**************************************************************************/

parse arg NamePart,DestDir  /* get the arguments */

Res = syscreateobject('CWMailFile',NamePart,DestDir,,'FAIL')  /* try to create a CWMAILFile class object in the destination dir using the name part -- simply fail if the class does not exist */

return Res  /* end of CreateCWMFObject */

/**************************************************************************/
DateApprox: procedure expose Global.  /* calculates a rough estimates of days since 1900-01-01 */
/**************************************************************************/

parse arg DateStr  /* get the argument */

DateDays = ''  /* start with nothing */

do while (DateStr >< '')  /* as long as we have something */
 parse var DateStr NextBit '-' DateStr  /* get the next bit */
 DateDays = DateDays||NextBit  /* add it to what we have */
end

return ((left(DateDays,4) * 365) + (substr(DateDays,5,2) * 30) + (substr(DateDays,7,2)))  /* end of DateApprox */

/**************************************************************************/
DateTimeDisplay: procedure expose Global.  /* rewrites a date/time stamp string if necessary */
/**************************************************************************/

parse arg OrgStamp,TimeTypes  /* get the arguments */

if (OrgStamp == '') then  /* if we have no date/time stamp */
do
 return ''  /* return nothing */
end

DayNames = 'Mon Tue Wed Thu Fri Sat Sun'  /* the names of the weekdays */
PrevDayName = ''  /* we have no previous day name yet */
MonthNames = 'Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec'  /* the names of the months */
parse var OrgStamp DayName Rest  /* get the day name */

if (right(DayName,1) == ',') then  /* if a day name reference was included, it should be followed by a comma */
do

 DayName = left(DayName,3)  /* get the first 3 characters of the day name */
 DayPos = wordpos(translate(DayName),translate(DayNames))  /* look for a semblance of the day name in the day names */

 if (DayPos > 0) then  /* if we find the day name */
 do
  DayName = word(DayNames,DayPos)  /* get the valid name */
 end
 else  /* if we do not find the day name */
 do
  DayName = ''  /* use nothing */
 end

 parse var Rest Day Rest  /* get the numerical day of the month */

end
else  /* if there was no day name reference */
do
 Day = DayName  /* the first bit was the numerical day of the month */
 DayName = ''  /* we have no day name */
end

parse var Rest MonthName Year Time OffsetStr  /* extract the other bits we need */
MonthName = left(MonthName,3)  /* get the first 3 characters of the month name */
Month = wordpos(translate(MonthName),translate(MonthNames))  /* look for a semblance of the month name in the month names */

if (Month == 0) then  /* if we do not find the month name */
do
 return OrgStamp  /* simply return the orignal */
end

parse var Time Hour ':' Minute ':' Second  /* get the time stamp components */

if (\datatype(Year||Day||Hour||Minute||Second,'W')) then  /* if the year, the day, or the time is not a whole number, the d/t stamp is a dud */
do
 return OrgStamp  /* simply return the orignal */
end

if (Year < 1000) then  /* if the year has less than 4 digits */
do

 if (Year < 50) then  /* if it's less than 50 */
 do
  Year = Year + 2000  /* add 2000 */
 end
 else  /* if it's 50 or over */
 do
  Year = Year + 1900  /* add 1900 */
 end

end

DateTimeStamp = ''  /* start with nothing */

do while (TimeTypes >< '')  /* go on while we have types left */

 if (DateTimeStamp >< '') then  /* if we already have something */
 do
  DateTimeStamp = DateTimeStamp||' '  /* add a space */
 end

 NewYear = Year  /* we have no new year value yet */
 NewMonth = Month  /* we have no new month value yet */
 NewDay = Day  /* we have no new day value yet */
 NewTime = Time  /* we have no new time value yet */
 NewDayName = DayName  /* we have no new day name value yet */
 parse var TimeTypes TimeType TimeTypes  /* get the next one */

 do while (pos(left(TimeType,1),Global.!ExtraCharacters) > 0)  /* as long as the first character is an extra */
  parse var TimeType AddChar +1 TimeType  /* separate the first character */
  DateTimeStamp = DateTimeStamp||AddChar  /* and add it to the output string */
 end

 if (TimeType >< '') then  /* if we have something left */
 do

  AddDateTimeStamp = ''  /* start with nothing */

  do while (pos(right(TimeType,1),Global.!ExtraCharacters) > 0)  /* as long as the last character is an extra */
   CutPoint = length(TimeType) - 1  /* where to cut */
   parse var TimeType TimeType +(CutPoint) AddChar  /* get the last character */
   AddDateTimeStamp = AddDateTimeStamp||AddChar  /* and add it to the additional output string */
  end

  ISODate = (right(TimeType,3) == 'ISO')  /* see if we want an ISO sorted date */
  TimeType = left(TimeType,1)  /* use just the initial of the time type */

  if (TimeType >< 'O') then  /* if we want UTC or system time rather than the original time */
  do

   if (OffsetStr >< '') then  /* if we have a time offset and perhaps more */
   do

    parse var OffsetStr Offset Rest  /* get what should be the time offset */

    if ((length(Offset) >< 5) | (pos(left(Offset,1),'+-') == 0) | (\datatype(right(Offset,4),'W'))) then  /* if the offset indicator is not of the [+/-]nnnn type */
    do

     OffsetN = '  GMT    UT   UTC   EDT   EST   CDT   CST   MDT   MST   PDT   PST'  /* non-numerical time offset indicators */
     Offsets = '+0000 +0000 +0000 -0400 -0500 -0500 -0600 -0600 -0700 -0700 -0800'  /* the equivalent offsets */
     OffsetPos = wordpos(Offset,OffsetN)  /* look for the offset string in our array */

     if (OffsetPos > 0) then  /* if we find it */
     do
      Offset = word(Offsets,OffsetPos)  /* convert the alpha offset to a numerical offset */
     end
     else  /* if we do not find it */
     do
      Offset = ''  /* use nothing */
     end

    end

   end
   else  /* if we have no time offset indicator at all */
   do
    Offset = ''  /* we have no offset either */
   end

   if (Offset >< '') then  /* if we have a valid offset */
   do

    if (TimeType = 'S') then  /* if we want local system time */
    do

     Offset = Offset + GetTimeZone(1)  /* get the local system time zone, contacting a time server if necessary, and add it to the message offset */

     if (Offset < 0) then  /* if the offset is negative */
     do
      Offset = '-'||right(substr(Offset,2),4,'0')  /* use this */
     end
     else  /* if the offset is zero or positive */
     do
      Offset = '+'||right(Offset,4,'0')  /* use this */
     end

    end

    if (Offset >< 0) then  /* if we have an actual offset */
    do

     MonthDays = '31 28 31 30 31 30 31 31 30 31 30 31'  /* the no. of days in each month */
     parse var Offset OffsetSign +1 OffsetHour +2 OffsetMinute .  /* get the offset components */

     if (OffsetSign = '-') then  /* if the offset is negative */
     do

      Hour = Hour + OffsetHour  /* add the offset hours */
      Minute = Minute + OffsetMinute  /* add the offset minutes */

      do while (Minute > 59)  /* as long as we run over the minute limit */
       Minute = Minute - 60  /* rewrite the minute */
       Hour = Hour + 1  /* add an hour */
      end

      do while (Hour > 23)  /* as long as we run over the hour limit */

       Hour = Hour - 24  /* rewrite the hour */
       NewDay = NewDay + 1  /* add a day */

       if (NewDayName >< '') then  /* if we have a day name */
       do
        NewDayName = word(DayNames||' Mon',wordpos(NewDayName,DayNames) + 1)  /* get the new day name */
       end

       MonthLen = word(MonthDays,NewMonth)  /* the no. of days in the date stamp month */

       if ((NewMonth == 2) & (LeapYear(NewYear))) then  /* if it is Feb in a leap year */
       do
        MonthLen = MonthLen + 1  /* add a day */
       end

       if (NewDay > MonthLen) then  /* if we run over the day limit */
       do

        if (NewMonth == 12) then  /* if it is in December */
        do
         NewMonth = 1  /* skip to the first month */
         NewYear = NewYear + 1  /* move on a year */
        end
        else  /* if it is not in Dec */
        do
         NewMonth = NewMonth + 1  /* just move up a month */
        end

        NewDay = 1  /* reset the day to 1 */

       end

      end

     end
     else  /* if the offset is positive */
     do

      Hour = Hour - OffsetHour  /* subtract the offset hours */
      Minute = Minute - OffsetMinute  /* subtract the offset minutes */

      do while (Minute < 0)  /* as long as we run below the minute limit */
       Minute = 60 + Minute  /* rewrite the minute */
       Hour = Hour - 1  /* subtract an hour */
      end

      do while (Hour < 0)  /* as long as we run below the hour limit */

       Hour = 24 + Hour  /* rewrite the hour */
       NewDay = NewDay - 1  /* subtract a day */

       if (NewDayName >< '') then  /* if we have a day name */
       do
        NewDayName = word('Sun '||DayNames,wordpos(NewDayName,DayNames))  /* get the new day name this way */
       end

       if (NewDay < 1) then  /* if we run below the day limit */
       do

        if (NewMonth == 1) then  /* if the month is Jan */
        do
         NewMonth = 12  /* skip to Dec */
         NewYear = NewYear - 1  /* go back a year */
        end
        else  /* if it is not Jan */
        do
         NewMonth = NewMonth - 1  /* just go back a month */
        end

        NewDay = word(MonthDays,NewMonth)  /* the no. of days in the new date stamp month */

        if ((NewMonth == 2) & (LeapYear(NewYear))) then  /* if it is Feb in a leap year */
        do
         NewDay = NewDay + 1  /* add a day */
        end

       end

      end

     end

     NewTime = right(Hour,2,'0')||':'||right(Minute,2,'0')||':'||right(Second,2,'0')  /* the new time stamp */

    end

   end

  end

  if (ISODate) then  /* if we want an ISO date */
  do
   NewStamp = NewYear||'-'||right(NewMonth,2,'0')||'-'||right(NewDay,2,'0')||' '||NewTime  /* this is our new d/t stamp */
  end
  else  /* if we want an RFC date */
  do
   NewStamp = NewDay||' '||word(MonthNames,NewMonth)||' '||NewYear||' '||NewTime  /* this is our new d/t stamp */
  end

  select  /* do one of the following */

   when (TimeType == 'O') then  /* if we want the original time */
   do

    if (OffsetStr >< '') then  /* if we have an offset (and perhaps more) */
    do
     NewStamp = NewStamp||' '||OffsetStr  /* add it as it is */
    end

   end

   when (TimeType == 'U') then  /* if we want UTC time */
   do
    NewStamp = NewStamp||' UTC'  /* add this */
   end

   otherwise  /* if none of the above */
   do
    nop  /* nothing */
   end

  end

  if (NewDayName >< PrevDayName) then  /* if we have a new day name that is not the same as the previous day name (always, the first time round) */
  do

   if (ISODate) then  /* if we want an ISO date */
   do
    NewStamp = NewStamp||' ('||NewDayName||')'  /* add it */
   end
   else  /* if we want an RFC date */
   do
    NewStamp = NewDayName||', '||NewStamp  /* add it */
   end

   PrevDayName = NewDayName  /* save the value for later use */

  end

  DateTimeStamp = DateTimeStamp||NewStamp||AddDateTimeStamp  /* add the results to what we have */

 end

end

return DateTimeStamp  /* end of DateTimeDisplay */

/**************************************************************************/
DateTimeRFC: procedure expose Global.  /* returns RFC date string */
/**************************************************************************/

parse arg ContactServers  /* get the argument */

ContactServers = (ContactServers == 1)  /* 1 is true */
Global.!Settings.TimeZone = GetTimeZone(ContactServers)  /* get and set the local time zone if necessary */
DateString = left(date('W'),3)||', '||date('N')||' '||time('N')||' '||Global.!Settings.TimeZone  /* prepare the message date string */

return DateString  /* end of DateTimeRFC */

/**************************************************************************/
DateTimeSys: procedure expose Global.  /* returns a date/time stamp string */
/**************************************************************************/

parse arg LongTime  /* get the argument */

LongTime = (LongTime == 1)  /* 1 is true */

if (LongTime) then  /* if we want a long time */
do
 TimeType = 'L'  /* use this */
end
else  /* if not */
do
 TimeType = 'N'  /* use this */
end

DateStr = date('S')  /* get the sorted date */
Stamp = left(DateStr,4)||'-'||,  /* start the date & time stamp with the year in 4 digits */
        substr(DateStr,5,2)||'-'||,  /* followed by the month in 2 digits */
        substr(DateStr,7,2)||' '||,  /* followed by the day of the month in 2 digits */
        time(TimeType)  /* followed by the normal or long time */

return Stamp  /* end of DateTimeSys */

/**************************************************************************/
DecodeB64: procedure expose Global.  /* decodes a B64 string */
/**************************************************************************/

parse arg B64Str  /* get the argument */

B64Chars = xrange('A','Z')||xrange('a','z')||xrange('0','9')||'+/'  /* define the base64 character set */
NoChars = '='||Global.!CRLF  /* these are to be skipped */
BitStr = ''  /* start with nothing */

do while (B64Str >< '')  /* as long as we have something left */

 parse var B64Str NextChar 2 B64Str  /* get the next character */

 if (pos(NextChar,NoChars) = 0) then  /* if it is not padding or part of a CRLF */
 do

  CharVal = pos(NextChar,B64Chars)  /* the value represented by the B64 character (plus one) */

  if (CharVal > 0) then  /* if it is a valid code */
  do
   BitStr = BitStr||right(x2b(d2x(CharVal - 1)),6,'0')  /* convert the value (minus one ) to 6 bits and add the result to the bit string */
  end
  else  /* if not */
  do
   call AddError 'Ignored invalid B64 code character: '||NextChar  /* report */
  end

 end

end

TextStr = ''  /* start with nothing */

do while(length(BitStr) > 7)  /* as long as we have at least a whole byte left */
 parse var BitStr NextChar 9 BitStr  /* get the first 8 bits */
 TextStr = TextStr||x2c(b2x(NextChar))  /* convert to a character and add to the result */
end

return TextStr  /* end of DecodeB64 */

/**************************************************************************/
DecodeHeaderLine: procedure expose Global.  /* decodes a header line if neccessary (and if possible) */
/**************************************************************************/

parse arg Text  /* get the argument */

CodeStart = '=?'  /* define the start of an encoded string */
CodeEnd = '?='  /* define the end of an encoded string */
NewText = ''  /* start with nothing */

do while (Text >< '')  /* as long as we have text left */

 NewStr = ''  /* start with nothing */
 parse var Text FrontBit (CodeStart) CharSet '?' Encoding '?' CodedBit (CodeEnd) Text  /* extract the bits we want */

 if (CharSet >< '') then  /* if we found a character set code, encoded text follows */
 do

  CharSet = translate(CharSet)  /* upper case */

  if (translate(Encoding) == 'Q') then  /* if the encoding is quoted-printable */
  do

   do while (CodedBit >< '')  /* as long as we have coded content */

    parse var CodedBit NextChar 2 CodedBit  /* get the first character of what we have left */

    if (NextChar == '=') then  /* if it is a "=" */
    do

     parse var CodedBit NextChar 3 CodedBit  /* get the next two characters */

     if (verify(translate(NextChar),'1234567890ABCDEF','NOMATCH') == 0) then  /* if the NextChar bit is a hex code */
     do
      NewBit = x2c(NextChar)  /* convert hex to character */
     end
     else  /* if not, we have malformed code */
     do
      NewBit = '_'  /* use this */
     end

    end
    else  /* if it is not a "=" */
    do
     NewBit = translate(NextChar,' ','_')  /* turn an underscore into a space */
    end

    NewStr = NewStr||NewBit  /* add the result to what we have */

   end

  end
  else  /* if it is B-64 */
  do
   NewStr = DecodeB64(CodedBit)  /* decode it */
  end

  select  /* do one of the following */

   when (CharSet == 'US-ASCII') then  /* if the character set is US-ASCII */
   do
    NewStr = translate(NewStr,' ','_')  /* decode underscores into spaces */
   end

   when (CharSet == 'UTF-8') then  /* if the character set is UTF-8 */
   do
    NewStr = DecodeUTF8(NewStr)  /* decode it */
    NewStr = translate(NewStr,Global.!FilterIn,xrange('80'x,'FF'x))  /* convert high bits from PC819 (ISO 8859-1) to the local code page */
   end

   when (CharSet == 'ISO-8859-1') then  /* if the character set is ISO-8859-1 */
   do
    NewStr = translate(NewStr,Global.!FilterIn,xrange('80'x,'FF'x))  /* convert high bits from PC819 (ISO 8859-1) to the local code page */
   end

   when (pos('WINDOWS-',CharSet) > 0) then  /* if the character set is a Windows subset */
   do
    NewStr = translate(NewStr,Global.!PC1004_PC819,xrange('80'x,'FF'x))  /* convert high bits from PC1004 to PC819 (ISO 8859-1) */
    NewStr = translate(NewStr,Global.!FilterIn,xrange('80'x,'FF'x))  /* convert high bits from PC819 (ISO 8859-1) to the local code page */
   end

   otherwise  /* if none of the above apply */
   do
    NewStr = '(...)'  /* we have nothing to show */
   end

  end

 end

 NewText = NewText||FrontBit||NewStr  /* add the bits we have */

end

NewText = strip(NewText,'B',' ')  /* remove excess spaces */

return NewText  /* end of DecodeHeaderLine */

/*********************************************************************/
DecodeUTF8: procedure expose Global.  /* converts UTF-8 text to ISO-8859-1 */
/*********************************************************************/

parse arg TextUTF  /* get the argument */

BadBytes = x2c('C0')||x2c('C1')||xrange('F7'x,'FF'x)  /* these should not be in any UTF-8 encoded text */
BadVals = '55296 56191 56192 56319 56320 57216 57343 65534 65535'  /* bad UCS values */
BytesToGo = 0  /* how many bytes left in a high-bit sequence */
BinBits = ''  /* the binary represention of a character */
TextISO = ''  /* the resulting text string */
GotBad = 0  /* nothing wrong yet */

call GetDefinitions Global.!ProgDir||'\ucs.txt','!UCSList',1  /* collect the UCS data if necessary */

do Index = 1 to length(TextUTF)  /* for each byte of the input string */

 NewChar = ''  /* nothing yet */
 UTFChar = substr(TextUTF,Index,1)  /* take the next byte */

 select  /* do one of the following */

  when (pos(UTFChar,BadBytes) > 0) then  /* if it is a bad one */
  do
   GotBad = 1  /* set a flag */
  end

  when (c2d(UTFChar) < 128) then  /* if it is ASCII */
  do

   if (BytesToGo > 0) then  /* if we are in mid-sequence */
   do
    GotBad = 1  /* set a flag */
   end

   NewChar = UTFChar  /* use the ASCII character */

  end

  otherwise  /* if neither of the above, it is a high-bit byte */
  do
  
   BinChar = x2b(c2x(UTFChar))  /* make it binary */
   parse var BinChar HiBits '0' LoBits  /* cut at the first 0 */

   select  /* do one of the following */

    when (length(HiBits) > 6) then  /* if we have more than 6 high bits */
    do
     GotBad = 1  /* set a flag */
    end

    when (length(HiBits) > 1) then  /* if we have a starter byte */
    do

     if ((BytesToGo > 0) | (LoBits == 0)) then  /* if we are in mid-sequence, or we have no low bits */
     do
      GotBad = 1  /* set a flag */
     end
     else  /* if all is well */
     do
      BinBits = LoBits  /* save the low bits as binary stuff */
      BytesToGo = (length(HiBits) - 1)  /* we will be looking for so many continuation bytes */
     end

    end

    otherwise  /* if neither of the above, this is a continuation byte */
    do

     if ((BytesToGo == 0) | ((BytesToGo > 1) & (BinBits == 0))) then  /* if we are not in a sequence, or we are in mid-sequence and there is no previous binary stuff */
     do
      GotBad = 1  /* set a flag */
     end
     else  /* if all is well */
     do

      BinBits = BinBits||LoBits  /* add the low bits to the binary stuff */
      BytesToGo = BytesToGo - 1  /* one less byte to go */

      if ((BytesToGo == 0) & (\GotBad)) then  /* if we are at the end of a sequence and we are not processing a rotten sequence */
      do

       UCSVal = x2d(b2x(BinBits))  /* convert the binary stuff to a decimal number */
       BinBits = ''  /* nothing left to convert */

       if (UCSVal < 256) then  /* if it is inside our range */
       do
        NewChar = d2c(UCSVal)  /* convert to ISO-8859-1 */
       end
       else  /* if it is a high UCS value */
       do

        if (wordpos(UCSVal,BadVals) > 1) then  /* if the value is in the bad values */
        do
         GotBad = 1  /* set a flag */
        end
        else  /* if the value looks O.K. */
        do

         if (symbol('Global.!UCSList.'||UCSVal) == 'VAR') then  /* if we have this one */
         do

          NewChar = Global.!UCSList.UCSVal  /* look up the character in the UCS list */
          parse var NewChar NewChar '#' Description  /* look for the components */
          NewCHar = strip(NewChar)  /* remove excess whitespace */
          Description = strip(Description)  /* remove excess whitespace */

          if (NewChar == '') then  /* if we found no character */
          do

           if (Description == '') then  /* if there is no description either */
           do
            NewChar = '_'  /* use this */
           end
           else  /* if we have just a description */
           do
            NewChar = '['||Description||']'  /* use this */
           end

          end

         end
         else  /* if this one is not in the list */
         do
          NewChar = '_'  /* use this */
         end

        end

       end

      end

     end

    end

   end

  end

 end

 if (length(NewChar) > 0) then  /* if we have new text */
 do

  if (GotBad) then  /* if we found a bad one */
  do
   TextISO = TextISO||'[?]'  /* add a marker */
   BinBits = ''  /* nothing to convert */
   BytesToGo = 0  /* nothing left to do */
   GotBad = 0  /* reset the bad text flag */
  end

  TextISO = TextISO||NewChar  /* add the new character */

 end

end

return TextISO  /* end of DecodeUTF8 */

/**************************************************************************/
DirCheck: procedure expose Global.  /* checks the existence of a directory */
/**************************************************************************/

parse arg DirSpec,Warning  /* get the argument */

Warning = (Warning == 1)  /* 1 = true */
CurDir = directory()  /* store the current dir */

if (directory(DirSpec) == '') then  /* if we cannot change to the specified dir */
do

 if (Warning) then  /* unless we want a quiet check */
 do
  call AddError 'Cannot find directory "'||DirSpec||'"'  /* report an error */
 end

 return 0  /* return no result */

end
else  /* if we did change to the specified dir */
do
 call directory CurDir  /* change back to the original dir */
end

return 1  /* end of DirCheck */

/**************************************************************************/
EditMessage: procedure expose Global.  /* edits address template or outgoing message file  */
/**************************************************************************/

MessFile = Global.!ProcFile  /* get the name of the file to edit */

if (\MessageSettings(MessFile,'0*******','MATCH')) then  /* if the file is not an outgoing message */
do
 call AddError 'Invalid message type'  /* report */
 return 0  /* and quit */
end

if (MessageSettings(MessFile,'**1*****','MATCH')) then  /* if the file has been sent, we are dealing with an old version */
do
 call ViewMessage MessFile  /* call the viewer routine */
 return 1  /* and quit */
end

MessageDir = strip(filespec('D',MessFile)||filespec('P',MessFile),'T','\')  /* the directory the message file is in */

if (pos(translate(Global.!AddressesDir),translate(MessageDir)) > 0) then  /* if it's an address folder file */
do
 call RunCommand Global.!Settings.RunEdit,MessFile  /* just run the edit command */
 return 1  /* and quit */
end

call SetObjectData MessFile,'OBJECTID='  /* reset any remaining object ID */
call MessageSettings MessFile,'*0******','CHANGE'  /* make the file not O.K. to send */

if (Global.!Settings.SendASCII) then  /* if all mail is to be sent as ASCII text */
do
 call MessageSettings MessFile,'***1****','CHANGE'  /* set the ASCII flag */
end

MessCont = GetFileContents(MessFile)  /* get the message file contents */

if (\MessageContents(MessCont,'Global.!MessHead.','Global.!MessBody')) then  /* if we cannot get the message contents sorted out */
do
 return 0  /* quit */
end

if (\HeaderContains('FROM',,'Global.!MessHead.')) then  /* if we have no "From:" line */
do
 call InsertHeaderLine 'Global.!MessHead.','From','"'||Global.!Settings.Name||'" <'||Global.!Settings.Address||'>',1  /* insert this at line 1 */
end

if (\HeaderContains('TO',,'Global.!MessHead.')) then  /* if we have no "To:" line */
do
 call InsertHeaderLine 'Global.!MessHead.','To','',2  /* insert this at line 2 */
end

if (\HeaderContains('SUBJECT',,'Global.!MessHead.')) then  /* if we have no "Subject:" line */
do
 call InsertHeaderLine 'Global.!MessHead.','Subject','',3  /* insert this at line 3 */
end

if ((\HeaderContains('REPLY-TO',,'Global.!MessHead.')) & (Global.!Settings.ReplyAddress >< '')) then  /* if we have no "Reply-to:" line, and we have a default reply address */
do

 AddText = '<'||Global.!Settings.ReplyAddress||'>'  /* start with his text */

 if (Global.!Settings.ReplyName >< '') then  /* if we have a reply name */
 do
  AddText = '"'||Global.!Settings.ReplyName||'" '||AddText  /* add the name in front of the address */
 end

 call InsertHeaderLine 'Global.!MessHead.','Reply-To',AddText,2  /* insert the result as a Reply-To header entry at line 2 */

end

AttDir = AttDirGet(MessFile,1)  /* get the attachments dir, creating it if necessary  */

if (AttDir == '') then  /* if we still have no att dir */
do
 return 0  /* quit */
end

AddDirTo = AddressMakeAddDir(AttDir,'TO')  /* create this folder if required */
AddDirCc = AddressMakeAddDir(AttDir,'CC')  /* create this folder if required */
AddDirAcc = AddressMakeAddDir(AttDir,'ACC')  /* create this folder if required */
AddDirBcc = AddressMakeAddDir(AttDir,'BCC')  /* create this folder if required */

if (Global.!Settings.OpenAttachBeforeEdit) then  /* if we want the folder open on the desktop */
do

 if (\sysopenobject(AttDir,0,1)) then  /* if we cannot open the attachments folder */
 do
  call AddError 'Cannot open attachments folder "'||AttDir||'"'  /* report */
 end

end

EditFile = Global.!TempDir||'\'||filespec('N',AttDir)||' (EDIT) '||CheckCommLine(strip(left(GetHeaderEntry('SUBJECT','Global.!MessHead.'),42)),':')  /* a matching edit file name in the temp dir */

if (\HeaderCheck()) then  /* if we cannot check the header entries */
do
 return 0  /* quit */
end

if (\PutFileContents(EditFile,FoldHeaderLines('Global.!MessHead.','','E')||Global.!CRLF||Global.!MessBody)) then  /* if we cannot write the edit text to the edit file after arranging the header for editing */
do
 return 0  /* just quit */
end

call sysfiletree EditFile,'Files.','FO','+****','-****'  /* reset the archive attribute of the edit file */
call RunCommand Global.!Settings.RunBeforeEdit,EditFile  /* try to run a command before editing */

if (Global.!Settings.EditCheck) then  /* if we want a conditional "O.K. to send" setting later on */
do
 call sysfiletree EditFile,'Files.','FO','','-****'  /* reset the Archive attribute */
end

ButtonID = MessBarAdd(MessFile,AttDir,AddDirTo,AddDirCc,AddDirAcc,AddDirBcc)  /* add the new file and folder to the message buttons bar */
call RunCommand Global.!Settings.RunEdit,EditFile  /* try to run the edit command */
call MessBarRemove ButtonID  /* delete the file and folder from the message buttons bar */
call RunCommand Global.!Settings.RunAfterEdit,EditFile  /* try to run a command after editing */

if ((\FileCheck(MessFile)) | (\FileCheck(EditFile))) then  /* if either file has ceased to exist */
do
 return 1  /* we're done */
end

KeepAdds = 0  /* reset a warning flag */
ToAdds = ''  /* start with nothing */

if (AddDirTo >< '') then  /* if we have this dir */
do

 ToAdds = AddressGetAdds(AttDir||'\'||AddDirTo)  /* look for additional To addresses */

 if (left(ToAdds,1) == Global.!Warning) then  /* if the addresses string starts with a warning marker */
 do
  ToAdds = substr(ToAdds,2)  /* remove it */
  KeepAdds = 1  /* set a warning flag */
 end

end

CcAdds = ''  /* start with nothing */

if (AddDirCc >< '') then  /* if we have this dir */
do

 CcAdds = AddressGetAdds(AttDir||'\'||AddDirCc)  /* look for additional Cc addresses */

 if (left(CcAdds,1) == Global.!Warning) then  /* if the addresses string starts with a warning marker */
 do
  CcAdds = substr(CcAdds,2)  /* remove it */
  KeepAdds = 1  /* set a warning flag */
 end

end

AccAdds = ''  /* start with nothing */

if (AddDirAcc >< '') then  /* if we have this dir */
do

 AccAdds = AddressGetAdds(AttDir||'\'||AddDirAcc)  /* look for additional Acc addresses */

 if (left(AccAdds,1) == Global.!Warning) then  /* if the addresses string starts with a warning marker */
 do
  AccAdds = substr(AccAdds,2)  /* remove it */
  KeepAdds = 1  /* set a warning flag */
 end

end

BccAdds = ''  /* start with nothing */

if (AddDirBcc >< '') then  /* if we have this dir */
do

 BccAdds = AddressGetAdds(AttDir||'\'||AddDirBcc)  /* look for additional Bcc addresses */

 if (left(BccAdds,1) == Global.!Warning) then  /* if the addresses string starts with a warning marker */
 do
  BccAdds = substr(BccAdds,2)  /* remove it */
  KeepAdds = 1  /* set a warning flag */
 end

end

if (Global.!Settings.EditCheck) then  /* if we want a conditional "O.K. to send" setting */
do
 call sysfiletree EditFile,'Files.','FO','+****'  /* see if the Archive attribute has been set */
 OKToSend = (Files.0 == 1)  /* if it has, the message is O.K. to send */
end
else  /* if we don't care if the message was edited or not */
do
 OKToSend = 1  /* it is always O.K. to send -- at least, for now */
end

GotAtt = AttDirClose(AttDir)  /* close the attachments dir; we get 1 = TRUE in return if there are any contents left */
Settings = MessageSettings(MessFile)  /* get the message settings from the original file (they may have been changed in the meantime) */
NewName = ''  /* we have no new name yet */
MessCont = GetFileContents(EditFile)  /* get the new edit file contents */
call sysfiledelete EditFile  /* get rid of the edit file */

if (\MessageContents(MessCont,'Global.!MessHead.','Global.!MessBody')) then  /* if we cannot get the new edit message contents sorted out */
do
 return 0  /* quit with an error */
end

if (\HeaderCheck(ToAdds,CcAdds,AccAdds,BccAdds)) then  /* if we cannot check the header entries and add any extra recipients */
do
 return 0  /* quit */
end

MessHead = FoldHeaderLines('Global.!MessHead.','','P')  /* arrange the header content (post-editing) */
MessCont = MessHead||Global.!CRLF||Global.!MessBody  /* the new message content */
call AttDirUnLink MessFile  /* unlink the attachments dir from the old message file */
call sysfiledelete MessFile  /* get rid of the original message file */

if (\PutFileContents(MessFile,MessCont)) then  /* if we cannot write the new content to the message file */
do
 return 0  /* quit with an error */
end

NewName = MakeTitle(MessFile,1,KeepAdds,0,1)  /* get a new name for the mail file, and insert warnings as necessary, and do object stuff */
OKToSend = (OKToSend & (pos(Global.!Warning,NewName) == 0))  /* if the title contains no warnings, it remains O.K. to send */

call MessageSettings MessFile,left(Settings,1)||,  /* change the message settings: copy the first position */
                              OKToSend||,  /* set the second position according to the OKToSend flag */
                              substr(Settings,3,2)||,  /* copy the third and fourth positions */
                              GotAtt||,  /* set the fifth position according to the GotAtt flag */
                              right(Settings,3),'CHANGE'  /* copy the sixth, seventh, and eighth positions */
call AttDirLink MessFile,AttDir  /* link the attachments dir to the new message file */

if (NewName >< '') then  /* if we have a new name for the message file */
do
 call SetObjectData MessFile,'REXXMAILREFRESH=YES;TITLE='||NewName  /* rename the file */
end

return 1  /* end of EditMessage */

/**********************************************************************/
EncodeB64: procedure expose Global.  /* encodes a text string */
/**********************************************************************/

parse arg Text  /* get the argument */

B64Chars = xrange('A','Z')||xrange('a','z')||xrange('0','9')||'+/'  /* define the base64 character set */
B64Str = ''  /* start with nothing */

do while (length(Text) > 3)  /* go on while the length is sufficient */

 parse var Text NextBlock 4 Text  /* get the next block of 3 characters */
 NextBits = x2b(c2x(NextBlock))  /* convert it to 24 bits */

 do 4  /* do 4 times */
  parse var NextBits NextSext 7 NextBits  /* get the next sextet */
  B64Str = B64Str||substr(B64Chars,x2d(b2x(NextSext))+1,1)  /* convert to decimal, get the corresponding B64 character, and add */
 end

end

TextLeft = length(Text)  /* the number of 8-bit characters left (1, 2, or 3) */

if (TextLeft > 0) then  /* if we have anything left */
do

 NextBits = x2b(c2x(Text))||copies('00',(3-TextLeft))  /* convert to bits and add zeroes */

 do (TextLeft + 1)  /* do so many times */
  parse var NextBits NextSext 7 NextBits  /* get the next sextet */
  B64Str = B64Str||substr(B64Chars,x2d(b2x(NextSext))+1,1)  /* convert to decimal, get the corresponding B64 character, and add */
 end

 B64Str = B64Str||copies('=',3 - TextLeft)  /* add this */

end

return B64Str  /* end of Encode64 */

/**************************************************************************/
ExpandRecipients: procedure expose Global.  /* expands a list of recipients into real addresses */
/**************************************************************************/

parse arg Recipients,ListNames  /* get the arguments */

ListNames = (ListNames == 1)  /* 1 = TRUE */
GotRecipients = ''  /* nothing yet */

do while (Recipients >< '')  /* as long as we have recipients left */

 parse var Recipients Recipient Recipients  /* get the next address from the list */

 if (pos(':;',Recipient) == 0) then  /* if the address does not contain a group indicator */
 do
  GotRecipients = GotRecipients||CheckRecipients(Recipient,GotRecipients)  /* check the recipient and add it to the list if O.K. */
 end
 else  /* if the address contains a group indicator */
 do

  parse upper var Recipient ListName ':;'.  /* get the list name in upper case */
  ListFile = Global.!ConfigurationDir||'\lists\'||ListName  /* the list file to look for */

  if (FileCheck(ListFile)) then  /* if there is a mail list file of that name */
  do

   call LogAction 'Including mailing list file "'||ListFile||'"'  /* report */
   ListCont = GetFileContents(ListFile)  /* get the contents of the list file */

   do while (ListCont >< '')  /* as long as we have content left */

    parse var ListCont NextLine (Global.!CRLF) ListCont  /* get the next line */

    if (NextLine >< '') then  /* if it is not an empty line */
    do

     if (left(NextLine,1) >< '#') then  /* if it is not a comment */
     do
      GotRecipients = GotRecipients||CheckRecipients(AddressFormat(NextLine,0,1,ListNames),GotRecipients,ListFile)  /* check the new recipients and add to the list if O.K. */
     end

    end

   end

  end
  else  /* if there is no mail list file of that name */
  do

   GotGroup = 0  /* nothing yet */
   call sysfiletree Global.!AddressesDir||'\*','Dirs.','DOS'  /* get all the address (sub)dirs */

   do DirIndex = 1 to Dirs.0  /* take each one we found */

    if (translate(filespec('N',Dirs.DirIndex)) == ListName) then  /* if the group is one we want */
    do

     GotGroup = 1  /* our group is a folder */
     call LogAction 'Including mail group folder "'||Dirs.DirIndex||'"'  /* report */
     call sysfiletree Dirs.DirIndex||'\*','Files.','FOS'  /* look for (template) files in the group dir and any subdirs */

     do Index = 1 to Files.0  /* take each one */
      GotRecipients = GotRecipients||CheckRecipients(GetRecipients(Files.Index),GotRecipients,Files.Index)  /* check the new the new recipients annd add to the check list if O.K. */
     end

    end

   end

   if (\GotGroup) then  /* if we did not find the list dir */
   do
    
    call sysfiletree Global.!AddressesDir||'\*','Files.','FOS'  /* look for (template) files in the address dir and any subdirs */

    do Index = 1 to Files.0  /* take each one */

     Keywords = GetObjectEA(Files.Index,'.KEYWORDS')  /* get the keywords block from the template file, if any */

     if (Keywords >< '') then  /* if we have something */
     do

      Keywords = translate(translate(Keywords),'  ',Global.!CRLF)  /* upper case, and convert CRLF to whitespace */

      if (wordpos(ListName,Keywords) > 0) then  /* if our list name is among the keywords */
      do
       call LogAction 'Including mail group message "'||Files.Index||'"'  /* report */
       GotRecipients = GotRecipients||CheckRecipients(GetRecipients(Files.Index),GotRecipients,Files.Index)  /* check the new recipients and add to the check list if O.K. */
      end

     end

    end

   end

  end

 end

end

return GotRecipients  /* end of ExpandRecipients */

/**************************************************************************/
FileCheck: procedure expose Global.  /* checks the existence of a file */
/**************************************************************************/

parse arg FileName,Warning  /* get the argument */

Warning = (Warning == 1)  /* 1 = true */

if (stream(FileName,'C','QUERY EXISTS') == '') then  /* if the file does not exist */
do

 if (Warning == 1) then  /* if we want a warning */
 do
  call AddError 'Cannot find file "'||FileName||'"'  /* report */
 end

 return 0  /* return no result */

end

return 1  /* end of FileCheck */

/**************************************************************************/
FileClose: procedure expose Global.  /* closes a file */
/**************************************************************************/

parse arg FileName  /* get the argument */

if (stream(FileName,'C','CLOSE') >< 'READY:') then  /* if we cannot close the file */
do
 call AddError 'Cannot close file "'||FileName||'"'  /* report */
 return 0  /* return with an error */
end

return 1  /* end of FileClose */

/**************************************************************************/
FileCopy: procedure expose Global.  /* copies a file */
/**************************************************************************/

parse arg SourceFile,TargetFile  /* get the arguments */

if (\FileOpen(TargetFile)) then  /* if we cannot open the source file for writing */
do
 return 0  /* quit with no success */
end

if (\FileOpen(SourceFile,'READ')) then  /* if we cannot open the source file for reading */
do
 return 0  /* quit with no success */
end

if (charout(TargetFile,charin(SourceFile,1,chars(SourceFile)),1) > 0) then  /* if we cannot write all the content to the file */
do
 call AddError 'Cannot write contents of "'||SourceFile||'" to "'||TargetFile||'"'  /* report */
 return 0  /* and quit */
end

call FileClose SourceFile  /* close the source file */
call FileClose TargetFile  /* close the source file */

return 1  /* end of FileCopy */

/**************************************************************************/
FileOpen: procedure expose Global.  /* attempts to open a file */
/**************************************************************************/

parse arg FileName,Operation  /* get the arguments */

if (FileName == '') then  /* if we have no file name */
do
 call AddError 'No file name to open'  /* report */
 return 0  /* quit with no success */
end

Attempts = 0  /* we haven't tried yet */
MaxAttempts = 20  /* stop after so many tries */
Delay = 0.005  /* start with this delay */

if (Operation >< '') then  /* if we have a specific operation */
do
 Operation = ' '||Operation  /* add a space */
end

do while ((stream(FileName,'C','OPEN'||Operation) >< 'READY:') & (Attempts < MaxAttempts))  /* go on until we succeed or reach the maximum no. of attempts */
 Attempts = Attempts + 1  /* up the attempts counter */
 call syssleep (Delay * Attempts)  /* wait a bit */
end

if (Attempts == MaxAttempts) then  /* if we reached the last try */
do
 call AddError 'Cannot open file "'||FileName||'"'  /* report */
 return 0  /* no success */
end

return 1  /* end of FileOpen */

/**************************************************************************/
FoldersClose: procedure expose Global.  /* closes the RexxMail message folders and resets the In folder icon if necessary */
/**************************************************************************/

call RunCommand Global.!Settings.RunBeforeClose  /* see if we can run an external command before closing the folders */
call Toolbar 'DELETE','Message Buttons',1  /* remove any remaining mail buttons toolbar */
call sysfiledelete Global.!TempDir||'\Mail_Bar.LST'  /* remove the mail bar list file */
call RexxMailCloseObject filespec('N',Global.!MainDir)  /* close the main folder -- EXTERNAL -- RXMLUTIL.DLL */
call RexxMailCloseObject filespec('N',Global.!AddressesDir)  /* close the addresses folder -- EXTERNAL -- RXMLUTIL.DLL */
call RexxMailCloseObject filespec('N',Global.!ConfigurationDir)  /* close the configuration folder -- EXTERNAL -- RXMLUTIL.DLL */
call RexxMailCloseObject filespec('N',Global.!In_ArchiveDir)  /* close the in archive folder -- EXTERNAL -- RXMLUTIL.DLL */
call RexxMailCloseObject filespec('N',Global.!InDir)  /* close the in folder -- EXTERNAL -- RXMLUTIL.DLL */
call RexxMailCloseObject filespec('N',Global.!OutDir)  /* close the out archive folder -- EXTERNAL -- RXMLUTIL.DLL */
call RexxMailCloseObject filespec('N',Global.!Out_ArchiveDir)  /* close the out folder -- EXTERNAL -- RXMLUTIL.DLL */

if (Global.!DraftsDir >< '') then  /* if we have a drafts folder */
do
 call RexxMailCloseObject filespec('N',Global.!DraftsDir)  /* close the drafts folder -- EXTERNAL -- RXMLUTIL.DLL */
end

call LogAction 'Closed user folders',1  /* report, quietly */
call RunCommand Global.!Settings.RunAfterClose  /* see if we can run an external command after closing the folders */

return 1  /* end of FoldersClose */

/**************************************************************************/
FoldersOpen: procedure expose Global.  /* opens the RexxMail message folders */
/**************************************************************************/

call WarnIcon 'HIDE',1  /* hide the mail warning program object, if any (no alarm bells) */
call RunCommand Global.!Settings.RunBeforeOpen  /* see if we need to run an external command before opening the folders */

do while (Global.!Settings.FoldersOpen >< '')  /* as long as we have folder values */

 parse upper var Global.!Settings.FoldersOpen NextKeyword Global.!Settings.FoldersOpen  /* get the next value in upper case */

 select  /* do one of the following */

  when (NextKeyword == 'INARCHIVE') then  /* if we have this one */
  do
   NextKeyword = 'IN_ARCHIVE'  /* insert an underscore */
  end

  when (NextKeyword == 'OUTARCHIVE') then  /* if we have this one */
  do
   NextKeyword = 'OUT_ARCHIVE'  /* insert an underscore */
  end

  otherwise  /* if neither of the above */
  do
   nop  /* zilch */
  end

 end

 OpenFolder = value('Global.!'||NextKeyword||'Dir')  /* the corresponding folder */

 if (OpenFolder >< '') then  /* if the folder exists (the drafts folder may not be present) */
 do

  if (\sysopenobject(OpenFolder,0,1)) then  /* if we cannot open the folder on the desktop */
  do
   call AddError 'Cannot open folder "'||OpenFolder||'"'  /* report */
  end

 end

end

call LogAction 'Opened user folders',1  /* report, quietly */
call ToolBar 'OPEN',,1  /* try to open the default toolbar, no warning bells */
call RunCommand Global.!Settings.RunAfterOpen  /* see if we need to run an external command after opening the folders */
call sysfiletree Global.!In_ArchiveDir||'\*','Dirs.','DOS'  /* look for In Archive subdirs */

do Index = 1 to Dirs.0  /* take each one */
 call ResetFolderIcon Dirs.Index  /* reset the folder icon if necessary */
end

call ResetFolderIcon Global.!In_ArchiveDir  /* reset the In Archive icon if necessary */
call TrimLogFile Global.!ActionLog,Global.!Settings.LogActionLines  /* trim if necessary */
call TrimLogFile Global.!ErrorLog,Global.!Settings.LogErrorLines  /* trim if necessary */
call TrimLogFile Global.!MailLog,Global.!Settings.LogMailLines  /* trim if necessary */
call CleanUpTemp  /* clean up temporary files and folders */

return 1  /* end of FoldersOpen */

/**************************************************************************/
FoldHeaderLines: procedure expose Global.  /* gets header info from original message file and returns folded and indented header text block */
/**************************************************************************/

parse arg HeadName,Keywords,Mode  /* get the arguments */

Header. = ''  /* start with nothing */
HeadLines = ''  /* start with nothing */
Reformats = 'FROM TO REPLY_TO CC ACC BCC APPARENTLY_FROM SENDER RESENT_FROM RESENT_TO RESENT_CC RESENT_BCC'  /* reformat these (@ddresses) */
DateLines = 'DATE RESENT_DATE'  /* reformat these if required (date/time stamps) */
Receivers = 'RECEIVED'  /* reformat these if required (route and date/time stamps) */

if (Keywords == '') then  /* if we do not have a list of keywords */
do

 HeadCount = value(HeadName||'0')  /* use all the existing header lines */

 do Index = 1 to HeadCount  /* for each line */
  Keyword = value(HeadName||Index||'.!Keyword')  /* the keyword */
  Header.Index.!Keyword = value(HeadName||Index||'.!Keyword')  /* copy the keyword */
  Header.Index.!Entry = value(HeadName||Index||'.!Entry')  /* copy the entry */
 end

end
else  /* if we have a list of keywords */
do

 HeadCount = 0  /* start a counter */

 do while (Keywords >< '')  /* while we have something left in this variable */

  parse var Keywords Keyword Keywords  /* get the next word from the list of keywords, in upper case */

  if (left(Keyword,1) == '*') then  /* if it starts with "*" */
  do
   Keyword = substr(Keyword,2)  /* use the rest */
   More = 1  /* we want more than the first occurrence */
  end
  else  /* if the word does not start with "*" */
  do
   More = 0  /* we don't want more */
  end

  Entry = GetHeaderEntry(translate(Keyword),HeadName,More)  /* look for this entry in the message header */

  if (Entry >< '') then  /* if we have an entry */
  do
   HeadCount = HeadCount + 1  /* up the counter */
   Header.HeadCount.!Keyword = Keyword  /* store the keyword */
   Header.HeadCount.!Entry = Entry  /* store the entry */
  end

 end

end

Indent = 1  /* the minimum indent */

if (pos(Mode,'EV') > 0) then  /* if we are editing or viewing */
do
 call GetDefinitions Global.!Translations,'!Translations',1  /* see if we need to set any keyword translations */
end

do Index = 1 to HeadCount  /* for each of the lines we collected  */

 UpKeyword = translate(translate(Header.Index.!Keyword,'_','-'))  /* upper case with hyphens turned into underscores */
 Header.Index.!Reformat = (wordpos(UpKeyword,Reformats) > 0)  /* if it in this list, set a flag */
 Header.Index.!DateLine = (wordpos(UpKeyword,DateLines) > 0)  /* if it in this list, set a flag */
 Header.Index.!Receiver = (wordpos(UpKeyword,Receivers) > 0)  /* if it in this list, set a flag */

 if (pos(Mode,'EV') > 0) then  /* if we are editing or viewing */
 do

  if (symbol('Global.!Translations.'||UpKeyword) == 'VAR') then  /* if this one exists */
  do
   Header.Index.!Keyword = Global.!Translations.UpKeyword  /* use the translation instead */
  end

  Indent = max(Indent,length(Header.Index.!Keyword))  /* if the keyword is longer than any previous one, update the indent */

 end

end

if (pos(Mode,'EV') > 0) then  /* if we want indented lines */
do
 Indent = Indent + 3  /* add 3 (blank, colon, blank) */
end

do Index = 1 to HeadCount  /* for each of the entries we collected  */

 if (Header.Index.!Reformat) then  /* if this is one we want to reformat, i.e. an @ddress entry */
 do

  Header.Index.!Entry = AddressFormat(Header.Index.!Entry,1,(pos(Mode,'EP') > 0))  /* get a formatted string of addresses, full, check if editing or post-editing */

  if ((Mode == 'V') & (Global.!Settings.MaxAddresses >< '')) then  /* if we are viewing, and we want long lists eliminated */
  do

   Counter = 1  /* start a counter */
   NextPos = 0  /* start here */

   do until (NextPos == 0)  /* go on until we run out of steam */

    NextPos = pos(d2c(0),Header.Index.!Entry,(NextPos + 1))  /* look for the next null byte spacer */

    if (NextPos > 0) then  /* if we find one */
    do
     Counter = Counter + 1  /* up the counter */
    end

   end

   if (Counter > Global.!Settings.MaxAddresses) then  /* if we have more addresses than we want to see */
   do
    Header.Index.!Entry = '[list of '||Counter||' recipients suppressed]'  /* use this instead */
   end

  end

  if (Global.!Settings.HeaderLineLength >< '') then  /* if we have a line length limit */
  do
   Header.Index.!Entry = AddressIndent(Header.Index.!Entry,Indent,(Global.!Settings.HeaderLineLength - Indent))  /* indent the address to fit the header */
  end

 end
 else  /* if this is not one we want to reformat */
 do

  if ((Header.Index.!DateLine) | (Header.Index.!Receiver)) then  /* if it is one with dates in it */
  do

   EntryLines = Header.Index.!Entry  /* copy the entry (which may contain -- More -- several lines) */
   Header.Index.!Entry = ''  /* start all over with nothing */
   
   do while (Entrylines >< '')  /* go on while we have stuff left */

    Receiver = ''  /* we have no receiver yet (we may be getting a "Received" line) */
    Recipient = ''  /* we have no recipient yet (we may be getting a "Received" line) */

    parse var EntryLines NextLine (Global.!CRLF) EntryLines  /* get the next entry */

    if (Header.Index.!Receiver) then  /* if we are dealing with a receiver entry */
    do

     LastSemiPos = lastpos(';',NextLine)  /* the last semicolon position */
     parse var NextLine RouteInfo =(LastSemiPos) +1 NextLine  /* get the different bits */

     do while ((RouteInfo >< '') & ((Receiver == '') | (Recipient = '')))  /* as long as we have something left and either result is still missing */

      parse var RouteInfo NextWord RouteInfo  /* get the next word */

      if ((Receiver == '') & (translate(NextWord) == 'BY')) then  /* if it is this */
      do
       Receiver = word(RouteInfo,1)  /* the receiver must be the second word, so store it till after we sort the date, if necessary */
      end

      if ((Recipient == '') & (translate(NextWord) == 'FOR')) then  /* if it is this */
      do
       Recipient = word(RouteInfo,1)  /* the recipient must be the second word, so store it */
      end

     end

    end

    if (Receiver >< '') then  /* if we have a receiver */
    do
     DateTimeType = word(Global.!Settings.DateTimeHeader,1)  /* use just the first word of the date/time stamp type string */
    end
    else  /* if we do not have a receiver */
    do
     DateTimeType = Global.!Settings.DateTimeHeader  /* use the whole date/time stamp type string */
    end

    NextLine = DateTimeDisplay(NextLine,DateTimeType)  /* get a formatted date/time stamp */
     
    if (Receiver >< '') then  /* if we have a receiver */
    do
     NextLine = NextLine||' '||Receiver  /* add this */
    end

    if (Recipient >< '') then  /* if we have a recipient */
    do
     NextLine = NextLine||' '||Recipient  /* add this */
    end

    if (Global.!Settings.HeaderLineLength >< '') then  /* if we have a line length limit */
    do
     NextLine = WordWrap(NextLine,(Global.!Settings.HeaderLineLength - Indent),0,copies(' ',Indent))  /* word-wrap, indent, no word breaks */
    end

    if (Header.Index.!Entry >< '') then  /* if this is not the first line */
    do
     NextLine = Global.!CRLF||copies(' ',Indent)||NextLine  /* add a new line and an indent */
    end

    Header.Index.!Entry = Header.Index.!Entry||NextLine  /* add the result to what we already have */

   end

  end
  else  /* if it is not a date entry we want to reformat */
  do

   if (Global.!Settings.HeaderLineLength >< '') then  /* if we have a line length limit */
   do
    Header.Index.!Entry = WordWrap(Header.Index.!Entry,(Global.!Settings.HeaderLineLength - Indent),0,copies(' ',Indent))  /* word-wrap, indent, no word breaks */
   end

  end

 end

end

do Index = 1 to HeadCount  /* for each of the lines we collected  */

 if (pos(Mode,'EV') > 0) then  /* if we are viewing or editing */
 do
  Filler = copies(' ',(Indent - length(Header.Index.!Keyword) - 2))  /* filler blanks */
 end
 else  /* if we the header is for machine use */
 do
  Filler = ''  /* no filler */
 end

 HeadLines = HeadLines||Header.Index.!Keyword||Filler||': '||Header.Index.!Entry||Global.!CRLF  /* add the full line */

end

return HeadLines  /* end of FoldHeaderLines */

/**************************************************************************/
FullHostName: procedure expose Global.  /* returns the full host name */
/**************************************************************************/

if (Global.!Settings.HostName == '') then  /* if we have no host name definition yet */
do

 Global.!Settings.HostName = value('HOSTNAME',,'OS2Environment')  /* try to get a host name from the OS/2 environment */

 if (Global.!Settings.HostName == '') then  /* if we still have no host name */
 do
  Global.!Settings.HostName = 'RexxMail-Machine'  /* use this dummy host name */
 end

end

parse var Global.!Settings.Address . '@' Domain  /* get the (real) sender's domain name */
FullHostName = Global.!Settings.HostName||'.'||Domain  /* complete the host name */

return FullHostName  /* end of FullHostName */

/**************************************************************************/
GetAddressList: procedure expose Global.  /* extract a list of names and/or addresses from a formatted list of addresses */
/**************************************************************************/

parse arg Addresses,AddressType  /* get the arguments */

Names = ''  /* start with nothing */
Separator = d2c(0)  /* the separator between full addresses */

do while (Addresses >< '')  /* as long as we have bits left */

 if (Names >< '') then  /* if we already have something */
 do
  Names = Names||'. '  /* add a separator */
 end

 parse var Addresses Address (Separator) Addresses  /* get the next address */

 if (words(Address) > 1) then  /* if the address contains more than one word */
 do
  NamePart = strip(subword(Address,1,words(Address)-1),'B','"')  /* all but the last must be the name, and get rid of any quotation marks */
  Address = subword(Address,words(Address))  /* the last is the address */
 end
 else  /* if the address contains just a single word */
 do
  NamePart = ''  /* we have no name part */
 end

 AddressType = translate(left(AddressType,1))  /* use just the first character, in upper case */

 select  /* do one of the following */

  when (AddressType == 'F') then  /* if we want it all */
  do

   if (NamePart >< '') then  /* if we have a name part */
   do
    NewName = NamePart||' '||Address  /* use this */
   end
   else  /* if we do not have a name part */
   do
    NewName = Address  /* use this */
   end

  end

  when (AddressType == 'N') then  /* if we want just the name */
  do

   if (NamePart >< '') then  /* if we have a name part */
   do
    NewName = NamePart  /* use it */
   end
   else  /* if we do not have a name part */
   do
    NewName = Address  /* use the address */
   end

  end

  otherwise  /* if we want just the address */
  do
   Address = strip(Address,'L','<')  /* get rid of the leading angle bracket */
   Address = strip(Address,'T','>')  /* get rid of the trailing angle bracket */
   NewName = Address  /* use the address */
  end

 end

 Names = Names||NewName  /* add the result to the names */

end

return Names  /* end of GetAddressList */

/**************************************************************************/
GetDefinitions: procedure expose Global.  /* retrieves definition file contents */
/**************************************************************************/

parse upper arg FileName,VarName,Named,Bare  /* get the arguments */

VarName = 'Global.'||VarName||'.'  /* complete the variable base name */

if (symbol(VarName||'0') == 'VAR') then  /* if we already have this one */
do
 return 1  /* quit with an O.K. result */
end

FileCont = GetFileContents(FileName)  /* get the file contents */
LeftSide = ''  /* no left side yet */
LineCount = 0  /* start a line counter */
Counter = 0  /* start a counter */
Named = (Named == 1)  /* 1 = TRUE */
Bare = (Bare == 1)  /* 1 = TRUE */
call value VarName||'0',Counter  /* store the counter */

If (FileCont == '') then  /* if there is nothing */
do
 return 1  /* quit, no problem */
end

do while (FileCont >< '')  /* while we have content left */

 LineCount = LineCount + 1  /* up the line counter */
 parse var FileCont NextLine (Global.!CRLF) FileCont  /* get the next line */
 NextLine = translate(NextLine,' ',d2c(9))  /* turn TABs into spaces */

 if ((NextLine >< '') & (left(NextLine,1) >< '#')) then  /* unless it is a comment or an empty line */
 do

  if (Named) then  /* if we want a named array */
  do

   parse var NextLine LeftSide '=' RightSide  /* get the bits */
   LeftSide = translate(translate(strip(LeftSide),'_','-'))  /* get rid of superfluous spaces, turn hyphens into underscores, then make it upper case */

   if (LeftSide == '') then  /* if we have nothing left */
   do
    call AddError 'Missing left side of definition in line '||LineCount||' of "'||FileName||'"'  /* report */
    return 0  /* quit */
   end

   RightSide = strip(RightSide)  /* get rid of superfluous spaces */

   if ((Bare) | (RightSide >< '')) then  /* unless we have no right side and we cannot have bare values */
   do
    call value VarName||LeftSide,RightSide  /* set the new value -- true case */
   end

  end
  else  /* if we want a numbered array */
  do

   Counter = Counter + 1  /* up the counter */

   if (left(NextLine,1) == ' ') then  /* if this is a folded line */
   do

    if (LeftSide == '') then  /* if we have no previous left side */
    do
     call AddError 'Missing left side of definition in line '||LineCount||' of "'||FileName||'"'  /* report */
     return 0  /* quit */
    end

    call value VarName||Counter||'.!Right',strip(NextLine)  /* use the line minus excess space */

   end
   else  /* if it is not a folded line */
   do

    parse var NextLine LeftSide '=' RightSide  /* get the bits */
    LeftSide = strip(LeftSide)  /* remove excess space */

    if (LeftSide == '') then  /* if we have nothing left */
    do
     call AddError 'Missing left side of definition in line '||LineCount||' of "'||FileName||'"'  /* report */
     return 0  /* quit */
    end

    call value VarName||Counter||'.!Right',strip(RightSide)  /* store the second bit minus excess space */

   end

   if (LeftSide == '_') then  /* if we have just an underscore */
   do
    call value VarName||Counter||'.!Left',''  /* use nothing */
   end
   else  /* if we have something else */
   do
    call value VarName||Counter||'.!Left',LeftSide  /* use the previous left side */
   end

  end

 end

end

call value VarName||'0',Counter  /* store the counter */

return 1  /* end of GetDefinitions */

/**************************************************************************/
GetFileContents: procedure expose Global.  /* gets the contents of a file */
/**************************************************************************/

parse arg FileName  /* get the argument */

if (\FileCheck(FileName)) then  /* if there is no such file */
do
 return ''  /* quit with nothing */
end

if (\FileOpen(FileName,'READ')) then  /* if we cannot open the file for reading */
do
 return ''  /* quit with nothing */
end

Contents = strip(charin(FileName,1,chars(FileName)),'T',d2c(26))  /* get the file contents and remove any EOF marker */

call FileClose FileName  /* close the file */

return Contents  /* end of GetFileContents */

/**************************************************************************/
GetFileEntry: procedure expose Global.  /* looks for keyword in definitions and returns first match */
/**************************************************************************/

parse upper arg FileName,SearchString,ReverseLookup  /* get the arguments in upper case */

ReverseLookup = (ReverseLookup == 1)  /* 1 = true */
NamePart = filespec('N',FileName)  /* the file name part */
NamePart = translate(substr(NamePart,1,(lastpos('.',NamePart) - 1)))  /* lose the extension */

if (\GetDefinitions(FileName,NamePart,0)) then  /* if we cannot find the definition info */
do
 return ''  /* quit */
end

do Index = 1 to Global.NamePart.0  /* look through the finds */

 if (ReverseLookup) then  /* if we are looking from right to left */
 do

  if (pos(SearchString,translate(Global.NamePart.Index.!Right)) > 0) then  /* if the right part contains what we are looking for */
  do
   return Global.NamePart.Index.!Left  /* this is what we want */
  end

 end
 else  /* if we are looking from left to right */
 do

  if (SearchString == translate(Global.NamePart.Index.!Left)) then  /* if the left part is the one we are looking for */
  do
   return Global.NamePart.Index.!Right  /* this is what we want */
  end

 end

end

return ''  /* end of GetFileEntry */

/**************************************************************************/
GetHeader: procedure expose Global.  /* gets one or more header entries from a (RexxMail) message file */
/**************************************************************************/

parse upper arg Keywords  /* get the arguments in upper case */

MessFile = Global.!ProcFile  /* the file to process */
MessCont = GetFileContents(MessFile)  /* get the file contents */

if (\MessageSettings(MessFile,,'CHECK')) then  /* if it is not a RexxMail message file */
do
 return 0  /* quit */
end

if (\MessageContents(MessCont,'Global.!MessHead.')) then  /* if we cannot get the header sorted out */
do
 return 0  /* and quit with nothing */
end

do while (Keywords >< '')  /* as long as we have keywords left */

 parse var Keywords Keyword Keywords  /* get the next one */

 Keyword = strip(Keyword,'T',':')  /* strip any trailing colon */

 select  /* do one of the following */

  when (Keyword == 'FULL') then  /* if we want the lot */
  do

   do Index = 1 to Global.!MessHead.0  /* for each header line */
    call lineout 'STDOUT:',Global.!MessHead.Index.!Keyword||': '||Global.!MessHead.Index.!Entry  /* send the entry to the standard output */
   end

  end

  when (Keyword == 'ALL') then  /* if we want all the entries */
  do

   do Index = 1 to Global.!MessHead.0  /* for each header line */
    call lineout 'STDOUT:',Global.!MessHead.Index.!Entry  /* send the entry to the standard output */
   end

  end

  otherwise  /* if it is something else */
  do

   HeadLine = GetHeaderEntry(Keyword,'Global.!MessHead.',1)  /* look for our entry or entries */

   if (HeadLine == '') then  /* if we found nothing */
   do
    return 0  /* quit with no success */
   end

   call lineout 'STDOUT:',HeadLine  /* send the result to the standard output */

  end

 end

end

return 1  /* end of GetHeader */

/**************************************************************************/
GetHeaderEntry: procedure expose Global.  /* returns the first entry or all entries ("More") in a message header with a matching key word */
/**************************************************************************/

parse arg Keyword,HeadName,More  /* get the arguments */

More = (More == 1)  /* 1 = true */
OutLine = ''  /* start with nothing */

do Index = 1 to value(HeadName||'0')  /* run through the header contents */

 if (translate(value(HeadName||Index||'.!Keyword')) == Keyword) then  /* if this is one we want */
 do

  if (\More) then  /* if we want just the one entry */
  do
   return value(HeadName||Index||'.!Entry')  /* return it */
  end

  if (value(HeadName||Index||'.!Entry') >< '') then  /* if we want more, and if it contains something */
  do

   Entry = value(HeadName||Index||'.!Entry')  /* start with this */

   if (OutLine >< '') then  /* if we already have something */
   do
    Entry = Global.!CRLF||Entry  /* add a new line */
   end

   OutLine = OutLine||Entry  /* add the new bit to the output text */

  end

 end

end

return OutLine  /* end of GetHeaderEntry */

/**************************************************************************/
GetObjectEA: procedure expose Global.  /* attempts to get an extended attribute from an object */
/**************************************************************************/

parse arg Object,EAName  /* get the arguments */

Attempts = 0  /* we haven't tried yet */
MaxAttempts = 20  /* stop after so many attempts */
Delay = 0.005  /* start with this delay */

do while ((sysgetea(Object,EAName,'EAString') >< 0) & (Attempts < MaxAttempts))  /* go on until we succeed or reach the maximum no. of attempts */
 Attempts = Attempts + 1  /* up the attempts counter */
 call syssleep (Delay * Attempts)  /* wait a bit */
end

if (Attempts == MaxAttempts) then  /* if we reached the last try */
do
 call AddError 'Cannot get extended attributes for object "'||Object||'"'  /* report */
 return ''  /* no success */
end

OutText = ''  /* nothing yet */

if (EAString >< '') then  /* if we actually found something */
do

 parse var EAString EAType 3 EARest  /* get the bits we want */

 if (EAType == 'DFFF'x) then  /* if it is a multiple value, multiple type entry */
 do
  parse var EARest . 3 Lines 5 EAString  /* get the stuff we want */
  Lines = c2d(reverse(Lines))  /* extract the number of comment lines */
 end
 else  /* if not, it must be simple ASCII */
 do
  Lines = 1  /* we have just one line to extract */
 end

 do Lines  /* take each line entry */

  parse var EAString EAType 3 EALength 5 EAString  /* get the bits we want */

  if (EAType == 'FDFF'x) then  /* if it is ASCII text (which it should be) */
  do

   EALength = c2d(reverse(EALength))  /* get the length of the next line */
   parse var EAString EAText +(EALength) EAString  /* get the bits we want */
   OutText = OutText||EAText  /* add the text to what we have */

   if (EAString >< '') then  /* if there is more */
   do
    OutText = OutText||Global.!CRLF  /* add a CRLF */
   end

  end
  else  /* if it is not ASCII */
  do
   call AddError 'Invalid extended attribute '||EAType||' from "'||Object||'"'  /* report */
   return ''  /* return no success */
  end

 end

end

return OutText  /* end of GetObjectEA */

/**************************************************************************/
GetRecipients: procedure expose Global.  /* get (bare) recipient addresses from a message file */
/**************************************************************************/

parse arg FileName  /* get the argument */

FileCont = GetFileContents(FileName)  /* get the file contents */

if (\MessageContents(FileCont,'Global.!ListHead.')) then  /* if we cannot get the header contents sorted out */
do
 return ''  /* quit with nothing */
end

AddRecipients = GetHeaderEntry('TO','Global.!ListHead.')||','||,  /* get the To recipients from the header */
                GetHeaderEntry('CC','Global.!ListHead.')||','||,  /* add the Cc recipients from the header */
                GetHeaderEntry('ACC','Global.!ListHead.')  /* add the Acc recipients from the header */
                GetHeaderEntry('BCC','Global.!ListHead.')  /* add the Bcc recipients from the header */
AddRecipients = AddressFormat(AddRecipients,0,1)  /* get a list of properly formatted and checked bare addresses */

return AddRecipients  /* end of GetRecipients */

/**************************************************************************/
GetRedirect: procedure expose Global.  /* check for a destination or action in a destination/action specs file */
/**************************************************************************/

parse arg MoveFile,Location,HeadName,BodyName  /* get the arguments */

NamePart = filespec('N',MoveFile)  /* the move file name */
NamePart = translate(substr(NamePart,1,(lastpos('.',NamePart) - 1)))  /* the move file name without the '.TXT' and in upper case */

if (\GetDefinitions(MoveFile,NamePart,0)) then  /* if we cannot get (or already have) the file contents */
do
 return ''  /* quit with nothing */
end

do MoveIndex = 1 to Global.NamePart.0  /* for each of the stored redirection lines */

 if ((Global.NamePart.MoveIndex.!Right == '') & (Global.NamePart.MoveIndex.!Left >< '')) then  /* if we hit a default action */
 do
  return Global.NamePart.MoveIndex.!Left  /* return with the corresponding action */
 end

 NextRecipe = Global.NamePart.MoveIndex.!Right  /* copy the recipe */
 GotHit = 0  /* no luck yet */

 do while ((NextRecipe >< '') & (\GotHit))  /* as long as we have something left and no luck */

  parse var NextRecipe NextTerm '&' NextRecipe  /* separate the first recipe term from any subsequent ANDed recipe terms */

  if (left(NextTerm,1) == '\') then  /* if it starts with a backslash */
  do
   parse var NextTerm . 2 NextTerm  /* lose it */
   NextTerm = strip(NextTerm)  /* remove excess whitespace */
   Invert = 1  /* invert the meaning of this one */
  end
  else  /* if there is no backslash */
  do
   Invert = 0  /* this one is normal */
  end

  if (pos(':',NextTerm) == 2) then  /* if it looks like a location */
  do

   if (Invert) then  /* if we are inverting */
   do
    GotHit = (translate(NextTerm) >< translate(Location))  /* if the location does not match, set a flag */
   end
   else  /* if it is a normal recipe */
   do
    GotHit = (translate(NextTerm) == translate(Location))  /* if we have a location hit, set a flag */
   end

  end
  else  /* if it is not a location */
  do

   parse var NextTerm MoveKeyword ':' MoveEntry  /* get the recipe term parts */
   MoveKeyword = translate(strip(MoveKeyword))  /* remove excess blanks and make upper case */
   MoveEntry = translate(strip(MoveEntry))  /* remove excess blanks and make upper case */
   MoveCont. = ''  /* start with nothing */
   Counter = 0  /* start a counter */

   if (pos(':',MoveEntry) == 2) then  /* if it looks like a location */
   do

    if (FileCheck(MoveEntry)) then  /* if it is an existing file */
    do

     if (FileOpen(MoveEntry,'READ')) then  /* if we can open the file */
     do

      do while (lines(MoveEntry))  /* while we have lines left */

       NextLine = strip(translate(linein(MoveEntry),' ',d2c(9)))  /* get the next line, convert any TABs to spaces and remove excess whitespace */

       if (NextLine >< '') then  /* if it is not empty */
       do

        if (left(NextLine,1) >< '#') then  /* if it is not a comment */
        do
         Counter = Counter + 1  /* up the counter */
         MoveCont.Counter = translate(NextLine)  /* store the line in upper case */
        end

       end

      end

      call FileClose MoveEntry  /* close the file */

     end

    end

   end

   MoveCont.0 = Counter  /* store the counter */

   if (MoveCont.0 == 0) then  /* if we have no file contents to check against */
   do
    MoveCont.0 = 1  /* we have one entry */
    MoveCont.1 = MoveEntry  /* and this is it */
   end

   if (MoveKeyword == 'BODY') then  /* if we are matching body content */
   do

    if (BodyName = '') then  /* if we have no body name */
    do
     call AddError 'Invalid redirection recipe (BODY cannot be used in collect.txt): '||NextTerm,1  /* report quietly (non-fatal) */
    end
    else  /* if we have a body name */
    do

     BodyCont = translate(value(BodyName))  /* the body content, in upper case */

     do Index = 1 to MoveCont.0  /* run through the entries */

      if (pos(MoveCont.Index,BodyCont) > 0) then  /* if this one is in the body */
      do
       GotHit = (\Invert)  /* if we are inverting the recipe, finding a hit is not a match */
       leave Index  /* exit the loop */
      end

     end

    end

   end
   else  /* if we are matching header content */
   do

    do HeadIndex = 1 to value(HeadName||'0')  /* run through the header lines */

     if ((MoveKeyword == translate(value(HeadName||HeadIndex||'.!Keyword'))) | (MoveKeyword == 'HEAD')) then  /* if it is the right keyword, or we are matching any header content */
     do

      do Index = 1 to MoveCont.0  /* run through the entries */

       if ((MoveCont.Index == '') | (pos(MoveCont.Index,translate(value(HeadName||HeadIndex||'.!Entry'))) > 0)) then  /* if there is no need to match content (just the keyword), or if the content is there */
       do
        GotHit = 1  /* we have a hit */
        leave HeadIndex  /* exit the outer loop */
       end

      end

     end

    end

    GotHit = (GotHit && Invert)  /* if we were inverting, no hit means a match */

   end

  end

  if (GotHit) then  /* if we have a hit */
  do

   if (NextRecipe == '') then  /* if there is nothing left to check */
   do
    call LogAction 'Redirect match ('||NamePart||') = "'||Global.NamePart.MoveIndex.!Right||'"; action = "'||Global.NamePart.MoveIndex.!Left||'"',1  /* report, quietly */
    return Global.NamePart.MoveIndex.!Left  /* return with the corresponding action */
   end
   else  /* if we still have a bit to check */
   do
    GotHit = 0  /* make sure we can enter the loop for the next bit */
   end

  end
  else  /* if we had no luck */
  do
   leave  /* exit the loop */
  end

 end

end

return ''  /* end of GetRedirect */

/**************************************************************************/
GetTimeZone: procedure expose Global.  /* gets the local time zone */
/**************************************************************************/

signal on halt name HaltGetTimeZone  /* handles halt locally */

parse arg ContactServers  /* get the argument */

if (Global.!Settings.TimeZone >< '') then  /* if we have a global time zone var */
do
 return Global.!Settings.TimeZone  /* simply return it */
end

ContactServers = (ContactServers == 1)  /* 1 = True */

if (\ContactServers) then  /* if we are not allowed to contact time servers */
do
 return '-0000'  /* return an unknown time zone */
end

TimeZone = ''  /* start with nothing */

do while ((TimeZone == '') & (Global.!Settings.TimeServers >< ''))  /* as long as we have no result, and we have time servers left */

 parse var Global.!Settings.TimeServers TimeServer Global.!Settings.TimeServers  /* get the next entry */
 parse var TimeServer TimeServer ':' PortNumber  /* see if we have a port number */

 if (PortNumber == '') then  /* if not */
 do
  PortNumber = 37  /* default to this (NTP) */
 end

 Socket = ServerConnect(TimeServer,PortNumber,Global.!Settings.TimeAttempts)  /* get the socket number for a server connection through the specified or default port */

 if (Socket == '') then  /* if we have no socket */
 do
  call LogAction 'Connection failed'  /* report a miss */
 end
 else  /* if we have a socket */
 do

  call LogAction 'Connection established'  /* report */
  SocketBytes = sockrecv(Socket,'TimeBytes',4)  /* see how many bytes we get from the server; look for 4 */
  call SocketClose Socket  /* close the socket */

  if (SocketBytes == 4) then  /* if we got 4 bytes */
  do

   call LogAction 'Time data retrieved'  /* report */
   SysTime = ((date('B') - 693595) * 86400) + time('S')  /* system seconds since midnight, 1900-01-01 */
   TimeBytes = reverse(TimeBytes)  /* reverse the time bytes */
   AbsTime = 0  /* start with nothing */

   do Index = 1 to 4  /* take each of the bytes, starting with the low order one */
    AbsTime = AbsTime + (c2d(substr(TimeBytes,Index,1)) * (256 ** (Index-1)))  /* absolute seconds since midnight, 1900-01-01 */
   end

   TimeDiff = SysTime - AbsTime  /* the time difference in seconds, plus or minus */

   if (sign(TimeDiff) == -1) then  /* if the time difference is negative */
   do
    TimeSign = '-'  /* we need a minus sign */
    TimeDiff = -1 * TimeDiff  /* reverse the sign of the time difference */
   end
   else  /* if the time difference is not negative */
   do
    TimeSign = '+'  /* we need a plus sign */
   end

   TimeDiff = (TimeDiff + 900) % 1800  /* the system offset, rounded to the nearest half hour */

   if ((TimeSign = '-') & (TimeDiff = 0)) then  /* if we have a negative time difference, and the 30-minute offset is 0, i.e. we are only a bit behind */
   do
    TimeSign = '+'  /* use a plus */
   end

   ZoneHours = (TimeDiff % 2) * 100  /* construct the hours number */
   ZoneHalfHours = (TimeDiff // 2) * 30  /* construct the half hours number */
   TimeZone = TimeSign||right((ZoneHours + ZoneHalfHours),4,'0')  /* the whole time zone */
   call LogAction 'UTC offset calculated as "'||TimeZone||'"'  /* report */

  end
  else  /* if we did not get 4 bytes */
  do
   call LogAction 'Time data retrieval failed'  /* report */
  end

 end

end

/**************************************************************************/
HaltGetTimeZone:  /* handles halt locally */
/**************************************************************************/

if (TimeZone == '') then  /* if we still have no time zone */
do
 TimeZone = '-0000'  /* use the "UTC elsewhere" string instead */
end

return TimeZone  /* end of GetTimeZone */

/**************************************************************************/
HeaderCheck: procedure expose Global.  /* checks header contents of outgoing messages */
/**************************************************************************/

parse arg ToAdds,CcAdds,AccAdds,BccAdds  /* get the arguments */

NewList = ''  /* start afresh */
OKKeywords = 'From Sender To Cc Acc Bcc Reply-To Subject Comments In-Reply-To References Keyphrases'  /* allow these keywords -- once */
UpOKKeywords = translate(OKKeywords)  /* make an upper case copy */
UpOKTranslate = ''  /* start with nothing */
call GetDefinitions Global.!Translations,'!Translations',1  /* see if we need to set any keyword translations */

do Index = 1 to words(UpOKKeywords)  /* for each of the upper case OK keywords */

 Keyword = translate(word(UpOKKeywords,Index),'_','-')  /* get the upper case OK keyword with underscores replacing hyphens */

 if (symbol('Global.!Translations.'||Keyword) == 'VAR') then  /* if we have a translation for this one */
 do
  UpOKTranslate = UpOKTranslate||' '||translate(translate(Global.!Translations.Keyword,d2c(0),' '))  /* add the corresponding keyword translation in upper case and with spaces replaced with null bytes */
 end
 else  /* if we have no translation */
 do
  UpOKTranslate = UpOKTranslate||' '||translate(Keyword)  /* add the original keyword in upper case */
 end

end

do Index = 1 to Global.!MessHead.0  /* run through the header content */

 if (left(Global.!MessHead.Index.!Keyword,1) == Global.!Warning) then  /* if the keyword starts with a marker */
 do
  Global.!MessHead.Index.!Keyword = substr(Global.!MessHead.Index.!Keyword,2)  /* use the remainder */
 end

 UpKeyword = translate(translate(Global.!MessHead.Index.!Keyword,d2c(0),' '))  /* make an upper case copy with spaces changed into null bytes */

 if (wordpos(UpKeyword,NewList) > 0) then  /* if the upper case keyword is in the new keywords list, we already have this one */
 do
  Global.!MessHead.Index.!Keyword = Global.!Warning||Global.!MessHead.Index.!Keyword  /* add a warning marker */
  Global.!MessHead.!HeaderOK = 0  /* we have a problem */
 end
 else  /* if it is a new keyword */
 do

  NewList = NewList||' '||UpKeyword  /* add the upper case keyword to the list of new keywords */
  GotPos = wordpos(UpKeyword,UpOKTranslate)  /* look for the upper case keyword position in the upper case OK translations list */

  if (GotPos > 0) then  /* if the upper case keyword is in the upper case OK translations list */
  do
   Global.!MessHead.Index.!Keyword = word(OKKeywords,GotPos)  /* substitute the RFC equivalent from the OK keywords list */
   UpKeyword = translate(Global.!MessHead.Index.!Keyword)  /* make a new upper case copy */
  end

  if (wordpos(UpKeyword,UpOKKeywords) > 0) then  /* if the upper case keyword is in the upper case OK keywords list */
  do

   select  /* do one of the following */

    when ((UpKeyword == 'TO') & (ToAdds >< '')) then  /* if we are looking at the "TO:" line and we have addresses to add */
    do
     Global.!MessHead.Index.!Entry = Global.!MessHead.Index.!Entry||', '||ToAdds  /* add any extra recipients */
     ToAdds = ''  /* get rid of the extra recipients */
    end

    when ((UpKeyword == 'CC') & (ToAdds >< '')) then  /* if we are looking at the "CC:" line and we have addresses to add */
    do
     Global.!MessHead.Index.!Entry = Global.!MessHead.Index.!Entry||', '||CcAdds  /* add any extra recipients */
     CcAdds = ''  /* get rid of the extra recipients */
    end

    when ((UpKeyword == 'ACC') & (ToAdds >< '')) then  /* if we are looking at the "ACC:" line and we have addresses to add */
    do
     Global.!MessHead.Index.!Entry = Global.!MessHead.Index.!Entry||', '||AccAdds  /* add any extra recipients */
     AccAdds = ''  /* get rid of the extra recipients */
    end

    when ((UpKeyword == 'BCC') & (ToAdds >< '')) then  /* if we are looking at the "BCC:" line and we have addresses to add */
    do
     Global.!MessHead.Index.!Entry = Global.!MessHead.Index.!Entry||', '||BccAdds  /* add any extra recipients */
     BccAdds = ''  /* get rid of the extra recipients */
    end

    otherwise  /* if none of the above */
    do
     nop  /* do nothing */
    end

   end

  end
  else  /* if it is not in the O.K. keywords list */
  do

   if ((left(UpKeyword,2) >< 'X-') | (verify(Global.!MessHead.Index.!Keyword,xrange('21'x,'7E'x),'NOMATCH') > 0))  then  /* if it is not an X-string keyword, or if it contains illegal characters, it's a rogue entry */
   do
    Global.!MessHead.Index.!Keyword = Global.!Warning||Global.!MessHead.Index.!Keyword  /* so add a warning marker */
    Global.!MessHead.!HeaderOK = 0  /* we have a problem */
   end

  end

 end

end

if (ToAdds >< '') then  /* if we still have these extra recipients hanging about, there was no "To:" line in the header */
do
 Global.!MessHead.0 = Global.!MessHead.0 + 1  /* up the lines counter */
 Index = Global.!MessHead.0  /* up the index */
 Global.!MessHead.Index.!Keyword = 'TO'  /* add the keyword */
 Global.!MessHead.Index.!Entry = ToAdds  /* and the entry */
end

if (CcAdds >< '') then  /* if we still have these extra recipients hanging about, there was no "Cc:" line in the header */
do
 Global.!MessHead.0 = Global.!MessHead.0 + 1  /* up the lines counter */
 Index = Global.!MessHead.0  /* up the index */
 Global.!MessHead.Index.!Keyword = 'CC'  /* add the keyword */
 Global.!MessHead.Index.!Entry = CcAdds  /* and the entry */
end

if (AccAdds >< '') then  /* if we still have these extra recipients hanging about, there was no "Acc:" line in the header */
do
 Global.!MessHead.0 = Global.!MessHead.0 + 1  /* up the lines counter */
 Index = Global.!MessHead.0  /* up the index */
 Global.!MessHead.Index.!Keyword = 'ACC'  /* add the keyword */
 Global.!MessHead.Index.!Entry = AccAdds  /* and the entry */
end

if (BccAdds >< '') then  /* if we still have these extra recipients hanging about, there was no "Bcc:" line in the header */
do
 Global.!MessHead.0 = Global.!MessHead.0 + 1  /* up the lines counter */
 Index = Global.!MessHead.0  /* up the index */
 Global.!MessHead.Index.!Keyword = 'BCC'  /* add the keyword */
 Global.!MessHead.Index.!Entry = BccAdds  /* and the entry */
end

return 1  /* end of HeaderCheck */

/**************************************************************************/
HeaderContains: procedure expose Global.  /* checks for specific header content */
/**************************************************************************/

parse arg Keyword,Entry,HeadName  /* get the arguments */

do Index = 1 to value(HeadName||'0')  /* run through the header content */

 KeywordOK = ((Keyword == '') | (Keyword == translate(value(HeadName||Index||'.!Keyword'))))  /* if we find the keyword, or we have no keyword, all is well */
 EntryOK = ((Entry == '') | (pos(Entry,translate(value(HeadName||Index||'.!Entry'))) > 0))  /* if we find the entry, or we have no entry, all is well */

 if (KeywordOK & EntryOK & (Keyword||Entry >< '')) then  /* if we found what we wanted */
 do
  return 1  /* return with a result */
 end

end

return 0  /* end of HeaderContains */

/**************************************************************************/
HeaderEncode: procedure expose Global.  /* encodes a header string if necessary */
/**************************************************************************/

parse arg MessHead  /* get the argument */

ISOStart = '=?iso-8859-1?Q?'  /* define the start of an ISO string */
ISOEnd = '?='  /* define the end of an ISO string */
NewHead = ''  /* start with nothing */
GotCRLF = (right(MessHead,2) == Global.!CRLF)  /* set a flag if we have a terminating CRLF sequence */

do until (MessHead == '')  /* go on until we run out of header */

 StartQuote = ''  /* nothing yet */
 EndQuote = ''  /* nothing yet */
 MIMEPrevWord = 0  /* there is no previous word, so it cannot have been a MIME word */
 parse var MessHead NextLine (Global.!CRLF) MessHead  /* get a line from the header */

 if (left(NextLine,1) == ' ') then  /* if the line starts with a space */
 do
  NewLine = ''  /* nothing yet */
  NextLine = strip(NextLine)  /* remove excess whitespace */
 end
 else  /* if the line does not start with a space */
 do
  parse var NextLine NewLine NextLine  /* start the new line with the first word, which must be a keyword */
 end

 do while (NextLine >< '')  /* go on until we run out of line */

  parse var NextLine NextWord NextLine  /* get the next word in the line */

  if (verify(NextWord,xrange('00'x,'7F'x),'NOMATCH') > 0) then  /* if the current word contains any high-bit characters */
  do

   NewWord = ''  /* start with nothing */

   if (left(NextWord,1) == '"') then  /* if the word starts with a quotation mark */
   do
    NextWord = substr(NextWord,2)  /* cut it off */
    StartQuote = '"'  /* we have a quotation mark */
   end

   if (right(NextWord,1) == '"') then  /* if the word ends with a quotation mark */
   do
    NextWord = left(NextWord,length(NextWord) - 1)  /* cut it off */
    EndQuote = '"'  /* we have a quotation mark */
   end

   do Index = 1 to length(NextWord)  /* run through the message text */

    NextChar = substr(NextWord,Index,1)  /* take each character */

    if ((c2d(NextChar) > 126) | (pos(NextChar,'_=') > 0)) then  /* if it's not in the OK range, or if we are in a MIME word and it is an underscore or equal sign */
    do
     NewWord = NewWord||'='||c2x(NextChar)  /* substitute an equals sign followed by the character's hex code */
    end
    else  /* if the character is in the OK range */
    do
     NewWord = NewWord||NextChar  /* simply add it to the output buffer */
    end

   end

   if (MIMEPrevWord) then  /* if the previous word was also a MIME word */
   do
    NewWord = '_'||NewWord  /* stick on an underscore */
   end
   else  /* if the previous word was not a MIME word */
   do
    NewWord = ' '||StartQuote||ISOStart||NewWord  /* stick on a space, any quotation mark, and an ISO start string */
    StartQuote = ''  /* nothing now */
    MIMEPrevWord = 1  /* set the MIME flag */
   end

   if (NextLine == '') then  /* if this is the last word */
   do
    NewWord = NewWord||ISOEnd||EndQuote  /* add the end code */
    EndQuote = ''  /* nothing now */
   end

  end
  else  /* if the word does not contain any high-bit characters */
  do

   if (MIMEPrevWord) then  /* if the previous word was a MIME word */
   do
    NewWord = ISOEnd||EndQuote||' '||NextWord  /* stick on an ISO end string and a space */
    EndQuote = ''  /* nothing now */
    MIMEPrevWord = 0  /* reset the MIME flag */
   end
   else  /* if the previous word was not a MIME word */
   do
    NewWord = ' '||NextWord  /* just stick on a space */
   end

  end

  NewLine = NewLine||NewWord  /* add it to the rest of the line */

 end

 NewHead = NewHead||NewLine  /* add the new line to the header */

 if (MessHead >< '') then  /* if there is more to come */
 do
  NewHead = NewHead||Global.!CRLF  /* add a CRLF to the header */
 end

end

if (GotCRLF >< '') then  /* if we had a terminating CRLF */
do
 NewHead = NewHead||Global.!CRLF  /* add a CRLF */
end

return NewHead  /* end of HeaderEncode */

/**************************************************************************/
HTMLFilter: procedure expose Global.  /* filters an HTML file to remove potentially malicious content */
/**************************************************************************/

signal on halt name HaltHTMLFilter  /* handles halt locally */

parse arg HTMLCont  /* get the argument */

HeaderTags = ''  /* start with nothing */
NewHTMLCont = ''  /* start with nothing */

if (pos('<HEAD>',translate(HTMLCont)) > 0) then  /* if we have a <HEAD> tag */
do
 InHead = 1  /* assume we're in the <HEAD> part to begin with to make sure 'stray' tags get picked up */
end
else  /* if we have no <HEAD> tag */
do
 InHead = 0  /* we're not in the <HEAD> part */
end

do while (HTMLCont >< '')  /* as long as we have something left */

 parse var HTMLCont FirstBit '<' TagContent '>' HTMLCont  /* get the next tag content and the part preceding it */

 if (TagContent >< '') then  /* if we have tag content */
 do

  NewContent = ''  /* start with nothing */

  do while (TagContent >< '')  /* run through the content */

   parse var TagContent BeforeBit (Global.!CRLF) TagContent  /* look for a bit terminated by a CRLF pair (FrontPage appears to consider this formatting) */
   NewContent = NewContent||BeforeBit  /* always use this bit */

   if (TagContent >< '') then  /* if there is more to come */
   do
    NewContent = NewContent||' '  /* add a space */
   end

  end

  TagContent = NewContent  /* copy the results */
  TagWord = translate(word(TagContent,1))  /* get the first tag word in upper case */

  if (TagWord == 'A') then  /* if the tag is a normal anchor link */
  do
   FirstBit = FirstBit||'<'||TagContent||'>'  /* restore the tag */
  end
  else  /* if the tag is not a normal anchor link */
  do

   if (Global.!Settings.HTMLSafe) then  /* if we want safe HTML */
   do

    URLPos = pos('HTTP://',translate(TagContent))  /* get the position of any external URL in the tag content (with or without --mandatory-- quotes) */

    if (URLPos > 0) then  /* if we have something */
    do

     parse var TagContent =(URLPos) URLContent .  /* get the URL content */
     TagContent = strip(TagContent,'T','"')  /* remove any double quotes that remain from the URL start */
     URLContent = strip(URLContent,'T','"')  /* remove any double quotes that remain at the end of the URL */

     if (URLContent >< '') then  /* if we have something */
     do

      select  /* do one of the following */

       when (TagWord == 'BODY') then  /* if this is the <BODY> tag, the URL must be an external background bitmap, so */
       do
        parse var TagContent FirstTagContent (URLContent) TagContent  /* split the tag content around the URL */
        TagContent = FirstTagContent||TagContent  /* rewrite the tag content without the URL */
       end

       when (TagWord == '!DOCTYPE') then  /* if this is the <!DOCTYPE... tag */
       do
        TagContent = ''  /* simply lose it */
       end

       otherwise  /* if it is neither of the above */
       do

        TagContent = ''  /* delete the tag content to signal that we have finished processing this tag */
        URLString = '<FONT SIZE="-1"><I>External '||TagWord||' link:</I>'||Global.!CRLF||,  /* small print explanatory text */
                    '<A HREF="'||URLContent||'">'||URLContent||'</A></FONT>'||Global.!CRLF  /* followed by the URL link */

        if (InHead) then  /* if we are still inside the <HEAD> part */
        do
         HeaderTags = HeaderTags||'<BR>'||Global.!CRLF||URLString  /* add a line break to HeaderTags, followed by the URL string */
        end
        else  /* if we are not inside the <HEAD> part */
        do
         FirstBit = FirstBit||Global.!CRLF||,  /* start a new source line */
                    '<HR SIZE=2>'||Global.!CRLF||URLString||,  /* add a horizontal rule followed by the URL string */
                    '<HR SIZE=2>'||Global.!CRLF  /* and end with another horizontal rule */
        end

       end

      end

     end

    end

   end

   if (TagContent >< '') then  /* if we still have tag content */
   do

    select  /* do one of the following */

     when (TagWord == 'HEAD') then  /* if the content starts with this tag */
     do
      FirstBit = FirstBit||'<HEAD>'||Global.!CRLF||,  /* restore the header tag */
                 '<META NAME="MSSmartTagsPreventParsing" CONTENT="TRUE">'||Global.!CRLF  /* add an anti-spam tag */
     end

     when (TagWord == '/HEAD') then  /* if the content starts with this tag */
     do
      FirstBit = FirstBit||'</HEAD>'||Global.!CRLF  /* restore the tag */
      InHead = 0  /* we're no longer inside the <HEAD> part */
     end

     when (TagWord == 'IMG') then  /* if the content starts with this tag */
     do
      TagContent = LocalURLTag(TagContent,'SRC')  /* look for a URL filename and change the tag content if we find it */
      FirstBit = FirstBit||'<'||TagContent||'>'  /* restore or redo the tag */
     end

     when (TagWord == 'BODY') then  /* if this is the 'BODY' tag */
     do

      TagContent = LocalURLTag(TagContent,'BACKGROUND')  /* look for a URL filename and change the tag content if we find it */
      FirstBit = FirstBit||'<'||TagContent||'>'  /* restore or redo the tag */

      if (HeaderTags >< '') then  /* if we have converted header tags */
      do
       FirstBit = FirstBit||Global.!CRLF||,  /* start a new source line */
                  '<HR SIZE=2>'||Global.!CRLF||,  /* add a horizontal rule */
                  '<B>NOTE:</B>'||Global.!CRLF||,  /* add bold text */
                  '<BR>'||Global.!CRLF||,  /* add a line break */
                  '<I>RexxMail removed the following header content:</I>'||Global.!CRLF||,  /* add italic text */
                  HeaderTags||,  /* add the rewritten header tags */
                  '<HR SIZE=2>'||Global.!CRLF  /* and end with another horizontal rule */
      end

     end

     otherwise  /* if none of the above apply */
     do
      FirstBit = FirstBit||'<'||TagContent||'>'  /* restore the tag */
     end

    end

   end

  end

  NewHTMLCont = NewHTMLCont||FirstBit  /* add the first bit to the new content */

 end

end

/**************************************************************************/
HaltHTMLFilter:  /* handles halt locally */
/**************************************************************************/

return NewHTMLCont  /* end of HTMLFilter */

/**************************************************************************/
HTMLToText: procedure expose Global.  /* converts (ISO 8859-1) HTML to plain text */
/**************************************************************************/

signal on halt name HaltHTMLToText  /* handles halt locally */

parse arg HTMLCont,GotUTF  /* get the arguments */

GotUTF = (GotUTF == 1)  /* 1 = TRUE */
Text = ''  /* start with nothing */
HTMLCont = strip(HTMLCont,'B',' ')  /* get rid of excess blanks */

if (Global.!Settings.BodyLineLength == '') then  /* if this one is undefined */
do
 SeparatorLength = 42  /* set the separator length to this */
end
else  /* if we have a value */
do
 SeparatorLength = Global.!Settings.BodyLineLength  /* use it */
end

if (translate(right(HTMLCont,7)) >< '</HTML>') then  /* if the HTML stuff does not end with the correct end tag */
do
 HTMLCont = HTMLCont||'</HTML>'  /* add it to make sure the text part gets bumped out */
end

HTMLCodes = copies('_ ',31),  /* 31 blanks */
            'nbsp',  /* non-breaking space */
            '_',  /* 1 blank */
            'quot',  /* double quotation mark */
            copies('_ ',3),  /* 3 blanks */
            'amp',  /* ampersand */
            copies('_ ',21),  /* 21 blanks */
            'lt',  /* less-than sign */
            '_',  /* 1 blank */
            'gt',  /* greater-than sign */
            copies('_ ',65),  /* 65 blanks */
            'euro',  /* euro sign */
            copies('_ ',32),  /* 32 blanks */
            'iexcl',  /* inverted exclamation mark */
            'cent',  /* cent sign */
            'pound',  /* pound sterling sign */
            'curren',  /* general currency sign */
            'yen',  /* yen sign */
            'brvbar',  /* broken vertical bar */
            'sect',  /* section sign */
            'uml',  /* umlaut (dieresis) */
            'copy',  /* copyright */
            'ordf',  /* feminine ordinal */
            'laquo',  /* left angle quote, guillemet left */
            'not',  /* not sign */
            'shy',  /* soft hyphen */
            'reg',  /* registered trademark */
            'macr',  /* macron accent */
            'deg',  /* degree sign */
            'plusmn',  /* plus or minus */
            'sup2',  /* superscript two */
            'sup3',  /* superscript three */
            'acute',  /* acute accent */
            'micro',  /* micro sign */
            'para',  /* paragraph sign */
            'middot',  /* middle dot */
            'cedil',  /* cedilla */
            'sup1',  /* superscript one */
            'ordm',  /* masculine ordinal */
            'raquo',  /* right angle quote, guillemet right */
            'frac14',  /* fraction one quarter */
            'frac12',  /* fraction one half */
            'frac34',  /* fraction three quarters */
            'iquest',  /* inverted question mark */
            'Agrave',  /* uppercase A, grave */
            'Aacute',  /* uppercase A, acute */
            'Acirc',  /* uppercase A, circumflex */
            'Atilde',  /* uppercase A, tilde */
            'Auml',  /* uppercase A, dieresis or Umlaut */
            'Aring',  /* uppercase A, ring */
            'AElig',  /* uppercase AE diphthong (ligature) */
            'Ccedil',  /* uppercase C, cedilla */
            'Egrave',  /* uppercase E, grave */
            'Eacute',  /* uppercase E, acute */
            'Ecirc',  /* uppercase E, circumflex */
            'Euml',  /* uppercase E, dieresis or Umlaut */
            'Igrave',  /* uppercase I, grave */
            'Iacute',  /* uppercase I, acute */
            'Icirc',  /* uppercase I, circumflex */
            'Iuml',  /* uppercase I, dieresis or Umlaut */
            'ETH',  /* uppercase Eth, Icelandic */
            'Ntilde',  /* uppercase N, tilde */
            'Ograve',  /* uppercase O, grave */
            'Oacute',  /* uppercase O, acute */
            'Ocirc',  /* uppercase O, circumflex */
            'Otilde',  /* uppercase O, tilde */
            'Ouml',  /* uppercase O, dieresis or Umlaut */
            'times',  /* multiply sign */
            'Oslash',  /* uppercase O, slash */
            'Ugrave',  /* uppercase U, grave */
            'Uacute',  /* uppercase U, acute */
            'Ucirc',  /* uppercase U, circumflex */
            'Uuml',  /* uppercase U, dieresis or Umlaut */
            'Yacute',  /* uppercase Y, acute */
            'THORN',  /* uppercase THORN, Icelandic */
            'szlig',  /* lowercase sharp s, sz ligature */
            'agrave',  /* lowercase a, grave */
            'aacute',  /* lowercase a, acute */
            'acirc',  /* lowercase a, circumflex */
            'atilde',  /* lowercase a, tilde */
            'auml',  /* lowercase a, dieresis or Umlaut */
            'aring',  /* lowercase a, ring */
            'aelig',  /* lowercase ae diphthong (ligature) */
            'ccedil',  /* lowercase c, cedilla */
            'egrave',  /* lowercase e, grave */
            'eacute',  /* lowercase e, acute */
            'ecirc',  /* lowercase e, circumflex */
            'euml',  /* lowercase e, dieresis or Umlaut */
            'igrave',  /* lowercase i, grave */
            'iacute',  /* lowercase i, acute */
            'icirc',  /* lowercase i, circumflex */
            'iuml',  /* lowercase i, dieresis or Umlaut */
            'eth',  /* lowercase eth, Icelandic */
            'ntilde',  /* lowercase n, tilde */
            'ograve',  /* lowercase o, grave */
            'oacute',  /* lowercase o, acute */
            'ocirc',  /* lowercase o, circumflex */
            'otilde',  /* lowercase o, tilde */
            'ouml',  /* lowercase o, dieresis or Umlaut */
            'divide',  /* division sign */
            'oslash',  /* lowercase o, slash */
            'ugrave',  /* lowercase u, grave */
            'uacute',  /* lowercase u, acute */
            'ucirc',  /* lowercase u, circumflex */
            'uuml',  /* lowercase u, dieresis or Umlaut */
            'yacute',  /* lowercase y, acute */
            'thorn',  /* lowercase thorn, Icelandic */
            'yuml'  /* lowercase y, dieresis or umlaut */
UniToISO = ' '||"'"||' f " ... [+] [++] ^ [0/00] S < OE '||"' '"||' " "  - -- ~ [TM] s > oe Y'  /* the UniCode to ISO conversion table */
UniCodes = '8364',  /* euro */
           '8218',  /* Single Low-9 Quotation Mark */
            '402',  /*  */
           '8222',  /* Double Low-9 Quotation Mark */
           '8230',  /* Horizontal Ellipsis */
           '8224',  /* Dagger */
           '8225',  /* Double Dagger */
            '710',  /* Modifier Letter Circumflex Accent */
           '8240',  /* Per Mille Sign */
            '352',  /* Latin Capital Letter S With Caron */
           '8249',  /* Single Left-Pointing Angle Quotation Mark */
            '338',  /* Latin Capital Ligature OE */
           '8216',  /* Left Single Quotation Mark */
           '8217',  /* Right Single Quotation Mark */
           '8220',  /* Left Double Quotation Mark */
           '8221',  /* Right Double Quotation Mark */
           '8226',  /* Bullet */
           '8211',  /* En Dash */
           '8212',  /* Em Dash */
            '732',  /* Small Tilde */
           '8482',  /* Trade Mark Sign */
            '353',  /* Latin Small Letter S With Caron */
           '8250',  /* Single Right-Pointing Angle Quotation Mark */
            '339',  /* Latin Small Ligature OE */
            '376'  /* Latin Capital Letter Y With Diaeresis */
IndentTags = 'DL',  /* <DL> */
             'OL',  /* <OL> */
             'UL'  /* <UL> */
UnIndentTags = '/DL',  /* </DL> */
               '/OL',  /* </OL> */
               '/UL'  /* </UL> */
NewLineTags = 'BR',  /* <BR> */
              'DIV',  /* <DIV> */
              'P',  /* <P> */
              'HR',  /* <HR> */
              'DT',  /* <DT> */
              'DD',  /* <DD> */
              'LI',  /* <LI> */
              'H1',  /* <H1> */
              '/H1',  /* </H1> */
              'H2',  /* <H2> */
              '/H2',  /* </H2> */
              'H3',  /* <H3> */
              '/H3',  /* </H3> */
              'H4',  /* <H4> */
              '/H4',  /* </H4> */
              'H5',  /* <H5> */
              '/H5',  /* </H5> */
              'H6',  /* <H6> */
              '/H6',  /* </H6> */
              'TR',  /* <TR> */
              '/TR',  /* </TR> */
              'PRE',  /* <PRE> */
              '/PRE',  /* </PRE> */
              'BLOCKQUOTE',  /* <BLOCKQUOTE> */
              '/BLOCKQUOTE',  /* </BLOCKQUOTE> */
              'TITLE',  /* <TITLE> */
              '/TITLE',  /* </TITLE> */
              '/HTML'  /* </HTML> */
AddText = ''  /* start with nothing to add */
TypeStack = ''  /* start with an empty type stack */
PreText = 0  /* no preformatted text yet */
IndentLevel = 0  /* no indent yet */
Indent = ''  /* no indent yet */
TagIndent = ''  /* no tag indent yet */
URLCount = 0  /* start a counter */
URLList. = ''  /* start with no URLs */

do while (HTMLCont >< '')  /* as long as we have HTML text left */

 parse var HTMLCont NextPart '<' TagCont '>' HTMLCont  /* get the next part before any tag, and keep the tag content */
 TagCont = strip(TagCont)  /* remove excess whitespace */

 if (TagCont >< '') then  /* if we have tag content */
 do

  TagCont = strip(TagCont,'L','<')  /* get rid of any double < */
  parse var TagCont TagCont '<' TagRest  /* look for any < within the tag content */

  if (TagRest >< '') then  /* if we did find a < in our tag */
  do
   HTMLCont = '<'||TagRest||HTMLCont  /* we must have a missing end > for our tag, so restore the next tag */
  end

  TagWord = translate(word(TagCont,1))  /* get the first word of the tag content in upper case */

  select  /* do one of the following */

   when (left(TagWord,3) == '!--') then  /* if it is the start of a comment */
   do

    if (right(TagCont,2) >< '--' ) then  /* if the tag content does not end with a comment terminator, the end tag we have must be part of the comment */
    do
     parse var HTMLCont . '-->' HTMLCont  /* so remove the remaining comment from the rest of the HTML text */
    end

    TagCont = ''  /* get rid of the tag content to prevent further processing */

   end

   when (TagWord == 'META') then  /* if it is the start of a meta tag */
   do

    if ((pos('UTF-8',translate(TagCont)) > 0) & (\GotUTF)) then  /* if we have UTF-8 content and it was not decoded before */
    do
     HTMLCont = DecodeUTF8(HTMLCont)  /* convert the HTML content */
     GotUTF = 1  /* set the flag just in case we find more than one UTF-8 meta tag */
    end

   end

   when (TagWord == 'SCRIPT') then  /* if it is the start of a script */
   do

    do until (TagWord == '/SCRIPT')  /* go on until we get to the end of the script */
     parse var HTMLCont . '<' TagCont '>' HTMLCont  /* look for the next tag */
     TagWord = translate(word(TagCont,1))  /* get the first word of the tag content in upper case */
    end

    HTMLCont = '<P>'||HTMLCont  /* add a new paragraph -- scripts may insert whitespace */
    TagCont = ''  /* get rid of the tag content to prevent further processing */

   end

   when (TagWord == 'STYLE') then  /* if it is the start of a style */
   do

    do until (TagWord == '/STYLE')  /* go on until we get to the end of the style */
     parse var HTMLCont . '<' TagCont '>' HTMLCont  /* look for the next tag */
     TagWord = translate(word(TagCont,1))  /* get the first word of the tag content in upper case */
    end

    TagCont = ''  /* get rid of the tag content to prevent further processing */

   end

   otherwise  /* if none of the above apply */
   do
    nop  /* do nothing */
   end

  end

 end

 if (NextPart >< '') then  /* if we have something before the tag */
 do

  do while (NextPart >< '')  /* as long as we have something left */

   parse var NextPart BeforePart '&' CharCode ';' NextPart  /* get the bit before any character code, and keep the code */

   if (CharCode >< '') then  /* if we have a character code */
   do

    if ((left(CharCode,1) == '#') & (datatype(substr(CharCode,2),'W'))) then  /* if the code is a character number */
    do

     CharCode = substr(CharCode,2)  /* lose the # */

     if (CharCode > 255) then  /* if it looks like unicode */
     do

      CharPos = wordpos(CharCode,UniCodes)  /* look for the code in our list */

      if (CharPos > 0) then  /* if we found it */
      do
       Character = word(UniToISO,CharPos)  /* convert the code to the corresponding ISO-8859-1 character(s) */
      end
      else  /* if not */
      do
       Character = '[#'||CharCode||']'  /* use the original code like this */
      end

     end
     else  /* if not, it should be an ISO-8859-1 3-digit code */
     do
      Character = d2c(CharCode)  /* simply convert the code to the corresponding ISO-8859-1 character */
     end

    end
    else  /* if the character code is not a number */
    do

     Character = d2c(wordpos(CharCode,HTMLCodes))  /* look for the character code in the HTML codes string and convert the code to the corresponding ISO-8859-1 character */

     if (Character == d2c(0)) then  /* if we did not find it */
     do
      Character = CharCode  /* use the original code; it's probably malformed text */
     end

    end

    BeforePart = BeforePart||Character  /* add the character to what we have */

   end

   AddText = AddText||BeforePart  /* add the result to what we have */

  end

 end

 if (TagCont >< '') then  /* if we have tag content */
 do

  select  /* do one of the following */

   when ((TagWord == 'A') & (Global.!Settings.HTMLURLList)) then  /* if it is an anchor and we want URLs listed */
   do

    parse var TagCont . '=' URLCont  /* get the URL content */

    if (URLCont >< '') then  /* if we have URL content */
    do

     if (pos('"',URLCont) > 0) then  /* if it looks like a correctly formatted URL */
     do
      parse var URLCont '"' URLCont '"'  /* get the bit we want */
     end

     URLCount = URLCount + 1  /* up the URL counter */
     URLList.URLCount = URLCont  /* store the URL */
     AddText = AddText||' [URL '||URLCount||'] '  /* add a marker to the text */

    end

   end

   when (PreText) then  /* if we are dealing with preformatted text */
   do

    if (TagWord == '/PRE') then  /* if the tag signals the end of preformatted text */
    do

     if (IndentLevel == 0) then  /* if we have no indent */
     do
      Text = Text||AddText  /* just add it */
     end
     else  /* if we have an indent value */
     do

      do while (AddText >< '')  /* walk through the part we have collected */
       parse var AddText NextBit (Global.!CRLF) AddText  /* get the bit up to the next CRLF */
       Text = Text||Indent||NextBit||Global.!CRLF  /* add it to what we have, with an indent */
      end

     end

     PreText = 0  /* reset the preformatted text flag */

    end
    else  /* if it is not the end of the preformatted text */
    do
     AddText = AddText||'<'  /* add the left angle bracket to what we have */
     HTMLCont = TagCont||'>'||HTMLCont  /* restore the remainder to be processed next time around */
    end

   end
   
   otherwise  /* if none of the above */
   do

    if (wordpos(TagWord,IndentTags||' '||UnIndentTags||' '||NewLineTags) > 0) then  /* if it is one of these */
    do

     NewText = ''  /* start with nothing */

     if (AddText >< '') then  /* if we have something to add */
     do

      AddText = strip(translate(AddText,'   ',d2c(9)||Global.!CRLF),'B',' ')  /* take what we have, turn TAB and CRLF into space, and remove excess space */

      do while (AddText >< '')  /* as long as we have something left */
       parse var AddText NextWord AddText  /* get the next word (to get rid of excess whitespace) */
       NewText = NewText||NextWord||' '  /* and add it to what we have, followed by a space */
      end

      NewText = strip(NewText,'T',' ')  /* remove the trailing blank */

     end

     if (NewText >< '') then  /* if we have something to add */
     do

      if (Text >< '') then  /* if previous text exists */
      do

       if (right(Text,2) == Global.!CRLF) then  /* if the previous text ends with a CRLF */
       do
        Text = Text||Indent  /* add an indent, if any */
       end

      end

      if (Global.!Settings.BodyLineLength >< '') then  /* if we want word-wrapped text */
      do
       NewText = WordWrap(NewText,(Global.!Settings.BodyLineLength - length(Indent),1,Indent))  /* word-wrap the new text part with a full indent, break words if necessary */
      end

      Text = Text||NewText||Global.!CRLF  /* add the new text and a new line */

     end
     else  /* if we have nothing to add */
     do

      if ((wordpos(TagWord,NewLineTags) > 0) & (Text >< '')) then  /* if it is one of these and we already have text */
      do

       if (right(Text,4) >< Global.!CRLF||Global.!CRLF) then  /* if the previous text does not end with a double CRLF */
       do
        Text = Text||Global.!CRLF  /* add a new line */
       end

      end

     end

     select  /* do one of the following */

      when (wordpos(TagWord,IndentTags) > 0) then  /* if it is one of these */
      do

       TypeStack = TagWord||' '||TypeStack  /* push the tag word onto the type stack */
       IndentLevel = IndentLevel + 1  /* move up */
       Indent = copies(' ',IndentLevel * 4)  /* the new indent string */

       if (TagWord == 'OL') then  /* if we are starting an ordered list */
       do
        OListCount.IndentLevel = 0  /* start a new list counter */
       end

      end

      when (wordpos(TagWord,UnIndentTags) > 0) then  /* if it is one of these */
      do

       if (TagWord == '/OL') then  /* if we are ending an ordered list */
       do
        drop OListCount.IndentLevel  /* drop the list counter */
       end

       TypeStack = subword(TypeStack,2)  /* remove the tag word from the type stack */
       IndentLevel = IndentLevel - 1  /* move back */
       Indent = copies(' ',IndentLevel * 4)  /* the new indent string */

      end

      otherwise  /* it must be a new line tag */
      do

       select  /* do one of the following */

        when (TagWord == 'P') then  /* if it is a paragraph marker */
        do
         Text = Text||Global.!CRLF  /* add another CRLF */
        end

        when (TagWord == 'PRE') then  /* if it is the start of preformatted text */
        do
         PreText = 1  /* set the preformatted text flag */
        end

        when (TagWord == 'DT') then  /* if it is a definition tag */
        do

         if (IndentLevel > 1) then  /* if we have an indent */
         do
          TagIndentLevel = IndentLevel - 1  /* go back one */
          TagIndent = copies(' ',TagIndentLevel * 4)  /* the new tag indent string */
         end
         else  /* if we have no indent */
         do
          TagIndentLevel = 1  /* "nothing" */
         end

         Text = Text||TagIndent||'  - '  /* tag indent plus a marker */

        end

        when (TagWord == 'DD') then  /* if it is a definition definition */
        do
         Text = Text||Indent  /* full indent */
        end

        when (TagWord == 'LI') then  /* if it is a list item */
        do

         if (IndentLevel > 1) then  /* if we have an indent */
         do
          TagIndentLevel = IndentLevel - 1  /* go back one */
          TagIndent = copies(' ',TagIndentLevel * 4)  /* the new tag indent string */
         end
         else  /* if we have no indent */
         do
          TagIndentLevel = 1  /* "nothing" */
         end

         if (word(TypeStack,1) == 'OL') then  /* if we are in an ordered list */
         do
          OListCount.IndentLevel = OListCount.IndentLevel + 1  /* up the current list counter */
          Text = Text||TagIndent||right(OListCount.IndentLevel,2,' ')||'. '  /* indent, the number, a full stop, and a space */
         end
         else  /* we must be in an unordered list */
         do
          Text = Text||TagIndent||'  - '  /* indent and a marker */
         end

        end

        when ((wordpos(TagWord,'HR H1 H2 H3 H4 H5 H6 TR TITLE') > 0) & (Global.!Settings.HTMLLines)) then  /* if it is one of these, and we want lines */
        do

         Text = Text||Indent||copies('_',(SeparatorLength - length(Indent)))||Global.!CRLF  /* full indent, a starting line and a new line */

         if (TagWord == 'HR') then  /* if it is a separator line we want */
         do
          Text = Text||Indent||copies(d2c(175),(SeparatorLength - length(Indent)))||Global.!CRLF  /* add an ending line */
         end

        end

        when ((wordpos(TagWord,'/H1 /H2 /H3 /H4 /H5 /TR /TITLE') > 0) & (Global.!Settings.HTMLLines)) then  /* if it is one of these, and we want lines */
        do
         Text = Text||Indent||copies(d2c(175),(SeparatorLength - length(Indent)))||Global.!CRLF  /* add an ending line */
        end

        otherwise  /* if it is none of the above */
        do
         nop  /* do nothing else */
        end

       end

      end

     end

    end

   end

  end

 end

end

do Index = 1 to URLCount  /* for each of the URLs we found in the text */
 Text = Text||Global.!CRLF||'URL '||right(Index,length(URLCount),' ')||' = '||URLList.Index  /* add it to the text on a new line */
end

/**************************************************************************/
HaltHTMLToText:  /* handles halt locally */
/**************************************************************************/

return Text  /* end of HTMLToText */

/**************************************************************************/
InsertHeaderLine: procedure expose Global.  /* inserts a header keyword and entry */
/**************************************************************************/

parse arg HeadName,Keyword,Entry,Position  /* get the arguments */

call value HeadName||'0',(value(HeadName||'0') + 1)  /* up the last header line number */

do Index = value(HeadName||'0') to Position by -1  /* starting with the last header line, and ending with the one at Position */
 call value HeadName||Index||'.!Keyword',value(HeadName||(Index - 1)||'.!Keyword')  /* copy the previous keyword */
 call value HeadName||Index||'.!Entry',value(HeadName||(Index - 1)||'.!Entry')  /* copy the previous entry */
end

call value HeadName||Position||'.!Keyword',KeyWord  /* store the new keyword */
call value HeadName||Position||'.!Entry',Entry  /* store the new entry */

return 1  /* end of InsertHeaderLine */

/**************************************************************************/
IsMessage: procedure expose Global.  /* checks for message content */
/**************************************************************************/

parse arg MessFile  /* get the argument */

MessCont = GetFileContents(MessFile)  /* get the file contents */

if (\MessageContents(MessCont,'Global.!MessHead.','Global.!MessBody')) then  /* if we cannot get the file contents sorted out */
do
 return 0  /* quit with an error */
end

IsMess = 0  /* start with nothing */
IsMess = IsMess + HeaderContains('RECEIVED',,'Global.!MessHead.')  /* look for this header line */
IsMess = IsMess + HeaderContains('DATE',,'Global.!MessHead.')  /* look for this header line */
IsMess = IsMess + HeaderContains('MESSAGE-ID',,'Global.!MessHead.')  /* look for this header line */
IsMess = IsMess + HeaderContains('FROM',,'Global.!MessHead.')  /* look for this header line */
IsMess = IsMess + HeaderContains('TO',,'Global.!MessHead.')  /* look for this header line */
IsMess = IsMess + HeaderContains('SUBJECT',,'Global.!MessHead.')  /* look for this header line */

if (IsMess < 3) then  /* if we did not find a minimum of message header lines */
do
 call AddError 'Not a message file : "'||MessFile||'"'  /* report */
 return 0  /* quit with an error */
end

return 1  /* end of IsMessage */

/**************************************************************************/
LeapYear: procedure expose Global.  /* check for leap year */
/**************************************************************************/

parse arg Year  /* get the year to analyze */

if (Year == '') then  /* if we have nothing */
do
 Year = substr(date('S'),1,4)  /* use the current year */
end

if (Year // 4 == 0) then  /* if the year is divisible by 4 */
do

 if (Year // 100 == 0) then  /* and if the year is divisible by 100 */
 do

  if (Year // 400 == 0) then  /* and if the year is divisible by 400 */
  do
   Leap = 1  /* it is a leap year */
  end
  else  /* if the year is divisible by 100, but not by 400 */
  do
   Leap = 0  /* it is not a leap year */
  end

 end
 else  /* if the year is divisible by 4, but not by 100 */
 do
  Leap = 1  /* it is a leap year */
 end

end
else  /* if the year is not divisible by 4 */
do
 Leap = 0  /* it is not a leap year */
end

return Leap  /* end of LeapYear */

/**************************************************************************/
LoadRexxMailUtil: procedure expose Global.  /* try to load the RexxMailUtil functions library if necessary -- EXTERNAL -- RXMLUTIL.DLL */
/**************************************************************************/

if (rxfuncquery('RexxMailDropFuncs') == 0) then  /* if it is already loaded */
do
 return 1  /* OK */
end

if (rxfuncadd('RexxMailLoadFuncs','RxMlUtil','RexxMailLoadFuncs') >< 0) then  /* if we cannot register the loading function */
do
 call AddError 'Cannot register the RexxMail utilities library loading function'  /* report */
 return 0  /* bad result */
end

if (RexxMailLoadFuncs() == 0) then  /* if we cannot load the functions */
do
 call AddError 'Cannot load the RexxMail utilities library'  /* report */
 return 0  /* bad result */
end

return 1  /* end of LoadRexxMailUtil */

/**************************************************************************/
LoadRexxSock: procedure expose Global.  /* try to load the REXX socket functions library if necessary */
/**************************************************************************/

if (rxfuncquery('SockDropFuncs') == 0) then  /* if it is already loaded */
do
 return 1  /* OK */
end

if (rxfuncadd('SockLoadFuncs','RxSock','SockLoadFuncs') >< 0) then  /* if we cannot register the loading function */
do
 call AddError 'Cannot register REXX socket library loading function'  /* report */
 return 0  /* bad result */
end

if (sockloadfuncs() == 0) then  /* if we cannot load the functions */
do
 call AddError 'Cannot load REXX socket library'  /* report */
 return 0  /* bad result */
end

return 1  /* end of LoadRexxSock */

/**************************************************************************/
LoadRexxUtil: procedure expose Global.  /* try to load the REXX utilities library if necessary */
/**************************************************************************/

if (rxfuncquery('SysDropFuncs') == 0) then  /* if it is already loaded */
do
 return 1  /* OK */
end

if (rxfuncadd('SysLoadFuncs','RexxUtil','SysLoadFuncs') >< 0) then  /* if we cannot register the loading function */
do
 call AddError 'Cannot register REXX system library loading function'  /* report */
 return 0  /* bad result */
end

if (SysLoadFuncs() == 0) then  /* if we cannot load the functions */
do
 call AddError 'Cannot load REXX system library'  /* report */
 return 0  /* bad result */
end

return 1  /* end of LoadRexxUtil */

/**************************************************************************/
LocalURLTag: procedure expose Global.  /* converts an URL to the filename of a local attachment file */
/**************************************************************************/

parse upper arg TagContent,URLString  /* get the arguments in upper case */

parse var TagContent FirstTagContent (URLString) '=' URLContent RestTagContent  /* get the bits we want */

if (URLContent >< '') then  /* if we have URL content */
do

 URLContent = strip(strip(URLContent),'B','"')  /* remove any excess space and quotation marks */

 if (left(URLContent,4) == 'CID:') then  /* if it starts with CID: */
 do

  URLContent = substr(strip(URLContent,'B','"'),5)  /* get rid of any double quotation marks and remove the first four characters ('CID:') */
  SourceString = '<'||URLContent||'>'  /* add angle brackets to get a source name */
  parse upper var Global.!MessBody FirstBodyPart (SourceString) LastBodyPart +513 .  /* get the body part before and a bit after the source name part, both in upper case */
  FirstBodyPart = left(reverse(FirstBodyPart),512)  /* reverse the first body part and take the first bit */
  parse var FirstBodyPart FileNamePart '=EMAN' Remainder  /* get the file name part that follows 'NAME=' */

  if (Remainder == '') then  /* if we have no remainder, we have no file name part, so */
  do
   parse var LastBodyPart 'NAME=' FileNamePart  /* look for it in the last body part */
  end
  else  /* if we do have something */
  do
   FileNamePart = reverse(FileNamePart)  /* turn the file name part the right way round */
  end

  parse var FileNamePart FileNamePart (Global.!CRLF) .  /* remove any extra lines */
  parse var FileNamePart FileNamePart ';' .  /* remove any extra stuff (which would be after a semicolon) */
  FileNamePart = strip(FileNamePart,'B','"')  /* remove any double quotation marks */

 end
 else  /* if the URL does not start with CID: */
 do
  URLContent = reverse(URLContent)  /* turn the URL content back to front */
  parse var URLContent URLContent '/' .  /* dump everything after the first (i.e. preceding the last) slash to get rid of a subdir path */
  FileNamePart = reverse(URLContent)  /* turn the URL content back the right way */
 end

 TagContent = FirstTagContent||URLString||'="'||FileNamePart||'" '||RestTagContent  /* create new tag content */

end

return TagContent  /* end of LocalURLTag */

/**************************************************************************/
LogAction: procedure expose Global.  /* logs an action */
/**************************************************************************/

parse arg Action,Quiet  /* get the argument */

Quiet = (Quiet == 1)  /* a value of 1 indicates True */

if (\Quiet) then  /* unless we want to keep quiet about it*/
do
 call ShowLine Action  /* write the log text to the standard output device after clipping it to fit */
end

WriteLog = Global.!Settings.LogActionLines  /* copy the global log lines value */

if (WriteLog == '') then  /* if we have an empty value */
do
 WriteLog = 1  /* use this */
end

if (WriteLog > 0) then  /* if we are to log errors */
do

 if (\FileOpen(Global.!ActionLog,'WRITE')) then  /* if we cannot open the file for writing */
 do
  MessText = 'Cannot open log file "'||Global.!ActionLog||'"'  /* the text to display/send */
  call ShowLine MessText  /* report */
  call SystemMessage 'Action log file write error.',MessText  /* send a message to the user if required */
  return 0  /* return with no success */
 end

 ActionText = ''  /* start with nothing */

 do while (Action >< '')  /* as long as we have text left */

  parse var Action NextBit (Global.!CRLF) Action  /* get the bit up to the next CRLF */

  if (NextBit >< '') then  /* if we have something */
  do
   ActionText = ActionText||' '||strip(NextBit,'B',' ')  /* remove excess space and add it to what we have */
  end

 end

 call lineout Global.!ActionLog,DateTimeSys()||' :'||ActionText  /* write a date/time stamp and the new action text */
 call FileClose Global.!ActionLog  /* close the log file */

end

return 1  /* end of LogAction */

/**************************************************************************/
LogErrors: procedure expose Global.  /* processes and logs collected error messages */
/**************************************************************************/

parse arg ErrorLines  /* get the argument */

if (ErrorLines == '') then  /* if we have no errors */
do
 return 0  /* return with no further action */
end

if (symbol('Global.!Settings.Silent') == 'VAR') then  /* if this one is defined */
do

 if (Global.!Settings.Silent) then  /* if we want a silent exit */
 do
  return 1  /* return with an error status */
 end

end
else  /* if the var is undefined, we were in the early stages ofthe main program */
do
 Global.!Settings.Silent = 0  /* no silence required */
end

if (symbol('Global.!Settings.SignalError') == 'VAR') then  /* if this one is defined */
do
 call SoundSignal Global.!Settings.SignalError  /* signal if required */
end
else  /* if we have no error beep definition */
do
 call beep 333,333  /* sound a simple beep */
end

ErrorText = 'Build: '||Global.!Build||Global.!CRLF  /* the first line of the explanatory text */

if (symbol('Global.!StoreCommands') == 'VAR') then  /* if the command-line store has been defined */
do

 if (Global.!StoreCommands >< '') then  /* if we have command line arguments */
 do
  ErrorText = ErrorText||'Arguments: '||Global.!StoreCommands||Global.!CRLF  /* the second line of the explanatory text */
 end

end

if (lastpos(Global.!CRLF,ErrorLines) > pos(Global.!CRLF,ErrorLines)) then  /* if the error message contains more than one CRLF pair, i.e. if there is more than one message */
do
 Plural = 's'  /* we need a plural s */
end
else  /* if we have just the one entry */
do
 Plural = ''  /* no plural s needed */
end
 
ErrorText = ErrorText||'Error'||Plural||':'||Global.!CRLF||ErrorLines  /* another line of explanatory text, followed by the error lines */
call ShowLine  /* empty line */
call lineout 'CON:',ErrorText  /* show the error text */

if ((pos('Fatal error: ',ErrorLines) == 0) & (pos('command',ErrorLines) > 0)) then  /* if we do not have a fatal error, and we do have one or more command errors */
do
 call ShowLine 'Use the /help command switch for more information'  /* send instructions to the console */
end

if (symbol('Global.!Settings.ErrorMail') == 'VAR') then  /* if the ErrorMail settings has been set */
do

 if (Global.!Settings.ErrorMail) then  /* if we want an error message to be sent to the user */
 do

  if (symbol('Global.!InDir') == 'VAR') then  /* if the InDir has been set */
  do

   if (Global.!InDir >< '') then  /* if we have an incoming messages dir */
   do

    StatusMess = ''  /* start with no status message */

    if (symbol('Global.!User') == 'VAR') then  /* if we have a user name */
    do
     StatusMess = StatusMess||'User          = "'||Global.!User||'"'||Global.!CRLF  /* report */
    end

    if (symbol('Global.!MainDir') == 'VAR') then  /* if we have a main folder name */
    do
     StatusMess = StatusMess||'Main folder   = "'||Global.!MainDir||'"'||Global.!CRLF  /* report */
    end

    call SystemMessage 'RexxMail error.',ErrorText||Global.!CRLF||StatusMess  /* insert the error text and send it with the status text as a message to the user */

   end

  end

 end

end

if (symbol('Global.!Settings.LogErrorLines') >< 'VAR') then  /* if the LogErrorLines settings has not been set (i.e. we bombed out early) */
do
 Global.!Settings.LogErrorLines = 1  /* set it to 1 */
end

if (Global.!Settings.LogErrorLines == 0) then  /* if we are not to log errors */
do
 return 1  /* quit */
end

if (symbol('Global.!ErrorLog') >< 'VAR') then  /* if the error log name has not been set (i.e. we bombed out very early -- this should not happen) */
do
 call ShowLine 'No error log file defined.'  /* show an error text */
end

if (\FileOpen(Global.!ErrorLog,'WRITE')) then  /* if we cannot open the log file for writing */
do
 call SystemMessage 'Error log write error.',ErrorText||Global.!CRLF||,  /* report */
                    'Cannot write to error log file:'||Global.!CRLF||,  /* report */
                    '  '||Global.!ErrorLog  /* report */
 return 0  /* quit */
end

call charout Global.!ErrorLog,DateTimeSys()||' : '  /* write a date/time stamp to the log file */
FirstLine = 1  /* this is the first line */

do while (ErrorText >< '')  /* as long as we have error text left */

 parse var ErrorText NextError (Global.!CRLF) ErrorText  /* get the next error message */

 if (FirstLine) then  /* if this is the first line */
 do
  FirstLine = 0  /* the next line is not */
 end
 else  /* if this is not the first line */
 do
  call charout Global.!ErrorLog,copies(' ',22)  /* write an indent */
 end

 call lineout Global.!ErrorLog,NextError  /* write the error text */

end

call FileClose Global.!ErrorLog  /* close the log file */

return 1  /* end of LogErrors */

/**************************************************************************/
LogMail: procedure expose Global.  /* log a sent or collected message */
/**************************************************************************/

WriteLog = Global.!Settings.LogMailLines  /* copy the global log lines value */

if (WriteLog == '') then  /* if we have an empty value */
do
 WriteLog = 1  /* use this */
end

if (WriteLog > 0) then  /* if we are to log */
do

 if (\FileOpen(Global.!MailLog,'WRITE')) then  /* if we cannot open the log file for writing */
 do
  call AddError 'Cannot open log file "'||Global.!MailLog||'"'  /* report an error */
  return 0  /* return with no success */
 end

 parse arg Direction,Sender,Recipient,RecipientsTo,RecipientsCc,RecipientsAcc,RecipientsBcc,Subject,MessDate,MessID,Bytes  /* get the arguments */
 OutLine = DateTimeSys()||' -- '||Direction||' -- From: '||Sender||' -- '  /* write a date/time stamp, the direction, the sender, and some separators */

 if (Recipient >< '') then  /* if we have a lone recipient (incoming mail only) */
 do
  OutLine = OutLine||'For: '||Recipient||' -- '  /* add the recipient and a separator */
 end

 OutLine = OutLine||'To: '||RecipientsTo||' -- '  /* add the To recipients and a separator */

 if (RecipientsCc >< '') then  /* if we have Cc recipients */
 do
  OutLine = OutLine||'Cc: '||RecipientsCc||' -- '  /* add the Cc recipients and a separator */
 end

 if (RecipientsAcc >< '') then  /* if we have Acc recipients */
 do
  OutLine = OutLine||'Acc: '||RecipientsAcc||' -- '  /* add the Acc recipients and a separator */
 end

 if (RecipientsBcc >< '') then  /* if we have Bcc recipients */
 do
  OutLine = OutLine||'Bcc: '||RecipientsBcc||' -- '  /* add the Bcc recipients and a separator */
 end

 MessDate = DateTimeDisplay(MessDate,'OriginalISO')||' / '||DateTimeDisplay(MessDate,'UTCISO')  /* format the message date string */
 OutLine = OutLine||'Subject: '||Subject||' -- Date: '||MessDate||' -- Message-Id: '||MessID||' -- Bytes: '||Bytes  /* add the subject, date, message ID, and message size */
 call lineout Global.!MailLog,OutLine  /* write the log entry */
 call FileClose Global.!MailLog  /* close the log file */

end

return 1  /* end of LogMail */

/**************************************************************************/
MakeAddressObject: procedure expose Global.  /* adds a template to an address book folder */
/**************************************************************************/

parse arg FileName,Contents,Subject,Comments,Keywords  /* get the arguments */

if (pos(translate(Global.!AddressesDir),translate(FileName)) == 1) then  /* if the address template file is to be created in an address (sub)folder */
do
 Target = Global.!AddressesDir  /* we start here */
 FilePart = substr(FileName,(length(Global.!AddressesDir) + 2))  /* use the remainder minus the leading backslash */
end
else  /* if we are creating a template elsewhere */
do
 Target = left(FileName,2)  /* we start in the drive root */
 FilePart = substr(Filename,4)  /* remove the drive root and backslash */
end

PathPart = strip(filespec('P',FilePart),'T','\')  /* the path part */
NamePart = filespec('N',FilePart)  /* the name part */
Settings = 'ALWAYSSORT=YES;ICONVIEW=FLOWED,MINI;WORKAREA=YES;BACKGROUND='||Global.!IconDir||'\Folders\Mail_Background.BMP'  /* the settings to use for the folder object */

do while (PathPart >< '')  /* as long as the path continues */

 parse var PathPart Folder '\' PathPart  /* get the first part of the path */

 if (syscreateobject('WPFolder',Folder,Target,,'FAIL')) then  /* if we can create the folder */
 do
  call SetObjectData Target||'\'||Folder,Settings  /* adjust its settings */
 end

 Target = Target||'\'||Folder  /* move up the path for any next bit */

end

call sysdestroyobject FileName  /* remove any existing instance of this object */
call CreateCWMFObject NamePart,Target  /* try to create a CWMAILFile class object in the target dir using the name part */

if (PutFileContents(FileName,Contents)) then  /* if we can write the text content to the file */
do

 if (Subject >< '') then  /* if we have a subject */
 do

  do while (left(Subject,2) == Global.!CRLF)  /* as long as it starts with an empty line */
   Subject = substr(Subject,3)  /* remove it */
  end

  do while (right(Subject,2) == Global.!CRLF)  /* as long as it ends with an empty line */
   Subject = left(Subject,length(Subject) - 2)  /* remove it */
  end

  if (\PutObjectEA(FileName,'.SUBJECT',Subject,0)) then  /* if we cannot attach it to the file */
  do
   return 0  /* return an error */
  end

 end

 if (Comments >< '') then  /* if we have comments */
 do

  do while (left(Comments,2) == Global.!CRLF)  /* as long as it starts with an empty line */
   Comments = substr(Comments,3)  /* remove it */
  end

  do while (right(Comments,2) == Global.!CRLF)  /* as long as it ends with an empty line */
   Comments = left(Comments,length(Comments) - 2)  /* remove it */
  end

  if (\PutObjectEA(FileName,'.COMMENTS',Comments,1)) then  /* if we cannot attach them to the file */
  do
   return 0  /* return an error */
  end

 end

 if (Keywords >< '') then  /* if we have one or more keywords (groups) */
 do

  do while (left(Keywords,2) == Global.!CRLF)  /* as long as it starts with an empty line */
   Keywords = substr(Keywords,3)  /* remove it */
  end

  do while (right(Keywords,2) == Global.!CRLF)  /* as long as it ends with an empty line */
   Keywords = left(Keywords,length(Keywords) - 2)  /* remove it */
  end

  if (\PutObjectEA(FileName,'.KEYPHRASES',Keywords,1)) then  /* if we cannot attach them to the file */
  do
   return 0  /* return an error */
  end

 end

 if (\MessageSettings(FileName,'00000000','CHANGE')) then  /* if we can not make it a fresh outgoing mail message file */
 do
  call AddError 'Cannot set address template message attributes'  /* report */
  return 0  /* and quit */
 end

 if (\SetObjectData(FileName,'REXXMAILREFRESH=YES;TEMPLATE=YES;')) then  /* if we cannot make the file a template etc. */
 do
  return 0  /* and quit */
 end

end

return 1  /* end of MakeAddressObject */

/**************************************************************************/
MakeAttCopy: procedure expose Global.  /* do the actual attachment copying etc. for CopyAttachment */
/**************************************************************************/

parse arg FileName,AttDir,MessFile  /* get the arguments */

if (syscopyobject(FileName,AttDir)) then  /* if we can copy the file to the attachments folder */
do

 if (MessFile >< '') then  /* if we have a message file spec */
 do
  call MessageSettings MessFile,'****1***','CHANGE'  /* change the message settings to show an attachment */
  call LogAction 'File "'||FileName||'" attached to "'||MessFile||'"',1  /* report, quietly */
 end
 else  /* if we do not have a message file spec */
 do
  call LogAction 'File "'||FileName||'" copied to attachments folder "'||AttDir||'"',1  /* report, quietly */
 end

end
else  /* if we cannot copy the file to it */
do

 if (MessFile >< '') then  /* if we have a message file spec */
 do
  call AddError 'Cannot attach file "'||FileName||'" to message file; attachment may already exist'  /* report an error */
 end
 else  /* if we do not have a message file spec */
 do
  call AddError 'Cannot copy file "'||FileName||'" to attachments folder; attachment may already exist'  /* report an error */
 end

end

return  /* end of MakeAttCopy */

/**************************************************************************/
MakeForwardReplyMessage: procedure expose Global.  /* generate a forward or reply message and optionally open it */
/**************************************************************************/

parse arg Switch Addresses  /* get the arguments */

Reply = (pos('REPLY',Switch) > 0)  /* set a flag if we are replying */
MessFile = Global.!ProcFile  /* the file name to process */

if (Reply) then  /* if we are replying */
do

 if (\MessageSettings(MessFile,'1*******','MATCH')) then  /* if it is not an incoming message */
 do
  call AddError 'Invalid message type'  /* report */
  return 0  /* and quit */
 end

 AttDirArg = ''  /* no argument when we create the attachments dir later */

end
else  /* if we are forwarding */
do

 if (\(MessageSettings(MessFile,'1*******','MATCH') | MessageSettings(MessFile,'0*1*****','MATCH'))) then  /* if it is not an incoming message or a previously sent message */
 do
  call AddError 'Invalid message type'  /* report */
  return 0  /* and quit */
 end

 AttDirArg = MessFile  /* the argument when we create the attachments dir later */

end

MessFileDir = strip(filespec('D',MessFile)||filespec('P',MessFile),'T','\')  /* get the path to the mail message */

if ((pos(translate(Global.!MainDir),translate(MessFileDir)) == 0) & (Global.!Settings.UseCurrentDir)) then  /* if it is not in the RexxMail (sub)folders, and we want the output in the original dir */
do
 OutDir = MessFileDir  /* we will create the output file in the same folder as the original */
end
else  /* if the original message is in the RexxMail folders, oe we do not wat to use the original dir */
do

 if (Global.!DraftsDir >< '') then  /* if there is a Drafts folder */
 do
  OutDir = Global.!DraftsDir  /* use it for our output */
 end
 else  /* if not */
 do
  OutDir = Global.!OutDir  /* create the output file in the 'Out' folder */
 end

end

AttDir = AttDirCreate(AttDirArg)  /* get an attachments dir name, do not link it to the message if we are replying (i.e. make it a temp one) */
GotAtt = UnpackMessage(MessFile,AttDir,0)  /* unpack the message into the attachments dir (do not postprocess for viewing) and get the view text */

if (Global.!MessBody == '') then  /* if we have no view text at all */
do
 return 0  /* quit */
end

OutText = 'From: "'||Global.!Settings.Name||'" <'||Global.!Settings.Address||'>'||Global.!CRLF  /* start with a prefilled "From:" line */
MessageID = GetHeaderEntry('MESSAGE-ID','Global.!MessHead.')  /* try to get the message ID from the original message */
Subject = GetHeaderEntry('SUBJECT','Global.!MessHead.')  /* try to get the subject from the original message */

if (Subject == '') then  /* if we have no subject */
do
 Subject = '(no subject)'  /* substitute this */
end

if (Reply) then  /* if we are replying */
do

 ForwardReplyText = Global.!Settings.ReplyText  /* use this for the forward/reply text setup */
 ReplyAddress = GetHeaderEntry('REPLY-TO','Global.!MessHead.')  /* look for "Reply-To:" addresses */

 if (ReplyAddress == '') then  /* if we have nothing */
 do

  ReplyAddress = GetHeaderEntry('FROM','Global.!MessHead.')  /* look for the "From:" address */

  if (ReplyAddress == '') then  /* if we have nothing */
  do

   ReplyAddress = GetHeaderEntry('RESENT-FROM','Global.!MessHead.')  /* get the last resender address, if any */

   if (ReplyAddress == '') then  /* if we have nothing */
   do

    ReplyAddress = GetHeaderEntry('APPARENTLY-FROM','Global.!MessHead.')  /* look for the apparent sender */

    if (ReplyAddress == '') then  /* if we have nothing */
    do
     ReplyAddress = GetHeaderEntry('SENDER','Global.!MessHead.')  /* look for a "Sender:" address */
    end

   end

  end

 end

 ToCcAddress = GetHeaderEntry('TO','Global.!MessHead.')  /* get the "To:" addresses */

 if (ToCcAddress == '') then  /* if there is no "To:" address */
 do
  ToCcAddress = GetHeaderEntry('APPARENTLY-TO','Global.!MessHead.')  /* look for an "Apparently-To:" address */
 end

 ToCcAddress = ToCcAddress||', '||GetHeaderEntry('CC','Global.!MessHead.')  /* add any "Cc:" addresses */
 RecipientAddress = ''  /* start with nothing */

 if (ToCcAddress >< '') then  /* if we have something */
 do

  Separator = d2c(0)  /* the separator for the next actions */
  ToCcAddress = AddressFormat(ToCcAddress,1,0)  /* format to get full addresses separated by null bytes */

  do while (ToCcAddress >< '')  /* as long as we have something */

   parse var ToCcAddress NextAddress (Separator) ToCcAddress  /* get the next address */
   BareAddress = reverse(NextAddress)  /* reverse the address */
   parse var BareAddress '>' BareAddress '<' .  /* get the bare address */
   BareAddress = reverse(translate(BareAddress))  /* make upper case and reverse back */

   if (wordpos('<'||BareAddress||'>',translate(RecipientAddress)) == 0) then  /* if it is a new address */
   do

    if (BareAddress >< translate(Global.!Settings.Address)) then  /* if it is not our own address */
    do
     RecipientAddress = RecipientAddress||', '||NextAddress  /* add it to the recipients list */
    end

   end

  end

 end

 ListAddress = GetHeaderEntry('LIST-POST','Global.!MessHead.')  /* get the list address, if any */

 if (ListAddress >< '') then  /* if we found something */
 do
  parse var ListAddress . ':' ListAddress '>'  /* get the bit we want */
  ListAddress = '<'||ListAddress||'>'  /* add angle brackets */
 end
 else  /* if not */
 do
  ListAddress = GetHeaderEntry('RESENT-FROM','Global.!MessHead.')  /* get the last resender address, if any */
 end

 select  /* do one of the following */

  when (pos('REPLYTOALL',Switch) == 1) then  /* if we are replying to everybody we can find */
  do

   SendTo = ReplyAddress  /* start with the reply address */

   if (RecipientAddress >< '') then  /* if we have recipient addresses */
   do
    SendTo = SendTo||', '||RecipientAddress  /* add them */
   end

   if (ListAddress >< '') then  /* if we have a list address */
   do
    SendTo = SendTo||', '||ListAddress  /* add it */
   end

  end

  when (pos('REPLYTOLIST',Switch) == 1) then  /* if we are replying to a list */
  do
   SendTo = ListAddress  /* use the list address */
  end

  when (pos('REPLYTORECIPIENTS',Switch) == 1) then  /* if we are replying to the recipients addresses */
  do
   SendTo = RecipientAddress  /* use the recipients addresses */
  end

  otherwise  /* if we have none of the above special cases */
  do
   SendTo = ReplyAddress  /* use the reply @address */
  end

 end

 if (SendTo == '') then  /* if we have no address */
 do
  call AddError 'Cannot find a reply address'  /* report */
  return 0  /* and return with an error */
 end

 if (pos('ASLIST',Switch) > 0) then  /* if we are replying through a new local list */
 do

  ListName = TempFileName(Global.!ConfigurationDir||'\lists')  /* get a unique file name in the Lists dir */

  if (ListName == '') then  /* if we cannot get a temp file */
  do
   return 0  /* quit with no success */
  end

  if (\PutFileContents(ListName,SendTo)) then  /* if we cannot write the addresses to the list file */
  do
   return 0  /* no success */
  end

  SendTo = filespec('N',ListName)||':;'  /* the list address to use in the new "To:" line below */

 end

 OutText = OutText||'To: '||SendTo||Global.!CRLF  /* add a "To:" line */

 if (MessageID >< '') then  /* if there is a message ID */
 do
  OutText = OutText||'In-Reply-To: '||MessageID||Global.!CRLF  /* add the reply info line */
 end

 if ((translate(left(Subject,3)) >< 'RE:') & (translate(left(Subject,4)) >< 'RE :')) then  /* if the subject does not start with "Re:" or "Re :" */
 do
  Subject = 'Re: '||Subject  /* add it to the header */
 end

end
else  /* if we are forwarding */
do

 ForwardReplyText = Global.!Settings.ForwardText  /* use this for the forward/reply text setup */
 OutText = OutText||'To: '||Addresses||Global.!CRLF  /* add a "To:" line, inserting any addresses we may have received */

 if (MessageID >< '') then  /* if there is a message ID */
 do
  OutText = OutText||'References: '||MessageID||Global.!CRLF  /* add a references line */
 end

 if (translate(word(Subject,1)) >< 'FWD:') then  /* if the subject does not start with "Fwd:" */
 do
  Subject = 'Fwd: '||Subject  /* add it */
 end

end

OutText = OutText||'Subject: '||Subject||Global.!EmptyLine  /* add the subject line to the output text followed by an empty line */
OutText = OutText||MakeInsertText(ForwardReplyText,'Global.!MessHead.',Global.!MessBody)  /* add the forward/reply text, if any */
OutFile = TempFileName(OutDir,,'FRR')  /* create a unique temporary file name */

if (OutFile == '') then  /* if we cannot get a temp file */
do
 return 0  /* quit with no success */
end

OutName = filespec('N',OutFile)  /* get the name part */
call CreateCWMFObject OutName,OutDir  /* try to create a CWMAILFile class object in the destination dir using the name part */

if (\FileOpen(OutFile,'WRITE')) then  /* if we cannot open the output file for writing */
do
 return 0  /* and return with an error */
end

call charout OutFile,OutText  /* write the output text to the new message file */
call FileClose OutFile  /* close the output file */

if (Reply) then  /* if we are replying */
do
 call sysdestroyobject AttDir  /* get rid of the attachments folder -- we need only the view file for a reply */
 GotAtt = 0  /* no attachments */
end
else  /* if we are forwarding or resending */
do

 if (\AttDirLink(OutFile,AttDir)) then  /* if we cannot link the attachments dir to the new message */
 do
  return 0  /* return with an error */
 end

 call sysfiletree AttDir||'\*','Files.','FOS'  /* have we got anything in the attachments dir(s)? */
 GotAtt = (Files.0 > 0)  /* set a flag if we have attachments */

end

call MessageSettings OutFile,'0000'||GotAtt||'000','CHANGE'  /* make it a new outgoing mail message file and set the attachments flag if necessary */

if (pos('OPEN',Switch) > 0) then  /* if we want the message opened on the desktop for editing */
do
 ObjectID = '<REXXMAIL_FRM_'||filespec('N',OutFile)||'>'  /* object ID */
 call SetObjectData OutFile,'OBJECTID='||ObjectID  /* set an object ID */
 call SetTitle OutFile  /* set the file title */
 call sysopenobject ObjectID,0,1  /* open the new message */
end
else  /* if we want to just leave it there */
do
 call SetTitle OutFile  /* set the file title */
end

return 1  /* end of MakeForwardReplyMessage */

/**************************************************************************/
MakeInsertText: procedure expose Global.  /* prepares text to insert in headers and reply/forwarded messages */
/**************************************************************************/

parse arg SetupString,HeadName,Body  /* get the arguments */

TextLine = SetupString  /* copy the setup string */
NewText = ''  /* nothing yet */

do while (TextLine >< '')  /* as long as we have a reply formula */

 parse var TextLine NextBit '#' TextLine  /* look for the next bit and anything following a placeholder */
 NewText = NewText||NextBit  /* start by adding the first bit */

 if (TextLine >< '') then  /* if we found more, it must include a macro */
 do

  AddText = ''  /* nothing to add yet */
  parse var TextLine NextChar +1 TextLine  /* get the next character */

  select  /* do one of the following */

   when (NextChar == '#') then  /* if it is this, it is an escape character */
   do
    AddText = '#'  /* restore the # */
   end

   when (NextChar == 'D') then  /* if it is this */
   do

    AddText = GetHeaderEntry('DATE',HeadName)  /* try to get a message date */

    if (AddText >< '') then  /* if we have something */
    do
     AddText = DateTimeDisplay(AddText,Global.!Settings.DateTimeBody)  /* turn it into the type of date display we need */
    end
    else  /* if we have no date */
    do
     AddText = 'an unknown date'  /* use this */
    end

   end

   when (NextChar == 'F') then  /* if it is this */
   do

    AddText = GetHeaderEntry('FROM',HeadName)  /* try to get a sender */

    if (AddText >< '') then  /* if we have something */
    do
     AddText = AddressIndent(AddressFormat(AddText,1,0))  /* reformat the address(es), full, no check, then replace null byte spacers with commas and spaces */
    end
    else  /* if we have no sender */
    do
     AddText = 'an unknown sender'  /* use this */
    end

   end

   when (NextChar == 'L') then  /* if it is this */
   do
    AddText = Global.!CRLF  /* add a new line */
   end

   when ((NextChar == 'O') & (Body >< '')) then  /* if it is this */
   do
    AddText = Body  /* use the original body text */
   end

   when ((NextChar == 'Q') & (Body >< '')) then  /* if it is this */
   do
    AddText = WordWrap('> '||Body,Global.!Settings.ReplyLineLength,1,'> ')  /* word-wrap the text with quotation signs, break if necessary */
   end

   when (NextChar == 'S') then  /* if it is this */
   do

    AddText = GetHeaderEntry('SUBJECT',HeadName)  /* try to get the subject */

    if (AddText == '') then  /* if we have nothing */
    do
     AddText = 'an unknown subject'  /* use this */
    end

   end

   when (NextChar == 'T') then  /* if it is this */
   do

    AddText = GetHeaderEntry('TO',HeadName)  /* try to get the recipient(s) */

    if (AddText >< '') then  /* if we have something */
    do
     AddText = AddressIndent(AddressFormat(AddText,1,0))  /* reformat the address(es), full, no check, then replace null byte spacers with commas and spaces */
    end
    else  /* if we have no recipients */
    do
     AddText = 'an unknown recipient'  /* use this */
    end

   end

   otherwise  /* if none of the above */
   do
    call AddError 'Invalid macro #'||NextChar||' in setup string: "'||SetupString||'"'  /* report (not fatal) */
   end

  end

  NewText = NewText||AddText  /* add the result */

 end

end

if (NewText >< '') then  /* if we have a result */
do
 NewText = NewText||Global.!CRLF  /* add a CRLF combo */
end

return NewText  /* end of MakeInsertText */

/**************************************************************************/
MakeNewMessage: procedure expose Global.  /* generate a new message to a recipient */
/**************************************************************************/

parse arg MailTo,Switch  /* get the arguments */

if (Global.!Settings.UseCurrentDir) then  /* if we want the new message in the current dir */
do
 MessDir = strip(directory(),'T','\')  /* use the current dir name without a trailing backslash (we need that bit if we are in a root dir) */
end
else  /* if we are to put the message in the default dir */
do

 if (Global.!DraftsDir >< '') then  /* if there is a Drafts folder */
 do
  MessDir = Global.!DraftsDir  /* use it for our output */
 end
 else  /* if not */
 do
  MessDir = Global.!OutDir  /* create the output file in the 'Out' folder */
 end

end

MessFile = TempFileName(MessDir,,'NEW')  /* get a unique temporary file name */

if (MessFile == '') then  /* if we cannot get a temp file */
do
 return 0  /* quit with no success */
end

MessName = filespec('N',MessFile)  /* get the name part */
call CreateCWMFObject MessName,MessDir  /* try to create a CWMAILFile class object in the destination dir using the name part */

if (\FileOpen(MessFile,'WRITE')) then  /* if we cannot open the message file for writing */
do
 call sysdestroyobject MessFile  /* get rid of the useless object */
 return 0  /* and return with an error */
end

call FileClose MessFile  /* close the Mess file */

AttDir = AttDirCreate(MessFile)  /* create an attachments dir and link it to the message file */
FromCont = 'From: "'||Global.!Settings.Name||'" <'||Global.!Settings.Address||'>'||Global.!CRLF  /* start with a prefilled "From:" line */
GotSubject = 0  /* no subject line yet */
GotAttach = 0  /* no attachments yet */

if (MailTo >< '') then  /* if we have a mailto string */
do

 parse var MailTo MailTo ' mailto:' .  /* remove any double mailto: entry (Mozilla!) */

 if ((left(MailTo,1) == '"') & (right(MailTo,1) == '"')) then  /* if the parameters start and end with this */
 do
  MailTo = strip(MailTo,'B','"')  /* remove the quotation marks */
 end

 MailTo = strip(MailTo,'B',' ')  /* remove any excess whitespace */
 parse var MailTo Recipients '?' Parameters  /* look for parameters */
 MessCont = 'To: '||translate(AddressFormat(Recipients,1,1),',',d2c(0))||Global.!CRLF  /* prepare a checked list of full addresses, convert nulls to commas, and store */

 do while (Parameters >< '')  /* as long as we have parameters left */

  parse var Parameters ParamName '=' ParamValue '&' Parameters  /* get the next bit */
  ParamName = translate(ParamName)  /* make the parameter name upper case */

  select  /* do one of the following */

   when (ParamName == 'FROM') then  /* if it is this one */
   do
    FromCont = 'From: '||translate(AddressFormat(ParamValue,1,1),',',d2c(0))||Global.!CRLF  /* prepare a checked list of full addresses, convert nulls to commas, and store */
   end

   when (ParamName == 'REPLY-TO') then  /* if it is this one */
   do
    MessCont = MessCont||'Reply-To: '||translate(AddressFormat(ParamValue,1,1),',',d2c(0))||Global.!CRLF  /* prepare a checked list of full addresses, convert nulls to commas, and store */
   end

   when (ParamName == 'SUBJECT') then  /* if it is this one */
   do
    MessCont = MessCont||'Subject: '||ParamValue||Global.!CRLF  /* store */
    GotSubject = 1  /* we have a subject line */
   end

   when (ParamName == 'CC') then  /* if it is this one */
   do
    MessCont = MessCont||'Cc: '||translate(AddressFormat(ParamValue,1,1),',',d2c(0))||Global.!CRLF  /* prepare a checked list of full addresses, convert nulls to commas, and store */
   end

   when (ParamName == 'ACC') then  /* if it is this one */
   do
    MessCont = MessCont||'Acc: '||translate(AddressFormat(ParamValue,1,1),',',d2c(0))||Global.!CRLF  /* prepare a checked list of full addresses, convert nulls to commas, and store */
   end

   when (ParamName == 'BCC') then  /* if it is this one */
   do
    MessCont = MessCont||'Bcc: '||translate(AddressFormat(ParamValue,1,1),',',d2c(0))||Global.!CRLF  /* prepare a checked list of full addresses, convert nulls to commas, and store */
   end

   when (ParamName == 'BODY') then  /* if it is this one */
   do

    call sysfiletree ParamValue,'Files.','FO'  /* look for a file */

    if (Files.0 == 0) then  /* if we do not find it */
    do
     call sysfiletree directory()||'\'||ParamValue,'Files.','FO'  /* look for the file spec in the current dir */
    end

    if (Files.0 == 1) then  /* if we find a file */
    do
     BodyPart = GetFileContents(Files.1)  /* use its contents */
    end
    else  /* if we find nothing */
    do
     BodyPart = ParamValue  /* simply use the whole parameter */
    end

    MessCont = MessCont||Global.!CRLF||BodyPart  /* add the body after an empty line */

   end

   when (ParamName == 'ATTACH') then  /* if it is this one */
   do

    do while (ParamValue >< '')  /* as long as we have a bit left */

     parse var ParamValue AttachFile ',' ParamValue  /* get the next file name in the list */

     call sysfiletree AttachFile,'Files.','FO'  /* look for the file spec */

     if (Files.0 == 0) then  /* if we do not find it */
     do
      call sysfiletree directory()||'\'||AttachFile,'Files.','FO'  /* look for the file spec in the current dir */
     end

     if (Files.0 == 0) then  /* if we still find nothing */
     do
      call AddError 'File not found "'||AttachFile||'"'  /* report */
     end

     do Index = 1 to Files.0  /* take each of the files we found */

      if (syscopyobject(Files.Index,AttDir)) then  /* if we can copy the attachment file to the attachment dir */
      do
       GotAttach = 1  /* set the attachment flag */
      end
      else  /* if we cannot copy it */
      do
       call AddError 'Cannot copy "'||Files.Index||'" to "'||AttDir||'"'  /* report */
      end

     end

    end

   end

   otherwise  /* if it is none of the above */
   do
    call AddError 'Unknown mailto: parameter '||ParamName||'='||ParamValue  /* report, not fatal */
   end

  end

 end

end
else  /* if we have no mailto string */
do
 MessCont = 'To: '||Global.!CRLF  /* use this */
end

if (\GotSubject) then  /* if we have no subject line */
do
 MessCont = MessCont||'Subject: '  /* add this */
end

MessCont = FromCont||MessCont  /* add the From line */

if (\PutFileContents(MessFile,MessCont)) then  /* if we can not write the content to the message file */
do
 call sysdestroyobject MessFile  /* get rid of the useless object */
 return 0  /* and return with an error */
end

MessCont = GetFileContents(MessFile)  /* get the message file contents */

if (\MessageContents(MessCont,'Global.!MessHead.')) then  /* if we cannot get the header sorted out */
do
 return 0  /* quit */
end

NewName = MakeTitle(MessFile,1,0,0,0)  /* get a new name for the mail file, and insert warnings as necessary, but do not do the object stuff */
OKToSend = (pos(Global.!Warning,NewName) == 0)  /* if the title contains no warnings, it is O.K. to send */
call MessageSettings MessFile,'0'||OKToSend||'00'||GotAttach||'000','CHANGE'  /* make it an outgoing mail message file and set the attachments flag if necessary */

if (pos('SEND',Switch) > 0) then  /* if we want the message sent */
do

 if (\OKToSend) then  /* if the message is not O.K. to send */
 do
  call AddError 'Message not ready for sending: "'||NewName||'"'  /* add a warning */
  call SetObjectData MessFile,'REXXMAILREFRESH=YES;TITLE='||NewName  /* set the new title */
  return 0  /* return with no result */
 end

 Global.!ProcFile = MessFile  /* specify the file to use */
 return SendMessage()  /* try to send the message and return with the result */

end

if (pos('OPEN',Switch) > 0) then  /* if we want the message opened on the desktop for editing */
do
 ObjectID = '<REXXMAIL_NEWMESSAGE_'||filespec('N',MessFile)||'>'  /* an object ID */
 call SetObjectData MessFile,'OBJECTID='||ObjectID  /* set the object ID */
 call SetTitle MessFile  /* set the file title */
 call sysopenobject ObjectID,0,1  /* open the message */
end
else  /* if we want to just leave it there */
do
 call SetTitle MessFile  /* set the file title */
end

return 1  /* end of MakeNewMessage */

/**************************************************************************/
MakeSendFile: procedure expose Global.  /* prepares the protofile to send */
/**************************************************************************/

parse arg MessCont,AttDir,Bare,ASCII  /* get the arguments */

SendFile = TempFileName(,,'SEND')  /* prepare a unique file name */

if (SendFile == '') then  /* if we cannot get a temp file */
do
 return ''  /* quit with no success */
end

call LogAction 'Creating send file "'||SendFile||'"',1  /* report, quietly */
Bare = (Bare == 1)  /* 1 = TRUE */
NamePart = filespec('N',SendFile)  /* the name part of the send file */
UniqueStr = date('S')||'.'||translate(left(time('L'),11),'.',':')||'.'||right(random(999),3,'0')  /* a unique string */
AddHead = 'Date: '||DateTimeRFC(1)||Global.!CRLF||,  /* a date line to add to the header (contact time servers if necessary) */
          'Message-Id: <'||UniqueStr||'@'||FullHostName()||'>'||Global.!CRLF||,  /* and a message ID line */
          'X-Mailer: '||Global.!BuildMess||Global.!CRLF||'          '||Global.!Copyright||Global.!CRLF  /* and a copyright line */

if (Global.!Settings.AddToHeader >< '') then  /* if we have additional header content */
do
 AddHead = AddHead||MakeInsertText(Global.!Settings.AddToHeader,'Global.!MessHead.')  /* add any additional content after processing any macros */
end

if (GetHeaderEntry('FROM','Global.!MessHead.') == '') then  /* if we have no sender */
do

 if (Global.!Settings.Address >< '') then  /* if we have an address */
 do

  AddHead = AddHead||'From: '  /* add this */

  if (Global.!Settings.Name >< '') then  /* if we have a name */
  do
   AddHead = AddHead||' "'||Global.!Settings.Name||'"'  /* add it */
  end

  AddHead = AddHead||' <'||Global.!Settings.Address||'>'||Global.!CRLF  /* add the address */

 end

end

AddHead = translate(AddHead,Global.!FilterOut,xrange('80'x,'FF'x))  /* convert the additional header stuff to PC819 (ISO-8859-1) */
AddHead = HeaderEncode(AddHead)  /* encode high-bit sequences in the additional header content */

call CreateCWMFObject NamePart,Global.!TempDir  /* try to create a CWMAILFile class object in the system temp dir using the name part */

if (Bare) then  /* if we are to send "bare" */
do

 if (\PutFileContents(SendFile,AddHead||MessCont)) then  /* if we cannot write the send file contents */
 do
  return ''  /* quit with nothing */
 end

 return SendFile  /* and return with its spec */

end

/**************************************************************************/
/* Preparing a normal, non-bare send file:                                */
/**************************************************************************/

parse var MessCont MessHead (Global.!EmptyLine) MessBody  /* split the message content into a header and a body */

if (symbol('Global.!SignatureText') >< 'VAR') then  /* if we have not set this var yet (i.e. this is the first time around) */
do
 Global.!SignatureText = GetFileContents(Global.!Signature)  /* get the signature text */
end

if ((Global.!SignatureText >< '') & (pos(Global.!CRLF||'-- '||Global.!CRLF,MessBody) == 0)) then  /* if we have a signature text and the message has no signature marker */
do
 MessBody = MessBody||Global.!CRLF||'-- '||Global.!CRLF||Global.!SignatureText  /* add the signature text to the message text after a signature marker */
end

MessHead = translate(MessHead,' '||copies('_',29),d2c(9)||xrange('00'x,'08'x)||xrange('0B'x,'0C'x)||xrange('0E'x,'1F'x))  /* convert header TABs to spaces and escape codes to underscores */
MessHead = translate(MessHead,Global.!FilterOut,xrange('80'x,'FF'x))  /* convert the header to PC819 (ISO-8859-1) */
MessBody = translate(MessBody,copies('_',29),xrange('00'x,'08'x)||xrange('0B'x,'0C'x)||xrange('0E'x,'1F'x))  /* convert body escape codes to underscores */
MessBody = translate(MessBody,Global.!FilterOut,xrange('80'x,'FF'x))  /* convert the body to PC819 (ISO-8859-1) */

if (ASCII) then  /* if we want ASCII text */
do
 MessHead = translate(MessHead,Global.!PC819_ASCII,xrange('80'x,'FF'x))  /* convert the header to ASCII */
 MessBody = translate(MessBody,Global.!PC819_ASCII,xrange('80'x,'FF'x))  /* convert the body text to ASCII */
end

MessHead = HeaderEncode(MessHead)  /* encode high-bit sequences in the header */
GotAttach = 0  /* no attachments yet */

if (AttDir >< '') then  /* if there is an attachments storage directory */
do

 call sysfiletree AttDir||'\*','Attach.','FOS'  /* look for attachment files in the attachments directory and any subdirectories */
 GotAttach = (Attach.0 > 0)  /* set a flag if we've got attachments */

 if (GotAttach) then  /* if we have something */
 do

  do Index = 1 to Attach.0  /* take each one */

   if (stream(Attach.Index,'C','QUERY SIZE') > 44040192) then  /* if the file is too big */
   do
    call AddError 'Attached file size exceeds processing limit : "'||filespec('N',Attach.Index)||'"'  /* report */
    return ''  /* and quit */
   end

  end

  call sysmkdir AttDir||'.ATT'  /* create a temp dir for the attachments */

  do Index = 1 to Attach.0  /* take each one */
   call FileCopy Attach.Index,AttDir||'.ATT\'||filespec('N',Attach.Index)  /* copy it to the temp dir */
  end

  CurDir = directory()  /* store the current dir */
  call directory AttDir||'.ATT'  /* switch to the temporary attachments dir */
  call RunCommand Global.!Settings.RunAttachOut  /* run a command on the attachments, if any */
  call directory CurDir  /* switch back to the original dir */
  call sysfiletree AttDir||'.ATT\*','Attach.','FO'  /* look for remaining attachment files in the temp directory */
  GotAttach = (Attach.0 > 0)  /* set a flag if we've got attachments */

 end

end

if ((GotAttach) | (\ASCII)) then  /* if we've got attachments, or we want quoted-printable text */
do
 MessHead = MessHead||'Mime-version: 1.0'||Global.!CRLF  /* add the MIME version header line */
end

if (GotAttach) then  /* if we've got attachments */
do

 MIMEBound = '=='||UniqueStr||'=='  /* the MIME boundary to use */

 if (Attach.0 > 1) then  /* if we have more than one attachment */
 do
  Plural = 's'  /* we need a plural "s" */
 end
 else  /*if there is only one attachment */
 do
  Plural = ''  /* we don't need the "s" */
 end

 MessHead = MessHead||'Content-type: multipart/mixed; boundary="'||MIMEBound||'"'||Global.!CRLF  /* add the MIME boundary header line */
 MessHead = MessHead||Global.!CRLF  /* add an empty line */
 MessHead = MessHead||'This is a multi-part message in MIME format.'||Global.!CRLF  /* add a plain text message line */
 MessHead = MessHead||'In addition to the message below it contains '||,  /* add an info line */
            Attach.0||' b64-encoded attachment'||Plural||'.'||Global.!CRLF  /* rest of the info line */
 MessHead = MessHead||Global.!CRLF||'--'||MIMEBound||Global.!CRLF  /* add an empty line followed by a MIME boundary */

end

if (ASCII) then  /* if we want ASCII text */
do

 if (GotAttach) then  /* if this is a MIME-enhanced message */
 do
  MessHead = MessHead||'Content-Type: text/plain; charset="us-ascii"'||Global.!CRLF  /* the first type line */
  MessHead = MessHead||'Content-Transfer-Encoding: 7bit'||Global.!CRLF  /* another one */
 end

end
else  /* if we want quoted-printable text */
do
 MessHead = MessHead||'Content-Type: text/plain; charset="iso-8859-1"'||Global.!CRLF  /* the first type line */
 MessHead = MessHead||'Content-Transfer-Encoding: quoted-printable'||Global.!CRLF  /* another one */
end

NewBody = ''  /* start with nothing */

if (ASCII) then  /* if we want ASCII text */
do
 NewBody = WordWrap(MessBody,76,1)  /* word-wrapped text, no indent, word breaks are allowed */
end
else  /* if we don't want ASCII text, we get quoted-printable */
do

 NewBody = ''  /* start with nothing */

 do while (MessBody >< '')  /* go through the contents of the body text */

  parse var MessBody NextChar 2 MessBody  /* get the next character from the text */

  if ((c2d(NextChar) > 126) | (NextChar == '=')) then  /* if it's not in the OK range, or if it is an equal sign */
  do
   NextChar = '='||c2x(NextChar)  /* substitute an equals sign followed by the character's hex code */
  end

  NewBody = NewBody||NextChar  /* add the character or character code to the new body text */

 end

 NewBody = WordWrap(NewBody,74,1,'',1)  /* word-wrap the new body text at 74/76 chars, word breaks allowed, no indent, Quoted-Printable) */

end

call charout SendFile,AddHead||MessHead||Global.!CRLF||NewBody  /* write the new message header parts and the new body text to the sendable file, separated by an empty line */
Success = 1  /* assume success */

if (GotAttach) then  /* if we've got one or more MIME attachments */
do

 Encoder = Global.!ProgDir||'\mmencode.exe'  /* the encoder file spec */

 if (\FileCheck(Encoder,1)) then  /* if we can't find the encoder */
 do
  Success = 0  /* no success */
 end
 else  /* if we find the decoder */
 do Index = 1 to Attach.0  /* take each of the attachment files */

  AttachName = filespec('N',Attach.Index)  /* get the name of the attachment file */
  MIMEType = ''  /* no MIME type yet */
  DotPos = lastpos('.',AttachName)  /* look for the last dos position in the file name */

  if (DotPos > 0) then  /* if we have a last dot */
  do
   AttachExt = substr(AttachName,DotPos + 1)  /* get the extension */
   MIMEType = GetFileEntry(Global.!MIMETypes,AttachExt,1)  /* try to match the extension with a MIME type, reverse lookup */
  end

  if (MIMEType == '') then  /* if we cannot match it */
  do
   MIMEType = 'application/octet-stream'  /* use this */
  end

  call charout SendFile,Global.!EmptyLine||,  /* start with an empty line */
                        '--'||MIMEBound||Global.!CRLF||,  /* followed by a MIME boundary */
                        'Content-Type: '||MIMEType||'; name="'||AttachName||'"'||Global.!CRLF||,  /* add a content line */
                        'Content-Transfer-Encoding: base64'||Global.!CRLF||,  /* add another one */
                        'Content-Disposition: attachment; filename="'||AttachName||'"'||Global.!EmptyLine  /* and another one, followed bij an empty line */
  call FileClose SendFile  /* close the new message file to allow MMENCODE.EXE access */
  address cmd Encoder||' < "'||Attach.Index||'" >> "'||SendFile||'"'  /* convert the attachment file to B64 code and add the result to SendFile [EXTERNAL] */
  call sysfiledelete Attach.Index  /* remove the attachment file copy */

  if (RC >< 0) then  /* if we get an error */
  do
   Success = 0  /* we have no success overall */
  end

  call FileOpen SendFile,'WRITE'  /* reopen the new message file for more writing */

 end

 call charout SendFile,Global.!EmptyLine||'--'||MIMEBound||'--'||Global.!CRLF  /* write an empty line followed by a MIME end boundary and a new line */

end

call FileClose SendFile  /* close the new message file */

if (\Success) then  /* if something went wrong */
do
 call sysfiledelete SendFile  /* get rid of the temp file -- no check, it may already have been deleted by another process */
 SendFile = ''  /* return nothing */
end

return SendFile  /* end of MakeSendFile */

/**************************************************************************/
MakeShadow: procedure expose Global.  /* creates a message shadow */
/**************************************************************************/

parse arg MessFile,ShadowDir,StartDir,MoveDir,OrgDir  /* get the arguments */

if (MoveDir == '') then  /* if we are not moving the message */
do
 return 0  /* quit */
end

if (translate(MoveDir) == translate(Global.!TempDir)) then  /* if we are not keeping the message */
do
 return 0  /* quit */
end

ShadowDir = strip(ShadowDir,'T','\')  /* strip off any trailing backslash */

if (ShadowDir == '') then  /* if we do not want a shadow */
do
 return 0  /* quit */
end

if (translate(ShadowDir) == 'LEAVE') then  /* if we want the shadow in the original message dir */
do

 if (OrgDir == '') then  /* if we have no original message dir spec */
 do
  ShadowDir = filespec('D',MessFile)||strip(filespec('P',MessFile),'T','\')  /* use this */
 end
 else  /* if we have an original message dir spec */
 do
  ShadowDir = OrgDir  /* use the original dir */
 end

end

if (translate(MoveDir) == translate(ShadowDir)) then  /* if we are moving the message to the shadow dir */
do
 return 0  /* quit */
end

if (substr(ShadowDir,2,1) >< ':') then  /* if it does not include a drive spec, we need to add the In or Out Archive dir */
do
 ShadowDir = StartDir||'\'||strip(ShadowDir,'L','\')  /* strip off any leading backslash and add the start dir */
end

call sysfiletree ShadowDir,'ShadowDir.','DO'  /* see if the dir exists */

if (ShadowDir.0 == 0) then  /* if it does not */
do

 if (\RexxMailMakePath(ShadowDir)) then  /* if we cannot make the dir -- EXTERNAL -- RXMLUTIL.DLL */
 do
  call AddError 'Cannot create "'||ShadowDir||'"'  /* report */
  return 0  /* no result */
 end

end

if (\syscreateshadow(MessFile,ShadowDir)) then  /* if we cannot create the shadow */
do
 call AddError 'Cannot create shadow of message file "'||MessFile||'" in "'||ShadowDir||'"'  /* report */
 return 0  /* quit with no result */
end

call LogAction 'Created shadow of message file in "'||ShadowDir||'"',1  /* report, quietly */

return 1  /* end of MakeShadow */

/**************************************************************************/
MakeTitle: procedure expose Global.  /* returns new title for mail file */
/**************************************************************************/

parse arg MessFile,Warnings,KeepAdds,Incoming,ObjectDo  /* get the parameters */

Warnings = (Warnings == 1)  /* 1 = true */
KeepAdds = (KeepAdds == 1)  /* 1 = true */
Incoming = (Incoming == 1)  /* 1 = true */
ObjectDo = (ObjectDo == 1)  /* 1 = true */
OutGoing = (MessageSettings(MessFile,'0*******','MATCH'))  /* look for an outgoing message type flag */

if (OutGoing & Warnings) then  /* if the message is outgoing and we want warnings */
do
 WarnStart = Global.!Warning||' '  /* start of a warning line */
 WarnEnd = ''  /* end of a warning line */
end
else  /* if the message is incoming, or we do not want warnings */
do
 WarnStart = '('  /* start of a warning line */
 WarnEnd = ')'  /* end of a warning line */
end

ObjectFrom = ''  /* start with nothing */
ObjectTo = ''  /* start with nothing */
ObjectCc = ''  /* start with nothing */
ObjectAcc = ''  /* start with nothing */
ObjectBcc = ''  /* start with nothing */
TitleDate = GetHeaderEntry('DATE','Global.!MessHead.')  /* get the date */

if (TitleDate >< '') then  /* if we have a date */
do
 TitleDate = translate(TitleDate,Global.!FilterOut,xrange('80'x,'FF'x))  /* convert to ISO */
 TitleDate = translate(TitleDate,Global.!PC819_ASCII,xrange('80'x,'FF'x))  /* convert to ASCII */
 ObjectDate = DateTimeDisplay(TitleDate,'UTCISO')  /* use a simple ISO UTC date/time string for the object date */
 TitleDate = DateTimeDisplay(TitleDate,Global.!Settings.DateTimeTitle)  /* rewrite the title date/time stamp if necessary */
end
else  /* if we have no date */
do
 ObjectDate = '_'  /* no object date */
 TitleDate = '_'  /* no title date */
end

TitleFrom = GetHeaderEntry('FROM','Global.!MessHead.')  /* get the sender */

if (TitleFrom == '') then  /* if we have nothing */
do

 TitleFrom = GetHeaderEntry('APPARENTLY-FROM','Global.!MessHead.')  /* get the sender this way */

 if (TitleFrom == '') then  /* if we have no sender */
 do
  TitleFrom = GetHeaderEntry('SENDER','Global.!MessHead.')  /* get the sender this way */
 end

end

if (TitleFrom == '') then  /* if we still have no sender */
do
 TitleFrom = WarnStart||'No FROM address'||WarnEnd  /* use this */
end
else  /* if we have something */
do

 TitleFrom = AddressFormat(TitleFrom,1,(OutGoing & Warnings))  /* get a reformatted address string, full, check if outgoing and we want warnings */

 if ((Warnings) & (pos(Global.!Warning,TitleFrom) > 0)) then  /* if it contains a marker */
 do
  TitleFrom = WarnStart||'Error in FROM address'||WarnEnd  /* insert a warning */
 end
 else  /* if there is no marker, or we do not want warnings */
 do
  TitleFrom = translate(TitleFrom,Global.!FilterOut,xrange('80'x,'FF'x))  /* convert to ISO */
  TitleFrom = translate(TitleFrom,Global.!PC819_ASCII,xrange('80'x,'FF'x))  /* convert to ASCII */
  ObjectFrom = left(GetAddressList(TitleFrom,Global.!Settings.SortAddress),64)  /* extract a list of names according to the SortAddress setting, and truncate the string for folder sorting use */
  TitleFrom = GetAddressList(TitleFrom,Global.!Settings.TitleAddress)  /* extract a list of names according to the TitleAddress setting */
 end

end

TitleTo = GetHeaderEntry('TO','Global.!MessHead.')  /* get the recipients */

if (TitleTo >< '') then  /* if we have one or more recipients */
do

 TitleTo = AddressFormat(TitleTo,1,(OutGoing & Warnings))  /* get a reformatted address string, full, check if outgoing and we want warnings */

 if ((Warnings) & (pos(Global.!Warning,TitleTo) > 0)) then  /* if it contains a marker and we want warnings */
 do
  TitleTo = WarnStart||'Error in TO address'||WarnEnd  /* insert a warning */
 end
 else  /* if there is no marker, or we do not want warnings */
 do
  TitleTo = translate(TitleTo,Global.!FilterOut,xrange('80'x,'FF'x))  /* convert to ISO */
  TitleTo = translate(TitleTo,Global.!PC819_ASCII,xrange('80'x,'FF'x))  /* convert to ASCII */
  ObjectTo = left(GetAddressList(TitleTo,Global.!Settings.SortAddress),64)  /* extract a list of names according to the SortAddress setting, and truncate the string for folder sorting use */
  TitleTo = GetAddressList(TitleTo,Global.!Settings.TitleAddress)  /* extract a list of names according to the TitleAddress setting */
 end

end

if (TitleTo == '') then  /* if we still have no recipient */
do
 TitleTo = WarnStart||'No TO address'||WarnEnd  /* use this */
end

TitleCc = GetHeaderEntry('CC','Global.!MessHead.')  /* look for CC addresses */

if (TitleCc >< '') then  /* if CC addresses were found */
do

 TitleCc = AddressFormat(TitleCc,1,(OutGoing & Warnings))  /* get a reformatted address string, full, check if outgoing and we want warnings */

 if ((Warnings) & (pos(Global.!Warning,TitleCc) > 0)) then  /* if it contains a marker and we want warnings */
 do
  TitleCc = WarnStart||'Error in CC address'||WarnEnd  /* create a warning line */
 end
 else  /* if there is no marker */
 do
  TitleCc = translate(TitleCc,Global.!FilterOut,xrange('80'x,'FF'x))  /* convert to ISO */
  TitleCc = translate(TitleCc,Global.!PC819_ASCII,xrange('80'x,'FF'x))  /* convert to ASCII */
  ObjectCc = left(GetAddressList(TitleCc,Global.!Settings.SortAddress),64)  /* extract a list of names according to the SortAddress setting, and truncate the string for folder sorting use */
  TitleCc = GetAddressList(TitleCc,Global.!Settings.TitleAddress)  /* extract a list of names according to the TitleAddress setting */
 end

end

TitleAcc = GetHeaderEntry('ACC','Global.!MessHead.')  /* look for ACC addresses */

if (TitleAcc >< '') then  /* if ACC addresses were found */
do

 TitleAcc = AddressFormat(TitleAcc,1,(OutGoing & Warnings))  /* get a reformatted address string, full, check if outgoing and we want warnings */

 if ((Warnings) & (pos(Global.!Warning,TitleAcc) > 0)) then  /* if it contains a marker and we want warnings */
 do
  AddLine = WarnStart||'Error in ACC address'||WarnEnd  /* add a warning line */
 end
 else  /* if there is no marker */
 do
  TitleAcc = translate(TitleAcc,Global.!FilterOut,xrange('80'x,'FF'x))  /* convert to ISO */
  TitleAcc = translate(TitleAcc,Global.!PC819_ASCII,xrange('80'x,'FF'x))  /* convert to ASCII */
  ObjectAcc = left(GetAddressList(TitleAcc,Global.!Settings.SortAddress),64)  /* extract a list of names according to the SortAddress setting, and truncate the string for folder sorting use */
  TitleAcc = GetAddressList(TitleAcc,Global.!Settings.TitleAddress)  /* extract a list of names according to the TitleAddress setting */
 end

end

TitleBcc = GetHeaderEntry('BCC','Global.!MessHead.')  /* look for BCC addresses */

if (TitleBcc >< '') then  /* if BCC addresses were found */
do

 TitleBcc = AddressFormat(TitleBcc,1,(OutGoing & Warnings))  /* get a reformatted address string, full, check if outgoing and we want warnings */

 if ((Warnings) & (pos(Global.!Warning,TitleBcc) > 0)) then  /* if it contains a marker and we want warnings */
 do
  AddLine = WarnStart||'Error in BCC address'||WarnEnd  /* add a warning line */
 end
 else  /* if there is no marker */
 do
  TitleBcc = translate(TitleBcc,Global.!FilterOut,xrange('80'x,'FF'x))  /* convert to ISO */
  TitleBcc = translate(TitleBcc,Global.!PC819_ASCII,xrange('80'x,'FF'x))  /* convert to ASCII */
  ObjectBcc = left(GetAddressList(TitleBcc,Global.!Settings.SortAddress),64)  /* extract a list of names according to the SortAddress setting, and truncate the string for folder sorting use */
  TitleBcc = GetAddressList(TitleBcc,Global.!Settings.TitleAddress)  /* extract a list of names according to the TitleAddress setting */
 end

end

TitleSubject = GetHeaderEntry('SUBJECT','Global.!MessHead.')  /* get the subject */

if (TitleSubject == '') then  /* if we have no subject */
do
 TitleSubject = WarnStart||'No subject'||WarnEnd  /* use this */
 ObjectSubject = ''  /* nothing */
end
else  /* if we have something */
do
 TitleSubject = translate(TitleSubject,Global.!FilterOut,xrange('80'x,'FF'x))  /* convert to ISO */
 TitleSubject = translate(TitleSubject,Global.!PC819_ASCII,xrange('80'x,'FF'x))  /* convert to ASCII */
 ObjectSubject = left(TitleSubject,64)  /* get the object data subject */
end

if (ObjectDo) then  /* if we want to do object stuff */
do
 Settings = 'REXXMAILDATE='||strip(ObjectDate)||';'||,  /* set the object date */
            'REXXMAILFROM='||strip(ObjectFrom)||';'||,  /* set the object sender */
            'REXXMAILTO='||strip(ObjectTo)||';'||,  /* set the object recipients */
            'REXXMAILCC='||strip(ObjectCc)||';'||,  /* set the object recipients */
            'REXXMAILACC='||strip(ObjectAcc)||';'||,  /* set the object recipients */
            'REXXMAILBCC='||strip(ObjectBcc)||';'||,  /* set the object recipients */
            'REXXMAILSUBJECT='||strip(ObjectSubject)  /* set the object subject */
 call SetObjectData MessFile,Settings  /* set the object data strings */
end

TitleCount = 0  /* start at 0 */
Title. = ''  /* start with nothing */

if (Incoming) then  /* if the message is incoming */
do
 TitleWords = translate(Global.!Settings.ObjectTitleIn)  /* copy the contents of the ObjectTitleIn setting, upper case */
end
else  /* if the message is outgoing */
do
 TitleWords = translate(Global.!Settings.ObjectTitleOut)  /* copy the contents of the ObjectTitleOut setting, upper case */
end

if (TitleWords == '') then  /* if we have no title words string setting */
do
 TitleCount = 1  /* up the counter */
 Title.TitleCount = DateTimeSys(1)  /* the file title to use is a date/time stamp with the long time */
end
else  /* if we have a title words string */
do

 if (Global.!Settings.TitleKeywords) then  /* if we want keywords */
 do
  call GetDefinitions Global.!Translations,'!Translations',1  /* see if we need to set any keyword translations */
 end

 TitleWordsCopy = TitleWords  /* copy the title words string */

 do while (TitleWordsCopy >< '')  /* as long as we have object title words left */

  TitleCount = TitleCount + 1  /* up the title lines counter */
  parse var TitleWordsCopy NextWord TitleWordsCopy  /* get the next word */

  if (Global.!Settings.TitleKeywords) then  /* if we want a keyword */
  do

   if (symbol('Global.!Translations.'||NextWord) == 'VAR') then  /* if we have a translation for this one */
   do
    Keyword = Global.!Translations.NextWord||': '  /* use it */
   end
   else  /* if there is no translation */
   do
    Keyword = Nextword||': '  /* use this */
   end

  end
  else  /* if not */
  do
   Keyword = ''  /* use nothing */
  end

  select  /* do one of the following */

   when (NextWord == 'DATE') then  /* if the next word is 'DATE' */
   do
    Title.TitleCount = Keyword||CheckCommLine(TitleDate)  /* get rid of problem characters and add the keyword, if any */
   end

   when ((NextWord == 'FROM') | (NextWord == 'SENDER')) then  /* if the next word is 'FROM' or 'SENDER' */
   do
    Title.TitleCount = Keyword||CheckCommLine(TitleFrom)  /* get rid of problem characters and add the keyword, if any */
   end

   when ((NextWord == 'TO') | (NextWord == 'RECIPIENTS')) then  /* if the next word is 'TO' or 'RECIPIENTS' */
   do
    Title.TitleCount = Keyword||CheckCommLine(TitleTo)  /* get rid of problem characters and add the keyword, if any */
   end

   when (NextWord == 'CC') then  /* if the next word is 'CC' */
   do

    if (TitleCc >< '') then  /* if we have a Cc line */
    do
     Title.TitleCount = Keyword||CheckCommLine(TitleCc)  /* get rid of problem characters and add the keyword, if any */
    end
    else  /* if we have no Cc line */
    do
     TitleCount = TitleCount - 1  /* lower the title lines counter */
    end

   end

   when (NextWord == 'ACC') then  /* if the next word is 'ACC' */
   do

    if (TitleAcc >< '') then  /* if we have an Acc line */
    do
     Title.TitleCount = Keyword||CheckCommLine(TitleAcc)  /* get rid of problem characters and add the keyword, if any */
    end
    else  /* if we have no Acc line */
    do
     TitleCount = TitleCount - 1  /* lower the title lines counter */
    end

   end

   when (NextWord == 'BCC') then  /* if the next word is 'BCC' */
   do

    if (TitleBcc >< '') then  /* if we have a Bcc line */
    do
     Title.TitleCount = Keyword||CheckCommLine(TitleBcc)  /* get rid of problem characters and add the keyword, if any */
    end
    else  /* if we have no Bcc line */
    do
     TitleCount = TitleCount - 1  /* lower the title lines counter */
    end

   end

   when (NextWord == 'SUBJECT') then  /* if the next word is 'SUBJECT' */
   do
    Title.TitleCount = Keyword||CheckCommLine(TitleSubject)  /* get rid of problem characters and add the keyword, if any */
   end

   otherwise  /* if it is not one of the above */
   do
    call AddError 'Illegal keyword in '||word('ObjectTitleOut ObjectTitleIn',(Incoming + 1))||' setting: '||NextWord  /* report a non-fatal error */
    TitleCount = TitleCount - 1  /* reduce the counter back to its previous value */
   end

  end

 end

end

if (KeepAdds) then  /* if some address insertion folder is still there, it must contain invalid files */
do
 TitleCount = TitleCount + 1  /* up the title lines counter */
 Title.TitleCount = WarnStart||'Invalid TO-CC-ACC-BCC file'||WarnEnd  /* add a warning */
end

if (\Global.!MessHead.!HeaderOK) then  /* if the header is not O.K. */
do
 TitleCount = TitleCount + 1  /* up the title lines counter */
 Title.TitleCount = WarnStart||'Illegal keyword'||WarnEnd  /* add a warning */
end

if (\Global.!MessHead.!BodyOK & Warnings) then  /* if we have no message body and we want warnings */
do
 TitleCount = TitleCount + 1  /* up the title lines counter */
 Title.TitleCount = WarnStart||'No message body'||WarnEnd  /* add a warning */
end

do Index = 2 to TitleCount  /* for each line in the title, except the first one */
 Title.Index = d2c(10)||Title.Index  /* add a leading new line */
end

NewTitle. = ''  /* start with nothing */
Titlelength = 0  /* start with nothing */
CutLength = Global.!Settings.TitleLineLength  /* cut each line at the maximum allowable line length */

do Index = 1 to TitleCount  /* for each line in the title */
 NewTitle.Index = strip(left(Title.Index,CutLength),'T',' ')  /* copy the contents, cut to size and strip trailing blanks */
 TitleLength = TitleLength + length(NewTitle.Index)  /* add the length to what we have */
end

MaxTitleLength = 240 - length(filespec('D',MessFile)||filespec('P',MessFile))  /* the maximum no. of characters in the title */

do while (TitleLength > MaxTitleLength)  /* as long as the full title is too long */

 Titlelength = 0  /* start with nothing */

 do Index = 1 to TitleCount  /* for each line in the title */

  if (length(NewTitle.Index) > CutLength) then  /* if this part exceeds the cutting length */
  do
   NewTitle.Index = strip(left(NewTitle.Index,CutLength),'T',' ')  /* reduce this one to the cutting length and strip any trailing blanks */
  end

  TitleLength = TitleLength + length(NewTitle.Index)  /* add the length to what we have */

 end

 CutLength = CutLength - 1  /* reduce the cutting length by one */

end

FullTitle = ''  /* start with nothing once more */

do Index = 1 to TitleCount  /* for each line in the title */

 if (NewTitle.Index >< Title.Index) then  /* if the contents are not what we originally had */
 do
  NewTitle.Index = left(NewTitle.Index,(length(NewTitle.Index) - 3))||'...'  /* remove the last 3 characters and add an ellipsis */
 end

 FullTitle = FullTitle||NewTitle.Index  /* add it to the full title */

end

if (right(FullTitle,1) >< '.') then  /* if the full title does not end in a full stop */
do
 FullTitle = FullTitle||'.'  /* add a dot */
end

return FullTitle  /* end of MakeTitle */

/**************************************************************************/
MessageContents: procedure expose Global.  /* extracts the unfolded, decoded header and the body from file contents */
/**************************************************************************/

parse arg MessCont,HeadName,BodyName  /* get the arguments */

if (HeadName >< '') then  /* if we have a header var name */
do
 call value HeadName,''  /* set the header variable to contain nothing */
 call value HeadName||'0',0  /* store the header line counter at 0 */
end

if (BodyName >< '') then  /* if we have a body var name */
do
 call value BodyName,''  /* set the body variable to contain nothing */
end

if (MessCont == '') then  /* if there is no content */
do
 return 1  /* quit */
end

parse var MessCont HeadPart (Global.!EmptyLine) BodyPart  /* split the contents into a header part and a body part */

if (HeadPart == '') then  /* if we have no header part */
do
 call AddError 'No header found'  /* report */
 return 0  /* return without success */
end

if (BodyPart >< '') then  /* if we have a body part */
do
 call value HeadName||'!BodyOK',1  /* the body is O.K. */
end
else  /* if not */
do
 call value HeadName||'!BodyOK',0  /* the body is not O.K. */
end

if (BodyName >< '') then  /* if we have a body name spec */
do

 do while (left(BodyPart,2) == Global.!CRLF)  /* as long as the body starts with an empty line */
  BodyPart = substr(BodyPart,3)  /* remove it */
 end

 call value BodyName,BodyPart  /* set the body variable to contain the body part */

end

call value HeadName||'!HeaderOK',1  /* the header is O.K. to start with */
HeadCount = 0  /* start a counter */
HeadPart = translate(HeadPart,' ',D2C(9))  /* convert any TABs to spaces */
FoldPoint = Global.!CRLF||' '  /* define a folding point */

do until (HeadPart == '')  /* go on until we have no header part left */

 parse var HeadPart NextPart (FoldPoint) HeadPart  /* get the next part ending in a folding point, if any */

 do while (NextPart >< '')  /* go on while we have content in our part */

  parse var Nextpart NextLine (Global.!CRLF) NextPart  /* get the next line */
  NextLine = strip(NextLine,'T',' ')  /* remove excess trailing spaces */

  if ((NextPart == '') & (HeadPart >< '')) then  /* if it is the last line of the part and we have folded stuff waiting */
  do
   HeadPart = NextLine||' '||strip(HeadPart,'L',' ')  /* remove excess leading spaces from the folded content and restore the last line */
  end
  else  /* if it is not the last line, or there is no more folded stuff */
  do
   parse var NextLine Keyword ':' Entry  /* get the components */
   HeadCount = HeadCount + 1  /* up the counter */
   call value HeadName||HeadCount||'.!Keyword',strip(KeyWord)  /* strip and store the keyword */
   call value HeadName||HeadCount||'.!Entry',DecodeHeaderLine(strip(Entry))  /* strip, decode, and store the entry */
  end

 end

end

call value HeadName||'0',HeadCount  /* store the header line counter */

return 1  /* end of MessageContents */

/**************************************************************************/
MessageSettings: procedure expose Global.  /* handles EA settings */
/**************************************************************************/

/**************************************************************************/
/* Flags:                                                                 */
/*   Position 0:                                                          */
/*     UNSET = outgoing message                                           */
/*     SET   = incoming message                                           */
/*   Position 1:                                                          */
/*     UNSET = not O.K. to send                                           */
/*     SET   = O.K. to send                                               */
/*   Position 2:                                                          */
/*     UNSET = not processed                                              */
/*     SET   = processed                                                  */
/*   Position 3:                                                          */
/*     UNSET = non-ASCII (i.e. quoted-printable text)                     */
/*     SET   = ASCII (i.e. 7-bit, word-wrapped, no TABs)                  */
/*   Position 4:                                                          */
/*     UNSET = no attachments                                             */
/*     SET   = attachments                                                */
/**************************************************************************/
/* Switch options:                                                        */
/*   CHANGE:   Changes the file settings to the settings provided,        */
/*             with "1" indicating "set", "0" indicating "reset", and     */
/*             "*" indicating "leave unchanged".                          */
/*   MATCH :   Looks for a match between the file settings and the        */
/*             settings provided. A "*" character serves as a "don't      */
/*             care" placeholder.                                         */
/*   CHECK :   See if we are dealing with a RexxMail message file.        */
/**************************************************************************/

parse upper arg MessFile,Setting,Action  /* get the arguments */

CurSettings = GetObjectEA(MessFile,'RXMLSETTINGS')  /* get the current settings */

if (Action == 'CHECK') then  /* if all we are doing is checking to see if we are dealing with a RexxMail mail file */
do

 if (CurSettings >< '') then  /* if we found a CurSettings string */
 do
  return (verify(CurSettings,'10','NOMATCH') == 0)  /* if CurSettings contains nothing but 1 and 0, it must be a RexxMail message file */
 end
 else  /* if we find nothing */
 do
  return 0  /* this cannot be a RexxMail message file */
 end

end

CurSettings = left(CurSettings,8,'0')  /* pad to 8 characters with 0 (i.e. "not set") */

if (Setting == '') then  /* if we received no action argument */
do
 return CurSettings  /* return the current value */
end

NewSettings = ''  /* start with nothing */

do Index = 1 to 8  /* take each of the characters in the action setting */

 if (substr(Setting,Index,1) == '*') then  /* if the action setting is * */
 do
  NewSettings = NewSettings||substr(CurSettings,Index,1)  /* copy the current setting */
 end
 else  /* if the new setting is not * (but 1 or 0) */
 do
  NewSettings = NewSettings||substr(Setting,Index,1)  /* copy the new setting */
 end

end

select  /* selects the action to take */

 when (Action == 'CHANGE') then  /* if we're asked to change the settings */
 do

  if (left(NewSettings,1) == 0) then  /* if the first position, i.e. the "Received" bit is not set */
  do

   if (substr(NewSettings,3,1) == 1) then  /* if the third position, i.e. the "Processed" bit is set */
   do
    TypeText = 'Mail Message Sent'  /* make it a sent mail message file type */
   end
   else  /* if the third position bit is unset */
   do
    TypeText = 'Mail Message Out'  /* make it an outgoing mail message file type */
   end

  end
  else  /* if the "Received" bit is set */
  do
   TypeText = 'Mail Message In'  /* make it a normal incoming mail message file type */
  end

  if (\PutObjectEA(MessFile,'.TYPE',TypeText)) then  /* if we cannot attach it to the file */
  do
   return 0  /* quit */
  end

  if (\PutObjectEA(MessFile,'RXMLSETTINGS',NewSettings)) then  /* if we cannot attach it to the file */
  do
   return 0  /* quit */
  end

  Colours = GetObjectEA(MessFile,'RXMLSTATIONERY')  /* get the stationery EA, if any */

  if (Colours == '') then  /* if we find nothing */
  do
   Colours = Global.!Settings.Stationery  /* use the default */
  end

  if (\SetObjectData(MessFile,'ICONFILE='||Global.!IconDir||'\Stationery\'||Colours||'\'||NewSettings||'.ICO')) then  /* if we cannot set the icon */
  do
   return 0  /* quit with an error */
  end

  return 1  /* return with success */

 end

 when (Action == 'MATCH') then  /* if the action is to check for a match */
 do
  return (NewSettings == CurSettings)  /* return O.K. if they match */
 end

 otherwise  /* if we have an unexpected action parameter */
 do
  return 0  /* return an error status */
 end

end

return 0  /* end of MessageSettings */

/**************************************************************************/
MessBarAdd: procedure expose Global.  /* adds content to the message button bar */
/**************************************************************************/

if (Global.!Settings.MessageToolBar == '') then  /* if we do not want a message toolbar */
do
 return ''  /* quit */
end

parse arg Message,AttDir,AddDirTo,AddDirCc,AddDirAcc,AddDirBcc  /* get the arguments */

Counter = MessBarStore(GetfileContents(Global.!TempDir||'\Mess_Bar.LST')) + 1  /* get the bar list contents and store the lot */
Global.!MessBar.Counter.!ID = date('S')||left(time('L'),11)||right(random(999),3,'0')  /* store a unique ID */
Global.!MessBar.Counter.!Message = Message  /* store */
Global.!MessBar.Counter.!AttDir = AttDir  /* store */
Global.!MessBar.Counter.!AddDirTo = AddDirTo  /* store */
Global.!MessBar.Counter.!AddDirCc = AddDirCc  /* store */
Global.!MessBar.Counter.!AddDirAcc = AddDirAcc  /* store */
Global.!MessBar.Counter.!AddDirBcc = AddDirBcc  /* store */
Global.!MessBar.0 = Counter  /* store the new count */

if (\MessBarWrite(Global.!TempDir||'\Mess_Bar.LST')) then  /* if we cannot write the results */
do
 return ''  /* quit */
end

call ToolBar 'CREATE','Message Buttons',1  /* create a new message bar */
call ToolBar 'OPEN','Message Buttons',1  /* show the new message bar */

return Global.!MessBar.Counter.!ID  /* end of MessToolbarAdd */

/**************************************************************************/
MessBarRemove: procedure expose Global.  /* removes content from the message button bar */
/**************************************************************************/

if (Global.!Settings.MessageToolbar == '') then  /* if we do not want a message toolbar */
do
 return ''  /* quit */
end

parse arg MessBarID  /* get the argument */

Counter = MessBarStore(GetfileContents(Global.!TempDir||'\Mess_Bar.LST'))  /* get the bar list contents and store the lot */

do Index = 1 to Counter  /* for each existing message bar item */

 if (Global.!MessBar.Index.!ID == MessBarID) then  /* it it's the one we want to remove */
 do

  if (Counter > 1) then  /* if we have more than one entry */
  do

   LastButOne = Counter - 1  /* the index number of the last item but one */

   do SubIndex = Index to LastButOne  /* for each of the remaining message bar items but one */
    NextIndex = SubIndex + 1  /* the next item's index number */
    Global.!MessBar.SubIndex.!ID = Global.!MessBar.NextIndex.!ID  /* copy the next ID */
    Global.!MessBar.SubIndex.!Message = Global.!MessBar.NextIndex.!Message  /* copy the next message */
    Global.!MessBar.SubIndex.!AttDir = Global.!MessBar.NextIndex.!AttDir  /* copy the next attachments dir */
    Global.!MessBar.SubIndex.!AddDirTo = Global.!MessBar.NextIndex.!AddDirTo  /* copy the additional To addresses dir */
    Global.!MessBar.SubIndex.!AddDirCc = Global.!MessBar.NextIndex.!AddDirCc  /* copy the additional Cc addresses dir */
    Global.!MessBar.SubIndex.!AddDirAcc = Global.!MessBar.NextIndex.!AddDirAcc  /* copy the additional Acc addresses dir */
    Global.!MessBar.SubIndex.!AddDirBcc = Global.!MessBar.NextIndex.!AddDirBcc  /* copy the additional Bcc addresses dir */
   end

   Index = Counter  /* make sure we exit the loop */

  end

  Global.!MessBar.0 = Global.!MessBar.0 - 1  /* one less entry */

 end

end

if (\MessBarWrite(Global.!TempDir||'\Mess_Bar.LST')) then  /* if we cannot write the results */
do
 return ''  /* quit */
end

if (Global.!MessBar.0 > 0) then  /* if we have something left */
do
 call ToolBar 'CREATE','Message Buttons',1  /* create a new message bar */
 call ToolBar 'OPEN','Message Buttons',1  /* show the new message bar */
end
else  /* if this was the last button */
do
 call ToolBar 'DELETE','Message Buttons',1  /* delete the message bar */
end

return 1  /* end of MessBarRemove */

/**************************************************************************/
MessBarStore: procedure expose Global.  /* stores content from the message button bar in global memory */
/**************************************************************************/

parse arg MessBarCont  /* get the argument */

Counter = 0  /* start with nothing */
Sep = d2c(0)  /* define a separator */
DoubleSep = Sep||Sep  /* define a double separator */

do while(MessBarCont >< '')  /* as long as we have something left */
 parse var MessBarCont NextMessID (Sep) NextMessage (Sep) NextAttDir (Sep) NextAddDirTo (Sep) NextAddDirCc (Sep) NextAddDirAcc (Sep) NextAddDirBcc (DoubleSep) MessBarCont  /* get the bits we want */
 Counter = Counter + 1  /* up the counter */
 Global.!MessBar.Counter.!ID = NextMessID  /* store the ID */
 Global.!MessBar.Counter.!Message = NextMessage  /* store */
 Global.!MessBar.Counter.!AttDir = NextAttDir  /* store */
 Global.!MessBar.Counter.!AddDirTo = NextAddDirTo  /* store */
 Global.!MessBar.Counter.!AddDirCc = NextAddDirCc  /* store */
 Global.!MessBar.Counter.!AddDirAcc = NextAddDirAcc  /* store */
 Global.!MessBar.Counter.!AddDirBcc = NextAddDirBcc  /* store */
end

Global.!MessBar.0 = Counter  /* store the counter */

return Counter  /* end of MessBarStore */

/**************************************************************************/
MessBarWrite: procedure expose Global.  /* writes message bar list content to a file */
/**************************************************************************/

parse arg MessBarFile  /* get the argument */

MessBarCont = ''  /* start with nothing */
Sep = d2c(0)  /* define a separator */
DoubleSep = Sep||Sep  /* define a double separator */

do Index = 1 to Global.!MessBar.0  /* for each existing message bar item */
 MessBarCont = MessBarCont||,  /* add the next items: */
               Global.!MessBar.Index.!ID||Sep||,  /* the message bar ID */
               Global.!MessBar.Index.!Message||Sep||,  /* the message spec */
               Global.!MessBar.Index.!AttDir||Sep||,  /* the attachments dir */
               Global.!MessBar.Index.!AddDirTo||Sep||,  /* the additional To addresses folder */
               Global.!MessBar.Index.!AddDirCc||Sep||,  /* the additional Cc addresses folder */
               Global.!MessBar.Index.!AddDirAcc||Sep||,  /* the additional Acc addresses folder */
               Global.!MessBar.Index.!AddDirBcc||DoubleSep  /* the additional Bcc addresses folder */
end

call sysfiledelete MessBarFile  /* zap the old file */

if (\PutFileContents(MessBarFile,MessBarCont)) then  /* if we cannot write the results */
do
 return 0  /* quit */
end

return 1  /* end of MessBarWrite */

/**************************************************************************/
MIMEDecode: procedure expose Global.  /* decodes MIME message contents */
/**************************************************************************/

signal on halt name HaltMIMEDecode  /* handles halt locally */

parse arg MessCont,AttDir  /* get the arguments */

Decoder = Global.!ProgDir||'\munpack.exe'  /* the MIME decoder program */

if (\FileCheck(Decoder,1)) then  /* if we cannot find the decoder (issue a warning) */
do
 return 0  /* quit */
end

CopyFile = AttDir||'.COPY'  /* the name of the copy file we will use to process the message contents */

if (\PutFileContents(CopyFile,MessCont)) then  /* if we cannot write the message contents to the copy file */
do
 return 0  /* quit */
end

address cmd Decoder||' -C "'||AttDir||'" -e -q -t "'||CopyFile||'"'  /* unpack the mail file into the attachments directory, adding MIME type EAs, quietly, and including text parts [EXTERNAL] */
call sysfiledelete CopyFile  /* get rid of the copy file */

return (RC == 0)  /* end of MIMEDecode */

/**************************************************************************/
HaltMIMEDecode:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
MoveAction: procedure expose Global.  /* returns a redirect action for a mail message file */
/**************************************************************************/

parse arg MessFile,MoveFile,StartDir  /* get the parameters */

if (\FileCheck(MoveFile)) then  /* if we can find no move list file */
do
 return ''  /* return with nothing */
end

MessDir = filespec('D',MessFile)||strip(filespec('P',MessFile),'T','\')  /* the message dir */
Action = GetRedirect(MoveFile,MessDir,'Global.!MessHead.','Global.!MessBody')  /* try to find a matching action */

if (Action == '') then  /* if we have nothing */
do
 return ''  /* nothing to do */
end

MoveDir = ''  /* we have no destination dir yet */

do while (Action >< '')  /* go on until we run out of steam */

 parse var Action NextPart '&' Action  /* get the next bit */
 NextPart = strip(NextPart)  /* lose excess blanks */

 select  /* do one of the following */

  when ((translate(word(NextPart,1)) == 'RUN') & (FileCheck(MessFile))) then  /* if it is a command to run and the message still exists */
  do
   call RunCommand subword(NextPart,2),MessFile  /* run the string as a command */
  end

  when ((translate(word(NextPart,1)) == 'SETSTATIONERY') & (FileCheck(MessFile))) then  /* if it is a stationery directive and the message still exists */
  do
   call PutObjectEA MessFile,'RXMLSTATIONERY',translate(strip(word(NextPart,2)))  /* write the stationery EA */
  end

  when (translate(NextPart) == 'SILENT') then  /* if we want silence for this one */
  do
   Global.!SilentOnce = 1  /* set a flag */
  end

  when (translate(NextPart) == 'SKIP') then  /* if it is the special skip string */
  do

   if (MoveDir >< '') then  /* if we already have a destination */
   do
    call AddError 'Invalid destination in "'||MoveFile||'" : '||MoveDir  /* report */
   end

   MoveDir = ''  /* there is no need to move this one */

  end

  when (translate(NextPart) == 'DELETE') then  /* if it is the special delete string */
  do

   if (MoveDir >< '') then  /* if we already have a destination */
   do
    call AddError 'Invalid destination in "'||MoveFile||'" : '||MoveDir  /* report */
    MoveDir = ''  /* leave the message where it is */
   end
   else  /* if all is well */
   do
    MoveDir = Global.!TempDir  /* use the temp dir to move the file out of sight to be deleted the next time the mail folders are opened */
   end

  end

  otherwise  /* if it is none of the above */
  do

   if (MoveDir >< '') then  /* if we already have a destination */
   do
    call AddError 'Invalid destination in "'||MoveFile||'" : '||MoveDir  /* report */
    MoveDir = ''  /* leave the message where it is */
   end
   else  /* if all is well */
   do
 
    MoveDir = strip(NextPart,'T','\')  /* strip off any trailing backslash */

    if (substr(MoveDir,2,1) >< ':') then  /* if it does not include a drive spec, we need to add the In or Out Archive dir */
    do
     MoveDir = StartDir||'\'||strip(MoveDir,'L','\')  /* strip off any leading backslash and add the start dir */
    end

   end

  end

 end

end

if ((\FileCheck(MessFile)) | (MoveDir == '') | ((translate(MoveDir) == translate(MessDir)) & (MessDir >< Global.!TempDir))) then  /* if the file is gone, or we still have no destination dir, or if we are to stay where we are (unless we are in the Temp dir) */
do
 return ''  /* quit */
end

call sysfiletree MoveDir,'MoveDir.','DO'  /* see if the dir exists */

if (MoveDir.0 == 0) then  /* if it does not */
do

 if (\RexxMailMakePath(MoveDir)) then  /* if we cannot make the dir -- EXTERNAL -- RXMLUTIL.DLL */
 do
  call AddError 'Cannot create "'||MoveDir||'"'  /* report */
  return ''  /* no result */
 end

end

return MoveDir  /* end of MoveAction */

/**************************************************************************/
MoveMessage: procedure expose Global.  /* moves a mail message file after sending or viewing, if necessary */
/**************************************************************************/

parse arg MessFile,MoveDir  /* get the parameters */

if (MoveDir == '') then  /* if we do not have a new location */
do
 return 0  /* return with no move */
end

MessDir = filespec('D',MessFile)||strip(filespec('P',MessFile),'T','\')  /* get the message dir */

if (translate(MoveDir) >< translate(MessDir)) then  /* if we really have a move to do */
do

 NewName = filespec('N',TempFileName(MessDir,,'MOVE'))  /* get a temp name in the message dir */

 if (NewName == '') then  /* if we cannot get a temp file */
 do
  return 0  /* quit with no success */
 end

 if (\SetObjectData(MessFile,'TITLE='||NewName)) then  /* if we cannot rename the file */
 do
  call AddError 'Cannot rename "'||MessFile||'"'  /* report */
  return 0  /* and quit with nothing */
 end

 if (\sysmoveobject(MessDir||'\'||NewName,MoveDir)) then  /* if we cannot move the file */
 do
  call AddError 'Cannot move "'||MessDir||'\'||NewName||'" to "'||MoveDir||'"'  /* report */
  call SetTitle MessDir||'\'||NewName  /* restore the file title */
  return 0  /* and quit with nothing */
 end

 call LogAction 'Message moved to "'||MoveDir||'"',1  /* report, quietly */
 MessFile = MoveDir||'\'||NewName  /* use this file spec */

end

call SetTitle MessFile  /* set the title */

return 1  /* end of MoveMessage */

/**************************************************************************/
POP3DeleteMess:  procedure expose Global.  /* delete a message from the POP3 server */
/**************************************************************************/

signal on halt name HaltPOP3DeleteMess  /* handles halt locally */

parse arg Socket,Index  /* get the arguments */

if (\SocketSendLine(Socket,'DELE '||Index,'+OK')) then  /* if we cannot send this and get the right reply */
do
 call AddError 'Cannot delete message (server no. '||Index||')'  /* report */
 return 0  /* return with an error */
end

call LogAction 'Message deleted from server'  /* report */

return 1  /* end of POP3DeleteMess */

/**************************************************************************/
HaltPOP3DeleteMess:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
POP3GetStatus: procedure expose Global.  /* get the number of waiting messages from the POP3 server */
/**************************************************************************/

signal on halt name HaltPOP3GetStatus  /* handles halt locally */

parse arg Socket  /* get the argument */

Reply = SocketSendLine(Socket,'STAT','?')  /* send this and get the reply */
parse var Reply Status Messages Bytes  /* get the status, the number of messages and the total byte count */

if (Status >< '+OK') then  /* if we did not get the right status reply */
do
 call AddError 'Cannot get message status from server'  /* report */
 return 0  /* quit with nothing */
end

if (Messages == 0) then  /* if there are no messages */
do
 call LogAction 'There are no waiting messages'  /* report */
end
else  /* if we have waiting messages */
do
 call ShowLine  /* skip a line on the display */
 call LogAction 'No. of waiting messages = '||Messages||'; total size = '||Bytes||' bytes'  /* report */
end

return Messages  /* end of POP3GetStatus */

/**************************************************************************/
HaltPOP3GetStatus:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
POP3Process: procedure expose Global.  /* handles mail retrieval from POP3 server */
/**************************************************************************/

signal on halt name HaltPOP3Process  /* handles halt locally */

parse arg Socket  /* get the argument */

AutoProcess = 0  /* no automation yet */
CollectOK = 1  /* we are collecting messages */
MessCount = 0  /* start with a message count of 0 */
Deleted = 0  /* we have not deleted any messages yet */
Collected = 0  /* we have not collected any messages yet */
Signalled = 0  /* nothing signalled yet */
SizeTotal = 0  /* no bytes collected yet */

if (\Global.!Settings.POP3Interactive) then  /* if we are not interactive */
do

 AutoProcess = (FileCheck(Global.!ControlColl))  /* if we can find the action file, we want automated processing */

 if (AutoProcess) then  /* if we have an action specs file, i.e. we are collecting messages automatically (even if the file is empty!) */
 do
  call LogAction 'Using collect action file "'||Global.!ControlColl||'"'  /* report */
 end

end

do until (\Global.!Settings.POP3Interactive)  /* go on looping if we are working interactively, otherwise run just once */

 if (POP3GetStatus(Socket) == 0) then  /* if we have no waiting messages */
 do

  if ((Deleted == 0) | (\Global.!Settings.POP3Interactive)) then  /* if we have not deleted any messages either, or if we are not processing interactively */
  do
   return Signalled  /* we're done */
  end
  else  /* if we have deleted messages, and we are processing interactively */
  do

   if (Deleted == 1) then  /* if we have only one deleted message */
   do
    Plural = ''  /* no plural s */
   end
   else  /* if we have more than one deleted message */
   do
    Plural = 's'  /* we need a plural s */
   end

   ValidKeys = '1Uu0Qq'||d2c(13)||d2c(27)  /* we will accept these keystrokes */
   call ShowLine  /* empty line */
   call ShowLine '             1 = Undelete '||Deleted||' message'||Plural||' on server'  /* info */
   call ShowLine '             0 = Quit'  /* info */
   call ShowLine  /* empty line */
   call charout 'CON:','             Your choice (1 U [Enter] | 0 Q [Esc]): U'||d2c(8)  /* prompt */

   do until (pos(Reply,ValidKeys) > 0)  /* go on until we get a valid keystroke */
    Reply = sysgetkey('NOECHO')  /* get the reply */
   end

   if (pos(Reply,'1Uu'||d2c(13)) > 0) then  /* if it is one of these */
   do

    call ShowLine 'U'  /* echo the key  */
    call ShowLine  /* empty line */

    if (POP3Reset(Socket)) then  /* if we can reset the server */
    do
     Deleted = 0  /* we have no deleted messages now */
    end

   end
   else  /* if it was another key, we'll quit */
   do
    call ShowLine 'Q'  /* complete the line */
    call ShowLine  /* empty line */
    return Signalled  /* we're done */
   end

  end

 end
 else  /* if we have waiting messages */
 do

  call LogAction 'Retrieving message list'  /* report */

  if (\SocketSendLine(Socket,'LIST','+OK')) then  /* if we cannot send this and get the right reply */
  do
   call AddError 'Cannot retrieve message list from server'  /* report */
   return Signalled  /* return with no success */
  end

  SortList. = ''  /* start with an empty sorting list */
  MessList. = ''  /* start with an empty message list */
  MessCount = 0  /* start with a message count of 0 */
  NextLine = SocketGetLine(Socket)  /* get a line from the POP3 socket */

  do while ((NextLine >< '.') & (Global.!SocketOK))  /* as long as it is not a full stop or an error condition */

   MessCount = MessCount + 1  /* up the message counter */
   parse var NextLine Number Bytes .  /* get the ingredients */

   if \(datatype(Number,'W') & (Number > 0)) then  /* if we have an insane message number value */
   do
    call AddError 'Cannot retrieve message number from server'  /* report */
    return Signalled  /* return with no success */
   end

   TempBytes = translate(Bytes,'  ',',.')  /* remove any "helpful" punctuation */
   Bytes = ''  /* start with nothing */

   do while (TempBytes >< '')  /* as long as we have something left */
    parse var TempBytes NextBit TempBytes  /* get the next bit */
    Bytes = Bytes||NextBit  /* add it to what we have */
   end

   if \(datatype(Bytes,'W') & (Bytes > 0)) then  /* if we have an insane byte count value */
   do
    Bytes = 0  /* use this */
   end

   SortList.MessCount = right(Bytes,12,'0')||right(Number,6,'0')  /* this will be used for sorting by size later */
   MessList.Number.!Bytes = Bytes  /* store the byte count */
   NextLine = SocketGetLine(Socket)  /* get the next line */

  end

  if (NextLine >< '.') then  /* if we did not get to the end of the list */
  do
   call AddError 'Cannot retrieve message list from server'  /* report */
   return Signalled  /* return with no success */
  end

  if (\Global.!Settings.Pop3Interactive) then  /* if we are not working interactively */
  do

   if (MessCount > 0) then  /* if we have something */
   do

    SortList.0 = MessCount  /* copy the message counter */

    if (Global.!Settings.POP3SortBySize) then  /* if we want to sort by size */
    do

     if (sysstemsort('SortList.') == 0) then  /* if we can sort the list in ascending order */
     do
      call LogAction 'Message list sorted; retrieving messages in order of size.'  /* report (failure to sort is non-fatal) */
     end

    end

   end

  end

  call LogAction 'Retrieving message headers'  /* report */
  call syscurstate 'OFF'  /* hide the cursor */
  IntroMess = 'Messages left to process: '  /* what to report */
  call charout 'CON:',IntroMess  /* report */
 
  do Index = 1 to MessCount  /* run through the list */

   parse var SortList.Index =13 MessNumber  /* retrieve the message number */
   MessNumber = strip(MessNumber,'L','0')  /* get rid of leading zeroes */

   if (\SocketSendLine(Socket,'TOP '||MessNumber||' 0','+OK')) then  /* if we cannot send this and get the right reply */
   do
    call AddError 'Cannot retrieve header of message '||Index||' (no. '||MessNumber||' on server)'  /* report */
    call syscurstate 'ON'  /* restore the cursor */
    return Signalled  /* quit with what we have collected and signalled */
   end

   call charout 'CON:',right((MessCount - Index + 1),length(MessCount),' ')||copies(d2c(8),length(MessCount))  /* show the remaining no. of files to process */
   SortList.Index = MessNumber  /* replace the sort list contents (bytes/number) with the message number */
   MessList.MessNumber.!For = '[unknown]'  /* the default recipient text */
   Global.!POP3List.MessNumber. = ''  /* nothing yet */
   HeadCount = 0  /* nothing yet */
   GotRecipient = 0  /* we have no recipient yet */
   NextLine = ''  /* nothing yet */
   StoreLine = SocketGetLine(Socket)  /* get a line from the POP3 socket */

   if ((StoreLine >< '.') & (Global.!SocketOK)) then  /* if we do not encounter a full stop or an error */
   do

    NextLine = SocketGetLine(Socket)  /* get another line from the POP3 socket */

    do while ((NextLine >< '.') & (Global.!SocketOK))  /* as long as we do not encounter a full stop or an error */

     if ((NextLine >< '') & (pos(left(NextLine,1),d2c(9)||d2c(32)) > 0)) then  /* if the next line starts with whitespace */
     do
      StoreLine = StoreLine||' '||strip(translate(NextLine,d2c(32),d2c(9)),'B',' ')  /* add it to what we have, removing any excess leading or trailing whitespace */
     end
     else  /* if the next line does not start with whitespace */
     do

      parse var StoreLine FirstWord ':' RestOfLine  /* get the bits we want from what we have in store */
      StoreLine = NextLine  /* start a new stored line (which will be empty if we have had the last header line) */
      FirstWord = translate(strip(FirstWord,'T',' '))  /* remove any trailing blanks from the first word, and make it upper case */
      RestOfLine = strip(RestOfLine)  /* get rid of excess whitespace */
      DecodedLine = DecodeHeaderLine(RestOfLine)  /* the decoded version of the header entry */
      HeadCount = HeadCount + 1  /* up the header lines counter */
      Global.!POP3List.MessNumber.HeadCount.!Keyword = FirstWord  /* store the keyword */
      Global.!POP3List.MessNumber.HeadCount.!Entry = RestOfLine||' | '||DecodedLine  /* store the entry in both raw and decoded forms */

      select  /* do one of the following */

       when (FirstWord == 'DATE') then  /* if it is this */
       do
        MessList.MessNumber.!Date = RestOfLine  /* store the original date string */
       end

       when (FirstWord == 'FROM') then  /* if it is this */
       do
        Sender = DecodedLine  /* store the decoded version */
        MessList.MessNumber.!From = AddressFormat(Sender,0,0)  /* store the bare address(es) */
        MessList.MessNumber.!FromFull = AddressIndent(AddressFormat(Sender,1,0),13,66)  /* format the address(es), full, no check, and indent the list to match the display */
       end

       when (FirstWord == 'TO') then  /* if it is this */
       do
        MessList.MessNumber.!To = AddressFormat(RestOfLine,0,0)  /* store the bare address(es) */
       end

       when (FirstWord == 'CC') then  /* if it is this */
       do
        MessList.MessNumber.!Cc = AddressFormat(DecodedLine,0,0)  /* use the decoded version, and store it bare */
       end

       when (FirstWord == 'SUBJECT') then  /* if it is this */
       do
        MessList.MessNumber.!Subject = DecodedLine  /* use the decoded version, and store it */
       end

       when (FirstWord == 'MESSAGE-ID') then  /* if it is this */
       do
        MessList.MessNumber.!MessageID = RestOfLine  /* store it */
       end

       when ((FirstWord == 'RECEIVED') & (\GotRecipient)) then  /* if it is this and we have no recipient yet */
       do

        parse var RestOfLine . ' for ' Recipient .  /* get the bit we want */
        Recipient = strip(Recipient,'T',';')  /* remove any trailing semicolon */

        if (Recipient >< '') then  /* if we have something */
        do
         MessList.MessNumber.!For = AddressFormat(Recipient,0,0)  /* store the bare address */
         GotRecipient = 1  /* we have our recipient */
        end

       end

       otherwise  /* if it is none of the above */
       do
        nop  /* ignore it */
       end

      end

     end

     NextLine = SocketGetLine(Socket)  /* get the next line */

    end

   end

   if (NextLine >< '.') then  /* if we did not get to the termination sequence */
   do
    call AddError 'Cannot retrieve header of message '||Index||' (no. '||MessNumber||' on server)'  /* report */
    call syscurstate 'ON'  /* restore the cursor */
    return Signalled  /* return with no success */
   end

   Global.!POP3List.MessNumber.0 = HeadCount  /* store the header lines counter */

  end

  call charout 'CON:',copies(d2c(8),length(IntroMess))||copies(' ',(length(IntroMess) + length(MessCount)))  /* clear the line */

  do Index = 1 to MessCount  /* run through the message info we collected */

   MessNumber = SortList.Index  /* get the message number */
   call ShowLine  /* empty line on the display */
   call ShowLine 'Message no.: '||Index||' of '||MessCount||' (no. '||MessNumber||' on server)'  /* show a string on screen */
   call ShowLine 'Message ID : '||MessList.MessNumber.!MessageID  /* add this */
   call ShowLine 'Size       : '||MessList.MessNumber.!Bytes||' bytes'  /* add this */
   call ShowLine 'Date       : '||DateTimeDisplay(MessList.MessNumber.!Date,'UTCISO')  /* add this */
   call ShowLine 'From       : '||MessList.MessNumber.!FromFull  /* add this */
   call ShowLine 'For        : '||MessList.MessNumber.!For  /* add this */
   call ShowLine 'Subject    : '||MessList.MessNumber.!Subject  /* add this */
   call LogAction 'Processing message; server no. '||MessNumber||'; '||MessList.MessNumber.!Bytes||' bytes; ID = '||MessList.MessNumber.!MessageID,1  /* report quietly */

   if (Global.!Settings.POP3Interactive) then  /* if we are processing interactively */
   do

    ValidKeys = '1Gg2Kk3Dd4Ss0Qq'||d2c(13)||d2c(27)  /* we will accept these keystrokes */
    UndelKey = ''  /* do not show the undelete key */
    call ShowLine  /* empty line */
    call ShowLine '             1 = Get message and delete from server'  /* info */
    call ShowLine '             2 = Keep message on server and get it'  /* info */
    call ShowLine '             3 = Delete message from server'  /* info */
    call ShowLine '             4 = Skip message'  /* info */

    if (Deleted > 0) then  /* if we have deleted messages */
    do

     if (Deleted == 1) then  /* if we have only one deleted message */
     do
      Plural = ''  /* no plural s */
     end
     else  /* if we have more than one deleted message */
     do
      Plural = 's'  /* we need a plural s */
     end

     call ShowLine '             5 = Undelete '||Deleted||' message'||Plural||' on server'  /* info */
     ValidKeys = ValidKeys||'5Uu'  /* we will now also accept these keystrokes */
     UndelKey = ' | 5 U'  /* show the undelete key */

    end

    call ShowLine '             0 = Quit'  /* info */
    call ShowLine  /* empty line */
    call charout 'CON:','             Your choice (1 G [Enter] | 2 K | 3 D | 4 S'||UndelKey||' | 0 Q [Esc]): G'||d2c(8)  /* prompt */

    do until (pos(Reply,ValidKeys) > 0)  /* go on until we get a valid keystroke */
     Reply = sysgetkey('NOECHO')  /* get the reply */
    end

    select  /* do one of the following */

     when (pos(Reply,'1Gg'||d2c(13)) > 0) then  /* if it is one of these */
     do
      call ShowLine 'G'  /* echo the key */
      Action = 'GET'  /* use this */
     end

     when (pos(Reply,'2Kk') > 0) then  /* if it is one of these */
     do
      call ShowLine 'K'  /* echo the key */
      Action = 'KEEP'  /* use this */
     end

     when (pos(Reply,'3Dd') > 0) then  /* if it is one of these */
     do
      call ShowLine 'D'  /* echo the key */
      Action = 'DELETE'  /* use this */
     end

     when (pos(Reply,'4Ss') > 0) then  /* if it is one of these */
     do
      call ShowLine 'S'  /* just echo the key */
      Action = 'SKIP'  /* use this */
     end

     when (pos(Reply,'5Uu') > 0) then  /* if it is one of these */
     do
      call ShowLine 'U'  /* echo the key */
      Action = 'UNDELETE'  /* use this */
     end

     when (pos(Reply,'0Qq'||d2c(27)) > 0) then  /* if it is one of these */
     do
      call ShowLine 'Q'  /* echo the key */
      Action = 'QUIT'  /* use this */
     end

     otherwise  /* this shouldn't occur */
     do
      Action = 'SKIP'  /* use this to be on the safe side */
     end

    end

    call ShowLine  /* empty line */

   end
   else  /* if we are not processing interactively */
   do

    if (AutoProcess) then  /* if we are processing using an action specs file */
    do

     Action = GetRedirect(Global.!ControlColl,'','Global.!POP3List.'||MessNumber||'.')  /* look for a matching action spec */

     if (Action == '') then  /* if we found nothing */
     do
      Action = 'GET'  /* use this */
     end

    end
    else  /* if we are simply getting all waiting mail */
    do
     Action = 'GET'  /* use this */
    end

   end

   select  /* do one of the following */

    when ((Action == 'GET') | (Action == 'KEEP')) then  /* if we have one of these */
    do

     if (CollectOK) then  /* if we are still collecting */
     do

      if (\Global.!Settings.POP3Interactive) then  /* if we are not processing interactively */
      do

       if (Global.!Settings.MaxCollectNumber >< '') then  /* if we have a collect limit */
       do

        if (Collected == Global.!Settings.MaxCollectNumber) then  /* if we have reached the limit */
        do
         call LogAction 'The message limit has been reached (MaxCollectNumber = '||Global.!Settings.MaxCollectNumber||').'  /* report */
         call POP3SkipMess MessNumber  /* skip the message */
         call LogMail '=_=',MessList.MessNumber.!From,MessList.MessNumber.!For,MessList.MessNumber.!To,MessList.MessNumber.!Cc,,,MessList.MessNumber.!Subject,MessList.MessNumber.!Date,MessList.MessNumber.!MessageID,MessList.MessNumber.!Bytes  /* log the message */
         Action = ''  /* no more action needed */
         CollectOK = 0  /* we will no longer collect messages */
        end

       end

       if (Action >< '') then  /* if we are to carry on */
       do

        if (Global.!Settings.MaxCollectSize >< '') then  /* if we have a size limit */
        do

         if (MessList.MessNumber.!Bytes > Global.!Settings.MaxCollectSize) then  /* if this one is over the limit */
         do
          call LogAction 'Message size ('||MessList.MessNumber.!Bytes||' bytes) exceeds limit (MaxCollectSize = '||Global.!Settings.MaxCollectSize||').'  /* report */
          call POP3SkipMess MessNumber  /* skip the message */
          call LogMail '===',MessList.MessNumber.!From,MessList.MessNumber.!For,MessList.MessNumber.!To,MessList.MessNumber.!Cc,,,MessList.MessNumber.!Subject,MessList.MessNumber.!Date,MessList.MessNumber.!MessageID,MessList.MessNumber.!Bytes  /* log the message */
          Action = ''  /* no more action required */
         end

        end

       end

       if (Action >< '') then  /* if we are to carry on */
       do

        if (Global.!Settings.MaxCollectSizeTotal >< '') then  /* if we have a total size limit */
        do

         if (MessList.MessNumber.!Bytes + SizeTotal > Global.!Settings.MaxCollectSizeTotal) then  /* if this one is over the limit */
         do
          call LogAction 'Total message size has been reached (MaxCollectSizeTotal = '||Global.!Settings.MaxCollectSizeTotal||').'  /* report */
          call POP3SkipMess MessNumber  /* skip the message */
          call LogMail '===',MessList.MessNumber.!From,MessList.MessNumber.!For,MessList.MessNumber.!To,MessList.MessNumber.!Cc,,,MessList.MessNumber.!Subject,MessList.MessNumber.!Date,MessList.MessNumber.!MessageID,MessList.MessNumber.!Bytes  /* log the message */
          Action = ''  /* no more action required */
          CollectOK = 0  /* we will no longer collect messages */
         end

        end

       end

      end

      if (Action >< '') then  /* if we are to carry on */
      do

       MessFile = POP3RetrieveMess(Socket,MessNumber,MessList.MessNumber.!Bytes)  /* if we can collect the message, we should get back a file name */

       if (MessFile >< '') then  /* if we get back a file name */
       do

        if ((Action == 'GET') & (\Global.!Settings.POP3KeepMessages)) then  /* unless we want to always keep the messages on the server */
        do

         if (POP3DeleteMess(Socket,MessNumber)) then  /* if we can delete the message */
         do
          Deleted = Deleted + 1  /* up the deleted messages counter */
         end

        end

        SizeTotal = SizeTotal + MessList.MessNumber.!Bytes  /* add the message size to the total download size */
        Collected = Collected + 1  /* up the counter */
        call LogMail '<==',MessList.MessNumber.!From,MessList.MessNumber.!For,MessList.MessNumber.!To,MessList.MessNumber.!Cc,,,MessList.MessNumber.!Subject,MessList.MessNumber.!Date,MessList.MessNumber.!MessageID,MessList.MessNumber.!Bytes  /* log the message */
        call RunCommand Global.!Settings.RunReceived,MessFile  /* run an optional external command on the message file */

        if (FileCheck(MessFile)) then  /* if the mail file is still there */
        do

         if (RegisterMessage(MessFile)) then  /* if the message gets registered */
         do

          if (SoundSignal(Global.!Settings.SignalReceived)) then  /* if a signal got sounded */
          do
           Signalled = Signalled + 1  /* up the collected and signalled messages counter */
          end

         end

        end

       end
       else  /* if we do not get back a filename (i.e. the message could not be retrieved or stored) */
       do
        Action = ''  /* no more action required */
        CollectOK = 0  /* stop collecting messages */
       end

      end

     end
     else  /* if we are no longer collecting */
     do
      call POP3SkipMess MessNumber  /* skip the message */
     end

    end

    when (Action == 'DELETE') then  /* if we have this */
    do

     if (\Global.!Settings.POP3KeepMessages) then  /* unless we want to keep the messages on the server */
     do

       if (POP3DeleteMess(Socket,MessNumber)) then  /* if we can delete the message */
       do
        call LogMail '=X=',MessList.MessNumber.!From,MessList.MessNumber.!For,MessList.MessNumber.!To,MessList.MessNumber.!Cc,,,MessList.MessNumber.!Subject,MessList.MessNumber.!Date,MessList.MessNumber.!MessageID,MessList.MessNumber.!Bytes  /* log the message as deleted */
        Deleted = Deleted + 1  /* up the deleted messages counter */
       end

     end

    end

    when (Action == 'SKIP') then  /* if we have this */
    do
     call POP3SkipMess MessNumber  /* skip the message */
    end

    when (Action == 'UNDELETE') then  /* if we have this */
    do

     if (POP3Reset(Socket)) then  /* if we can reset the server */
     do
      Deleted = 0  /* we have no deleted messages now */
     end

     Index = MessCount  /* make sure we exit the current counted loop to rescan the list */

    end

    when (Action == 'QUIT') then  /* if we have this */
    do
     return Signalled  /* quit */
    end

    otherwise  /* if none of the above */
    do
     call AddError 'Incorrect entry in collect action file: "'||Action||'"'  /* report */
     call POP3SkipMess MessNumber  /* skip the message */
    end

   end

  end

 end

 call ShowLine  /* empty line on the display */

end

/**************************************************************************/
HaltPOP3Process:  /* handles halt locally */
/**************************************************************************/

call LogAction 'Messages on server : '||MessCount  /* report */
call LogAction 'Messages deleted   : '||Deleted  /* report */
call LogAction 'Messages collected : '||Collected  /* report */
call LogAction 'Messages signalled : '||Signalled  /* report */
call LogAction 'Bytes collected    : '||SizeTotal  /* report */

return Signalled  /* end of POP3Process */

/**************************************************************************/
POP3Reset: procedure expose Global.  /* resets the POP3 server */
/**************************************************************************/

signal on halt name HaltPOP3Reset  /* handles halt locally */

parse arg Socket  /* get the argument */

if (\SocketSendLine(Socket,'RSET','+OK')) then  /* if we cannot reset the server */
do
 call AddError 'Cannot undelete messages on server'  /* report */
 return 0  /* return with an error */
end

call LogAction 'Messages undeleted'  /* report */

return 1  /* end of POP3Reset */

/**************************************************************************/
HaltPOP3Reset:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
POP3RetrieveMess: procedure expose Global.  /* retrieve a message from the POP3 server */
/**************************************************************************/

signal on halt name HaltPOP3RetrieveMess  /* handles halt locally */
signal on notready name POP3IOError  /* to handle I/O errors */

parse arg Socket,Index,ByteCount  /* get the arguments */

MailFile = TempFileName(,,'POP3')  /* get a unique file name in the system temp folder */

if (MailFile == '') then  /* if we cannot get a temp file */
do
 return ''  /* quit with no success */
end

if (\SocketSendLine(Socket,'RETR '||Index,'+OK')) then  /* if we cannot send this and get the right reply */
do
 call AddError 'Cannot retrieve message (server no. '||Index||') from server'  /* report */
 return ''  /* return with no success */
end

if (\FileOpen(MailFile)) then  /* if we cannot open the file for normal writing */
do
 call AddError 'Cannot open '||MailFile  /* report */
 return ''  /* return with no success */
end

call lineout MailFile,'Received: by '||FullHostName()||'; '||DateTimeRFC(1)  /* start with a "Received:" line (contact time servers if necessary) */
InBytes = 0  /* we have collected nothing yet */
ShowStats = Global.!Settings.ShowProgress  /* if we want stats, set a local flag */

if (ShowStats) then  /* if we want stats */
do
 signal on syntax name POP3TimeBug  /* go here if the REXX time() bug strikes */
 call time 'R'  /* reset the timer */
 signal on syntax name Syntax  /* use the normal syntax error routine */
 call syscurstate 'OFF'  /* switch off the cursor */
 call ProgressBar 'Bytes retrieved = ',0,ByteCount,0  /* start a progress bar */
 TransTime = 0  /* no transmission time yet */
 ElapsedTime = 0  /* no elapsed time yet */
end

POP3TimeBugDone:  /* we come back here after the REXX time() bug strikes */

NextLine = SocketGetLine(Socket)  /* get the next line */

do while ((NextLine >< '.') & (Global.!SocketOK))  /* as long as do not get a full stop or an error */

 if (left(NextLine,1) == '.') then  /* if the line starts with a full stop */
 do
  NextLine = substr(NextLine,2)  /* skip the first character */
 end

 call lineout MailFile,NextLine  /* write the line to the file */
 InBytes = InBytes + length(NextLine) + 2  /* up the byte count */

 if (ShowStats) then  /* if we want stats */
 do

  signal on syntax name POP3TimeBug  /* go here if the REXX time() bug strikes */
  ElapsedTime = ElapsedTime + time('R')  /* update the elapsed time and reset the timer */
  signal on syntax name Syntax  /* use the normal syntax error routine */

  if (ElapsedTime > 0.2) then  /* if the elapsed time is sufficient */
  do
   TransTime = TransTime + ElapsedTime  /* update the transmission time */
   ElapsedTime = 0  /* start all over again */
   call ProgressBar 'Bytes retrieved = ',InBytes,ByteCount,TransTime  /* show the status */
  end

 end

 NextLine = SocketGetLine(Socket)  /* get the next line */

end

if (InBytes < ByteCount) then  /* if we have fewer bytes than expected, we may have a rogue message on our hands */
do
 call lineout MailFile,'[RexxMail warning: message terminator arrived sooner than expected'  /* insert a warning line in the message */
 call lineout MailFile,' no. of bytes expected = '||ByteCount  /* insert a warning line in the message */
 call lineout MailFile,' no. of bytes received = '||InBytes||']'  /* insert a warning line in the message */
end

call FileClose MailFile  /* close the file */

if (ShowStats) then  /* if we want stats */
do
 call ProgressBar  /* clear the status display */
 call syscurstate 'ON'  /* switch the cursor back on */
end

call LogAction 'Message retrieved'  /* report */
signal off notready  /* no more need to locally handle I/O errors */

return MailFile  /* end of POP3RetrieveMess */

/**************************************************************************/
POP3TimeBug:  /* handles the Classic REXX time() bug */
/**************************************************************************/

signal on syntax name Syntax  /* use the normal syntax error routine */

ShowStats = 0  /* no more statistics for this one */
call ProgressBar  /* clear the status display */
call syscurstate 'ON'  /* switch the cursor back on */
call LogAction 'REXX Time() error; do not synchronize clock while collecting mail. Resuming.'  /* report */

signal POP3TimeBugDone  /* go here */

/**************************************************************************/
POP3IOError:  /* handles local I/O errors */
/**************************************************************************/

call LogErrors 'I/O error while retrieving message'  /* report */

/**************************************************************************/
HaltPOP3RetrieveMess:  /* handles halt locally */
/**************************************************************************/

call FileClose MailFile  /* try to close the file */
call sysfiledelete MailFile  /* get rid of the file */
signal off notready  /* no more need to locally handle I/O errors */

return ''  /* quit with nothing */

/**************************************************************************/
POP3SkipMess:  procedure expose Global.  /* skips a message on the POP3 server (i.e. does nothing; just reports) */
/**************************************************************************/

signal on halt name HaltPOP3SkipMess  /* handles halt locally */

parse arg Index  /* get the argument */

call LogAction 'Message skipped'  /* report */

return 1  /* end of POP3SkipMess */

/**************************************************************************/
HaltPOP3SkipMess:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
ProgressBar: procedure expose Global.  /* displays a byte counter, timer and progress bar, and optionally erases the lot */
/**************************************************************************/

parse arg Text,CurCount,TotalCount,TransTime  /* get the arguments */

CurRow = word(syscurpos(),1)  /* get the current row number */
ScreenWidth = word(systextscreensize(),2)  /* get the screen width */

if (Text == '') then  /* if we want to clean up */
do

 call syscurpos CurRow,0  /* return the cursor to its original position */
 call charout 'CON:',copies(' ',(ScreenWidth - 1))  /* erase the byte counter line */

 if (ScreenWidth > 20) then  /* if we have a progress bar */
 do
  call syscurpos (CurRow + 1),0  /* move to the start of the progress bar line */
  call charout 'CON:',copies(' ',(ScreenWidth - 1))  /* erase the line */
 end

 call syscurpos CurRow,0  /* return the cursor to its original position */

 return  /* quit */

end

if (wordpos(Global.!CodePage,'437 850') > 0) then  /* if we are running either of these code pages */
do
 FGChar = ''  /* use this foreground bar character */
 BGChar = ''  /* use this background bar character */
end
else  /* if we have some other code page */
do
 FGChar = '#'  /* use this foreground bar character */
 BGChar = '_'  /* use this background bar character */
end

if (TotalCount > 0) then  /* if we have a total byte count */
do

 if (CurRow == word(systextscreensize(),1) - 1) then  /* if we are at the bottom row of the screen */
 do
  call ShowLine  /* skip to the next line to move the existing text up one line */
  CurRow = max((CurRow - 1),0)  /* position the byte counter a row up, but not past the edge of the screen */
  call syscurpos CurRow,0  /* reposition the cursor */
 end

 CountLength = length(TotalCount)  /* the maximum length of the byte count string */
 Filler = '0'  /* pad with zeroes */

end
else  /* if we have no maximum byte count */
do
 CountLength = 9  /* use this (enough for 999,999,999 bytes) */
 Filler = ' '  /* pad with blanks */
end

call charout 'CON:',Text||right(CurCount,CountLength,Filler)  /* show the current byte count */

if (TotalCount > 0) then  /* if we have a total byte count */
do

 call charout 'CON:','/'||TotalCount||' - Time left = '  /* report */

 if (CurCount == 0) then  /* if we have no bytes yet, we are just starting out */
 do
  call charout 'CON:','__:__:__'  /* report nothing */
 end
 else  /* if we are already under way */
 do
  SecsLeft = format((TotalCount - CurCount) / CurCount * TransTime,,0)  /* calculate the number of whole seconds left */
  HoursLeft = right(SecsLeft % 3600,2,'0')  /* calculate the hours left */
  SecsLeft = SecsLeft // 3600  /* calculate the seconds left without the hours */
  MinsLeft = right(SecsLeft % 60,2,'0')  /* calculate the minutes left */
  SecsLeft = right(SecsLeft // 60,2,'0')  /* calculate the seconds left without the minutes */
  call charout 'CON:',HoursLeft||':'||MinsLeft||':'||SecsLeft  /* report the time left */
 end

 if (ScreenWidth > 20) then  /* if we have room for a progress bar */
 do

  if (CurCount <= TotalCount) then  /* if we are still within a useful range */
  do
   ProgCount = format((CurCount / TotalCount) * (ScreenWidth - 4),,0)  /* the number of progress blocks to show on the progress bar */
   call syscurpos (CurRow + 1),0  /* move to the next row */
   call charout 'CON:','  '||copies(FGChar,ProgCount)||copies(BGChar,(ScreenWidth - 4 - ProgCount))  /* show the progress bar */
  end

 end

end

call syscurpos CurRow,0  /* return the cursor to its original position */

return  /* end of ProgressBar */

/**************************************************************************/
PutFileContents: procedure expose Global.  /* writes the contents of a file */
/**************************************************************************/

signal on halt name HaltPutFileContents  /* handles halt locally */

parse arg FileName,Content  /* get the arguments */

if (\FileOpen(FileName)) then  /* if we cannot open the file for writing */
do
 return 0  /* quit with no result */
end

if (charout(FileName,Content,1) >< 0) then  /* if we cannot write the full content */
do
 return 0  /* quit with no result */
end

call FileClose FileName  /* close the file */

return 1  /* end of PutFileContents */

/**************************************************************************/
HaltPutFileContents:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
PutObjectEA: procedure expose Global.  /* attempts to attach an extended attribute to an object */
/**************************************************************************/

parse arg Object,EAName,EAString,MVT  /* get the arguments */

Lines = 0  /* no lines yet */
EAValue = ''  /* nothing yet */
MVT = (MVT == 1)  /* 1 = true */

do while (length(EAString) > 0)  /* run through the EA string */
 parse var EAString EALine (Global.!CRLF) EAString  /* get the next line of the EA string */
 EAValue = EAValue||'FDFF'x||reverse(right(d2c(Length(EALine)),2,'00'x))||EALine  /* add another ASCII string to the EA contents */
 Lines = Lines + 1  /* up the counter */
end

if ((Lines > 1) | (MVT)) then  /* if we have more than one line, or if we want an MVT entry */
do
 EAValue = 'DFFF'x||'0000'x||reverse(right(d2c((Lines)),2,'00'x))||EAValue  /* prepare the full EA */
end

Attempts = 0  /* we haven't tried yet */
MaxAttempts = 20  /* stop after so many tries */
Delay = 0.005  /* start with this delay */

do while ((sysputea(Object,EAName,EAValue) >< 0) & (Attempts < MaxAttempts))  /* go on until we succeed or reach the maximum no. of attempts */
 Attempts = Attempts + 1  /* up the attempts counter */
 call syssleep (Delay * Attempts)  /* wait a bit */
end

if (Attempts == MaxAttempts) then  /* if we reached the last try */
do
 call AddError 'Cannot attach extended attributes to object "'||Object||'"'  /* report */
 return 0  /* no success */
end

return 1  /* end of PutObjectEA */

/**************************************************************************/
RedirectAdd: procedure expose Global.  /* processes redirection lines from command line */
/**************************************************************************/

parse arg Switch Type Action '=' Recipe  /* get the arguments */

Recipe = strip(Recipe)  /* remove excess whitespace */

if (Recipe == '') then  /* if we have no recipe */
do
 call AddError 'Missing redirection recipe'  /* report */
 return 0  /* quit */
end

if (Action == '') then  /* if we have no action */
do
 call AddError 'Missing redirection action'  /* report */
 return 0  /* quit */
end

if (wordpos(translate(Type),'COLLECT REGISTER VIEW SEND') == 0) then  /* if we have an invalid type */
do
 call AddError 'Invalid redirection type: "'||Type||'"'  /* report */
 return 0  /* and quit */
end

MessFile = Global.!ProcFile  /* the file to process */
OutFile = Global.!ConfigurationDir||'\'||Type||'.txt'  /* the redirection file to use */

if (pos('#',Recipe) > 0) then  /* if it looks like we have a macro */
do

 MessCont = GetFileContents(MessFile)  /* get the message file contents */

 if (\MessageContents(MessCont,'Global.!MessHead.')) then  /* if we cannot get the message header sorted out */
 do
  return 0  /* quit with no attachments */
 end

 NewRecipe = ''  /* start with nothing */

 do while ((Recipe >< '') & (pos('#',Recipe) > 0))  /* as long as we have something left with a macro in it */

  parse var Recipe FirstBit '#' +1 Macro +1 Recipe  /* get the interesting bits */

  NewRecipe = NewRecipe||FirstBit  /* restore the first bit */

  if (Macro == '') then  /* if we have no macro content */
  do
   call AddError 'Missing macro specification'  /* report */
   return 0  /* quit */
  end

  if (Macro == '#') then  /* if we have this */
  do
   NewRecipe = NewRecipe||'#'  /* simply use it */
  end
  else  /* if it is something else */
  do

   Macro = translate(Macro)  /* make the macro upper case */
   MacroType = pos(Macro,'ABCFTDS')  /* is the macro one of the valid ones? */

   if (MacroType == 0) then  /* if it is not */
   do
    call AddError 'Invalid macro: "#'||Macro||'"'  /* report */
    return 0  /* quit */
   end

   Keyword = word('ACC BCC CC FROM TO DATE SUBJECT',MacroType)  /* get the corresponding keyword */
   NewBit = GetHeaderEntry(Keyword,'Global.!MessHead.')  /* get the corresponding header value */

   if (NewBit == '') then  /* if there is no such entry */
   do
    call AddError 'No '||Keyword||' header entry found'  /* report */
    return 0  /*quit */
   end

   if (MacroType > 4) then  /* if it is a date or subject line */
   do
    NewRecipe = NewRecipe||Keyword||' : '||NewBit  /* add the new bit to what we have, including the keyword */
   end
   else  /* if it is one containing addresses */
   do

    NewBit = AddressFormat(NewBit,0,0)  /* get a list of bare addresses */

    do while (NewBit >< '')  /* as long as we have something left */

     parse var NewBit NextAddress NewBit  /* get the next address */
     NewRecipe = NewRecipe||Keyword||' : '||NextAddress  /* add the keyword, a colon, and the address */

     if (NewBit >< '') then  /* if there is more */
     do
      NewRecipe = NewRecipe||' & '  /* add this */
     end

    end

   end

  end

 end

 Recipe = NewRecipe||Recipe  /* we have a new recipe (add any old recipe that may be left over) */

end

select  /* do one of the following */

 when (Switch == 'REDIRECTADD') then  /* if we have this one */
 do

  Comment = copies('#',76)||Global.!CRLF||'# AddRedirect '||DateTimeSys()||' :'  /* a comment line */

  if (\PutFileContents(OutFile,Comment||Global.!CRLF||Action||' = '||Recipe||Global.!EmptyLine||GetFileContents(OutFile))) then  /* if we cannot stick our new redirect line at the front of the redirect file */
  do
   return 0  /* quit */
  end

 end

 otherwise  /* if none of the above */
 do
  nop  /* this should not happen */
 end

end

return 1  /* end of RedirectAdd */

/**************************************************************************/
RegisterMessage: procedure expose Global.  /* registers incoming messages, optionally moving them to a special folder and setting the folder's icon */
/**************************************************************************/

signal on halt name HaltRegisterMessage  /* handles halt locally */

parse arg MessFile  /* get the arguments */

if (MessFile == '') then  /* if we have none */
do
 MessFile = Global.!ProcFile  /* get the file name to process */
end

if (\IsMessage(MessFile)) then  /* if the file is not a message */
do
 return 0  /* return with no result */
end

GotAtt = (HeaderContains('CONTENT-TYPE','MULTIPART/MIXED','Global.!MessHead.'))  /* if we have this header entry, we have one or more attachments */
MoveDir = MoveAction(MessFile,Global.!ControlRegi,Global.!In_ArchiveDir)  /* see if the message needs to be moved to anywhere special */

if (\FileCheck(MessFile)) then  /* if the message got zapped along the way (e.g. by a "RUN" redirection action) */
do
 return 0  /* quit */
end

if (MoveDir == '') then  /* if we get no special destination */
do
 MoveDir = Global.!InDir  /* simply use the In dir */
end

if (\DirCheck(MoveDir)) then  /* if the destination dir does not exist (non-fatal) */
do
 MoveDir = directory()  /* use the current dir */
end

MailFile = TempFileName(MoveDir,,'REGISTER')  /* get a unique file name */

if (MailFile == '') then  /* if we get no file name */
do
 return 0  /* return with an error */
end

call CreateCWMFObject filespec('N',MailFile),MoveDir  /* try to create a CWMAILFile class object in the destination dir using the name part */

if (\FileCopy(MessFile,MailFile)) then  /* if we cannot copy the message contents to the new mail file */
do
 call AddError 'Cannot copy "'||MessFile||'" to "'||MailFile||'"'  /* report */
 return 0  /* and quit */
end

Stationery = GetObjectEA(MessFile,'RXMLSTATIONERY')  /* look for a stationery directive */
call sysdestroyobject MessFile  /* get rid of the original file */

if (Stationery >< '') then  /* if we found a stationery directive */
do
 call PutObjectEA MailFile,'RXMLSTATIONERY',Stationery  /* write the stationery EA to the new file */
end

call MessageSettings MailFile,'1000'||GotAtt||'000','CHANGE'  /* set the incoming message status: "received & unread" mail message status, any attachments?, and complete with unset bits */
call MakeShadow MailFile,Global.!Settings.ShadowRegister,Global.!In_ArchiveDir,MoveDir,Global.!InDir  /* create a shadow if necessary */
NewName = MakeTitle(MailFile,1,0,1,1)  /* get the new title and insert warnings, and do object stuff */
call SetObjectData MailFile,'REXXMAILREFRESH=YES;REXXMAILATTACHMENT='||word('No Yes',(GOTATT + 1))||';TITLE='||NewName  /* set the file attributes */
ArchDir = translate(Global.!In_ArchiveDir)  /* make upper case copy */
MoveDir = translate(MoveDir)  /* make upper case */

if (pos(ArchDir,MoveDir) == 1) then  /* if the folder is a RexxMail In Archive subfolder */
do
 
 parse var MoveDir (ArchDir) '\' MoveDir  /* remove the first bit */

 do while (MoveDir >< '')  /* as long as we have bits left */
  parse var MoveDir NextDir '\' MoveDir  /* get the next bit */
  ArchDir = ArchDir||'\'||NextDir  /* restore the dir path */
  call SetObjectData ArchDir,'ICONFILE='||Global.!IconDir||'\Folders\Unread_0.ICO'  /* set the closed unread mail folder icon */
  call SetObjectData ArchDir,'ICONNFILE=1,'||Global.!IconDir||'\Folders\Unread_1.ICO'  /* set the animated unread mail folder icon */
 end

end

return 1  /* end of RegisterMessage */

/**************************************************************************/
HaltRegisterMessage:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
RemoveFromHeader: procedure expose Global.  /* removes an entry from a message header */
/**************************************************************************/

parse arg MessHead,Keyword  /* get the arguments */

NewHead = ''  /* start with nothing */

do while (MessHead >< '')  /* as long as we have header content */

 parse var MessHead NextLine (Global.!CRLF) MessHead  /* get the next line */

 if (translate(left(NextLine,(length(Keyword) + 1))) == Keyword||':') then  /* if we have the entry we want to lose */
 do

  do until (left(NextLine,1) >< ' ')  /* go on until we reach an unfolded line */
   parse var MessHead NextLine (Global.!CRLF) MessHead  /* get the next line */
  end

  NewHead = NewHead||NextLine||Global.!CRLF  /* restore the last line */
  
 end
 else  /* if not */
 do
  NewHead = NewHead||NextLine||Global.!CRLF  /* restore the line */
 end

end

return NewHead  /* end of RemoveFromHeader */

/**************************************************************************/
ResendMessage: procedure expose Global.  /* resend a previously sent message */
/**************************************************************************/

MessFile = Global.!ProcFile  /* the file name to process */

if (\MessageSettings(MessFile,'0*1*****','MATCH')) then  /* if it is not a previously sent message */
do
 call AddError 'Invalid message type'  /* report */
 return 0  /* and quit */
end

if (SMTPSendMessage(MessFile,1)) then  /* if we can resend the message */
do
 call SoundSignal Global.!Settings.SignalSent  /* signal if required */
end

return 1  /* end of ResendMessage */

/**************************************************************************/
ResetFolderIcon: procedure expose Global.  /* resets a folder icon if necessary */
/**************************************************************************/

parse arg Folder  /* get the argument */

ResetIcon = 1  /* assume we are to reset the message folder icon to begin with */
call sysfiletree Folder||'\*','Files.','FO'  /* look for files in the In folder */

do Index = 1 to Files.0  /* run through the lot */

 if (MessageSettings(Files.Index,,'CHECK')) then  /* if it is a RexxMail message file */
 do

  if (MessageSettings(Files.Index,'**0*****','MATCH')) then  /* if the file is unprocessed, i.e. unread */
  do
   ResetIcon = 0  /* the icon stays put */
  end

 end

end

if (\ResetIcon) then  /* if we are to reset the message dir icon */
do
 return 0  /* return with no action */
end

if (Folder == Global.!In_ArchiveDir) then  /* if it is the In Archive folder */
do
 call SetObjectData Folder,'ICONFILE='||Global.!IconDir||'\Folders\In_Archive_0.ICO'  /* specify the normal closed icon */
 call SetObjectData Folder,'ICONNFILE=1,'||Global.!IconDir||'\Folders\In_Archive_1.ICO'  /* specify the normal animated icon */
end
else  /* if it is a subfolder */
do
 call SetObjectData Folder,'ICONFILE='||Global.!IconDir||'\Folders\Normal_0.ICO'  /* specify the normal closed icon */
 call SetObjectData Folder,'ICONNFILE=1,'||Global.!IconDir||'\Folders\Normal_1.ICO'  /* specify the normal animated icon */
end

return 1  /* end of ResetFolderIcon */

/**************************************************************************/
RunCommand: procedure expose Global.  /* runs an external command if necessary */
/**************************************************************************/

signal on halt name HaltRunCommand  /* handles halt locally */

parse arg CommLine,CommFile  /* get the arguments */

if (CommLine == '') then  /* if we have no command definition */
do
 return ''  /* quit with nothing */
end

if (CommFile >< '') then  /* if we have a file spec */
do

 if (pos('#N',CommLine) > 0) then  /* if we have a filename macro */
 do
  parse var CommLine FirstBit '#N' LastBit  /* look for the filename macro in the command line definition */
  CommLine = FirstBit||CommFile||LastBit  /* change the command to include the file spec */
 end

end

call LogAction 'Command configuration option parsed as:'||Global.!CRLF||,  /* report */
               '  '||CommLine,1  /* report, quietly */
OrgDir = directory()  /* store the current directory name */

signal off Error  /* we'll handle errors ourselves */
signal off Failure  /* we'll handle failures ourselves */
address cmd CommLine  /* run the command */
signal on Error  /* back to where we were */
signal on Failure  /* back to where we were */

if (RC == '') then  /* if we get nothing in return */
do
 RC = 0  /* use this */
end

return RC  /* end of RunCommand */

/**************************************************************************/
HaltRunCommand:  /* handles halt locally */
/**************************************************************************/

return 1  /* no success */

/**************************************************************************/
SendMessage: procedure expose Global.  /* sends an outgoing message */
/**************************************************************************/

signal on halt name HaltSendMessage  /* handles halt locally */

parse arg Bare  /* get the argument */

MessFile = Global.!ProcFile  /* get the name of the file to send */

if (\FileCheck(MessFile)) then  /* if the file no longer exists */
do
 return 0  /* quit with no result */
end

Bare = (Bare == 1)  /* 1 = TRUE */
NoSend = 0  /* assume we are actually sending the message */

if (\Bare) then  /* if we are not sending a bare message */
do

 NoSend = MessageSettings(MessFile,'001*****','MATCH')  /* if the file is outgoing and already processed, set a flag */

 if (\NoSend & (\MessageSettings(MessFile,'01******','MATCH'))) then  /* if we are actually sending and the file is not outgoing and O.K. for processing */
 do
  return 0  /* quit with no result */
 end

end

SendCont = GetFileContents(MessFile)  /* get the file contents */

if (\MessageContents(SendCont,'Global.!MessHead.')) then  /* if we cannot get the message file header sorted out */
do
 return 0  /* quit */
end

MoveDir = (MoveAction(MessFile,Global.!ControlSend,Global.!Out_ArchiveDir))  /* see if the file is to be moved to another folder */

if (\FileCheck(MessFile)) then  /* if the message got zapped (e.g. by a "RUN" action) */
do
 return 0  /* quit */
end

if (NoSend) then  /* if we are not actually sending */
do

 if (MoveDir >< '') then  /* if the send file is to be moved */
 do
  call MoveMessage MessFile,MoveDir,0  /* moved it to the other folder */
 end

 return 1  /* quit */

end

if (\Bare) then  /* if the message is not to be sent "bare" */
do

 if (Global.!Settings.SendASCII) then  /* if we want mail sent as ASCII text */
 do
  call MessageSettings MessFile,'***1****','CHANGE'  /* set the ASCII flag (which may already be set */
 end

 ASCII = MessageSettings(MessFile,'***1****','MATCH')  /* if we find an ASCII EA, this message should be sent as ASCII */
 MessSett = MessageSettings(MessFile)  /* get the message settings */
 AttDir = AttDirGet(MessFile)  /* get the attachments dir */

end
else  /* if we are sending "bare" */
do
 AttDir = ''  /* no attachments dir */
end

SendFile = MakeSendFile(SendCont,AttDir,Bare,ASCII)  /* try to get a sendable file */

if (SendFile == '') then  /* if we do not get back a sendable file name */
do
 return 0  /* quit */
end

call RunCommand Global.!Settings.RunBeforeSend,SendFile  /* run any external command before sending */
RunSendResult = RunCommand(Global.!Settings.RunSend,SendFile)  /* see if we can run an external send program */

if (RunSendResult == '') then  /* if we could not run an external send command, i.e. it does not exist */
do

 if (\SMTPSendMessage(SendFile)) then  /* if the internal routine to send the file fails */
 do
  return 0  /* quit with no success */
 end

end
else  /* if we had an external command to run */
do

 if (RunSendResult >< 0) then  /* if the command failed */
 do
  return 0  /* quit with no success */
 end

end

call SoundSignal Global.!Settings.SignalSent  /* signal if required */
call RunCommand Global.!Settings.RunAfterSend,SendFile  /* run any external command after sending */

if ((\Bare) & (FileCheck(SendFile))) then  /* if the message was sent "not bare" and if the send file still exists (it may have been deleted by an external command) */
do

 if (AttDir >< '') then  /* if there is an attachments dir */
 do
  call sysdestroyobject AttDir  /* remove it */
 end

 call sysdestroyobject MessFile  /* get rid of the message file */

 if (MoveDir >< Global.!TempDir) then  /* if we are keeping the sent message */
 do

  MessDir = filespec('D',MessFile)||strip(filespec('P',MessFile),'T','\')  /* the original message dir */

  if (MoveDir == '') then  /* if the send file is not being moved to anywhere in particular */
  do
   MoveDir = MessDir  /* use the original message folder */
  end

  call MessageSettings SendFile,left(MessSett,1)||'01'||right(MessSett,5),'CHANGE'  /* new sent mail message setting: Processed and Not O.K. for processing */
  call MakeShadow SendFile,Global.!Settings.ShadowSend,Global.!Out_ArchiveDir,MoveDir,MessDir  /* create a shadow if necessary */
  call MoveMessage SendFile,MoveDir  /* move the send file */

 end

end

call ShowLine  /* skip a line on the display */

return 1  /* end of SendMessage */

/**************************************************************************/
HaltSendMessage:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
ServerConnect: procedure expose Global.  /* connects to a server */
/**************************************************************************/

signal on halt name HaltServerConnect  /* handles halt locally */

parse arg Server,Port,MaxAttempts  /* get the arguments */

if (MaxAttempts == 0) then  /* if no attempts are to be made */
do
 return ''  /* return with no socket */
end

if (sockinit() >< 0) then  /* if we cannot initialize the socket buffers */
do
 call AddError 'Socket initialization error'  /* report */
 return ''  /* return with no socket */
end

Socket = socksocket('AF_INET','SOCK_STREAM','IPPROTO_TCP')  /* create a socket for TCP protocol */

if (Socket < 0) then  /* if we did not get a socket */
do
 call AddError 'Cannot get socket'  /* report */
 return ''  /* return with no socket */
end

if (verify(Server,'1234567890.','NOMATCH') == 0) then  /* if we have nothing but digits and dots, assume it was a dotted address */
do
 Address.!Addr = Server  /* use the server address as it is */
end
else  /* if not, asssume it was a mnemonic name */
do

 call LogAction 'Resolving host name "'||Server||'"'  /* report */
 Attempts = 0  /* no try yet */
 GotHost = 0  /* no host contacted yet */

 do while ((\GotHost) & (Attempts < MaxAttempts))  /* go on until we get a result, or get timed out */

  GotHost = sockgethostbyname(Server,'Host.!')  /* if we can get the SMTP server's IP address by name, all is well */

  if (\GotHost) then  /* if we did not get the right result */
  do
   call syssleep 1  /* wait a second */
   Attempts = Attempts + 1  /* up the Attempts counter */
   call LogAction 'Retrying'  /* report */
  end

 end

 if (\GotHost) then  /* if we did not get the right result */
 do
  call AddError 'Cannot resolve host name "'||Server||'"'  /* report */
  return ''  /* return with no socket */
 end

 Address.!Addr = Host.!Addr  /* define the connect address */

end

Address.!Family = 'AF_INET'  /* define the connection type */
Address.!Port = Port  /* define the port to use */

if (Server >< Address.!Addr) then  /* if we have a menomnic name in addition to a dotted address */
do
 AddMessage = '"'||Server||'" at '  /* add this to the next message */
end
else  /* if not */
do
 AddMessage = ''  /* add nothing */
end

call LogAction 'Connecting to host '||AddMessage||Address.!Addr||' through port '||Port  /* report */
Attempts = 0  /* no attempts yet */
GotConn = 0  /* no connection yet */

do while ((\GotConn) & (Attempts < MaxAttempts))  /* go on until we get a result, or get timed out */

 GotConn = (sockconnect(Socket,'Address.!') == 0)  /* if we can open the socket connection, all is well */

 if (\GotConn) then  /* if we did not get the right result */
 do
  call syssleep 1  /* wait a second */
  Attempts = Attempts + 1  /* up the attempts counter */
  call LogAction 'Retrying'  /* report */
 end

end

if (\GotConn) then  /* if we did not get the right result */
do
 call AddError 'Cannot connect to host "'||Server||'"'  /* report */
 return ''  /* return with no success */
end

return Socket  /* end of ServerConnect */

/**************************************************************************/
HaltServerConnect:  /* handles halt locally */
/**************************************************************************/

return ''  /* no success */

/**************************************************************************/
ServerDisconnect: procedure expose Global.  /* disconnect from a server */
/**************************************************************************/

signal on halt name HaltServerDisconnect  /* handles halt locally */

parse arg Socket,Server  /* get the arguments */

call LogAction 'Disconnecting from "'||Server||'"'  /* report */

if (\SocketSendLine(Socket,'QUIT','221 +OK')) then  /* if we do not get 221 or +OK in return when we send this line */
do
 call AddError 'Error while disconnecting from "'||Server||'"'  /* report */
end

call SocketClose Socket  /* close the socket */

return 1  /* end of ServerDisconnect */

/**************************************************************************/
HaltServerDisconnect:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
SetMessageType: procedure expose Global.  /* sets message type attributes */
/**************************************************************************/

parse upper arg Switch  /* get the argument, in upper case */

if (Switch == '') then  /* if we have no switch */
do
 return 0  /* quit */
end

MessFile = Global.!ProcFile  /* the file name to process */

if (\IsMessage(MessFile)) then  /* if the file is not a message */
do
 return 0  /* no result */
end

MessSettings = (MessageSettings(MessFile))  /* get the message file's status settings */

if (Switch == 'SETMESSAGEICON') then  /* if we are simply updating */
do
 Settings = left(MessSettings,3)||'*****'  /* the new settings */
end
else  /* if we are not updating */
do

 if (substr(MessSettings,1,1) == 0) then  /* if it is an outgoing message */
 do

  if (substr(MessSettings,3,1) == 0) then  /* if the message has not yet been sent */
  do

   if (Switch == 'SETASCIIQP') then  /* if we are to toggle the ASCII/Q-P setting */
   do

    if (substr(MessSettings,4,1) == 1) then  /* if the ASCII flag has already been set */
    do
     Switch = 'SETQP'  /* reset it */
    end
    else  /* if the ASCII flag has not been set yet */
    do
     Switch = 'SETASCII'  /* set it */
    end

   end
   else  /* if we are not to toggle the ASCII/Q-P setting */
   do

    if (Switch == 'SETOKNOTOKTOSEND') then  /* if we are to toggle the new/sendable setting */
    do

     if (substr(MessSettings,2,1) == 1) then  /* if the sendable flag has already been set */
     do
      Switch = 'SETNOTOKTOSEND'  /* reset it */
     end
     else  /* if the sendable flag has not been set yet */
     do
      Switch = 'SETOKTOSEND'  /* set it */
     end

    end

   end

  end

 end

 SwitchString = 'SETNEW SETNOTOKTOSEND SETOKTOSEND SETSENT SETASCII SETQP SETRECEIVED SETVIEWED'  /* the available types */
 SettingsString = '00000000 000***** 010***** 001***** ***1**** ***0**** 10000000 101*****'  /* the corresponding settings strings */
 SwitchPos = wordpos(Switch,SwitchString)  /* look for the switch in the switch string */

 if (SwitchPos == 0) then  /* if the switch is not there */
 do
  call AddError 'Invalid switch : "'||Switch||'"'  /* report */
  return 0  /* quit */
 end

 Settings = word(SettingsString,SwitchPos)  /* get the settings that correspond with the message type */

end

if (\MessageSettings(MessFile,Settings,'CHANGE')) then  /* if we cannot attach the mail message settings */
do
 call AddError 'Cannot change file settings'  /* report */
 return 0  /* quit */
end

return 1  /* end of SetMessageType */

/**************************************************************************/
SetObjectData: procedure expose Global.  /* attempts to change an object's settings */
/**************************************************************************/

parse arg Object,Settings  /* get the arguments */

Attempts = 0  /* we haven't tried yet */
MaxAttempts = 20  /* stop after so many tries */
Delay = 0.005  /* start with this delay */

do while ((\syssetobjectdata(Object,Settings)) & (Attempts < MaxAttempts))  /* go on until we succeed or reach the maximum no. of attempts */
 Attempts = Attempts + 1  /* up the attempts counter */
 call syssleep (Delay * Attempts)  /* wait a bit */
end

if (Attempts = MaxAttempts) then  /* if we reached the last try */
do
 call AddError 'Cannot adjust settings for object "'||Object||'" : '||Settings  /* report */
 return 0  /* no success */
end

return 1  /* end of SetObjectData */

/**************************************************************************/
SetProgramObjectIcons: procedure expose Global.  /* sets program object icons */
/**************************************************************************/

parse arg IconFolder  /* get any argument */

if (IconFolder == '') then  /* if there is none */
do
 IconFolder = Global.!IconDir||'\Objects'  /* use the default */
end
else  /* if we have a folder spec */
do

 if (substr(IconFolder,2,2) >< ':\') then  /* if we do not have a drive spec */
 do
  IconFolder = Global.!IconDir||'\'||strip(IconFolder,'L','\')  /* look for the folder in the icon dir */
 end

end

call sysfiletree IconFolder||'\*.ico','Icons.','FO'  /* look for the icons */

if (Icons.0 == 0) then  /* if none are found */
do
 call AddError 'Cannot find icon files in folder "'||IconDir||'"'  /* report */
 return 0  /* and quit */
end

do Index = 1 to Icons.0  /* for each one found */

 IconName = filespec('N',Icons.Index)  /* get the bare name */
 parse upper var IconName IDName '.ICO'  /* get the object ID name in upper case */

 if (IDName >< 'WARNING') then  /* unless we have this special case */
 do

  if (SetObjectData('<REXXMAIL_'||IDName||'>','ICONFILE='||Icons.Index)) then  /* if we can set the icon */
  do
   call LogAction 'Icon of object <REXXMAIL_'||IDName||'> set to "'||Icons.Index||'"',1  /* log it, quietly */
  end

 end

end

return 1  /* end of SetProgramObjectIcons */

/**************************************************************************/
SetStationery: procedure expose Global.  /* sets message stationery */
/**************************************************************************/

MessFile = Global.!ProcFile  /* the file to use */

if (Global.!Settings.Stationery == '') then  /* if we have no stationery defined */
do
 call AddError 'No stationery defined'  /* report */
 return 0  /* quit */
end

call PutObjectEA MessFile,'RXMLSTATIONERY',Global.!Settings.Stationery  /* write the stationery EA to the file */
call MessageSettings MessFile,'********','CHANGE'  /* refresh the message status */

return 1  /* end of SetStationery */

/**************************************************************************/
SetTitle: procedure expose Global.  /* sets message title */
/**************************************************************************/

parse arg MessFile  /* look for an argument */

if (MessFile == '') then  /* if there is none */
do
 MessFile = Global.!ProcFile  /* the file name to process */
end

if (\MessageSettings(MessFile,,'CHECK')) then  /* if it is not a RexxMail message */
do
 return 0  /* quit */
end

MessCont = GetFileContents(MessFile)  /* get the message file contents */

if (\MessageContents(MessCont,'Global.!MessHead.')) then  /* if we cannot get the header sorted out */
do
 return 0  /* quit */
end

NewName = MakeTitle(MessFile,1,0,MessageSettings(MessFile,'1*******','MATCH'),1)  /* get a new name for the mail file, and insert warnings */

if (SetObjectData(MessFile,'REXXMAILREFRESH=YES;TITLE='||NewName)) then  /* if we can set the new title */
do
 NewObject = filespec('D',MessFile)||filespec('P',MessFile)||NewName  /* the full object spec */
end
else  /* if the retitle operation failed */
do
 NewObject = MessFile  /* nothing changes */
end

return NewObject  /* end of SetTitle */

/**************************************************************************/
ShowLine: procedure expose Global.  /* displays a (truncated) line of text on the screen */
/**************************************************************************/

parse arg Text  /* get the argument */

if (Text >< '') then  /* if there is something to show */
do
 Text = left(Text,min(length(Text),(word(systextscreensize(),2) - 1)))  /* truncate if necessary */
end

call lineout 'CON:',Text  /* show the (truncated) text, if any */

return 1  /* end of ShowLine */

/**************************************************************************/
SMTPSendMessage: procedure expose Global.  /* sends mail to an SMTP mail relay */
/**************************************************************************/

signal on halt name HaltSMTPSendMessage  /* handles halt locally */

parse arg SendFile,Resending  /* get the arguments */

Socket = ''  /* no socket yet */

if (Global.!Settings.SMTPServer == '') then  /* if we have no server name */
do
 call AddError 'Missing configuration entry: SMTPServer'  /* report */
 Global.!Break = 1  /* break off processing of further messages */
 return 0  /* and quit with no success */
end

SMTPFile = TempFileName(,,'SMTP')  /* the name of a temp file */

if (SMTPFile == '') then  /* if we cannot get a temp file */
do
 return 0  /* quit with no success */
end

call LogAction 'Creating SMTP file "'||SMTPFile||'"',1  /* report, quietly */
MessCont = GetFileContents(SendFile)  /* get the sendable message file contents */
parse var MessCont MessHead (Global.!EmptyLine) MessBody  /* get the message parts */
MessHead = MessHead||Global.!CRLF  /* make sure the header ends with a CRLF */

if (\MessageContents(MessHead,'Global.!SendHead.')) then  /* if we cannot get the header contents sorted out */
do
 return 0  /* quit */
end

Resending = (Resending == 1)  /* 1 is true */
MessFrom = GetHeaderEntry('FROM','Global.!SendHead.',1)  /* get all the From: lines */

if (pos(Global.!CRLF,MessFrom) > 0) then  /* if we have more than one entry */
do
 call AddError 'Illegal "From:" line in header'  /* report */
 return 0  /* and quit */
end

MessFrom = AddressFormat(MessFrom,0,0)  /* format to get bare addresses */

if (MessFrom == '') then  /* if we have no from @ddress */
do
 call AddError 'Missing "From:" address in header'  /* report */
 return 0  /* and quit */
end

MessSender = GetHeaderEntry('SENDER','Global.!SendHead.',1)  /* get all the Sender: lines */

if (pos(Global.!CRLF,MessSender) > 0) then  /* if we have more than one entry */
do
 call AddError 'Illegal "Sender:" line in header'  /* report */
 return 0  /* and quit */
end

MessSender = AddressFormat(MessSender,0,0)  /* format to get bare address */

if (MessSender >< '') then  /* if we have an explicit sender line */
do
 MailFrom = MessSender  /* use it as the return @ddress */
end
else  /* if there is no explicit sender line */
do
 MailFrom = Global.!Settings.Address  /* use the local @ddress */
end

MessTo = GetHeaderEntry('TO','Global.!SendHead.',1)  /* get all the To: lines */

if (MessTo == '') then  /* if we have no To: recipient */
do
 call AddError 'Missing "To:" line in header'  /* report */
 return 0  /* and quit */
end

if (pos(Global.!CRLF,MessTo) > 0) then  /* if we have more than one entry */
do
 call AddError 'Illegal "To:" line in header'  /* report */
 return 0  /* and quit */
end

MessCc = GetHeaderEntry('CC','Global.!SendHead.',1)  /* get all the Cc: lines */

if (MessCc >< '') then  /* if we have one or more CC entries */
do

 if (pos(Global.!CRLF,MessCc) > 0) then  /* if we have more than one entry */
 do
  call AddError 'Illegal "Cc:" line in header'  /* report */
  return 0  /* and quit */
 end

end

MessAcc = GetHeaderEntry('ACC','Global.!SendHead.',1)  /* get all the Acc: lines */

if (MessAcc >< '') then  /* if we have one or more ACC entries */
do

 MessHead = RemoveFromHeader(MessHead,'ACC')  /* remove the ACC entry from the message header */

 if (pos(Global.!CRLF,MessAcc) > 0) then  /* if we have more than one entry */
 do
  call AddError 'Illegal "Acc:" line in header'  /* report */
  return 0  /* and quit */
 end

end

MessBcc = GetHeaderEntry('BCC','Global.!SendHead.',1)  /* get all the Bcc: lines */

if (MessBcc >< '') then  /* if we have one or more BCC entries */
do

 MessHead = RemoveFromHeader(MessHead,'BCC')  /* remove the BCC entry from the message header */

 if (pos(Global.!CRLF,MessBcc) > 0) then  /* if we have more than one entry */
 do
  call AddError 'Illegal "Bcc:" line in header'  /* report */
  return 0  /* and quit */
 end

end

MessDate = GetHeaderEntry('DATE','Global.!SendHead.',1)  /* get all the Date: lines */

if (pos(Global.!CRLF,MessDate) > 0) then  /* if we have more than one entry */
do
 call AddError 'Illegal "Date:" line in header'  /* report */
 return 0  /* and quit */
end

MessSubject = GetHeaderEntry('SUBJECT','Global.!SendHead.',1)  /* get all the Subject: lines */

if (pos(Global.!CRLF,MessSubject) > 0) then  /* if we have more than one entry */
do
 call AddError 'Illegal "Subject:" line in header'  /* report */
 return 0  /* and quit */
end

MessID = GetHeaderEntry('MESSAGE-ID','Global.!SendHead.',1)  /* get all the Message-ID: lines */

if (pos(Global.!CRLF,MessID) > 0) then  /* if we have more than one entry */
do
 call AddError 'Illegal "Message-ID:" line in header'  /* report */
 return 0  /* and quit */
end

parse var MessID . '@' HostName '>'  /* get the host name */

if (Resending) then  /* if the message is being resent */
do
 MessHead = MessHead||'Resent-Message-Id: <'||date('S')||'.'||translate(left(time('L'),11),'.',':')||'.'||right(random(999),3,'0')||'@'||HostName||'>'||Global.!CRLF||,  /* add this line */
                      'Resent-Date: '||DateTimeRFC(1)||Global.!CRLF||,  /* add this line (contact time servers if necessary) */
                      'Resent-From: <'||Global.!Settings.Address||'>'||Global.!CRLF  /* add this line */
end

if (MessFrom >< Global.!Settings.Address) then  /* if the From sender is not the local user */
do

 if ((\Resending) & (MessSender == '')) then  /* if we are not resending and there is no previous Sender entry */
 do
  MessHead = MessHead||'Sender: "'||Global.!Settings.Name||'" <'||Global.!Settings.Address||'>'||Global.!CRLF  /* add a "Sender:" line */
 end

 SenderBit = ' ('||MailFrom||')'  /* we will insert this later in screen and log messages */

end
else  /* if they are the same */
do
 SenderBit = ''  /* no additional sender text required */
end

/**************************************************************************/
/* Expand the lists of recipients                                         */
/**************************************************************************/

Global.!NamesList = ''  /* start with nothing */
Global.!NoNames = 0  /* start a counter at nothing */

MessTo = strip(ExpandRecipients(AddressFormat(MessTo,0,0,0),0))  /* format to get bare addresses and expand the list of To: recipients */
MessCc = strip(ExpandRecipients(AddressFormat(MessCc,0,0,0),0))  /* format to get bare addresses and expand the list of Cc: recipients */
MessAcc = strip(ExpandRecipients(AddressFormat(MessAcc,0,0,1),1))  /* format to get bare addresses --creating a names list-- and expand the list of Acc: recipients */
MessBcc = strip(ExpandRecipients(AddressFormat(MessBcc,0,0,0),0))  /* format to get bare addresses and expand the list of Bcc: recipients */
Recipients = MessTo||' '||MessCc||' '||MessAcc||' '||MessBcc  /* all our recipients */

if ((Global.!NamesList >< '') | (Global.!NoNames > 0)) then  /* if we have a list of recipient names to include in the header, or any unnamed recipients */
do

 if (Global.!NoNames > 0) then  /* if we have unnamed recipients */
 do

  if (Global.!NoNames == 1) then  /* if we have just one unnamed recipient */
  do
   Plural = ''  /* no plural "s" needed */
  end
  else  /* if we have more than one */
  do
   Plural = 's'  /* we will add a plural "s" below */
  end

  if (Global.!NamesList >< '') then  /* if we have a list of recipient names */
  do
   Global.!NamesList = Global.!NamesList||','||Global.!CRLF||' and '||Global.!NoNames||' unnamed recipient'||Plural  /* add this */
  end
  else  /* if we have no list of names */
  do
   Global.!NamesList = Global.!CRLF||' '||Global.!NoNames||' unnamed recipient'||Plural  /* add this */
  end

 end

 Global.!NamesList = translate(Global.!NamesList,Global.!FilterOut,xrange('80'x,'FF'x))  /* convert to PC819 (ISO-8859-1) */
 Global.!NamesList = HeaderEncode(Global.!NamesList)  /* encode high-bit content */
 Global.!NamesList = 'Comments: ACC (Addressless Carbon Copy):'||Global.!NamesList||'.'||Global.!CRLF  /* add the keyword and a comments line */

end

/**************************************************************************/
/* Write the sendable file                                                */
/**************************************************************************/

if (\PutFileContents(SMTPFile,MessHead||Global.!NamesList||Global.!EmptyLine||MessBody)) then  /* if we cannot write the SMTP data file */
do
 return 0  /* quit */
end

/**************************************************************************/
/* Connect to the server if necessary                                     */
/**************************************************************************/

if (Global.!SMTPConnect >< '') then  /* if we already have an SMTP connection */
do
 Socket = Global.!SMTPConnect  /* use it */
end
else  /* if we do not have an SMTP connection yet */
do

 parse var Global.!Settings.SMTPServer Global.!Settings.SMTPServer ':' PortNumber  /* look for a port number */

 if (PortNumber == '') then  /* if there is none */
 do
  PortNumber = 25  /* use this (SMTP) */
 end

 Socket = ServerConnect(Global.!Settings.SMTPServer,PortNumber,Global.!Settings.SMTPAttempts)  /* get the socket number for a server connection through the specified or default port */

 if (Socket == '') then  /* if we have no socket */
 do
  Global.!Break = 1  /* break off processing of further messages */
  return 0  /* quit without success */
 end

 Global.Socket.SocketBuffer = ''  /* start with an empty socket buffer */

 if (\SocketAnswer(Socket,220)) then  /* if we don't get a 220 return code */
 do
  call AddError 'No response from '||Global.!Settings.SMTPServer  /* report */
  call SocketClose Socket  /* close the socket */
  return 0  /* return with no success */
 end

 if ((Global.!Settings.SMTPUser >< '') & (Global.!Settings.SMTPPassword >< '')) then  /* if we have a user/password combo */
 do
  HelloString = 'EHLO'  /* use this to announce our presence */
 end
 else  /* if we do not have a user/password combo */
 do
  HelloString = 'HELO'  /* use this to announce our presence */
 end
  
 if (\SocketSendLine(Socket,HelloString||' '||HostName,250)) then  /* if don't get 250 in return when we knock */
 do
  call AddError HostName||' not accepted'  /* report */
  call ServerDisconnect Socket,Global.!Settings.SMTPServer  /* disconnect */
  Global.!Break = 1  /* break off processing of further messages */
  return 0  /* return with no success */
 end

 if ((Global.!Settings.SMTPUser >< '') & (Global.!Settings.SMTPPassword >< '')) then  /* if we have a user/password combo */
 do
  
  AuthString = EncodeB64(d2c(0)||Global.!Settings.SMTPUser||d2c(0)||Global.!Settings.SMTPPassword)  /* B64-encode the authentication string */

  if (\SocketSendLine(Socket,'AUTH PLAIN '||AuthString,235)) then  /* if don't get 235 in return when we send this line */
  do
   call AddError 'Authentication string not accepted'  /* report */
   call ServerDisconnect Socket,Global.!Settings.SMTPServer  /* disconnect */
   Global.!Break = 1  /* break off processing of further messages */
   return 0  /* return with no success */
  end

 end

 Global.!SMTPConnect = Socket  /* we have an SMTP connection */

end

/**************************************************************************/
/* Send the message(s)                                                    */
/**************************************************************************/

if (\SocketSendLine(Socket,'MAIL FROM:<'||MailFrom||'>',250)) then  /* if we don't get 250 in return when we send this line */
do
 call AddError 'MAIL FROM:<'||MailFrom||'> not accepted by server'  /* report */
 call ServerDisconnect Socket,Global.!Settings.SMTPServer  /* disconnect */
 Global.!Break = 1  /* break off processing of further messages */
 return 0  /* quit without success */
end

if (Resending) then  /* if we are resending */
do
 LogText = 'Res'  /* start with this */
 Indent = 23  /* the indent to use */
end
else  /* if not */
do
 LogText = 'S'  /* start with this */
 Indent = 21  /* the indent to use */
end

LogText = LogText||'ending message from '||MailFrom||SenderBit||Global.!CRLF||,  /* start a log text with this */
          copies(' ',(Indent - 3))||'to '  /* and this */

RecipientCount = 0  /* we have no successful recipients yet */

do while (Recipients >< '')  /* as long as we have recipients left in the "normal" list */

 parse var Recipients Recipient Recipients  /* get the next recipient from the list */

 if (RecipientCount > 0) then  /* if we are not on the first line anymore */
 do
  LogText = LogText||Global.!CRLF||copies(' ',Indent)  /* add a new line and an indent to the log text */
 end

 if (SocketSendLine(Socket,'RCPT TO:<'||Recipient||'>',250 251)) then  /* if we get 250 or 251 in return when we send this line */
 do
  RecipientCount = RecipientCount + 1  /* up the counter */
  LogText = LogText||Recipient  /* add the recipient to the log text */
 end
 else  /* if we failed to enter the recipient */
 do
  call LogAction LogText,1  /* report, quietly */
  call AddError 'RCPT TO:<'||Recipient||'> not accepted by server'  /* report */
  call ServerDisconnect Socket,Global.!Settings.SMTPServer  /* disconnect */
  return 0  /* quit without success */
 end

end

call LogAction LogText,1  /* report, quietly */

if (\SocketSendLine(Socket,'DATA',354)) then  /* if we don't get 354 in return when we send this line */
do
 return 0  /* return with an error */
end

ByteCount = stream(SMTPFile,'C','QUERY SIZE')  /* get the length of the data block to send */

if \(datatype(ByteCount,'W') & (ByteCount > 42)) then  /* if we have an insane value */
do
 call AddError 'Cannot determine SMTP file size'  /* report */
 return 0  /* return with an error */
end

if (\FileOpen(SMTPFile,'READ')) then  /* if we cannot open the SMTP file for reading */
do
 return 0  /* return with an error */
end

OutBytes = 0  /* we have sent nothing yet */
StatusOK = 1  /* all is well */
ShowStats = Global.!Settings.ShowProgress  /* if we want stats, set a local flag */

if (ShowStats) then  /* if we want stats */
do
 signal on syntax name SMTPTimeBug  /* we go here if the REXX time() bug strikes */
 call time 'R'  /* reset the timer */
 signal on syntax name Syntax  /* use the normal syntax error routine */
 call syscurstate 'OFF'  /* switch off the cursor */
 call ProgressBar 'Bytes sent = ',OutBytes,ByteCount,0  /* start a progress bar */
 TransTime = 0  /* no transmission time yet */
 ElapsedTime = 0  /* no elapsed time yet */
end

SMTPTimeBugDone:  /* we return here after the REXX time() bug strikes */

do while ((lines(SMTPFile)) & (StatusOK))  /* as long as we have data to send and all is well */

 DataLine = linein(SMTPFile)  /* get the next line */

 if (DataLine >< '') then  /* if it is not an empty line */
 do

  if (left(DataLine,1) == '.') then  /* if the line starts with a period */
  do
   DataLine = '.'||DataLine  /* stick an extra period on the front */
  end

 end

 StatusOK = SocketSendLine(Socket,DataLine)  /* if we can send this to the socket, all is well */
 OutBytes = OutBytes + length(DataLine) + 2  /* up the line bytes count (plus the terminating CRLF) */

 if ((StatusOK) & (ShowStats)) then  /* if we want stats and all is still well */
 do

  signal on syntax name SMTPTimeBug  /* we go here if the REXX time() bug strikes */
  ElapsedTime = ElapsedTime + time('R')  /* update the elapsed time and reset the timer */
  signal on syntax name Syntax  /* use the normal syntax error routine */

  if (ElapsedTime > 0.1) then  /* if the elapsed time is sufficient */
  do
   TransTime = TransTime + ElapsedTime  /* update the transmission time */
   ElapsedTime = 0  /* start all over again */
   call ProgressBar 'Bytes sent = ',OutBytes,ByteCount,TransTime  /* show the status */
  end

 end

end

call FileClose SMTPFile  /* close the SMTP data file */
call sysfiledelete SMTPFile  /* zap the SMTP data file */

if (ShowStats) then  /* if we want stats */
do
 call ProgressBar  /* erase the status display */
 call syscurstate 'ON'  /* switch the cursor back on */
end

if (\StatusOK) then  /* if we had an error */
do
 call AddError 'Error sending message to <'||Recipient||'>'  /* report */
 call ServerDisconnect Socket,Global.!Settings.SMTPServer  /* disconnect */
 return 0  /* quit without success */
end

if (\SocketSendLine(Socket,'.',250)) then  /* if we don't get 250 in return when we send this line */
do
 return 0  /* quit */
end

if (RecipientCount == 1) then  /* if we have exactly 1 recipient */
do
 Plural = ''  /* we need no plural 's' */
end
else  /* if we have 0 or multiple recipients */
do
 Plural = 's'  /* we need a plural 's' */
end

call LogAction 'Message sent to '||RecipientCount||' recipient'||Plural||'; '||OutBytes||' bytes; ID = '||MessID  /* report */
call LogMail '==>',MessFrom||SenderBit,,MessTo,MessCc,MessAcc,MessBcc,MessSubject,MessDate,MessID,OutBytes  /* log the message */

if (Global.!FilesToDo == 0) then  /* if we have done the last file */
do
 call ServerDisconnect Socket,Global.!Settings.SMTPServer  /* disconnect */
 Global.!SMTPConnect = ''  /* we have no SMTP connection any more */
end

return 1  /* end of SMTPSendMessage */

/**************************************************************************/
SMTPTimeBug:  /* handles the Classic REXX time() bug */
/**************************************************************************/

signal on syntax name Syntax  /* use the normal syntax error routine */

ShowStats = 0  /* no more statistics for this one */
call ProgressBar  /* erase the status display */
call syscurstate 'ON'  /* switch the cursor back on */
call LogAction 'REXX Time() error; do not synchronize the clock while sending mail. Resuming.'  /* report */

signal SMTPTimeBugDone  /* go here */

/**************************************************************************/
HaltSMTPSendMessage:  /* handles halt locally */
/**************************************************************************/

if (Socket >< '') then  /* if we have a socket */
do
 call ServerDisconnect Socket,Global.!Settings.SMTPServer  /* disconnect */
end

Global.!SMTPConnect = ''  /* we have no SMTP connection any more */

return 0  /* no success */

/**************************************************************************/
SocketAnswer: procedure expose Global.  /* see if we got the right answer */
/**************************************************************************/

signal on halt name HaltSocketAnswer  /* handles halt locally */

parse arg Socket,ReplyCodes  /* get the arguments */

do until ((Answer >< '') | (\Global.!SocketOK))  /* go on until we get a result or an error */

 Answer = SocketGetLine(Socket)  /* get the next line from the socket (buffer) */

 if ((Answer >< '') & (Global.!SocketOK)) then  /* if we have something and there is no error */
 do

  if (substr(Answer,4,1) == '-') then  /* if the fourth character is a hyphen, we have part of a multiline reply */
  do
   Answer = ''  /* dump this line */
  end
  else  /* if we have the final line of a multiline reply, or a single-line reply */
  do

   if (ReplyCodes == '') then  /* if we have no reply codes to match */
   do
    return Answer  /* return the full answer */
   end

   if (wordpos(word(Answer,1),ReplyCodes) > 0) then  /* if the first word of the answer is in of the reply codes we want */
   do
    return 1  /* return success */
   end
   else  /* if the first word is not in the list of reply codes */
   do
    call AddError 'Error from server: '||subword(Answer,2)  /* report */
    return 0  /* no success */
   end

  end

 end
 else  /* if we get no answer, or an error */
 do
  return 0  /* no success */
 end

end

return 1  /* end of SocketAnswer */

/**************************************************************************/
HaltSocketAnswer:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
SocketClose: procedure expose Global.  /* closes the IP socket */
/**************************************************************************/

signal on halt name HaltSocketClose  /* handles halt locally */

parse arg Socket  /* get the argument */

if (socksoclose(Socket) >< 0) then  /* close the socket */
do
 call AddError 'Error closing socket'  /* report */
end

return 1  /* end of SocketClose */

/**************************************************************************/
HaltSocketClose:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
SocketGetLine: procedure expose Global.  /* get a line from a socket */
/**************************************************************************/

signal on halt name HaltSocketGetLine  /* handles halt locally */

parse arg Socket  /* get the argument */

Global.!SocketOK = 1  /* all is well for now */

do while (pos(Global.!CRLF,Global.Socket.SocketBuffer) == 0)  /* as long as we have no CRLF in the buffer */

 if (sockrecv(Socket,'SocketData',8096) > 0) then  /* if we can get data from the socket */
 do
  Global.Socket.SocketBuffer = Global.Socket.SocketBuffer||SocketData  /* add the results to the buffer */
 end
 else  /* if we cannot get anything from the socket */
 do
  call AddError 'Connection closed'  /* report */
  Global.!SocketOK = 0  /* we have an error */
 end

end

parse var Global.Socket.SocketBuffer NextLine (Global.!CRLF) Global.Socket.SocketBuffer  /* take the bit before the CRLF from the buffer */

if (Global.!Settings.SocketMonitor) then  /* if we want to monitor socket data */
do
 call lineout 'STDOUT:',DateTimeSys()||' <= '||NextLine  /* send this info to STDOUT */
end

return NextLine  /* end of SocketGetLine */

/**************************************************************************/
HaltSocketGetLine:  /* handles halt locally */
/**************************************************************************/

return ''  /* no success */

/**************************************************************************/
SocketSendLine: procedure expose Global.  /* send a line to a socket */
/**************************************************************************/

signal on halt name HaltSocketSendLine  /* handles halt locally */

parse arg Socket,DataToSend,ReplyCodes  /* get the argument */

DataToSend = DataToSend||Global.!CRLF  /* add a CRLF to the data to be sent */
DataLength = length(DataToSend)  /* get the length of the data to be sent */

do while (DataLength > 0)  /* as long as we have data left to send */

 SentLength = socksend(Socket,DataToSend)  /* see how many bytes we can send */

 if (SentLength <= 0) then  /* if we sent no data */
 do
  call AddError 'Connection closed by server'  /* report */
  return 0  /* quit; all is not well */
 end

 parse var DataToSend DataSent +(SentLength) DataToSend  /* split the data into the relevant parts */
 DataLength = DataLength - SentLength  /* how much more to go? */

 if (Global.!Settings.SocketMonitor) then  /* if we are monitoring the socket traffic */
 do
  call charout 'STDOUT:',DateTimeSys()||' => '||DataSent  /* send this info to STDOUT */
 end

end

if (ReplyCodes == '') then  /* if we have no reply codes to match */
do
 return 1  /* all is well */
end

if (ReplyCodes == '?') then  /* if we want to see the answer itself */
do
 ReplyCodes = ''  /* go on without any reply codes */
end

return SocketAnswer(Socket,ReplyCodes)  /* if we get the correct answer, all is well */

/**************************************************************************/
HaltSocketSendLine:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
SoundSignal: procedure expose Global.  /* sound a signal (or not, as the case may be) */
/**************************************************************************/

signal on halt name HaltSoundSignal  /* handles halt locally */

parse arg SignalString  /* get the argument */

if (Global.!Settings.Silent) then  /* if we want to suppress all beeps */
do
 return 0  /* return with nothing */
end

if (Global.!SilentOnce) then  /* if we want to suppress this beep */
do
 Global.!SilentOnce = 0  /* reset the flag */
 return 0  /* return with nothing */
end

if (SignalString >< '') then  /* if there is a signal string */
do

 if (word(SignalString,1) == 'BEEP') then  /* if it starts with this, it is a beep sequence */
 do

  SignalString = subword(SignalString,2)  /* use the rest */

  do while (SignalString >< '')  /* as long as we have something left */
   parse var SignalString NextSignal SignalString  /* get the next beep bit */
   interpret 'call beep '||NextSignal  /* and do it */
  end

 end
 else  /* if it is not a beep string, it must be an external command */
 do
  call RunCommand SignalString  /* try to run the command */
 end

end

return 1  /* end of SoundSignal */

/**************************************************************************/
HaltSoundSignal:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
SystemMessage: procedure expose Global.  /* sends a system message to the current user */
/**************************************************************************/

signal on halt name HaltSystemMessage  /* handles halt locally */

parse arg Subject,MessText  /* get the arguments */

MessCont = ''  /* nothing yet */

if (Global.!Settings.Name == '') then  /* if we do not have a user name */
do
 Global.!Settings.Name = 'RexxMail user'  /* use this */
end

if (Global.!Settings.Address == '') then  /* if we do not have a user address */
do
 Global.!Settings.Address = 'rexxmail-user@localhost'  /* use this */
end

MessCont = 'Date: '||DateTimeRFC(0)||Global.!CRLF||,  /* start with a date line, but do not contact a time server */
           'From: "RexxMail system messages" <rexxmail-messages@localhost>'||Global.!CRLF||,  /* add a sender line */
           'To: "'||Global.!Settings.Name||'" <'||Global.!Settings.Address||'>'||Global.!CRLF||,  /* add a recipient line */
           'Subject: '||Subject||Global.!EmptyLine||,  /* add a subject line and an empty line */
           MessText  /* add the message text */
MessFile = TempFileName(,,'SYSTEM')  /* get a unique file name in the system temp dir */

if (MessFile == '') then  /* if we cannot get a temp file */
do
 return 0  /* quit with no success */
end

if (\PutFileContents(MessFile,MessCont)) then  /* if we cannot write the file contents */
do
 return 0  /* quit with no success */
end

if (RegisterMessage(MessFile)) then  /* if the message gets registered */
do
 return SoundSignal(Global.!Settings.SignalSystem)  /* if a signal was sounded, return with the result */
end

/**************************************************************************/
HaltSystemMessage:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
TempFileName: procedure expose Global.  /* returns a unique file name */
/**************************************************************************/

signal on halt name HaltTempFileName  /* handles halt locally */

parse arg Location,Name,Extension  /* get the arguments */

if (Location == '') then  /* if we have no location */
do
 Location = Global.!TempDir  /* use the temp dir */
end
else  /* if we have something */
do
 Location = strip(Location,'T','\')  /* remove any trailing backslash */
end

if (Name >< '') then  /* if we have a name */
do
 Name = CheckCommLine(Name)||'_????'  /* use this */
end
else  /* if we have no name */
do
 Name = 'RXML????'  /* use this */
end

if (Extension >< '') then  /* if we have an extension */
do
 Extension = '.'||CheckCommLine(Extension)  /* it needs a leading dot and awkward characters seen to */
end

TempFile = systempfilename(Location||'\'||Name||Extension)  /* this is the file name */

if (TempFile == '') then  /* if we cannot get a temp file */
do
 call AddError 'Cannot open temporary file : "'||Location||'\'||Name||Extension||'"'  /* report */
end

return TempFile  /* end of TempFileName */

/**************************************************************************/
HaltTempFileName:  /* handles halt locally */
/**************************************************************************/

return ''  /* no success */

/**************************************************************************/
Toolbar: procedure expose Global.  /* creates/deletes/opens a RexxMail toolbar */
/**************************************************************************/

signal on halt name HaltToolbar  /* handles halt locally */

parse arg Action,ToolbarName,Quiet  /* get the arguments */

Quiet = (Quiet == 1)  /* 1 = TRUE */
MessBar = 0  /* no flag yet */

if (ToolbarName >< '') then  /* if we have a name */
do
 ToolbarName = translate(ToolbarName,' ',d2c(9))  /* convert TABs to spaces */
 ObjectID = '<REXXMAIL_TOOLBAR_'||Global.!User||'_'||translate(translate(ToolbarName,'_',' '))||'>'  /* add the upper case name, with underscores instead of spaces, to the object ID */
 OldObjectID = '<REXXMAIL_TOOLBAR_'||translate(translate(ToolbarName,'_',' '))||'>'  /* add the upper case name, with underscores instead of spaces, to the object ID */
 MessBar = (ObjectID == '<REXXMAIL_TOOLBAR_'||Global.!User||'_MESSAGE_BUTTONS>')  /* set a flag */
end
else  /* if not */
do
 ToolbarName = 'RexxMail Toolbar'  /* use this */
 ObjectID = '<REXXMAIL_TOOLBAR_'||Global.!User||'>'  /* use this */
 OldObjectID = '<REXXMAIL_TOOLBAR>'  /* use this */
end

if (syssetobjectdata(OldObjectID,'OBJECTID='||ObjectID)) then  /* if we can update the object ID -- do not use SetObjectData()! */
do
 call LogAction 'Updated object ID of RexxMail toolbar "'||ToolbarName||'" to : '||ObjectID,1  /* report, quietly */
end

Action = translate(Action)  /* make the action upper case */

select  /* do one of the following */

 when (Action == 'CREATE') then  /* if we have this */
 do

  if (MessBar) then  /* if we want a mail bar */
  do
   CreateOption = 'REPLACE'  /* update if it already exists */
   FPObjects = Global.!MessBar.1.!Message  /* start with the first button */
  end
  else  /* if not */
  do
   CreateOption = 'FAIL'  /* fail if it already exists */
   FPObjects = '<REXXMAIL_COLLECT_SENDREADY>,'||,  /* include this program object button */
               '<REXXMAIL_COLLECT_POP3INTERACTIVE>,'||,  /* include this program object button */
               '<REXXMAIL_FORWARD>,'||,  /* include this program object button */
               '<REXXMAIL_REPLY>,'||,  /* include this program object button */
               '<REXXMAIL_ADDRESSIMPORT>,'||,  /* include this program object button */
               '<REXXMAIL_UTILITY>,'||,  /* include this program object button */
               '<REXXMAIL_REFERENCE>,'||,  /* include this program object button */
               '<REXXMAIL_TUTORIAL>,'||,  /* include this program object button */
               '<REXXMAIL_CLOSE>'  /* include this object program button */
  end

  Settings = 'OBJECTID='||ObjectID||';'||,  /* the object ID */
             'FPOBJECTS='||FPObjects||';'||,  /* include these objects */
             'LPACTIONSTYLE=OFF;'||,  /* no system action buttons */
             'LPHIDECTLS=YES'  /* hide the frame controls */

  if (syscreateobject('WPLaunchPad',ToolbarName,Global.!MainDir,Settings,CreateOption)) then  /* if we created the toolbar object in the Main folder */
  do

   if (MessBar) then  /* if we want a mail bar */
   do

    do Index = 2 to Global.!MessBar.0  /* for each mail bar item except the first one (which we used to create the button bar) */
     call SetObjectData ObjectID,'FPOBJECTS='||Global.!MessBar.Index.!Message  /* add the next button object (this is a kludge to avoid overextending the FPOBJECTS directive) */
    end

    do Index = 1 to Global.!MessBar.0  /* for each mail bar item */

     if (Global.!MessBar.Index.!AddDirTo >< '') then  /* if we have an additional To addresses dir spec */
     do
      AddDirTo = Global.!MessBar.Index.!AttDir||'\'||Global.!MessBar.Index.!AddDirTo  /* the complete additional To addresses folder */
     end
     else  /* if not */
     do
      AddDirTo = ''  /* nothing */
     end

     if (Global.!MessBar.Index.!AddDirCc >< '') then  /* if we have an additional Cc addresses dir spec */
     do
      AddDirCc = Global.!MessBar.Index.!AttDir||'\'||Global.!MessBar.Index.!AddDirCc  /* the complete additional Cc addresses folder */
     end
     else  /* if not */
     do
      AddDirCc = ''  /* nothing */
     end

     if (Global.!MessBar.Index.!AddDirAcc >< '') then  /* if we have an additional Acc addresses dir spec */
     do
      AddDirAcc = Global.!MessBar.Index.!AttDir||'\'||Global.!MessBar.Index.!AddDirAcc  /* the complete additional Acc addresses folder */
     end
     else  /* if not */
     do
      AddDirAcc = ''  /* nothing */
     end

     if (Global.!MessBar.Index.!AddDirBcc >< '') then  /* if we have an additional Bcc addresses dir spec */
     do
      AddDirBcc = Global.!MessBar.Index.!AttDir||'\'||Global.!MessBar.Index.!AddDirBcc  /* the complete additional Bcc addresses folder */
     end
     else  /* if not */
     do
      AddDirBcc = ''  /* nothing */
     end

     call SetObjectData ObjectID,'DRAWEROBJECTS='||Index||','||Global.!MessBar.Index.!AttDir||','||AddDirTo||','||AddDirCc||','||AddDirAcc||','||AddDirBcc  /* add the drawer objects */
    end

    AddSettings = ''  /* use the default settings to start with */
    CopySettings = Global.!Settings.MessageToolbar  /* copy the toolbar settings */

    do while (CopySettings >< '')  /* run through the settings */

     parse upper var CopySettings NextSetting CopySettings  /* get the next one in upper case */

     if (NextSetting >< 'YES') then  /* unless it is simply the default command marker */
     do
      AddSettings = AddSettings||'LP'||NextSetting||'=YES;'  /* add it to what we have */
     end

    end

   end
   else  /* if it is not a message toolbar */
   do
    AddSettings = 'LPDRAWERTEXT=YES;'||,  /* text in drawers */
                  'LPCLOSEDRAWER=NO;'||,  /* do not close drawers after use */
                  'LPFLOAT=YES;'||,  /* float to top */
                  'LPHIDECTLS=YES;'||,  /* hide the window controls */
                  'LPSMALLICONS=YES;'||,  /* use small icons */
                  'LPTEXT=YES;'||,  /* show text on buttons */
                  'LPVERTICAL=YES'  /* show vertically */
   end

   call SetObjectData ObjectID,AddSettings  /* try to adjust some settings */
   Action = 'Created'  /* use this later on */

  end
  else  /* if we failed */
  do
   Action = 'Cannot create'  /* use this later on */
  end

 end

 when (Action == 'DELETE') then  /* if we have this */
 do

  if (sysdestroyobject(ObjectID)) then  /* if we deleted the toolbar */
  do
   Action = 'Deleted'  /* use this later on */
  end
  else  /* if we failed */
  do

   if (Quiet) then  /* if we want no alarms */
   do
    Action = ''  /* no report necessary */
   end
   else  /* if we want alarms */
   do
    Action = 'Cannot delete'  /* use this later on */
   end

  end

 end

 when (Action == 'OPEN') then  /* if we have this */
 do

  if (sysopenobject(ObjectID,121,0)) then  /* if we opened the toolbar */
  do
   Action = 'Opened'  /* use this later on */
  end
  else  /* if we failed */
  do

   if (Quiet) then  /* if we want no alarms */
   do
    Action = ''  /* no report necessary */
   end
   else  /* if we want alarms */
   do
    Action = 'Cannot open'  /* use this later on */
   end

  end

 end

 otherwise  /* this should not occur */
 do
  Action = ''  /* nothing to do */
 end

end

if (Action >< '') then  /* if we want a report */
do

 if (word(Action,1) == 'Cannot') then  /* if the action failed */
 do
  call AddError Action||' toolbar: "'||ToolbarName||'"'  /* report */
  return 0  /* return no go */
 end

 call LogAction Action||' toolbar: "'||ToolbarName||'"',1  /* report, quietly */

end

return 1  /* end of Toolbar */

/**************************************************************************/
HaltToolbar:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
TrimLogFile: procedure expose Global.  /* rewrites a log file if necessary */
/**************************************************************************/

signal on halt name HaltTrimLogFile  /* handles halt locally */

parse arg LogFile,KeepLines  /* get the arguments */

if (KeepLines == '') then  /* if we have an empty value here, we do not have to trim this log file (there is no size limit) */
do
 return 1  /* quit */
end

if (KeepLines == 0) then  /* if we have a zero value here, we do not have to trim this log file (there is no new log writing) */
do
 return 1  /* quit */
end

if (\FileOpen(LogFile)) then  /* if we cannot open the log file */
do
 return 0  /* quit */
end

Counter = 0  /* start a counter */
LogLines. = ''  /* start with nothing */

do while (lines(LogFile))  /* as long as we have log lines left */
 Counter = Counter + 1  /* up the counter */
 LogLines.Counter = linein(LogFile)  /* store the next line */
end

call FileClose LogFile  /* close the file */
StartLine = max(1,(Counter - KeepLines + 1))  /* if the first line we want to keep is less than 1, start at the first line */

if (StartLine > 1) then  /* if we want a new file */
do

 call sysfiledelete LogFile  /* get rid of the old file */

 if (\FileOpen(LogFile)) then  /* if we cannot open a new log file */
 do
  return 0  /* quit */
 end

 do Index = StartLine to Counter  /* take each of the lines we want */
  call lineout LogFile,LogLines.Index  /* write it to the log file */
 end

 call FileClose LogFile  /* close the file */

end

return 1  /* end of TrimLogFile */

/**************************************************************************/
HaltTrimLogFile:  /* handles halt locally */
/**************************************************************************/

call FileClose LogFile  /* close the file */

return 0  /* no success */

/**************************************************************************/
UnpackMessage: procedure expose Global.  /* unpacks received/sent mail files */
/**************************************************************************/

signal on halt name HaltUnpackMessage  /* handles halt locally */

parse arg MessFile,AttDir,Process  /* get the arguments */

Process = (Process == 1)  /* 1 = True */
MessCont = GetFileContents(MessFile)  /* get the message file contents */

if (\MessageContents(MessCont,'Global.!MessHead.','Global.!MessBody')) then  /* if we cannot get the message contents sorted out */
do
 return 0  /* quit with no attachments */
end

if (Global.!MessBody == '') then  /* if there is no message body */
do
 Global.!MessBody = '[RexxMail found no message text]'  /* use this */
 return 0  /* quit with no attachments */
end

/**************************************************************************/
/* If we got this far, we have body content of some sort                  */
/**************************************************************************/

if (\HeaderContains('CONTENT-TYPE',,'Global.!MessHead.')) then  /* if we do not have a Content-Type header line */
do
 Global.!MessBody = WordWrap(Global.!MessBody,Global.!Settings.BodyLineLength,1)  /* word-wrap the body if necessary, breaks are O.K. */
 return 0  /* quit with no attachments */
end

/**************************************************************************/
/* If we got this far, we have MIME body content of some sort             */
/**************************************************************************/

call directory Global.!TempDir  /* change to the temp dir -- this is a kludge to make munpack.exe work and to make sure we can remove the attachments dir later */
CTLine = translate(GetHeaderEntry('CONTENT-TYPE','Global.!MessHead.'))  /* look for a content type line in the header */
CTELine = translate(GetHeaderEntry('CONTENT-TRANSFER-ENCODING','Global.!MessHead.'))  /* look for a content transfer encoding line */
HTMLKeep = ((\Process) | (Global.!Settings.HTMLKeep))  /* if we are not processing for viewing, or if we want to keep the HTML in any case, set a flag */

if ((left(CTLine,5) == 'TEXT/') | (CTLine == '')) then  /* if we have a single-part text message, or a rogue one */
do

 Attachments = 0  /* no attachments yet */

 select  /* do one of the following */

  when (pos('BASE64',CTELine) > 0) then  /* if we have B64-encoded text */
  do
   Global.!MessBody = DecodeB64(Global.!MessBody)  /* decode the body text */
  end

  when (pos('QUOTED-PRINTABLE',CTELine) > 0) then  /* if we have quoted-printable text */
  do

   if (MIMEDecode(MessCont,AttDir)) then  /* if we can decode the message contents */
   do
    Global.!MessBody = GetFileContents(AttDir||'\part1')  /* this must be the message text */
    call sysfiledelete AttDir||'\part1'  /* remove it */
   end

  end

  otherwise  /* if none of the above */
  do
   nop  /* zilch */
  end

 end

 select  /* do one of the following */

  when (pos('UTF-8',CTLine) > 0) then  /* if it uses the UTF-8 encoding */
  do
   Global.!MessBody = DecodeUTF8(Global.!MessBody)  /* convert the message body content from UTF-8 to ISO-8859-1 */
  end

  when (pos('WINDOWS-',CTLine) > 0) then  /* if it uses a Windows character set */
  do
   Global.!MessBody = translate(Global.!MessBody,Global.!PC1004_PC819,xrange('80'x,'FF'x))  /* convert the message body content from PC1004 to PC819 (ISO-8859-1) */
  end

  otherwise  /* if neither of the above */
  do
   nop  /* assume it's ISO-8859-1, for RexxMail cannot handle anything else */
  end

 end

 if ((pos('TEXT/HTML',CTLine) > 0) | ((left(Global.!MessBody,1) == '<') & (right(Global.!MessBody,1) == '>') & (pos('<HTML',translate(Global.!MessBody)) > 0) & (pos('</',Global.!MessBody) > 0))) then  /* if it is an HTML message or if it looks like HTML */
 do

  if (HTMLKeep) then  /* if we want to keep the HTML */
  do
   call PutFileContents AttDir||'\Message.HTML',HTMLFilter(Global.!MessBody)  /* write the filtered HTML to a file */
   call PutObjectEA AttDir||'\Message.HTML','.TYPE','text/html'  /* set the file type */
   Attachments = 1  /* we have an attachment */
  end

  Global.!MessBody = HTMLToText(Global.!MessBody)  /* convert the HTML body to text */

 end

 Global.!MessBody = translate(Global.!MessBody,Global.!FilterIn,xrange('80'x,'FF'x))  /* assume we have ISO text and convert the message body content to the local code page */
 Global.!MessBody = WordWrap(Global.!MessBody,Global.!Settings.BodyLineLength,1)  /* word-wrap the body if necessary, breaks are O.K. */

 return Attachments  /* quit */

end

/**************************************************************************/
/* If we got this far, we have a multipart message or an exotic message   */
/**************************************************************************/

if (\MIMEDecode(MessCont,AttDir)) then  /* if we cannot decode the message contents */
do
 return 0  /* quit with no attachments */
end

call sysfiletree AttDir||'\*','AttFiles.','FOS'  /* look for files in the attachments dir and any subdirs that may have been created */
call sysfiletree AttDir||'\part*','PartFiles.','FO'  /* look for message part files in the attachments dir */
Attachments = (AttFiles.0 - PartFiles.0)  /* the number of attachments -- for now */
TextPart = ''  /* no text part yet */
HTMLPart = ''  /* no HTML part yet */
BodyPart = ''  /* no body part yet */

do Index = 1 to PartFiles.0  /* for each message part file we found */

 MIMEType = translate(GetObjectEA(PartFiles.Index,'MIME-TYPE'))  /* get the MIME type EA in upper case */

 if (left(MIMEType,5) == 'TEXT/') then  /* if we have a text part */
 do

  TextCont = GetFileContents(PartFiles.Index)  /* get the file contents */
  call sysfiledelete PartFiles.Index  /* zap the part file */

  if (TextCont >< '') then  /* if we have something */
  do

   select  /* do one of the following */

    when (MIMEType == 'TEXT/ENRICHED') then  /* if we have enriched text */
    do

     if (TextPart >< '') then  /* if we already have plain text */
     do
      TextCont = ''  /* lose the enriched bit */
     end
 
    end

    when ((MIMEType == 'TEXT/HTML') | ((left(TextCont,1) == '<') & (right(TextCont,1) == '>') & (pos('<HTML',translate(TextCont)) > 0) & (pos('</',TextCont) > 0))) then  /* if it looks like HTML */
    do

     if (HTMLKeep) then  /* if we want to keep the HTML */
     do
      call PutFileContents AttDir||'\Message_part_'||Index||'.HTML',HTMLFilter(TextCont)  /* write the filtered HTML to a file */
      call PutObjectEA AttDir||'\Message_part_'||Index||'.HTML','.TYPE','text/html'  /* set the file type */
      Attachments = Attachments + 1  /* we have one more attachment */
     end
 
     if ((pos('CHARSET="UTF-8"',translate(Global.!MessBody)) > 0) | (pos('CHARSET=UTF-8',translate(Global.!MessBody)) > 0)) then  /* if we can assume it is UTF-8 */
     do
      TextCont = DecodeUTF8(TextCont)  /* decode it */
      GotUTF = 1  /* we found UTF-8 and decoded it */
     end
     else  /* if there is no wrapper indication about UTF-8 text */
     do
      GotUTF = 0  /* we found no UTF-8 */
     end

     if (HTMLPart >< '') then  /* if we have previous HTML content */
     do
      HTMLPart = HTMLPart||copies('_',42)||Global.!CRLF  /* add a separator */
     end

     HTMLPart = HTMLPart||HTMLToText(TextCont,GotUTF)  /* convert the HTML to text and add to any existing HTML part (include the UTF-8 flag) */

    end

    otherwise  /* if neither of the above, assume we have normal text */
    do

     if ((pos('CHARSET="UTF-8"',translate(Global.!MessBody)) > 0) | (pos('CHARSET=UTF-8',translate(Global.!MessBody)) > 0)) then  /* if we can assume it is UTF-8 */
     do
      TextCont = DecodeUTF8(TextCont)  /* decode it */
     end

     if (TextPart >< '') then  /* if we have previous text content */
     do
      TextPart = TextPart||Global.!CRLF||copies('_',42)||Global.!CRLF  /* add a separator */
     end

     TextPart = TextPart||TextCont  /* add to any existing text part */

    end

   end

  end

 end
 else  /* if we have something else than HTML or plain text */
 do

  if (MIMEType >< '') then  /* if we have a MIME type */
  do
   call SetObjectData PartFiles.Index,'TITLE='||translate(MIMEType,'__','/\')  /* change the title to show what's in it */
  end

  Attachments = Attachments + 1  /* we have one more attachment */

 end

 if (pos('MULTIPART/',CTLine) > 0) then  /* if we have a proper multipart message */
 do

  select  /* do one of the following */

   when ((pos('/ALTERNATIVE',CTLine) > 0) & (HTMLPart >< '')) then  /* if we have an alternative multipart message and we have HTML */
   do
    TextPart = ''  /* lose the text part */
   end

   when ((pos('/MIXED',CTLine) > 0) | (pos('/RELATED',CTLine) > 0)) then  /* if we have a mixed or related multipart message */
   do

    if ((pos('/ALTERNATIVE',translate(Global.!MessBody)) > 0)  & (HTMLPart >< '')) then  /* if it looks like an alternative message part and we have HTML */
    do
     TextPart = ''  /* lose the text part */
    end

   end

   otherwise  /* if it is something else */
   do

    if (HTMLPart >< '') then  /* if we have HTML content */
    do
     HTMLPart = copies('_',42)||Global.!CRLF||'[text conversion of html content follows:]'||Global.!CRLF||HTMLPart  /* add a header */
    end

   end

  end

 end

 BodyPart = TextPart||HTMLPart  /* fit the parts together */

end

if (Attachments > 0) then  /* if we have attachments in the attachments dir */
do
 
 call sysfiletree AttDir||'\*','AttFiles.','FOS'  /* look for files in the attachments dir and any subdirs that may have been created */

 do Index = 1 to AttFiles.0  /* take each one */
  call SetObjectData AttFiles.Index,'TITLE='||CheckCommLine(filespec('N',AttFiles.Index))  /* change the title if necessary */
 end

end

if (BodyPart >< '') then  /* if we have body text */
do

 do while (right(BodyPart,4) == Global.!CRLF||Global.!CRLF)  /* as long as the body text ends in two new lines */
  BodyPart = left(BodyPart,(length(BodyPart) - 2))  /* remove one */
 end

end

if (BodyPart >< '') then  /* if we still have body text */
do
 BodyPart = translate(BodyPart,Global.!FilterIn,xrange('80'x,'FF'x))  /* convert to local code page */
 Global.!MessBody = WordWrap(BodyPart,Global.!Settings.BodyLineLength,1)  /* this is our viewing content after any word-wrapping that may be required (breaks O.K.) */
end
else  /* if we have no body text */
do
 Global.!MessBody = '[RexxMail found no message text]'  /* use this */
end

return (Attachments > 0)  /* end of UnpackMessage */

/**************************************************************************/
HaltUnpackMessage:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
UpdateSettings: procedure expose Global.  /* updates the current user's configuration file */
/**************************************************************************/

signal on halt name HaltUpdateSettings  /* handles halt locally */

parse arg OldSettingsFile,NewSettingsFile  /* get the arguments */

call LogAction 'Updating configuration file "'||OldSettingsFile||'"'  /* report */
MessText = ''  /* start with no message text */
BakFile = OldSettingsFile||'.backup'  /* the backup file */

if (FileCheck(OldSettingsFile)) then  /* if the RexxMail configuration file exists */
do

 OldSettingsCont = GetFileContents(OldSettingsFile)  /* get the contents of the old settings file */

 if (FileCheck(BakFile)) then  /* if the backup file already exists */
 do

  if (sysfiledelete(BakFile) > 0) then  /* if we cannot delete the old backup file */
  do
   call AddError 'Cannot delete "'||BakFile||'"'  /* report */
   return 0  /* and quit */
  end

 end

 if (\PutFileContents(BakFile,OldSettingsCont)) then  /* if we cannot write the export file contents to the backup file */
 do
  call AddError 'Cannot copy "'||OldSettingsFile||'" to "'||BakFile||'"'  /* report */
  return 0  /* and quit */
 end

 if (sysfiledelete(OldSettingsFile) > 0) then  /* if we cannot delete the existing export file */
 do
  call AddError 'Cannot delete "'||OldSettingsFile||'"'  /* report */
  return 0  /* and quit */
 end

end
else  /* if we have no file */
do
 OldSettingsCont = ''  /* we have no content */
end

NewSettingsCont = GetFileContents(NewSettingsFile)  /* get the contents of the new settings file */
OldSettings. = ''  /* start with nothing */
Counter = 1  /* start a counter */

do while(OldSettingsCont >< '')  /* as long as we have old settings content left */

 parse var OldSettingsCont NextLine (Global.!CRLF) OldSettingsCont  /* get the next line */
 StripLine = strip(translate(NextLine,' ',d2c(9)),'B',' ')  /* turn TABs into blanks, then get rid of extra blanks at either end */

 if (StripLine == '') then  /* if we have an empty line */
 do
  OldSettings.Counter._Comment = OldSettings.Counter._Comment||Global.!CRLF  /* add an empty line to the comment part */
 end
 else  /* if it is not an empty line */
 do

  if (left(StripLine,1) == '#') then  /* if it is a comment */
  do
   OldSettings.Counter._Comment = OldSettings.Counter._Comment||NextLine||Global.!CRLF  /* add the original line to the comment part */
  end
  else  /* if it is not a comment */
  do
   OldSettings.Counter._Entry = NextLine  /* store the original entry */
   OldSettings.Counter._KeyWord = translate(word(StripLine,1))  /* the keyword is the first word of the stripped line, in upper case */
   Counter = Counter + 1  /* up the counter to start a new entry */
  end

 end

end

OldSettings.0 = Counter  /* save  the counter */
NewSettings. = ''  /* start with nothing */
Counter = 0  /* restart the counter, this time at 0 */

do while(NewSettingsCont >< '')  /* as long as we have new settings content left */

 Counter = Counter + 1  /* up the counter */
 parse var NewSettingsCont NextPart (Global.!EmptyLine) NewSettingsCont  /* get the next part */
 NewSettings.Counter._Part = NextPart  /* save the part */

 do while (NextPart >< '')  /* as long as we have something left */
  parse var NextPart NextLine (Global.!CRLF) NextPart  /* get the next line, so we end up with just the last line */
 end

 NewSettings.Counter._Keyword = translate(word(NextLine,1))  /* the keyword is the first word; use upper case */

end

NewSettings.0 = Counter  /* save  the counter */
NewSettings = ''  /* start with nothing */
InvalidStuff = Global.!CRLF||copies('#',76)||Global.!CRLF||,  /* the first line of the warning to add before an invalid or obsolete entry */
               '# INVALID SETTING'||Global.!CRLF||,  /* the second line of the warning to add before an invalid or obsolete entry */
               '#'||Global.!CRLF||,  /* the comment before an invalid or obsolete entry */
               '# This setting is invalid or has become obsolete.'||Global.!CRLF||,  /* the comment before an invalid or obsolete entry */
               '# Please remove it.'||Global.!CRLF||,  /* the comment before an invalid or obsolete entry */
               '#'||Global.!CRLF||,  /* the comment before an invalid or obsolete entry */
               '# '  /* the comment before an invalid or obsolete entry */
GotVersion = 0  /* we have no version setting yet */
GotUpdate = 0  /* we are not updating as far as we know (we may be correcting an invalid setting) */
GotInvalid = 0  /* we have no invalid or obsolete entries yet */

do Counter = 1 to OldSettings.0  /* run through the old settings */

 Found = 0  /* we haven't found anything yet */

 do Index = 1 to NewSettings.0  /* run through the new settings */

  if (OldSettings.Counter._KeyWord == NewSettings.Index._KeyWord) then  /* if the two keywords are the same */
  do
   NewSettings.Index._Keyword = ''  /* get rid of the new keyword */
   Found = 1  /* we've found it */
  end

 end

 if (Found) then  /* if the old keyword was found in the new file */
 do
  NewSettings = NewSettings||OldSettings.Counter._Comment||OldSettings.Counter._Entry||Global.!CRLF  /* add the comment and entry to the new settings */
 end
 else  /* if the old keyword was not found in the new file */
 do

  NewSettings = NewSettings||OldSettings.Counter._Comment  /* always add the user's own comment part */

  if (OldSettings.Counter._KeyWord >< '') then  /* if we have an old keyword */
  do

   if (OldSettings.Counter._KeyWord == 'VERSION') then  /* if it is the 'Version' setting */
   do

    GotVersion = 1  /* we have a version setting */

    if (Global.!Build >< Global.!Settings.Version) then  /* if we are updating or reverting */
    do
     OldSettings.Counter._Entry = left(OldSettings.Counter._Entry,length(OldSettings.Counter._Entry) - 15)  /* get the user's indent for the version entry */
     NewSettings = NewSettings||OldSettings.Counter._Entry||Global.!Build||Global.!CRLF  /* insert the version string, retaining the original formatting */
     GotUpdate = 1  /* we have an update on our hands */
    end
    else  /* if we are not updating */
    do
     NewSettings = NewSettings||OldSettings.Counter._Entry||Global.!CRLF  /* insert the original version string */
    end

   end
   else  /* if it is not the 'Version' setting */
   do
    NewSettings = NewSettings||InvalidStuff||OldSettings.Counter._Entry||Global.!EmptyLine  /* add the commented-out entry to the new settings with a comment */
    GotInvalid = 1  /* we have at least one invalid or obsolete entry */
   end

  end

 end

end

if (GotInvalid) then  /* if we have commented out one or more invalid or obsolete entries */
do
 MessText = '- One or more invalid or obsolete entries have been commented out.'||Global.!CRLF  /* this is the message text */
end

if (GotVersion) then  /* if we already had a version setting */
do

 if (GotUpdate) then  /* if we have performed an update */
 do
  MessText = '- The version number entry has been changed to '||Global.!Build||'.'||Global.!CRLF||MessText  /* add this to the start of the message text */
 end

end
else  /* if we have no version number */
do
 VersionStuff = copies('#',76)||Global.!CRLF||,  /* start with a separator */
                '# REXXMAIL BUILD NUMBER'||Global.!CRLF||,  /* add a title */
                '#'||Global.!CRLF||,  /* add an empty comment line */
                '# DO NOT REMOVE THIS ENTRY!'||Global.!CRLF||,  /* add a warning */
                '#'||Global.!CRLF||,  /* add an empty comment line */
                'Version = '||Global.!Build  /* add the version entry */
 NewSettings = VersionStuff||Global.!EmptyLine||NewSettings  /* add the lot to what we have */
 MessText = '- A version number entry has been added.'||Global.!CRLF||MessText  /* add this to the start of the message text */
end

NewStuff = ''  /* no new stuff yet */

do Index = 1 to NewSettings.0  /* run through the new settings again */

 if (NewSettings.Index._KeyWord >< '') then  /* if the keyword still exists */
 do
  NewStuff = NewStuff||Global.!EmptyLine||NewSettings.Index._Part  /* add the complete entry to the new stuff */
 end

end

if (NewStuff >< '') then  /* if we have new stuff */
do
 NewSettings = NewSettings||Global.!CRLF||,  /* new line */
               copies('#',76)||Global.!CRLF||,  /* add a separator */
               '# NEW SETTINGS'||Global.!CRLF||,  /* add a header */
               '#'||Global.!CRLF||,  /* and an empty comment line */
               '# The following entries are new for this build of RexxMail, or they were not'||Global.!CRLF||,  /* add text */
               '# found in the existing configuration file.'||Global.!CRLF||,  /* add text */
               copies('#',76)||,  /* add a separator */
               NewStuff  /* and the new stuff */
 MessText = MessText||'- One or more new configuration entries have been added;'||Global.!CRLF  /* add this to the message text */
 MessText = MessText||'  please edit the new configuration entries now.'||Global.!CRLF  /* add this to the message text */
end

do while (right(NewSettings,2) == Global.!CRLF)  /* as long as the new settings end with a CRLF */
 NewSettings = left(NewSettings,length(NewSettings) - 2)  /* remove it */
end

if (\PutFileContents(OldSettingsFile,NewSettings||Global.!CRLF)) then  /* if we cannot write the new settings to the RexxMail settings file with a CRLF at the end */
do
 return 0  /* quit */
end

if (MessText >< '') then  /* if we have something to report */
do
 MessText = 'Your configuration file '||Global.!CRLF||,  /* add this to the front of the message text */
            '  '||OldSettingsFile||Global.!CRLF||,  /* followed by this */
            'has been updated:'||Global.!CRLF||MessText  /* and this */
 call SystemMessage 'RexxMail configuration updated.',MessText  /* send a message to the user */
end

return 1  /* end of UpdateSettings */

/**************************************************************************/
HaltUpdateSettings:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
ViewMessage: procedure expose Global.  /* views received mail messages; also resets folder icon if no unread messages remain */
/**************************************************************************/

signal on halt name HaltViewMessage  /* handles halt locally */

MessFile = Global.!ProcFile  /* the file name to process */
AttDir = AttDirCreate(MessFile)  /* get an attachments dir */

if (AttDir == '') then  /* if we have no attachments dir */
do
 return 0  /* quit */
end

GotAtt = UnpackMessage(MessFile,AttDir,1)  /* unpack the mail file into the attachment dir (process for viewing) */
InComing = (MessageSettings(MessFile,'1*******','MATCH'))  /* set a flag if the message is incoming */
MoveDir = ''  /* no need to move the message yet */

if (InComing) then  /* if the message is incoming */
do

 HeaderEntries = Global.!Settings.ViewHeaderIn  /* use this */

 if (MessageSettings(MessFile,'**0*****','MATCH')) then  /* if it is an unviewed message */
 do

  if ((pos(translate(Global.!MainDir),translate(MessFile)) == 1) | (\Global.!Settings.UseCurrentDir)) then  /* if the message is in a RexxMail folder, or if we want all viewed messages redirected */
  do

   MoveDir = MoveAction(MessFile,Global.!ControlView,Global.!In_ArchiveDir)  /* see if we need to move the file to another folder (and perhaps change the stationery) */

   if (\FileCheck(MessFile)) then  /* if the message got zapped (e.g. by a "RUN" action) */
   do
    return 0  /* quit */
   end

  end

  call MessageSettings MessFile,'**1*****','CHANGE'  /* new read mail message setting: Processed */

 end

 if (GotAtt) then  /* if we have attachments */
 do
  CurDir = directory()  /* store the current dir */
  call directory AttDir  /* switch to the attachments directory */
  call RunCommand Global.!Settings.RunAttachIn,MessFile  /* run an external command in the attachments dir if necessary */
  call directory CurDir  /* switch back to the original dir */
  call sysfiletree AttDir||'\*','PreFiles.','FOS'  /* see what's in the attachments dir */
  GotAtt = (PreFiles.0 > 0)  /* have we still got attachments? */
 end

end
else  /* if the message is outgoing, i.e. already sent */
do
 HeaderEntries = Global.!Settings.ViewHeaderOut  /* use this */
end

ViewFile = AttDir||' (VIEW) '  /* start with this */

if (GotAtt) then  /* if we (still) have attachments */
do

 address cmd 'attrib /s -a "'||AttDir||'\*" > NUL'  /* a kludge: the sysfiletree function does not always work */

 if (Global.!Settings.OpenAttachBeforeView) then  /* if we want the attachments folder opened */
 do
  call sysopenobject AttDir,0,1  /* open the attachments folder on the desktop */
 end

end
else  /* if not */
do
 call sysdestroyobject AttDir  /* remove the attachments dir */
 AttDir = ''  /* nothing left */
end

MessHead = FoldHeaderLines('Global.!MessHead.',HeaderEntries,'V')  /* arrange the header lines for viewing */
ViewFile = ViewFile||CheckCommLine(strip(left(GetHeaderEntry('SUBJECT','Global.!MessHead.'),42)),':')  /* the full view file name */

if (\PutFileContents(ViewFile,MessHead||Global.!CRLF||Global.!MessBody)) then  /* if we cannot fill the view file with the view text */
do
 return 0  /* quit */
end

call RunCommand Global.!Settings.RunBeforeView,ViewFile  /* try to run a command before viewing the view file */
ButtonID = MessBarAdd(MessFile,AttDir)  /* add the message to the message button bar */
call RunCommand Global.!Settings.RunView,ViewFile  /* try to run a command to view the view file */
call MessBarRemove ButtonID  /* delete the message and folder buttons */
call RunCommand Global.!Settings.RunAfterView,ViewFile  /* try to run a command see after viewing the view file */
call sysdestroyobject ViewFile  /* get rid of the view file */

if (GotAtt) then  /* if we had attachments */
do

 if (Global.!Settings.CloseAttachAfterView) then  /* if we normally want the attachments folder closed */
 do

  GotAtt = 0  /* no need to keep the attachments folder yet */

  if (Incoming) then  /* if the message was incoming */
  do

   call sysfiletree AttDir||'\*','PostFiles.','FOS'  /* what have we got (left) in the attachments dir? */

   do Index = 1 to PostFiles.0  /* run through the list of files we have now */

    Existing = 0  /* assume the file is not an existing one (i.e. it was not there before) */

    do Counter = 1 to PreFiles.0  /* run through the list of files we had before */

     if (PostFiles.Index == PreFiles.Counter) then  /* if we have a match */
     do
      call sysfiletree PostFiles.Index,'ArchFiles.','FO','+****'  /* see if the archive attribute of the file was set */
      Existing = (ArchFiles.0 == 0)  /* if it has not, the file hasn't changed */
     end

    end

    if (\Existing) then  /* if the file was not an existing one */
    do
     GotAtt = 1  /* we have at least 1 change, se let's keep the attachments folder */
    end

   end

  end

 end

 if (\GotAtt) then  /* if we have no attachments to keep */
 do
  call sysdestroyobject AttDir  /* remove the attachments dir */
 end

end

if (FileCheck(MessFile)) then  /* if the message file still exists */
do
 call MakeShadow MessFile,Global.!Settings.ShadowView,Global.!In_ArchiveDir,MoveDir  /* create a shadow if necessary */
 call MoveMessage MessFile,MoveDir  /* move the file if necessary */
end

if (pos(translate(Global.!In_ArchiveDir),translate(MessFile)) == 1) then  /* if the message folder is an In Archive folder */
do
 call ResetFolderIcon strip(filespec('D',MessFile)||filespec('P',MessFile),'T','\')  /* reset the folder icon if necessary */
end

return 1  /* end of ViewMessage */

/**************************************************************************/
HaltViewMessage:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
ViewRawMessage: procedure expose Global.  /* shows the raw message contents */
/**************************************************************************/

signal on halt name HaltViewRawMessage  /* handles halt locally */

MessFile = Global.!ProcFile  /* get the file name to process */
Tempfile = TempFileName(,,'RAW')  /* get a temp file name */

if (TempFile == '') then  /* if we cannot get a temp file */
do
 return 0  /* quit with no success */
end

if (\FileCopy(MessFile,TempFile)) then  /* if we cannot copy the message file to the temp file */
do
 call AddError 'Cannot copy "'||MessFile||'" to "'||TempFile||'"'  /* report */
 return 0  /* and quit */
end

call RunCommand Global.!Settings.RunBeforeViewRaw,TempFile  /* try to run a command before viewing the raw file */
call RunCommand Global.!Settings.RunViewRaw,TempFile  /* see if we can view the raw file */
call RunCommand Global.!Settings.RunAfterViewRaw,TempFile  /* try to run a command after viewing the raw file */
call sysfiledelete TempFile  /* get rid of the temporary file */

return 1  /* end of ViewRawMessage */

/**************************************************************************/
HaltViewRawMessage:  /* handles halt locally */
/**************************************************************************/

return 0  /* no success */

/**************************************************************************/
WarnIcon: procedure expose Global.  /* shows that mail has arrived, or deletes/hides its own program object */
/**************************************************************************/

parse upper arg Action,Quiet  /* get the argument, if any */

Quiet = (Quiet == 1)  /* 1 = TRUE */
ObjectID = '<REXXMAIL_WARNING_OBJECT_'||Global.!User||'>'  /* the "Mail!" warning program object ID */

if (syssetobjectdata('<REXXMAIL_WARNING_OBJECT>','OBJECTID='||ObjectID)) then  /* if we can update the object ID */
do
 call LogAction 'Updated Mail warning object to : '||ObjectID,1  /* report, quietly */
end

select  /* do one of the following */

 when (Action == 'DELETE') then  /* if we are to delete the warning object */
 do

  if (sysdestroyobject(ObjectID)) then  /* if we can delete the object */
  do
   Action = 'deleted'  /* use this later on */
  end
  else  /* if we failed */
  do
   Action = 'Cannot delete'  /* use this later on */
  end

 end

 when (Action == 'HIDE') then  /* if we are to hide the warning object */
 do

  if (syssetobjectdata(ObjectID,'NOTVISIBLE=YES')) then  /* if we can hide the object -- do not use SetObjectData()! */
  do
   Action = 'made invisible'  /* use this later on */
  end
  else  /* if we failed */
  do

   if (Quiet) then  /* if we want no alarm */
   do
    Action = ''  /* no report necessary */
   end
   else  /* if we want alarms */
   do
    Action = 'Cannot hide'  /* use this later on */
   end

  end

 end

 when (Action == 'SHOW') then  /* if we are to show the warning object */
 do

  if (syssetobjectdata(ObjectID,'NOTVISIBLE=NO')) then  /* if we can show the object -- do not use SetObjectData()! */
  do
   Action = 'made visible'  /* use this later on */
  end
  else  /* if we failed */
  do
   Action = 'Cannot show'  /* use this later on */
  end

 end

 otherwise  /* if none of the above */
 do

  MainOpen = 0  /* no main folder found yet */

  if (sysqueryswitchlist('Stem.') == 0) then  /* if we can store the window list contents in Stem. */
  do

   do Index = 1 to Stem.0  /* take each entry */

    Entry = translate(Stem.Index)  /* make the list entry upper case */

    if ((pos(translate(filespec('N',Global.!MainDir)),Entry) == 1) & (pos(' VIEW',Entry) == (length(Entry) - 4))) then  /* if it looks identical to the main mail folder */
    do
     MainOpen = 1  /* the mail folder must be open on the desktop, so set a flag */
    end

   end

  end

  if (\MainOpen) then  /* if the main mail folder is not open on the desktop */
  do

   if (syssetobjectdata(ObjectID,'NOTVISIBLE=NO')) then  /* if we can show the object -- do not use SetObjectData()!  */
   do
    Action = 'made visible'  /* use this later on */
   end
   else  /* if we failed, assume the object does not exist */
   do

    Settings = 'EXENAME='||Global.!ProgDir||'\RexxMail.CMD;'||,  /* opening the object will launch RexxMail */
               'PARAMETERS=/OPEN %;'||,  /* use the "/Open" switch and do not accept additional parameters */
               'MINIMIZED=YES;'||,  /* the program will run minimized */
               'OBJECTID='||ObjectID||';'||,  /* use the object ID defined earlier */
               'ICONFILE='||Global.!IconDir||'\Objects\Warning.ICO;'  /* attach the mail warning icon file */

    if (syscreateobject('WPProgram','You have mail','<WP_DESKTOP>',Settings)) then  /* if we can create a new program object */
    do
     Action = 'created'  /* use this later on */
    end
    else  /* if we failed */
    do
     Action = 'Cannot create'  /* use this later on */
    end

   end

  end

 end

end

if (Action >< '') then  /* if we want a report */
do

 if (word(Action,1) == 'Cannot') then  /* if the action failed */
 do
  call AddError Action||' mail warning object'  /* report */
  return 0  /* return no go */
 end

 call LogAction 'Mail warning object '||Action,1  /* report, quietly */

end

return 1  /* end of WarnIcon */

/**************************************************************************/
WordWrap: procedure expose Global.  /* word-wraps a block of text */
/**************************************************************************/

parse arg Text,MaxLength,Break,IndentString,QuotedP  /* get the arguments */

if ((MaxLength == '') & (IndentString == '')) then  /* if we have no maximum line length value and no indent string */
do
 return Text  /* do nothing */
end

if (MaxLength >< '') then  /* if we have a maximum length */
do

 if (MaxLength <= 0) then  /* if we have an impossible value */
 do
  Return Text  /* do nothing */
 end

 CutPoint = MaxLength + 1  /* start the cut at the maximum length plus one */

end

QuotedP = (QuotedP == 1)  /* 1 = TRUE */
Break = (Break == 1)  /* 1 = TRUE */
NewText = ''  /* start with nothing */

if (QuotedP) then  /* if we want Quoted-Printable text */
do
 ReturnString = '='  /* add this before a CRLF */
end
else  /* if we want normal text */
do
 ReturnString = ''  /* nothing before a CRLF */
end

do while (Text >< '')  /* as long as we have text */

 parse var Text NextLine (Global.!CRLF) Text  /* get the next line */

 if (MaxLength >< '') then  /* if we have a maximum length */
 do

  do while (length(NextLine) > MaxLength)  /* as long as the line is longer than the limit */

   parse var NextLine NextBit =(CutPoint) NextLine  /* get the next bit of the maximum length */

   if ((right(NextBit,1) == ' ') | (left(NextLine,1) == ' ')) then  /* if we cut the line just before or after a space, i.e. we cut between two words */
   do

    NextBit = strip(NextBit,'T',' ')  /* get rid of trailing spaces */
    NextLine = strip(NextLine,'L',' ')  /* get rid of leading spaces on the remaining bit of the line */

    if (QuotedP) then  /* if we are doing Quoted-Printable */
    do
     NextBit = NextBit||' '  /* add a trailing space */
    end

   end
   else  /* if we cut through a word */
   do

    LastSpace = lastpos(' ',NextBit)  /* look for the last space in this bit */

    if (LastSpace > 0) then  /* if we found a space */
    do

     parse var NextBit NextBit =(LastSpace) MoreBit  /* cut off the bit starting with the space and save it for the next line */
     NextBit = strip(NextBit,'T',' ')  /* get rid of trailing spaces */
     NextLine = strip(MoreBit,'L',' ')||NextLine  /* get rid of the leading space and stick this bit back on to what's left of the original text line */

     if (QuotedP) then  /* if we are doing Quoted-Printable */
     do
      NextBit = NextBit||' '  /* add a trailing space */
     end

    end
    else  /* if we found no space */
    do

     if (Break) then  /* if we can break words */
     do

      LastEqual = lastpos('=',NextBit)  /* look for the last = in this bit */

      if (LastEqual > 1) then  /* if we found it and it is not the first character in the line */
      do
       parse var NextBit NextBit =(LastEqual) MoreBit  /* cut off the bit starting with the = and save it for the next line */
       NextLine = MoreBit||NextLine  /* stick this bit back on to what's left of the original text line */
      end

     end
     else  /* if we must not break words */
     do
      parse var NextLine MoreBit NextLine  /* get the first word of the remaining bit of the line */
      NextBit = NextBit||MoreBit  /* and add it to what we have */
     end

    end

   end

   NewText = NewText||NextBit  /* add the bit we have to the new text */

   if (NextLine >< '') then  /* if there is more of the same line to come */
   do
    NewText = NewText||ReturnString||Global.!CRLF||IndentString  /* add a new line with an indent */
   end

  end

 end

 NewText = NewText||NextLine  /* add the remaining bit of the line to the text */

 if (Text >< '') then  /* if there is more text to come */
 do
   NewText = NewText||Global.!CRLF||IndentString  /* add a new line with an indent */
 end

end

return NewText  /* end of WordWrap */

