/*
 * EXIFREAD.C
 *
 * Extracts fields from EXIF format files
 *
 * Thanks to TsuruZoh Tachibanaya, for compiling the description
 * of the format; See:
 *   http://www.ba.wakwak.com/~tsuruzoh/Computer/Digicams/exif-e.html
 * Additional info from exif.org:
 *   http://www.exif.org/
 *
 * (c) 2003 Stephan Dahl
 * Permission granted to use for any purpose.
 * Author disclaims any usefulness whatsoever.
 */

#define DEBUG

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#define MARKSTART     0xFF
#define SOIMARK       0xD8
#define EOIMARK       0xD9
#define APP1MARK      0xE1
#define SOSMARK       0xDA
#define TAGMAKE       0x010F
#define TAGMODEL      0x0110
#define TAGORIENT     0x0112
#define TAGXRES       0x011A
#define TAGYRES       0x011B
#define TAGRESUNIT    0x0128
#define TAGDATETIMEE  0x0132
#define TAGYCBCRPOS   0x0213
#define TAGSUBEXIF    0x8769
#define TAGEXPOTIME   0x829A
#define TAGFNUMBER    0x829D
#define TAGEXIFVER    0x9000
#define TAGDATETIMEO  0x9003
#define TAGDATETIMED  0x9004
#define TAGCOMPCONF   0x9101
#define TAGCOMPRESS   0x9102
#define TAGSHUTSPEED  0x9201
#define TAGAPERTURE   0x9202
#define TAGEXPBIAS    0x9204
#define TAGMAXAPERT   0x9205
#define TAGMETERMODE  0x9207
#define TAGFLASH      0x9209
#define TAGFOCALLEN   0x920A
#define TAGMAKERNOTE  0x927C
#define TAGUSRCOMMENT 0x9286
#define TAGFLASHPIXV  0xA000
#define TAGCOLSPACE   0xA001
#define TAGIMGWIDTH   0xA002
#define TAGIMGHEIGHT  0xA003
#define TAGINTEROPO   0xA005
#define TAGFOCALXRES  0xA20E
#define TAGFOCALYRES  0xA20F
#define TAGFOCALRESU  0xA210
#define TAGSENSMETH   0xA217
#define TAGFILESRC    0xA300
/*
#define TAG   0xA401
#define TAG   0xA402
#define TAG   0xA403
#define TAG   0xA404
#define TAG   0xA406
*/

static char* orientationText[] = {
    "ERROR",
    "top, left side",
    "top, right side",
    "bottom, right side",
    "bottom, left side",
    "left side, top",
    "right side, top",
    "right side, bottom",
    "left side, bottom",
    "ERROR"};
static char* resUnitText[] = {
    "ERROR",
    "no-unit",
    "inch",
    "cm",
    "ERROR"};
static char* YCbCrText[] = {
		"ERROR",
		"Center Point",
		"Datum Point",
		"ERROR"};
static char* compConfText[] = {
		"ERROR",
		"Y",
		"Cb",
		"Cr",
		"Red",
		"Green",
		"Blue",
		"ERROR"};
static char *flashText[] = {
		"Flash did not fire",
		"Flash fired",
		"Error 3",
		"Error 4",
		"Error 5",
		"Flash fired but strobe return light not detected",
		"Error 6",
		"Flash fired and strobe return light detected",
		"ERROR"};

enum states {INIT=0, SOI, INSTREAM, INMARKER, INAPP1, END};
enum alignments {INTELALIGN=0, MOTOROLAALIGN};

int bytealign=0;

long filepos=0;

int getnext(FILE *fin) {
  filepos=filepos+1;
  return fgetc(fin);
}

long makeInt(char *buf, int size) {
  int mult=1;
  long val=0;
  int i,byte;

#ifdef DEBUG
printf("makeInt(");
for (i=0;i<size;i++) printf("%02X",buf[i]);
printf(",%d):",size);
#endif
  for (i=0; i<size; i++) {
    if (bytealign==MOTOROLAALIGN) byte=buf[size-i-1]; else byte=buf[i];
    val=val+byte*mult;
    mult=mult*256;
  }
#ifdef DEBUG
printf(" %ld\n",val);
#endif
  return val;
}

char *meterModeText(int mode) {
	static char *texts[] = {
		"unknown",
		"average",
		"center weighted average",
		"spot",
		"multi-spot",
		"multi-segment",
		"partial"
		"other",
		"ERROR"};
	switch(mode) {
		case 0: return texts[0]; break;
		case 1: return texts[1]; break;
		case 2: return texts[2]; break;
		case 3: return texts[3]; break;
		case 4: return texts[4]; break;
		case 5: return texts[5]; break;
		case 6: return texts[6]; break;
		case 255: return texts[7]; break;
		default: break;
	}
	return texts[8];
}

void handleInterOpIFD(char *srcbuf, char *buf) {
  int i;
  int directorySize,tag,format;
  long noComponents,data;

#ifdef DEBUG
printf("handleInterOp(\n");
for (i=0;i<60;i++) {printf("%02X",buf[i]); if (i%8==7) printf("\n");}
printf("...)\n");
#endif
  directorySize = makeInt(buf,2);
  buf+=2;
#ifdef DEBUG
printf("%d dir entries\n",directorySize);
#endif
  for (i=0; i<directorySize; i++) {
    tag = makeInt(buf+i*12,2);
    format = makeInt(buf+i*12+2,2);
    noComponents = makeInt(buf+i*12+4,4);
    data = makeInt(buf+i*12+8,4);
#ifdef DEBUG
printf("InterOp SubIFD-%d: %04X %04X %ld %ld\n",i,tag,format,noComponents,data);
#endif
    switch(tag) {
      default:
				printf("Unhandled InterOp SubIFD-%d: %04X %04X %ld %ld\n",
					i,tag,format,noComponents,data);
        break;
    }
  }
}

void handleMakerNote(char *srcbuf, char *buf) {
  int i;
  int directorySize,tag,format;
  long noComponents,data;

#ifdef DEBUG
printf("handleMakerNote(\n");
for (i=0;i<606;i++) {printf("%02X.",buf[i]); if (i%8==7) printf(" "); if (i%16==15) printf("\n");}
printf("...)\n");
#endif
  directorySize = makeInt(buf,2);
  buf+=2;
#ifdef DEBUG
printf("%d dir entries\n",directorySize);
#endif
  for (i=0; i<directorySize; i++) {
    tag = makeInt(buf+i*12,2);
    format = makeInt(buf+i*12+2,2);
    noComponents = makeInt(buf+i*12+4,4);
    data = makeInt(buf+i*12+8,4);
#ifdef DEBUG
printf("MakerNote-%d: %04X %04X %ld %ld\n",i,tag,format,noComponents,data);
#endif
    switch(tag) {
      default:
				printf("Unhandled MakerNote-%d: %04X %04X %ld %ld\n",
					i,tag,format,noComponents,data);
        break;
    }
  }
}

void handleSubIFD(char *srcbuf, char *buf) {
  int i;
  int directorySize,tag,format;
  long noComponents,data;

#ifdef DEBUG
printf("handleSubEXIF(\n");
for (i=0;i<60;i++) {printf("%02X",buf[i]); if (i%8==7) printf("\n");}
printf("...)\n");
#endif
  directorySize = makeInt(buf,2);
  buf+=2;
#ifdef DEBUG
printf("%d dir entries\n",directorySize);
#endif
  for (i=0; i<directorySize; i++) {
    tag = makeInt(buf+i*12,2);
    format = makeInt(buf+i*12+2,2);
    noComponents = makeInt(buf+i*12+4,4);
    data = makeInt(buf+i*12+8,4);
#ifdef DEBUG
printf("EXIF SubIFD-%d: %04X %04X %ld %ld\n",i,tag,format,noComponents,data);
#endif
    switch(tag) {
      case TAGEXPOTIME   :
        printf("Exposure: %ld/%ld sec\n",makeInt(srcbuf+data-10,4),makeInt(srcbuf+data-10+4,4));
        break;
      case TAGFNUMBER    :
        printf("F-number: %ld/%ld\n",makeInt(srcbuf+data-10,4),makeInt(srcbuf+data-10+4,4));
        break;
      case TAGEXIFVER    :
        printf("Exif version: %4.4s\n",buf+i*12+8);
        break;
      case TAGDATETIMEO  :
        printf("Original Datetime: %s\n",srcbuf+data-10);
        break;
      case TAGDATETIMED  :
        printf("Digitized Datetime: %s\n",srcbuf+data-10);
        break;
      case TAGCOMPCONF   :
      	printf("Components Configuration: %s%s%s\n",
      		compConfText[min(makeInt(buf+i*12+8,1),7)],
      		compConfText[min(makeInt(buf+i*12+8+1,1),7)],
      		compConfText[min(makeInt(buf+i*12+8+2,1),7)]);
      	break;
      case TAGCOMPRESS   :
      	printf("Est. Compression: %ld/%ld\n",makeInt(srcbuf+data-10,4),makeInt(srcbuf+data-10+4,4));
      	break;
      case TAGSHUTSPEED  :
      	printf("Shutter speed: %ld/%ld\n",makeInt(srcbuf+data-10,4),makeInt(srcbuf+data-10+4,4));
      	break;
      case TAGAPERTURE   :
      	printf("Aperture: %ld/%ld\n",makeInt(srcbuf+data-10,4),makeInt(srcbuf+data-10+4,4));
      	break;
      case TAGEXPBIAS    :
      	printf("Exposure Bias: %ld/%ld\n",makeInt(srcbuf+data-10,4),makeInt(srcbuf+data-10+4,4));
      	break;
      case TAGMAXAPERT   :
      	printf("Lens max aperture: %ld/%ld\n",makeInt(srcbuf+data-10,4),makeInt(srcbuf+data-10+4,4));
      	break;
      case TAGMETERMODE  :
      	printf("Metering mode: %s\n",meterModeText(makeInt(buf+i*12+8,2)));
      	break;
      case TAGFLASH      :
      	printf("Flash: %s\n",flashText[min(makeInt(buf+i*12+8,1),2)]);
      	break;
      case TAGFOCALLEN   :
      	printf("Focal Length: %ld/%ld mm\n",makeInt(srcbuf+data-10,4),makeInt(srcbuf+data-10+4,4));
      	break;
      case TAGMAKERNOTE  :
      	handleMakerNote(buf,buf+data);
      	break;
      case TAGUSRCOMMENT :break;
      case TAGFLASHPIXV  :
        printf("FlashPix version: %4.4s\n",buf+i*12+8);
        break;
      case TAGCOLSPACE   :
      	printf("Color space: ");
      	if (data==1)
      		printf("sRGB");
      	else if (data==65535)
      		printf("uncalibrated");
      	else
      		printf("ERROR");
        printf("\n");
      	break;
      case TAGIMGWIDTH   :
        printf("Img Width: %ld\n",data);
        break;
      case TAGIMGHEIGHT  :
        printf("Img Height: %ld\n",data);
        break;
      case TAGINTEROPO   :
      	handleInterOpIFD(buf,buf+data-20);
      	break;
      case TAGFOCALXRES  :
      	printf("Focal X Resolution: %ld/%ld\n",makeInt(srcbuf+data-10,4),makeInt(srcbuf+data-10+4,4));
      	break;
      case TAGFOCALYRES  :
      	printf("Focal Y Resolution: %ld/%ld\n",makeInt(srcbuf+data-10,4),makeInt(srcbuf+data-10+4,4));
      	break;
      case TAGFOCALRESU  :
        printf("Focal Resolution Units: %s\n",resUnitText[min(makeInt(buf+i*12+8,2),4)]);
        break;
      case TAGSENSMETH   :
      	printf("Sensing method: %s\n",(data==2)?"chip color area":"unknown");
      	break;
      case TAGFILESRC    :
      	printf("File source: %s\n",(data==3)?"digital still camera":"unknown");
      	break;
      default:
				printf("Unhandled EXIF SubIFD-%d: %04X %04X %ld %ld\n",
					i,tag,format,noComponents,data);
        break;
    }
  }
}


void handleIFD(char *buf) {
  int i;
  int directorySize,tag,format;
  long noComponents,data;

#ifdef DEBUG
printf("handleIFD(\n");
for (i=0;i<60;i++) {printf("%02X",buf[i]); if (i%8==7) printf("\n");}
printf("...)\n");
#endif
  directorySize = makeInt(buf,2);
  buf+=2;
#ifdef DEBUG
printf("%d dir entries\n",directorySize);
#endif
  for (i=0; i<directorySize; i++) {
    tag = makeInt(buf+i*12,2);
    format = makeInt(buf+i*12+2,2);
    noComponents = makeInt(buf+i*12+4,4);
    data = makeInt(buf+i*12+8,4);
#ifdef DEBUG
printf("IFD0-%d: %04X %04X %ld %ld\n",i,tag,format,noComponents,data);
#endif
    switch(tag) {
      case TAGMAKE:
        printf("Make: %s\n",buf+data-10);
        break;
      case TAGMODEL:
        printf("Model: %s\n",buf+data-10);
        break;
      case TAGDATETIMEE:
        printf("Edit Datetime: %s\n",buf+data-10);
        break;
      case TAGORIENT:
        printf("Orientation: (0,0) at %s\n",orientationText[min(makeInt(buf+i*12+8,2),9)]);
        break;
      case TAGXRES:
        printf("X resolution: %ld/%ld\n",makeInt(buf+data-10+4,4),makeInt(buf+data-10,4));
        break;
      case TAGYRES:
        printf("Y resolution: %ld/%ld\n",makeInt(buf+data-10+4,4),makeInt(buf+data-10,4));
        break;
      case TAGRESUNIT:
        printf("Units: %s\n",resUnitText[min(makeInt(buf+i*12+8,2),4)]);
        break;
			case TAGYCBCRPOS:
				printf("YCbCrPositioning: %s\n",YCbCrText[min(makeInt(buf+i*12+8,2),3)]);
				break;
      case TAGSUBEXIF:
        handleSubIFD(buf,buf+data-10);
        break;
      default:
        break;
    }
  }
}

void handleAPP1(char *buf) {
  int i;
#ifdef DEBUG
printf("handleAPP1(");
for (i=0;i<10;i++) printf("%02X",buf[i]);
printf("...)\n");
#endif
  if (strncmp(buf,"Exif\x00\x00",6)) {
    printf("APP1 EXIF header is not Exif\n");
    exit(8);
  }
  if (strncmp(buf+6,"II",2)==0) {
    bytealign=INTELALIGN;
#ifdef DEBUG
printf("Intel Align: LSB first\n");
#endif
  }
  else if (strncmp(buf+6,"MM",2)==0) {
    bytealign=MOTOROLAALIGN;
#ifdef DEBUG
printf("Motorola Align: MSB first\n");
#endif
  }
  else {
    printf("APP1 TIFF header contains ...\n");
    exit(8);
  }
  handleIFD(buf+14);
}


int main(int argc, char **argv) {
  FILE *fin;                /* input files */
  int  state;
  int  c, c1, c2;           /* char in */
  int  rc;
  int  markersize;
  char *buf;

  if (argc!=2) {
    printf("Usage: EXIFREAD exiffile\n");
    exit(8);
  }
  if (!(fin = fopen(argv[1],"rb"))) {
    fprintf(stderr,"Unable to open file %s\n",argv[1]);
    exit(12);
  }
  state=INIT;
  while(!feof(fin) && state!=END) {
    c = getnext(fin);
    switch(state) {
      case INIT:
        if (c==MARKSTART) {
          c = getnext(fin);
          switch(c) {
            case SOIMARK:
              state=SOI;
              break;
            default:
              printf("Unexpected character %02X at position %ld\n",c,filepos);
              exit(8);
          }
        }
        else {
          printf("Unexpected character %02X at position %ld\n",c,filepos);
          exit(8);
        }
        break;
      case SOI:
        if (c==MARKSTART) {
          c = getnext(fin);
          switch(c) {
            case SOSMARK:
              state=INSTREAM;
              c1 = getnext(fin);
              c2 = getnext(fin);
              markersize=c1*256+c2;
              rc = fseek(fin,markersize-2,SEEK_CUR);
              filepos+=markersize-2;
              if (rc!=0) {
                printf("Error %d from fseek at %ld\n",rc,filepos);
                exit(8);
              }
              break;
            case APP1MARK:
              c1 = getnext(fin);
              c2 = getnext(fin);
              markersize=c1*256+c2;
#ifdef DEBUG
printf("At %ld: In APP1 Marker: Size %d\n",filepos,markersize);
#endif
              buf=malloc(markersize-2); /* Size-2 */
              rc=fread(buf,sizeof(char),markersize-2,fin);
              if (rc!=markersize-2) {
                printf("At %ld: Error reading %d characters into APP1 - read %d\n",filepos,markersize-2,rc);
                exit(8);
              }
              filepos+=rc;
              handleAPP1(buf);
              free(buf);
              break;
            default:
              c1 = getnext(fin);
              c2 = getnext(fin);
              markersize=c1*256+c2;
#ifdef DEBUG
printf("At %ld: In marker %02X: Size %d\n",filepos,c,markersize);
#endif
              rc = fseek(fin,markersize-2,SEEK_CUR);
              filepos+=markersize-2;
              if (rc!=0) {
                printf("Error %d from fseek at %ld\n",rc,filepos);
                exit(8);
              }
              break;
          }
        }
        else {
          printf("Unexpected character %02X at position %ld\n",c,filepos);
          exit(8);
        }
        break;
      case INSTREAM:
        if (c==MARKSTART) {
          c = getnext(fin);
          if (c==EOIMARK) {
#ifdef DEBUG
printf("At %ld: End marker\n",filepos);
#endif
            state=END;
          }
        }
        break;
      default:
        printf("Undefined state %d at position %ld\n",state,filepos);
        exit(8);
    }
  }
  return 0;
}


