;;; s7 test suite
;;;
;;; sources include 
;;;   clisp test suite
;;;   sbcl test suite
;;;   Paul Dietz's CL test suite
;;;   R Kelsey, W Clinger, and J Rees r5rs.html (and r6rs.html)
;;;   A Jaffer's r4rstest.scm (the inspiration for this...)
;;;   guile test suite
;;;   gauche test suite
;;;   gambit test suite
;;;   sacla test suite
;;;   Kent Dybvig's "The Scheme Programming Language"
;;;   Brad Lucier (who also pointed out many bugs)
;;;   GSL tests
;;;   Abramowitz and Stegun, "Handbook of Mathematical Functions"
;;;   Weisstein, "Encyclopedia of Mathematics"
;;;   the arprec package of David Bailey et al
;;;   Maxima, William Schelter et al
;;;   H Cohen, "A Course in Computational Algebraic Number Theory"
;;;   N Higham, "Accuracy and Stability of Numerical Algorithms"
;;;   various mailing lists and websites (see individual cases below)

(define with-bignums (provided? 'gmp))  ; scheme number has any number of bits
					; we assume s7_Double is double, and s7_Int is long long int
                                        ;   a few of the bignum tests assume the default bignum-precision is 128

;;; (define pi 3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582231725359408128481117450284102701938521105559644622948954930382)
;;; (set! *safety* 1)

;;; --------------------------------------------------------------------------------

(if (and (defined? 'current-time) ; in Snd
	 (defined? 'mus-rand-seed))
    (set! (mus-rand-seed) (current-time)))

(define (ok? otst ola oexp)
  (let ((result (catch #t ola
		       (lambda args 
			 (if (not (eq? oexp 'error)) 
			     (begin (display args) (newline)))
			 'error))))
    ;(format #t "~A: ~A -> ~A~%" otst result oexp)
    (if (not (equal? result oexp))
	(format #t "~A: ~A got ~S but expected ~S~%~%" (port-line-number) otst result oexp))))

(defmacro test (tst expected) ;(display tst) (newline)
  `(ok? ',tst (lambda () ,tst) ,expected))

(define (tok? otst ola)
  (let* ((data #f)
	 (result (catch #t ola (lambda args (set! data args) 'error))))
    (if (or (not result)
	    (eq? result 'error))
	(format #t "~A: ~A got ~S ~A~%~%" (port-line-number) otst result (or data "")))))

(defmacro test-t (tst) ;(display tst) (newline)
  `(tok? ',tst (lambda () ,tst)))

(defmacro test-e (tst op arg) ;(display tst) (newline)
  `(let ((result (catch #t (lambda () ,tst) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (format #t "~A: (~A ~S) got ~S but expected 'error~%~%" (port-line-number) ,op ,arg result))))


(define (op-error op result expected)
  
  (define (conjugate n) 
    (make-rectangular (real-part n) (- (imag-part n))))
  
  (if (and (real? result)
	   (real? expected))
      (/ (abs (- result expected)) (max 1.0 (abs expected)))
      (case op
	((acosh)
	 (/ (magnitude (- (cosh result) (cosh expected)))
	    (max 0.001 (magnitude (cosh expected)))))
	((asin)
	 (/ (min (magnitude (- (sin result) (sin expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (* 10 (magnitude (sin expected))))))
	((acos)
	 (/ (min (magnitude (- (cos result) (cos expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (magnitude (cos expected)))))
	((asinh)
	 (/ (magnitude (- (sinh result) (sinh expected)))
	    (max 0.001 (magnitude (sinh expected)))))
	((atanh)
	 (/ (min (magnitude (- (tanh result) (tanh expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (magnitude (tanh expected)))))
	((atan)
	 (/ (min (magnitude (- (tan result) (tan expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (magnitude (tan expected)))))
	((cosh)
	 (/ (min (magnitude (- result expected))
		 (magnitude (- result (- expected))))
	    (max 0.001 (magnitude expected))))
	(else (/ (magnitude (- result expected)) (max 0.001 (magnitude expected)))))))


;;; relative error (/ (abs (- x res) (abs x)))

(define error-12 1e-12)
(define error-6  1e-6)

(define (number-ok? tst result expected)
  (if (not (eq? result expected))
      (if (or (and (not (number? expected))
		   (not (eq? result expected)))
	      (and (nan? expected)
		   (not (nan? result)))
	      (and (nan? result)
		   (not (nan? expected)))
	      (and (number? expected)
		   (or (not (number? result))
		       (nan? result)))
	      (and (rational? expected)
		   (rational? result)
		   (not (= result expected)))
	      (and (or (rational? expected) 
		       (rational? result))
		   (real? expected)
		   (real? result)
		   (> (abs (- result expected)) error-12))
	      (and (pair? tst)
		   (> (op-error (car tst) result expected) error-6)))
	  (begin
	    (format #t "~A: ~A got ~A~Abut expected ~A~%" 
		    (port-line-number) tst result 
		    (if (and (rational? result) (not (rational? expected)))
			(format #f " (~A) " (* 1.0 result))
			" ")
		    expected)
	    
	    (if (and (not (number? expected))
		     (not (eq? result expected)))
		(format #t ", (eq? ~A ~A) -> #f" result expected)
		(if (and (number? expected)
			 (or (not (number? result))
			     (nan? result)))
		    (begin
		      (if (not (number? result))
			  (format #t ", (number? ~A) but not (number? ~A)" expected result)
			  (format #t ", (number? ~A) but (nan? ~A)" expected result)))
		    (if (and (rational? expected)
			     (rational? result)
			     (not (= result expected)))
			(format #t ", exact results but not (= ~A ~A): ~A" expected result (= result expected))
			(if (and (or (rational? expected) 
				     (rational? result))
				 (real? expected)
				 (real? result)
				 (> (abs (- result expected)) error-12))
			    (format #t ", rational results but diff > ~A: ~A" error-12 (abs (- result expected)))
			    (if (and (pair? tst)
				     (< (op-error (car tst) result expected) error-6))
				(let ((n result))
				  (format #t ", result not internally consistent")
				  (if (and (integer? n) 
					   (or (not (= (denominator n) 1))
					       (not (= n (numerator n)))
					       (not (zero? (imag-part n)))
					       (not (= (floor n) (ceiling n) (truncate n) (round n) n))
					       (not (= n (real-part n)))))
				      (format #t ", ~A integer but den: ~A, num: ~A, imag: ~A, real: ~A, floors: ~A ~A ~A ~A"
					      n (denominator n) (numerator n) (imag-part n) (real-part n)
					      (floor n) (ceiling n) (truncate n) (round n))
				      (if (and (rational? n)
					       (not (integer? n))
					       (or (not (zero? (imag-part n)))
						   (= (denominator n) 1)
						   (= (denominator n) 0)
						   (not (= n (real-part n)))
						   (not (= n (/ (numerator n) (denominator n))))))
					  (format #t ", ~A ratio but imag: ~A, den: ~A, real: ~A, ~A/~A=~A"
						  n (imag-part n) (denominator n) (real-part n) 
						  (numerator n) (denominator n) (* 1.0 (/ (numerator n) (denominator n))))
					  (if (and (real? n)
						   (not (rational? n))
						   (or (not (zero? (imag-part n)))
						       (not (= n (real-part n)))))
					      (format #t ", ~A real but rational: ~A, imag: ~A, real: ~A"
						      n (rational? n) (imag-part n) (real-part n))
					      (format #t ", ~A complex but real? ~A, imag: ~A, ~A+~A=~A"
						      n (real? n) (imag-part n) (real-part n) (imag-part n)
						      (+ (real-part n) (* 0+i (imag-part n)))))))))))))
	    (newline) (newline)))))

(define (nok? otst ola oexp)
  (let ((result (catch #t ola (lambda args 'error))))
     (number-ok? otst result oexp)))

(defmacro num-test (tst expected) ;(display tst) (newline)
  `(nok? ',tst  (lambda () ,tst) ,expected))

(define-macro (num-test-1 proc val tst expected)
  `(let ((result (catch #t (lambda () ,tst) (lambda args 'error))))
     (number-ok? (list ,proc ,val) result ,expected)))

(define-macro (num-test-2 proc val1 val2 tst expected)
  `(let ((result (catch #t (lambda () ,tst) (lambda args 'error))))
     (number-ok? (list ,proc ,val1 ,val2) result ,expected)))

(define-macro (reinvert n op1 op2 arg)
  (let ((body `(,op2 (,op1 ,arg))))
    (do ((i 1 (+ i 1)))
	((= i n))
      (set! body `(,op2 (,op1 ,body))))
    body))

(define-macro (recompose n op arg)
  (define (recompose-1 n)
    (if (= n 1)
	`(,op ,arg)
	`(,op ,(recompose-1 (- n 1)))))
  (recompose-1 n))


(if (symbol-access 'val) (set! (symbol-access 'val) #f)) ; might get here from snd-test

(define _ht_ (make-hash-table))



;;; --------------------------------------------------------------------------------
;;; GENERIC STUFF
;;; --------------------------------------------------------------------------------

;;; eq?
(test (eq? 'a 3) #f)
(test (eq? #t 't) #f)
(test (eq? "abs" 'abc) #f)
(test (eq? "hi" '(hi)) #f)
(test (eq? "hi" "hi") #f)
(test (eq? "()" '()) #f)
(test (eq? '(1) '(1)) #f)
(test (eq? '(#f) '(#f)) #f)
(test (eq? #\a #\b) #f)
(test (eq? #t #t) #t)
(test (eq? #f #f) #t)
(test (eq? #f #t) #f)
(test (eq? (null? '()) #t) #t)
(test (eq? (null? '(a)) #f) #t)
(test (eq? (cdr '(a)) '()) #t)
(test (eq? 'a 'a) #t)
(test (eq? 'a 'b) #f)
(test (eq? 'a (string->symbol "a")) #t)
(test (eq? (symbol "a") (string->symbol "a")) #t)
(test (eq? :a :a) #t)
(test (eq? ':a 'a) #f)
(test (eq? ':a ':a) #t)
(test (eq? :a a:) #f)
(test (eq? ':a 'a:) #f)
(test (eq? 'a: 'a:) #t)
(test (eq? ':a: 'a:) #f)
(test (eq? 'a (symbol "a")) #t)
(test (eq? :: '::) #t)
;(test (eq? ': (symbol->keyword (symbol ""))) #t)
(test (eq? ':a (symbol->keyword (symbol "a"))) #t) ; but not a:
(test (eq? '(a) '(b)) #f)
(test (let ((x '(a . b))) (eq? x x)) #t)
(test (let ((x (cons 'a 'b))) (eq? x x)) #t)
(test (eq? (cons 'a 'b) (cons 'a 'b)) #f)
(test (eq? "abc" "cba") #f)
(test (let ((x "hi")) (eq? x x)) #t)
(test (eq? (string #\h #\i) (string #\h #\i)) #f)
(test (eq? '#(a) '#(b)) #f)
(test (let ((x (vector 'a))) (eq? x x)) #t)
(test (eq? (vector 'a) (vector 'a)) #f)
(test (eq? car car) #t)
(test (eq? car cdr) #f)
(test (let ((x (lambda () 1))) (eq? x x)) #t)
(test (let ((x (lambda () 1))) (let ((y x)) (eq? x y))) #t)
(test (eq? 'abc 'abc) #t)
(test (eq? eq? eq?) #t)
(test (eq? (if #f 1) 1) #f)
(test (eq? '() '(#||#)) #t)
(test (eq? '() '(#|@%$&|#)) #t)
(test (eq? '#||#hi 'hi) #t) ; ??
(test (eq? '; a comment
         hi 'hi) #t) ; similar:
    (test (cadr '#| a comment |#(+ 1 2)) 1)
    (test `(+ 1 ,@#||#(list 2 3)) '(+ 1 2 3))
    (test `(+ 1 ,#||#(+ 3 4)) '(+ 1 7))
    ;; but not splitting the ",@" or splitting a number:
    (test (+ 1 2.0+#||#3i) 'error)
    (test `(+ 1 ,#||#@(list 2 3)) 'error)
(test (eq? #||# (#|%%|# append #|^|#) #|?|# (#|+|# list #|<>|#) #||#) #t)
(test (eq? '() ;a comment
	   '()) #t)
(test (eq? 3/4 3) #f)
(test (eq? '() '()) #t)
(test (eq? '() '(  )) #t)
(test (eq? '()'()) #t)
(test (eq? '()(list)) #t)
(test (eq? '() (list)) #t)
(test (eq? (begin) (append)) #t)
(test (let ((lst (list 1 2 3))) (eq? lst (apply list lst))) #t)

(test (eq? ''2 '2) #f)
(test (eq? '2 '2) #t) ; unspecified??
(test (eq? '2 2) #t)
(test (eq? ''2 ''2) #f)
(test (eq? ''#\a '#\a) #f)
(test (eq? '#\a #\a) #t) ; was #f 
(test (eq? 'car car) #f)
(test (eq? '() ()) #t)
(test (eq? ''() '()) #f)
(test (eq? '#f #f) #t)
(test (eq? '#f '#f) #t)
(test (eq? #f '  #f) #t)
(test (eq? '()'()) #t) ; no space
(test (#||# eq? #||# #f #||# #f #||#) #t)
(test (eq? (current-input-port) (current-input-port)) #t)
(test (let ((f (lambda () (quote (1 . "H"))))) (eq? (f) (f))) #t)
(test (let ((f (lambda () (cons 1 (string #\H))))) (eq? (f) (f))) #f)
(test (eq? *stdin* *stdin*) #t)
(test (eq? *stdout* *stderr*) #f)

(display ";this should display #t: ")
(begin #| ; |# (display #t))
(newline)

(test (;
       eq? ';!
       (;)()#
	);((")";
       ;"#|)#""
       '#|";"|#(#|;|#); ;#
	 ;\;"#"#f 
	       )#t)

(test (+ #| this is a comment |# 2 #| and this is another |# 3) 5)
(test (eq? #| a comment |# #f #f) #t)
(test (eq? #| a comment |##f #f) #t)  ; ??
(test (eq? #| a comment | ##f|##f #f) #t) ; ??
(test (eq? #||##||##|a comment| ##f|##f #f) #t)

(test (+ ;#|
            3 ;|#
            4)
      7)
(test (+ #| ; |# 3
		 4)
      7)
;;; Snd's listener is confused by (eq? #||##||##|a;comment|" ##f|##f #f) etc

(test (eq? (if #f #t) (if #f 3)) #t)

(test (eq?) 'error)           ; "this comment is missing a double-quote
(test (eq? #t) 'error)        #| "this comment is missing a double-quote |#
(test (eq? #t #t #t) 'error)  #| and this has redundant starts #| #| |#
(test (eq? #f . 1) 'error)
(test (eq #f #f) 'error)

(let ((things (vector #t #f #\space '() "" 0 1 3/4 1+i 1.5 '(1 .2) '#() (vector) (vector 1) (list 1) 'f 't #\t)))
  (do ((i 0 (+ i 1)))
      ((= i (- (vector-length things) 1)))
    (do ((j (+ i 1) (+ j 1)))
	((= j (vector-length things)))
      (if (eq? (vector-ref things i) (vector-ref things j))
	  (format #t ";(eq? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j))))))

;;; these are defined at user-level in s7 -- why are other schemes so coy about them?
(test (eq? (if #f #f) #<unspecified>) #t)
(test (eof-object? #<eof>) #t)
(test (eq? (symbol->value '_?__undefined__?_) #<undefined>) #t)
(test (eq? #<eof> #<eof>) #t)
(test (eq? #<undefined> #<undefined>) #t)
(test (eq? #<unspecified> #<unspecified>) #t)
(test (eq? #<eof> #<undefined>) #f)
(test (eq? #<eof> '()) #f)

(test (let () (define-macro (hi a) `(+ 1 ,a)) (eq? hi hi)) #t)
(test (let () (define (hi a) (+ 1 a)) (eq? hi hi)) #t)
(test (let ((x (lambda* (hi (a 1)) (+ 1 a)))) (eq? x x)) #t)

(test (eq? quasiquote quasiquote) #t)
(test (eq? `quasiquote 'quasiquote) #t)
(test (eq? 'if (keyword->symbol :if)) #t)
(test (eq? 'if (string->symbol "if")) #t)
(test (eq? (copy lambda) (copy 'lambda)) #f)
(test (eq? if 'if) #f)
(test (eq? if 'if) #f)
(test (eq? if (keyword->symbol :if)) #f)
(test (eq? if (string->symbol "if")) #f)
(test (eq? lambda and) #f)
(test (eq? let let*) #f)
(test (eq? quote quote) #t)
(test (eq? '"hi" '"hi") #f) ; guile also
(test (eq? '"" "") #f)
(test (eq? '"" '"") #f)
(test (eq? "" "") #f)
(test (eq? #() '#()) #f)
(test (eq? #() #()) #f)
(test (eq? '#() '#()) #f)
(test (let ((v #())) (eq? v #())) #f)
(test (let ((v '#())) (eq? v '#())) #f)
(test (let ((v #())) (eq? v v)) #t)


;;; eqv?
(test (eqv? 'a 3) #f)
(test (eqv? #t 't) #f)
(test (eqv? "abs" 'abc) #f)
(test (eqv? "hi" '(hi)) #f)
(test (eqv? "()" '()) #f)
(test (eqv? '(1) '(1)) #f)
(test (eqv? '(#f) '(#f)) #f)
(test (eqv? #\a #\b) #f)
(test (eqv? #\a #\a) #t)
(test (eqv? (integer->char 255) (string-ref (string #\x (integer->char 255) #\x) 1)) #t)
(test (eqv? (integer->char #xf0) (integer->char #x70)) #f)
(test (eqv? #\space #\space) #t)
(test (let ((x (string-ref "hi" 0))) (eqv? x x)) #t)
(test (eqv? #t #t) #t)
(test (eqv? #f #f) #t)
(test (eqv? #f #t) #f)
(test (eqv? (null? '()) #t) #t)
(test (eqv? (null? '(a)) #f) #t)
(test (eqv? (cdr '(a)) '()) #t)
(test (eqv? 'a 'a) #t)
(test (eqv? 'a 'b) #f)
(test (eqv? 'a (string->symbol "a")) #t)
(test (eqv? '(a) '(b)) #f)
(test (let ((x '(a . b))) (eqv? x x)) #t)
(test (let ((x (cons 'a 'b))) (eqv? x x)) #t)
(test (eqv? (cons 'a 'b) (cons 'a 'b)) #f)
(test (eqv? "abc" "cba") #f)
(test (let ((x "hi")) (eqv? x x)) #t)
(test (eqv? (string #\h #\i) (string #\h #\i)) #f)
(test (eqv? '#(a) '#(b)) #f)
(test (let ((x (vector 'a))) (eqv? x x)) #t)
(test (eqv? (vector 'a) (vector 'a)) #f)
(test (eqv? car car) #t)
(test (eqv? car cdr) #f)
(test (let ((x (lambda () 1))) (eqv? x x)) #t)
(test (eqv? (lambda () 1) (lambda () 1)) #f)
(test (let () (define (make-adder x) (lambda (y) (+ x y))) (eqv? (make-adder 1) (make-adder 1))) #f)
(test (eqv? 9/2 9/2) #t)
(test (eqv? quote quote) #t)
(test (eqv? "hi" "hi") #f) ; unspecified 
(test (eqv? #() #()) #f)   ; unspecified

(let ((c1 (let ((x 32))
	    (lambda () x)))
      (c2 (let ((x 123))
	    (lambda () x))))
  (test (eqv? c1 c2) #f)
  (test (eqv? c1 c1) #t))

(test (eqv? most-positive-fixnum most-positive-fixnum) #t)
(test (eqv? most-positive-fixnum most-negative-fixnum) #f)
(test (eqv? 9223372036854775807 9223372036854775806) #f)
(test (eqv? 9223372036854775807 -9223372036854775808) #f)
(test (eqv? -9223372036854775808 -9223372036854775808) #t)
(test (eqv? 123456789/2 123456789/2) #t)
(test (eqv? 123456789/2 123456787/2) #f)
(test (eqv? -123456789/2 -123456789/2) #t)
(test (eqv? 2/123456789 2/123456789) #t)
(test (eqv? -2/123456789 -2/123456789) #t)
(test (eqv? 2147483647/2147483646 2147483647/2147483646) #t)
(test (eqv? 3/4 12/16) #t)
(test (eqv? 1/1 1) #t)
(test (eqv? 312689/99532 833719/265381) #f)
(test (let ((x 3.141)) (eqv? x x)) #t)
(test (let ((x 1+i)) (eqv? x x)) #t)
(test (let* ((x 3.141) (y x)) (eqv? x y)) #t)
(test (let* ((x 1+i) (y x)) (eqv? x y)) #t)
(test (let* ((x 3/4) (y x)) (eqv? x y)) #t)
(test (eqv? 1.0 1.0) #t)
(test (eqv? 0.6 0.6) #t)
(test (eqv? 0.6 0.60) #t)
(test (eqv? 1+i 1+i) #t)
(test (eqv? -3.14 -3.14) #t)
(test (eqv? 1e2 1e2) #t)
(test (eqv? #i3/5 #i3/5) #t)
(test (eqv? #e0.6 #e0.6) #t)
(test (eqv? 1 1.0) #f)
(test (eqv? 1/2 0.5) #f)
(test (eqv? 1 1/1) #t)
(test (eqv? 0.5 5e-1) #t)

(test (eqv? (cons 'a 'b) (cons 'a 'c)) #f)
(test (eqv? eqv? eqv?) #t)
(test (eqv? '#(1) '#(1)) #f)
(test (eqv? '(1) '(1)) #f)
(test (eqv? '() '()) #t)
(test (eqv? '() (list)) #t)
(test (eqv? '(()) '(())) #f)

(let ((things (vector #t #f #\space '() "" 0 1 3/4 1+i 1.5 '(1 .2) '#() (vector) (vector 1) (list 1) 'f 't #\t)))
  (do ((i 0 (+ i 1)))
      ((= i (- (vector-length things) 1)))
    (do ((j (+ i 1) (+ j 1)))
	((= j (vector-length things)))
      (if (eqv? (vector-ref things i) (vector-ref things j))
	  (format #t ";(eqv? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j))))))

(test (eqv?) 'error)
(test (eqv? #t) 'error)
(test (eqv? #t #t #t) 'error)
(test (eqv #f #f) 'error)

(test (eqv? ''2 '2) #f)
(test (eqv? '2 '2) #t)
(test (eqv? '2 2) #t)
(test (eqv? ''2 ''2) #f)
(test (eqv? ''#\a '#\a) #f)
(test (eqv? '#\a #\a) #t)
(test (eqv? 'car car) #f)
(test (eqv? '() ()) #t)
(test (eqv? ''() '()) #f)
(test (eqv? '#f #f) #t)
(test (eqv? '#f '#f) #t)
(test (eqv? #<eof> #<eof>) #t)
(test (eqv? #<undefined> #<undefined>) #t)
(test (eqv? #<unspecified> #<unspecified>) #t)
(test (eqv? (if #f #f) #<unspecified>) #t)
(test (eqv? #<eof> #<undefined>) #f)
(test (eqv? #<eof> '()) #f)
(test (let () (define-macro (hi a) `(+ 1 ,a)) (eqv? hi hi)) #t)
(test (let () (define (hi a) (+ 1 a)) (eqv? hi hi)) #t)
(test (let ((x (lambda* (hi (a 1)) (+ 1 a)))) (eqv? x x)) #t)

(if with-bignums
    (begin
      (test (eqv? (bignum "1+i") (bignum "1+i")) #t)
      (test (eqv? (bignum "1+i") 1+i) #t)
      (test (eqv? 1+i (bignum "1+i")) #t)
      ))



;;; equal?
(test (equal? 'a 3) #f)
(test (equal? #t 't) #f)
(test (equal? "abs" 'abc) #f)
(test (equal? "hi" '(hi)) #f)
(test (equal? "()" '()) #f)
(test (equal? '(1) '(1)) #t)
(test (equal? '(#f) '(#f)) #t)
(test (equal? '(()) '(() . ())) #t)
(test (equal? #\a #\b) #f)
(test (equal? #\a #\a) #t)
(test (let ((x (string-ref "hi" 0))) (equal? x x)) #t)
(test (equal? #t #t) #t)
(test (equal? #f #f) #t)
(test (equal? #f #t) #f)
(test (equal? (null? '()) #t) #t)
(test (equal? (null? '(a)) #f) #t)
(test (equal? (cdr '(a)) '()) #t)
(test (equal? 'a 'a) #t)
(test (equal? 'a 'b) #f)
(test (equal? 'a (string->symbol "a")) #t)
(test (equal? '(a) '(b)) #f)
(test (equal? '(a) '(a)) #t)
(test (let ((x '(a . b))) (equal? x x)) #t)
(test (let ((x (cons 'a 'b))) (equal? x x)) #t)
(test (equal? (cons 'a 'b) (cons 'a 'b)) #t)
(test (equal?(cons 'a 'b)(cons 'a 'b)) #t) ; no space
(test (equal? "abc" "cba") #f)
(test (equal? "abc" "abc") #t)
(test (let ((x "hi")) (equal? x x)) #t)
(test (equal? (string #\h #\i) (string #\h #\i)) #t)
(test (equal? '#(a) '#(b)) #f)
(test (equal? '#(a) '#(a)) #t)
(test (let ((x (vector 'a))) (equal? x x)) #t)
(test (equal? (vector 'a) (vector 'a)) #t)
(test (equal? '#(1 2) (vector 1 2)) #t)
(test (equal? '#(1.0 2/3) (vector 1.0 2/3)) #t)
(test (equal? '#(1 2) (vector 1 2.0)) #f) ; 2 not equal 2.0!
(test (equal? '(1 . 2) (cons 1 2)) #t)
(test (equal? '(1 #||# . #||# 2) (cons 1 2)) #t)
(test (- '#||#1) -1) ; hmm
(test (equal? '#(1 "hi" #\a) (vector 1 "hi" #\a)) #t)
(test (equal? '#((1 . 2)) (vector (cons 1 2))) #t)
(test (equal? '#(1 "hi" #\a (1 . 2)) (vector 1 "hi" #\a (cons 1 2))) #t)
(test (equal? '#(#f hi (1 2) 1 "hi" #\a (1 . 2)) (vector #f 'hi (list 1 2) 1 "hi" #\a (cons 1 2))) #t)
(test (equal? '#(#(1) #(1)) (vector (vector 1) (vector 1))) #t)
(test (equal? '#(()) (vector '())) #t)
(test (equal? '#("hi" "ho") (vector "hi" '"ho")) #t)
(test (equal? `#(1) '#(1)) #t)
(test (equal? ``#(1) #(1)) #t)
(test (equal? '`#(1) #(1)) #t)
(test (equal? ''#(1) #(1)) #f)
(test (equal? ''#(1) '#(1)) #f)
(test (equal? (list 1 "hi" #\a) '(1 "hi" #\a)) #t)
(test (equal? (list 1.0 2/3) '(1.0 2/3)) #t)
(test (equal? (list 1 2) '(1 2.0)) #f)
(test (equal? '#(1.0+1.0i) (vector 1.0+1.0i)) #t)
(test (equal? (list 1.0+1.0i) '(1.0+1.0i)) #t)
(test (equal? '((())) (list (list (list)))) #t)
(test (equal? car car) #t)
(test (equal? car cdr) #f)
(test (let ((x (lambda () 1))) (equal? x x)) #t)
(test (equal? (lambda () 1) (lambda () 1)) #f)
(test (equal? 9/2 9/2) #t)
(test (equal? #((())) #((()))) #t)
(test (equal? "123""123") #t);no space
(test (equal? """") #t)#|nospace|#
(test (equal? #()#()) #t)
(test (equal? #()()) #f)
(test (equal? ()"") #f)
(test (equal? "hi""hi") #t)
(test (equal? #<eof> #<eof>) #t)
(test (equal? #<undefined> #<undefined>) #t)
(test (equal? #<unspecified> #<unspecified>) #t)
(test (equal? (if #f #f) #<unspecified>) #t)
(test (equal? #<eof> #<undefined>) #f)
(test (equal? #<eof> '()) #f)
(test (let () (define-macro (hi a) `(+ 1 ,a)) (equal? hi hi)) #t)
(test (let () (define (hi a) (+ 1 a)) (equal? hi hi)) #t)
(test (let ((x (lambda* (hi (a 1)) (+ 1 a)))) (equal? x x)) #t)
(test (equal? ``"" '"") #t)
(test (let ((pws (make-procedure-with-setter (lambda () 1) (lambda (x) x)))) (equal? pws pws)) #t)
(test (equal? if :if) #f)

(test (equal? most-positive-fixnum most-positive-fixnum) #t)
(test (equal? most-positive-fixnum most-negative-fixnum) #f)
(test (equal? pi pi) #t)
(test (equal? 9223372036854775807 9223372036854775806) #f)
(test (equal? 9223372036854775807 -9223372036854775808) #f)
(test (equal? -9223372036854775808 -9223372036854775808) #t)
(test (equal? 123456789/2 123456789/2) #t)
(test (equal? 123456789/2 123456787/2) #f)
(test (equal? -123456789/2 -123456789/2) #t)
(test (equal? 2/123456789 2/123456789) #t)
(test (equal? -2/123456789 -2/123456789) #t)
(test (equal? 2147483647/2147483646 2147483647/2147483646) #t)
(test (equal? 3/4 12/16) #t)
(test (equal? 1/1 1) #t)
(test (equal? 312689/99532 833719/265381) #f)
(test (let ((x 3.141)) (equal? x x)) #t)
(test (let ((x 1+i)) (equal? x x)) #t)
(test (let* ((x 3.141) (y x)) (equal? x y)) #t)
(test (let* ((x 1+i) (y x)) (equal? x y)) #t)
(test (let* ((x 3/4) (y x)) (equal? x y)) #t)

(test (let ((x 3.141)) (equal? x x)) #t)
(test (equal? 3 3) #t)
(test (equal? 3 3.0) #f)
(test (equal? 3.0 3.0) #t)
(test (equal? 3-4i 3-4i) #t)
(test (equal? (string #\c) "c") #t)
(test (equal? equal? equal?) #t)
(test (equal? (cons 1 (cons 2 3)) '(1 2 . 3)) #t)
(test (equal? '() '()) #t)
(test (equal? '() (list)) #t)
(test (equal? (cdr '   ''0) '((quote 0))) #t)
(test (equal? "\n" "\n") #t)
(test (equal? #f ((lambda () #f))) #t)
(test (equal? (+) 0) #t)
(test (equal? (recompose 32 list '(1)) (recompose 32 list (list 1))) #t)
(test (equal? (recompose 100 list '(1)) (recompose 100 list (list 1))) #t)
(test (equal? (recompose 32 vector 1) (recompose 32 vector 1)) #t)
(test (equal? (reinvert 32 list vector 1) (reinvert 32 list vector 1)) #t)
(test (equal? (recompose 32 (lambda (a) (cons 1 a)) '()) (recompose 32 (lambda (a) (cons 1 a)) '())) #t)
(test (equal? (recompose 32 (lambda (a) (list 1 a)) '()) (recompose 32 (lambda (a) (list 1 a)) '())) #t)

(test (equal? "asd""asd") #t) ; is this the norm?
(let ((streq (lambda (a b) (equal? a b)))) (test (streq "asd""asd") #t))

(let ((things (vector #t #f #\space '() "" 0 1 3/4 1+i 1.5 '(1 .2) '#() (vector 1) (list 1) 'f 't #\t)))
  (do ((i 0 (+ i 1)))
      ((= i (- (vector-length things) 1)))
    (do ((j (+ i 1) (+ j 1)))
	((= j (vector-length things)))
      (if (equal? (vector-ref things i) (vector-ref things j))
	  (format #t ";(equal? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j))))))

(test (equal?) 'error)
(test (equal? #t) 'error)
(test (equal? #t #t #t) 'error)
(test (equal #t #t) 'error)

(test (call-with-exit (lambda (return) (return (equal? return return)))) #t)
(test (call-with-exit (lambda (return) (call-with-exit (lambda (quit) (return (equal? return quit)))))) #f)
(test (call/cc (lambda (return) (return (equal? return return)))) #t)
(test (let hiho ((i 0)) (equal? hiho hiho)) #t)
(test (let hiho ((i 0)) (let hoho ((i 0)) (equal? hiho hoho))) #f)
(test (equal? + *) #f)
(test (equal? lambda lambda) #t)
(test (equal? lambda lambda*) #f)
(test (equal? let let) #t)
(test (equal? let letrec) #f)
(test (equal? define define) #t)
(test (equal? + ((lambda (a) a) +)) #t)
(test (let ((x "hi")) (define (hi) x) (equal? (hi) (hi))) #t)

;; so (eq? 3/4 3/4) is #f, (eqv? 3/4 3/4) is #t,
;;    (eqv? #(1) #(1)) is #f, (equal? #(1) #(1)) is #t
;;    (equal? 3 3.0) is #f, (= 3 3.0) is #t
;; in s7 
;;    (eq? 0.0 0.0) is #t,
;;    (eq? 2.0 2.0) is #f
(test (equal? .0 0.) #t)
(test (equal? 
       (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) (list) (cons 1 2) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>)
       (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) (list) (cons 1 2) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>))
      #t)
(test (equal? 
       (vector "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>)
       (vector "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>))
      #t)
(test (equal? (make-string 3) (make-string 3)) #t)
(test (equal? (make-list 3) (make-list 3)) #t)
(test (equal? (make-vector 3) (make-vector 3)) #t)

(if with-bignums
    (begin
      (test (equal? (/ (* 5 most-positive-fixnum) (* 3 most-negative-fixnum)) -46116860184273879035/27670116110564327424) #t)
      ))


;;; boolean?
(test (boolean? #f) #t)
(test (boolean? #t) #t)
(test (boolean? 0) #f)
(test (boolean? 1) #f)
(test (boolean? "") #f)
(test (boolean? #\0) #f)
(test (boolean? '()) #f)
(test (boolean? '#()) #f)
(test (boolean? 't) #f)
(test (boolean? (list)) #f)
(test ( boolean? #t) #t)
(test (boolean? boolean?) #f)
(test (boolean? or) #f)
(test (   ; a comment 
       boolean?  ;;; and another
       #t
       )
      #t)

(for-each
 (lambda (arg)
   (if (boolean? arg)
       (format #t ";(boolean? ~A) -> #t?~%" arg)))
 (list "hi" '(1 2) (integer->char 65) 1 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) (if #f #f) #<eof> #<undefined>))

(test (recompose 12 boolean? #f) #t)

(test (boolean?) 'error)
(test (boolean? #f #t) 'error)
(test (boolean #f) 'error)
(test (boolean? (lambda (x) #f)) #f)
(test (boolean? and) #f)
(test (boolean? if) #f)
(test (boolean? (values)) #f)
;(test (boolean? else) #f) ; this could also be an error -> unbound variable, like (symbol? else)


;;; not
(test (not #f) #t)
(test (not #t) #f)
(test (not (not #t)) #t)
(test (not 0) #f)
(test (not 1) #f)
(test (not '()) #f)
(test (not 't) #f)
(test (not (list)) #f)
(test (not (list 3)) #f)
(test (not 'nil) #f)
(test (not not) #f)
(test (not "") #f)
(test (not lambda) #f)
(test (not quote) #f)

(for-each
 (lambda (arg)
   (if (not arg)
       (format #t ";(not ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) #<eof> #<undefined> (if #f #f)))

(test (recompose 12 not #f) #f)

(test (not) 'error)
(test (not #f #t) 'error)
(test (not and) #f)
(test (not case) #f)



;;; symbol?
(test (symbol? 't) #t)
(test (symbol? "t") #f)
(test (symbol? '(t)) #f)
(test (symbol? #t) #f)
(test (symbol? 4) #f)
(test (symbol? 'foo) #t)
(test (symbol? (car '(a b))) #t)
(test (symbol? 'nil) #t)
(test (symbol? '()) #f)
(test (symbol? #()) #f)
(test (symbol? #f) #f)
(test (symbol? 'car) #t)
(test (symbol? car) #f)
(test (symbol? '#f) #f)
(test (symbol? #()) #f)
(test (symbol? :hi) #t)
(test (symbol? hi:) #t)
(test (symbol? :hi:) #t)
(test (symbol? ::) #t)
(test (symbol? ':) #t)
(test (symbol? '|) #t)
(test (symbol? '|') #t)
(test (symbol? '@) #t)
;(test (symbol? '#:) #t) ; confusable given guile-style keywords
(test (symbol? #b1) #f)
(test (symbol? 'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789) #t) ;M Gran
(test (symbol? (vector-ref '#(1 a 34) 1)) #t)
(test (if (symbol? '1+) (symbol? '0e) #t) #t)
(test (symbol? 'begin) #t)
(test (symbol? 'if) #t)
(test (symbol? (keyword->symbol :if)) #t)
(test (symbol? (string->symbol "if")) #t)
(test (symbol? if) #f)
(test (symbol? quote) #f)

(for-each
 (lambda (arg)
   (if (symbol? arg)
       (format #t ";(symbol? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) #<eof> #<undefined>))

(test (symbol?) 'error)
(test (symbol? 'hi 'ho) 'error)
(test (symbol? 'hi 3) 'error)
(test (symbol? 3 3) 'error)
(test (symbol? 3 'hi) 'error)
(test (symbol 'hi) 'error) ; symbol takes a string

;;; "Returns #t if obj is a symbol, otherwise returns #f" (r5|6rs.html)
(test (symbol? begin) #f) ; ?? this is an error in Guile, it was #t in s7
(test (symbol? expt) #f)
(test (symbol? if) #f)
(test (symbol? and) #f)
(test (symbol? lambda) #f)
(test (symbol? 'let) #t)
(test (symbol? call/cc) #f)
(test (symbol? '1.2.3) #t)
(test (symbol? '1.2) #f)
(test (symbol? ''1.2) #f)
(test (symbol? '"hi") #f)

(test (let ((sym000000000000000000000 3))
	(let ((sym000000000000000000001 4))
	  (+ sym000000000000000000000 sym000000000000000000001)))
      7)


;;; procedure?
(test (procedure? car) #t)
(test (procedure? procedure?) #t)
(test (procedure? 'car) #f)
(test (procedure? (lambda (x) x)) #t)
(test (procedure? '(lambda (x) x)) #f)
(test (call/cc procedure?) #t) ; ??
(test (let ((a (lambda (x) x)))	(procedure? a)) #t)
(test (letrec ((a (lambda () (procedure? a)))) (a)) #t)
(test (let ((a 1)) (let ((a (lambda () (procedure? a)))) (a))) #f)
(test (let () (define (hi) 1) (procedure? hi)) #t)
(test (let () (define-macro (hi a) `(+ ,a 1)) (procedure? hi)) #f)
(test (procedure? begin) #f)
(test (procedure? lambda) #f)
(test (procedure? (lambda* ((a 1)) a)) #t)
(test (procedure? and) #f)
(test (procedure? 'let) #f)
(test (procedure? (make-procedure-with-setter (lambda () 1) (lambda (x) x))) #t)
(if with-bignums (test (procedure? (bignum "1e100")) #f))
(test (procedure? quasiquote) #f)
(let () (define-macro (hi a) `(+ ,a 1)) (test (procedure? hi) #f))
(test (procedure? (make-random-state 1234)) #f)
(test (procedure? pi) #f)
(test (procedure? cond) #f)
(test (procedure? do) #f)
(test (procedure? set!) #f)

(for-each
 (lambda (arg)
   (if (procedure? arg)
       (format #t ";(procedure? ~A) -> #t?~%" arg)))
 (list "hi" _ht_ (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f #() (if #f #f)))

(test (procedure?) 'error)
(test (procedure? abs car) 'error)
(test (procedure abs) 'error)

;; these are questionable -- an applicable object is a procedure
(test (procedure? "hi") #f)
(test (procedure? '(1 2)) #f)
(test (procedure? #(1 2)) #f)





;;; --------------------------------------------------------------------------------
;;; CHARACTERS
;;; --------------------------------------------------------------------------------

(test (eqv? '#\  #\space) #t)
(test (eqv? #\newline '#\newline) #t)

;;; char?
(test (char? #\a) #t)
(test (char? #\() #t)
(test (char? #\space) #t)
(test (char? '#\newline) #t)
(test (char? #\1) #t)
(test (char? #\$) #t)
(test (char? #\.) #t)
(test (char? #\\) #t)
(test (char? #\)) #t)
(test (char? #\%) #t)
(test (char? '#\space) #t)
(test (char? '#\ ) #t)
(test (char? '#\newline) #t)
(test (char? '#\a) #t)
(test (char? '#\8) #t)
(test (char? #\-) #t)
(test (char? #\n) #t)
(test (char? #\() #t)
(test (char? #e1) #f)
(test (char? #\#) #t)
(test (char? #\x) #t)
(test (char? #\o) #t)
(test (char? #\b) #t)
(test (char? #b101) #f)
(test (char? #o73) #f)
(test (char? #x73) #f)
(test (char? 'a) #f)
(test (char? 97) #f)
(test (char? "a") #f)
(test (char? (string-ref "hi" 0)) #t)
(test (char? (string-ref (make-string 1) 0)) #t)
(test (char? #\") #t)
(test (char? #\') #t)
(test (char? #\`) #t)
(test (char? #\@) #t)
(test (char? #<eof>) #f)
(test (char? '1e311) #f)

(for-each
 (lambda (arg)
   (if (char? arg)
       (format #t ";(char? ~A) -> #t?~%" arg)))
 (list "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #f #t (if #f #f) (lambda (a) (+ a 1))))

(test (char? begin) #f)

(do ((i 0 (+ i 1)))
    ((= i 256))
  (if (not (char? (integer->char i)))
      (format #t ";(char? (integer->char ~A)) -> #f?~%" i)))

(test (char?) 'error)
(test (char? #\a #\b) 'error)
(test (char #\a) 'error)

(test (char? #\x65) #t)
(test (char? #\x000000000065) #t)
(test (char? #\x0) #t)
(test (char=? #\x000 #\null) #t)
(test (char=? #\x08 #\x8) #t)
(test (char=? #\x0e #\xe) #t) ; Guile thinks both of these names are bogus
(test (char=? #\x00e #\xe) #t)
(test (char=? #\x0000e #\xe) #t)
(test (char=? #\x00000000e #\xe) #t) ; hmmm -- surely this is a bug
(test (char? #\xff) #t)
;; any larger number is a reader error

(test (eval-string "(char? #\\x#b0)") 'error)

(test (eval-string "(char? #\\100)") 'error)
(test (eval-string "(char? #\\x-65)") 'error)
(test (eval-string "(char? #\\x6.5)") 'error)
(test (eval-string "(char? #\\x6/5)") 'error)
(test (eval-string "(char? #\\x6/3)") 'error)
(test (eval-string "(char? #\\x6+i)") 'error)
(test (eval-string "(char? #\\x6asd)") 'error)
(test (eval-string "(char? #\\x6#)") 'error)
(test (eval-string "(char? #\\x#b0") 'error)
(test (eval-string "(char? #\\x#e0.0") 'error)
(test (eval-string "(char? #\\x-0") 'error)
(test (eval-string "(char? #\\x#e0e100") 'error)
(test (eval-string "(char? #\\x1.4") 'error)

(test (char=? #\x6a #\j) #t)

(test (char? #\return) #t)
(test (char? #\null) #t)
(test (char? #\nul) #t)
(test (char? #\linefeed) #t)
(test (char? #\tab) #t)
(test (char? #\space) #t)
(test (char=? #\null #\nul) #t)
(test (char=? #\newline #\linefeed) #t)
(test (char=? #\return #\xd) #t)
(test (char=? #\nul #\x0) #t)
;(test (char? #\ÿ) #t) ; this seems to involve unwanted translations in emacs?
(test (eval-string (string-append "(char? " (format #f "#\\~C" (integer->char 255)) ")")) #t)
(test (eval-string (string-append "(char? " (format #f "#\\~C" (integer->char 127)) ")")) #t)
(test (apply char? (list (integer->char 255))) #t)

(num-test (let ((str (make-string 258 #\space)))
	    (do ((i 1 (+ i 1)))
		((= i 256))
	      (string-set! str i (integer->char i)))
	    (string-set! str 257 (integer->char 0))
	    (string-length str))
	  258)


(let ((a-to-z (list #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\x #\y #\z))
      (cap-a-to-z (list #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\X #\Y #\Z))
      (mixed-a-to-z (list #\a #\B #\c #\D #\e #\F #\g #\H #\I #\j #\K #\L #\m #\n #\O #\p #\Q #\R #\s #\t #\U #\v #\X #\y #\Z))
      (digits (list #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
  
;;; char-upper-case?
  (test (char-upper-case? #\a) #f)
  (test (char-upper-case? #\A) #t)
  
  (for-each
   (lambda (arg)
     (if (not (char-upper-case? arg))
	 (format #t ";(char-upper-case? ~A) -> #f?~%" arg)))
   cap-a-to-z)
  
  (for-each
   (lambda (arg)
     (if (char-upper-case? arg)
	 (format #t ";(char-upper-case? ~A) -> #t?~%" arg)))
   a-to-z)
  
  ;; non-alpha chars are "unspecified" here
  
  (test (char-upper-case? 1) 'error)
  (test (char-upper-case?) 'error)
  (test (char-upper-case? 1) 'error)
  (test (char-upper-case?) 'error)
  (test (char-upper-case? #\a #\b) 'error)
  (test (char-upper-case #\a) 'error)

  
;;; char-lower-case?
  (test (char-lower-case? #\A) #f)
  (test (char-lower-case? #\a) #t)
  
  (for-each
   (lambda (arg)
     (if (not (char-lower-case? arg))
	 (format #t ";(char-lower-case? ~A) -> #f?~%" arg)))
   a-to-z)
  
  (for-each
   (lambda (arg)
     (if (char-lower-case? arg)
	 (format #t ";(char-lower-case? ~A) -> #t?~%" arg)))
   cap-a-to-z)
  
  (test (char-lower-case? 1) 'error)
  (test (char-lower-case?) 'error)
  (test (char-lower-case? 1) 'error)
  (test (char-lower-case?) 'error)
  (test (char-lower-case? #\a #\b) 'error)
  (test (char-lower-case #\a) 'error)

;;  (test (char-lower-case? #\xb5) #t)  ; what is this?  in Snd it's #t, in ex1 it's #f -- is this a locale choice?
  (test (char-lower-case? #\xb6) #f)

  (for-each
   (lambda (c)
     (test (and (not (char-upper-case? c)) 
		(not (char-lower-case? c))) #t))
   (map integer->char (list 0 1 2 3 32 33 34 170 182 247)))


  
;;; char-upcase
  (test (char-upcase #\A) #\A)
  (test (char-upcase #\a) #\A)
  (test (char-upcase #\?) #\?)
  (test (char-upcase #\$) #\$)
  (test (char-upcase #\.) #\.)
  (test (char-upcase #\\) #\\)
  (test (char-upcase #\5) #\5)
  (test (char-upcase #\)) #\))
  (test (char-upcase #\%) #\%)
  (test (char-upcase #\0) #\0)
  (test (char-upcase #\_) #\_)
  (test (char-upcase #\space) #\space)
  (test (char-upcase #\newline) #\newline)
  (test (char-upcase #\null) #\null)
  
  (for-each
   (lambda (arg1 arg2)
     (if (not (char=? (char-upcase arg1) arg2))
	 (format #t ";(char-upcase ~A) != ~A?~%" arg1 arg2)))
   a-to-z
   cap-a-to-z)
  
  (do ((i 1 (+ i 1)))
      ((= i 256))
    (if (and (not (char=? (integer->char i) (char-upcase (integer->char i))))
	     (not (char-alphabetic? (integer->char i))))
	(format #t ";(char-upcase ~A) -> ~A but not alphabetic?~%" (integer->char i) (char-upcase (integer->char i)))))

  (test (recompose 12 char-upcase #\a) #\A)
  (test (reinvert 12 char-upcase char-downcase #\a) #\a)

  (test (char-upcase) 'error)
  (test (char-upcase #\a #\b) 'error)
  (test (char-upcase #<eof>) 'error)
  (test (char-upcase #f) 'error)
  (test (char-upcase (list)) 'error)


  
;;; char-downcase
  (test (char-downcase #\A) #\a)
  (test (char-downcase #\a) #\a)
  (test (char-downcase #\?) #\?)
  (test (char-downcase #\$) #\$)
  (test (char-downcase #\.) #\.)
  (test (char-downcase #\_) #\_)
  (test (char-downcase #\\) #\\)
  (test (char-downcase #\5) #\5)
  (test (char-downcase #\)) #\))
  (test (char-downcase #\%) #\%)
  (test (char-downcase #\0) #\0)
  (test (char-downcase #\space) #\space)
  
  (for-each
   (lambda (arg1 arg2)
     (if (not (char=? (char-downcase arg1) arg2))
	 (format #t ";(char-downcase ~A) != ~A?~%" arg1 arg2)))
   cap-a-to-z
   a-to-z)

  (test (recompose 12 char-downcase #\A) #\a)

  (test (char-downcase) 'error)
  (test (char-downcase #\a #\b) 'error)  


;;; char-numeric?  
  (test (char-numeric? #\a) #f)
  (test (char-numeric? #\5) #t)
  (test (char-numeric? #\A) #f)
  (test (char-numeric? #\z) #f)
  (test (char-numeric? #\Z) #f)
  (test (char-numeric? #\0) #t)
  (test (char-numeric? #\9) #t)
  (test (char-numeric? #\space) #f)
  (test (char-numeric? #\;) #f)
  (test (char-numeric? #\.) #f)
  (test (char-numeric? #\-) #f)
  (test (char-numeric? (integer->char 200)) #f)
  (test (char-numeric? (integer->char 128)) #f)
  (test (char-numeric? (integer->char 216)) #f) ; 0 slash
  (test (char-numeric? (integer->char 189)) #f) ; 1/2
  
  (for-each
   (lambda (arg)
     (if (char-numeric? arg)
	 (format #t ";(char-numeric? ~A) -> #t?~%" arg)))
   cap-a-to-z)
  
  (for-each
   (lambda (arg)
     (if (char-numeric? arg)
	 (format #t ";(char-numeric? ~A) -> #t?~%" arg)))
   a-to-z)

  (test (char-numeric?) 'error)
  (test (char-numeric? #\a #\b) 'error)  

  
;;; char-whitespace?
  (test (char-whitespace? #\a) #f)
  (test (char-whitespace? #\A) #f)
  (test (char-whitespace? #\z) #f)
  (test (char-whitespace? #\Z) #f)
  (test (char-whitespace? #\0) #f)
  (test (char-whitespace? #\9) #f)
  (test (char-whitespace? #\space) #t)
  (test (char-whitespace? #\tab) #t)
  (test (char-whitespace? #\newline) #t)
  (test (char-whitespace? #\return) #t)
  (test (char-whitespace? #\linefeed) #t)
  (test (char-whitespace? #\null) #f)
  (test (char-whitespace? #\;) #f)
  (test (char-whitespace? #\xb) #t)
  (test (char-whitespace? #\x0b) #t)
  (test (char-whitespace? #\xc) #t)
  (test (char-whitespace? #\xd) #t) ; #\return
  (test (char-whitespace? #\xe) #f) 

  (for-each
   (lambda (arg)
     (if (char-whitespace? arg)
	 (format #t ";(char-whitespace? ~A) -> #t?~%" arg)))
   mixed-a-to-z)
  
  (for-each
   (lambda (arg)
     (if (char-whitespace? arg)
	 (format #t ";(char-whitespace? ~A) -> #t?~%" arg)))
   digits)

  (test (char-whitespace?) 'error)
  (test (char-whitespace? #\a #\b) 'error)   
 
  
;;; char-alphabetic?
  (test (char-alphabetic? #\a) #t)
  (test (char-alphabetic? #\$) #f)
  (test (char-alphabetic? #\A) #t)
  (test (char-alphabetic? #\z) #t)
  (test (char-alphabetic? #\Z) #t)
  (test (char-alphabetic? #\0) #f)
  (test (char-alphabetic? #\9) #f)
  (test (char-alphabetic? #\space) #f)
  (test (char-alphabetic? #\;) #f)
  (test (char-alphabetic? #\.) #f)
  (test (char-alphabetic? #\-) #f)
  (test (char-alphabetic? #\_) #f)
  (test (char-alphabetic? #\^) #f)
  (test (char-alphabetic? #\[) #f)

  ;(test (char-alphabetic? (integer->char 200)) #t) ; ??
  (test (char-alphabetic? (integer->char 127)) #f)  ; backspace
  
  (for-each
   (lambda (arg)
     (if (char-alphabetic? arg)
	 (format #t ";(char-alphabetic? ~A) -> #t?~%" arg)))
   digits)
  
  (for-each
   (lambda (arg)
     (if (not (char-alphabetic? arg))
	 (format #t ";(char-alphabetic? ~A) -> #f?~%" arg)))
   mixed-a-to-z)

  (test (char-alphabetic?) 'error)
  (test (char-alphabetic? #\a #\b) 'error)  

  (for-each
   (lambda (op)
     (for-each
      (lambda (arg)
	(test (op arg) 'error))
      (list "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
	    3.14 3/4 1.0+1.0i #f #t (if #f #f) (lambda (a) (+ a 1)))))
   (list char-upper-case? char-lower-case? char-upcase char-downcase char-numeric? char-whitespace? char-alphabetic?))


  
  (test 
   (let ((unhappy '()))
     (do ((i 0 (+ i 1))) 
	 ((= i 256)) 
       (let* ((ch (integer->char i))
	      (chu (char-upcase ch))
	      (chd (char-downcase ch)))
	 (if (or (and (not (char=? ch chu))
		      (not (char=? ch (char-downcase chu))))
		 (and (not (char=? ch chd))
		      (not (char=? ch (char-upcase chd))))
		 (and (not (char=? ch chd))
		      (not (char=? ch chu)))
		 (not (char-ci=? chu chd))
		 (not (char-ci=? ch chu))
		 (and (char-alphabetic? ch)
		      (or (not (char-alphabetic? chd))
			  (not (char-alphabetic? chu))))
		 (and (char-numeric? ch)
		      (or (not (char-numeric? chd))
			  (not (char-numeric? chu))))
		 (and (char-whitespace? ch)
		      (or (not (char-whitespace? chd))
			  (not (char-whitespace? chu))))
		 (and (char-alphabetic? ch)
		      (char-whitespace? ch))
		 (and (char-numeric? ch)
		      (char-whitespace? ch))
		 (and (char-alphabetic? ch)
		      (char-numeric? ch)))
	     ;; there are characters that are alphabetic but the result of char-upcase is not an upper-case character
	     ;; 223 for example, or 186 for lower case
	     (set! unhappy (cons (format #f "~C: ~C ~C (~D)~%" ch chu chd i) unhappy)))))
     unhappy)
   '())
  

  
  (for-each
   (lambda (op)
     (for-each
      (lambda (arg)
	(test (op #\a arg) 'error))
      (list "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
	    3.14 3/4 1.0+1.0i #f #t (if #f #f) (lambda (a) (+ a 1)))))
   (list char=? char<? char<=? char>? char>=? char-ci=? char-ci<? char-ci<=? char-ci>? char-ci>=?))

  (for-each
   (lambda (op)
     (for-each
      (lambda (arg)
	(test (op arg #\a) 'error))
      (list "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
	    3.14 3/4 1.0+1.0i #f #t (if #f #f) (lambda (a) (+ a 1)))))
   (list char=? char<? char<=? char>? char>=? char-ci=? char-ci<? char-ci<=? char-ci>? char-ci>=?))


;;; char=?
  (test (char=? #\d #\d) #t)
  (test (char=? #\A #\a) #f)
  (test (char=? #\d #\x) #f)
  (test (char=? #\d #\D) #f)
  (test (char=? #\a #\a) #t)
  (test (char=? #\A #\B) #f)
  (test (char=? #\a #\b) #f)
  (test (char=? #\9 #\0) #f)
  (test (char=? #\A #\A) #t)
  (test (char=? #\  #\space) #t)
  (let ((i (char->integer #\space)))
    (test (char=? (integer->char i) #\space) #t))
  (test (char=? (integer->char (char->integer #\")) #\") #t)
  (test (char=? #\x65 #\e) #t)
  
  (test (char=? #\d #\d #\d #\d) #t)
  (test (char=? #\d #\d #\x #\d) #f)
  (test (char=? #\d #\y #\x #\c) #f)
  (test (apply char=? cap-a-to-z) #f)
  (test (apply char=? mixed-a-to-z) #f)
  (test (apply char=? digits) #f)
  (test (char=? #\d #\c #\d) #f)

  (test (char=? #\a) 'error)
  (test (char=?) 'error)
  (test (char=? #\a 0) 'error)
  (test (char=? #\a #\b 0) 'error)
  

;;; char<?  
  (test (char<? #\z #\0) #f)
  (test (char<? #\d #\x) #t)
  (test (char<? #\d #\d) #f)
  (test (char<? #\d #\x) #t)
  (test (char<? #\A #\B) #t)
  (test (char<? #\a #\b) #t)
  (test (char<? #\9 #\0) #f)
  (test (char<? #\A #\A) #f)
  (test (char<? #\space #\space) #f)
  
  (test (char<? #\a #\e #\y #\z) #t)
  (test (char<? #\a #\e #\e #\y) #f)
  (test (apply char<? a-to-z) #t)
  (test (apply char<? cap-a-to-z) #t)
  (test (apply char<? mixed-a-to-z) #f)
  (test (apply char<? digits) #t)
  (test (apply char<? (reverse a-to-z)) #f)
  (test (apply char<? (reverse cap-a-to-z)) #f)
  (test (apply char<? (reverse mixed-a-to-z)) #f)
  (test (apply char<? (reverse digits)) #f)
  (test (char<? #\b #\c #\a) #f)
  (test (char<? #\B #\B #\A) #f)
  (test (char<? #\b #\c #\e) #t)
  (test (char<? (integer->char #xf0) (integer->char #x70)) #f)

  (test (char<?) 'error)
  (test (char<? #\b #\a "hi") 'error)
  (test (char<? #\b #\a 0) 'error)
  (test (char<? (integer->char 0) (integer->char 255)) #t)
  
  

;;; char<=?
  (test (char<=? #\d #\x) #t)
  (test (char<=? #\d #\d) #t)
  
  (test (char<=? #\a #\e #\y #\z) #t)
  (test (char<=? #\a #\e #\e #\y) #t)
  (test (char<=? #\A #\B) #t)
  (test (char<=? #\a #\b) #t)
  (test (char<=? #\9 #\0) #f)
  (test (char<=? #\A #\A) #t)
  (test (char<=? #\space #\space) #t)
  
  (test (char<=? #\a #\e #\y #\z) #t)
  (test (char<=? #\a #\e #\e #\y) #t)
  (test (char<=? #\e #\e #\d #\y) #f)
  (test (apply char<=? a-to-z) #t)
  (test (apply char<=? cap-a-to-z) #t)
  (test (apply char<=? mixed-a-to-z) #f)
  (test (apply char<=? digits) #t)
  (test (apply char<=? (reverse a-to-z)) #f)
  (test (apply char<=? (reverse cap-a-to-z)) #f)
  (test (apply char<=? (reverse mixed-a-to-z)) #f)
  (test (apply char<=? (reverse digits)) #f)
  (test (char<=? #\b #\c #\a) #f)
  (test (char<=? #\B #\B #\A) #f)
  (test (char<=? #\b #\c #\e) #t)
  
  (test (char<=? #\b #\a "hi") 'error)
  (test (char<=? #\b #\a 0) 'error)
  (test (char<=?) 'error)


  
;;; char>?
  (test (char>? #\e #\d) #t)
  (test (char>? #\z #\a) #t)
  (test (char>? #\A #\B) #f)
  (test (char>? #\a #\b) #f)
  (test (char>? #\9 #\0) #t)
  (test (char>? #\A #\A) #f)
  (test (char>? #\space #\space) #f)
  
  (test (char>? #\d #\c #\b #\a) #t)
  (test (char>? #\d #\d #\c #\a) #f)
  (test (char>? #\e #\d #\b #\c #\a) #f)
  (test (apply char>? a-to-z) #f)
  (test (apply char>? cap-a-to-z) #f)
  (test (apply char>? mixed-a-to-z) #f)
  (test (apply char>? digits) #f)
  (test (apply char>? (reverse a-to-z)) #t)
  (test (apply char>? (reverse cap-a-to-z)) #t)
  (test (apply char>? (reverse mixed-a-to-z)) #f)
  (test (apply char>? (reverse digits)) #t)
  (test (char>? #\d #\c #\a) #t)
  (test (char>? #\d #\c #\c) #f)
  (test (char>? #\B #\B #\C) #f)
  (test (char>? #\b #\c #\e) #f)
  (test (char>? (integer->char #xf0) (integer->char #x70)) #t)

  (test (char>? #\a #\b "hi") 'error)
  (test (char>? #\a #\b 0) 'error)
  (test (char>?) 'error)

  
  
;;; char>=?
  (test (char>=? #\e #\d) #t)
  (test (char>=? #\A #\B) #f)
  (test (char>=? #\a #\b) #f)
  (test (char>=? #\9 #\0) #t)
  (test (char>=? #\A #\A) #t)
  (test (char>=? #\space #\space) #t)
  
  (test (char>=? #\d #\c #\b #\a) #t)
  (test (char>=? #\d #\d #\c #\a) #t)
  (test (char>=? #\e #\d #\b #\c #\a) #f)
  (test (apply char>=? a-to-z) #f)
  (test (apply char>=? cap-a-to-z) #f)
  (test (apply char>=? mixed-a-to-z) #f)
  (test (apply char>=? digits) #f)
  (test (apply char>=? (reverse a-to-z)) #t)
  (test (apply char>=? (reverse cap-a-to-z)) #t)
  (test (apply char>=? (reverse mixed-a-to-z)) #f)
  (test (apply char>=? (reverse digits)) #t)
  (test (char>=? #\d #\c #\a) #t)
  (test (char>=? #\d #\c #\c) #t)
  (test (char>=? #\B #\B #\C) #f)
  (test (char>=? #\b #\c #\e) #f)

  (test (char>=? #\a #\b "hi") 'error)
  (test (char>=? #\a #\b 0) 'error)
  (test (char>=?) 'error)

  
  
;;; char-ci=?
  (test (char-ci=? #\A #\B) #f)
  (test (char-ci=? #\a #\B) #f)
  (test (char-ci=? #\A #\b) #f)
  (test (char-ci=? #\a #\b) #f)
  (test (char-ci=? #\9 #\0) #f)
  (test (char-ci=? #\A #\A) #t)
  (test (char-ci=? #\A #\a) #t)
  (test (char-ci=? #\a #\A) #t)
  (test (char-ci=? #\space #\space) #t)
  
  (test (char-ci=? #\d #\D #\d #\d) #t)
  (test (char-ci=? #\d #\d #\X #\d) #f)
  (test (char-ci=? #\d #\Y #\x #\c) #f)
  (test (apply char-ci=? cap-a-to-z) #f)
  (test (apply char-ci=? mixed-a-to-z) #f)
  (test (apply char-ci=? digits) #f)
  (test (char-ci=? #\d #\c #\d) #f)

  (test (char-ci=?) 'error)
  (test (char-ci=? #\a #\b 0) 'error)
  

  
;;; char-ci<?
  (test (char-ci<? #\A #\B) #t)
  (test (char-ci<? #\a #\B) #t)
  (test (char-ci<? #\A #\b) #t)
  (test (char-ci<? #\a #\b) #t)
  (test (char-ci<? #\9 #\0) #f)
  (test (char-ci<? #\0 #\9) #t)
  (test (char-ci<? #\A #\A) #f)
  (test (char-ci<? #\A #\a) #f)
  (test (char-ci<? #\Y #\_) #t)
  (test (char-ci<? #\\ #\J) #f)
  (test (char-ci<? #\_ #\e) #f)
  (test (char-ci<? #\t #\_) #t)
  (test (char-ci<? #\a #\]) #t)
  (test (char-ci<? #\z #\^) #t)
  
  (test (char-ci<? #\b #\a "hi") 'error)
  (test (char-ci<? #\b #\a 0) 'error)
  (test (char-ci>? (integer->char #xf0) (integer->char #x70)) #t)

#|
;;; this tries them all:
  (do ((i 0 (+ i 1)))
      ((= i 256))
    (do ((k 0 (+ k 1)))
	((= k 256))
      (let ((c1 (integer->char i))
	    (c2 (integer->char k)))
	(for-each
	 (lambda (op1 op2)
	   (if (not (eq? (op1 c1 c2) (op2 (string c1) (string c2))))
	       (format #t ";(~A|~A ~A ~A) -> ~A|~A~%" op1 op2 c1 c2 (op1 c1 c2) (op2 (string c1) (string c2)))))
	 (list char=? char<? char<=? char>? char>=? char-ci=? char-ci<? char-ci<=? char-ci>? char-ci>=?)
	 (list string=? string<? string<=? string>? string>=? string-ci=? string-ci<? string-ci<=? string-ci>? string-ci>=?)))))
|#
  
  (test (char-ci<? #\d #\D #\d #\d) #f)
  (test (char-ci<? #\d #\d #\X #\d) #f)
  (test (char-ci<? #\d #\Y #\x #\c) #f)
  (test (apply char-ci<? cap-a-to-z) #t)
  (test (apply char-ci<? mixed-a-to-z) #t)
  (test (apply char-ci<? digits) #t)
  (test (char-ci<? #\d #\c #\d) #f)
  (test (char-ci<? #\b #\c #\a) #f)
  (test (char-ci<? #\b #\C #\e) #t)
  (test (char-ci<? #\3 #\? #\Z #\[) #t)
  
  (test (char-ci>? #\a #\b "hi") 'error)
  (test (char-ci>? #\a #\b 0) 'error)


  
;;; char-ci>?
  (test (char-ci>? #\A #\B) #f)
  (test (char-ci>? #\a #\B) #f)
  (test (char-ci>? #\A #\b) #f)
  (test (char-ci>? #\a #\b) #f)
  (test (char-ci>? #\9 #\0) #t)
  (test (char-ci>? #\A #\A) #f)
  (test (char-ci>? #\A #\a) #f)
  (test (char-ci>? #\^ #\a) #t)
  (test (char-ci>? #\_ #\e) #t)
  (test (char-ci>? #\[ #\S) #t)
  (test (char-ci>? #\\ #\l) #t)
  (test (char-ci>? #\t #\_) #f)
  (test (char-ci>? #\a #\]) #f)
  (test (char-ci>? #\z #\^) #f)
  (test (char-ci>? #\] #\X) #t)
  
  (test (char-ci>? #\d #\D #\d #\d) #f)
  (test (char-ci>? #\d #\d #\X #\d) #f)
  (test (char-ci>? #\d #\Y #\x #\c) #f)
  (test (apply char-ci>? cap-a-to-z) #f)
  (test (apply char-ci>? mixed-a-to-z) #f)
  (test (apply char-ci>? (reverse mixed-a-to-z)) #t)
  (test (apply char-ci>? digits) #f)
  (test (char-ci>? #\d #\c #\d) #f)
  (test (char-ci>? #\b #\c #\a) #f)
  (test (char-ci>? #\d #\C #\a) #t)
  
  
;;; char-ci<=?
  (test (char-ci<=? #\A #\B) #t)
  (test (char-ci<=? #\a #\B) #t)
  (test (char-ci<=? #\A #\b) #t)
  (test (char-ci<=? #\a #\b) #t)
  (test (char-ci<=? #\9 #\0) #f)
  (test (char-ci<=? #\A #\A) #t)
  (test (char-ci<=? #\A #\a) #t)
  (test (char-ci<=? #\` #\H) #f)
  (test (char-ci<=? #\[ #\m) #f)
  (test (char-ci<=? #\j #\`) #t)
  (test (char-ci<=? #\\ #\E) #f)
  (test (char-ci<=? #\t #\_) #t)
  (test (char-ci<=? #\a #\]) #t)
  (test (char-ci<=? #\z #\^) #t)
  
  (test (char-ci<=? #\d #\D #\d #\d) #t)
  (test (char-ci<=? #\d #\d #\X #\d) #f)
  (test (char-ci<=? #\d #\Y #\x #\c) #f)
  (test (apply char-ci<=? cap-a-to-z) #t)
  (test (apply char-ci<=? mixed-a-to-z) #t)
  (test (apply char-ci<=? digits) #t)
  (test (char-ci<=? #\d #\c #\d) #f)
  (test (char-ci<=? #\b #\c #\a) #f)
  (test (char-ci<=? #\b #\c #\C) #t)
  (test (char-ci<=? #\b #\C #\e) #t)
  
  (test (char-ci<=? #\b #\a "hi") 'error)
  (test (char-ci<=? #\b #\a 0) 'error)


  
;;; char-ci>=?
  (test (char-ci>=? #\A #\B) #f)
  (test (char-ci>=? #\a #\B) #f)
  (test (char-ci>=? #\A #\b) #f)
  (test (char-ci>=? #\a #\b) #f)
  (test (char-ci>=? #\9 #\0) #t)
  (test (char-ci>=? #\A #\A) #t)
  (test (char-ci>=? #\A #\a) #t)
  (test (char-ci>=? #\Y #\_) #f)
  (test (char-ci>=? #\` #\S) #t)
  (test (char-ci>=? #\[ #\Y) #t)
  (test (char-ci>=? #\t #\_) #f)
  (test (char-ci>=? #\a #\]) #f)
  (test (char-ci>=? #\z #\^) #f)
  
  (test (char-ci>=? #\d #\D #\d #\d) #t)
  (test (char-ci>=? #\d #\d #\X #\d) #f)
  (test (char-ci>=? #\d #\Y #\x #\c) #f)
  (test (apply char-ci>=? cap-a-to-z) #f)
  (test (apply char-ci>=? mixed-a-to-z) #f)
  (test (apply char-ci>=? (reverse mixed-a-to-z)) #t)
  (test (apply char-ci>=? (reverse mixed-a-to-z)) #t)
  (test (apply char-ci>=? digits) #f)
  (test (char-ci>=? #\d #\c #\d) #f)
  (test (char-ci>=? #\b #\c #\a) #f)
  (test (char-ci>=? #\d #\D #\a) #t)
  (test (char-ci>=? #\\ #\J #\+) #t)

  (test (char-ci>=? #\a #\b "hi") 'error)
  (test (char-ci>=? #\a #\b 0) 'error)

  ) ; end let with a-to-z



;;; integer->char
;;; char->integer
(test (integer->char (char->integer #\.)) #\.)
(test (integer->char (char->integer #\A)) #\A)
(test (integer->char (char->integer #\a)) #\a)
(test (integer->char (char->integer #\space)) #\space)
(test (char->integer (integer->char #xf0)) #xf0)

(do ((i 0 (+ i 1)))
    ((= i 256)) 
  (if (not (= (char->integer (integer->char i)) i)) 
      (format #t ";char->integer ~D ~A != ~A~%" i (integer->char i) (char->integer (integer->char i)))))

(test (reinvert 12 integer->char char->integer 60) 60)

(test (char->integer 33) 'error)
(test (char->integer) 'error)
(test (integer->char) 'error)
(test (integer->char (expt 2 31)) 'error)
(test (integer->char (expt 2 32)) 'error)
(test (integer->char 12 14) 'error)
(test (char->integer #\a #\b) 'error)
;(test (char->integer #\ÿ) 255) ; emacs confusion?
(test (eval-string (string-append "(char->integer " (format #f "#\\~C" (integer->char 255)) ")")) 255)

(for-each
 (lambda (arg)
   (test (char->integer arg) 'error))
 (list -1 1 0 123456789 "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (integer->char arg) 'error))
 (list -1 257 123456789 -123456789 #\a "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (#\a) 'error)
(test (#\newline 1) 'error)




;;; --------------------------------------------------------------------------------
;;; STRINGS
;;; --------------------------------------------------------------------------------

;;; string?
(test (string? "abc") #t)
(test (string? ':+*/-) #f)
(test (string? "das ist einer der teststrings") #t)
(test (string? '(das ist natuerlich falsch)) #f)
(test (string? "aaaaaa") #t)
(test (string? #\a) #f)
(test (string? "\"\\\"") #t)
(test (string? lambda) #f)
(test (string? format) #f)

(for-each
 (lambda (arg)
   (test (string? arg) #f))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (string?) 'error)
(test (string? "hi" "ho") 'error)
(test (string? #\null) #f)



;;; string=?
(test (string=? "foo" "foo") #t)
(test (string=? "foo" "FOO") #f)
(test (string=? "foo" "bar") #f)
(test (string=? "FOO" "FOO") #t)
(test (string=? "A" "B") #f)
(test (string=? "a" "b") #f)
(test (string=? "9" "0") #f)
(test (string=? "A" "A") #t)
(test (string=? "" "") #t)
(test (string=? (string #\newline) (string #\newline)) #t)

(test (string=? "A" "B" "a") #f)
(test (string=? "A" "A" "a") #f)
(test (string=? "A" "A" "A") #t)
(test (string=? "foo" "foo" "foo") #t)
(test (string=? "foo" "foo" "") #f)
(test (string=? "foo" "foo" "fOo") #f)

(test (string=? "foo" "FOO" 1.0) 'error)

(test (let ((str (string #\" #\1 #\\ #\2 #\")))	(string=? str "\"1\\2\"")) #t)
(test (let ((str (string #\\ #\\ #\\)))	(string=? str "\\\\\\")) #t)
(test (let ((str (string #\")))	(string=? str "\"")) #t)
(test (let ((str (string #\\ #\"))) (string=? str "\\\"")) #t)
(test (let ((str (string #\space #\? #\)))) (string=? str " ?)")) #t)
(test (let ((str (string #\# #\\ #\t))) (string=? str "#\\t")) #t)
(test (string=? (string #\x (integer->char #xf0) #\x) (string #\x (integer->char #x70) #\x)) #f)
(test (string=? (string #\x (integer->char #xf0) #\x) (string #\x (integer->char #xf0) #\x)) #t)

(test (string=? (string) "") #t)
(test (string=? (string) (make-string 0)) #t)
(test (string=? (string-copy (string)) (make-string 0)) #t)
(test (string=? "" (make-string 0)) #t)
(test (string=? "" (string-append)) #t)
(test (string=? (string #\space #\newline) " \n") #t)

(test (string=? "......" "...\ ...") #t)
(test (string=? "......" "...\
...") #t)
(test (string=? "" "\ \ \ \ \ \ \ ") #t)
(test (string=? "\n" (string #\newline)) #t)
(test (string=? "\
\
\
\
" "") #t)
(test (string=? "" (string #\null)) #f)
(test (string=? (string #\null #\null) (string #\null)) #f)
(test (string=? "" "asd") #f)
(test (string=? "asd" "") #f)
(test (string=? "xx" (make-string 2 #\x) (string #\x #\x) (list->string (list #\x #\x)) (substring "axxb" 1 3) (string-append "x" "x")) #t)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "124")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2)) #f)
(test "\x3012" "012")

(for-each
 (lambda (arg)
   (test (string=? "hi" arg) 'error)
   (test (string=? arg "hi") 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))


;; this strikes me as highly dubious
(test (call-with-input-string "1\n2" (lambda (p) (read p))) 1)
(test (call-with-input-string "1\\ \n2" (lambda (p) (read p))) (symbol "1\\"))
(test (call-with-input-string "1\ 2" (lambda (p) (read p))) 12)
(test (call-with-input-string "1\ \ \ 2" (lambda (p) (read p))) 12)

(test (call-with-input-string "1\
2" (lambda (p) (read p))) 12)

(test (call-with-input-string "1\ \
2" (lambda (p) (read p))) 12)

(test (let ((xyzzy 32)) (call-with-input-string "xy\
zzy" (lambda (p) (read p)))) 'xyzzy)

(test (let ((xyzzy 32)) (call-with-input-string "xy\
zzy" (lambda (p) (eval (read p))))) 32)

(test (let ((xyzzy 32)) (call-with-input-string "(set! xyzzy;\
 this is presumably a comment
 321)" (lambda (p) (eval (read p)))) xyzzy) 321)

(test (let ((xyzzy 32)) (call-with-input-string "(set! xyzzy;\
 this is presumably a comment;\
 and more commentary
 321)" (lambda (p) (eval (read p)))) xyzzy) 321)

(test (eval-string "1\ 2") 12)
(test (length "\ \ \ \ \ \ \ ") 0)
(test (eval-string "(length \"\\ 1\")") 1)
(test (eval-string "\"\\ \x30\\ \"") "0") ; (integer->char #x30) = "0"



;;; string<?
(test (string<? "aaaa" "aaab") #t)
(test (string<? "aaaa" "aaaaa") #t)
(test (string<? "" "abcdefgh") #t)
(test (string<? "a" "abcdefgh") #t)
(test (string<? "abc" "abcdefgh") #t)
(test (string<? "cabc" "abcdefgh") #f)
(test (string<? "abcdefgh" "abcdefgh") #f)
(test (string<? "xyzabc" "abcdefgh") #f)
(test (string<? "abc" "xyzabcdefgh") #t)
(test (string<? "abcdefgh" "") #f)
(test (string<? "abcdefgh" "a") #f)
(test (string<? "abcdefgh" "abc") #f)
(test (string<? "abcdefgh" "cabc") #t)
(test (string<? "abcdefgh" "xyzabc") #t)
(test (string<? "xyzabcdefgh" "abc") #f)
(test (string<? "abcdef" "bcdefgh") #t)
(test (string<? "" "abcdefgh") #t)
(test (string<? "" "") #f)
(test (string<? "A" "B") #t)
(test (string<? "a" "b") #t)
(test (string<? "9" "0") #f)
(test (string<? "A" "A") #f)

(test (string<? "A" "B" "A") #f)
(test (string<? "A" "A" "B") #f)
(test (string<? "A" "A" "A") #f)
(test (string<? "B" "B" "C") #f)
(test (string<? "foo" "foo" "foo") #f)
(test (string<? "foo" "foo" "") #f)
(test (string<? "foo" "foo" "fOo") #f)

(test (string<? "foo" "fo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<? s1 s2)) #f)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<? s1 s2)) #t)
(test (not (string<? "foo\x0a" "foo\x0a")) #t)
(test (string<? "foo\x0a" "foo\x0b") #t)

(test (string<? (string (integer->char #xf0)) (string (integer->char #x70))) #f) 

(for-each
 (lambda (arg)
   (test (string<? "hi" arg) 'error)
   (test (string<? arg "hi") 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))



;;; string>?
(test (string>? "aaab" "aaaa") #t)
(test (string>? "aaaaa" "aaaa") #t)
(test (string>? "" "abcdefgh") #f)
(test (string>? "a" "abcdefgh") #f)
(test (string>? "abc" "abcdefgh") #f)
(test (string>? "cabc" "abcdefgh") #t)
(test (string>? "abcdefgh" "abcdefgh") #f)
(test (string>? "xyzabc" "abcdefgh") #t)
(test (string>? "abc" "xyzabcdefgh") #f)
(test (string>? "abcdefgh" "") #t)
(test (string>? "abcdefgh" "a") #t)
(test (string>? "abcdefgh" "abc") #t)
(test (string>? "abcdefgh" "cabc") #f)
(test (string>? "abcdefgh" "xyzabc") #f)
(test (string>? "xyzabcdefgh" "abc") #t)
(test (string>? "abcde" "bc") #f)
(test (string>? "bcdef" "abcde") #t)
(test (string>? "bcdef" "abcdef") #t)
(test (string>? "" "") #f)
(test (string>? "A" "B") #f)
(test (string>? "a" "b") #f)
(test (string>? "9" "0") #t)
(test (string>? "A" "A") #f)

(test (string>? "A" "B" "a") #f)
(test (string>? "C" "B" "A") #t)
(test (string>? "A" "A" "A") #f)
(test (string>? "B" "B" "A") #f)
(test (string>? "foo" "foo" "foo") #f)
(test (string>? "foo" "foo" "") #f)
(test (string>? "foo" "foo" "fOo") #f)

(test (string>? "foo" "fooo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>? s1 s2)) #t)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>? s1 s2)) #f)

(test (string>? (string (integer->char #xf0)) (string (integer->char #x70))) #t) ; ??

(for-each
 (lambda (arg)
   (test (string>? "hi" arg) 'error)
   (test (string>? arg "hi") 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))



;;; string<=?
(test (string<=? "aaa" "aaaa") #t)
(test (string<=? "aaaaa" "aaaa") #f)
(test (string<=? "a" "abcdefgh") #t)
(test (string<=? "abc" "abcdefgh") #t)
(test (string<=? "aaabce" "aaabcdefgh") #f)
(test (string<=? "cabc" "abcdefgh") #f)
(test (string<=? "abcdefgh" "abcdefgh") #t)
(test (string<=? "xyzabc" "abcdefgh") #f)
(test (string<=? "abc" "xyzabcdefgh") #t)
(test (string<=? "abcdefgh" "") #f)
(test (string<=? "abcdefgh" "a") #f)
(test (string<=? "abcdefgh" "abc") #f)
(test (string<=? "abcdefgh" "cabc") #t)
(test (string<=? "abcdefgh" "xyzabc") #t)
(test (string<=? "xyzabcdefgh" "abc") #f)
(test (string<=? "abcdef" "bcdefgh") #t)
(test (string<=? "" "") #t)
(test (string<=? "A" "B") #t)
(test (string<=? "a" "b") #t)
(test (string<=? "9" "0") #f)
(test (string<=? "A" "A") #t)

(test (string<=? "A" "B" "C") #t)
(test (string<=? "C" "B" "A") #f)
(test (string<=? "A" "B" "B") #t)
(test (string<=? "A" "A" "A") #t)
(test (string<=? "B" "B" "A") #f)
(test (string<=? "foo" "foo" "foo") #t)
(test (string<=? "foo" "foo" "") #f)
(test (string<=? "foo" "foo" "fooo") #t)

(test (string<=? "foo" "fo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #f)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #t)

(for-each
 (lambda (arg)
   (test (string<=? "hi" arg) 'error)
   (test (string<=? arg "hi") 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))



;;; string>=?
(test (string>=? "aaaaa" "aaaa") #t)
(test (string>=? "aaaa" "aaaa") #t)
(test (string>=? "aaa" "aaaa") #f)
(test (string>=? "" "abcdefgh") #f)
(test (string>=? "a" "abcdefgh") #f)
(test (string>=? "abc" "abcdefgh") #f)
(test (string>=? "cabc" "abcdefgh") #t)
(test (string>=? "abcdefgh" "abcdefgh") #t)
(test (string>=? "xyzabc" "abcdefgh") #t)
(test (string>=? "abc" "xyzabcdefgh") #f)
(test (string>=? "abcdefgh" "") #t)
(test (string>=? "abcdefgh" "a") #t)
(test (string>=? "abcdefgh" "abc") #t)
(test (string>=? "abcdefgh" "cabc") #f)
(test (string>=? "abcdefgh" "xyzabc") #f)
(test (string>=? "xyzabcdefgh" "abc") #t)
(test (string>=? "bcdef" "abcdef") #t)
(test (string>=? "A" "B") #f)
(test (string>=? "a" "b") #f)
(test (string>=? "9" "0") #t)
(test (string>=? "A" "A") #t)
(test (string>=? "" "") #t)

(test (string>=? "A" "B" "C") #f)
(test (string>=? "C" "B" "A") #t)
(test (string>=? "C" "B" "B") #t)
(test (string>=? "A" "B" "B") #f)
(test (string>=? "A" "A" "A") #t)
(test (string>=? "B" "B" "A") #t)
(test (string>=? "B" "B" "C") #f)
(test (string>=? "foo" "foo" "foo") #t)
(test (string>=? "foo" "foo" "") #t)
(test (string>=? "foo" "foo" "fo") #t)

(test (string>=? "fo" "foo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #t)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #t)

(for-each
 (lambda (arg)
   (test (string>=? "hi" arg) 'error)
   (test (string>=? arg "hi") 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))



;;; string-ci=?
(test (string-ci=? "A" "B") #f)
(test (string-ci=? "a" "B") #f)
(test (string-ci=? "A" "b") #f)
(test (string-ci=? "a" "b") #f)
(test (string-ci=? "9" "0") #f)
(test (string-ci=? "A" "A") #t)
(test (string-ci=? "A" "a") #t)
(test (string-ci=? "" "") #t)
(test (string-ci=? "aaaa" "AAAA") #t)
(test (string-ci=? "aaaa" "Aaaa") #t)

(test (string-ci=? "A" "B" "a") #f)
(test (string-ci=? "A" "A" "a") #t)
(test (string-ci=? "A" "A" "a") #t)
(test (string-ci=? "foo" "foo" "foo") #t)
(test (string-ci=? "foo" "foo" "") #f)
(test (string-ci=? "foo" "Foo" "fOo") #t)

(test (string-ci=? "foo" "GOO" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "124")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)

(for-each
 (lambda (arg)
   (test (string-ci=? "hi" arg) 'error)
   (test (string-ci=? arg "hi") 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))


#|
(let ((size 15)
      (tries 10000))
  (let ((str1 (make-string size))
	(str2 (make-string size)))
    (do ((i 0 (+ i 1)))
	((= i tries))
      (do ((k 0 (+ k 1)))
	  ((= k size))
	(set! (str1 k) (integer->char (random 128)))
	(if (> (random 10) 4)
	    (set! (str2 k) (char-upcase (str1 k)))
	    (set! (str2 k) (char-downcase (str1 k)))))
      (if (not (string-ci=? str1 str2))
	  (format #t "not =: ~S ~S~%" str1 str2))
      (if (and (string-ci<? str1 str2)
	       (string-ci>=? str1 str2))
	  (format #t "< : ~S ~S~%" str1 str2))
      (if (and (string-ci>? str1 str2)
	       (string-ci<=? str1 str2))
	  (format #t "> : ~S ~S~%" str1 str2)))))
|#

;;; string-ci<?
(test (string-ci<? "a" "Aa") #t)
(test (string-ci<? "A" "B") #t)
(test (string-ci<? "a" "B") #t)
(test (string-ci<? "A" "b") #t)
(test (string-ci<? "a" "b") #t)
(test (string-ci<? "9" "0") #f)
(test (string-ci<? "0" "9") #t)
(test (string-ci<? "A" "A") #f)
(test (string-ci<? "A" "a") #f)
(test (string-ci<? "" "") #f)

(test (string-ci<? "t" "_") #t)
(test (string-ci<? "a" "]") #t)
(test (string-ci<? "z" "^") #t)
(test (string-ci<? "]4.jVKo\\\\^:\\A9Z4" "MImKA[mNv1`") #f)

(test (string-ci<? "A" "B" "A") #f)
(test (string-ci<? "A" "A" "B") #f)
(test (string-ci<? "A" "A" "A") #f)
(test (string-ci<? "B" "B" "C") #f)
(test (string-ci<? "B" "b" "C") #f)
(test (string-ci<? "foo" "foo" "foo") #f)
(test (string-ci<? "foo" "foo" "") #f)
(test (string-ci<? "foo" "foo" "fOo") #f)
(test (string-ci<? "34ZsfQD<obff33FBPFl" "7o" "9l7OM" "FC?M63=" "rLM5*J") #t)
(test (string-ci<? "NX7" "-;h>P" "DMhk3Bg") #f)
(test (string-ci<? "+\\mZl" "bE7\\e(HaW5CDXbPi@U_" "B_") #t)

(test (string-ci<? (string (integer->char #xf0)) (string (integer->char #x70))) #f) 

(test (string-ci<? "foo" "fo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "124")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)
(test (let ((s1 "123") (s2 "12")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)

(for-each
 (lambda (arg)
   (test (string-ci<? "hi" arg) 'error)
   (test (string-ci<? arg "hi") 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))




;;; string-ci>?
(test (string-ci>? "Aaa" "AA") #t)
(test (string-ci>? "A" "B") #f)
(test (string-ci>? "a" "B") #f)
(test (string-ci>? "A" "b") #f)
(test (string-ci>? "a" "b") #f)
(test (string-ci>? "9" "0") #t)
(test (string-ci>? "A" "A") #f)
(test (string-ci>? "A" "a") #f)
(test (string-ci>? "" "") #f)
(test (string-ci>? "Z" "DjNTl0") #t)
(test (string-ci>? "2399dt7BVN[,A" "^KHboHV") #f)

(test (string-ci>? "t" "_") #f)
(test (string-ci>? "a" "]") #f)
(test (string-ci>? "z" "^") #f)
(test (string-ci>? "R*95oG.k;?" "`2?J6LBbLG^alB[fMD") #f)
(test (string-ci>? "]" "X") #t)

(test (string-ci>? "A" "B" "a") #f)
(test (string-ci>? "C" "b" "A") #t)
(test (string-ci>? "a" "A" "A") #f)
(test (string-ci>? "B" "B" "A") #f)
(test (string-ci>? "foo" "foo" "foo") #f)
(test (string-ci>? "foo" "foo" "") #f)
(test (string-ci>? "foo" "foo" "fOo") #f)
(test (string-ci>? "ZNiuEa@/V" "KGbKliYMY" "9=69q3ica" ":]") #f)
(test (string-ci>? "^" "aN@di;iEO" "7*9q6uPmX9)PaY,6J" "15vH") #t)

(test (string-ci>? "foo" "fooo" 1.0) 'error)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #t)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #f)

(for-each
 (lambda (arg)
   (test (string-ci>? "hi" arg) 'error)
   (test (string-ci>? arg "hi") 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))




;;; string-ci<=?
(test (string-ci<=? "A" "B") #t)
(test (string-ci<=? "a" "B") #t)
(test (string-ci<=? "A" "b") #t)
(test (string-ci<=? "a" "b") #t)
(test (string-ci<=? "9" "0") #f)
(test (string-ci<=? "A" "A") #t)
(test (string-ci<=? "A" "a") #t)
(test (string-ci<=? "" "") #t)
(test (string-ci<=? ":LPC`" ",O0>affA?(") #f)

(test (string-ci<=? "t" "_") #t)
(test (string-ci<=? "a" "]") #t)
(test (string-ci<=? "z" "^") #t)
(test (string-ci<=? "G888E>beF)*mwCNnagP" "`2uTd?h") #t)

(test (string-ci<=? "A" "b" "C") #t)
(test (string-ci<=? "c" "B" "A") #f)
(test (string-ci<=? "A" "B" "B") #t)
(test (string-ci<=? "a" "A" "A") #t)
(test (string-ci<=? "B" "b" "A") #f)
(test (string-ci<=? "foo" "foo" "foo") #t)
(test (string-ci<=? "foo" "foo" "") #f)
(test (string-ci<=? "FOO" "fOo" "fooo") #t)
(test (string-ci<=? "78mdL82*" "EFaCrIdm@_D+" "eMu\\@dSSY") #t)
(test (string-ci<=? "`5pNuFc3PM<rNs" "e\\Su_raVNk6HD" "vXnuN7?S0?S(w+M?p") #f)

(test (string-ci<=? "fOo" "fo" 1.0) 'error)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #f)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #t)

(for-each
 (lambda (arg)
   (test (string-ci<=? "hi" arg) 'error)
   (test (string-ci<=? arg "hi") 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))




;;; string-ci>=?
(test (string-ci>=? "A" "B") #f)
(test (string-ci>=? "a" "B") #f)
(test (string-ci>=? "A" "b") #f)
(test (string-ci>=? "a" "b") #f)
(test (string-ci>=? "9" "0") #t)
(test (string-ci>=? "A" "A") #t)
(test (string-ci>=? "A" "a") #t)
(test (string-ci>=? "" "") #t)
(test (string-ci>=? "5d7?[o[:hop=ktv;9)" "p^r9;TAXO=^") #f)

(test (string-ci>=? "t" "_") #f)
(test (string-ci>=? "a" "]") #f)
(test (string-ci>=? "z" "^") #f)
(test (string-ci>=? "jBS" "`<+s[[:`l") #f)

(test (string-ci>=? "A" "b" "C") #f)
(test (string-ci>=? "C" "B" "A") #t)
(test (string-ci>=? "C" "B" "b") #t)
(test (string-ci>=? "a" "B" "B") #f)
(test (string-ci>=? "A" "A" "A") #t)
(test (string-ci>=? "B" "B" "A") #t)
(test (string-ci>=? "B" "b" "C") #f)
(test (string-ci>=? "foo" "foo" "foo") #t)
(test (string-ci>=? "foo" "foo" "") #t)
(test (string-ci>=? "foo" "foo" "fo") #t)
(test (string-ci>=? "tF?8`Sa" "NIkMd7" "f`" "1td-Z?teE" "-ik1SK)hh)Nq].>") #t)
(test (string-ci>=? "Z6a8P" "^/VpmWwt):?o[a9\\_N" "8[^h)<KX?[utsc") #f)

(test (string-ci>=? "fo" "foo" 1.0) 'error)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #t)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #f)

(for-each
 (lambda (arg)
   (test (string-ci>=? "hi" arg) 'error)
   (test (string-ci>=? arg "hi") 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))




;;; string-length
(test (string-length "abc") 3)
(test (string-length "") 0)
(test (string-length (string)) 0)
(test (string-length "\"\\\"") 3)
(test (string-length (string #\newline)) 1)
(test (string-length "hi there") 8)
(test (string-length "\"") 1)
(test (string-length "\\") 1)
(test (string-length "\n") 1)
(test (string-length (make-string 100 #\a)) 100)
(test (string-length "1\\2") 3)
(test (string-length "1\\") 2)
(test (string-length "hi\\") 3)
(test (string-length "\\\\\\\"") 4)
(test (string-length "A ; comment") 11)
(test (string-length "#| comment |#") 13)
(test (string-length "'123") 4)
(test (string-length '"'123") 4)
(test (let ((str (string #\# #\\ #\t))) (string-length str)) 3)

(test (string-length "#\\(") 3)
(test (string-length ")()") 3)
(test (string-length "(()") 3)
(test (string-length "(string #\\( #\\+ #\\space #\\1 #\\space #\\3 #\\))") 44)
(test (string-length) 'error)
(test (string-length "hi" "ho") 'error)
(test (string-length "..\ ..") 4)
(test (string-length (string #\null)) 1) ; ??
(test (string-length (string #\null #\null)) 2) ; ??
(test (string-length (string #\null #\newline)) 2) ; ??
(test (string-length ``"hi") 2) ; ?? and in s7 ,"hi" is "hi" as with numbers

(for-each
 (lambda (arg)
   (test (string-length arg) 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))




;;; string
(for-each
 (lambda (arg)
   (test (string #\a arg) 'error)
   (test (string #\a #\null arg) 'error)
   (test (string arg) 'error))
 (list '() (list 1) '(1 . 2) "a" #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (string) "")
(test (string #\a #\b #\c) "abc")
(test (string #\a) "a")
(test (map string '(#\a #\b)) '("a" "b"))
(test (map string '(#\a #\b) '(#\c #\d)) '("ac" "bd"))
(test (map string '(#\a #\b #\c) '(#\d #\e #\f) '(#\g #\h #\i)) '("adg" "beh" "cfi"))
(test (map string "abc" "def" "ghi") '("adg" "beh" "cfi"))
(test (string #\" #\# #\") "\"#\"")
(test (string #\\ #\\ #\# #\\ #\# #\#) "\\\\#\\##")
(test (string #\' #\' #\` #\") '"''`\"")
;;; some schemes accept \' and other such sequences in a string, but the spec only mentions \\ and \"
(test (string '()) 'error)
(test (string "j" #\a) 'error)



;;; make-string
(test (make-string 0) "")
(test (make-string 3 #\a) "aaa")
(test (make-string 0 #\a) "")
(test (make-string 3 #\space) "   ")
(test (let ((hi (make-string 3 #\newline))) (string-length hi)) 3)

(test (make-string -1) 'error)
(test (make-string -0) "")
(test (make-string 2 #\a #\b) 'error)
(test (make-string) 'error)
(test (make-string most-positive-fixnum) 'error)
(test (make-string most-negative-fixnum) 'error)

(for-each
 (lambda (arg)
   (test (make-string 3 arg) 'error))
 (list "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (make-string arg #\a) 'error))
 (list #\a "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (make-string arg) 'error))
 (list #\a "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))



;;; string-ref
(test (string-ref "abcdef-dg1ndh" 0) #\a)
(test (string-ref "abcdef-dg1ndh" 1) #\b)
(test (string-ref "abcdef-dg1ndh" 6) #\-)
(test (string-ref "\"\\\"" 1) #\\)
(test (string-ref "\"\\\"" 2) #\")
(test (string-ref "12\ 34" 2) #\3)

(test (let ((str (make-string 3 #\x))) (set! (string-ref str 1) #\a) str) "xax")

(test (string-ref "abcdef-dg1ndh" 20) 'error)
(test (string-ref "abcdef-dg1ndh") 'error)
(test (string-ref "abcdef-dg1ndh" -3) 'error)
(test (string-ref) 'error)
(test (string-ref 2) 'error)
(test (string-ref "\"\\\"" 3) 'error)
(test (string-ref "" 0) 'error)  
(test (string-ref "" 1) 'error)
(test (string-ref "hiho" (expt 2 32)) 'error)
(test (char=? (string-ref (string #\null) 0) #\null) #t)
(test (char=? (string-ref (string #\1 #\null #\2) 1) #\null) #t)
(test (char=? ("1\x002" 1) #\null) #t)
(test (char=? (string-ref (string #\newline) 0) #\newline) #t)
(test (char=? (string-ref (string #\space) 0) #\space) #t)

(for-each
 (lambda (arg)
   (test (string-ref arg 0) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-ref "hiho" arg) 'error))
 (list #\a -1 123 4 "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test ("hi" 1) #\i)
(test (("hi" 1) 0) 'error)
(test ("hi" 1 2) 'error)
(test ("" 0) 'error)
(test (set! ("" 0) #\a) 'error)
(test (set! ("hi" 1 2) #\a) 'error)
(test (set! ("hi" 1) #\a #\b) 'error)
(test ("hi") 'error)
(test ("") 'error)
(test ((let () "hi")) 'error)
(test ((let () "hi") 0) #\h)



;;; string-copy
(test (let ((hi (string-copy "hi"))) (string-set! hi 0 #\H) hi) "Hi")
(test (let ((hi (string-copy "hi"))) (string-set! hi 1 #\H) hi) "hH")
(test (let ((hi (string-copy "\"\\\""))) (string-set! hi 0 #\a) hi) "a\\\"")
(test (let ((hi (string-copy "\"\\\""))) (string-set! hi 1 #\a) hi) "\"a\"")
(test (let ((hi (string #\a #\newline #\b))) (string-set! hi 1 #\c) hi) "acb")
(test (string-copy "ab") "ab")
(test (string-copy "") "")
(test (string-copy "\"\\\"") "\"\\\"")
(test (let ((hi "abc")) (eq? hi (string-copy hi))) #f)
(test (let ((hi (string-copy (make-string 8 (integer->char 0))))) (string-fill! hi #\a) hi) "aaaaaaaa") ; is this result widely accepted?
(test (string-copy (string-copy (string-copy "a"))) "a")
(test (string-copy (string-copy (string-copy ""))) "")
(test (string-copy "a\x00b") "a\x00b") ; prints normally as "a" however
(test (string-copy (string #\1 #\null #\2)) (string #\1 #\null #\2))
(test (string-copy) 'error)
(test (string-copy "hi" "ho") 'error)

(for-each
 (lambda (arg)
   (test (string-copy arg) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (length (string-copy (string #\null))) 1)



;;; string-set!
(let ((str (make-string 10 #\x)))
  (string-set! str 3 (integer->char 0))
  (test (string=? str "xxx") #f)
  (test (char=? (string-ref str 4) #\x) #t)
  (string-set! str 4 #\a)
  (test (string=? str "xxx") #f)
  (test (char=? (string-ref str 4) #\a) #t)
  (string-set! str 3 #\x)
  (test (string=? str "xxxxaxxxxx") #t))

(test (string-set! "hiho" 1 #\c) #\c)
(test (set! ("hi" 1 2) #\i) 'error)
(test (set! ("hi" 1) "ho") 'error)
(test (set! ("hi") #\i) 'error)
(test (let ((x "hi") (y 'x)) (string-set! y 0 #\x) x) 'error)

(test (let ((hi (make-string 3 #\a)))
	(string-set! hi 1 (let ((ho (make-string 4 #\x)))
			    (string-set! ho 1 #\b)
			    (string-ref ho 0)))
	hi)
      "axa")

(test (string-set! "hiho" (expt 2 32) #\a) 'error)

(test (let ((hi (string-copy "hi"))) (string-set! hi 2 #\H) hi) 'error)
(test (let ((hi (string-copy "hi"))) (string-set! hi -1 #\H) hi) 'error)
(test (let ((g (lambda () "***"))) (string-set! (g) 0 #\?)) #\?)
(test (string-set! "" 0 #\a) 'error)
(test (string-set! "" 1 #\a) 'error)
(test (string-set! (string) 0 #\a) 'error)
(test (string-set! (symbol->string 'lambda) 0 #\a) #\a)
(test (let ((ho (make-string 0 #\x))) (string-set! ho 0 #\a) ho) 'error)
(test (let ((str "hi")) (string-set! (let () str) 1 #\a) str) "ha") ; (also in Guile)
(test (let ((x 2) (str "hi")) (string-set! (let () (set! x 3) str) 1 #\a) (list x str)) '(3 "ha"))
(test (let ((str "hi")) (set! ((let () str) 1) #\b) str) "hb")
(test (let ((str "hi")) (string-set! (let () (string-set! (let () str) 0 #\x) str) 1 #\x) str) "xx")
(test (let ((str "hi")) (string-set! (let () (set! str "hiho") str) 3 #\x) str) "hihx") ; ! (this works in Guile also)

(for-each
 (lambda (arg)
   (test (string-set! arg 0 #\a) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-set! "hiho" arg #\a) 'error))
 (list #\a -1 123 4 "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-set! "hiho" 0 arg) 'error))
 (list 1 "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (equal? (let ((str "hiho")) (string-set! str 2 #\null) str) "hi") #f)
(test (string=? (let ((str "hiho")) (string-set! str 2 #\null) str) "hi") #f)
(test (let* ((s1 "hi") (s2 s1)) (string-set! s2 1 #\x) s1) "hx")
(test (let* ((s1 "hi") (s2 (copy s1))) (string-set! s2 1 #\x) s1) "hi")

(test (eq? (car (catch #t (lambda () (set! ("hi") #\a)) (lambda args args))) 'wrong-number-of-args) #t)
(test (eq? (car (catch #t (lambda () (set! ("hi" 0 0) #\a)) (lambda args args))) 'wrong-number-of-args) #t) ; (vector-set! 1 ...)
(test (eq? (car (catch #t (lambda () (set! (("hi" 0) 0) #\a)) (lambda args args))) 'syntax-error) #t) ; (set! (1 ...))



;;; string-fill!
(test (string-fill! "hiho" #\c) #\c)
(test (string-fill! "" #\a) #\a)
(test (string-fill! "hiho" #\a) #\a)
(test (let ((g (lambda () "***"))) (string-fill! (g) #\?)) #\?)
(test (string-fill!) 'error)
(test (string-fill! "hiho" #\a #\b) 'error)

(test (let ((hi (string-copy "hi"))) (string-fill! hi #\s) hi) "ss")
(test (let ((hi (string-copy ""))) (string-fill! hi #\x) hi) "")
(test (let ((str (make-string 0))) (string-fill! str #\a) str) "")
(test (let ((hi (make-string 8 (integer->char 0)))) (string-fill! hi #\a) hi) "aaaaaaaa") ; is this result widely accepted?
(test (recompose 12 string-copy "xax") "xax")
(test (let ((hi (make-string 3 #\x))) (recompose 12 (lambda (a) (string-fill! a #\a) a) hi)) "aaa")
(test (let ((hi (make-string 3 #\x))) (recompose 12 (lambda (a) (string-fill! hi a)) #\a) hi) "aaa")
(test (let ((str (string #\null #\null))) (fill! str #\x) str) "xx")

(for-each
 (lambda (arg)
   (test (let ((hiho "hiho")) (string-fill! hiho arg) hiho) 'error))
 (list 1 "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-fill! arg #\a) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))



;;; substring
(test (substring "ab" 0 0) "")
(test (substring "ab" 1 1) "")
(test (substring "ab" 2 2) "")
(test (substring "ab" 0 1) "a")
(test (substring "ab" 1 2) "b")
(test (substring "ab" 0 2) "ab")
(test (substring "hi there" 3 6) "the")
(test (substring "hi there" 0 (string-length "hi there")) "hi there")
(test (substring "" 0 0) "")
(let ((str "012345"))
  (let ((str1 (substring str 2 4)))
    (string-set! str1 1 #\x)
    (test (string=? str "012345") #t)
    (let ((str2 (substring str1 1)))
      (set! (str2 0) #\z)
      (test (string=? str "012345") #t)
      (test (string=? str1 "2x") #t)
      (test (string=? str2 "z") #t))))
(test (substring (substring "hiho" 0 2) 1) "i")
(test (substring (substring "hiho" 0 2) 2) "")
(test (substring (substring "hiho" 0 2) 0 1) "h")
(test (substring "hi\nho" 3 5) "ho")
(test (substring (substring "hi\nho" 1 4) 2) "h")
(test (substring (substring "hi\nho" 3 5) 1 2) "o")
(test (substring "hi\"ho" 3 5) "ho")
(test (substring (substring "hi\"ho" 1 4) 2) "h")
(test (substring (substring "hi\"ho" 3 5) 1 2) "o")
(test (substring "01\ \ 34" 2) "34")
(test (let* ((s1 "0123456789") (s2 (substring s1 1 3))) (string-set! s2 1 #\x) s1) "0123456789")
(test (substring (substring "" 0 0) 0 0) "")
(test (substring (format #f "") 0 0) "")

(test (substring "012" 3) "")
(test (substring "012" 10) 'error)
(test (substring "012" most-positive-fixnum) 'error)
(test (substring "012" -1) 'error)
(test (substring "012" 3 3) "")
(test (substring "012" 3 4) 'error)
(test (substring "012" 3 2) 'error)
(test (substring "012" 3 -2) 'error)
(test (substring "012" 3 0) 'error)
(test (substring "012" 0) "012")
(test (substring "012" 2) "2")
(test (substring "" 0) "")

(test (recompose 12 (lambda (a) (substring a 0 3)) "12345") "123")
(test (reinvert 12 (lambda (a) (substring a 0 3)) (lambda (a) (string-append a "45")) "12345") "12345")

(test (substring "ab" 0 3) 'error)
(test (substring "ab" 3 3) 'error)
(test (substring "ab" 2 3) 'error)
(test (substring "" 0 1) 'error)
(test (substring "" -1 0) 'error)
(test (substring "abc" -1 0) 'error)
(test (substring "hiho" (expt 2 32) (+ 2 (expt 2 32))) 'error)
(test (substring) 'error)
(test (substring "hiho" 0 1 2) 'error)
(test (substring "1234" -1 -1) 'error)
(test (substring "1234" 1 0) 'error)
(test (substring "" most-positive-fixnum 1) 'error)

(let ((str "0123456789"))
  (string-set! str 5 #\null)
  (test (substring str 6) "6789")
  (test (substring str 5 5) "")
  (test (substring str 4 5) "4")
  (test (substring str 5 6) "\x00")
  (test (substring str 5 7) "\x006")
  (test (substring str 4 7) "4\x006"))

(for-each
 (lambda (arg)
   (test (substring "hiho" arg 0) 'error))
 (list "hi" #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (substring "0123" arg) 'error)
   (test (substring "hiho" 1 arg) 'error))
 (list "hi" #\a -1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (substring arg 1 2) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))



;;; string-append
(test (string-append "hi" "ho") "hiho")
(test (string-append "hi") "hi")
(test (string-append "hi" "") "hi")
(test (string-append "hi" "" "ho") "hiho")
(test (string-append "" "hi") "hi")
(test (string-append) "")
(test (string-append "a" (string-append (string-append "b" "c") "d") "e") "abcde")
(test (string-append "a" "b" "c" "d" "e") "abcde")
(test (string-append (string-append) (string-append (string-append))) "")
(test (let ((hi "hi")) (let ((ho (string-append hi))) (eq? hi ho))) #f)
(test (let ((hi "hi")) (let ((ho (string-append hi))) (string-set! ho 0 #\a) hi)) "hi")
(test (let ((hi "hi")) (set! hi (string-append hi hi hi hi)) hi) "hihihihi")
(test (string-append '()) 'error)
(test (string=? (string-append "012" (string #\null) "456") 
		(let ((str "0123456")) (string-set! str 3 #\null) str))
      #t)
(test (string=? (string-append "012" (string #\null) "356") 
		(let ((str "0123456")) (string-set! str 3 #\null) str))
      #f)
(test (string-append """hi""ho""") "hiho")
(test (let* ((s1 "hi") (s2 (string-append s1 s1))) (string-set! s2 1 #\x) s1) "hi")
(test (let* ((s1 "hi") (s2 (string-append s1))) (string-set! s2 1 #\x) s1) "hi")
(test (length (string-append (string #\x #\y (integer->char 127) #\z) (string #\a (integer->char 0) #\b #\c))) 8)

(test (length (string-append "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc")) 915)
(test (length (string-append (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c))) 915)


(num-test (letrec ((hi (lambda (str n)
			 (if (= n 0)
			     str
			     (hi (string-append str "a") (- n 1))))))
	    (string-length (hi "" 100)))
	  100)

(test (let* ((str "hiho")
	     (str1 "ha")
	     (str2 (string-append str1 str)))
	(string-set! str2 1 #\x)
	(string-set! str2 4 #\x)
	(and (string=? str "hiho")
	     (string=? str1 "ha")
	     (string=? str2 "hxhixo")))
      #t)
(test (let* ((str (string-copy "hiho"))
	     (str1 (string-copy "ha"))
	     (str2 (string-append str1 str)))
	(string-set! str1 1 #\x)
	(string-set! str 2 #\x)
	(and (string=? str "hixo")
	     (string=? str1 "hx")
	     (string=? str2 "hahiho")))
      #t)

(let ((s1 (string #\x #\null #\y))
      (s2 (string #\z #\null)))
  (test (string=? (string-append s1 s2) (string #\x #\null #\y #\z #\null)) #t)
  (test (string=? (string-append s2 s1) (string #\z #\null #\x #\null #\y)) #t))

(test (recompose 12 string-append "x") "x")
(test (recompose 12 (lambda (a) (string-append a "x")) "a") "axxxxxxxxxxxx")
(test (recompose 12 (lambda (a) (string-append "x" a)) "a") "xxxxxxxxxxxxa")

(test (length (string-append "\\?" "hi")) 4)
(test (string-append "hi" 1) 'error)
(test (eval-string "(string-append \"\\?\")") 'error) ; guile mailing list
(test (eval-string "(string-append \"\\?\" \"hi\")") 'error) ; guile mailing list
(for-each
 (lambda (arg)
   (test (string-append "hiho" arg) 'error)
   (test (string-append arg "hi") 'error)
   (test (string-append "a" "b" arg) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))


(test (let ((str (make-string 4 #\x))
	    (ctr 0))
	(for-each
	 (lambda (c)
	   (string-set! str ctr c)
	   (set! ctr (+ ctr 1)))
	 "1234")
	str)
      "1234")

(test (let ((str (make-string 8 #\x))
	    (ctr 0))
	(for-each
	 (lambda (c1 c2)
	   (string-set! str ctr c1)
	   (string-set! str (+ ctr 1) c2)
	   (set! ctr (+ ctr 2)))
	 "1234"
	 "hiho")
	str)
      "1h2i3h4o")

#|
(let ((size 1024))
  (let ((str (make-string size)))
    (do ((i 0 (+ i 1)))
	((= i size))
      (set! (str i) (integer->char (+ 1 (modulo i 255)))))
    (let ((str1 (string-copy str)))
      (test (string? str1) #t)
      (test (string-length str1) 1024)
      (test (string-ref str1 556) (string-ref str 556))
      (test (string=? str str1) #t)
      (test (string<=? str str1) #t)
      (test (string>=? str str1) #t)
      (test (string-ci=? str str1) #t)
      (test (string-ci<=? str str1) #t)
      (test (string-ci>=? str str1) #t)
      (test (string<? str str1) #f)
      (test (string>? str str1) #f)
      (test (string-ci<? str str1) #f)
      (test (string-ci>? str str1) #f)
      (test (substring str 123 321) (substring str1 123 321))

      (string-set! str1 1000 #\space)
      (test (string=? str str1) #f)
      (test (string<=? str str1) #f)
      (test (string>=? str str1) #t)
      (test (string-ci=? str str1) #f)
      (test (string-ci<=? str str1) #f)
      (test (string-ci>=? str str1) #t)
      (test (string<? str str1) #f)
      (test (string>? str str1) #t)
      (test (string-ci<? str str1) #f)
      (test (string-ci>? str str1) #t)

      (test (string-length (string-append str str1)) 2048)
      )))
|#


;;; string->list
;;; list->string
(test (string->list "abc") (list #\a #\b #\c))
(test (string->list "") '())
(test (string->list (make-string 0)) '())
(test (string->list (string #\null)) '(#\null))
(test (string->list (string)) '())
(test (string->list (substring "hi" 0 0)) '())
(test (string->list (list->string (list #\a #\b #\c))) (list #\a #\b #\c))
(test (string->list (list->string '())) '())
(test (list->string (string->list "abc")) "abc")
(test (list->string (string->list "hi there")) "hi there")
(test (list->string (string->list "&*#%^@%$)~@")) "&*#%^@%$)~@")
(test (list->string (string->list "")) "")
(test (let* ((str "abc")
	     (lst (string->list str)))
	(and (string=? str "abc")
	     (equal? lst (list #\a #\b #\c))))
      #t)
(test (list->string '()) "")

(test (list->string (list #\a #\b #\c)) "abc")
(test (list->string (list)) "")

(test (list->string (list #\" #\# #\")) "\"#\"")
(test (list->string (list #\\ #\\ #\# #\\ #\# #\#)) "\\\\#\\##")
(test (list->string (list #\' #\' #\` #\")) '"''`\"")

(test (reinvert 12 string->list list->string "12345") "12345")

(test (string->list) 'error)
(test (list->string) 'error)
(test (string->list "hi" "ho") 'error)
(test (list->string '() '(1 2)) 'error)
(test (string->list " hi ") '(#\space #\h #\i #\space))
(test (string->list (string (integer->char #xf0) (integer->char #x70))) (list (integer->char #xf0) (integer->char #x70)))

(for-each
 (lambda (arg)
   (test (string->list arg) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (let ((x (cons #\a #\b))) (set-cdr! x x) (list->string x)) 'error)
(test (let ((lst (list #\a #\b))) (set! (cdr (cdr lst)) lst) (list->string lst)) 'error)
(test (let ((lst (list #\a #\b))) (set! (cdr (cdr lst)) lst) (apply string lst)) 'error)

(for-each
 (lambda (arg)
   (test (list->string arg) 'error))
 (list "hi" #\a 1 ''foo '(1 . 2) (cons #\a #\b) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(let ((str (list->string '(#\x #\space #\null #\x))))
  (test (length str) 4)
  (test (str 1) #\space)
  (test (str 2) #\null)
  (test (str 3) #\x)
  (test (object->string str) "\"x \\x00x\"")
  (let ((lst (string->list str)))
    (test lst '(#\x #\space #\null #\x))))

(let ((strlen 8))
  (let ((str (make-string strlen)))
    (do ((i 0 (+ i 1)))
	((= i 10))
      (do ((k 0 (+ k 1)))
	  ((= k strlen))
	(set! (str k) (integer->char (random 256))))
      (let ((lst (string->list str)))
	(let ((newstr (list->string lst)))
	  (let ((lstlen (length lst))
		(newstrlen (length newstr)))
	    (if (or (not (= lstlen strlen newstrlen))
		    (not (string=? newstr str)))
		(format #t ";string->list->string: ~S -> ~A -> ~S~%" str lst newstr))))))))

#|
(define (all-strs len file)
  (let* ((funny-chars (list #\` #\# #\, #\@ #\' #\" #\. #\( #\) #\\))
	 (num-chars (length funny-chars)))
    (let ((ctrs (make-vector len 0)))

      (do ((i 0 (+ i 1)))
	  ((= i (expt num-chars len)))
	(let ((carry #t))
	  (do ((k 0 (+ k 1)))
	      ((or (= k len)
		   (not carry)))
	    (vector-set! ctrs k (+ 1 (vector-ref ctrs k)))
	    (if (= (vector-ref ctrs k) num-chars)
		(vector-set! ctrs k 0)
		(set! carry #f)))

	  (let ((strlst '()))
	    (do ((k 0 (+ k 1)))
		((= k len))
	      (let ((c (list-ref funny-chars (vector-ref ctrs k))))
		(set! strlst (cons c strlst))))

	    (let ((str (list->string strlst)))
	      (format file "(test (and (string=? ~S (string ~{#\\~C~^ ~})) (equal? '~A (string->list ~S))) #t)~%" str strlst strlst str))))))))

(call-with-output-file "strtst.scm"
  (lambda (p)
    (do ((len 3 (+ len 1)))
	((= len 5))
      (all-strs len p))))

(load "strtst.scm")
|#

(test (and (string=? "\"" (string #\")) (equal? '(#\") (string->list "\""))) #t)
(test (and (string=? "#\\" (string #\# #\\)) (equal? '(#\# #\\) (string->list "#\\"))) #t)
(test (and (string=? "#(" (string #\# #\()) (equal? '(#\# #\() (string->list "#("))) #t)
(test (and (string=? "\"@" (string #\" #\@)) (equal? '(#\" #\@) (string->list "\"@"))) #t)
(test (and (string=? "\";" (string #\" #\;)) (equal? '(#\" #\;) (string->list "\";"))) #t)
(test (and (string=? ")(" (string #\) #\()) (equal? '(#\) #\() (string->list ")("))) #t)
(test (and (string=? "`)#" (string #\` #\) #\#)) (equal? '(#\` #\) #\#) (string->list "`)#"))) #t)
(test (and (string=? "##\\" (string #\# #\# #\\)) (equal? '(#\# #\# #\\) (string->list "##\\"))) #t)
(test (and (string=? "#\"(" (string #\# #\" #\()) (equal? '(#\# #\" #\() (string->list "#\"("))) #t)
(test (and (string=? "#.@" (string #\# #\. #\@)) (equal? '(#\# #\. #\@) (string->list "#.@"))) #t)
(test (and (string=? ",`@" (string #\, #\` #\@)) (equal? '(#\, #\` #\@) (string->list ",`@"))) #t)
(test (and (string=? "',@" (string #\' #\, #\@)) (equal? '(#\' #\, #\@) (string->list "',@"))) #t)
(test (and (string=? "\"#@" (string #\" #\# #\@)) (equal? '(#\" #\# #\@) (string->list "\"#@"))) #t)
(test (and (string=? "\")\"" (string #\" #\) #\")) (equal? '(#\" #\) #\") (string->list "\")\""))) #t)
(test (and (string=? ")#(" (string #\) #\# #\()) (equal? '(#\) #\# #\() (string->list ")#("))) #t)
(test (and (string=? "`(,@" (string #\` #\( #\, #\@)) (equal? '(#\` #\( #\, #\@) (string->list "`(,@"))) #t)
(test (and (string=? "`)#\"" (string #\` #\) #\# #\")) (equal? '(#\` #\) #\# #\") (string->list "`)#\""))) #t)
(test (and (string=? "#\"'#" (string #\# #\" #\' #\#)) (equal? '(#\# #\" #\' #\#) (string->list "#\"'#"))) #t)
(test (and (string=? "#(@\\" (string #\# #\( #\@ #\\)) (equal? '(#\# #\( #\@ #\\) (string->list "#(@\\"))) #t)
(test (and (string=? "#(\\\\" (string #\# #\( #\\ #\\)) (equal? '(#\# #\( #\\ #\\) (string->list "#(\\\\"))) #t)
(test (and (string=? ",,.@" (string #\, #\, #\. #\@)) (equal? '(#\, #\, #\. #\@) (string->list ",,.@"))) #t)
(test (and (string=? ",@`\"" (string #\, #\@ #\` #\")) (equal? '(#\, #\@ #\` #\") (string->list ",@`\""))) #t)
(test (and (string=? "\"'\")" (string #\" #\' #\" #\))) (equal? '(#\" #\' #\" #\)) (string->list "\"'\")"))) #t)
(test (and (string=? "\")#\"" (string #\" #\) #\# #\")) (equal? '(#\" #\) #\# #\") (string->list "\")#\""))) #t)
(test (and (string=? "(\\`)" (string #\( #\\ #\` #\))) (equal? '(#\( #\\ #\` #\)) (string->list "(\\`)"))) #t)
(test (and (string=? "))\"'" (string #\) #\) #\" #\')) (equal? '(#\) #\) #\" #\') (string->list "))\"'"))) #t)
(test (and (string=? "\\,\\\"" (string #\\ #\, #\\ #\")) (equal? '(#\\ #\, #\\ #\") (string->list "\\,\\\""))) #t)
(test (and (string=? "\\\"`\"" (string #\\ #\" #\` #\")) (equal? '(#\\ #\" #\` #\") (string->list "\\\"`\""))) #t)
(test (and (string=? "\\\\#\"" (string #\\ #\\ #\# #\")) (equal? '(#\\ #\\ #\# #\") (string->list "\\\\#\""))) #t)




;;; symbol->string
;;; string->symbol
(test (symbol->string 'hi) "hi")
(test (string->symbol (symbol->string 'hi)) 'hi)
(test (eq? (string->symbol "hi") 'hi) #t)
(test (eq? (string->symbol "hi") (string->symbol "hi")) #t)

(test (string->symbol "hi") 'hi)

(test (let ((str (symbol->string 'hi)))
	(catch #t (lambda () (string-set! str 1 #\x)) (lambda args 'error)) ; can be disallowed
	(symbol->string 'hi))
      "hi")

(test (symbol->string 'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789)
      "sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789")
(test (string->symbol "sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789")
      'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789)
(test (let ((sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 32))
	(+ sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 1))
      33)

(test (symbol->string (string->symbol "hi there")) "hi there")
(test (symbol->string (string->symbol "Hi There")) "Hi There")
(test (symbol->string (string->symbol "HI THERE")) "HI THERE")
(test (symbol->string (string->symbol "")) "")
(test (symbol? (string->symbol "(weird name for a symbol!)")) #t)
(test (symbol->string (string->symbol "()")) "()")
(test (symbol->string (string->symbol (string #\"))) "\"")
(test (symbol->string 'quote) "quote")
(test (symbol->string if) 'error)
(test (symbol->string quote) 'error)

(test (symbol? (string->symbol "0")) #t)
(test (symbol? (symbol "0")) #t)
(test (string->symbol "0e") '0e)
(test (string->symbol "1+") '1+)
(test (symbol? (string->symbol "1+i")) #t)
(test (string->symbol ":0") ':0)
(test (symbol? (string->symbol " hi") ) #t)
(test (symbol? (string->symbol "hi ")) #t)
(test (symbol? (string->symbol "")) #t)

(test (reinvert 12 string->symbol symbol->string "hiho") "hiho")

(test (symbol->string) 'error)
(test (string->symbol) 'error)
(test (symbol->string 'hi 'ho) 'error)
(test (string->symbol "hi" "ho") 'error)

(test (symbol? (string->symbol (string #\x (integer->char 255) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 8) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 128) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 200) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 255) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 20) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 2) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 7) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 17) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 170) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 0) #\x))) #t)       ; but the symbol's name here is "x"
(test (eq? (string->symbol (string #\x (integer->char 0) #\x)) 'x) #t)        ;   hmmm...
(test (symbol? (string->symbol (string #\x #\y (integer->char 127) #\z))) #t) ; xy(backspace)z

(test (symbol? (string->symbol (string #\; #\" #\)))) #t)
(test (let (((symbol ";")) 3) (symbol ";")) 'error)
(test (symbol? (symbol "")) #t)
(test (symbol? (symbol (string))) #t)
(test (symbol? (symbol (make-string 0))) #t)
(test (symbol? (symbol (string-append))) #t)

(for-each
 (lambda (arg)
   (test (symbol->string arg) 'error))
 (list #\a 1 "hi" '() (list 1) '(1 . 2) #f (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string->symbol arg) 'error)
   (test (symbol arg) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (symbol? (string->symbol (string arg))) #t)
   (test (symbol? (symbol (string arg))) #t))
 (list #\; #\, #\. #\) #\( #\" #\' #\` #\x33 #\xff #\x7f #\# #\]))

(test (symbol) 'error)
(test (symbol "hi" "ho") 'error)



;;; symbol->value
(let ((sym 0))
  (test (symbol->value 'sym) 0)
  (for-each
   (lambda (arg)
     (set! sym arg)
     (test (symbol->value 'sym) arg))
   (list #\a 1 '() (list 1) '(1 . 2) #f (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
	 3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1)))))

(for-each
 (lambda (arg)
   (test (symbol->value arg) 'error)
   (test (symbol->value 'abs arg) 'error))
 (list #\a 1 '() (list 1) "hi" '(1 . 2) #f (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1))))
  
(test (symbol->value) 'error)
(test (symbol->value 'hi 'ho) 'error)

(test (symbol->value 'abs (initial-environment)) abs)
(test (symbol->value 'abs (global-environment)) abs)
(test (symbol->value 'lambda) lambda)
(test (symbol->value 'do) do)
(test (symbol->value do) 'error)
(test (symbol->value 'macroexpand) macroexpand)
(test (symbol->value 'quasiquote) quasiquote)
(test (symbol->value 'else) else)
(test (symbol->value :hi) :hi)
(test (symbol->value hi:) hi:)
(test (symbol->value '#<eof>) 'error) ; ??
(test (symbol? '#<eof>) #f)
(test (let ((a1 32)) (let () (symbol->value 'a1 (current-environment)))) 32)
(test (let ((a1 32)) (let ((a1 0)) (symbol->value 'a1 (current-environment)))) 0)
(test (let ((a1 32)) (let ((a1 0)) (symbol->value 'b1 (current-environment)))) #<undefined>)
(test (symbol->value 'abs '()) 'error)
(test (let ((a1 (let ((b1 32)) (lambda () b1)))) (symbol->value 'b1 (procedure-environment a1))) 32)
(test (let ((x #f)) (set! x (let ((a1 (let ((b1 32)) (lambda () b1)))) a1)) (symbol->value 'b1 (procedure-environment x))) 32)
(test (symbol->value 'if) if)
(test (symbol->value if) 'error)
(test ((symbol->value (define (hi a) (+ a 1))) 2) 3)
(test ((symbol->value (define-macro (hi a) `(+ ,a 1))) 2) 3)
(test (let ((mac (symbol->value (define-macro (hi a) `(+ ,a 1))))) (mac 3)) 4)



(test (let ((name "hiho"))
	(string-set! name 2 #\null)
	(symbol? (string->symbol name)))
      #t)


#|
(let ((str "(let ((X 3)) X)"))
  (do ((i 0 (+ i 1)))
      ((= i 256))
    (catch #t
	   (lambda ()
	     (if (symbol? (string->symbol (string (integer->char i))))
		 (catch #t
			(lambda ()
			  (set! (str 7) (integer->char i))
			  (set! (str 13) (integer->char i))
			  (let ((val (eval-string str)))
			    (format #t "ok: ~S -> ~S~%" str val)))
			(lambda args
			  (format #t "bad but symbol: ~S~%" str))))) ; 11 12 # ' , . 
	   (lambda args
	     (format #t "bad: ~C~%" (integer->char i))))))  ; # ( ) ' " . ` nul 9 10 13 space 0..9 ;

(let ((str "(let ((XY 3)) XY)"))
  (do ((i 0 (+ i 1)))
      ((= i 256))
    (do ((k 0 (+ k 1)))
	((= k 256))
      (catch #t
	     (lambda ()
	       (if (symbol? (string->symbol (string (integer->char i))))
		   (catch #t
			  (lambda ()
			    (set! (str 7) (integer->char i))
			    (set! (str 8) (integer->char k))
			    (set! (str 14) (integer->char i))
			    (set! (str 15) (integer->char k))
			    (let ((val (eval-string str)))
			      (format #t "ok: ~S -> ~S~%" str val)))
			  (lambda args
			    (format #t "bad but symbol: ~S~%" str))))) ; 11 12 # ' , . 
	     (lambda args
	       (format #t "bad: ~C~%" (integer->char i)))))))  ; # ( ) ' " . ` nul 9 10 13 space 0..9 ;
|#



;;; --------------------------------------------------------------------------------
;;; LISTS
;;; --------------------------------------------------------------------------------

;;; cons
(test (cons 'a '()) '(a))
(test (cons '(a) '(b c d)) '((a) b c d))
(test (cons "a" '(b c)) '("a" b c))
(test (cons 'a 3) '(a . 3))
(test (cons '(a b) 'c) '((a b) . c))
(test (cons '() '()) '(()))
(test (cons '() 1) '(() . 1))
(test (cons 1 2) '(1 . 2))
(test (cons 1 '()) '(1))
(test (cons '() 2) '(() . 2))
(test (cons 1 (cons 2 (cons 3 (cons 4 '())))) '(1 2 3 4))
(test (cons 'a 'b) '(a . b))
(test (cons 'a (cons 'b (cons 'c '()))) '(a b c))
(test (cons 'a (list 'b 'c 'd)) '(a b c d))
(test (cons 'a (cons 'b (cons 'c 'd))) '(a b c . d))
(test '(a b c d e) '(a . (b . (c . (d . (e . ()))))))
(test (cons (cons 1 2) (cons 3 4)) '((1 . 2) 3 . 4))
(test (list (cons 1 2) (cons 3 4)) '((1 . 2) (3 . 4)))
(test (cons (cons 1 (cons 2 3)) 4) '((1 . (2 . 3)) . 4))
(test (cons (cons 1 (cons 2 '())) (cons 1 2)) '((1 2) . (1 . 2)))
(test (let ((lst (list 1 2))) (list (apply cons lst) lst)) '((1 . 2) (1 2)))
(test (let ((lst (list 1 2))) (list lst (apply cons lst))) '((1 2) (1 . 2)))
(test (cdadr (let ((lst (list 1 2))) (list (apply cons lst) lst))) '(2))
(test (cons '+ '=) '(+ . =))
(test (cons .(cadddr 10)) (cons cadddr 10))
(test (cons 1 '()) '(
                      1
		       ))



;;; car
(test (car (list 1 2 3)) 1)
(test (car (cons 1 2)) 1)
(test (car (list 1)) 1)
(test (car '(1 2 3)) 1)
(test (car '(1)) 1)
(test (car '(1 . 2)) 1)
(test (car '((1 2) 3)) '(1 2))
(test (car '(((1 . 2) . 3) 4)) '((1 . 2) . 3))
(test (car (list (list) (list 1 2))) '())
(test (car '(a b c)) 'a)
(test (car '((a) b c d)) '(a))
(test (car (reverse (list 1 2 3 4))) 4)
(test (car (list 'a 'b 'c 'd 'e 'f 'g)) 'a)
(test (car '(a b c d e f g)) 'a)
(test (car '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((((1 2 3) 4) 5) (6 7)))
(test (car '(a)) 'a)
(test (car '(1 ^ 2)) 1)
(test (car '(1 .. 2)) 1)
(test (car ''foo) 'quote)
(test (car '(1 2 . 3)) 1)
(test (car (cons 1 '())) 1)
(test (car (if #f #f)) 'error)
(test (car '()) 'error)
(test (car #(1 2)) 'error)
(test (car '#(1 2)) 'error)

(for-each
 (lambda (arg)
   (if (not (equal? (car (cons arg '())) arg))
       (format #t ";(car '(~A)) returned ~A?~%" arg (car (cons arg '()))))
   (test (car arg) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(test (reinvert 12 car (lambda (a) (cons a '())) '(1)) '(1))



;;; cdr
(test (cdr (list 1 2 3)) '(2 3))
(test (cdr (cons 1 2)) 2)
(test (cdr (list 1)) '())
(test (cdr '(1 2 3)) '(2 3))
(test (cdr '(1)) '())
(test (cdr '(1 . 2)) 2)
(test (cdr '((1 2) 3)) '(3))
(test (cdr '(((1 . 2) . 3) 4)) '(4))
(test (cdr (list (list) (list 1 2))) '((1 2)))
(test (cdr '(a b c)) '(b c))
(test (cdr '((a) b c d)) '(b c d))
(test (equal? (cdr (reverse (list 1 2 3 4))) 4) #f)
(test (equal? (cdr (list 'a 'b 'c 'd 'e 'f 'g)) 'a) #f)
(test (cdr '((((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f) g)) '(g))
(test (cdr '(a)) '())
(test (cdr '(a b c d e f g)) '(b c d e f g))
(test (cdr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((((u v w) x) y) ((q w e) r) (a b c) e f g))
(test (cdr ''foo) '(foo))
(test (cdr (cons (cons 1 2) (cons 3 4))) '(3 . 4))
(test (cdr '(1 2 . 3)) '(2 . 3))
(test (cdr (if #f #f)) 'error)
(test (cdr '()) 'error)

(for-each
 (lambda (arg)
   (if (not (equal? (cdr (cons '() arg)) arg))
       (format #t ";(cdr '(() ~A) -> ~A?~%" arg (cdr (cons '() arg))))
   (test (cdr arg) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(let* ((a (list 1 2 3))
       (b a))
  (set! (car a) (cadr a)) 
  (set! (cdr a) (cddr a))
  (test a (list 2 3))
  (test b a))

(define (cons-r a b n) (if (= 0 n) (cons a b) (cons (cons-r (+ a 1) (+ b 1) (- n 1)) (cons-r (- a 1) (- b 1) (- n 1)))))
(define (list-r a b n) (if (= 0 n) (list a b) (list (list-r (+ a 1) (+ b 1) (- n 1)) (list-r (- a 1) (- b 1) (- n 1)))))

(define lists (list (list 1 2 3)
		    (cons 1 2)
		    (list 1)
		    (list)
		    (list (list 1 2) (list 3 4))
		    (list (list 1 2) 3)
		    '(1 . 2)
		    '(a b c)
		    '((a) b (c))
		    '((1 2) (3 4))
		    '((1 2 3) (4 5 6) (7 8 9))
		    '(((1) (2) (3)) ((4) (5) (6)) ((7) (8) (9)))
		    '((((1 123) (2 124) (3 125) (4 126)) ((5) (6) (7) (8)) ((9) (10) (11) (12)) ((13) (14) (15) (16)))
		      (((21 127) (22 128) (23 129) (24 130)) ((25) (26) (27) (28)) ((29) (30) (31) (32)) ((33) (34) (35) (36)))
		      (((41 131) (42 132) (43 133) (44 134)) ((45) (46) (47) (48)) ((49) (50) (51) (52)) ((53) (54) (55) (56)))
		      (((61 135) (62 136) (63 137) (64 138)) ((65) (66) (67) (68)) ((69) (70) (71) (72)) ((73) (74) (75) (76)))
		      321)
		    (cons 1 (cons 2 (cons 3 4)))
		    (cons (cons 2 (cons 3 4)) 5)
		    (cons '() 1)
		    (cons 1 '())
		    (cons '() '())
		    (list 1 2 (cons 3 4) 5 (list (list 6) 7))
		    (cons-r 0 0 4)
		    (cons-r 0 0 5)
		    (cons-r 0 0 10)
		    (list-r 0 0 3)
		    (list-r 0 0 7)
		    (list-r 0 0 11)
		    ''a
		    ))


;;; cxr
(define (caar-1 x) (car (car x)))
(define (cadr-1 x) (car (cdr x)))
(define (cdar-1 x) (cdr (car x)))
(define (cddr-1 x) (cdr (cdr x)))
(define (caaar-1 x) (car (car (car x))))
(define (caadr-1 x) (car (car (cdr x))))
(define (cadar-1 x) (car (cdr (car x))))
(define (caddr-1 x) (car (cdr (cdr x))))
(define (cdaar-1 x) (cdr (car (car x))))
(define (cdadr-1 x) (cdr (car (cdr x))))
(define (cddar-1 x) (cdr (cdr (car x))))
(define (cdddr-1 x) (cdr (cdr (cdr x))))
(define (caaaar-1 x) (car (car (car (car x)))))
(define (caaadr-1 x) (car (car (car (cdr x)))))
(define (caadar-1 x) (car (car (cdr (car x)))))
(define (caaddr-1 x) (car (car (cdr (cdr x)))))
(define (cadaar-1 x) (car (cdr (car (car x)))))
(define (cadadr-1 x) (car (cdr (car (cdr x)))))
(define (caddar-1 x) (car (cdr (cdr (car x)))))
(define (cadddr-1 x) (car (cdr (cdr (cdr x)))))
(define (cdaaar-1 x) (cdr (car (car (car x)))))
(define (cdaadr-1 x) (cdr (car (car (cdr x)))))
(define (cdadar-1 x) (cdr (car (cdr (car x)))))
(define (cdaddr-1 x) (cdr (car (cdr (cdr x)))))
(define (cddaar-1 x) (cdr (cdr (car (car x)))))
(define (cddadr-1 x) (cdr (cdr (car (cdr x)))))
(define (cdddar-1 x) (cdr (cdr (cdr (car x)))))
(define (cddddr-1 x) (cdr (cdr (cdr (cdr x)))))

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (format #t ";(~A ~S) -> ~S, (~A-1): ~S?~%" name lst val1 name val2))))
    lists))
 (list 'caar 'cadr 'cdar 'cddr 'caaar 'caadr 'cadar 'cdaar 'caddr 'cdddr 'cdadr 'cddar 
       'caaaar 'caaadr 'caadar 'cadaar 'caaddr 'cadddr 'cadadr 'caddar 'cdaaar 
       'cdaadr 'cdadar 'cddaar 'cdaddr 'cddddr 'cddadr 'cdddar)
 
 (list caar cadr cdar cddr caaar caadr cadar cdaar caddr cdddr cdadr cddar 
       caaaar caaadr caadar cadaar caaddr cadddr cadadr caddar cdaaar 
       cdaadr cdadar cddaar cdaddr cddddr cddadr cdddar)
 
 (list caar-1 cadr-1 cdar-1 cddr-1 caaar-1 caadr-1 cadar-1 cdaar-1 caddr-1 cdddr-1 cdadr-1 cddar-1 
       caaaar-1 caaadr-1 caadar-1 cadaar-1 caaddr-1 cadddr-1 cadadr-1 caddar-1 cdaaar-1 
       cdaadr-1 cdadar-1 cddaar-1 cdaddr-1 cddddr-1 cddadr-1 cdddar-1))



(test (equal? (cadr (list 'a 'b 'c 'd 'e 'f 'g)) 'b) #t)
(test (equal? (cddr (list 'a 'b 'c 'd 'e 'f 'g)) '(c d e f g)) #t)
(test (equal? (caddr (list 'a 'b 'c 'd 'e 'f 'g)) 'c) #t)
(test (equal? (cdddr (list 'a 'b 'c 'd 'e 'f 'g)) '(d e f g)) #t)
(test (equal? (cadddr (list 'a 'b 'c 'd 'e 'f 'g)) 'd) #t)
(test (equal? (cddddr (list 'a 'b 'c 'd 'e 'f 'g)) '(e f g)) #t)
(test (equal? (caadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '((u v w) x)) #t)
(test (equal? (cadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(6 7)) #t)
(test (equal? (cdaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(5)) #t)
(test (equal? (cdadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(y)) #t)
(test (equal? (cddar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '()) #t)
(test (equal? (caaaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(1 2 3)) #t)
(test (equal? (caadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 6) #t)
(test (equal? (caaddr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(q w e)) #t)
(test (equal? (cadaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 5) #t)
(test (equal? (cadadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 'y) #t)
(test (equal? (caddar (list (list (list (list (list 1 2 3) 4) 5) 1 6 (list 5 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 6) #t)
(test (equal? (cadddr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(a b c)) #t)
(test (equal? (cdaaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(4)) #t)
(test (equal? (cdaadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(x)) #t)
(test (equal? (cdadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(7)) #t)

(test (caar '((a) b c d e f g)) 'a)
(test (cadr '(a b c d e f g)) 'b)
(test (cdar '((a b) c d e f g)) '(b))
(test (cddr '(a b c d e f g)) '(c d e f g))
(test (caaar '(((a)) b c d e f g)) 'a)
(test (caadr '(a (b) c d e f g)) 'b)
(test (cadar '((a b) c d e f g)) 'b)
(test (caddr '(a b c d e f g)) 'c)
(test (cdaar '(((a b)) c d e f g)) '(b))
(test (cdadr '(a (b c) d e f g)) '(c))
(test (cddar '((a b c) d e f g)) '(c))
(test (cdddr '(a b c d e f g)) '(d e f g))
(test (caaaar '((((a))) b c d e f g)) 'a)
(test (caaadr '(a ((b)) c d e f g)) 'b)
(test (caadar '((a (b)) c d e f g)) 'b)
(test (caaddr '(a b (c) d e f g)) 'c)
(test (cadaar '(((a b)) c d e f g)) 'b)
(test (cadadr '(a (b c) d e f g)) 'c)
(test (caddar '((a b c) d e f g)) 'c)
(test (cadddr '(a b c d e f g)) 'd)
(test (cdaaar '((((a b))) c d e f g)) '(b))
(test (cdaadr '(a ((b c)) d e f g)) '(c))
(test (cdadar '((a (b c)) d e f g)) '(c))
(test (cdaddr '(a b (c d) e f g)) '(d))
(test (cddaar '(((a b c)) d e f g)) '(c))
(test (cddadr '(a (b c d) e f g)) '(d))
(test (cdddar '((a b c d) e f g)) '(d))
(test (cddddr '(a b c d e f g)) '(e f g))
(test (cadr '(1 2 . 3)) 2)
(test (cddr '(1 2 . 3)) 3)
(test (cadadr '''1) 1)
(test (cdadr '''1) '(1))

;; sacla
(test (caar '((a) b c)) 'a)
(test (cadr '(a b c)) 'b)
(test (cdar '((a . aa) b c)) 'aa)
(test (cddr '(a b . c)) 'c)
(test (caaar '(((a)) b c)) 'a)
(test (caadr '(a (b) c)) 'b)
(test (cadar '((a aa) b c)) 'aa)
(test (caddr '(a b c)) 'c)
(test (cdaar '(((a . aa)) b c)) 'aa)
(test (cdadr '(a (b . bb) c)) 'bb)
(test (cddar '((a aa . aaa) b c)) 'aaa)
(test (cdddr '(a b c . d)) 'd)
(test (caaaar '((((a))) b c)) 'a)
(test (caaadr '(a ((b)) c)) 'b)
(test (caadar '((a (aa)) b c)) 'aa)
(test (caaddr '(a b (c))) 'c)
(test (cadaar '(((a aa)) b c)) 'aa)
(test (cadadr '(a (b bb) c)) 'bb)
(test (caddar '((a aa aaa) b c)) 'aaa)
(test (cadddr '(a b c d)) 'd)
(test (cdaaar '((((a . aa))) b c)) 'aa)
(test (cdaadr '(a ((b . bb)) c)) 'bb)
(test (cdadar '((a (aa . aaa)) b c)) 'aaa)
(test (cdaddr '(a b (c . cc))) 'cc)
(test (cddaar '(((a aa . aaa)) b c)) 'aaa)
(test (cddadr '(a (b bb . bbb) c)) 'bbb)
(test (cdddar '((a aa aaa . aaaa) b c)) 'aaaa)
(test (cddddr '(a b c d . e)) 'e)

(test (caar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((1 2 3) 4) 5))
(test (cadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((u v w) x) y))
(test (cdar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((6 7)))
(test (cddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((q w e) r) (a b c) e f g))
(test (caaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((1 2 3) 4))
(test (caadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((u v w) x))
(test (cadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(6 7))
(test (caddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((q w e) r))
(test (cdaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(5))
(test (cdadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(y))
(test (cddar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '())
(test (cdddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((a b c) e f g))
(test (caaaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(1 2 3))
(test (caaadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(u v w))
(test (caadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 6)
(test (caaddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(q w e))
(test (cadaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 5)
(test (cadadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 'y)
(test (cadddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(a b c))
(test (cdaaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(4))
(test (cdaadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(x))
(test (cdadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(7))
(test (cdaddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(r))
(test (cddaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '())
(test (cddadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '())
(test (cddddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(e f g))

(test (cadr '(a b c d e f g)) 'b)
(test (cddr '(a b c d e f g)) '(c d e f g))
(test (caddr '(a b c d e f g)) 'c)
(test (cdddr '(a b c d e f g)) '(d e f g))
(test (cadddr '(a b c d e f g)) 'd)
(test (cddddr '(a b c d e f g)) '(e f g))

(test (caar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((a . b) c . d))
(test (caar '(((a . b) c . d) (e . f) g . h)) '(a . b))
(test (caar '((a . b) c . d)) 'a)
(test (cadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((i . j) k . l))
(test (cadr '(((a . b) c . d) (e . f) g . h)) '(e . f))
(test (cadr '((a . b) c . d)) 'c)
(test (cdar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((e . f) g . h))
(test (cdar '(((a . b) c . d) (e . f) g . h)) '(c . d))
(test (cdar '((a . b) c . d)) 'b)
(test (cddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((m . n) o . p))
(test (cddr '(((a . b) c . d) (e . f) g . h)) '(g . h))
(test (cddr '((a . b) c . d)) 'd)
(test (caaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(a . b))
(test (caaar '(((a . b) c . d) (e . f) g . h)) 'a)
(test (caadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(i . j))
(test (caadr '(((a . b) c . d) (e . f) g . h)) 'e)
(test (cddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(g . h))
(test (cddar '(((a . b) c . d) (e . f) g . h)) 'd)
(test (cdddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(o . p))
(test (cdddr '(((a . b) c . d) (e . f) g . h)) 'h)
(test (caaaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'a)
(test (caaadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'i)
(test (caddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'g)
(test (cadddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'o)
(test (cdaaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'b)
(test (cdaadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'j)
(test (cdddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'h)
(test (cddddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'p)

(test (cadr ''foo) 'foo)

(test (caar '((a) b c)) 'a)
(test (cadr '(a b c)) 'b)
(test (cdar '((a . aa) b c)) 'aa)
(test (cddr '(a b . c)) 'c)
(test (caaar '(((a)) b c)) 'a)
(test (caadr '(a (b) c)) 'b)
(test (cadar '((a aa) b c)) 'aa)
(test (caddr '(a b c)) 'c)
(test (cdaar '(((a . aa)) b c)) 'aa)
(test (cdadr '(a (b . bb) c)) 'bb)
(test (cddar '((a aa . aaa) b c)) 'aaa)
(test (cdddr '(a b c . d)) 'd)
(test (caaaar '((((a))) b c)) 'a)
(test (caaadr '(a ((b)) c)) 'b)
(test (caadar '((a (aa)) b c)) 'aa)
(test (caaddr '(a b (c))) 'c)
(test (cadaar '(((a aa)) b c)) 'aa)
(test (cadadr '(a (b bb) c)) 'bb)
(test (caddar '((a aa aaa) b c)) 'aaa)
(test (cadddr '(a b c d)) 'd)
(test (cdaaar '((((a . aa))) b c)) 'aa)
(test (cdaadr '(a ((b . bb)) c)) 'bb)
(test (cdadar '((a (aa . aaa)) b c)) 'aaa)
(test (cdaddr '(a b (c . cc))) 'cc)
(test (cddaar '(((a aa . aaa)) b c)) 'aaa)
(test (cddadr '(a (b bb . bbb) c)) 'bbb)
(test (cdddar '((a aa aaa . aaaa) b c)) 'aaaa)
(test (cddddr '(a b c d . e)) 'e)

(test (recompose 10 cdr '(1 2 3 4 5 6 7 8 9 10 11 12)) '(11 12))
(test (recompose 10 car '(((((((((((1 2 3)))))))))))) '(1 2 3))

(test (cons 1 . 2) 'error)
(test (eval-string "(1 . 2 . 3)") 'error)
(test (car (list)) 'error)
(test (car '()) 'error)
(test (cdr (list)) 'error)
(test (cdr '()) 'error)
(test (caddar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 'error)
(test (cdddar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 'error)
(test (caar '(a b c d e f g)) 'error)
(test (cdar '(a b c d e f g)) 'error)
(test (caaar '(a b c d e f g)) 'error)
(test (caadr '(a b c d e f g)) 'error)
(test (cadar '(a b c d e f g)) 'error)
(test (cdaar '(a b c d e f g)) 'error)
(test (cdadr '(a b c d e f g)) 'error)
(test (cddar '(a b c d e f g)) 'error)
(test (caaaar '(a b c d e f g)) 'error)
(test (caaadr '(a b c d e f g)) 'error)
(test (caadar '(a b c d e f g)) 'error)
(test (caaddr '(a b c d e f g)) 'error)
(test (cadaar '(a b c d e f g)) 'error)
(test (cadadr '(a b c d e f g)) 'error)
(test (caddar '(a b c d e f g)) 'error)
(test (cdaaar '(a b c d e f g)) 'error)
(test (cdaadr '(a b c d e f g)) 'error)
(test (cdadar '(a b c d e f g)) 'error)
(test (cdaddr '(a b c d e f g)) 'error)
(test (cddaar '(a b c d e f g)) 'error)
(test (cddadr '(a b c d e f g)) 'error)
(test (cdddar '(a b c d e f g)) 'error)
(test (caar 'a) 'error)
(test (caar '(a)) 'error)
(test (cadr 'a) 'error)
(test (cadr '(a . b)) 'error)
(test (cdar 'a) 'error)
(test (cdar '(a . b)) 'error)
(test (cddr 'a) 'error)
(test (cddr '(a . b)) 'error)
(test (caaar 'a) 'error)
(test (caaar '(a)) 'error)
(test (caaar '((a))) 'error)
(test (caadr 'a) 'error)
(test (caadr '(a . b)) 'error)
(test (caadr '(a b)) 'error)
(test (cadar 'a) 'error)
(test (cadar '(a . b)) 'error)
(test (cadar '((a . c) . b)) 'error)
(test (caddr 'a) 'error)
(test (caddr '(a . b)) 'error)
(test (caddr '(a c . b)) 'error)
(test (cdaar 'a) 'error)
(test (cdaar '(a)) 'error)
(test (cdaar '((a . b))) 'error)
(test (cdadr 'a) 'error)
(test (cdadr '(a . b)) 'error)
(test (cdadr '(a b . c)) 'error)
(test (cddar 'a) 'error)
(test (cddar '(a . b)) 'error)
(test (cddar '((a . b) . b)) 'error)
(test (cdddr 'a) 'error)
(test (cdddr '(a . b)) 'error)
(test (cdddr '(a c . b)) 'error)
(test (caaaar 'a) 'error)
(test (caaaar '(a)) 'error)
(test (caaaar '((a))) 'error)
(test (caaaar '(((a)))) 'error)
(test (caaadr 'a) 'error)
(test (caaadr '(a . b)) 'error)
(test (caaadr '(a b)) 'error)
(test (caaadr '(a (b))) 'error)
(test (caadar 'a) 'error)
(test (caadar '(a . b)) 'error)
(test (caadar '((a . c) . b)) 'error)
(test (caadar '((a c) . b)) 'error)
(test (caaddr 'a) 'error)
(test (caaddr '(a . b)) 'error)
(test (caaddr '(a c . b)) 'error)
(test (caaddr '(a c b)) 'error)
(test (cadaar 'a) 'error)
(test (cadaar '(a)) 'error)
(test (cadaar '((a . b))) 'error)
(test (cadaar '((a b))) 'error)
(test (cadadr 'a) 'error)
(test (cadadr '(a . b)) 'error)
(test (cadadr '(a b . c)) 'error)
(test (cadadr '(a (b . e) . c)) 'error)
(test (caddar 'a) 'error)
(test (caddar '(a . b)) 'error)
(test (caddar '((a . b) . b)) 'error)
(test (caddar '((a b . c) . b)) 'error)
(test (cadddr 'a) 'error)
(test (cadddr '(a . b)) 'error)
(test (cadddr '(a c . b)) 'error)
(test (cadddr '(a c e . b)) 'error)
(test (cdaaar 'a) 'error)
(test (cdaaar '(a)) 'error)
(test (cdaaar '((a))) 'error)
(test (cdaaar '(((a . b)))) 'error)
(test (cdaadr 'a) 'error)
(test (cdaadr '(a . b)) 'error)
(test (cdaadr '(a b)) 'error)
(test (cdaadr '(a (b . c))) 'error)
(test (cdadar 'a) 'error)
(test (cdadar '(a . b)) 'error)
(test (cdadar '((a . c) . b)) 'error)
(test (cdadar '((a c . d) . b)) 'error)
(test (cdaddr 'a) 'error)
(test (cdaddr '(a . b)) 'error)
(test (cdaddr '(a c . b)) 'error)
(test (cdaddr '(a c b . d)) 'error)
(test (cddaar 'a) 'error)
(test (cddaar '(a)) 'error)
(test (cddaar '((a . b))) 'error)
(test (cddaar '((a b))) 'error)
(test (cddadr 'a) 'error)
(test (cddadr '(a . b)) 'error)
(test (cddadr '(a b . c)) 'error)
(test (cddadr '(a (b . e) . c)) 'error)
(test (cdddar 'a) 'error)
(test (cdddar '(a . b)) 'error)
(test (cdddar '((a . b) . b)) 'error)
(test (cdddar '((a b . c) . b)) 'error)
(test (cddddr 'a) 'error)
(test (cddddr '(a . b)) 'error)
(test (cddddr '(a c . b)) 'error)
(test (cddddr '(a c e . b)) 'error)




;;; length
(test (length (list 'a 'b 'c 'd 'e 'f)) 6)
(test (length (list 'a 'b 'c 'd)) 4)
(test (length (list 'a (list 'b 'c) 'd)) 3)
(test (length '()) 0)
(test (length '(this-that)) 1)
(test (length '(this - that)) 3)
(test (length '(a b)) 2)
(test (length '(a b c)) 3)
(test (length '(a (b) (c d e))) 3)
(test (length (list 1 (cons 1 2))) 2)
(test (length (list 1 (cons 1 '()))) 2)

(for-each
 (lambda (arg)
   (test (length arg) 'error))
 (list (integer->char 65) #f 'a-symbol abs quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(test (length 'x) 'error)
(test (length (cons 1 2)) -1)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (infinite? (length x)) #t))
(test (length '(1 2 . 3)) -2)
(test (length) 'error)
(test (length '(1 2 3) #(1 2 3)) 'error)



;;; reverse
(test (reverse '(a b c d)) '(d c b a))
(test (reverse '(a b c))  '(c b a))
(test (reverse '(a (b c) d (e (f))))  '((e (f)) d (b c) a))
(test (reverse '()) '())
(test (reverse (list 1 2 3)) '(3 2 1))
(test (reverse (list 1)) '(1))
(test (reverse (list)) (list))
(test (reverse '(1 2 3)) (list 3 2 1))
(test (reverse '(1)) '(1))
(test (reverse '((1 2) 3)) '(3 (1 2)))
(test (reverse '(((1 . 2) . 3) 4)) '(4 ((1 . 2) . 3)))
(test (reverse (list (list) (list 1 2))) '((1 2) ()))
(test (reverse '((a) b c d)) '(d c b (a)))
(test (reverse (reverse (list 1 2 3 4))) (list 1 2 3 4))
(test (reverse ''foo) '(foo quote))
(test (let ((x (list 1 2 3 4)))
	(let ((y (reverse x)))
	  (and (equal? x (list 1 2 3 4))
	       (equal? y (list 4 3 2 1)))))
      #t)
(test (letrec ((hi (lambda (lst n)
		     (if (= n 0)
			 lst
			 (hi (reverse lst) (- n 1))))))
	(hi (list 1 2 3) 100))
      (list 1 2 3))
(test (let ((var (list 1 2 3))) (reverse (cdr var)) var) (list 1 2 3))
(test (let ((var '(1 2 3))) (reverse (cdr var)) var) '(1 2 3))
(test (let ((var (list 1 (list 2 3)))) (reverse (cdr var)) var) (list 1 (list 2 3)))
(test (let ((var '(1 (2 3)))) (reverse (cdr var)) var) '(1 (2 3)))
(test (let ((var (list (list 1 2) (list 3 4 5)))) (reverse (car var)) var) '((1 2) (3 4 5)))
(test (let ((x '(1 2 3))) (list (reverse x) x)) '((3 2 1) (1 2 3)))
(test (reverse '(1 2)) '(2 1))
(test (reverse '(1 2 3)) '(3 2 1))
(test (reverse '(1 2 3 4)) '(4 3 2 1))

(for-each
 (lambda (lst)
   (if (list? lst)
       (if (not (equal? lst (reverse (reverse lst))))
	   (format #t ";(reverse (reverse ~A)) -> ~A?~%" lst (reverse (reverse lst))))))
 lists)

(for-each
 (lambda (lst)
   (if (list? lst)
       (if (not (equal? lst (reverse (reverse (reverse (reverse lst))))))
	   (format #t ";(reverse...(4x) ~A) -> ~A?~%" lst (reverse (reverse (reverse (reverse lst))))))))
 lists)

(test (let ((x (list 1 2 3))) (list (recompose 32 reverse x) x)) '((1 2 3) (1 2 3)))
(test (let ((x (list 1 2 3))) (list (recompose 31 reverse x) x)) '((3 2 1) (1 2 3)))

(test (reverse (cons 1 2)) '(2 . 1))
(test (reverse '(1 . 2)) '(2 . 1))
(test (reverse '(1 2 . 3)) '(3 2 1))
(test (reverse) 'error)
(test (reverse '(1 2 3) '(3 2 1)) 'error)

(for-each
 (lambda (arg)
   (test (reverse arg) 'error))
 (list (integer->char 65) #f 'a-symbol abs quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))



;;; reverse!
(test (reverse! '(1 . 2)) 'error)
(test (reverse! (cons 1 2)) 'error)
(test (reverse! (cons 1 (cons 2 3))) 'error)
(test (reverse!) 'error)
(test (reverse! '(1 2 3) '(3 2 1)) 'error)

(test (reverse! '(a b c d)) '(d c b a))
(test (reverse! '(a b c))  '(c b a))
(test (reverse! '(a (b c) d (e (f))))  '((e (f)) d (b c) a))
(test (reverse! '()) '())
(test (reverse! (list 1 2 3)) '(3 2 1))
(test (reverse! (list 1)) '(1))
(test (reverse! (list)) (list))
(test (reverse! '(1 2 3)) (list 3 2 1))
(test (reverse! '(1)) '(1))
(test (reverse! '((1 2) 3)) '(3 (1 2)))
(test (reverse! '(((1 . 2) . 3) 4)) '(4 ((1 . 2) . 3)))
(test (reverse! (list (list) (list 1 2))) '((1 2) ()))
(test (reverse! '((a) b c d)) '(d c b (a)))
(test (reverse! (reverse! (list 1 2 3 4))) (list 1 2 3 4))
(test (reverse! ''foo) '(foo quote))
(test (reverse (reverse! (list 1 2 3))) (list 1 2 3))
(test (reverse (reverse! (reverse! (reverse (list 1 2 3))))) (list 1 2 3))

(test (let ((x (list 1 2 3))) (recompose 31 reverse! x)) '(3 2 1))
(test (reverse! '(1 2 . 3)) 'error)

(let* ((lst1 (list 1 2 3))
       (lst2 (apply list '(4 5 6)))
       (lst3 (sort! (reverse! (append lst1 lst2)) <)))
  (test lst3 '(1 2 3 4 5 6))
  (define (lt . args)
    args)
  (set! lst3 (sort! (apply reverse! (lt lst3)) >))
  (test lst3 '(6 5 4 3 2 1)))

(for-each
 (lambda (arg)
   (test (reverse! arg) 'error))
 (list (integer->char 65) #f 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) #(1 2 3) "hiho" (lambda (a) (+ a 1))))



;;; pair?
(test (pair? 'a) #f)
(test (pair? '()) #f)
(test (pair? '(a b c)) #t)
(test (pair? (cons 1 2)) #t)
(test (pair? ''()) #t)
(test (pair? #f) #f)
(test (pair? (make-vector 6)) #f)
(test (pair? #t) #f)
(test (pair? '(a . b)) #t)
(test (pair? '#(a b))  #f)
(test (pair? (list 1 2)) #t)
(test (pair? (list)) #f)
(test (pair? ''foo) #t)
(test (pair? (list 'a 'b 'c 'd 'e 'f)) #t)
(test (pair? '(this-that)) #t)
(test (pair? '(this - that)) #t)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (pair? x) #t))
(test (pair? (list 1 (cons 1 2))) #t)
(test (pair? (list 1 (cons 1 '()))) #t)
(test (pair? (cons 1 '())) #t)
(test (pair? (cons '() '())) #t)
(test (pair? (cons '() 1)) #t)
(test (pair? (list (list))) #t)
(test (pair? '(())) #t)
(test (pair? (cons 1 (cons 2 3))) #t)
(test (pair?) 'error)
(test (pair? `'1) #t)
(test (pair? begin) #f)
(test (pair? 'begin) #f)
(test (pair? ''begin) #t)
(test (pair? list) #f)

(for-each
 (lambda (arg)
   (if (pair? arg)
       (format #t ";(pair? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))



;;; list?
(test (list? 'a) #f)
(test (list? '()) #t)
(test (list? '(a b c)) #t)
(test (list? (cons 1 2)) #f)
(test (list? ''()) #t)
(test (list? #f) #f)
(test (list? (make-vector 6)) #f)
(test (list? #t) #f)
(test (list? '(a . b)) #f)
(test (list? '#(a b))  #f)
(test (list? (list 1 2)) #t)
(test (list? (list)) #t)
(test (list? ''foo) #t)
(test (list? ''2) #t)
(test (list? (list 'a 'b 'c 'd 'e 'f)) #t)
(test (list? '(this-that)) #t)
(test (list? '(this - that)) #t)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (list? x) #f))
(test (list? (list 1 (cons 1 2))) #t)
(test (list? (list 1 (cons 1 '()))) #t)
(test (list? (cons 1 '())) #t)
(test (list? (cons '() '())) #t)
(test (list? (cons '() 1)) #f)
(test (list? (list (list))) #t)
(test (list? '(())) #t)
(test (list? '(1 2 . 3)) #f)
(test (list? (cons 1 (cons 2 3))) #f)
(test (list? '(1 . ())) #t)

(test (list? '(1 2) '()) 'error)
(test (list?) 'error)
(for-each
 (lambda (arg)
   (if (list? arg)
       (format #t ";(list? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))



;;; null?
(test (null? 'a) '#f)
(test (null? '()) #t)
(test (null? '(a b c)) #f)
(test (null? (cons 1 2)) #f)
(test (null? ''()) #f)
(test (null? #f) #f)
(test (null? (make-vector 6)) #f)
(test (null? #t) #f)
(test (null? '(a . b)) #f)
(test (null? '#(a b))  #f)
(test (null? (list 1 2)) #f)
(test (null? (list)) #t)
(test (null? ''foo) #f)
(test (null? (list 'a 'b 'c 'd 'e 'f)) #f)
(test (null? '(this-that)) #f)
(test (null? '(this - that)) #f)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (null? x) #f))
(test (null? (list 1 (cons 1 2))) #f)
(test (null? (list 1 (cons 1 '()))) #f)
(test (null? (cons 1 '())) #f)
(test (null? (cons '() '())) #f)
(test (null? (cons '() 1)) #f)
(test (null? (list (list))) #f)
(test (null? '(())) #f)
(test (null? '#()) #f)
(test (null? (make-vector '(2 0 3))) #f)
(test (null? "") #f)
(test (null? lambda) #f)
(test (null? cons) #f)

(test (null? () '()) 'error)
(test (null?) 'error)

(for-each
 (lambda (arg)
   (if (null? arg)
       (format #t ";(null? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) #<eof> #<undefined> (values) (lambda (a) (+ a 1))))



;;; set-car!
(test (let ((x (cons 1 2))) (set-car! x 3) x) (cons 3 2))
(test (let ((x (list 1 2))) (set-car! x 3) x) (list 3 2))
(test (let ((x (list (list 1 2) 3))) (set-car! x 22) x) (list 22 3))
(test (let ((x (cons 1 2))) (set-car! x '()) x) (cons '() 2))
(test (let ((x (list 1 (list 2 3 4)))) (set-car! x (list 5 (list 6))) x) (list (list 5 (list 6)) (list 2 3 4)))
(test (let ((x '(((1) 2) (3)))) (set-car! x '((2) 1)) x) '(((2) 1) (3)))
(test (let ((x ''foo)) (set-car! x "hi") x) (list "hi" 'foo))
(test (let ((x '((1 . 2) . 3))) (set-car! x 4) x) '(4 . 3))
(test (let ((x '(1 . 2))) (set-car! x (cdr x)) x) '(2 . 2))
(test (let ((x '(1 . 2))) (set-car! x x) (list? x)) #f)
(test (let ((x (list 1))) (set-car! x '()) x) '(()))
(test (let ((x '(((1 2) . 3) 4))) (set-car! x 1) x) '(1 4))
(test (let ((lst (cons 1 (cons 2 3)))) (set-car! (cdr lst) 4) lst) (cons 1 (cons 4 3)))
(test (let ((lst (cons 1 (cons 2 3)))) (set-car! lst 4) lst) (cons 4 (cons 2 3)))
(test (let ((x (list 1 2))) (set! (car x) 0) x) (list 0 2))
(test (let ((x (cons 1 2))) (set! (cdr x) 0) x) (cons 1 0))
(test (let ((x (list 1 2))) (set-car! x (list 3 4)) x) '((3 4) 2))
(test (let ((x (cons 1 2))) (set-car! x (list 3 4)) x) '((3 4) . 2))
(test (let ((x (cons (list 1 2) 3))) (set-car! (car x) (list 3 4)) x) '(((3 4) 2) . 3))

(test (set-car! '() 32) 'error)
(test (set-car! () 32) 'error)
(test (set-car! (list) 32) 'error)
(test (set-car! 'x 32) 'error)
(test (set-car! #f 32) 'error)
(test (set-car!) 'error)
(test (set-car! '(1 2) 1 2) 'error)
(test (let ((c (cons 1 2))) (set-car! c #\a) (car c)) #\a)
(test (let ((c (cons 1 2))) (set-car! c #()) (car c)) #())
(test (let ((c (cons 1 2))) (set-car! c #f) (car c)) #f)
(test (let ((c (cons 1 2))) (set-car! c _ht_) (car c)) _ht_)


;;; set-cdr!
(test (let ((x (cons 1 2))) (set-cdr! x 3) x) (cons 1 3))
(test (let ((x (list 1 2))) (set-cdr! x 3) x) (cons 1 3))
(test (let ((x (list (list 1 2) 3))) (set-cdr! x 22) x) '((1 2) . 22))
(test (let ((x (cons 1 2))) (set-cdr! x '()) x) (list 1))
(test (let ((x (list 1 (list 2 3 4)))) (set-cdr! x (list 5 (list 6))) x) '(1 5 (6)))
(test (let ((x '(((1) 2) (3)))) (set-cdr! x '((2) 1)) x) '(((1) 2) (2) 1))
(test (let ((x ''foo)) (set-cdr! x "hi") x) (cons 'quote "hi"))
(test (let ((x '((1 . 2) . 3))) (set-cdr! x 4) x) '((1 . 2) . 4))
(test (let ((x '(1 . 2))) (set-cdr! x (cdr x)) x) '(1 . 2))
(test (let ((x '(1 . 2))) (set-cdr! x x) (list? x)) #f)
(test (let ((x (list 1))) (set-cdr! x '()) x) (list 1))
(test (let ((x '(1 . (2 . (3 (4 5)))))) (set-cdr! x 4) x) '(1 . 4))
(test (let ((lst (cons 1 (cons 2 3)))) (set-cdr! (cdr lst) 4) lst) (cons 1 (cons 2 4)))
(test (let ((x (cons (list 1 2) 3))) (set-cdr! (car x) (list 3 4)) x) '((1 3 4) . 3))
(test (let ((x (list 1 2))) (set-cdr! x (list 4 5)) x) '(1 4 5))
(test (let ((x (cons 1 2))) (set-cdr! x (list 4 5)) x) '(1 4 5)) ;!
(test (let ((x (cons 1 2))) (set-cdr! x (cons 4 5)) x) '(1 4 . 5))

(test (set-cdr! '() 32) 'error)
(test (set-cdr! () 32) 'error)
(test (set-cdr! (list) 32) 'error)
(test (set-cdr! 'x 32) 'error)
(test (set-cdr! #f 32) 'error)
(test (set-cdr!) 'error)
(test (set-cdr! '(1 2) 1 2) 'error)
(test (let ((c (cons 1 2))) (set-cdr! c #\a) (cdr c)) #\a)
(test (let ((c (cons 1 2))) (set-cdr! c #()) (cdr c)) #())
(test (let ((c (cons 1 2))) (set-cdr! c #f) (cdr c)) #f)
(test (let ((c (cons 1 2))) (set-cdr! c _ht_) (cdr c)) _ht_)
(test (let ((c (cons 1 2))) (set-cdr! c (list 3)) c) '(1 3))




;;; list-ref
(test (list-ref (list 1 2) 1) 2)
(test (list-ref '(a b c d) 2) 'c)
(test (list-ref (cons 1 2) 0) 1) ; !!
(test (list-ref ''foo 0) 'quote)
(test (list-ref '((1 2) (3 4)) 1) '(3 4))
(test (list-ref (list-ref (list (list 1 2) (list 3 4)) 1) 1) 4)
(test (let ((x (list 1 2 3))) (list-ref x (list-ref x 1))) 3)
(test (list-ref '(1 2 . 3) 1) 2)
(test (list-ref '(1 2 . 3) 2) 'error) ; hmm...
(test ('(1 2 . 3) 0) 1)
(test ('(1 . 2) 0) 1)

(test (let ((lst (list 1 2))) (set! (list-ref lst 1) 0) lst) (list 1 0))
(test (((lambda () list)) 'a 'b 'c) '(a b c))
(test (apply ((lambda () list)) (list 'a 'b 'c) (list 'c 'd 'e)) '((a b c) c d e))
(test (((lambda () (values list))) 1 2 3) '(1 2 3))
(test (apply list 'a 'b '(c)) '(a b c))

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (format #t ";(~A ~A) -> ~A ~A?~%" name lst val1 val2))))
    lists))
 (list 'list-ref:0 'list-ref:1 'list-ref:2 'list-ref:3)
 (list car cadr caddr cadddr)
 (list (lambda (l) (list-ref l 0)) (lambda (l) (list-ref l 1)) (lambda (l) (list-ref l 2)) (lambda (l) (list-ref l 3))))

(for-each
 (lambda (arg)
   (test (list-ref (list 1 arg) 1) arg))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 0)) 1)
(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 1)) 1)
(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 100)) 1)

(test (list-ref '((1 2 3) (4 5 6)) 1) '(4 5 6))
(test (list-ref '((1 2 3) (4 5 6)) 1 2) 6)
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1) '((7 8 9) (10 11 12)))
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0) '(7 8 9))
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2) 9)
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 3) 'error)
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2 0) 'error)

(test ('((1 2 3) (4 5 6)) 1) '(4 5 6))
(test ('((1 2 3) (4 5 6)) 1 2) 6)
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1) '((7 8 9) (10 11 12)))
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0) '(7 8 9))
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2) 9)
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 3) 'error)
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2 0) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (L 1)) '(4 5 6))
(test (let ((L '((1 2 3) (4 5 6)))) (L 1 2)) 6)
(test (let ((L '((1 2 3) (4 5 6)))) (L 1 2 3)) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1)) '((7 8 9) (10 11 12)))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1 0)) '(7 8 9))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1 0 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1 0 2 3)) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) ((L 1) 2)) 6)
(test (let ((L '((1 2 3) (4 5 6)))) (((L 1) 2) 3)) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L 1) 0)) '(7 8 9))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (((L 1) 0) 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L 1 0) 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L 1) 0 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((((L 1) 0) 2) 3)) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (list-ref (L 1) 2)) 6)
(test (let ((L '((1 2 3) (4 5 6)))) (list-ref ((L 1) 2) 3)) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-ref (L 1) 0)) '(7 8 9))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((list-ref (L 1) 0) 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-ref (((L 1) 0) 2) 3)) 'error)

(let ((zero 0)
      (one 1)
      (two 2)
      (three 3))
  (test (list-ref '((1 2 3) (4 5 6)) one) '(4 5 6))
  (test (list-ref '((1 2 3) (4 5 6)) 1 two) 6)
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one) '((7 8 9) (10 11 12)))
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero) '(7 8 9))
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero two) 9)
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero three) 'error)
  
  (test ('((1 2 3) (4 5 6)) one) '(4 5 6))
  (test ('((1 2 3) (4 5 6)) 1 two) 6)
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one) '((7 8 9) (10 11 12)))
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero) '(7 8 9))
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero two) 9)
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero three) 'error)
  
  (test (let ((L '((1 2 3) (4 5 6)))) (L one)) '(4 5 6))
  (test (let ((L '((1 2 3) (4 5 6)))) (L 1 two)) 6)
  (test (let ((L '((1 2 3) (4 5 6)))) (L 1 2 3)) 'error)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L one)) '((7 8 9) (10 11 12)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L one zero)) '(7 8 9))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L one zero two)) 9)
  
  (test (let ((L '((1 2 3) (4 5 6)))) ((L one) two)) 6)
  (test (let ((L '((1 2 3) (4 5 6)))) (((L one) two) 3)) 'error)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L one) zero)) '(7 8 9))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (((L one) zero) two)) 9)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L one zero) two)) 9)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L one) 0 two)) 9)
  
  (test (let ((L '((1 2 3) (4 5 6)))) (list-ref (L one) two)) 6)
  (test (let ((L '((1 2 3) (4 5 6)))) (list-ref ((L one) two) 3)) 'error)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-ref (L one) zero)) '(7 8 9))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((list-ref (L one) zero) two)) 9))


(test (list-ref '() 0) 'error)
(test (list-ref (list 1 2) 2) 'error)
(test (list-ref (list 1 2) -1) 'error)
(test (list-ref (list 1 2) 1.3) 'error)
(test (list-ref (list 1 2) 1/3) 'error)
(test (list-ref (list 1 2) 1+2.0i) 'error)
(test (list-ref (cons 1 2) 1) 'error)
(test (list-ref (cons 1 2) 2) 'error)
(test (list-ref (list 1 2 3) (expt 2 32)) 'error)
(test (list-ref '(1 2 3) 1 2) 'error)
(test (list-ref) 'error)
(test (list-ref '(1 2)) 'error)
(test ('(0)) 'error)
(test ((0)) 'error)
(test ('(1 2 3) 1) 2)
(test ((list 1 2 3) 2) 3)
(test ((list)) 'error)
(test ((list 1) 0 0) 'error)
(test ((list 1 (list 2 3)) 1 1) 3)
(test ((append '(3) '() '(1 2)) 0) 3)
(test ((append '(3) '() 1) 0) 3)
(test ((append '(3) '() 1) 1) 'error)
;; this works with 0 because:
(test ((cons 1 2) 0) 1)
(test (list-ref (cons 1 2) 0) 1)
(test (((list (list 1 2 3)) 0) 0) 1)
(test (((list (list 1 2 3)) 0) 1) 2)
(test (((list (list 1 2 3)) 0 1)) 'error) ; see below
(test (let ((lst (list (list 1 2 3)))) (lst 0 1)) 2) 
(test ((list (list 1 2 3)) 0 1) 2)
(test (list-ref (list (list 1 2)) 0 ()) 'error)

(let ((lst (list 1 2)))
  (for-each
   (lambda (arg)
     (test (list-ref (list 1 2) arg) 'error)
     (test ((list 1 2) arg) 'error)
     (test (lst arg) 'error))
   (list "hi" (integer->char 65) #f '(1 2) '() 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
	 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1)))))




;;; list-set!
(test (let ((x (list 1))) (list-set! x 0 2) x) (list 2))
(test (let ((x (cons 1 2))) (list-set! x 0 3) x) '(3 . 2))
(test (let ((x (cons 1 2))) (list-set! x 1 3) x) 'error)
(test (let ((x '((1) 2))) (list-set! x 0 1) x) '(1 2))
(test (let ((x '(1 2))) (list-set! x 1 (list 3 4)) x) '(1 (3 4)))
(test (let ((x ''foo)) (list-set! x 0 "hi") x ) '("hi" foo))
(test (let ((x (list 1 2))) (list-set! x 0 x) (list? x)) #t)
(test (let ((x (list 1 2))) (list-set! x 1 x) (list? x)) #t)
(test (let ((x 2) (lst '(1 2))) (list-set! (let () (set! x 3) lst) 1 23) (list x lst)) '(3 (1 23)))
(test (apply list-set! '((1 2) (3 2)) 1 '(1 2)) 2)

(test (list-set! '(1 2 3) 1 4) 4)
(test (set-car! '(1 2) 4) 4)
(test (set-cdr! '(1 2) 4) 4)
(test (fill! (list 1 2) 4) 4)
(test (fill! '() 1) 1)
(test (list-set! '(1 2 . 3) 1 23) 23)
(test (list-set! '(1 2 . 3) 2 23) 'error)
(test (set! ('(1 2 . 3) 1) 23) 23)

(for-each
 (lambda (arg)
   (test (let ((x (list 1 2))) (list-set! x 0 arg) (list-ref x 0)) arg))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 32) L) '((1 2 3) 32))
(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 0 32) L) '((1 2 3) (32 5 6)))
(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 0 2 32) L) 'error)
(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 3 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 32) L) '(((1 2 3) (4 5 6)) 32))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 0 32) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 0 2 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 0 2 1 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 4 2 32) L) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1) 32) L) '((1 2 3) 32))
(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1 0) 32) L) '((1 2 3) (32 5 6)))
(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1 0 2) 32) L) 'error)
(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1 3) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1) 32) L) '(((1 2 3) (4 5 6)) 32))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 0) 32) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 0 2) 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 0 2 1) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 4 2) 32) L) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (set! ((L 1) 0) 32) L) '((1 2 3) (32 5 6)))
(test (let ((L '((1 2 3) (4 5 6)))) (set! (((L 1) 0) 2) 32) L) 'error)
(test (let ((L '((1 2 3) (4 5 6)))) (set! ((L 1) 3) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L 1) 0) 32) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((L 1) 0) 2) 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((((L 1) 0) 2) 1) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((L 1) 4) 2) 32) L) 'error)
(test (let ((L '(((1 2 3))))) (set! ((L 0) 0 1) 32) L) '(((1 32 3))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L 1 0) 2) 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (L 0 0 1) 32) L) '((((1 2 3) 32) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L 0) 0 1 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L 0 0) 1 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L 0 0 1) 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L 0) 0) 1 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L 0 0) 1) 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((((L 0) 0) 1) 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '(1 2 3))) (let ((L1 (list L))) (set! ((car L1) 1) 32) L)) '(1 32 3))

(let ((zero 0)
      (one 1)
      (two 2)
      (three 3)
      (thirty-two 32))
  (test (let ((L '((1 2 3) (4 5 6)))) (list-set! L one thirty-two) L) '((1 2 3) 32))
  (test (let ((L '((1 2 3) (4 5 6)))) (list-set! L one zero thirty-two) L) '((1 2 3) (32 5 6)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L one thirty-two) L) '(((1 2 3) (4 5 6)) 32))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L one zero thirty-two) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L one zero two thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  
  (test (let ((L '((1 2 3) (4 5 6)))) (set! (L one) thirty-two) L) '((1 2 3) 32))
  (test (let ((L '((1 2 3) (4 5 6)))) (set! (L one zero) thirty-two) L) '((1 2 3) (32 5 6)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L one) thirty-two) L) '(((1 2 3) (4 5 6)) 32))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L one zero) thirty-two) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L one zero two) thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  
  (test (let ((L '((1 2 3) (4 5 6)))) (set! ((L one) zero) thirty-two) L) '((1 2 3) (32 5 6)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L one) zero) thirty-two) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((L one) zero) two) thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  (test (let ((L '(((1 2 3))))) (set! ((L zero) zero one) thirty-two) L) '(((1 32 3))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L one zero) two) thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (L zero zero one) thirty-two) L) '((((1 2 3) 32) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L zero) zero one two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L zero zero) one two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L zero 0 one) two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L zero) zero) one two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L zero zero) one) two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((((L zero) zero) one) two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '(1 2 3))) (let ((L1 (list L))) (set! ((car L1) one) thirty-two) L)) '(1 32 3)))
  
(test (let ((x '(1)) (y '(2))) (set! ((if #t x y) 0) 32) (list x y)) '((32) (2)))
  
(test (list-set! '() 0 1) 'error)
(test (list-set! '() -1 1) 'error)
(test (list-set! '(1) 1 2) 'error)
(test (list-set! '(1 2 3) -1 2) 'error)
(test (list-set! '(1) 1.5 2) 'error)
(test (list-set! '(1) 3/2 2) 'error)
(test (list-set! '(1) 1+3i 2) 'error)
(test (list-set! '(1 2 3) 1 2 3) 'error)
(test (list-set! (list 1 2 3) (expt 2 32)  0) 'error)
(test (list-set! (list 1 2) '() 1) 'error)

(for-each
 (lambda (arg)
   (test (list-set! (list 1 2) arg arg) 'error)
   (test (list-set! arg 1 2) 'error)
   (test (list-set! (list 1 2) arg 1) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))



;;; list
(test (let ((tree1 (list 1 (list 1 2) (list (list 1 2 3)) (list (list (list 1 2 3 4)))))) tree1) '(1 (1 2) ((1 2 3)) (((1 2 3 4)))))
(test (let ((tree2 (list "one" (list "one" "two") (list (list "one" "two" "three"))))) tree2) '("one" ("one" "two") (("one" "two" "three"))))
(test (let ((tree1 (list 1 (list 1 2) (list 1 2 3) (list 1 2 3 4)))) tree1) '(1 (1 2) (1 2 3) (1 2 3 4)))
(test (let ((tree1 (list 1 (list 1 2))) (tree2 (list 1 (list 1 2)))) tree2) '(1 (1 2)))
(test (let ((tree1 (list 1 (list 1 2))) (tree2 (list 1 (list 1 2)))) (eqv? tree1 tree2)) #f)
(test (let ((tree1 (list ''a (list ''b ''c))) (tree2 (list ''a (list ''b ''c)))) tree2) '('a ('b 'c)))
(test (let ((lst (list 1 (list 2 3)))) lst) '(1 (2 3)))
(test (let* ((lst (list 1 (list 2 3))) (slst lst)) slst) '(1 (2 3)))
(test (list 1) '(1))
(test (let ((a 1)) (list a 2)) '(1 2))
(test (let ((a 1)) (list 'a '2)) '(a 2))
(test (let ((a 1)) (list 'a 2)) '(a 2))
(test (list) '())
(test (let ((a (list 1 2))) a) '(1 2))
(test (let ((a (list 1 2))) (list 3 4 'a (car (cons 'b 'c)) (+ 6 -2))) '(3 4 a b 4))
(test (list) '())
(test (length (list quote do map call/cc lambda define if begin set! let let* cond and or for-each)) 15)
(test (list 1(list 2)) '(1(2)))
(test (list 1 2 . 3) 'error)
;(test (list 1 2 , 3) 'error) ; ,3 -> 3 in the reader now
(test (list 1 2 ,@ 3) 'error)



;;; list-tail
(test (list-tail '(1 2 3) 0) '(1 2 3))
(test (list-tail '(1 2 3) 2) '(3))
(test (list-tail '(1 2 3) 3) '())
(test (list-tail '(1 2 3 . 4) 2) '(3 . 4))
(test (list-tail '(1 2 3 . 4) 3) 4)
(test (let ((x (list 1 2 3))) (eq? (list-tail x 2) (cddr x))) #t)
(test (list-tail '() 0) '())
(test (list-tail '() 1) 'error)
(test (list-tail '(1 2 3) 4) 'error)
(test (list-tail '() -1) 'error)
(test (list-tail (list 1 2) 2) '())
(test (list-tail (cons 1 2) 0) '(1 . 2))
(test (list-tail (cons 1 2) 1) 2)
(test (list-tail (cons 1 2) 2) 'error)
(test (list-tail (cons 1 2) -1) 'error)
(test (list-tail ''foo 1) '(foo))
(test (list-tail '((1 2) (3 4)) 1) '((3 4)))
(test (list-tail (list-tail '(1 2 3) 1) 1) '(3))
(test (list-tail (list-tail (list-tail '(1 2 3 4) 1) 1) 1) '(4))
(test (list-tail '(1 2) (list-tail '(0 . 1) 1)) '(2))

(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 0) x))
(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 1) (cdr x)))
(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 100) x))

(let ((x (list 1 2 3)))
  (let ((y (list-tail x 1)))
    (set! (y 1) 32)
    (test (equal? y '(2 32)) #t)
    (test (equal? x '(1 2 32)) #t))) ; list-tail is not like substring

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (format #t ";(~A ~A) -> ~A ~A?~%" name lst val1 val2))))
    lists))
 (list 'list-tail:0 'list-tail:1 'list-tail:2 'list-tail:3 'list-tail:4)
 (list (lambda (l) l) cdr cddr cdddr cddddr)
 (list (lambda (l) (list-tail l 0)) (lambda (l) (list-tail l 1)) (lambda (l) (list-tail l 2)) (lambda (l) (list-tail l 3)) (lambda (l) (list-tail l 4))))

(test (list-tail (list 1 2) 3) 'error)
(test (list-tail (list 1 2) -1) 'error)
(test (list-tail (list 1 2) 1.3) 'error)
(test (list-tail (list 1 2) 1/3) 'error)
(test (list-tail (list 1 2) 1+2.0i) 'error)
(test (list-tail '(1 2 . 3)) 'error)
(test (list-tail '(1 2 . 3) 1) '(2 . 3))
(test (list-tail '(1 2 . 3) 0) '(1 2 . 3))
(test (list-tail (list 1 2 3) (+ 1 (expt 2 32))) 'error)
(test (list-tail) 'error)
(test (list-tail '(1)) 'error)
(test (list-tail '(1) 1 2) 'error)
(test (set! (list-tail (list 1 2 3)) '(32)) 'error) ; should this work?

(for-each
 (lambda (arg)
   (test (list-tail (list 1 2) arg) 'error)
   (test (list-tail arg 0) 'error))
 (list "hi" -1 3 most-negative-fixnum most-positive-fixnum 
       (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) #<eof> #() #(1 2 3) (lambda (a) (+ a 1))))




;;; assq
(let ((e '((a 1) (b 2) (c 3))))
  (test (assq 'a e) '(a 1))
  (test (assq 'b e) '(b 2))
  (test (assq 'd e) #f))
(test (assq (list 'a) '(((a)) ((b)) ((c))))  #f)

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assq #t e) (list #t 1))
    (test (assq #f e) (list #f 2))
    (test (assq 'a e) (list 'a 3))
    (test (assq xcons e) (list xcons 4))
    (test (assq xvect e) (list xvect 5))
    (test (assq xlambda e) (list xlambda 6))
    (test (assq xstr e) (list xstr 7))
    (test (assq car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3))))
  (test (assq 1+i e) #f)
  (test (assq 3.0 e) #f)
  (test (assq 5/3 e) #f))

(test (assq 'x (cdr (assq 'a '((b . 32) (a . ((a . 12) (b . 32) (x . 1))) (c . 1))))) '(x . 1))

(test (assq #f '(#f 2 . 3)) #f)
(test (assq #f '((#f 2) . 3)) '(#f 2))
(test (assq '() '((() 1) (#f 2))) '(() 1))
(test (assq '() '((1) (#f 2))) #f)
(test (assq #() '((#f 1) (() 2) (#() 3))) #f)  ; (eq? #() #()) -> #f

(test (assq 'b '((a . 1) (b . 2) () (c . 3) #f)) '(b . 2))
(test (assq 'c '((a . 1) (b . 2) () (c . 3) #f)) '(c . 3))
(test (assq 'b '((a . 1) (b . 2) () (c . 3) . 4)) '(b . 2))
(test (assq 'c '((a . 1) (b . 2) () (c . 3) . 4)) '(c . 3))
(test (assq 'b (list '(a . 1) '(b . 2) '() '(c . 3) #f)) '(b . 2))
(test (assq 'asdf (list '(a . 1) '(b . 2) '() '(c . 3) #f)) #f)
(test (assq "" (list '("a" . 1) '("" . 2) '(#() . 3))) #f) ; since (eq? "" "") is #f
(test (assq 'a '((a . 1) (a . 2) (a . 3))) '(a . 1)) ; is this specified?
(test (assq 'a '((b . 1) (a . 2) (a . 3))) '(a . 2))

;; check the even/odd cases
(let ((odd '((3 . 1) (a . 2) (3.0 . 3) (b . 4) (3/4 . 5) (c . 6) (#(1) . 7) (d . 8)))
      (even '((e . 1) (3 . 2) (a . 3) (3.0 . 4) (b . 5) (3/4 . 6) (c . 7) (#(1) . 8) (d . 9))))
  (test (assq 'a odd) '(a . 2))
  (test (assq 'a even) '(a . 3))
  (test (assq 3/4 odd) #f)
  (test (assq 3/4 even) #f)
  (test (assq 3.0 odd) #f)
  (test (assq 3.0 even) #f)
  (test (assq #(1) odd) #f)
  (test (assq #(1) even) #f))


;;; assv
(test (assv 1 '(1 2 . 3)) #f)
(test (assv 1 '((1 2) . 3)) '(1 2))

(let ((e '((a 1) (b 2) (c 3))))
  (test (assv 'a e) '(a 1))
  (test (assv 'b e) '(b 2))
  (test (assv 'd e) #f))
(test (assv (list 'a) '(((a)) ((b)) ((c))))  #f)

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assv #t e) (list #t 1))
    (test (assv #f e) (list #f 2))
    (test (assv 'a e) (list 'a 3))
    (test (assv xcons e) (list xcons 4))
    (test (assv xvect e) (list xvect 5))
    (test (assv xlambda e) (list xlambda 6))
    (test (assv xstr e) (list xstr 7))
    (test (assv car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3) (#\a 4) ("hiho" 5))))
  (test (assv 1+i e) '(1+i 1))
  (test (assv 3.0 e) '(3.0 2))
  (test (assv 5/3 e) '(5/3 3))
  (test (assv #\a e) '(#\a 4))
  (test (assv "hiho" e) #f))

(let ((e '(((a) 1) (#(a) 2) ("c" 3))))
  (test (assv '(a) e) #f)
  (test (assv '#(a) e) #f)
  (test (assv (string #\c) e) #f))

(let ((lst '((2 . a) (3 . b))))
  (set-cdr! (assv 3 lst) 'c)
  (test lst '((2 . a) (3 . c))))

(test (assv '() '((() 1) (#f 2))) '(() 1))
(test (assv '() '((1) (#f 2))) #f)
(test (assv #() '((#f 1) (() 2) (#() 3))) #f)  ; (eqv? #() #()) -> #f ??

(test (assv 'b '((a . 1) (b . 2) () (c . 3) #f)) '(b . 2))
(test (assv 'c '((a . 1) (b . 2) () (c . 3) #f)) '(c . 3))
(test (assv 'b '((a . 1) (b . 2) () (c . 3) . 4)) '(b . 2))
(test (assv 'c '((a . 1) (b . 2) () (c . 3) . 4)) '(c . 3))
(test (assv 'asdf '((a . 1) (b . 2) () (c . 3) . 4)) #f)
(test (assv 'd '((a . 1) (b . 2) () (c . 3) (d . 5))) '(d . 5))
(test (assv 'a '((a . 1) (a . 2) (a . 3))) '(a . 1)) ; is this specified?
(test (assv 'a '((b . 1) (a . 2) (a . 3))) '(a . 2))

(let ((odd '((3 . 1) (a . 2) (3.0 . 3) (b . 4) (3/4 . 5) (c . 6) (#(1) . 7) (d . 8)))
      (even '((e . 1) (3 . 2) (a . 3) (3.0 . 4) (b . 5) (3/4 . 6) (c . 7) (#(1) . 8) (d . 9))))
  (test (assv 'a odd) '(a . 2))
  (test (assv 'a even) '(a . 3))
  (test (assv 3 odd) '(3 . 1))
  (test (assv 3 even) '(3 . 2))
  (test (assv 3/4 odd) '(3/4 . 5))
  (test (assv 3/4 even) '(3/4 . 6))
  (test (assv 3.0 odd) '(3.0 . 3))
  (test (assv 3.0 even) '(3.0 . 4))
  (test (assv #(1) odd) #f)
  (test (assv #(1) even) #f))



;;; assoc
(let ((e '((a 1) (b 2) (c 3))))
  (test (assoc 'a e) '(a 1))
  (test (assoc 'b e) '(b 2))
  (test (assoc 'd e) #f))
(test (assoc (list 'a) '(((a)) ((b)) ((c))))  '((a)))

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assoc #t e) (list #t 1))
    (test (assoc #f e) (list #f 2))
    (test (assoc 'a e) (list 'a 3))
    (test (assoc xcons e) (list xcons 4))
    (test (assoc xvect e) (list xvect 5))
    (test (assoc xlambda e) (list xlambda 6))
    (test (assoc xstr e) (list xstr 7))
    (test (assoc car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3) (#\a 4) ("hiho" 5))))
  (test (assoc 1+i e) '(1+i 1))
  (test (assoc 3.0 e) '(3.0 2))
  (test (assoc 5/3 e) '(5/3 3))
  (test (assoc #\a e) '(#\a 4))
  (test (assoc "hiho" e) '("hiho" 5)))

(let ((e '(((a) 1) (#(a) 2) ("c" 3))))
  (test (assoc '(a) e) '((a) 1))
  (test (assoc '#(a) e) '(#(a) 2))
  (test (assoc (string #\c) e) '("c" 3)))

(test (assoc 'a '((b c) (a u) (a i))) '(a u))
(test (assoc 'a '((b c) ((a) u) (a i))) '(a i))
(test (assoc (list 'a) '(((a)) ((b)) ((c))))  '((a)))
(test (assoc 5 '((2 3) (5 7) (11 13))) '(5 7))
(test (assoc 'key '()) #f)
(test (assoc 'key '(() ())) 'error)
(test (assoc '() '()) #f)
(test (assoc 1 '((1 (2)) (((3) 4)))) '(1 (2)))

(test (assoc '() 1) 'error)
(test (assoc (cons 1 2) 1) 'error)
(test (assoc (let ((x (cons 1 2))) (set-cdr! x x)) 1) 'error)
(test (assoc '((1 2) .3) 1) 'error)
(test (assoc ''foo quote) 'error)
(test (assoc 3 '((a . 3)) abs =) 'error)
(test (assoc 1 '(1 2 . 3)) 'error)
(test (assoc 1 '((1 2) . 3)) '(1 2))
(test (assoc 1 '((1) (1 3) (1 . 2))) '(1))
(test (assoc 1 '((1 2 . 3) (1 . 2))) '(1 2 . 3))
(test (assoc '(((1 2))) '((1 2) ((1 2) 3) (((1 2) 3) 4) ((((1 2) 3) 4) 5))) #f)
(test (assoc '(((1 2))) '((1 2) ((1 2)) (((1 2))) ((((1 2)))))) '((((1 2)))))
(test (assoc 'a '((a . 1) (a . 2) (a . 3))) '(a . 1)) ; is this specified?
(test (assoc 'a '((b . 1) (a . 2) (a . 3))) '(a . 2))

(test (assoc '() '((() 1) (#f 2))) '(() 1))
(test (assoc '() '((1) (#f 2))) #f)
(test (assoc #() '((#f 1) (() 2) (#() 3))) '(#() 3))

(for-each
 (lambda (arg)
   (test (assoc arg (list (list 1 2) (list arg 3))) (list arg 3)))
 (list "hi" (integer->char 65) #f 'a-symbol #() abs 3/4 #\f #t (if #f #f)))

(test (assoc 'b '((a . 1) (b . 2) () (c . 3) #f)) '(b . 2))
(test (assoc 'c '((a . 1) (b . 2) () (c . 3) #f)) '(c . 3))
(test (assoc 'b '((a . 1) (b . 2) () (c . 3) . 4)) '(b . 2))
(test (assoc 'c '((a . 1) (b . 2) () (c . 3) . 4)) '(c . 3))
(test (assoc 'c '((a . 1) (b . 2) () (c . 3) (c . 4) . 4)) '(c . 3))
(test (assoc 'asdf '((a . 1) (b . 2) () (c . 3) (c . 4) . 4)) #f)
(test (assoc "" (list '("a" . 1) '("" . 2) '(#() . 3))) '("" . 2))
(test (assoc assoc (list (cons abs 1) (cons assoc 2) (cons + 3))) (cons assoc 2))

(let ((odd '((3 . 1) (a . 2) (3.0 . 3) (b . 4) (3/4 . 5) (c . 6) (#(1) . 7) (d . 8)))
      (even '((e . 1) (3 . 2) (a . 3) (3.0 . 4) (b . 5) (3/4 . 6) (c . 7) (#(1) . 8) (d . 9))))
  (test (assoc 'a odd) '(a . 2))
  (test (assoc 'a even) '(a . 3))
  (test (assoc 3 odd) '(3 . 1))
  (test (assoc 3 even) '(3 . 2))
  (test (assoc 3/4 odd) '(3/4 . 5))
  (test (assoc 3/4 even) '(3/4 . 6))
  (test (assoc 3.0 odd =) '(3 . 1)) 
  (test (assoc 3.0 odd) '(3.0 . 3)) 
  (test (assoc 3.0 even) '(3.0 . 4))
  (test (assoc #(1) odd) '(#(1) . 7))
  (test (assoc #(1) even) '(#(1) . 8)))

(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) =) '(3 . c))
(test (assoc 3 '((1 . a) (2 . b) (31 . c) (4 . d)) =) #f)
(test (assoc 3 '() =) #f)
(test (assoc 3.0 '((1 . a) (2 . b) (3 . c) (4 . d)) =) '(3 . c))
(test (assoc #\a '((#\A . 1) (#\b . 2)) char=?) #f)
(test (assoc #\a '((#\A . 1) (#\b . 2)) char-ci=?) '(#\A . 1))
(test (assoc #\a '((#\A . 1) (#\b . 2)) (lambda (a b) (char-ci=? a b))) '(#\A . 1))
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) #(1)) 'error)
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) abs) 'error)
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) quasiquote) 'error)
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda (a b c) (= a b))) 'error)
(test (assoc 3.0 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda* (a b c) (= a b))) '(3 . c))
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda (a) (= a 1))) 'error)
(test (assoc 4.0 '((1 . a) (2 . b) (3 . c) (4 . d)) (make-procedure-with-setter = =)) '(4 . d))
(test (catch #t (lambda () (assoc 4.0 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda (a b) (error 'assoc a)))) (lambda args (car args))) 'assoc)
(test (call-with-exit (lambda (go) (assoc 4.0 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda (a b) (go 'assoc))))) 'assoc)
(test (assoc 3 '((#\a . 3) (#() . 2) (3.0 . 1) ("3" . 0))) #f)
(test (assoc 3 '((#\a . 3) (#() . 2) (3.0 . 1) ("3" . 0)) (lambda (a b) (= a b))) 'error)
(test (assoc 3 '((#\a . 3) (#() . 2) (3.0 . 1) ("3" . 0)) (lambda (a b) (and (number? b) (= a b)))) '(3.0 . 1)) ; is this order specified?
(test (let ((lst (list (cons 1 2) (cons 3 4) (cons 5 6)))) (set! (cdr (cdr lst)) lst) (assoc 3 lst)) '(3 . 4))
(test (let ((lst '((1 . 2) (3 . 4) . 5))) (assoc 3 lst)) '(3 . 4))
(test (let ((lst '((1 . 2) (3 . 4) . 5))) (assoc 5 lst)) #f)
(test (let ((lst '((1 . 2) (3 . 4) . 5))) (assoc 3 lst =)) '(3 . 4))
(test (let ((lst '((1 . 2) (3 . 4) . 5))) (assoc 5 lst =)) #f)
(test (assoc 3 '((1 . 2) . 3)) #f)
(test (assoc 1 '((1 . 2) . 3)) '(1 . 2))
(test (assoc 3 '((1 . 2) . 3) =) #f)
(test (assoc 1 '((1 . 2) . 3) =) '(1 . 2))
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (and (assoc 2 lst =) lst)) '((1 . 2) (2 . 3) (3 . 4)))
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr lst)) lst) (assoc 2 lst)) '(2 . 3))
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr lst)) lst) (assoc 2 lst =)) '(2 . 3))
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr lst)) lst) (assoc 4 lst)) #f)
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr lst)) lst) (assoc 4 lst =)) #f)
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr (cdr lst))) lst) (assoc 3 lst =)) '(3 . 4))
(test (assoc '(1 2) '((a . 3) ((1 2) . 4))) '((1 2) . 4))
(test (assoc '(1 2) '((a . 3) ((1 2) . (3 4)))) '((1 2) 3 4))
(test (assoc '(1 2) '((a . 3) ((1 2) . (3 . 4)))) '((1 2) 3 . 4))
(test (cdr (assoc '(1 2) '((a . 3) ((1 2) . (3 . 4))))) (cons 3 4))

(test (assoc #t (list 1 2) #()) 'error)
(test (assoc #t (list 1 2) (integer->char 127)) 'error)
(test (assoc #t (list 1 2) (lambda (x y) (+ x 1))) 'error) ; (+ #t 1)
(test (assoc #t (list 1 2) abs) 'error)
(test (assoc #t (list 1 2) (lambda args args)) 'error)
(test (assoc 1 '((3 . 2) 3) =) 'error)
(test (assoc 1 '((1 . 2) 3) =) '(1 . 2)) ; this is like other trailing error unchecked cases -- should we check?



;;; memq
(test (memq 'a '(a b c)) '(a b c))
(test (memq 'a (list 'a 'b 'c)) '(a b c))
(test (memq 'b '(a b c)) '(b c))
(test (memq 'a '(b c d)) #f)
(test (memq (list 'a) '(b (a) c))  #f)
(test (memq 'a '(b a c a d a)) '(a c a d a))
(let ((v (vector 'a))) (test (memq v (list 'a 1.2 v "hi")) (list v "hi")))
(test (memq #f '(1 a #t "hi" #f 2)) '(#f 2))
(test (memq eq? (list 2 eqv? 1 eq?)) (list eq?))
(test (memq eq? (list 2 eqv? 2)) #f)
(test (memq 6 (memq 5 (memq 4 (memq 3 (memq 2 (memq 1 '(1 2 3 4 5 6))))))) '(6))
(test (memq 'a (cons 'a 'b)) '(a . b))
(test (memq 'a (list a b . c)) 'error)
(test (memq) 'error)
(test (memq 'a) 'error)
(test (memq 'a 'b) 'error)
(test (memq 'a '(a b . c)) '(a b . c))
(test (memq 'b '(a b . c)) '(b . c))
(test (memq 'c '(a b . c)) #f) ; or should it be 'c?
(test (memq '() '(1 () 3)) '(() 3))
(test (memq '() '(1 2)) #f)
(test (memq 'a '(c d a b c)) '(a b c))
(test (memq 'a '(c d f b c)) #f)
(test (memq 'a '()) #f)
(test (memq 'a '(c d a b . c)) '(a b . c))
(test (memq 'a '(c d f b . c)) #f)
(test (let ((x (cons 1 2))) (memq x (list x (cons 3 4)))) '((1 . 2) (3 . 4)))
(test (pair? (let ((x (lambda () 1))) (memq x (list 1 2 x 3)))) #t)
(test (memq memq (list abs + memq car)) (list memq car))
(test (memq 'a '(a a a)) '(a a a)) ;?
(test (memq 'a '(b a a)) '(a a))
(test (memq "hi" '(1 "hi" 2)) #f)
(test (let ((str "hi")) (memq str (list 1 str 2))) '("hi" 2))
(test (memq #\a '(1 #f #\a 2)) '(#\a 2))

(let ((odd '(3 a 3.0 b 3/4 c #(1) d))
      (even '(e 3 a 3.0 b 3/4 c #(1) d)))
  (test (memq 'a odd) '(a 3.0 b 3/4 c #(1) d))
  (test (memq 'a even) '(a 3.0 b 3/4 c #(1) d))
  (test (memq 3/4 odd) #f)
  (test (memq 3/4 even) #f)
  (test (memq 3.0 odd) #f)
  (test (memq 3.0 even) #f)
  (test (memq #(1) odd) #f)
  (test (memq #(1) even) #f))



;;; memv
(test (memv 101 '(100 101 102)) '(101 102))
(test (memv 101 (list 100 101 102)) '(101 102))
(test (memv 3.4 '(1.2 2.3 3.4 4.5)) '(3.4 4.5))
(test (memv 3.4 '(1.3 2.5 3.7 4.9)) #f)
(let ((ls (list 'a 'b 'c)))
  (set-car! (memv 'b ls) 'z)
  (test ls '(a z c)))
(test (memv 1 (cons 1 2)) '(1 . 2))
(test (memv 'a (list 'a 'b . 'c)) 'error)
(test (memv 'a '(a b . c)) '(a b . c))
(test (memv 'asdf '(a b . c)) #f)
(test (memv) 'error)
(test (memv 'a) 'error)
(test (memv 'a 'b) 'error)
(test (memv 'c '(a b c)) '(c))
(test (memv 'c '(a b . c)) #f)
(test (memv ''a '('a b c)) #f)
(test (let ((x (cons 1 2))) (memv x (list (cons 1 2) (cons 3 4)))) #f)
(test (let ((x (cons 1 2))) (memv x (list x (cons 3 4)))) '((1 . 2) (3 . 4)))
(test (memv 'a '(a a a)) '(a a a)) ;?
(test (memv 'a '(b a a)) '(a a))
(test (memv "hi" '(1 "hi" 2)) #f)
(test (memv #\a '(1 #f #\a 2)) '(#\a 2))

(let ((odd '(3 a 3.0 b 3/4 c #(1) d))
      (even '(e 3 a 3.0 b 3/4 c #(1) d)))
  (test (memv 'a odd) '(a 3.0 b 3/4 c #(1) d))
  (test (memv 'a even) '(a 3.0 b 3/4 c #(1) d))
  (test (memv 3/4 odd) '(3/4 c #(1) d))
  (test (memv 3/4 even) '(3/4 c #(1) d))
  (test (memv 3.0 odd) '(3.0 b 3/4 c #(1) d))
  (test (memv 3.0 even) '(3.0 b 3/4 c #(1) d))
  (test (memv #(1) odd) #f)
  (test (memv #(1) even) #f))
(test (memv #(1) '(1 #(1) 2)) #f)
(test (memv '() '(1 () 2)) '(() 2))



;;; member
(test (member (list 'a) '(b (a) c)) '((a) c))
(test (member "b" '("a" "c" "b")) '("b"))
(test (member 1 '(3 2 1 4)) '(1 4))
(test (member 1 (list 3 2 1 4)) '(1 4))
(test (member car (list abs car modulo)) (list car modulo))
(test (member do (list quote map do)) (list do))
(test (member 5/2 (list 1/3 2/4 5/2)) '(5/2))
(test (member 'a '(a b c d)) '(a b c d))
(test (member 'b '(a b c d)) '(b c d))
(test (member 'c '(a b c d)) '(c d))
(test (member 'd '(a b c d)) '(d))
(test (member 'e '(a b c d)) #f)
(test (member 1 (cons 1 2)) '(1 . 2))
(test (member 'a (list a b . c)) 'error)
(test (member 1 '(1 2 . 3)) '(1 2 . 3))
(test (member 2 '(1 2 . 3)) '(2 . 3))
(test (member 3 '(1 2 . 3)) #f)
(test (member 4 '(1 2 . 3)) #f)
(test (member) 'error)
(test (member 'a) 'error)
(test (member 'a 'b) 'error)
(test (member '() '(1 2 3)) #f)
(test (member '() '(1 2 ())) '(()))
(test (member #() '(1 () 2 #() 3)) '(#() 3))
(test (member #2d((1 2) (3 4)) '(1 #() #2d((1 2) (1 2)))) #f)
(test (member #2d((1 2) (3 4)) '(1 #() #2d((1 2) (3 4)))) '(#2d((1 2) (3 4))))
(test (let ((x (cons 1 2))) (member x (list (cons 1 2) (cons 3 4)))) '((1 . 2) (3 . 4)))
(test (let ((x (list 1 2))) (member x (list (cons 1 2) (list 1 2)))) '((1 2)))
(test (member ''a '('a b c)) '('a b c))
(test (member 'a '(a a a)) '(a a a)) ;?
(test (member 'a '(b a a)) '(a a))
(test (member (member 3 '(1 2 3 4)) '((1 2) (2 3) (3 4) (4 5))) '((3 4) (4 5)))
(test (member "hi" '(1 "hi" 2)) '("hi" 2))
(test (member #\a '(1 #f #\a 2)) '(#\a 2))

(for-each
 (lambda (arg)
   (test (member arg (list 1 2 arg 3)) (list arg 3)))
 (list "hi" (integer->char 65) #f 'a-symbol abs 3/4 #\f #t (if #f #f) '(1 2 (3 (4))) most-positive-fixnum))

(test (member 3 . (1 '(2 3))) 'error)
(test (member 3 '(1 2 3) = =) 'error)
(test (member 3 . ('(1 2 3))) '(3))
(test (member 3 . ('(1 2 3 . 4))) '(3 . 4))
(test (member . (3 '(1 2 3))) '(3))
(test (member '(1 2) '(1 2)) #f)
(test (member '(1 2) '((1 2))) '((1 2)))
(test (member . '(quote . ((quote)))) #f)
(test (member . '(quote . ((quote) .()))) #f)
(test (member '(((1))) '((((1).()).()).())) '((((1)))))
(test (member '((1)) '(1 (1) ((1)) (((1))))) '(((1)) (((1)))))
(test (member member (list abs car memq member +)) (list member +))

(let ((odd '(3 a 3.0 b 3/4 c #(1) d))
      (even '(e 3 a 3.0 b 3/4 c #(1) d)))
  (test (member 'a odd) '(a 3.0 b 3/4 c #(1) d))
  (test (member 'a even) '(a 3.0 b 3/4 c #(1) d))
  (test (member 3/4 odd) '(3/4 c #(1) d))
  (test (member 3/4 even) '(3/4 c #(1) d))
  (test (member 3.0 odd) '(3.0 b 3/4 c #(1) d))
  (test (member 3.0 even) '(3.0 b 3/4 c #(1) d))
  (test (member #(1) odd) '(#(1) d))
  (test (member #(1) even) '(#(1) d)))

(test (member 3 '(1 2 3 4) =) '(3 4))
(test (member 3 '() =) #f)
(test (member 3 '(1 2 4 5) =) #f)
(test (member 4.0 '(1 2 4 5) =) '(4 5))
(test (member #\a '(#\b #\A #\c) char=?) #f)
(test (member #\a '(#\b #\A #\c) char-ci=?) '(#\A #\c))
(test (member #\a '(#\b #\A #\c) (lambda (a b) (char-ci=? a b))) '(#\A #\c))
(test (char=? (car (member #\a '(#\b #\a))) #\a) #t)
(test (char=? (car (member #\a '(#\b #\a) (lambda (a b) (char=? a b)))) #\a) #t)
(test (member 3 '(1 2 3 4) <) '(4))
(test (member 3 '((1 2) (3 4)) member) '((3 4)))
(test (member 3 '(((1 . 2) (4 . 5)) ((3 . 4))) assoc) '(((3 . 4))))
(test (member '(#f #f #t) '(0 1 2) list-ref) '(2))
(test (let ((v (vector 1 2 3))) (member v (list 0 v) vector-fill!)) '(0 #(0 0 0)))

(test (member 3 '(1 2 3) abs) 'error)
(test (member 3 '(1 2 3) quasiquote) 'error)
(test (member 3 '(1 2 3) (lambda (a b c) (= a b))) 'error)
(test (member 3 '(1 2 3) (lambda* (a b c) (= a b))) '(3))
(test (member 3 '(1 2 3 4) (make-procedure-with-setter = =)) '(3 4))
(test (catch #t (lambda () (member 3 '(1 2 3) (lambda (a b) (error 'member a)))) (lambda args (car args))) 'member)
(test (call-with-exit (lambda (go) (member 3 '(1 2 3) (lambda (a b) (go 'member))))) 'member)
(test (member 'a '(a a a) eq?) '(a a a))
(test (member 'a '(b a a) eqv?) '(a a))
(test (member 3.0 '(1 #\a (3 . 3) abs #() 3+i)) #f)
(test (member 3.0 '(1 #\a (3 . 3) abs #() 3+i) (lambda (a b) (= (real-part a) b))) 'error)
(test (member 3.0 '(1 #\a (3 . 3) abs #() 3+i) (lambda (a b) (and (number? b) (= (real-part b) a)))) '(3+i))
;; is it guaranteed that in the comparison function the value is 1st and the list member 2nd?
(test (member 4 '((1 2 3) (4 5 6) (7 8 9)) member) '((4 5 6) (7 8 9)))
(test (member 4 '(1 2 3) member) 'error)
(test (member 4 '((1 2) (3 5) 7) (lambda (a b) (member a (map (lambda (c) (+ c 1)) b)))) '((3 5) 7))
(test (member 4 '((1 2) (3 5) 7) (lambda (a b) (assoc a (map (lambda (c) (cons (+ c 1) c)) b)))) '((3 5) 7))

(test (member 4 '(1 2 3 4 . 5)) '(4 . 5))
(test (member 4 '(1 2 3 4 . 5) =) '(4 . 5))
(test (member 4 '(1 2 3 . 4)) #f)
(test (member 4 '(1 2 3 . 4) =) #f)
(test (let ((lst (list 1 2 3))) (and (member 2 lst =) lst)) '(1 2 3))
(test (pair? (let ((lst (list 1 2 3))) (set! (cdr (cdr lst)) lst) (member 2 lst))) #t)
(test (pair? (let ((lst (list 1 2 3))) (set! (cdr (cdr lst)) lst) (member 2 lst =))) #t)
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr lst)) lst) (member 4 lst)) #f)
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr lst)) lst) (member 4 lst =)) #f)
(test (pair? (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) lst) (member 3 lst =))) #t)
(test (pair? (let ((lst (list 1 2 3 4))) (set! (cdr (cdr (cdr lst))) (cdr (cdr lst))) (member 3 lst =))) #t)
(test (let ((lst (list 1 2 3 4))) (set! (cdr (cdr (cdr lst))) (cdr (cdr lst))) (member 5 lst =)) #f)
(test (let ((lst (list 1 2 3 4))) (set! (cdr (cdr (cdr lst))) (cdr lst)) (member 4 lst =)) #f)
(test (let ((lst '(1 2 3 5 6 9 10))) (member 3 lst (let ((last (car lst))) (lambda (a b) (let ((result (= (- b last) a))) (set! last b) result))))) '(9 10))
(test (let ((lst '(1 2 3 5 6 9 10))) (member 2 lst (let ((last (car lst))) (lambda (a b) (let ((result (= (- b last) a))) (set! last b) result))))) '(5 6 9 10))
(test (member 1 '() =) #f)
(test (member 1 #(1) =) 'error)
(test (member 3 '(5 4 3 2 1) >) '(2 1))
(test (member 3 '(5 4 3 2 1) >=) '(3 2 1))
(test (member '(1 2) '((1) (1 . 2) (1 2 . 3) (1 2 3) (1 2) 1 . 2)) '((1 2) 1 . 2))
(test (member '(1 2 . 3) '((1) (1 . 2) (1 2 . 3) (1 2 3) (1 2) 1 . 2)) '((1 2 . 3) (1 2 3) (1 2) 1 . 2))

(let ()
  (define-macro (do-list lst . body) 
    `(member #t ,(cadr lst) (lambda (a b) 
			      (let ((,(car lst) b)) 
				,@body 
				#f))))
  (let ((sum 0))
    (do-list (x '(1 2 3)) (set! sum (+ sum x)))
    (test (= sum 6) #t)))

(let ()
  (define (tree-member a lst) 
    (member a lst (lambda (c d) 
		    (if (pair? d) 
			(tree-member c d) 
			(equal? c d)))))
  (test (tree-member 1 '(2 3 (4 1) 5)) '((4 1) 5))
  (test (tree-member -1 '(2 3 (4 1) 5)) #f)
  (test (tree-member 1 '(2 3 ((4 (1) 5)))) '(((4 (1) 5)))))

(let ((lst (list 1 2 3)))
  (set! (cdr (cdr (cdr lst))) lst)
  (test (member 2 lst) (member 2 lst equal?)))

(let ((lst (list 1 2 3)))
  (set! (cdr (cdr (cdr lst))) lst)
  (test (member 4 lst) (member 4 lst equal?)))

(let ((lst (list 1 2 3 4)))
  (set! (cdr (cdr (cdr (cdr lst)))) lst)
  (test (member 4 lst) (member 4 lst equal?)))

(let ((lst (list 1 2 3 4)))
  (set! (cdr (cdr (cdr (cdr lst)))) (cdr lst))
  (test (member 4 lst) (member 4 lst equal?)))

(for-each
  (lambda (arg lst)
    (test (member arg lst eq?) (memq arg lst))
    (test (member arg lst eqv?) (memv arg lst))
    (test (member arg lst equal?) (member arg lst)))
  (list 'a #f (list 'a) 'a 1 3/4 #(1) "hi")
  (list '(a b c) '(1 "hi" #t #f 2) '(b (a) c) '(d a b . c) '(1 3/4 23) '(1 3/4 23) '(a 1 #(1) 23) '(1 "hi" 23)))

(for-each
 (lambda (op)
   (test (op) 'error)
   (for-each
    (lambda (arg)
      (let ((result (catch #t (lambda () (op arg)) (lambda args 'error))))
	(if (not (eq? result 'error))
	    (format #t ";(~A ~A) returned ~A?~%" op arg result))
	(test (op arg '() arg) 'error)
	(test (op arg) 'error)))
    (list '() "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand hook-functions 
	  3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1)))))
 (list cons car cdr set-car! set-cdr! caar cadr cdar cddr caaar caadr cadar cdaar caddr cdddr cdadr cddar 
       caaaar caaadr caadar cadaar caaddr cadddr cadadr caddar cdaaar cdaadr cdadar cddaar cdaddr cddddr cddadr cdddar
       assq assv memq memv list-ref list-tail))

(for-each
 (lambda (op)
   (test (op '(1) '(2)) 'error))
 (list reverse car cdr caar cadr cdar cddr caaar caadr cadar cdaar caddr cdddr cdadr cddar 
       caaaar caaadr caadar cadaar caaddr cadddr cadadr caddar cdaaar cdaadr cdadar cddaar cdaddr cddddr cddadr cdddar
       list-ref list-tail list-set!))

(for-each
 (lambda (op)
   (for-each
    (lambda (arg)
      (let ((result (catch #t (lambda () (op #f arg)) (lambda args 'error))))
	(if (not (eq? result 'error))
	    (format #t ";(~A #f ~A) returned ~A?~%" op arg result))))
    (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
	  3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1)))))
 (list assq assv assoc memq memv member))




;;; append
(test (append '(a b c) '()) '(a b c))
(test (append '() '(a b c)) '(a b c))
(test (append '(a b) '(c d)) '(a b c d))
(test (append '(a b) 'c) '(a b . c))
(test (equal? (append (list 'a 'b 'c) (list 'd 'e 'f) '() '(g)) '(a b c d e f g)) #t)
(test (append (list 'a 'b 'c) (list 'd 'e 'f) '() (list 'g)) '(a b c d e f g))
(test (append (list 'a 'b 'c) 'd) '(a b c . d))
(test (append '() '()) '())
(test (append '() (list 'a 'b 'c)) '(a b c))
(test (append) '())
(test (append '() 1) 1)
(test (append 'a) 'a)
(test (append '(x) '(y))  '(x y))
(test (append '(a) '(b c d)) '(a b c d))
(test (append '(a (b)) '((c)))  '(a (b) (c)))
(test (append '(a b) '(c . d))  '(a b c . d))
(test (append '() 'a)  'a)
(test (append '(a b) (append (append '(c)) '(e) 'f)) '(a b c e . f))
(test (append ''foo 'foo) '(quote foo . foo))
(test (append '() (cons 1 2)) '(1 . 2))
(test (append '() '() '()) '())
(test (append (cons 1 2)) '(1 . 2))

(test (append #f) #f)
(test (append '() #f) #f)
(test (append '(1 2) #f) '(1 2 . #f))
(test (append '() '() #f) #f)
(test (append '() '(1 2) #f) '(1 2 . #f))
(test (append '(1 2) '() #f) '(1 2 . #f))
(test (append '(1 2) '(3 4) #f) '(1 2 3 4 . #f))
(test (append '() '() '() #f) #f)
(test (append '(1 2) '(3 4) '(5 6) #f) '(1 2 3 4 5 6 . #f))
(test (append () () #()) #())
(test (append () ((lambda () #f))) #f)

(test (append (begin) do) do)
(test (append if) if)
(test (append quote) quote)
(test (append 0) 0) ; is this correct?
(test (append '() 0) 0)
(test (append '() '() 0) 0)
(test (let* ((x '(1 2 3)) (y (append x '()))) (eq? x y)) #f) ; check that append returns a new list
(test (let* ((x '(1 2 3)) (y (append x '()))) (equal? x y)) #t)
(test (let* ((x (list 1 2 3)) (y (append x (list)))) (eq? x y)) #f) 
(test (append '(1) 2) '(1 . 2))
(let ((x (list 1 2 3)))
  (let ((y (append x '())))
    (set-car! x 0)
    (test (= (car y) 1) #t)))
(let ((x (list 1 2 3)))
  (let ((y (append x '())))
    (set-cdr! x 0)
    (test (and (= (car y) 1)
	       (= (cadr y) 2)
	       (= (caddr y) 3))
	  #t)))

(test (let ((xx (list 1 2))) (recompose 12 (lambda (x) (append (list (car x)) (cdr x))) xx)) '(1 2))

(test (append 'a 'b) 'error)
(test (append 'a '()) 'error)
(test (append (cons 1 2) '()) 'error)
(test (append '(1) 2 '(3)) 'error)
(test (append '(1) 2 3) 'error)
(test (let ((lst (list 1 2 3))) (append lst lst)) '(1 2 3 1 2 3))
(test (append ''1 ''1) '(quote 1 quote 1))
(test (append '(1 2 . 3) '(4)) 'error)
(test (append '(1 2 . 3)) '(1 2 . 3))
(test (append '(4) '(1 2 . 3)) '(4 1 2 . 3))
(test (append '() 12 . ()) 12)
(test (append '(1) 12) '(1 . 12))
(test (append '(1) 12 . ()) '(1 . 12))
(test (append '() '() '(1) 12) '(1 . 12))
(test (append '(1) '(2) '(3) 12) '(1 2 3 . 12))
(test (append '(((1))) '(((2)))) '(((1)) ((2))))
(test (append '() . (2)) 2)
(test (append . (2)) 2)
(test (append ''() '()) ''())

(for-each
 (lambda (arg)
   (test (append arg) arg)
   (test (append '() arg) arg)
   (test (append '() '() '() arg) arg))
 (list "hi" #\a #f 'a-symbol _ht_ (make-vector 3) abs 1 3.14 3/4 1.0+1.0i 0/0 #t #<unspecified> #<eof> '() #() (list 1 2) (cons 1 2) #(0) (lambda (a) (+ a 1))))
(test (append not) not)



(test (eval-string "(list #b)") 'error)
(test (eval-string "(char? #\\spaces)") 'error)
(test (eval-string "(car '( . 1))") 'error)
(test (eval-string "(car '(. ))") 'error)
(test (eval-string "(car '( . ))") 'error)
(test (eval-string "(car '(. . . ))") 'error)
(test (eval-string "'#( . 1)") 'error)
(test (eval-string "'(1 2 . )") 'error)
(test (eval-string "'#(1 2 . )") 'error)
(test (eval-string "(+ 1 . . )") 'error)
(test (eval-string "(car '(1 . ))") 'error)
(test (eval-string "(car '(1 . . 2))") 'error)
(test (eval-string "'#( . )") 'error)
(test (eval-string "'#(1 . )") 'error)
(test (eval-string "'#(. . . )") 'error)
(test (eval-string "'#(1 . . 2)") 'error)
(test (eval-string "'(. 1)") 'error)
(test (eval-string "'#(. 1)") 'error)
(test (eval-string "'(. )") 'error)
(test (eval-string "'#(. )") 'error)
(test (eval-string "(list 1 . 2)") 'error)
(test (eval-string "(+ 1 . 2)") 'error)
(test (eval-string "(car '@#`')") 'error)
(test (eval-string "(list . )") 'error)
(test (eval-string "'#( .)") 'error)
(test (eval-string "(car '( .))") 'error)
(test (eval-string "(let ((. 3)) .)") 'error)
(test (eval-string "#0d()") 'error)
(test (eval-string "`#0d()") 'error)
(test (eval-string "'#t:") 'error) ; guile interprets this as #t : and complains unbound variable :
(test (eval-string "#t1") 'error)  ;   similarly this is #t 1 in guile
(test (eval-string "'#(1 . 2)") 'error)
(test (eval-string "#(1 2 . 3)") 'error)
(test (eval-string "'#'") 'error)
(test (eval-string "#b") 'error)





;;; --------------------------------------------------------------------------------
;;; VECTORS
;;; --------------------------------------------------------------------------------


;;; vector?
(test (vector? (make-vector 6)) #t)
(test (vector? (make-vector 6 #\a)) #t)
(test (vector? (make-vector 0)) #t)
;; (test (vector? #*1011) #f)
(test (vector? '#(0 (2 2 2 2) "Anna")) #t)
(test (vector? '#()) #t)
(test (vector? '#("hi")) #t)
(test (vector? (vector 1)) #t)
(test (let ((v (vector 1 2 3))) (vector? v)) #t)

(for-each
 (lambda (arg)
   (test (vector? arg) #f))
 (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (vector?) 'error)
(test (vector? #() #(1)) 'error)
(test (vector? begin) #f)
(test (vector? vector?) #f)

;;; make a shared ref -- we'll check it later after enough has happened that an intervening GC is likely

(define check-shared-vector-after-gc #f)
(let ((avect (make-vector '(6 6) 32)))
  (do ((i 0 (+ i 1)))
      ((= i 6))
    (do ((j 0 (+ j 1)))
	((= j 6))
      (set! (avect i j) (cons i j))))
  (set! check-shared-vector-after-gc (avect 3)))




;;; make-vector
(test (let ((v (make-vector 3 #f))) (and (vector? v) (= (vector-length v) 3) (eq? (vector-ref v 1) #f))) #t)
(test (let ((v (make-vector 1 1))) (and (vector? v) (= (vector-length v) 1) (vector-ref v 0))) 1)
(test (let ((v (make-vector 0 1))) (and (vector? v) (= (vector-length v) 0))) #t)
(test (do ((vec (make-vector 5)) (i 0 (+ i 1))) ((= i 5) vec) (vector-set! vec i i)) '#(0 1 2 3 4))
(test (let ((v (make-vector 5))) (for-each (lambda (i) (vector-set! v i (* i i))) '(0 1 2 3 4)) v) '#(0 1 4 9 16))
(test (make-vector 2 'hi) '#(hi hi))
(test (make-vector 0) '#())
(test (make-vector -0) #())
(test (make-vector 0 'hi) '#())
(test (make-vector 3 (make-vector 1 'hi)) '#(#(hi) #(hi) #(hi)))
(test (make-vector 3 '#(hi)) '#(#(hi) #(hi) #(hi)))
(test (make-vector 9/3 (list)) '#(() () ()))
(test (make-vector 3/1 (make-vector 1 (make-vector 1 'hi))) '#(#(#(hi)) #(#(hi)) #(#(hi))))

(test (let ((v (make-vector 3 0))) (set! (vector-ref v 1) 32) v) #(0 32 0))

(for-each
 (lambda (arg)
   (test (vector-ref (make-vector 1 arg) 0) arg))
 (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))

(test (make-vector) 'error)
(test (make-vector 1 #f #t) 'error)
(test (make-vector 1 2 3) 'error)
(test (make-vector most-positive-fixnum) 'error)
(test (make-vector most-negative-fixnum) 'error)
(test (make-vector '(2 -2)) 'error)
(test (make-vector (list 2 -2 -3)) 'error)
(test (make-vector (cons 2 3)) 'error)
(test (make-vector '(2 3 . 4)) 'error)
(test (make-vector '(2 (3))) 'error)
(test (make-vector most-negative-fixnum) 'error)

(for-each
 (lambda (arg)
   (test (make-vector arg) 'error)
   (test (make-vector (list 2 arg)) 'error))
 (list #\a '() -1 #f "hi" 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))




;;; vector
(test (vector 1 2 3) '#(1 2 3))
(test (vector 1 '(2) 3) '#(1 (2) 3))
(test (vector) '#())
(test (vector (vector (vector))) '#(#(#())))
(test (vector (vector) (vector) (vector)) '#(#() #() #()))
(test (vector (list)) '#(()))
(test '#(1 #\a "hi" hi) (vector 1 #\a "hi" 'hi))
(test (let ((v (make-vector 4 "hi")))
	(vector-set! v 0 1)
	(vector-set! v 1 #\a)
	(vector-set! v 3 'hi)
	v)
      '#(1 #\a "hi" hi))
(let ((x 34))
  (test (vector x 'x) '#(34 x)))

(for-each
 (lambda (arg)
   (test (vector-ref (vector arg) 0) arg))
 (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))




;;; vector->list
;;; list->vector
(test (vector->list '#(0)) (list 0))
(test (vector->list (vector)) '())
(test (vector->list '#(a b c)) '(a b c))
(test (vector->list '#(#(0) #(1))) '(#(0) #(1)))
(test (vector? (list-ref (let ((v (vector 1 2))) (vector-set! v 1 v) (vector->list v)) 1)) #t)

(test (list->vector '()) '#())
(test (list->vector '(a b c)) '#(a b c))
(test (list->vector (list (list 1 2) (list 3 4))) '#((1 2) (3 4)))
(test (list->vector ''foo) '#(quote foo))
(test (list->vector (list)) '#())
(test (list->vector (list 1)) '#(1))
(test (list->vector (list (list))) '#(()))
(test (list->vector (list 1 #\a "hi" 'hi)) '#(1 #\a "hi" hi))
(test (list->vector ''1) #(quote 1))
(test (list->vector '''1) #(quote '1))

(for-each
 (lambda (arg)
   (if (list? arg)
       (test (vector->list (list->vector arg)) arg)))
 lists)
(set! lists '())

(test (list->vector (vector->list (vector))) '#())
(test (list->vector (vector->list (vector 1))) '#(1))
(test (vector->list (list->vector (list))) '())
(test (vector->list (list->vector (list 1))) '(1))

(test (reinvert 12 vector->list list->vector #(1 2 3)) #(1 2 3))

(test (vector->list) 'error)
(test (list->vector) 'error)
(test (vector->list #(1) #(2)) 'error)
(test (list->vector '(1) '(2)) 'error)

(for-each
 (lambda (arg)
   (test (vector->list arg) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol "hi" abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (let ((x (cons #\a #\b))) (set-cdr! x x) (list->vector x)) 'error)
(test (list->vector (cons 1 2)) 'error)
(test (list->vector '(1 2 . 3)) 'error)
(test (let ((lst (list #\a #\b))) (set! (cdr (cdr lst)) lst) (list->vector lst)) 'error)
(test (let ((lst (list #\a #\b))) (set! (cdr (cdr lst)) lst) (apply vector lst)) 'error)

(for-each
 (lambda (arg)
   (test (list->vector arg) 'error))
 (list "hi" #\a 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))




;;; vector-length
(test (vector-length (vector)) 0)
(test (vector-length (vector 1)) 1)
(test (vector-length (make-vector 128)) 128)
(test (vector-length '#(a b c d e f)) 6)
(test (vector-length '#()) 0)
(test (vector-length (vector #\a (list 1 2) (vector 1 2))) 3)
(test (vector-length '#(#(#(hi)) #(#(hi)) #(#(hi)))) 3)
(test (vector-length (vector 1 2 3 4)) 4)
(test (vector-length (let ((v (vector 1 2))) (vector-set! v 1 v) v)) 2)
(test (vector-length (let ((v (vector 1 2))) (vector-set! v 1 v) (vector-ref v 1))) 2)

(test (vector-length) 'error)
(test (vector-length #(1) #(2)) 'error)

(for-each
 (lambda (arg)
   (test (vector-length arg) 'error))
 (list "hi" #\a 1 '() '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))




;;; vector-ref
(test (vector-ref '#(1 1 2 3 5 8 13 21) 5) 8)
(test (vector-ref '#(1 1 2 3 5 8 13 21) (let ((i (round (* 2 (acos -1))))) (if (inexact? i) (inexact->exact i)  i))) 13)
(test (let ((v (make-vector 1 0))) (vector-ref v 0)) 0)
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref v 1)) (list 2))
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref v 2)) '#(#\a #\a #\a))
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref (vector-ref v 2) 1)) #\a)
(test (vector-ref '#(a b c) 1) 'b)
(test (vector-ref '#(()) 0) '())
(test (vector-ref '#(#()) 0) '#())
(test (vector-ref (vector-ref (vector-ref '#(1 (2) #(3 (4) #(5))) 2) 2) 0) 5)
(test (let ((v (vector 1 2))) (vector-set! v 1 v) (eq? (vector-ref v 1) v)) #t)

(test (vector-ref) 'error)
(test (vector-ref #(1)) 'error)
(test (vector-ref #(1) 0 0) 'error)
(test (vector-ref '() 0) 'error)

(test (let ((v (make-vector 1 0))) (vector-ref v 1)) 'error)
(test (let ((v (make-vector 1 0))) (vector-ref v -1)) 'error)
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref (vector-ref v 2) 3)) 'error)
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref (vector-ref v 3) 0)) 'error)
(test (vector-ref (vector) 0) 'error)
(test (vector-ref '#() 0) 'error)
(test (vector-ref '#() -1) 'error)
(test (vector-ref '#() 1) 'error)
(test (vector-ref #(1 2 3) (floor .1)) 1)
(test (vector-ref #(1 2 3) (floor 0+0i)) 1)

(test (#(1 2) 1) 2)
(test (#(1 2) 1 2) 'error)
(test ((#("hi" "ho") 0) 1) #\i)
(test (((vector (list 1 2) (cons 3 4)) 0) 1) 2)
(test ((#(#(1 2) #(3 4)) 0) 1) 2)
(test ((((vector (vector (vector 1 2) 0) 0) 0) 0) 0) 1)
(test ((((list (list (list 1 2) 0) 0) 0) 0) 0) 1)
(test ((((list (list (list 1 2) 0) 0) 0) 0) ((((vector (vector (vector 1 2) 0) 0) 0) 0) 0)) 2)
(test (#(1 2) -1) 'error)
(test (#()) 'error)
(test (#(1)) 'error)
(test (#2d((1 2) (3 4))) 'error)
(test (apply (make-vector '(1 2))) 'error)

(test (eval-string "#2/3d(1 2)") 'error)
(test (eval-string "#2.1d(1 2)") 'error)
(test (#(#(#(#t))) 0 0 0) #t)


(let ((v #(1 2 3)))
  (for-each
   (lambda (arg)
     (test (vector-ref arg 0) 'error)
     (test (v arg) 'error)
     (test (v arg 0) 'error)
     (test (vector-ref v arg) 'error)
     (test (vector-ref v arg 0) 'error)
     (test (vector-ref #2d((1 2) (3 4)) 0 arg) 'error))
   (list "hi" '() #() #\a -1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
	 3.14 3/4 1.0+1.0i #t (lambda (a) (+ a 1)) (make-hash-table))))


(test (vector-ref '#(#(1 2 3) #(4 5 6)) 1) '#(4 5 6))
(test (vector-ref '#(#(1 2 3) #(4 5 6)) 1 2) 6)
(test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1) '#(#(7 8 9) #(10 11 12)))
(test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0) '#(7 8 9))
(test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2) 9)
(test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 3) 'error)
(test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2 0) 'error)

(test ('#(#(1 2 3) #(4 5 6)) 1) '#(4 5 6))
(test ('#(#(1 2 3) #(4 5 6)) 1 2) 6)
(test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1) '#(#(7 8 9) #(10 11 12)))
(test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0) '#(7 8 9))
(test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2) 9)
(test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 3) 'error)
(test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2 0) 'error)

(test (let ((L '#(#(1 2 3) #(4 5 6)))) (L 1)) '#(4 5 6))
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (L 1 2)) 6)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (L 1 2 3)) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1)) '#(#(7 8 9) #(10 11 12)))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1 0)) '#(7 8 9))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1 0 2)) 9)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1 0 2 3)) 'error)

(test (let ((L '#(#(1 2 3) #(4 5 6)))) ((L 1) 2)) 6)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (((L 1) 2) 3)) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L 1) 0)) '#(7 8 9))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (((L 1) 0) 2)) 9)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L 1 0) 2)) 9)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L 1) 0 2)) 9)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((((L 1) 0) 2) 3)) 'error)

(test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-ref (L 1) 2)) 6)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-ref ((L 1) 2) 3)) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-ref (L 1) 0)) '#(7 8 9))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((vector-ref (L 1) 0) 2)) 9)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-ref (((L 1) 0) 2) 3)) 'error)


(let ((zero 0)
      (one 1)
      (two 2)
      (three 3)
      (thirty-two 32))
  (test (vector-ref '#(#(1 2 3) #(4 5 6)) one) '#(4 5 6))
  (test (vector-ref '#(#(1 2 3) #(4 5 6)) one two) 6)
  (test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one) '#(#(7 8 9) #(10 11 12)))
  (test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero) '#(7 8 9))
  (test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero two) 9)
  
  (test ('#(#(1 2 3) #(4 5 6)) one) '#(4 5 6))
  (test ('#(#(1 2 3) #(4 5 6)) one two) 6)
  (test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one) '#(#(7 8 9) #(10 11 12)))
  (test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero) '#(7 8 9))
  (test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero two) 9)
  
  (test (let ((L '#(#(1 2 3) #(4 5 6)))) (L one)) '#(4 5 6))
  (test (let ((L '#(#(1 2 3) #(4 5 6)))) (L one two)) 6)
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L one)) '#(#(7 8 9) #(10 11 12)))
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L one zero)) '#(7 8 9))
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L one zero two)) 9)
  
  (test (let ((L '#(#(1 2 3) #(4 5 6)))) ((L one) two)) 6)
  (test (let ((L '#(#(1 2 3) #(4 5 6)))) (((L one) two) 3)) 'error)
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L one) zero)) '#(7 8 9))
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (((L one) zero) two)) 9)
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L one zero) two)) 9)
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L one) zero two)) 9)
  
  (test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-ref (L one) two)) 6)
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-ref (L one) zero)) '#(7 8 9))
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((vector-ref (L one) zero) two)) 9))

(test ((#(#(:hi) #\a (3)) (#("hi" 2) 1)) (#2d((#() ()) (0 #(0))) 1 ('(cons 0) 1))) 3)
(test (#(1 2 3) (#(1 2 3) 1)) 3)
(test ((#(#(1 2)) (#(1 0) 1)) (#(3 2 1 0) 2)) 2)
(test (apply min (#(1 #\a (3)) (#(1 2) 1))) 3) ; i.e vector ref here 2 levels -- (#(1 2) 1) is 2 and (#(1 #\a (3)) 2) is (3) 


;;; vector-set!
(test (let ((vec (vector 0 '(2 2 2 2) "Anna"))) (vector-set! vec 1 '("Sue" "Sue")) vec) '#(0 ("Sue" "Sue") "Anna"))
(test (let ((v (vector 1 2 3))) (vector-set! v 1 32) v) '#(1 32 3))
(let ((v (make-vector 8 #f)))
  (for-each
   (lambda (arg)
     (vector-set! v 1 arg)
     (test (vector-ref v 1) arg))
   (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
	 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1)))))
(test (let ((v (vector 1 2 3))) (vector-set! v 1 0) v) '#(1 0 3))
(test (let ((v (vector #f))) (vector-set! v 0 (vector)) v) '#(#()))
(test (let ((v (vector 1 (list 2) (vector 1 2 3)))) (vector-set! (vector-ref v 2) 0 21) v) '#(1 (2) #(21 2 3)))

(test (vector-set! (vector 1 2) 0 4) 4)
(test (vector-set!) 'error)
(test (vector-set! #(1)) 'error)
(test (vector-set! #(1) 0) 'error)
(test (vector-set! #(1) 0 0 1) 'error)
(test (vector-set! #(1) 0 0 1 2 3) 'error)
(test (vector-set! #(1) #(0) 1) 'error)
(test (vector-set! '#(1 2) 0 2) 2)
(test (let ((x 2) (v (vector 1 2))) (vector-set! (let () (set! x 3) v) 1 23) (list x v)) '(3 #(1 23)))

(for-each
 (lambda (arg)
   (test (vector-set! arg 0 0) 'error))
 (list "hi" '() #\a -1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (lambda (a) (+ a 1)) (make-hash-table)))

(let ((v (vector 1 2 3)))
  (for-each
   (lambda (arg)
     (test (vector-set! v arg 0) 'error))
   (list "hi" '() #() #\a -1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
	 3.14 3/4 1.0+1.0i #t (make-vector 3) (lambda (a) (+ a 1)))))

(for-each
 (lambda (arg)
   (test (vector-set! arg 0 0) 'error))
 (list "hi" '() #\a 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(let ((v (vector)))
  (test (vector-set! v 0 0) 'error)
  (test (vector-set! v 1 0) 'error)
  (test (vector-set! v -1 0) 'error))
(test (vector-set! #() 0 123) 'error)
(test (vector-set! #(1 2 3) 0 123) 123)
(test (let ((v #(1 2 3))) (set! (v 0) '(+ 1 2)) v) #((+ 1 2) 2 3))
(test (let ((v #(1 2 3))) (set! (v '(+ 1 1)) 2) v) 'error)
(test (let ((v #(1 2 3))) (set! (v (+ 1 1)) 2) v) #(1 2 2))

(test (let ((g (lambda () '#(1 2 3)))) (vector-set! (g) 0 #\?) (g)) #(#\? 2 3))
(test (let ((g (lambda () '(1 . 2)))) (set-car! (g) 123) (g)) '(123 . 2))
(test (let ((g (lambda () '(1 2)))) (list-set! (g) 0 123) (g)) '(123 2))
(test (let ((g (lambda () (symbol->string 'hi)))) (string-set! (g) 1 #\a) (symbol->string 'hi)) "hi")

(test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-set! L 1 32) L) '#(#(1 2 3) 32))
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-set! L 1 0 32) L) '#(#(1 2 3) #(32 5 6)))
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-set! L 1 0 2 32) L) 'error)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-set! L 1 3 32) L) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 32) L) '#(#(#(1 2 3) #(4 5 6)) 32))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 0 32) L) '#(#(#(1 2 3) #(4 5 6)) #(32 #(10 11 12))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 0 2 32) L) '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 0 2 1 32) L) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 4 2 32) L) 'error)

(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! (L 1) 32) L) '#(#(1 2 3) 32))
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! (L 1 0) 32) L) '#(#(1 2 3) #(32 5 6)))
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! (L 1 0 2) 32) L) 'error)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! (L 1 3) 32) L) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1) 32) L) '#(#(#(1 2 3) #(4 5 6)) 32))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 0) 32) L) '#(#(#(1 2 3) #(4 5 6)) #(32 #(10 11 12))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 0 2) 32) L) '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 0 2 1) 32) L) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 4 2) 32) L) 'error)

(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! ((L 1) 0) 32) L) '#(#(1 2 3) #(32 5 6)))
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! (((L 1) 0) 2) 32) L) 'error)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! ((L 1) 3) 32) L) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! ((L 1) 0) 32) L) '#(#(#(1 2 3) #(4 5 6)) #(32 #(10 11 12))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (((L 1) 0) 2) 32) L) '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! ((((L 1) 0) 2) 1) 32) L) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (((L 1) 4) 2) 32) L) 'error)
(test (let ((L '#(#(#(1 2 3))))) (set! ((L 0) 0 1) 32) L) '#(#(#(1 32 3))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! ((L 1 0) 2) 32) L) '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))

(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! (L 0 0 1) 32) 
	L) 
      '#(#(#(#(1 2 3) 32) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! ((L 0) 0 1 2) 32) 
	L) 
      '#(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! ((L 0 0) 1 2) 32) 
	L) 
      '#(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! ((L 0 0 1) 2) 32) 
	L) 
      '#(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! (((L 0) 0) 1 2) 32) 
	L) 
      '#(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! (((L 0 0) 1) 2) 32) 
	L) 
      '#(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! ((((L 0) 0) 1) 2) 32) 
	L) 
      '#(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))


(test (eq? (car (catch #t (lambda () (set! (#(1)) 2)) (lambda args args))) 'wrong-number-of-args) #t)
(test (eq? (car (catch #t (lambda () (set! (#(1) 0 0) 2)) (lambda args args))) 'wrong-type-arg) #t) ; (vector-set! 1 ...)
(test (eq? (car (catch #t (lambda () (set! ((#(1) 0) 0) 2)) (lambda args args))) 'syntax-error) #t) ; (set! (1 ...))
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (eq? (car (catch #t (lambda () (set! ((L) 1) 32) L) (lambda args args))) 'wrong-number-of-args)) #t)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (eq? (car (catch #t (lambda () (set! ((L)) 32) L) (lambda args args))) 'wrong-number-of-args)) #t)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (eq? (car (catch #t (lambda () (set! ((L 1) 2)) L) (lambda args args))) 'syntax-error)) #t)



;;; vector-fill!
(test (fill! (vector 1 2) 4) 4)

(test (let ((v (vector 1 2 3))) (vector-fill! v 0) v) '#(0 0 0))
(test (let ((v (vector))) (vector-fill! v #f) v) '#())
(let ((v (make-vector 8 #f)))
  (for-each
   (lambda (arg)
     (vector-fill! v arg)
     (test (vector-ref v 1) arg))
   (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
	 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1)))))

(test (let ((str "hi") (v (make-vector 3))) (vector-fill! v str) (string-set! (vector-ref v 0) 1 #\a) str) "ha")
(test (let ((lst (list 1 2)) (v (make-vector 3))) (vector-fill! v lst) (list-set! (vector-ref v 0) 1 #\a) lst) '(1 #\a))

(test (let ((v (vector 1 2 3))) (vector-set! v -1 0)) 'error)
(test (let ((v (vector 1 2 3))) (vector-set! v 3 0)) 'error)
(test (vector-fill! '#(1 2) 2) 2)
(test (vector-fill! #() 0) 0)
(test (vector-fill! (vector) 0) 0)
(test (let ((v (vector 1))) (vector-fill! v 32) (v 0)) 32)
(test (let ((v (make-vector 11 0))) (vector-fill! v 32) (v 10)) 32)
(test (let ((v (make-vector 16 0))) (vector-fill! v 32) (v 15)) 32)
(test (let ((v (make-vector 3 0))) (vector-fill! v 32) (v 1)) 32)
(test (let ((v (make-vector 3 0))) (fill! v 32) (v 1)) 32)

(for-each
 (lambda (arg)
   (test (vector-fill! arg 0) 'error))
 (list "hi" #\a '() 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(if with-bignums
    (let ((v (make-vector 2 0.0)))
      (vector-fill! v 1180591620717411303424)
      (num-test (v 1) (expt 2 70))
      (vector-fill! v 3/1180591620717411303424)
      (num-test (v 0) 3/1180591620717411303424)
      (vector-fill! v 1180591620717411303424.0)
      (num-test (v 1) 1180591620717411303424.0)
      (vector-fill! v (make-rectangular (expt 2 70) 1.0))
      (num-test (v 0) (make-rectangular (expt 2 70) 1.0))))

(let ((v (make-vector 3)))
  (vector-fill! v v)
  (test (v 0) v)
  (set! (v 1) 32)
  (test ((v 0) 1) 32))
  

(test (let ((sum 0)) (for-each (lambda (n) (set! sum (+ sum n))) (vector 1 2 3)) sum) 6)
(test (let ((sum 0)) (for-each (lambda (n m) (set! sum (+ sum n (- m)))) (vector 1 2 3) (vector 4 5 6)) sum) -9)
(test (let () (for-each (lambda (n) (error "oops")) (vector)) #f) #f)
(test (let ((sum 0)) (for-each (lambda (n m p) (set! sum (+ sum n (- m) (* 2 p)))) (vector 1 2 3) (vector 4 5 6) (vector 6 7 8)) sum) 33)
(test (let ((sum 0)) (for-each (lambda (n) (for-each (lambda (m) (set! sum (+ sum (* m n)))) (vector 1 2 3))) (vector 4 5 6)) sum) 90)
(test (call/cc (lambda (return) (for-each (lambda (n) (return "oops")) (vector 1 2 3)))) "oops")
(test (call/cc (lambda (return) (for-each (lambda (n) (if (even? n) (return n))) (vector 1 3 8 7 9 10)))) 8)

(for-each
 (lambda (data)
   (let ((v data)
	 (c #f)
	 (y 0))
     
     (do ((i 0 (+ i 1)))
	 ((= i 10))
       (set! (v i) i))
     
     (let ((tag 
	    (call/cc
	     (lambda (exit)
	       
	       (for-each
		(lambda (x)
		  
		  (call/cc
		   (lambda (return)
		     (set! c return)))
		  
		  (if (and (even? (inexact->exact x))
			   (> x y) 
			   (< x 10)) 
		      (begin 
			(set! (v (inexact->exact y)) 100)
			(set! y x) 
			(exit x)) 
		      (set! y x)))
		v)))))
       
       (if (and (number? tag) (< tag 10))
	   (c)))
     
     (let ((correct (vector 0 100 2 100 4 100 6 100 8 9)))
       (do ((i 0 (+ i 1)))
	   ((= i 10))
	 (if (not (= (correct i) (inexact->exact (v i))))
	     (format #t ";for-each call/cc data: ~A~%" v))))))
 
 (list (make-vector 10)
;       (make-vct 10)
       (make-list 10)))


(test (map (lambda (n) (+ 1 n)) (vector 1 2 3)) '(2 3 4))
(test (map (lambda (n m) (- n m)) (vector 1 2 3) (vector 4 5 6)) '(-3 -3 -3))
(test (map (lambda (n m p) (+ n m p)) (vector 1 2 3) (vector 4 5 6) (vector 6 7 8)) '(11 14 17))
(test (map (lambda (n) (map (lambda (m) (* m n)) (vector 1 2 3))) (vector 4 5 6)) '((4 8 12) (5 10 15) (6 12 18)))
(test (call/cc (lambda (return) (map (lambda (n) (return "oops")) (vector 1 2 3)))) "oops")
(test (call/cc (lambda (return) (map (lambda (n) (if (even? n) (return n))) (vector 1 3 8 7 9 10)))) 8)



(test (vector? (symbol-table)) #t)
(let* ((st (symbol-table))
       (len (length st))
       (loc -1))
  (do ((i 0 (+ i 1)))
      ((or (>= loc 0) 
	   (= i len)))
    (if (pair? (st i))
	(set! loc i)))
  (if (>= loc 0)
      (begin
	(test (symbol? (((symbol-table) loc) 0)) #t)
	(let ((old-table (symbol-table))
	      (old-list ((symbol-table) loc)))
	  ;; try to clobber it...
	  (vector-fill! (symbol-table) #())
	  (set! ((symbol-table) loc) 1)
	  (test (list? ((symbol-table) loc)) #t)
	  (test (symbol? (((symbol-table) loc) 0)) #t)
	  (test (equal? old-list ((symbol-table) loc)) #t)))))

(let ((v (make-vector 3 (vector 1 2))))
  (test (equal? (v 0) (v 1)) #t)
  (test (eq? (v 0) (v 1)) #t)
  (test (eqv? (v 0) (v 1)) #t))

(let ((v (vector (vector 1 2) (vector 1 2) (vector 1 2))))
  (test (equal? (v 0) (v 1)) #t)
  (test (eq? (v 0) (v 1)) #f)
  (test (eqv? (v 0) (v 1)) #f))

(let ((v (vector (vector (vector (vector 1 2) 3) 4) 5)))
  (test (v 0) #(#(#(1 2) 3) 4))
  (test (v 1) 5)
  (test (((v 0) 0) 1) 3)
  (test ((((v 0) 0) 0) 1) 2))

(test (make-vector 1 (make-vector 1 (make-vector 1 0))) #(#(#(0))))


(let ((v1 (make-vector 3 1)))
  (num-test (v1 1) 1)
  (set! (v1 1) 2)
  (num-test (v1 1) 2)
  (let ((i0 0)
	(i2 2))
    (num-test (v1 i0) 1)
    (num-test (vector-ref v1 i2) 1)
    (set! (v1 i0) 0)
    (num-test (v1 0) 0)
    (set! (v1 i0) i2)
    (num-test (v1 i0) i2))
  (test (vector-dimensions v1) '(3))
  (set! v1 (make-vector '(3 2)))
  (test (vector-dimensions v1) '(3 2))
  (vector-set! v1 1 1 0)
  (num-test (vector-ref v1 1 1) 0)
  (let ((i0 1)
	(i1 1)
	(i2 32))
    (set! (v1 i0 i1) i2)
    (num-test (vector-ref v1 1 1) 32)
    (num-test (v1 i0 i1) i2)
    (vector-set! v1 0 1 3)
    (num-test (v1 0 1) 3)
    (num-test (v1 1 1) 32))
  (set! v1 (make-vector '(2 4 3) 1))
  (test (vector-dimensions v1) '(2 4 3))      
  (num-test (vector-ref v1 1 1 1) 1)
  (vector-set! v1 0 0 0 32)
  (num-test (v1 0 0 0) 32)
  (set! (v1 0 1 1) 3)
  (num-test (v1 0 1 1) 3))

(for-each
 (lambda (arg)
   (test (vector-dimensions arg) 'error))
 (list "hi" -1 0 #\a 'a-symbol '(1 . 2) '(1 2 3) 3.14 3/4 1.0+1.0i #t abs #<eof> #<unspecified> (lambda () 1)))
(test (vector-dimensions) 'error)
(test (vector-dimensions #() #()) 'error)
(test (vector-dimensions #()) '(0))
(test (vector-dimensions (vector)) '(0))
(test (vector-dimensions (vector 0)) '(1))
(test (vector-dimensions (vector-ref #2d((1 2 3) (3 4 5)) 0)) '(3))
(test (vector-dimensions (vector-ref #3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) 0)) '(2 3))
(test (vector-dimensions (vector-ref #3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) 0 1)) '(3))
(test (set! (vector-dimensions #(1 2)) 1) 'error)
(test (let ((v #(1 2 3))) (set! (car (vector-dimensions v)) 0) v) #(1 2 3))

(let ((old-len *vector-print-length*))
  (let ((vect1 #3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))))
	(vect2 #2d((1 2 3 4 5 6) (7 8 9 10 11 12)))
	(vect3 #(1 2 3 4 5 6 7 8 9 10 11 12 13 14))
	(vect4 #3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12)))))
    (do ((i 1 (+ i 1)))
	((= i 15))
      (set! *vector-print-length* i)
      (let ((str (object->string vect1)))
	(test str (case i
		    ((1) "#3D(((1 ...)...)...)")
		    ((2) "#3D(((1 2 ...)...)...)")
		    ((3) "#3D(((1 2 3)...)...)")
		    ((4) "#3D(((1 2 3) (3 ...))...)")
		    ((5) "#3D(((1 2 3) (3 4 ...))...)")
		    ((6) "#3D(((1 2 3) (3 4 5))...)")
		    ((7) "#3D(((1 2 3) (3 4 5)) ((5 ...)...))")
		    ((8) "#3D(((1 2 3) (3 4 5)) ((5 6 ...)...))")
		    ((9) "#3D(((1 2 3) (3 4 5)) ((5 6 1)...))")
		    ((10) "#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 ...)))")
		    ((11) "#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 ...)))")
		    ((12) "#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2)))")
		    ((13) "#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2)))")
		    ((14) "#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2)))"))))

      (let ((str (object->string vect4)))
	(test str (case i
		    ((1) "#3D(((1 ...)...)...)")
		    ((2) "#3D(((1 2)...)...)")
		    ((3) "#3D(((1 2) (3 ...)...)...)")
		    ((4) "#3D(((1 2) (3 4)...)...)")
		    ((5) "#3D(((1 2) (3 4) (5 ...))...)")
		    ((6) "#3D(((1 2) (3 4) (5 6))...)")
		    ((7) "#3D(((1 2) (3 4) (5 6)) ((7 ...)...))")
		    ((8) "#3D(((1 2) (3 4) (5 6)) ((7 8)...))")
		    ((9) "#3D(((1 2) (3 4) (5 6)) ((7 8) (9 ...)...))")
		    ((10) "#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10)...))")
		    ((11) "#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 ...)))")
		    ((12) "#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12)))")
		    ((13) "#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12)))")
		    ((14) "#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12)))"))))

      (let ((str (object->string vect2)))
	(test str (case i
		    ((1) "#2D((1 ...)...)")
		    ((2) "#2D((1 2 ...)...)")
		    ((3) "#2D((1 2 3 ...)...)")
		    ((4) "#2D((1 2 3 4 ...)...)")
		    ((5) "#2D((1 2 3 4 5 ...)...)")
		    ((6) "#2D((1 2 3 4 5 6)...)")
		    ((7) "#2D((1 2 3 4 5 6) (7 ...))")
		    ((8) "#2D((1 2 3 4 5 6) (7 8 ...))")
		    ((9) "#2D((1 2 3 4 5 6) (7 8 9 ...))")
		    ((10) "#2D((1 2 3 4 5 6) (7 8 9 10 ...))")
		    ((11) "#2D((1 2 3 4 5 6) (7 8 9 10 11 ...))")
		    ((12) "#2D((1 2 3 4 5 6) (7 8 9 10 11 12))")
		    ((13) "#2D((1 2 3 4 5 6) (7 8 9 10 11 12))")
		    ((14) "#2D((1 2 3 4 5 6) (7 8 9 10 11 12))"))))

      (let ((str (object->string vect3)))
	(test str (case i
		    ((1) "#(1 ...)")
		    ((2) "#(1 2 ...)")
		    ((3) "#(1 2 3 ...)")
		    ((4) "#(1 2 3 4 ...)")
		    ((5) "#(1 2 3 4 5 ...)")
		    ((6) "#(1 2 3 4 5 6 ...)")
		    ((7) "#(1 2 3 4 5 6 7 ...)")
		    ((8) "#(1 2 3 4 5 6 7 8 ...)")
		    ((9) "#(1 2 3 4 5 6 7 8 9 ...)")
		    ((10) "#(1 2 3 4 5 6 7 8 9 10 ...)")
		    ((11) "#(1 2 3 4 5 6 7 8 9 10 11 ...)")
		    ((12) "#(1 2 3 4 5 6 7 8 9 10 11 12 ...)")
		    ((13) "#(1 2 3 4 5 6 7 8 9 10 11 12 13 ...)")
		    ((14) "#(1 2 3 4 5 6 7 8 9 10 11 12 13 14)")))))

    (let ((vect5 (make-vector '(2 3))))
      (set! (vect5 0 0) vect1)
      (set! (vect5 0 1) vect2)
      (set! (vect5 0 2) vect3)
      (set! (vect5 1 0) vect4)
      (set! (vect5 1 1) (vector 1 2 3))
      (set! (vect5 1 2) #2d())

      (do ((i 1 (+ i 1)))
	  ((= i 15))
	(set! *vector-print-length* i)
	(let ((str (object->string vect5)))
	  (test str (case i

		      ((1) "#2D((#3D(((1 ...)...)...) ...)...)")
		      ((2) "#2D((#3D(((1 2 ...)...)...) #2D((1 2 ...)...) ...)...)")
		      ((3) "#2D((#3D(((1 2 3)...)...) #2D((1 2 3 ...)...) #(1 2 3 ...))...)")
		      ((4) "#2D((#3D(((1 2 3) (3 ...))...) #2D((1 2 3 4 ...)...) #(1 2 3 4 ...)) (#3D(((1 2) (3 4)...)...) ...))")
		      ((5) "#2D((#3D(((1 2 3) (3 4 ...))...) #2D((1 2 3 4 5 ...)...) #(1 2 3 4 5 ...)) (#3D(((1 2) (3 4) (5 ...))...) #(1 2 3) ...))")
		      ((6) "#2D((#3D(((1 2 3) (3 4 5))...) #2D((1 2 3 4 5 6)...) #(1 2 3 4 5 6 ...)) (#3D(((1 2) (3 4) (5 6))...) #(1 2 3) #2D()))")
		      ((7) "#2D((#3D(((1 2 3) (3 4 5)) ((5 ...)...)) #2D((1 2 3 4 5 6) (7 ...)) #(1 2 3 4 5 6 7 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 ...)...)) #(1 2 3) #2D()))")
		      ((8) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 ...)...)) #2D((1 2 3 4 5 6) (7 8 ...)) #(1 2 3 4 5 6 7 8 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 8)...)) #(1 2 3) #2D()))")
		      ((9) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 1)...)) #2D((1 2 3 4 5 6) (7 8 9 ...)) #(1 2 3 4 5 6 7 8 9 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 8) (9 ...)...)) #(1 2 3) #2D()))")
		      ((10) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 ...))) #2D((1 2 3 4 5 6) (7 8 9 10 ...)) #(1 2 3 4 5 6 7 8 9 10 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10)...)) #(1 2 3) #2D()))")
		      ((11) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 ...))) #2D((1 2 3 4 5 6) (7 8 9 10 11 ...)) #(1 2 3 4 5 6 7 8 9 10 11 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 ...))) #(1 2 3) #2D()))")
		      ((12) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) #2D((1 2 3 4 5 6) (7 8 9 10 11 12)) #(1 2 3 4 5 6 7 8 9 10 11 12 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12))) #(1 2 3) #2D()))")
		      ((13) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) #2D((1 2 3 4 5 6) (7 8 9 10 11 12)) #(1 2 3 4 5 6 7 8 9 10 11 12 13 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12))) #(1 2 3) #2D()))")
		      ((14) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) #2D((1 2 3 4 5 6) (7 8 9 10 11 12)) #(1 2 3 4 5 6 7 8 9 10 11 12 13 14)) (#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12))) #(1 2 3) #2D()))")))))))

  (set! *vector-print-length* old-len))
  
(let ((v (make-vector '(2 2))))
  (set! (v 0 0) 1)
  (set! (v 0 1) 2)
  (set! (v 1 0) 3)
  (set! (v 1 1) 4)
  (set! (v 0 1) #2d((1 2) (3 4)))
  (test (object->string v) "#2D((1 #2D((1 2) (3 4))) (3 4))"))

(let ((v #2d((1 2) (3 4)))) 
  (set! (v 0 1) #2d((1 2) (3 4))) 
  (test (object->string v) "#2D((1 #2D((1 2) (3 4))) (3 4))"))

(let ((v (make-vector '(2 3))))
  (do ((i 0 (+ i 1)))
      ((= i 2))
    (do ((j 0 (+ j 1)))
	((= j 3))
      (set! (v i j) (list i j))))
  (test (v 0 0) '(0 0))
  (test ((v 1 2) 0) 1)
  (test (v 1 2 0) 'error)
  (test (object->string v) "#2D(((0 0) (0 1) (0 2)) ((1 0) (1 1) (1 2)))"))

(test (let ((v1 (make-vector '(3 2) 1))
	    (v2 (make-vector '(3 2) 2))
	    (sum 0))
	(for-each (lambda (n m) (set! sum (+ sum n m))) v1 v2)
	sum)
      18)
(test (vector->list (make-vector '(2 3) 1)) '(1 1 1 1 1 1))
(test (vector->list #2d((1 2) (3 4))) '(1 2 3 4))
(test (list->vector '((1 2) (3 4))) #((1 2) (3 4)))
(test (vector->list (make-vector (list 2 0))) '())
(test (vector-dimensions #2d((1 2 3))) '(1 3))

(test (#2d((1 2 3) (4 5 6)) 0 0) 1)
(test (#2d((1 2 3) (4 5 6)) 0 1) 2)
(test (#2d((1 2 3) (4 5 6)) 1 1) 5)
(test (#3D(((1 2) (3 4)) ((5 6) (7 8))) 0 0 0) 1)
(test (#3D(((1 2) (3 4)) ((5 6) (7 8))) 1 1 0) 7)
(test (#4d((((1) (2)) ((3) (4)) ((5) (6)))) 0 0 0 0) 1)
(test (vector? #2d((1 2) (3 4))) #t)
(test ((#2d((1 #2d((2 3) (4 5))) (6 7)) 0 1) 1 0) 4)
(test ((((((((((#10D((((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))))) (((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1)))))))))) 0) 0) 0) 0) 0) 0) 0) 0) 0) 0) 1)
(test (#10D((((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))))) (((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1)))))))))) 0 0 0 0 0 0 0 0 0 0) 1)
(let ((v (make-vector (make-list 100 1) 0)))
  (test (equal? v #100D((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((0))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))) #t)
  (test (apply v (make-list 100 0)) 0)
  (test (v 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0) 0))

;;; eval-string here else these are read errors
(test (eval-string "#3D(((1 2) (3 4)) ((5 6) (7)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) ((5) (7 8)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) (() (7 8)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) ((5 6) (7 8 9)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) (5 (7 8 9)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) ((5 6) (7 . 8)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) ((5 6) (7 8 . 9)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) ((5 6) ()))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) ((5 6)))") 'error)

(test (vector-dimensions #3D(((1 2) (3 4)) ((5 6) (7 8)))) '(2 2 2))
(test (vector-dimensions #2d((1 2 3) (4 5 6))) '(2 3))
(test (vector-dimensions #4d((((1) (2)) ((3) (4)) ((5) (6))))) '(1 3 2 1))

(test (vector-length #3D(((1 2) (3 4)) ((5 6) (7 8)))) 8)
(test (length #2d((1 2 3) (4 5 6))) 6)

(test (#2d((1 (2) 3) (4 () 6)) 0 1) '(2))
(test (#2d((1 (2) 3) (4 () 6)) 1 1) '())
(test (#2d((1 (2) 3) (4 6 ())) 1 2) '())
(test (#2d((() (2) ()) (4 5 6)) 0 2) '())

(test (equal? (make-vector 0) (make-vector '(0))) #t)
(test (equal? #() (make-vector '(0))) #t)

(test (equal? #2d((1 2) (3 4)) #2D((1 2) (3 4))) #t)
(test (eq? #2d((1 2) (3 4)) #2D((1 2) (3 4))) #f)
(test (eqv? #2d((1 2) (3 4)) #2D((1 2) (3 4))) #f)
(test (make-vector (1 . 2) "hi") 'error)
(test (make-vector (cons 1 2) "hi") 'error)
(test (equal? (make-vector 0) (vector)) #t)
(test (equal? #() (vector)) #t)

(let ((v (make-vector '(2 3) 0)))
  (num-test (vector-length v) 6)
  (test (vector-dimensions v) '(2 3))
  (num-test (v 0 0) 0)
  (num-test (v 1 2) 0)
  (test (v 2 2) 'error)
  (test (v 2 -1) 'error)
  (test (v 2 0) 'error)
  (set! (v 0 1) 1)
  (num-test (v 0 1) 1)
  (num-test (v 1 0) 0)
  (set! (v 1 2) 2)
  (num-test (v 1 2) 2)
  (test (set! (v 2 2) 32) 'error)
  (test (set! (v 1 -1) 0) 'error)
  (test (set! (v 2 0) 0) 'error)
  (num-test (vector-ref v 0 1) 1)
  (num-test (vector-ref v 1 2) 2)
  (test (vector-ref v 2 2) 'error)
  (test (vector-ref v 1 -1) 'error)
  (vector-set! v 1 1 64)
  (num-test (vector-ref v 1 1) 64)
  (num-test (vector-ref v 0 0) 0)
  (test (vector-ref v 1 2 3) 'error)
  (test (vector-set! v 1 2 3 4) 'error)
  (test (v 1 1 1) 'error)
  (test (set! (v 1 1 1) 1) 'error))



(let ((v1 (make-vector '(3 2) 0))
      (v2 (make-vector '(2 3) 0))
      (v3 (make-vector '(2 3 4) 0))
      (v4 (make-vector 6 0))
      (v5 (make-vector '(2 3) 0)))
  (test (equal? v1 v2) #f)
  (test (equal? v1 v3) #f)
  (test (equal? v1 v4) #f)
  (test (equal? v2 v2) #t)
  (test (equal? v3 v2) #f)
  (test (equal? v4 v2) #f)
  (test (equal? v5 v2) #t)
  (test (equal? v4 v3) #f)
  (test (vector-dimensions v3) '(2 3 4))
  (test (vector-dimensions v4) '(6))
  (num-test (v3 1 2 3) 0)
  (set! (v3 1 2 3) 32)
  (num-test (v3 1 2 3) 32)
  (num-test (vector-length v3) 24)
  (num-test (vector-ref v3 1 2 3) 32)
  (vector-set! v3 1 2 3 -32)
  (num-test (v3 1 2 3) -32)
  (test (v3 1 2) '#(0 0 0 -32))
  (test (set! (v3 1 2) 3) 'error)
  (test (vector-ref v3 1 2) '#(0 0 0 -32))
  (test (vector-set! v3 1 2 32) 'error))

(test (let ((v #2d((1 2) (3 4)))) (vector-fill! v #t) v) #2D((#t #t) (#t #t)))

(test (eval-string "#2d((1 2) #2d((3 4) 5 6))") 'error)
(test (string=? (object->string #2d((1 2) (3 #2d((3 4) (5 6))))) "#2D((1 2) (3 #2D((3 4) (5 6))))") #t)
(test (string=? (object->string #3d(((#2d((1 2) (3 4)) #(1)) (#3d(((1))) 6)))) "#3D(((#2D((1 2) (3 4)) #(1)) (#3D(((1))) 6)))") #t)

(test (make-vector '(2 -2)) 'error)
(test (make-vector '(2 1/2)) 'error)
(test (make-vector '(2 1.2)) 'error)
(test (make-vector '(2 2+i)) 'error)
(test (make-vector '(2 "hi")) 'error)

(let ((v (make-vector '(1 1 1) 32)))
  (test (vector? v) #t)
  (test (equal? v #()) #f)
  (test (vector->list v) '(32))
  (test (vector-ref v 0) '#2D((32)))
  (test (vector-set! v 0 0) 'error)
  (test (vector-ref v 0 0) #(32))
  (test (vector-set! v 0 0 0) 'error)
  (test (vector-ref v 0 0 0) 32)
  (test (let () (vector-set! v 0 0 0 31) (vector-ref v 0 0 0)) 31)
  (test (vector-length v) 1)
  (test (vector-dimensions v) '(1 1 1))
  (test (object->string v) "#3D(((31)))")
  )

(test (vector? #3D(((32)))) #t)
(test (equal? #3D(((32))) #()) #f)
(test (vector->list #3D(((32)))) '(32))
(test (#3D(((32))) 0) '#2D((32)))
(test (set! (#3D(((32))) 0) 0) 'error)
(test (#3D(((32))) 0 0) '#(32))
(test (set! (#3D(((32))) 0 0) 0) 'error)
(test (#3D(((32))) 0 0 0) 32)
(test (vector-length #3D(((32)))) 1)
(test (vector-dimensions #3D(((32)))) '(1 1 1))
(test (object->string #3D(((32)))) "#3D(((32)))")


(let ((v1 (make-vector '(1 0))))
  (test (vector? v1) #t)
  (test (equal? v1 #()) #f)
  (test (vector->list v1) '())
  (test (vector-ref v1 0) 'error)
  (test (vector-set! v1 0 0) 'error)
  (test (vector-ref v1 0 0) 'error)
  (test (vector-set! v1 0 0 0) 'error)
  (test (vector-length v1) 0)
  (test (vector-dimensions v1) '(1 0))
  (test (object->string v1) "#2D()")
  )

(let ((v2 (make-vector '(10 3 0))))
  (test (vector? v2) #t)
  (test (equal? v2 #()) #f)
  (test (vector->list v2) '())
  (test (vector-ref v2) 'error)
  (test (vector-set! v2 0) 'error)
  (test (vector-ref v2 0) 'error)
  (test (vector-set! v2 0 0) 'error)
  (test (vector-ref v2 0 0) 'error)
  (test (vector-set! v2 0 0 0) 'error)
  (test (vector-ref v2 1 2 0) 'error)
  (test (vector-set! v2 1 2 0 0) 'error)
  (test (vector-length v2) 0)
  (test (vector-dimensions v2) '(10 3 0))
  (test (object->string v2) "#3D()")
  )

(let ((v3 (make-vector '(10 0 3))))
  (test (vector? v3) #t)
  (test (equal? v3 #()) #f)
  (test (vector->list v3) '())
  (test (vector-ref v3) 'error)
  (test (vector-set! v3 0) 'error)
  (test (vector-ref v3 0) 'error)
  (test (vector-set! v3 0 0) 'error)
  (test (vector-ref v3 0 0) 'error)
  (test (vector-set! v3 0 0 0) 'error)
  (test (vector-ref v3 1 0 2) 'error)
  (test (vector-set! v3 1 0 2 0) 'error)
  (test (vector-length v3) 0)
  (test (vector-dimensions v3) '(10 0 3))
  (test (object->string v3) "#3D()")
  )

(test (((#(("hi") ("ho")) 0) 0) 1) #\i)
(test (string-ref (list-ref (vector-ref #(("hi") ("ho")) 0) 0) 1) #\i)

(test (equal? #2D() (make-vector '(0 0))) #t)
(test (equal? #2D() (make-vector '(1 0))) #f)
(test (equal? (make-vector '(2 2) 2) #2D((2 2) (2 2))) #t)
(test (equal? (make-vector '(2 2) 2) #2D((2 2) (1 2))) #f)
(test (equal? (make-vector '(1 2 3) 0) (make-vector '(1 2 3) 0)) #t)
(test (equal? (make-vector '(1 2 3) 0) (make-vector '(1 3 2) 0)) #f)
(test (make-vector '1 2 3) 'error)

(test (set! (vector) 1) 'error)
(test (set! (make-vector 1) 1) 'error)
(test (equal? (make-vector 10 '()) (make-hash-table 10)) #f)
(test (equal? #() (copy #())) #t)
(test (equal? #2d() (copy #2d())) #t)
(test (fill! #() 1) 1)
(test (fill! #2d() 1) 1)

(test (equal? #2d((1 2) (3 4)) (copy #2d((1 2) (3 4)))) #t)
(test (equal? #3d() #3d(((())))) #f)
(test (equal? #3d() #3d()) #t)
(test (equal? #1d() #1d()) #t)
(test (equal? #3d() #2d()) #f)
(test (equal? #3d() (copy #3d())) #t)
(test (equal? #2d((1) (2)) #2d((1) (3))) #f)
(test (equal? #2d((1) (2)) (copy #2d((1) (2)))) #t)
(test (equal? (make-vector '(3 0 1)) (make-vector '(3 0 2))) #f)
(test (eval-string "#0d()") 'error)

(let ((v1 (make-vector '(3 2 1) #f))
      (v2 (make-vector '(3 2 1) #f)))
  (test (equal? v1 v2) #t)
  (set! (v2 0 0 0) 1)
  (test (equal? v1 v2) #f))
(test (equal? (make-vector '(3 2 1) #f) (make-vector '(1 2 3) #f)) #f)

(test (map (lambda (n) n) #2d((1 2) (3 4))) '(1 2 3 4))
(test (let ((vals '())) (for-each (lambda (n) (set! vals (cons n vals))) #2d((1 2) (3 4))) vals) '(4 3 2 1))
(test (map (lambda (x y) (+ x y)) #2d((1 2) (3 4)) #1d(4 3 2 1)) '(5 5 5 5))
(test (let ((vals '())) (for-each (lambda (x y) (set! vals (cons (+ x y) vals))) #2d((1 2) (3 4)) #1d(4 3 2 1)) vals) '(5 5 5 5))

(let ((v #2D((#(1 2) #(3 4)) (#2d((5 6) (7 8)) #2D((9 10 11) (12 13 14))))))
  (test (v 0 0) #(1 2))
  (test (v 0 1) #(3 4))
  (test (v 1 0) #2d((5 6) (7 8)))
  (test (v 1 1) #2D((9 10 11) (12 13 14)))
  (test ((v 1 0) 0 1) 6)
  (test ((v 0 1) 1) 4)
  (test ((v 1 1) 1 2) 14))

(let ((v #2D((#((1) #(2)) #(#(3) (4))) (#2d(((5) #(6)) (#(7) #(8))) #2D((#2d((9 10) (11 12)) (13)) (14 15))))))
  (test (v 0 0) #((1) #(2)))
  (test (v 0 1) #(#(3) (4)))
  (test (v 1 0) #2D(((5) #(6)) (#(7) #(8))))
  (test (v 1 1) #2D((#2D((9 10) (11 12)) (13)) (14 15)))
  (test ((v 1 0) 0 1) #(6))
  (test (((v 1 0) 0 1) 0) 6)
  (test ((v 0 1) 1) '(4))
  (test (((v 1 1) 0 0) 1 0) 11))


(test (let ((V #2D((1 2 3) (4 5 6)))) (V 0)) '#(1 2 3))
(test (let ((V #2D((1 2 3) (4 5 6)))) (V 1)) '#(4 5 6))
(test (let ((V #2D((1 2 3) (4 5 6)))) (V 2)) 'error)
(test (let ((V #2D((1 2 3) (4 5 6)))) (set! (V 1) 0)) 'error)
(test (let ((V #2D((1 2 3) (4 5 6)))) (let ((V1 (V 0))) (set! (V1 1) 32) V)) '#2D((1 32 3) (4 5 6)))
(test (let ((V #2D((1 2 3) (4 5 6)))) (let ((V1 (V 0))) (set! (V1 3) 32) V)) 'error)

(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 1)) '#2D((7 8 9) (10 11 12)))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 1 1)) '#(10 11 12))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 0 1)) '#(4 5 6))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 2 1)) 'error)
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((V 0) 1)) '#(4 5 6))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((V 0) 1) 1) 32) V) '#3D(((1 2 3) (4 32 6)) ((7 8 9) (10 11 12))))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-set! V 0 1 1 32) V) '#3D(((1 2 3) (4 32 6)) ((7 8 9) (10 11 12))))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-set! V 1 1 0 32) V) '#3D(((1 2 3) (4 5 6)) ((7 8 9) (32 11 12))))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-length (V 1))) 6)
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-dimensions (V 1))) '(2 3))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-length (V 0 1))) 3)
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-dimensions (V 0 1))) '(3))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12)))) (one 1) (zero 0)) 
	(let ((V1 (V one zero))
	      (sum 0))
	  (for-each (lambda (n) (set! sum (+ sum n))) V1)
	  sum))
      24) ; 7 8 9
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12)))) (two 2) (one 1) (zero 0)) 
	(let ((V10 (V one zero))
	      (V00 (V zero zero))
	      (V01 (V zero one))
	      (V11 (V one one))
	      (sum 0))
	  (for-each (lambda (n0 n1 n2 n3) (set! sum (+ sum n0 n1 n2 n3))) V00 V01 V10 V11)
	  sum))
      78)

(let ((old-vlen *vector-print-length*))
  (set! *vector-print-length* 32)
  (test (object->string (make-vector '(8 8) 0)) "#2D((0 0 0 0 0 0 0 0) (0 0 0 0 0 0 0 0) (0 0 0 0 0 0 0 0) (0 0 0 0 0 0 0 0)...)")
  (test (object->string (make-vector 64 0)) "#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ...)")
  (test (object->string (make-vector 32 0)) "#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0)")
  (test (object->string (make-vector 33 0)) "#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ...)")
  (test (object->string (make-vector '(8 4) 0)) "#2D((0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0))")
  (set! *vector-print-length* old-vlen))

(let ((old-vlen *vector-print-length*))
  (set! *vector-print-length* 1024) ; check the many-() case
  (test (object->string (make-vector '(2 1 2 1 2 1 2 1 2 1 2 1 2 1) 0)) "#14D((((((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))) (((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))))) (((((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))) (((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))))))")

  (test (object->string (make-vector '(16 1 1 1 1 1 1 1 1 1 1 1 1 1) 0)) "#14D((((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))))")

;;; now see if our shared vector has survived...
  (test (and (vector? check-shared-vector-after-gc)
	     (= (length check-shared-vector-after-gc) 6)
	     (do ((i 0 (+ i 1))
		  (happy #t))
		 ((= i 6) happy)
	       (if (or (not (pair? (check-shared-vector-after-gc i)))
		       (not (equal? (check-shared-vector-after-gc i) (cons 3 i))))
		   (set! happy #f))))
	#t)
  (set! check-shared-vector-after-gc #f)

  (set! *vector-print-length* old-vlen))  





;;; -------- circular structures --------

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (apply + lst) 'error))

(let ((l1 (list 1)))
  (test (object->string (list l1 1 l1)) "(#1=(1) 1 #1#)"))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (append '(1) lst)) "(1 . #1=(1 2 3 . #1#))"))
(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (append lst '()) 'error)) 

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (sort! lst <) 'error))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (list lst)) "(#1=(1 2 3 . #1#))"))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (make-list 4 lst)) "(#1=(1 2 3 . #1#) #1# #1# #1#)"))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (vector lst lst)) "#(#1=(1 2 3 . #1#) #1#)"))

(let ((lst `(+ 1 2 3)))
   (set! (cdr (cdddr lst)) (cddr lst)) 
   (test (object->string lst) "(+ 1 . #1=(2 3 . #1#))"))


(let ((x (list 1 2)))
  (test (equal? x x) #t)
  (test (equal? x (cdr x)) #f)
  (test (equal? x '()) #f))
(let ((x (list 1 (list 2 3) (list (list 4 (list 5)))))
      (y (list 1 (list 2 3) (list (list 4 (list 5))))))
  (test (equal? x y) #t))
(let ((x (list 1 (list 2 3) (list (list 4 (list 5)))))
      (y (list 1 (list 2 3) (list (list 4 (list 5) 6)))))
  (test (equal? x y) #f))

(test (length '()) 0)
(test (length (cons 1 2)) -1)
(test (length '(1 2 3)) 3)

(test (let ((lst (list))) (fill! lst 0) lst) '())
(test (let ((lst (list 1))) (fill! lst 0) lst) '(0))
(test (let ((lst (list 1 2))) (fill! lst 0) lst) '(0 0))
(test (let ((lst (list 1 (list 2 3)))) (fill! lst 0) lst) '(0 0))
(test (let ((lst (cons 1 2))) (fill! lst 0) lst) '(0 . 0))
(test (let ((lst (cons 1 (cons 2 3)))) (fill! lst 0) lst) '(0 0 . 0))
(let ((lst (make-list 3)))
  (fill! lst lst)
  (test lst (lst 0))
  (set! (lst 1) 32)
  (test ((lst 0) 1) 32))

(let ((lst1 (list 1 2))) 
  (test (length lst1) 2)
  (list-set! lst1 0 lst1)
  (test (length lst1) 2) ; its car is a circular list, but it isn't
  (test (list->string lst1) 'error)
  (let ((lst2 (list 1 2)))
    (set-car! lst2 lst2)
    (test (equal? lst1 lst2) #t)
    (test (eq? lst1 lst2) #f)
    (test (eqv? lst1 lst2) #f)
    (test (pair? lst1) #t)
    (test (null? lst1) #f)
    (test (car lst2) lst2)
    (test (car lst1) lst1)
    (test (let ()
	    (fill! lst1 32)
	    lst1)
	  '(32 32))))

(let ((lst1 (list 1))) 
  (test (length lst1) 1)
  (set-cdr! lst1 lst1)
  (test (infinite? (length lst1)) #t)
  (test (null? lst1) #f)
  (test (pair? lst1) #t)
  (let ((lst2 (cons 1 '())))
    (set-cdr! lst2 lst2)
    (test (equal? lst1 lst2) #t)
    (set-car! lst2 0)
    (test (equal? lst1 lst2) #f)
    (test (infinite? (length lst2)) #t)))

(let ((lst1 (list 1))) 
  (set-cdr! lst1 lst1)
  (test (list-tail lst1 0) lst1)
  (test (list-tail lst1 3) lst1)
  (test (list-tail lst1 10) lst1))

(let ((lst1 (let ((lst (list 'a))) 
	      (set-cdr! lst lst)
	      lst)))
  (test (apply lambda lst1 (list 1)) 'error)) ; lambda parameter 'a is used twice in the lambda argument list !

(let ((lst1 (list 1))
      (lst2 (list 1)))
  (set-car! lst1 lst2)
  (set-car! lst2 lst1)
  (test (equal? lst1 lst2) #t)
  (test (length lst1) 1)
  (let ((lst3 (list 1)))
    (test (equal? lst1 lst3) #f)
    (set-cdr! lst3 lst3)
    (test (equal? lst1 lst3) #f)))

(let ((lst1 (list 'a 'b 'c)))
  (set! (cdr (cddr lst1)) lst1)
  (test (infinite? (length lst1)) #t)
  (test (memq 'd lst1) #f)
  (test (memq 'a lst1) lst1)
  (test (memq 'b lst1) (cdr lst1)))

(let ((lst1 (list 1 2 3)))
  (list-set! lst1 1 lst1)
  (test (object->string lst1) "#1=(1 #1# 3)"))

(let ((lst1 (let ((lst (list 1))) 
	      (set-cdr! lst lst)
	      lst)))
  (test (list-ref lst1 9223372036854775807) 'error)
  (test (list-set! lst1 9223372036854775807 2) 'error)
  (test (list-tail lst1 9223372036854775807) 'error)
  (test (make-vector lst1 9223372036854775807) 'error)
  (test (map (lambda (x) x) lst1) 'error)
  (test (map (lambda (x y) x) lst1 lst1) 'error)
  (test (for-each (lambda (x) x) lst1) 'error)
  (test (for-each (lambda (x y) x) lst1 lst1) 'error)
  (test (map (lambda (x y) (+ x y)) lst1 '(1 2 3)) '(2 3 4))
  )

(test (let ((lst '(a b c)))
	(set! (cdr (cddr lst)) lst)
	(map cons lst '(0 1 2 3 4 5)))
      '((a . 0) (b . 1) (c . 2) (a . 3) (b . 4) (c . 5)))

(test (copy (list 1 2 (list 3 4))) '(1 2 (3 4)))
(test (copy (cons 1 2)) '(1 . 2))
(test (copy '(1 2 (3 4) . 5)) '(1 2 (3 4) . 5))
(test (copy '()) '())

(test (object->string (let ((l1 (list 0 1))) (set! (l1 1) l1) (copy l1))) "(0 #1=(0 #1#))")
(test (object->string (let ((lst (list 1 2))) (set! (cdr lst) lst) (copy lst))) "(1 . #1=(1 . #1#))")
(test (object->string (let ((l1 (list 1 2))) (copy (list l1 4 l1)))) "(#1=(1 2) 4 #1#)")
(test (object->string (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) (cdr lst)) (copy lst))) "(1 2 3 . #1=(2 3 . #1#))")

(test (reverse '(1 2 (3 4))) '((3 4) 2 1))
(test (reverse '(1 2 3)) '(3 2 1))
(test (reverse '()) '())
(test (let ((lst (list 1 2 3))) (set! (lst 2) lst) (object->string (reverse lst))) "(#1=(1 2 #1#) 2 1)")
(test (let ((l1 (cons 1 '()))) (set-cdr! l1 l1) (object->string (reverse l1))) "(#1=(1 . #1#) 1 1 1)")


(test (equal? (vector 0) (vector 0)) #t)
(test (equal? (vector 0 #\a "hi" (list 1 2 3)) (vector 0 #\a "hi" (list 1 2 3))) #t)
(test (let ((v (vector 0))) (equal? (vector v) (vector v))) #t)

(let ((v1 (make-vector 1 0)))
  (set! (v1 0) v1)
  (test (vector? v1) #t)
  (let ((v2 (vector 0)))
    (vector-set! v2 0 v2)
    (test (vector-length v1) 1)
    (test (equal? v1 v2) #t)
    (test (equal? (vector-ref v1 0) v1) #t)
    (test (equal? (vector->list v1) (list v1)) #t)
    (vector-fill! v1 0)
    (test (equal? v1 (vector 0)) #t)
    (let ((v3 (copy v2)))
      (test (equal? v2 v3) #t)
      (vector-set! v3 0 0)
      (test (equal? v3 (vector 0)) #t))
    ))

(let ((v1 (make-vector 1 0))
      (v2 (vector 0)))
  (set! (v1 0) v2)
  (set! (v2 0) v1)
  (test (equal? v1 v2) #t)) 

(let* ((l1 (list 1 2))
       (v1 (vector 1 2))
       (l2 (list 1 l1 2))
       (v2 (vector l1 v1 l2)))
  (vector-set! v1 0 v2)
  (list-set! l1 1 l2)
  (test (equal? v1 v2) #f))

(let ((v1 (make-vector 1 0)))
  (set! (v1 0) v1)
  (let ((v2 (vector 0)))
    (vector-set! v2 0 v2)
    (test (equal? v1 v2) #t)))

(let ((v1 (make-vector 1 0)))
  (set! (v1 0) v1)
  (test (object->string v1) "#1=#(#1#)"))

(let ((l1 (cons 0 '()))) 
  (set-cdr! l1 l1) 
  (test (list->vector l1) 'error))

(let ((lst (list "nothing" "can" "go" "wrong")))
  (let ((slst (cddr lst))
	(result '()))
    (set! (cdr (cdddr lst)) slst)
    (test (do ((i 0 (+ i 1))
	       (l lst (cdr l)))
	      ((or (null? l) (= i 12))
	       (reverse result))
	    (set! result (cons (car l) result)))
	  '("nothing" "can" "go" "wrong" "go" "wrong" "go" "wrong" "go" "wrong" "go" "wrong"))))

#|
;;; here is a circular function
(let ()
  (define (cfunc)
    (begin
      (display "cfunc! ")
      #f))

  (let ((clst (procedure-source cfunc)))
    (set! (cdr (cdr (car (cdr (cdr clst)))))
	  (cdr (car (cdr (cdr clst))))))

  (cfunc))
|#

(test (let ((l (list 1 2))) 
	(list-set! l 0 l) 
	(string=? (object->string l) "#1=(#1# 2)")) 
      #t)
(test (let ((lst (list 1)))
	(set! (car lst) lst)
	(set! (cdr lst) lst)
	(string=? (object->string lst) "#1=(#1# . #1#)"))
      #t)
(test (let ((lst (list 1)))
	(set! (car lst) lst)
	(set! (cdr lst) lst)
	(equal? (car lst) (cdr lst)))
      #t)
(test (let ((lst (cons 1 2))) 
	(set-cdr! lst lst)
	(string=? (object->string lst) "#1=(1 . #1#)"))
      #t)
(test (let ((lst (cons 1 2))) 
	(set-car! lst lst)
	(string=? (object->string lst) "#1=(#1# . 2)"))
      #t)
(test (let ((lst (cons (cons 1 2) 3))) 
	(set-car! (car lst) lst)
	(string=? (object->string lst) "#1=((#1# . 2) . 3)"))
      #t)
(test (let ((v (vector 1 2))) 
	(vector-set! v 0 v) 
	(string=? (object->string v) "#1=#(#1# 2)")) 
      #t)
(test (let* ((l1 (list 1 2)) (l2 (list l1))) 
	(list-set! l1 0 l1) 
	(string=? (object->string l2) "(#1=(#1# 2))")) 
      #t)

(test (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) lst) (object->string lst)) "#1=(1 2 3 . #1#)")
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) (cdr lst)) (object->string lst)) "(1 . #1=(2 3 . #1#))")
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) (cdr (cdr lst))) (object->string lst)) "(1 2 . #1=(3 . #1#))")
(test (let ((lst (list 1 2 3))) (set! (car lst) (cdr lst)) (object->string lst)) "(#1=(2 3) . #1#)")
(test (let ((lst (list 1 2 3))) (set! (car (cdr lst)) (cdr lst)) (object->string lst)) "(1 . #1=(#1# 3))")
(test (let ((lst (list 1 2 3))) (set! (car (cdr lst)) lst) (object->string lst)) "#1=(1 #1# 3)")
(test (let ((l1 (list 1))) (let ((l2 (list l1 l1))) (object->string l2))) "(#1=(1) #1#)")

(test (let* ((v1 (vector 1 2)) (v2 (vector v1))) 
	(vector-set! v1 1 v1) 
	(string=? (object->string v2) "#(#1=#(1 #1#))")) 
      #t)
(test (let ((v1 (make-vector 3 1))) 
	(vector-set! v1 0 (cons 3 v1)) 
	(string=? (object->string v1) "#1=#((3 . #1#) 1 1)")) 
      #t)
(test (let ((h1 (make-hash-table 11))
	    (old-print-length *vector-print-length*))
	(set! *vector-print-length* 32)
	(hash-table-set! h1 "hi" h1)
	(let ((result (object->string h1)))
	  (set! *vector-print-length* old-print-length)
	  (let ((val (string=? result "#1=#<hash-table (\"hi\" . #1#)>")))
	    (if (not val)
		(format #t ";hash display:~%  ~A~%" (object->string h1)))
	    val)))
      #t)

(test (let* ((l1 (list 1 2))
	     (v1 (vector 1 2))
	     (l2 (list 1 l1 2))
	     (v2 (vector l1 v1 l2)))
	(vector-set! v1 0 v2)
	(list-set! l1 1 l2)
	(string=? (object->string v2) "#2=#(#1=(1 #3=(1 #1# 2)) #(#2# 2) #3#)"))
      #t)

(test (let ((l1 (list 1 2))
	    (l2 (list 1 2)))
	(set! (car l1) l2)
	(set! (car l2) l1)
	(object->string (list l1 l2)))
      "(#1=(#2=(#1# 2) 2) #2#)")

(test (let* ((l1 (list 1 2)) 
	     (l2 (list 3 4)) 
	     (l3 (list 5 l1 6 l2 7)))
	(set! (cdr (cdr l1)) l1) 
	(set! (cdr (cdr l2)) l2)
	(string=? (object->string l3) "(5 #1=(1 2 . #1#) 6 #2=(3 4 . #2#) 7)"))
      #t)
(test (let* ((lst1 (list 1 2))
	     (lst2 (list (list (list 1 (list (list (list 2 (list (list (list 3 (list (list (list 4 lst1 5))))))))))))))
	(set! (cdr (cdr lst1)) lst1)
	(string=? (object->string lst2) "(((1 (((2 (((3 (((4 #1=(1 2 . #1#) 5))))))))))))"))
      #t)


(test (equal? '(a) (list 'a)) #t)
(test (equal? '(a b . c) '(a b . c)) #t)
(test (equal? '(a b (c . d)) '(a b (c . d))) #t)
(test (equal? (list "hi" "hi" "hi") '("hi" "hi" "hi")) #t)
(let ((l1 (list "hi" "hi" "hi"))
      (l2 (list "hi" "hi" "hi")))
  (fill! l1 "ho")
  (test (equal? l1 l2) #f)
  (fill! l2 (car l1))
  (test (equal? l1 l2) #t))
(let ((lst (list 1 2 3 4))) 
  (fill! lst "hi") 
  (test (equal? lst '("hi" "hi" "hi" "hi")) #t))
(let ((vect (vector 1 2 3 4)))
  (fill! vect "hi")
  (test (equal? vect #("hi" "hi" "hi" "hi")) #t))
(let ((lst (list 1 2 (list 3 4) (list (list 5) 6))))
  (test (equal? lst '(1 2 (3 4) ((5) 6))) #t)
  (fill! lst #f)
  (test (equal? lst '(#f #f #f #f)) #t))
(let ((lst (list 1 2 3 4)))
  (set! (cdr (cdddr lst)) lst)
  (test (equal? lst lst) #t)
  (test (eq? lst lst) #t)
  (test (eqv? lst lst) #t)
  (fill! lst #f)
  (test (object->string lst) "#1=(#f #f #f #f . #1#)")
  (let ((l1 (copy lst)))
    (test (equal? lst l1) #t)
    (test (eq? lst l1) #f)
    (test (eqv? lst l1) #f)))

(test (let ((lst (list "hi" "hi" "hi"))) (fill! lst "hi") (equal? lst '("hi" "hi" "hi"))) #t)
(test (let ((lst (list "hi" "hi"))) (fill! lst "hi") (equal? lst '("hi" "hi"))) #t)
(test (let ((lst (list 1 2 3 4))) (fill! lst "hi") (equal? lst '("hi" "hi" "hi" "hi"))) #t)


(let ((lst '(#\( #\) #\* #\+ #\, #\- #\. #\/ #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9 #\: #\; #\< #\= #\> #\? #\@ #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z #\[ #\\ #\] #\^ #\_ #\` #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z #\{ #\| #\} #\~)))
  (let ((str (apply string lst)))
    (let ((lstr (list->string lst)))
      (let ((strl (string->list str)))
	(test (eq? str str) #t)
	(test (eq? str lstr) #f)
	(test (eqv? str str) #t)
	(test (eqv? str lstr) #f)
	(test (equal? str lstr) #t)
	(test (equal? str str) #t)
	(test (eq? lst strl) #f)	
	(test (eqv? lst strl) #f)	
	(test (equal? lst strl) #t)
	(let ((l2 (copy lst))
	      (s2 (copy str)))
	  (test (eq? l2 lst) #f)
	  (test (eq? s2 str) #f)
	  (test (eqv? l2 lst) #f)
	  (test (eqv? s2 str) #f)
	  (test (equal? l2 lst) #t)
	  (test (equal? s2 str) #t))))))


(let ((vect #(#\( #\) #\* #\+ #\, #\- #\. #\/ #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9 #\: #\; #\< #\= #\> #\? #\@ #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z #\[ #\\ #\] #\^ #\_ #\` #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z #\{ #\| #\} #\~)))
  (let ((lst (vector->list vect)))
    (let ((vect1 (list->vector lst)))
	(test (eq? lst lst) #t)
	(test (eq? lst vect) #f)
	(test (eqv? lst lst) #t)
	(test (eqv? lst vect) #f)
	(test (equal? vect1 vect) #t)
	(test (equal? lst lst) #t)
	(test (eq? vect vect1) #f)	
	(test (eqv? vect vect1) #f)	
	(test (equal? vect vect1) #t)
	(let ((l2 (copy vect))
	      (s2 (copy lst)))
	  (test (eq? l2 vect) #f)
	  (test (eq? s2 lst) #f)
	  (test (eqv? l2 vect) #f)
	  (test (eqv? s2 lst) #f)
	  (test (equal? l2 vect) #t)
	  (test (equal? s2 lst) #t)))))

(let* ((vals (list "hi" #\A 1 'a #(1) abs _ht_ quasiquote macroexpand make-type hook-functions 
		   3.14 3/4 1.0+1.0i #\f '(1 . 2)))
       (vlen (length vals)))
  (do ((i 0 (+ i 1)))
      ((= i 20))
    (let* ((size (max 1 (random 20)))
	   (vect (make-vector size '())))
      (do ((n 0 (+ n 1)))
	  ((= n size))
	(let ((choice (random 4))
	      (len (random 4)))
	  (if (= choice 0)
	      (let ((v (make-vector len)))
		(do ((k 0 (+ k 1)))
		    ((= k len))
		  (vector-set! v k (list-ref vals (random vlen))))
		(vector-set! vect n v))
	      (if (= choice 1)
		  (let ((lst (make-list len #f)))
		    (do ((k 0 (+ k 1)))
			((= k len))
		      (list-set! lst k (list-ref vals (random vlen))))
		    (vector-set! vect n lst))
		  (vector-set! vect n (list-ref vals (random vlen)))))))
      (test (eq? vect vect) #t)
      (test (eqv? vect vect) #t)
      (test (equal? vect vect) #t)
      (let ((lst1 (vector->list vect)))
	(let ((lst2 (copy lst1)))
	  (test (eq? lst1 lst2) #f)
	  (test (eqv? lst1 lst2) #f)
	  (test (equal? lst1 lst2) #t))))))

(let* ((lst1 (list 1 2 3))
       (vec1 (vector 1 2 lst1)))
  (list-set! lst1 2 vec1)
  (let* ((lst2 (list 1 2 3))
	 (vec2 (vector 1 2 lst2)))
    (list-set! lst2 2 vec2)
    (test (equal? lst1 lst2) #t)
    (test (equal? vec1 vec2) #t)
    (vector-set! vec1 1 vec1)
    (test (equal? lst1 lst2) #f)
    (test (equal? vec1 vec2) #f)
    ))
  
(let* ((base (list #f))
       (lst1 (list 1 2 3))
       (vec1 (vector 1 2 base)))
  (list-set! lst1 2 vec1)
  (let* ((lst2 (list 1 2 3))
	 (vec2 (vector 1 2 base)))
    (list-set! lst2 2 vec2)
    (set! (car lst1) lst1)
    (set! (car lst2) lst2)
    (set! (cdr (cddr lst1)) base)
    (set! (cdr (cddr lst2)) base)
    (test (equal? lst1 lst2) #t)
    (test (equal? vec1 vec2) #t)
    (test (object->string lst1) "#1=(#1# 2 #(1 2 #2=(#f)) . #2#)")))

(let ((base (list 0 #f)))
  (let ((lst1 (list 1 base 2))
	(lst2 (list 1 base 2)))
    (set! (cdr (cdr base)) base)
    (test (equal? lst1 lst2) #t)))

(let ((base1 (list 0 #f))
      (base2 (list 0 #f)))
  (let ((lst1 (list 1 base1 2))
	(lst2 (list 1 base2 2)))
    (set! (cdr (cdr base1)) lst2)
    (set! (cdr (cdr base2)) lst1)
    (test (equal? lst1 lst2) #t)
    (test (object->string lst1) "#1=(1 (0 #f 1 (0 #f . #1#) 2) 2)")))

(let ()
  (define-macro (c?r path)

  (define (X-marks-the-spot accessor tree)
    (if (pair? tree)
	(or (X-marks-the-spot (cons 'car accessor) (car tree))
	    (X-marks-the-spot (cons 'cdr accessor) (cdr tree)))
	(if (eq? tree 'X) accessor #f)))

  (let ((body 'lst))
    (for-each
     (lambda (f)
       (set! body (list f body)))
     (reverse (X-marks-the-spot '() path)))

    `(make-procedure-with-setter
      (lambda (lst) 
	,body)
      (lambda (lst val)
	(set! ,body val)))))

  (define (copy-tree lis)
    (if (pair? lis)
	(cons (copy-tree (car lis))
	      (copy-tree (cdr lis)))
	lis))

  (let* ((l1 '(0 (1 (2 (3 (4 (5 (6 (7 (8))))))))))
	 (l2 (list 0 (list 1 (list 2 (list 3 (list 4 (list 5 (list 6 (list 7 (list 8))))))))))
	 (l3 (copy-tree l1))
	 (cxr (c?r (0 (1 (2 (3 (4 (5 (6 (7 (X))))))))))))
    (set! (cxr l1) 3)
    (set! (cxr l2) 4)
    (test (equal? l1 l2) #f)
    (test (equal? l1 l3) #f)
    (set! (cxr l2) 3)
    (test (cxr l2) 3)
    (test (cxr l1) 3)
    (test (cxr l3) 8)
    (test (equal? l1 l2) #t)
    (test (equal? l2 l3) #f))

  (let* ((l1 '(0 (1 (2 (3 (4 (5 (6 (7 (8))))))))))
	 (l2 (list 0 (list 1 (list 2 (list 3 (list 4 (list 5 (list 6 (list 7 (list 8))))))))))
	 (l3 (copy-tree l1))
	 (cxr (c?r (0 (1 (2 (3 (4 (5 (6 (7 (8 . X))))))))))))
    (set! (cxr l1) l1)
    (set! (cxr l2) l2)
    (test (equal? l1 l2) #t)
    (test (equal? l1 l3) #f)
    (test (object->string l2) "#1=(0 (1 (2 (3 (4 (5 (6 (7 (8 . #1#)))))))))"))

  (let* ((l1 '(0 ((((((1))))))))
	 (l2 (copy-tree l1))
	 (cxr (c?r (0 ((((((1 . X))))))))))
    (set! (cxr l1) l2)
    (set! (cxr l2) l1)
    (test (equal? l1 l2) #t))

  (let* ((l1 '(0 1 (2 3) 4 5))
	 (cxr (c?r (0 1 (2 3 . X) 4 5))))
    (set! (cxr l1) (cdr l1))
    (test (object->string l1) "(0 . #1=(1 (2 3 . #1#) 4 5))"))

  (let* ((l1 '(0 1 (2 3) 4 5))
	 (l2 '(6 (7 8 9) 10))
	 (cxr1 (c?r (0 1 (2 3 . X) 4 5)))
	 (cxr2 (c?r (6 . X)))
	 (cxr3 (c?r (6 (7 8 9) 10 . X)))
	 (cxr4 (c?r (0 . X))))
    (set! (cxr1 l1) (cxr2 l2))
    (set! (cxr3 l2) (cxr4 l1))
    (test (object->string l1) "(0 . #1=(1 (2 3 (7 8 9) 10 . #1#) 4 5))")
    (test (cadr l1) 1)
    (test (cadddr l1) 4)
    )

  (let ((l1 '((a . 2) (b . 3) (c . 4)))
	(cxr (c?r ((a . 2) (b . 3) (c . 4) . X))))
    (set! (cxr l1) (cdr l1))
    (test (assq 'a l1) '(a . 2))
    (test (assv 'b l1) '(b . 3))
    (test (assoc 'c l1) '(c . 4))
    (test (object->string l1) "((a . 2) . #1=((b . 3) (c . 4) . #1#))")
    (test (assq 'asdf l1) #f)
    (test (assv 'asdf l1) #f)
    (test (assoc 'asdf l1) #f)
    )

  (let ((l1 '(a b c d e))
	(cxr (c?r (a b c d e . X))))
    (set! (cxr l1) (cddr l1))
    (test (memq 'b l1) (cdr l1))
    (test (memv 'c l1) (cddr l1))
    (test (member 'd l1) (cdddr l1))
    (test (object->string l1) "(a b . #1=(c d e . #1#))")
    (test (memq 'asdf l1) #f)
    (test (memv 'asdf l1) #f)
    (test (member 'asdf l1) #f)
    (test (pair? (member 'd l1)) #t) ; #1=(d e c . #1#)
    )

  (let ((ctr 0)
	(x 0))
    (let ((lst `(call-with-exit (lambda (return) (set! x (+ x 1)) (if (> x 10) (return x) 0)))))
      (let ((acc1 (c?r (call-with-exit (lambda (return) . X))))
	    (acc2 (c?r (call-with-exit (lambda (return) (set! x (+ x 1)) (if (> x 10) (return x) 0) . X)))))
	(set! (acc2 lst) (acc1 lst))
	(test (eval lst) 11))))
  )
  
(let ((v #2d((1 2) (3 4))))
  (set! (v 1 0) v)
  (test (object->string v) "#1=#2D((1 2) (#1# 4))")
  (test (length v) 4)
  (test ((((v 1 0) 1 0) 1 0) 0 0) 1))

(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (lst 100) 2)
  (test ((cdddr (cdddr (cdddr lst))) 100) 2)
  (set! (lst 100) 32)
  (test (object->string lst) "#1=(1 32 3 . #1#)"))

(let* ((l1 (list 1 2))
       (l2 (list l1 l1)))
  (set! (l1 0) 32)
  (test (equal? l2 '((32 2) (32 2))) #t))

(let ((q (list 1 2 3 4)))
  (set! (cdr (cdddr q)) q) 
  (test (car q) 1)
  (set! (car q) 5)
  (set! q (cdr q))
  (test (car q) 2)
  (test (object->string q) "#1=(2 3 4 5 . #1#)"))

(let ()
  (define (make-node prev data next) (vector prev data next))
  (define prev (make-procedure-with-setter (lambda (node) (node 0)) (lambda (node val) (set! (node 0) val))))
  (define next (make-procedure-with-setter (lambda (node) (node 2)) (lambda (node val) (set! (node 2) val))))
  (define data (make-procedure-with-setter (lambda (node) (node 1)) (lambda (node val) (set! (node 1) val))))
  (let* ((head (make-node () 0 ()))
	 (cur head))
    (do ((i 1 (+ i 1)))
	((= i 8))
      (let ((next-node (make-node cur i ())))
	(set! (next cur) next-node)
	(set! cur (next cur))))
    (set! (next cur) head)
    (set! (prev head) cur)
    (test (object->string head) "#1=#(#7=#(#6=#(#5=#(#4=#(#3=#(#2=#(#8=#(#1# 1 #2#) 2 #3#) 3 #4#) 4 #5#) 5 #6#) 6 #7#) 7 #1#) 0 #8#)")
#|
    ;; in CL:
    (let* ((head (vector nil 0 nil))
	   (cur head))
      (do ((i 1 (+ i 1)))
	  ((= i 8))
	(let ((node (vector nil i nil)))
	  (setf (aref node 0) cur)
	  (setf (aref cur 2) node)
	  (setf cur node)))
      (setf (aref head 0) cur)
      (setf (aref cur 2) head)
      (format t "~A~%" head)) -> "#1=#(#2=#(#3=#(#4=#(#5=#(#6=#(#7=#(#8=#(#1# 1 #7#) 2 #6#) 3 #5#) 4 #4#) 5 #3#) 6 #2#) 7 #1#) 0 #8#)"
|#
    (let ((ahead (do ((cur head (next cur))
		      (dat '() (cons (data cur) dat)))
		     ((member (data cur) dat)
		      (reverse dat)))))
      (let ((behind (do ((cur (prev head) (prev cur))
			 (dat '() (cons (data cur) dat)))
			((member (data cur) dat)
			 dat))))
	(test (equal? ahead behind) #t)))))

(let ()
  (define (make-node prev data next) (list prev data next))
  (define prev (make-procedure-with-setter (lambda (node) (node 0)) (lambda (node val) (set! (node 0) val))))
  (define next (make-procedure-with-setter (lambda (node) (node 2)) (lambda (node val) (set! (node 2) val))))
  (define data (make-procedure-with-setter (lambda (node) (node 1)) (lambda (node val) (set! (node 1) val))))
  (let* ((head (make-node () 0 ()))
	 (cur head))
    (do ((i 1 (+ i 1)))
	((= i 8))
      (let ((next-node (make-node cur i ())))
	(set! (next cur) next-node)
	(set! cur (next cur))))
    (set! (next cur) head)
    (set! (prev head) cur)
    (test (object->string head) "#1=(#7=(#6=(#5=(#4=(#3=(#2=(#8=(#1# 1 #2#) 2 #3#) 3 #4#) 4 #5#) 5 #6#) 6 #7#) 7 #1#) 0 #8#)")
    (let ((ahead (do ((cur head (next cur))
		      (dat '() (cons (data cur) dat)))
		     ((member (data cur) dat)
		      (reverse dat)))))
      (let ((behind (do ((cur (prev head) (prev cur))
			 (dat '() (cons (data cur) dat)))
			((member (data cur) dat)
			 dat))))
	(test (equal? ahead behind) #t))))
  (let* ((head (make-node () 0 ()))
	 (cur head))
    (do ((i 1 (+ i 1)))
	((= i 32))
      (let ((next-node (make-node cur i ())))
	(set! (next cur) next-node)
	(set! cur (next cur))))
    (set! (next cur) head)
    (set! (prev head) cur)
    (test (object->string head) "#1=(#31=(#30=(#29=(#28=(#27=(#26=(#25=(#24=(#23=(#22=(#21=(#20=(#19=(#18=(#17=(#16=(#15=(#14=(#13=(#12=(#11=(#10=(#9=(#8=(#7=(#6=(#5=(#4=(#3=(#2=(#32=(#1# 1 #2#) 2 #3#) 3 #4#) 4 #5#) 5 #6#) 6 #7#) 7 #8#) 8 #9#) 9 #10#) 10 #11#) 11 #12#) 12 #13#) 13 #14#) 14 #15#) 15 #16#) 16 #17#) 17 #18#) 18 #19#) 19 #20#) 20 #21#) 21 #22#) 22 #23#) 23 #24#) 24 #25#) 25 #26#) 26 #27#) 27 #28#) 28 #29#) 29 #30#) 30 #31#) 31 #1#) 0 #32#)")))

(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (append lst lst ())) 'error)
(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (object->string (append (list lst) (list lst) ()))) "(#1=(1 2 3 . #1#) #1#)")

(let ((ht (make-hash-table 3)))
  (set! (ht "hi") ht)
  (test (object->string ht) "#1=#<hash-table (\"hi\" . #1#)>")
  (test (equal? (ht "hi") ht) #t))

(let ((l1 '(0)) (l2 '(0))) 
  (set! (car l1) l1) (set! (cdr l1) l1) (set! (car l2) l2) (set! (cdr l2) l2)
  (test (object->string l1) "#1=(#1# . #1#)")
  (test (equal? l1 l2) #t)
  (set! (cdr l1) l2)
  (test (object->string l1) "#1=(#1# . #2=(#2# . #2#))")
  (test (equal? l1 l2) #t)
  (set! (cdr l1) '())
  (test (equal? l1 l2) #f))

(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (map (lambda (a b)
	       (+ a b))
	     (list 4 5 6)
	     lst)
	'(5 7 9)))
(test (let ((lst (list 1 2 3)) 
	    (result '()))
	(set! (cdr (cddr lst)) lst)
	(for-each (lambda (a b)
		    (set! result (cons (+ a b) result)))
		  (list 4 5 6)
		  lst)
	result)
      '(9 7 5))
(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (map (lambda (a b)
	       (+ a b))
	     (vector 4 5 6)
	     lst)
	'(5 7 9)))
(test (let ((lst (list 1 2 3)))
	(set! (cdr (cddr lst)) lst)
	(map (lambda (a b)
	       (+ a b))
	     (vector 4 5 6 7 8 9 10)
	     lst))
      '(5 7 9 8 10 12 11))
(test (map (lambda (a) a) '(0 1 2 . 3)) '(0 1 2))
(test (let ((ctr 0)) (for-each (lambda (a) (set! ctr (+ ctr a))) '(1 2 . 3)) ctr) 3)
(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (map (lambda (a b)
	       (+ a b))
	     '()
	     lst)
	'()))
(test (let ((lst (list 1 2 3))
	    (ctr 0))
	(set! (cdr (cddr lst)) lst)
	(for-each (lambda (a b)
		    (set! ctr (+ ctr (+ a b))))
		  lst '())
	ctr)
      0)

(test (let ((lst (list 1))) (set! (cdr lst) (car lst)) (object->string lst)) "(1 . 1)")
(test (let ((lst (list 1))) (set! (car lst) (cdr lst)) (object->string lst)) "(())")

(test (let ((lst (list 1 2 3))) (fill! lst lst) (object->string lst)) "#1=(#1# #1# #1#)")
(test (let ((lst (vector 1 2 3))) (fill! lst lst) (object->string lst)) "#1=#(#1# #1# #1#)")
(test (let ((lst #2d((1) (1)))) (fill! lst lst) (object->string lst)) "#1=#2D((#1#) (#1#))")

(let ((ctr 0) (lst `(let ((x 3)) (set! ctr (+ ctr 1)) (set! (cdr (cddr lst)) `((+ x ctr))) (+ x 1))))
  (test (eval lst) 4)
  (test (eval lst) 5)
  (test (eval lst) 6))
  

(let ()
  (define fact         ; Reini Urban, http://autocad.xarch.at/lisp/self-mod.lsp.txt
    (let ((old '())
	  (result '()))
      
      (define (last lst)
	(list-tail lst (- (length lst) 1)))
      
      (define (butlast lis)
	(let ((len (length lis)))
	  (if (<= len 1) '()
	      (let ((result '()))
		(do ((i 0 (+ i 1))
		     (lst lis (cdr lst)))
		    ((= i (- len 1)) (reverse result))
		  (set! result (cons (car lst) result)))))))
      
      (lambda (n)
	(cond ((zero? n) 1)
	      (#t 
	       (set! old (procedure-source fact))
	       (set! fact (apply lambda '(n)
				       `((cond 
					 ,@(butlast (cdr (car (cdr (cdr old)))))
					 ((= n ,n) ,(let ()
						      (set! result (* n (fact (- n 1))))
						      result))
					 ,@(last (cdr (car (cdr (cdr old)))))))))
	       result)))))

  (test (fact 3) 6)
  (test (fact 5) 120)
  (test (fact 2) 2))

(test (let ((f #f))
	(set! f (lambda () 
		  (let* ((code (procedure-source f))
			 (pos (- (length code) 1)))
		    (set! (code pos) (+ (code pos) 1)))
		  1))
	(f) (f) (f))
      4)

(let* ((x (list 1 2 3)) ; from Lambda the Ultimate I think -- I lost the reference
       (y (list 4 5))	
       (z (cons (car x) (cdr y)))
       (w (append y z))
       (v (cons (cdr x) (cdr y))))
  (set-car! x 6)
  (set-car! y 7)
  (set-cdr! (cdr x) (list 8))
  (test (object->string (list x y z w v)) "((6 . #3=(2 8)) (7 . #1=(5)) #2=(1 . #1#) (4 5 . #2#) (#3# . #1#))"))
;; guile gets this result, but prints it as: ((6 2 8) (7 5) (1 5) (4 5 1 5) ((2 8) 5))


(let ()
  (define (for-each-permutation func vals)          ; for-each-combination -- use for-each-subset below
    ;; apply func to every permutation of vals: 
    ;;   (for-each-permutation (lambda args (format #t "~{~A~^ ~}~%" args)) '(1 2 3))
    (define (pinner cur nvals len)
      (if (= len 1)
	  (apply func (cons (car nvals) cur))
	  (do ((i 0 (+ i 1)))                       ; I suppose a named let would be more Schemish
	      ((= i len))
	    (let ((start nvals))
	      (set! nvals (cdr nvals))
	      (let ((cur1 (cons (car nvals) cur)))  ; add (car nvals) to our arg list
		(set! (cdr start) (cdr nvals))      ; splice out that element and 
		(pinner cur1 (cdr start) (- len 1)) ;   pass a smaller circle on down
		(set! (cdr start) nvals))))))       ; restore original circle
    (let ((len (length vals)))
      (set-cdr! (list-tail vals (- len 1)) vals)    ; make vals into a circle
      (pinner '() vals len)
      (set-cdr! (list-tail vals (- len 1)) '())))   ; restore its original shape

  ;; t224 applies this to +/*

  (let ((perms '((3 1 2) (1 3 2) (1 2 3) (2 1 3) (2 3 1) (3 2 1)))
	(pos '()))
    (for-each-permutation
     (lambda args
       (call-with-exit
	(lambda (ok)
	  (let ((ctr 0))
	    (for-each
	     (lambda (a)
	       (if (equal? a args)
		   (begin
		     (set! pos (cons ctr pos))
		     (ok)))
	       (set! ctr (+ ctr 1)))
	     perms)))))
     '(1 2 3))
    (test pos '(5 4 3 2 1 0)))
  )





;;; --------------------------------------------------------------------------------
;;; HASH-TABLES
;;; --------------------------------------------------------------------------------

(let ((ht (make-hash-table)))
  (test (hash-table? ht) #t)
  (test (equal? ht ht) #t)
  (test (let () (hash-table-set! ht 'key 3.14) (hash-table-ref ht 'key)) 3.14)
  (test (let () (hash-table-set! ht "ky" 3.14) (hash-table-ref ht "ky")) 3.14)
  (test (let () (hash-table-set! ht 123 "hiho") (hash-table-ref ht 123)) "hiho")
  (test (let () (hash-table-set! ht 3.14 "hi") (hash-table-ref ht 3.14)) "hi")
  (test (let () (hash-table-set! ht pi "hiho") (hash-table-ref ht pi)) "hiho")
  (test (hash-table-ref ht "123") #f)
  (let ((ht1 (copy ht)))
    (test (hash-table? ht1) #t)
    (test (= (length ht) (length ht1)) #t)
    (test (equal? ht ht1) #t)
    (test (eq? ht ht) #t)
    (test (eqv? ht ht) #t)
    (set! (ht 'key) 32)
    (set! (ht1 'key) 123)
    (test (and (= (ht 'key) 32) (= (ht1 'key) 123)) #t)
    (set! (ht "key") 321)
    (test (ht "key") 321)
    (test (ht 'key) 32)
    (set! (ht 123) 43)
    (set! (ht "123") 45)
    (test (ht 123) 43)
    (test (ht "123") 45))
  (test (let () (set! (hash-table-ref ht 'key) 32) (hash-table-ref ht 'key)) 32)

  (for-each
   (lambda (arg)
     (test (let () (hash-table-set! ht 'key arg) (hash-table-ref ht 'key)) arg))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2))))

(for-each
 (lambda (arg)
   (test (hash-table-set! arg 'key 32) 'error))
 (list "hi" '() -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((hi (make-hash-table 7)))
  (test (object->string hi) "#<hash-table>")
  (set! (hi 1) "1")
  (test (object->string hi) "#<hash-table (1 . \"1\")>")
  (set! (hi -1) "-1")
  (test (object->string hi) "#<hash-table (-1 . \"-1\") (1 . \"1\")>")
  (set! (hi 9) "9")
  (test (object->string hi) "#<hash-table (9 . \"9\") (-1 . \"-1\") (1 . \"1\")>")
  (set! (hi -9) "-9")
  (test (object->string hi) "#<hash-table (-9 . \"-9\") (9 . \"9\") (-1 . \"-1\") (1 . \"1\")>")
  (test (hi 1) "1")
  (test (hi -9) "-9")
  (set! (hi 2) "2")
  (test (object->string hi) "#<hash-table (-9 . \"-9\") (9 . \"9\") (-1 . \"-1\") (1 . \"1\") (2 . \"2\")>")

  (let ((old-plen *vector-print-length*))
    (set! *vector-print-length* 3)
    (test (object->string hi) "#<hash-table (-9 . \"-9\") (9 . \"9\") (-1 . \"-1\") ...>")
    (set! *vector-print-length* 0)
    (test (object->string hi) "#<hash-table ...>")
    (test (object->string (hash-table)) "#<hash-table>")
    (set! *vector-print-length* old-plen))
  )

(let ((ht (make-hash-table 277)))
  (test (hash-table? ht) #t)
  (test (>= (hash-table-size ht) 277) #t)
  (test (let () (hash-table-set! ht 'key 3.14) (hash-table-ref ht 'key)) 3.14)
  (test (let () (hash-table-set! ht "ky" 3.14) (hash-table-ref ht "ky")) 3.14)
  (for-each
   (lambda (arg)
     (test (let () (hash-table-set! ht 'key arg) (hash-table-ref ht 'key)) arg))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2))))

(for-each
 (lambda (arg)
   (test (hash-table? arg) #f))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(test (hash-table? (make-vector 3 '())) #f)
(test (let ((ht (make-hash-table))) (set! (ht 'a) 123) (map values ht)) '((a . 123)))

(let ((ht (make-hash-table)))	
  (test (hash-table-ref ht 'not-a-key) #f)
  (test (hash-table-ref ht "not-a-key") #f)
  (hash-table-set! ht 'key 3/4)
  (hash-table-set! ht "key" "hi")
  (test (hash-table-ref ht "key") "hi")
  (test (hash-table-ref ht 'key) 3/4)
  
  (hash-table-set! ht 'asd 'hiho)
  (test (hash-table-ref ht 'asd) 'hiho)
  (hash-table-set! ht 'asd 1234)
  (test (hash-table-ref ht 'asd) 1234))

(for-each
 (lambda (arg)
   (test (hash-table-ref arg 'key) 'error))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((ht1 (make-hash-table 653))
      (ht2 (make-hash-table 277)))
  (test (equal? ht1 ht2) #f)
  (hash-table-set! ht1 'key 'hiho)
  (hash-table-set! ht2 (hash-table-ref ht1 'key) 3.14)
  (test (>= (hash-table-size ht1) 653) #t)
  (test (hash-table-ref ht2 'hiho) 3.14)
  (test (hash-table-ref ht2 (hash-table-ref ht1 'key)) 3.14))

(let ((ht1 (make-hash-table)))
   (set! (ht1 1) 'hi)
   (let ((ht2 (make-hash-table)))
      (set! (ht2 1) ht1)
      (test ((ht2 1) 1) 'hi)))

(test (hash-table?) 'error)
(test (hash-table? 1 2) 'error)

(test (make-hash-table most-positive-fixnum) 'error)
(test (make-hash-table most-negative-fixnum) 'error)
(test (make-hash-table 10 1) 'error)

(let ((ht (make-hash-table)))
  (test (hash-table? ht ht) 'error)
  (test (hash-table-ref ht #\a #\b) 'error)
  (test (hash-table-ref ht) 'error)
  (test (hash-table-ref) 'error)
  (test (hash-table-set!) 'error)
  (test (hash-table-set! ht) 'error)
  (test (hash-table-set! ht #\a) 'error)
  (test (hash-table-set! ht #\a #\b #\c) 'error)
  (test (fill! ht 123) 'error)
  (set! (ht 'key) 32)
  (test (ht 'key) 32)
  (set! (ht :key) 123)
  (test (ht 'key) 32)
  (test (ht :key) 123)
  (fill! ht '())
  (test (ht 'key) #f))

(let ((ht (make-hash-table)))
  (test (hash-table-set! ht #\a 'key) 'key)
  (for-each
   (lambda (arg)
     (test (hash-table-set! ht arg 3.14) 3.14))
   (list #\a '#(1 2 3) 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
  (for-each
   (lambda (arg)
     (test (hash-table-ref ht arg) 3.14))
   (list #\a '#(1 2 3) 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
  (test (hash-table-size ht 123) 'error))

(for-each
 (lambda (arg)
   (test (hash-table-size arg) 'error))
 (list "hi" -1 0 #\a 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
(test (hash-table-size) 'error)

(for-each
 (lambda (arg)
   (test (make-hash-table arg) 'error)
   (test (make-hash-table-iterator arg) 'error))
 (list "hi" -1 0 #\a 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ()
 (define ht (make-hash-table))
 (set! (ht 123) "123")
 (set! (ht 456) "456")
 (define hti (make-hash-table-iterator ht))
 (let ((vals (list (hti) (hti))))
   (if (not (equal? (sort! vals (lambda (a b) (< (car a) (car b)))) '((123 . "123") (456 . "456"))))
       (format #t ";hash-table-iterator: ~A~%" vals))
   (let ((val (hti)))
     (if (not (null? val))
	 (format #t ";hash-table-iterator at end: ~A~%" val)))
   (let ((val (hti)))
     (if (not (null? val))
	 (format #t ";hash-table-iterator at end (2): ~A~%" val)))))

(test (make-hash-table-iterator) 'error)
(test (make-hash-table-iterator (make-hash-table) 1) 'error)

(let ((ht1 (make-hash-table))
      (ht2 (make-hash-table)))
  (test (equal? ht1 ht2) #t)
  (test (equal? ht1 (make-vector (hash-table-size ht1) '())) #f)
  (hash-table-set! ht1 'key 'hiho)
  (test (equal? ht1 ht2) #f)
  (hash-table-set! ht2 'key 'hiho)
  (test (equal? ht1 ht2) #t)

  (hash-table-set! ht1 'a '())
  (test (ht1 'a) '())
  )

(let ((ht (make-hash-table 1)))
  (test (>= (length ht) 1) #t)
  (set! (ht 1) 32)
  (test (>= (length ht) 1) #t))

(let ((ht (hash-table '("hi" . 32) '("ho" . 1))))
  (test (ht "hi") 32)
  (test (ht "ho") 1))

(let ((ht (hash-table)))
  (test (hash-table? ht) #t)
  (test (>= (length ht) 461) #t)
  (test (ht 1) #f))

(for-each
 (lambda (arg)
   (test (hash-table arg) 'error))
 (list "hi" -1 0 #\a 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t abs #<eof> #<unspecified> (lambda () 1)))

(test (set! (hash-table) 1) 'error)
(test (set! (make-hash-table) 1) 'error)

;; no null hash-tables?

(let ((ht (make-hash-table)))
  ;; these are broken on purpose -- ht has no entries, so its lenth is 0, so we don't check arg nums etc
  (test (map (lambda (x) x) ht) '())
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 0)
  (test (map (lambda (x y) (cons x y)) (list 1 2 3) ht) '())
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) #(1 2 3) ht) ctr) 0)
  (test (map (lambda (x y) (cons x y)) ht "123") '())
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht '()) ctr) 0)

  (let ((rt (reverse ht)))
    (test (map (lambda (x) x) rt) '())
    (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) rt) ctr) 0))

  (set! (ht 1) 32)
  ;; these need to be independent of entry order
  
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(32))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 1)
  (test (map (lambda (x y) (cons x y)) '() ht) '())
  (test (let ((ctr 0)) (for-each (lambda (x y) (set! ctr (+ ctr 1))) ht "") ctr) 0)
  (test (sort! (map (lambda (x y) (max (cdr x) y)) ht (list 1 2 3)) <) '(32))
  (test (let ((ctr 0)) (for-each (lambda (x y) (set! ctr (max (cdr x) y))) ht #(1 2 3)) ctr) 32)

  (let ((rt (reverse ht)))
    (test (equal? (rt 32) 1) #t)
    (test (equal? (rt 1) #f) #t)
    (test (ht (rt 32)) 32)
    (test (sort! (map (lambda (x) (cdr x)) rt) <) '(1))
    (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) rt) ctr) 1)
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht)
    (set! (rt 32) 123)
    (test (rt 32) 123)
    (test (ht 32) #f)
    (test (ht 1) 32))

  (set! (ht 2) 1)
  (test (ht (ht 2)) 32)
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(1 32))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 2)
  (set! (ht 3) 123)
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(1 32 123))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 3)
  (test (let ((ctr 0)) (for-each (lambda (x y) (set! ctr (+ ctr 1))) ht '(1)) ctr) 1)
  (test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr 1))) "12" ht '(1)) ctr) 1)
  (test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr 1))) "12" ht '(1 2)) ctr) 2)
  (test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr 1))) "12345" ht '(1 2 3 4 5 6)) ctr) 3)
  
  (test (sort! (map (lambda (x y) (max x (cdr y))) (list -1 -2 -3 -4) ht) <) '(1 32 123))
  (test (let ((sum 0)) (for-each (lambda (x y) (set! sum (+ sum x (cdr y)))) #(10 20 30) ht) sum) 216)
  
  (let ((rt (reverse ht)))
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht))
  
  (set! (ht (list 1 2 3)) "hi")
  (test (ht '(1 2 3)) "hi")
  (test (ht 2) 1)
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 4)
  (set! (ht "hi") 2)
  (test (ht "hi") 2)
  (test (ht (ht (ht "hi"))) 32)

  (let ((rt (reverse ht)))
    (test (rt "hi") '(1 2 3))
    (test (rt 2) "hi")
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht)
    (set! (rt 2) "ho")
    (test (rt 2) "ho")
    (test (ht '(1 2 3)) "hi")
    (set! (rt 123) 321)
    (test (rt 123) 321)
    (test (ht 3) 123))

  (fill! ht '())
  (set! (ht "hi") 1)
  (set! (ht "hoi") 2)
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(1 2))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 2)
  
  (let ((rt (reverse ht)))
    (test (rt 2) "hoi")
    (set! (rt 2) "ha")
    (test (ht "hoi") 2))

  (set! (ht #\a) #\b)
  (test (ht #\a) #\b)
  (test (ht "hi") 1)

  (fill! ht '())
  (set! (ht #(1)) #(2))
  (test (ht #(1)) #(2))
  (set! (ht '(1)) '(3))
  (set! (ht "1") "4")
  (set! (ht ht) "5")
  (test (ht ht) "5")
  (test (ht '(1)) '(3))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 4)  
    
  (let ((rt (reverse ht)))
    (test (rt "5") ht)
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht))
  )  

(let ((ht1 (make-hash-table 32))
      (ht2 (make-hash-table 1024)))
  (do ((i 0 (+ i 1)))
      ((= i 256))
    (let ((str (number->string i)))
      (set! (ht1 str) i)
      (set! (ht2 i) str)))
  (let ((cases 0))
    (for-each
     (lambda (a b)
       (if (not (equal? (string->number (car a)) (cdr a)))
	   (format #t ";hash-table for-each (str . i): ~A?~%" a))
       (if (not (equal? (number->string (car b)) (cdr b)))
	   (format #t ";hash-table for-each (i . str): ~A?~%" b))
       (set! cases (+ cases 1)))
     ht1 ht2)
    (if (not (= cases 256))
	(format #t ";hash-table for-each cases: ~A~%" cases)))
  (let ((iter1 (make-hash-table-iterator ht1))
	(iter2 (make-hash-table-iterator ht2)))
    (let ((cases 0))
      (do ((a (iter1) (iter1))
	   (b (iter2) (iter2)))
	  ((or (null? a)
	       (null? b)))
	(if (not (equal? (string->number (car a)) (cdr a)))
	    (format #t ";hash-table iter1 (str . i): ~A?~%" a))
	(if (not (equal? (number->string (car b)) (cdr b)))
	    (format #t ";hash-table iter2 (i . str): ~A?~%" b))
	(set! cases (+ cases 1)))
      (if (not (= cases 256))
	  (format #t ";hash-table iter1/2 cases: ~A~%" cases)))))

(let ((ht (make-hash-table 31)))
  (let ((ht1 (make-hash-table 31)))
    (set! (ht1 'a1) 'b1)
    (set! (ht 'a0) ht1)
    (test ((ht 'a0) 'a1) 'b1)
    (test (hash-table-ref ht 'a0 'a1) 'b1)
    (test (ht 'a0 'a1) 'b1)))

;; there's no real need for multidim hashes:

(let ((ht (make-hash-table)))
   (set! (ht (cons 'a 1)) 'b)
   (set! (ht (cons 'a 2)) 'c)
   (set! (ht (cons 'b 1)) 'd)
   (test (ht '(a . 1)) 'b)
   (test (ht '(b . 1)) 'd)
   (set! (ht '(a . 2)) 32)
   (test (ht '(a . 2)) 32))

(let ((ht (make-hash-table)))
  (set! (ht 1.0) 'a)
  (set! (ht 2.0) 'b)
  (set! (ht 3.0) 'c)
  (test (ht 2.0) 'b)
  (set! (ht 2.0) 'd)
  (test (ht 2.0) 'd)
  (test (ht 0.0) #f)
  (test (ht 1.0) 'a))

(let ((ht (make-hash-table)))
  (test (ht) 'error)
  (test (ht 0 1) 'error))

(let ()
  (define-macro (memoize f)
    `(define ,f (let ((ht (make-hash-table))
		      (old-f ,f))
		  (lambda args
		    (let ((val (ht args)))
		      (if val
			  (val 0)
			  (let ((new-val (apply old-f args)))
			    (set! (ht args) (list new-val))
			    new-val)))))))

  (define (our-abs num) (abs num))
  (memoize our-abs)
  (num-test (our-abs -1) 1)
  (with-environment (procedure-environment our-abs)
    (test (ht '(-1)) '(1))))		    


(let ((ht (make-hash-table)))
  (test (eq? (car (catch #t (lambda () (set! (ht) 2)) (lambda args args))) 'wrong-number-of-args) #t)
  (test (eq? (car (catch #t (lambda () (set! (ht 0 0) 2)) (lambda args args))) 'wrong-number-of-args) #t)
  (test (eq? (car (catch #t (lambda () (set! ((ht 0) 0) 2)) (lambda args args))) 'syntax-error) #t))




;;; --------------------------------------------------------------------------------
;;; HOOKS
;;; --------------------------------------------------------------------------------

;;; hook?
;;; hook-arity
;;; hook-functions
;;; hook-documentation
;;; make-hook
;;; hook-apply
;;; hook

(for-each
 (lambda (arg)
   (for-each
    (lambda (func)
      (let ((val (catch #t (lambda () (func arg)) (lambda args 'error))))
	(if (not (eq? val 'error))
	    (format #t ";(~A ~A) got ~A, but expected 'error~%" func arg val))))
    (list hook-arity hook-documentation hook-functions hook hook-apply))
   (test (hook? arg) #f))
 (list "hi" #f (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f #<eof> #<undefined> #<unspecified>))

(let ((h (make-hook '(1 0 #f) "a hook")))
  (test (hook? h) #t)
  (test (hook-functions h) '())
  (test (hook-arity h) '(1 0 #f))
  (test (hook-documentation h) "a hook")
  (test (hook-apply h '(0)) (h 0))
  (let ((f1 (lambda (x) x)))
    (set! (hook-functions h) (list f1))
    (test (member f1 (hook-functions h)) (list f1))
    (test (hook-functions h) (list f1))
    (test (hook-apply h '(1)) (h 1))
    (test (apply (car (hook-functions h)) '(1)) 1)
    (set! (hook-functions h) '())
    (test (hook-functions h) '())
    (test (set! (hook-functions h) (list (lambda (x y) (+ x y)))) 'error)
    (let ((f2 (lambda* args (car args))))
      (set! (hook-functions h) (list f1 f2))
      (test (hook-functions h) (list f1 f2))
      (test (hook-apply h '(23)) (h 23))))
  (for-each
   (lambda (arg)
     (test (set! (hook-functions h) arg) 'error))
   (list "hi" #f (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f #<eof> #<undefined> #<unspecified>)))

(let ((h (make-hook '(1 1 #f))))
  (test (hook? h) #t)
  (test (hook-functions h) '())
  (test (hook-arity h) '(1 1 #f))
  (test (hook-documentation h) "")
  (test (hook-apply h '(0 1)) (h 0 1))
  (let ((f1 (lambda* (x (y 12)) (+ x y))))
    (set! (hook-functions h) (list f1))
    (test (member f1 (hook-functions h)) (list f1))
    (test (hook-functions h) (list f1))
    (test (hook-apply h '(1 2)) (h 1 2))
    (test (apply (car (hook-functions h)) '(1 2)) 3)
    (set! (hook-functions h) '())
    (test (hook-functions h) '())
    (set! (hook-functions h) (list (lambda* ((x 1) (y 2)) (+ x y))))
    (test (hook-apply h '(23)) (h 23))))

(let ((h (make-hook)))
  (test (hook? h) #t)
  (test (hook-functions h) '())
  (test (hook-arity h) '(0 0 #f))
  (test (hook-documentation h) "")
  (test (hook-apply h '()) (h))
  (let ((f1 (lambda () 0)))
    (set! (hook-functions h) (list f1))
    (test (member f1 (hook-functions h)) (list f1))
    (test (hook-functions h) (list f1))
    (test (hook-apply h '()) (h))
    (test (apply (car (hook-functions h)) '()) 0)
    (set! (hook-functions h) '())
    (test (hook-functions h) '())))

(let ((h (hook (lambda x (error 'out-of-hook 32)))))
  (let ((val (catch 'out-of-hook
		    (lambda ()
		      (h 123)
		      (error 'oops "too far"))
		    (lambda args args))))
    (if (not (equal? val '(out-of-hook (32))))
	(format #t ";hook error 1: ~A~%" val))

    (let ((val (call-with-exit
		(lambda (return)
		  (set! (hook-functions h) (list (lambda x (return x))))
		  (hook-apply h (list "hi"))
		  123))))
      (if (not (equal? val '("hi")))
	  (format #t ";hook error 2: ~A~%" val)))))

(test (make-hook '(1 2 #f) "hi" 3) 'error)
(let ((h (make-hook)))
  (test (hook-functions) 'error)
  (test (hook-functions h h) 'error)
  (test (hook-arity) 'error)
  (test (hook-arity h h) 'error)
  (test (hook-documentation) 'error)
  (test (hook-documentation h h) 'error)
  (test (hook-apply) 'error)
  (test (hook-apply h 1) 'error)
  (test (hook?) 'error)
  (test (hook? h h) 'error)
  (test (hook h) 'error)
  (test (hook? (hook)) #t)
  (test (hook-arity (hook)) '(0 0 #f))
  (test (hook-functions (hook)) '())
  (test (hook-documentation (hook)) ""))
(test (make-hook '()) 'error)
(test (make-hook '(1)) 'error)
(test (make-hook '(1 2)) 'error)
(test (make-hook '(1 2 #f 3)) 'error)
(test (make-hook '(1.0 2 #f)) 'error)
(test (make-hook '(1 2/3 #f)) 'error)
(test (make-hook '(1 0 1)) 'error)
(test (make-hook '(-1 0 #f)) 'error)
(test (make-hook '(1 -1 #f)) 'error)
(test (make-hook '(1 . 2)) 'error)
(test (make-hook '(1 2 . #f)) 'error)
(test (make-hook (list 1 0/0 #f)) 'error)
(test (make-hook (list 1 1/0 #f)) 'error)
(test (hook? (make-hook 1)) #t)
(test (make-hook -1) 'error)
(test (let ((lst (list 1 2 #f))) (set! (cdr (cdr (cdr lst))) lst) (make-hook lst)) 'error)

(let ((h (make-hook '(1 0 #f)))
      (x 0))
  (set! (hook-functions h) (list (lambda (y) (set! x (+ x y))) (lambda (z) (set! x (+ x z)))))
  (h 2)
  (num-test x 4)
  (apply h (list 3))
  (num-test x 10)
  (hook-apply h '(4))
  (num-test x 18))

(let ((h1 (make-hook))
      (h2 (make-hook))
      (x 0))
  (let ((f1 (lambda () (h2)))
	(f2 (lambda () (set! x (+ x 123)))))
    (set! (hook-functions h1) (list f1 f2))
    (set! (hook-functions h2) (list (lambda () (set! x (+ x 321)))))
    (h1)
    (num-test x 444)))

(let ((h (make-hook))
      (x 0))
  (define (f1) (if (< x 10) (begin (set! x (+ x 1)) (f1))))
  (set! (hook-functions h) (list f1))
  (h)
  (num-test x 10))

(let ((h (make-hook '(1 0 #f))))
  (define-macro (mac a) `(+ 1 ,a))
  (test (set! (hook-functions h) (list (lambda () 3))) 'error)
  (test (set! (hook-functions h) (list (lambda (x y) 3))) 'error)
  (test (pair? (set! (hook-functions h) (list (lambda x 3)))) #t)
  (test (pair? (set! (hook-functions h) (list (lambda* ((x 1)) 3)))) #t)
  (test (pair? (set! (hook-functions h) (list (lambda* (x (y 1)) 3)))) #t)
  (test (pair? (set! (hook-functions h) (list (lambda (x . z) 3)))) #t)
  (test (pair? (set! (hook-functions h) (list abs sin list + /))) #t)
  (test (set! (hook-functions h) (list quasiquote)) 'error)
  (test (call-with-exit (lambda (return) (set! (hook-functions h) (list return)))) 'error)
  (test (set! (hook-functions h) (list mac)) 'error)
  (test (set! (hook-functions h) (list h)) 'error)
  (test (pair? (set! (hook-functions h) (list hook-functions))) #t)
  )

(test (hook-arity (hook (lambda () 1))) '(0 0 #f))
(test (hook-arity (hook (lambda (x) 1))) '(1 0 #f))
(test (hook-arity (hook (lambda (x . y) 1))) '(1 0 #t))
(test (hook-arity (hook (lambda* ((x 1) (y 2)) 1))) '(0 2 #f))

(test (hook? (hook (lambda (x) x) (lambda (y) 1))) #t)
(test (hook (lambda (x) x) (lambda () 1)) 'error)
(test (hook (lambda (x) x) (lambda (x y) 1)) 'error)
(test (hook? (hook (lambda (x) x) (lambda x 1))) #t)
(test (hook? (hook (lambda () 1) (lambda x 1))) #t)
(test (hook? (hook (lambda x 1) (lambda x 1))) #t)
(test (hook? (hook (lambda (x y) x) (lambda* (x (y 1) (z 2)) 1))) #t)
(test (hook (lambda (x y) x) (lambda* (x) 1)) 'error)
(test (hook (lambda* (x (y 1)) x) (lambda* (x) 1)) 'error)

(let ((h (make-hook '(1 1 #f))))
  (set! (hook-functions h) (list (lambda* (x (y 1)) (+ x y)) (lambda z 2)))
  (test (hook-apply h '(0)) (h 0))
  (test (hook-apply h '(1 2)) (h 1 2))
  (test (hook-apply h '(1 2 3)) 'error)
  (test (h 1 2 3) 'error)
  (test (hook-apply h '()) 'error)
  (test (h) 'error)
  (set! (hook-functions h) (list (lambda x x)))
  (test (pair? (hook-functions h)) #t)
  (set! (hook-functions h) '())
  (test (hook-apply h '(1 2)) (h 1 2))
  )

(let ((h (make-hook '(1 0 #t))))
  (test (object->string h) "#<hook>")
  (test (reverse h) 'error)
  (test (length h) 'error)
  (test (fill! h) 'error)
  (test (hook-arity h) '(1 0 #t))
  (set! (hook-functions h) (list (lambda (x . y) x)))
  (test (hook-apply h '(1 2 3)) (h 1 2 3))
  (test (set! (hook-functions h) (cons (lambda (x . y) x) (lambda (x . y) x))) 'error)
  (let ((lst (list (lambda (x . y) x))))
    (set! (cdr lst) lst)
    (test (set! (hook-functions h) lst) 'error)))

(let ((h (make-hook '(1 0 #f)))
      (sum 0))
  (set! (hook-functions h) (list (lambda (x) (set! sum (+ sum 1)) (if (> x 0) (h (- x 1)) 0))))
  (h 3)
  (if (not (= sum 4)) (format #t ";hook called by hook function: ~A~%" sum)))

(let ((h1 (make-hook '(1 0 #f))))
  (let ((h2 (copy h1)))
    (test (hook-arity h2) '(1 0 #f))
    (test (hook-functions h2) '())
    (test (equal? h1 h2) #t)
    (test (eq? h1 h2) #f)
    (set! (hook-functions h1) (list (lambda (a) (+ a 1))))
    (test (hook-functions h2) '())
    (set! (hook-functions h1) '())
    (set! (hook-functions h2) (list (lambda (a) (+ a 1))))
    (test (hook-functions h1) '())
    (let ((x 0))
      (set! (hook-functions h1) (list (lambda (a) (set! x (+ a 1)))))
      (test (let () (h1 3) (= x 4)) #t)
      (let ((h3 (copy h1)))
	(test (equal? (hook-functions h1) (hook-functions h3)) #t)
	(test (equal? h1 h3) #t)
	(test (equal? h2 h3) #f)
	(h3 4)
	(test x 5)))))

(let* ((h1 (make-hook '(1 0 #f)))
       (x 0)
       (p1 (make-procedure-with-setter (lambda (a) (set! x a)) (lambda (a b) (set! x (+ a b))))))
  (set! (hook-functions h1) (list p1))
  (h1 123)
  (test x 123))



;;; --------------------------------------------------------------------------------
;;; PORTS
;;; --------------------------------------------------------------------------------


(define start-input-port (current-input-port))
(define start-output-port (current-output-port))

(test (input-port? (current-input-port)) #t)
(test (input-port? *stdin*) #t)
(test (input-port? (current-output-port)) #f)
(test (input-port? *stdout*) #f)
(test (input-port? (current-error-port)) #f)
(test (input-port? *stderr*) #f)

(for-each
 (lambda (arg)
   (if (input-port? arg)
       (format #t ";(input-port? ~A) -> #t?~%" arg)))
 (list "hi" #f (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f #<eof> #<undefined> #<unspecified>))

(test (call-with-input-file "s7test.scm" input-port?) #t)
(if (not (eq? start-input-port (current-input-port)))
    (format #t "call-with-input-file did not restore current-input-port? ~A from ~A~%" start-input-port (current-input-port)))

(test (let ((this-file (open-input-file "s7test.scm"))) (let ((res (input-port? this-file))) (close-input-port this-file) res)) #t)
(if (not (eq? start-input-port (current-input-port)))
    (format #t "open-input-file clobbered current-input-port? ~A from ~A~%" start-input-port (current-input-port)))

(test (call-with-input-string "(+ 1 2)" input-port?) #t)
(test (let ((this-file (open-input-string "(+ 1 2)"))) (let ((res (input-port? this-file))) (close-input-port this-file) res)) #t)

;;; read
;;; write
(test (+ 100 (call-with-input-string "123" (lambda (p) (values (read p) 1)))) 224)

(test (call-with-input-string
       "1234567890"
       (lambda (p)
	 (call-with-input-string
	  "0987654321"
	  (lambda (q)
            (+ (read p) (read q))))))
      2222222211)

(test (call-with-input-string
       "12345 67890"
       (lambda (p)
	 (call-with-input-string
	  "09876 54321"
	  (lambda (q)
            (- (+ (read p) (read q)) (read p) (read q))))))
      -99990)

(call-with-output-file "empty-file" (lambda (p) #f))
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read-char p)))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read p)))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read-byte p)))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read-line p)))) #t)
(test (load "empty-file") #<unspecified>)
(test (call-with-input-file "empty-file" (lambda (p) (port-closed? p))) #f)

(let ()
  (define (io-func) (lambda (p) (eof-object? (read-line p))))
  (test (call-with-input-file (let () "empty-file") (io-func)) #t))

(let ((p1 #f))
  (call-with-output-file "empty-file" (lambda (p) (set! p1 p) (write-char #\a p)))
  (test (port-closed? p1) #t))
(test (call-with-input-file "empty-file" (lambda (p) (and (char=? (read-char p) #\a) (eof-object? (read-char p))))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (and (string=? (symbol->string (read p)) "a") (eof-object? (read p))))) #t) ; Guile also returns a symbol here
(test (call-with-input-file "empty-file" (lambda (p) (and (char=? (integer->char (read-byte p)) #\a) (eof-object? (read-byte p))))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (and (string=? (read-line p) "a") (eof-object? (read-line p))))) #t)

(test (call-with-input-string "(lambda (a) (+ a 1))" (lambda (p) (let ((f (eval (read p)))) (f 123)))) 124)
(test (call-with-input-string "(let ((x 21)) (+ x 1))" (lambda (p) (eval (read p)))) 22)
(test (call-with-input-string "(1 2 3) (4 5 6)" (lambda (p) (list (read p) (read p)))) '((1 2 3) (4 5 6)))

(test (let ()
	(call-with-output-file "empty-file" (lambda (p) (write '(lambda (a) (+ a 1)) p)))
	(call-with-input-file "empty-file" (lambda (p) (let ((f (eval (read p)))) (f 123)))))
      124)
(test (let ()
	(call-with-output-file "empty-file" (lambda (p) (write '(let ((x 21)) (+ x 1)) p)))
	(call-with-input-file "empty-file" (lambda (p) (eval (read p)))))
      22)
(test (let ()
	(call-with-output-file "empty-file" (lambda (p) (write '(1 2 3) p) (write '(4 5 6) p)))
	(call-with-input-file "empty-file" (lambda (p) (list (read p) (read p)))))
      '((1 2 3) (4 5 6)))

(call-with-output-file "empty-file" (lambda (p) (for-each (lambda (c) (write-char c p)) "#b11")))
(test (call-with-input-file "empty-file" (lambda (p) 
					   (and (char=? (read-char p) #\#) 
						(char=? (read-char p) #\b) 
						(char=? (read-char p) #\1) 
						(char=? (read-char p) #\1) 
						(eof-object? (read-char p))))) 
      #t)
(test (call-with-input-file "empty-file" (lambda (p) 
					   (and (= (read p) 3) 
						(eof-object? (read p))))) 
      #t)
(test (call-with-input-file "empty-file" (lambda (p) 
					   (and (= (read-byte p) (char->integer #\#))
						(= (read-byte p) (char->integer #\b))
						(= (read-byte p) (char->integer #\1))
						(= (read-byte p) (char->integer #\1))
						(eof-object? (read-byte p))))) 
      #t)
(test (call-with-input-file "empty-file" (lambda (p) 
					   (and (string=? (read-line p) "#b11") 
						(eof-object? (read-line p))))) 
      #t)
(test (load "empty-file") 3)
(let ((p1 (make-procedure-with-setter (lambda (p) (and (= (read p) 3) (eof-object? (read p)))) (lambda (p) #f))))
  (test (call-with-input-file "empty-file" p1) #t))


(test (reverse *stdin*) 'error)
(test (fill! (current-output-port)) 'error)
(test (length *stderr*) 'error)

;; these apparently jump out of the enclosing load too
(for-each
 (lambda (arg)
   (test (load arg) 'error)
   (test (load "empty-file" arg) 'error))
 (list '() (list 1) '(1 . 2) #f #\a 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #f #t (if #f #f) (lambda (a) (+ a 1))))
(test (load) 'error)
(test (load "empty-file" (current-environment) 1) 'error)
(test (load "not a file") 'error)
(test (load "") 'error)
(test (load "/home/bil/cl") 'error)

(test (output-port? (current-input-port)) #f)
(test (output-port? *stdin*) #f)
(test (output-port? (current-output-port)) #t)
(test (output-port? *stdout*) #t)
(test (output-port? (current-error-port)) #t)
(test (output-port? *stderr*) #t)

(write-char #\space (current-output-port))
(write " " (current-output-port))
(newline (current-output-port))


(for-each
 (lambda (arg)
   (if (output-port? arg)
       (format #t ";(output-port? ~A) -> #t?~%" arg)))
 (list "hi" #f '() 'hi (integer->char 65) 1 (list 1 2) _ht_ '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f))

(for-each
 (lambda (arg)
   (test (read-line '() arg) 'error)
   (test (read-line arg) 'error))
 (list "hi" (integer->char 65) 1 #f _ht_ (list) (cons 1 2) (list 1 2) (make-vector 3) 3.14 3/4 1.0+1.0i #\f))

(test (call-with-output-file "tmp1.r5rs" output-port?) #t)
(if (not (eq? start-output-port (current-output-port)))
    (format #t "call-with-output-file did not restore current-output-port? ~A from ~A~%" start-output-port (current-output-port)))

(test (let ((this-file (open-output-file "tmp1.r5rs"))) (let ((res (output-port? this-file))) (close-output-port this-file) res)) #t)
(if (not (eq? start-output-port (current-output-port)))
    (format #t "open-output-file clobbered current-output-port? ~A from ~A~%" start-output-port (current-output-port)))

(test (let ((val #f)) (call-with-output-string (lambda (p) (set! val (output-port? p)))) val) #t)
(test (let ((res #f)) (let ((this-file (open-output-string))) (set! res (output-port? this-file)) (close-output-port this-file) res)) #t)

(for-each
 (lambda (arg)
   (if (eof-object? arg)
       (format #t ";(eof-object? ~A) -> #t?~%" arg)))
 (list "hi" '() '(1 2) -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #f #t (if #f #f) #<undefined> (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (let ((val (catch #t
		     (lambda () (port-closed? arg))
		     (lambda args 'error))))
     (if (not (eq? val 'error))
	 (format #t ";(port-closed? ~A) -> ~S?~%" arg val))))
 (list "hi" '(1 2) -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #f #t (if #f #f) #<undefined> #<eof> (lambda (a) (+ a 1))))

(test (port-closed?) 'error)
(test (port-closed? (current-input-port) (current-output-port)) 'error)

(call-with-output-file "tmp1.r5rs" (lambda (p) (display "3.14" p)))
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (read p) (let ((val (read p))) (eof-object? val)))) #t)

(test (call-with-input-file "tmp1.r5rs" (lambda (p) (read-char p))) #\3)
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (peek-char p))) #\3)
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (peek-char p) (read-char p))) #\3)
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (list->string (list (read-char p) (read-char p) (read-char p) (read-char p))))) "3.14")
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (list->string (list (read-char p) (peek-char p) (read-char p) (read-char p) (peek-char p) (read-char p))))) "3..144")

(for-each
 (lambda (arg)
   (call-with-output-file "tmp1.r5rs" (lambda (p) (write arg p)))
   (test (call-with-input-file "tmp1.r5rs" (lambda (p) (read p))) arg))
 (list "hi" -1 #\a 1 'a-symbol (make-vector 3 0) 3.14 3/4 .6 1.0+1.0i #f #t (list 1 2 3) (cons 1 2)
       '(1 2 . 3) '() '((1 2) (3 . 4)) '(()) (list (list 'a "hi") #\b 3/4) ''a
       (string #\a #\null #\b) "" "\"hi\""
       (integer->char 128) (integer->char 127) (integer->char 255) #\space #\null #\newline #\tab
       #() #2d((1 2) (3 4)) #3d()
       #<eof> #<undefined> #<unspecified>
       most-negative-fixnum
       (if with-bignums 1239223372036854775808 123)
       (if with-bignums 144580536300674537151081081515762353325831/229154728370723013560448485454219755525522 11/10)
       (if with-bignums 221529797579218180403518826416685087012.0 1000.1)
       (if with-bignums 1239223372036854775808+1239223372036854775808i 1000.1-1234i)

       ))

(for-each
 (lambda (arg)
   (call-with-output-file "tmp1.r5rs" (lambda (p) (write arg p)))
   (test (call-with-input-file "tmp1.r5rs" (lambda (p) (eval (read p)))) arg)) ; so read -> symbol?
 (list *stdout* *stdin* *stderr*
       abs + quasiquote
  
;       (hash-table '(a . 1) '(b . 2)) (hash-table)
;       0/0 (real-part (log 0))
;;; for these we need nan? and infinite? since equal? might be #f
;       (lambda (a) (+ a 1))
; pws?
;       (current-output-port)
;       (make-random-state 1234)
;       (symbol ":\"")

;;; macroexpand 
;;; (let () (define-macro (hi1 a) `(+ ,a 1)) hi1)
;;; and how could a continuation work in general?        
       ))

;;; (call-with-input-file "tmp1.r5rs" (lambda (p) (read p))) got (symbol ":\"") but expected (symbol ":\"")


;;; r4rstest
(let* ((write-test-obj '(#t #f a () 9739 -3 . #((test) "te \" \" st" "" test #() b c)))
       (load-test-obj (list 'define 'foo (list 'quote write-test-obj))))
  
  (define (check-test-file name)
    (let ((val (call-with-input-file
		   name
		 (lambda (test-file)
		   (test (read test-file) load-test-obj)
		   (test (eof-object? (peek-char test-file)) #t)
		   (test (eof-object? (read-char test-file)) #t)
		   (input-port? test-file)))))
      (if (not (eq? val #t))
	  (format #t "input-port? in call-with-input-file? returned ~A from ~A~%" val name))))
  
  (test (call-with-output-file
	    "tmp1.r5rs"
	  (lambda (test-file)
	    (write-char #\; test-file)
	    (display #\; test-file)
	    (display ";" test-file)
	    (write write-test-obj test-file)
	    (newline test-file)
	    (write load-test-obj test-file)
	    (output-port? test-file))) #t)
  (check-test-file "tmp1.r5rs")
  
  (let ((test-file (open-output-file "tmp2.r5rs")))
    (test (port-closed? test-file) #f)
    (write-char #\; test-file)
    (display #\; test-file)
    (display ";" test-file)
    (write write-test-obj test-file)
    (newline test-file)
    (write load-test-obj test-file)
    (test (output-port? test-file) #t)
    (close-output-port test-file)
    (check-test-file "tmp2.r5rs")))


(call-with-output-file "tmp1.r5rs" (lambda (p) (display "3.14" p)))
(test (with-input-from-file "tmp1.r5rs" (lambda () (read))) 3.14)
(if (not (eq? start-input-port (current-input-port)))
    (format #t "with-input-from-file did not restore current-input-port? ~A from ~A~%" start-input-port (current-input-port)))

(test (with-input-from-file "tmp1.r5rs" (lambda () (eq? (current-input-port) start-input-port))) #f)

(test (with-output-to-file "tmp1.r5rs" (lambda () (eq? (current-output-port) start-output-port))) #f)
(if (not (eq? start-output-port (current-output-port)))
    (format #t "with-output-to-file did not restore current-output-port? ~A from ~A~%" start-output-port (current-output-port)))


(let ((newly-found-sonnet-probably-by-shakespeare 
       "This is the story, a sad tale but true \
        Of a programmer who had far too little to do.\
        One day as he sat in his hut swilling stew, \
        He cried \"CLM takes forever, it's stuck in a slough!,\
        Its C code is slow, too slow by a few.\
        Why, with just a small effort, say one line or two,\
        It could outpace a no-op, you could scarcely say 'boo'\"!\
        So he sat in his kitchen and worked like a dog.\
        He typed and he typed 'til his mind was a fog. \
        Now 6000 lines later, what wonders we see!  \
        CLM is much faster, and faster still it will be!\
        In fact, for most cases, C beats the DSP!  \
        But bummed is our coder; he grumbles at night.  \
        That DSP code took him a year to write.  \
        He was paid many dollars, and spent them with glee,\
        But his employer might mutter, this result were he to see."))
  
  (call-with-output-file "tmp1.r5rs"
    (lambda (p)
      (write newly-found-sonnet-probably-by-shakespeare p)))
  
  (let ((sonnet (with-input-from-file "tmp1.r5rs"
		  (lambda ()
		    (read)))))
    (if (or (not (string? sonnet))
	    (not (string=? sonnet newly-found-sonnet-probably-by-shakespeare)))
	(format #t "write/read long string returned: ~A~%" sonnet)))
  
  (let ((file (open-output-file "tmp1.r5rs")))
    (let ((len (string-length newly-found-sonnet-probably-by-shakespeare)))
      (write-char #\" file)
      (do ((i 0 (+ i 1)))
	  ((= i len))
	(let ((chr (string-ref newly-found-sonnet-probably-by-shakespeare i)))
	  (if (char=? chr #\")
	      (write-char #\\ file))
	  (write-char chr file)))
      (write-char #\" file)
      (close-output-port file)))
  
  (let ((file (open-input-file "tmp1.r5rs")))
    (let ((sonnet (read file)))
      (close-input-port file)
      (if (or (not (string? sonnet))
	      (not (string=? sonnet newly-found-sonnet-probably-by-shakespeare)))
	  (format #t "write-char/read long string returned: ~A~%" sonnet)))))

(let ((file (open-output-file "tmp1.r5rs")))
  (for-each
   (lambda (arg)
     (write arg file)
     (write-char #\space file))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))
  (close-output-port file))

(let ((file (open-input-file "tmp1.r5rs")))
  (for-each
   (lambda (arg)
     (let ((val (read file)))
       (if (not (equal? val arg))
	   (format #t "read/write ~A returned ~A~%" arg val))))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))
  (close-input-port file))

(with-output-to-file "tmp1.r5rs"
  (lambda ()
    (write lists)))

(let ((val (with-input-from-file "tmp1.r5rs"
	     (lambda ()
	       (read)))))
  (if (not (equal? val lists))
      (format #t "read/write lists returned ~A~%" val)))

(if (not (string=? "" (with-output-to-string (lambda () (display "")))))
    (format #t "with-output-to-string null string?"))

(let ((str (with-output-to-string
	     (lambda ()
	       (with-input-from-string "hiho123"
		 (lambda ()
		   (do ((c (read-char) (read-char)))
		       ((eof-object? c))
		     (display c))))))))
  (if (not (string=? str "hiho123"))
      (format #t "with string ports: ~S?~%" str)))


(if (not (eof-object? (with-input-from-string "" (lambda () (read-char)))))
    (format #t ";input from null string not #<eof>?~%")
    (let ((EOF (with-input-from-string "" (lambda () (read-char)))))
      (if (not (eq? (with-input-from-string "" (lambda () (read-char)))
		    (with-input-from-string "" (lambda () (read-char)))))
	  (format #t "#<eof> is not eq? to itself?~%"))
      (if (char? EOF)
	  (do ((c 0 (+ c 1)))
	      ((= c 256))
	    (if (char=? EOF (integer->char c))
		(format #t "#<eof> is char=? to ~C~%" (integer->char c)))))))

(test (+ 100 (call-with-output-file "tmp.r5rs" (lambda (p) (write "1" p) (values 1 2)))) 103)
(test (+ 100 (with-output-to-file "tmp.r5rs" (lambda () (write "2") (values 1 2)))) 103)

(let ((str (with-output-to-string
	     (lambda ()
	       (with-input-from-string "hiho123"
		 (lambda ()
		   (do ((c (read-char) (read-char)))
		       ((or (not (char-ready?))
			    (eof-object? c)))
		     (display c))))))))
  (if (not (string=? str "hiho123"))
      (format #t "with string ports: ~S?~%" str)))

(let ((str (with-output-to-string
	     (lambda ()
	       (with-input-from-string ""
		 (lambda ()
		   (do ((c (read-char) (read-char)))
		       ((eof-object? c))
		     (display c))))))))
  (if (not (string=? str ""))
      (format #t "with string ports and null string: ~S?~%" str)))

(let ((str (with-output-to-string ; this is from the guile-user mailing list, I think -- don't know who wrote it
	     (lambda ()
	       (with-input-from-string "A2B5E3426FG0ZYW3210PQ89R."
		 (lambda ()
		   (call/cc
		    (lambda (hlt)
		      (define (nextchar)
			(let ((c (read-char)))
			  (if (and (char? c) 
				   (char=? c #\space))
			      (nextchar) 
			      c)))
		      
		      (define inx
			(lambda()
			  (let in1 ()
			    (let ((c (nextchar)))
			      (if (char-numeric? c)
				  (let ((r (nextchar)))
				    (let out*n ((n (- (char->integer c) (char->integer #\0))))
				      (out r)
				      (if (not (zero? n))
					  (out*n (- n 1)))))
				  (out c))
			      (in1)))))
		      
		      (define (move-char c)
			(write-char c)
			(if (char=? c #\.)
			    (begin (hlt))))
		      
		      (define outx
			(lambda()
			  (let out1 ()
			    (let h1 ((n 16))
			      (move-char (in))
			      (move-char (in))
			      (move-char (in))
			      (if (= n 1)
				  (begin (out1))
				  (begin (write-char #\space) (h1 (- n 1))) )))))
		      
		      (define (in)
			(call/cc (lambda(return)
				   (set! outx return)
				   (inx))))
		      
		      (define (out c)
			(call/cc (lambda(return) 
				   (set! inx return)
				   (outx c))))
		      (outx)))))))))
  (if (not (string=? str "ABB BEE EEE E44 446 66F GZY W22 220 0PQ 999 999 999 R."))
      (format #t "call/cc with-input-from-string str: ~A~%" str)))

(let ((badfile "tmp1.r5rs"))
  (let ((p (open-output-file badfile)))
    (close-output-port p))
  (load badfile))

(for-each
 (lambda (str)
   ;;(test (eval-string str) 'error)
   ;; eval-string is confused somehow
   (test (with-input-from-string str (lambda () (read))) 'error))
 (list "\"\\x" "\"\\x0" "`(+ ," "`(+ ,@" "#2d(" "#\\"))

(let ((loadit "tmp1.r5rs"))
  (let ((p (open-output-file loadit)))
    (display "(define s7test-var 314) (define (s7test-func) 314) (define-macro (s7test-mac a) `(+ ,a 2))" p)
    (newline p)
    (close-output-port p)
    (load loadit)
    (test (= s7test-var 314) #t)
    (test (s7test-func) 314)
    (test (s7test-mac 1) 3)
    (set! p (open-output-file loadit)) ; hopefully this starts a new file
    (display "(define s7test-var 3) (define (s7test-func) 3) (define-macro (s7test-mac a) `(+ ,a 1))" p)
    (newline p)
    (close-output-port p)
    (load loadit)
    (test (= s7test-var 3) #t)
    (test (s7test-func) 3)
    (test (s7test-mac 1) 2)
    ))

(test (+ 100 (with-input-from-string "123" (lambda () (values (read) 1)))) 224)

(for-each
 (lambda (op)
   (for-each
    (lambda (arg) ;(format #t ";(~A ~A)~%" op arg)
      (test (op arg) 'error))
    (list (integer->char 65) 1 0 -1 (list 1) (cons 1 2) #f 'a-symbol (make-vector 3) abs lambda with-environment
	  _ht_ quasiquote macroexpand make-type hook-functions 
	  3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1)))))
 (list char-ready? set-current-output-port set-current-input-port set-current-error-port
       close-input-port close-output-port open-input-file open-output-file
       read-char peek-char read 
       (lambda (arg) (write-char #\a arg))
       (lambda (arg) (write "hi" arg))
       (lambda (arg) (display "hi" arg))
       call-with-input-file with-input-from-file call-with-output-file with-output-to-file))

(with-output-to-file "tmp1.r5rs"
  (lambda ()
    (display "this is a test")
    (newline)))
    
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (integer->char (read-byte p)))) #\t)
(test (with-input-from-string "123" (lambda () (read-byte))) 49)
;(test (with-input-from-string "1/0" (lambda () (read))) 'error) ; this is a reader error in CL
;;; this test causes trouble when s7test is called from snd-test -- I can't see why

(let ((bytes (vector #o000 #o000 #o000 #o034 #o000 #o001 #o215 #o030 #o000 #o000 #o000 #o022 #o000 
		     #o000 #o126 #o042 #o000 #o000 #o000 #o001 #o000 #o000 #o000 #o000 #o000 #o001)))
  (with-output-to-file "tmp1.r5rs"
    (lambda ()
      (for-each
       (lambda (b)
	 (write-byte b))
       bytes)))
  
  (let ((ctr 0))
    (call-with-input-file "tmp1.r5rs"
      (lambda (p)	
	(if (not (string=? (port-filename p) "tmp1.r5rs")) (display (port-filename p)))	
	(let loop ((val (read-byte p)))
	  (if (eof-object? val)
	      (if (not (= ctr 26))
		  (format #t "read-byte done at ~A~%" ctr))
	      (begin
		(if (not (= (bytes ctr) val))
		    (format #t "read-byte bytes[~D]: ~A ~A~%" ctr (bytes ctr) val))
		(set! ctr (+ 1 ctr))
		(loop (read-byte p))))))))
  
  (let ((ctr 0))
    (call-with-input-file "tmp1.r5rs"
      (lambda (p)
	(let loop ((val (read-char p)))
	  (if (eof-object? val)
	      (if (not (= ctr 26))
		  (format #t "read-char done at ~A~%" ctr))
	      (begin
		(if (not (= (bytes ctr) (char->integer val)))
		    (format #t "read-char bytes[~D]: ~A ~A~%" ctr (bytes ctr) (char->integer val)))
		(set! ctr (+ 1 ctr))
		(loop (read-char p))))))))
  )

(with-output-to-file "tmp1.r5rs"
  (lambda ()
    (if (not (string=? (port-filename (current-output-port)) "tmp1.r5rs")) (display (port-filename (current-output-port))))
    (display "(+ 1 2) 32")
    (newline)
    (display "#\\a  -1")))

(with-input-from-file "tmp1.r5rs"
  (lambda ()
    (if (not (string=? (port-filename (current-input-port)) "tmp1.r5rs")) (display (port-filename (current-input-port))))
    (let ((val (read)))
      (if (not (equal? val (list '+ 1 2)))
	  (format #t "read: ~A~%" val)))
    (let ((val (read)))
      (if (not (equal? val 32))
	  (format #t "read: ~A~%" val)))
    (let ((val (read)))
      (if (not (equal? val #\a))
	  (format #t "read: ~A~%" val)))
    (let ((val (read)))
      (if (not (equal? val -1))
	  (format #t "read: ~A~%" val)))
    (let ((val (read)))
      (if (not (eof-object? val))
	  (format #t "read: ~A~%" val)))
    (let ((val (read)))
      (if (not (eof-object? val))
	  (format #t "read again: ~A~%" val)))))

(let ((port #f))
  (call-with-exit
   (lambda (go)
     (call-with-input-string "0123456789"
       (lambda (p)
	 (set! port p)
	 (if (not (char=? (peek-char p) #\0))
	     (format #t ";peek-char input-string: ~A~%" (peek-char p)))
	 (go)))))
  (if (not (input-port? port))
      (format #t ";c/e-> c/is -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format #t ";c/e -> c/is -> closed? ~A~%" port)
	    (close-input-port port)))))

(call-with-output-file "tmp1.r5rs" (lambda (p) (display "0123456789" p)))

(let ((port #f))
  (call-with-exit
   (lambda (go)
     (call-with-input-file "tmp1.r5rs"
       (lambda (p)
	 (set! port p)
	 (if (not (char=? (peek-char p) #\0))
	     (format #t ";peek-char input-file: ~A~%" (peek-char p)))
	 (go)))))
  (if (not (input-port? port))
      (format #t ";c/e -> c/if -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format #t ";c/e -> c/if -> closed? ~A~%" port)
	    (close-input-port port)))))

(let ((port #f))
  (call-with-exit
   (lambda (go)
     (dynamic-wind
	 (lambda () #f)
	 (lambda ()
	   (call-with-input-string "0123456789"
             (lambda (p)
	       (set! port p)
	       (if (not (char=? (peek-char p) #\0))
		   (format #t ";peek-char input-string 1: ~A~%" (peek-char p)))
	       (go))))
	 (lambda ()
	   (close-input-port port)))))
  (if (not (input-port? port))
      (format #t ";c/e -> dw -> c/is -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format #t ";c/e -> dw -> c/is -> closed? ~A~%" port)
	    (close-input-port port)))))

(let ((port #f))
  (call-with-exit
   (lambda (go)
     (dynamic-wind
	 (lambda () #f)
	 (lambda ()
	   (call-with-input-file "tmp1.r5rs"
            (lambda (p)
	      (set! port p)
	      (if (not (char=? (peek-char p) #\0))
		  (format #t ";peek-char input-file: ~A~%" (peek-char p)))
	      (go))))
	 (lambda ()
	   (close-input-port port)))))
  (if (not (input-port? port))
      (format #t ";c/e -> dw -> c/if -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format #t ";c/e -> dw -> c/if -> closed? ~A~%" port)
	    (close-input-port port)))))

(let ((port #f))
  (catch #t
    (lambda ()
     (call-with-input-string "0123456789"
       (lambda (p)
	 (set! port p)
	 (if (not (char=? (peek-char p) #\0))
	     (format #t ";peek-char input-string: ~A~%" (peek-char p)))
	 (error 'oops))))
    (lambda args #f))
  (if (not (input-port? port))
      (format #t ";catch -> c/is -> error -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format #t ";catch -> c/is -> error -> closed? ~A~%" port)
	    (close-input-port port)))))

(let ((port #f))
  (catch #t
    (lambda ()
     (call-with-input-file "tmp1.r5rs"
       (lambda (p)
	 (set! port p)
	 (if (not (char=? (peek-char p) #\0))
	     (format #t ";peek-char input-file: ~A~%" (peek-char p)))
	 (error 'oops))))
    (lambda args #f))
  (if (not (input-port? port))
      (format #t ";catch -> c/if -> error -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format #t ";catch -> c/if -> error -> closed? ~A~%" port)
	    (close-input-port port)))))

(test (with-output-to-string (lambda () (write (string (integer->char 4) (integer->char 8) (integer->char 20) (integer->char 30))))) "\"\\x04\\x08\\x14\\x1e\"")
(test (string-length "\x04\x08\x14\x1e") 4)
(test (char->integer (string-ref "\x0" 0)) 0)
(test (char->integer (string-ref "\x0e" 0)) 14)
(test (char->integer (string-ref "\x1e" 0)) 30)
(test (char->integer (string-ref "\xff" 0)) 255)
(test (string=?
        "\"\\x01\\x02\\x03\\x04\\x05\\x06\\x07\\x08\\x09x\\x0b\\x0c\\x0d\\x0e\\x0f\\x10\\x11\\x12\\x13\\x14\\x15\\x16\\x17\\x18\\x19\\x1a\\x1b\\x1c\\x1d\\x1e\\x1f !\\\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\\x7f\\x80\\x81\\x82\\x83\\x84\\x85\\x86\\x87\\x88\\x89\\x8a\\x8b\\x8c\\x8d\\x8e\\x8f\\x90\\x91\\x92\\x93\\x94\\x95\\x96\\x97\\x98\\x99\\x9a\\x9b\\x9c\\x9d\\x9e\\x9f\\xa0¡¢£¤¥¦§¨©ª«¬\\xad®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖ×ØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõö÷øùúûüýþÿ\""             "\"\\x01\\x02\\x03\\x04\\x05\\x06\\x07\\x08\\x09x\\x0b\\x0c\\x0d\\x0e\\x0f\\x10\\x11\\x12\\x13\\x14\\x15\\x16\\x17\\x18\\x19\\x1a\\x1b\\x1c\\x1d\\x1e\\x1f !\\\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\\x7f\\x80\\x81\\x82\\x83\\x84\\x85\\x86\\x87\\x88\\x89\\x8a\\x8b\\x8c\\x8d\\x8e\\x8f\\x90\\x91\\x92\\x93\\x94\\x95\\x96\\x97\\x98\\x99\\x9a\\x9b\\x9c\\x9d\\x9e\\x9f\\xa0¡¢£¤¥¦§¨©ª«¬\\xad®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖ×ØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõö÷øùúûüýþÿ\"") 
      #t)
(test (string=? "\x61\x42\x63" "aBc") #t)



(for-each
 (lambda (arg)
   (test (char-ready? arg) 'error))
 (list "hi" -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #f #t (if #f #f) (lambda (a) (+ a 1))))


;;; -------- format --------
;;; format

(test (format #f "hiho") "hiho")
(test (format #f "") "")
(test (format #f "" 1) 'error)
(test (format #f "a") "a")
(test (format #f "a\x00b") "a")

(test (format #f "~~") "~") ; guile returns this, but clisp thinks it's an error
(test (format #f "~~~~") "~~")
(test (format #f "a~~") "a~")
(test (format #f "~~a") "~a")
(test (format #f "~A" "") "")
(test (format #f "~{~^~A~}" '()) "")
(test (format #f "~{~^~{~^~A~}~}" '(())) "")
(test (format #f "~P" 1) "")
(test (format #f "~P" #\a) 'error)
(test (format #f "~0T") "")
(test (format #f "") "")
(test (format #f "~*~*" 1 2) "")
(test (format #f "~20,'~D" 3) "~~~~~~~~~~~~~~~~~~~3")
(test (format #f "~0D" 123) "123")
(test (format #f "~-1D" 123) 'error)
(test (format #f "~+1D" 123) 'error)
(test (format #f "~1.D" 123) 'error)
(test (format #f "~20,'-1D" 123) 'error)

(test (format #f "hiho~%ha") (string-append "hiho" (string #\newline) "ha"))
(test (format #f "~%") (string #\newline))
(test (format #f "~%ha") (string-append (string #\newline) "ha"))
(test (format #f "hiho~%") (string-append "hiho" (string #\newline)))

(for-each
 (lambda (arg res)
   (let ((val (catch #t (lambda () (format #f "~A" arg)) (lambda args 'error))))
     (if (or (not (string? val))
	     (not (string=? val res)))
	 (begin (display "(format #f \"~A\" ") (display arg) 
		(display " returned \"") (display val) 
		(display "\" but expected \"") (display res) (display "\"") 
		(newline)))))
 (list "hiho"  -1  #\a  1   #f   #t  '#(1 2 3)   3.14   3/4  1.5+1.5i '()  '#(())  (list 1 2 3) '(1 . 2) 'hi)
 (list "hiho" "-1" "a" "1" "#f" "#t" "#(1 2 3)" "3.14" "3/4" "1.5+1.5i"   "()" "#(())" "(1 2 3)"    "(1 . 2)" "hi"))

(test (format #f "hi ~A ho" 1) "hi 1 ho")
(test (format #f "hi ~a ho" 1) "hi 1 ho")
(test (format #f "~a~A~a" 1 2 3) "123")
(test (format #f "~a~~~a" 1 3) "1~3")
(test (format #f "~a~%~a" 1 3) (string-append "1" (string #\newline) "3"))

(for-each
 (lambda (arg res)
   (let ((val (catch #t (lambda () (format #f "~S" arg)) (lambda args 'error))))
     (if (or (not (string? val))
	     (not (string=? val res)))
	 (begin (display "(format #f \"~S\" ") (display arg) 
		(display " returned \"") (display val) 
		(display "\" but expected \"") (display res) (display "\"") 
		(newline)))))
 (list "hiho"  -1  #\a  1   #f   #t  '#(1 2 3)   3.14   3/4  1.5+1.5i '()  '#(())  (list 1 2 3) '(1 . 2) 'hi)
 (list "\"hiho\"" "-1" "#\\a" "1" "#f" "#t" "#(1 2 3)" "3.14" "3/4" "1.5+1.5i"   "()" "#(())" "(1 2 3)"    "(1 . 2)" "hi"))

(test (format #f "hi ~S ho" 1) "hi 1 ho")
(test (format #f "hi ~S ho" "abc") "hi \"abc\" ho")
(test (format #f "~s~a" #\a #\b) "#\\ab")
(test (format #f "~C~c~C" #\a #\b #\c) "abc")

(test (format #f "~{~A~}" '(1 2 3)) "123")
(test (format #f "asb~{~A ~}asb" '(1 2 3 4)) "asb1 2 3 4 asb")
(test (format #f "asb~{~A ~A.~}asb" '(1 2 3 4)) "asb1 2.3 4.asb")
(test (format #f ".~{~A~}." '()) "..")

(test (format #f "~{~A ~A ~}" '(1 "hi" 2 "ho")) "1 hi 2 ho ")
(test (format #f "~{.~{+~A+~}.~}" (list (list 1 2 3) (list 4 5 6))) ".+1++2++3+..+4++5++6+.")
(test (format #f "~{~s ~}" '(fred jerry jill)) "fred jerry jill ")
(test (format #f "~{~s~^ ~}" '(fred jerry jill)) "fred jerry jill")
(test (format #f "~{~s~^~^ ~}" '(fred jerry jill)) "fred jerry jill")
(test (format #f "~{.~{~A~}+~{~A~}~}" '((1 2) (3 4 5) (6 7 8) (9))) ".12+345.678+9")
(test (format #f "~{.~{+~{-~A~}~}~}" '(((1 2) (3 4 5)))) ".+-1-2+-3-4-5")
(test (format #f "~{.~{+~{-~A~}~}~}" '(((1 2) (3 4 5)) ((6) (7 8 9)))) ".+-1-2+-3-4-5.+-6+-7-8-9")

(test (format #f "~A ~* ~A" 1 2 3) "1  3")
(test (format #f "~*" 1) "")
(test (format #f "~{~* ~}" '(1 2 3)) "   ")
(test (format #f "~A" catch) "catch")
(test (format #f "this is a ~
             sentence") "this is a sentence")
(test (format #f "~{~C~}" "hi") "hi")
(test (format #f "~{~C~}" #(#\h #\i)) "hi")

(test (format #f "~{.~{~C+~}~}" '((#\h #\i) (#\h #\o))) ".h+i+.h+o+")
(test (format #f "~{.~{~C+~}~}" '("hi" "ho")) ".h+i+.h+o+")
(test (format #f "~{.~{~C+~}~}" #("hi" "ho")) ".h+i+.h+o+")
(test (format #f "~{.~{~C+~}~}" #(#(#\h #\i) #(#\h #\o))) ".h+i+.h+o+")

; (format #f "~{.~{~C~+~}~}" #2d((#\h #\i) (#\h #\o))) error?? -- this is documented...
(test (format #f "~{~A~}" #2D((1 2) (3 4))) "1234") ; this seems inconsistent with:
(test (format #f "~{~A~}" '((1 2) (3 4))) "(1 2)(3 4)")
(test (format #f "~{~A ~}" #2d((1 2) (3 4))) "1 2 3 4 ")
(test (format #f "1~\
a2" 3) "132")

;; ~nT handling is a mess -- what are the defaults?  which is column 1? do we space up to or up to and including?

(test (format #f "asdh~20Thiho") "asdh               hiho")
(test (format #f "asdh~2Thiho") "asdhhiho")
(test (format #f "a~Tb") "ab")
(test (format #f "0123456~4,8Tb") "0123456    b")
(test (format #f "0123456~0,8Tb") "0123456b")
(test (format #f "0123456~10,8Tb") "0123456          b")
(test (format #f "0123456~1,0Tb") "0123456b")
(test (format #f "0123456~1,Tb") "0123456b")
(test (format #f "0123456~1,Tb") "0123456b")
(test (format #f "0123456~,Tb") "0123456b")
(test (format #f "0123456~7,10Tb") "0123456         b")
(test (format #f "0123456~8,10tb") "0123456          b")
(test (format #f "0123456~3,12tb") "0123456       b")
(test (format #f "~40TX") "                                       X")
(test (format #f "X~,8TX~,8TX") "X      X       X")
(test (format #f "X~8,TX~8,TX") "X      XX")
(test (format #f "X~8,10TX~8,10TX") "X                X         X")
(test (format #f "X~8,0TX~8,0TX") "X      XX")
(test (format #f "X~0,8TX~0,8TX") "X      X       X")
(test (format #f "X~1,8TX~1,8TX") "X       X       X")
(test (format #f "X~,8TX~,8TX") "X      X       X") ; ??
(test (format #f "X~TX~TX") "XXX") ; clisp and sbcl say "X X X" here and similar differences elsewhere -- is it colnum or colinc as default if no comma?
(test (format #f "X~2TX~4TX") "XX X")
(test (format #f "X~0,0TX~0,0TX") "XXX")
(test (format #f "X~0,TX~0,TX") "XXX")
(test (format #f "X~,0TX~,0TX") "XXX")

(test (= (length (substring (format #f "~%~10T.") 1)) (length (format #f "~10T."))) #t)
(test (= (length (substring (format #f "~%-~10T.~%") 1)) (length (format #f "-~10T.~%"))) #t)
(test (string=? (format #f "~%|0 1 2|~21T|5  8  3  2|~%~
                              |1 2 3| |0 1 2 3|~21T|8 14  8  6|~%~
                              |2 3 0| |1 2 3 0| = ~21T|3  8 13  6|~%~
                              |3 0 1| |2 3 0 1|~21T|2  6  6 10|~%")
		"
|0 1 2|             |5  8  3  2|
|1 2 3| |0 1 2 3|   |8 14  8  6|
|2 3 0| |1 2 3 0| = |3  8 13  6|
|3 0 1| |2 3 0 1|   |2  6  6 10|
") #t)

(test (format #f "~12,''D" 1) "'''''''''''1")
(test (let ((str "~12,'xD")) (set! (str 5) #\space) (format #f str 1)) "           1")
(test (format #f "~12,' D" 1) "           1")
(test (format #f "~12,'\\D" 1) "\\\\\\\\\\\\\\\\\\\\\\1")
(test (format #f "~12,'\"D" 1) "\"\"\"\"\"\"\"\"\"\"\"1")
(test (format #f "~12,'~D" 1) "~~~~~~~~~~~1")
(test (format #f "~12,',d" 1) ",,,,,,,,,,,1")
(test (format #f "~12,',,d" 1) 'error)
(test (format #f "~12,,d" 1) 'error)

(test (string=? (format #f "~%~&" ) (string #\newline)) #t)
(test (string=? (format #f "~%a~&" ) (string #\newline #\a #\newline)) #t)
(test (string=? (format #f "~%~%") (string #\newline #\newline)) #t)
(test (string=? (format #f "~10T~%~&~10T.") (format #f "~10T~&~&~10T.")) #t)
(test (string=? (format #f "~10T~&~10T.") (format #f "~10T~%~&~&~&~&~10T.")) #t)

(test (format #f "~2,1F" 0.5) "0.5")
(test (format #f "~:2T") 'error)
(test (format #f "~2,1,3F" 0.5) 'error)
(test (format #f "~<~W~>" 'foo) 'error)
(test (format #f "~{12") 'error)
(test (format #f "~{}") 'error)
(test (format #f "~{}" '(1 2)) 'error)
(test (format #f "{~}" '(1 2)) 'error)
(test (format #f "~{~{~}}" '(1 2)) 'error)
(test (format #f "~}" ) 'error)
(test (format #f "#|~|#|") 'error)
(test (format #f "~1.5F" 1.5) 'error)
(test (format #f "~1+iF" 1.5) 'error)
(test (format #f "~1,1iF" 1.5) 'error)
(test (format #f "~0" 1) 'error)
(test (format #f "~1") 'error)
(test (format #f "~^" 1) 'error)
(test (format #f "~.0F" 1.0) 'error)
(test (format #f "~1.0F" 1.0) 'error)
(test (format #f "~-1F" 1.0) 'error)
(test (format #f "~^") "")
(test (format #f "~D~" 9) 'error)
(test (format #f "~&" 9) 'error)
(test (format #f "~D~100T~D" 1 1) "1                                                                                                  1")
(test (format #f ".~P." 1) "..")
(test (format #f ".~P." 1.0) "..")
(test (format #f ".~P." 1.2) ".s.")
(test (format #f ".~P." 2) ".s.")
(test (format #f ".~p." 1) "..")
(test (format #f ".~p." 1.0) "..")
(test (format #f ".~p." 1.2) ".s.")
(test (format #f ".~p." 2) ".s.")
(test (format #f ".~@P." 1) ".y.")
(test (format #f ".~@P." 1.0) ".y.")
(test (format #f ".~@P." 1.2) ".ies.")
(test (format #f ".~@P." 2) ".ies.")
(test (format #f ".~@p." 1) ".y.")
(test (format #f ".~@p." 1.0) ".y.")
(test (format #f ".~@p." 1.2) ".ies.")
(test (format #f ".~@p." 2) ".ies.")

(test (format #f (string #\~ #\a) 1) "1")
(test (format #f (format #f "~~a") 1) "1")
(test (format #f (format #f "~~a") (format #f "~D" 1)) "1")
(test (format #f "~A" (quasiquote quote)) "quote")

(test (format #f "~f" (/ 1 3)) "1/3") ; hmmm -- should it call exact->inexact?
(test (format #f "~f" 1) "1")
(test (format #f "~F" most-positive-fixnum) "9223372036854775807")

(if with-bignums
    (begin
      (test (format #f "~A" -7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601) "-7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601")
      (test (format #f "~D" -7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601) "-7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601")
    ))
(test (format #f "~@F" 1.23) 'error)
(test (format #f "~{testing ~D ~C ~}" (list 0 #\( 1 #\) 2 #\* 3 #\+ 4 #\, 5 #\- 6 #\. 7 #\/ 8 #\0 9 #\1 10 #\2 11 #\3 12 #\4 13 #\5 14 #\6 15 #\7 16 #\8 17 #\9 18 #\: 19 #\; 20 #\< 21 #\= 22 #\> 23 #\? 24 #\@ 25 #\A 26 #\B 27 #\C 28 #\D 29 #\E 30 #\F 31 #\G 32 #\H 33 #\I 34 #\J 35 #\K 36 #\L 37 #\M 38 #\N 39 #\O 40 #\P 41 #\Q 42 #\R 43 #\S 44 #\T 45 #\U 46 #\V 47 #\W 48 #\X 49 #\Y 50 #\( 51 #\) 52 #\* 53 #\+ 54 #\, 55 #\- 56 #\. 57 #\/ 58 #\0 59 #\1 60 #\2 61 #\3 62 #\4 63 #\5 64 #\6 65 #\7 66 #\8 67 #\9 68 #\: 69 #\; 70 #\< 71 #\= 72 #\> 73 #\? 74 #\@ 75 #\A 76 #\B 77 #\C 78 #\D 79 #\E 80 #\F 81 #\G 82 #\H 83 #\I 84 #\J 85 #\K 86 #\L 87 #\M 88 #\N 89 #\O 90 #\P 91 #\Q 92 #\R 93 #\S 94 #\T 95 #\U 96 #\V 97 #\W 98 #\X 99 #\Y))
      "testing 0 ( testing 1 ) testing 2 * testing 3 + testing 4 , testing 5 - testing 6 . testing 7 / testing 8 0 testing 9 1 testing 10 2 testing 11 3 testing 12 4 testing 13 5 testing 14 6 testing 15 7 testing 16 8 testing 17 9 testing 18 : testing 19 ; testing 20 < testing 21 = testing 22 > testing 23 ? testing 24 @ testing 25 A testing 26 B testing 27 C testing 28 D testing 29 E testing 30 F testing 31 G testing 32 H testing 33 I testing 34 J testing 35 K testing 36 L testing 37 M testing 38 N testing 39 O testing 40 P testing 41 Q testing 42 R testing 43 S testing 44 T testing 45 U testing 46 V testing 47 W testing 48 X testing 49 Y testing 50 ( testing 51 ) testing 52 * testing 53 + testing 54 , testing 55 - testing 56 . testing 57 / testing 58 0 testing 59 1 testing 60 2 testing 61 3 testing 62 4 testing 63 5 testing 64 6 testing 65 7 testing 66 8 testing 67 9 testing 68 : testing 69 ; testing 70 < testing 71 = testing 72 > testing 73 ? testing 74 @ testing 75 A testing 76 B testing 77 C testing 78 D testing 79 E testing 80 F testing 81 G testing 82 H testing 83 I testing 84 J testing 85 K testing 86 L testing 87 M testing 88 N testing 89 O testing 90 P testing 91 Q testing 92 R testing 93 S testing 94 T testing 95 U testing 96 V testing 97 W testing 98 X testing 99 Y ")

(let ((old-len *vector-print-length*))
  (let ((vect1 #3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))))
	(vect2 #2d((1 2 3 4 5 6) (7 8 9 10 11 12)))
	(vect3 #(1 2 3 4 5 6 7 8 9 10 11 12 13 14))
	(vect4 #3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12)))))
    (do ((i 0 (+ i 2)))
	((>= i 10))
      (set! *vector-print-length* i)
      (test (object->string vect1) (format #f "~A" vect1))
      (test (object->string vect2) (format #f "~A" vect2))
      (test (object->string vect3) (format #f "~A" vect3))
      (test (object->string vect4) (format #f "~A" vect4))))
  (set! *vector-print-length* 0)
  (test (format #f "~A" #()) "#()")
  (set! *vector-print-length* old-len))

(test (format #f "~D" 123) "123")
(test (format #f "~X" 123) "7b")
(test (format #f "~B" 123) "1111011")
(test (format #f "~O" 123) "173")

(test (format #f "~10D" 123) "       123")
(test (format #f "~10X" 123) "        7b")
(test (format #f "~10B" 123) "   1111011")
(test (format #f "~10O" 123) "       173")

(test (format #f "~D" -123) "-123")
(test (format #f "~X" -123) "-7b")
(test (format #f "~B" -123) "-1111011")
(test (format #f "~O" -123) "-173")

(test (format #f "~10D" -123) "      -123")
(test (format #f "~10X" -123) "       -7b")
(test (format #f "~10B" -123) "  -1111011")
(test (format #f "~10O" -123) "      -173")

(test (format #f "~d" 123) "123")
(test (format #f "~x" 123) "7b")
(test (format #f "~b" 123) "1111011")
(test (format #f "~o" 123) "173")

(test (format #f "~10d" 123) "       123")
(test (format #f "~10x" 123) "        7b")
(test (format #f "~10b" 123) "   1111011")
(test (format #f "~10o" 123) "       173")

(test (format #f "~d" -123) "-123")
(test (format #f "~x" -123) "-7b")
(test (format #f "~b" -123) "-1111011")
(test (format #f "~o" -123) "-173")

(test (format #f "~10d" -123) "      -123")
(test (format #f "~10x" -123) "       -7b")
(test (format #f "~10b" -123) "  -1111011")
(test (format #f "~10o" -123) "      -173")

(test (format #f "~D" most-positive-fixnum) "9223372036854775807")
(test (format #f "~D" (+ 1 most-negative-fixnum)) "-9223372036854775807")
      
(test (format #f "~X" most-positive-fixnum) "7fffffffffffffff")
(test (format #f "~X" (+ 1 most-negative-fixnum)) "-7fffffffffffffff")
      
(test (format #f "~O" most-positive-fixnum) "777777777777777777777")
(test (format #f "~O" (+ 1 most-negative-fixnum)) "-777777777777777777777")
      
(test (format #f "~B" most-positive-fixnum) "111111111111111111111111111111111111111111111111111111111111111")
(test (format #f "~B" (+ 1 most-negative-fixnum)) "-111111111111111111111111111111111111111111111111111111111111111")
      
(num-test (inexact->exact most-positive-fixnum) most-positive-fixnum)

(test (format #f "~0D" 123) "123")
(test (format #f "~0X" 123) "7b")
(test (format #f "~0B" 123) "1111011")
(test (format #f "~0O" 123) "173")

(test (format #f "" 1) 'error)
(test (format #f "hiho" 1) 'error)
(test (format #f "a~%" 1) 'error) ; some just ignore extra args

(for-each
 (lambda (arg)
   (let ((result (catch #t (lambda () (format arg "hiho")) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (begin (display "(format ") (display arg) (display " \"hiho\")")
		(display " returned ") (display result) 
		(display " but expected 'error")
		(newline)))))
 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i 'hi :hi #<eof> abs (lambda () 1) '#(()) (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (let ((result (catch #t (lambda () (format #f arg)) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (begin (display "(format #f ") (display arg) (display ")")
		(display " returned ") (display result) 
		(display " but expected 'error")
		(newline)))))
 (list -1 #\a 1 #f #t '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs (lambda () 1) '#(()) (list 1 2 3) '(1 . 2)))

(test (format #f "hi ~A ho" 1 2) 'error)
(test (format #f "hi ~A ho") 'error)
(test (format #f "hi ~S ho") 'error)
(test (format #f "hi ~S ho" 1 2) 'error)
(test (format #f "~C" 1) 'error)
(test (format #f "123 ~R 321" 1) 'error)
(test (format #f "123 ~,3R 321" 1) 'error)
(test (format #f "~,2,3,4D" 123) 'error)

(test (format #f "hi ~Z ho") 'error)
(test (format #f "hi ~+ ho") 'error)
(test (format #f "hi ~# ho") 'error)
(test (format #f "hi ~, ho") 'error)

(test (format #f "hi ~} ho") 'error)
(test (format #f "hi {ho~}") 'error)

(test (format #f "asb~{~A asd" '(1 2 3)) 'error)
(test (format #f "~{~A~}" 1 2 3) 'error)
(test (format #f "asb~{~}asd" '(1 2 3)) 'error)
(test (format #f "asb~{ ~}asd" '(1 2 3)) 'error)
(test (format #f "asb~{ . ~}asd" '(1 2 3)) 'error)
(test (format #f "asb~{ hiho~~~}asd" '(1 2 3)) 'error)

#|
(do ((i 0 (+ i 1))) ((= i 256)) 
  (let ((chr (integer->char i)))
    (format #t "~D: ~A ~A ~D~%" i (format #f "~S" (string chr)) (string chr) (length (format #f "~S" (string chr))))))
|#

(for-each
 (lambda (arg)
   (test (format #f "~F" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~D" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~P" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~X" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~C" arg) 'error))
 (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) abs #f #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f arg 123) 'error))
 (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) abs #f #t (if #f #f) (lambda (a) (+ a 1))))

(test (format #f "~{~A ~A ~}" '(1 "hi" 2)) 'error)
(for-each
 (lambda (arg)
   (let ((result (catch #t (lambda () (format #f "~F" arg)) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (begin (display "(format #f \"~F\" ") (display arg)
		(display ") returned ") (display result) 
		(display " but expected 'error")
		(newline)))))
 (list #\a '#(1 2 3) "hi" '() 'hi abs (lambda () 1) '#(()) (list 1 2 3) '(1 . 2)))

(test (format #f "~D") 'error)
(test (format () "hi") 'error)
(test (format #f "~F" "hi") 'error)
(test (format #f "~D" #\x) 'error)
(test (format #f "~C" (list 1 2 3)) 'error)
(test (format #f "~1/4F" 1.4) 'error)
(test (format #f "~1.4F" 1.4) 'error)
(test (format #f "~F" (real-part (log 0.0))) "-inf.0")
(test (let ((val (format #f "~F" (/ (real-part (log 0.0)) (real-part (log 0.0)))))) (or (string=? val "nan.0") (string=? val "-nan.0"))) #t)
(test (format #f "~1/4T~A" 1) 'error)
(test (format #f "~T") "")
(test (format #f "~@P~S" 1 '(1)) "y(1)")
(test (format #f ".~A~*" 1 '(1)) ".1")
(test (format #f "~*~*~T" 1 '(1)) "")

(test (format #f "~A" 'AB\c) "(symbol \"AB\\\\c\")")
(test (format #f "~S" 'AB\c) "(symbol \"AB\\\\c\")")
(test (format #f "~A" '(AB\c () xyz)) "((symbol \"AB\\\\c\") () xyz)")
(test (format #f "~,2f" 1234567.1234) "1234567.12")
(test (format #f "~5D" 3) "    3")
(test (format #f "~5,'0D" 3) "00003")
(test (format #f "++~{-=~s=-~}++" (quote (1 2 3))) "++-=1=--=2=--=3=-++")

(test (format) 'error)
(for-each
 (lambda (arg)
   (test (format arg) 'error))
 (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t (if #f #f) (lambda (a) (+ a 1))))
(test (format "hi") "hi") ; !?
(test (format "~A ~D" 1/3 2) "1/3 2")
(test (format "") "")

;; from slib/formatst.scm
(test (string=? (format #f "abc") "abc") #t)
(test (string=? (format #f "~a" 10) "10") #t)
(test (string=? (format #f "~a" -1.2) "-1.2") #t)
(test (string=? (format #f "~a" 'a) "a") #t)
(test (string=? (format #f "~a" #t) "#t") #t)
(test (string=? (format #f "~a" #f) "#f") #t)
(test (string=? (format #f "~a" "abc") "abc") #t)
(test (string=? (format #f "~a" '#(1 2 3)) "#(1 2 3)") #t)
(test (string=? (format #f "~a" '()) "()") #t)
(test (string=? (format #f "~a" '(a)) "(a)") #t)
(test (string=? (format #f "~a" '(a b)) "(a b)") #t)
(test (string=? (format #f "~a" '(a (b c) d)) "(a (b c) d)") #t)
(test (string=? (format #f "~a" '(a . b)) "(a . b)") #t)
(test (string=? (format #f "~a ~a" 10 20) "10 20") #t)
(test (string=? (format #f "~a abc ~a def" 10 20) "10 abc 20 def") #t)
(test (string=? (format #f "~d" 100) "100") #t)
(test (string=? (format #f "~x" 100) "64") #t)
(test (string=? (format #f "~o" 100) "144") #t)
(test (string=? (format #f "~b" 100) "1100100") #t)
(test (string=? (format #f "~10d" 100) "       100") #t)
(test (string=? (format #f "~10,'*d" 100) "*******100") #t)
(test (string=? (format #f "~c" #\a) "a") #t)
(test (string=? (format #f "~c" #\space) " ") #t)
(test (string=? (format #f "~C" #\x91) "\x91") #t)
(test (string=? (format #f "~C" #\x9) "\x09") #t)
(test (string=? (format #f "~C" #\~) "~") #t)
(test (string=? (format #f "~A" #\x91) "\x91") #t)
(test (string=? (format #f "~S" #\x91) "#\\x91") #t)
(test (string=? (format #f "~A" (string->symbol "hi")) "hi") #t)
(test (string=? (format #f "~S" (string->symbol "hi")) "hi") #t)
(test (string=? (format #f "~A" (string->symbol ";\\\";")) "(symbol \";\\\\\\\";\")") #t)
(test (string=? (format #f "~S" (string->symbol ";\\\";")) "(symbol \";\\\\\\\";\")") #t)
(test (string=? (format #f "~A" (string->symbol (string #\, #\. #\# #\; #\" #\\ #\' #\`))) "(symbol \",.#;\\\"\\\\'`\")") #t)

(test (string=? (format #f "~~~~") "~~") #t)
(test (string=? (format #f "~s" "abc") "\"abc\"") #t)
(test (string=? (format #f "~s" "abc \\ abc") "\"abc \\\\ abc\"") #t)
(test (string=? (format #f "~a" "abc \\ abc") "abc \\ abc") #t)
(test (string=? (format #f "~s" "abc \" abc") "\"abc \\\" abc\"") #t)
(test (string=? (format #f "~a" "abc \" abc") "abc \" abc") #t)
(test (string=? (format #f "~s" #\space) "#\\space") #t)
(test (string=? (format #f "~s" #\newline) "#\\newline") #t)
(test (string=? (format #f "~s" #\a) "#\\a") #t)
(test (string=? (format #f "~a" '(a "b" c)) "(a \"b\" c)") #t)
(test (string=? (format #f "abc~
         123") "abc123") #t)
(test (string=? (format #f "abc~
123") "abc123") #t)
(test (string=? (format #f "abc~
") "abc") #t)
(test (string=? (format #f "~{ ~a ~}" '(a b c)) " a  b  c ") #t)
(test (string=? (format #f "~{ ~a ~}" '()) "") #t)
(test (string=? (format #f "~{ ~a ~}" "") "") #t)
(test (string=? (format #f "~{ ~a ~}" #()) "") #t)
(test (string=? (format #f "~{ ~a,~a ~}" '(a 1 b 2 c 3)) " a,1  b,2  c,3 ") #t)
(test (string=? (format #f "abc ~^ xyz") "abc ") #t)
(test (format (values #f "~A ~D" 1 2)) "1 2")
(test (format #f "~A~^" 1) "1") ; clisp agrees here
(test (format #f "~A~*~* ~A" (values 1 2 3 4)) "1 4")
(test (format #f "~^~A~^~*~*~^ ~^~A~^" (values 1 2 3 4)) "1 4")

(test (string=? (format #f "~B" 123) "1111011") #t)
(test (string=? (format #f "~B" 123/25) "1111011/11001") #t)
(test (string=? (format #f "~B" 123.25) "1111011.01") #t)
(test (string=? (format #f "~B" 123+i) "1111011.0+1.0i") #t)

(test (string=? (format #f "~D" 123) "123") #t)
(test (string=? (format #f "~D" 123/25) "123/25") #t)

(test (string=? (format #f "~O" 123) "173") #t)
(test (string=? (format #f "~O" 123/25) "173/31") #t)
(test (string=? (format #f "~O" 123.25) "173.2") #t)
(test (string=? (format #f "~O" 123+i) "173.0+1.0i") #t)

(test (string=? (format #f "~X" 123) "7b") #t)
(test (string=? (format #f "~X" 123/25) "7b/19") #t)
(test (string=? (format #f "~X" 123.25) "7b.4") #t)
(test (string=? (format #f "~X" 123+i) "7b.0+1.0i") #t)

(test (string=? (format #f "~A" "hi") (format #f "~S" "hi")) #f)
(test (string=? (format #f "~A" #\a) (format #f "~S" #\a)) #f)
(for-each
 (lambda (arg)
   (test (string=? (format #f "~A" arg) (format #f "~S" arg)) #t))
 (list 1 1.0 #(1 2 3) '(1 2 3) '(1 . 2) '() #f #t abs #<eof> #<unspecified> 'hi '\a))
(test (length (format #f "~S" (string #\\))) 4)                  ; "\"\\\\\""
(test (length (format #f "~S" (string #\a))) 3)                  ; "\"a\""
(test (length (format #f "~S" (string #\null))) 6)               ; "\"\\x00\""
(test (length (format #f "~S" (string (integer->char #xf0)))) 3) ; "\"ð\""
(test (length (format #f "~S" (string #\"))) 4)                  ; "\""

(test (format #f "~F" 3.0) "3.000000")
(test (format #f "~G" 3.0) "3.0")
(test (format #f "~E" 3.0) "3.000000e+00")
(test (format #f "~F" 3.14159) "3.141590")
(test (format #f "~G" 3.14159) "3.14159")
(test (format #f "~E" 3.14159) "3.141590e+00")
(test (format #f "~,2F" 3.14159) "3.14")
(test (format #f "~,2G" 3.14159) "3.1")
(test (format #f "~,2E" 3.14159) "3.14e+00")
(test (format #f "~12F" 3.14159) "    3.141590")
(test (format #f "~12G" 3.14159) "     3.14159")
(test (format #f "~12E" 3.14159) "3.141590e+00")
(test (format #f "~12,3F" 3.14159) "       3.142")
(test (format #f "~12,3G" 3.14159) "        3.14")
(test (format #f "~12,3E" 3.14159) "   3.142e+00")
(test (format #f "~12,'xD" 1) "xxxxxxxxxxx1")
(test (format #f "~12,'xF" 3.14159) "xxxx3.141590")
(test (format #f "~12,'xG" 3.14159) "xxxxx3.14159")
(test (format #f "~12,'xE" 3.14159) "3.141590e+00")
(test (format #f "~12,'\\F" 3.14159) "\\\\\\\\3.141590")
(test (format #f "~20,20G" 3.0) "                   3.0")
(test (format #f "~20,20F" 3.0) "3.00000000000000000000")
(test (format #f "~20,20E" 3.0) "3.00000000000000000000e+00")

(test (format #f "~,3B" 0.99999) "0.111")
(test (format #f "~,3O" 0.99999) "0.777")
(test (format #f "~,3F" 0.99999) "1.000")
(test (format #f "~,3X" 0.99999) "0.fff")

(test (format #f "~-2F" 0.0) 'error)
(test (format #f "~,-2F" 0.0) 'error)
(test (format #f "~2/3F" 0.0) 'error)
(test (format #f "~2.3F" 0.0) 'error)
(test (format #f "~2,1,3,4F" 0.0) 'error)
(test (format #f "~'xF" 0.0) 'error)
(test (format #f "~3,3" pi) 'error)
(test (format #f "~3," pi) 'error)
(test (format #f "~3" pi) 'error)
(test (format #f "~," pi) 'error)
(test (format #f "~'," pi) 'error)
(test (format #f "~'" pi) 'error)

(test (format #f "~*" 1.0) "")
(test (format #f "~D" 1.0) "1.000000e+00")
(test (format #f "~O" 1.0) "1.0")
(test (format #f "~P" 1.0) "")
(test (format #f "~P" '(1 2 3)) 'error)
(test (format #f "~\x00T") 'error)
(test (format #f "~9,'(T") "((((((((")
(test (format #f "~0F" 1+1i) "1.000000+1.000000i")
(test (format #f "~9F" 1) "        1")
(test (format #f "~,0F" 3.14) "3.0")
(test (format #f "~,0F" 1+1i) "1+1i")
(test (format #f "~,0X" 1+1i) "1.0+1.0i")
(test (format #f "~,9g" 1+1i) "1+1i")
(test (format #f "~,1e" 3.14) "3.1e+00")
(test (format #f "~9,0F" 3.14) "        3.0")
(test (format #f "~9,1F" 3.14) "      3.1")
(test (format #f "~9,2F" 3.14) "     3.14")
(test (format #f "~9,3F" 3.14) "    3.140")
(test (format #f "~9,4F" 3.14) "   3.1400")
(test (format #f "~9,5F" 3.14) "  3.14000")
(test (format #f "~9,6F" 3.14) " 3.140000")
(test (format #f "~9,7F" 3.14) "3.1400000")
(test (format #f "~9,8F" 3.14) "3.14000000")
(test (format #f "~9,9F" 3.14) "3.140000000")
(test (format #f "~9,9G" 1+1i) "     1+1i")
(test (format #f "~9,0e" 1+1i) "1e+00+1e+00i")
(test (format #f "~9,1e" 1+1i) "1.0e+00+1.0e+00i")
(test (format #f "~9,2e" 1+1i) "1.00e+00+1.00e+00i")
(test (format #f "~9,3e" 1+1i) "1.000e+00+1.000e+00i")
(test (format #f "~9,4e" 1+1i) "1.0000e+00+1.0000e+00i")
(test (format #f "~9,5e" 1+1i) "1.00000e+00+1.00000e+00i")
(test (format #f "~9,6e" 1+1i) "1.000000e+00+1.000000e+00i")
(test (format #f "~9,7e" 1+1i) "1.0000000e+00+1.0000000e+00i")
(test (format #f "~9,8e" 1+1i) "1.00000000e+00+1.00000000e+00i")
(test (format #f "~9,9e" 1+1i) "1.000000000e+00+1.000000000e+00i")
(test (format #f "~9,0x" 3.14) "      3.0")
(test (format #f "~9,1x" 3.14) "      3.2")
(test (format #f "~9,2x" 3.14) "     3.23")
(test (format #f "~9,3x" 3.14) "    3.23d")
(test (format #f "~9,4x" 3.14) "   3.23d7")
(test (format #f "~9,5x" 3.14) "   3.23d7")
(test (format #f "~9,6x" 3.14) " 3.23d70a")
(test (format #f "~9,7x" 3.14) "3.23d70a3")
(test (format #f "~9,8x" 3.14) "3.23d70a3d")
(test (format #f "~9,9x" 3.14) "3.23d70a3d7")
(test (format #f "~9,0b" 3.14) "     11.0")
(test (format #f "~9,1b" 3.14) "     11.0")
(test (format #f "~9,2b" 3.14) "     11.0")
(test (format #f "~9,3b" 3.14) "   11.001")
(test (format #f "~9,4b" 3.14) "   11.001")
(test (format #f "~9,5b" 3.14) "   11.001")
(test (format #f "~9,6b" 3.14) "   11.001")
(test (format #f "~9,7b" 3.14) "11.0010001")
(test (format #f "~9,8b" 3.14) "11.00100011")
(test (format #f "~9,9b" 3.14) "11.001000111")
(test (format #f "~0,'xf" 1) "1")
(test (format #f "~1,'xf" 1) "1")
(test (format #f "~2,'xf" 1) "x1")
(test (format #f "~3,'xf" 1) "xx1")
(test (format #f "~4,'xf" 1) "xxx1")
(test (format #f "~5,'xf" 1) "xxxx1")
(test (format #f "~6,'xf" 1) "xxxxx1")
(test (format #f "~7,'xf" 1) "xxxxxx1")
(test (format #f "~8,'xf" 1) "xxxxxxx1")
(test (format #f "~9,'xf" 1) "xxxxxxxx1")
(test (format #f "~11,'xf" 3.14) "xxx3.140000")
(test (format #f "~12,'xf" 3.14) "xxxx3.140000")
(test (format #f "~13,'xf" 3.14) "xxxxx3.140000")
(test (format #f "~14,'xf" 3.14) "xxxxxx3.140000")
(test (format #f "~15,'xf" 3.14) "xxxxxxx3.140000")
(test (format #f "~16,'xf" 3.14) "xxxxxxxx3.140000")
(test (format #f "~17,'xf" 3.14) "xxxxxxxxx3.140000")
(test (format #f "~18,'xf" 3.14) "xxxxxxxxxx3.140000")
(test (format #f "~19,'xf" 3.14) "xxxxxxxxxxx3.140000")
(test (format #f "~,f" 1.0) "1.000000")
(test (format #f "~,,f" 1.0) 'error)
(test (format #f "~p" '(1 2 3)) 'error) ; these are not errors in CL
(test (format #f "~p" #(())) 'error)
(test (format #f "~p" 'hi) 'error)
(test (format #f "~p" abs) 'error)
(test (format #f "~p" 1+i) 'error)
(test (format #f "~@p" '(1 2 3)) 'error)
(test (format #f "~@p" #(())) 'error)
(test (format #f "~@p" 'hi) 'error)
(test (format #f "~@p" abs) 'error)
(test (format #f "~{~{~A~^~} ~}" '((hi 1))) "hi1 ")
(test (format #f "~{~{~A~^~} ~}" '((1 2) (3 4))) "12 34 ")
(test (format #f "~{~{~A~} ~}" '((1 2) (3 4))) "12 34 ")
(test (format #f "~{~{~A~} ~}" '(())) " ")
(test (format #f "~{~{~A~} ~}" '((()))) "() ")
(test (format #f "~{~{~F~} ~}" '(())) " ")
(test (format #f "~{~{~C~} ~}" '(())) " ")
(test (format #f "~{~C ~}" '()) "")
(test (format #f "~C ~^" #\a) "a ") ; CL ignores pointless ~^
(test (format #f "~^~A" #f) "#f")
(test (format #f "~^~^~A" #f) "#f")
(test (format #f "~*~*~A~*" 1 2 3 4) "3")
(test (format #f "~{~*~A~}" '(1 2 3 4)) "24")
(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (format #f "~A" lst)) "#1=(1 2 3 . #1#)")
(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (format #f "~{~A~}" lst)) 'error)
(test (format #f "~{~A~}" (cons 1 2)) 'error)
(test (format #f "~{~A~}" '(1 2 3 . 4)) 'error)
(test (format #f "~20,vF" 3.14) 'error)
(test (format #f "~{~C~^ ~}" "hiho") "h i h o")
(test (format #f "~{~A ~}" #(1 2 3 4)) "1 2 3 4 ")
(test (let ((v (vector 1))) (set! (v 0) v) (format #f "~A" v)) "#1=#(#1#)")
(test (let ((v (vector 1))) (set! (v 0) v) (format #f "~{~A~}" v)) "#1=#(#1#)")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" '(((1 2) (3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" '((#(1 2) #(3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" #(((1 2) (3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" #(#((1 2) (3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" #(#(#(1 2) (3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" #(#(#(1 2) #(3 4)))) "1 2 3 4")
(test (format #f "~{~{~C~^ ~}~^ ~}" (list "hiho" "xxx")) "h i h o x x x")
(test (format #f "~{~{~A~}~}" '((1 . 2) (3 . 4))) 'error)
(test (format #f "~{~A~^ ~}" '((1 . 2) (3 . 4))) "(1 . 2) (3 . 4)") 
(test (format #f "~{~A ~}" (hash-table '(a . 1) '(b . 2))) "(b . 2) (a . 1) ")
(test (format #f "~{~A ~}" (hash-table)) "")

(let ((ctr ((cadr (make-type :getter (lambda (a b) b) :length (lambda (a) 4))))))
  (test (format #f "~{~A~^ ->~}" ctr) "0 ->1 ->2 ->3"))
(let ((ctr ((cadr (make-type :getter (lambda (a b) (+ b 3)) :length (lambda (a) 4))))))
  (test (format #f "~{~A~^ ->~}" ctr) "3 ->4 ->5 ->6"))

(test (format #f "~{ ~,-tF ~}" '()) "") ; hmm -- it's ignoring the unimplemented format directive -- should we add an error check?
(let ((ctr ((cadr (make-type :getter (lambda (a b) (car b)) :length (lambda (a) 4))))))
  (test (format #f "~{~A~^ ->~}" ctr) 'error))
(let ((ctr ((cadr (make-type :getter (lambda (a b) (+ b 1)) :length (lambda (a) 'hi))))))
  (test (format #f "~{~A~^ ->~}" ctr) 'error))

(test (format #f "~10,'-T") "---------")
(test (format #f "~10,'\\T") "\\\\\\\\\\\\\\\\\\")
(test (format #f "~10,'\"T") "\"\"\"\"\"\"\"\"\"")
(test (format #f "~10,'-T12345~20,'-T") "---------12345-----")
(test (format #f "~10,')T") ")))))))))")

(test (format #f "~,0F" 1.4) "1.0")
(test (format #f "~,0F" 1.5) "2.0")
(test (format #f "~,0F" 1.6) "2.0")
(test (format #f "~,0F" 0.4) "0.0")
(test (format #f "~,0F" 0.5) "0.0")
(test (format #f "~,0F" 0.6) "1.0")
(test (format #f "~,-0F" 1.4) 'error)
(test (format #f "~, 0F" 1.4) 'error)
(test (format #f "~*1~*" 1) 'error)
(test (format #f "~*1~A" 1) 'error)

(test (reverse (format #f "~{~A~}" '((1 2) (3 4)))) ")4 3()2 1(")
(test (string->symbol (format #f "~A" '(1 2))) (symbol "(1 2)"))

(test (string->number (format #f "~A" -1)) -1)
(test (string->number (format #f "~S" -1)) -1)
(test (string->number (format #f "~F" -1)) -1)
(test (string->number (format #f "~D" -1)) -1)
(test (string->number (format #f "~G" -1)) -1)
(test (string->number (format #f "~E" -1)) -1)
(test (string->number (format #f "~B" -1)) -1)
(test (string->number (format #f "~X" -1)) -1)
(test (string->number (format #f "~O" -1)) -1)
(num-test (string->number (format #f "~A" 1.5)) 1.5)
(num-test (string->number (format #f "~S" 1.5)) 1.5)
(num-test (string->number (format #f "~F" 1.5)) 1.5)
(num-test (string->number (format #f "~D" 1.5)) 1.5)
(num-test (string->number (format #f "~G" 1.5)) 1.5)
(num-test (string->number (format #f "~E" 1.5)) 1.5)
(num-test (string->number (format #f "~B" 1.5)) 1.1)
(num-test (string->number (format #f "~X" 1.5)) 1.8)
(num-test (string->number (format #f "~O" 1.5)) 1.4)
(num-test (string->number (format #f "~A" 1+1i)) 1+1i)
(num-test (string->number (format #f "~S" 1+1i)) 1+1i)
(num-test (string->number (format #f "~F" 1+1i)) 1+1i)
(num-test (string->number (format #f "~D" 1+1i)) 1+1i)
(num-test (string->number (format #f "~G" 1+1i)) 1+1i)
(num-test (string->number (format #f "~E" 1+1i)) 1+1i)
(num-test (string->number (format #f "~B" 1+1i)) 1+1i)
(num-test (string->number (format #f "~X" 1+1i)) 1+1i)
(num-test (string->number (format #f "~O" 1+1i)) 1+1i)
(test (string->number (format #f "~A" 3/4)) 3/4)
(test (string->number (format #f "~S" 3/4)) 3/4)
(test (string->number (format #f "~F" 3/4)) 3/4)
(test (string->number (format #f "~D" 3/4)) 3/4)
(test (string->number (format #f "~G" 3/4)) 3/4)
(test (string->number (format #f "~E" 3/4)) 3/4)
(test (string->number (format #f "~B" 3/4)) 11/100)
(test (string->number (format #f "~X" 3/4)) 3/4)
(test (string->number (format #f "~O" 3/4)) 3/4)
(num-test (string->number (format #f "~A" 0+1i)) 0+1i)
(num-test (string->number (format #f "~S" 0+1i)) 0+1i)
(num-test (string->number (format #f "~F" 0+1i)) 0+1i)
(num-test (string->number (format #f "~D" 0+1i)) 0+1i)
(num-test (string->number (format #f "~G" 0+1i)) 0+1i)
(num-test (string->number (format #f "~E" 0+1i)) 0+1i)
(num-test (string->number (format #f "~B" 0+1i)) 0+1i)
(num-test (string->number (format #f "~X" 0+1i)) 0+1i)
(num-test (string->number (format #f "~O" 0+1i)) 0+1i)

(test (format "~G" 1e10) "1e+10")
(test (format "~F" 1e10) "10000000000.000000")
(test (format "~E" 1e10) "1.000000e+10")
(test (format "~A" 1e10) "10000000000.0")
(test (format "~D" 1e10) "1.000000e+10")

(test (format #f "~P{T}'" 1) "{T}'")
(test (format #f "~") 'error)
(test (format #f "~B&B~X" 1.5 1.5) "1.1&B1.8")
(test (format #f ",~~~A~*1" 1 1) ",~11")
(test (format #f "~D~20B" 0 0) "0                   0")
(test (format #f "~D~20B" 1 1) "1                   1")
(test (format #f "~10B" 1) "         1")
(test (format #f "~10B" 0) "         0")
(test (format #f "~100B" 1) "                                                                                                   1")
(test (length (format #f "~1000B" 1)) 1000)
(test (format #f "~D~20D" 3/4 3/4) "3/4                 3/4")
(test (length (format #f "~20D" 3/4)) 20)
(test (format #f "~20B" 3/4) "              11/100")
(test (length (format #f "~20B" 3/4)) 20)
(test (format #f "~D~20B" 3/4 3/4) "3/4              11/100")
(test (format #f "~X~20X" 21/33 21/33) "7/b                 7/b")
(test (format #f "~D~20,'.B" 3/4 3/4) "3/4..............11/100")
(test (format #f "~20g" 1+i) "                1+1i")
(test (length (format #f "~20g" 1+i)) 20)
(test (format #f "~20f" 1+i) "  1.000000+1.000000i")
(test (length (format #f "~20f" 1+i)) 20)
(test (format #f "~20x" 17+23i) "          11.0+17.0i")
(test (length (format #f "~20x" 17+23i)) 20)

(let ()
  (define* (clean-string e (precision 3))
    (format #f (format #f "(~~{~~,~DF~~^ ~~})" precision) e))
  (test (clean-string '(1.123123 -2.31231323 3.141592653589 4/3) 1) "(1.1 -2.3 3.1 4/3)")
  (test (clean-string '(1.123123 -2.31231323 3.141592653589 4/3)) "(1.123 -2.312 3.142 4/3)")
  (test (clean-string '(1.123123 -2.31231323 3.141592653589 4/3) 6) "(1.123123 -2.312313 3.141593 4/3)"))

(if with-bignums
    (begin
      (test (format #f "~P" (bignum "1")) "s")
      (test (format #f "~P" (bignum "1.0")) "s")
      (test (format #f "~10,' D" (bignum "1")) "         1")
      (test (format #f "~10,' D" (bignum "3/4")) "       3/4")
      (test (format #f "~10,'.D" (bignum "3/4")) ".......3/4")
      (test (format #f "~10D" (bignum "3/4")) "       3/4")
      (test (length (format #f "~100D" (bignum "34"))) 100)
      (test (format #f "~50F" (bignum "12345678.7654321")) "                                1.23456787654321E7")
      ))


(call-with-output-file "tmp1.r5rs" (lambda (p) (format p "this ~A ~C test ~D" "is" #\a 3)))
(let ((res (call-with-input-file "tmp1.r5rs" (lambda (p) (read-line p)))))
  (if (not (string=? res "this is a test 3"))
      (begin 
	(display "call-with-input-file + format to \"tmp1.r5rs\" ... expected \"this is a test 3\", but got \"")
	(display res) (display "\"?") (newline))))

(let ((val (format #f "line 1~%line 2~%line 3")))
  (with-input-from-string val
    (lambda ()
      (let ((line1 (read-line)))
	(test (string=? line1 "line 1") #t))
      (let ((line2 (read-line)))
	(test (string=? line2 "line 2") #t))
      (let ((line3 (read-line)))
	(test (string=? line3 "line 3") #t))
      (let ((eof (read-line)))
	(test (eof-object? eof) #t))
      (let ((eof (read-line)))
	(test (eof-object? eof) #t)))))


(let ((val (format #f "line 1~%line 2~%line 3")))
  (call-with-input-string val
			  (lambda (p)
			    (let ((line1 (read-line p #t)))
			      (test (string=? line1 (string-append "line 1" (string #\newline))) #t))
			    (let ((line2 (read-line p #t)))
			      (test (string=? line2 (string-append "line 2" (string #\newline))) #t))
			    (let ((line3 (read-line p #t)))
			      (test (string=? line3 "line 3") #t))
			    (let ((eof (read-line p #t)))
			      (test (eof-object? eof) #t))
			    (let ((eof (read-line p #t)))
			      (test (eof-object? eof) #t)))))

(let ((res #f)) 
  (let ((this-file (open-output-string))) 
    (format this-file "this ~A ~C test ~D" "is" #\a 3)
    (set! res (get-output-string this-file))
    (close-output-port this-file))
  (if (not (string=? res "this is a test 3"))
      (begin 
	(display "open-output-string + format ... expected \"this is a test 3\", but got \"")
	(display res) (display "\"?") (newline))))

(let ((res1 #f)
      (res2 #f)
      (res3 #f))
  (let ((p1 (open-output-string)))
    (format p1 "~D" 0)
    (let ((p2 (open-output-string)))
      (format p2 "~D" 1)
      (let ((p3 (open-output-string)))
	(if (not (string=? (get-output-string p1) "0"))
	    (format #t ";format to nested ports, p1: ~S~%" (get-output-string p1)))	
	(if (not (string=? (get-output-string p2) "1"))
	    (format #t ";format to nested ports, p2: ~S~%" (get-output-string p2)))	
	(format p3 "~D" 2)
	(format p2 "~D" 3)
	(format p1 "~D" 4)
	(format p3 "~D" 5)
	(set! res3 (get-output-string p3))
	(close-output-port p3)
	(if (not (string=? (get-output-string p1) "04"))
	    (format #t ";format to nested ports after close, p1: ~S~%" (get-output-string p1)))	
	(if (not (string=? (get-output-string p2) "13"))
	    (format #t ";format to nested ports after close, p2: ~S~%" (get-output-string p2))))
      (format (or p1 p3) "~D" 6)
      (format (and p1 p2) "~D" 7)
      (set! res1 (get-output-string p1))
      (close-output-port p1)
      (if (not (string=? (get-output-string p2) "137"))
	  (format #t ";format to nested ports after 2nd close, p2: ~S~%" (get-output-string p2)))
      (format p2 "~D" 8)
      (set! res2 (get-output-string p2))
      (test (get-output-string p1) 'error)
      (close-output-port p2)))
  (if (not (string=? res1 "046"))
      (format #t ";format to nested ports, res1: ~S~%" res1))
  (if (not (string=? res2 "1378"))
      (format #t ";format to nested ports, res2: ~S~%" res2))
  (if (not (string=? res3 "25"))
      (format #t ";format to nested ports, res3: ~S~%" res3)))

(test (call/cc (lambda (return) 
		 (let ((val (format #f "line 1~%line 2~%line 3")))
		   (call-with-input-string val
					   (lambda (p) (return "oops"))))))
      "oops")

(format #t "format #t: ~D" 1)
(format (current-output-port) " output-port: ~D! (this is testing output ports)~%" 2)

(call-with-output-file "tmp1.r5rs"
  (lambda (p)
    (display 1 p)
    (write 2 p)
    (write-char #\3 p)
    (format p "~D" 4)
    (write-byte (char->integer #\5) p)
    (call-with-output-file "tmp2.r5rs"
      (lambda (p)
	(display 6 p)
	(write 7 p)
	(write-char #\8 p)
	(format p "~D" 9)
	(write-byte (char->integer #\0) p)
	(newline p)))
    (call-with-input-file "tmp2.r5rs"
      (lambda (pin)
	(display (read-line pin) p)))
    (newline p)))

(test (call-with-input-file "tmp1.r5rs"
	(lambda (p)
	  (read-line p)))
      "1234567890")

(call-with-output-file "tmp1.r5rs"
  (lambda (p)
    (format p "12345~%")
    (format p "67890~%")))

(call-with-input-file "tmp1.r5rs"
  (lambda (p)
    (test (read-char p) #\1)
    (test (read-byte p) (char->integer #\2))
    (test (peek-char p) #\3)
    (test (char-ready? p) #t)
    (test (read-line p) "345")
    (test (read-line p) "67890")))

(let ((op1 (set-current-output-port (open-output-file "tmp1.r5rs"))))
  (display 1)
  (write 2)
  (write-char #\3)
  (format #t "~D" 4) ; #t -> output port
  (write-byte (char->integer #\5))
  (let ((op2 (set-current-output-port (open-output-file "tmp2.r5rs"))))
    (display 6)
    (write 7)
    (write-char #\8)
    (format #t "~D" 9)
    (write-byte (char->integer #\0))
    (newline)
    (close-output-port (current-output-port))
    (set-current-output-port op2)
    (let ((ip1 (set-current-input-port (open-input-file "tmp2.r5rs"))))
      (display (read-line))
      (close-input-port (current-input-port))
      (set-current-input-port ip1))
    (newline)
    (close-output-port (current-output-port))
    (set-current-output-port op1)))

(let ((old-op1 (current-output-port))
      (op1 (open-output-file "tmp1.r5rs")))
  (set! (current-output-port) op1)
  (display 1)
  (write 2)
  (write-char #\3)
  (format #t "~D" 4) ; #t -> output port
  (write-byte (char->integer #\5))
  (let ((old-op2 (current-output-port))
	(op2 (open-output-file "tmp2.r5rs")))
    (set! (current-output-port) op2)
    (display 6)
    (write 7)
    (write-char #\8)
    (format #t "~D" 9)
    (write-byte (char->integer #\0))
    (newline)
    (close-output-port (current-output-port))
    (set! (current-output-port) old-op2)
    (let ((old-ip1 (current-input-port))
	  (ip1 (open-input-file "tmp2.r5rs")))
      (set! (current-input-port) ip1)
      (display (read-line))
      (close-input-port (current-input-port))
      (set! (current-input-port) old-ip1))
    (newline)
    (close-output-port (current-output-port))
    (set! (current-output-port) old-op1)))

(test (call-with-input-file "tmp1.r5rs"
	(lambda (p)
	  (read-line p)))
      "1234567890")

(for-each 
 (lambda (op)
   (for-each
    (lambda (arg)
      (test (op arg display) 'error))
    (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t (if #f #f) (lambda (a) (+ a 1)))))
 (list call-with-output-file call-with-input-file
       call-with-output-string call-with-input-string
       with-input-from-string with-input-from-file
       with-output-to-file))

(for-each 
 (lambda (op)
   (for-each
    (lambda (arg)
      (test (op arg) 'error))
    (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t (if #f #f) (lambda (a) (+ a 1)))))
 (list open-output-file open-input-file 
       open-input-string))

(for-each
 (lambda (op)
   (for-each 
    (lambda (arg)
      (test (op "hi" arg) 'error))
    (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t (if #f #f) (lambda (a) (+ a 1)))))
 (list write display write-byte newline write-char 
       read read-char read-byte peek-char char-ready? read-line))

(for-each 
 (lambda (arg)
   (test (write-char arg) 'error)
   (test (write-byte arg) 'error)
   (test (read-char arg) 'error)
   (test (read-byte arg) 'error)
   (test (peek-char arg) 'error))
 (list "hi" 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (op)
   (for-each
    (lambda (arg)
      (test (op arg) 'error))
    (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs (if #f #f) (lambda (a) (+ a 1)))))
 (list set-current-input-port set-current-error-port set-current-output-port close-input-port close-output-port))

(let ((hi (open-output-string)))
  (test (get-output-string hi) "")
  (close-output-port hi)
  (test (get-output-string hi) 'error))

(test (open-output-string "hiho") 'error)
(test (with-output-to-string "hi") 'error)
(test (call-with-output-string "hi") 'error)

(test (get-output-string 1 2) 'error)
(test (get-output-string) 'error)
(for-each 
 (lambda (arg)
   (test (get-output-string arg) 'error))
 (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs (if #f #f) (lambda (a) (+ a 1))))

;; since read of closed port will generate garbage, it needs to be an error,
;;   so I guess write of closed port should also be an error

(let ((hi (open-output-string)))
  (close-output-port hi)
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'closed-port))
   (list (lambda (p) (display 1 p))
	 (lambda (p) (write 1 p))
	 (lambda (p) (write-char #\a p))
	 (lambda (p) (write-byte 0 p))
	 (lambda (p) (format p "hiho"))
	 set-current-output-port
	 set-current-input-port
	 set-current-error-port
	 newline)))

(let ((hi (open-input-string "hiho")))
  (test (get-output-string hi) 'error)
  (close-input-port hi)
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'closed-port))
   (list read read-char read-byte peek-char char-ready? read-line 
	 port-filename port-line-number
	 set-current-output-port
	 set-current-input-port
	 set-current-error-port
	 )))
  
(test (close-output-port (open-input-string "hiho")) 'error)
(test (close-input-port (open-output-string)) 'error)
(test (set! (port-filename) "hiho") 'error)
(test (set! (port-closed (current-output-port)) "hiho") 'error)

(let* ((new-error-port (open-output-string))
       (old-error-port (set-current-error-port new-error-port)))
  (catch #t
	 (lambda ()
	   (format #f "~R" 123))
	 (lambda args
	   (format (current-error-port) "oops")))
  (let ((str (get-output-string new-error-port)))
    (set-current-error-port old-error-port)
    (test str "oops")))


(let ((hi (open-input-string "hiho")))
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'input-port))
   (list (lambda (p) (display 1 p))
	 (lambda (p) (write 1 p))
	 (lambda (p) (write-char #\a p))
	 (lambda (p) (write-byte 0 p))
	 (lambda (p) (format p "hiho"))
	 newline))
  (close-input-port hi))

(let ((hi (open-output-string)))
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'output-port))
   (list read read-char read-byte peek-char char-ready? read-line))
  (close-output-port hi))

(test (output-port? (current-error-port)) #t)
(test (and (not (null? (current-error-port))) (input-port? (current-error-port))) #f)

(call-with-output-file "tmp1.r5rs"
  (lambda (p)
    (test (get-output-string p) 'error)
    (do ((i 0 (+ i 1)))
	((= i 256))
      (write-byte i p))))

(call-with-input-file "tmp1.r5rs"
  (lambda (p)
    (test (get-output-string p) 'error)
    (do ((i 0 (+ i 1)))
	((= i 256))
      (let ((b (read-byte p)))
	(if (not (= b i))
	    (format #t "read-byte got ~A, expected ~A~%" b i))))
    (let ((eof (read-byte p)))
      (if (not (eof-object? eof))
	  (format #t "read-byte at end: ~A~%" eof)))
    (let ((eof (read-byte p)))
      (if (not (eof-object? eof))
	  (format #t "read-byte at end: ~A~%" eof)))))

(call-with-output-file "tmp1.r5rs"
  (lambda (p)
    (do ((i 0 (+ i 1)))
	((= i 256))
      (write-char (integer->char i) p))))

(define our-eof #f)

(call-with-input-file "tmp1.r5rs"
  (lambda (p)
    (do ((i 0 (+ i 1)))
	((= i 256))
      (let ((b (read-char p)))
	(if (or (not (char? b))
		(not (char=? b (integer->char i))))
	    (format #t "read-char got ~A, expected ~A (~D: char? ~A)~%" b (integer->char i) i (char? (integer->char i))))))
    (let ((eof (read-char p)))
      (if (not (eof-object? eof))
	  (format #t "read-char at end: ~A~%" eof))
      (set! our-eof eof))
    (let ((eof (read-char p)))
      (if (not (eof-object? eof))
	  (format #t "read-char again at end: ~A~%" eof)))))

(test (eof-object? (integer->char 255)) #f)
(test (eof-object? our-eof) #t)
(test (char->integer our-eof) 'error)
(test (char? our-eof) #f)
(test (eof-object? ((lambda () our-eof))) #t)

(for-each
 (lambda (op)
   (test (op *stdout*) 'error)
   (test (op *stderr*) 'error)
   (test (op (current-output-port)) 'error)
   (test (op (current-error-port)) 'error)
   (test (op '()) 'error))
 (list read read-line read-char read-byte peek-char char-ready?))

(for-each
 (lambda (op)
   (test (op #\a *stdin*) 'error)
   (test (op #\a (current-input-port)) 'error)
   (test (op #\a '()) 'error))
 (list write display write-char))
	 
(test (write-byte 0 *stdin*) 'error)
(test (newline *stdin*) 'error)
(test (format *stdin* "hiho") 'error)

(test (port-filename *stdin*) "*stdin*")	 
(test (port-filename *stdout*) "*stdout*")	 
(test (port-filename *stderr*) "*stderr*")	

(test (input-port? *stdin*) #t) 
(test (output-port? *stdin*) #f) 
(test (port-closed? *stdin*) #f)
(test (input-port? *stdout*) #f) 
(test (output-port? *stdout*) #t) 
(test (port-closed? *stdout*) #f)
(test (input-port? *stderr*) #f) 
(test (output-port? *stderr*) #t) 
(test (port-closed? *stderr*) #f)

(test (port-line-number *stdin*) 0)
(test (port-line-number *stdout*) 'error)
(test (port-line-number *stderr*) 'error)

(test (open-input-file "[*not-a-file!*]-") 'error)
(test (call-with-input-file "[*not-a-file!*]-" (lambda (p) p)) 'error)
(test (with-input-from-file "[*not-a-file!*]-" (lambda () #f)) 'error)

(test (open-input-file "") 'error)
(test (call-with-input-file "" (lambda (p) p)) 'error)
(test (with-input-from-file "" (lambda () #f)) 'error)

;(test (open-output-file "/bad-dir/badness/[*not-a-file!*]-") 'error)
;(test (call-with-output-file "/bad-dir/badness/[*not-a-file!*]-" (lambda (p) p)) 'error)
;(test (with-output-to-file "/bad-dir/badness/[*not-a-file!*]-" (lambda () #f)) 'error)

(with-output-to-file "tmp.r5rs"
  (lambda ()
    (write-char #\a)
    (with-output-to-file "tmp1.r5rs"
      (lambda ()
	(format #t "~C" #\b)
	(with-output-to-file "tmp2.r5rs"
	  (lambda ()
	    (display #\c)))
	(display (with-input-from-file "tmp2.r5rs"
		   (lambda ()
		     (read-char))))))
    (with-input-from-file "tmp1.r5rs"
      (lambda ()
	(write-byte (read-byte))
	(write-char (read-char))))))

(with-input-from-file "tmp.r5rs"
  (lambda ()
    (test (read-line) "abc")))

(with-input-from-file "tmp.r5rs" ; this assumes tmp.r5rs has "abc" as above
  (lambda ()
    (test (read-char) #\a)
    (test (eval-string "(+ 1 2)") 3)
    (test (read-char) #\b)
    (with-input-from-string "(+ 3 4)"
      (lambda ()
	(test (read) '(+ 3 4))))
    (test (read-char) #\c)))

(test (eval-string (object->string (with-input-from-string "(+ 1 2)" (lambda () (read))))) 3)
(test (eval (eval-string "(with-input-from-string \"(+ 1 2)\" (lambda () (read)))")) 3)
(test (eval-string "(eval (with-input-from-string \"(+ 1 2)\" (lambda () (read))))") 3)
(test (eval-string (object->string (eval-string (format #f "(+ 1 2)")))) 3)

;;; -------- test that we can plow past errors --------

(if (and (defined? 'file-exists?) ; (ifdef name ...)?
	 (file-exists? "tests.data"))
    (delete-file "tests.data"))

(call-with-output-file "tests.data"
  (lambda (p)
    (format p "start ")
    (catch #t 
      (lambda () 
	(format p "next ") (abs "hi") (format p "oops "))
      (lambda args
	'error))
    (format p "done\n")))

(let ((str (call-with-input-file "tests.data" 
             (lambda (p) 
	       (read-line p)))))
  (if (or (not (string? str))
	  (not (string=? str "start next done")))
      (format #t ";call-with-output-file + error -> ~S~%" str)))

(let ((str (call-with-input-file "tests.data" 
             (lambda (p) 
	       (catch #t
		      (lambda ()
			(read-char p)
			(abs "hi")
			(read-char p))
		      (lambda args "s"))))))
  (if (or (not (string? str))
	  (not (string=? str "s")))
      (format #t ";call-with-input-file + error -> ~S~%" str)))


(if (and (defined? 'file-exists?)
	 (file-exists? "tests.data"))
    (delete-file "tests.data"))

(with-output-to-file "tests.data"
  (lambda ()
    (format #t "start ")
    (catch #t 
      (lambda () 
	(format #t "next ") (abs "hi") (format #t "oops "))
      (lambda args
	'error))
    (format #t "done\n")))

(let ((str (with-input-from-file "tests.data" 
             (lambda () 
	       (read-line)))))
  (if (or (not (string? str))
	  (not (string=? str "start next done")))
      (format #t ";with-output-to-file + error -> ~S~%" str)))

(let ((str (with-input-from-file "tests.data" 
             (lambda () 
	       (catch #t
		      (lambda ()
			(read-char)
			(abs "hi")
			(read-char))
		      (lambda args "s"))))))
  (if (or (not (string? str))
	  (not (string=? str "s")))
      (format #t ";with-input-from-file + error -> ~S~%" str)))

(test (call-with-output-string newline) (string #\newline))
(test (call-with-output-string append) "")

(let ((str (call-with-output-string
	    (lambda (p)
	      (format p "start ")
	      (catch #t 
		     (lambda () 
		       (format p "next ") (abs "hi") (format p "oops "))
		     (lambda args
		       'error))
	      (format p "done")))))
  (if (or (not (string? str))
	  (not (string=? str "start next done")))
      (format #t ";call-with-output-string + error -> ~S~%" str)))

(let ((str (with-output-to-string
	    (lambda ()
	      (format #t "start ")
	      (catch #t 
		     (lambda () 
		       (format #t "next ") (abs "hi") (format #t "oops "))
		     (lambda args
		       'error))
	      (format #t "done")))))
  (if (or (not (string? str))
	  (not (string=? str "start next done")))
      (format #t ";with-output-to-string + error -> ~S~%" str)))

(test (with-output-to-string (lambda () (format (current-output-port) "a test ~D" 123))) "a test 123")
;(test (with-output-to-string (lambda () (format *stdout* "a test ~D" 1234))) "a test 1234")

(let ((str (call-with-input-string "12345"
	    (lambda (p)
	      (catch #t
		     (lambda ()
		       (read-char p)
		       (abs "hi")
		       (read-char p))
		     (lambda args "s"))))))
  (if (or (not (string? str))
	  (not (string=? str "s")))
      (format #t ";call-with-input-string + error -> ~S~%" str)))

(let ((str (with-input-from-string "12345"
	    (lambda ()
	      (catch #t
		     (lambda ()
		       (read-char)
		       (abs "hi")
		       (read-char))
		     (lambda args "s"))))))
  (if (or (not (string? str))
	  (not (string=? str "s")))
      (format #t ";with-input-from-string + error -> ~S~%" str)))

(for-each
 (lambda (arg)
   (test (port-line-number arg) 'error)
   (test (port-filename arg) 'error))
 (list "hi" -1 0 #\a 'a-symbol '#(1 2 3) '(1 . 2) '(1 2 3) 3.14 3/4 1.0+1.0i #t abs #<eof> #<unspecified> (lambda () 1)))

(test (catch #t (lambda () (eval-string (port-filename))) (lambda args #f)) #f)
(test (symbol? (string->symbol (port-filename))) #t)

(for-each
 (lambda (arg)
   (test
    (with-input-from-string (format #f "~A" arg)
      (lambda ()
	(read)))
    arg))
 (list 1 3/4 '(1 2) #(1 2) :hi #f #t))

(num-test (with-input-from-string "3.14" (lambda () (read))) 3.14)
(num-test (with-input-from-string "3.14+2i" (lambda () (read))) 3.14+2i)
(num-test (with-input-from-string "#x2.1" (lambda () (read))) 2.0625)
(test (with-input-from-string "'hi" (lambda () (read))) ''hi)
(test (with-input-from-string "'(1 . 2)" (lambda () (read))) ''(1 . 2))


(test
 (let ((cin #f)
       (cerr #f))
   (catch #t
	  (lambda ()
	    (with-input-from-string "123"
	      (lambda ()
		(set! cin (current-input-port))
		(error 'testing "jump out"))))
	  (lambda args
	    (set! cerr #t)))
   (format #f "~A ~A" cin cerr))
 "<port string input (closed)> #t")

(test
 (let ((cout #f)
       (cerr #f))
   (catch #t
	  (lambda ()
	    (with-output-to-string
	      (lambda ()
		(set! cout (current-output-port))
		(error 'testing "jump out"))))
	  (lambda args
	    (set! cerr #t)))
   (format #f "~A ~A" cout cerr))
 "<port string output (closed)> #t")

(call-with-output-file "tmp1.r5rs"
  (lambda (p)
    (display "1" p)
    (newline p)
    (newline p)
    (display "2345" p)
    (newline p)))

(call-with-input-file "tmp1.r5rs"
  (lambda (p)
    (test (read-line p) "1")
    (test (read-line p) "")
    (test (read-line p) "2345")
    (test (eof-object? (read-line p)) #t)))

(let ((p (open-output-file "tmp1.r5rs" "a")))
  (display "678" p)
  (newline p)
  (close-output-port p))

(test (let ((p (open-output-file "tmp1.r5rs" "xyzzy"))) (close-output-port p)) 'error)
(test (let ((p (open-input-file "tmp1.r5rs" "xyzzy"))) (close-input-port p)) 'error)

(call-with-input-file "tmp1.r5rs"
  (lambda (p)
    (test (read-line p) "1")
    (test (read-line p) "")
    (test (read-line p) "2345")
    (test (read-line p) "678")
    (test (eof-object? (read-line p)) #t)))

(for-each
 (lambda (arg)
   (test (port-filename arg) 'error))
 (list "hi" -1 #\a 1 0 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t '() (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (port-filename arg) 'error))
 (list "hi" -1 #\a 1 0 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t '() (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (open-input-file "s7test.scm" arg) 'error)
   (test (open-output-file "test.data" arg) 'error))
 (list -1 #\a 1 0 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t '() (list 1 2 3) '(1 . 2)))

(test (current-input-port '()) 'error)
(test (current-output-port '()) 'error)
(test (current-error-port '()) 'error)

(for-each
 (lambda (op)
   (let ((tag (catch #t (lambda () (op)) (lambda args 'error))))
     (if (not (eq? tag 'error))
	 (format #t ";(~A) -> ~A (expected 'error)~%" op tag))))
 (list set-current-input-port set-current-error-port set-current-output-port 
       close-input-port close-output-port
       write display write-byte write-char format                     ; newline
       ;read read-char read-byte peek-char char-ready? read-line      ; these can default to current input
       call-with-output-file call-with-input-file
       call-with-output-string call-with-input-string
       with-input-from-string with-input-from-file
       with-output-to-file
       open-output-file open-input-file 
       open-input-string))

(for-each
 (lambda (op)
   (let ((tag (catch #t (lambda () (op 1 2 3 4 5)) (lambda args 'error))))
     (if (not (eq? tag 'error))
	 (format #t ";(~A 1 2 3 4 5) -> ~A (expected 'error)~%" op tag))))
 (list set-current-input-port set-current-error-port set-current-output-port 
       close-input-port close-output-port
       write display write-byte write-char format newline
       read read-char read-byte peek-char char-ready? read-line
       call-with-output-file call-with-input-file
       call-with-output-string call-with-input-string
       with-input-from-string with-input-from-file
       with-output-to-file
       open-output-file open-input-file 
       open-input-string))

;;; (string-set! (with-input-from-string "\"1234\"" (lambda () (read))) 1 #\a)

(test (>= (length (with-output-to-string (lambda () (write (make-string 512 #\tab))))) 512) #t)
(test (>= (length (with-output-to-string (lambda () (write (make-string 512 #\newline))))) 512) #t)
(test (>= (length (with-output-to-string (lambda () (write (make-string 512 #\"))))) 512) #t)
(test (>= (length (with-output-to-string (lambda () (write (make-string 512 #\x65))))) 512) #t)

(if (and (defined? 'file-exists?)
	 (file-exists? "/home/bil/test"))
    (let ((old-path *load-path*))
      (set! *load-path* (cons "/home/bil/test" *load-path*))

      (with-output-to-file "/home/bil/test/load-path-test.scm"
	(lambda ()
	  (format #t "(define (load-path-test) *load-path*)~%")))

      (load "load-path-test.scm")
      (if (or (not (defined? 'load-path-test))
	      (not (equal? *load-path* (load-path-test))))
	  (format #t ";*load-path*: ~S, but ~S~%" *load-path* (load-path-test)))
      (set! *load-path* old-path)))




;;; -------- poke at the reader --------

(test (cdr '(1 ."a")) "a")
(test (cadr '(1 .#d2)) '.#d2)
(test '(1 .(2 3)) '(1 2 3))
(test '(1 .(2 3)) '(1 . (2 3)))
(test (+ .(2 .(3))) 5)
(test (cadr '(1 '0,)) ''0,)
(test (equal? 3 ' 3) #t)
(test (equal? '   
	             3 3) #t)
(test (equal? '"hi" ' "hi") #t)
(test (equal? '#\a '    #\a) #t)
(test (let ((nam()e 1)) 1) 'error)
(test (let ((nam""e 1)) nam""e) 'error) ; this was 1 originally
(test (cadr '(1 ']x)) '']x)
(test `1 1)
(test (equal? '(1 .(1 .())) '(1 1)) #t)
(test (equal? '("hi"."ho") ' ("hi" . "ho")) #t)
(test (equal? '("hi""ho") '("hi" "ho")) #t)
(test '("""""") '("" "" ""))
(test '(#|;"();|#) '())
(test '(#||##\# #||##b1) '(#\# 1))
(test '((). '()) '(() quote ()))
(test '(1. . .2) '(1.0 . 0.2))
(test (equal? '(().()) '(())) #t)
(test (equal? '(()()) '(() ())) #t)
(test (equal? '(()..()) '(() .. ())) #t)
(test '((().()).()) '((())))
(test '(((().()).()).()) '(((()))))
(test '((().(().())).()) '((() ())))
(test '((()().(().()))) '((() () ())))
(test '(1 .;
	  2) '(1 . 2))
(test (vector .(1 .(2))) #(1 2))
(test (vector 0. .(.1)) #(0.0 0.1))

;; currently \ -> (), ` -> #<eof> etc -- not sure these matter
(test (char? #\#) #t)
(test (eval-string "'#<vct>") 'error)
(test (eval-string "'(#<vct>)") 'error)
(test (car `(,.1e0)) .1)
(test (car `(,.1E0)) .1)
(test (let ((x "hi")) (set! x"asdf") x) "asdf")
(test (let ((x 1)) (set! x(list 1 2)) x) '(1 2))
(num-test (let ((x 1)) (set!;"
			x;)
			12.;(
			);#|
	       x) 12.0)
(test (let ((\x00}< 1) (@:\t{ 2)) (+ \x00}< @:\t{)) 3)
(test (let ((?#||#\ 1) (\n\r\t 2) (.1e+2+ie 3)) (+ ?#||#\ \n\r\t .1e+2+ie)) 6)
(test (let ((@,@'[1] 1) (\,| 2)) (+ @,@'[1] \,|)) 3)
(test (list"0"0()#()#\a"""1"'x(list)+(cons"""")#f) (list "0" 0 () #() #\a "" "1" 'x (list) + '("" . "") #f))
(test (let ((x, 1)) x,) 1)
(test (length (eval-string (string #\' #\( #\1 #\space #\. (integer->char 200) #\2 #\)))) 2) ; will be -1 if dot is for improper list, 3 if dot is a symbol
(test (eval-string "(list \\\x001)") 'error)
(test (eval-string "(list \\\x00 1)") 'error)
(test (+ `,0(angle ```,`11)) 0)
(test (map . (char->integer "123")) '(49 50 51))
(test (map .(values "0'1")) '(#\0 #\' #\1))
(test (map /""'(123)) '())
(num-test (+ 1 .()) 1)

;; how is ...#(... parsed?
(test (eval-string "'(# (1))") 'error)
(test (let ((lst (eval-string "'(#(1))"))) (and (= (length lst) 1) (vector? (car lst)))) #t)                     ; '(#(1))
(test (let ((lst (eval-string "'(#\ (1))"))) (and (= (length lst) 1) (vector? (car lst)))) #t)                   ; '(#(1))
(test (let ((lst (eval-string "'(-#(1))"))) (and (= (length lst) 2) (symbol? (car lst)) (pair? (cadr lst)))) #t) ; '(-# (1))
(test (let ((lst (eval-string "'(1#(1))"))) (and (= (length lst) 2) (symbol? (car lst)) (pair? (cadr lst)))) #t) ; '(1# (1))
(test (let ((lst (eval-string "'('#(1))"))) (and (= (length lst) 1) (vector? (cadar lst)))) #t)                  ; '((quote #(1)))
(test (let ((lst (eval-string "'(()#())"))) (and (= (length lst) 2) (null? (car lst)) (vector? (cadr lst)))) #t) ; '(() #())
(test (let ((lst (eval-string "'(().())"))) (and (= (length lst) 1) (null? (car lst)))) #t)                      ; '(())
(test (let ((lst (eval-string "'(()-())"))) (and (= (length lst) 3) (null? (car lst)) (null? (caddr lst)))) #t)  ; '(() - ())
(test (let ((lst (eval-string "'(().#())"))) (and (= (length lst) 3) (null? (car lst)) (null? (caddr lst)))) #t) ; '(() .# ())
(test (let ((lst (eval-string "'((). #())"))) (and (= (length lst) -1) (null? (car lst)) (vector? (cdr lst)))) #t) ; '(() . #())
(test (let ((lst (eval-string "'(\"\"#())"))) (and (= (length lst) 2) (string? (car lst)) (vector? (cadr lst)))) #t) ; '("" #())
(test (length (car '("#\\("))) 3)
(test (length (car '("#\\\""))) 3)
(test (char=? ((car '("#\\\"")) 2) #\") #t)
(test (length '(()#\(())) 3)
(test (length (eval-string "'(()#\\(())")) 3)
(test (char=? ((eval-string "'(()#\\#())") 1) #\#) #t)
(test (length (list""#t())) 3)
(test (length (list""#())) 2)
(test (length (eval-string "'(#xA(1))")) 2)
(test (length '(#xA""#(1))) 3)
(test (length (eval-string "'(#xA\"\"#(1))")) 3)
(test (length (eval-string "'(1#f)")) 1)
(test (eval-string "'(#f#())") 'error)
(test (length '(#f())) 2)
(test (length '(#f"")) 2)
(test (eval-string "#F") 'error)
(test (eval-string "'(#<eof>#<eof>)") 'error)
(test (eval-string "'(#<eof>#())") 'error)
(test (equal? '('#()) '(#())) #f)
(test (equal? (list '#()) '(#())) #t)
(test (equal? '('#()) '('#())) #t)
(test (equal? '('#()) '(`#())) #f) ; ! [guile agrees]
(test (equal? '('()) '(`())) #f) ; ! quote != quasiquote [guile agrees]
(test (equal? '('(1)) '(`(1))) #t) ; !! but lists are different? [guile says #f]
(test (equal? '('#(1)) '(`#(1))) #f) ; ! [guile agrees]
(test (equal? '('#()) '(#())) #f)
(test (equal? '(`#()) '(`#())) #t)
(test (equal? '#() `#()) #t)
(test (equal? (list '#()) (list `#())) #t)
(test (equal? (list '#()) '(`#())) #t)
(test (equal? '(`#()) '(#())) #t)
(test (equal? `#() '#()) #t) ; and also (1) () #(1) etc
(test (equal? `'#() ''#()) #t) ; "
(test (equal? '`#() ''#()) #f) ; ! it equals '#()
(test (equal? '`#() ``#()) #t)
;; gah -- `#(...) should be removed from s7
;; but there is still the strangeness that `'() is not the same as '`() -- quasiquote returns '() not (), and similarly for #() except...
;; there are actually 3 cases here, the 3rd is (equal? `''() ``'()) -> #f, but the #() case is #t

#|
"(equal? ''() ``())" -> #f
"(equal? ''(1) ``(1))" -> #t 
"(equal? ''#() ``#())" -> #f
"(equal? ''#(1) ``#(1))" -> #f
"(equal? ''1 ``1)" -> #f
"(equal? ''#f ``#f)" -> #f

"(equal? `''() ``'())" -> #f 
"(equal? `''(1) ``'(1))" -> #t
"(equal? `''#() ``'#())" -> #t
"(equal? `''#(1) ``'#(1))" -> #t
"(equal? `''1 ``'1)" -> #t
"(equal? `''#f ``'#f)" -> #t

;; see t242.scm

(define (check-strs str1 str2)
  (let* ((expr (format #f "(equal? ~A ~A)" (string-append str1 "()") (string-append str2 "()")))
	 (val (catch #t 
		     (lambda () (eval-string expr))
		     (lambda args 'error))))
    (format #t "--------~%~S -> ~S~%" expr val)
    (set! expr (format #f "(equal? ~A ~A)" (string-append str1 "(1)") (string-append str2 "(1)")))
    (let ((val (catch #t 
		      (lambda () (eval-string expr))
		      (lambda args 'error))))
      (format #t "~S -> ~S~%" expr val))
    (set! expr (format #f "(equal? ~A ~A)" (string-append str1 "#()") (string-append str2 "#()")))
    (let ((val (catch #t 
		      (lambda () (eval-string expr))
		      (lambda args 'error))))
      (format #t "~S -> ~S~%" expr val))
    (set! expr (format #f "(equal? ~A ~A)" (string-append str1 "#(1)") (string-append str2 "#(1)")))
    (let ((val (catch #t 
		      (lambda () (eval-string expr))
		      (lambda args 'error))))
      (format #t "~S -> ~S~%" expr val))
    (set! expr (format #f "(equal? ~A ~A)" (string-append str1 "1") (string-append str2 "1")))
    (let ((val (catch #t 
		      (lambda () (eval-string expr))
		      (lambda args 'error))))
      (format #t "~S -> ~S~%" expr val))
    (set! expr (format #f "(equal? ~A ~A)" (string-append str1 "#f") (string-append str2 "#f")))
    (let ((val (catch #t 
		      (lambda () (eval-string expr))
		      (lambda args 'error))))
      (format #t "~S -> ~S~%" expr val))))

(let ((strs '()))
  (do ((i 0 (+ i 1)))
      ((= i 4))
    (let ((c1 ((vector #\' #\` #\' #\`) i))
	  (c2 ((vector #\' #\' #\` #\`) i)))
      (do ((k 0 (+ k 1)))
	  ((= k 4))
	(let ((d1 ((vector #\' #\` #\' #\`) k))
	      (d2 ((vector #\' #\' #\` #\`) k)))
	  (let ((str1 (string c1 c2))
		(str2 (string d1 d2)))
	    (if (not (member (list str1 str2) strs))
		(begin
		  (check-strs str1 str2)
		  (set! strs (cons (list str1 str2) strs))
		  (set! strs (cons (list str2 str1) strs))))))))))

(let ((strs '()))
  (do ((i 0 (+ i 1)))
      ((= i 8))
    (let ((c1 ((vector #\' #\` #\' #\` #\' #\` #\' #\`) i))
	  (c2 ((vector #\' #\' #\` #\` #\' #\' #\` #\`) i))
	  (c3 ((vector #\' #\' #\' #\' #\` #\` #\` #\`) i)))
      (do ((k 0 (+ k 1)))
	  ((= k 8))
	(let ((d1 ((vector #\' #\` #\' #\` #\' #\` #\' #\`) k))
	      (d2 ((vector #\' #\' #\` #\` #\' #\' #\` #\`) k))
	      (d3 ((vector #\' #\' #\' #\' #\` #\` #\` #\`) k)))
	  (let ((str1 (string c1 c2 c3))
		(str2 (string d1 d2 d3)))
	    (if (not (member (list str1 str2) strs))
		(begin
		  (check-strs str1 str2)
		  (set! strs (cons (list str1 str2) strs))
		  (set! strs (cons (list str2 str1) strs))))))))))
|#

#|
(do ((i 0 (+ i 1)))
    ((= i 256))
  (if (and (not (= i (char->integer #\))))
	   (not (= i (char->integer #\"))))
      (let ((str (string #\' #\( #\1 #\space #\. (integer->char i) #\2 #\))))
	(catch #t
	       (lambda ()
		 (let ((val (eval-string str)))
		   (format #t "[~D] ~A -> ~S (~S ~S)~%" i str val (car val) (cdr val))))
	       (lambda args
		 (format #t "[~D] ~A -> ~A~%" i str args))))))

(let ((chars (vector (integer->char 0) #\newline #\space #\tab #\. #\, #\@ #\= #\x #\b #\' #\` #\# #\] #\[ #\} #\{ #\( #\) #\1 #\i #\+ #\- #\e #\_ #\\ #\" #\: #\; #\> #\<)))
  (let ((nchars (vector-length chars)))
    (do ((len 2 (+ len 1)))
	((= len 3))
      (let ((str (make-string len))
	    (ctrs (make-vector len 0)))

	(do ((i 0 (+ i 1)))
	    ((= i (expt nchars len)))

	  (let ((carry #t))
	    (do ((k 0 (+ k 1)))
		((or (= k len)
		     (not carry)))
	      (vector-set! ctrs k (+ 1 (vector-ref ctrs k)))
	      (if (= (vector-ref ctrs k) nchars)
		  (vector-set! ctrs k 0)
		  (set! carry #f)))
	    (do ((k 0 (+ k 1)))
		((= k len))
	      (string-set! str k (vector-ref chars (vector-ref ctrs k)))))

	  (format #t "~A -> " str)
	  (catch #t
		 (lambda ()
		   (let ((val (eval-string str)))
		     (format #t " ~S (~S ~S)~%" val (car val) (cdr val))))
		 (lambda args
		   (format #t " ~A~%" args))))))))
|#

(let ((äåæéîå define)
      (ìåîçôè length)
      (äï do)
      (ìåô* let*)
      (éæ if)
      (áâó abs)
      (ìïç log)
      (óåô! set!))

  (äåæéîå (óòã-äõòáôéïî å)
    (ìåô* ((ìåî (ìåîçôè å))
           (åø0 (å 0))
           (åø1 (å (- ìåî 2)))
           (áìì-ø (- åø1 åø0))
           (äõò 0.0))
      (äï ((é 0 (+ é 2)))
          ((>= é (- ìåî 2)) äõò)
        (ìåô* ((ø0 (å é))
               (ø1 (å (+ é 2)))
               (ù0 (å (+ é 1))) ; 1/ø ø ðïéîôó
               (ù1 (å (+ é 3)))
               (áòåá (éæ (< (áâó (- ù0 ù1)) .0001)
                         (/ (- ø1 ø0) (* ù0 áìì-ø))
                         (* (/ (- (ìïç ù1) (ìïç ù0)) 
                               (- ù1 ù0)) 
                            (/ (- ø1 ø0) áìì-ø)))))
         (óåô! äõò (+ äõò (áâó áòåá)))))))

  (num-test (óòã-äõòáôéïî (list 0 1 1 2)) 0.69314718055995)
  (num-test (óòã-äõòáôéïî (vector 0 1 1 2)) 0.69314718055995))

(test (let ((ÿa 1)) ÿa) 1)
(test (+ (let ((!a 1)) !a) (let (($a 1)) $a) (let ((%a 1)) %a) (let ((&a 1)) &a) (let ((*a 1)) *a) (let ((+a 1)) +a) (let ((-a 1)) -a) (let ((.a 1)) .a) (let ((/a 1)) /a) (let ((0a 1)) 0a) (let ((1a 1)) 1a) (let ((2a 1)) 2a) (let ((3a 1)) 3a) (let ((4a 1)) 4a) (let ((5a 1)) 5a) (let ((6a 1)) 6a) (let ((7a 1)) 7a) (let ((8a 1)) 8a) (let ((9a 1)) 9a) (let ((<a 1)) <a) (let ((=a 1)) =a) (let ((>a 1)) >a) (let ((?a 1)) ?a) (let ((@a 1)) @a) (let ((Aa 1)) Aa) (let ((Ba 1)) Ba) (let ((Ca 1)) Ca) (let ((Da 1)) Da) (let ((Ea 1)) Ea) (let ((Fa 1)) Fa) (let ((Ga 1)) Ga) (let ((Ha 1)) Ha) (let ((Ia 1)) Ia) (let ((Ja 1)) Ja) (let ((Ka 1)) Ka) (let ((La 1)) La) (let ((Ma 1)) Ma) (let ((Na 1)) Na) (let ((Oa 1)) Oa) (let ((Pa 1)) Pa) (let ((Qa 1)) Qa) (let ((Ra 1)) Ra) (let ((Sa 1)) Sa) (let ((Ta 1)) Ta) (let ((Ua 1)) Ua) (let ((Va 1)) Va) (let ((Wa 1)) Wa) (let ((Xa 1)) Xa) (let ((Ya 1)) Ya) (let ((Za 1)) Za) (let (([a 1)) [a) (let ((\a 1)) \a) (let ((]a 1)) ]a) (let ((^a 1)) ^a) (let ((_a 1)) _a) (let ((aa 1)) aa) (let ((ba 1)) ba) (let ((ca 1)) ca) (let ((da 1)) da) (let ((ea 1)) ea) (let ((fa 1)) fa) (let ((ga 1)) ga) (let ((ha 1)) ha) (let ((ia 1)) ia) (let ((ja 1)) ja) (let ((ka 1)) ka) (let ((la 1)) la) (let ((ma 1)) ma) (let ((na 1)) na) (let ((oa 1)) oa) (let ((pa 1)) pa) (let ((qa 1)) qa) (let ((ra 1)) ra) (let ((sa 1)) sa) (let ((ta 1)) ta) (let ((ua 1)) ua) (let ((va 1)) va) (let ((wa 1)) wa) (let ((xa 1)) xa) (let ((ya 1)) ya) (let ((za 1)) za) (let (({a 1)) {a) (let ((|a 1)) |a) (let ((}a 1)) }a) (let ((~a 1)) ~a) (let (( a 1))  a) (let ((¡a 1)) ¡a) (let ((¢a 1)) ¢a) (let ((£a 1)) £a) (let ((¤a 1)) ¤a) (let ((¥a 1)) ¥a) (let ((¦a 1)) ¦a) (let ((§a 1)) §a) (let ((¨a 1)) ¨a) (let ((©a 1)) ©a) (let ((ªa 1)) ªa) (let ((«a 1)) «a) (let ((¬a 1)) ¬a) (let ((­a 1)) ­a) (let ((®a 1)) ®a) (let ((¯a 1)) ¯a) (let ((°a 1)) °a) (let ((±a 1)) ±a) (let ((²a 1)) ²a) (let ((³a 1)) ³a) (let ((´a 1)) ´a) (let ((µa 1)) µa) (let ((¶a 1)) ¶a) (let ((·a 1)) ·a) (let ((¸a 1)) ¸a) (let ((¹a 1)) ¹a) (let ((ºa 1)) ºa) (let ((»a 1)) »a) (let ((¼a 1)) ¼a) (let ((½a 1)) ½a) (let ((¾a 1)) ¾a) (let ((¿a 1)) ¿a) (let ((Àa 1)) Àa) (let ((Áa 1)) Áa) (let ((Âa 1)) Âa) (let ((Ãa 1)) Ãa) (let ((Äa 1)) Äa) (let ((Åa 1)) Åa) (let ((Æa 1)) Æa) (let ((Ça 1)) Ça) (let ((Èa 1)) Èa) (let ((Éa 1)) Éa) (let ((Êa 1)) Êa) (let ((Ëa 1)) Ëa) (let ((Ìa 1)) Ìa) (let ((Ía 1)) Ía) (let ((Îa 1)) Îa) (let ((Ïa 1)) Ïa) (let ((Ða 1)) Ða) (let ((Ña 1)) Ña) (let ((Òa 1)) Òa) (let ((Óa 1)) Óa) (let ((Ôa 1)) Ôa) (let ((Õa 1)) Õa) (let ((Öa 1)) Öa) (let ((×a 1)) ×a) (let ((Øa 1)) Øa) (let ((Ùa 1)) Ùa) (let ((Úa 1)) Úa) (let ((Ûa 1)) Ûa) (let ((Üa 1)) Üa) (let ((Ýa 1)) Ýa) (let ((Þa 1)) Þa) (let ((ßa 1)) ßa) (let ((àa 1)) àa) (let ((áa 1)) áa) (let ((âa 1)) âa) (let ((ãa 1)) ãa) (let ((äa 1)) äa) (let ((åa 1)) åa) (let ((æa 1)) æa) (let ((ça 1)) ça) (let ((èa 1)) èa) (let ((éa 1)) éa) (let ((êa 1)) êa) (let ((ëa 1)) ëa) (let ((ìa 1)) ìa) (let ((ía 1)) ía) (let ((îa 1)) îa) (let ((ïa 1)) ïa) (let ((ða 1)) ða) (let ((ña 1)) ña) (let ((òa 1)) òa) (let ((óa 1)) óa) (let ((ôa 1)) ôa) (let ((õa 1)) õa) (let ((öa 1)) öa) (let ((÷a 1)) ÷a) (let ((øa 1)) øa) (let ((ùa 1)) ùa) (let ((úa 1)) úa) (let ((ûa 1)) ûa) (let ((üa 1)) üa) (let ((ýa 1)) ýa) (let ((þa 1)) þa) (let ((ÿa 1)) ÿa)) 181)

;;; there are about 50 non-printing chars, some of which would probably work as well


;; (eval-string "(eval-string ...)") is not what it appears to be -- the outer call
;;    still sees the full string when it evaluates, not the string that results from
;;    the inner call.



;;; -------- object->string
;;; object->string

(test (string=? (object->string 32) "32") #t)
(test (string=? (object->string 32.5) "32.5") #t)
(test (string=? (object->string 32/5) "32/5") #t)
(test (object->string 1+i) "1+1i")
(test (string=? (object->string "hiho") "\"hiho\"") #t)
(test (string=? (object->string 'symb) "symb") #t)
(test (string=? (object->string (list 1 2 3)) "(1 2 3)") #t)
(test (string=? (object->string (cons 1 2)) "(1 . 2)") #t)
(test (string=? (object->string '#(1 2 3)) "#(1 2 3)") #t)
(test (string=? (object->string +) "+") #t)
(test (object->string (object->string (object->string "123"))) "\"\\\"\\\\\\\"123\\\\\\\"\\\"\"")
(test (object->string #<eof>) "#<eof>")
(test (object->string (if #f #f)) "#<unspecified>")
(test (object->string #<undefined>) "#<undefined>")
(test (object->string #f) "#f")
(test (object->string #t) "#t")
(test (object->string '()) "()")
(test (object->string #()) "#()")
(test (object->string "") "\"\"")
(test (object->string abs) "abs")
(test (object->string lambda) "lambda")
(test (object->string +) "+")
(test (object->string +) "+")
(test (object->string '''2) "''2")
(test (object->string (lambda () #f)) "#<closure>")
(test (call-with-exit (lambda (return) (object->string return))) "#<goto>")
(test (call/cc (lambda (return) (object->string return))) "#<continuation>")
(test (let () (define-macro (hi a) `(+ 1 ,a)) (object->string hi)) "#<macro>")
(test (let () (define (hi a) (+ 1 a)) (object->string hi)) "hi")
(test (let () (define* (hi a) (+ 1 a)) (object->string hi)) "hi")
(test (object->string dynamic-wind) "dynamic-wind")
(test (object->string (make-procedure-with-setter (lambda () 1) (lambda (val) val))) "#<procedure-with-setter>")
(test (object->string object->string) "object->string")
(test (object->string 'if) "if")
(test (object->string begin) "begin")
(test (object->string let) "let")

(test (object->string #\n #f) "n")
(test (object->string #\n) "#\\n")
(test (object->string #\r) "#\\r")
(test (object->string #\r #f) "r")
(test (object->string #\t #f) "t")
(test (object->string #\t) "#\\t")

(test (object->string "a\x00b" #t) "\"a\\x00b\"")
(test (object->string "a\x00b" #f) "a")

#|
(do ((i 0 (+ i 1))) 
    ((= i 256)) 
  (let ((c (integer->char i))) 
    (let ((str (object->string c))) 
      (if (and (not (= (length str) 3))       ; "#\\a"
	       (or (not (char=? (str 2) #\x))
		   (not (= (length str) 5)))) ; "#\\xee"
	  (format #t "(#t) ~C: ~S~%" c str))
      (set! str (object->string c #f))
      (if (not (= (length str) 1))
	  (format #t "(#f) ~C: ~S~%" c str)))))
this prints:
(#t) : "#\\null"
(#f) : ""
(#t) : "#\\x1"
(#t) : "#\\x2"
(#t) : "#\\x3"
(#t) : "#\\x4"
(#t) : "#\\x5"
(#t) : "#\\x6"
(#t) : "#\\x7"
(#t): "#\\x8"
(#t) 	: "#\\tab"
(#t) 
: "#\\newline"
(#t) 
     : "#\\xb"
(#t) 
     : "#\\xc"
: "#\\return"
(#t) : "#\\xe"
(#t) : "#\\xf"
(#t)  : "#\\space"
|#

(test (object->string #\x30) "#\\0")
(test (object->string #\x91) "#\\x91")
(test (object->string #\x10) "#\\x10")
(test (object->string #\xff) "#\\xff")
(test (object->string #\x55) "#\\U")
(test (object->string #\x7e) "#\\~")
(test (object->string #\newline) "#\\newline")
(test (object->string #\return) "#\\return")
(test (object->string #\tab) "#\\tab")
(test (object->string #\null) "#\\null")
(test (object->string #\space) "#\\space")
(test (object->string ''#\a) "'#\\a")
(test (object->string (list 1 '.' 2)) "(1 .' 2)")
(test (object->string (quote (quote))) "(quote)")
(test (object->string (quote quote)) "quote")
(test (object->string (quote (quote (quote)))) "'(quote)")

(test (object->string) 'error)
(test (object->string 1 2) 'error)
(test (object->string 1 #f #t) 'error)
(test (object->string abs) "abs")
(test(let ((val 0)) (cond (else (set! val (object->string else)) 1)) val) "else")
(test (cond (else (object->string else))) "else")
(test (object->string (string->symbol (string #\; #\" #\)))) "(symbol \";\\\")\")")

(test (object->string "hi" #f) "hi")
(test (object->string "h\\i" #f) "h\\i")
(test (object->string -1.(list? -1e0)) "-1.0")

(test (object->string catch) "catch")
(test (object->string lambda) "lambda")
(test (object->string dynamic-wind) "dynamic-wind")
;(test (object->string else) "else") ; this depends on previous code
(test (object->string do) "do")

(for-each
 (lambda (arg)
   (test (object->string 1 arg) 'error))
 (list "hi" -1 #\a 1 0 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i '() (list 1 2 3) '(1 . 2)))

(test (symbol? (string->symbol (object->string "" #f))) #t)
(test (string->symbol (object->string #(1 #\a (3)) #f)) (symbol "#(1 #\\a (3))"))
(test (string->list (object->string #(1 2) #f)) '(#\# #\( #\1 #\space #\2 #\)))
(test (string->list (object->string #(1 #\a (3)) #f)) '(#\# #\( #\1 #\space #\# #\\ #\a #\space #\( #\3 #\) #\)))
(test (reverse (object->string #2D((1 2) (3 4)) #f))  "))4 3( )2 1((D2#")




;;; --------------------------------------------------------------------------------
;;; CONTROL OPS
;;; --------------------------------------------------------------------------------

(define control-ops (list lambda define quote if begin set! let let* letrec cond case and or do
			  call/cc eval apply for-each map values call-with-values dynamic-wind))
(for-each
 (lambda (op)
   (if (not (eq? op op))
       (format #t "~A not eq? to itself?~%" op)))
 control-ops)

(for-each
 (lambda (op)
   (if (not (eqv? op op))
       (format #t "~A not eqv? to itself?~%" op)))
 control-ops)

(for-each
 (lambda (op)
   (if (not (equal? op op))
       (format #t "~A not equal? to itself?~%" op)))
 control-ops)

(define question-ops (list boolean? eof-object? string?
		           number? integer? real? rational? complex? char?
			   list? vector? pair? null?))

(for-each
 (lambda (ques)
   (for-each
    (lambda (op)
      (if (ques op)
	  (format #t ";(~A ~A) returned #t?~%" ques op)))
    control-ops))
 question-ops)

(let ((unspecified (if #f #f)))
  (for-each
   (lambda (op)
     (if (op unspecified)
	 (format #t ";(~A #<unspecified>) returned #t?~%" op)))
   question-ops))

(for-each 
 (lambda (s)
   (if (not (symbol? s))
       (format #t ";(symbol? ~A returned #f?~%" s)))
 '(+ - ... !.. $.+ %.- &.! *.: /:. <-. =. >. ?. ~. _. ^.))



;;; --------------------------------------------------------------------------------
;;; if
;;; --------------------------------------------------------------------------------

(test ((if #f + *) 3 4) 12)
(test (if (> 3 2) 'yes 'no) 'yes)
(test (if (> 2 3) 'yes 'no) 'no)
(test (if (> 3 2) (- 3 2) (+ 3 2)) 1)
(test (if (> 3 2) 1) 1)
(test (if '() 1 2) 1)
(test (if 't 1 2) 1)
(test (if #t 1 2) 1)
(test (if '#() 1 2) 1)
(test (if 1 2 3) 2)
(test (if 0 2 3) 2)
(test (if (list) 2 3) 2)
(test (if "" 2 3) 2)
(test (eq? (if #f #f) (if #f #f)) #t) ; I assume there's only one #<unspecified>!
(test (if . (1 2)) 2)
(test (if (if #f #f) #f #t) #f)
(test (if 1/0 0 1) 0)

(test (let ((a #t) (b #f) (c #t) (d #f)) (if (if (if (if d d c) d b) d a) 'a 'd)) 'a)
(test (let ((a #t) (b #f) (c #t) (d #f)) (if a (if b (if c (if d d c) c) 'b) 'a)) 'b)
(test (let ((a #t) (b #f) (c #t) (d #f)) (if b (if a (if d 'gad) 'gad) (if d 'gad 'a))) 'a)

(let ((a #t))
  (for-each
   (lambda (arg)
     (test (if a arg 'gad) arg))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2))))

(let ((a #t))
  (for-each
   (lambda (arg)
     (test (if (not a) 'gad arg) arg))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2))))

(test (let ((ctr 0) (a #t)) (if a (let ((b ctr)) (set! ctr (+ ctr 1)) (list b ctr)) (let ((c ctr)) (set! ctr (+ ctr 100)) (list c ctr)))) (list 0 1))

(test (if if if if) if)
(test (((if if if) if if) if if 'gad) if)
(test (if if (if if if) if) if)
(test (let ((car if)) (car #t 0 1)) 0)
(test ((car (list if)) #t 0 1) 0)
(test (symbol->string 'if) "if")
(test (if (and if (if if if)) if 'gad) if)
(test (let ((ctr 0)) (if (let () (set! ctr (+ ctr 1)) (= ctr 1)) 0 1)) 0)
(test (let ((ctr 0)) (if (let () (set! ctr (+ ctr 1)) (if (= ctr 1) (> 3 2) (< 3 2))) 0 1)) 0)
(test (        if (> 3 2) 1 2) 1)
(test (let ((alist (list (list map 1) (list car 2) (list if 3) (list do 4)))) (assoc if alist)) (list if 3))
(test (let ((alist (list (list map 1) (list car 2) (list if 3) (list do 4)))) (assv if alist)) (list if 3))
(test (let ((alist (list (list map 1) (list car 2) (list if 3) (list do 4)))) (assq if alist)) (list if 3))
(test (let ((alist (list map car if do))) (member if alist)) (list if do))
(test (let ((alist (list map car if do))) (memv if alist)) (list if do))
(test (let ((alist (list map car if do))) (memq if alist)) (list if do))
(test ((vector-ref (vector if) 0) #t 1 2) 1)
(test ((vector-ref (make-vector 1 if) 0) #t 1 2) 1)
(test ((if #t + -) 3 4) 7)
(test (list (if 0 1 2)) (list 1))
(test ((car (list if map)) #f 1 2) 2)
(test (let ((ctr 0)) (if (= ctr 0) (let () (set! ctr (+ ctr 1)) (if (= ctr 1) 2 3)) (let () (set! ctr (+ ctr 1)) (if (= ctr 1) 4 5)))) 2)
(test (let ((x (cons 1 2))) (set-cdr! x x) (if x 1 2)) 1)
(test (let ((ctr 0)) (if (let ((ctr 123)) (set! ctr (+ ctr 1)) (= ctr 124)) (let () (set! ctr (+ ctr 100)) ctr) (let () (set! ctr (+ ctr 1000)) ctr)) ctr) 100)
(test (let () (if #t (define (hi a) a)) (hi 1)) 1)
(test (let () (if #f (define (hi a) (+ a 1)) (define (hi a) a)) (hi 1)) 1)

(test (let ((ctr 0)) (call/cc (lambda (exit) (if (> 3 2) (let () (exit ctr) (set! ctr 100) ctr) #f)))) 0)
(test (let ((ctr 0)) (call/cc (lambda (exit) (if (< 3 2) #f (let () (exit ctr) (set! ctr 100) ctr))))) 0)
(test (let ((ctr 0)) (call/cc (lambda (exit) (if (let () (exit ctr) (set! ctr 100) ctr) 123 321)))) 0)
(test (let ((ctr 0)) (if (> 3 2) (call/cc (lambda (exit) (set! ctr (+ ctr 1)) (exit ctr))) #f) ctr) 1)

(test (let ((ctr 0))
	(do ((x 0 (+ x 1)))
	    ((= x 12))
	  (if (> x 0)
	      (if (> x 1)
		  (if (> x 2)
		      (if (> x 3)
			  (if (> x 4)
			      (if (> x 5)
				  (if (> x 6)
				      (if (> x 7)
					  (if (> x 8)
					      (if (> x 9)
						  (if (> x 10)
						      (set! ctr (+ ctr 1000))
						      (set! ctr (- ctr 1)))
						  (set! ctr (- ctr 2)))
					      (set! ctr (- ctr 3)))
					  (set! ctr (- ctr 4)))
				      (set! ctr (- ctr 5)))
				  (set! ctr (- ctr 6)))
			      (set! ctr (- ctr 7)))
			  (set! ctr (- ctr 8)))
		      (set! ctr (- ctr 9)))
		  (set! ctr (- ctr 10)))
	      (set! ctr (- ctr 11))))
	ctr)
      934)

(test (let ((ctr 0))
	(do ((x 0 (+ x 1)))
	    ((= x 12))
	  (if (> x 0)
	      (if (> x 1)
		  (if (> x 2)
		      (if (> x 3)
			  (if (> x 4)
			      (if (> x 5)
				  (if (> x 6)
				      (if (> x 7)
					  (if (> x 8)
					      (if (> x 9)
						  (if (> x 10)
						      (set! ctr (+ ctr 1000))
						      (set! ctr (- ctr 1)))
						  (set! ctr (- ctr 2)))
					      (set! ctr (- ctr 3)))
					  (set! ctr (- ctr 4))))))))
		  (set! ctr (- ctr 10)))
	      (set! ctr (- ctr 11))))
	ctr)
      969)

(test (if #f) 'error)
(test (if (< 2 3)) 'error)
(test (if #f 1 2 3) 'error)
(test (if 1 2 3 4) 'error)
(test (if #f 1 else 2) 'error)
(test (if) 'error)
(test ('+ '1 '2) 'error)
(test (if 1 . 2) 'error)
(test (if 1 2 . 3) 'error)
(test (if . 1) 'error)
(test (if _no_var_ 1) 'error)
(test (if (values) (values) (values) 1) 'error)
(num-test (+ 1 (if #t (values 3 4) (values 5 6)) 2) 10)
(let ()
  (define (bad a) (if a 1 2 3))
  (test (bad #f) 'error)
  (test (bad #t) 'error))





;;; --------------------------------------------------------------------------------
;;; quote
;;; --------------------------------------------------------------------------------

(test (quote a) 'a)
(test 'a (quote a))
(test '1 1)
(test '1/4 1/4)
(test '(+ 2 3) '(+ 2 3))
(test '"hi" "hi")
(test '#\a #\a)
(test '#f #f)
(test '#t #t)
(test '#b1 1)
(test (= 1/2 '#e#b1e-1) #t)
(test '() '())
(test '(1 . 2) (cons 1 2))
(test #(1 2) '#(1 2))
(test (+ '1 '2) 3)
(test (+ '1 '2) '3)
(test (+ ' 1 '   2) '    3)
(test (char? '#\a) #t)
(test (string? '"hi") #t)
(test (boolean? '#t) #t)
(test (if '#f 2 3) 3)
(test (if '#t 2 3) 2)
(test (vector? '#()) #t)
(test (char? (quote #\a)) #t)
(test (string? (quote "hi")) #t)
(test (boolean? (quote #t)) #t)
(test (if (quote #f) 2 3) 3)
(test (if (quote #t) 2 3) 2)
(test (vector? (quote #())) #t)
(test (+ (quote 1) (quote 2)) (quote 3))
(test (list? (quote ())) #t)
(test (pair? (quote (1 . 2))) #t)
(test (+ '1.0 '2.0) 3.0)
(test (+ '1/2 '3/2) 2)
(test (+ '1.0+1.0i '-2.0) -1.0+1.0i)
(test (let ((hi 2)) (equal? hi 'hi)) #f)
(test ''1 (quote (quote 1)))
(test ''a (quote (quote a)))
(test (symbol? '#f) #f)
(test (symbol? '.') #t)
(test ''quote (quote (quote quote)))
(test (+ (cadr ''3) (cadadr '''4) (cadr (cadr (cadr ''''5)))) 12)
(test (+ (cadr ' '   3) (cadadr '  
  '    ' 4)) 7)
(test (+ '#| a comment |#2 3) 5)
(test (+ ' #| a comment |# 2 3) 5)
(test (eq? lambda 'lambda) #f)
(test (equal? + '+) #f)
(test (eq? '() ()) #t) ; s7 specific

(test (quote) 'error)
(test (quote . -1) 'error)
(test (quote 1 1) 'error)
(test (quote . 1) 'error)
(test (quote . (1 2)) 'error)
(test (quote 1 . 2) 'error)
(test (symbol? '1'1) #t) 
(test (apply '+ (list 1 2)) 'error)

(test (equal? '(1 2 '(3 4)) '(1 2 (3 4))) #f)
(test (equal? '(1 2 '(3 4)) (quote (list 1 2 (quote (list 3 4))))) #f)
(test (equal? (list-ref '(1 2 '(3 4)) 2) '(3 4)) #f)
(test (equal? '(1 2 '(3 4)) (list 1 2 (list 'quote (list 3 4)))) #t)
(test (equal? '(1 2 ''(3 4)) (list 1 2 (list 'quote (list 'quote (list 3 4))))) #t)
(test (equal? '('3 4) (list (list 'quote 3) 4)) #t)
(test (equal? '('3 4) (list 3 4)) #f)
(test (equal? '('() 4) (list (list 'quote '()) 4)) #t)
(test (equal? '('('4)) (list (list quote (list (list quote 4))))) #f)
(test (equal? '('('4)) (list (list 'quote (list (list 'quote 4))))) #t) 
(test (equal? '('('4)) '((quote ((quote 4))))) #t)
(test (equal? '1 ''1) #f)
(test (equal? ''1 ''1) #t)
(test (equal? '(1 '(1 . 2)) (list 1 (cons 1 2))) #f)
(test (equal? #(1 #(2 3)) '#(1 '#(2 3))) #f)
(test (equal? #(1) #('1)) #f)
(test (equal? #(()) #('())) #f)

(test (eqv? #\a (quote #\a)) #t)
(test (eqv? 1 (quote 1)) #t)
(test (eqv? 0 (quote 0)) #t)
(test (equal? #(1 2 3) (quote #(1 2 3))) #t)
(test (eqv? 3.14 (quote 3.14)) #t)
(test (eqv? 3/4 (quote 3/4)) #t)
(test (eqv? 1+1i (quote 1+1i)) #t)
(test (eq? #f (quote #f)) #t)
(test (eq? #t (quote #t)) #t)
(test (eq? '() (quote ())) #t)
(test (equal? '(1 2 3) (quote (1 2 3))) #t)
(test (equal? '(1 . 2) (quote (1 . 2))) #t)
(test ('abs -1) 'error)
(test ('"hi" 0) #\h)

(test (''begin 1) 'begin)
(test (''let ((x 1)) ('set! x 3) x) 'error)
(test ('and #f) 'error)
(test ('and 1 #f) 'error)
(test ('begin 1) 'error)
(test ('cond ('define '#f)) 'error)
(test ('let ((x 1)) ('set! x 3) x) 'error)
(test ('let* () ('define x 3) x) 'error)
(test ('or #f) 'error)
(test ('quote 3) 'error)
(test ((copy quote) 1) 1)
(test ((copy quote) quote) 'quote)
(test ((lambda (q) (let ((x 1)) (q x))) quote) 'x) ; these two are strange -- not sure about them, but Guile 1.8 is the same
(test ((lambda (s c) (s c)) quote #f) 'c)
;;; ((lambda (lambda) (lambda (else))) quote) -> '(else)
(test ((quote and) #f) 'error)
(test ((values quote) 1) 1)

;; see also quasiquote




;;; --------------------------------------------------------------------------------
;;; for-each
;;; --------------------------------------------------------------------------------

(test (let ((v (make-vector 5))) (for-each (lambda (i) (vector-set! v i (* i i))) '(0 1 2 3 4)) v) '#(0 1 4 9 16))
(test (let ((ctr 0) (v (make-vector 5))) (for-each (lambda (i) (vector-set! v ctr (* i i)) (set! ctr (+ ctr 1))) '(0 1 2 3 4)) v) '#(0 1 4 9 16))
(for-each (lambda (x) (display "for-each should not have called this")) '())
(test (let ((ctr 0)) (for-each (lambda (x y) (if (= x y) (set! ctr (+ ctr 1)))) '(1 2 3 4 5 6) '(2 3 3 4 7 6)) ctr) 3)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(0 1) '(2 3) '(4 5)) ctr) 15)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(1) '(3) '(5)) ctr) 9)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '() '() '()) ctr) 0)
(test (let () (for-each abs '(1 2)) 1) 1)
(test (let ((ctr 0)) (for-each (lambda (a) (for-each (lambda (b) (set! ctr (+ ctr 1))) '(0 1))) '(2 3 4)) ctr) 6)
(test (let ((sum 0)) (for-each (lambda args (set! sum (+ sum (apply + args)))) '(0 1 2) '(2 1 0) '(3 4 5) '(5 4 3) '(6 7 8) '(8 7 6)) sum) 72)
(test (let ((sum 0)) (for-each (lambda (a b . args) (set! sum (+ sum a b (apply + args)))) '(0 1 2) '(2 1 0) '(3 4 5) '(5 4 3) '(6 7 8) '(8 7 6)) sum) 72)
(test (let ((sum 0)) (for-each (lambda (a b . args) (set! sum (+ sum a b (apply + args)))) '(0 1 2) '(2 1 0)) sum) 6)
(test (let () (for-each + '(0 1 2) '(2 1 0)) 0) 0)
(test (let () () ()) '())
(test (for-each + ()) #<unspecified>)
(test (let ((sum 0)) (for-each (lambda a (set! sum (+ sum (apply + a)))) '(1 2 3)) sum) 6)
(test (let ((sum 0)) (for-each (lambda* ((a 1)) (set! sum (+ sum a))) '(1 2 3)) sum) 6)
(test (let ((sum 0)) (for-each (lambda (a . b) (set! sum (+ sum a))) '(1 2 3)) sum) 6)
(test (let ((sum 0) (lst (list 1 2 3))) (for-each (lambda (a b c) (set! sum (+ sum a b c))) lst lst lst) sum) 18)
(test (let ((sum 0) (lst (vector 1 2 3))) (for-each (lambda (a b c) (set! sum (+ sum a b c))) lst lst lst) sum) 18)

(test (let ((d 0))
	(for-each (let ((a 0))
		    (for-each (lambda (b) (set! a (+ a b))) (list 1 2))
		    (lambda (c) (set! d (+ d c a))))
		  (list 3 4 5))
	d)
      21)
(test (let ((d 0))
	(for-each (lambda (c)
		    (let ((a 0))
		      (for-each (lambda (b) (set! a (+ a b))) (list 1 2))
		      (set! d (+ d a c))))
		  (list 3 4 5))
	d)
      21)

(test (let ((ctr 0)) 
	(let ((val (call/cc 
		    (lambda (exit) 
		      (for-each (lambda (a) 
				  (if (> a 3) (exit a)) 
				  (set! ctr (+ ctr 1))) 
				(list 0 1 2 3 4 5)))))) 
	  (list ctr val)))
      (list 4 4))

(test (call-with-current-continuation
       (lambda (exit)
	 (for-each 
	  (lambda (x) 
	    (if (negative? x) (exit x)))
	  '(54 0 37 -3 245 19))
	 #t))
      -3)

(test (let ((ctr 0)
	    (cont #f)
	    (lst '()))
	(let ((val (call/cc 
		    (lambda (exit) 
		      (for-each (lambda (a) 
				  (if (and (not cont) (= a 2))
				      (exit a)) 
				  (if (and cont (= a 5)) 
				      (exit a))
				  (call/cc (lambda (c) (set! cont c)))
				  (set! lst (cons ctr lst))
				  (set! ctr (+ ctr 1)))
				(list 0 1 2 3 4 5)))))) 
	  (if (< val 5)
	      (cont))
	  (list ctr val lst)))
      (list 5 5 (list 4 3 2 1 0)))

(test (let ((lst '())) 
	(for-each (lambda (a) (set! lst (cons a lst))) 
		  (let ((lst '())) 
		    (for-each (lambda (b) (set! lst (cons b lst))) 
			      (list 1 2 3)) 
		    lst)) 
	lst) 
      (list 1 2 3))

;;; this is an infinite loop?
					; (let ((cont #f)) (call/cc (lambda (x) (set! cont x))) (for-each cont (list 1 2 3)))
(test (call/cc (lambda (x) (for-each x (list 1 2 3)))) 1) ; map also gives 1 ... perhaps not actually legal?

(test (let ((ctr 0))
	(for-each 
	 (lambda (x)
	   (for-each
	    (lambda (x y)
	      (for-each 
	       (lambda (x y z)
		 (set! ctr (+ x y z)))
	       (list x (+ x 1))
	       (list y (+ y 2))
	       (list (+ x y) (- x y))))
	    (list (+ x 3) (+ x 4) (+ x 5))
	    (list (- x 3) (- x 4) (- x 5))))
	 (list 1 2 3 4 5))
	ctr)
      23)

(for-each
 (lambda (a)
   (if (not (string=? a "hi"))
       (format #t "yow: ~S" a)))
 (list "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi"))


;; now some mixed cases
(test (let ((sum 0)) (for-each (lambda (n m) (set! sum (+ sum n m))) (list 1 2) (vector 3 4)) sum) 10)
(test (let ((sum 0)) (for-each (lambda (n m) (set! sum (+ sum n m))) (vector 1 2) (list 3 4)) sum) 10)
(test (let ((sum 0)) (for-each (lambda (n m p) (set! sum (+ sum n m))) (vector 1 2) (list 3 4) (vector 5 6)) sum) 10)
(test (let ((sum 0)) (for-each (lambda (n m p) (if (char=? p #\x) (set! sum (+ sum n m)))) (vector 1 2 3) (list 3 4 5) "xax") sum) 12)

(test (let* ((x (list (list 1 2 3))) (y (apply for-each abs x))) x) '((1 2 3)))

(test (for-each (lambda (x) (display "for-each should not have called this"))) 'error)
(test (for-each (lambda () 1) '()) #<unspecified>)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(1) '(3) '()) ctr) 0)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(0 1) '(2 3) '(4 5 6)) ctr) 15)
(test (for-each (lambda (a b) (+ a b)) (list 1)) 'error)
(test (for-each (lambda (a b) (+ a b)) (list 1) (list)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1)) 'error)
(test (for-each (lambda (a b) (+ a b)) (list 1) (list 2) (list 3)) 'error)
(test (for-each (lambda (a b) (+ a b)) (list 1) (list 1 2)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1 2 3)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1 2) (list)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1 2) (list 1 2)) 'error)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (cons 1 2)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (cons 1 2) (list 1 2)) #<unspecified>)
(test (for-each (lambda (a) (+ a 1)) (list 1) (list 2)) 'error)
(test (for-each (lambda (a) (+ a 1)) #\a) 'error)
(test (for-each (lambda (a) (+ a 1)) (cons 1 2)) #<unspecified>)
(test (let ((sum 0)) (for-each (lambda (a b . args) (set! sum (+ sum a b (apply + args)))) '(0 1 2)) sum) 'error)
(test (for-each (lambda (a) a) '(1 2 . 3)) #<unspecified>)
(test (for-each #(0 1 2) #(2 1 0)) #<unspecified>)
(for-each
 (lambda (arg)
   (test (for-each arg (list 1)) #<unspecified>))
 (list (list 1 2 3) #(1 2 3) "hi"))

(for-each
 (lambda (op)
   (test (for-each op '()) 'error)
   (test (for-each op "") 'error)
   (test (for-each op #(1 2 3) '()) 'error)
   (test (for-each op #() (list) (string)) 'error))
 (list 0 '() #f #t 'a-symbol :hi #\a #<eof> #<unspecified> #<undefined> 0.0 1+i 1/2 1/0 0/0 *stdout* (current-input-port)))
(for-each
 (lambda (arg)
   (test (for-each arg (list 1)) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))
(for-each
 (lambda (arg)
   (test (for-each (lambda (n m) n) (list 1) arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))
(for-each
 (lambda (arg)
   (test (for-each (lambda (a) a) arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))

(test (for-each) 'error)
(test (for-each #t) 'error)
(test (for-each map #t) 'error)

(test (for-each abs '() abs) 'error)
(test (for-each abs '(1) '#(1)) 'error)
(test (let ((vals '())) (for-each for-each (list (lambda (a) (set! vals (cons (abs a) vals)))) (list (list -1 -2))) vals) '(2 1))
(test (let ((c #f)) (for-each (lambda (x) (set! c x)) "a") c) #\a)
(test (let ((c #f)) (for-each (lambda (x) (set! c x)) "") c) #f)
(test (let ((c #f)) (for-each (lambda (x) (set! c x)) (string #\null)) c) #\null)

(test (let ((L (list 1 2 3 4 5)) (sum 0)) (for-each (lambda (x) (set-cdr! (cddr L) 5) (set! sum (+ sum x))) L) sum) 6)
; map (below) has more tests along this line

(test (for-each ="") #<unspecified>)
(test (for-each =""=) 'error)
(test (for-each = "" 123) 'error)
(test (for-each = () 123) 'error)
(test (for-each =()=) 'error)
(test (for-each abs "") #<unspecified>)
(test (for-each null? () #() "") #<unspecified>)
(test (for-each null? () #() 0 "") 'error)
(test (for-each define '(a) '(3)) #<unspecified>)
(test (for-each '(()) #()) #<unspecified>)
(test (for-each '(1 2 . 3) '(1 . 2)) #<unspecified>)
(test (for-each '(()) '()) #<unspecified>)
(test (for-each #2D((1 2) (3 4)) '(1)) #<unspecified>)
(test (for-each "a\x00b" #(1 2)) #<unspecified>)
(test (for-each #(1 (3)) '(1)) #<unspecified>)
(test (for-each '((1 (2)) (((3) 4))) '(1)) #<unspecified>)
(test (for-each "hi" '(1)) #<unspecified>)
(test (for-each #() #()) #<unspecified>)
(test (for-each '(1 . 2) #()) #<unspecified>)
(test (let ((ht (hash-table '(a . 1) '(b . 2)))) (for-each ht ht)) #<unspecified>)
(test (let ((ht (hash-table '(a . 1) '(b . 2)))) (let ((sum 0)) (for-each (lambda (c) (set! sum (+ sum (cdr c)))) ht) sum)) 3)
(test (let ((ht (hash-table '(a . 1) '(b . 2)))) (for-each ht '(a b))) #<unspecified>)
(test (for-each ''2 '(1)) #<unspecified>)
(let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) (test (for-each lst lst) 'error))
(let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) (test (for-each #() lst) 'error))
(test (for-each 1 "hi" '()) 'error)
(test (for-each 0 #() '()) 'error)
(test (for-each #\a #(1 2) '(3 4) "") 'error)

(let ((ctr ((cadr (make-type :getter (lambda (a b) b) :length (lambda (a) (- (expt 2 31) 1)))))) 
      (sum 0))
  (test (call-with-exit 
	 (lambda (go)
	   (for-each (lambda (a) 
		       (set! sum (+ sum a)) 
		       (if (> sum 100) 
			   (go sum))) 
		     ctr)))
	105))

(let ((ctr ((cadr (make-type :getter (lambda (a b) b) :length (lambda (a) most-positive-fixnum)))))
      (sum 0))
  (gc)
  (test (call-with-exit 
	 (lambda (go) 
	   (for-each (lambda (a) 
		       (set! sum (+ sum a)) 
		       (if (> sum 100) 
			   (go sum))) 
		     ctr)))
	105))

(let ((ctr ((cadr (make-type :getter (lambda (a b) (+ b 1))
			     :length (lambda (a) 'hi))))))
  (test (for-each (lambda (x) x) ctr) 'error))

(let ((ctr ((cadr (make-type :getter (lambda (a b) (car b))
			     :length (lambda (a) 4))))))
  (test (for-each (lambda (x) x) ctr) 'error))

(let ((x 0))
  (let ((p1 (make-procedure-with-setter (lambda (a) (set! x (+ x a))) (lambda (a b) (+ a b)))))
    (for-each p1 '(1 2 3))
    (test x 6))
  (set! x 0)
  (for-each (lambda args (set! x (+ x (car args)))) '(1 2 3))
  (test x 6)
  (set! x 0)
  (for-each (lambda* (a (b 2)) (set! x (+ x a))) '(1 2 3))
  (test x 6)
  (set! x 0)
  (for-each (lambda args (set! x (+ x (car args) (cadr args)))) '(1 2 3) '(3 2 1))
  (test x 12)
  (set! x 0)
  (for-each (lambda* (a (b 2)) (set! x (+ x a b))) '(1 2 3) '(3 2 1))
  (test x 12)
  (set! x 0)
  (for-each (lambda* (a (b 2)) (set! x (+ x a b))) '(1 2 3))
  (test x 12))

(test (let ((lst '(1 2 3)) (sum 0)) (define-macro (hi a) `(set! sum (+ sum (+ 1 ,a)))) (for-each hi lst) sum) 9)
  




;;; --------------------------------------------------------------------------------
;;; map
;;; --------------------------------------------------------------------------------

(test (map cadr '((a b) (d e) (g h))) '(b e h))
(test (map (lambda (n) (expt n n)) '(1 2 3 4 5)) '(1 4 27 256 3125))
(test (map + '(1 2 3) '(4 5 6)) '(5 7 9))

(test (apply vector (map (lambda (i) (* i i)) '(0 1 2 3 4))) '#(0 1 4 9 16))
(map (lambda (x) (display "map should not have called this")) '())
(test (let ((ctr 0)) (map (lambda (x y) (if (= x y) (set! ctr (+ ctr 1))) ctr) '(1 2 3 4 5 6) '(2 3 3 4 7 6))) (list 0 0 1 2 2 3))
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '(0 1) '(2 3) '(4 5))) (list 6 15))
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '(1) '(3) '(5))) (list 9))
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '() '() '())) '())
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1 2)) (list 2 4))
(test (map abs '(1 -2)) (list 1 2))
(test (map + '(0 1 2) '(2 1 0) '(3 4 5) '(5 4 3) '(6 7 8) '(8 7 6)) (list 24 24 24))
(test (map (lambda (a) (cons a (map (lambda (b) (+ b 1)) (list 0 1 2)))) (list 3 4 5)) '((3 1 2 3) (4 1 2 3) (5 1 2 3)))
(test (map (lambda (a) (+ a 1)) (map (lambda (b) (+ b 1)) (map (lambda (c) (+ c 1)) (list 0 1 2)))) '(3 4 5))
(test (map (lambda args (apply + args)) '(0 1 2) '(3 4 5) '(6 7 8) '(9 10 11) '(12 13 14)) '(30 35 40))
(test (map (lambda (a b . args) (+ a b (apply + args))) '(0 1 2) '(3 4 5) '(6 7 8) '(9 10 11) '(12 13 14)) '(30 35 40))
(test (map (lambda (a b . args) (+ a b (apply + args))) '(0 1 2) '(3 4 5)) '(3 5 7))
(test (map (lambda args args) '(1 2 3)) '((1) (2) (3)))
(test (map + () ()) ())
(test (map + (#(#() #()) 1)) '())
(test (map + #(1) #(1) #(1)) '(3))
(test (map list '(a b c)) '((a) (b) (c)))
(test (map (lambda (a b) (- a b)) (list 1 2) (vector 3 4)) '(-2 -2))
(test (map (lambda (a b c) (if (char=? a #\a) (+ b c) (- b c))) "axa" (list 1 2 3) (vector 4 5 6)) '(5 -3 9))
(test (map vector (memv 1 (list 1 2 3))) '(#(1) #(2) #(3)))
(test (map append #(1 2 3)) '(1 2 3))
(test (map eval '((+ 1 2) (* 3 4))) '(3 12))

(test (let* ((x (list (list 1 2 3))) (y (apply map abs x))) (list x y)) '(((1 2 3)) (1 2 3)))
(test (let* ((x (quote ((1 2) (3 4)))) (y (apply map ash x))) (list x y)) '(((1 2) (3 4)) (8 32)))
(test (let* ((x (quote ((1 2 3) (4 5 6) (7 8 9)))) (y (apply map + x))) (list x y)) '(((1 2 3) (4 5 6) (7 8 9)) (12 15 18)))
(test (map * (map + '(1 2 3) '(4 5 6)) '(1 2 3)) '(5 14 27))
(test (apply map * (apply map + '(1 2 3) '((4 5 6))) '((1 2 3))) '(5 14 27))
(test (let* ((x (lambda () '(1 2 3))) (y (apply map - (list (x))))) (x)) '(1 2 3))

(test (let ((d 0))
	(map (let ((a 0))
	       (map (lambda (b) (set! a (+ a b))) (list 1 2))
	       (lambda (c) (set! d (+ d c a)) d))
	     (list 3 4 5)))
      (list 6 13 21))
(test (let ((d 0))
	(map (lambda (c)
	       (let ((a 0))
		 (map (lambda (b) (set! a (+ a b))) (list 1 2))
		 (set! d (+ d a c))
		 d))
	     (list 3 4 5)))
      (list 6 13 21))

(test (let ((ctr 0))
	(let ((val (call/cc 
		    (lambda (exit) 
		      (map (lambda (a) 
			     (if (> a 3) (exit a)) 
			     (set! ctr (+ ctr 1))
			     ctr)
			   (list 0 1 2 3 4 5))))))
	  (list ctr val)))
      (list 4 4))

(test (call-with-current-continuation
       (lambda (exit)
	 (map 
	  (lambda (x) 
	    (if (negative? x) (exit x))
	    x)
	  '(54 0 37 -3 245 19))))
      -3)

(test (let ((ctr 0)
	    (cont #f)
	    (lst '()))
	(let ((val (call/cc 
		    (lambda (exit) 
		      (map (lambda (a) 
			     (if (and (not cont) (= a 2))
				 (exit a)) 
			     (if (and cont (= a 5)) 
				 (exit a))
			     (call/cc (lambda (c) (set! cont c)))
			     (set! lst (cons ctr lst))
			     (set! ctr (+ ctr 1))
			     ctr)
			   (list 0 1 2 3 4 5))))))
	  (if (< val 5)
	      (cont))
	  (list ctr val lst)))
      (list 5 5 (list 4 3 2 1 0)))

(let ()
  (define (tree-add x lst)
    (define (tree-add-1 lst-1)
      (map (lambda (a)
	     (if (pair? a) (tree-add-1 a) (+ a x)))
	   lst-1))
    (tree-add-1 lst))
  (test (tree-add 12 '((1 2) ((3)) 4 5)) '((13 14) ((15)) 16 17)))

(test (map (lambda (a) a) (map (lambda (b) b) (list 1 2 3))) (list 1 2 3))
(test (map cons '(a b c) '(() () ())) '((a) (b) (c)))
(test (map (lambda a (list a)) '(1 2 3)) '(((1)) ((2)) ((3))))
(test (map (lambda* a (list a)) '(1 2 3)) '(((1)) ((2)) ((3))))
(test (map (lambda* (a) (list a)) '(1 2 3)) '((1) (2) (3)))
(test (map (lambda* ((a 0)) (list a)) '(1 2 3)) '((1) (2) (3)))
(test (map (lambda* ((a 0) (b 1)) (list a)) '(1 2 3)) '((1) (2) (3)))
(test (map (lambda (a . b) (list a)) '(1 2 3)) '((1) (2) (3)))
(test (map list '(1 2 3)) '((1) (2) (3)))
(test (map (lambda a (apply list a)) '(1 2 3)) '((1) (2) (3)))
(test (map (lambda a (apply values a)) '(1 2 3)) '(1 2 3))
(test (map (lambda a (values a)) '(1 2 3)) '((1) (2) (3)))
(test (map (lambda a (append a)) '(1 2 3)) '((1) (2) (3)))
(test (map values '(1 2 3)) '(1 2 3))
;(test ((lambda* ('a) quote) 1) 1)
(test (procedure? (car (map lambda '(()) '((1))))) #t)
(test (procedure? (car (map lambda '((x)) '(((+ x 1)))))) #t)
(test (map #(0 1 2) #(2 1 0)) '(2 1 0))
(test (map quasiquote '((quasiquote 1) (quasiquote 2))) '(1 2))
(test (map (lambda (a b) (a b)) (map lambda '((x) (y) (z)) '((+ x x) (* y y) (expt z z))) (list 1 2 3)) '(2 4 27))
(test (map apply (map lambda '((x) (y) (z)) '((+ x x) (* y y) (expt z z))) '((1) (2) (3))) '(2 4 27))

#|
(let ((val '())) (list (map (lambda a (set! val (cons a val)) a) '(1 2 3)) val))
((#3=(1) #2=(2) #1=(3)) (#1# #2# #3#))
|#

(test (map list "hi") '((#\h) (#\i)))
(test (map string "hi") '("h" "i"))
(test (map vector "hi") '(#(#\h) #(#\i)))
(test (map char-upcase "hi") '(#\H #\I))
(test (map append #(#() #())) '(#() #()))

(test (map abs '() abs) 'error)
(test (map (lambda (x) (display "map should not have called this"))) 'error)
(test (map (lambda () 1) '()) '())
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '(1) '(3) '())) '())
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z))) '(0 1) '(2 3) '(4 5 6))) '(6 15))

(test (map (lambda (a b) (+ a b)) (list 1)) 'error)
(test (map (lambda (a b) (+ a b)) (list 1) (list)) '())
(test (map (lambda (a b) (+ a b)) (list 1) (list 2)) (list 3))
(test (map (lambda (a b) (+ a b)) (list 1)) 'error)
(test (map (lambda (a b) (+ a b)) (list 1) (list 2) (list 3)) 'error)
(test (map (lambda (a b) (+ a b)) (list 1) (list 1 2)) '(2))
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1)) '(2))
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1 2 3)) '(2 4))
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1)) '(2))
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1 2) (list)) '())
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1 2) (list 1 2)) 'error)
(test (map (lambda (a b) (+ a b)) (list 1 2) (cons 1 2)) '(2))

(test (map (lambda . (x y z 8)) '(1 2 3))  'error) ; (y unbound) but other schemes ignore unused args
(test (map (lambda . (x 8)) '(1 2)) '(8 8)) 

(test (map (lambda (a) (+ a 1)) (list 1) (list 2)) 'error)
(test (map (lambda (a) (+ a 1)) #\a) 'error)
(test (map (lambda (a) (+ a 1)) (cons 1 2)) '(2))
(test (map (lambda (a b . args) (+ a b (apply + args))) '(0 1 2)) 'error)
(test (map (lambda (a) a) '(1 2 . 3)) '(1 2))
(test (map) 'error)
(test (map #t) 'error)
(test (map set-cdr! '(1 2 3)) 'error)
(test (map (lambda (a b) (set-cdr! a b) b) '((1) (2) (3)) '(4 5 6)) '(4 5 6))
(test (let ((str "0123")) (set! (str 2) #\null) (map append str)) '(#\0 #\1 #\null #\3))

(test (map ((lambda () abs)) '(-1 -2 -3)) '(1 2 3))
(test (apply map ((lambda () abs)) (list (list -1 -2 -3))) '(1 2 3))
(test (apply apply map ((lambda () abs)) (list (list (list -1 -2 -3)))) '(1 2 3))
(test (apply apply apply map ((lambda () abs)) '((((-1 -2 -3))))) '(1 2 3))
(test (apply apply apply (list (list map abs (list (list -1 -2 -3))))) '(1 2 3))
(test (apply + (apply apply apply (list (list map abs (list (list -1 -2 -3)))))) 6)
(test (apply map vector (values (list (vector 1 2)))) '(#(1) #(2)))
(test (apply map string (list "123")) '("1" "2" "3"))
(test (map string "123") '("1" "2" "3"))
(test (map "hi" '(0 1)) '(#\h #\i))
(test (map (list 2 3) '(0 1)) '(2 3))
(test (map #(2 3) '(1 0)) '(3 2))
(for-each
 (lambda (arg)
   (test (map arg (list 1)) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))
(for-each
 (lambda (arg)
   (test (map (lambda (n m) n) (list 1) arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))
(for-each
 (lambda (arg)
   (test (map (lambda (a) a) arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))

(test (map (lambda (a1 a2 a3 a4 a5 a6 a7 a8 a9 a10)
	     (max a1 a2 a3 a4 a5 a6 a7 a8 a9 a10))
	   (list 6 7 8 9 10)
	   (list 21 22 23 24 25)
	   (list 16 17 18 19 20)
	   (list 11 12 13 14 15)
	   (list 26 27 28 29 30)
	   (list 1 2 3 4 5)
	   (list 36 37 38 39 40)
	   (list 41 42 43 44 45)
	   (list 46 47 48 49 50)
	   (list 31 32 33 34 35))
      (list 46 47 48 49 50))
  
(test (map map (list abs) (list (list -1))) '((1)))
(test (map map (list map) (list (list abs)) (list (list (list -1)))) '(((1))))
(test (map map (list map) (list (list map)) (list (list (list abs))) (list (list (list (list -1 -3))))) '((((1 3)))))
(test (map map (list lcm) (vector #(1 2))) '((1 2)))
(test (map map (list integer?) (list (vector "hi" 1 2/3))) '((#f #t #f)))
(test (map map (list char-lower-case?) (list "hAba")) '((#t #f #t #t)))
(test (map map (list char-lower-case? char-upper-case?) (list "hAba" "HacDf")) '((#t #f #t #t) (#t #f #f #t #f)))
(test (map map (list + -) (list (list 1 2) (list 3 4))) '((1 2) (-3 -4)))
(test (map map (list map map) (list (list + -) (list - +)) '(((1 2) (3 4)) ((4 5) (6 7)))) '(((1 2) (-3 -4)) ((-4 -5) (6 7))))

(test (map member (list 1 2 3) (list (list 1 2 3) (list 1 3 4) (list 3 4 5))) '((1 2 3) #f (3 4 5)))
(test (map - (list 1 2 3) (list 1 2 3) (list 1 3 4) (list 3 4 5)) '(-4 -7 -9))
(test (map - (list 1 2 3) (list 1 2 3 'hi) (list 1 3 4 #\a "hi") (list 3 4 5)) '(-4 -7 -9))
(test (let () (define (mrec a b) (if (<= b 0) (list a) (map mrec (list a) (list (- b 1))))) (mrec (list 1 2) 5)) '(((((((1 2))))))))
(test (map append '(3/4)) '(3/4))
(test (map list '(1.5)) '((1.5)))
(test (map vector '("hi")) '(#("hi")))
(test (map object->string '(:hi (1 2) (()))) '(":hi" "(1 2)" "(())"))
(test (map map (list for-each) (list (list abs)) (list (list (list 1 2 3)))) '((#<unspecified>)))
(test (map map (list vector) '((#(1 #\a (3))))) '((#(#(1 #\a (3))))))
(test (apply map map (list cdr) '((((1 2) (3 4 5))))) '(((2) (4 5))))
(test (apply map map (list char-upcase) '(("hi"))) '((#\H #\I)))
(test (apply map map (list *) '(((1 2)) ((3 4 5)))) '((3 8))) ; (* 1 3) (* 2 4)
(test (map apply (list map) (list map) (list (list *)) '((((1 2)) ((3 4 5))))) '(((3 8))))
(test (map map (list magnitude) '((1 . 2))) '((1))) ; magnitude is called once with arg 1
(test (map magnitude '(1 . 2)) '(1))
(test (map call/cc (list (lambda (r1) 1) (lambda (r2) (r2 2 3)) (lambda (r3) (values 4 5)))) '(1 2 3 4 5))
(test (map call/cc (list number? continuation?)) '(#f #t))

;; from scheme working group 
(test (let ((L (list 1 2 3 4 5))) (map (lambda (x) (set-cdr! (cddr L) 5) x) L)) '(1 2 3))
(test (let ((L (list 1 2))) (map (lambda (x) (set! (cdr (cdr L)) L) x) L)) '(1 2))
(test (let ((L (list 1 2))) (object->string (map (lambda (x) (set! (car (cdr L)) L) x) L))) "(1 #1=(1 #1#))")
;;;(test (let ((L (list 1 2))) (map (lambda (x) (set-cdr! L L) x) L)) '(1 2)) ;?? this depends on when we cdr? infinite loop in Guile
;;;(let ((L (list 1 2 3 4 5))) (map (lambda (x) (set-cdr! L '()) x) L)) ; another similar case -- s7 doesn't notice what happened
;;;  does that mean a GC during this map would leave us accessing freed memory? 
;;;  I think not because the original list is held by map (eval) locals that are protected
;;;  we simply stepped on something after looking at it, similar to:
(test (let ((L (list 1 2 3 4 5))) (map (lambda (x) (set-car! L 123) x) L)) '(1 2 3 4 5))
(test (let ((L (list 1 2 3 4 5))) (map (lambda (x) (set-cdr! (cddr L) (list 6 7 8)) x) L)) '(1 2 3 6 7))
;;; we could do something similar with strings:
(test (let ((S "12345")) (map (lambda (x) (set! (S 2) #\null) x) S)) '(#\1 #\2 #\null #\4 #\5))
;;; (length S) is still 5 even with the embedded null
(test (let ((L (list 1 2 3))) (map (lambda (x) (set! L (list 6 7 8)) x) L)) '(1 2 3))
(test (let ((L1 (list 1 2 3)) (L2 (list 4 5 6 7))) (map (lambda (x1 x2) (set-cdr! (cdr L1) '()) (cons x1 x2)) L1 L2)) '((1 . 4) (2 . 5)))
(test (let ((L (list 1 2 3))) (map (lambda (x) (set-car! (cddr L) 32) x) L)) '(1 2 32))
;;; should these notice the increased length?:
(test (let ((L1 (list 1 2)) (L2 (list 6 7 8 9))) (map (lambda (x y) (set-cdr! (cdr L1) (list 10 11 12 13 14)) (cons x y)) L1 L2)) '((1 . 6) (2 . 7)))
(test (let ((L1 (list 1)) (L2 (list 6 7 8))) (map (lambda (x y) (set-cdr! L1 (list 10 11 12)) (cons x y)) L1 L2)) '((1 . 6)))
(test (let ((L1 (list 1 2))) (map (lambda (x) (set-cdr! (cdr L1) (list 10 11 12)) x) L1)) '(1 2))
;;; a similar case could be made from hash-tables
(test (let ((H (hash-table '(a . 3) '(b . 4)))) (map (lambda (x) (set! (H 'c) 32) (cdr x)) H)) '(3 4))
(test (let ((H (hash-table '(a . 3) '(b . 4)))) (map (lambda (x) (set! (H 'b) 32) (cdr x)) H)) '(3 32))

;; in that 1st example, the set-cdr! is not the problem (map supposedly can treat its args in any order),
;;   any set! will do:
(test (let ((x 0)) (map (lambda (y) (set! x (+ x y)) x) '(1 2 3 4))) '(1 3 6 10))

(test (map begin '(1 2 3)) '(1 2 3))
(let ((funcs (map (lambda (lst) (eval `(lambda ,@lst))) '((() #f) ((arg) (+ arg 1))))))
  (test ((car funcs)) #f)
  (test ((cadr funcs) 2) 3))

(test (map = #() =) 'error)
(test (map ="") '())
(test (map abs ()) ())
(test (map abs "") ())
(test (map abs "123" "") ()) ; should this be an error -- arity is wrong?
(test (map abs "123" "" #f) 'error)
(test (map null? () #() "") ())
(test (map null? () #() 0 "") 'error)
(test (map '(()) #()) '())
(test (map '(1 2 . 3) '(1 . 2)) '(2))
(test (map '(()) '()) '())
(test (map #2D((1 2) (3 4)) '(1)) '(#(3 4)))
(test (map "a\x00b" #(1 2)) '(#\null #\b))
(test (map #(1 (3)) '(1)) '((3)))
(test (map '((1 (2)) (((3) 4))) '(1)) '((((3) 4))))
(test (map "hi" '(1)) '(#\i))
(test (map #() #()) '())
(test (map '(1 . 2) #()) '())
(test (map ''2 '(1)) '(2))
(test (((map lambda '((x)) '(1 2 . 3)) 0) 0) 1)
(test (((map lambda '(()) #(1 2)) 0)) 1)
(test (((map lambda '((x)) '((+ x 1))) 0) 32) 33)
(test (map #() '()) '()) ; hmmm -- (map '() '()) is an error
(test (map "" "") '())
(test (map (let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) lst) '(0)) '(1))
(let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) (test (map lst lst) 'error))
(test (map 1 "hi" '()) 'error)
(test (map 0 #() '()) 'error)
(test (map #\a #(1 2) '(3 4) "") 'error)
(test (map or '(1 2 . 3)) '(1 2))
(test (map or "a\x00b") '(#\a #\null #\b))
(test (map cond '((1 2) (3 4))) '(2 4)) ; (cond (1 2)) -> 2
(test (map begin "hi") '(#\h #\i))
(test (map quote "hi") '(#\h #\i))
(test (map (begin #(1 (3))) '(1)) '((3)))
(test (map (''2 0) ''2) 'error)
(test (map (apply lambda 'a '(-1)) '((1 2))) '(-1))
(test (map do '(()) '((1 2))) '(2))
(test (map case '(1) '(((-1 1) 2) 3)) '(2))
(test (map let '(()) "a\x00b") '(#\a))
(test (map "hi" '(0 1) '(0 1)) 'error)
(test (map '((1 2) (3 4)) '(0 1) '(0 1)) '(1 4))
(test (map #2d((1 2) (3 4)) '(0 1) '(0 1)) '(1 4))
(test (map #2d((1 2) (3 4)) '(0 1)) '(#(1 2) #(3 4)))
(let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) (test (map (lambda (a) a) lst) 'error))
(let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) (test (map (lambda (a) a) lst lst) 'error))
(test (map "hi" ('((1)) 0)) '(#\i))
(test (map "hi" ('((1 0)) 0)) '(#\i #\h))
(test (let ((ht (hash-table '(a . 1) '(b . 2)))) (map ht ht)) '(#f #f))
(test (let ((ht (hash-table '(a . 1) '(b . 2)))) (let ((lst (map (lambda (c) (cdr c)) ht))) (or (equal? lst '(1 2)) (equal? lst '(2 1))))) #t)
(test (let ((ht (hash-table '(a . 1) '(b . 2)))) (map ht '(a b))) '(1 2))

(let ((pws (make-procedure-with-setter (lambda (a) a) (lambda (a b) b))))
  (test (map append pws) 'error)
  (test (map pws '(1 2 3)) '(1 2 3)))

(test (map abs '(1 2 . 3)) '(1 2)) ;; ?? Guile says wrong type arg here
(test (map + '(1) '(1 2 . 3)) '(2))
(test (map abs '(1 . 2)) '(1))
;; problematic because last thing is completely ignored:
(test (map abs '(1 . "hi")) '(1))
(test (map floor '(1 . "hi")) '(1))

(let ((ctr ((cadr (make-type :getter (lambda (a b) b) :length (lambda (a) (- (expt 2 31) 1)))))) 
      (sum 0))
  (test (call-with-exit 
	 (lambda (go)
	   (map (lambda (a) 
		  (set! sum (+ sum a)) 
		  (if (> sum 100) 
		      (go sum))
		  sum)
		ctr)))
	105))

(let ((ctr ((cadr (make-type :getter (lambda (a b) (+ b 1))
			     :length (lambda (a) 'hi))))))
  (test (map (lambda (x) x) ctr) 'error))

(let ((ctr ((cadr (make-type :getter (lambda (a b) (car b))
			     :length (lambda (a) 4))))))
  (test (map (lambda (x) x) ctr) 'error))

(for-each
 (lambda (op)
   (test (map op '()) 'error)
   (test (map op "") 'error)
   (test (map op #() (list) (string)) 'error))
 (list 0 '() #f #t 'a-symbol :hi #\a #<eof> #<unspecified> #<undefined> 0.0 1+i 1/2 1/0 0/0 *stdout* (current-input-port)))

(test (map append (make-vector (list 2 0))) '())
(let ((p1 (make-procedure-with-setter (lambda (a) (+ a 1)) (lambda (a b) (+ a b)))))
  (test (map p1 '(1 2 3)) '(2 3 4)))
(test (map (lambda args (+ (car args) 1)) '(1 2 3)) '(2 3 4))
(test (map (lambda* (a (b 2)) (+ a 1)) '(1 2 3)) '(2 3 4))
(let ((p1 (make-procedure-with-setter (lambda (a b) (+ a b)) (lambda (a b c) (+ a b c)))))
  (test (map p1 '(1 2 3) '(3 2 1)) '(4 4 4)))
(test (map (lambda args (+ (car args) (cadr args))) '(1 2 3) '(3 2 1)) '(4 4 4))
(test (map (lambda* (a (b 2)) (+ a b)) '(1 2 3) '(3 2 1)) '(4 4 4))
(test (map (lambda* (a (b 2)) (+ a b)) '(1 2 3)) '(3 4 5))
(test (map (lambda* ((a 1) (b (map (lambda (c) (+ c 1)) (list 1 2)))) (+ a (apply + b))) (list 4 5 6)) '(9 10 11))
(test (let ((lst (list 0 1 2))) (map (lambda* ((a 1) (b (for-each (lambda (c) (set! (lst c) (+ (lst c) 1))) (list 0 1 2)))) a) lst)) '(0 2 4))

(test (let ((lst '(1 2 3))) (define-macro (hiho a) `(+ 1 ,a)) (map hiho lst)) '(2 3 4))
(test (let ((lst '(1 2 3))) (define-macro (hiho a b) `(+ 1 ,a (* 2 ,b))) (map hiho lst lst)) '(4 7 10))
(test (let ((lst '(1 2 3))) (define-macro (hi1 a) `(+ 1 ,a)) (define-macro (hiho a b) `(+ 1 ,a (* 2 ,b))) (map hiho lst (map hi1 lst))) '(6 9 12))
(test (let ((lst '(1 2 3))) (define-macro (hiho a b) `(+ 1 ,a (* 2 ,b))) (map hiho lst (map (symbol->value (define-macro (hi1 a) `(+ 1 ,a))) lst))) '(6 9 12))
(test (let ((lst '(1 2 3))) (define-macro (hi a) `(+ 1 ,a)) (define-macro (ho b) `(+ 1 (hi ,b))) (map ho lst)) '(3 4 5))
(test (let ((lst '(1 2 3))) (define-macro* (hi a (b 2)) `(+ 1 ,a (* 2 ,b))) (map hi lst)) '(6 7 8))
(test (let ((lst '(1 2 3))) (define-macro* (hi a (b 2)) `(+ 1 ,a (* 2 ,b))) (map hi lst (map hi lst))) '(14 17 20))

(let ()
  (define (hi)
    (map (lambda (a) (a 0))
	 (list (vector 1 2 3)
	       (string #\a #\b #\c)
	       (list 'e 'f 'g))))
  (test (hi) '(1 #\a e)))


#|
;;; this is from the r6rs comment site
(let ((resume #f)
       (results '()))
   (set! results
         (cons (map (lambda (x)
                      (call/cc (lambda (k)
                                 (unless resume (set! resume k))
                                 0)))
                    '(#f #f))
               results ))
   (display results)(newline)
   (if resume
       (let ((resume* resume))
         (set! resume #f)
         (resume* 1))))

With a careful implementation of MAP, a new list is returned every
time, so that the displayed results are

   ((0 0))
   ((1 0) (0 0))
   ((1 1) (1 0) (0 0))
|#





;;; --------------------------------------------------------------------------------
;;; do
;;; --------------------------------------------------------------------------------

(test (do () (#t 1)) 1)
(for-each
 (lambda (arg)
   (test (do () (#t arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (do ((i arg)) (#t i)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))

(test (do ((i 0 (+ i 1))) ((= i 3) #f)) #f)
(test (do ((i 0 (+ i 1))) ((= i 3) i)) 3)
(test (do ((vec (make-vector 5)) (i 0 (+ i 1))) ((= i 5) vec) (vector-set! vec i i)) '#(0 1 2 3 4))
(test (let ((x '(1 3 5 7 9))) (do ((x x (cdr x)) (sum 0 (+ sum (car x))))  ((null? x) sum))) 25)
(test (do ((i 4 (- i 1)) (a 1 (* a i))) ((zero? i) a)) 24)
(test (do ((i 2 (+ i 1))) ((> i 0) 123)) 123)

(test (do () (() ()) ()) '())
(test (do () ('() '())) '())
(test (do () ('())) '())
(test (do () (())) '())
(test (do) 'error)

(test (let ((x 0) (y 0)) (set! y (do () (#t (set! x 32) 123))) (list x y)) (list 32 123))
(test (let ((i 32)) (do ((i 0 (+ i 1)) (j i (+ j 1))) ((> j 33) i))) 2)
(test (let ((i 0)) (do () ((> i 1)) (set! i (+ i 1))) i) 2)
(test (let ((i 0) (j 0)) (do ((k #\a)) (#t i) (set! i (char->integer k)) (set! j (+ j i)))) 0)
(test (let ((i 0) (j 0)) (do ((k #\a)) ((> i 1) j) (set! i (char->integer k)) (set! j (+ j i)))) (char->integer #\a))
(test (let ((x 0)) (do ((i 0 (+ i 2)) (j 1 (* j 2))) ((= i 4) x) (set! x (+ x i j)))) 5)
(test (let ((sum 0)) (do ((lst '(1 2 3 4) (cdr lst))) ((null? lst) sum) (set! sum (+ sum (car lst))))) 10)
(test (do ((i 0 (+ 1 i))) ((= i 4) (do ((i 0 (+ i 2))) ((= i 10) i)))) 10)
(test (let ((i 0)) (do ((i 1 (+ i 1))) ((= i 3) i))) 3)
(test (let ((j 0)) (do ((i 0 (+ i 1))) ((= i 3) (+ i j)) (do ((j 0 (+ j i 1))) ((> j 3) j)))) 3)
(test (let ((add1 (lambda (a) (+ a 1)))) (do ((i 0 (add1 i))) ((= i 10) (add1 i)))) 11)
(test (do ((i 0 (do ((j 0 (+ j 1))) ((= j i) (+ i 1))))) ((= i 3) i)) 3)
(test (do ((i 0 (do ((i 0 (+ i 1))) ((= i 3) i)))) ((= i 3) i)) 3)
(test (let ((i 123)) (do ((i 0 (+ i 1)) (j i (+ j i))) ((> j 200) i))) 13)
(test (do ((i 0 (+ i 1))) ((> i 3) i) (set! i (* i 10))) 11)
(test (do ((i 123) (j 0 (+ j i))) ((= j 246) i)) 123)
(test (do ((i 123 i) (j 0 (+ j i))) ((= j 246) i)) 123)
(test (do ((i 0 i)) (i i)) 0)
(test (do ((i 1 i)) (i i (+ i i) (+ i i i))) 3)
(test (do ((i 1)) (#t 1) 123) 1)
(test (do ((i 0 (+ i j)) (j 0 (+ j 1))) (#t 1)) 1)
(test (do ((i 0 j) (j 0 (+ j 1))) ((= j 3) i)) 2) ; uh, lessee... lexical scoping...
(test (do ((i 1 j) (j 0 k) (k 0 m) (m 0 (+ i j k))) ((> m 10) (list i j k m))) (list 4 5 8 11))
(test (let ((i 10) (j 11) (k 12)) (do ((i i j) (j j k) (k k m) (m (+ i j k) (+ i j k))) ((> m 100) (list i j k m)))) (list 33 56 78 122))
(test (do ((i 0 (let () (set! j 3) (+ i 1))) (j 0 (+ j 1))) ((= i 3) j)) 4)
(test (let ((i 0)) (do () ((= i 3) (* i 2)) (set! i (+ i 1)))) 6)
(num-test (do ((i 0 (- i 1))) ((= i -3) i)) -3)
(num-test (do ((i 1/2 (+ i 1/2))) ((> i 2) i)) 5/2)
(num-test (do ((i 0.0 (+ i 0.1))) ((>= i 0.9999) i)) 1.0)
(num-test (do ((i 0 (- i 1/2))) ((< i -2) i)) -5/2)
(num-test (do ((i 0+i (+ i 0+i))) ((> (magnitude i) 2) i)) 0+3i)
(test (let ((x 0)) 
	(do ((i 0 (+ i 1)))
	    ((> i 4) x) 
	  (set! x (+ x i))
	  (set! i (+ i 0.5))))
      4.5)
(test (do ((i 0 1)) ((> i 0) i)) 1)
(test (do ((i 1.0+i 3/4)) ((= i 3/4) i)) 3/4)
(test (do ((i 0 "hi")) ((not (number? i)) i)) "hi")
(test (do ((i "hi" 1)) ((number? i) i)) 1)
(test (do ((i #\c "hi")) ((string? i) i)) "hi")
(test (do ((i #\c +)) ((not (char? i)) i)) +)
(test (let ((j 1)) (do ((i 0 j)) ((= i j) i))) 1)
(test (let ((j 1)) (do ((i 0 j)) ((= i j) i) (set! j 2))) 2)
(test (do ((j 1 2) (i 0 j)) ((= i j) i)) 2)
(test (let ((old+ +) (j 0)) (do ((i 0 (old+ i 1))) ((or (< i -3) (> i 3))) (set! old+ -) (set! j (+ j i))) j) -6)
(test (let ((old+ +) (j 0)) (do ((i 0 (+ i 1))) ((or (< i -3) (> i 3))) (set! + -) (set! j (old+ j i))) (set! + old+) j) -6)
(test (do ((i 0 (case i ((0) 1) ((1) "hi")))) ((string? i) i)) "hi")
(test (do ((i if +)) ((equal? i +) i)) +)
(test (let ((k 0)) (do ((j 0 (+ j 1)) (i 0 ((if (= i 0) + -) i 1))) ((= j 5)) (set! k (+ k i))) k) 2)
(test (let ((j -10) (k 0)) (do ((i 0 (+ i j)) (j 2)) ((> i 4) k) (set! k (+ k i)))) 6)
(test (let ((j -10) (k 0)) (do ((i j (+ i j)) (j 2)) ((> i 4) k) (set! k (+ k i)))) -24)
(test (let ((j -10) (k 0)) (do ((i j (+ i j)) (j 2)) ((= i j) k) (set! k (+ k i)))) -30)
(test (let ((j -10) (k 0)) (do ((i j (+ i j)) (j 2)) ((= i j) j) (set! k (+ k i)))) 2)
(test (let ((equal =)) (do ((i 0 (+ i 1))) ((equal i 3) i))) 3)
(test (let ((equal =)) (do ((i 0 (+ i 1))) ((equal i 3) i) (set! equal >))) 4)
(test (do ((equal =) (i 0 (+ i 1))) ((equal i 3) i)) 3)
(test (do ((equal = >) (i 0 (+ i 1))) ((equal i 3) i)) 4)
(test (do ((j 0) (plus + -) (i 0 (plus i 1))) ((= i -1) j) (set! j (+ j 1))) 3)
(test (let ((expr `(+ i 1))) (do ((j 0) (i 0 (eval expr))) ((= i 3) j) (set! j (+ j 1)))) 3)
(test (let ((expr `(+ i 1))) (do ((j 0) (i 0 (eval expr))) ((= i -3) j) (set! j (+ j 1)) (if (= j 3) (set! expr `(- i 1))))) 7)
(test (do ((i 0 (+ i 1))) ((or (= i 12) (not (number? i)) (> (expt 2 i) 32)) (expt 2 i))) 64)
(test (let ((k 0)) (do ((i 0 (+ i 1))) ((let () (set! k (+ k 1)) (set! i (+ i 1)) (> k 3)) i))) 7)
(num-test (do ((i 0 (+ i 1))) ((> i 3) i) (set! i (* .9 i))) 3.439)
(test (let ((v #(0 0 0))) (do ((i 0 (+ i 1))) ((= i 3) v) (set! (v i) i))) #(0 1 2))
(test (let ((v (list 0 0 0))) (do ((i 0 (+ i 1))) ((= i 3) v) (set! (v i) i))) '(0 1 2))
(test (let ((sum 0)) ((do ((i 0 (+ i 1))) ((> i 64) (lambda () sum)) (set! sum (+ sum i))))) 2080)
(test (do ((lst '() (cons i lst)) (i 0 (+ i 1))) ((> i 6) (reverse lst))) '(0 1 2 3 4 5 6))

(test (let ((lst '(1 2 3))
	    (v (vector 0 0 0)))
	(do ((l lst (map (lambda (a) (+ a 1)) (cdr l))))
	    ((null? l))
	  (set! (v (- (length l) 1)) (apply + l)))
	v)
      #(5 7 6))

(test (let ((lst '(1 2 3)))
	(map (lambda (a)
	       (let ((! 1))
		 (do ((i 0 (+ i 1))
		      (sum 0))
		     ((= i a) sum)
		   (set! sum (+ sum a)))))
	     lst))
      '(1 4 9))

(test (let ((sum 0)) (do ((i_0 0 (+ i_0 0))(i_1 1 (+ i_1 1))(i_2 2 (+ i_2 2))(i_3 3 (+ i_3 3))(i_4 4 (+ i_4 4))(i_5 5 (+ i_5 5))(i_6 6 (+ i_6 6))(i_7 7 (+ i_7 7))(i_8 8 (+ i_8 8))(i_9 9 (+ i_9 9))(i_10 10 (+ i_10 10))(i_11 11 (+ i_11 11))(i_12 12 (+ i_12 12))(i_13 13 (+ i_13 13))(i_14 14 (+ i_14 14))(i_15 15 (+ i_15 15))(i_16 16 (+ i_16 16))(i_17 17 (+ i_17 17))(i_18 18 (+ i_18 18))(i_19 19 (+ i_19 19))(i_20 20 (+ i_20 20))(i_21 21 (+ i_21 21))(i_22 22 (+ i_22 22))(i_23 23 (+ i_23 23))(i_24 24 (+ i_24 24))(i_25 25 (+ i_25 25))(i_26 26 (+ i_26 26))(i_27 27 (+ i_27 27))(i_28 28 (+ i_28 28))(i_29 29 (+ i_29 29))(i_30 30 (+ i_30 30))(i_31 31 (+ i_31 31))(i_32 32 (+ i_32 32))(i_33 33 (+ i_33 33))(i_34 34 (+ i_34 34))(i_35 35 (+ i_35 35))(i_36 36 (+ i_36 36))(i_37 37 (+ i_37 37))(i_38 38 (+ i_38 38))(i_39 39 (+ i_39 39)))
    ((= i_1 10) sum)
  (set! sum (+ sum i_0 i_1 i_2 i_3 i_4 i_5 i_6 i_7 i_8 i_9 i_10 i_11 i_12 i_13 i_14 i_15 i_16 i_17 i_18 i_19 i_20 i_21 i_22 i_23 i_24 i_25 i_26 i_27 i_28 i_29 i_30 i_31 i_32 i_33 i_34 i_35 i_36 i_37 i_38 i_39))))
      35100)

(let () (define (jtest) (let ((j 0)) (do ((i 0 (+ i 1))) ((= i 10) j) (if (= i 3) (set! j i))))) (test (jtest) 3))
(let () (define (jtest1) (let ((j (vector 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (set! (j 0) i))))) (test (jtest1) 3))
(let () (define (jtest2) (let ((j (vector 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (vector-set! j 0 i))))) (test (jtest2) 3))
(let () (define (jtest3) (let ((j (vector 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (set! (vector-ref j 0) i))))) (test (jtest3) 3))
(let () (define (jtest4) (let ((j (list 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (set! (j 0) i))))) (test (jtest4) 3))
(let () (define (jtest5) (let ((j (list 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (set! (car j) i))))) (test (jtest5) 3))
(let () (define (jtest6) (let ((j (list 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (set-car! j i))))) (test (jtest6) 3))
(let () (define (jtest7) (let ((j (list 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (list-set! j 0 i))))) (test (jtest7) 3))
(let () (define (jtest8) (let ((j #f)) (do ((i 0 (+ i 1))) ((= i 10) (car j)) (if (= i 3) (set! j (list i)))))) (test (jtest8) 3))
(let () (define (jtest9) (let ((j #f)) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (set! j (vector i)))))) (test (jtest9) 3))
(let () (define (jtest10) (let ((j (cons 1 2))) (do ((i 0 (+ i 1))) ((= i 10) j) (if (= i 3) (set-car! j i))))) (test (jtest10) '(3 . 2)))
(let () (define (jtest11) (let ((j (cons 1 2))) (do ((i 0 (+ i 1))) ((= i 10) j) (if (= i 3) (set! j (cons 0 i)))))) (test (jtest11) '(0 . 3)))
;; (let ((f #f)) (define (jtest12) (do ((i 0 (+ i 1))) ((= i 10) (f)) (if (= i 3) (set! f (lambda () i))))) (test (jtest12) 3))
;; this lambda business is a separate issue

(test (let () (define (step-it a) (+ a 1)) (define (hi) (do ((i 0 (step-it i))) ((= i 3) i))) (hi) (hi)) 3)

(test (call-with-exit (lambda (return) (do () () (if #t (return 123))))) 123)
(test (call-with-exit (lambda (return) (do () (#f) (if #t (return 123))))) 123)
(test (call-with-exit (lambda (return) (do ((i 0 (+ i 1))) () (if (= i 100) (return 123))))) 123)
(test (call-with-exit (lambda (return) (do () ((return 123))))) 123)
(test (call-with-exit (lambda (return) (do () (#t (return 123))))) 123)

(test (do () (/ 0)) 0)
(test (do () (+)) '())
(test (do () (+ +) *) +)

(if with-bignums
    (begin
      (num-test (do ((i 24444516448431392447461 (+ i 1))
		     (j 0 (+ j 1)))
		    ((>= i 24444516448431392447471) j))
		10)
      (num-test (do ((i 0 (+ i 24444516448431392447461))
		     (j 0 (+ j 1)))
		    ((>= i 244445164484313924474610) j))
		10)
      (num-test (do ((i 4096 (* i 2))
		     (j 0 (+ j 1)))
		    ((= i 4722366482869645213696) j))
		60)))

(test (do ((i 9223372036854775805 (+ i 1))
	   (j 0 (+ j 1)))
	  ((>= i 9223372036854775807) j))
      2)
(test (do ((i -9223372036854775805 (- i 1))
	   (j 0 (+ j 1)))
	  ((<= i -9223372036854775808) j))
      3)

(num-test (do ((x (list 1 2 3) (cdr x)) (j -1)) ((null? x) j) (set! j (car x))) 3)

(test (let ((x 0)) 
	(do ((i 0 (+ i 1)))
	    ((= i (do ((j 0 (+ j 1))) ((= j 2) (+ j 1)))))
	  (set! x (+ x i)))
	x)
      3)
(test (let ((x 0)) 
	(do ((i 0 (+ i (do ((j 0 (+ j 1))) ((= j 2) 1)))))
	    ((= i 3) x)
	  (set! x (+ x i))))
      3)
(test (let ((x 0)) 
	(do ((i 0 (+ i (do ((j 0 (+ j 1))) ((= j 2) 1)))))
	    ((= i 3) (do ((j 0 (+ j 1))) ((= j 5) x) (set! x j)))
	  (set! x (+ x i))))
      4)

(test (call-with-exit (lambda (exit) (do ((i 0 (+ i 1))) ((= i 100) i) (if (= i 2) (exit 321))))) 321)
(test (call-with-exit (lambda (exit) (do ((i 0 (if (= i 3) (exit 321) (+ i 1)))) ((= i 100) i)))) 321)
(test (call-with-exit (lambda (exit) (do ((i 0 (+ i 1))) ((= i 10) (exit 321))))) 321)
(test (call-with-exit (lambda (exit) (do ((i 0 (+ i 1))) ((= i 10) i) (if (= i -2) (exit 321))))) 10)
(test (do ((x 0 (+ x 1)) (y 0 (call/cc (lambda (c) c)))) ((> x 5) x) #f) 6)
(test (let ((happy #f)) (do ((i 0 (+ i 1))) (happy happy) (if (> i 3) (set! happy i)))) 4)

(test (+ (do ((i 0 (+ i 1))) ((= i 3) i)) (do ((j 0 (+ j 1))) ((= j 4) j))) 7)
(test (do ((i (if #f #f))) (i i)) (if #f #f))
(test (do ((i (if #f #f)) (j #f i)) (j j)) (if #f #f))

(test (let ((cont #f)
	    (j 0)
	    (k 0))
	(call/cc (lambda (exit) 
		   (do ((i 0 (+ i 1))) 
		       ((= i 100) i) 
		     (set! j i)
		     (call/cc (lambda (r) (set! cont r)))
		     (if (= j 2) (exit))
		     (set! k i))))
	(if (= j 2)
	    (begin
	      (set! j 3)
	      (cont))
	    (list j k)))
      (list 99 99))

(test (call/cc (lambda (r) (do () (#f) (r 1)))) 1)
(test (let ((hi (lambda (x) (+ x 1)))) (do ((i 0 (hi i))) ((= i 3) i))) 3)
(test (do ((i 0 (+ i 1))) (list 1) ((= i 3) #t)) 1) ; a typo originally -- Guile and Gauche are happy with it
(test (do () (1 2) 3) 2)

;; from sacla tests
(test (let ((rev (lambda (list)
		   (do ((x list (cdr x))
			(reverse '() (cons (car x) reverse)))
		       ((null? x) reverse)))))
	(and (null? (rev '()))
	     (equal? (rev '(0 1 2 3 4)) '(4 3 2 1 0))))
      #t)

(test (let ((nrev (lambda (list)
		    (do ((f1st (if (null? list) '() (cdr list)) (if (null? f1st) '() (cdr f1st)))
			 (s2nd list f1st)
			 (t3rd '() s2nd))
			((null? s2nd) t3rd)
		      (set-cdr! s2nd t3rd)))))
	(and (null? (nrev '()))
	     (equal? (nrev (list 0 1 2 3 4)) '(4 3 2 1 0))))
      #t)

(test (do ((temp-one 1 (+ temp-one 1))
	   (temp-two 0 (- temp-two 1)))
	  ((> (- temp-one temp-two) 5) temp-one))
      4)

(test (do ((temp-one 1 (+ temp-one 1))
	   (temp-two 0 (+ temp-one 1)))     
	  ((= 3 temp-two) temp-one))
      3)

(let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
  (test (do ((i 0 (+ 1 i))
	     (n #f)
	     (j 9 (- j 1)))
	    ((>= i j) vec)
	  (set! n (vector-ref vec i))
	  (vector-set! vec i (vector-ref vec j))
	  (vector-set! vec j n))
	'#(9 8 7 6 5 4 3 2 1 0)))

(test (do ((i 0 (+ i 1))) (#t i) (error "do evaluated its body?")) 0)
(test (do '() (#t 1)) 'error)
(test (do . 1) 'error)
(test (do ((i i i)) (i i)) 'error)
(test (do ((i 0 i (+ i 1))) (i i)) 'error)
(test (do ((i)) (#t i)) 'error)
(test (do ((i 0 (+ i 1))) #t) 'error)
(test (do 123 (#t 1)) 'error)
(test (do ((i 1)) (#t . 1) 1) 'error)
(test (do ((i 1) . 1) (#t 1) 1) 'error)
(test (do ((i 1) ()) (= i 1)) 'error)
(test (do ((i 0 . 1)) ((= i 1)) i) 'error)
(test (do ((i 0 (+ i 1))) ((= i 3)) (set! i "hiho")) 'error)
(test (let ((do+ +)) (do ((i 0 (do+ i 1))) ((= i 3)) (set! do+ abs))) 'error)
(test (do () . 1) 'error)
(test (do ((i)) (1 2)) 'error)
(test (do (((i))) (1 2)) 'error)
(test (do ((i 1) ((j))) (1 2)) 'error)
(test (do (((1))) (1 2)) 'error)
(test (do ((pi 1 2)) (#t pi)) 'error)
(test (do ((1+i 2 3)) (#t #t)) 'error)
(test (do ((1.2 2 3)) (#t #t)) 'error)
(test (do (((1 . 2) "hi" (1 2))) (#t 1)) 'error)
(test (do ((() () ())) (#t #t)) 'error)
(test (do (("hi" "hi")) ("hi")) 'error)
(test (do ((:hi 1 2)) (#t :hi)) 'error)
(test (do ((i 0 (abs ()))) ((not (= i 0)) i)) 'error)
(test (do ((i j) (j i)) (i i)) 'error)
(test (do ((i 0 0) . ((j 0 j))) (#t j)) 0)
(test (do ((i 0 1 . 2)) (#t i)) 'error)
(test (do ((i 0 "hi")) ((string? i) . i)) 'error)
(test (do ((i 0 j)) (#t i)) 0) ; guile also -- (do ((i 0 (abs "hi"))) (#t i)) etc (do ((i 0 1)) (#t i) (abs "hi"))
(test (do ((i 0 1) . (j 0 0)) ((= i 1) i) i) 'error)
(test (do ((i 0 1) ((j 0 0)) ((= i 1) i)) i) 'error)
(test (do #f) 'error)
(test (do () #f) 'error)
(test (do () #()) 'error)
(test (do '((i 1)) ()) 'error)
(test (do #() ()) 'error)
(test (do ((#() 1)) ()) 'error)
(test (do ((1)) ()) 'error)
(test (do ((i 1) . #(a 1)) ()) 'error)
(test (do () ((3 4))) 'error)
(test (do ((i 1)) '()) '())
(test (do . (() (#t 1))) 1)
(test (do () . ((#t 1))) 1)
(test (do ((i 1 (+ i 1))) . ((() . ()))) '())

(test (define-constant) 'error)
(test (define-constant _asdf_ 2 3) 'error)
(test (define-constant pi 3) 'error) ; except in Utah
(test (define-constant pi . 3) 'error)
(define-constant __do_step_var_check__ 1)
(test (do ((__do_step_var_check__ 2 3)) (#t #t)) 'error)
(test (let ((__do_step_var_check__ 2)) 1) 'error)
(test (let () (set! __do_step_var_check__ 2)) 'error)
(test (let ((__do_step_var_access_1__ #f))
	(set! (symbol-access '__do_step_var_access_1__) (list #f #f #f))
	(do ((__do_step_var_access_1__ 1 2)) (#t __do_step_var_access_1__)))
      1)

(test (let ((__do_step_var_access_1__ #f))
	(set! (symbol-access '__do_step_var_access_1__) (list #f (lambda (x y) (error "do step var is being set!"))
							      (lambda (x y) (+ y 1))))
	(do ((__do_step_var_access_1__ 1 32)) (#t __do_step_var_access_1__)))
      2) 
(test (do ((hi #3d(((1 2) (3 4)) ((5 6) (7 8))) (hi 1))) ((equal? hi 8) hi)) 8)
(test (do ((i 0 ('((1 2) (3 4)) 0 1))) ((not (= i 0)) i)) 2)
(test (do () (#t (+ 1 2 3))) 6)
(test (do ((f + *) (j 1 (+ j 1))) ((= j 2) (apply f (list j j)))) 4)
(test (do ((f lambda) (j 1 (+ j 1))) ((= j 2) ((f (a) (+ a j)) 3))) 5)

(let ()
  (define-macro (add-1 x) `(+ ,x 1))
  (test (do ((i 0 (add-1 i))) ((= i 3) i)) 3)
  (test (do ((i 0 (add-1 i))) ((= i 3) (add-1 i))) 4))

(test (let ((j #f))
	(do ((i 0 (let ((x 0))
		    (dynamic-wind
			(lambda ()
			  (set! x i))
			(lambda ()
			  (+ x 1))
			(lambda ()
			  (if (> x 3)
			      (set! j #t)))))))
	    (j i)))
      5)
(test (let ((j 0)) (do ((i 0 (eval-string "(+ j 1)"))) ((= i 4) j) (set! j i))) 3)
(test (do ((i (do ((i (do ((i 0 (+ i 1)))
			  ((= i 3) (+ i 1)))
		      (do ((j 0 (+ j 1)))
			  ((= j 3)) (+ j i))))
		  ((> (do ((k 0 (+ k 1)))
			  ((= k 2) (* k 4)))
		      (do ((n 0 (+ n 1)))
			  ((= n 3) n)))
		   (do ((m 0 (+ m 1)))
		       ((= m 3) (+ m i)))))
	      i))
	  ((> i 6) i))
      7)

(test (let ((L (list 1 2))) 
	(do ((sum 0 (+ sum (car lst))) 
	     (i 0 (+ i 1)) 
	     (lst L (cdr lst))) 
	    ((or (null? lst) 
		 (> i 10)) 
	     sum) 
	  (set-cdr! (cdr L) L))) 
      16)

;;; optimizer checks
(num-test (let ((x 0)) (do ((i 1.0 (+ i 1))) ((> i 3)) (set! x (+ x i))) x) 6.0)
(num-test (let ((x 0)) (do ((i 1 4)) ((> i 3)) (set! x (+ x i))) x) 1)
(num-test (let ((x 0)) (do ((i 1 ((if #t + -) i 1))) ((> i 3)) (set! x (+ x i))) x) 6)
(num-test (let ((x 0)) (do ((i 1 (+))) ((> i 0)) (set! x (+ x i))) x) 0)
(num-test (let ((x 0)) (do ((i 1 (+ 1))) ((> i 0)) (set! x (+ x i))) x) 0)
(num-test (let ((x 0)) (do ((i 1 (+ 1 i 2))) ((> i 10)) (set! x (+ x i))) x) 22)
(num-test (let ((x 0)) (do ((i 1 (+ 1.0 i))) ((> i 3)) (set! x (+ x i))) x) 6.0)
(num-test (let ((x 0)) (do ((i 1 (+ 1 pi))) ((> i 2)) (set! x (+ x i))) x) 1)
(num-test (do ((i 0 (+ 1 pi))) ((> i 2) i)) (+ pi 1.0))
(num-test (let ((x 0)) (do ((i 0 (+ i 8796093022208))) ((> i 0)) (set! x (+ x i))) x) 0)
(num-test (let ((x 0)) (do ((i 0 (+ i 8796093022208))) ((> i 17592186044416)) (set! x (+ x i))) x) (+ (expt 2 44) (expt 2 43)))
(num-test (let ((x 0)) (do ((i 1 (* i 2))) ((> i 10)) (set! x (+ x i))) x) 15)
(num-test (do ((i 0 (+ i 1))) ((> i 2) i) (set! i (+ i 3.0))) 4.0)
(num-test (let ((x 0)) (let ((add +)) (do ((i 0 (add i 1))) ((< i -2)) (set! add -) (set! x (+ x i)))) x) -3)
(num-test (let ((equals =) (x 0)) (do ((i 0 (+ i 1))) ((equals i 3) x) (set! x (+ x i)))) 3)
(num-test (let ((equals =) (x 0)) (do ((i 0 (+ i 1))) ((equals i 3) x) (set! x (+ x i)) (set! i (* i 1.0)))) 3.0)
(num-test (let ((equals =) (x 0)) (do ((i 0 (+ i 1))) ((equals i 3) x) (set! x (+ x i)) (set! equals >))) 6)
(num-test (let ((equals =) (x 0)) (do ((i 0 (+ i 1))) ((equals i 3) x) (set! x (+ x i)) (set! equals =))) 3)
(num-test (let ((equals =) (x 0)) (do ((i 0 (+ i 1))) ((equals i 3) (set! x (+ x 1)) x) (set! x (+ x i)) (set! equals =))) 4)
(num-test (do ((i 0 (+ i 1))) ((> i 3) i) (set! i (expt 2 60))) (+ 1 (expt 2 60)))
(num-test (let ((x 0) (n 3)) (do ((i 0 (+ i 1))) ((= i n) x) (set! x (+ x i)))) 3)
(num-test (let ((x 0) (n 3)) (do ((i 0 (+ i 1))) ((= 1 1) x) (set! x (+ x i)))) 0)
(num-test (let ((x 0) (n (expt 2 50))) (do ((i 0 (+ i n))) ((= i (expt 2 51)) x) (set! x (+ x i)))) (expt 2 50))
(num-test (let ((x 0) (n 31.0)) (do ((i 0 (+ i 1))) ((= i n) x) (set! x (+ x i)) (set! n 3))) 3)
(num-test (let ((x 0)) (do ((i 0 (+ i 1/2))) ((= i 3) x) (set! x (+ x i)))) 15/2)
(num-test (let ((x 0)) (do ((i 0 (+ i 1+i))) ((> (magnitude i) 3) x) (set! x (+ x i)))) 3+3i)
(num-test (call-with-exit (lambda (r) (do () () (r 1)))) 1)
(num-test (call-with-exit (lambda (r) (do () (#t 10 14) (r 1)))) 14)
(num-test (do ((i 0 (+ i 1))) (#t 10 12)) 12)
(num-test (do ((i 0 (+ i 1))) ((= i 3) i)) 3)
(num-test (do ((i 0 (+ i 1))) ((> i 3) i)) 4)
(num-test (do ((i 0 (+ i 1))) ((< i 3) i)) 0)
(num-test (do ((i 0 (+ i 1))) ((<= i 3) i)) 0)
(num-test (do ((i 0 (+ i 1))) ((>= i 3) i)) 3)
(num-test (do ((i 0 (+ i 1))) ((>= 3 i) i)) 0)
(num-test (do ((i 0 (+ i 1))) ((> 3 i) i)) 0)
(num-test (do ((i 0 (+ i 1))) ((< 3 i) i)) 4)
(num-test (do ((i 0 (+ i 1))) ((<= 3 i) i)) 3)
(num-test (let ((n 3)) (do ((i 0 (+ i 1))) ((> i n) i))) 4)
(num-test (let ((n 3)) (do ((i 0 (+ i 1))) ((< n i) i))) 4)
(num-test (do ((i 10 (- i 1))) ((= i 0) i)) 0)
(num-test (do ((i 10 (- 1 i))) ((< i 0) i)) -9)
(num-test (do ((i 10 (- i 3))) ((< i 0) i)) -2)

;;; check an optimizer bug
(define _do_call_cc_end_ 1)
(define (call-cc-do-test)
  (do ((i 0 (+ i 1)))
      ((= i _do_call_cc_end_))
    (let ((ctr 0)) 
      (call/cc (lambda (exit) 
		 (if (> 3 2) 
		     (let () 
		       (exit ctr) 
		       (set! ctr 100) ctr) 
		     #f)))))
  (do ((i 0 (+ 1 i)))
      ((= i _do_call_cc_end_))
    (let ((ctr 0)) 
      (call/cc (lambda (exit) 
		 (if (> 3 2) 
		     (let () 
		       (exit ctr) 
		       (set! ctr 100) ctr) 
		     #f))))))
(call-cc-do-test)

;;; more optimizer checks
(let () (define (do-test) (do ((i 0 (+ i 1))) ((= i 10)) (display i))) (test (with-output-to-string (lambda () (do-test))) "0123456789"))
(let () (define (do-test) (do ((i 0 (+ 1 i))) ((= i 10)) (display i))) (test (with-output-to-string (lambda () (do-test))) "0123456789"))
(let ((start 0)) (define (do-test) (do ((i start (+ i 1))) ((= i 10)) (display i))) (test (with-output-to-string (lambda () (do-test))) "0123456789"))
(let ((start 0) (end 10)) (define (do-test) (do ((i start (+ i 1))) ((= i end)) (display i))) (test (with-output-to-string (lambda () (do-test))) "0123456789"))
(let ((start 0) (end 10)) (define (do-test) (do ((i start (+ i 1))) ((= end i)) (display i))) (test (with-output-to-string (lambda () (do-test))) "0123456789"))
(let () (define (do-test) (do ((i 0 (+ i 1))) ((= i 10)) (let ((k i)) (display k)))) (test (with-output-to-string (lambda () (do-test))) "0123456789"))
(let () (define (do-test) (do ((i 0 (+ i 2))) ((= i 20)) (display (/ i 2)))) (test (with-output-to-string (lambda () (do-test))) "0123456789"))
(let () (define (do-test) (do ((i 0 (+ i 1))) ((= i 10)) (let ((a (+ 1 2))) (display #\0)))) (test (with-output-to-string (lambda () (do-test))) "0000000000"))

(let () (define (do-test) (do ((i 0 (+ i 1))) ((= i 10)) (let ((j 0)) (set! j i) (display j)))) (test (with-output-to-string (lambda () (do-test))) "0123456789"))
(let () (define (do-test) (do ((i 0 (+ i 1))) ((= i 10)) (let ((j 0)) (display i)))) (test (with-output-to-string (lambda () (do-test))) "0123456789"))
(let () (define (do-test) (do ((i 0 (+ i 1))) ((= i 10)) (let ((j 0)) (set! j 32) (display i)))) (test (with-output-to-string (lambda () (do-test))) "0123456789"))
(let () (define (do-test) (do ((i 0 (+ i 1))) ((= i 10)) (let ((j i)) (display j)))) (test (with-output-to-string (lambda () (do-test))) "0123456789"))
(let () (define (do-test) (do ((i 0 (+ i 1))) ((= i 5)) (let ((j (+ i 1))) (let ((i j)) (display (- i 1)))))) (test (with-output-to-string (lambda () (do-test))) "01234"))





;;; --------------------------------------------------------------------------------
;;; set!
;;; --------------------------------------------------------------------------------

(test (let ((a 1)) (set! a 2) a) 2)
(for-each
 (lambda (arg)
   (test (let ((a 0)) (set! a arg) a) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))

(test (let ((a 1)) (call/cc (lambda (r) (set! a (let () (if (= a 1) (r 123)) 321)))) a) 1)
(test (let ((a (lambda (b) (+ b 1)))) (set! a (lambda (b) (+ b 2))) (a 3)) 5)
(test (let ((a (lambda (x) (set! x 3) x))) (a 1)) 3)

(test (let ((x (vector 1 2 3))) (set! (x 1) 32) x) #(1 32 3))
(test (let* ((x (vector 1 2 3))
	     (y (lambda () x)))
	(set! ((y) 1) 32)
	x)
      #(1 32 3))
(test (let* ((x (vector 1 2 3))
	     (y (lambda () x))
	     (z (lambda () y)))
	(set! (((z)) 1) 32)
	x)
      #(1 32 3))

(test (let ((a 1)) (set! a)) 'error)
(test (let ((a 1)) (set! a 2 3)) 'error)
(test (let ((a 1)) (set! a . 2)) 'error)
(test (let ((a 1)) (set! a 1 . 2)) 'error)
(test (let ((a 1)) (set! a a) a) 1)
(test (set! "hi" 1) 'error)
(test (set! 'a 1) 'error)
(test (set! 1 1) 'error)
(test (set! (list 1 2) 1) 'error)
(test (set! (let () 'a) 1) 'error)
(test (set!) 'error)
(test (set! #t #f) 'error)
(test (set! '() #f) 'error)
(test (set! #(1 2 3) 1) 'error)
(test (set! (call/cc (lambda (a) a)) #f) 'error)
(test (set! 3 1) 'error)
(test (set! 3/4 1) 'error)
(test (set! 3.14 1) 'error)
(test (set! #\a 12) 'error)
(test (set! (1 2) #t) 'error)
(test (set! _not_a_var_ 1) 'error)
(test (set! (_not_a_pws_) 1) 'error)

(test (let ((a (lambda (x) (set! a 3) x))) (list (a 1) a)) 'error)
(test (let ((a (let ((b 1)) (set! a 3) b))) a) 'error)            
(test (let ((a (lambda () "hi"))) (set! (a) "ho")) 'error)
(test (let ((a (let ((b 1)) (set! a 3) b))) a) 'error) 

(test (set! . -1) 'error)
(test (set!) 'error)
(test (let ((x 1)) (set! x x x)) 'error)
(test (let ((x 1)) (set! x x) x) 1)
(test (set! set! 123) 'error)
(test (set! (cons 1 2) 3) 'error)
(test (let ((var 1) (val 2)) (set! var set!) (var val 3) val) 3)
(test (let ((var 1) (val 2)) (set! var +) (var val 3)) 5)
(test (let ((sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 1))
	(set! sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 2)
	sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789)
      2)

(test (let ((x '(1)) (y '(2))) (set! ((if #t x y) 0) 32) x) '(32))
(test (let ((hi 0)) (set! hi 32)) 32)
(test (let ((hi 0)) ((set! hi ('((1 2) (3 4)) 0)) 0)) 1)

(test (set! #<undefined> 1) 'error)
(test (set! #<eof> 1) 'error)
(test (set! #<unspecified> 1) 'error)
(test (let ((x 0)) (define-macro (hi) 'x) (set! (hi) 3) x) 'error)

(test (set! ("hi" . 1) #\a) 'error)
(test (set! (#(1 2) . 1) 0) 'error)
(test (set! ((1 . 2)) 3) 'error)
(test (let ((lst (list 1 2))) (set! (lst . 0) 3) lst) 'error)
(test (let ((lst (list 1 2))) (set! (list-ref lst . 1) 2)) 'error)
(test (let ((v #2d((1 2) (3 4)))) (set! (v 0 . 0) 2) v) 'error)
(test (set! ('(1 2) . 0) 1) 'error)
(test (set! ('(1 2) 0) 3) 3)
(test (set! (''(1 . 2)) 3) 'error)
(test (set! (''(1 2)) 3) 'error)
(test (set! ('(1 . 2)) 3) 'error)
(test (set! ('(1 2)) 3) 'error)
(test (set! (''(1 2) 0 0) 3) 'error)
(test (set! (#(1 2) 0 0) 3) 'error)
(test (let ((x 1)) (set! (quasiquote . x) 2) x) 'error)
(test (let ((x 1)) (set! (quasiquote x) 2) x) 'error)
(test (set! `,(1) 3) 'error)
(test (set! (1) 3) 'error)
(test (set! `,@(1) 3) 'error)
(test (let ((x 0)) (set! x 1 . 2)) 'error)
(test (let ((x 0)) (apply set! x '(3))) 'error) ; ;set!: can't alter immutable object: 0
(test (let ((x 0)) (apply set! 'x '(3)) x) 3)
(test (set! (#(a 0 (3)) 1) 0) 0)
(test (set! ('(a 0) 1) 0) 0)
(test (apply set! (apply list (list ''(1 2 3) 1)) '(32)) 32)
(let ()
  (define-macro (symbol-set! var val) `(apply set! ,var (list ,val)))
  (test (let ((x 32) (y 'x)) (symbol-set! y 123) (list x y)) '(123 x)))
(test (set! ('(1 2) 1 . 2) 1) 'error)
(test (set! ('((1 2) 1) () . 1) 1) 'error)
(test (set! ('(1 1) () . 1) 1) 'error)

(test (let () (define (hi) (let ((x 1000)) (set! x (+ x 1)) x)) (hi) (hi)) 1001)
(test (let () (define (hi) (let ((x 1000.5)) (set! x (+ x 1)) x)) (hi) (hi)) 1001.5)
(test (let () (define (hi) (let ((x 3/2)) (set! x (+ x 1)) x)) (hi) (hi)) 5/2)
(test (let () (define (hi) (let ((x "asdf")) (set! x (+ x 1)) x)) (hi) (hi)) 'error)




;;; --------------------------------------------------------------------------------
;;; or
;;; --------------------------------------------------------------------------------

(test (or (= 2 2) (> 2 1)) #t)
(test (or (= 2 2) (< 2 1)) #t)
(test (or #f #f #f) #f)
(test (or) #f)
(test (or (memq 'b '(a b c)) (+ 3 0)) '(b c))
(test (or 3 9) 3)
(test (or #f 3 asdf) 3) ; "evaluation stops immediately"
(test (or 3 (/ 1 0) (display "or is about to exit!") (exit)) 3)

(for-each
 (lambda (arg)
   (test (or arg) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) #<eof> #<unspecified> '(1 . 2)))

(test (call-with-input-file "s7test.scm"
	(lambda (p)
	  (let ((loc 0))
	    (let loop ((val (read-char p)))
	      (or (eof-object? val)
		  (> loc 1000) ; try to avoid the read-error stuff
		  (begin
		    (set! loc (+ 1 loc))
		    (loop (read-char p)))))
	    (> loc 1000))))
      #t)

(test (or (and (or (> 3 2) (> 3 4)) (> 2 3)) 4) 4)
(test (or or) or)
(test (or (or (or))) #f)
(test (or (or (or) (and))) #t)
(test (let ((a 1)) (or (let () (set! a 2) #f) (= a 1) (let () (set! a 3) #f) (and (= a 3) a) (let () (set! a 4) #f) a)) 3)
(test (or '#f '()) '())
(test (call/cc (lambda (r) (or #f (> 3 2) (r 123) 321))) #t)
(test (call/cc (lambda (r) (or #f (< 3 2) (r 123) 321))) 123)
(test (+ (or #f (not (null? '())) 3) (or (zero? 1) 2)) 5)
(test (or 0) 0)
(test (if (or) 1 2) 2)

(test (or . 1) 'error)
(test (or #f . 1) 'error)
(test (or . (1 2)) 1)
(test (or . ()) (or))
; (test (or 1 . 2) 1) ; this fluctuates

(test (let () (or (define (hi a) a)) (hi 1)) 1)
(test (let () (or #t (define (hi a) a)) (hi 1)) 'error)
(test (let () (and (define (hi a) a) (define (hi a) (+ a 1))) (hi 1)) 2) ; guile agrees with this
(test ((lambda (arg) (arg #f 123)) or) 123)
(test (let ((oar or)) (oar #f 43)) 43)
(test (let ((oar #f)) (set! oar or) (oar #f #f 123)) 123)




;;; --------------------------------------------------------------------------------
;;; and
;;; --------------------------------------------------------------------------------

(test (and (= 2 2) (> 2 1)) #t)
(test (and (= 2 2) (< 2 1)) #f)
(test (and 1 2 'c '(f g)) '(f g))
(test (and) #t)
(test (and . ()) (and))
(test (and 3) 3)
(test (and (memq 'b '(a b c)) (+ 3 0)) 3)
(test (and 3 9) 9)
(test (and #f 3 asdf) #f) ; "evaluation stops immediately"
(test (and 3 (zero? 1) (/ 1 0) (display "and is about to exit!") (exit)) #f)
(test (if (and) 1 2) 1)
(test (if (+) 1 2) 1)
(test (if (*) 1 2) 1)
(test (and (if #f #f)) (if #f #f))

(for-each
 (lambda (arg)
   (test (and arg) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call-with-input-file "s7test.scm"
	(lambda (p)
	  (let ((loc 0))
	    (let loop ((val (read-char p)))
	      (and (not (eof-object? val))
		   (< loc 1000)
		   (begin
		     (set! loc (+ 1 loc))
		     (loop (read-char p)))))
	    (>= loc 1000))))
      #t)

(test (and (or (and (> 3 2) (> 3 4)) (> 2 3)) 4) #f)
(test (and and) and)
(test (and (and (and))) #t)
(test (and (and (and (and (or))))) #f)
(test (let ((a 1)) (and (let () (set! a 2) #t) (= a 1) (let () (set! a 3) #f) (and (= a 3) a) (let () (set! a 4) #f) a)) #f)
(test (and '#t '()) '())
(test (call/cc (lambda (r) (and #t (> 3 2) (r 123) 321))) 123)
(test (call/cc (lambda (r) (and #t (< 3 2) (r 123) 321))) #f)
(test (+ (and (null? '()) 3) (and (zero? 0) 2)) 5)

(test (and . #t) 'error)
(test (and 1 . 2) 'error)
(test (and . (1 2)) 2)

(test (let () (and (define (hi a) a)) (hi 1)) 1)
(test (let () (and #f (define (hi a) a)) (hi 1)) 'error)
(test (+ 1 (and (define (hi a) a) (hi 2))) 3)




;;; --------------------------------------------------------------------------------
;;; cond
;;; --------------------------------------------------------------------------------

(test (cond ('a)) 'a)
(test (cond (3)) 3)
(test (cond (#f 'a) ('b)) 'b)
(test (cond (#t 'a) (#t 'b)) 'a)
(test (cond ((> 3 2) 'greater) ((< 3 2) 'less)) 'greater)
(test (cond((> 3 2)'greater)((< 3 2)'less)) 'greater)
(test (cond ((> 3 3) 'greater) ((< 3 3) 'less)  (else 'equal)) 'equal)
(test (cond ((assv 'b '((a 1) (b 2))) => cadr)  (else #f)) 2)
(test (cond (#f 2) (else 5)) 5)
(test (cond (1 2) (else 5)) 2)
(test (cond (1 => (lambda (x) (+ x 2))) (else 8)) 3)
(test (cond ((+ 1 2))) 3)
(test (cond ((zero? 1) 123) ((= 1 1) 321)) 321)
(test (cond ('() 1)) 1)
(test (let ((x 1)) (cond ((= 1 2) 3) (else (* x 2) (+ x 3)))) 4)
(test (let((x 1))(cond((= 1 2)3)(else(* x 2)(+ x 3)))) 4)
(test (let ((x 1)) (cond ((= x 1) (* x 2) (+ x 3)) (else 32))) 4)
(test (let ((x 1)) (cond ((= x 1) (let () (set! x (* x 2))) (+ x 3)) (else 32))) 5)
(test (let ((x 1)) (cond ((= x 2) (let () (set! x (* x 2))) (+ x 3)) (else 32))) 32)
(test (let ((x 1)) (cond ((= x 2) 3) (else (let () (set! x (* x 2))) (+ x 3)))) 5)
(test (cond ((= 1 2) 3) (else 4) (else 5)) 4) ; this should probably be an error
(test (cond (1 2 3)) 3)
(test (cond (1 2) (3 4)) 2)
(test (cond ((= 1 2) 3) ((+ 3 4))) 7)
(test (cond ((= 1 1) (abs -1) (+ 2 3) (* 10 2)) (else 123)) 20)
(test (let ((a 1)) (cond ((= a 1) (set! a 2) (+ a 3)))) 5)
(test (let ((a 1)) (cond ((= a 2) (+ a 2)) (else (set! a 3) (+ a 3)))) 6)
(test (cond ((= 1 1))) #t)
(test (cond ((= 1 2) #f) (#t)) #t)
(test (cond ((+ 1 2))) 3)
(test (cond ((cons 1 2))) '(1 . 2))
(test (cond (#f #t) ((string-append "hi" "ho"))) "hiho")
(test (cond ('() 3) (#t 4)) 3)
(test (cond ((list) 3) (#t 4)) 3)
;;; (cond (1 1) (asdf 3)) -- should this be an error?
(test (cond (+ 0)) 0)
(test (cond (lambda ())) ())
(test (cond . ((1 2) ((3 4)))) 2)
(test (cond (define #f)) #f)
(test (let () (cond ((> 2 1) (define x 32) x) (#t 1)) x) 32) ; ? a bit strange
(test (let ((x 1)) (+ x (cond ((> x 0) (define x 32) x)) x)) 65)

(for-each
 (lambda (arg)
   (test (cond ((or arg) => (lambda (x) x))) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (cond ((+ 1 2) => (lambda (x) (+ 1 x)))) 4)
(test (cond ((cons 1 2) => car)) 1)
(test (cond ((values 1 2) => +)) 3)
(test (cond (1 2 => +)) 'error)
(test (cond ((begin 1 2) => +)) 2)
(test (cond ((values -1) => abs)) 1)
(test (cond ((= 1 2) => +) (#t => not)) #f)
(test (cond ((* 2 3) => (let () -))) -6)
(test (cond ((* 2 3) => (cond ((+ 3 4) => (lambda (a) (lambda (b) (+ b a))))))) 13)
(test (let ((x 1)) ((cond ((let () (set! x 2) #f) => boolean?) (lambda => (lambda (a) (apply a '((b) (+ b 123)))))) x)) 125)
(test (cond ((values 1 2 3) => '(1 (2 3 (4 5 6 7 8))))) 7)
(test (cond ((values #f #f) => equal?)) #t) ; (values #f #f) is not #f
(test (let () (cond (#t (define (hi a) a))) (hi 1)) 1)
(test (let () (cond (#f (define (hi a) a))) (hi 1)) 'error)
(test (let () (cond ((define (hi a) a) (hi 1)))) 1)

(test (cond (else 1)) 1)
(test (call/cc (lambda (r) (cond ((r 4) 3) (else 1)))) 4)
(test (cond ((cond (#t 1)))) 1)
(test (symbol? (cond (else else))) #f)
(test (equal? else (cond (else else))) #t)
(test (cond (#f 2) ((cond (else else)) 1)) 1)

(for-each
 (lambda (arg)
   (test (cond (#t arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (cond (arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (cond (#f 1) (else arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (cond (arg => (lambda (x) x))) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (cond ((let () 1) => (let ((x 2)) (lambda (n) (+ n x))))) 3)
(test (cond ((let () 1) => (let ((x 2)) (cond (3 => (let ((y 4)) (lambda (n) (lambda (m) (+ n m x y))))))))) 10)

(test (let ((=> 3)) (cond (=> =>))) 3)
(test (cond (cond 'cond)) 'cond)
(test (cond (3 => (lambda args (car args)))) 3)
(test (cond (3 => (lambda (a . b) a))) 3)
(test (cond ((list 3 4) => (lambda (a . b) b))) '())
(test (cond) 'error)
					;(test (cond ((= 1 2) 3) (else 4) (4 5)) 'error) ; trailing ignored 
(test (cond ((+ 1 2) => (lambda (a b) (+ a b)))) 'error)
(test (equal? (cond (else)) else) #t)
(test (cond (#t => 'ok)) 'error)
(test (cond (else =>)) 'error)
(test (cond ((values -1) => => abs)) 'error)
(test (cond ((values -1) =>)) 'error)
(test (cond (cond (#t 1))) 'error)
(test (cond 1) 'error)
(test (cond) 'error)
(test (cond (1 . 2) (else 3)) 'error)
(test (cond (#f 2) (else . 4)) 'error)
(test (cond ((values 1 2) => (lambda (x y) #t))) #t)
(test (cond #t) 'error)
(test (cond 1 2) 'error)
(test (cond 1 2 3) 'error)
(test (cond 1 2 3 4) 'error)
(test (cond (1 => (lambda (x y) #t))) 'error)
(test (cond . 1) 'error)
(test (cond ((1 2)) . 3) 'error)
(test (cond (1 => + abs)) 'error)
(test (cond (1 =>)) 'error)
(test (cond ((values 1 2) => + abs)) 'error)
;(test (cond (else => not)) 'error) ; to heck with this
(test (let ((else 3)) (cond ((= else 3) 32) (#t 1))) 32)
(test (let ((else #f)) (cond (else 32) (#t 1))) 1)
(test (cond #((1 2))) 'error)

(test (let ((=> 3)) (cond (1 =>))) 3)
(test (let ((=> 3)) (cond (1 => abs))) abs)
(test (let ((=> 3) (else 4)) (cond (else => abs))) abs)
(test (let ((=> 3)) (cond (1 => "hi"))) "hi")

(test (let ((x 0))
	(cond ((let ((y x)) (set! x 1) (= y 1)) 0)
	      ((let ((y x)) (set! x 1) (= y 1)) 1)
	      (#t 2)))
      1)

(let ((c1 #f)
      (x 1))
  (let ((y (cond ((let ()
		    (call/cc
		     (lambda (r)
		       (set! c1 r)
		       (r x))))
		  => (lambda (n) (+ n 3)))
		 (#t 123))))
    (if (= y 4) (begin (set! x 2) (c1 321)))
    (test (list x y) '(2 324))))

(let ((c1 #f)
      (x 1))
  (let ((y (cond (x => (lambda (n) 
			 (call/cc
			  (lambda (r)
			    (set! c1 r)
			    (r (+ 3 x))))))
		 (#t 123))))
    (if (= y 4) (begin (set! x 2) (c1 321)))
    (test (list x y) '(2 321))))




;;; -------- cond-expand --------
;;; cond-expand

(test (let ()
	(cond-expand (guile )
		     (s7 (define (hi a) a)))
	(hi 1))
      1)
(test (let ((x 0))
	(cond-expand (guile (format #t ";oops~%"))
		     (else (set! x 32)))
	x)
      32)
(test (let ()
	(cond-expand
	 (guile 
	  (define (hi a) (+ a 1)))
	 ((or common-lisp s7)
	  (define (hi a) a)))
	(hi 1))
      1)
(test (let ()
	(cond-expand
	 ((not guile)
	  (define (hi a) a))
	 (else 
	  (define (hi a) (+ a 1))))
	(hi 1))
      1)
(test (let ()
	(cond-expand 
	 ((and s7 dfls-exponents)
	  (define (hi a) a))
	 (else 
	  (define (hi a) (+ a 1))))
	(hi 1))
      1)
(test (let ()
	(cond-expand 
	 ((or s7 guile)
	  (define (hi a) a))
	 (else 
	  (define (hi a) (+ a 1))))
	(hi 1))
      1)
(test (let ()
	(cond-expand 
	 ((and s7 dfls-exponents unlikely-feature)
	  (define (hi a) a))
	 (else 
	  (define (hi a) (+ a 1))))
	(hi 1))
      2)




;;; --------------------------------------------------------------------------------
;;; case
;;; --------------------------------------------------------------------------------

(test (case (* 2 3) ((2 3 5 7) 'prime) ((1 4 6 8 9) 'composite))  'composite)
(test (case (car '(c d)) ((a e i o u) 'vowel) ((w y) 'semivowel) (else 'consonant)) 'consonant)
(test (case 3.1 ((1.3 2.4) 1) ((4.1 3.1 5.4) 2) (else 3)) 2)
(test (case 3/2 ((3/4 1/2) 1) ((3/2) 2) (else 3)) 2)
(test (case 3 ((1) 1 2 3) ((2) 2 3 4) ((3) 3 4 5)) 5)
(test (case 1+i ((1) 1) ((1/2) 1/2) ((1.0) 1.0) ((1+i) 1+i)) 1+i)
(test (case 'abs ((car cdr) 1) ((+ cond) 2) ((abs) 3) (else 4)) 3)
(test (case #\a ((#\b) 1) ((#\a) 2) ((#\c) 3)) 2)
(test (case (boolean? 1) ((#t) 2) ((#f) 1) (else 0)) 1)
(test (case 1 ((1 2 3) (case 2 ((1 2 3) 3)))) 3)
(test (case 1 ((1 2) 1) ((3.14 2/3) 2)) 1)
(test (case 1 ((1 2) 1) ((#\a) 2)) 1)
(test (case 1 ((1 2) 1) ((#\a) 2) ((car cdr) 3) ((#f #t) 4)) 1)
(test (case #f ((1 2) 1) ((#\a) 2) ((car cdr) 3) ((#f #t) 4)) 4)
(test (case 1 ((#t) 2) ((#f) 1) (else 0)) 0)
(test (let ((x 1)) (case x ((x) "hi") (else "ho"))) "ho")
(test (let ((x 1)) (case x ((1) "hi") (else "ho"))) "hi")
(test (let ((x 1)) (case x (('x) "hi") (else "ho"))) "ho")
(test (let ((x 1)) (case 'x ((x) "hi") (else "ho"))) "hi")
(test (case '() ((()) 1)) 1)
;;; but not (case #() ((#()) 1)) because (eqv? #() #()) is #f
(test (let ((x '(1))) (eval `(case ',x ((,x) 1) (else 0)))) 1)    ; but we can overcome that!
(test (let ((x #())) (eval `(case ',x ((,x) 1) (else 0)))) 1)
(test (case ''2 (('2) 1) (else 0)) 0)

(test (case else ((#f) 2) ((#t) 3) ((else) 4) (else 5)) 5)          ; (eqv? 'else else) is #f (Guile says "unbound variable: else")
(test (case #t ((#f) 2) ((else) 4) (else 5)) 5)                     ; else is a symbol here         
(test (equal? (case 0 ((0) else)) else) #t)
(test (cond ((case 0 ((0) else)) 1)) 1)

(test (let ((x 1)) (case x ((2) 3) (else (* x 2) (+ x 3)))) 4)
(test (let ((x 1)) (case x ((1) (* x 2) (+ x 3)) (else 32))) 4)
(test (let ((x 1)) (case x ((1) (let () (set! x (* x 2))) (+ x 3)) (else 32))) 5)
(test (let ((x 1)) (case x ((2) (let () (set! x (* x 2))) (+ x 3)) (else 32))) 32)
(test (let ((x 1)) (case x ((2) 3) (else (let () (set! x (* x 2))) (+ x 3)))) 5)
(test (let((x 1))(case x((2)3)(else(let()(set! x(* x 2)))(+ x 3)))) 5)
(test (let ((x 1)) (case x ((2) 3) (else 4) (else 5))) 'error)

(test (case '() ((()) 2) (else 1)) 2)    ; car: (), value: (), eqv: 1, null: 1 1
(test (case '() (('()) 2) (else 1)) 1)   ; car: (quote ()), value: (), eqv: 0, null: 0 1
(test (case () (('()) 2) (else 1)) 1)    ; car: (quote ()), value: (), eqv: 0, null: 0 1
(test (case () ((()) 2) (else 1)) 2)     ; car: (), value: (), eqv: 1, null: 1 1

;;; this is a difference between '() and () ?
;;; (eqv? '() '()) -> #t and (eqv? '() ()) is #t so it's the lack of evaluation in the search case whereas the index is evaluated
;;; equivalent to:
 
(test (case 2 (('2) 3) (else 1)) 1)      ; car: (quote 2), value: 2, eqv: 0, null: 0 0
(test (case '2 (('2) 3) (else 1)) 1)     ; car: (quote 2), value: 2, eqv: 0, null: 0 0
(test (case '2 ((2) 3) (else 1)) 3)      ; car: 2, value: 2, eqv: 1, null: 0 0
(test (case 2 ((2) 3) (else 1)) 3)       ; car: 2, value: 2, eqv: 1, null: 0 0

(test (case '(()) ((()) 1) (((())) 2) (('()) 3) (('(())) 4) ((((()))) 5) (('((()))) 6) (else 7)) 7) ; (eqv? '(()) '(())) is #f

(test (let ((x 1)) (case (+ 1 x) ((0 "hi" #f) 3/4) ((#\a 1+3i '(1 . 2)) "3") ((-1 'hi 2 2.0) #\f))) #\f)
(test (case (case 1 ((0 2) 3) (else 2)) ((0 1) 2) ((4 2) 3) (else 45)) 3)
(test (case 3/4 ((0 1.0 5/6) 1) (("hi" 'hi 3/4) 2) (else 3)) 2)
(test (case (case (+ 1 2) (else 3)) ((3) (case (+ 2 2) ((2 3) 32) ((4) 33) ((5) 0)))) 33)
(test (let ((x 1)) (case x ((0) (set! x 12)) ((2) (set! x 32))) x) 1)

(test (case 1 (else #f)) #f)
(test (let () (case 0 ((0) (define (hi a) a)) (else (define (hi a) (+ a 1)))) (hi 1)) 1)
(test (let () (case 1 ((0) (define (hi a) a)) (else (define (hi a) (+ a 1)))) (hi 1)) 2)
(test (let () (case (define (hi a) a) ((hi) (hi 1)))) 1)

(for-each
 (lambda (arg)
   (test (case 1 ((0) 'gad) ((1 2 3) arg) (else 'gad)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (case arg ((0) 'gad) ((1 2 3) arg) (else 'gad)) 'gad))
 (list "hi" -1 #\a 0 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call/cc (lambda (r) (case 1 ((1) (r 123) #t) (else #f)))) 123)
(test (call/cc (lambda (r) (case 1 ((0) 0) (else (r 123) #f)))) 123)

(test (case '() ((1) 1) ('() 2)) 2)
(test (case (list) ((1) 1) ('() 2)) 2)
(test (case '() ((1) 1) ((()) 2)) 2)
(test (case (list) ((1) 1) ((()) 2)) 2)
(test (case #<eof> ((#<eof>) 1)) 1)
(test (case #\newline ((#\newline) 1)) 1)
(test (case 'c (else => (lambda (x) (symbol? x)))) #t)

; case use eqv? -- why not case-equal?
(test (case "" (("") 1)) #<unspecified>)
(test (case abs ((abs) 1)) #<unspecified>)

(test (case) 'error)
(test (case 1) 'error)
(test (case 1 . "hi") 'error)
(test (case 1 ("hi")) 'error)
(test (case 1 ("a" "b")) 'error)
(test (case 1 (else #f) ((1) #t)) 'error)
(test (case "hi" (("hi" "ho") 123) ("ha" 321)) 'error)
(test (case) 'error)
(test (case . 1) 'error)
(test (case 1 . 1) 'error)
(test (case 1 (#t #f) ((1) #t)) 'error)
(test (case 1 (#t #f)) 'error)
(test (case -1 ((-1) => abs)) 1)
(test (case 1 (else =>)) 'error)
(test (case 1 (else => + - *)) 'error)
(test (case #t ((1 2) (3 4)) -1) 'error)
(test (case 1 1) 'error)
(test (case 1 ((2) 1) . 1) 'error)
(test (case 1 (2 1) (1 1)) 'error)
(test (case 1 (else)) 'error)
(test (case () ((1 . 2) . 1) . 1) 'error)
(test (case 1 ((1))) 'error)
(test (case 1 ((else))) 'error)
(test (case 1 ((2) 3) ((1))) 'error)
(test (case 1 ((1)) 1 . 2) 'error)
(test (case () ((()))) 'error)
(test (case 1 (else 3) . 1) 'error)
(test (case 1 ((1 2)) (else 3)) 'error)
(test (case 1 ('(1 2) 3) (else 4)) 4)
(test (case 1 (('1 2) 3) (else 4)) 4)
;;; (test (case 1 ((1 . 2) 3) (else 4)) 'error) ; ?? in guile it's an error
;;; (test (case 1 ((1 2 . 3) 3) (else 4)) 'error)
(test (case 1 (('1 . 2) 3) (else 4)) 'error)
(test (case 1 ((1 . (2)) 3) (else 4)) 3)
(test (case 1 ((1 2) . (3)) (else 4)) 3)
(test (case 1 ((2) 3) (else)) 'error)
(test (case 1 ((2) 3) ()) 'error)
(test (case 1 ((2) 3) (() 2)) 'error) ; ?? in Guile this is #<unspecified>; our error is confusing: ;case clause key list () is not a list or 'else'
(test (case '() ('() 2)) 2)            ; ?? error??
(test (case '() ((()) 2)) 2) 
(test (case 1 else) 'error)
(test (case 1 (((1) 1) 2) (else 3)) 2) ; the (1) can't be matched -- should it be an error?
(test (case 1 ((1) . (else 3))) 3)     ; ?? guile says "unbound variable: else"
(test (case . (1 ((2) 3) ((1) 2))) 2)
(test (case 1 (#(1 2) 3)) 'error)
(test (case 1 #((1 2) 3)) 'error)

(test (case 'case ((case) 1) ((cond) 3)) 1)
(test (case 101 ((0 1 2) 200) ((3 4 5 6) 600) ((7) 700) ((8) 800) ((9 10 11 12 13) 1300) ((14 15 16) 1600) ((17 18 19 20) 2000) ((21 22 23 24 25) 2500) ((26 27 28 29) 2900) ((30 31 32) 3200) ((33 34 35) 3500) ((36 37 38 39) 3900) ((40) 4000) ((41 42) 4200) ((43) 4300) ((44 45 46) 4600) ((47 48 49 50 51) 5100) ((52 53 54) 5400) ((55) 5500) ((56 57) 5700) ((58 59 60) 6000) ((61 62) 6200) ((63 64 65) 6500) ((66 67 68 69) 6900) ((70 71 72 73) 7300) ((74 75 76 77) 7700) ((78 79 80) 8000) ((81) 8100) ((82 83) 8300) ((84 85 86 87) 8700) ((88 89 90 91 92) 9200) ((93 94 95) 9500) ((96 97 98) 9800) ((99) 9900) ((100 101 102) 10200) ((103 104 105 106 107) 10700) ((108 109) 10900) ((110 111) 11100) ((112 113 114 115) 11500) ((116) 11600) ((117) 11700) ((118) 11800) ((119 120) 12000) ((121 122 123 124 125) 12500) ((126 127) 12700) ((128) 12800) ((129 130) 13000) ((131 132) 13200) ((133 134 135 136) 13600) ((137 138) 13800)) 10200)
(test (case most-positive-fixnum ((-1231234) 0) ((9223372036854775807) 1) (else 2)) 1)
(test (case most-negative-fixnum ((123123123) 0) ((-9223372036854775808) 1) (else 2)) 1)
(test (case 0 ((3/4 "hi" #t) 0) ((#f #() -1) 2) ((#\a 0 #t) 3) (else 4)) 3)
(test (case 3/4 ((3/4 "hi" #t) 0) ((#f #() hi) 2) ((#\a 0 #t) 3) (else 4)) 0)
(test (case 'hi ((3/4 "hi" #t) 0) ((#f #() hi) 2) ((#\a 0 #t) 3) (else 4)) 2)
(test (case #f ((3/4 "hi" #t) 0) ((#f #() hi) 2) ((#\a 0 #t) 3) (else 4)) 2)
(test (case 3 ((3/4 "hi" #t) 0) ((#f #() hi) 2) ((#\a 0 #t) 3) (else 4)) 4)
(test (case 0 ((values 0 1) 2) (else 3)) 2)

(test (let ((else 3)) (case 0 ((1) 2) (else 3))) 'error) ; also if else is set!
(test (let ((else 3)) (case else ((3) else))) 3)
(test (case 0 ((1) #t) ((2 else 3) #f) ((0) 0)) 0) ; should this be an error? (it isn't in Guile)
(test (case 0 ((1) #t) ((else) #f) ((0) 0)) 0)

(test (let ((x 0)) (let ((y (case 1 ((2) (set! x (+ x 3))) ((1) (set! x (+ x 4)) (+ x 2))))) (list x y))) '(4 6))

;;; one thing that will hang case I think: circular key list



;;; --------------------------------------------------------------------------------
;;; lambda
;;; --------------------------------------------------------------------------------

(test (procedure? (lambda (x) x)) #t)
(test ((lambda (x) (+ x x)) 4) 8)
(test (let ((reverse-subtract (lambda (x y) (- y x)))) (reverse-subtract 7 10)) 3)
(test (let ((add4 (let ((x 4)) (lambda (y) (+ x y))))) (add4 6)) 10)
(test ((lambda x x) 3 4 5 6) (list 3 4 5 6))
(test ((lambda (x y . z) z) 3 4 5 6) (list 5 6))
(test ((lambda (a b c d e f) (+ a b c d e f)) 1 2 3 4 5 6) 21)
(test (let ((foo (lambda () 9))) (+ (foo) 1)) 10)
(test (let ((a 1)) (let ((f (lambda (x) (set! a x) a))) (let ((c (f 123))) (list c a)))) (list 123 123))
(test (let ((a 1) (b (lambda (a) a))) (b 3)) 3)
(test (let ((ctr 0)) (letrec ((f (lambda (x) (if (> x 0) (begin (set! ctr (+ ctr 1)) (f (- x 1))) 0)))) (f 10) ctr)) 10)
(test (let ((f (lambda (x) (car x)))) (f '(4 5 6))) 4)
(test ((lambda () ((lambda (x y) ((lambda (z) (* (car z) (cdr z))) (cons x y))) 3 4))) 12)
(test (let ((ctr 0)) (define (f) (set! ctr (+ ctr 1)) ctr) (let ((x (f))) (let ((y (f))) (list x y ctr)))) (list 1 2 2))

(test (let ((x 5)) (define foo (lambda (y) (bar x y))) (define bar (lambda (a b) (+ (* a b) a))) (foo (+ x 3))) 45)
(test (let ((x 5)) (letrec ((foo (lambda (y) (bar x y))) (bar (lambda (a b) (+ (* a b) a)))) (foo (+ x 3)))) 45)
(num-test (let () (define compose (lambda (f g) (lambda args (f (apply g args))))) ((compose sqrt *) 12 75))  30.0)
(test (let ((f (lambda () (lambda (x y) (+ x y))))) ((f) 1 2)) 3)
(test ((lambda (x) (define y 4) (+ x y)) 1) 5)
(test ((lambda(x)(define y 4)(+ x y))1) 5)
(test ((lambda () (define (y x) (+ x 1)) (y 1))) 2)
(test ((lambda (x) 123 (let ((a (+ x 1))) a)) 2) 3)
(test ((lambda (x) "documentation" (let ((a (+ x 1))) a)) 2) 3)
(test ((lambda (x) (x 1)) (lambda (y) (+ y 1))) 2)
(test (let ((a 1)) (let ((b (lambda (x) (define y 1) (define z 2) (define a 3) (+ x y z a)))) (b a))) 7)
(test ((lambda (f x) (f x x)) + 11) 22)
(test ((lambda () (+ 2 3))) 5)
(test (let ((x (let () (lambda () (+ 1 2))))) (x)) 3)
(test (cond (0 => (lambda (x) x))) 0)
(test ((lambda () "hiho")) "hiho")
(test ((lambda()()))())
(test (procedure-source (apply lambda (list) (list (list)))) '(lambda () ()))

(test (letrec ((f (lambda (x) (g x)))
	       (g (lambda (x) x)))
	(let ((top (f 1)))
	  (set! g (lambda (x) (- x)))
	  (+ top (f 1))))
      0)

(for-each
 (lambda (arg)
   (test ((lambda (x) x) arg) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((list-length
       (lambda (obj)
	 (call-with-current-continuation
	  (lambda (return)
	    (letrec ((r (lambda (obj) (cond ((null? obj) 0)
					    ((pair? obj) (+ (r (cdr obj)) 1))
					    (else (return #f))))))
	      (r obj)))))))
  (test (list-length '(1 2 3 4)) 4)
  (test (list-length '(a b . c)) #f))

(test (let ((samples (vector 0 1 2 3 4 5 6 7 8 9 10)))
	(let ((make-scaler 
	       (lambda (start end)
		 (letrec ((ctr start)
			  (us (lambda (them)
				(vector-set! samples ctr (* 2 (vector-ref samples ctr)))
				(set! ctr (+ ctr 2))
				(if (<= ctr end)
				    (them us)))))
		   us))))
	  ((make-scaler 0 11)
	   (make-scaler 1 11))) 
	samples)
      (vector 0 2 4 6 8 10 12 14 16 18 20))

(test ((lambda (x . y) y) 1 2 '(3 . 4)) '(2 (3 . 4)))
(test ((lambda (x . y) y) 1) '())
(test ((lambda x x) '()) '(()))
(test ((lambda x x)) '())
(test ((lambda (x) x) '()) '())
(test ((lambda (x) (+ x ((lambda (x) (+ x 1)) 2))) 3) 6)
(test ((lambda (x) (define y 1) (+ x y)) 2) 3)
(test ((lambda (a) "this is a doc string" a) 1) 1)
;;; ideally ((lambda (a) "hiho" (define x 1) x) 1) -> 1 but I'm not sure it's r5rs-ish
(test (let ((g (lambda () '3))) (= (g) 3)) #t)
(test ((((lambda () lambda)) () 1)) 1)

(test (let () ; PLP Scott p168
	(define A
	  (lambda ()
	    (let* ((x 2)
		   (C (lambda (P)
			(let ((x 4))
			  (P))))
		   (D (lambda ()
			x))
		   (B (lambda ()
			(let ((x 3))
			  (C D)))))
	      (B))))
	(A))
      2)

#|
;;; here s7 "do" uses set!
(test (let ((funcs (make-vector 3 #f)))
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (vector-set! funcs i (lambda () (+ i 1))))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)
|#

;;; the equivalent named let version:
(test (let ((funcs (make-vector 3 #f)))
	(let loop ((i 0))
	  (if (< i 3)
	      (begin
		(vector-set! funcs i (lambda () (+ i 1)))
		(loop (+ i 1)))))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((i 1))
	(let ((func1 (lambda () i)))
	  (let ((i 2))
	    (let ((func2 (lambda () i)))
	      (+ (func1) (func2))))))
      3)

(test (let ((funcs (make-vector 3 #f)))
	(map
	 (lambda (i)
	   (vector-set! funcs i (lambda () (+ i 1))))
	 (list 0 1 2))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((funcs (make-vector 3 #f)))
	(for-each
	 (lambda (i)
	   (vector-set! funcs i (lambda () (+ i 1))))
	 (list 0 1 2))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((funcs (make-vector 3 #f)))
	(sort! (list 0 1 2)
	 (lambda (i j)
	   (vector-set! funcs i (lambda () (+ i 1))) 
	   (> i j)))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((funcs (make-vector 3 #f)))
	(member 4 (list 0 1 2)
	 (lambda (j i)
	   (vector-set! funcs i (lambda () (+ i 1)))
	   #f))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((funcs (make-vector 3 #f)))
	(assoc 4 (list (cons 0 0) (cons 1 0) (cons 2 0))
	 (lambda (j i) 
	   (vector-set! funcs i (lambda () (+ i 1)))
	   #f))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((func #f))
	(define (func1 x)
	  (set! func (lambda () (+ x 1))))
	(func1 1)
	(+ (func)
	   (let ()
	     (func1 2)
	     (func))))
      5)

(test (((lambda (x) (lambda () (+ x 1))) 32)) 33)

(test (let ((func #f))
	(define (func1 x)
	  (set! func (lambda () (string-append x "-"))))
	(func1 "hi")
	(string-append (func)
		       (let ()
			 (func1 "ho")
			 (func))))
      "hi-ho-")

(test (let ((func1 #f)
	    (func2 #f))
	(let ((x 1))
	  (set! func1 (lambda () x))
	  (set! func2 (lambda (y) (set! x y) y)))
	(+ (func1)
	   (let ()
	     (func2 32)
	     (func1))))
      33)

(test (let ((funcs (make-vector 3)))
	(let ((hi (lambda (a) (vector-set! funcs (- a 1) (lambda () a)))))
	  (hi 1) (hi 2) (hi 3)
	  (+ ((vector-ref funcs 0))
	     ((vector-ref funcs 1))
	     ((vector-ref funcs 2)))))
      6)

(test (let ((hi (lambda (a) (+ a 1)))
	    (ho (lambda (a) (a 32))))
	(+ (hi (hi (hi 1)))
	   (ho hi)))
      37)

(test (let ((x 0)
	    (b 4)
	    (f1 #f)
	    (f2 #f))
	(let ((x 1))
	  (let ((x 2))
	    (set! f1 (lambda (a) (+ a b x)))))
	(let ((x 3))
	  (let ((b 5))
	    (set! f2 (lambda (a) (+ a b x)))))
	(+ (f1 10) (f2 100)))  ; (+ 10 4 2) (+ 100 5 3)
      124)

(test ((if (> 3 2) + -) 3 2) 5)
(test (let ((op +)) (op 3 2)) 5)
(test (((lambda () +)) 3 2) 5)
(test ((car (cons + -)) 3 2) 5)
(test ((do ((i 0 (+ i 1))) ((= i 3) +) ) 3 2) 5)
(test (((lambda (x) x) (lambda (x) x)) 3) 3)
(test ((((lambda (x) x) (lambda (x) x)) (lambda (x) x)) 3) 3)
(test (((lambda (x) (lambda (y) x)) 3) 4) 3)
(test (((lambda (x) (lambda (x) x)) 3) 4) 4)
(test (let ((x 32)) (((lambda (x) (lambda (y) x)) 3) x)) 3)
(test ((call/cc (lambda (return) (return +))) 3 2) 5)
(test ((call-with-values (lambda () (values +)) (lambda (x) x)) 3 2) 5)
(test ((case '+ ((+) +)) 3 2) 5)
(test ((case '+ ((-) -) (else +)) 3 2) 5)
(test ((call/cc (lambda (return) (dynamic-wind (lambda () #f) (lambda () (return +)) (lambda () #f)))) 3 2) 5)
(test (+ 1 ((call/cc (lambda (return) (dynamic-wind (lambda () #f) (lambda () (return +)) (lambda () #f)))) 3 2) 2) 8)
(test (let ((lst (list + -))) ((car lst) 1 2 3)) 6)
(test (let ((a +)) ((let ((b -)) (if (eq? a b) a *)) 2 3)) 6)
(test ((list-ref (list + - * /) 0) 2 3) 5)
(test (((if #t list-ref oops) (list + - * /) 0) 2 3) 5)
(test ((((car (list car cdr)) (list car cdr)) (list + -)) 2 3) 5)
(test (let ()
	(define function lambda)
	(define hiho (function (a) (+ a 1)))
	(hiho 2))
      3)
(test ((lambda (a b c d e f g h i j k l m n o p q r s t u v x y z)
	 (+ a b c d e f g h i j k l m n o p q r s t u v x y z))
       1 2 3 4 5 6 7 8 9 11 12 13 14 15 16 17 18 19 21 22 23 24 25 26 27)
      348)
(test ((lambda (x) "a useless string" x) 32) 32)
(test ((lambda (>< =0=? .arg.) (+ >< =0=? .arg.)) 1 2 3) 6)

(test
 (let ()
   (begin
     (define f1 #f)
     (define f2 #f)
     (let ((lv 32))
       (set! f1 (lambda (a) (+ a lv)))
       (set! f2 (lambda (a) (- a lv)))))
   (+ (f1 1) (f2 1)))
 2)

(test ((lambda () => abs)) 'error)
(test ((lambda () => => 3)) 'error)
;; actually, both Guile and Gauche accept
;; ((lambda () + 3)) and (begin + 3)
;; but surely => is an undefined variable in this context?

(test (lambda) 'error)
(test (lambda (a) ) 'error)
;; should this be an error: (lambda (a) (define x 1)) ?
(test (lambda . 1) 'error)
(test ((lambda . (x 1))) 1)
(test ((lambda . ((x . y) 2)) 1) 2)
(test ((lambda (x) . (x)) 1) 1)
(test ((lambda . ((x) . (x))) 1) 1)
(test ((lambda . (x . (x))) 1) '(1))
(test ((lambda . ((x . ()) x)) 1) 1)
(test (eval-string "((lambda . (x 1 . 3)) 1)") 'error)

(test (lambda 1) 'error)
(test (lambda (x 1) x) 'error)
(test (lambda "hi" 1) 'error)
(test (lambda (x x) x) 'error)
(test ((lambda (x x) x) 1 2) 'error) 
(test (lambda (x "a")) 'error)
(test ((lambda (x y) (+ x y a)) 1 2) 'error)
(test ((lambda ())) 'error)
(test (lambda (x (y)) x) 'error)
(test ((lambda (x) x . 5) 2) 'error)
(test (lambda (1) #f) 'error)
(test (eval-string "(lambda (x . y z) x)") 'error) 
(test ((lambda () 1) 1) 'error)
(test ((lambda (()) 1) 1) 'error)
(test ((lambda (x) x) 1 2) 'error)
(test ((lambda (x) x)) 'error)
(test ((lambda ("x") x)) 'error)
(test ((lambda "x" x)) 'error)
(test ((lambda (x . "hi") x)) 'error)
(test (let ((hi (lambda (a 0.0) (b 0.0) (+ a b)))) (hi)) 'error)
(test (object->string
       ((lambda (arg)
	  (list arg
		(list (quote quote)
		      arg)))
	(quote (lambda (arg)
		 (list arg
		       (list (quote quote)
			     arg))))))
      "(#1=(lambda (arg) (list arg (list 'quote arg))) '#1#)")
      
(test ((apply lambda '((a) (+ a 1))) 2) 3)
(test ((apply lambda '(() #f))) #f)
(test ((apply lambda '(arg arg)) 3) '(3))
(test ((apply lambda* '((a (b 1)) (+ a b))) 3 4) 7)
(test ((apply lambda* '((a (b 1)) (+ a b))) 3) 4)

(test (lambda #(a b) a) 'error)
(test (lambda* (#(a 1)) a) 'error)

(test ((lambda (a) a) #<eof>) #<eof>)
(test ((lambda () (let ((a #<undefined>)) a))) #<undefined>)

(let ()
  (define (hi a) (+ a x))
  ;(format #t "hi: ~S~%" (procedure-source hi))
  (test ((apply let '((x 32)) (list (procedure-source hi))) 12) 44))
;; i.e. make a closure from (let ((x 32)) <procedure-source hi>)



;;; --------------------------------------------------------------------------------
;;; begin
;;; --------------------------------------------------------------------------------

(test (begin) '()) ; I think Guile returns #<unspecified> here
(test (begin (begin)) '())
(test (let () (begin) #f) #f)
(test (let () (begin (begin (begin (begin)))) #f) #f)
(test (let () (begin (define x 2) (define y 1)) (+ x y)) 3)
(test (let () (begin (define x 0)) (begin (set! x 5) (+ x 1)))  6)
(test (let () (begin (define first car)) (first '(1 2))) 1)
(test (let () (begin (define x 3)) (begin (set! x 4) (+ x x))) 8)
(test (let () (begin (define x 0) (define y x) (set! x 3) y)) 0)         ; the let's block confusing global defines
(test (let () (begin (define x 0) (define y x) (begin (define x 3) y))) 0)
(test (let () (begin (define y x) (define x 3) y)) 'error)               ; guile says 3
(test (let ((x 12)) (begin (define y x) (define x 3) y)) 12)             ; guile says 3 which is letrec-style?
(test (begin (define (x) y) (define y 4) (x)) 4)
;; (let ((x 12)) (begin (define y x) y)) is 12
(test (let ((x 3)) (begin x)) 3)
(test (begin 3) 3)
(test (begin . (1 2)) 2)
(test (begin . ()) (begin))
(test (begin . 1) 'error)
(test (begin 1 . 2) 'error)

(if (equal? (begin 1) 1)
    (begin
      (test (let () (begin (define x 0)) (set! x (begin (begin 5))) (begin ((begin +) (begin x) (begin (begin 1))))) 6)      
      
      (test (let ((x 5))
	      (begin (begin (begin)
			    (begin (begin (begin) (define foo (lambda (y) (bar x y)))
					  (begin)))
			    (begin))
		     (begin)
		     (begin)
		     (begin (define bar (lambda (a b) (+ (* a b) a))))
		     (begin))
	      (begin)
	      (begin (foo (+ x 3))))
	    45)
      
      (for-each
       (lambda (arg)
	 (test (begin arg) arg))
       (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
      
      (test (if (= 1 1) (begin 2) (begin 3)) 2)
      ))

(test ((lambda (x) (begin (set! x 1) (let ((a x)) (+ a 1)))) 2) 2)
;;; apparently these can be considered errors or not (guile says error, stklos and gauche do not)
(test (begin (define x 0) (+ x 1)) 1)
(test ((lambda () (begin (define x 0) (+ x 1)))) 1)
(test (let ((f (lambda () (begin (define x 0) (+ x 1))))) (f)) 1)

(test ((lambda () (begin (define x 0)) (+ x 1))) 1)
(test (let ((f (lambda () (begin (define x 0)) (+ x 1)))) (f)) 1)
(test (let ((x 32)) (begin (define x 3)) x) 3)
(test ((lambda (x) (begin (define x 3)) x) 32) 3)
(test (let* ((x 32) (y x)) (define x 3) y) 32)

(test (let ((z 0)) (begin (define x 32)) (begin (define y x)) (set! z y) z) 32)
(test (let((z 0))(begin(define x 32))(begin(define y x))(set! z y)z) 32)
(test (let ((z 0)) (begin (define x 32) (define y x)) (set! z y) z) 32)        
(test (let () (begin (define b 1) (begin (define a b) (define b 3)) a)) 1)
(test (let () (begin (begin (define a1 1) (begin (define a1 b1) (define b1 3))) a1)) 'error)
(test (let () (begin (begin (define (a3) 1)) (begin (define (a3) b3) (define b3 3)) (a3))) 3) ; yow
(test (let () (begin (begin (define (a) 1)) (a))) 1)
(test (let ((a 1)) (begin (define a 2)) a) 2)
(test (+ 1 (begin (values 2 3)) 4) 10)
(test (+ 1 (begin (values 5 6) (values 2 3)) 4) 10)
(test (let ((hi 0)) (begin (values (define (hi b) (+ b 1))) (hi 2))) 3)




;;; --------------------------------------------------------------------------------
;;; apply
;;; --------------------------------------------------------------------------------

(test (apply (lambda (a b) (+ a b)) (list 3 4)) 7)
(test (apply + 10 (list 3 4)) 17)
(test (apply list '()) '())
(test (apply + '(1 2)) 3)
(test (apply - '(1 2)) -1)
(test (apply max 3 5 '(2 7 3)) 7)
(test (apply cons '((+ 2 3) 4)) '((+ 2 3) . 4))
(test (apply + '()) 0)
(test (apply + (list 3 4)) 7)
(test (apply + '()) 0)
(test (apply + 2 '(3)) 5)
(test (apply + 2 3 '()) 5)
(test (apply + '(2 3)) 5)
(test (apply list 1 '(2 3)) (list 1 2 3))
(test (apply apply (list list 1 2 '(3))) (list 1 2 3))
(test (vector? (apply make-vector '(1))) #t)
(test (apply make-vector '(1 1)) '#(1))
(test (apply make-vector '((1) 1)) '#(1))
(test (let ((f +)) (apply f '(1 2))) 3)
(test (let* ((x '(1 2 3)) (y (apply list x))) (eq? x y)) #t) ; is this standard? -- no, Guile says #f
(test (apply min '(1 2 3 5 4 0 9)) 0)
(test (apply min 1 2 4 3 '(4 0 9)) 0)
(test (apply vector 1 2 '(3)) '#(1 2 3))
(test (apply (lambda (x . y) x) (list 1 2 3)) 1)
(test (apply * (list 2 (apply + 1 2 '(3)))) 12)
(test (apply (if (> 3 2) + -) '(3 2)) 5)
(test (let ((x (list 1 2))) (eq? x (append '() x))) #t) ;; ?? guile says #t also
(test (apply (lambda* args args) 1 2 3 '(4 5 6 (7))) '(1 2 3 4 5 6 (7))) ; from lisp bboard

(for-each
 (lambda (arg)
   (test (apply (lambda (x) x) (list arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (apply cadadr (list '''4)) 4)
(test (apply string-ref "hi" '(0)) #\h)
(test (let ((x (string-copy "hi"))) (apply string-set! x 0 '(#\c)) x) "ci")
(test (apply apply (list + '(3  2))) 5)
(test (apply apply apply apply (list (list (list + '(3  2))))) 5)
(test (apply + 1 2 (list 3 4)) 10)
(test ((apply cdr '((1 2) (3 4)) ()) 0) '(3 4))
(test ((apply car '((1 2) (3 4)) ()) 1) 2)
(test ((apply cadr '((1 2) (3 4)) ()) 1) 4)

(test (apply +) 0)
(test (apply + #f) 'error)
(test (apply #f '(2 3)) 'error)
(test (apply make-vector '(1 2 3)) 'error)
(test (apply + 1) 'error)
(test (apply) 'error)
(test (apply 1) 'error)
(test (apply . 1) 'error)
(test (apply car ''foo) 'error)
(test (apply + '(1 . 2)) 'error)
(test (apply + '(1 2 . 3)) 'error)
(test (apply '() '()) 'error)
(test (apply list '(1 . 2) '()) '((1 . 2)))
(test (apply (lambda (x) x) _ht_) 'error)
(test (apply + '#(1 2 3)) 'error)

(for-each
 (lambda (arg)
   (test (apply arg '(1)) 'error)
   (test (apply abs arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #t)) 

(test (apply "hi" '(1)) #\i)
(test (apply '(1 2 3) '(1)) 2)
(test (apply #(1 2 3) '(2)) 3)
(test (apply #2D((1 2) (3 4)) 0 0 ()) 1)
(test (apply '((1 2) (3 4)) 1 0 ()) 3)
(test (let ((ht (make-hash-table))) (set! (ht "hi") 32) (apply ht '("hi"))) 32)

(test (let ((x (list 1 2))) (set-cdr! x x) (apply + x)) 'error)
(test (apply + '(1 2 . 3)) 'error)
(test (apply + '(1 2) (list 3 4)) 'error)
(test (let () (define (mrec a b) (if (<= b 0) (list a) (apply mrec (list a) (list (- b 1))))) (mrec (list 1 2) 5)) '(((((((1 2))))))))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (apply + lst) 'error))

(test (apply values (values (cons 1 ()))) 1)
(test (+ (apply values (values (list 1 2)))) 3)
(test (port-filename) (apply port-filename (list)))
(num-test (apply atan (#(1 #\a (3)) (max (values 1 2)))) 1.2490457723983)
(test (apply #2D((1 2) (3 4)) (list (floor (acosh 1)))) #(1 2)) 
(test ((apply values (list + 1 2)) 3) 6)
(num-test (* 0-2i (acosh (asin 0.0))) pi)
(test (apply truncate (lognot (min 1)) (list)) -2)
(num-test (apply /(list 11 11)) 1)

(test (apply dynamic-wind (list (lambda () #f) (lambda () 1) (lambda () #f))) 1)
(test (apply call-with-exit (list (lambda (exit) 1))) 1)
(test (apply call-with-exit (list (lambda (exit) (exit 1) 32))) 1)
(test (apply catch (list #t (lambda () 1) (lambda args 'error))) 1)
(test (apply eval '((+ 1 2))) 3)
(test (apply eval '()) 'error) ; (eval) is an error -- should it be? (eval ()) is ()
(test (apply eval '(())) '())
(test (apply eval-string '("(+ 1 2)")) 3) 
(test (let () (apply begin '((define x 1) (define y x) (+ x y)))) 2)
(test (apply begin '()) (begin))
(test (apply if '(#f 1 2)) 2)
(test (let ((x 1)) (apply set! '(x 3)) x) 3)
(test (let ((x 3)) (apply set! (list (values 'x 32))) x) 32)
(test (let ((x 1)) (apply cond '(((= x 2) 3) ((= x 1) 32)))) 32)
(test (apply and '((= 1 1) (> 2 3))) #f)
(test (apply and '()) (and))
(test (apply or '((= 1 1) (> 2 3))) #t)
(test (apply or '()) (or))
(test (let () (apply define '(x 32)) x) 32)
(test (let () (apply define* '((hi (a 1) (b 2)) (+ a b))) (hi 32)) 34)
(test ((apply lambda '((n) (+ n 1))) 2) 3)
(test ((apply lambda* '(((n 1)) (+ n 1)))) 2)
(test (apply let '(((x 1)) (+ x 2))) 3)
(test (apply let* '(((x 1) (y (* 2 x))) (+ x y))) 3)
(test (let () (apply define-macro `((hiho a) `(+ ,a 1))) (hiho 2)) 3)
(test (let () (apply defmacro `(hiho (a) `(+ ,a 1))) (hiho 2)) 3)
(test (let () (apply defmacro* `(hiho ((a 2)) `(+ ,a 1))) (hiho)) 3)
(test (let () (apply define-macro* `((hiho (a 2)) `(+ ,a 1))) (hiho)) 3)
(test (apply do '(((i 0 (+ i 1))) ((= i 3) i))) 3)
(test (apply case '(1 ((2 3) 4) ((1 5) 32))) 32)
(test (+ (apply values '(1 2 3))) 6)
(test (apply quote '(1)) 1)
(test (apply quote '()) 'error) ; (quote) is an error
(test (let () (apply letrec '(() (define x 9) x))) 9)
(test ((lambda (n) (apply n '(((x 1)) (+ x 2)))) let) 3)
(test ((apply lambda (list (apply let (list (list) (quote (list (apply case '(0 ((0 1) 'n))))))) (quasiquote (+ n 1)))) 2) 3)
(test (apply let '((x 1)) '((+ x 1))) 2)
(test ((apply make-procedure-with-setter (list (lambda (x) (+ x 1)) (lambda (x y) (+ x y)))) 23) 24)
(test (apply (apply make-procedure-with-setter (list (lambda (x) (+ x 1)) (lambda (x y) (+ x y)))) '(23)) 24)

(test (apply 'begin) 'error)
(test (apply and) #t)
(test (apply begin) '())
(test (apply if '((> 1 2) 3 4)) 4)
(test (apply or) #f)
(test (apply quote '(1)) 1)




;;; --------------------------------------------------------------------------------
;;; define
;;; --------------------------------------------------------------------------------

;;; trying to avoid top-level definitions here

(let ()
  (define x 2)
  (test (+ x 1) 3)
  (set! x 4)
  (test (+ x 1) 5)
  (let ()
    (define (tprint x) #t)
    (test (tprint 56) #t)
    (let ()
      (define first car)
      (test (first '(1 2)) 1)
      (let ()
	(define foo (lambda () (define x 5) x))
	(test (foo) 5)
	(let ()
	  (define (foo x) ((lambda () (define x 5) x)) x)
	  (test (foo 88) 88))))))


(test (letrec ((foo (lambda (arg) (or arg (and (procedure? foo) (foo 99)))))) (define bar (foo #f)) (foo #f)) 99)
(test (letrec ((foo 77) (bar #f) (retfoo (lambda () foo))) (define baz (retfoo)) (retfoo)) 77)

(test (let () (define .. 1) ..) 1)

(test (let () (define (hi a) (+ a 1)) (hi 2)) 3)
(test (let () (define (hi a . b) (+ a (cadr b) 1)) (hi 2 3 4)) 7)
(test (let () (define (hi) 1) (hi)) 1)
(test (let () (define (hi . a) (apply + a)) (hi 1 2 3)) 6)

(for-each
 (lambda (arg)
   (test (let () (define x arg) x) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test ((lambda (x) (define (hi a) (+ a 1)) (hi x)) 1) 2)
(test (let ((x 2)) (define f (lambda (y) (+ y x))) (f 3)) 5)
(begin (define r5rstest-plus (lambda (x y) (+ x y))) (define r5rstest-x 32))
(test (r5rstest-plus r5rstest-x 3) 35)

(test (let () (define (asdf a) (define (asdf a) (+ a 1)) (+ a (asdf a))) (asdf 4)) 9)
(test (let ((asdf 1)) (define (asdf a) (define (asdf a) (+ a 1)) (+ a (asdf a))) (asdf 4)) 9)
(test (let () (define (a1 a) (define (a2 a) (define (a3 a) (define (a4 a) (+ a 1)) (+ (a4 a) 1)) (+ (a3 a) 1)) (+ (a2 a) 1)) (a1 0)) 4)

(test (let () (define (hi1 a) (define (hi1 b) (+ b 1)) (hi1 a)) (hi1 1)) 2)
(test (let () (define (hi1 a) (begin (define (hi1 b) (+ b 1))) (hi1 a)) (hi1 1)) 2)
(test (let ((j 0) (k 0))
	(define (hi1 a)
	  (let ((hi1 (lambda (b) 
		       (set! k (+ k 1)) 
		       (hi1 (- b 1)))))
	    (if (<= a 0)
		(list j k)
		(begin
		  (set! j (+ j 1))
		  (hi1 (- a 1))))))
	(hi1 3))
      '(2 2))

(test (procedure? (let () (define (a) a) (a))) #t)

(test (define) 'error)
(test (define*) 'error)
(test (define x) 'error)
(test (define . x) 'error)
(test (define x 1 2) 'error)
(test (define x x) 'error)
(test (define (x 1)) 'error)
(test (define (x)) 'error)
(test (define 1 2) 'error)
(test (define "hi" 2) 'error)
(test (define :hi 2) 'error)
(test (define x 1 2) 'error)
(test (define x 1 . 2) 'error)
(test (define x . 1) 'error)
(test (define x (lambda ())) 'error)
(test (define #<eof> 3) 'error)
(test (define (#<undefined>) 4) 'error)
(test (define (:hi a) a) 'error)
(test (define (hi: a) a) 'error)
(test (define (#b1 a) a) 'error)
(test (define (hi #b1) #b1) 'error)
(test (let() (define #(hi a) a)) 'error)
(test (let () (define hi (lambda args args)) (hi 1 . 2)) 'error)
(test (let () (define . 1) 1) 'error)
(test (let () (define func (do () (#t (lambda (y) 2)))) (func 1)) 2)
(test (let () (define* x 3)) 'error)

;; y combinator example from some CS website
(let ()
  (define Y
    (lambda (X)
      ((lambda (procedure)
         (X (lambda (arg) ((procedure procedure) arg))))
       (lambda (procedure)
         (X (lambda (arg) ((procedure procedure) arg)))))))

  (define M
    (lambda (func-arg)
      (lambda (l)
        (if (null? l)
            'no-list
            (if (null? (cdr l))
                (car l)
                (max (car l) (func-arg (cdr l))))))))

  (test ((Y M) '(4 5 6 3 4 8 6 2)) 8))

(test (((lambda (X)
	  ((lambda (procedure)
	     (X (lambda (arg) ((procedure procedure) arg))))
	   (lambda (procedure)
	     (X (lambda (arg) ((procedure procedure) arg))))))
	(lambda (func-arg)
	  (lambda (n)
	    (if (zero? n)
		1
		(* n (func-arg (- n 1)))))))
       5)
      120)

(let ()
  (define (Cholesky:decomp P)
    ;; from Marco Maggi based on a Scheme bboard post
    ;; (Cholesky:decomp '((2 -2) (-2 5))) -> ((1.4142135623731 0) (-1.4142135623731 1.7320508075689))
    (define (Cholesky:make-square L)
      (define (zero-vector n)
	(if (zero? n)
	    '()
	    (cons 0 (zero-vector (- n 1)))))
      (map (lambda (v)
	     (append v (zero-vector (- (length L) (length v)))))
	   L))
    (define (Cholesky:add-element P L i j)
      (define (Cholesky:smaller P)
	(if (null? (cdr P))
	    '()
	    (reverse (cdr (reverse P)))))
      (define (Cholesky:last-row L)
	(car (reverse L)))
      (define (matrix:element A i j)
	(list-ref (list-ref A i) j))
      (define (Cholesky:make-element P L i j)
	(define (Cholesky:partial-sum L i j)
	  (let loop ((k j))
	    (case k
	      ((0) 0)
	      ((1) (* (matrix:element L i 0)
		      (matrix:element L j 0)))
	      (else
	       (+ (* (matrix:element L i k)
		     (matrix:element L j k))
		  (loop (- k 1)))))))
	(let ((x (- (matrix:element P i j)
		    (Cholesky:partial-sum L i j))))
	  (if (= i j)
	      (sqrt x)
	      (/ x (matrix:element L j j)))))
      (if (zero? j)
	  (append L `((,(Cholesky:make-element P L i j))))
	  (append (Cholesky:smaller L)
		  (list (append
			 (Cholesky:last-row L)
			 (list (Cholesky:make-element P L i j)))))))
    (Cholesky:make-square
     (let iter ((i 0) (j 0) (L '()))
       (if (>= i (length P))
	   L
	   (iter (if (= i j) (+ 1 i) i)
		 (if (= i j) 0 (+ 1 j))
		 (Cholesky:add-element P L i j))))))
  (let* ((lst (Cholesky:decomp '((2 -2) (-2 5))))
	 (lst0 (car lst))
	 (lst1 (cadr lst)))
    (if (or (> (abs (- (car lst0) (sqrt 2))) .0001)
	    (not (= (cadr lst0) 0))
	    (> (abs (+ (car lst1) (sqrt 2))) .0001)
	    (> (abs (- (cadr lst1) (sqrt 3))) .0001))
	(format #t ";cholesky decomp: ~A~%" lst))))

(let ()
  (define* (a1 (b (let ()
		    (define* (a1 (b 32)) b)
		    (a1))))
    b)
  (test (a1) 32)
  (test (a1 1) 1))

(test (let ((x 1)) (cond (else (define x 2))) x) 2)
(test (let ((x 1)) (and (define x 2)) x) 2)
(test (let () (begin 1)) 1)
(test (let () (begin (define x 1) x)) 1)

(let ()
  (define (f64 arg0 arg1 arg2 arg3 arg4 arg5 arg6 arg7 arg8 arg9 arg10 arg11 arg12 arg13 arg14 arg15 arg16 arg17 arg18 arg19 arg20 arg21 arg22 arg23 arg24 arg25 arg26 arg27 arg28 arg29 arg30 arg31 arg32 arg33 arg34 arg35 arg36 arg37 arg38 arg39 arg40 arg41 arg42 arg43 arg44 arg45 arg46 arg47 arg48 arg49 arg50 arg51 arg52 arg53 arg54 arg55 arg56 arg57 arg58 arg59 arg60 arg61 arg62 arg63 arg64) 
    (+ arg0 arg1 arg2 arg3 arg4 arg5 arg6 arg7 arg8 arg9 arg10 arg11 arg12 arg13 arg14 arg15 arg16 arg17 arg18 arg19 arg20 arg21 arg22 arg23 arg24 arg25 arg26 arg27 arg28 arg29 arg30 arg31 arg32 arg33 arg34 arg35 arg36 arg37 arg38 arg39 arg40 arg41 arg42 arg43 arg44 arg45 arg46 arg47 arg48 arg49 arg50 arg51 arg52 arg53 arg54 arg55 arg56 arg57 arg58 arg59 arg60 arg61 arg62 arg63 arg64))
  (test (f64 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 57 58 59 60 61 62 63 64) 
	2080))

#|
(let ((n 12))
  (let ((nums (do ((lst '() (cons i lst))
		   (i 0 (+ i 1)))
		  ((> i n) (reverse lst)))))
    (format #t "(let ((f~D (lambda (~{arg~D~^ ~})~%    (+ ~{arg~D~^ ~}))))~%  (f~D ~{~D~^ ~}))~%" n nums nums n nums)))
|#

(test (let ((f128 (lambda (arg128 arg127 arg126 arg125 arg124 arg123 arg122 arg121 arg120 arg119 arg118 arg117 arg116 arg115 arg114 arg113 arg112 arg111 arg110 arg109 arg108 arg107 arg106 arg105 arg104 arg103 arg102 arg101 arg100 arg99 arg98 arg97 arg96 arg95 arg94 arg93 arg92 arg91 arg90 arg89 arg88 arg87 arg86 arg85 arg84 arg83 arg82 arg81 arg80 arg79 arg78 arg77 arg76 arg75 arg74 arg73 arg72 arg71 arg70 arg69 arg68 arg67 arg66 arg65 arg64 arg63 arg62 arg61 arg60 arg59 arg58 arg57 arg56 arg55 arg54 arg53 arg52 arg51 arg50 arg49 arg48 arg47 arg46 arg45 arg44 arg43 arg42 arg41 arg40 arg39 arg38 arg37 arg36 arg35 arg34 arg33 arg32 arg31 arg30 arg29 arg28 arg27 arg26 arg25 arg24 arg23 arg22 arg21 arg20 arg19 arg18 arg17 arg16 arg15 arg14 arg13 arg12 arg11 arg10 arg9 arg8 arg7 arg6 arg5 arg4 arg3 arg2 arg1 arg0)
		    (+ arg128 arg127 arg126 arg125 arg124 arg123 arg122 arg121 arg120 arg119 arg118 arg117 arg116 arg115 arg114 arg113 arg112 arg111 arg110 arg109 arg108 arg107 arg106 arg105 arg104 arg103 arg102 arg101 arg100 arg99 arg98 arg97 arg96 arg95 arg94 arg93 arg92 arg91 arg90 arg89 arg88 arg87 arg86 arg85 arg84 arg83 arg82 arg81 arg80 arg79 arg78 arg77 arg76 arg75 arg74 arg73 arg72 arg71 arg70 arg69 arg68 arg67 arg66 arg65 arg64 arg63 arg62 arg61 arg60 arg59 arg58 arg57 arg56 arg55 arg54 arg53 arg52 arg51 arg50 arg49 arg48 arg47 arg46 arg45 arg44 arg43 arg42 arg41 arg40 arg39 arg38 arg37 arg36 arg35 arg34 arg33 arg32 arg31 arg30 arg29 arg28 arg27 arg26 arg25 arg24 arg23 arg22 arg21 arg20 arg19 arg18 arg17 arg16 arg15 arg14 arg13 arg12 arg11 arg10 arg9 arg8 arg7 arg6 arg5 arg4 arg3 arg2 arg1 arg0))))
	(f128 128 127 126 125 124 123 122 121 120 119 118 117 116 115 114 113 112 111 110 109 108 107 106 105 104 103 102 101 100 99 98 97 96 95 94 93 92 91 90 89 88 87 86 85 84 83 82 81 80 79 78 77 76 75 74 73 72 71 70 69 68 67 66 65 64 63 62 61 60 59 58 57 56 55 54 53 52 51 50 49 48 47 46 45 44 43 42 41 40 39 38 37 36 35 34 33 32 31 30 29 28 27 26 25 24 23 22 21 20 19 18 17 16 15 14 13 12 11 10 9 8 7 6 5 4 3 2 1 0))
      8256)

(test (let ((f512 (lambda (arg0 arg1 arg2 arg3 arg4 arg5 arg6 arg7 arg8 arg9 arg10 arg11 arg12 arg13 arg14 arg15 arg16 arg17 arg18 arg19 arg20 arg21 arg22 arg23 arg24 arg25 arg26 arg27 arg28 arg29 arg30 arg31 arg32 arg33 arg34 arg35 arg36 arg37 arg38 arg39 arg40 arg41 arg42 arg43 arg44 arg45 arg46 arg47 arg48 arg49 arg50 arg51 arg52 arg53 arg54 arg55 arg56 arg57 arg58 arg59 arg60 arg61 arg62 arg63 arg64 arg65 arg66 arg67 arg68 arg69 arg70 arg71 arg72 arg73 arg74 arg75 arg76 arg77 arg78 arg79 arg80 arg81 arg82 arg83 arg84 arg85 arg86 arg87 arg88 arg89 arg90 arg91 arg92 arg93 arg94 arg95 arg96 arg97 arg98 arg99 arg100 arg101 arg102 arg103 arg104 arg105 arg106 arg107 arg108 arg109 arg110 arg111 arg112 arg113 arg114 arg115 arg116 arg117 arg118 arg119 arg120 arg121 arg122 arg123 arg124 arg125 arg126 arg127 arg128 arg129 arg130 arg131 arg132 arg133 arg134 arg135 arg136 arg137 arg138 arg139 arg140 arg141 arg142 arg143 arg144 arg145 arg146 arg147 arg148 arg149 arg150 arg151 arg152 arg153 arg154 arg155 arg156 arg157 arg158 arg159 arg160 arg161 arg162 arg163 arg164 arg165 arg166 arg167 arg168 arg169 arg170 arg171 arg172 arg173 arg174 arg175 arg176 arg177 arg178 arg179 arg180 arg181 arg182 arg183 arg184 arg185 arg186 arg187 arg188 arg189 arg190 arg191 arg192 arg193 arg194 arg195 arg196 arg197 arg198 arg199 arg200 arg201 arg202 arg203 arg204 arg205 arg206 arg207 arg208 arg209 arg210 arg211 arg212 arg213 arg214 arg215 arg216 arg217 arg218 arg219 arg220 arg221 arg222 arg223 arg224 arg225 arg226 arg227 arg228 arg229 arg230 arg231 arg232 arg233 arg234 arg235 arg236 arg237 arg238 arg239 arg240 arg241 arg242 arg243 arg244 arg245 arg246 arg247 arg248 arg249 arg250 arg251 arg252 arg253 arg254 arg255 arg256 arg257 arg258 arg259 arg260 arg261 arg262 arg263 arg264 arg265 arg266 arg267 arg268 arg269 arg270 arg271 arg272 arg273 arg274 arg275 arg276 arg277 arg278 arg279 arg280 arg281 arg282 arg283 arg284 arg285 arg286 arg287 arg288 arg289 arg290 arg291 arg292 arg293 arg294 arg295 arg296 arg297 arg298 arg299 arg300 arg301 arg302 arg303 arg304 arg305 arg306 arg307 arg308 arg309 arg310 arg311 arg312 arg313 arg314 arg315 arg316 arg317 arg318 arg319 arg320 arg321 arg322 arg323 arg324 arg325 arg326 arg327 arg328 arg329 arg330 arg331 arg332 arg333 arg334 arg335 arg336 arg337 arg338 arg339 arg340 arg341 arg342 arg343 arg344 arg345 arg346 arg347 arg348 arg349 arg350 arg351 arg352 arg353 arg354 arg355 arg356 arg357 arg358 arg359 arg360 arg361 arg362 arg363 arg364 arg365 arg366 arg367 arg368 arg369 arg370 arg371 arg372 arg373 arg374 arg375 arg376 arg377 arg378 arg379 arg380 arg381 arg382 arg383 arg384 arg385 arg386 arg387 arg388 arg389 arg390 arg391 arg392 arg393 arg394 arg395 arg396 arg397 arg398 arg399 arg400 arg401 arg402 arg403 arg404 arg405 arg406 arg407 arg408 arg409 arg410 arg411 arg412 arg413 arg414 arg415 arg416 arg417 arg418 arg419 arg420 arg421 arg422 arg423 arg424 arg425 arg426 arg427 arg428 arg429 arg430 arg431 arg432 arg433 arg434 arg435 arg436 arg437 arg438 arg439 arg440 arg441 arg442 arg443 arg444 arg445 arg446 arg447 arg448 arg449 arg450 arg451 arg452 arg453 arg454 arg455 arg456 arg457 arg458 arg459 arg460 arg461 arg462 arg463 arg464 arg465 arg466 arg467 arg468 arg469 arg470 arg471 arg472 arg473 arg474 arg475 arg476 arg477 arg478 arg479 arg480 arg481 arg482 arg483 arg484 arg485 arg486 arg487 arg488 arg489 arg490 arg491 arg492 arg493 arg494 arg495 arg496 arg497 arg498 arg499 arg500 arg501 arg502 arg503 arg504 arg505 arg506 arg507 arg508 arg509 arg510 arg511 arg512)
    (+ arg0 arg1 arg2 arg3 arg4 arg5 arg6 arg7 arg8 arg9 arg10 arg11 arg12 arg13 arg14 arg15 arg16 arg17 arg18 arg19 arg20 arg21 arg22 arg23 arg24 arg25 arg26 arg27 arg28 arg29 arg30 arg31 arg32 arg33 arg34 arg35 arg36 arg37 arg38 arg39 arg40 arg41 arg42 arg43 arg44 arg45 arg46 arg47 arg48 arg49 arg50 arg51 arg52 arg53 arg54 arg55 arg56 arg57 arg58 arg59 arg60 arg61 arg62 arg63 arg64 arg65 arg66 arg67 arg68 arg69 arg70 arg71 arg72 arg73 arg74 arg75 arg76 arg77 arg78 arg79 arg80 arg81 arg82 arg83 arg84 arg85 arg86 arg87 arg88 arg89 arg90 arg91 arg92 arg93 arg94 arg95 arg96 arg97 arg98 arg99 arg100 arg101 arg102 arg103 arg104 arg105 arg106 arg107 arg108 arg109 arg110 arg111 arg112 arg113 arg114 arg115 arg116 arg117 arg118 arg119 arg120 arg121 arg122 arg123 arg124 arg125 arg126 arg127 arg128 arg129 arg130 arg131 arg132 arg133 arg134 arg135 arg136 arg137 arg138 arg139 arg140 arg141 arg142 arg143 arg144 arg145 arg146 arg147 arg148 arg149 arg150 arg151 arg152 arg153 arg154 arg155 arg156 arg157 arg158 arg159 arg160 arg161 arg162 arg163 arg164 arg165 arg166 arg167 arg168 arg169 arg170 arg171 arg172 arg173 arg174 arg175 arg176 arg177 arg178 arg179 arg180 arg181 arg182 arg183 arg184 arg185 arg186 arg187 arg188 arg189 arg190 arg191 arg192 arg193 arg194 arg195 arg196 arg197 arg198 arg199 arg200 arg201 arg202 arg203 arg204 arg205 arg206 arg207 arg208 arg209 arg210 arg211 arg212 arg213 arg214 arg215 arg216 arg217 arg218 arg219 arg220 arg221 arg222 arg223 arg224 arg225 arg226 arg227 arg228 arg229 arg230 arg231 arg232 arg233 arg234 arg235 arg236 arg237 arg238 arg239 arg240 arg241 arg242 arg243 arg244 arg245 arg246 arg247 arg248 arg249 arg250 arg251 arg252 arg253 arg254 arg255 arg256 arg257 arg258 arg259 arg260 arg261 arg262 arg263 arg264 arg265 arg266 arg267 arg268 arg269 arg270 arg271 arg272 arg273 arg274 arg275 arg276 arg277 arg278 arg279 arg280 arg281 arg282 arg283 arg284 arg285 arg286 arg287 arg288 arg289 arg290 arg291 arg292 arg293 arg294 arg295 arg296 arg297 arg298 arg299 arg300 arg301 arg302 arg303 arg304 arg305 arg306 arg307 arg308 arg309 arg310 arg311 arg312 arg313 arg314 arg315 arg316 arg317 arg318 arg319 arg320 arg321 arg322 arg323 arg324 arg325 arg326 arg327 arg328 arg329 arg330 arg331 arg332 arg333 arg334 arg335 arg336 arg337 arg338 arg339 arg340 arg341 arg342 arg343 arg344 arg345 arg346 arg347 arg348 arg349 arg350 arg351 arg352 arg353 arg354 arg355 arg356 arg357 arg358 arg359 arg360 arg361 arg362 arg363 arg364 arg365 arg366 arg367 arg368 arg369 arg370 arg371 arg372 arg373 arg374 arg375 arg376 arg377 arg378 arg379 arg380 arg381 arg382 arg383 arg384 arg385 arg386 arg387 arg388 arg389 arg390 arg391 arg392 arg393 arg394 arg395 arg396 arg397 arg398 arg399 arg400 arg401 arg402 arg403 arg404 arg405 arg406 arg407 arg408 arg409 arg410 arg411 arg412 arg413 arg414 arg415 arg416 arg417 arg418 arg419 arg420 arg421 arg422 arg423 arg424 arg425 arg426 arg427 arg428 arg429 arg430 arg431 arg432 arg433 arg434 arg435 arg436 arg437 arg438 arg439 arg440 arg441 arg442 arg443 arg444 arg445 arg446 arg447 arg448 arg449 arg450 arg451 arg452 arg453 arg454 arg455 arg456 arg457 arg458 arg459 arg460 arg461 arg462 arg463 arg464 arg465 arg466 arg467 arg468 arg469 arg470 arg471 arg472 arg473 arg474 arg475 arg476 arg477 arg478 arg479 arg480 arg481 arg482 arg483 arg484 arg485 arg486 arg487 arg488 arg489 arg490 arg491 arg492 arg493 arg494 arg495 arg496 arg497 arg498 arg499 arg500 arg501 arg502 arg503 arg504 arg505 arg506 arg507 arg508 arg509 arg510 arg511 arg512))))
  (f512 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 57 58 59 60 61 62 63 64 65 66 67 68 69 70 71 72 73 74 75 76 77 78 79 80 81 82 83 84 85 86 87 88 89 90 91 92 93 94 95 96 97 98 99 100 101 102 103 104 105 106 107 108 109 110 111 112 113 114 115 116 117 118 119 120 121 122 123 124 125 126 127 128 129 130 131 132 133 134 135 136 137 138 139 140 141 142 143 144 145 146 147 148 149 150 151 152 153 154 155 156 157 158 159 160 161 162 163 164 165 166 167 168 169 170 171 172 173 174 175 176 177 178 179 180 181 182 183 184 185 186 187 188 189 190 191 192 193 194 195 196 197 198 199 200 201 202 203 204 205 206 207 208 209 210 211 212 213 214 215 216 217 218 219 220 221 222 223 224 225 226 227 228 229 230 231 232 233 234 235 236 237 238 239 240 241 242 243 244 245 246 247 248 249 250 251 252 253 254 255 256 257 258 259 260 261 262 263 264 265 266 267 268 269 270 271 272 273 274 275 276 277 278 279 280 281 282 283 284 285 286 287 288 289 290 291 292 293 294 295 296 297 298 299 300 301 302 303 304 305 306 307 308 309 310 311 312 313 314 315 316 317 318 319 320 321 322 323 324 325 326 327 328 329 330 331 332 333 334 335 336 337 338 339 340 341 342 343 344 345 346 347 348 349 350 351 352 353 354 355 356 357 358 359 360 361 362 363 364 365 366 367 368 369 370 371 372 373 374 375 376 377 378 379 380 381 382 383 384 385 386 387 388 389 390 391 392 393 394 395 396 397 398 399 400 401 402 403 404 405 406 407 408 409 410 411 412 413 414 415 416 417 418 419 420 421 422 423 424 425 426 427 428 429 430 431 432 433 434 435 436 437 438 439 440 441 442 443 444 445 446 447 448 449 450 451 452 453 454 455 456 457 458 459 460 461 462 463 464 465 466 467 468 469 470 471 472 473 474 475 476 477 478 479 480 481 482 483 484 485 486 487 488 489 490 491 492 493 494 495 496 497 498 499 500 501 502 503 504 505 506 507 508 509 510 511 512))
      131328)


(let ((x 32))
  (define (f1) x)
  (define x 33)
  (test (f1) 33))
    
(let ()
  (define (f2 a) (+ a 1))
  (define (f1 a) (f2 a))
  (define (f2 a) (- a))
  (test (f1 12) -12))





;;; --------------------------------------------------------------------------------
;;; values
;;; call-with-values
;;; --------------------------------------------------------------------------------


(test (call-with-values (lambda () (values 1 2 3)) +) 6)
(test (call-with-values (lambda () (values 4 5)) (lambda (a b) b))  5)
(test (call-with-values (lambda () (values 4 5)) (lambda (a b) (+ a b))) 9)
(test (call-with-values * -) -1) ; yeah, right... (- (*))
(test (values 1) 1)
(test (call-with-values (lambda () (values 1 2 3 4)) list) (list 1 2 3 4))
(test (+ (values 1) (values 2)) 3)
(test (+ (values '1) (values '2)) 3)
(test (if (values #t) 1 2) 1)
(test (if (values '#t) 1 2) 1)
(test (if (values #f) 1 2) 2)
(test (if (values #f #f) 1 2) 1)
(test (equal? (values #t #t)) #t)
(test (call-with-values (lambda () 4) (lambda (x) x)) 4)
(test (let () (values 1 2 3) 4) 4)
(test (apply + (values '())) 0)
(test (+ (values 1 2 3)) 6)
(test (let ((f (lambda () (values 1 2 3)))) (+ (f))) 6)
(num-test (log (values 8 2)) 3)
(test (* (values 2 (values 3 4))) 24)
(test (* (values (+ (values 1 2)) (- (values 3 4)))) -3)
(test (list (values 1 2) (values 3) 4) '(1 2 3 4))
(test (let ((f1 (lambda (x) (values x (+ x 1)))) (f2 (lambda () (values 2)))) (+ (f1 3) (* 2 (f2)))) 11)
(test (+ (let () (values 1 2)) 3) 6)
(test (let () (values 1 2) 4) 4)
(test (let () + (values 1 2) 4) 4)
(test (string-ref (values "hiho" 2)) #\h)
(test (vector-ref (values (vector 1 2 3)) 1) 2)
(test (+ (values (+ 1 (values 2 3)) 4) 5 (values 6) (values 7 8 (+ (values 9 10) 11))) 66)
(test (+ (if (values) (values 1 2) (values 3 4)) (if (null? (values)) (values 5 6) (values 7 8))) 18) ; (values) is now #<unspecified> (sort of)
(test (+ (cond (#f (values 1 2)) (#t (values 3 4))) 5) 12)
(test (+ (cond (#t (values 1 2)) (#f (values 3 4))) 5) 8)
(test (apply + (list (values 1 2))) 3)
(test (apply + (list ((lambda (n) (values n (+ n 1))) 1))) 3)
(test (+ (do ((i 0 (+ i 1))) ((= i 3) (values i (+ i 1))))) 7)
(test (+ (with-input-from-string "(values 1 2 3)" (lambda () (eval (read)))) 2) 8)
(test (< (values 1 2 3)) #t)
(test (apply (values + 1 2) '(3)) 6)
(test (let () (define-macro (hi a) `(+ 1 ,a)) (hi (values 1 2 3))) 7)
(test (+ 1 (eval-string "(values 2 3 4)")) 10)
(test (+ 1 (eval '(values 2 3 4))) 10)
(test (or (values #t) #f) #t)
(test (and (values #t) #f) #f)
(test (let ((x 1)) (set! x (values 32)) x) 32)
(test (let ((x #(32 33))) ((values x) 0)) 32)
(test (let ((x #(32 33))) (set! ((values x) 0) 123) x) #(123 33))
(test (list-ref '(1 (2 3)) (values 1 1)) 3)
(test (list-ref (values '(1 (2 3)) 1 1)) 3)
(test (list-ref ((lambda () (values '(1 (2 3)) 1 1)))) 3)
(test (set! (values) 1) 'error)
(test (+ (values (begin (values 1 2)) (let ((x 1)) (values x (+ x 1))))) 6)
(test (vector 1 (values 2 3) 4) #(1 2 3 4))
(test (vector (values 1 (values 2 3) (values (values 4)))) #(1 2 3 4))
(test(+ 1 (values (values (values 2) 3) (values (values (values 4)) 5) 6) 7) 28)
(test (map (values values #(1 2))) '(1 2))
(test ((values values) (values 0)) 0)
(test ((object->string values) (abs 1)) #\a)
(test (list? (values 1 2 3)) 'error)
(test (list? (values 1)) #f)
(test (list? (values (list 1 2 3))) #t)

(test (let ((x 1)) (set! x (values)) x) 'error)
(test (let ((x 1)) (set! x (values 1 2 3)) x) 'error)
(test (let ((x 1)) (set! x (values 2)) x) 2)
(test (let ((x 1)) (set! (values x) 2) x) 'error) ; (no generalized set for values, so (values x) is not the same as x
(test (let ((x #(0 1))) (set! (values x 0 32)) x) 'error)
(test (let ((var (values 1 2 3))) var) 'error)
(test (let* ((var (values 1 2 3))) var) 'error)
(test (letrec ((var (values 1 2 3))) var) 'error)
(test (let ((x ((lambda () (values 1 2))))) x) 'error)
(test (+ 1 ((lambda () ((lambda () (values 2 3)))))) 6)
(test (let () (define (hi) (symbol? (values 1 2 3))) (hi)) 'error)
(test (let () (define (hi) (symbol? (values))) (hi)) #f) ; this is consistent with earlier such cases: (boolean? (values))
(test (let () (define (hi) (symbol? (values 'a))) (hi)) #t)
(test (let () (define (hi) (symbol? (values 1))) (hi)) #f)

(test (let ((str "hi")) (string-set! (values str 0 #\x)) str) "xi")
(test (values if) if)
(test (values quote) quote)

(test ((values '(1 (2 3)) 1 1)) 3)
(test (let ((x #(32 33))) ((values x 0))) 32)
(test (+ 1 (apply values '(2 3 4))) 10)
(test (+ 1 ((lambda args (apply values args)) 2 3 4)) 10)
(test (apply begin '(1 2 3)) 3)
(test (let ((x 1)) ((values set!) x 32) x) 32)

(test (or (values #t #f) #f) #t)
(test (or (values #f #f) #f) #f)
(test (or (values #f #t) #f) #t)
(test (or (values #f #f) #t) #t)
(test (or (values 1 2) #f) 1)
(test (+ 1 (or (values 2 3) 4)) 3)
(test (+ 1 (and 2 (values 3 4)) 5) 13)
(test (and (values) 1) 1)
(test (and (values 1 2 #f) 4) #f)
(test (and (values 1 2 3) 4) 4)
(test (length (values '())) 0)
(test (length (values #(1 2 3 4))) 4)
(test (vector? (values #())) #t)
(test (map + (values '(1 2 3) #(1 2 3))) '(2 4 6))
(test (map + (values '(1 2 3)) (values #(1 2 3))) '(2 4 6))
(test (map + (values '(1 2 3) #(4 5 6)) (values '(7 8 9))) '(12 15 18))

(test (let ((x 1)) 
	(and (let () (set! x 2) #f) 
	     (let () (set! x 3) #f)) 
	x) 2)
(test (let ((x 1)) 
	(and (values (let () (set! x 2) #f) 
		     (let () (set! x 3) #f)))
	x) 3)

(test (+ (values 1 2) 3) 6)
(test (+ (values 1 (values 2))) 3)
(test (list (values 1 2)) '(1 2))
(test (+ 6 (values 1 (values 2 3) 4 ) 5) 21)
(test (+ ((lambda (x) (values (+ 1 x))) 2) 3) 6)
(test (list ((lambda (x) (values (+ 1 x))) 2)) '(3))
(test (+ (begin (values 1 2))) 3)
(test (+ 1 (let () (values 1 2))) 4)
(test (apply (values + 1 2) (list 3)) 6)
(test ((lambda* ((a 1) (b 2)) (list a b)) (values :a 3)) '(3 2))
(test (+ (values (values 1 2) (values 4 5))) 12)
(test (+ (begin 3 (values 1 2) 4)) 4)
(test (map (lambda (x) (if #f x (values))) (list 1 2)) '())
(test (map (lambda (x) (if #f x (begin (values)))) (list 1 2)) '())
(test (map (lambda (x) (if (odd? x) (values x (* x 20)) (values))) (list 1 2 3 4)) '(1 20 3 60))
(test (map (lambda (x) (if (odd? x) (apply values '(1 2 3)) (values))) (list 1 2 3 4)) '(1 2 3 1 2 3))
(test (object->string (map (lambda (x) (if (odd? x) (values x (* x 20)) (values))) (list 1 2 3 4))) "(1 20 3 60)") ; make sure no "values" floats through
(test (map (lambda (x) (if (odd? x) (values x (* x 20) (cons x (+ x 1))) (values))) (list 1 2 3 4 5 6)) '(1 20 (1 . 2) 3 60 (3 . 4) 5 100 (5 . 6)))
(test (* 2 (case 1 ((2) (values 3 4)) ((1) (values 5 6)))) 60)
(test (* 2 (case 1 ((2) (values 3 4)) (else (values 5 6)))) 60)
(test (* 2 (case 1 ((1) (values 3 4)) (else (values 5 6)))) 24)
(test (+ (values (* 3 2) (abs (values -1)))) 7)
(test (+ (let ((x 1)) (values x (+ x 1))) (if #f #f (values 2 3))) 8)

(test (let ((sum 0)) (for-each (lambda (n m p) (set! sum (+ sum n m p))) (values (list 1 2 3) (list 4 5 6) (list 7 8 9))) sum) 45)
(test (map (lambda (n m p) (+ n m p)) (values (list 1 2 3) (list 4 5 6) (list 7 8 9))) '(12 15 18))
(test (string-append (values "123" "4" "5") "6" (values "78" "90")) "1234567890")
(test (+ (dynamic-wind (lambda () #f) (lambda () (values 1 2 3)) (lambda () #f)) 4) 10)

(for-each
 (lambda (arg)
   (test (values arg) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (call-with-values (lambda () (values arg arg)) (lambda (a b) b)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call-with-values (lambda () (values "hi" 1 3/2 'a)) (lambda (a b c d) (+ b c))) 5/2)
					;(test (call-with-values values (lambda arg arg)) '())
(test (string-ref (values "hi") 1) #\i)
(test ((lambda (a b) (+ a b)) ((lambda () (values 1 2)))) 3)

(test (list (letrec ((split (lambda (ls)
			      (if (or (null? ls) (null? (cdr ls)))
				  (values ls '())
				  (call-with-values
				      (lambda () (split (cddr ls)))
				    (lambda (odds evens)
				      (values (cons (car ls) odds)
					      (cons (cadr ls) evens))))))))
	      (split '(a b c d e f))))
      '((a c e) (b d f)))

(test (call-with-values (lambda () (call/cc (lambda (k) (k 2 3)))) (lambda (x y) (list x y))) '(2 3))
(test (+ (call/cc (lambda (return) (return (values 1 2 3)))) 4) 10)

(test (let ((values 3)) (+ 2 values)) 5)
(test (let ((a (values 1))) a) 1)

(test (call-with-values (lambda () 2) (lambda (x) x)) 2)
(test (call-with-values (lambda () -1) abs) 1)
(test (call-with-values (lambda () (values -1)) abs) 1)
(test (call-with-values (lambda () (values -1)) (lambda (a) (abs a))) 1)

(test (call-with-values 
	  (lambda ()
	    (values
	     (call-with-values (lambda () (values 1 2 3)) +)
	     (call-with-values (lambda () (values 1 2 3 4)) *)))
	(lambda (a b)
	  (- a b)))
      -18)

(test (call-with-values 
	  (lambda ()
	    (values
	     (call-with-values (lambda () (values 1 2 3)) +)
	     (call-with-values (lambda () (values 1 2 3 4)) *)))
	(lambda (a b)
	  (+ (* a (call-with-values (lambda () (values 1 2 3)) +))
	     (* b (call-with-values (lambda () (values 1 2 3 4)) *)))))
      612)

(test (call-with-values (lambda (x) (+ x 1)) (lambda (y) y)) 'error)
(test (+ (values . 1)) 'error)
(for-each
 (lambda (arg)
   (test (call-with-values arg arg) 'error))
 (list "hi" -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
(test (call-with-values (lambda () (values -1 2)) abs) 'error)

(test (multiple-value-bind (a b) (values 1 2) (+ a b)) 3)
(test (multiple-value-bind (a) 1 a) 1)
(test (multiple-value-bind (a . rest) (values 1 2 3) (+ a (apply + rest))) 6)
(test (multiple-value-bind a (values 1 2 3) a) '(1 2 3))

(test (let ((a 1)
	    (b 2))
	(multiple-value-set! (a b) (values 32 64))
	(+ a b))
      96)
(test (let ((add (lambda (a b) (values (+ a 1) (+ b 1))))) (+ 1 (add 2 3))) 8)
(test (min (values 1 2) (values 3 0)) 0)
(test ((lambda* ((a 1) (b 2)) (list a b)) (values :b 231)) '(1 231))
(test (cons (values 1 2) (values 3 4)) 'error)

(test (cond ((values) 3) (#t 4)) 3)          ; an error in Guile "zero values returned"
(test (cond ((values (values)) 3) (#t 4)) 3) ; same
(test (+ (cond (#t (values 1 2)))) 3)        ; 1 in guile
(test (+ (cond ((values 3 4) => (lambda (a) a)))) 'error)
(test (+ (cond ((values 3 4) => (lambda (a b) (values a b))))) 7)
(test (+ 1 (cond ((values 2 3))) 4) 10)
(test (+ 1 (values)) 'error)

(test (case (values 1) ((1) 2) (else 3)) 2)
(test (case (values 1 2) ((1) 2) (else 3)) 3)
(test (case (values 1) (((values 1)) 2) (else 3)) 3)
(test (case (values 1 2) (((values 1 2)) 2) (else 3)) 3)

(test ((values) 0) 'error)
(test ((values "hi") 1) #\i)
(test (string-ref (values "hi") 0) #\h)
(test (string-ref (values "hi" "ho") 0) 'error)
(test (let ((str "hi")) (set! ((values str) 0) #\x) str) "xi")
(test (let ((str "hi")) (string-set! (values str) 0 #\x) str) "xi")
(test (let ((str "hi")) (set! (values str 0) #\x) str) 'error)
(test (let ((str "hi")) (string-set! (values str 0) #\x) str) "xi")

(test ((values 1 2 3) 0) 'error)
(test ((values "hi" "ho") 1) 'error)
(test ((values + 1 2 3)) 6)
(test ((values + 1 2) 3) 6)
(test ((values +) 1 2 3) 6)
(test ((values "hi" 0)) #\h)
(test ((values + 1) (values 2 3) 4) 10)
(test ((values - 10)) -10)
(test ((values - -10) 0) -10) ; looks odd but it's (- -10 0) that is (- a) != (- a 0)
(test ((values - 2 3) 0) -1)
(test ((values - 2 3) 1) -2)
(test ((values - 2 3) 2) -3)  ; it's actually (- 2 3 2) -> -3

(test (let ((str "hi")) (set! ((values str 0) 0) #\x) str) 'error)
(test (let ((str "hi")) (set! ((values str) 0) #\x) str) "xi")
(test (+ (let ((x 0)) (do ((i (values 0) (+ i 1))) (((values = i 10)) (values x 2 3)) (set! x (+ x i)))) 4) 54)

(test (map values (list (values 1 2) (values 3 4))) '(1 2 3 4))
(test (let () (define-macro (hi a) `(+ 1 ,a)) (hi (values 2 3 4))) 10)
(test (let () (+ 4 (let () (values 1 2 3)) 5)) 15)
(test (let* () (+ 4 (let () (values 1 2 3)) 5)) 15)
(test (let () (+ 4 (let* () (values 1 2 3)) 5)) 15)
(test (letrec () (+ 4 (let () (values 1 2 3)) 5)) 15)
(test (let () (+ 4 (letrec () (values 1 2 3)) 5)) 15)
(test (letrec* () (+ 4 (let () (values 1 2 3)) 5)) 15)
(test (let* () (+ 4 (letrec* () (values 1 2 3)) 5)) 15)

(test (cons (values 1 2)) '(1 . 2))
(test (number->string (values 1 2)) "1")
(test (object->string (values)) "#<unspecified>")
(test (equal? (values) #<unspecified>) #f) ; hmmm -- this means that
(test (equal? (begin) (begin (values))) #f) ;   maybe this is a bug, but we want this:
(test (map (lambda (x) (if #f x #<unspecified>)) (list 1 2)) '(#<unspecified> #<unspecified>))
(test (equal? (values) (if #f #f)) #f)
(test (substring (values "hi") (values 1 2)) "i")
(test (cond (call-with-exit (values "hi"))) "hi")
(test (procedure-arity (cond (values))) '(0 0 #t)) ; values as a procedure here
(test (procedure-arity 'values) '(0 0 #t))
(test (values (begin (values "hi"))) "hi")
(test (< (values (values 1 2))) #t)

(test (let ((lst (list 0)))
	(set-cdr! lst lst)
	(format (values #f "~A" lst)))
      "#1=(0 . #1#)")

(let ()
  (define (mv n)
    (define (mv-1 a)
      (values a (+ a 1)))
    (define (mv-2 b)
      (values b (* b 2)))
    (values n (mv-1 n) (mv-2 n)))
  (test (list (mv 2)) '(2 2 3 2 4))
  (test (+ (mv 1) (mv 3)) 26))

(let ()
  (define (fib n)
    (define (1st a b) 
      a)
    (define (fib-1 n)
      (if (< n 3)
	  (values 1 1)
	  (values (+ (fib-1 (- n 1))) 
		  (1st (fib-1 (- n 1))))))
    (1st (fib-1 n)))
  (test (fib 8) 21)
  (test (fib 13) 233))

(let ()
  (define (fib n)
    (define (1st a b) 
      a)
    (define (2nd a b) 
      (values (+ a b) a))
    (define (fib-1 n)
      (if (< n 3)
	  (2nd 1 0)
	  (2nd (fib-1 (- n 1)))))
    (1st (fib-1 n)))
  (test (fib 8) 21)
  (test (fib 13) 233))

(let ()
  (define (flatten lst) ; flatten via values and map
    (define (flatten-1 lst)
      (cond ((null? lst) (values))
	    ((not (pair? lst)) lst)
	    (#t (values (flatten-1 (car lst))
			(flatten-1 (cdr lst))))))
    (map values (list (flatten-1 lst))))

  (test (flatten '(1 2 3)) '(1 2 3))
  (test (flatten '()) '())
  (test (flatten '((1) 2 (3 4) (6 (7)))) '(1 2 3 4 6 7))
  (test (flatten '(1 ((((2)) 3)))) '(1 2 3))
  (test (flatten '(1 () 2)) '(1 2))
  (test (flatten '((1 () 2) ())) '(1 2))
  (test (flatten '(() 1 ((2 (3)) () 4))) '(1 2 3 4))
  (test (flatten '((1) 2 ((3 4) 5) ((())) (((6))) 7 8 ())) '(1 2 3 4 5 6 7 8))
  (test (flatten '(() 1 () ((2 (1)) 4) (3 2) ())) '(1 2 1 4 3 2))
  )

(let ()
  (define (flatten! lst) ; in-place flatten
    (if (not (pair? lst))
	lst
	(let loop ((L lst))
	  (if (pair? (car L))
	      (let ((end (cdr L))
		    (p (car L)))
		(set! (car L) (car p))
		(set! (cdr L) (cdr p))
		(set! (cdr (list-tail L (- (length p) 1))) end)
		(loop L))
	      (if (not (null? (cdr L)))
		  (if (null? (car L))
		      (begin
			(set! (car L) (cadr L))
			(set! (cdr L) (cddr L))
			(loop L))
		      (loop (cdr L)))))
	  (if (equal? lst '(()))
	      '()
	      (let ((len (length lst)))
		(if (null? (car (list-tail lst (- len 1))))
		    (set! (cdr (list-tail lst (- len 2))) '()))
		lst)))))

  (test (flatten! '(1 2 3)) '(1 2 3))
  (test (flatten! '()) '())
  (test (flatten! '((1) 2 (3 4) (6 (7)))) '(1 2 3 4 6 7))
  (test (flatten! '(1 ((((2)) 3)))) '(1 2 3))
  (test (flatten! '(1 () 2)) '(1 2))
  (test (flatten! '((1 () 2) ())) '(1 2))
  (test (flatten! '(() 1 ((2 (3)) () 4))) '(1 2 3 4))
  (test (flatten! '((1) 2 ((3 4) 5) ((())) (((6))) 7 8 ())) '(1 2 3 4 5 6 7 8))
  (test (flatten! '(() 1 () ((2 (1)) 4) (3 2) ())) '(1 2 1 4 3 2))
  )

(let ()
  (define (flatten x) ; standard flatten
    (cond ((null? x) '())
          ((not (pair? x)) (list x))
          (#t (append (flatten (car x))
		      (flatten (cdr x))))))

  (test (flatten '(1 2 3)) '(1 2 3))
  (test (flatten '()) '())
  (test (flatten '((1) 2 (3 4) (6 (7)))) '(1 2 3 4 6 7))
  (test (flatten '(1 ((((2)) 3)))) '(1 2 3))
  (test (flatten '(1 () 2)) '(1 2))
  (test (flatten '((1 () 2) ())) '(1 2))
  (test (flatten '(() 1 ((2 (3)) () 4))) '(1 2 3 4))
  (test (flatten '((1) 2 ((3 4) 5) ((())) (((6))) 7 8 ())) '(1 2 3 4 5 6 7 8))
  (test (flatten '(() 1 () ((2 (1)) 4) (3 2) ())) '(1 2 1 4 3 2))
  )

(test (let () (define (hi a) (+ (abs a) (values 1 2 3))) (hi -4)) 10)
(let ()
  (define (hi a)
    (let ((x 0)
	  (again #f))
      (let ((y (+ (abs a) (call/cc (lambda (r) (set! again r) 1)))))
	(set! x (+ x y))
	(if (< x 3) (again 1))
	x)))
  (test (hi 0) 3))





;;; --------------------------------------------------------------------------------
;;; let
;;; let*
;;; letrec
;;; --------------------------------------------------------------------------------

(test (let ((x 2) (y 3)) (* x y)) 6)
(test (let ((x 32)) (let ((x 3) (y x)) y)) 32)
(test (let ((x 32)) (let* ((x 3) (y x)) y)) 3)
(test (let ((x 2) (y 3)) (let ((x 7) (z (+ x y))) (* z x))) 35)
(test (let ((x 2) (y 3)) (let* ((x 7)  (z (+ x y))) (* z x))) 70)
(test (letrec ((even? (lambda (n)  (if (zero? n) #t (odd? (- n 1))))) (odd? (lambda (n)  (if (zero? n) #f (even? (- n 1)))))) (even? 88))  #t)
(test (let loop ((numbers '(3 -2 1 6 -5)) 
		 (nonneg '()) 
		 (neg '())) 
	(cond ((null? numbers) 
	       (list nonneg neg)) 
	      ((>= (car numbers) 0)  
	       (loop (cdr numbers) (cons (car numbers) nonneg)  neg)) 
	      ((< (car numbers) 0)  
	       (loop (cdr numbers) nonneg (cons (car numbers) neg))))) 
      '((6 1 3) (-5 -2)))
(test(let((i 1)(j 2))(+ i j))3)

(test (let ((x 3)) (define x 5) x) 5)
(test (let* () (define x 8) x) 8)
(test (letrec () (define x 9) x) 9)
(test (letrec ((x 3)) (define x 10) x) 10)
(test (let foo () 1) 1)
(test (let ((f -)) (let f ((n (f 1))) n)) -1)
(test (let () 1 2 3 4) 4)
(test (+ 3 (let () (+ 1 2))) 6)

(test (let ((x 1)) (let ((x 32) (y x)) y)) 1)
(test (let ((x 1)) (letrec ((y (if #f x 1)) (x 32)) 1)) 1)
(test (let ((x 1)) (letrec ((y (lambda () (+ 1 x))) (x 32)) (y))) 33) 
(test (let ((x 1)) (letrec ((y (* 0 x)) (x 32)) y)) 'error)
(test (let* ((x 1) (f (letrec ((y (lambda () (+ 1 x))) (x 32)) y))) (f)) 33)
(test (letrec ((x 1) (y (let ((x 2)) x))) (+ x y)) 3)
(test (letrec ((f (lambda () (+ x 3))) (x 2)) (f)) 5)
(test (let* ((x 1) (x 2)) x) 2)
(test (let* ((x 1) (y x)) y) 1)
(test (let ((x 1)) (let ((x 32) (y x)) (+ x y))) 33)
(test (let ((x 1)) (let* ((x 32) (y x)) (+ x y))) 64)
(test (let ((x 'a) (y '(b c))) (cons x y)) '(a b c))
(test (let ((x 0) (y 1)) (let ((x y) (y x)) (list x y))) (list 1 0))
(test (let ((x 0) (y 1)) (let* ((x y) (y x)) (list x y))) (list 1 1))
(test (letrec ((sum (lambda (x) (if (zero? x) 0 (+ x (sum (- x 1))))))) (sum 5)) 15)
(test (let ((divisors (lambda (n) (let f ((i 2)) (cond ((>= i n) '()) ((integer? (/ n i)) (cons i (f (+ i 1)))) (else (f (+ i 1)))))))) (divisors 32)) '(2 4 8 16))
(test (let ((a -1)) (let loop () (if (not (positive? a)) (begin (set! a (+ a 1)) (loop)))) a) 1)
(test (let () (let () (let () '()))) '())
(test (let ((x 1)) (let ((y 0)) (begin (let ((x (* 2 x))) (set! y x))) y)) 2)
(test (let* ((x 1) (x (+ x 1)) (x (+ x 2))) x) 4)
(test (let ((.. 2) (.... 4) (..... +)) (..... .. ....)) 6)

(test (let () (begin (define x 1)) x) 1)
(test (let ((define 1)) define) 1)
(test (let ((y 1)) (begin (define x 1)) (+ x y)) 2)
(test (let ((: 0)) (- :)) 0) 
;; this only works if we haven't called (string->symbol "") making : into a keyword (see also other cases below)
;; perhaps I should document this weird case -- don't use : as a variable name

(test ((let ((x 2))
	 (let ((x 3))
	   (lambda (arg) (+ arg x))))
       1)
      4)

(test ((let ((x 2))
	 (define (inner arg) (+ arg x))
	 (let ((x 32))
	   (lambda (arg) (inner (+ arg x)))))
       1)
      35)

(test ((let ((inner (lambda (arg) (+ arg 1))))
	 (let ((inner (lambda (arg) (inner (+ arg 2)))))
	   inner))
       3)
      6)

(test ((let ()
	 (define (inner arg) (+ arg 1))
	 (let ((inner (lambda (arg) (inner (+ arg 2)))))
	   inner))
       3)
      6)

(test ((let ((x 11))
	 (define (inner arg) (+ arg x))
	 (let ((inner (lambda (arg) (inner (+ (* 2 arg) x)))))
	   inner))
       3)
      28)

(test ((let ((x 11))
	 (define (inner arg) (+ arg x))
	 (let ((x 2))
	   (lambda (arg) (inner (+ (* 2 arg) x)))))
       3)
      19)

(test (let ((f1 (lambda (arg) (+ arg 1))))
	(let ((f1 (lambda (arg) (f1 (+ arg 2)))))
	  (f1 1)))
      4)

(test (let ((f1 (lambda (arg) (+ arg 1))))
	(let* ((f1 (lambda (arg) (f1 (+ arg 2)))))
	  (f1 1)))
      4)

(test (let ((f1 (lambda (arg) (+ arg 1))))
	(let* ((x 32)
	       (f1 (lambda (arg) (f1 (+ x arg)))))
	  (f1 1)))
      34)

(test ((let ((x 11))
	 (define (inner arg) (+ arg x))
	 (let ((x 2)
	       (inner (lambda (arg) (inner (+ (* 2 arg) x)))))
	   inner))
       3)
      28)

(test ((let ((x 11))
	 (define (inner arg) (+ arg x))
	 (let* ((x 2)
		(inner (lambda (arg) (inner (+ (* 2 arg) x)))))
	   inner))
       3)
      19)

(test (let ((x 1))
	(let* ((f1 (lambda (arg) (+ x arg)))
	       (x 32))
	  (f1 1)))
      2)

(test (let ((inner (lambda (arg) (+ arg 1))))
	(let ((inner (lambda (arg) (+ (inner arg) 1))))
	  (inner 1)))
      3)
(test (let ((inner (lambda (arg) (+ arg 1))))
	(let* ((inner (lambda (arg) (+ (inner arg) 1))))
	  (inner 1)))
      3)

(test (let ((caller #f)) (let ((inner (lambda (arg) (+ arg 1)))) (set! caller inner)) (caller 1)) 2)
(test (let ((caller #f)) (let ((x 11)) (define (inner arg) (+ arg x)) (set! caller inner)) (caller 1)) 12)

(test (let ((caller #f)) 
	(let ((x 11)) 
	  (define (inner arg) 
	    (+ arg x)) 
	  (let ((y 12))
	    (let ((inner (lambda (arg) 
			   (+ (inner x) y arg)))) ; 11 + 11 + 12 + arg
	      (set! caller inner))))
	(caller 1))
      35)

(test (let ((caller #f)) 
	(let ((x 11)) 
	  (define (inner arg) 
	    (+ arg x)) 
	  (let* ((y 12) 
		 (inner (lambda (arg) 
			  (+ (inner x) y arg)))) ; 11 + 11 + 12 + arg
	    (set! caller inner))) 
	(caller 1))
      35)


(test (let* ((f1 3) (f1 4)) f1) 4)
(test (let ((f1 (lambda () 4))) (define (f1) 3) (f1)) 3)

(test (let ((j -1)
	    (k 0))
	(do ((i 0 (+ i j))
	     (j 1))
	    ((= i 3) k)
	  (set! k (+ k i))))
      3)

(test (let ((j (lambda () -1))
	    (k 0))
	(do ((i 0 (+ i (j)))
	     (j (lambda () 1)))
	    ((= i 3) k)
	  (set! k (+ k i))))
      3)


(test (let ((j (lambda () 0))
	    (k 0))
	(do ((i (j) (j))
	     (j (lambda () 1) (lambda () (+ i 1))))
	    ((= i 3) k)
	  (set! k (+ k i))))
      3) ; 6 in Guile which follows the spec

(test (let ((k 0)) (do ((i 0 (+ i 1)) (j 0 (+ j i))) ((= i 3) k) (set! k (+ k j)))) 1)

#|
(test (let ((j (lambda () 0))
	    (i 2)
	    (k 0))
	(do ((i (j) (j))
	     (j (lambda () i) (lambda () (+ i 1))))
	    ((= i 3) k)
	  (set! k (+ k i))))
      3) ; or 2?

(test (let ((f #f))
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (let ()
	    (define (x) i)
	    (if (= i 1) (set! f x))))
	(f))
      1)
|#

(test (let ((x 1))
	(let ()
	  (define (f) x)
	  (let ((x 0))
	    (define (g) (set! x 32) (f))
	    (g))))
      1)

(test (let ((a 1))
	(let ()
	  (if (> a 1)
	      (begin
		(define a 2)))
	  a))
      1)

(test (let ((a 1))
	(let ()
	  (if (= a 1)
	      (begin
		(define a 2)))
	  a))
      2)

(let ((x 123))
  (define (hi b) (+ b x))
  (let ((x 321))
    (test (hi 1) 124)
    (set! x 322)
    (test (hi 1) 124))
  (set! x 124)
  (test (hi 1) 125)
  (let ((x 321)
	(y (hi 1)))
    (test y 125))
  (let* ((x 321)
	 (y (hi 1)))
    (test y 125))
  (test (hi 1) 125))

(test (let ((j 0)
	    (k 0))
	(let xyz
	    ((i 0))
	  (let xyz
	      ((i 0))
	    (set! j (+ j 1))
	    (if (< i 3)
		(xyz (+ i 1))))
	  (set! k (+ k 1))
	  (if (< i 3)
	      (xyz (+ i 1))))
	(list j k))
      (list 16 4))

(test (let ((x 123)) (begin (define x 0)) x) 0) ; this strikes me as weird, since (let ((x 123) (x 0)) x) is illegal, so...
(test (let ((x 123)) (begin (define (hi a) (+ x a)) (define x 0)) (hi 1)) 1) ; is non-lexical reference?

(for-each
 (lambda (arg)
   (test (let ((x arg)) x) arg))
 (list "hi" -1 #\a "" '() '#() (current-output-port) 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t abs (list 1 2 3) '(1 . 2)))

(test (let ((x 1)) (= 1 (let ((y 2)) (set! x y) x)) (+ x 1)) 3)
(test (let ((x 1)) (let ((xx (lambda (a) (set! x a) a))) (= 1 (xx 2))) (+ x 1)) 3)
(test (let ((x 32)) (begin (define x 123) (define (hi a) (+ a 1))) (hi x)) 124)
(test (let () (begin (define x 123) (define (hi a) (+ a 1))) (hi x)) 124)


					;(let ((initial-chars "aA!$%&*/:<=>?^_~")
					;      (subsequent-chars "9aA!$%&*+-./:<=>?@^_~")
					;      (ctr 0))
					;  (format #t ";(let (")
					;  (do ((i 0 (+ i 1)))
					;      ((= i (string-length initial-chars)))
					;    (format #t ";(~A ~D) " (string (string-ref initial-chars i)) ctr)
					;    (set! ctr (+ ctr 1)))
					;
					;  (do ((i 0 (+ i 1)))
					;      ((= i (string-length initial-chars)))
					;    (do ((k 0 (+ k 1)))
					;	((= k (string-length subsequent-chars)))
					;      (format #t ";(~A ~D) " (string (string-ref initial-chars i) (string-ref subsequent-chars k)) ctr)
					;      (set! ctr (+ ctr 1))))
					;
					;  (format #t ")~%  (+ ")
					;  (do ((i 0 (+ i 1)))
					;      ((= i (string-length initial-chars)))
					;    (format #t "~A " (string (string-ref initial-chars i))))
					;
					;  (do ((i 0 (+ i 1)))
					;      ((= i (string-length initial-chars)))
					;    (do ((k 0 (+ k 1)))
					;	((= k (string-length subsequent-chars)))
					;      (format #t "~A " (string (string-ref initial-chars i) (string-ref subsequent-chars k)))))
					;
					;  (format #t "))~%"))

(num-test (let ((a 0) (A 1) (! 2) ($ 3) (% 4) (& 5) (* 6) (/ 7) (| 8) (< 9) (= 10) (> 11) (? 12) (^ 13) (_ 14) (~ 15) (a9 16) (aa 17) (aA 18) (a! 19) (a$ 20) (a% 21) (a& 22) (a* 23) (a+ 24) (a- 25) (a. 26) (a/ 27) (a| 28) (a< 29) (a= 30) (a> 31) (a? 32) (a@ 33) (a^ 34) (a_ 35) (a~ 36) (A9 37) (Aa 38) (AA 39) (A! 40) (A$ 41) (A% 42) (A& 43) (A* 44) (A+ 45) (A- 46) (A. 47) (A/ 48) (A| 49) (A< 50) (A= 51) (A> 52) (A? 53) (A@ 54) (A^ 55) (A_ 56) (A~ 57) (!9 58) (!a 59) (!A 60) (!! 61) (!$ 62) (!% 63) (!& 64) (!* 65) (!+ 66) (!- 67) (!. 68) (!/ 69) (!| 70) (!< 71) (!= 72) (!> 73) (!? 74) (!@ 75) (!^ 76) (!_ 77) (!~ 78) ($9 79) ($a 80) ($A 81) ($! 82) ($$ 83) ($% 84) ($& 85) ($* 86) ($+ 87) ($- 88) ($. 89) ($/ 90) ($| 91) ($< 92) ($= 93) ($> 94) ($? 95) ($@ 96) ($^ 97) ($_ 98) ($~ 99) (%9 100) (%a 101) (%A 102) (%! 103) (%$ 104) (%% 105) (%& 106) (%* 107) (%+ 108) (%- 109) (%. 110) (%/ 111) (%| 112) (%< 113) (%= 114) (%> 115) (%? 116) (%@ 117) (%^ 118) (%_ 119) (%~ 120) (&9 121) (&a 122) (&A 123) (&! 124) (&$ 125) (&% 126) (&& 127) (&* 128) (&+ 129) (&- 130) (&. 131) (&/ 132) (&| 133) (&< 134) (&= 135) (&> 136) (&? 137) (&@ 138) (&^ 139) (&_ 140) (&~ 141) (*9 142) (*a 143) (*A 144) (*! 145) (*$ 146) (*% 147) (*& 148) (** 149) (*+ 150) (*- 151) (*. 152) (*/ 153) (*| 154) (*< 155) (*= 156) (*> 157) (*? 158) (*@ 159) (*^ 160) (*_ 161) (*~ 162) (/9 163) (/a 164) (/A 165) (/! 166) (/$ 167) (/% 168) (/& 169) (/* 170) (/+ 171) (/- 172) (/. 173) (// 174) (/| 175) (/< 176) (/= 177) (/> 178) (/? 179) (/@ 180) (/^ 181) (/_ 182) (/~ 183) (|9 184) (ca 185) (CA 186) (|! 187) (|$ 188) (|% 189) (|& 190) (|* 191) (|+ 192) (|- 193) (|. 194) (|/ 195) (cc 196) (|< 197) (|= 198) (|> 199) (|? 200) (|@ 201) (|^ 202) (|_ 203) (|~ 204) (<9 205) (<a 206) (<A 207) (<! 208) (<$ 209) (<% 210) (<& 211) (<* 212) (<+ 213) (<- 214) (<. 215) (</ 216) (<| 217) (<< 218) (<= 219) (<> 220) (<? 221) (<@ 222) (<^ 223) (<_ 224) (<~ 225) (=9 226) (=a 227) (=A 228) (=! 229) (=$ 230) (=% 231) (=& 232) (=* 233) (=+ 234) (=- 235) (=. 236) (=/ 237) (=| 238) (=< 239) (== 240) (=> 241) (=? 242) (=@ 243) (=^ 244) (=_ 245) (=~ 246) (>9 247) (>a 248) (>A 249) (>! 250) (>$ 251) (>% 252) (>& 253) (>* 254) (>+ 255) (>- 256) (>. 257) (>/ 258) (>| 259) (>< 260) (>= 261) (>> 262) (>? 263) (>@ 264) (>^ 265) (>_ 266) (>~ 267) (?9 268) (?a 269) (?A 270) (?! 271) (?$ 272) (?% 273) (?& 274) (?* 275) (?+ 276) (?- 277) (?. 278) (?/ 279) (?| 280) (?< 281) (?= 282) (?> 283) (?? 284) (?@ 285) (?^ 286) (?_ 287) (?~ 288) (^9 289) (^a 290) (^A 291) (^! 292) (^$ 293) (^% 294) (^& 295) (^* 296) (^+ 297) (^- 298) (^. 299) (^/ 300) (^| 301) (^< 302) (^= 303) (^> 304) (^? 305) (^@ 306) (^^ 307) (^_ 308) (^~ 309) (_9 310) (_a 311) (_A 312) (_! 313) (_$ 314) (_% 315) (_& 316) (_* 317) (_+ 318) (_- 319) (_. 320) (_/ 321) (_| 322) (_< 323) (_= 324) (_> 325) (_? 326) (_@ 327) (_^ 328) (__ 329) (_~ 330) (~9 331) (~a 332) (~A 333) (~! 334) (~$ 335) (~% 336) (~& 337) (~* 338) (~+ 339) (~- 340) (~. 341) (~/ 342) (~| 343) (~< 344) (~= 345) (~> 346) (~? 347) (~@ 348) (~^ 349) (~_ 350) (~~ 351) )
	    (+ a A ! $ % & * / | < = > ? ^ _ ~ a9 aa aA a! a$ a% a& a* a+ a- a. a/ a| a< a= a> a? a@ a^ a_ a~ A9 Aa AA A! A$ A% A& A* A+ A- A. A/ A| A< A= A> A? A@ A^ A_ A~ !9 !a !A !! !$ !% !& !* !+ !- !. !/ !| !< != !> !? !@ !^ !_ !~ $9 $a $A $! $$ $% $& $* $+ $- $. $/ $| $< $= $> $? $@ $^ $_ $~ %9 %a %A %! %$ %% %& %* %+ %- %. %/ %| %< %= %> %? %@ %^ %_ %~ &9 &a &A &! &$ &% && &* &+ &- &. &/ &| &< &= &> &? &@ &^ &_ &~ *9 *a *A *! *$ *% *& ** *+ *- *. */ *| *< *= *> *? *@ *^ *_ *~ /9 /a /A /! /$ /% /& /* /+ /- /. // /| /< /= /> /? /@ /^ /_ /~ |9 ca CA |! |$ |% |& |* |+ |- |. |/ cc |< |= |> |? |@ |^ |_ |~ <9 <a <A <! <$ <% <& <* <+ <- <. </ <| << <= <> <? <@ <^ <_ <~ =9 =a =A =! =$ =% =& =* =+ =- =. =/ =| =< == => =? =@ =^ =_ =~ >9 >a >A >! >$ >% >& >* >+ >- >. >/ >| >< >= >> >? >@ >^ >_ >~ ?9 ?a ?A ?! ?$ ?% ?& ?* ?+ ?- ?. ?/ ?| ?< ?= ?> ?? ?@ ?^ ?_ ?~ ^9 ^a ^A ^! ^$ ^% ^& ^* ^+ ^- ^. ^/ ^| ^< ^= ^> ^? ^@ ^^ ^_ ^~ _9 _a _A _! _$ _% _& _* _+ _- _. _/ _| _< _= _> _? _@ _^ __ _~ ~9 ~a ~A ~! ~$ ~% ~& ~* ~+ ~- ~. ~/ ~| ~< ~= ~> ~? ~@ ~^ ~_ ~~ ))
	  61776)

(test (let ()(+ (let ((x 0) (y 1) (z 2) )(+ x y (let ((x 3) )(+ x (let ()(+ (let ()
									      (+ (let ((x 0) (y 1) (z 2) )(+ x y z (let ((x 3) )(+ x (let ((x 4) (y 5) (z 6) )
																       (+ x y z (let ()(+ (let ((x 7) )(+ x (let ()(+ (let ((x 8) (y 9) )
																							(+ x (let ((x 10) (y 11) (z 12) )(+ x  ))))))))))))))))))))))))))
      50)
(test  (let* ((x 0) (y x) )(+ x y (let ()(+ (let ((x 2) )(+ x (let ()(+ (let ((x 4) )
									  (+ x (let ((x 5) )(+ x (let ((x 6) (y x) (z y) )(+ x (let ((x 7) (y x) )
																 (+ x (let ((x 8) (y x) )(+ x y (let ((x 9) (y x) (z y) )(+ x ))))))))))))))))))))
       48)
(test (let* ((x 0) (y x) )(+ x y (let* ()(+ (let* ((x 2) )(+ x (let* ()(+ (let* ((x 4) )
									    (+ x (let* ((x 5) )(+ x (let* ((x 6) (y x) (z y) )(+ x (let* ((x 7) (y x) )
																     (+ x (let* ((x 8) (y x) )(+ x y (let* ((x 9) (y x) (z y) )(+ x ))))))))))))))))))))
      49)

(test (let ((!@$%^&*~|}{?><.,/`_-+=:! 1)) (+ !@$%^&*~|}{?><.,/`_-+=:! 1)) 2)
(test (let ((:hi 1)) :hi) 'error)
(test (let ((:hi: 1)) :hi:) 'error)
(test (let ((hi: 1)) hi) 'error)
(let ((1.0+2j (lambda (a) (+ a 1.0+2i))))
  (num-test (1.0+2j 3+i) 4.0+3i))

(test (let func ((a 1) (b 2)) (set! b a) (if (> b 0) (func (- a 1) b)) b) 1)
(test (let func ((a 1) (b 2)) (set! b a) (if (> b 0) (func (- a 1) b) b)) 0)
(test (let loop ((numbers '(3 -2 1 6 -5))
		 (nonneg '())
		 (neg '()))
	(cond ((null? numbers) (list nonneg neg))
	      ((>= (car numbers) 0)
	       (loop (cdr numbers)
		     (cons (car numbers) nonneg)
		     neg))
	      ((< (car numbers) 0)
	       (loop (cdr numbers)
		     nonneg
		     (cons (car numbers) neg)))))   
      '((6 1 3) (-5 -2)))
(test (let ((b '(1 2 3)))
	(let* ((a b)
	       (b (cons 0 a)))
	  (let b ((a b))
	    (if (null? a)
		'done
		(b (cdr a))))))
      'done)
(test (let lp ((x 100))
	(if (positive? x)
	    (lp (- x 1))
	    x))
      0)
(test (let func ((a 1) (b 2) (c 3)) (+ a b c (if (> a 1) (func (- a 1) (- b 1) (- c 1)) 0))) 6)
(test (let func ((a 1) (b 2) (c 3)) (+ a b c (if (> a 0) (func (- a 1) (- b 1) (- c 1)) 0))) 9)
(test (let func () 1) 1)
(test (let ((a 1)) (let func () (if (> a 1) (begin (set! a (- a 1)) (func)) 0))) 0)
(test (let func1 ((a 1)) (+ (let func2 ((a 2)) a) a)) 3)
(test (let func1 ((a 1)) (+ (if (> a 0) (func1 (- a 1)) (let func2 ((a 2)) (if (> a 0) (func2 (- a 1)) 0))) a)) 1)
(test (let func ((a (let func ((a 1)) a))) a) 1)
(test (let ((i 3)) (let func () (set! i (- i 1)) (if (> i 0) (func))) i) 0)
(test (let func ((a 1)) (define (func a) 2) (func 1)) 2)
(test (let func ((a 1)) (define func (lambda (a) (func a))) (if (> a 1) (func (- a 1)) 0)) 0)
(test (let loop ((i 0)) (let loop ((i 0)) (if (< i 1) (loop (+ i 1)))) i) 0)
(test (let * ((i 0)) (if (< i 1) (* (+ i 1))) i) 0)
(test (let ((j 123)) (define (f g) (set! j 0) (g 0)) (let loop ((i 1)) (if (> i 0) (f loop))) j) 0)
(test (procedure? (let loop () loop)) #t)
(test (let loop1 ((func 0)) (let loop2 ((i 0)) (if (not (procedure? func)) (loop1 loop2)) func)) 0)
(test (let ((k 0)) (let ((x (let xyz ((i 0)) (set! k (+ k 1)) xyz))) (x 0)) k) 2)
(test (let ((hi' 3) (a'b 2)) (+ hi' a'b)) 5)
(test (let ((hi''' 3) (a'''b 2)) (+ hi''' a'''b)) 5)
(test (let ((f (let func ((i 0)) (if (= i 0) func (if (> i 1) (+ i (func (- i 1))) 1))))) (map f '(1 2 3))) '(1 3 6))
(test (let ((x 0)) (let ((f (lambda (a) (+ a x)))) (map (let () (set! x (+ x 1)) f) '(1 2 3)))) '(2 3 4))

(let ((enter 0)
      (exit 0)
      (inner 0))
  (define (j1) 
    (set! enter (+ enter 1))
    (let ((result 
	   (let hiho
	       ((i 0))
	     (set! inner (+ inner 1))
	     (if (< i 3) 
		 hiho
		 i))))
      (set! exit (+ exit 1))
      result))

  (let ((j2 (j1)))
    (test (and (procedure? j2) (= enter 1) (= exit 1) (= inner 1)) #t)
    (let ((result (j2 1)))
      (test (and (procedure? result) (= enter 1) (= exit 1) (= inner 2)) #t)
      (set! result (j2 3))
      (test (and (= result 3) (= enter 1) (= exit 1) (= inner 3)) #t))))


(let ()
  (define (block-comment-test a b c)
    (+ a b c))

  (let ((val (block-comment-test 
#|
	    a comment
|#
	    1 #| this is a |# 
#|
            another comment
|#
 2 #| this is b |# 3)))

    (test val 6)))


(test (letrec* ((p (lambda (x)
		     (+ 1 (q (- x 1)))))
		(q (lambda (y)
		     (if (zero? y)
			 0
			 (+ 1 (p (- y 1))))))
		(x (p 5))
		(y x))
	       y)
      5)
(test (letrec ((p (lambda (x)
		     (+ 1 (q (- x 1)))))
		(q (lambda (y)
		     (if (zero? y)
			 0
			 (+ 1 (p (- y 1))))))
		(x (p 5))
		(y x))
	       y)
      'error)
(test (let* ((p (lambda (x)
		     (+ 1 (q (- x 1)))))
		(q (lambda (y)
		     (if (zero? y)
			 0
			 (+ 1 (p (- y 1))))))
		(x (p 5))
		(y x))
	       y)
      'error)

(test (let ((x 1) ((y 2))) x) 'error)
(test (let ((x 1 2 3)) x) 'error)
(test (let ((+ 1 2)) 2) 'error)
(test (let* ((x 1 2)) x) 'error)
(test (letrec ((x 1 2)) x) 'error)
(test (letrec* ((x 1 2)) x) 'error)
(test (let ((x 1 . 2)) x) 'error)
(test (let ((x 1 , 2)) x) 'error)
(test (let ((x . 1)) x) 'error)
(test (let* ((x . 1)) x) 'error)
(test (letrec ((x . 1)) x) 'error)
(test (letrec* ((x . 1)) x) 'error)
(test (let hi ()) 'error)
(test (let* ((x -1) 2) 3) 'error)
(test (let ((x -1) 2) 3) 'error)
(test (letrec ((x -1) 2) 3) 'error)
(test (let ((pi 3)) pi) 'error)
(test (let* ((pi 3)) pi) 'error)
(test (letrec ((pi 3)) pi) 'error)

(test (let) 'error)
(test (let*) 'error)
(test (letrec) 'error)
(test (let . 1) 'error)
(test (let* (x)) 'error)
(test (let (x) 1) 'error)
(test (let ((x)) 3) 'error)
(test (let ((x 1) y) x) 'error)
(test (let* x ()) 'error)
(test (let* ((1 2)) 3) 'error)
(test (let () ) 'error)
(test (let '() 3) 'error)
(test (let* ((x 1))) 'error)
(test (let ((x 1)) (letrec ((x 32) (y x)) (+ 1 y))) 'error) ; #<unspecified> seems reasonable if not the 1+ 
(test (let ((x 1)) (letrec ((y x) (x 32)) (+ 1 y))) 'error)
(test (let ((x 1)) (letrec ((y x) (x 32)) 1)) 1)
(test (let ((x 1)) (letrec ((y (let () (+ x 1))) (x 32)) (+ 1 y))) 'error)
(test (let ((x 1)) (letrec ((y (let ((xx (+ x 1))) xx)) (x 32)) (+ 1 y))) 'error)
(test (let ((x 32)) (letrec ((y (apply list `(* ,x 2))) (x 1)) y)) '(* #<undefined> 2))
(test (letrec) 'error)
(test (letrec*) 'error)
(test (let ((x . 1)) x) 'error)
(test (letrec* ((and #2D((1 2) (3 4)) 3/4))) 'error)
(test (letrec* ((hi "" #\a))) 'error)

(test (let #((a 1)) a) 'error)
(test (let* #((a 1)) a) 'error)
(test (letrec #((a 1)) a) 'error)
(test (letrec* #((a 1)) a) 'error)

;; (let *((a 1)) a) -> 1 ; * is named let name?
(test (letrec *((a 1)) a) 'error)
(test (letrec* *((a 1)) a) 'error)
(test (letrec* (((a 1) 2)) a) 'error)
(test (letrec* (#(a 1) 2) a) 'error)
(test (letrec* ((a a)) a) #<undefined>) ; hmm -- guile says Variable used before given a value: a
(test (let . (((a 1)) a)) 1)
(test (let '((a 1)) a) 'error)
(test (let (((x 1)) 2) 3) 'error)
(test (let ((#f 1)) #f) 'error)
(test (let (()) #f) 'error)
(test (let (lambda () ) #f) 'error)
(test (let ((f1 3) (f1 4)) f1) 'error) ; not sure about this
;;   (let () (define (f1) 3) (define (f1) 4) (f1))
(test (let ((asdf (lambda (a) (if (> a 0) (asdf (- a 1)) 0)))) (asdf 3)) 'error)
(test (let* ((asdf (lambda (a) (if (> a 0) (asdf (- a 1)) 0)))) (asdf 3)) 'error)
(test (let (('a 3)) 1) 'error)
(test (let ((#\a 3)) #\a) 'error)
;;      (test (let ((#z1 2)) 1) 'error)
(test (let ('a 3) 1) 'error)
(test (let 'a 1) 'error)
(test (let* func ((a 1)) a) 'error)
(test (letrec func ((a 1)) a) 'error)
(test (letrec* func ((a 1)) a) 'error)

(test (let ((1 3)) 3) 'error)
(test (let ((#t 3)) 3) 'error)
(test (let ((() 3)) 3) 'error)
(test (let ((#\c 3)) 3) 'error)
(test (let (("hi" 3)) 3) 'error)
(test (let ((:hi 3)) 3) 'error)

(test (let 1 ((i 0)) i) 'error)
(test (let #f ((i 0)) i) 'error)
(test (let "hi" ((i 0)) i) 'error)
(test (let #\c ((i 0)) i) 'error)
(test (let :hi ((i 0)) i) 'error)

(test (let func ((a 1) . b) a) 'error)
(test (let func a . b) 'error)
(test (let let func ((a 1)) func) 'error)
(test (let func 1 ((x 1)) x) 'error)
(test (let func ((a 1) . b) (if (> a 0) (func (- a 1) 2 3) b)) 'error)
(test (let func ((a . 1)) a) 'error)
(test (let func (a . 1) a) 'error)
(test (let ((a 1) . b) a) 'error)
(test (let* ((a 1) . b) a) 'error)
(test (let func ((a func) (i 1)) i) 'error)
(test (let func ((i 0)) (if (< i 1) (func))) 'error)
(test (let func (let ((i 0)) (if (< i 1) (begin (set! i (+ i 1)) (func))))) 'error)
(test (let ((x 0)) (set! x (+ x 1)) (begin (define y 1)) (+ x y)) 2)
(test (let loop loop) 'error)
(test (let loop (loop)) 'error)
(test (let loop ((i 0) (loop 1)) i) 0) ; this used to be an error, Guile also returns 0

(test (letrec ((cons 1 (quote ())) . #(1)) 1) 'error)
(test (letrec ((a 1) . 2) 1) 'error)
(test (let* ((a 1) (b . 2) . 1) (())) 'error)
(test (let "" 1) 'error)
(test (let "hi" 1) 'error)
(test (let #(1) 1) 'error)
(test (let __hi__ #t) 'error)
(test (let* hi () 1) 'error)
(test (letrec (1 2) #t) 'error)
(test (letrec* (1 2) #t) 'error)
(test (let hi (()) 1) 'error)
(test (let hi a 1) 'error)

;;; these ought to work, but see s7.c under EVAL: (it's a speed issue)
;(test (let let ((i 0)) (if (< i 3) (let (+ i 1)) i)) 3)
;(test (let () (define (if a) a) (if 1)) 1)
;(test (let begin ((i 0)) (if (< i 3) (begin (+ i 1)) i)) 3)


;;; from the scheme wiki
;;; http://community.schemewiki.org/?sieve-of-eratosthenes

(let ((results '(2)))
  (define (primes n) 
    (let ((pvector (make-vector (+ 1 n) #t))) ; if slot k then 2k+1 is a prime 
      (let loop ((p 3) ; Maintains invariant p = 2j + 1 
		 (q 4) ; Maintains invariant q = 2j + 2jj 
		 (j 1) 
		 (k '()) 
		 (vec pvector)) 
	(letrec ((lp (lambda (p q j k vec) 
		       (loop (+ 2 p) 
			     (+ q (- (* 2 (+ 2 p)) 2)) 
			     (+ 1 j) 
			     k 
			     vec))) 
		 (eradicate (lambda (q p vec) 
			      (if (<= q n) 
				  (begin (vector-set! vec q #f) 
					 (eradicate (+ q p) p vec)) 
				  vec)))) 
          (if (<= j n) 
	      (if (eq? #t (vector-ref vec j)) 
		  (begin (set! results (cons p results))
			 (lp p q j q (eradicate q p vec))) 
		  (lp p q j k vec)) 
	      (reverse results))))))
  (test (primes 10) '(2 3 5 7 11 13 17 19)))

(test (let ((gvar 32)) (define (hi1 a) (+ a gvar)) (let ((gvar 0)) (hi1 2))) 34)
(test (let ((gvar 32)) (define-macro (hi2 b) `(* gvar ,b)) (define (hi1 a) (+ (hi2 a) gvar)) (let ((gvar 0)) (hi1 2))) 96)
(test (let ((gvar 32)) (define-macro (hi2 b) `(* gvar ,b)) (define (hi1 a) (+ a gvar)) (let ((gvar 0)) (hi1 (hi2 2)))) 32)
(test (let ((gvar 32)) (define-macro (hi2 b) `(* gvar ,b)) (define (hi1 a) (+ (a 2) gvar)) (let ((gvar 0)) (define (hi2 a) (* a 2)) (hi1 hi2))) 36)
(test (let ((gvar 32)) (define-macro (hi2 b) `(* gvar ,b)) (define (hi1 a) (+ (a 2) gvar)) (let ((gvar 0) (hi2 (lambda (a) (hi2 a)))) (hi1 hi2))) 96)
(test (let ((gvar 32)) (define-macro (hi2 b) `(* gvar ,b)) (define (hi1 a) (+ (a 2) gvar)) (let* ((gvar 0) (hi2 (lambda (a) (hi2 a)))) (hi1 hi2))) 32)
(test (let () ((let ((gvar 32)) (define-macro (hi2 b) `(* gvar ,b)) (define (hi1 a) (+ (hi2 2) gvar)) hi1) 2)) 96)
(test (let ((gvar 0)) ((let ((gvar 1)) (define-macro (hi2 b) `(+ gvar ,b)) (define (hi1 a) (let ((gvar 2)) (hi2 a))) hi1) 2)) 4)
(test (let ((gvar 0)) (define-macro (hi2 b) `(+ gvar ,b)) ((let ((gvar 1)) (define (hi1 a) (let ((gvar 2)) (a 2))) hi1) hi2)) 4)
(test (let ((gvar 0)) (define-macro (hi2 b) `(+ gvar ,b)) ((let ((gvar 1)) (define (hi1 a) (a 2)) hi1) hi2)) 3)
(test (let () (define-macro (hi2 b) `(+ gvar ,b)) ((let ((gvar 1)) (define (hi1 a) (a 2)) hi1) hi2)) 3)
(test (let ((y 1) (x (let ((y 2) (x (let ((y 3) (x 4)) (+ x y)))) (+ x y)))) (+ x y)) 10)
(test (let ((x 0)) 
	(+ (let ((x 1) (y (+ x 1))) 
	  (+ (let ((x 2) (y (+ x 1))) 
	    (+ (let ((x 3) (y (+ x 1))) 
	      (+ (let ((x 4) (y (+ x 1))) 
		(+ (let ((x 5) (y (+ x 1)))
		  (+ (let ((x 6) (y (+ x 1))) 
		    (+ (let ((x 7) (y (+ x 1)))
			 (+ x y)) x)) x)) x)) x)) x)) x)) x)) 35)

(test (let loop ((lst (list 1 2)) 
		 (i 0) 
		 (sum 0))
	(if (or (null? lst)
		(> i 10))
	    sum
	    (begin
	      (set-cdr! (cdr lst) lst)
	      (loop (cdr lst) (+ i 1) (+ sum (car lst))))))
      16)

;;; these are confusing:
;(letrec ((if 0.0)) ((lambda () (if #t "hi")))) -> "hi"
;(let ((let 0)) let) -> 0
;(let* ((lambda 0)) ((lambda () 1.5))) -> 1.5 ; syntax error in Guile
;(let* ((lambda 0)) lambda) -> 0

;; from test-submodel.scm, from MIT I think
(test (letrec ((factorial
		(lambda (n)
		  (if (<= n 0) 1 (* n (factorial (- n 1)))))))
	(factorial 3))
      6)

(test (letrec ((iter-fact
		(lambda (n)
		  (letrec
		      ((helper (lambda (n p)
				 (if (<= n 0) p (helper (- n 1) (* n p))))))
		    (helper n 1)))))
	(iter-fact 3))
      6)

(test (letrec ((y-factorial
		(lambda (n)
		  (letrec ((y
			    (lambda (f)
			      ((lambda (x)
				 (f (lambda (z) ((x x) z))))
			       (lambda (x)
				 (f (lambda (z) ((x x) z)))))))
			   (fact-def
			    (lambda (fact)
			      (lambda (n)
				(if (<= n 0)
				    1
				    (* n (fact (- n 1))))))))
		    ((y fact-def) n)))))
	(y-factorial 3))
      6)

(test (let ((x 1)) (let ((x 0) (y x)) (cons x y))) '(0 . 1))
(test (let ((x 1)) (let* ((x 0) (y x)) (cons x y))) '(0 . 0))
(test (let ((x 1)) (letrec ((x 0) (y x)) (cons x y))) '(0 . #<undefined>))
(test (let ((x 1)) (letrec* ((x 0) (y x)) (cons x y))) '(0 . 0))

(test (let ((x 1)) (let ((x 0) (y (let () (set! x 2) x))) (cons x y))) '(0 . 2))
(test (let ((x 1)) (letrec ((x 0) (y (let () (set! x 2) x))) (cons x y))) '(0 . 2))
(test (let ((x 1)) (let* ((x 0) (y (let () (set! x 2) x))) (cons x y))) '(2 . 2))
(test (let ((x 1)) (letrec* ((x 0) (y (let () (set! x 2) x))) (cons x y))) '(2 . 2))

(test (letrec ((x x)) x) #<undefined>) ; weird
(test (letrec ((x y) (y x)) x) #<undefined>)

(test (procedure? (letrec ((x (lambda () x))) x)) #t)
(test (procedure? (letrec ((x (lambda () x))) (x))) #t)
(test (letrec ((x (lambda () x))) (equal? x (x))) #t)  ; !
(test (letrec ((x (lambda () x))) (equal? x ((x)))) #t)  ; !

(test (letrec* ()) 'error)
(test (letrec* ((x 1 x)) x) 'error)
(test (letrec ((x (let () (set! y 1) y)) (y (let () (set! y (+ y 1)) y))) (list x y)) '(1 2)) ; !

(test (letrec ((x 1) (y x)) (list x y)) '(1 #<undefined>)) ; guile says '(1 1)
(test (letrec ((y x) (x 1)) (list x y)) '(1 #<undefined>)) ; guile says '(1 1)
(test (letrec ((x 1) (y (let () (set! x 2) x))) (list x y)) '(1 2))
(test (letrec ((history (list 9))) ((lambda (n) (begin (set! history (cons history n)) history)) 8)) '((9) . 8))
(test (((call/cc (lambda (k) k)) (lambda (x) x)) 'HEY!) 'HEY!)

(let ((sequence '()))
  ((call-with-current-continuation
    (lambda (goto)
      (letrec ((start
		(lambda ()
		  (begin (set! sequence (cons 'start sequence))
			 (goto next))))
	       (froz
		(lambda ()
		  (begin (set! sequence (cons 'froz sequence))
			 (goto last))))
	       (next
		(lambda ()
		  (begin (set! sequence (cons 'next sequence))
			 (goto froz))))
	       (last
		(lambda ()
		  (begin (set! sequence (cons 'last sequence))
			 #f))))
	start))))
  (test (reverse sequence) '(start next froz last)))

(let ()
  (define thunk 'dummy-thunk)

  (define (make-fringe-thunk tree)
    (call-with-exit
     (lambda (return-to-repl)
       (cond ((pair? tree) (begin (make-fringe-thunk (car tree))
				  (make-fringe-thunk (cdr tree))))
	     ((null? tree) (begin (set! thunk (lambda () 'done)) 'null))
	     (else (call/cc
		    (lambda (cc)
		      (begin
			(set! thunk
			      (lambda ()
				(begin (display tree) (cc 'leaf))))
			(return-to-repl 'thunk-set!)))))))))

  (define tr '(() () (((1 (( (() 2 (3 4)) (((5))) )) ))) ))
  (test (make-fringe-thunk tr) 'null)
  (test (thunk) 'done))

;;; evaluation order matters, but in s7 it's always left -> right
(test (let ((x 1)) (+ x (let () (define x 2) x))) 3)
(test (let ((x 1)) (+ (begin (define x 2) x) x)) 4)
(test (let ((x 1)) (+ x (begin (define x 2) x))) 3) 
(test (let ((x 1)) (+ x (begin (set! x 2) x))) 3)
(test (let ((x 1)) (+ (begin (set! x 2) x) x)) 4)
(test (let ((x 1)) ((if (= x 1) + -) x (begin (set! x 2) x))) 3) 



;;; --------------------------------------------------------------------------------
;;; call/cc
;;; --------------------------------------------------------------------------------
;;; some of these were originally from Al Petrovsky, Scott G Miller, Matthias Radestock, J H Brown, Dorai Sitaram, 
;;;   and probably others.

(let ((calls (make-vector 3 #f))
      (travels (make-vector 5 0))
      (ctr 0))
  (set! (travels 0) (+ (travels 0) 1))
  (call/cc (lambda (c0) (set! (calls 0) c0)))
  (set! (travels 1) (+ (travels 1) 1))
  (call/cc (lambda (c1) (set! (calls 1) c1)))
  (set! (travels 2) (+ (travels 2) 1))
  (call/cc (lambda (c2) (set! (calls 2) c2)))
  (set! (travels 3) (+ (travels 3) 1))
  (let ((ctr1 ctr))
    (set! ctr (+ ctr1 1))
    (if (< ctr1 3)
	((calls ctr1) ctr1)))
  (set! (travels 4) (+ (travels 4) 1))
  (test travels #(1 2 3 4 1)))

(let ((calls (make-vector 5 #f))
      (travels (make-vector 5 0))
      (ctr2 0))
  (let loop ((ctr 0))
    (if (< ctr 3)
	(begin
	  (set! (travels ctr) (+ (travels ctr) 1))
	  (call/cc (lambda (c0) (set! (calls ctr) c0)))
	  (loop (+ ctr 1)))))
  (set! (travels 3) (+ (travels 3) 1))
  (let ((ctr1 ctr2))
    (set! ctr2 (+ ctr1 1))
    (if (< ctr1 3)
	((calls ctr1) ctr1)))
  (set! (travels 4) (+ (travels 4) 1))
  (test travels #(1 2 3 4 1)))

(let ((c1 #f)
      (c2 #f)
      (c3 #f)
      (x0 0)
      (x1 0)
      (x2 0)
      (x3 0))
  (let ((x (+ 1 
	      (call/cc
	       (lambda (r1)
		 (set! c1 r1)
		 (r1 2)))
	      (call/cc
	       (lambda (r2)
		 (set! c2 r2)
		 (r2 3)))
	      (call/cc
	       (lambda (r3)
		 (set! c3 r3)
		 (r3 4)))
	      5)))
    (if (= x0 0) 
	(set! x0 x)
	(if (= x1 0)
	    (set! x1 x)
	    (if (= x2 0)
		(set! x2 x)
		(if (= x3 0)
		    (set! x3 x)))))
    (if (= x 15)
	(c1 6))
    (if (= x 19)
	(c2 7))
    (if (= x 23)
	(c3 8))
    (test (list x x0 x1 x2 x3) '(27 15 19 23 27))))

(let ((c1 #f) (c2 #f) (c3 #f) (x0 0) (x1 0) (x2 0) (x3 0) (y1 0) (z0 0) (z1 0) (z2 0) (z3 0))
  (let* ((y 101)
	 (x (+ y 
	      (call/cc
	       (lambda (r1)
		 (set! c1 r1)
		 (r1 2)))
	      (call/cc
	       (lambda (r2)
		 (set! c2 r2)
		 (r2 3)))
	      (call/cc
	       (lambda (r3)
		 (set! c3 r3)
		 (r3 4)))
	      5))
	 (z (+ x y)))
    (set! y1 y)
    (if (= x0 0) 
	(begin
	  (set! x0 x)
	  (set! z0 z))
	(if (= x1 0)
	    (begin
	      (set! x1 x)
	      (set! z1 z))
	    (if (= x2 0)
		(begin
		  (set! x2 x)
		  (set! z2 z))
		(if (= x3 0)
		    (begin
		      (set! x3 x)
		      (set! z3 z))))))
    (if (= x 115)
	(c1 6))
    (if (= x 119)
	(c2 7))
    (if (= x 123)
	(c3 8))
    (test (list x x0 x1 x2 x3 y1 z0 z1 z2 z3) '(127 115 119 123 127 101 216 220 224 228))))

(let ((c1 #f)
      (c2 #f)
      (c3 #f)
      (x0 0)
      (x1 0)
      (x2 0)
      (x3 0))
  (let ((x (+ 1 
	      (call/cc
	       (lambda (r1)
		 (set! c1 r1)
		 (r1 2)))
	      (call/cc
	       (lambda (r2)
		 (set! c2 r2)
		 (r2 3)))
	      (call/cc
	       (lambda (r3)
		 (set! c3 r3)
		 (r3 4)))
	      5)))
    (if (= x0 0) 
	(set! x0 x)
	(if (= x1 0)
	    (set! x1 x)
	    (if (= x2 0)
		(set! x2 x)
		(if (= x3 0)
		    (set! x3 x)))))
    (if (= x 15)
	(c1 6 1))
    (if (= x 20)
	(c2 7 2 3))
    (if (= x 29)
	(c3 8 3 4 5))
    (test (list x x0 x1 x2 x3) '(45 15 20 29 45))))
;; 45 = (+ 1 6 1 7 2 3 8 3 4 5 5)

(let ((x 0)
      (c1 #f)
      (results '()))
  (set! x (call/cc
	   (lambda (r1)
	     (set! c1 r1)
	     (r1 2))))
  (set! results (cons x results))
  (if (= x 2) (c1 32))
  (test results '(32 2)))

(let ((x #(0))
      (y #(0))
      (c1 #f))
  (set! ((call/cc
	   (lambda (r1)
	     (set! c1 r1)
	     (r1 x)))
	 0) 32)
  (if (= (y 0) 0) (c1 y))
  (test (and (equal? x #(32)) (equal? y #(32))) #t))

(test (call/cc (lambda (k) ((call/cc (lambda (top) (k (+ 1 (call/cc (lambda (inner) (top inner))))))) 2))) 3)

(let* ((next-leaf-generator (lambda (obj eot)
			      (letrec ((return #f)
				       (cont (lambda (x)
					       (recur obj)
					       (set! cont (lambda (x) (return eot)))
					       (cont #f)))
				       (recur (lambda (obj)
						(if (pair? obj)
						    (for-each recur obj)
						    (call-with-current-continuation
						     (lambda (c)
						       (set! cont c)
						       (return obj)))))))
				(lambda () (call-with-current-continuation
					    (lambda (ret) (set! return ret) (cont #f)))))))
       (leaf-eq? (lambda (x y)
		   (let* ((eot (list 'eot))
			  (xf (next-leaf-generator x eot))
			  (yf (next-leaf-generator y eot)))
		     (letrec ((loop (lambda (x y)
				      (cond ((not (eq? x y)) #f)
					    ((eq? eot x) #t)
					    (else (loop (xf) (yf)))))))
		       (loop (xf) (yf)))))))
  
  (test (leaf-eq? '(a (b (c))) '((a) b c)) #t)
  (test (leaf-eq? '(a (b (c))) '((a) b c d)) #f))

(test (let ((r #f)
	    (a #f)
	    (b #f)
	    (c #f)
	    (i 0))
	(let () 
	  (set! r (+ 1 (+ 2 (+ 3 (call/cc (lambda (k) (set! a k) 4))))
		     (+ 5 (+ 6 (call/cc (lambda (k) (set! b k) 7))))))
	  (if (not c) 
	      (set! c a))
	  (set! i (+ i 1))
	  (case i
	    ((1) (a 5))
	    ((2) (b 8))
	    ((3) (a 6))
	    ((4) (c 4)))
	  r))
      28)

(test (let ((r #f)
	    (a #f)
	    (b #f)
	    (c #f)
	    (i 0))
	(let () 
	  (set! r (+ 1 (+ 2 (+ 3 (call/cc (lambda (k) (set! a k) 4))))
		     (+ 5 (+ 6 (call/cc (lambda (k) (set! b k) 7))))))
	  (if (not c) 
	      (set! c a))
	  (set! i (+ i 1))
	  (case i
	    ((1) (b 8))
	    ((2) (a 5))
	    ((3) (b 7))
	    ((4) (c 4)))
	  r))
      28)

(test (let ((k1 #f)
	    (k2 #f)
	    (k3 #f)
	    (state 0))
	(define (identity x) x)
	(define (fn)
	  ((identity (if (= state 0)
			 (call/cc (lambda (k) (set! k1 k) +))
			 +))
	   (identity (if (= state 0)
			 (call/cc (lambda (k) (set! k2 k) 1))
			 1))
	   (identity (if (= state 0)
			 (call/cc (lambda (k) (set! k3 k) 2))
			 2))))
	(define (check states)
	  (set! state 0)
	  (let* ((res '())
		 (r (fn)))
	    (set! res (cons r res))
	    (if (null? states)
		res
		(begin (set! state (car states))
		       (set! states (cdr states))
		       (case state
			 ((1) (k3 4))
			 ((2) (k2 2))
			 ((3) (k1 -)))))))
	(map check '((1 2 3) (1 3 2) (2 1 3) (2 3 1) (3 1 2) (3 2 1))))
      '((-1 4 5 3) (4 -1 5 3) (-1 5 4 3) (5 -1 4 3) (4 5 -1 3) (5 4 -1 3)))

(let ((c1 #f))
  (let ((x ((call/cc (lambda (r1) (set! c1 r1) (r1 "hiho"))) 0)))
    (if (char=? x #\h)
	(c1 "asdf"))
    (test x #\a)))

(test (let ((x '())
	    (y 0))
	(call/cc 
	 (lambda (escape)
	   (let* ((yin ((lambda (foo) 
			  (set! x (cons y x))
			  (if (= y 10)
			      (escape x)
			      (begin
				(set! y 0)
				foo)))
			(call/cc (lambda (bar) bar))))
		  (yang ((lambda (foo) 
			   (set! y (+ y 1))
			   foo)
			 (call/cc (lambda (baz) baz)))))
	     (yin yang)))))
      '(10 9 8 7 6 5 4 3 2 1 0))

(test (let ((c #f))
	(let ((r '()))
	  (let ((w (let ((v 1))
		     (set! v (+ (call-with-current-continuation
				 (lambda (c0) (set! c c0) v))
				v))
		     (set! r (cons v r))
		     v)))
	    (if (<= w 1024) (c w) r))))
      '(2048 1024 512 256 128 64 32 16 8 4 2))

(test (let ((cc #f)
	    (r '()))
	(let ((s (list 1 2 3 4 (call/cc (lambda (c) (set! cc c) 5)) 6 7 8)))
	  (if (null? r)
	      (begin (set! r s) (cc -1))
	      (list r s))))
      '((1 2 3 4 5 6 7 8) (1 2 3 4 -1 6 7 8)))

(test (let ((count 0))
        (let ((first-time? #t)
              (k (call/cc values)))
          (if first-time?
              (begin
                (set! first-time? #f)
                (set! count (+ count 1))
                (k values))
              (void)))
        count)
      2)

(let ((c #f)
      (vals '()))
  (let ((val (+ 1 2 (call/cc (lambda (r) (set! c r) (r 3))))))
    (set! vals (cons val vals))
    (if (< val 20) (c (+ val 1)))
    (test vals '(22 18 14 10 6))))
(let ((c #f)
      (vals '()))
  (let ((val (+ 1 2 (call/cc (lambda (r) (set! c r) (r 3))))))
    (set! vals (cons val vals))
    (if (< val 20) (apply c vals))
    (test vals '(36 18 9 6))))
(let ((c #f)
      (vals '()))
  (let ((val (+ 1 2 (call/cc (lambda (r) (set! c r) (r 3))))))
    (set! vals (cons val vals))
    (if (< val 20) (c (apply values vals)))
    (test vals '(36 18 9 6))))

(test (procedure? (call/cc call/cc)) #t)
(test (call/cc (lambda (c) (0 (c 1)))) 1)
(test (call/cc (lambda (k) (k "foo"))) "foo")
(test (call/cc (lambda (k) "foo")) "foo")
(test (call/cc (lambda (k) (k "foo") "oops")) "foo")
(test (call/cc (lambda (return) (catch #t (lambda () (error 'hi "")) (lambda args (return "oops"))))) "oops")
(test (call/cc (lambda (return) (catch #t (lambda () (return 1)) (lambda args (return "oops"))))) 1)
(test (catch #t (lambda () (call/cc (lambda (return) (return "oops")))) (lambda arg 1)) "oops")
(test (call/cc (if (< 2 1) (lambda (return) (return 1)) (lambda (return) (return 2) 3))) 2)
(test (call/cc (let ((a 1)) (lambda (return) (set! a (+ a 1)) (return a)))) 2)
(test (call/cc (lambda (return) (let ((hi return)) (hi 2) 3))) 2)
(test (let () (define (hi) (call/cc func)) (define (func a) (a 1)) (hi)) 1)
(test (((call/cc (call/cc call/cc)) call/cc) (lambda (a) 1)) 1)
(test (+ 1 (eval-string "(+ 2 (call-with-exit (lambda (return) (return 3))) 4)") 5) 15)
(test (+ 1 (eval '(+ 2 (call-with-exit (lambda (return) (return 3))) 4)) 5) 15)
(test (call-with-exit) 'error)
(test (call-with-exit s7-version s7-version) 'error)
(test (call/cc) 'error)
(test (call/cc s7-version s7-version) 'error)
(test (call/cc (lambda () 1)) 'error)
(test (call/cc (lambda (a b) (a 1))) 'error)
(test (+ 1 (call/cc (lambda (k) (k #\a)))) 'error)
(test (+ 1 (call-with-exit (lambda (k) (k #\a)))) 'error)
(test ((call/cc (lambda (return) (call/cc (lambda (cont) (return cont))) list)) 1) '(1)) ; from Guile mailing list -- this strikes me as very strange

(test (call/cc begin) 'error)
(test (call/cc quote) 'error)

(let ((p1 (make-procedure-with-setter (lambda (k) (k 3)) (lambda (k a) (k a)))))
  (test (call/cc p1) 3)
  (test (call-with-exit p1) 3))

;;; guile/s7 accept: (call/cc (lambda (a . b) (a 1))) -> 1
;;; same:            (call/cc (lambda arg ((car arg) 1))) -> 1

(test (let ((listindex (lambda (e l)
			 (call/cc (lambda (not_found)
				    (letrec ((loop 
					      (lambda (l)
						(cond
						 ((null? l) (not_found #f))
						 ((equal? e (car l)) 0)
						 (else (+ 1 (loop (cdr l))))))))
				      (loop l)))))))
	(listindex 1 '(0 3 2 4 8)))
      #f)

(test (let ((product (lambda (li)
		       (call/cc (lambda (return)
				  (let loop ((l li))
				    (cond
				     ((null? l) 1)
				     ((= (car l) 0) (return 0))
				     (else (* (car l) (loop (cdr l)))))))))))
	(product '(1 2 3 0 4 5 6)))
      0)

(test (let ((lst '()))
	((call/cc
	  (lambda (goto)
	    (letrec ((start (lambda () (set! lst (cons "start" lst)) (goto next)))
		     (next  (lambda () (set! lst (cons "next" lst))  (goto last)))
		     (last  (lambda () (set! lst (cons "last" lst)) (reverse lst))))
	      start)))))
      '("start" "next" "last"))

(test (let ((cont #f))
	(letrec ((x (call-with-current-continuation (lambda (c) (set! cont c) 0)))
		 (y (call-with-current-continuation (lambda (c) (set! cont c) 0))))
	  (if cont
	      (let ((c cont))
		(set! cont #f)
		(set! x 1)
		(set! y 1)
		(c 0))
	      (+ x y))))
      0)

(test (letrec ((x (call-with-current-continuation
		   (lambda (c)
		     (list #t c)))))
	(if (car x)
	    ((cadr x) (list #f (lambda () x)))
	    (eq? x ((cadr x)))))
      #t)

(test (call/cc (lambda (c) (0 (c 1)))) 1)

(test (let ((member (lambda (x ls)
		      (call/cc
		       (lambda (return)
			 (do ((ls ls (cdr ls)))
			     ((null? ls) #f)
			   (if (equal? x (car ls))
			       (return ls))))))))
	(list (member 'd '(a b c))
	      (member 'b '(a b c))))
      '(#f (b c)))


;;; call-with-exit
(test (+ 2 (call/cc (lambda (k) (* 5 (k 4))))) 6)
(test (+ 2 (call/cc (lambda (k) (* 5 (k 4 5 6))))) 17)
(test (+ 2 (call/cc (lambda (k) (* 5 (k (values 4 5 6)))))) 17)
(test (+ 2 (call/cc (lambda (k) (* 5 (k 1 (values 4 5 6)))))) 18)
(test (+ 2 (call/cc (lambda (k) (* 5 (k 1 (values 4 5 6) 1))))) 19)
(test (+ 2 (call-with-exit (lambda (k) (* 5 (k 4))))) 6)
(test (+ 2 (call-with-exit (lambda (k) (* 5 (k 4 5 6))))) 17)
(test (+ 2 (call-with-exit (lambda (k) (* 5 (k (values 4 5 6)))))) 17)
(test (+ 2 (call-with-exit (lambda (k) (* 5 (k 1 (values 4 5 6)))))) 18)
(test (+ 2 (call-with-exit (lambda (k) (* 5 (k 1 (values 4 5 6) 1))))) 19)
(test (+ 2 (call-with-exit (lambda* ((hi 1)) (hi 1)))) 3)
(test (call-with-exit (lambda (hi) (((hi 1)) #t))) 1) ; !! (jumps out of list evaluation)
(test (call-with-exit (lambda* args ((car args) 1))) 1)
(test ((call-with-exit (lambda (return) (return + 1 2 3)))) 6)
(test ((call-with-exit (lambda (return) (apply return (list + 1 2 3))))) 6)
(test ((call/cc (lambda (return) (return + 1 2 3)))) 6)

(test (+ 2 (values 3 (call-with-exit (lambda (k1) (k1 4))) 5)) 14)
(test (+ 2 (call-with-exit (lambda (k1) (values 3 (k1 4) 5))) 8) 14)
(test (+ 2 (call-with-exit (lambda (k1) (values 3 (k1 4 -3) 5))) 8) 11)

(test (call-with-exit (let () (lambda (k1) (k1 2)))) 2)
(test (+ 2 (call/cc (let () (call/cc (lambda (k1) (k1 (lambda (k2) (k2 3)))))))) 5)
(test (+ 2 (call/cc (call/cc (lambda (k1) (k1 (lambda (k2) (k2 3))))))) 5)
(test (call-with-exit (lambda arg ((car arg) 32))) 32)
(test (call-with-exit (lambda arg ((car arg) 32)) "oops!") 'error)
(test (call-with-exit (lambda (a b) a)) 'error)
(test (call-with-exit (lambda (return) (apply return '(3)))) 3)
(test (call-with-exit (lambda (return) (apply return (list  (cons 1 2))) (format #t "; call-with-exit: we shouldn't be here!"))) (cons 1 2))
(test (call/cc (lambda (return) (apply return (list  (cons 1 2))) (format #t "; call/cc: we shouldn't be here!"))) (cons 1 2))
(test (procedure? (call-with-exit (lambda (return) (call-with-exit return)))) #t)
(test (call-with-exit (lambda (return) #f) 1) 'error)
(test (+ (call-with-exit ((lambda () (lambda (k) (k 1 2 3)))))) 6)

(test (let ((x 0))
	(define (quit z1) (z1 1) (set! x 1))
	(call-with-exit
	 (lambda (z)
	   (set! x 2)
	   (quit z)
	   (set! x 3)))
	x)
      2)

(test (let ((x (call/cc (lambda (k) k))))
	(x (lambda (y) "hi")))
      "hi")

(test (((call/cc (lambda (k) k)) (lambda (x) x)) "hi") "hi")

(test (let ((return #f)
	    (lst '()))
	(let ((val (+ 1 (call/cc 
			 (lambda (cont) 
			   (set! return cont) 
			   1)))))
	  (set! lst (cons val lst)))
	(if (= (length lst) 1)
	    (return 10)
	    (if (= (length lst) 2)
		(return 20)))
	(reverse lst))
      '(2 11 21))

(test (let ((r1 #f)
	    (r2 #f)
	    (lst '()))
	(define (somefunc x y)
	  (+ (* 2 (expt x 2)) (* 3 y) 1))
	(let ((val (somefunc (call/cc
			      (lambda (c1)
				(set! r1 c1)
				(c1 1)))
			     (call/cc
			      (lambda (c2)
				(set! r2 c2)
				(c2 1))))))
	  (set! lst (cons val lst)))
	(if (= (length lst) 1)
	    (r1 2)
	    (if (= (length lst) 2)
		(r2 3)))
	(reverse lst))
      '(6 12 18))

(let ((tree->generator
       (lambda (tree)
	 (let ((caller '*))
	   (letrec
	       ((generate-leaves
		 (lambda ()
		   (let loop ((tree tree))
		     (cond ((null? tree) 'skip)
			   ((pair? tree)
			    (loop (car tree))
			    (loop (cdr tree)))
			   (else
			    (call/cc
			     (lambda (rest-of-tree)
			       (set! generate-leaves
				     (lambda ()
				       (rest-of-tree 'resume)))
			       (caller tree))))))
		   (caller '()))))
	     (lambda ()
	       (call/cc
		(lambda (k)
		  (set! caller k)
		  (generate-leaves)))))))))
  (let ((same-fringe? 
	 (lambda (tree1 tree2)
	   (let ((gen1 (tree->generator tree1))
		 (gen2 (tree->generator tree2)))
	     (let loop ()
	       (let ((leaf1 (gen1))
		     (leaf2 (gen2)))
		 (if (eqv? leaf1 leaf2)
		     (if (null? leaf1) #t (loop))
		     #f)))))))
    
    (test (same-fringe? '(1 (2 3)) '((1 2) 3)) #t)
    (test (same-fringe? '(1 2 3) '(1 (3 2))) #f)))

(let ()
  (define (a-func)
    (call-with-exit
     (lambda (go)
       (lambda ()
	 (go + 32 1)))))

  (define (b-func)
    (call/cc
     (lambda (go)
       (lambda ()
	 (go + 32 1)))))

  (test ((a-func)) 'error) ;invalid-escape-function
  (test ((b-func)) 33))

(test ((call-with-exit
	(lambda (go)
	  (lambda ()
	    (eval-string "(go + 32 1)")))))
      'error)


;;; (test ((call/cc (lambda (go) (lambda () (eval-string "(go + 32 1)"))))) 33)
;;; this is ok in the listener, but exits the load in this context

(test ((call/cc
	(lambda (go-1)
	  (call/cc
	   (lambda (go)
	     (lambda ()
	       (go (go-1 + 32 1))))))))
      33)

(for-each
 (lambda (arg)
   (test (let ((ctr 0))
	   (let ((val (call/cc (lambda (exit)
				 (do ((i 0 (+ i 1)))
				     ((= i 10) 'gad)
				   (set! ctr (+ ctr 1))
				   (if (= i 1)
				       (exit arg)))))))
	     (and (equal? val arg)
		  (= ctr 2))))
	 #t))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (let ((ctr 0))
	   (let ((val (call/cc (lambda (exit)
				 (do ((i 0 (+ i 1)))
				     ((= i 10) arg)
				   (set! ctr (+ ctr 1))
				   (if (= i 11)
				       (exit 'gad)))))))
	     (and (equal? val arg)
		  (= ctr 10))))
	 #t))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (let ((c #f)
	    (r (string-copy "testing-hiho")))
	(let ((v (call/cc (lambda (c0) (set! c c0) (list #\a 0)))))
	  (let ((chr (car v))
		(index (cadr v)))
	    (string-set! r index chr)
	    (set! index (+ index 1))
	    (if (<= index 8) 
		(c (list (integer->char (+ 1 (char->integer chr))) index)) 
		r))))
      "abcdefghiiho")

(test (let ((x 0)
	    (again #f))
	(call/cc (lambda (r) (set! again r)))
	(set! x (+ x 1))
	(if (< x 3) (again))
	x)
      3)

(test (let* ((x 0)
	     (again #f)
	     (func (lambda (r) (set! again r))))
	(call/cc func)
	(set! x (+ x 1))
	(if (< x 3) (again))
	x)
      3)

(test (let* ((x 0)
	     (again #f))
	(call/cc (let ()
		   (lambda (r) (set! again r))))
	(set! x (+ x 1))
	(if (< x 3) (again))
	x)
      3)

(test (let ((x 0)
	    (xx 0))
	(let ((cont #f))
	  (call/cc (lambda (c) (set! xx x) (set! cont c)))
	  (set! x (+ x 1))
	  (if (< x 3)	(cont))
	  xx))
      0)

(test (call/cc procedure?) #t)
(test (procedure? (call/cc (lambda (a) a))) #t)

(for-each
 (lambda (arg)
   (test (call/cc (lambda (a) arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((a (call/cc (lambda (a) a))))
  (test (eq? a a) #t)
  (test (eqv? a a) #t)
  (test (equal? a a) #t)
  (for-each
   (lambda (ques)
     (if (ques a)
	 (format #t ";(~A ~A) returned #t?~%" ques a)))
   question-ops))

(test (let ((conts (make-vector 4 #f)))
	(let ((lst '()))
	  (set! lst (cons (+ (call/cc (lambda (a) (vector-set! conts 0 a) 0))
			     (call/cc (lambda (a) (vector-set! conts 1 a) 0))
			     (call/cc (lambda (a) (vector-set! conts 2 a) 0))
			     (call/cc (lambda (a) (vector-set! conts 3 a) 0)))
			  lst))
	  (let ((len (length lst)))
	    (if (< len 4)
		((vector-ref conts (- len 1)) (+ len 1))
		(reverse lst)))))
      '(0 2 5 9))

(test (let ((conts '()))
	(let ((lst '()))
	  (set! lst (cons (+ (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) 1))
			     (* (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) 1))
				(+ (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) 1))
				   (* (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) 1)) 2))))
			  lst))
	  (let ((len (length lst)))
	    (if (<= len 4)
		((list-ref conts (- len 1)) (+ len 1))
		(reverse lst)))))
					; (+ 1 (* 1 (+ 1 (* 1 2)))) to start
					; (+ 1 ...          2     )
					; (+ 1 ...     3    [1]   )
					; (+ 1 ...4    [1]        )
					; (+ 5   [1]              )
      '(4 6 6 13 8))

(test (let ((conts (make-vector 4 #f)))
	(let ((lst '()))
	  (set! lst (cons (+ (call/cc (lambda (a) (if (not (vector-ref conts 0)) (vector-set! conts 0 a)) 0))
			     (call/cc (lambda (a) (if (not (vector-ref conts 1)) (vector-set! conts 1 a)) 0))
			     (call/cc (lambda (a) (if (not (vector-ref conts 2)) (vector-set! conts 2 a)) 0))
			     (call/cc (lambda (a) (if (not (vector-ref conts 3)) (vector-set! conts 3 a)) 0)))
			  lst))
	  (let ((len (length lst)))
	    (if (< len 4)
		((vector-ref conts (- len 1)) (+ len 1))
		(reverse lst)))))
      '(0 2 3 4))

(test (let ((conts '()))
	(let ((lst '()))
	  (set! lst (cons (+ (if (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) #f)) 1 0)
			     (* (if (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) #f)) 2 1)
				(+ (if (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) #f)) 1 0)
				   (* (if (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) #f)) 2 1) 2))))
			  lst))
	  (let ((len (length lst)))
	    (if (<= len 4)
		((list-ref conts (- len 1)) #t)
		(reverse lst)))))
					; (+ 0 (* 1 (+ 0 (* 1 2)))) to start
					; (+ 0 ...          2     )
					; (+ 0 ...     1   [1]    )
					; (+ 0 ...2   [0]         )
					; (+ 1   [1]              )
      '(2 4 3 4 3))

(test (let ((call/cc 2)) (+ call/cc 1)) 3)
(test (+ 1 (call/cc (lambda (r) (r 2 3 4))) 5) 15)
(test (string-ref (call/cc (lambda (s) (s "hiho" 1)))) #\i)

(let ((r5rs-ratify (lambda (ux err)
		     (if (= ux 0.0) 
			 0
			 (let ((tt 1) 
			       (a1 0) 
			       (b2 0) 
			       (a2 1) 
			       (b1 1) 
			       (a 0)  
			       (b 0)
			       (ctr 0)
			       (x (/ 1 ux)))
			   (call-with-current-continuation
			    (lambda (return)
			      (do ()
				  (#f)
				(set! a (+ (* a1 tt) a2)) 
				(set! b (+ (* tt b1) b2))
					;(format #t "~A ~A~%" a (- b a))
				(if (or (<= (abs (- ux (/ a b))) err)
					(> ctr 1000))
				    (return (/ a b)))
				(set! ctr (+ 1 ctr))
				(if (= x tt) (return))
				(set! x (/ 1 (- x tt))) 
				(set! tt (floor x))
				(set! a2 a1) 
				(set! b2 b1) 
				(set! a1 a) 
				(set! b1 b)))))))))
  
  (test (r5rs-ratify (/ (log 2.0) (log 3.0)) 1/10000000) 665/1054)
  (if (positive? 2147483648)
      (test (r5rs-ratify (/ (log 2.0) (log 3.0)) 1/100000000000) 190537/301994)))

#|
(let ((max-diff 0.0)
      (max-case 0.0)
      (err 0.01)
      (epsilon 1e-16))
  (do ((i 1 (+ i 1))) 
      ((= i 100)) 
    (let ((x (/ i 100.)))
      (let ((vals (cr x err))) 
	(if (not (= (car vals) (cadr vals))) 
	    (let ((r1 (car vals))
		  (r2 (cadr vals)))
	      (let ((diff (abs (- r1 r2))))
		(if (> diff max-diff)
		    (begin
		      (set! max-diff diff)
		      (set! max-case x))))
	      (if (> (abs (- r1 x)) (+ err epsilon))
		(format #t "(rationalize ~A ~A) is off: ~A -> ~A~%" x err r1 (abs (- r1 x))))
	      (if (> (abs (- r2 x)) (+ err epsilon))
		(format #t "(ratify ~A ~A) is off: ~A -> ~A~%" x err r2 (abs (- r2 x))))
	      (if (< (denominator r2) (denominator r1))
		  (format #t "(ratify ~A ~A) is simpler? ~A ~A~%" x err r1 r2)))))))
  (list max-case max-diff (cr max-case err)))
|#

(for-each
 (lambda (arg)
   (test (let ((ctr 0)) 
	   (let ((val (call/cc 
		       (lambda (exit) 
			 (for-each (lambda (a) 
				     (if (equal? a arg) (exit arg))
				     (set! ctr (+ ctr 1))) 
				   (list 0 1 2 3 arg 5)))))) 
	     (list ctr val)))
	 (list 4 arg)))
 (list "hi" -1 #\a 11 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t '(1 . 2)))

(test (+ 2 (call/cc (lambda (rtn) (+ 1 (let () (begin (define x (+ 1 (rtn 3)))) x))))) 5)


;;; others from stackoverflow.com Paul Hollingsworth etc:

(test (procedure? (call/cc (lambda (k) k))) #t)
(test (call/cc (lambda (k) (+ 56 (k 3)))) 3)
(test (apply
       (lambda (k i) 
	 (if (> i 5) 
	     i 
	     (k (list k (* 2 i)))))
       (call/cc (lambda (k) (list k 1))))
      8)
(test (apply
       (lambda (k i n) (if (= i 0) n (k (list k (- i 1) (* i n)))))
       (call/cc (lambda (k) (list k 6 1))))
      720)
(test (let* ((ka (call/cc (lambda (k) `(,k 1)))) (k (car ka)) (a (cadr ka)))
	(if (< a 5) (k `(,k ,(* 2 a))) a))
      8)

(test (apply (lambda (k i n) (if (eq? i 0) n (k (list k (- i 1) (* i n))))) (call/cc (lambda (k) (list k 6 1)))) 720)
(test ((call/cc (lambda (k) k)) (lambda (x) 5)) 5)

(let ()
  (define (generate-one-element-at-a-time a-list)
    (define (generator)
      (call/cc control-state)) 
    (define (control-state return)
      (for-each 
       (lambda (an-element-from-a-list)
	 (set! return
	       (call/cc
		(lambda (resume-here)
		  (set! control-state resume-here)
		  (return an-element-from-a-list)))))
       a-list)
      (return 'you-fell-off-the-end-of-the-list))
    generator)
  (let ((gen (generate-one-element-at-a-time (list 3 2 1))))
    (test (gen) 3)
    (test (gen) 2)
    (test (gen) 1)
    (test (gen) 'you-fell-off-the-end-of-the-list)))

;;; from Ferguson and Duego "call with current continuation patterns"
(test (let ()
	(define count-to-n
	  (lambda (n)
	    (let ((receiver 
		   (lambda (exit-procedure)
		     (let ((count 0))
		       (letrec ((infinite-loop
				 (lambda ()
				   (if (= count n)
				       (exit-procedure count)
				       (begin
					 (set! count (+ count 1))
					 (infinite-loop))))))
			 (infinite-loop))))))
	      (call/cc receiver))))
	(count-to-n 10))
      10)

(test (let ()
	(define product-list
	  (lambda (nums)
	    (let ((receiver
		   (lambda (exit-on-zero)
		     (letrec ((product
			       (lambda (nums)
				 (cond ((null? nums) 1)
				       ((zero? (car nums)) (exit-on-zero 0))
				       (else (* (car nums)
						(product (cdr nums))))))))
		       (product nums)))))
	      (call/cc receiver))))
	(product-list '(1 2 3 0 4 5)))
      0)

(begin
  (define fact
    ((lambda (f)
       ((lambda (u) (u (lambda (x)
			 (lambda (n) ((f (u x)) n)))))
	(call/cc (call/cc (call/cc 
			   (call/cc (call/cc (lambda (x) x))))))))
     (lambda (f) (lambda (n)
		   (if (<= n 0) 1 (* n (f (- n 1))))))))
  (test (map fact '(5 6 7)) '(120 720 5040)))

;; http://okmij.org/ftp/Scheme/callcc-calc-page.html

(test (let ()
	(define product-list
	  (lambda (nums)
	    (let ((receiver
		   (lambda (exit-on-zero)
		     (letrec ((product
			       (lambda (nums) 
				 (cond ((null? nums) 1)
				       ((number? (car nums))
					(if (zero? (car nums))
					    (exit-on-zero 0)
					    (* (car nums)
					       (product (cdr nums)))))
				       (else (* (product (car nums))
						(product (cdr nums))))))))
		       (product nums)))))
	      (call/cc receiver))))
	(product-list '(1 2 (3 4) ((5)))))
      120)

(test (call/cc (lambda () 0)) 'error)
(test (call/cc (lambda (a) 0) 123) 'error)
(test (call/cc) 'error)
(test (call/cc abs) 'error)
(for-each
 (lambda (arg)
   (test (call-with-exit arg) 'error)
   (test (call-with-current-continuation arg) 'error)
   (test (call/cc arg) 'error))
 (list "hi" -1 '() #(1 2) _ht_ #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call/cc . 1) 'error)
(test (call/cc abs) 'error)
(test (+ 1 (call/cc (lambda (r1) (call/cc (lambda (r2) (r1 2 3))))) 4) 10)
(test (+ 1 (call/cc (lambda (r1) (+ 5 (call/cc (lambda (r2) (r2 2 3)))))) 4) 15)


#|
;;; from bug-guile
(define k #f)
(define result #f)
(define results '())
(set! result (map (lambda (x)
                    (if x x (call/cc (lambda (c)
                                       (set! k c)
                                       1))))
                  '(#t #f)))
(set! results (cons result results))
(write results)
(newline)
(if (< (cadr result) 5)
    (k (+ 1 (cadr result))))
(newline)

the claim is that this should return 

((#t 1))
((#t 2) (#t 1))
((#t 3) (#t 2) (#t 1))
((#t 4) (#t 3) (#t 2) (#t 1))
((#t 5) (#t 4) (#t 3) (#t 2) (#t 1))

but I think that depends on how we interpret the sequence of top-level statements.
The test should be written:

(let* ((k #f)
       (results '()))
  (let ((result (map (lambda (x)
		       (if x x (call/cc (lambda (c)
					  (set! k c)
					  1))))
		     '(#t #f))))
    (set! results (cons result results))
    (write results)
    (newline)
    (if (< (cadr result) 5)
	(k (+ 1 (cadr result))))
    (newline)))

and then s7 is not following r6rs because it stops at 

((#t 1))
((1 . #1=(#t 2)) #1#)

saying cadr is not a number. I don't think this example is correct in any case --
who says the continuation has to restart the map from the top?
|#

(let ((cont #f))
  (let ((x (* (call/cc
	       (lambda (return)
		 (set! cont return)
		 (return 3 4))))))
    (if (= x 12)
	(cont 5 6 7))
    (test x 210)))

;; Guile handles this very differently


(test (let ((cont #f)) (call-with-exit (lambda (return) (set! cont return))) (cont 1)) 'error)
(test (let ((cont #f)) (call-with-exit (lambda (return) (set! cont return))) (apply cont)) 'error)
(test (let ((cont #f)) (call-with-exit (lambda (return) (set! cont return) (cont 1))) (apply cont)) 'error)
(test (let ((cont #f)) (call-with-exit (lambda (return) (set! cont return) (cont 1))) (cont 1)) 'error)
(test (procedure? (call-with-exit append)) #t)
(test (procedure? (call-with-exit values)) #t)
(test (procedure? (car (call-with-exit list))) #t)
(test (call-with-exit (call-with-exit append)) 'error)
(test (continuation? (call/cc (call/cc append))) #t)
(test (procedure? (call-with-exit call-with-exit)) #t)
(test (vector? (call-with-exit vector)) #t)
(test (call-with-exit ((lambda args procedure?))) #t)
(test (call-with-exit (let ((x 3)) (define (return y) (y x)) return)) 3)

(test (let ((c1 #f)) (call-with-exit (lambda (c2) (call-with-exit (lambda (c3) (set! c1 c3) (c2))))) (c1)) 'error)
(test (let ((c1 #f)) (call/cc (lambda (c2) (call-with-exit (lambda (c3) (set! c1 c3) (c2))))) (c1)) 'error)
(test (let ((cont #f)) (catch #t (lambda () (call-with-exit (lambda (return) (set! cont return) (error 'testing " a test")))) (lambda args 'error)) (apply cont)) 'error)
(test (let ((cont #f)) (catch #t (lambda () (call-with-exit (lambda (return) (set! cont return) (error 'testing " a test")))) (lambda args 'error)) (cont 1)) 'error)
(test (let ((e (call-with-exit (lambda (go) (lambda () (go 1)))))) (e)) 'error)

(test (let ((cc #f)
	    (doit #t)
	    (ctr 0))
	(let ((ok (call-with-exit
		   (lambda (c3)
		     (call/cc (lambda (ret) (set! cc ret)))
		     (c3 (let ((res doit)) (set! ctr (+ ctr 1)) (set! doit #f) res))))))
	  (if ok (cc)))
	ctr)
      2)

(test (let ((val (call-with-exit (lambda (ret) (let ((ret1 ret)) (ret1 2) 3))))) val) 2)
(test (call-with-exit (lambda (return) (sort! '(3 2 1) return))) 'error)

;;; this one from Rick
(test (eval '(call/cc (lambda (go) (go 9) 0))) 9)
(test (eval-string "(call/cc (lambda (go) (go 9) 0))") 9)
(test (call-with-exit (lambda (return) (call/cc (lambda (go) (go 9) 0)) (return 1) 2)) 1)

(num-test (/ 1 (call/cc (lambda (go) (go 9) 0))) 1/9)

(test (call/cc (lambda (g) (call/cc (lambda (f) (f 1)) (g 2)))) 2) ; !! guile agrees! (evaluating the extraneous arg jumps)
(test (call/cc (lambda (g) (abs -1 (g 2)))) 2)                     ; perhaps this should be an error
(test (call/cc (lambda (g) (if #t #f #f (g 2)))) 'error)

(test ((call-with-exit (lambda (go) (go go))) eval) 'error)
(test ((call/cc (lambda (go) (go go))) eval) eval)
(test (call-with-exit quasiquote) 'error)

(test (call-with-exit (lambda (go) (if (go 1) (go 2) (go 3)))) 1)
(test (call-with-exit (lambda (go) (set! (go 1) 2))) 'error) 
(test (call-with-exit (lambda (go) (let ((x 1) (y (go x))) #f))) 'error)
(test (call-with-exit (lambda (go) (let* ((x 1) (y (go x))) #f))) 1)
(test (call-with-exit (lambda (go) (letrec ((x 1) (y (go x))) #f))) #<undefined>)
(test (call-with-exit (lambda (go) (letrec* ((x 1) (y (go x))) #f))) 1)
(test (call-with-exit (lambda (go) (case (go 1) ((go 2) 3) (else 4)))) 1)
(test (call-with-exit (lambda (go) (case go ((go 2) 3) (else 4)))) 4)
(test (call-with-exit (lambda (go) (case 2 ((go 2) 3) (else 4)))) 3)
(test (call-with-exit (lambda (go) (eq? go go))) #t)
(test (call-with-exit (lambda (go) (case 'go ((go 2) 3) (else 4)))) 3)
(test (call-with-exit (lambda (go) (go (go (go 1))))) 1)
(test (call-with-exit (lambda (go) (quasiquote (go 1)))) '(go 1))
(test (call-with-exit (lambda (go) ((lambda* (a (go 1)) a) (go 2) 3))) 2)
(test (call-with-exit (lambda (go) ((lambda* (a (go 1)) a) 2))) 2) ; default arg not evaluated if not needed
(test (call-with-exit (lambda (go) ((lambda* (a (go 1)) a)))) #f) ; lambda_star_argument_default_value in s7 explicitly desires this
(test (call-with-exit (lambda (go) ((lambda (go) go) 1))) 1)
(test (call-with-exit (lambda (go) (quote (go 1)) 2)) 2)
(test (call-with-exit (lambda (go) (and (go 1) #f))) 1)
(test (call-with-exit (lambda (go) (dynamic-wind (lambda () (go 1) 11) (lambda () (go 2) 12) (lambda () (go 3) 13)))) 1)

(test (eval '(call/cc (lambda (go) (if (go 1) (go 2) (go 3))))) 1)
(test (eval '(call/cc (lambda (go) (set! (go 1) 2)))) 'error) 
(test (eval '(call/cc (lambda (go) (let ((x 1) (y (go x))) #f)))) 'error)
(test (eval '(call/cc (lambda (go) (let* ((x 1) (y (go x))) #f)))) 1)
(test (eval '(call/cc (lambda (go) (letrec ((x 1) (y (go x))) #f)))) #<undefined>)
(test (eval '(call/cc (lambda (go) (letrec* ((x 1) (y (go x))) #f)))) 1)
(test (eval '(call/cc (lambda (go) (case (go 1) ((go 2) 3) (else 4))))) 1)
(test (eval '(call/cc (lambda (go) (case go ((go 2) 3) (else 4))))) 4)
(test (eval '(call/cc (lambda (go) (case 2 ((go 2) 3) (else 4))))) 3)
(test (eval '(call/cc (lambda (go) (eq? go go)))) #t)
(test (eval '(call/cc (lambda (go) (case 'go ((go 2) 3) (else 4))))) 3)
(test (eval '(call/cc (lambda (go) (go (go (go 1)))))) 1)
(test (eval '(call/cc (lambda (go) (quasiquote (go 1))))) '(go 1))
(test (eval '(call/cc (lambda (go) ((lambda* (a (go 1)) a) (go 2))))) 2)
(test (eval '(call/cc (lambda (go) ((lambda* (a (go 1)) a) 2)))) 2)
(test (eval '(call/cc (lambda (go) ((lambda* (a (go 1)) a))))) #f)
(test (eval '(call/cc (lambda (go) ((lambda (go) go) 1)))) 1)
(test (eval '(call/cc (lambda (go) (quote (go 1)) 2))) 2)
(test (eval '(call/cc (lambda (go) (and (go 1) #f)))) 1)
(test (eval '(call/cc (lambda (go) (dynamic-wind (lambda () (go 1) 11) (lambda () (go 2) 12) (lambda () (go 3) 13))))) 1)

(test (call-with-exit (lambda (go) (eval '(go 1)) 2)) 1) 
(test (call-with-exit (lambda (go) (eval-string "(go 1)") 2)) 1)
(test (call-with-exit (lambda (go) `(,(go 1) 2))) 1)
;;; (test (call-with-exit (lambda (go) `#(,(go 1) 2))) 'error) ; this is s7's choice -- read time #(...)
(test (call-with-exit (lambda (go) (case 0 ((0) (go 1) (go 2))))) 1)
(test (call-with-exit (lambda (go) (cond (1 => go)) 2)) 1)
(test (call-with-exit (lambda (go) (((cond ((go 1) => go)) 2)))) 1)
(test (call-with-exit (lambda (go) (cond (1 => (go 2))))) 2)

(test (call-with-exit (lambda (go) (go (eval '(go 1))) 2)) 1)
(test (+ 10 (call-with-exit (lambda (go) (go (eval '(go 1))) 2))) 11)
(test (call-with-exit (lambda (go) (go (eval-string "(go 1)")) 2)) 1)
(test (call-with-exit (lambda (go) (eval-string "(go 1)") 2)) 1) 
(test (call-with-exit (lambda (go) ((eval 'go) 1) 2)) 1)  
(test (eval-string "(call/cc (lambda (go) (if (go 1) (go 2) (go 3))))") 1)
(test (call-with-exit (lambda (quit) ((lambda* ((a (quit 32))) a)))) 32)
(test ((call-with-exit (lambda (go) (go quasiquote))) go) 'go)

(test (let ((c #f))
	(let ((val -1))
	  (set! val (call/cc 
		     (lambda (c1)
		       (call-with-exit 
			(lambda (c2)
			  (call-with-exit 
			   (lambda (c3)
			     (call/cc 
			      (lambda (c4)
				(set! c c4)
				(c1 (c2 0)))))))))))
	  (if (= val 0) (c 5))
	  val))
      5)




;;; --------------------------------------------------------------------------------
;;; dynamic-wind
;;; --------------------------------------------------------------------------------

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (dynamic-wind
			(lambda () (set! ctr1 (+ ctr1 1)))
			(lambda () (set! ctr2 (+ ctr2 1)) (+ 1 ctr2))
			(lambda () (set! ctr3 (+ ctr3 1))))))
	  (= ctr1 ctr2 ctr3 (- ctr4 1) 1)))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (catch 'dw
			   (lambda ()
			     (dynamic-wind
				 (lambda () (set! ctr1 (+ ctr1 1)))
				 (lambda () (set! ctr2 (+ ctr2 1)) (error 'dw "dw-error") ctr2)
				 (lambda () (set! ctr3 (+ ctr3 1)))))
			   (lambda args (car args)))))
	  (and (eq? ctr4 'dw)
	       (= ctr1 1) (= ctr2 1) (= ctr3 1))))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (catch 'dw
			   (lambda ()
			     (dynamic-wind
				 (lambda () (set! ctr1 (+ ctr1 1)))
				 (lambda () (error 'dw "dw-error") (set! ctr2 (+ ctr2 1)) ctr2)
				 (lambda () (set! ctr3 (+ ctr3 1)))))
			   (lambda args (car args)))))
	  (and (eq? ctr4 'dw)
	       (= ctr1 1) (= ctr2 0) (= ctr3 1))))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (catch #t
			   (lambda ()
			     (dynamic-wind
				 (lambda () (set! ctr1 (+ ctr1 1)) (error 'dw-init "dw-error"))
				 (lambda () (set! ctr2 (+ ctr2 1)) (error 'dw "dw-error") ctr2)
				 (lambda () (set! ctr3 (+ ctr3 1)))))
			   (lambda args (car args)))))
	  (and (eq? ctr4 'dw-init)
	       (= ctr1 1) (= ctr2 0) (= ctr3 0))))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (catch #t
			   (lambda ()
			     (dynamic-wind
				 (lambda () (set! ctr1 (+ ctr1 1)))
				 (lambda () (set! ctr2 (+ ctr2 1)) ctr2)
				 (lambda () (set! ctr3 (+ ctr3 1)) (error 'dw-final "dw-error"))))
			   (lambda args (car args)))))
	  (and (eq? ctr4 'dw-final)
	       (= ctr1 1) (= ctr2 1) (= ctr3 1))))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (catch #t
			   (lambda ()
			     (dynamic-wind
				 (lambda () (set! ctr1 (+ ctr1 1)))
				 (lambda () (set! ctr2 (+ ctr2 1)) ctr2)
				 (lambda () (error 'dw-final "dw-error") (set! ctr3 (+ ctr3 1)))))
			   (lambda args (car args)))))
	  (and (eq? ctr4 'dw-final)
	       (= ctr1 1) (= ctr2 1) (= ctr3 0))))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (call/cc (lambda (exit)
			       (dynamic-wind
				   (lambda () (set! ctr1 (+ ctr1 1)))
				   (lambda () (exit ctr2) (set! ctr2 (+ ctr2 1)) ctr2)
				   (lambda () (set! ctr3 (+ ctr3 1)) 123))))))
	  (and (= ctr1 ctr3 1)
	       (= ctr2 ctr4 0))))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (call/cc (lambda (exit)
			       (dynamic-wind
				   (lambda () (exit ctr1) (set! ctr1 (+ ctr1 1)))
				   (lambda () (set! ctr2 (+ ctr2 1)) ctr2)
				   (lambda () (set! ctr3 (+ ctr3 1))))))))
	  (= ctr1 ctr2 ctr3 ctr4 0)))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (call/cc (lambda (exit)
			       (dynamic-wind
				   (lambda () (set! ctr1 (+ ctr1 1)))
				   (lambda () (set! ctr2 (+ ctr2 1)) ctr2)
				   (lambda () (exit ctr3) (set! ctr3 (+ ctr3 1))))))))
	  (and (= ctr1 ctr2 1)
	       (= ctr3 ctr4 0))))
      #t)

(test (let ((path '())  
	    (c #f)) 
	(let ((add (lambda (s)  
		     (set! path (cons s path))))) 
	  (dynamic-wind  
	      (lambda () (add 'connect))  
	      (lambda () (add (call-with-current-continuation  
			       (lambda (c0) (set! c c0) 'talk1))))  
	      (lambda () (add 'disconnect))) 
	  (if (< (length path) 4) 
	      (c 'talk2) 
	      (reverse path)))) 
      '(connect talk1 disconnect  connect talk2 disconnect))


(for-each
 (lambda (arg)
   (test (dynamic-wind (lambda () #f) (lambda () arg) (lambda () #f)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (dynamic-wind (lambda () #f) (lambda () #f) (lambda () #f)) #f)
(test (+ 1 (dynamic-wind (lambda () #f) (lambda () (values 2 3 4)) (lambda () #f)) 5) 15)

(test (let ((identity (lambda (a) a)))
        (let ((x '())
              (c #f))
          (dynamic-wind
	      (lambda () (set! x (cons 'a x)))
	      (lambda ()
		(dynamic-wind
		    (lambda () (set! x (cons 'b x)))
		    (lambda ()
		      (dynamic-wind
			  (lambda () (set! x (cons 'c x)))
			  (lambda () (set! c (call/cc identity)))
			  (lambda () (set! x (cons 'd x)))))
		    (lambda () (set! x (cons 'e x))))
		(dynamic-wind
		    (lambda () (set! x (cons 'f x)))
		    (lambda () (if c (c #f)))
		    (lambda () (set! x (cons 'g x)))))
	      (lambda () (set! x (cons 'h x))))
          (reverse x)))
      '(a b c d e f g b c d e f g h))


(test (list (dynamic-wind 
		(lambda () #f)
		(lambda () (values 'a 'b 'c))
		(lambda () #f)))
      (list 'a 'b 'c))

(test (let ((dynamic-wind 1)) (+ dynamic-wind 2)) 3)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((val (dynamic-wind
		       (lambda () #f)
		       (lambda ()
			 (set! ctr1 1)
			 (call/cc
			  (lambda (exit)
			    (exit 123)
			    (set! ctr2 2)
			    321)))
		       (lambda ()
			 (set! ctr3 3)))))
	  (and (= ctr1 1) (= ctr2 0) (= ctr3 3) (= val 123))))
      #t)

(test (let ((ctr1 0))
	(let ((val (dynamic-wind
		       (let ((a 1))
			 (lambda ()
			   (set! ctr1 a)))
		       (let ((a 10))
			 (lambda ()
			   (set! ctr1 (+ ctr1 a))
			   ctr1))
		       (let ((a 100))
			 (lambda ()
			   (set! ctr1 (+ ctr1 a)))))))
	  (and (= ctr1 111) (= val 11))))
      #t)

(test (let ((ctr1 0))
	(let ((val (+ 3 (dynamic-wind
			    (let ((a 1))
			      (lambda ()
				(set! ctr1 a)))
			    (let ((a 10))
			      (lambda ()
				(set! ctr1 (+ ctr1 a))
				ctr1))
			    (let ((a 100))
			      (lambda ()
				(set! ctr1 (+ ctr1 a)))))
		      1000)))
	  (and (= ctr1 111) (= val 1014))))
      #t)

(test (let ((n 0))
	(call-with-current-continuation
	 (lambda (k)
	   (dynamic-wind
	       (lambda ()
		 (set! n (+ n 1))
		 (k))
	       (lambda ()
		 (set! n (+ n 2)))
	       (lambda ()
		 (set! n (+ n 4))))))
	n)
      1)

(test (let ((n 0))
	(call-with-current-continuation
	 (lambda (k)
	   (dynamic-wind
	       (lambda () #f)
	       (lambda ()
		 (dynamic-wind
		     (lambda () #f)
		     (lambda ()
		       (set! n (+ n 1))
		       (k))
		     (lambda ()
		       (set! n (+ n 2))
					;(k)
		       )))
	       (lambda ()
		 (set! n (+ n 4))))))
	n)
      7)

(test (let ((n 0))
	(call-with-current-continuation
	 (lambda (k)
	   (dynamic-wind
	       (lambda () #f)
	       (lambda ()
		 (dynamic-wind
		     (lambda () #f)
		     (lambda ()
		       (dynamic-wind
			   (lambda () #f)
			   (lambda ()
			     (set! n (+ n 1))
			     (k))
			   (lambda ()
			     (if (= n 1)
				 (set! n (+ n 2))))))
		     (lambda ()
		       (if (= n 3)
			   (set! n (+ n 4))))))
	       (lambda ()
		 (if (= n 7)
		     (set! n (+ n 8)))))))
	n)
      15)

(test (dynamic-wind) 'error)
(test (dynamic-wind (lambda () #f)) 'error)
(test (dynamic-wind (lambda () #f) (lambda () #f)) 'error)
(test (dynamic-wind (lambda (a) #f) (lambda () #f) (lambda () #f)) 'error)
(test (dynamic-wind (lambda () #f) (lambda (a b) #f) (lambda () #f)) 'error)
(test (dynamic-wind (lambda () #f) (lambda () #f) (lambda (a) #f)) 'error)
(test (dynamic-wind (lambda () 1) #f (lambda () 2)) 'error)
(test (dynamic-wind . 1) 'error)
(test (dynamic-wind () () ()) 'error)
(test (dynamic-wind () _ht_ ()) 'error)

(for-each
 (lambda (arg)
   (test (dynamic-wind arg (lambda () #f) (lambda () #f)) 'error)
   (test (dynamic-wind (lambda () #f) arg (lambda () #f)) 'error)
   (test (dynamic-wind (lambda () #f) (lambda () #f) arg) 'error))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (dynamic-wind (let ((x 1)) (lambda () x)) ((lambda () (lambda () 2))) s7-version) 2)
(test (let ((x 0)) (dynamic-wind (lambda () (set! x 1)) ((lambda () (set! x 32) (lambda () x))) (let () (set! x 44) (lambda () x)))) 1)
(test (let ((x 0)) (dynamic-wind (lambda () (set! x (+ x 1))) ((lambda () (set! x 32) (lambda () x))) (let () (set! x 44) (lambda () x)))) 45)
(test (let ((x 0)) (dynamic-wind (lambda () (set! x (+ x 1))) ((lambda () (set! x 32) (lambda () x))) (let () (lambda () x)))) 33)
(test (let ((x 0)) (dynamic-wind (lambda () (set! x (+ x 1))) ((lambda () (set! x (+ x 32)) (lambda () x))) (let () (lambda () (set! x (+ x 100)) x)))) 33)

(let ()
  (define-macro (make-thunk val) `(lambda () ,val))
  (test (dynamic-wind (make-thunk 0) (make-thunk 1) (make-thunk 2)) 1))

;;; from scheme wiki
;;; http://community.schemewiki.org/?hose-the-repl
;;; jorgen-schafer

(test (let loop ()  
	(call-with-exit
	 (lambda (k)  
	   (dynamic-wind  
	       (lambda () #t)  
	       (lambda () (let loop () (loop)))  
	       k))) 
	(loop))
      'error)
;; that example calls to mind a bunch like it:
(test (call-with-exit (lambda (k) (dynamic-wind (lambda () #t) (lambda () (let loop () (loop))) k))) 'error)
(test (call-with-exit (lambda (k) (dynamic-wind (lambda () #t) k (lambda () #t)))) 'error)
(test (call-with-exit (lambda (k) (dynamic-wind k (lambda () #f) (lambda () #t)))) 'error)

(test (call-with-exit (lambda (k) (procedure-documentation k))) "")
(test (call-with-exit (lambda (k) (procedure-arity k))) '(0 0 #t))
(test (call-with-exit (lambda (k) (procedure-source k))) '())
(test (procedure-arity (call-with-exit (lambda (k) (make-procedure-with-setter k k)))) '(0 0 #t))
(test (procedure-arity (make-procedure-with-setter vector-ref vector-set!)) '(2 0 #t))
(test (let ((pws (make-procedure-with-setter vector-ref vector-set!))) 
	(let ((pws1 (make-procedure-with-setter pws vector-set!))) 
	  (let ((v (vector 1 2))) 
	    (set! (pws1 v 1) 32) 
	    (pws1 v 1))))
      32)
(test (call-with-exit (lambda (k) (map k '(1 2 3)))) 1)
(test (call-with-exit (lambda (k) (for-each k '(1 2 3)))) 1)
(test (call-with-exit (lambda (k) (catch #t k k))) 'error)
(test (call-with-exit (lambda (k) (catch #t (lambda () #f) k))) #f)
(test (call-with-exit (lambda (k) (catch #t (lambda () (error 'an-error)) k))) 'error)
(test (procedure? (call-with-exit (lambda (return) (call-with-exit return)))) #t)
;(test (call-with-exit (lambda (k) (sort! '(1 2 3) k))) 'error) -- currently returns (values 2 3) which is plausible
(test (sort! '(1 2 3) (lambda () #f)) 'error)
(test (sort! '(1 2 3) (lambda (a) #f)) 'error)
(test (sort! '(1 2 3) (lambda (a b c) #f)) 'error)
(test (let () (define-macro (asdf a b) `(< ,a ,b)) (sort! '(1 2 3) asdf)) 'error)
(test (let () (let asdf () (sort! '(1 2 3) asdf))) 'error)
(test (let () (let asdf () (map asdf '(1 2 3)))) 'error)
(test (let () (let asdf () (for-each asdf '(1 2 3)))) 'error)
(test (dynamic-wind quasiquote s7-version s7-version) 'error)

(test (let ((ctr 0))
	(call-with-exit
	 (lambda (exit)
	   (let asdf
	       ()
	     (set! ctr (+ ctr 1))
	     (if (> ctr 2)
		 (exit ctr))
	     (dynamic-wind
		 (lambda () #f)
		 (lambda () #f)
		 asdf)))))
      3)

(test (let ((ctr 0))
	(dynamic-wind
	    (lambda () #f)
	    (lambda ()
	      (call-with-exit
	       (lambda (exit)
		 (catch #t
			(lambda ()
			  (error 'error))
			(lambda args
			  (exit 'error)))
		 (set! ctr 1))))
	    (lambda ()
	      (set! ctr (+ ctr 2))))
	ctr)
      2)
(test (call-with-exit (lambda (r1) (call-with-exit (lambda (r2) (call-with-exit (lambda (r3) (r1 12) (r2 1))) (r1 2))) 3)) 12)
(test (call-with-exit (lambda (r1) (call-with-exit (lambda (r2) (call-with-exit (lambda (r3) (r2 12) (r2 1))) (r1 2))) 3)) 3)
(test (call-with-exit (lambda (r1) (call-with-exit (lambda (r2) (call-with-exit (lambda (r3) (r3 12) (r2 1))) (r1 2))) 3)) 2)

(let ((pws (make-procedure-with-setter < >))) (test (sort! '(2 3 1 4) pws) '(1 2 3 4)))
(test (call-with-exit (lambda (k) (call-with-input-string "123" k))) 'error)
(test (call-with-exit (lambda (k) (call-with-input-file "tmp1.r5rs" k))) 'error)
(test (call-with-exit (lambda (k) (call-with-output-file "tmp1.r5rs" k))) 'error)
(test (call-with-exit (lambda (k) (call-with-output-string k))) 'error)
(let ((pws (make-procedure-with-setter (lambda (a) (+ a 1)) (lambda (a b) b))))
  (test (procedure? pws) #t)
  (test (map pws '(1 2 3)) '(2 3 4))
  (test (apply pws '(1)) 2))
(test (let ((ctr 0)) (call-with-exit (lambda (top-exit) (set! ctr (+ ctr 1)) (call-with-exit top-exit) (set! ctr (+ ctr 16)))) ctr) 1)

(test (let () (+ 5 (call-with-exit (lambda (return) (return 1 2 3) 4)))) 11)
(test (+ 5 (call-with-exit (lambda (return) (return 1)))) 6)
(test (+ 5 (call-with-exit (lambda (return) (return)))) 'error)

(test (let ((cur '()))
	(define (step pos)
	  (dynamic-wind
	      (lambda ()
		(set! cur (cons pos cur)))
	      (lambda ()
		(set! cur (cons (+ pos 1) cur))
		(if (< pos 40)
		    (step (+ pos 10)))
		(set! cur (cons (+ pos 2) cur))
		cur)
	      (lambda ()
		(set! cur (cons (+ pos 3) cur)))))
	(reverse (step 0)))
      '(0 1 10 11 20 21 30 31 40 41 42 43 32 33 22 23 12 13 2))


(test (let ((cur '()))
	(define (step pos)
	  (dynamic-wind
	      (lambda ()
		(set! cur (cons pos cur)))
	      (lambda ()
		(set! cur (cons (+ pos 1) cur))
		(if (< pos 40)
		    (step (+ pos 10))
		    (error 'all-done))
		(set! cur (cons (+ pos 2) cur))
		cur)
	      (lambda ()
		(set! cur (cons (+ pos 3) cur)))))
	(catch 'all-done
	       (lambda ()
		 (reverse (step 0)))
	       (lambda args (reverse cur))))
      '(0 1 10 11 20 21 30 31 40 41 43 33 23 13 3))

(test (let ((cur '()))
	(define (step pos ret)
	  (dynamic-wind
	      (lambda ()
		(set! cur (cons pos cur)))
	      (lambda ()
		(set! cur (cons (+ pos 1) cur))
		(if (< pos 40)
		    (step (+ pos 10) ret)
		    (ret (reverse cur)))
		(set! cur (cons (+ pos 2) cur))
		cur)
	      (lambda ()
		(set! cur (cons (+ pos 3) cur)))))
	(list (call-with-exit
	       (lambda (ret)
		 (step 0 ret)))
	      (reverse cur)))
      '((0 1 10 11 20 21 30 31 40 41) (0 1 10 11 20 21 30 31 40 41 43 33 23 13 3)))

(test (let ()
	(catch #t
	       (lambda ()
		 (eval-string "(error 'hi \"hi\")"))
	       (lambda args
		 'error)))
      'error)
(test (let ()
	(catch #t
	       (lambda ()
		 (eval-string "(+ 1 #\\a)"))
	       (lambda args
		 'oops)))
      'oops)

(test (let ()
	(call-with-exit
	 (lambda (return)
	   (eval-string "(return 3)"))))
      3)
(test (let ()
	(call-with-exit
	 (lambda (return)
	   (eval-string "(abs (+ 1 (if #t (return 3))))"))))
      3)

(test (let ((val (catch #t
			(lambda ()
			  (eval-string "(catch 'a (lambda () (+ 1 __asdf__)) (lambda args 'oops))"))
			(lambda args 'error))))
	val)
      'error)

(test (let ((val (catch #t
			(lambda ()
			  (eval `(catch 'a (lambda () (+ 1 __asdf__)) (lambda args 'oops))))
			(lambda args 'error))))
	val)
      'error)


#|
;; this exits the s7test load
(test (let ()
	(call/cc
	 (lambda (return)
	   (eval-string "(return 3)"))))
      3)
|#

(let ((x 0)
      (y 0)
      (z 0))
  (define (dw1 a c) 
    (dynamic-wind
	(lambda ()
	  (set! x (+ x 1)))
	(lambda ()
	  (set! y (+ y 1))
	  (or (and (>= a c) a)
	      (dw1 (+ a 1) c)))
	(lambda ()
	  (set! z (+ z 1))
	  (set! y (- y 1)))))
  (let ((val (dw1 0 8)))
    (test (list val x y z) (list 8 9 0 9))))

(let ((x 0)
      (y 0)
      (z 0))
  (define (dw1 a c) 
    (catch #t
	   (lambda ()
	     (dynamic-wind
		 (lambda ()
		   (set! x (+ x 1)))
		 (lambda ()
		   (set! y (+ y 1))
		   (or (and (>= a c) a)
		       (dw1 (+ a 1) c)))
		 (lambda ()
		   (set! z (+ z 1))
		   (set! y (= y 1))))) ; an error after the 1st call because we have (= #f 1)
	   (lambda args 'error)))
  (let ((val (dw1 0 4)))
    (test val 'error)))

(let ((x 0)
      (y 0)
      (z 0))
  (define (dw1 a c) 
    (catch #t
	   (lambda ()
	     (dynamic-wind
		 (lambda ()
		   (set! x (+ x 1)))
		 (lambda ()
		   (set! y (= y 1)) ; an error after the 1st call because we have (= #f 1)
		   (or (and (>= a c) a)
		       (dw1 (+ a 1) c)))
		 (lambda ()
		   (set! z (+ z 1))
		   (set! y (= y 1)))))
	   (lambda args 'error)))
  (let ((val (dw1 0 4)))
    (test val 'error)))

(let ((x 0)
      (y 0)
      (z 0))
  (define (dw1 a c) 
    (catch #t
	   (lambda ()
	     (dynamic-wind
		 (lambda ()
		   (set! x (= x 1))) ; an error after the 1st call because we have (= #f 1)
		 (lambda ()
		   (set! y (= y 1)) 
		   (or (and (>= a c) a)
		       (dw1 (+ a 1) c)))
		 (lambda ()
		   (set! z (+ z 1))
		   (set! y (= y 1)))))
	   (lambda args 'error)))
  (let ((val (dw1 0 4)))
    (test val 'error)))

(let ((x 0)
      (y 0)
      (z 0))
  (let ((val (call-with-exit
	      (lambda (r)
		(catch #t
		       (lambda ()
			 (dynamic-wind
			     (lambda ()
			       (set! x (+ x 1)))
			     (lambda ()
			       (set! y (+ y 1))
			       (r y))
			     (lambda ()
			       (set! z (+ z 1)))))
		       (lambda args 'error))))))
    (test (list val z) '(1 1))))

(let ((x 0)
      (y 0)
      (z 0))
  (let ((val (catch #t
		    (lambda ()
		      (dynamic-wind
			  (lambda ()
			    (set! x (+ x 1)))
			  (lambda ()
			    (call-with-exit
			     (lambda (r)
			       (set! y r)
			       x)))
			  (lambda ()
			    (set! z (+ z 1))
			    (y z))))
		    (lambda args 'error))))
    (test val 'error)))




;;; --------------------------------------------------------------------------------
;;; quasiquote
;;; --------------------------------------------------------------------------------

(test `(1 2 3) '(1 2 3))
(test `() '())
(test `(list ,(+ 1 2) 4)  '(list 3 4))
(test `(1 ,@(list 1 2) 4) '(1 1 2 4))
(test `#(1 ,@(list 1 2) 4) '#(1 1 2 4))
(test `(a ,(+ 1 2) ,@(map abs '(4 -5 6)) b) '(a 3 4 5 6 b))
(if (eqv? 2 (sqrt 4))
    (test `#(10 5 ,(sqrt 4) ,@(map sqrt '(16 9)) 8) '#(10 5 2 4 3 8))) ; inexactness foolishness
(test `(a `(b ,(+ 1 2) ,(foo ,(+ 1 3) d) e) f) '(a `(b ,(+ 1 2) ,(foo 4 d) e) f))
(test (let ((name1 'x) (name2 'y)) `(a `(b ,,name1 ,',name2 d) e)) '(a `(b ,x ,'y d) e))
(test `(1 2 ,(* 9 9) 3 4) '(1 2 81 3 4))
(test `(1 ,(+ 1 1) 3) '(1 2 3))                     
(test `(,(+ 1 2)) '(3))
(test `(,'a . ,'b) (cons 'a 'b))
(test `(,@'() . foo) 'foo)
(test `(1 , 2) '(1 2))
(test `(1 , @(list 2 3)) 'error) ; ?? this is an error in Guile and Clisp
(test `(1 ,@ (list 2 3)) '(1 2 3)) ; seems a bit arbitrary
(test `(1 ,@(list)) '(1))
(test `(1 ,@()) '(1))
(test `(1 ,@'()) '(1))
(test `(1 . ,()) '(1))
(test `(1 , #|a comment|# 2) '(1 2))
(test `(1 ,@ #|a comment|# (list 2 3)) '(1 2 3))
(test `(1 , ;a comment
                       2) '(1 2))
(test `(1 #||#,2) '(1 2))
(test `(1 #||#,@(list #||# 2 3 #||#)) '(1 2 3))
(test (eval ``(+ 1 ,@,@'('(2 3)))) '(+ 1 2 3))
(test (eval ``(+ 1   ,@ #||#  ,@   '('(2 3)))) '(+ 1 2 3))
(test `(,1 ,1) '(1 1))
(test `(,1 ,`,1) '(1 1))
(test `(,1 ,`,@(list 1)) '(1 1))
(test `(,1 ,`,`,1) '(1 1))
(test `(,1 ,`,@'(1)) '(1 1))
(test `(,1 ,`,@`(1)) '(1 1))
(test `(,1 ,`,@`(,1)) '(1 1))
(test `(,1 ,@`,@(list (list 1))) '(1 1))
(test (eval ``(,,1 ,@,@(list (quote (list 1))))) '(1 1))
(test (eval ``(,,1 ,@,@(list `(list 1)))) '(1 1))
(test (eval (eval ```(,,,1 ,@,@,@(list '(list '(list 1)))))) '(1 1))
(test (+ 1 (eval (eval ```,@,,@(list ''(list 2 3))))) 6)
(test (+ 1 (eval (eval (eval ````,@,,,@(list '''(list 2 3)))))) 6)
(test (apply + `(1 ,@`(2 ,@(list 3)))) 6)
(test (eval `(- ,@()',1)) -1)
(test (eval `(,- ,@()'1)) -1)
(test (eval (eval ``(- ,@,@'(,1())))) -1)
(test (eval (eval ``(,@,@'(- ,1())))) -1)
(test (eval (eval ``(,- ,@,@'(1())))) -1)
(test (eval (eval ``(,- ,@'(,@()1)))) -1)
(test (eval (eval ``(- ,@,@',().(1)))) -1)
(test (quasiquote quote) 'quote)
(test (eval (list quasiquote (list values #t))) (list values #t))

;; from gauche
(let ((quasi0 99)
      (quasi1 101)
      (quasi2 '(a b))
      (quasi3 '(c d)))
  (test `,quasi0 99)
  (test `,quasi1 101)
  (test `(,(cons 1 2)) '((1 . 2)))
  (test `(,(cons 1 2) 3) '((1 . 2) 3))
  (test `(,quasi0 3) '(99 3))
  (test `(3 ,quasi0) '(3 99))
  (test `(,(+ quasi0 1) 3) '(100 3))
  (test `(3 ,(+ quasi0 1)) '(3 100))
  (test `(,quasi1 3) '(101 3))
  (test `(3 ,quasi1) '(3 101))
  (test `(,(+ quasi1 1) 3) '(102 3))
  (test `(3 ,(+ quasi1 1)) '(3 102))
  (test `(1 ,@(list 2 3) 4) '(1 2 3 4))
  (test `(1 2 ,@(list 3 4)) '(1 2 3 4))
  (test `(,@quasi2 ,@quasi3) '(a b c d))
  (test `(1 2 . ,(list 3 4)) '(1 2 3 4))
  (test `(,@quasi2 . ,quasi3) '(a b c d))
  (test `#(,(cons 1 2) 3) '#((1 . 2) 3))
;  (test `#(,quasi0 3) '#(99 3))
;  (test `#(,(+ quasi0 1) 3) '#(100 3))
;  (test `#(3 ,quasi1) '#(3 101))
;  (test `#(3 ,(+ quasi1 1)) '#(3 102))
  (test `#(1 ,@(list 2 3) 4) '#(1 2 3 4))
  (test `#(1 2 ,@(list 3 4)) '#(1 2 3 4))
;  (test `#(,@quasi2 ,@quasi3) '#(a b c d))
;  (test `#(,@quasi2 ,quasi3) '#(a b (c d)))
;  (test `#(,quasi2  ,@quasi3) '#((a b) c d))

;;; the vector quasiquote comma-eval takes place in the global environment so
;;;   (let ((x 0)) (let ((y `#(,(begin (define x 32) x)))) (list x y))) -> '(0 #(32)) and defines x=32 in the top level

  (test `#() '#())
  (test `#(,@(list)) '#())
  (test `(,@(list 1 2) ,@(list 1 2)) '(1 2 1 2))
  (test `(,@(list 1 2) a ,@(list 1 2)) '(1 2 a 1 2))
  (test `(a ,@(list 1 2) ,@(list 1 2)) '(a 1 2 1 2))
  (test `(,@(list 1 2) ,@(list 1 2) a) '(1 2 1 2 a))
  (test `(,@(list 1 2) ,@(list 1 2) a b) '(1 2 1 2 a b))
  (test `(,@(list 1 2) ,@(list 1 2) . a) '(1 2 1 2 . a))
  (test `(,@(list 1 2) ,@(list 1 2) . ,(cons 1 2)) '(1 2 1 2 1 . 2))
  (test `(,@(list 1 2) ,@(list 1 2) . ,quasi2) '(1 2 1 2 a b))
  (test `(,@(list 1 2) ,@(list 1 2) a . ,(cons 1 2)) '(1 2 1 2 a 1 . 2))
  (test `(,@(list 1 2) ,@(list 1 2) a . ,quasi3) '(1 2 1 2 a c d))
  (test `#(,@(list 1 2) ,@(list 1 2)) '#(1 2 1 2))
  (test `#(,@(list 1 2) a ,@(list 1 2)) '#(1 2 a 1 2))
  (test `#(a ,@(list 1 2) ,@(list 1 2)) '#(a 1 2 1 2))
  (test `#(,@(list 1 2) ,@(list 1 2) a) '#(1 2 1 2 a))
  (test `#(,@(list 1 2) ,@(list 1 2) a b) '#(1 2 1 2 a b))
;  (test `(1 `(1 ,2 ,,(+ 1 2)) 1) '(1 `(1 ,2 ,3) 1))
;  (test `(1 `(1 ,,quasi0 ,,quasi1) 1) '(1 `(1 ,99 ,101) 1))
  (test `(1 `(1 ,@2 ,@,(list 1 2))) '(1 `(1 ,@2 ,@(1 2))))
  (test `(1 `(1 ,@,quasi2 ,@,quasi3)) '(1 `(1 ,@(a b) ,@(c d))))
  (test `(1 `(1 ,(,@quasi2 x) ,(y ,@quasi3))) '(1 `(1 ,(a b x) ,(y c d))))
;  (test `#(1 `(1 ,2 ,,(+ 1 2)) 1) '#(1 `(1 ,2 ,3) 1))
;  (test `#(1 `(1 ,,quasi0 ,,quasi1) 1) '#(1 `(1 ,99 ,101) 1))
  (test `#(1 `(1 ,@2 ,@,(list 1 2))) '#(1 `(1 ,@2 ,@(1 2))))
;  (test `#(1 `(1 ,@,quasi2 ,@,quasi3)) '#(1 `(1 ,@(a b) ,@(c d))))
;  (test `#(1 `(1 ,(,@quasi2 x) ,(y ,@quasi3))) '#(1 `(1 ,(a b x) ,(y c d))))
;  (test `(1 `#(1 ,(,@quasi2 x) ,(y ,@quasi3))) '(1 `#(1 ,(a b x) ,(y c d))))
  )

(test `#2d((1 ,(* 3 2)) (,@(list 2) 3)) #2D((1 6) (2 3)))
(test `#3d() #3D())
(test `#3D((,(list 1 2) (,(+ 1 2) 4)) (,@(list (list 5 6)) (7 8))) #3D(((1 2) (3 4)) ((5 6) (7 8))))
(test (eval-string "`#2d(1 2)") 'error)
(test (eval-string "`#2d((1) 2)") 'error)
(test (eval-string "`#2d((1 2) (3 4) (5 6 7))") 'error)
(test `#2d((1 2)) #2D((1 2)))

(let ((x 3)
      (y '(a b c)))
  (test `(1 . ,2) '(1 . 2))
  (test `(1 2 . ,3) '(1 2 . 3))
  (test `(1 x . ,3) '(1 x . 3))
  (test `(1 x . ,x) '(1 x . 3))
  (test `(1 . ,(list 2 3)) '(1 2 3))
  (test `(1 ,@(list 2 3)) '(1 2 3))
;;;  (test `(1 . ,@('(2 3))) '(1 2 3))
  (test `(1 ,(list 2 3)) '(1 (2 3)))
  (test `(1 . (list 2 3)) '(1 list 2 3))
  (test `(x . ,x) '(x . 3))
  (test `(y . ,y) '(y a b c))
  (test `(,x ,@y ,x) '(3 a b c 3))
  (test `(,x ,@y . ,x) '(3 a b c . 3))
  (test `(,y ,@y) '((a b c) a b c))
  (test `(,@y . ,y) '(a b c a b c))

  (test (object->string `(,y . ,y)) "(#1=(a b c) . #1#)")
  (test (object->string `(y ,y ,@y ,y . y)) "(y #1=(a b c) a b c #1# . y)")

  (test (eval ``(1 . ,,2)) '(1 . 2))
  (test (eval ``(y . ,,x)) '(y . 3))
  (test (eval ``(,y . ,x)) '((a b c) . 3))
  (test (eval ``(,@y . x)) '(a b c . x))
  (test (eval ``(,x . ,y)) '(3 a b c))
  (test (eval ``(,,x . y)) '(3 . y))
  (test (eval ``(,,x ,@y)) '(3 a b c))  ;; in clisp `(,y . ,@(y)) -> *** - READ: the syntax `( ... . ,@form) is invalid
  )
(test (let ((.' '(1 2))) `(,@.')) '(1 2))

(test (let ((hi (lambda (a) `(+ 1 ,a))))
	(hi 2))
      '(+ 1 2))

(test (let ((hi (lambda (a) `(+ 1 ,@a))))
	(hi (list 2 3)))
      '(+ 1 2 3))

(test (let ((hi (lambda (a) `(let ((b ,a)) ,(+ 1 a)))))
	(hi 3))
      '(let ((b 3)) 4))

(test (let ((x '(a b c)))
	`(x ,x ,@x foo ,(cadr x) bar ,(cdr x) baz ,@(cdr x)))
      '(x (a b c) a b c foo b bar (b c) baz b c))

(test (let ((x '(a b c)))
	`(,(car `(,x))))
      '((a b c)))

(test (let ((x '(a b c)))
	`(,@(car `(,x))))
      '(a b c))

(test (let ((x '(a b c)))
	`(,(car `(,@x))))
      '(a))

(test (let ((x '(a b c)))
	``,,x)
      '(a b c))

(test (let ((x '(a b c)))
	`,(car `,x))
      'a)

(test (let ((x '(2 3)))
	`(1 ,@x 4))
      '(1 2 3 4))

(test `#(1 ,(/ 12 2)) '#(1 6))
(test ((lambda () `#(1 ,(/ 12 2)))) '#(1 6))

(test (let ((x '(2 3)))
	`(1 ,@(map (lambda (a) (+ a 1)) x)))
      '(1 3 4))

;;; these are from the scheme bboard
(test (let ((x '(1 2 3))) `(0 . ,x)) '(0 1 2 3))
(test (let ((x '(1 2 3))) `(0 ,x)) '(0 (1 2 3)))
;(test (let ((x '(1 2 3))) `#(0 ,x)) '#(0 (1 2 3)))
;(test (let ((x '(1 2 3))) `#(0 . ,x)) '#(0 1 2 3))

(test `#(,most-positive-fixnum 2) #(9223372036854775807 2))

(test (let () (define-macro (tryqv . lst) `(map abs ',lst)) (tryqv 1 2 3 -4 5)) '(1 2 3 4 5))
(test (let () (define-macro (tryqv . lst) `(map abs '(,@lst))) (tryqv 1 2 3 -4 5)) '(1 2 3 4 5))
(test (let () (define-macro (tryqv . lst) `(map abs (vector ,@lst))) (tryqv 1 2 3 -4 5)) '(1 2 3 4 5))

(for-each
 (lambda (str)
   (let ((val (catch #t
		     (lambda () (eval-string str))
		     (lambda args 'error))))
     (if (not (eqv? val -1))
	 (format #t "~S = ~S?~%" str val))))
 (list "( '(.1 -1)1)" "( - '`-00 1)" "( - .(,`1/1))" "( - .(`1) )" "( -(/ .(1)))" "( / 01 -1 )" "(' '` -1(/ ' 1))" "(' (-01 )0)" 
       "(' `'`` -1 1 01)" "(''-1 .(1))" "('(, -1 )0)" "('(,-1)'000)" "('(,-1)00)" "('(-1  -.0)0)" "('(-1 '1`)0)" "('(-1 .-/-)0)" "('(-1()),0)"
       "('(-1) 0)" "('(10. -1 )1)" "(-  '`1)" "(-  `1 1 1)" "(- '  1)" "(- ' 1)" "(- '1 .())" "(- '` ,``1)" "(- '` 1)" "(- '`, `1)" "(- '`,`1)" 
       "(- '``1)" "(- (''1 1))" "(- (- ' 1 0))" "(- (-(- `1)))" "(- (`  (1)0))" "(- ,`, `1)" "(- ,`1 . ,())" "(- . ( 0 1))" "(- . ( `001))" 
       "(- .(', 01))" "(- .('`,1))" "(- .('``1))" "(- .(,,1))" "(- .(01))" "(- .(1))" "(- .(`,1))" "(- .(`,`1))" "(- .(`1))" "(- ` ,'1)" 
       "(- ` -0 '1)" "(- ` 1 )" "(- ` `1)" "(- `, 1)" "(- `,1)" "(- `,`,1)" "(- `,`1)" "(- ``,,1)" "(- ``,1)" "(- ``1 )" "(- ```,1)" 
       "(- ```1)" "(-(  / 1))" "(-( -  -1 ))" "(-( `,- 1 0))" "(-(' (1)0))" "(-('(,1)00))" "(-(- '`1) 0)" "(-(- -1))" "(-(/(/(/ 1))))" 
       "(-(`'1 '1))" "(-(`(,1 )'0))" "(-(`,/ ,1))" "(/ '-1 '1 )" "(/ ,, `,-1)" "(/ ,11 -11)" "(/ 01 (- '1))" "(/ `1 '`-1)" 
       "(/(- '1)  )" "(/(- '1)1)" "(/(- 001)1)" "(/(- 1),,1)" "(/(/ -1))" "(` ,- 1)" "(` `,(-1)0)" "(`' , -1 1)" "(/(- -001(+)))"
       "(`' -1 '1/1)" "(`','-1 ,1)" "(`(,-1)-00)" "(`(-0 -1)1)" "(`(-1 -.')0)" "(`(-1 1')0)" "(`(` -1)'`0)" "(`, - 1)" "(`,- '1)" "(`,- .(1))" 
       "(`,- 1 )" "(`,- `,1)" "(`,- `1)" "(`,/ . (-1))" "(``,,- `01)" "('''-1 '1 '1)" "(/ `-1 1)" "(/ .( -1))" "(-(+(+)0)1)" "(/ ,`,`-1/1)" 
       "(-(*).())" "(*(- +1))" "(-(`,*))" "(-(+)'1)" "(+(-(*)))" "(-(+(*)))" "(-(+)(*))" "(-(/(*)))" "(*(-(*)))" "(-(*(*)))" "(/(-(*)))" "(-(+(*)))"
       "(/ .(-1))" "(-(*))" "(- 000(*))" "(-(*(+ 1)))" "(- .((*)))" "(- +0/10(*))" "(-(`,/ .(1)))" "(+ .(' -01))" "(-(''1 01))" "(- -1/1 +0)"
       "(- `,'0 `01)" "( - `,(+)'1)" "(+(- . (`1)))" "(* '`,``-1)" "(-(+ -0)1)" "(+ +0(-(*)))" "(+(- '+1 ))" "(+ '-01(+))" "(`, -(+)1)" 
       "(`,+ 0 -1)" "(-(/(/(*))))" "(`,* .( -1))" "(-(*(*(*))))" "(`,@(list +)-1)" "(* (- 1) )" "(`, - (* ))" "(/(- (* 1)))"
       "(- -0/1(*))" "(`(,-1)0)" "(/(-(*).()))" "(* ````-1)" "(-(+(*)0))" "(-(* `,(*)))" "(- +1(*)1)" "(- (` ,* ))" "(/(-(+ )1))" "(`,* -1(*))" 
       "(` ,- .(1))" "(+(`,-( *)))" "( /(`,- 1))" "(`(1 -1)1)" "(*( -(/(*))))" "(- -1(-(+)))" "(* ``,,-1)" "(-(+(+))1)" "( +(*(-(*))))"
       "(-(+)`0(*))" "(-(+(+(*))))" "(-(+ .(01)))" "(/(*(-(* ))))" "(/ (-(* 1)))" "( /(-(/(*))))" "(+ -1 0/1)" "(/(-( +(*))))" "(*( -(`,*)))"
       "(* 1(/ 1)-1)" "(+ 0(- ',01))" "(+(-(-(+ -1))))" "(- 0(/(+(* ))))" "(-(+)( *)0)"
       ))

#|
(let ((chars (vector #\/ #\. #\0 #\1 #\- #\, #\( #\) #\' #\@ #\` #\space))
      (size 14))
  (let ((str (make-string size))
	(clen (length chars)))
    (set! (str 0) #\()
    (do ((i 0 (+ i 1)))
	((= i 10000000))
      (let ((parens 1))
	(do ((k 1 (+ k 1)))
	    ((= k size))
	  (set! (str k) (chars (random clen)))
	  (if (char=? (str k) #\()
	      (set! parens (+ parens 1))
	      (if (char=? (str k) #\))
		  (begin
		    (set! parens (- parens 1))
		    (if (negative? parens)
			(begin
			  (set! (str k) #\space)
			  (set! parens 0)))))))
	(let ((str1 str)
	      (happy (char=? (str (- size 1)) #\))))
	  (if (> parens 0)
	      (begin
		(set! str1 (make-string (+ size parens) #\)))
		(set! happy #t)
		(do ((k 0 (+ k 1)))
		    ((= k size))
		  (set! (str1 k) (str k)))))
	  (set! (-s7-symbol-table-locked?) #t)
	  (if (and happy
		   (not (char=? (str1 1) #\))))
	      (catch #t 
		     (lambda ()
		       (let ((num (eval-string str1)))
			 (if (and (number? num)
				  (eqv? num -1))
			     (format #t "~S ~%" str1))))
		     (lambda args
		       'error)))
	  (set! (-s7-symbol-table-locked?) #f))))))
#|

(test (= 1 '+1 `+1 '`1 `01 ``1) #t)
(test (''- 1) '-)
(test (`'- 1) '-)
(test (``- 1) '-)
(test ('`- 1) '-)
(test (''1 '1) 1)
(test ((quote (quote 1)) 1) 1) ; (quote (quote 1)) -> ''1 = (list 'quote 1), so ((list 'quote 1) 1) is 1!
(test (list 'quote 1) ''1) ; guile says #t
(test (list-ref ''1 1) 1)  ; same
(test (''1 ```1) 1)
(test (cond `'1) 1)
(test ```1 1)
(test ('''1 1 1) 1)
(test (`',1 1) 1)
(test (- `,-1) 1)
;;; some weirder cases...
(test (begin . `''1) ''1)
(test (`,@''1) 1)
(test (`,@ `'1) 1)
(test (`,@''.'.) '.'.)
(test #(`,1) #(1))
(test `#(,@'(1)) #(1))
(test `#(,`,@''1) #(quote 1))
(test `#(,@'(1 2 3)) #(1 2 3))
(test `#(,`,@'(1 2 3)) #(1 2 3)) ; but #(`,@'(1 2 3)) -> #(({apply} {values} '(1 2 3)))
(test (apply . `''1) 'error) ; '(quote quote 1)) ; (apply {list} 'quote ({list} 'quote 1)) -> ;quote: too many arguments '1
(test (apply - 1( )) -1)               ; (apply - 1 ())
(num-test (apply - 1.()) -1.0)
(num-test (apply - .1()) -0.1)
(num-test (apply - .1 .(())) -0.1)
(num-test (apply - .1 .('(1))) -0.9)
(test (apply - -1()) 1)                ; (apply - -1 ())
(test (apply . `(())) '())             ; (apply {list} ())
(test (apply . ''(1)) 1)               ; (apply quote '(1))
(test (apply . '`(1)) 1)               ; (apply quote ({list} 1))
(test (apply . `(,())) '())            ; (apply {list} ())
(test (apply . `('())) ''())           ; (apply {list} ({list} 'quote ()))
(test (apply . `(`())) '())            ; (apply {list} ())
(test (apply - `,1()) -1)              ; (apply - 1 ())
(test (apply - ``1()) -1)              ; (apply - 1 ())
(test (apply ''1 1()) 1)               ; (apply ''1 1 ())
(test (apply .(- 1())) -1)             ; (apply - 1 ())
(test (apply - .(1())) -1)             ; (apply - 1 ())
(test (apply . `(1())) '(1))           ; (apply {list} 1 ())
(test (apply . ''(())) '())
(test (apply . `((()))) '(()))

;; make sure the macro funcs really are constants
(test (defined? '{list}) #t)
(test (let () (set! {list} 2)) 'error)
(test (let (({list} 2)) {list}) 'error)
(test (defined? '{values}) #t)
(test (let () (set! {values} 2)) 'error)
(test (let (({values} 2)) {values}) 'error)
(test (defined? '{apply}) #t)
(test (let () (set! {apply} 2)) 'error)
(test (let (({apply} 2)) {apply}) 'error)
(test (defined? '{append}) #t)
(test (let () (set! {append} 2)) 'error)
(test (let (({append} 2)) {append}) 'error)
(test (defined? '{multivector}) #t)
(test (let () (set! {multivector} 2)) 'error)
(test (let (({multivector} 2)) {multivector}) 'error)

(test (+ 1 ((`#(,(lambda () 0) ,(lambda () 2) ,(lambda () 4)) 1))) 3) ; this calls vector each time, just like using vector directly
(test (+ 1 ((`(,(lambda () 0)  ,(lambda () 2) ,(lambda () 4)) 1))) 3)

(test (object->string (list 'quote 1 2)) "(quote 1 2)")
(test (object->string (list 'quote 'quote 1)) "(quote quote 1)")
(test (object->string (list 'quote 1 2 3)) "(quote 1 2 3)")
;;; but (object->string (list 'quote 1)) -> "'1" -- is this correct?
;;; (equal? (quote 1) '1) -> #t and (equal? (list 'quote 1) ''1) -> #t
;;; see comment s7.c in list_to_c_string -- we're following Clisp here
(test (object->string (list 'quote 1)) "'1")
(test (object->string (cons 'quote 1)) "(quote . 1)")
(test (object->string (list 'quote)) "(quote)")
(let ((lst (list 'quote 1)))
  (set! (cdr (cdr lst)) lst)
  (test (object->string lst) "#1=(quote 1 . #1#)"))
(let ((lst (list 'quote)))
  (set! (cdr lst) lst)
  (test (object->string lst) "#1=(quote . #1#)"))
(test (object->string quasiquote) "quasiquote")


;; from Guile mailing list
(test (let ((v '(\())))
	(and (pair? v)
	     (symbol? (car v)) ; \
	     (null? (cadr v))))
      #t)
;; this gets a read-error in Snd because the listener gets confused

#|
unquote outside qq:
(',- 1)
(',1 1 )
(',,= 1) -> (unquote =)
(',@1 1) -> 1
#(,1) -> #((unquote 1)) i.e. vector has 1 element the list '(unquote 1)
#(,@1) -> #((unquote ({apply} {values} 1)))
#(,,,1) -> #((unquote (unquote (unquote 1))))
is this a bug?
#(`'`1) -> #(({list} 'quote 1))

why are these different (read-time `#() ? )
:`#(,@(list 1 2 3))
#(1 2 3)
:(quasiquote #(,@(list 1 2 3)))
#((unquote ({apply} {values} (list 1 2 3))))
|#

(test (quasiquote) 'error)
(test (quasiquote 1 2 3) 'error)
(let ((d 1)) (test (quasiquote (a b c ,d)) '(a b c 1)))
(test (let ((a 2)) (quasiquote (a ,a))) (let ((a 2)) `(a ,a)))
(test (quasiquote 4) 4)
(test (quasiquote (list (unquote (+ 1 2)) 4)) '(list 3 4))
(test (quasiquote (1 2 3)) '(1 2 3))
(test (quasiquote ()) '())
(test (quasiquote (list ,(+ 1 2) 4))  '(list 3 4))
(test (quasiquote (1 ,@(list 1 2) 4)) '(1 1 2 4))
(test (quasiquote (a ,(+ 1 2) ,@(map abs '(4 -5 6)) b)) '(a 3 4 5 6 b))
(test (quasiquote (1 2 ,(* 9 9) 3 4)) '(1 2 81 3 4))
(test (quasiquote (1 ,(+ 1 1) 3)) '(1 2 3))                     
(test (quasiquote (,(+ 1 2))) '(3))
(test (quasiquote (,@'() . foo)) 'foo)
(test (quasiquote (1 , 2)) '(1 2))
(test (quasiquote (,1 ,1)) '(1 1))
(test (quasiquote (,1 ,(quasiquote ,1))) '(1 1))
(test (quasiquote (,1 ,(quasiquote ,@(list 1)))) '(1 1))
(test (quasiquote (,1 ,(quasiquote ,(quasiquote ,1)))) '(1 1))
(test (quasiquote (,1 ,(quasiquote ,@'(1)))) '(1 1))
(test (quasiquote (,1 ,(quasiquote ,@(quasiquote (1))))) '(1 1))
(test (quasiquote (,1 ,(quasiquote ,@(quasiquote (,1))))) '(1 1))
(test (quasiquote (,1 ,@(quasiquote ,@(list (list 1))))) '(1 1))
(test `(+ ,(apply values '(1 2))) '(+ 1 2))
(test `(apply + (unquote '(1 2))) '(apply + (1 2)))
(test (eval (list (list quasiquote +) -1)) -1)

(test (apply quasiquote '((1 2 3))) '(1 2 3))
(test (quasiquote (',,= 1)) 'error)
(test (quasiquote (',,@(1 2) 1)) 'error)
(test (quasiquote 1.1 . -0) 'error)

(test `(1 ,@2) 'error)
(test `(1 ,@(2 . 3)) 'error)
(test `(1 ,@(2 3)) 'error)
(test `(1 , @ (list 2 3)) 'error) ; unbound @ ! (guile also)

(test (call-with-exit quasiquote) 'error)
(test (call-with-output-string quasiquote) 'error)
(test (map quasiquote '(1 2 3))  'error)
(test (for-each quasiquote '(1 2 3))  'error)
(test (sort! '(1 2 3) quasiquote) 'error)
(test (quasiquote . 1) 'error)
(test (let ((x 3)) (quasiquote . x)) 'error)
(num-test `,#e.1 1/10)
(num-test `,,,-1 -1)
(num-test `,``,1 1)
(test (equal? ` 1 ' 1) #t)
(test (+ ` 1 `  2) `   3)
(test ` ( + ,(- 3 2) 2) '(+ 1 2))
(test (quasiquote #(1)) `#(1))

(test `(+ ,@(map sqrt '(1 4 9)) 2) '(+ 1 2 3 2))
(test (let ((sqrt (lambda (a) (* a a)))) `(+ ,@(map sqrt '(1 4 9)) 2)) '(+ 1 16 81 2))
(test `(+ ,(sqrt 9) 4) '(+ 3 4))
(test (let ((sqrt (lambda (a) (* a a)))) `(+ ,(sqrt 9) 4)) '(+ 81 4))
(test `(+ ,(let ((sqrt (lambda (a) (* a a)))) (sqrt 9)) 4) '(+ 81 4))
(test `(+ (let ((sqrt (lambda (a) (* a a)))) ,(sqrt 9)) 4) '(+ (let ((sqrt (lambda (a) (* a a)))) 3) 4))
(test `(+ ,(apply values (map sqrt '(1 4 9))) 2) '(+ 1 2 3 2))
(test (let ((sqrt (lambda (a) (* a a)))) `(+ ,(apply values (map sqrt '(1 4 9))) 2)) '(+ 1 16 81 2))
(test (let ((sqrt (lambda (a) (* a a)))) `(+ (unquote (apply values (map sqrt '(1 4 9)))) 2)) '(+ 1 16 81 2))



;;; --------------------------------------------------------------------------------
;;; -------- s7 specific stuff --------
;;; --------------------------------------------------------------------------------

;;; keyword?
;;; make-keyword
;;; keyword->symbol
;;; symbol->keyword

(for-each
 (lambda (arg)
   (test (keyword? arg) #f))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(test (cond ((cond (())) ':)) ':)
(test (keyword? :#t) #t)
(test (eq? #t :#t) #f)
;(test (keyword? '#:t) #f)  ; these 2 are fooled by the Guile-related #: business (which is still supported)
;(test (keyword? '#:#t) #f)
(test (keyword? :-1) #t)
(test (keyword? (symbol ":#(1 #\\a (3))")) #t)
(test (keyword? (make-keyword (object->string #(1 #\a (3)) #f))) #t)
(test (keyword? begin) #f)
(test (keyword? if) #f)

(let ((kw (make-keyword "hiho")))
  (test (keyword? kw) #t)
  (test (keyword->symbol kw) 'hiho)
  (test (symbol->keyword 'hiho) kw)
  (test (keyword->symbol (symbol->keyword 'key)) 'key)
  (test (symbol->keyword (keyword->symbol (make-keyword "hi"))) :hi)
  (test (keyword? :a-key) #t)
  (test (keyword? ':a-key) #t)
  (test (keyword? ':a-key:) #t)
  (test (keyword? 'a-key:) #t)
  (test (symbol? (keyword->symbol :hi)) #t)
  (test (keyword? (keyword->symbol :hi)) #f)
  (test (symbol? (symbol->keyword 'hi)) #t)
  (test (equal? kw :hiho) #t)
  (test ((lambda (arg) (keyword? arg)) :hiho) #t)
  (test ((lambda (arg) (keyword? arg)) 'hiho) #f)
  (test ((lambda (arg) (keyword? arg)) kw) #t)
  (test ((lambda (arg) (keyword? arg)) (symbol->keyword 'hiho)) #t)
  (test (make-keyword "3") :3)
  (test (keyword? :3) #t)
  (test (keyword? ':3) #t)
  (test (equal? :3 3) #f)
  (test (equal? (keyword->symbol :3) 3) #f)
  (test (equal? (symbol->value (keyword->symbol :3)) 3) #f) ; 3 as a symbol has value #<undefined>

#|
  (let ()
    (apply define (symbol "3") '(32))
    (test (symbol->value (symbol "3")) 32) ; hmmm
    (apply define (list (symbol "3") (lambda () 32)))
    (test (symbol->value (symbol "3")) 32))
|#

  (test (keyword? '3) #f)
  (test (keyword? ':) #f)
  (test (keyword? '::) #t)
  (test (keyword? ::) #t)
  (test (keyword? ::a) #t)
  (test (eq? ::a ::a) #t)
  (test (eq? (keyword->symbol ::a) :a) #t)
  (test (eq? (symbol->keyword :a) ::a) #t)
  (test (symbol->string ::a) "::a")
  (test ((lambda* (:a 32) ::a) 0) 'error) ; :a is a constant
  (test (eq? :::a::: :::a:::) #t)
  (test (keyword? a::) #t)
  (test (keyword->symbol ::) ':)
  (test (keyword? :optional) #t)
  (test (symbol->string (keyword->symbol hi:)) "hi")
  (test (symbol->string (keyword->symbol :hi)) "hi")
  (test (keyword? (make-keyword (string #\x (integer->char 128) #\x))) #t)
  (test (keyword? (make-keyword (string #\x (integer->char 200) #\x))) #t)
  (test (keyword? (make-keyword (string #\x (integer->char 255) #\x))) #t)
  (test (make-keyword ":") ::)
  (test (make-keyword (string #\")) (symbol ":\""))
  (test (keyword? (make-keyword (string #\"))) #t)
  (test (keyword->symbol (make-keyword (string #\"))) (symbol "\""))
  )

(test (symbol->keyword 'begin) :begin)
(test (symbol->keyword 'quote) :quote)
(test (symbol->keyword if) 'error)
(test (symbol->keyword quote) 'error)

(test (let ((:hi 3)) :hi) 'error)
(test (set! :hi 2) 'error)
(test (define :hi 3) 'error)

(let ((strlen 8))
  (let ((str (make-string strlen)))
    (do ((i 0 (+ i 1)))
	((= i 10))
      (do ((k 0 (+ k 1)))
	  ((= k strlen))
	(set! (str k) (integer->char (+ 1 (random 255)))))
      (let ((key (make-keyword str)))
	(let ((newstr (symbol->string (keyword->symbol key))))
	  (if (not (string=? newstr str))
	      (format #t ";make-keyword -> string: ~S -> ~A -> ~S~%" str key newstr)))))))

(let ()
  (define* (hi a b) (+ a b))
  (test (hi 1 2) 3)
  (test (hi :b 3 :a 1) 4)
  (test (hi b: 3 a: 1) 4))

(for-each
 (lambda (arg)
   (test (make-keyword arg) 'error))
 (list -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (keyword->symbol arg) 'error))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (symbol->keyword arg) 'error))
 (list "hi" -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(test (keyword?) 'error)
(test (keyword? 1 2) 'error)
(test (make-keyword) 'error)
(test (make-keyword 'hi 'ho) 'error)
(test (keyword->symbol) 'error)
(test (keyword->symbol :hi :ho) 'error)
(test (symbol->keyword) 'error)
(test (symbol->keyword 'hi 'ho) 'error)



;;; gensym
(for-each
 (lambda (arg)
   (test (gensym arg) 'error))
 (list -1 #\a 1 'hi _ht_ '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(test (gensym "hi" "ho") 'error)

(test (symbol? (gensym)) #t)
(test (symbol? (gensym "temp")) #t)
(test (eq? (gensym) (gensym)) #f)
(test (eqv? (gensym) (gensym)) #f)
(test (equal? (gensym) (gensym)) #f)
(test (keyword? (gensym)) #f)
(test (let* ((a (gensym)) (b a)) (eq? a b)) #t)
(test (let* ((a (gensym)) (b a)) (eqv? a b)) #t)

(let ((sym (gensym)))
  (test (eval `(let ((,sym 32)) (+ ,sym 1))) 33))

(let ((sym1 (gensym))
      (sym2 (gensym)))
  (test (eval `(let ((,sym1 32) (,sym2 1)) (+ ,sym1 ,sym2))) 33))

(test (let ((hi (gensym))) (eq? hi (string->symbol (symbol->string hi)))) #t)
(test (let () (define-macro (hi a) (let ((var (gensym ";"))) `(let ((,var ,a)) (+ 1 ,var)))) (hi 1)) 2)
(test (let () (define-macro (hi a) (let ((funny-name (string->symbol (string #\;)))) `(let ((,funny-name ,a)) (+ 1 ,funny-name)))) (hi 1)) 2)
(test (let () (define-macro (hi a) (let ((funny-name (string->symbol "| e t c |"))) `(let ((,funny-name ,a)) (+ 1 ,funny-name)))) (hi 2)) 3)

(let ((funny-name (string->symbol "| e t c |")))
  (define-macro (hi a) 
    `(define* (,a (,funny-name 32)) (+ ,funny-name 1)))
  (hi func)
  (test (func) 33)
  (test (func 1) 2)
  ;(procedure-source func) '(lambda* ((| e t c | 32)) (+ | e t c | 1))
  (test (apply func (list (symbol->keyword funny-name) 2)) 3)
  )

(let ((funny-name (string->symbol "| e t c |")))
  (apply define* `((func (,funny-name 32)) (+ ,funny-name 1)))
  (test (apply func (list (symbol->keyword funny-name) 2)) 3))



(test (provided?) 'error)
(test (or (null? *features*) (pair? *features*)) #t)
(test (provided? 1 2 3) 'error)
(provide 's7test)
(test (provided? 's7test) #t)
(test (provided? 'not-provided!) #f)
(test (provided? 'begin) #f)
(test (provided? if) 'error)
(test (provided? quote) 'error)

(test (provide quote) 'error)
(test (provide 1 2 3) 'error)
(test (provide) 'error)
(test (provide lambda) 'error)

(provide 's7test) ; should be a no-op
(let ((count 0))
  (for-each
   (lambda (p)
     (if (eq? p 's7test)
	 (set! count (+ count 1)))
     (if (not (provided? p))
	 (format #t ";~A is in *features* but not provided? ~A~%" p *features*)))
   *features*)
  (if (not (= count 1))
      (format #t ";*features* has ~D 's7test entries? ~A~%" count *features*)))

(for-each
 (lambda (arg)
   (test (provide arg) 'error))
 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (provided? arg) 'error))
 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (set! *gc-stats* arg) 'error))
 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() '#(()) (list 1 2 3) '(1 . 2)))
(test *gc-stats* #f)

(let ((f (sort! *features* (lambda (a b) (string<? (object->string a #f) (object->string b #f))))))
  (let ((last 'not-in-*features*))
    (for-each
     (lambda (p)
       (if (eq? p last)
	   (format #t ";*features has multiple ~A? ~A~%" p *features*))
       (set! last p))
     f)))

(for-each
 (lambda (arg)
   (test (set! *safety* arg) 'error)
   (test (set! *features* arg) 'error)
   (test (set! *load-path* arg) 'error)
   (test (set! *#readers* arg) 'error)
   )
 (list #\a '#(1 2 3) 3.14 3/4 1.0+1.0i abs 'hi #t #f '#(())))
(test (let ((*features* 123)) *features*) 'error)
(test (let ((*safety* '(1 2 3))) *safety*) 'error)
(test (set! *load-path* (list 1 2 3)) 'error)

(test (integer? *vector-print-length*) #t)
(test (or (null? *#readers*) (pair? *#readers*)) #t)
(test (or (null? *load-path*) (pair? *load-path*)) #t)
(test (vector? *error-info*) #t)

(test (let () (set! *error-info* 2)) 'error)
(test (let ((*error-info* 2)) *error-info*) 'error)

(let ((old-len *vector-print-length*))
  (for-each
   (lambda (arg)
     (test (set! *vector-print-length* arg) 'error))
   (list -1 #\a '#(1 2 3) 3.14 3/4 1.0+1.0i abs 'hi '() #t #f '#(()) (list 1 2 3) '(1 . 2)))
  (set! *vector-print-length* old-len))

(let ((old-hook (hook-functions *unbound-variable-hook*))
      (hook-val #f))
  (set! (hook-functions *unbound-variable-hook*) (list (lambda (sym) (set! hook-val sym) 123)))
  (catch #t
	 (lambda ()
	   (+ 1 one-two-three))
	 (lambda args 'error))
  (test (equal? one-two-three 123) #t)
  (test (equal? hook-val 'one-two-three) #t)
  (set! (hook-functions *unbound-variable-hook*) old-hook))

(for-each
 (lambda (arg)
   (test (set! *unbound-variable-hook* arg) 'error)
   (test (set! *error-hook* arg) 'error))
 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '#(())))

(let ((old-load-hook (hook-functions *load-hook*))
      (val #f))
  (with-output-to-file "load-hook-test.scm"
    (lambda ()
      (format #t "(define (load-hook-test val) (+ val 1))")))
  (set! *load-hook* 
	(lambda (file) 
	  (if (or val
		  (defined? 'load-hook-test))
	      (format #t ";*load-hook*: ~A ~A?~%" val load-hook-test))
	  (set! val file)))
  (load "load-hook-test.scm")
  (if (or (not (string? val))
	  (not (string=? val "load-hook-test.scm")))
      (format #t ";*load-hook-test* file: ~S~%" val))
  (if (not (defined? 'load-hook-test))
      (format #t ";load-hook-test function not defined?~%")
      (if (not (= (load-hook-test 1) 2))
	  (format #t ";load-hook-test: ~A~%" (load-hook-test 1))))
  (set! (hook-functions *load-hook*) old-load-hook))

(let ((old-vlen *vector-print-length*))
  (set! *vector-print-length* 0)
  (test (format #f "~A" #()) "#()")
  (test (format #f "~A" #(1 2 3 4)) "#(...)")
  (set! *vector-print-length* 1)
  (test (format #f "~A" #()) "#()")
  (test (format #f "~A" #(1)) "#(1)")
  (test (format #f "~A" #(1 2 3 4)) "#(1 ...)")
  (set! *vector-print-length* 2)
  (test (format #f "~A" #(1 2 3 4)) "#(1 2 ...)")
  (set! *vector-print-length* old-vlen))

(if with-bignums
    (let ((old-vlen *vector-print-length*))
      (set! *vector-print-length* (bignum "0"))
      (test (format #f "~A" #()) "#()")
      (test (format #f "~A" #(1 2 3 4)) "#(...)")
      (set! *vector-print-length* (bignum "1"))
      (test (format #f "~A" #()) "#()")
      (test (format #f "~A" #(1)) "#(1)")
      (test (format #f "~A" #(1 2 3 4)) "#(1 ...)")
      (set! *vector-print-length* (bignum "2"))
      (test (format #f "~A" #(1 2 3 4)) "#(1 2 ...)")
      (set! *vector-print-length* old-vlen)))


;;; -------- sort!
;;; sort!

(test (sort! '(2 3) <) '(2 3))
(test (sort! '(3 2) <) '(2 3))
(test (sort! '(12 3) <) '(3 12))

(test (sort! '(1 2 3) <) '(1 2 3))
(test (sort! '(1 3 2) <) '(1 2 3))
(test (sort! '(2 1 3) <) '(1 2 3))
(test (sort! '(2 3 1) <) '(1 2 3))
(test (sort! '(3 1 2) <) '(1 2 3))
(test (sort! '(3 2 1) <) '(1 2 3))

(test (sort! '(1 2 3) (lambda (a b) (> a b))) '(3 2 1))
(test (sort! #(2 3) <) #(2 3))
(test (sort! #(12 3) <) #(3 12))

(test (sort! #(1 2 3) <) #(1 2 3))
(test (sort! #(1 3 2) <) #(1 2 3))
(test (sort! #(2 1 3) <) #(1 2 3))
(test (sort! #(2 3 1) <) #(1 2 3))
(test (sort! #(3 1 2) <) #(1 2 3))
(test (sort! #(3 2 1) <) #(1 2 3))

(test (sort! #(1 2 3 4) <) #(1 2 3 4))
(test (sort! #(1 2 4 3) <) #(1 2 3 4))
(test (sort! #(1 3 2 4) <) #(1 2 3 4))
(test (sort! #(1 3 4 2) <) #(1 2 3 4))
(test (sort! #(1 4 2 3) <) #(1 2 3 4))
(test (sort! #(1 4 3 2) <) #(1 2 3 4))
(test (sort! #(2 1 3 4) <) #(1 2 3 4))
(test (sort! #(2 1 4 3) <) #(1 2 3 4))
(test (sort! #(2 3 4 1) <) #(1 2 3 4))
(test (sort! #(2 3 1 4) <) #(1 2 3 4))
(test (sort! #(2 4 3 1) <) #(1 2 3 4))
(test (sort! #(2 4 1 3) <) #(1 2 3 4))
(test (sort! #(3 1 2 4) <) #(1 2 3 4))
(test (sort! #(3 1 4 2) <) #(1 2 3 4))
(test (sort! #(3 2 4 1) <) #(1 2 3 4))
(test (sort! #(3 2 1 4) <) #(1 2 3 4))
(test (sort! #(3 4 1 2) <) #(1 2 3 4))
(test (sort! #(3 4 2 1) <) #(1 2 3 4))
(test (sort! #(4 1 2 3) <) #(1 2 3 4))
(test (sort! #(4 1 3 2) <) #(1 2 3 4))
(test (sort! #(4 2 1 3) <) #(1 2 3 4))
(test (sort! #(4 2 3 1) <) #(1 2 3 4))
(test (sort! #(4 3 2 1) <) #(1 2 3 4))
(test (sort! #(4 3 1 2) <) #(1 2 3 4))

(let ((f (lambda (a b) 
	   (< (car a) (car b)))))
  (test (sort! (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 1 "1") (cons 2 "2") (cons 4 "4") (cons 3 "3")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 1 "1") (cons 3 "3") (cons 2 "2") (cons 4 "4")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 1 "1") (cons 3 "3") (cons 4 "4") (cons 2 "2")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 1 "1") (cons 4 "4") (cons 2 "2") (cons 3 "3")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 1 "1") (cons 4 "4") (cons 3 "3") (cons 2 "2")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 2 "2") (cons 1 "1") (cons 3 "3") (cons 4 "4")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 2 "2") (cons 1 "1") (cons 4 "4") (cons 3 "3")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 2 "2") (cons 3 "3") (cons 4 "4") (cons 1 "1")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 2 "2") (cons 3 "3") (cons 1 "1") (cons 4 "4")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 2 "2") (cons 4 "4") (cons 3 "3") (cons 1 "1")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 2 "2") (cons 4 "4") (cons 1 "1") (cons 3 "3")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 3 "3") (cons 1 "1") (cons 2 "2") (cons 4 "4")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 3 "3") (cons 1 "1") (cons 4 "4") (cons 2 "2")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 3 "3") (cons 2 "2") (cons 4 "4") (cons 1 "1")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 3 "3") (cons 2 "2") (cons 1 "1") (cons 4 "4")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 3 "3") (cons 4 "4") (cons 1 "1") (cons 2 "2")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 3 "3") (cons 4 "4") (cons 2 "2") (cons 1 "1")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 4 "4") (cons 1 "1") (cons 2 "2") (cons 3 "3")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 4 "4") (cons 1 "1") (cons 3 "3") (cons 2 "2")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 4 "4") (cons 2 "2") (cons 1 "1") (cons 3 "3")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 4 "4") (cons 2 "2") (cons 3 "3") (cons 1 "1")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 4 "4") (cons 3 "3") (cons 2 "2") (cons 1 "1")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4")))
  (test (sort! (list (cons 4 "4") (cons 3 "3") (cons 1 "1") (cons 2 "2")) f) (list (cons 1 "1") (cons 2 "2") (cons 3 "3") (cons 4 "4"))))

(test (sort! #(5 1 2 3 4) <) #(1 2 3 4 5))
(test (sort! #(5 1 2 4 3) <) #(1 2 3 4 5))
(test (sort! #(5 1 3 2 4) <) #(1 2 3 4 5))
(test (sort! #(5 1 3 4 2) <) #(1 2 3 4 5))
(test (sort! #(5 1 4 2 3) <) #(1 2 3 4 5))
(test (sort! #(5 1 4 3 2) <) #(1 2 3 4 5))
(test (sort! #(5 2 1 3 4) <) #(1 2 3 4 5))
(test (sort! #(5 2 1 4 3) <) #(1 2 3 4 5))
(test (sort! #(5 2 3 4 1) <) #(1 2 3 4 5))
(test (sort! #(5 2 3 1 4) <) #(1 2 3 4 5))
(test (sort! #(5 2 4 3 1) <) #(1 2 3 4 5))
(test (sort! #(5 2 4 1 3) <) #(1 2 3 4 5))
(test (sort! #(5 3 1 2 4) <) #(1 2 3 4 5))
(test (sort! #(5 3 1 4 2) <) #(1 2 3 4 5))
(test (sort! #(5 3 2 4 1) <) #(1 2 3 4 5))
(test (sort! #(5 3 2 1 4) <) #(1 2 3 4 5))
(test (sort! #(5 3 4 1 2) <) #(1 2 3 4 5))
(test (sort! #(5 3 4 2 1) <) #(1 2 3 4 5))
(test (sort! #(5 4 1 2 3) <) #(1 2 3 4 5))
(test (sort! #(5 4 1 3 2) <) #(1 2 3 4 5))
(test (sort! #(5 4 2 1 3) <) #(1 2 3 4 5))
(test (sort! #(5 4 2 3 1) <) #(1 2 3 4 5))
(test (sort! #(5 4 3 2 1) <) #(1 2 3 4 5))
(test (sort! #(5 4 3 1 2) <) #(1 2 3 4 5))
(test (sort! #(1 5 2 3 4) <) #(1 2 3 4 5))
(test (sort! #(1 5 2 4 3) <) #(1 2 3 4 5))
(test (sort! #(1 5 3 2 4) <) #(1 2 3 4 5))
(test (sort! #(1 5 3 4 2) <) #(1 2 3 4 5))
(test (sort! #(1 5 4 2 3) <) #(1 2 3 4 5))
(test (sort! #(1 5 4 3 2) <) #(1 2 3 4 5))
(test (sort! #(2 5 1 3 4) <) #(1 2 3 4 5))
(test (sort! #(2 5 1 4 3) <) #(1 2 3 4 5))
(test (sort! #(2 5 3 4 1) <) #(1 2 3 4 5))
(test (sort! #(2 5 3 1 4) <) #(1 2 3 4 5))
(test (sort! #(2 5 4 3 1) <) #(1 2 3 4 5))
(test (sort! #(2 5 4 1 3) <) #(1 2 3 4 5))
(test (sort! #(3 5 1 2 4) <) #(1 2 3 4 5))
(test (sort! #(3 5 1 4 2) <) #(1 2 3 4 5))
(test (sort! #(3 5 2 4 1) <) #(1 2 3 4 5))
(test (sort! #(3 5 2 1 4) <) #(1 2 3 4 5))
(test (sort! #(3 5 4 1 2) <) #(1 2 3 4 5))
(test (sort! #(3 5 4 2 1) <) #(1 2 3 4 5))
(test (sort! #(4 5 1 2 3) <) #(1 2 3 4 5))
(test (sort! #(4 5 1 3 2) <) #(1 2 3 4 5))
(test (sort! #(4 5 2 1 3) <) #(1 2 3 4 5))
(test (sort! #(4 5 2 3 1) <) #(1 2 3 4 5))
(test (sort! #(4 5 3 2 1) <) #(1 2 3 4 5))
(test (sort! #(4 5 3 1 2) <) #(1 2 3 4 5))
(test (sort! #(1 2 5 3 4) <) #(1 2 3 4 5))
(test (sort! #(1 2 5 4 3) <) #(1 2 3 4 5))
(test (sort! #(1 3 5 2 4) <) #(1 2 3 4 5))
(test (sort! #(1 3 5 4 2) <) #(1 2 3 4 5))
(test (sort! #(1 4 5 2 3) <) #(1 2 3 4 5))
(test (sort! #(1 4 5 3 2) <) #(1 2 3 4 5))
(test (sort! #(2 1 5 3 4) <) #(1 2 3 4 5))
(test (sort! #(2 1 5 4 3) <) #(1 2 3 4 5))
(test (sort! #(2 3 5 4 1) <) #(1 2 3 4 5))
(test (sort! #(2 3 5 1 4) <) #(1 2 3 4 5))
(test (sort! #(2 4 5 3 1) <) #(1 2 3 4 5))
(test (sort! #(2 4 5 1 3) <) #(1 2 3 4 5))
(test (sort! #(3 1 5 2 4) <) #(1 2 3 4 5))
(test (sort! #(3 1 5 4 2) <) #(1 2 3 4 5))
(test (sort! #(3 2 5 4 1) <) #(1 2 3 4 5))
(test (sort! #(3 2 5 1 4) <) #(1 2 3 4 5))
(test (sort! #(3 4 5 1 2) <) #(1 2 3 4 5))
(test (sort! #(3 4 5 2 1) <) #(1 2 3 4 5))
(test (sort! #(4 1 5 2 3) <) #(1 2 3 4 5))
(test (sort! #(4 1 5 3 2) <) #(1 2 3 4 5))
(test (sort! #(4 2 5 1 3) <) #(1 2 3 4 5))
(test (sort! #(4 2 5 3 1) <) #(1 2 3 4 5))
(test (sort! #(4 3 5 2 1) <) #(1 2 3 4 5))
(test (sort! #(4 3 5 1 2) <) #(1 2 3 4 5))
(test (sort! #(1 2 3 5 4) <) #(1 2 3 4 5))
(test (sort! #(1 2 4 5 3) <) #(1 2 3 4 5))
(test (sort! #(1 3 2 5 4) <) #(1 2 3 4 5))
(test (sort! #(1 3 4 5 2) <) #(1 2 3 4 5))
(test (sort! #(1 4 2 5 3) <) #(1 2 3 4 5))
(test (sort! #(1 4 3 5 2) <) #(1 2 3 4 5))
(test (sort! #(2 1 3 5 4) <) #(1 2 3 4 5))
(test (sort! #(2 1 4 5 3) <) #(1 2 3 4 5))
(test (sort! #(2 3 4 5 1) <) #(1 2 3 4 5))
(test (sort! #(2 3 1 5 4) <) #(1 2 3 4 5))
(test (sort! #(2 4 3 5 1) <) #(1 2 3 4 5))
(test (sort! #(2 4 1 5 3) <) #(1 2 3 4 5))
(test (sort! #(3 1 2 5 4) <) #(1 2 3 4 5))
(test (sort! #(3 1 4 5 2) <) #(1 2 3 4 5))
(test (sort! #(3 2 4 5 1) <) #(1 2 3 4 5))
(test (sort! #(3 2 1 5 4) <) #(1 2 3 4 5))
(test (sort! #(3 4 1 5 2) <) #(1 2 3 4 5))
(test (sort! #(3 4 2 5 1) <) #(1 2 3 4 5))
(test (sort! #(4 1 2 5 3) <) #(1 2 3 4 5))
(test (sort! #(4 1 3 5 2) <) #(1 2 3 4 5))
(test (sort! #(4 2 1 5 3) <) #(1 2 3 4 5))
(test (sort! #(4 2 3 5 1) <) #(1 2 3 4 5))
(test (sort! #(4 3 2 5 1) <) #(1 2 3 4 5))
(test (sort! #(4 3 1 5 2) <) #(1 2 3 4 5))
(test (sort! #(1 2 3 4 5) <) #(1 2 3 4 5))
(test (sort! #(1 2 4 3 5) <) #(1 2 3 4 5))
(test (sort! #(1 3 2 4 5) <) #(1 2 3 4 5))
(test (sort! #(1 3 4 2 5) <) #(1 2 3 4 5))
(test (sort! #(1 4 2 3 5) <) #(1 2 3 4 5))
(test (sort! #(1 4 3 2 5) <) #(1 2 3 4 5))
(test (sort! #(2 1 3 4 5) <) #(1 2 3 4 5))
(test (sort! #(2 1 4 3 5) <) #(1 2 3 4 5))
(test (sort! #(2 3 4 1 5) <) #(1 2 3 4 5))
(test (sort! #(2 3 1 4 5) <) #(1 2 3 4 5))
(test (sort! #(2 4 3 1 5) <) #(1 2 3 4 5))
(test (sort! #(2 4 1 3 5) <) #(1 2 3 4 5))
(test (sort! #(3 1 2 4 5) <) #(1 2 3 4 5))
(test (sort! #(3 1 4 2 5) <) #(1 2 3 4 5))
(test (sort! #(3 2 4 1 5) <) #(1 2 3 4 5))
(test (sort! #(3 2 1 4 5) <) #(1 2 3 4 5))
(test (sort! #(3 4 1 2 5) <) #(1 2 3 4 5))
(test (sort! #(3 4 2 1 5) <) #(1 2 3 4 5))
(test (sort! #(4 1 2 3 5) <) #(1 2 3 4 5))
(test (sort! #(4 1 3 2 5) <) #(1 2 3 4 5))
(test (sort! #(4 2 1 3 5) <) #(1 2 3 4 5))
(test (sort! #(4 2 3 1 5) <) #(1 2 3 4 5))
(test (sort! #(4 3 2 1 5) <) #(1 2 3 4 5))
(test (sort! #(4 3 1 2 5) <) #(1 2 3 4 5))

(test (sort! #(3 1 2 1 4 1) <) #(1 1 1 2 3 4))
(test (sort! #(1 1 1) <) #(1 1 1))
(test (sort! #(1 2 3) (lambda (a b) (> a b))) #(3 2 1))
(test (equal? (sort! (list 3 4 8 2 0 1 5 9 7 6) <) (list 0 1 2 3 4 5 6 7 8 9)) #t)
(test (equal? (sort! (list 3 4 8 2 0 1 5 9 7 6) (lambda (a b) (< a b))) (list 0 1 2 3 4 5 6 7 8 9)) #t)
(test (equal? (sort! (list) <) '()) #t)
(test (equal? (sort! (list 1) <) '(1)) #t)
(test (equal? (sort! (list 1 1 1) <) '(1 1 1)) #t)
(test (equal? (sort! (list 0 1 2 3 4 5 6 7 8 9) <) '(0 1 2 3 4 5 6 7 8 9)) #t)
(test (equal? (sort! (list #\a #\l #\o #\h #\a) char<?) '(#\a #\a #\h #\l #\o)) #t)
(test (equal? (sort! (list "tic" "tac" "toe") string<?) '("tac" "tic" "toe")) #t)
(test (equal? (sort! (list 3 4 8 2 0 1 5 9 7 6) >) (reverse (list 0 1 2 3 4 5 6 7 8 9))) #t)
(test (equal? (sort! '((3 . 1) (2 . 8) (5 . 9) (4 . 7) (6 . 0)) (lambda (a b) (< (car a) (car b)))) '((2 . 8) (3 . 1) (4 . 7) (5 . 9) (6 . 0))) #t)
(test (equal? (sort! '((3 . 1) (2 . 8) (5 . 9) (4 . 7) (6 . 0)) (lambda (a b) (< (cdr a) (cdr b)))) '((6 . 0) (3 . 1) (4 . 7) (2 . 8) (5 . 9))) #t)
(test (equal? (sort! (list (list 1 2) (list 4 3 2) (list) (list 1 2 3 4)) (lambda (a b) (> (length a) (length b)))) '((1 2 3 4) (4 3 2) (1 2) ())) #t)
(test (equal? (sort! '((1 2 3) (4 5 6) (7 8 9)) (lambda (a b) (> (car a) (car b)))) '((7 8 9) (4 5 6) (1 2 3))) #t)
(test (equal? (sort! (list #\b #\A #\B #\a #\c #\C) char<?) '(#\A #\B #\C #\a #\b #\c)) #t)
(test (equal? (sort! (list (list 'u 2) (list 'i 1) (list 'a 7) (list 'k 3) (list 'c 4) (list 'b 6))
		     (lambda (a b) (< (cadr a) (cadr b))))
	      '((i 1) (u 2) (k 3) (c 4) (b 6) (a 7)))
      #t)
(test (equal? (sort! (sort! '(1 2 3) >) <) '(1 2 3)) #t)
(test (sort! #2d((1 2) (3 4)) >) #2D((4 3) (2 1))) ; ?!?
(test (sort! #2d((1 4) (3 2)) >) #2D((4 3) (2 1))) ; ??!!?? this is not what anyone would expect
(test (sort! '(3 2 1) (lambda (a b c) #f)) 'error)
(test (sort! '(3 2 1) (lambda* (a b c) (< a b))) '(1 2 3))
(test (sort! '(3 2 1) (lambda (a) #f)) 'error)
(test (sort! '(3 2 1) (lambda* (a) #f)) 'error)
(test (sort! '(3 1 2 4) (lambda args (< (car args) (cadr args)))) '(1 2 3 4))

(test (equal? (sort! (vector 3 4 8 2 0 1 5 9 7 6) <) (vector 0 1 2 3 4 5 6 7 8 9)) #t)
(test (equal? (sort! '#() <) '#()) #t)
(test (sort! '(1 2 . 3) <) 'error)
(test (sort! #(1 3 8 7 5 6 4 2) (lambda (a b) (if (even? a) (or (odd? b) (< a b)) (and (odd? b) (< a b))))) #(2 4 6 8 1 3 5 7))
(let ((ninf (real-part (log 0.0))) (pinf (- (real-part (log 0.0))))) (test (sort! (list pinf 0.0 ninf) <) (list ninf 0.0 pinf)))
(test (sort! '(1 1 1) <) '(1 1 1))

(test (call/cc (lambda (return) (sort! '(1 2 3) (lambda (a b) (return "oops"))))) "oops")
(let ((p1 (make-procedure-with-setter (lambda (a b) (< a b)) (lambda (a b) (error 'oops)))))
  (test (sort! '(3 1 2 4) p1) '(1 2 3 4)))
(let ((p1 (make-procedure-with-setter (lambda* (a (b 2)) (< a b)) (lambda (a b) (error 'oops)))))
  (test (sort! '(3 1 2 4) p1) '(1 2 3 4)))
(let ((p1 (make-procedure-with-setter (lambda args (< (car args) (cadr args))) (lambda (a b) (error 'oops)))))
  (test (sort! '(3 1 2 4) p1) '(1 2 3 4)))

(test (let ((v (make-vector 1000)))
	(do ((i 0 (+ i 1)))
	    ((= i 1000))
	  (vector-set! v i (random 100.0)))
	(set! v (sort! v >))
	(call-with-exit
	 (lambda (return)
	   (do ((i 0 (+ i 1)))
	       ((= i 999) #t)
	     (if (<= (v i) (v (+ i 1)))
		 (return #f))))))
      #t)

(test (let ((v '()))
	(do ((i 0 (+ i 1)))
	    ((= i 1000))
	  (set! v (cons (random 100.0) v)))
	(set! v (sort! v >))
	(apply > v))
      #t)

(test (sort! (list 3 2 1) (lambda (m n) (let ((vals (sort! (list m n) <))) (< m n)))) '(1 2 3))

(test (let ((lst '()))
	(do ((i 0 (+ i 1)))
	    ((= i 4))
	  (set! lst (cons (random 1.0) lst)))
	(let ((vals (sort! lst (lambda (m n)
				 (let ((lst1 (list 1 2 3)))
				   (sort! lst1 <))
				 (< m n)))))
	  (apply < vals)))
      #t)


(let ((v (make-vector 8)))
  (do ((i 0 (+ i 1)))
      ((= i 10))
    (do ((k 0 (+ k 1)))
	((= k 8))
      (set! (v k) (- (random 1.0) 0.5)))
    (let ((v1 (copy v)))
      (sort! v <)
      (if (not (apply < (vector->list v)))
	  (format #t ";(sort! ~A <) -> ~A?" v1 v)))))

(test (sort!) 'error)
(test (sort! '(1 2 3) < '(3 2 1)) 'error)
(test (sort! '(1 2 3)) 'error)
(test (sort! '(1 2 3) 1) 'error)
(test (sort! '(1 2 3) < <) 'error)
(test (sort! (cons 3 2) <) 'error)
(test (sort! (list 1 0+i) <) 'error)
(test (sort! (list "hi" "ho") <) 'error)
(test (sort! '(1 2 #t) <) 'error)
(test (sort! '(1 2 . #t) <) 'error)
(test (sort! '(#\c #\a #\b) <) 'error)
(test (sort! (begin) if) '())

(test (sort! (list) <) '())
(test (sort! (vector) <) #())
(test (sort! (list #\a) <) '(#\a)) ; I guess this is reasonable
(test (sort! (list #("hi")) <) '(#("hi")))
(test (sort! (append (sort! (append (sort! () <) ()) <) ()) <) '())
(test (sort! (append (sort! (append (sort! '(1 2) <) '(1 2)) <) '(1 2)) <) '(1 1 1 2 2 2))
(test (let ((lst (list 3 1 12 4 1)))
      (sort! lst (lambda (a b)
                   (let ((val (map (lambda (n) (+ n 1)) (list a b))))
                     (apply < val)))))
      '(1 1 3 4 12))
(test (sort! '(#\c #\a #\b) (lambda (a b) (string<? (string a) (string b)))) '(#\a #\b #\c))

(for-each
 (lambda (arg)
   (test (sort! arg <) 'error))
 (list -1 #\a 1 0 "" "hiho" (make-hash-table) :hi 'a-symbol 3.14 3/4 1.0+1.0i #f #t))

(for-each
 (lambda (arg)
   (test (sort! '(1 2 3) arg) 'error))
 (list -1 #\a 1 0 'a-symbol 3.14 3/4 1.0+1.0i #f #t #(1) '(1) "hi" abs :hi))

(test (sort! '(1 2 "hi" 3) <) 'error)
(test (sort! '(1 -2 "hi" 3) (lambda (a b) 
			     (let ((a1 (if (number? a) a (length a)))
				   (b1 (if (number? b) b (length b))))
			       (< a1 b1))))
      '(-2 1 "hi" 3))

(let ((ok #f))
  (catch #t
	 (lambda ()
	   (dynamic-wind
	       (lambda () #f)
	       (lambda () (sort! '(1 2 "hi" 3) <))
	       (lambda () (set! ok #t))))
	 (lambda args 'error))
  (if (not ok) (format #t "dynamic-wind out of sort! skipped cleanup?~%")))


(let ((lst (list 1 2 3 9 8 7)))
  (let ((val (catch #t
		    (lambda ()
		      (sort! (copy lst)
			     (lambda (a b)
			       (if (< a b) (error 'sort-error "a < b"))
			       #t)))
		    (lambda args (car args)))))
    (if (not (eq? val 'sort-error))
	(format #t ";sort! with error: ~A~%" val)))

  (let ((val (call-with-exit
	      (lambda (return)
		(sort! (copy lst)
		       (lambda (a b)
			 (if (< a b) (return 'sort-error))
			 #t))))))
    (if (not (eq? val 'sort-error))
	(format #t ";sort! call-with-exit: ~A~%" val)))

  (let ((val (call/cc
	      (lambda (return)
		(sort! (copy lst)
		       (lambda (a b)
			 (if (< a b) (return 'sort-error))
			 #t))))))
    (if (not (eq? val 'sort-error))
	(format #t ";sort! call/cc: ~A~%" val)))
  )

(let ((old-safety *safety*))
  (set! *safety* 1)
  (test (sort! #(1 2 3) (lambda (a b) (and #t (= a b)))) 'error)
  (set! *safety* old-safety))

(if (defined? 'make-vct)
    (let ((v (vct 1 4 2 34 2)))
      (test (length v) 5)
      (test (v 1) 4.0)
      (sort! v <)
      (test v (vct 1 2 2 4 34))
      (set! (v 1) 32)
      (sort! v >)
      (test v (vct 34 32 4 2 1))
      (test v (copy v))
      (test (reverse v) (vct 1 2 4 32 34))
      (test (object->string v) "#<vct[len=5]: 34.000 32.000 4.000 2.000 1.000>")
      (test (fill! v 1.0) (vct 1 1 1 1 1))
      ))

(let* ((vtype (make-type :getter vector-ref
			 :setter vector-set!
			 :length vector-length))
       (v? (car vtype))
       (make-v (cadr vtype))
       (v-ref (caddr vtype)))
  (let ((v1 (make-v (make-vector 6 0))))
    (set! (v1 0) 1)
    (set! (v1 1) 11)
    (set! (v1 2) 111)
    (set! (v1 3) 1111)
    (set! (v1 4) 11111)
    (set! (v1 5) -1)
    (sort! v1 <)
    (test (v1 0) -1)
    (test (v1 3) 111)))
	   




;;; -------- catch --------
;;; catch

(define (catch-test sym)
  (let ((errs '()))
    (catch 'a1
	 (lambda ()
	   (catch 'a2
		  (lambda ()
		    (catch 'a3
			   (lambda ()
			     (catch 'a4
				    (lambda ()
				      (error sym "hit error!"))
				    (lambda args
				      (set! errs (cons 'a4 errs))
				      'a4)))
			   (lambda args
			     (set! errs (cons 'a3 errs))
			     'a3)))
		  (lambda args
		    (set! errs (cons 'a2 errs))
		    'a2)))
	 (lambda args
	   (set! errs (cons 'a1 errs))
	   'a1))
    errs))

(test (catch-test 'a1) '(a1))
(test (catch-test 'a2) '(a2))
(test (catch-test 'a3) '(a3))
(test (catch-test 'a4) '(a4))

(define (catch-test-1 sym)
  (let ((errs '()))
    (catch 'a1
	 (lambda ()
	   (catch 'a2
		  (lambda ()
		    (catch 'a3
			   (lambda ()
			     (catch 'a4
				    (lambda ()
				      (error sym "hit error!"))
				    (lambda args
				      (set! errs (cons 'a4 errs))
				      (error 'a3)
				      'a4)))
			   (lambda args
			     (set! errs (cons 'a3 errs))
			     (error 'a2)
			     'a3)))
		  (lambda args
		    (set! errs (cons 'a2 errs))
		    (error 'a1)
		    'a2)))
	 (lambda args
	   (set! errs (cons 'a1 errs))
	   'a1))
    errs))

(test (catch-test-1 'a1) '(a1))
(test (catch-test-1 'a2) '(a1 a2))
(test (catch-test-1 'a3) '(a1 a2 a3))
(test (catch-test-1 'a4) '(a1 a2 a3 a4))

(test (catch #t (catch #t (lambda () (lambda () 1)) (lambda args 'oops)) (lambda args 'error)) 1)
(test (catch #t (catch #t (lambda () (error 'oops)) (lambda args (lambda () 1))) (lambda args 'error)) 1)
(test ((catch #t (lambda () (error 'oops)) (lambda args (lambda () 1)))) 1)
(test ((catch #t (lambda () (error 'oops)) (catch #t (lambda () (lambda args (lambda () 1))) (lambda args 'error)))) 1)
(test (catch #t (dynamic-wind (lambda () #f) (lambda () (lambda () 1)) (lambda () #f)) (lambda args 'error)) 1)
(test (dynamic-wind (catch #t (lambda () (lambda () #f)) (lambda args 'error)) (lambda () 1) (lambda () #f)) 1)
(test (dynamic-wind ((lambda () (lambda () #f))) (lambda () 1) (((lambda () (lambda () (lambda () #t)))))) 1)
(test (catch #t ((lambda () (lambda () 1))) (lambda b a)) 1)
(test (map (catch #t (lambda () abs) abs) '(-1 -2 -3)) '(1 2 3))
(test (catch + (((lambda () lambda)) () 1) +) 1)
(test (catch #t + +) 'error)
(test (string? (catch + s7-version +)) #t)
(test (string? (apply catch + s7-version (list +))) #t)
(test (catch #t (lambda () (catch '#t (lambda () (error '#t)) (lambda args 1))) (lambda args 2)) 1)
(test (catch #t (lambda () (catch "hi" (lambda () (error "hi")) (lambda args 1))) (lambda args 2)) 2) ; guile agrees with this
(test (let ((str (list 1 2))) (catch #t (lambda () (catch str (lambda () (error str)) (lambda args 1))) (lambda args 2))) 1)
(test (let ((str "hi")) (catch #t (lambda () (catch str (lambda () (error str)) (lambda args 1))) (lambda args 2))) 2) ; this doesn't make sense
(test (let () (abs (catch #t (lambda () -1) (lambda args 0)))) 1)

#|
(for-each 
 (lambda (str) 
   (format #t "~A ~A~%" str (catch #t (lambda () 
					(catch str (lambda () 
						     (error str))  ; use throw for guile
					       (lambda args 1))) 
				   (lambda args 2)))) 
 (list "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

        s7     Guile
"hi"    2       2
()      1       2
(1)     1       2
(1 . 2) 1       2
#f      1       2
a-symbol 1      1
#(#<unspecified> #<unspecified> #<unspecified>) 1 2
abs     1       2
3.14    1       2
3/4     1       2
1+1i    1       2
#t      1       1
#<unspecified> 1 2
#<closure> 1    2
|#

(let ((x 0))
  (catch #t
	 (lambda ()
	   (catch #t
		  (lambda ()
		    (+ 1 __asdf__))
		  (lambda args
		    (set! x (+ x 1))
		    (+ 1 __asdf__))))
	 (lambda args
	   (set! x (+ x 1))))
  (test x 2))

(test (let ((x 0))
	(catch 'a
	     (lambda ()
	       (catch 'b
		      (lambda ()
			(catch 'a
			       (lambda ()
				 (error 'a))
			       (lambda args
				 (set! x 1))))
		      (lambda args
			(set! x 2))))
	     (lambda args
	       (set! x 3)))
	x)
      1)

(test (catch) 'error)
(test (catch s7-version) 'error)
(test (catch #t s7-version) 'error)
(test (catch #t s7-version + +) 'error)



;;; error

(test (catch #t (lambda () (error 'oops 1)) (let () (lambda args (caadr args)))) 1)
(test (catch #t (lambda () (error 'oops 1)) (let ((x 3)) (lambda args (+ x (caadr args))))) 4)
(test (catch #t (let () (lambda () (error 'oops 1))) (let ((x 3)) (lambda args (+ x (caadr args))))) 4)
(test (catch #t (let ((x 2)) (lambda () (error 'oops x))) (let ((x 3)) (lambda args (+ x (caadr args))))) 5)
(test (catch #t (let ((x 2)) ((lambda () (lambda () (error 'oops x))))) (let ((x 3)) (lambda args (+ x (caadr args))))) 5)

(test (let ((pws (make-procedure-with-setter (lambda () (+ 1 2)) (lambda (a) (+ a 2)))))
	(catch #t pws (lambda (tag type) tag)))
      3)
(test (let ((pws (make-procedure-with-setter (lambda () (error 'pws 3) 4) (lambda (a) (+ a 2)))))
	(catch #t pws (lambda (tag type) tag)))
      'pws)
(test (let ((pws (make-procedure-with-setter (lambda (a b) a) (lambda (a b) (+ a 2)))))
	(catch #t (lambda () (error 'pws-error 3)) pws))
      'pws-error)

(for-each
 (lambda (tag)
   (let ((val (catch tag (lambda () (error tag "an error") 123) (lambda args (car args)))))
     (if (not (equal? tag val))
	 (format #t ";catch ~A -> ~A~%" tag val))))
 (list :hi '() #() #<eof> #f #t #<unspecified> car #\a 32 9/2))

(for-each
 (lambda (tag)
   (let ((val (catch #t (lambda () (error tag "an error") 123) (lambda args (car args)))))
     (if (not (equal? tag val))
	 (format #t ";catch #t (~A) -> ~A~%" tag val))))
 (list :hi '() #<eof> #f #t #<unspecified> car #\a 32 9/2 '(1 2 3) '(1 . 2) #(1 2 3) #()))

(for-each
 (lambda (tag)
   (test (catch #t tag (lambda args (car args))) 'error)
   (test (catch #t (lambda () #f) tag) 'error))
 (list :hi '() #<eof> #f #t #<unspecified> #\a 32 9/2 '(1 2 3) '(1 . 2) #(1 2 3) #()))

;; (error <string>...) throws 'no-catch which makes it harder to check
(let ((val (catch #t (lambda () (error "hi") 123) (lambda args (car args)))))
  (if (not (eq? val 'no-catch))
      (format #t ";catch #t, tag is string -> ~A~%" val)))

(for-each
 (lambda (tag)
   (let ((val (catch tag (lambda () (error #t "an error") 123) (lambda args (car args)))))
     (if (not (equal? #t val))
	 (format #t ";catch ~A -> ~A (#t)~%" tag val))))
 (list :hi '() #<eof> #f #t #<unspecified> car #\a 32 9/2))

(let ((tag 'tag)) (test (catch (let () tag) (lambda () (set! tag 123) (error 'tag "tag") tag) (lambda args (car args))) 'tag))

(let ()
  (define (check-reerror x)
    (catch #t
      (lambda ()
	(define (our-func x)
	  (case x
	    ((0) (error 'zero))
	    ((1) (error 'one))
	    (else (error 'else))))
	(catch #t 
  	  (lambda () 
	    (our-func x))
	  (lambda args
	    (if (eq? (car args) 'one)
		(our-func (+ x 1))
		(apply error args)))))
      (lambda args
	(let ((type (car args)))
	  (case type
	    ((zero) 0)
	    ((one) 1)
	    (else 2))))))
  (test (check-reerror 0) 0)
  (test (check-reerror 1) 2)
  (test (check-reerror 2) 2))

(test (catch 'hiho
	     (lambda ()
	       (define (f1 a)
		 (error 'hiho a))
	       (* 2 (catch 'hiho
			   (lambda ()
			     (f1 3))
			   (lambda args (caadr args)))))
	     (lambda args (caadr args)))
      6)

(test (let ()
	(define (f1 a)
	  (error 'hiho a))
	(catch 'hiho
	       (lambda ()
		 (* 2 (catch 'hiho
			     (lambda ()
			       (f1 3))
			     (lambda args (caadr args)))))
	       (lambda args (caadr args))))
      6)

(test (catch 'hiho
	     (lambda ()
	       (let ((f1 (catch 'hiho
				(lambda ()
				  (lambda (a) (error 'hiho 3)))
				(lambda args args))))
		 (f1 3)))
	     (lambda args (caadr args)))
      3)


(test (error) 'error)
(test (let ((x 1))
	(let ((val (catch #\a
			  (lambda ()
			    (set! x 0)
			    (error #\a "an error")
			    (set! x 2))
			  (lambda args
			    (if (equal? (car args) #\a)
				(set! x (+ x 3)))
			    x))))
	  (= x val 3)))
      #t)
(test (let ((x 1))
	(let ((val (catch 32
			   (lambda ()
			     (catch #\a
				    (lambda ()
				      (set! x 0)
				      (error #\a "an error: ~A" (error 32 "another error!"))
				      (set! x 2))
				    (lambda args
				      (if (equal? (car args) #\a)
					  (set! x (+ x 3)))
				      x)))
			   (lambda args 
			     (if (equal? (car args) 32)
				 (set! x (+ x 30)))))))
	  (= x val 30)))
      #t)

#|
(let ((old-error-hook (hook-functions *error-hook*))
      (tag #f)
      (args #f))
  (set! (hook-functions *error-hook*)
	(list (lambda (etag eargs)
		(set! tag etag)
		(set! args eargs))))
  (error 'tag 1 2 3)
  (test (and (equal? tag 'tag)
	     (equal? args '(1 2 3))))
  (set! (hook-functions *error-hook*) old-error-hook))
|#
;;; can't include this because it interrupts the load




;;; --------------------------------------------------------------------------------

(define (last-pair l) ; needed also by loop below
  (if (pair? (cdr l)) 
      (last-pair (cdr l)) l))
  

(let ()
  ;; from guile-user I think
  ;; (block LABEL FORMS...)
  ;;
  ;; Execute FORMS.  Within FORMS, a lexical binding named LABEL is
  ;; visible that contains an escape function for the block.  Calling
  ;; the function in LABEL with a single argument will immediatly stop
  ;; the execution of FORMS and return the argument as the value of the
  ;; block.  If the function in LABEL is not invoked, the value of the
  ;; block is the value of the last form in FORMS.
  
  (define-macro (block label . forms)
    `(let ((body (lambda (,label) ,@forms))
	   (tag (gensym "return-")))
       (catch tag
	      (lambda () (body (lambda (val) (error tag val))))
	      (lambda (tag val) val))))
  
  ;; (with-return FORMS...)
  ;;
  ;; Equivalent to (block return FORMS...)
  
  (define-macro (with-return . forms)
    `(block return ,@forms))
  
  ;; (tagbody TAGS-AND-FORMS...)
  ;;
  ;; TAGS-AND-FORMS is a list of either tags or forms.  A TAG is a
  ;; symbol while a FORM is everything else.  Normally, the FORMS are
  ;; executed sequentially.  However, control can be transferred to the
  ;; forms following a TAG by invoking the tag as a function.  That is,
  ;; within the FORMS, there is a lexical binding for each TAG with the
  ;; symbol that is the tag as its name.  The bindings carry functions
  ;; that will execute the FORMS following the respective TAG.
  ;;
  ;; The value of a tagbody is always `#f'.
  
  (define (transform-tagbody forms)
    (let ((start-tag (gensym "start-"))
	  (block-tag (gensym "block-")))
      (let loop ((cur-tag start-tag)
		 (cur-code '())
		 (tags-and-code '())
		 (forms forms))
	(cond
	 ((null? forms)
	  `(block ,block-tag
		  (letrec ,(reverse! (cons (list cur-tag `(lambda () ,@(reverse! (cons `(,block-tag #f) cur-code)))) tags-and-code))
		    (,start-tag))))
	 ((symbol? (car forms))
	  (loop (car forms)
		'()
		(cons (list cur-tag `(lambda () ,@(reverse! (cons `(,(car forms)) cur-code)))) tags-and-code)
		(cdr forms)))
	 (else
	  (loop cur-tag
		(cons (car forms) cur-code)
		tags-and-code
		(cdr forms)))))))
  
  (define-macro (tagbody . forms)
    (transform-tagbody forms))
  
  (define (first_even l)
    (with-return
     (tagbody
      continue
      (if (not (not (null? l)))
	  (break))
      (let ((e (car l)))
	(if (not (number? e))
	    (break))
	(if (even? e)
	    (return e))
	(set! l (cdr l)))
      (continue)
      break)
     (return #f)))
  
  (let ((val (first_even '(1 3 5 6 7 8 9))))
    (if (not (equal? val (list 6)))
	(format #t "first_even (tagbody, gensym, reverse!) (6): '~A~%" val)))
  
  (let ((hi (lambda* (a) a)))
    (test (hi 1) 1)
    (test (hi) #f)          ; all args are optional
    (test (hi :a 32) 32)    ; all args are keywords
    (test (hi 1 2) 'error)  ; extra args
    
    (for-each
     (lambda (arg)
       (test (hi arg) arg)
       (test (hi :a arg) arg))
     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2)))
    
    (test (hi :b 1) 'error))
  
  (let ((hi (lambda* ((a 1)) a)))
    (test (hi 2) 2)
    (test (hi) 1)
    (test (hi :a 2) 2)
    
    (for-each
     (lambda (arg)
       (test (hi arg) arg)
       (test (hi :a arg) arg))
     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2))))
  
  (let ((hi (lambda* (a (b "hi")) (list a b))))
    (test (hi) (list #f "hi"))
    (test (hi 1) (list 1 "hi"))
    (test (hi 1 2) (list 1 2))
    (test (hi :b 1) (list #f 1))
    (test (hi :a 1) (list 1 "hi"))
    (test (hi 1 :b 2) (list 1 2))
    (test (hi :b 3 :a 1) (list 1 3))
    (test (hi :a 3 :b 1) (list 3 1))
    (test (hi 1 :a 3) 'error)
    (test (hi 1 2 :a 3) 'error) ; trailing (extra) args
    (test (hi :a 2 :c 1) 'error)
    (test (hi 1 :c 2) 'error)
    
    (for-each
     (lambda (arg)
       (test (hi :a 1 :b arg) (list 1 arg))
       (test (hi :a arg) (list arg "hi"))
       (test (hi :b arg) (list #f arg))
       (test (hi arg arg) (list arg arg)))
     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2))))
  
  (let ((hi (lambda* (a :key (b 3) :optional c) (list a b c))))
    (test (hi) (list #f 3 #f))
    (test (hi 1) (list 1 3 #f))
    (test (hi :c 32) (list #f 3 32))
    (test (hi :c 32 :b 43 :a 54) (list 54 43 32))
    (test (hi 1 2 3) (list 1 2 3))
    (test (hi :b 32) (list #f 32 #f))
    (test (hi 1 2 :c 32) (list 1 2 32)))
  
  (let ((hi (lambda* (a :rest b) (list a b))))
    (test (hi 1 2 3) (list 1 (list 2 3)))
    (test (hi) (list #f ()))
    (test (hi :a 2) (list 2 '()))
    (test (hi :b 3) (list #f 3)))
  
  (let ((hi (lambda* (a :rest b :rest c) (list a b c))))
    (test (hi 1 2 3 4 5) (list 1 (list 2 3 4 5) (list 3 4 5))))
  
  (let ((hi (lambda* ((a 3) :key (b #t) :optional (c pi) :rest d) (list a b c d))))
    (test (hi) (list 3 #t pi ()))
    (test (hi 1 2 3 4) (list 1 2 3 (list 4))))
  
  (let ((hi (lambda* ((a 'hi)) (equal? a 'hi))))
    (test (hi) #t)
    (test (hi 1) #f)
    (test (hi 'hi) #t)
    (test (hi :a 1) #f))
  
  (let* ((x 32)
	 (hi (lambda* (a (b x)) (list a b))))
    (test (hi) (list #f 32))
    (test (hi :a 1) (list 1 32)))
  
  (let ((hi (lambda* (a . b) (list a b))))
    (test (hi 1 2 3) (list 1 (list 2 3)))
    (test (hi) (list #f ()))
    (test (hi :a 2) (list 2 '()))
    (test (hi :b 3) (list #f 3)))
  
  (let ((hi (lambda* ((a 0.0) :optional (b 0.0)) (+ a b))))
    (num-test (hi 1.0) 1.0)
    (num-test (hi 1.0 2.0) 3.0)
    (num-test (hi) 0.0)
    (num-test (+ (hi) (hi 1.0) (hi 1.0 2.0)) 4.0)
    (num-test (+ (hi 1.0) (hi) (hi 1.0 2.0)) 4.0)
    (num-test (+ (hi 1.0) (hi 1.0 2.0) (hi)) 4.0)
    (num-test (+ (hi 1.0 2.0) (hi) (hi 1.0)) 4.0))
  
  (test (let ((hi (lambda*))) (hi)) 'error)
  (test (let ((hi (lambda* #f))) (hi)) 'error)
  (test (let ((hi (lambda* "hi" #f))) (hi)) 'error)
  (test (let ((hi (lambda* ("hi") #f))) (hi)) 'error)
  (test (let ((hi (lambda* (a 0.0) a))) (hi)) 'error)
  (test (let ((hi (lambda* (a . 0.0) a))) (hi)) 'error)
  (test (let ((hi (lambda* ((a . 0.0)) a))) (hi)) 'error)
  (test (let ((hi (lambda* ((a 0.0 "hi")) a))) (hi)) 'error)
  (test (let ((hi (lambda* ((a 0.0 . "hi")) a))) (hi)) 'error)
  (test (let ((hi (lambda* ((a)) a))) (hi)) 'error)
  (test (let ((hi (lambda* (a 0.0) (b 0.0) (+ a b)))) (hi)) 'error)
  
  (test (let () (define* (hi) 0) (hi)) 0)
  (test (let () (define* (hi a . b) b) (hi 1 2 3)) '(2 3))
  (test (let () (define* (hi a . b) b) (hi :a 1 2 3)) '(2 3))
  (test (let () (define* (hi a . b) b) (hi 1)) '())
  (test (let () (define* (hi a . b) b) (hi :a 1)) '())
  (test (let () (define* (hi a . b) b) (hi)) '())
  
  (test (let () (define* (hi a :rest b) b) (hi 1 2 3)) '(2 3))
  (test (let () (define* (hi a :rest b) b) (hi :a 1 2 3)) '(2 3))
  (test (let () (define* (hi a :rest b) b) (hi 1)) '())
  (test (let () (define* (hi a :rest b) b) (hi :a 1)) '())
  (test (let () (define* (hi a :rest b) b) (hi)) '())
  
  (test (let () (define* (hi :key a :rest b) b) (hi 1 2 3)) '(2 3))
  (test (let () (define* (hi :key a :rest b) b) (hi :a 1 2 3)) '(2 3))
  (test (let () (define* (hi :key a :rest b) b) (hi 1)) '())
  (test (let () (define* (hi :key a :rest b) b) (hi :a 1)) '())
  (test (let () (define* (hi :key a :rest b) b) (hi)) '())
  
  (test (let () (define* (hi :optional a :rest b) b) (hi 1 2 3)) '(2 3))
  (test (let () (define* (hi :optional a :rest b) b) (hi :a 1 2 3)) '(2 3))
  (test (let () (define* (hi :optional a :rest b) b) (hi 1)) '())
  (test (let () (define* (hi :optional a :rest b) b) (hi :a 1)) '())
  (test (let () (define* (hi :optional a :rest b) b) (hi)) '())
  
  (test (let () (define* (hi (a 1) . b) b) (hi 1 2 3)) '(2 3))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi)) '(#f 22 ()))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :a 1)) '(1 22 ()))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :b 1)) '(#f 1 ()))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :c 1)) '(#f 22 1))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :a 1 2)) '(1 2 ()))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :b 1 2 3)) 'error) ; b set twice
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :c 1 2 3)) '(#f 2 (3)))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :b 1 :a 2 3)) '(2 1 (3)))

  (test (let () (define* (hi (a 1) :allow-other-keys) a) (hi)) 1)
  (test (let () (define* (hi (a 1) :allow-other-keys) a) (hi :b :a :a 3)) 3)
  (test (let () (define* (hi (a 1) :allow-other-keys) a) (hi :b 3)) 1)
  (test (let () (define* (hi (a 1) :allow-other-keys) a) (hi :a 3)) 3)
  (test (let () (define* (hi (a 1) :allow-other-keys) a) (hi a: 3)) 3)
  (test (let () (define* (hi (a 1) :allow-other-keys) a) (hi 3)) 3)
  (test (let () (define* (hi (a 1) :allow-other-keys) a) (hi 3 :b 2)) 3)
  (test (let () (define* (hi (a 1) :allow-other-keys) a) (hi :c 1 :a 3 :b 2)) 3)
  (test (let () (define* (hi (a 1) :optional :key :allow-other-keys) a) (hi :c 1 :a 3 :b 2)) 3)
  (test (let () (define* (hi :optional :key :rest a :allow-other-keys) a) (hi :c 1 :a 3 :b 2)) '(:c 1 :a 3 :b 2))
  
  (test (let () (define* (hi :optional (a 1) :optional (b 2)) a)) 'error)
  (test (let () (define* (hi :optional :optional (a 2)) a) (hi 21)) 'error)
  (test (let () (define* (hi optional: (a 1)) a) (hi 1)) 'error)
  (test (let () (define* (hi :optional: (a 1)) a) (hi 1)) 'error)
  (test (let () (define* (hi :key (a 1) :key (b 2)) a)) 'error)
  (test (let () (define* (hi :key (a 1) :optional (b 2) :allow-other-keys :allow-other-keys) a)) 'error)
  (test (let () (define* (hi :optional (a 1) :key :allow-other-keys) a) (hi :c 1 :a 3 :b 2)) 3)
  (test (let () (define* (hi :key :optional :allow-other-keys) 1) (hi :c 1 :a 3 :b 2)) 1)
  (test (let () (define* (hi :key :optional :allow-other-keys) 1) (hi)) 1)
  (test (let () (define* (hi (a 1) :allow-other-keys) a) (hi :a 2 32)) 'error)
  (test (let () (define* (hi (a 1) :allow-other-keys) a) (hi 2 32)) 'error)

  (test (let () (define* (hi (a 1) :rest c :allow-other-keys) (list a c)) (hi :a 3 :b 2)) '(3 (:b 2)))
  (test (let () (define* (hi (a 1) :rest c) (list a c)) (hi :a 3 :b 2)) '(3 (:b 2)))

  (test (let () (define* (hi (a 1) (b 2) :allow-other-keys) (list a b)) (hi :c 21 :b 2)) '(1 2))
  (test (let () (define hi (lambda* ((a 1) (b 2) :allow-other-keys) (list a b))) (hi :c 21 :b 2)) '(1 2))
  (test (let () (define-macro* (hi (a 1) (b 2) :allow-other-keys) `(list ,a ,b)) (hi :c 21 :b 2)) '(1 2))

  (test (let () (define* (f (a :b)) a) (list (f) (f 1) (f :c) (f :a :c) (f :a 1) (f))) '(:b 1 :c :c 1 :b))
  (test (let () (define* (f a (b :c)) b) (f :b 1 :d)) 'error)

  ;; some of these are questionable
  (test ((lambda* ((x (lambda () 1))) (x))) 1)
  (test ((lambda* ((x x) else) (+ x else)) 1 2) 3)
  (test (symbol? ((lambda* ((y y)) y))) #t)
  (test (symbol? ((lambda* ((y y) :key) y))) #t)
  (test (procedure-arity (lambda* ((a 1) :allow-other-keys) a)) '(0 1 #f))
  (test (procedure-arity (lambda* (:allow-other-keys) 34)) '(0 0 #f))
  (test ((lambda* (:allow-other-keys) 34) :a 32) 34)
  (test (procedure-arity (lambda* ((a 1) :rest b :allow-other-keys) a)) '(0 1 #t))
  (test ((lambda* ((y x) =>) (list y =>)) 1 2) '(1 2))
  (test ((lambda* (=> (y x)) (list y =>)) 1) '(x 1))
  (test ((lambda* ((y #2D((1 2) (3 4)))) (y 1 0))) 3)
  (test ((lambda* ((y (symbol "#(1 #\\a (3))")) x) -1)) -1)
  (test ((lambda* ((y (symbol "#(1 #\\a (3))")) x) y)) (symbol "#(1 #\\a (3))"))
  (test ((lambda* ((y #(1 #\a (3)))) (y 0))) 1)
  (test ((lambda* ((y ()) ()) y)) 'error)
  (test ((lambda* ((y ()) (x)) y)) 'error)
  (test ((lambda* ((=> "") else) else) else) #f)
  (test ((lambda* (x (y x)) y) 1) #f)
  (test ((lambda* (x (y x)) (let ((x 32)) y)) 1) #f)
  (test ((lambda* ((x 1) (y x)) y)) 1)
  (test ((lambda* ((x 1) (y (+ x 1))) y)) 2)
  (test ((lambda* ((x y) (y x)) y)) 'y)              ; I'd expect unbound variable or something here
  (test (let ((z 2)) ((lambda* ((x z) (y x)) y))) 2) ; hmmm
  (test (keyword? ((lambda* ((x :-)) x))) #t)
  (test ((lambda* ((- 0)) -) :- 1) 1)
  (test ((apply lambda* (list (list (list (string->symbol "a") 1)) (string->symbol "a"))) (symbol->keyword (string->symbol "a")) 3) 3)
  (test ((lambda* (:allow-other-keys) 1) :a 321) 1)
  (test ((lambda* (:rest (a 1)) a)) 'error)
  (test ((lambda* (:rest a) a)) '())
  (test ((lambda* (:rest (a 1)) 1)) 'error)
  (test (let ((b 2)) ((lambda* (:rest (a (let () (set! b 3) 4))) b))) 'error)
  (test (let ((b 2)) ((lambda* ((a (let () (set! b 3) 4))) b))) 3)
  (test ((lambda* (:rest hi :allow-other-keys (x x)) x)) 'error)
  (test ((lambda* (:rest x y) (list x y)) 1 2 3) '((1 2 3) 2))
  (test ((lambda* (:rest '((1 2) (3 4)) :rest (y 1)) 1)) 'error)
  (test ((lambda* (:rest (list (quote (1 2) (3 4))) :rest (y 1)) 1)) 'error)
  (test ((lambda* ((x ((list 1 2) 1))) x)) 2)
  (test ((lambda* ((y ("hi" 0))) y)) #\h)
  (test ((lambda* ((x ((lambda* ((x 1)) x)))) x)) 1)
  (test ((lambda* (:rest) 3)) 'error)
  (test ((lambda* (:rest 1) 3)) 'error)
  (test ((lambda* (:rest :rest) 3)) 'error)
  (test ((lambda* (:rest :key) 3)) 'error)
  (test ((lambda* ((: 1)) :)) 1)
  (test ((lambda* ((: 1)) :) :: 21) 21)
  (test ((lambda* ((a 1)) a) a: 21) 21)
  (test ((lambda* ((a 1)) a) :a: 21) 'error)
  (test (let ((func (let ((a 3)) (lambda* ((b (+ a 1))) b)))) (let ((a 21)) (func))) 4)
  (test (let ((a 21)) (let ((func (lambda* ((b (+ a 1))) b))) (let ((a 3)) (func)))) 22)
  (test (let ((a 21)) (begin (define-macro* (func (b (+ a 1))) b) (let ((a 3)) (func)))) 4)
  (test ((lambda* (:rest x :allow-other-keys y) x) 1) 'error)
  (test ((lambda* (:allow-other-keys x) x) 1) 'error)
  (test ((lambda* (:allow-other-keys . x) x) 1 2) 'error)
  (test ((lambda* (:optional . y) y) 1 2 3) '(1 2 3))
  (test ((lambda* (:optional . y) y)) '())
  (test ((lambda* (:rest . (x)) x) 1 2) '(1 2))
  (test ((lambda* (:rest . (x 1)) x) 1 2) 'error)
  (test ((lambda* (:rest . (x)) x)) '())
  (test ((lambda* (:optional . (x)) x) 1) 1)
  (test ((lambda* (:optional . (x 1)) x) 1) 'error)
  (test ((lambda* (:optional . (x)) x)) #f)
  (test ((lambda* (:optional . (x)) x) 1 2) 'error)
  (test ((lambda* (x :key) x) 1) 1)
  (test ((lambda* (:key :optional :rest x :allow-other-keys) x) 1) '(1))
  (test (lambda* (key: x) x) 'error)
  (test (lambda* (:key: x) x) 'error)
  (test ((lambda* x x) 1) '(1))
  (test (lambda* (((x) 1)) x) 'error)
  (test ((lambda* ((a: 3)) a:) :a: 4) 'error)
  (test ((lambda* ((a 3)) a) a: 4) 4)

  ;; not sure the next 4 aren't errors
  (test ((lambda* (:key . x) x) :x 1) '(:x 1))
  (test ((lambda* (:key . x) x)) '())
  (test ((lambda* (:optional . y) y) :y 1) '(:y 1))
  (test ((lambda* (:rest a b c) (list a b c)) 1 2 3 4) '((1 2 3 4) 2 3))

  (test (let ((x 3)) (define* (f (x x)) x) (let ((x 32)) (f))) 3)
  (test (let ((x 3)) (define-macro* (f (x x)) `,x) (let ((x 32)) (f))) 32)

  (test (let () (define (x x) x) (x 1)) 1)
  (test (procedure? (let () (define* (x (x #t)) x) (x x))) #t)
  (test (procedure? (let () (define* (x (x x)) x) (x (x x)))) #t)
  (test (procedure? (let () (define* (x (x x)) x) (x))) #t)
  (test (apply + ((lambda* ((x (values 1 2 3))) x))) 6)
  (test ((lambda* ((x (lambda* ((y (+ 1 2))) y))) (x))) 3)
  ;; (let () (define* (x (x (x))) :optional) (x (x x))) -> segfault infinite loop in prepare_closure_star

  ;;; define-macro
  ;;; define-macro*
  ;;; define-bacro
  ;;; define-bacro*

  (test (let ((x 0)) (define-macro* (hi (a (let () (set! x (+ x 1)) x))) `(+ 1 ,a)) (list (let ((x -1)) (list (hi) x)) x)) '((1 0) 0))
  (test (let ((x 0)) (define-bacro* (hi (a (let () (set! x (+ x 1)) x))) `(+ 1 ,a)) (list (let ((x -1)) (list (hi) x)) x)) '((1 0) 0))
  (test (let ((x 0)) (define-macro* (hi (a (let () (set! x (+ x 1)) x))) `(+ x ,a)) (list (let ((x -1)) (list (hi) x)) x)) '((-1 0) 0))
  (test (let ((x 0)) (define-bacro* (hi (a (let () (set! x (+ x 1)) x))) `(+ x ,a)) (list (let ((x -1)) (list (hi) x)) x)) '((-1 0) 0))

  (test (let ((x 0)) (define-macro* (hi (a (let () (set! x (+ x 1)) x))) `(let ((x -1)) (+ x ,a))) (list (hi) x)) '(-1 0)) 
  (test (let ((x 0)) (let ((x -1)) (+ x (let () (set! x (+ x 1)) x)))) -1)
  (test (let ((x 0)) (define-macro (hi a) `(let ((x -1)) (+ x ,a))) (list (hi (let () (set! x (+ x 1)) x)) x)) '(-1 0))
  (test (let () (define-macro (hi a) `(let ((b 1)) (+ ,a b))) (hi (+ 1 b))) 3)
  (test (let ((b 321)) (define-macro (hi a) `(let ((b 1)) (+ ,a b))) (hi b)) 2)
  (test (let ((b 321)) (define-macro* (hi (a b)) `(let ((b 1)) (+ ,a b))) (hi b)) 2)
  (test (let ((b 321)) (define-macro* (hi (a b)) `(let ((b 1)) (+ ,a b))) (hi)) 2) ; ???
  (test (let ((x 0)) (define-macro* (hi (a (let () (set! x (+ x 1)) x))) `(+ ,a ,a)) (hi)) 3) ; ??? -- default val is substituted directly
  ;; but (let () (define-macro* (hi a (b a)) `(+ ,a ,b)) (hi 1)) -> "a: unbound variable" -- "a" itself is substituted, but does not exist at expansion(?)

  ;; can we implement bacros via define-macro* default args?
  ;;  I don't think so -- macro arguments can't be evaluated in that environment because 
  ;;  only the default values have been set (on the previous parameters).

  ;; bacro ignores closure in expansion but macro does not:
  (test (let ((x 1)) (define-macro (ho a) `(+ ,x ,a)) (let ((x 32)) (ho 3))) 4)
  (test (let ((x 1)) (define-macro (ho a) `(+ x ,a)) (let ((x 32)) (ho 3))) 35)
  (test (let ((x 1)) (define-bacro (ho a) `(+ ,x ,a)) (let ((x 32)) (ho 3))) 35)
  (test (let ((x 1)) (define-bacro (ho a) `(+ x ,a)) (let ((x 32)) (ho 3))) 35)

  (test (let ((x 1)) (define-macro* (ho (a x)) `(+ ,x ,a)) (let ((x 32)) (ho))) 33)
  (test (let ((x 1)) (define-macro* (ho (a x)) `(+ x ,a)) (let ((x 32)) (ho))) 64)
  (test (let ((x 1)) (define-bacro* (ho (a x)) `(+ x ,a)) (let ((x 32)) (ho))) 64)
  (test (let ((x 1)) (define-bacro* (ho (a x)) `(+ ,x ,a)) (let ((x 32)) (ho))) 64)

  (test (let ((x 1)) (define-macro* (ho (a (+ x 0))) `(+ ,x ,a)) (let ((x 32)) (ho))) 33)  ;; (+ 32 (+ x 0)) !?! macroexpand is confusing?
  (test (let ((x 1)) (define-macro* (ho (a (+ x 0))) `(+ x ,a)) (let ((x 32)) (ho))) 64)   ;; (+ x (+ x 0))
  (test (let ((x 1)) (define-bacro* (ho (a (+ x 0))) `(+ x ,a)) (let ((x 32)) (ho))) 64 )
  (test (let ((x 1)) (define-bacro* (ho (a (+ x 0))) `(+ ,x ,a)) (let ((x 32)) (ho))) 64 )

  (test (let () (define-macro* (hi :rest a) `(list ,@a)) (hi)) '())
  (test (let () (define-macro* (hi :rest a) `(list ,@a)) (hi 1)) '(1))
  (test (let () (define-macro* (hi :rest a) `(list ,@a)) (hi 1 2 3)) '(1 2 3))
  (test (let () (define-macro* (hi a :rest b) `(list ,a ,@b)) (hi 1 2 3)) '(1 2 3))
  (test (let () (define-macro* (hi (a 1) :rest b) `(list ,a ,@b)) (hi)) '(1))
  (test (let () (define-macro* (hi (a 1) :rest b) `(list ,a ,@b)) (hi 2)) '(2))
  (test (let () (define-macro* (hi (a 1) :rest b) `(list ,a ,@b)) (hi :a 2)) '(2))
  (test (let () (define-macro* (hi (a 1) :rest b :allow-other-keys) `(list ,a ,@b)) (hi :a 2 :b 3)) '(2 :b 3))

;  (test (let () (define-macro ,@a 23)) 'error)
;  (test (let () (define-macro ,a 23)) 'error)
; maybe this isn't right

  (test ((lambda* ((a 1) :allow-other-keys) a) :b 1 :a 3) 3)
  (test (let () (define-macro* (hi (a 1) :allow-other-keys) `(list ,a)) (hi :b 2 :a 3)) '(3))
  (test ((lambda* ((a 1) :rest b :allow-other-keys) b) :c 1 :a 3) '())
  (test ((lambda* ((a 1) :rest b :allow-other-keys) b) :b 1 :a 3) 'error) 
  ;; that is the rest arg is not settable via a keyword and it's an error to try to
  ;;   do so, even if :allow-other-keys -- ??

  (test (let ((x 1)) (define* (hi (a x)) a) (let ((x 32)) (hi))) 1)
  (test (let ((x 1)) (define* (hi (a (+ x 0))) a) (let ((x 32)) (hi))) 1)
  (test (let ((x 1)) (define* (hi (a (+ x "hi"))) a) (let ((x 32)) (hi))) 'error)
  (test (let ((x 1)) (define-macro* (ho (a (+ x "hi"))) `(+ x ,a)) (let ((x 32)) (ho))) 'error)

  ;; define-macro* default arg expr does not see definition-time closure:
  (test (let ((mac #f))
	  (let ((a 32))
	    (define-macro* (hi (b (+ a 1))) `(+ ,b 2))
	    (set! mac hi))
	  (mac))
	'error) ; ";a: unbound variable, line 4"

  (test ((lambda* ((x (let ()
			(define-macro (hi a)
			  `(+ ,a 1))
			(hi 2))))
		  (+ x 1)))
	4)
  (test (let () 
	  (define-macro* (hi (x (let ()
				  (define-macro (hi a)
				    `(+ ,a 1))
				  (hi 2))))
	    `(+ ,x 1)) 
	  (hi))
	4)

  (test (let () (define* (hi b) b) (procedure? hi)) #t)
  
  (test (let ()
	  (define (hi a) a)
	  (let ((tag (catch #t
			    (lambda () (hi 1 2 3))
			    (lambda args (car args)))))
	    (eq? tag 'wrong-number-of-args)))
	#t)
  
  (test (let ()
	  (define (hi a) a)
	  (let ((tag (catch #t
			    (lambda () (hi))
			    (lambda args (car args)))))
	    (eq? tag 'wrong-number-of-args)))
	#t)
  
  (test (let ()
	  (define* (hi a) a)
	  (let ((tag (catch #t
			    (lambda () (hi 1 2 3))
			    (lambda args (car args)))))
	    (eq? tag 'wrong-number-of-args)))
	#t)

  (test (let () (define (hi :a) :a) (hi 1)) 'error)
  (test (let () (define* (hi :a) :a) (hi 1)) 'error)
  (test (let () (define* (hi (:a 2)) a) (hi 1)) 'error)
  (test (let () (define* (hi (a 1) (:a 2)) a) (hi 1)) 'error)
  (test (let () (define* (hi (pi 1)) pi) (hi 2)) 'error)
  (test (let () (define* (hi (:b 1) (:a 2)) a) (hi)) 'error)

  (test (let () (define* (hi (a 1) (a 2)) a) (hi 2)) 'error)
  (test (let () (define (hi a a) a) (hi 1 2)) 'error)
  (test (let () (define hi (lambda (a a) a)) (hi 1 1)) 'error)
  (test (let () (define hi (lambda* ((a 1) (a 2)) a)) (hi 1 2)) 'error)
  (test (let () (define (hi (a 1)) a) (hi 1)) 'error)

  (let () 
    (define* (hi (a #2d((1 2) (3 4)))) (a 1 0))
    (test (hi) 3)
    (test (hi #2d((7 8) (9 10))) 9))

  (let () (define* (f :rest a) a) (test (f :a 1) '(:a 1)))
  (let () (define* (f :rest a :rest b) (list a b)) (test (f :a 1 :b 2) '((:a 1 :b 2) (1 :b 2))))

  (test (lambda :hi 1) 'error)
  (test (lambda (:hi) 1) 'error)
  (test (lambda (:hi . :hi) 1) 'error)
  (test (lambda (i . i) 1 . 2) 'error)
  (test (lambda (i i i i) (i)) 'error)
  (test (lambda "hi" 1) 'error)
  (test (lambda* ((i 1) i i) i) 'error)
  (test (lambda* ((a 1 2)) a) 'error)
  (test (lambda* ((a . 1)) a) 'error)
  (test (lambda* ((0.0 1)) 0.0) 'error)

  (test ((lambda* ((b 3) :rest x (c 1)) (list b c x)) 32) '(32 1 ()))
  (test ((lambda* ((b 3) :rest x (c 1)) (list b c x)) 1 2 3 4 5) '(1 3 (2 3 4 5)))
  ;; these are in s7.html
  (test ((lambda* ((a 1) :rest b :rest c) (list a b c)) 1 2 3 4 5) '(1 (2 3 4 5) (3 4 5)))

  (test (let () (define-macro (hi a a) `(+ ,a 1)) (hi 1 2)) 'error)

  ;;; procedure-arity
  (test (procedure-arity car) '(1 0 #f))
  (test (procedure-arity 'car) '(1 0 #f))
  (test (procedure-arity +) '(0 0 #t))
  (test (procedure-arity '+) '(0 0 #t))
  (test (procedure-arity log) '(1 1 #f))
  (test (procedure-arity '/) '(1 0 #t))
  (test (procedure-arity catch) '(3 0 #f))
  (test (procedure-arity) 'error)
  (test (procedure-arity abs abs) 'error)
  (test (procedure-arity "hi") 'error)
;;;  (test (let () (set! (car (procedure-arity abs)) 0) (procedure-arity abs)) '(1 0 #f))

  (test (procedure-arity vector-set!) '(3 0 #t))
  (test (let ((hi (lambda () 1))) (procedure-arity hi)) '(0 0 #f))
  (test (let ((hi (lambda (a) 1))) (procedure-arity hi)) '(1 0 #f))
  (test (let ((hi (lambda (a b) 1))) (procedure-arity hi)) '(2 0 #f))
  (test (let ((hi (lambda (a . b) 1))) (procedure-arity hi)) '(1 0 #t))
  (test (let ((hi (lambda a 1))) (procedure-arity hi)) '(0 0 #t))
  
  (test (let () (define (hi) 1) (procedure-arity hi)) '(0 0 #f))
  (test (let () (define (hi a) a) (procedure-arity hi)) '(1 0 #f))
  (test (let () (define* (hi a) a) (procedure-arity hi)) '(0 1 #f))
  (test (let () (define* (hi a . b) a) (procedure-arity hi)) '(0 1 #t))
  (test (let () (define* (hi (a 1) (b 2)) a) (procedure-arity hi)) '(0 2 #f))
  (test (let ((hi (lambda* (a) 1))) (procedure-arity hi)) '(0 1 #f))
  (test (call/cc (lambda (func) (procedure-arity func))) '(0 0 #t))

  (test (procedure-arity (lambda* (a :rest b) a)) '(0 1 #t))
  (test (procedure-arity (lambda* (:optional a :rest b) a)) '(0 1 #t))
  (test (procedure-arity (lambda* (:optional a :key b :rest c) a)) '(0 2 #t))
  (test (procedure-arity (lambda* (:optional a b) a)) '(0 2 #f))
  (test (procedure-arity (lambda* (:rest args) args)) '(0 0 #t))
  (test (procedure-arity (lambda* (a :optional b . c) a)) '(0 2 #t))
  (test (procedure-arity (lambda* (:rest a . b) a)) '(0 0 #t))
  (test (procedure-arity (lambda* (:key :optional a) a)) '(0 1 #f))
  (test (procedure-arity (lambda* a a)) '(0 0 #t))
  (test (let () (define-macro (hi a) `(+ ,a 1)) (procedure-arity hi)) 'error)
  (test (procedure-arity (make-procedure-with-setter (lambda (a) a) (lambda (a b) a))) '(1 0 #f))
  (test (procedure-arity (make-procedure-with-setter (lambda (a . b) a) (lambda (a b) a))) '(1 0 #t))
  (test (procedure-arity (make-procedure-with-setter (lambda* (a :optional b) a) (lambda (a b) a))) '(0 2 #f))
    
  (for-each
   (lambda (arg)
     (test (procedure-arity arg) 'error))
   (list -1 #\a #f _ht_ 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

  (define (for-each-subset func args)
    (let* ((arity (procedure-arity func))
	   (min-args (car arity))
	   (max-args (if (caddr arity)
			 (length args)
			 (+ min-args (cadr arity))))
	   (subsets '()))
      
      (define (subset source dest len)
	(if (null? source)
	    (begin
	      (if (member dest subsets)
		  (format #t ";got ~S twice in for-each-subset: ~S~%" dest args))
	      (set! subsets (cons dest subsets))
	      (if (<= min-args len max-args)
		  (apply func dest)))
	    (begin
	      (subset (cdr source) (cons (car source) dest) (+ len 1))
	      (subset (cdr source) dest len))))
      
      (subset args '() 0)))

  (test (let ((ctr 0))
	  (for-each-subset (lambda args (set! ctr (+ ctr 1))) '(1 2 3 4))
	  ctr)
	16)
  (test (let ((ctr 0))
	  (for-each-subset (lambda (arg) (set! ctr (+ ctr 1))) '(1 2 3 4))
	  ctr)
	4)
  (test (let ((ctr 0))
	  (for-each-subset (lambda (arg1 arg2 arg3) (set! ctr (+ ctr 1))) '(1 2 3 4))
	  ctr)
	4)
  (test (let ((ctr 0))
	  (for-each-subset (lambda* (arg1 arg2 arg3) (set! ctr (+ ctr 1))) '(1 2 3 4))
	  ctr)
	15)
  (test (let ((ctr 0))
	  (for-each-subset (lambda () (set! ctr (+ ctr 1))) '(1 2 3 4))
	  ctr)
	1)

  (define (snarf func lst)
    "(snarf func lst) repeatedly applies func to as many elements of lst as func can take"
    (let ((arity (procedure-arity func)))
      (if (caddr arity)
	  (apply func lst)
	  (let ((n (+ (car arity) (cadr arity)))
		(lst-len (length lst)))
	    (if (< lst-len (car arity))
		(error 'wrong-number-of-args ";snarf func requires ~A args, but got ~A, ~A" (car arity) lst-len lst)
		(if (<= lst-len n)
		    (apply func lst)
		    (if (not (zero? (modulo (length lst) n)))
			(error 'wrong-number-of-args ";snarf will take ~A args at a time, but got ~A in ~A" n lst-len lst)
			;; ideally this would accept partial lists (i.e. left-over < req),
			;;   but then we also need to notice that case in the list-tail below
			(let ()
			  
			  (define (snarf-1 len f args)
			    (if (not (null? args))
				(let* ((last (list-tail args (- len 1)))
				       (rest (cdr last)))
				  (dynamic-wind
				      (lambda ()
					(set! (cdr last) '()))
				      (lambda ()
					(apply func args))
				      (lambda ()
					(set! (cdr last) rest)))
				  (snarf-1 len f rest))))
			  
			  (snarf-1 n func lst)))))))))

  (test (let ((lst '(1 2 3 4))) (catch #t (lambda () (snarf (lambda (a b) (format #t "~A ~A~%" a b c)) lst)) (lambda args 'error)) lst) '(1 2 3 4))
  (test (snarf (lambda (a b) (format #t "~A ~A~%" a b)) '(1 2 3 4 5)) 'error)
  (test (snarf (lambda (a b) (format #t "~A ~A~%" a b)) '(1)) 'error)
  (test (let ((x 0)) (snarf (lambda (a) (set! x (+ x a))) '(1 2 3)) x) 6)
  (test (let ((x 0)) (snarf (lambda (a b) (set! x (+ x a b))) '(1 2 3 4)) x) 10)
  (test (let ((x 0)) (snarf (lambda* (a b) (set! x (+ x a b))) '(1 2 3 4)) x) 10)
  (test (let ((x 0)) (snarf (lambda a (set! x (apply + a))) '(1 2 3 4)) x) 10)
  (test (let ((x 0)) (snarf (lambda* (a b (c 0)) (set! x (+ x a b c))) '(1 2)) x) 3)

  (test (let ((c 1)) 
	  (define* (a :optional (b c)) b) 
	  (set! c 2) 
	  (a))
	2)
  
  (test (let ((c 1)) 
	  (define* (a :optional (b c)) b) 
	  (let ((c 32)) 
	    (a)))
	1)
  
  (test (let ((c 1)) 
	  (define* (a (b (+ c 1))) b) 
	  (set! c 2) 
	  (a))
	3)
  
  (test (let ((c 1))
	  (define* (a (b (+ c 1))) b)
	  (set! c 2)
	  (let ((c 123))
	    (a)))
	3)
  
  (test (let* ((cc 1)
	       (c (lambda () (set! cc (+ cc 1)) cc)))
	  (define* (a (b (c))) b)
	  (list cc (a) cc))
	(list 1 2 2))

  (for-each
   (lambda (arg)
     (test (trace arg) 'error)
     (test (untrace arg) 'error))
   (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

  (let ((sum 0))
    (define (hiho a b c) (* a b c))
    (set! *trace-hook* (lambda (f args) (set! sum (apply + args))))
    (trace hiho)
    (hiho 2 3 4)
    (untrace hiho)
    (set! *trace-hook* '())
    (test sum 9))

  (test (hook-arity *trace-hook*) '(2 0 #f))
  (test (hook-documentation *trace-hook*) "*trace-hook* customizes tracing.  Its functions take 2 arguments, the function being traced, and its current arguments.")
  (test (hook-functions *trace-hook*) '())

  (let ((sum 0))
    (define (hiho a b c) (* a b c))
    (set! *trace-hook* (list (lambda (f args) (set! sum (apply + args)))))
    (trace hiho)
    (hiho 2 3 4)
    (untrace hiho)
    (set! *trace-hook* '())
    (test sum 9))

  (for-each
   (lambda (arg)
     (test (set! *trace-hook* arg) 'error)
     (test (set! *unbound-variable-hook* arg) 'error)
     (test (set! *error-hook* arg) 'error)
     (test (set! *load-hook* arg) 'error)

     (test (set! (hook-functions *trace-hook*) arg) 'error)
     (test (set! (hook-functions *unbound-variable-hook*) arg) 'error)
     (test (set! (hook-functions *error-hook*) arg) 'error)
     (test (set! (hook-functions *load-hook*) arg) 'error)

     (test (set! (hook-functions *trace-hook*) (list arg)) 'error)
     (test (set! (hook-functions *unbound-variable-hook*) (list arg)) 'error)
     (test (set! (hook-functions *error-hook*) (list arg)) 'error)
     (test (set! (hook-functions *load-hook*) (list arg)) 'error)
     )
   (list -1 #\a '#(1 2 3) 3.14 3/4 1.0+1.0i 'hi :hi #<eof> #(1 2 3) '#(()) "hi" '(1 . 2) '(1 2 3)))

  (for-each
   (lambda (arg)
     (eval-string (format #f "(define (func) ~S)" arg))
     (let ((source (procedure-source func)))
       (let ((val (func)))
	 (test val arg))))
   (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) ':hi "hi"))
  
  (test (string=? (let () (define (hi) "this is a string" 1) (procedure-documentation hi)) "this is a string") #t)
  (test (string=? (let () (define (hi) "this is a string" 1) (help hi)) "this is a string") #t)
  (test (string=? (let () (define (hi) "this is a string") (procedure-documentation hi)) "this is a string") #t)
  (test (string=? (let () (define (hi) "this is a string") (hi)) "this is a string") #t)
  (test (string=? (let () (define* (hi (a "a string")) a) (procedure-documentation hi)) "") #t)
  (test (string=? (let () (define* (hi (a "a string")) "another string" a) (procedure-documentation hi)) "another string") #t)
  (test (string=? (let () (define (hi a) "hi doc" (define (ho b) "ho doc" b) (ho a)) (procedure-documentation hi)) "hi doc") #t)
  (test (set! (procedure-documentation abs) "X the unknown") 'error)
  (test (let ((str (procedure-documentation abs))) (set! ((procedure-documentation abs) 1) #\x) (equal? str (procedure-documentation abs))) #t)
  (test (let ((str (procedure-documentation abs))) (fill! (procedure-documentation abs) #\x) (equal? str (procedure-documentation abs))) #t)
  (let ()
    (define-macro (amac a) "this is a string" `(+ ,a 1))
    (test (procedure-documentation amac) "this is a string"))

  (for-each
   (lambda (arg)
     (test (procedure-environment arg) 'error))
   (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

  (test (let ()
	  (define (hi a)
	    (let ((func (symbol->value '__func__ (procedure-environment hi))))
	      (list (if (symbol? func) func (car func))
		    a)))
	  (hi 1))
	(list 'hi 1))

  (test (let ()
	  (define hi (let ((a 32)) 
		       (lambda (b) 
			 (+ a b))))
	  (define ho (with-environment 
		      (procedure-environment hi) 
		      (lambda (b) 
			(+ a b))))
	  (list (hi 1) (ho 1)))
	(list 33 33))

  (test (let ()
	  (define (hi a) (+ a 1))
	  (with-environment (procedure-environment hi) 
            ((eval (procedure-source hi)) 2)))
	3)

  (let ()
    (define (where-is func)
      (let* ((env (procedure-environment func))
	     (addr (symbol->value '__func__ env)))
	(if (not (pair? addr))
	    ""
	    (list (format #f "~A[~D]" (cadr addr) (caddr addr))
		  addr))))
    (let ((e (where-is ok?)))
      (test (and (pair? (cadr e))
		 (< ((cadr e) 2) 100)) ; this depends on where ok? is in this file
	    #t)
      (test (and (pair? (cadr e))
		 (string=? (symbol->string (car (cadr e))) "ok?"))
	    #t)
      (test (and (pair? (cadr e))
		 (let ((name (cadr (cadr e))))
		   (and (string? name)
			(call-with-exit
			 (lambda (oops)
			   (let ((len (length name)))
			     (do ((i 0 (+ i 1)))
				 ((= i len) #t)
			       (if (and (not (char-alphabetic? (name i)))
					(not (char=? (name i) #\/))
					(not (char=? (name i) #\\))
					(not (char=? (name i) #\.))
					(not (char=? (name i) #\-))
					(not (char-numeric? (name i))))
				   (begin
				     (format #t "ok? file name: ~S~%" name)
				     (oops #f))))))))))
	    #t)))

  (let ()
    (define-macro (window func beg end . body)
      `(call-with-exit
	(lambda (quit)
	  (do ((notes ',body (cdr notes)))
	      ((null? notes))
	    (let* ((note (car notes))
		   (note-beg (cadr note)))
	      (if (<= ,beg note-beg)
		  (if (> note-beg (+ ,beg ,end))
		      (quit)
		      (,func note))))))))
    
    (test 
     (let ((n 0))
       (window (lambda (a-note) (set! n (+ n 1))) 0 1 
	       (fm-violin 0 1 440 .1) 
	       (fm-violin .5 1 550 .1) 
	       (fm-violin 3 1 330 .1))
       n)
     2)
    
    (test 
     (let ((notes 0)
	   (env #f))
       (set! env (current-environment))
       (window (with-environment env (lambda (n) (set! notes (+ notes 1)))) 0 1 
	       (fm-violin 0 1 440 .1) 
	       (fm-violin .5 1 550 .1) 
	       (fm-violin 3 1 330 .1))
       notes)
     2))

  (test (let ()
	  (define-macro (window func beg end . body)
	    `(let ((e (current-environment)))
	       (call-with-exit
		(lambda (quit)
		  (do ((notes ',body (cdr notes)))
		      ((null? notes))
		    (let* ((note (car notes))
			   (note-beg (cadr note)))
		      (if (<= ,beg note-beg)
			  (if (> note-beg (+ ,beg ,end))
			      (quit)
			      ((with-environment e ,func) note)))))))))
	  
	  (let ((notes 0))
	    (window (lambda (n) (set! notes (+ notes 1))) 0 1 
		    (fm-violin 0 1 440 .1) 
		    (fm-violin .5 1 550 .1) 
		    (fm-violin 3 1 330 .1))
	    notes))
	2)


  (for-each
   (lambda (arg)
     (test (continuation? arg) #f))
   (list -1 #\a 1 #f _ht_ '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2) "hi" (lambda () 1)))
  
  (test (let ((cont #f)) 
	  (and (call/cc (lambda (x) (set! cont x) (continuation? x)))
	       (continuation? cont)))
	#t)
  (test (let ((cont #f)) 
	  (or (call-with-exit (lambda (x) (set! cont x) (continuation? x)))
	      (continuation? cont)))
	#f) ; x is not a continuation
	
  (test (continuation?) 'error)
  (test (continuation? 1 2) 'error)

  
  (test (string? (s7-version)) #t)
  (test (s7-version 1) 'error)

;;; eval
;;; eval-string

  (test (eval-string "(+ 1 2)") 3)
  (test (eval '(+ 1 2)) 3)
  (test (eval `(+ 1 (eval `(* 2 3)))) 7)
  (test (eval `(+ 1 (eval-string "(* 2 3)"))) 7)
  (test (eval-string "(+ 1 (eval-string \"(* 2 3)\"))") 7)
  (test (eval `(+ 1 2 . 3)) 'error)
  (test (eval-string) 'error)
  (test (eval) 'error)
  (test (eval-string "") #f)
  (test (eval ()) ())
  (test (eval-string "1" () ()) 'error)
  (test (eval () () ()) 'error)
  (test (eval "1") "1")
  (test (eval-string #t) 'error)
  (test (eval #(+ 1 2)) #(+ 1 2))

  (test (eval '(begin (define __eval_var__ 1) __eval_var__) (global-environment)) 1)
  (test (let () __eval_var__) 1)
  (test (eval-string "(begin (define __eval_var1__ 12) __eval_var1__)" (global-environment)) 12)
  (test (let () __eval_var1__) 12)
  (test (let () (eval '(begin (define __eval_var2__ 123) __eval_var__) (current-environment)) __eval_var2__) 123)
  (test (let () __eval_var2__) 'error)


  (test (apply "hi" 1 ()) #\i)
  (test (eval ("hi" 1)) #\i)
  (test (apply + 1 1 (cons 1 (quote ()))) 3)
  (test (eq? (eval (quote (quote ()))) ()) #t)
  (test (apply (cons (quote cons) (cons 1 (quote ((quote ()))))) 1 ()) 1) ; essentially ((list 'cons 1 ...) 1) => 1
  (test (eval ((cons (quote cons) (cons 1 (quote ((quote ()))))) 1)) 1)
  (test (eval (eval (list '+ 1 2))) 3)

  (test (eval if) if)
  (test (eval quote) quote)
  (test (eval (eval (list define* #(1)))) 'error)
  (test (eval (eval (list lambda* ()))) 'error)
  (test (eval (eval (list letrec "hi"))) 'error)
  (test (eval (eval (cons defmacro 1))) 'error)
  (test (eval (eval (cons quote "hi"))) 'error)
  (test (eval (eval (list and "hi"))) "hi")

  (test (apply + (+ 1) ()) 1)
  (test (apply #(1) (+) ()) 1)
  (test (apply + (+) ()) 0)
  (test (eval #()) #())
  (test (apply (lambda () #f)) #f)
  (test (eval '(if #f #f)) (if #f #f))
  (test (let ((ho 32)) (symbol? (eval (eval (eval (eval '''''ho)))))) #t)
  (test (eval '(case 0 ((1) 2) ((0) 1))) 1)
  (test (eval '(cond ((= 1 2) 3) (#t 4))) 4)

  (test (eval-string (string-append "(list 1 2 3)" (string #\newline) (string #\newline))) (list 1 2 3))
  (eval-string (string-append "(define evalstr_1 32)" (string #\newline) "(define evalstr_2 2)"))
  (test (eval-string "(+ evalstr_1 evalstr_2)") 34)
  (eval-string (string-append "(set! evalstr_1 3)" "(set! evalstr_2 12)"))
  (test (eval-string "(+ evalstr_1 evalstr_2)") 15)
  
  (test (+ (eval `(values 1 2 3)) 4) 10)
  (test (+ (eval-string "(values 1 2 3)") 4) 10)
  (test (+ 1 (eval-string "(+ 2 3)") 4) 10)
  (test ((eval-string "(lambda (a) (+ a 1))") 2) 3)
  (test (eval ((eval-string "(lambda (a) (list '+ a 1))") 2)) 3)
  (test (eval-string "(+ 1 (eval (list '+ 1 2)))") 4)

  (num-test (eval-string "\ +\ \ 1") 1)
  (num-test (eval-string "'\ `\ 1") 1)
  (num-test (eval-string "\ 1\ +i") 1+1i)
  (num-test (eval-string "\x32\x37/\x37\ ") 27/7)
  (num-test (eval-string "\ '-1\ ") -1)
  (num-test (eval-string "\ .\ 10") 0.1)
  (num-test (eval-string "#\ i\ -\x34") -4.0)
  (num-test (eval-string "1\ 1.\x37\ ") 11.7)
  (num-test (eval-string "1/\ \ 1\x30") 1/10)
  (num-test (eval-string "#\ xe\ \x36\ ") 230)
  (num-test (eval-string "\ \ \x35\ .  ") 5.0)
  (num-test (eval-string "#x01/\ \x34") 1/4)
  (num-test (eval-string "-\ 1\ .\x35\ e\ 0") -1.5)

  (for-each
   (lambda (arg)
     (test (eval-string arg) 'error))
   (list -1 0 1 512 #\a '#(1 2 3) 3.14 2/3 1.5+0.3i 1+i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2) (lambda () 1)))
  (for-each
   (lambda (arg)
     (test (eval-string "(+ 1 2)" arg) 'error))
   (list -1 0 1 512 #\a '#(1 2 3) 3.14 2/3 1.5+0.3i 1+i 'hi abs "hi" '#(()) (lambda () 1)))

  
  (test (string=? (procedure-documentation abs) "(abs x) returns the absolute value of the real number x") #t)
  (test (string=? (help abs) "(abs x) returns the absolute value of the real number x") #t)
  (test (string=? (procedure-documentation 'abs) "(abs x) returns the absolute value of the real number x") #t)
  (test (let ((hi (lambda (x) "this is a test" (+ x 1)))) 
	  (list (hi 1) (procedure-documentation hi)))
	(list 2 "this is a test"))
  (test (procedure-documentation (lambda* (a b) "docs" a)) "docs")
  (test (procedure-documentation (lambda* (a b) "" a)) "")
  (test (procedure-documentation (lambda* (a b) a)) "")
  (test (procedure-documentation (call-with-exit (lambda (c) c))) "")
  (test (procedure-documentation (call/cc (lambda (c) c))) "")
  (test (procedure-arity (call-with-exit (lambda (c) c))) '(0 0 #t))
  (test (procedure-arity (call/cc (lambda (c) c))) '(0 0 #t))
  
  
  (if (not (provided? 'snd))
      (for-each
       (lambda (arg)
	 (test (procedure-documentation arg) 'error)
	 (test (help arg) #f))
       (list -1 #\a #f _ht_ 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi")))
  
  (test (let ((hi (lambda (x) (+ x 1)))) (procedure-source hi)) '(lambda (x) (+ x 1)))
  ;; (test (procedure-with-setter? symbol-access) #t)
  (test (procedure-documentation symbol-access) "(symbol-access sym) is a procedure-with-setter that adds or removes controls on how a symbol accesses its current binding.")
  
  (for-each
   (lambda (arg)
     (test (procedure-source arg) 'error))
   (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

  (test (procedure-documentation) 'error)
  (test (procedure-documentation abs abs) 'error)
  (test (procedure-arity) 'error)
  (test (procedure-arity abs abs) 'error)
  (test (procedure-source) 'error)
  (test (procedure-source abs abs) 'error)
  (test (procedure-source quasiquote) 'error)
  (test (let () (define-macro (hi a) `(+ 1 ,a)) (cadr (caddr (procedure-source hi)))) '(lambda (a) ({list} '+ 1 a)))

  (let ((p (make-procedure-with-setter (lambda (a) (+ a 1)) (lambda (a b) (+ a b)))))
    (test (object->string (procedure-source p)) "(lambda (a) (+ a 1))")
    (let ((p1 p)
	  (p2 (make-procedure-with-setter (lambda (a) "pws doc" (+ a 1)) (lambda (a b) (+ a b)))))
      (test (equal? p p1) #t)
      (test (equal? p1 p2) #f)
      (test (procedure-documentation p2) "pws doc")
      (test (apply p2 '(2)) 3)))

  (test (procedure-documentation hook-functions) "(hook-functions hook) returns the list of functions on the hook. It is settable;  (set! (hook-functions hook) (cons func (hook-functions hook))) adds func to the current list.")

  (test (make-list 0) '())
  (test (make-list 0 123) '())
  (test (make-list 1) '(#f))
  (test (make-list 1 123) '(123))
  (test (make-list 1 '()) '(()))
  (test (make-list 2) '(#f #f))
  (test (make-list 2 1) '(1 1))
  (test (make-list 2/1 1) '(1 1))
  (test (make-list 2 (make-list 1 1)) '((1) (1)))
  (test (make-list -1) 'error)
  (test (make-list -0) '())
  (test (make-list most-negative-fixnum) 'error)
  (test (make-list most-positive-fixnum) 'error)
  
  (for-each
   (lambda (arg)
     (test (make-list arg) 'error))
   (list #\a '#(1 2 3) 3.14 3/4 1.0+1.0i 0.0 1.0 '() #t 'hi '#(()) (list 1 2 3) '(1 . 2) "hi" (- (real-part (log 0.0)))))

  (for-each
   (lambda (arg)
     (test ((make-list 1 arg) 0) arg))
   (list #\a '#(1 2 3) 3.14 3/4 1.0+1.0i '() #f 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

  (test (make-list) 'error)
  (test (make-list 1 2 3) 'error)
  (test (let ((lst (make-list 2 (make-list 1 0)))) (eq? (lst 0) (lst 1))) #t)

  
  (test (let () (defmacro hiho (a) `(+ ,a 1)) (hiho 3)) 4)
  (test (let () (defmacro hiho () `(+ 3 1)) (hiho)) 4)
  (test (let () (defmacro hiho () `(+ 3 1)) (hiho 1)) 'error)
  (test (let () (defmacro hi (a) `(+ ,@a)) (hi (1 2 3))) 6)
  (test (let () (defmacro hi (a) `(+ ,a 1) #f) (hi 2)) #f)
  
  (test (let () (define-macro (hiho a) `(+ ,a 1)) (hiho 3)) 4)
  (test (let () (define-macro (hiho) `(+ 3 1)) (hiho)) 4)
  (test (let () (define-macro (hiho) `(+ 3 1)) (hiho 1)) 'error)
  (test (let () (define-macro (hi a) `(+ ,@a)) (hi (1 2 3))) 6)
  (test (let () (define-macro (hi a) `(+ ,a 1) #f) (hi 2)) #f)
  (test (let () (define-macro (mac1 a) `',a) (equal? (mac1 (+ 1 2)) '(+ 1 2))) #t)
  (test (let () (define-macro (hi . a) `,@a) (hi 1)) 1)
  
  (test (let () (defmacro hi (a) `(+ , a 1)) (hi 1)) 2)
  (test (let () (defmacro hi (a) `(eval `(+ ,,a 1))) (hi 1)) 2)
  (test (let () (defmacro hi (a) `(eval (let ((a 12)) `(+ ,,a 1)))) (hi 1)) 2)
  (test (let () (defmacro hi (a) `(eval (let ((a 12)) `(+ ,a 1)))) (hi 1)) 13)
  (test (let () (defmacro hi (a) `(eval (let ((a 12)) `(let ((a 100)) (+ ,a 1))))) (hi 1)) 13)
  (test (let () (defmacro hi (a) `(eval (let ((a 12)) `(let ((a 100)) (+ a 1))))) (hi 1)) 101)
  
  (test (let () (defmacro hi (q) ``(,,q)) (hi (* 2 3))) '(6))
  (test (let () (defmacro hi (q) `(let ((q 32)) `(,,q))) (hi (* 2 3))) '(6))
  (test (let () (defmacro hi (q) `(let ((q 32)) `(,q))) (hi (* 2 3))) '(32))
  (test (let () (defmacro hi (q) `(let () ,@(list q))) (hi (* 2 3))) 6)

  (test (let () (define-macro (tst a) ``(+ 1 ,,a)) (tst 2)) '(+ 1 2))
  (test (let () (define-macro (tst a) ```(+ 1 ,,,a)) (eval (tst 2))) '(+ 1 2))
  (test (let () (define-macro (tst a) ``(+ 1 ,,a)) (tst (+ 2 3))) '(+ 1 5))
  (test (let () (define-macro (tst a) ``(+ 1 ,@,a)) (tst '(2 3))) '(+ 1 2 3))
  (test (let () (define-macro (tst a) ``(+ 1 ,,@a)) (tst (2 3))) '(+ 1 2 3))
  (test (let () (define-macro (tst a) ```(+ 1 ,,,@a)) (eval (tst (2 3)))) '(+ 1 2 3))
  (test (let () (define-macro (tst a) ```(+ 1 ,,@,@a)) (eval (tst ('(2 3))))) '(+ 1 2 3))
  (test (let () (define-macro (tst a) ````(+ 1 ,,,,@a)) (eval (eval (eval (tst (2 3)))))) 6)
  (test (let () (define-macro (tst a) ``(+ 1 ,@,@a)) (tst ('(2 3)))) '(+ 1 2 3))
  (test (let () (define-macro (tst a b) `(+ 1 ,a (apply * `(2 ,,@b)))) (tst 3 (4 5))) 44)
  (test (let () (define-macro (tst . a) `(+ 1 ,@a)) (tst 2 3)) 6)
  (test (let () (define-macro (tst . a) `(+ 1 ,@a (apply * `(2 ,,@a)))) (tst 2 3)) 18)
  (test (let () (define-macro (tst a) ```(+ 1 ,@,@,@a)) (eval (tst ('('(2 3)))))) '(+ 1 2 3))

  (test (let () (define-macro (hi a) `(+ ,a 1)) (procedure? hi)) #f)
  (test (let () (define-macro (hi a) `(let ((@ 32)) (+ @ ,a))) (hi @)) 64)
  (test (let () (define-macro (hi @) `(+ 1 ,@@)) (hi (2 3))) 6) ; ,@ is ambiguous
  (test (let () (define-macro (tst a) `(+ 1 (if (> ,a 0) (tst (- ,a 1)) 0))) (tst 3)) 4)
  (test (let () (define-macro (hi a) (if (list? a) `(+ 1 ,@a) `(+ 1 ,a))) (* (hi 1) (hi (2 3)))) 12)

  (test (let ((x 1)) (eval `(+ 3 ,x))) 4)
  (test (let ((x 1)) (eval (eval `(let ((x 2)) `(+ 3 ,x ,,x))))) 6)
  (test (let ((x 1)) (eval (eval (eval `(let ((x 2)) `(let ((x 3)) `(+ 10 ,x ,,x ,,,x))))))) 16)
  (test (let ((x 1)) (eval (eval (eval (eval `(let ((x 2)) `(let ((x 3)) `(let ((x 30)) `(+ 100 ,x ,,x ,,,x ,,,,x))))))))) 136)

  (test (let () (define-bacro (hiho a) `(+ ,a 1)) (hiho 3)) 4)
  (test (let () (define-bacro (hiho) `(+ 3 1)) (hiho)) 4)
  (test (let () (define-bacro (hiho) `(+ 3 1)) (hiho 1)) 'error)
  (test (let () (define-bacro (hi a) `(+ ,@a)) (hi (1 2 3))) 6)
  (test (let () (define-bacro (hi a) `(+ ,a 1) #f) (hi 2)) #f)
  (test (let () (define-bacro (mac1 a) `',a) (equal? (mac1 (+ 1 2)) '(+ 1 2))) #t)
  (test (let () (define-bacro (tst a) ``(+ 1 ,,a)) (tst 2)) '(+ 1 2))
  (test (let () (define-bacro (tst a) ```(+ 1 ,,,a)) (eval (tst 2))) '(+ 1 2))
  (test (let () (define-bacro (tst a) ``(+ 1 ,,a)) (tst (+ 2 3))) '(+ 1 5))
  (test (let () (define-bacro (tst a) ``(+ 1 ,@,a)) (tst '(2 3))) '(+ 1 2 3))
  (test (let () (define-bacro (tst a) ``(+ 1 ,,@a)) (tst (2 3))) '(+ 1 2 3))
  (test (let () (define-bacro (tst a) ```(+ 1 ,,,@a)) (eval (tst (2 3)))) '(+ 1 2 3))
  (test (let () (define-bacro (tst a) ```(+ 1 ,,@,@a)) (eval (tst ('(2 3))))) '(+ 1 2 3))
  (test (let () (define-bacro (tst a) ````(+ 1 ,,,,@a)) (eval (eval (eval (tst (2 3)))))) 6)
  (test (let () (define-bacro (tst a) ``(+ 1 ,@,@a)) (tst ('(2 3)))) '(+ 1 2 3))
  (test (let () (define-bacro (tst a b) `(+ 1 ,a (apply * `(2 ,,@b)))) (tst 3 (4 5))) 44)
  (test (let () (define-bacro (tst . a) `(+ 1 ,@a)) (tst 2 3)) 6)
  (test (let () (define-bacro (tst . a) `(+ 1 ,@a (apply * `(2 ,,@a)))) (tst 2 3)) 18)
  (test (let () (define-bacro (tst a) ```(+ 1 ,@,@,@a)) (eval (tst ('('(2 3)))))) '(+ 1 2 3))
  (test (let () (define-bacro (hi a) `(+ ,a 1)) (procedure? hi)) #f)
  (test (let () (define-bacro (hi a) `(let ((@ 32)) (+ @ ,a))) (hi @)) 64)
  (test (let () (define-bacro (hi @) `(+ 1 ,@@)) (hi (2 3))) 6) ; ,@ is ambiguous
  (test (let () (define-bacro (tst a) `(+ 1 (if (> ,a 0) (tst (- ,a 1)) 0))) (tst 3)) 4)
  (test (let () (define-bacro (hi a) (if (list? a) `(+ 1 ,@a) `(+ 1 ,a))) (* (hi 1) (hi (2 3)))) 12)

  (test (let () (define-bacro (hiho a) `(+ ,a 1)) (macro? hiho)) #t)
  (test (let () (define-bacro* (hiho (a 1)) `(+ ,a 1)) (macro? hiho)) #t)
  (test (let () (define-macro (hiho a) `(+ ,a 1)) (macro? hiho)) #t)
  (test (let () (define-macro* (hiho (a 1)) `(+ ,a 1)) (macro? hiho)) #t)

  (let ()
    (define-macro (i_ arg)
      `(with-environment (initial-environment) ,arg))

    (define-bacro* (mac b)
      `((i_ let) ((a 12)) 
	((i_ +) a ,(symbol->value b)))) 
    ;; this assumes the 'b' value is a symbol
    ;; (let ((a 1)) (mac (* a 2))) is an error
    ;; use eval, not symbol->value to generalize it

    (test (let ((a 32) 
		(+ -)) 
	    (mac a))
	  44))

#|
  (define-macro (when test . body)
    `((apply lambda (list '(test) '(if test (let () ,@body)))) ,test))

  (define-macro (when test . body)
    `(if ,test (let () ,@body)))

  (define-macro (when test . body)
    `((lambda (test) (if test (let () ,@body))) ,test))

  (define-macro (when test . body)
    `(let ((func (apply lambda `(() ,,@body))))
       (if ,test (func))))
|#

  (test (defmacro) 'error)
  (test (define-macro) 'error)
  (test (defmacro 1 2 3) 'error)
  (test (define-macro (1 2) 3) 'error)
  (test (defmacro a) 'error)
  (test (define-macro (a)) 'error)
  (test (defmacro a (1) 2) 'error)
  (test (define-macro (a 1) 2) 'error)
  (test (defmacro . a) 'error)
  (test (define-macro . a) 'error)
  (test (define :hi 1) 'error)
  (test (define hi: 1) 'error)
  (test (define-macro (:hi a) `(+ ,a 1)) 'error)
  (test (defmacro :hi (a) `(+ ,a 1)) 'error)
  (test (defmacro hi (1 . 2) 1) 'error)
  (test (defmacro hi 1 . 2) 'error)
  (test (defmacro : "" . #(1)) 'error)
  (test (defmacro : #(1) . :) 'error)
  (test (defmacro hi ()) 'error)
  (test (define-macro (mac . 1) 1) 'error)
  (test (define-macro (mac 1) 1) 'error)
  (test (define-macro (a #()) 1) 'error)
  (test (define-macro (i 1) => (j 2)) 'error)
  (test (define hi 1 . 2) 'error)
  (test (defmacro hi hi . hi) 'error)
  (test (define-macro (hi hi) . hi) 'error)
  (test (((lambda () (define-macro (hi a) `(+ 1 ,a)) hi)) 3) 4)

  (test (let () (define-macro (hi a b) `(list ,@a . ,@b)) (hi (1 2) ((2 3)))) '(1 2 2 3))
  (test (let () (define-macro (hi a b) `(list ,@a . ,b)) (hi (1 2) (2 3))) '(1 2 2 3))
  (test (let () (define-macro (hi a b) `(list ,@a ,@b)) (hi (1 2) (2 3))) '(1 2 2 3))

  (let ((vals #(0 0)))
    (let ()
      (define (hi a) (+ 1 a))
      (define (use-hi b) (hi b))
      (set! (vals 0) (use-hi 1))
      (define (hi a) (+ 2 a))
      (set! (vals 1) (use-hi 1))
      (test vals #(2 3)))
    (let ()
      (defmacro hi (a) `(+ 1 ,a))
      (define (use-hi b) (hi b))
      (set! (vals 0) (use-hi 1))
      (defmacro hi (a) `(+ 2 ,a))
      (set! (vals 1) (use-hi 1))
      (test vals #(2 3)))
    (let ()
      (define (use-hi b) (hhi b))
      (defmacro hhi (a) `(+ 1 ,a))
      (set! (vals 0) (use-hi 1))
      (defmacro hhi (a) `(+ 2 ,a))
      (set! (vals 1) (use-hi 1))
      (test vals #(2 3))))

  (test (let ()
	  (define-macro (hanger name-and-args)
	    `(define ,(car name-and-args)
	       (+ ,@(map (lambda (arg) arg) (cdr name-and-args)))))
	  (hanger (hi 1 2 3))
	  hi)
	6)
  (test (let ()
	  (define-macro (hanger name-and-args)
	    `(define-macro (,(car name-and-args))
	       `(+ ,@(map (lambda (arg) arg) (cdr ',name-and-args)))))
	  (hanger (hi 1 2 3))
	  (hi))
	6)

  (let ()
    ;; inspired by Doug Hoyte, "Let Over Lambda"
    (define (mcxr path lst)
      (define (cxr-1 path lst)
	(if (null? path)
	    lst
	    (if (char=? (car path) #\a)
		(cxr-1 (cdr path) (car lst))
		(cxr-1 (cdr path) (cdr lst)))))
      (let ((p (string->list (symbol->string path))))
	(if (char=? (car p) #\c)
	    (set! p (cdr p)))
	(let ((p (reverse p)))
	  (if (char=? (car p) #\r)
	      (set! p (cdr p)))
	  (cxr-1 p lst))))
    
    (test (mcxr 'cr '(1 2 3)) '(1 2 3))
    (test (mcxr 'cadddddddr '(1 2 3 4 5 6 7 8)) 8)
    (test (mcxr 'caadadadadadadadr '(1 (2 (3 (4 (5 (6 (7 (8))))))))) 8)
    
    (define-macro (cxr path lst)
      (let ((p (string->list (symbol->string path))))
	(if (char=? (car p) #\c)
	    (set! p (cdr p)))
	(let ((p (reverse p)))
	  (if (char=? (car p) #\r)
	      (set! p (cdr p)))
	  (let ((func 'arg))
	    (for-each
	     (lambda (f)
	       (set! func (list (if (char=? f #\a) 'car 'cdr) func)))
	     p)
	    `((lambda (arg) ,func) ,lst)))))
    
    (test (cxr car '(1 2 3)) 1)
    (test (cxr cadddddddr '(1 2 3 4 5 6 7 8)) 8)
    (test (cxr caadadadadadadadr '(1 (2 (3 (4 (5 (6 (7 (8))))))))) 8)
    )

  ;; this is the best of them!
  (let ()
    (define-macro (c?r path)
      ;; here "path" is a list and "X" marks the spot in it that we are trying to access
      ;; (a (b ((c X)))) -- anything after the X is ignored, other symbols are just placeholders
      ;; c?r returns a function that gets X

      ;; maybe ... for cdr? (c?r (a ...);  right now it's using dot: (c?r (a . X)) -> cdr
      
      ;; (c?r (a b X)) -> caddr, 
      ;; (c?r (a (b X))) -> cadadr
      ;; ((c?r (a a a X)) '(1 2 3 4 5 6)) -> 4
      ;; ((c?r (a (b c X))) '(1 (2 3 4))) -> 4
      ;; ((c?r (((((a (b (c (d (e X)))))))))) '(((((1 (2 (3 (4 (5 6)))))))))) -> 6
      ;; ((c?r (((((a (b (c (X (e f)))))))))) '(((((1 (2 (3 (4 (5 6)))))))))) -> 4
      ;; (procedure-source (c?r (((((a (b (c (X (e f))))))))))) -> (lambda (lst) (car (car (cdr (car (cdr (car (cdr (car (car (car (car lst))))))))))))
      
      (define (X-marks-the-spot accessor tree)
	(if (pair? tree)
	    (or (X-marks-the-spot (cons 'car accessor) (car tree))
		(X-marks-the-spot (cons 'cdr accessor) (cdr tree)))
	    (if (eq? tree 'X)
		accessor
		#f)))
      
      (let ((accessor (X-marks-the-spot '() path)))
	(if (not accessor)
	    (error "can't find the spot! ~A" path)
	    (let ((len (length accessor)))
	      (if (< len 5)                   ; it's a built-in function
		  (let ((name (make-string (+ len 2))))
		    (set! (name 0) #\c)
		    (set! (name (+ len 1)) #\r)
		    (do ((i 0 (+ i 1))
			 (a accessor (cdr a)))
			((= i len))
		      (set! (name (+ i 1)) (if (eq? (car a) 'car) #\a #\d)))
		    (string->symbol name))
		  (let ((body 'lst))          ; make a new function to find the spot
		    (for-each
		     (lambda (f)
		       (set! body (list f body)))
		     (reverse accessor))
		    `(lambda (lst) ,body)))))))
    
    (test ((c?r (a b X)) (list 1 2 3 4)) 3)
    (test ((c?r (a (b X))) '(1 (2 3) ((4)))) 3)
    (test ((c?r (a a a X)) '(1 2 3 4 5 6)) 4)
    (test ((c?r (a (b c X))) '(1 (2 3 4))) 4)
    (test ((c?r (((((a (b (c (d (e X)))))))))) '(((((1 (2 (3 (4 (5 6)))))))))) 6)
    (test ((c?r (((((a (b (c (X (e f)))))))))) '(((((1 (2 (3 (4 (5 6)))))))))) 4))

  (let ()
    (define-macro (nested-for-each args func . lsts)
      (let ((body `(,func ,@args)))
	(for-each
	 (lambda (arg lst)
	   (set! body `(for-each
			(lambda (,arg)
			  ,body)
			,lst)))
	 args lsts)
	body))
    
    ;;(nested-for-each (a b) + '(1 2) '(3 4)) ->
    ;;  (for-each (lambda (b) (for-each (lambda (a) (+ a b)) '(1 2))) '(3 4))
    
    (define-macro (nested-map args func . lsts)
      (let ((body `(,func ,@args)))
	(for-each
	 (lambda (arg lst)
	   (set! body `(map
			(lambda (,arg)
			  ,body)
			,lst)))
	 args lsts)
	body))
    
    ;;(nested-map (a b) + '(1 2) '(3 4))
    ;;   ((4 5) (5 6))
    ;;(nested-map (a b) / '(1 2) '(3 4))
    ;;   ((1/3 2/3) (1/4 1/2))

    (test (nested-map (a b) + '(1 2) '(3 4)) '((4 5) (5 6)))
    (test (nested-map (a b) / '(1 2) '(3 4)) '((1/3 2/3) (1/4 1/2)))
    )
    
  (let ()
    (define-macro (define-curried name-and-args . body)	
      `(define ,@(let ((newlst `(begin ,@body)))
		   (define (rewrap lst)
		     (if (pair? (car lst))
			 (begin
			   (set! newlst (cons 'lambda (cons (cdr lst) (list newlst))))
			   (rewrap (car lst)))
			 (list (car lst) (list 'lambda (cdr lst) newlst))))
		   (rewrap name-and-args))))

    (define-curried (((((f a) b) c) d) e) (* a b c d e))
    (test (((((f 1) 2) 3) 4) 5) 120)
    (define-curried (((((f a b) c) d e) f) g) (* a b c d e f g))
    (test (((((f 1 2) 3) 4 5) 6) 7) 5040)
    (define-curried (((foo)) x) (+ x 34))
    (test (((foo)) 300) 334)
    (define-curried ((foo-1) x) (+ x 34))
    (test ((foo-1) 200) 234)
    )


  
  (define-macro (eval-case key . clauses)
    ;; case with evaluated key-lists
    `(cond ,@(map (lambda (lst)
		    (if (pair? (car lst))
			(cons `(member ,key (list ,@(car lst)))
			      (cdr lst))
			lst))
		  clauses)))

  (test (let ((a 1) (b 2)) (eval-case 1 ((a) 123) ((b) 321) (else 0))) 123)
  (test (let ((a 1) (b 2) (c 3)) (eval-case 3 ((a c) 123) ((b) 321) (else 0))) 123)
  (test (let ((a 1) (b 2)) (eval-case 3 ((a) 123) ((b) 321) (((+ a b)) -1) (else 0))) -1)
  (test (let ((a 1) (b 2)) (eval-case 6 ((a (* (+ a 2) b)) 123) ((b) 321) (((+ a b)) -1) (else 0))) 123)

  (test (let ()
	  (define (set-cadr! a b)
	    (set-car! (cdr a) b)
	    b)
	  (let ((lst (list 1 2 3)))
	    (set-cadr! lst 32)
	    lst))
	'(1 32 3))

  ;;; macro?
  (test (macro? eval-case) #t)
  (test (macro? pi) #f)
  (test (macro? quasiquote) #t) ; s7_define_macro in s7.c
  (test (let ((m quasiquote)) (macro? m)) #t)
  (test (macro? macroexpand) #t)
  (test (macro? cond) #f)
  (test (macro? letrec) #f)

  ;; not ideal: (let () (define (hi a) (+ a 1)) (macroexpand (hi 2))) ->
  ;;              ;+ argument 1, (hi 2), is pair but should be a number
  ;;              ;    (+ a 1)

  (for-each
   (lambda (arg)
     (test (macro? arg) #f))
   (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() car abs (lambda () 1) #2d((1 2) (3 4)) _ht_ #f 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))
  (test (macro?) 'error)
  
  (define-macro (fully-expand form)
    (define (expand form)
      ;; walk form looking for macros, expand any that are found
      (if (pair? form)
	  (if (macro? (car form))
	      (expand ((eval (procedure-source (car form))) form))
	      (cons (expand (car form))
		    (expand (cdr form))))
	  form))
    (expand form))

  (define fe1-called #f)
  (define-macro (fe1 a) (set! fe1-called #t) `(+ ,a 1))
  (define fe2-called #f)
  (define-macro (fe2 b) (set! fe2-called #f) `(+ (fe1 ,b) 2))
  (fully-expand (define (fe3 c) (+ (fe2 c) (fe1 (+ c 1)))))
  (set! fe1-called #f)
  (set! fe2-called #f)
  (let ((val (fe3 3)))
    (if (or (not (= val 11))
	    fe1-called
	    fe2-called)
	(format #t "fully-expand: ~A ~A ~A ~A~%" val (procedure-source fe3) fe1-called fe2-called)))

  (test (let ()
	  (define-macro (pop sym)
	    (let ((v (gensym "v")))
	      `(let ((,v (car ,sym)))
		 (set! ,sym (cdr ,sym))
		 ,v)))
	  (let ((lst (list 1 2 3)))
	    (let ((val (pop lst)))
	      (and (= val 1)
		   (equal? lst (list 2 3))))))
	#t)

  (define-macro (destructuring-bind lst expr . body)
    `(let ((ex ,expr))
       
       (define (flatten lst)
	 (cond ((null? lst) '())
	       ((pair? lst)
		(if (pair? (car lst))
		    (append (flatten (car lst)) (flatten (cdr lst)))
		    (cons (car lst) (flatten (cdr lst)))))
	       (#t lst)))
       
       (define (structures-equal? l1 l2)
	 (if (pair? l1)
	     (and (pair? l2)
		  (structures-equal? (car l1) (car l2))
		  (structures-equal? (cdr l1) (cdr l2)))
	     (not (pair? l2))))
       
       (if (not (structures-equal? ',lst ex))
	   (error "~A and ~A do not match" ',lst ex))
       
       (let ((names (flatten ',lst))
	     (vals (flatten ex)))
	 (apply (eval (list 'lambda names ',@body)) vals))))
  
  (test (destructuring-bind (a b) (list 1 2) (+ a b)) 3)
  (test (destructuring-bind ((a) b) (list (list 1) 2) (+ a b)) 3)
  (test (destructuring-bind (a (b c)) (list 1 (list 2 3)) (+ a b c)) 6)
  (test (let ((x 1)) (destructuring-bind (a b) (list x 2) (+ a b))) 3)

  (defmacro once-only (names . body)
    (let ((gensyms (map (lambda (n) (gensym)) names)))
      `(let (,@(map (lambda (g) `(,g (gensym))) gensyms))
	 `(let (,,@(map (lambda (g n) ``(,,g ,,n)) gensyms names))
	    ,(let (,@(map (lambda (n g) `(,n ,g)) names gensyms))
	       ,@body)))))

  (let ()
    (defmacro hiho (start end) 
      (once-only (start end) 
	`(list ,start ,end (+ 2 ,start) (+ ,end 2))))

    (test (let ((ctr 0)) 
	    (let ((lst (hiho (let () (set! ctr (+ ctr 1)) ctr) 
			     (let () (set! ctr (+ ctr 1)) ctr))))
	      (list ctr lst)))
	  '(2 (1 2 3 4))))

  (define-bacro (once-only-1 names . body)
    `(let (,@(map (lambda (name) `(,name ,(eval name))) names))
       ,@body))

  (let ()
    (define-bacro (hiho start end) 
      (once-only-1 (start end) 
	`(list ,start ,end (+ 2 ,start) (+ ,end 2))))

    (test (let ((ctr 0)) 
	    (let ((lst (hiho (let () (set! ctr (+ ctr 1)) ctr) 
			     (let () (set! ctr (+ ctr 1)) ctr))))
	      (list ctr lst)))
	  '(2 (1 2 3 4))))

  (defmacro with-gensyms (names . body)
    `(let ,(map (lambda (n) `(,n (gensym))) names)
       ,@body))

  (define-macro (define-clean-macro name-and-args . body)
    ;; the new backquote implementation breaks this slightly -- it's currently confused about unquoted nil in the original
    (let ((syms ()))
      
      (define (walk func lst)
	(if (and (func lst)
		 (pair? lst))
	    (begin
	      (walk func (car lst))
	      (walk func (cdr lst)))))
      
      (define (car-member sym lst)
	(if (null? lst)
	    #f
	    (if (eq? sym (caar lst))
		(cdar lst)
		(car-member sym (cdr lst)))))
      
      (define (walker val)
	(if (pair? val)
	    (if (eq? (car val) 'quote)
		(or (car-member (cadr val) syms)
		    (and (pair? (cadr val))
			 (or (and (eq? (caadr val) 'quote) ; 'sym -> (quote (quote sym))
				  val)
			     (append (list 'list) 
				     (walker (cadr val)))))
		    (cadr val))
		(cons (walker (car val))
		      (walker (cdr val))))
	    (or (car-member val syms)
		val)))
      
      (walk (lambda (val)
	      (if (and (pair? val)
		       (eq? (car val) 'quote)
		       (symbol? (cadr val))
		       (not (car-member (cadr val) syms)))
		  (set! syms (cons 
			      (cons (cadr val) 
				    (gensym (symbol->string (cadr val))))
			      syms)))
	      (or (not (pair? val))
		  (not (eq? (car val) 'quote))
		  (not (pair? (cadr val)))
		  (not (eq? (caadr val) 'quote))))
	    body)
      
      (let* ((new-body (walker body))
	     (new-syms (map (lambda (slot)
			      (list (cdr slot) '(gensym)))
			    syms))
	     (new-globals 
	      (let ((result '()))
		(for-each
		 (lambda (slot)
		   (if (defined? (car slot))
		       (set! result (cons
				     (list 'set! (cdr slot) (car slot))
				     result))))
		 syms)
		result)))
	
	`(define-macro ,name-and-args 
	   (let ,new-syms
	     ,@new-globals
	     `(begin ,,@new-body))))))


  (define-macro (define-immaculo name-and-args . body)
    (let* ((gensyms (map (lambda (g) (gensym)) (cdr name-and-args)))
	   (args (cdr (copy name-and-args)))
	   (name (car name-and-args))
	   (set-args (map (lambda (a g) `(list ',g ,a)) args gensyms))
	   (get-args (map (lambda (a g) `(quote (cons ',a ,g))) args gensyms))
	   (blocked-args (map (lambda (a) `(,a ',a)) args))
	   (new-body (list (eval `(let (,@blocked-args) ,@body)))))
      `(define-macro ,name-and-args
	 `(let ,(list ,@set-args)
	    ,(list 'with-environment 
		   (append (list 'augment-environment) 
			   (list (list 'procedure-environment ,name)) 
			   (list ,@get-args))
		   ',@new-body)))))
  
  (test (let ()
	  (define-clean-macro (hi a) `(+ ,a 1))
	  (hi 1))	  
	2)
  
  (test (let ()
	  (define-immaculo (hi a) `(+ ,a 1))
	  (hi 1))	  
	2)
  
  (test (let ()
	  (define-clean-macro (hi a) `(+ ,a 1))
	  (let ((+ *)
		(a 12))
	    (hi a)))
	13)
  
  (test (let ()
	  (define-immaculo (hi a) `(+ ,a 1))
	  (let ((+ *)
		(a 12))
	    (hi a)))
	13)
  
;; define-clean-macro is making no-longer-correct assumptions about quasiquote -- I think I'll just put these aside
;  (test (let ()
;	  (define-clean-macro (hi a) `(let ((b 23)) (+ b ,a)))
;	  (hi 2))
;	25)
  
  (test (let ()
	  (define-immaculo (hi a) `(let ((b 23)) (+ b ,a)))
	  (hi 2))
	25)
  
;  (test (let ()
;	  (define-clean-macro (hi a) `(let ((b 23)) (+ b ,a)))
;	  (let ((+ *)
;		(b 12))
;	    (hi b)))
;	35)
  
  (test (let ()
	  (define-immaculo (hi a) `(let ((b 23)) (+ b ,a)))
	  (let ((+ *)
		(b 12))
	    (hi b)))
	35)
  
;  (test (let ()
;	  (define-clean-macro (mac a b) `(let ((c (+ ,a ,b))) (let ((d 12)) (* ,a ,b c d))))
;	  (mac 2 3))
;	360)
  
  (test (let ()
	  (define-immaculo (mac a b) `(let ((c (+ ,a ,b))) (let ((d 12)) (* ,a ,b c d))))
	  (mac 2 3))
	360)
  
;  (test (let ()
;	  (define-clean-macro (mac a b) `(let ((c (+ ,a ,b))) (let ((d 12)) (* ,a ,b c d))))
;	  (let ((c 2)
;		(d 3))
;	    (mac c d)))
;	360)
  
  (test (let ()
	  (define-immaculo (mac a b) `(let ((c (+ ,a ,b))) (let ((d 12)) (* ,a ,b c d))))
	  (let ((c 2)
		(d 3))
	    (mac c d)))
	360)
  
  (test (let ()
	  (define-clean-macro (mac a . body)
	    `(+ ,a ,@body))
	  (mac 2 3 4))
	9)

  (test (let ()
	  (define-clean-macro (mac a . body)
	    `(+ ,a ,@body))
	  (let ((a 2)
		(+ *))
	    (mac a (- 5 a) (* a 2))))
	9)

  (test (let ()
	  (define-clean-macro (mac) (let ((a 1)) `(+ ,a 1)))
	  (mac))
	2)

  (test (let ()
	  (define-immaculo (mac) (let ((a 1)) `(+ ,a 1)))
	  (mac))
	2)

  (test (let ()
	  (define-immaculo (hi a) `(list 'a ,a))
	  (hi 1))
	(list 'a 1))

  (test (let ()
	  (define-immaculo (mac c d) `(let ((a 12) (b 3)) (+ a b ,c ,d)))
	  (let ((a 21) (b 10) (+ *)) (mac a b)))
	46)

;  (test (let ((values 32)) (define-macro (hi a) `(+ 1 ,@a)) (hi (2 3))) 6)
;  (test (let ((list 32)) (define-macro (hi a) `(+ 1 ,@a)) (hi (2 3))) 6)
;  (test (let () (define-macro (hi a) `(let ((apply 32)) (+ apply ,@a))) (hi (2 3))))
  (test (let () (define-macro (hi a) `(+ 1 (if ,(= a 0) 0 (hi ,(- a 1))))) (hi 3)) 4)
  (test (let () (define-macro (hi a) `(+ 1 ,a)) ((if #t hi abs) -3)) -2)
  (test (let () (apply define-macro '((m a) `(+ 1 ,a))) (m 2)) 3)
  (test (let () (apply (eval (apply define-macro '((m a) `(+ 1 ,a)))) '(3))) 4)
  (test (let () (apply (eval (apply define '((hi a) (+ a 1)))) '(2))) 3)
  (test (let () ((eval (apply define '((hi a) (+ a 1)))) 3)) 4)
  (test (let () ((eval (apply define-macro '((m a) `(+ 1 ,a)))) 3)) 4)
  (test (let () ((symbol->value (apply define '((hi a) (+ a 1)))) 3)) 4)
  (test (let () ((symbol->value (apply define-macro '((m a) `(+ 1 ,a)))) 3)) 4)
  (test (let () 
	  (define-macro (mu args . body)
	    (let ((m (gensym)))
	      `(symbol->value (apply define-macro '((,m ,@args) ,@body)))))
	  ((mu (a) `(+ 1 ,a)) 3))
	4)
  (test (let () (define-macro (hi a) `(+ 1 ,a)) (map hi '(1 2 3))) '(2 3 4))
  (test (let () (define-macro (hi a) `(+ ,a 1)) (apply hi '(4))) 5)
  (test (let () 
	  (define-macro (hi a) `(+ ,a 1))
	  (define (fmac mac) (apply mac '(4)))
	  (fmac hi))
	5)
  (test (let () 
	  (define (make-mac)
	    (define-macro (hi a) `(+ ,a 1))
	    hi)
	  (let ((x (make-mac)))
	    (x 2)))
	3)

  (define-macro* (_mac1_) `(+ 1 2))
  (test (_mac1_) 3)
  (define-macro* (_mac2_ a) `(+ ,a 2))
  (test (_mac2_ 1) 3)
  (test (_mac2_ :a 2) 4)
  (define-macro* (_mac3_ (a 1)) `(+ ,a 2))
  (test (_mac3_) 3)
  (test (_mac3_ 3) 5)
  (test (_mac3_ :a 0) 2)
  (define-macro* (_mac4_ (a 1) (b 2)) `(+ ,a ,b))
  (test (_mac4_) 3)
  (test (_mac4_ :b 3) 4)
  (test (_mac4_ 2 :b 3) 5)
  (test (_mac4_ :b 10 :a 12) 22)
  (test (_mac4_ :a 4) 6)

  (define-bacro* (_mac21_) `(+ 1 2))
  (test (_mac21_) 3)
  (define-bacro* (_mac22_ a) `(+ ,a 2))
  (test (_mac22_ 1) 3)
  (test (_mac22_ :a 2) 4)
  (define-bacro* (_mac23_ (a 1)) `(+ ,a 2))
  (test (_mac23_) 3)
  (test (_mac23_ 3) 5)
  (test (_mac23_ :a 0) 2)
  (define-bacro* (_mac24_ (a 1) (b 2)) `(+ ,a ,b))
  (test (_mac24_) 3)
  (test (_mac24_ :b 3) 4)
  (test (_mac24_ 2 :b 3) 5)
  (test (_mac24_ :b 10 :a 12) 22)
  (test (_mac24_ :a 4) 6)  
  
  (defmacro* _mac11_ () `(+ 1 2))
  (test (_mac11_) 3)
  (defmacro* _mac12_ (a) `(+ ,a 2))
  (test (_mac12_ 1) 3)
  (test (_mac12_ :a 2) 4)
  (defmacro* _mac13_ ((a 1)) `(+ ,a 2))
  (test (_mac13_) 3)
  (test (_mac13_ 3) 5)
  (test (_mac13_ :a 0) 2)
  (defmacro* _mac14_ ((a 1) (b 2)) `(+ ,a ,b))
  (test (_mac14_) 3)
  (test (_mac14_ :b 3) 4)
  (test (_mac14_ 2 :b 3) 5)
  (test (_mac14_ :b 10 :a 12) 22)
  (test (_mac14_ :a 4) 6)

  (define-bacro (symbol-set! var val) `(set! ,(symbol->value var) ,val))
  (test (let ((x 32) (y 'x)) (symbol-set! y 123) (list x y)) '(123 x))

  (define-bacro (symbol-eset! var val) `(set! ,(eval var) ,val))
  (test (let ((x '(1 2 3)) (y `(x 1))) (symbol-eset! y 123) (list x y)) '((1 123 3) (x 1)))
  (test (let ((x #(1 2 3)) (y `(x 1))) (symbol-eset! y 123) (list x y)) '(#(1 123 3) (x 1)))

  (let ()
    (define-macro (hi a) `````(+ ,,,,,a 1))
    (test (eval (eval (eval (eval (hi 2))))) 3)

    (define-macro (hi a) `(+ ,@@a))
    (test (hi (1 2 3)) 'error)

    (define-macro (hi @a) `(+ ,@@a))
    (test (hi (1 2 3)) 6))



  (let ((old-readers *#readers*))

    ;; testing *#readers* is slightly tricky because the reader runs before we evaluate the test expression
    ;;    so in these cases, the new reader use is always in a string 

    (set! *#readers* (list (cons #\s (lambda (str) 123))))
    (let ((val (eval-string "(+ 1 #s1)"))) ; force this into the current reader
      (test val 124))
    (set! *#readers* '())

    (set! *#readers* 
	  (cons (cons #\t (lambda (str) 
			    (string->number (substring str 1) 12)))
		*#readers*))
    (num-test (string->number "#tb") 11)
    (num-test (string->number "#t11.3") 13.25)
    (num-test (string->number "#e#t11.3") 53/4)
    (num-test (string->number "#t#e1.5") 17/12)
    (num-test (string->number "#i#t1a") 22.0)
    (num-test (string->number "#t#i1a") 22.0) ; ??? this is analogous to #x#i1a = 26.0
    (num-test (string->number "#t#t1a") 22.0)
    (num-test (string->number "#t#t#t1a") 22.0)
    (test (eval-string "#t") #t)
    (test (eval-string "#T1") 'error)

    (set! *#readers*
	  (cons (cons #\. (lambda (str)
			    (if (string=? str ".") (eval (read)) #f)))
		*#readers*))

    (test (eval-string "'(1 2 #.(* 3 4) 5)") '(1 2 12 5))
    (num-test (string->number "#t1a") 22)
    (test (eval-string "'(1 #t(2))") '(1 #t (2)))
    (test (string->number "#t1r") #f)

    (set! *#readers* (list (cons #\t (lambda (str) 
				       ;; in the duplicated case: "t#t..."
				       (if (< (length str) 3)
					   (string->number (substring str 1) 12)
					   (and (not (char=? (str 1) #\#)) 
						(not (char=? (str 2) #\t)) 
						(string->number (substring str 1) 12)))))))
    (test (string->number "#t#t1a") #f)

    (set! *#readers* (cons (cons #\x (lambda (str) 
				       (or (if (< (length str) 3)
					       (string->number (substring str 1) 7)
					       (and (not (char=? (str 1) #\#)) 
						    (not (char=? (str 2) #\x)) 
						    (string->number (substring str 1) 7)))
					   'error)))
			   *#readers*))

    (num-test (string->number "#x12") 9)
    (num-test (string->number "#x-142.1e-1") -11.30612244898)
    (num-test (string->number "#e#x-142.1e-1") -554/49)
    (num-test (string->number "#t460.88") 648.72222222222)
    (num-test (string->number "#e#ta.a") 65/6)
    (num-test (string->number "#x1") 1)
    (test (string->number "#te") #f)
    (num-test (string->number "#x10") 7)
    (test (string->number "#x17") #f)
    (num-test (string->number "#x106") 55)
    (test (string->number "#x#t1") #f)

    (let ()
      (define (read-in-radix str radix)
	;; no error checking, only integers
	(define (char->digit c)
	  (cond ((char-numeric? c)
		 (- (char->integer c) (char->integer #\0)))
		((char-lower-case? c)
		 (+ 10 (- (char->integer c) (char->integer #\a))))
		(#t
		 (+ 10 (- (char->integer c) (char->integer #\A))))))
	(let* ((negative (char=? (str 0) #\-))
	       (len (length str))
	       (j (if (or negative (char=? (str 0) #\+)) 2 1))) ; 1st char is "z"
	  (do ((sum (char->digit (str j))
		    (+ (* sum radix) (char->digit (str j)))))
	      ((= j (- len 1)) sum)
	    (set! j (+ j 1)))))
      
      (set! *#readers* (list (cons #\z (lambda (str) (read-in-radix str 32)))))
      (num-test (string->number "#z1p") 57)
      )
      
    (let ((p1 (make-procedure-with-setter (lambda (str) (string->number (substring str 1) 12)) (lambda (a) a))))
      (set! *#readers* (list (cons #\t p1)))
      (num-test (string->number "#ta") 10)
      (num-test (string->number "#t11.6") 13.5)
      (num-test (string->number "#e#t11.6") 27/2))

    (set! *#readers* old-readers)
    
    (num-test (string->number "#x106") 262)
    (num-test (string->number "#x17") 23)
    )

;;; (call-with-exit (lambda (exit) (set! *#readers* (cons (cons #\p (lambda (str) (exit 23))) *#readers*)) #p123))
  
  (begin
    (define-macro (hi a) `(+ ,a 1))
    (test (hi 2) 3)
    (let ()
      (define (ho b) (+ 1 (hi b)))
      (test (ho 1) 3))
    (let ((hi 32))
      (test (+ hi 1) 33))
    (letrec ((hi (lambda (a) (if (= a 0) 0 (+ 2 (hi (- a 1)))))))
      (test (hi 3) 6))
    (letrec* ((hi (lambda (a) (if (= a 0) 0 (+ 2 (hi (- a 1)))))))
      (test (hi 3) 6))
    (test (equal? '(hi 1) (quote (hi 1))) #t)
    (test (list? '(hi 1)) #t)
    (test (list? '(((hi 1)))) #t)
    (test (equal? (vector (hi 1)) '#(2)) #t)
    (test (symbol? (vector-ref '#(hi) 0)) #t))

  (define-macro (define-with-goto name-and-args . body)
    ;; run through the body collecting label accessors, (label name)
    ;; run through getting goto positions, (goto name)
    ;; tie all the goto's to their respective labels (via set-cdr! essentially)
    
    (define (find-accessor type)
      (let ((labels '()))
	(define (gather-labels accessor tree)
	  (if (pair? tree)
	      (if (equal? (car tree) type)
		  (begin
		    (set! labels (cons (cons (cadr tree) 
					     (let ((body 'lst))
					       (for-each
						(lambda (f)
						  (set! body (list f body)))
						(reverse (cdr accessor)))
					       (make-procedure-with-setter
						(apply lambda '(lst) (list body))
						(apply lambda '(lst val) `((set! ,body val))))))
				       labels))
		    (gather-labels (cons 'cdr accessor) (cdr tree)))
		  (begin
		    (gather-labels (cons 'car accessor) (car tree))
		    (gather-labels (cons 'cdr accessor) (cdr tree))))))
	(gather-labels '() body)
	labels))
    (let ((labels (find-accessor 'label))
	  (gotos (find-accessor 'goto)))
      (if (not (null? gotos))
	  (for-each
	   (lambda (goto)
	     (let* ((name (car goto))
		    (goto-accessor (cdr goto))
		    (label (assoc name labels))
		    (label-accessor (and label (cdr label))))
	       (if label-accessor
		   (set! (goto-accessor body) (label-accessor body))
		   (error 'bad-goto "can't find label: ~S" name))))
	   gotos))
      `(define ,name-and-args
	 (let ((label (lambda (name) #f))
	       (goto (lambda (name) #f)))
	   ,@body))))
  
  (let ()
    (define-with-goto (g1 a)
      (let ((x 1))
	(if a
	    (begin
	      (set! x 2)
	      (goto 'the-end)
	      (set! x 3))
	    (set! x 4))
	(label 'the-end)
	x))

    (define-with-goto (g2 a)
      (let ((x a))
	(label 'start)
	(if (< x 4)
	    (begin
	      (set! x (+ x 1))
	      (goto 'start)))
	x))
    
    (test (g1 #f) 4)
    (test (g1 #t) 2)
    (test (g2 1) 4)
    (test (g2 32) 32))

  
  (let ()
    (define special-value
      (let ((type (make-type)))
	((cadr type) 'special)))

    (test (eq? special-value special-value) #t)
    (test (eqv? special-value special-value) #t)
    (test (equal? special-value special-value) #t)
    (test (procedure? special-value) #f)
    (for-each
     (lambda (arg)
       (test (or (eq? arg special-value)
		 (eqv? arg special-value)
		 (equal? arg special-value))
	     #f))
       (list "hi" -1 #\a 1 'special 3.14 3/4 1.0+1.0i #f #t '(1 . 2) #<unspecified> #<undefined>))

    (begin
      (define rec? #f)
      (define make-rec #f)
      (define rec-a #f)
      (define rec-b #f)

      (let* ((rec-type (make-type))
	     (? (car rec-type))
	     (make (cadr rec-type))
	     (ref (caddr rec-type)))

	(set! make-rec (lambda* ((a 1) (b 2))
				(make (vector a b))))

	(set! rec? (lambda (obj)
		     (? obj)))
  
	(set! rec-a (make-procedure-with-setter
		     (lambda (obj)
		       (and (rec? obj)
			    (vector-ref (ref obj) 0)))
		     (lambda (obj val)
		       (if (rec? obj)
			   (vector-set! (ref obj) 0 val)))))

	(set! rec-b (make-procedure-with-setter
		     (lambda (obj)
		       (and (rec? obj)
			    (vector-ref (ref obj) 1)))
		     (lambda (obj val)
		       (if (rec? obj)
			   (vector-set! (ref obj) 1 val)))))))

    (let ((hi (make-rec 32 '(1 2))))
      (test (rec? hi) #t)
      (test (equal? hi hi) #t)
      (test (rec? 32) #f)
      (test (rec-a hi) 32)
      (test (rec-b hi) '(1 2))
      (set! (rec-b hi) 123)
      (test (rec-b hi) 123)
      (let ((ho (make-rec 32 '(1 2))))
	(test (eq? hi ho) #f)
	(test (eqv? hi ho) #f)
	(test (equal? hi ho) #f)
	(set! (rec-b ho) 123)
	(test (equal? hi ho) #t))
      (let ((ho (make-rec 123 '())))
	(test (eq? hi ho) #f)
	(test (eqv? hi ho) #f)
	(test (equal? hi ho) #f))
      (test (copy hi) 'error)
      (test (fill! hi 1.0) 'error)
      (test (object->string hi) "#<anonymous-type #(32 123)>")
      (test (length hi) 'error)
      (test (reverse hi) 'error)
      (test (for-each abs hi) 'error)
      (test (map abs hi) 'error)
      (test (hi 1) 'error)
      (test (set! (hi 1) 2) 'error)
      )

    (let ((typo (make-type :equal (lambda (a b) (equal? a b)))))
      (let ((a ((cadr typo) 123))
	    (b ((cadr typo) 321))
	    (c ((cadr typo) 123)))
	(test (procedure? a) #f)
	(test (equal? a b) #f)
	(test (eq? a a) #t)
	(test (eq? a b) #f)
	(test (eqv? a a) #t)
	(test (eqv? a b) #f)
	(test (equal? a c) #t)
	(test (equal? b c) #f)))

    (test (let ((typo (make-type :equal (lambda (a b) (= (abs (- a b)) 2)))))
	    (let ((a ((cadr typo) 1))
		  (b ((cadr typo) 3))
		  (c ((cadr typo) 1)))
	      (and (equal? a b)
		   (not (equal? a c))
		   (equal? b c))))
	  #t)

    (test (((cadr (make-type :getter (lambda (a b) (vector-ref a b)))) (vector 1 2 3)) 1) 2)
    (test (((cadr (make-type :getter (lambda (a b) (+ 100 (vector-ref a b))))) (vector 1 2 3)) 1) 102)
    (test (length ((cadr (make-type :length (lambda (a) (vector-length a)))) (vector 1 2 3))) 3)
    (test (length ((cadr (make-type :length (lambda (a) (+ 100 (vector-length a))))) (vector 1 2 3))) 103)
    (test (string=? (object->string ((cadr (make-type)) 1)) "#<anonymous-type 1>") #t)
    (test (string=? (object->string ((cadr (make-type :name "hiho")) 123)) "#<hiho 123>") #t)
    (test (string=? (object->string ((cadr (make-type :print (lambda (a) (format #f "#<typo: ~A>" a)))) 1)) "#<typo: 1>") #t)

    (test (let* ((type (make-type :setter (lambda (a b c) (vector-set! a b c))))
		 (t? (car type))
		 (make-t (cadr type))
		 (t-ref (caddr type))
		 (newt (make-t (vector 1 2 3))))
	    (set! (newt 1) 123)
	    (vector-ref (t-ref newt) 1))
	  123)

    (let ((rec1 (make-type))
	  (rec2 (make-type)))
      (let ((rec1? (car rec1))
	    (rec2? (car rec2))
	    (make-rec1 (cadr rec1))	
	    (make-rec2 (cadr rec2))
	    (rec1-ref (caddr rec1))
	    (rec2-ref (caddr rec2)))
	(let ((r1 (make-rec1 123))
	      (r2 (make-rec2 123)))
	  (test (and (rec1? r1)
		     (rec2? r2)
		     (not (rec1? r2))
		     (not (rec2? r1))
		     (= (rec1-ref r1) (rec2-ref r2)))
		#t))))

    (let ((rec3? (car (make-type)))
	  (rec4? (car (make-type #f))))
      (for-each
       (lambda (arg)
	 (test (rec3? arg) #f)
	 (test (rec4? arg) #f))
       (list "hi" -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t '(1 . 2))))

    (test (number? ((cadr (make-type #f #f)) 0)) #f)
    (test ((cadr (make-type :name 123))) 'error)
    (test ((cadr (make-type :length "hiho"))) 'error)
    (test ((cadr (make-type :print (lambda (a b) (= a b)))) "hi") 'error)
    (test ((cadr (make-type :length (lambda () 1))) "hi") 'error)
    (test (length ((cadr (make-type :length (lambda (a) (+ 1 a)))) (vector 1 2 3))) 'error)
    (test (call-with-exit (lambda (exit) (length ((cadr (make-type :length (lambda (a) (exit 32)))) 1)))) 32)
    (test (call-with-exit (lambda (exit) (fill! ((cadr (make-type :fill (lambda (a n) (exit 32)))) 1) 0))) 32)

    (test (let* ((vzt (make-type :name "vzt" 
				 :length (lambda (v) (vector-length v))
				 :getter (lambda (v n) (vector-ref v n))))
		 (make-vzt (cadr vzt)))
	    (let ((v (make-vzt (vector 1 2 3))))
	      (let ((sum 0)) 
		(for-each (lambda (n) (set! sum (+ sum n))) v)
		sum)))
	  6)

    (test (let* ((rec-type (make-type))
		 (? (car rec-type))
		 (make (cadr rec-type))
		 (ref (caddr rec-type)))
	    (let ((val-1 (make "hi")))
	      (let ((val-2 (make val-1)))
		(let ((val-3 (make val-2)))
		  (ref (ref (ref val-3)))))))
	  "hi")

    (test (let* ((rec1-type (make-type))
		 (?1 (car rec1-type))
		 (make1 (cadr rec1-type))
		 (ref1 (caddr rec1-type)))
	    (let* ((rec2-type (make-type))
		   (?2 (car rec2-type))
		   (make2 (cadr rec2-type))
		   (ref2 (caddr rec2-type)))
	      (let ((val-1 (make1 "hi")))
		(let ((val-2 (make2 "hi")))
		  (let ((val-3 (make1 val-2)))
		    (and (string=? (ref2 (ref1 val-3)) "hi")
			 (not (equal? val-1 val-2))
			 (?1 val-1)
			 (?2 val-2)
			 (not (?2 val-3))))))))
	  #t)

    (test (let* ((rec-type (make-type :name "rec"))
		 (make (cadr rec-type))
		 (ref (caddr rec-type)))
	    (let ((val (make "hi")))
	      (ref (ref val))))
	  'error)

    (test (let* ((rec1-type (make-type))
		 (make1 (cadr rec1-type))
		 (ref1 (caddr rec1-type)))
	    (let* ((rec2-type (make-type))
		   (make2 (cadr rec2-type)))
	      (let ((val-1 (make1 "hi")))
		(let ((val-2 (make2 val-1)))
		  (ref1 val-2)))))
	  'error)

    (test (make-type (make-type)) 'error)
    (for-each
     (lambda (arg)
       (test (make-type arg) 'error)
       (test (make-type :getter arg) 'error)
       (test (make-type :setter arg) 'error)
       (test (make-type :length arg) 'error)
       (test (make-type :print arg) 'error)
       (test (make-type :equal arg) 'error)
       (test (make-type :copy arg) 'error)
       (test (make-type :fill arg) 'error)
       (test (make-type :name arg) 'error))
     (list #\a 'a-symbol 1.0+1.0i #t #(1 2) '() 3/4 3.14 '(1 . 2)))
    (test (make-type :print (lambda () #f)) 'error)
    (test (make-type :print (lambda (a b) #f)) 'error)
    (test (make-type :getter (lambda () #f)) 'error)
    (test (make-type :setter (lambda () #f)) 'error)
    (test (make-type :setter (lambda (a) #f)) 'error)
    (test (make-type :length (lambda () #f)) 'error)
    (test (make-type :length (lambda (a b) #f)) 'error)
    (test (make-type :equal (lambda () #f)) 'error)
    (test (make-type :equal (lambda (a) #f)) 'error)
    (test (make-type :equal (lambda (a b c) #f)) 'error)
    (test (make-type :copy (lambda () #f)) 'error)
    (test (make-type :copy (lambda (a b) #f)) 'error)
    (test (make-type :fill (lambda () #f)) 'error)
    (test (make-type :fill (lambda (a) #f)) 'error)
    (test (make-type :fill (lambda (a b c) #f)) 'error)

    (let ((t (make-type)))
      (let ((t? (car t))
	    (make-t (cadr t))
	    (t-ref (caddr t)))
	(test (make-t 1 2) 'error)
	(test (t? (make-t)) #t)
	(test (t-ref (make-t)) #f)
	(test (t? 1 2) 'error)
	(test (t?) 'error)
	(test (t-ref) 'error)
	(test (t-ref 1 2) 'error)
	(for-each
	 (lambda (arg)
	   (test (t-ref arg) 'error))
	 (list #\a 'a-symbol 1.0+1.0i #t #(1 2) '() 3/4 3.14 #() "hi" :hi 1 #f #t '(1 . 2)))))

    (let ()
      (define make-float-vector #f)
      (define float-vector? #f)
      (define float-vector #f)
      
      (let* ((fv-type (make-type 
		       :getter vector-ref :length length :copy copy :fill fill!
		       :setter (lambda (obj index value)
				 (if (not (real? value))
				     (error 'wrong-type-arg-error "float-vector element must be real: ~S" value))
				 (vector-set! obj index (* 1.0 value)))
		       :name "float-vector"))
	     (fv? (car fv-type))
	     (make-fv (cadr fv-type))
	     (fv-ref (caddr fv-type)))
	
	(set! make-float-vector 
	      (lambda* (len (initial-element 0.0))
		       (if (not (real? initial-element))
			   (error 'wrong-type-arg-error "make-float-vector initial element must be real: ~S" initial-element))
		       (make-fv (make-vector len (* 1.0 initial-element)))))
	
	(set! float-vector? fv?)
	
	(set! float-vector
	      (lambda args
		(let* ((len (length args))
		       (fv (make-float-vector len))
		       (v (fv-ref fv)))
		  (do ((lst args (cdr lst))
		       (i 0 (+ i 1)))
		      ((null? lst) fv)
		    (let ((arg (car lst)))
		      (if (not (real? arg))
			  (error 'wrong-type-arg-error "float-vector element must be real: ~S in ~S" arg args))
		      (set! (v i) (* 1.0 arg))))))))

      (let ((val (catch #t
			(lambda ()
			  (let ((t (make-type)))
			    (for-each
			     (lambda (n) 
			       (format #t ";for-each called non-applicable type!~%"))
			     ((cadr t) 0))))
			(lambda args 'error))))
	(if (not (eq? val 'error))
	    (format #t ";for-each on non-applicable type: ~A~%" val)))
      
      (let ((val (catch #t
			(lambda ()
			  (let ((t (make-type :length (lambda (x) 3))))
			    (for-each
			     (lambda (n) 
			       (format #t ";for-each called non-applicable type!~%"))
			     ((cadr t) 0))))
			(lambda args 'error))))
	(if (not (eq? val 'error))
	    (format #t ";for-each on non-applicable type: ~A~%" val)))
      
      (let ((v (make-float-vector 3 0.0)))
	(test (length v) 3)
	(set! (v 1) 32.0)
	(test (v 0) 0.0)
	(test (v 1) 32.0)
	(test (eq? v v) #t)
	(test (eq? v (float-vector 0.0 32.0 0.0)) #f)
	(test (equal? v (float-vector 0.0 32.0 0.0)) #t)
	(test (map + (list 1 2 3) (float-vector 1 2 3)) '(2.0 4.0 6.0))
	(test (reverse (float-vector 1.0 2.0 3.0)) (float-vector 3.0 2.0 1.0))
	(test (copy (float-vector 1.0 2.0 3.0)) (float-vector 1.0 2.0 3.0))
	(test (let () (fill! v 1.0) v) (float-vector 1.0 1.0 1.0))
	(test (object->string v) "#<float-vector #(1.0 1.0 1.0)>")
	(test (let ((v (float-vector 1.0 2.0 3.0))) (map v (list 2 1 0))) '(3.0 2.0 1.0))
	(test (v -1) 'error)
	(test (v 32) 'error)
	(for-each
	 (lambda (arg)
	   (test (v arg) 'error))
	 (list #\a 'a-symbol 1.0+1.0i #f #t abs #(1 2) '() 3/4 3.14 '(1 . 2)))

	(test (map (lambda (a b)
		     (floor (apply + (map + (list a b) (float-vector a b)))))
		   (float-vector 1 2 3) (float-vector 4 5 6))
	      '(10 14 18))

	(test (set! (v 0) "hi") 'error)
	(test (set! (v -1) "hi") 'error)
	(test (set! (v 32) "hi") 'error)
	(for-each
	 (lambda (arg)
	   (test (set! (v 0) arg) 'error))
	 (list #\a 'a-symbol 1.0+1.0i #f #t abs #(1 2) '() '(1 . 2)))
	(test (let ((sum 0.0))
		(for-each
		 (lambda (x)
		   (set! sum (+ sum x)))
		 (float-vector 1.0 2.0 3.0))
		sum)
	      6.0)
	(test (length v) 3)
	))

    (let ()
      (define-macro (blet* names bindings . body)
	`(begin
	   ,@(map (lambda (name)
		    `(define ,name #f))
		  names)
	   (let* ,bindings
	     ,@body)))
      
      (blet* (make-adjustable-vector adjustable-vector? adjust-vector)
	     
	     ((av-type (make-type :name "adjustable-vector"
				  :getter (lambda (obj index)
					    ((car obj) index))
				  :setter (lambda (obj index value)
					    (set! ((car obj) index) value))
				  :length (lambda (obj)
					    (vector-length (car obj)))
				  :print (lambda (obj)
					   (object->string (car obj)))))
	      (av? (car av-type))
	      (make-av (cadr av-type))
	      (av-ref (caddr av-type)))
	     
	     (set! make-adjustable-vector (lambda args 
					    (make-av (list (apply make-vector args)))))
	     (set! adjustable-vector? av?)
	     (set! adjust-vector (lambda* (obj new-length initial-element)
					  (let* ((new-vector (make-vector new-length initial-element))
						 (copy-len (min new-length (length obj))))
					    (do ((i 0 (+ i 1)))
						((= i copy-len))
					      (set! (new-vector i) (obj i)))
					    (set! (car (av-ref obj)) new-vector)))))
      
      (let ((v (make-adjustable-vector 3 #f)))
	(test (length v) 3)
	(test (v 0) #f)
	(set! (v 1) 32.0)
	(adjust-vector v 10 #f)
	(test (length v) 10)
	(test (v 1) 32.0))

      (blet* (rec-a rec? rec-b make-rec)
	     
	     ((rec-type (make-type :name "rec" :length length :copy copy :fill fill!))
	      (? (car rec-type))
	      (make (cadr rec-type))
	      (ref (caddr rec-type)))
	     
	     (set! make-rec (lambda* ((a 1) (b 2))
				     (make (vector a b))))
	     
	     (set! rec? ?)
	     
	     (set! rec-a (make-procedure-with-setter
			  (lambda (obj)
			    (and (rec? obj)
				 (vector-ref (ref obj) 0)))
			  (lambda (obj val)
			    (if (rec? obj)
				(vector-set! (ref obj) 0 val)))))
	     
	     (set! rec-b (make-procedure-with-setter
			  (lambda (obj)
			    (and (rec? obj)
				 (vector-ref (ref obj) 1)))
			  (lambda (obj val)
			    (if (rec? obj)
				(vector-set! (ref obj) 1 val))))))
      
      (let ((r1 (make-rec)))
	(let ((r2 (copy r1)))
	  (test (eq? r1 r2) #f)
	  (test (rec? r2) #t)
	  (test (rec-a r1) 1)
	  (test (rec-b r1) 2)
	  (test (rec-a r2) 1)
	  (test (rec-b r2) 2)
	  (set! (rec-b r2) 32)
	  (test (rec-b r2) 32)
	  (test (rec-b r1) 2)
	  (fill! r2 123)
	  (test (rec-a r1) 1)
	  (test (rec-b r1) 2)
	  (test (rec-a r2) 123)
	  (test (rec-b r2) 123)
	  )
	))


    (define (notify-if-set var notifier)
      (set! (symbol-access var) (list #f notifier #f)))
    
    (define constant-access 
      (list #f
	    (lambda (symbol new-value) 
	      (error "can't change constant ~A's value to ~A" symbol new-value))
	    (lambda (symbol new-value) 
	      (error "can't bind constant ~A to a new value, ~A" symbol new-value))))
    
    (define-macro (define-global-constant symbol value)
      `(begin
	 (define ,symbol ,value)
	 (set! (symbol-access ',symbol) constant-access)
	 ',symbol))
    
    (define-macro (let-constant vars . body)
      (let ((varlist (map car vars)))
	`(let ,vars
	   ,@(map (lambda (var)
		    `(set! (symbol-access ',var) constant-access))
		  varlist)
	   ,@body)))
    
    (define-macro (define-integer var value)
      `(begin
	 (define ,var ,value)
	 (set! (symbol-access ',var) 
	       (list #f
		     (lambda (symbol new-value)
		       (if (real? new-value)
			   (floor new-value)
			   (error "~A can only take an integer value, not ~S" symbol new-value)))
		     #f))
	 ',var))
    
    (define (trace-var var)
      (let* ((cur-access (symbol-access var))
	     (cur-set (and cur-access (cadr cur-access))))
	(set! (symbol-access var)
	      (list (and cur-access (car cur-access))
		    (lambda (symbol new-value) 
		      (format #t "~A set to ~A~%" symbol new-value) 
		      (if cur-set 
			  (cur-set symbol new-value)
			  new-value))
		    (and cur-access (caddr cur-access))
		    cur-access))))
    
    (define (untrace-var var)
      (if (and (symbol-access var)
	       (cdddr (symbol-access var)))
	  (set! (symbol-access var) (cadddr (symbol-access var)))))

    (define-integer _int_ 32)
    (test _int_ 32)
    (set! _int_ 1.5)
    (test _int_ 1)

    (for-each
     (lambda (arg)
       (test (symbol-access arg) 'error)
       (test (set! (symbol-access _int_) arg) 'error))
     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() '#(()) (list 1 2 3) '(1 . 2) "hi"))
    
    (test (symbol-access) 'error)
    (test (symbol-access '_int_ 2) 'error)
    (test (symbol-access 'abs) #f)
    (test (symbol-access 'xyzzy) #f)
    (test (set! (symbol-access _int_) '()) 'error)
    (test (set! (symbol-access _int_) '(#f)) 'error)
    (test (set! (symbol-access _int_) '(#f #f)) 'error)
    (test (set! (symbol-access _int_) '(#f #f #f #f)) 'error)

    (let ((_x1_ #f))
      (set! (symbol-access '_x1_) (list #f 
					(lambda (x y) 'error)
					(lambda (x y) 'error)))
      (test (set! _x1_ 32) 'error)
      (test (let ((_x1_ 32)) 2) 'error))
    (set! (symbol-access '_x1_) #f)
    (let ((_x1_ 0))
      (set! (symbol-access '_x1_) (list #f 
					(lambda (x y) 'error)
					#f))
      (test (set! _x1_ 32) 'error)
      (test (let ((_x1_ 32)) _x1_) 32))
    (set! (symbol-access '_x1_) #f)
    (let ((_x1_ 0))
      (set! (symbol-access '_x1_) (list #f 
					(lambda (x y) 0)
					(lambda (x y) (* y 2))))
      (test (begin (set! _x1_ 32) _x1_) 0)
      (test (let ((_x1_ 32)) _x1_) 64))
    (set! (symbol-access '_x1_) #f)
    (let ((_x1_ 0))
      (set! (symbol-access '_x1_) (list #f 
					(lambda (x y) (symbol->value x))
					(lambda (x y) (+ 2 (symbol->value x)))))
      (test (begin (set! _x1_ 32) _x1_) 0)
      (test (let ((_x1_ 32)) _x1_) 2))

    (define _x3_ 3)
    (set! (symbol-access '_x3_) (list #f (lambda (a b) b) (lambda (a b) b)))
    (test (let ((_x3_ 32)) _x3_) 32)
    (test (let ((_x3_ 32)) (set! _x3_ 1) _x3_) 1)
    (test (let ((_x3_ 32)) (letrec ((_x3_ 1)) _x3_)) 1)

    (let ((ctr ((cadr (make-type :getter (lambda (a b) b) :length (lambda (a) 3)))))
	  (sum 0))
      (for-each (lambda (a b) (set! sum (+ sum a b))) ctr ctr)
      (test sum 6))

    (let ()
      (define-macro (enum . args)
	`(for-each define ',args ((cadr (make-type :getter (lambda (a b) b) 
						   :length (lambda (a) ,(length args)))))))
      (enum zero one two three)
      (test (+ zero one two three) 6))

    (let ((ctr ((cadr (make-type :getter (lambda (a b) b) :length (lambda (a) 4))))) (sum 0))
      (test (map (lambda (a b) (+ a b)) ctr ctr) '(0 2 4 6))
      (test (map (lambda (a b c) (+ a b c)) ctr ctr ctr) '(0 3 6 9))
      (test (map (lambda (a b) (+ a b)) #(0 1 2 3) ctr) '(0 2 4 6))
      (test (map (lambda (a b) (+ a b)) ctr #(0 1 2 3)) '(0 2 4 6))
      (test (map (lambda (a) a) ctr) '(0 1 2 3))
      (test (map ctr '(1 2 3 4)) '(1 2 3 4))
      (test (map ctr ctr) '(0 1 2 3))
      (test (for-each ctr ctr) #<unspecified>)
      (test (map ctr '()) '())
      (test (for-each ctr #()) #<unspecified>)
      )

    (let ((ctr ((cadr (make-type :getter (lambda (a b) (car (map append (list b a)))) :length (lambda (a) (length (map abs '(-1 -2 -3))))))))
	  (sum 0))
      (for-each (lambda (a b) (set! sum (+ sum a b))) ctr ctr)
      (test sum 6))

    (let ((ctr ((cadr (make-type :getter (lambda (a b) (for-each append (list b a)) b) :length (lambda (a) (for-each abs '(-1 -2 -3)) 3)))))
	  (sum 0))
      (for-each (lambda (a b) (set! sum (+ sum a b))) ctr ctr)
      (test sum 6))

    (let ((ctr ((cadr (make-type :getter (lambda (a b) (car (map append (list b a)))) :length (lambda (a) (length (map abs '(-1 -2 -3)))))))))
      (test (map (lambda (a b) (+ a b)) ctr ctr) '(0 2 4)))

    ))

#|
;;; these tests are problematic -- they might not fail as hoped, or they might generate unwanted troubles
(let ((bad-ideas "
                      (define (bad-idea)
                        (let ((lst '(1 2 3)))
                          (let ((result (list-ref lst 1)))
                            (list-set! lst 1 (* 2.0 16.6))
                            (gc)
                            result)))

                      (define (bad-idea-1)
                        (let ((lst #(1 2 3)))
                          (let ((result (vector-ref lst 1)))
                            (vector-set! lst 1 (* 2.0 16.6))
                            (gc)
                             result)))
                      "))
  (with-output-to-file "tmp1.r5rs" (lambda () (display bad-ideas)))
  (load "tmp1.r5rs"))

(num-test (bad-idea) 2)
(let ((val (bad-idea)))
  (if (equal? val 33.2)
      (set! val (bad-idea)))
  (if (equal? val 33.2)
      (format #t ";bad-idea 3rd time: ~A~%" val)))
(num-test (bad-idea-1) 2)
(let ((val (bad-idea-1)))
  (if (equal? val 33.2)
      (set! val (bad-idea-1)))
  (if (equal? val 33.2)
      (format #t ";bad-idea-1 3rd time: ~A~%" val)))
(set! *safety* 1)
(load "tmp1.r5rs")
(num-test (bad-idea) 2)
(num-test (bad-idea) 33.2)
(num-test (bad-idea) 33.2)
(num-test (bad-idea-1) 2)
(num-test (bad-idea-1) 33.2)
(num-test (bad-idea-1) 33.2)
(set! *safety* 0)
|#

(test (quit 0) 'error)

(define-expansion (_expansion_ a) `(+ ,a 1))
(test (_expansion_ 3) 4)
(test (macroexpand (_expansion_ 3)) `(+ 3 1))
(let () (define-macro (hi a) `(+ ,a 1)) (test (macroexpand (hi 2)) '(+ 2 1)))
(test '(_expansion_ 3) (quote (_expansion_ 3)))
(test (_expansion_ (+ (_expansion_ 1) 2)) 5)

(test (let () (define-constant __c1__ 32) __c1__) 32)
(test (let () __c1__) 'error)
(test (let ((__c1__ 3)) __c1__) 'error)
(test (let* ((__c1__ 3)) __c1__) 'error)
(test (letrec ((__c1__ 3)) __c1__) 'error)
(test (let () (define (__c1__ a) a) (__c1__ 3)) 'error)
(test (let () (set! __c1__ 3)) 'error)

;;; constant?
(test (constant? '__c1__) #t)
(test (constant? pi) #t)
(test (constant? 'pi) #t) ; take that, Clisp!
(test (constant? 12345) #t)
(test (constant? 3.14) #t)
(test (constant? :asdf) #t) 
(test (constant? 'asdf) #f)
(test (constant? "hi") #t) 
(test (constant? #\a) #t) 
(test (constant? #f) #t) 
(test (constant? #t) #t) 
(test (constant? '()) #t) 
(test (constant? ()) #t) 
(test (constant? '(a)) #t) 
(test (constant? '*features*) #f)
(test (let ((a 3)) (constant? 'a)) #f)
(test (constant? 'abs) #f)
(test (constant? abs) #t)
(test (constant? most-positive-fixnum) #t)
(test (constant? (/ (log 0))) #t)       ; nan.0 is a constant as a number I guess
(test (constant? 1/0) #t)
(test (constant? (log 0)) #t)
(test (constant?) 'error)
(test (constant? 1 2) 'error)
(test (constant? #<eof>) #t) ; ?
(test (constant? '-) #f)
(test (constant? ''-) #t)
(test (constant? '''-) #t)
(test (constant? '1) #t)
(test (constant? 1/2) #t)
(test (constant? 'with-environment) #t)
(test (constant? with-environment) #t)

;; and some I wonder about -- in CL's terms, these always evaluate to the same thing, so they're constantp
;;   but Clisp:
;;     (constantp (cons 1 2)) ->NIL
;;     (constantp #(1 2)) -> T
;;     (constantp '(1 . 2)) -> NIL
;; etc -- what a mess!

(test (constant? (cons 1 2)) #t)
(test (constant? #(1 2)) #t)
(test (constant? (list 1 2)) #t)
(test (constant? (vector 1 2)) #t)
(test (let ((v (vector 1 2))) (constant? v)) #t) ;!!
;; it's returning #t unless the arg is a symbol that is not a keyword or a defined constant
;; (it's seeing the value of v, not v):
(test (let ((v (vector 1 2))) (constant? 'v)) #f)
;; that is something that can be set! is not a constant?

(if with-bignums
    (begin
      (test (constant? 1624540914719833702142058941) #t)
      (test (constant? 1624540914719833702142058941.4) #t)
      (test (constant? 7151305879464824441563197685/828567267217721441067369971) #t)))

(test (constant? lambda) #t)   ; like abs?
(test (constant? (lambda () 1)) #t)
(test (constant? ''3) #t)
(test (constant? (if #f #f)) #t)
(test (constant? 'x) #f)
(test (let ((x 'x)) (and (not (constant? x)) (equal? x (eval x)))) #t)
(test (and (constant? (list + 1)) (not (equal? (list + 1) (eval (list + 1))))) #t)

;; not sure this is the right thing...
;; but CL makes no sense: 
;; [3]> (constantp (vector 1))
;; T
;; [4]> (constantp (cons 1 2))
;; NIL
;; [5]> (constantp (list 1))
;; NIL
;; [7]> (constantp "hi")
;; T
;; (setf (elt "hi" 1) #\a)
;; #\a
;; at least they finally agree that pi is a constant!

(let ()
  (define-constant __hi__ (vector 3 0))
  (set! (__hi__ 1) 231)
  (test __hi__ #(3 231)))
;; that is, hi is the constant as a vector, not the vector elements


;;; defined?
(test (defined? 'pi) #t)
(test (defined? 'pi (global-environment)) #t)
(test (defined? 'abs (global-environment)) #t)
(test (defined? 'abs (current-environment)) #t)
(test (let ((__c2__ 32)) (defined? '__c2__)) #t)
(test (let ((__c2__ 32)) (defined? '__c2__ (current-environment))) #t)
(test (let ((__c2__ 32)) (defined? '__c3__ (current-environment))) #f)
(test (let ((__c2__ 32)) (defined? '__c2__ (global-environment))) #f)
(test (let ((__c2__ 32)) (defined? '__c3__ (global-environment))) #f)
(test (defined?) 'error)
(test (defined? 'a 'b) 'error)
(for-each
 (lambda (arg)
   (test (defined? arg) 'error)
   (test (defined? 'abs arg) 'error))
 (list -1 #\a 1 _ht_ '#(1 2 3) 3.14 3/4 1.0+1.0i '() #f '#(()) (list 1 2 3) '(1 . 2) "hi"))
(test (defined? 'lambda car) 'error)
(test (defined? lambda gensym) 'error)
(test (defined? 'lambda defined?) 'error)
(test (defined? 'define car) 'error)
(test (defined? 'abs (augment-environment '())) #t) ; nil = global now
(test (defined? lambda) 'error)
(test (defined? 'lambda) #t)
(test (defined? 'dynamic-wind) #t)
(test (defined? 'asdaf) #f)
(test (defined? ':asdaf) #f)
(test (defined? :asdaf) #f)
(test (defined? 'ok?) #t)
(test (defined? 'test-t) #t)
(test (defined? 'quasiquote) #t)
(test (defined? (symbol "123")) #f)
(test (defined? (symbol "+")) #t)
(test (defined? ''+) 'error)
(test (defined? 'if) #t)
(test (defined? if) 'error)
(test (defined? quote) 'error)


;;; environment?
;;; global-environment
;;; initial-environment
;;; current-environment
;;; augment-environment
;;; with-environment

(for-each
 (lambda (arg)
   (test (environment? arg) #f))
 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() #f '#(()) (list 1 2 3) '(1 . 2) "hi" '((a . 1))))
(let () (test (environment? (initial-environment)) #t))
(test (environment? (current-environment)) #t)
(test (environment? (global-environment)) #t)
(test (environment? (augment-environment '())) #t)
(test (environment? (augment-environment! '())) #t)
(test (environment? (augment-environment (augment-environment '()) '(a . 1))) #t)
(test (environment? (augment-environment! (augment-environment! '()) '(a . 1))) #t)
(test (environment? (augment-environment '() '(a . 1))) #t)
(test (environment? (augment-environment! '() '(a . 1))) #t)
(let ((f1 (lambda (a) (+ a 1)))
      (f2 (lambda* ((a 2)) (+ a 1))))
  (define (hi a) (+ a 1))
  (define* (ho (a 1)) (+ a 1))
  (test (environment? (procedure-environment hi)) #t)
  (test (environment? (procedure-environment ho)) #t)
  (test (environment? (procedure-environment f1)) #t)
  (test (environment? (procedure-environment f2)) #t)
  (test (environment? (procedure-environment abs)) #t))

(let ()
  (apply augment-environment! (current-environment)
	 (with-environment (initial-environment)
	   (let ()
	     (define (lognor n1 n2) (lognot (logior n1 n2)))
	     (define (logit n1) n1)

	     (map (lambda (binding)
		    (cons (string->symbol 
			   (string-append "library:" (symbol->string (car binding))))
			  (cdr binding)))
		  (car (environment->list (current-environment)))))))
  (test (library:lognor 1 2) -4))

(for-each
 (lambda (arg)
   (test (environment->list arg) 'error))
 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() #f '#(()) (list 1 2 3) '(1 . 2) "hi" '((a . 1))))

(test (current-environment 1) 'error)
(test (global-environment 1) 'error)
(test (initial-environment 1) 'error)
(test (let () (set! initial-environment 2)) 'error)
(test (let ((initial-environment 2)) initial-environment) 'error)

(test (eq? (global-environment) '()) #f)
(test (eq? (global-environment) (global-environment)) #t)
(test (eq? (global-environment) (initial-environment)) #f)
(test (eqv? (global-environment) (global-environment)) #t)
(test (eqv? (global-environment) (initial-environment)) #f)
(test (equal? (global-environment) (global-environment)) #t)
(test (equal? (global-environment) (initial-environment)) #f)
(test (equal? (current-environment) (initial-environment)) #f)

(test (let () (augment-environment! (initial-environment) (cons 'a 32)) (symbol->value 'a (initial-environment))) #<undefined>)
(test (let ((caar 123)) (+ caar (with-environment (initial-environment) (caar '((2) 3))))) 125)
(test (let ()
	(+ (let ((caar 123)) 
	     (+ caar (with-environment (initial-environment) 
                       (let ((val (caar '((2) 3)))) 
			 (set! caar -1) 
			 (+ val caar))))) ; 124
	   (let ((caar -123)) 
	     (+ caar (with-environment (initial-environment) 
                       (let ((val (caar '((20) 3)))) 
			 (set! caar -2) 
			 (+ val caar))))) ; -105
	   (caar '((30) 3)))) ; 30 + 19
      49)
      
#|
(let ((old+ +))
  (let ((vals 
	 (list (let ()
		 (define a 32)
		 (define p +)
		 (define (f b) (+ a b))
		 (set! a 1)
		 (let ((t1 (f 2)))
		   (set! + -)
		   (let ((t2 (f 2)))
		     (let ((t3 (equal? p +)))
		       (list t1 t2 t3)))))
	       
	       ;; s7: (3 -1 #f) ; this is now (3 3 #f) which strikes me as correct
	       ;; guile: (3 3 #f)
	       
	       (let ()
		 (define a 32)
		 (define p old+)
		 (define (f b) (p a b))
		 (set! a 1)
		 (let ((t1 (f 2)))
		   (set! p -)
		   (let ((t2 (f 2)))
		     (let ((t3 (equal? p old+)))
		       (list t1 t2 t3)))))
	       
	       ;; s7 (3 -1 #t)
	       ;; guile (3 -1 #t)
	       )))
    (set! + old+)
    (test (car vals) (cadr vals))))
|#

(let ((old+ +))
  (define (f x) (with-environment (initial-environment) (+ x 1)))
  (set! + -)
  (test (+ 1 1) 0)
  (test (f 1) 2)
  (set! + old+))

(let ((old+ +))
  (let ((f #f))
    (let ((+ -))
      (set! f (lambda (a) (+ 1 a))))
    (test (f 2) -1)
    (set! + *)
    (test (f 2) -1)
    (set! + old+)))

(test (let ((a 1)) (eval '(+ a b) (augment-environment (current-environment) (cons 'b 32)))) 33)
(test (let ((a 1)) (+ (eval '(+ a b) (augment-environment (current-environment) (cons 'b 32))) a)) 34)
(test (let ((a 1)) (+ (eval '(+ a b) (augment-environment (current-environment) (cons 'b 32) (cons 'a 12))) a)) 45)
(test (let ((a 2)) (eval '(+ a 1) (augment-environment (current-environment)))) 3)
(test (let ((a 1)) (+ (eval '(+ a b) (augment-environment (augment-environment (current-environment) (cons 'b 32)) (cons 'a 12))) a)) 45)
(test (eval (list + 'a (eval (list - 'b) (augment-environment (initial-environment) (cons 'b 1)))) 
	    (augment-environment (initial-environment) (cons 'a 2))) 
      1)

(test (let ((a 1)) (eval-string "(+ a b)" (augment-environment (current-environment) (cons 'b 32)))) 33)
(test (let ((a 1)) (+ (eval-string "(+ a b)" (augment-environment (current-environment) (cons 'b 32))) a)) 34)
(test (let ((a 1)) (+ (eval-string "(+ a b)" (augment-environment (current-environment) (cons 'b 32) (cons 'a 12))) a)) 45)
(test (let ((a 2)) (eval-string "(+ a 1)" (augment-environment (current-environment)))) 3)
(test (let ((a 1)) (+ (eval-string "(+ a b)" (augment-environment (augment-environment (current-environment) (cons 'b 32)) (cons 'a 12))) a)) 45)
(test (eval-string (string-append "(+ a " (number->string (eval (list - 'b) (augment-environment (initial-environment) (cons 'b 1)))) ")")
		   (augment-environment (initial-environment) (cons 'a 2)))
      1)

(test (augment-environment) 'error)
(for-each
 (lambda (arg)
   (test (augment-environment arg '(a . 32)) 'error)
   (test (augment-environment! arg '(a . 32)) 'error))
 (list -1 #\a 1 3.14 3/4 1.0+1.0i "hi" 'hi #() #f _ht_))

(let ((e (augment-environment (current-environment)
			      (cons 'a 32)
			      (cons 'b 12))))
  (test (eval '(+ a b) e) 44)
  (test (eval '(+ a b c) (augment-environment e (cons 'c 3))) 47)
  (test (eval '(+ a b) (augment-environment e (cons 'b 3))) 35)
  (test (eval-string "(+ a b)" e) 44)
  (test (eval-string "(+ a b c)" (augment-environment e (cons 'c 3))) 47)
  (test (eval-string "(+ a b)" (augment-environment e (cons 'b 3))) 35)
  )

(test (with-environment (augment-environment '() '(a . 1)) (defined? 'a)) #t)
(test (defined? 'a (augment-environment '() '(a . 1))) #t)
(test (defined? 'b (augment-environment '() '(a . 1))) #f)
(test (defined? 'a '((a . 1))) 'error)
(test (defined? 'a '((a . 1) 2)) 'error)
(test (defined? 'a (augment-environment '())) #f)

(test (symbol->value 'a (augment-environment '() '(a . 1))) 1)
(test (symbol->value 'b (augment-environment '() '(a . 1))) #<undefined>)
(test (symbol->value 'a '((a . 1))) 'error)
(test (symbol->value 'a '((a . 1) 2)) 'error)

(test (eval 'a (augment-environment '() '(a . 1))) 1)
(test (eval 'a (augment-environment '() '(b . 1))) 'error)
(test (eval 'a '((a . 1))) 'error)
(test (eval 'a '((a . 1) 2)) 'error)

(test (eval-string "a" (augment-environment '() '(a . 1))) 1)
(test (eval-string "a" (augment-environment '() '(b . 1))) 'error)
(test (eval-string "a" '((a . 1))) 'error)
(test (eval-string "a" '((a . 1) 2)) 'error)

(test (with-environment (augment-environment '() '(a . 1)) a) 1)
(test (with-environment (augment-environment '()) 1) 1)
(test (with-environment (augment-environment '() '(b . 1)) a) 'error)
(test (with-environment '((a . 1)) a) 'error)
(test (with-environment '((a . 1) 2) a) 'error)

(for-each
 (lambda (arg)
   (test (augment-environment (current-environment) arg) 'error)
   (test (augment-environment! (current-environment) arg) 'error))
 (list -1 #\a #(1 2 3) 3.14 3/4 1.0+1.0i 'hi "hi" abs '#(()) (list 1 2 3) '(1 . 2) (lambda () 1)))

(test (with-environment (augment-environment (current-environment) (cons '+ (lambda args (apply * args)))) (+ 1 2 3 4)) 24)
(test (with-environment (current-environment) (let ((x 1)) x)) 1)

(test (let ((x 12))
	(let ((e (current-environment)))
	  (let ((x 32))
	    (with-environment e (* x 2)))))
      24)

(test (let ((e #f)) (let ((x 2) (y 3)) (set! e (current-environment))) (let ((x 0) (y 0)) (with-environment e (+ x y)))) 5)
(test (let ((e #f)) (let () (define (func a b) (set! e (current-environment)) (+ a b)) (func 1 2)) (with-environment e (+ a b))) 3)
(test (let ((e #f)
	    (f #f))
	(let ()
	  (define (func a b) 
	    (set! e (current-environment)) 
	    (+ a b))
	  (set! f func)
	  (func 1 2))
	(let ((val (with-environment e (+ a b))))
	  (f 3 4)
	  (list val (with-environment e (+ a b)))))
      '(3 7))

(test (with-environment) 'error)
(test (with-environment 1) 'error)
(test (with-environment () 1) 'error)
(test (with-environment (current-environment)) 'error) ; ?? perhaps this should be #<unspecified> 
(for-each
 (lambda (arg)
   (test (with-environment arg #f) 'error))
 (list -1 #\a #(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" abs '#(()) (list 1 2 3) '(1 . 2) (lambda () 1)))

(test (with-environment (augment-environment (augment-environment '()) '(a . 1)) 1) 1)
(test (with-environment (augment-environment (augment-environment '()) '(a . 1)) a) 1)
(test (with-environment (current-environment) 1) 1)
(test (let ((a 1))
	(+ (with-environment
	    (augment-environment (current-environment) (cons 'a 10))
	    a)
	   a))
      11)
(test (let ((a 1))
	(+ (with-environment
	    (augment-environment (current-environment) (cons 'a 10))
	    (+ a
	       (with-environment
		(augment-environment (current-environment) (cons 'a 100))
		a)))
	   a))
      111)
(test (let ((a 1))
	(+ (with-environment
	    (augment-environment (current-environment) (cons 'a 10))
	    (+ a
	       (with-environment
		(augment-environment (current-environment) (cons 'b 100))
		a)))
	   a))
      21)
(test (let ((a 1))
	(let ((e (current-environment)))
	  (+ (with-environment
	      (augment-environment (current-environment) (cons 'a 10))
	      (+ a
		 (with-environment e a)))
	   a)))
      12)
(test (let ((a 1))
	(let ((e (current-environment)))
	  (+ (with-environment
	      (augment-environment! (augment-environment (current-environment) (cons 'a 10)) (cons 'a 20))
	      (+ a
		 (with-environment e a)))
	   a)))
      22)
(test (let ((a 1))
	(+ (with-environment
	    (augment-environment (current-environment) (cons 'a 10))
	    (+ (let ((b a))
		 (augment-environment! (current-environment) (cons 'a 20))
		 (+ a b))
	       a))
	   a))
      41)
(test (let ((a 1))
	(let ((e (current-environment)))
	  (+ (with-environment
	      (augment-environment e (cons 'a 10))
	      (+ a
		 (with-environment e a)))
	   a)))
      'error) ; "e" is not in the current-environment at the top, so it's not in the nested env

(test (let ((x 3))
	(augment-environment! (current-environment)
          (cons 'y 123))
	(+ x y))
      126)

(test (let ()
	(define (hiho a) (+ a b))
	(augment-environment! (procedure-environment hiho) (cons 'b 21)) ; hmmm...
	(hiho 1))
      22)

(test (let ()
	(define hiho (let ((x 32)) (lambda (a) (+ a x b))))
	(augment-environment! (procedure-environment hiho) (cons 'b 10) (cons 'x 100))
	(hiho 1))
      111)

(test (let ()
	(define hiho (let () 
		       (define (hi b) 
			 (+ b 1)) 
		       (lambda (a) 
			 (hi a))))
	(augment-environment! (procedure-environment hiho) (cons 'hi (lambda (b) (+ b 123))))
	(hiho 2))
      125)

(test (let () ; here's one way for multiple functions to share a normal scheme closure
	(define f1 (let ((x 23))
		     (lambda (a)
		       (+ x a))))
	(define f2
	  (with-environment (procedure-environment f1)
            (lambda (b)
	      (+ b (* 2 x)))))
	(+ (f1 1) (f2 1)))
      71)

(test (augment-environment!) 'error)
(test (augment-environment 3) 'error)
(test (augment-environment! 3) 'error)

(test (let ((e (current-environment))) (environment? e)) #t)
(test (let ((f (lambda (x) (environment? x)))) (f (current-environment))) #t)
(test (let ((e (augment-environment! '() '(a . 1)))) (environment? e)) #t)
(test (let ((e (augment-environment! '() '(a . 1)))) ((lambda (x) (environment? x)) e)) #t)
(test (environment? ((lambda () (current-environment)))) #t)
(test (environment? ((lambda (x) x) (current-environment))) #t)
(test (let ((e (let ((x 32)) (lambda (y) (let ((z 123)) (current-environment))))))
	(eval `(+ x y z) (e 1)))
      156)
(test (let ((e #f)) (set! e (let ((x 32)) (lambda (y) (let ((z 123)) (procedure-environment e)))))
	   (eval `(+ x 1) (e 1)))
      33)


(test (catch #t
	     (lambda ()
	       (with-environment (current-environment)
		 (error 'testing "a test")
		 32))
	     (lambda args (car args)))
      'testing)
(test (call-with-exit
       (lambda (go)
	 (with-environment (current-environment)
	   (go 1)
	   32)))
      1)

(test (let ((x 0))
	(call-with-exit
	 (lambda (go)
	   (with-environment (augment-environment! (current-environment) (cons 'x 123))
            (go 1))))
	x)
      0)
(test (let ((x 1))
	(+ x (call-with-exit
	      (lambda (go)
		(with-environment (augment-environment! (current-environment) (cons 'x 123))
                  (go x))))
	   x))
      125)

(test (let ((x 0))
	(catch #t
          (lambda ()
	    (with-environment (augment-environment! (current-environment) (cons 'x 123))
              (error 'oops) x)) 
	  (lambda args x)))
      0)
(test (call-with-exit (lambda (c) (0 (c 1)))) 1)
(test (call-with-exit (lambda (k) (k "foo"))) "foo")
(test (call-with-exit (lambda (k) "foo")) "foo")
(test (call-with-exit (lambda (k) (k "foo") "oops")) "foo")
(test (let ((memb (lambda (x ls)
		    (call-with-exit
		     (lambda (break)
		       (do ((ls ls (cdr ls)))
			   ((null? ls) #f)
			 (if (equal? x (car ls))
			     (break ls))))))))
	(list (memb 'd '(a b c))
	      (memb 'b '(a b c))))
      '(#f (b c)))

(let ((x 1))
  (define y (call-with-exit (lambda (return) (set! x (return 32)))))
  (test (and (= x 1) (= y 32)) #t)
  (set! y (call-with-exit (lambda (return) ((lambda (a b c) (set! x a)) 1 2 (return 33)))))
  (test (and (= x 1) (= y 33)) #t)
  (set! y (call-with-exit (lambda (return) ((lambda (a b) (return a) (set! x b)) 2 3))))
  (test (and (= x 1) (= y 2)) #t))

(test (apply "hi" '(1 2)) 'error)
(test ("hi" 1 2) 'error)
(test (apply '(1 2) '(1 2)) 'error)
(test ((list 1 2 3) 1 2) 'error)

(test (apply "hi" '(1)) #\i)
(test ("hi" 1) #\i)
(test (apply '(1 2) '(1)) 2)
(test ((list 1 2 3) 1) 2)

(test (let ((pi 3)) pi) 'error)
(test (let ((:key 1)) :key) 'error)
(test (let ((:3 1)) 1) 'error)
(test (let ((3 1)) 1) 'error)
(test (let ((3: 1)) 1) 'error)
(test (let ((optional: 1)) 1) 'error)
(test (let ((x_x_x 32)) (let () (define-constant x_x_x 3) x_x_x) (set! x_x_x 31) x_x_x) 'error)


;;; make-procedure-with-setter
(test (let ((local 123))
	(define pws-test (make-procedure-with-setter
			  (lambda () local)
			  (lambda (val) (set! local val))))
	(pws-test))
      123)

(test (let ((local 123))
	(define pws-test (make-procedure-with-setter
			  (lambda () local)
			  (lambda (val) (set! local val))))
	(pws-test 32))
      'error)

(test (let ((local 123))
	(define pws-test (make-procedure-with-setter
			  (lambda () local)
			  (lambda (val) (set! local val))))
	(set! (pws-test 32) 123))
      'error)

(test (call-with-exit 
       (lambda (return) 
	 (let ((local 123))
	   (define pws-test (make-procedure-with-setter
			     (lambda () (return "oops"))
			     (lambda (val) (set! local val))))
	   (pws-test))))
      "oops")
(test (call-with-exit 
       (lambda (return)
	 (let ((local 123))
	   (define pws-test (make-procedure-with-setter
			     (lambda () 123)
			     (lambda (val) (return "oops"))))
	   (set! (pws-test) 1))))
      "oops")

(test (let ((local 123))
	(define pws-test (make-procedure-with-setter
			  (lambda () local)
			  (lambda (val) (set! local val))))
	(set! (pws-test) 321)
	(pws-test))
      321)

(test (let ((v (vector 1 2 3)))
	(define vset (make-procedure-with-setter
		      (lambda (loc)
			(vector-ref v loc))
		      (lambda (loc val)
			(vector-set! v loc val))))
	(let ((lst (list vset)))
	  (let ((val (vset 1)))
	    (set! (vset 1) 32)
	    (let ((val1 (vset 1)))
	      (set! ((car lst) 1) 3)
	      (list val val1 (vset 1))))))
      (list 2 32 3))

(let ((local 123))
  (define pws-test (make-procedure-with-setter
		    (lambda () local)
		    (lambda (val) (set! local val))))
  (test (procedure-with-setter? pws-test) #t)
  (test (pws-test) 123)
  (set! (pws-test) 32)
  (test (pws-test) 32)
  (set! (pws-test) 0)
  (test (pws-test) 0))

(let ((local 123))
  (define pws-test (make-procedure-with-setter
		    (lambda (val) (+ local val))
		    (lambda (val new-val) (set! local new-val) (+ local val))))
  (test (pws-test 1) 124)
  (set! (pws-test 1) 32)
  (test (pws-test 2) 34)
  (set! (pws-test 3) 0)
  (test (pws-test 3) 3))


(test (make-procedure-with-setter) 'error)
(test (make-procedure-with-setter abs) 'error)
(test (make-procedure-with-setter 1 2) 'error)
(test (make-procedure-with-setter (lambda () 1) (lambda (a) a) (lambda () 2)) 'error)
(test (make-procedure-with-setter (lambda () 1) 2) 'error)

(for-each
 (lambda (arg)
   (test (make-procedure-with-setter arg (lambda () #f)) 'error)
   (test (make-procedure-with-setter (lambda () #f) arg) 'error))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((pws (make-procedure-with-setter vector-ref vector-set!)))
  (let ((v (vector 1 2 3)))
    (test (procedure-with-setter? pws) #t)
    (test (procedure-with-setter? pws pws) 'error)
    (test (pws v 1) 2)
    (set! (pws v 1) 32)
    (test (pws v 1) 32)
    (test (procedure-arity pws) '(2 0 #t))
    (test (procedure-arity (procedure-setter pws)) '(3 0 #t))))

(for-each
 (lambda (arg)
   (test (procedure-with-setter? arg) #f))
 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" '#(()) abs (lambda () #f) (list (lambda () #f) (lambda (val) val)) (list 1 2 3) '(1 . 2) #<eof> #<unspecified> #<undefined>))

(test (procedure-with-setter?) 'error)
(test (call-with-exit (lambda (return) (procedure-with-setter? return))) #f)
(test (procedure-with-setter? quasiquote) #f)
;; (test (procedure-with-setter? -s7-symbol-table-locked?) #t)
;; (test (procedure-with-setter? '-s7-symbol-table-locked?) #f) ; this parallels (procedure? 'abs) -> #f but seems inconsistent with other *? funcs

(define (procedure-setter-arity proc) (procedure-arity (procedure-setter proc)))
(test (let ((pws (make-procedure-with-setter (lambda () 1) (lambda (a) a)))) (procedure-setter-arity pws)) '(1 0 #f))
(test (let ((pws (make-procedure-with-setter (lambda () 1) (lambda (a b c) a)))) (procedure-setter-arity pws)) '(3 0 #f))
(test (let ((pws (make-procedure-with-setter (lambda () 1) (lambda (a . b) a)))) (procedure-setter-arity pws)) '(1 0 #t))
(test (let ((pws (make-procedure-with-setter (lambda () 1) (lambda* (a (b 1)) a)))) (procedure-setter-arity pws)) '(0 2 #f))
(test (let ((pws (make-procedure-with-setter (lambda () 1) (lambda* (a :rest b) a)))) (procedure-setter-arity pws)) '(0 1 #t))
;; (test (procedure-setter-arity symbol-access) '(2 0 #f))
(test (let ((pws (make-procedure-with-setter (lambda args (apply + args)) (lambda args (apply * args))))) (pws 2 3 4)) 9)
(test (let ((pws (make-procedure-with-setter (lambda args (apply + args)) (lambda args (apply * args))))) (set! (pws 2 3 4) 5)) 120)
(let ((x 0)) 
  (let ((pws (make-procedure-with-setter
	      (let ((y 1)) ((lambda () (set! x (+ x y)) (lambda () x))))
	      (let ((y 2)) ((lambda () (set! x (* x y)) (lambda (z) (set! x (+ x z)))))))))
    (test x 2)
    (set! (pws) 3)
    (test x 5)))

(let ((p1 (make-procedure-with-setter (lambda () 1) (lambda (a) a))))
  (let ((p2 (make-procedure-with-setter p1 p1)))
    (test (p2) 1)))
(let () (define-macro (hi a) `(+ ,a 1)) (test (make-procedure-with-setter hi hi) 'error))
(test (make-procedure-with-setter quasiquote call/cc) 'error)
(test ((make-procedure-with-setter call-with-exit call/cc) (lambda (a) (a 1))) 1)
(test (length (make-procedure-with-setter < >)) 'error)

(let ((p1 (make-procedure-with-setter (lambda (a) (+ a 1)) (lambda (a b) (+ a b)))))
  (let ((p2 (make-procedure-with-setter p1 p1)))
    (test (p2 1) 2)))



;; generic length/reverse/copy/fill!
;;; copy
;;; fill!

(test (length (list 1 2)) 2)
(test (length "hiho") 4)
(test (length (vector 1 2)) 2)
(test (>= (length (make-hash-table 7)) 7) #t)
(test (length '()) 0)
(test (length (#(#() #()) 1)) 0)
(test (length abs) 'error)

(test (copy 3) 3)
(test (copy 3/4) 3/4)
(test (copy "hi") "hi")
(test (copy 'hi) 'hi)
(test (copy (list 1 2 3)) (list 1 2 3))
(test (copy (vector 0.0)) (vector 0.0))
(test (copy #\f) #\f)
(test (copy (list 1 (list 2 3))) (list 1 (list 2 3)))
(test (copy (cons 1 2)) (cons 1 2))
(test (let ((x (list 1 2 3))) (eq? (copy x) x)) #f)
(test (let ((x (list 1 2 3))) (equal? (copy x) x)) #t)
(test (let ((x #(1 2 3))) (eq? (copy x) x)) #f)
(test (let ((x #(1 2 3))) (equal? (copy x) x)) #t)
(test (let ((x "hi")) (eq? (copy x) x)) #f)
(test (let ((x "hi")) (equal? (copy x) x)) #t)
(test (copy '(1 2 . 3)) '(1 2 . 3))
(test (copy (+)) 0)
(test (copy +) +)
(test (copy (#(#() #()) 1)) #())
(test (copy #f) #f)
(test (copy '()) '())
(test (copy #()) #())
(test (copy #2d((1 2) (3 4))) #2d((1 2) (3 4)))
(test (let ((f (lambda () 1))) ((copy f))) 1) ; here copy actually returns f: 
(test (let ((f (lambda () 1))) (eq? (copy f) f)) #t)
(test (copy 1.0) 1.0)
(test (copy 1.0+i) 1.0+i)
(test (copy "") "")
(test (copy #t) #t)
(test (copy (string #\a #\null #\b)) "a\x00b")
(test (copy #<eof>) #<eof>)
(test ((copy abs) -123) 123)
(test (copy ''1) ''1)
(test (copy '''1) '''1)
(test (hook? (copy (make-hook 1))) #t)
(test (copy not) not)
(test (copy "a\x00b") "a\x00b")
(test (infinite? (copy (log 0.0))) #t)
(test (nan? (copy 1/0)) #t)
(test (copy if) if)
(test (copy quote) quote)


(test (reverse "hi") "ih")
(test (reverse "") "")
(test (reverse "123") "321")
(test (reverse "1234") "4321")
(test (reverse "a\x00b") "b\x00a")
(test (reverse #()) #())
(test (reverse #(1 2 3)) #(3 2 1))
(test (reverse #(1 2 3 4)) #(4 3 2 1))
(test (reverse #2D((1 2) (3 4))) #2D((4 3) (2 1)))
(test (reverse (string #\a #\null #\b)) "b\x00a")
(test (reverse abs) 'error)

(if (not (provided? 'gmp))
    (let ((r1 (make-random-state 1234)))
      (random 1.0 r1)
      (let ((r2 (copy r1)))
	(let ((v1 (random 1.0 r1))
	      (v2 (random 1.0 r2)))
	  (test (= v1 v2) #t)
	  (let ((v3 (random 1.0 r1)))
	    (random 1.0 r1)
	    (random 1.0 r1)
	    (let ((v4 (random 1.0 r2)))
	      (test (= v3 v4) #t)))))))

(if (provided? 'gmp)
    (let ((i (copy (bignum "1")))
	  (r (copy (bignum "3/4")))
	  (f (copy (bignum "1.5")))
	  (c (copy (bignum "1.0+1.0i"))))
      (test (= i (bignum "1")) #t)
      (test (= r (bignum "3/4")) #t)
      (test (= f (bignum "1.5")) #t)
      (test (= c (bignum "1.0+1.0i")) #t)))

(let ((str (string #\1 #\2 #\3)))
  (fill! str #\x)
  (test str "xxx"))
(let ((v (vector 1 2 3)))
  (fill! v 0.0)
  (test v (vector 0.0 0.0 0.0)))
(let ((lst (list 1 2 (list (list 3) 4))))
  (fill! lst 100)
  (test lst '(100 100 100)))
(let ((cn (cons 1 2)))
  (fill! cn 100)
  (test cn (cons 100 100)))
(test (fill! 1 0) 'error)
(test (fill! 'hi 0) 'error)

(test (fill!) 'error)
(test (copy) 'error)
(test (fill! '"hi") 'error)
(test (fill! (begin) if) if)

(for-each
 (lambda (arg)
   (test (fill! arg 1) 'error))
 (list (integer->char 65) #f 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (let ((str (string #\a #\b)))
     (test (fill! str arg) 'error)))
 (list "hi" '(1 2 3) #() #f 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(let ((c1 #f))
  (call/cc
   (lambda (c)
     (test (reverse c) 'error)
     (test (fill! c) 'error)
     (test (length c) 'error)
     (test (eq? c c) #t) ; is this the norm?
     (test (equal? c c) #t)
     (test (equal? c (copy c)) #t)
     (set! c1 c)))
  (test (continuation? c1) #t))

(let ((c1 #f))
  (call-with-exit
   (lambda (c)
     (test (reverse c) 'error)
     (test (fill! c) 'error)
     (test (length c) 'error)
     (test (eq? c c) #t) 
     (test (equal? c c) #t)
     (test (equal? c (copy c)) #t)
     (set! c1 c)))
  (test (procedure? c1) #t))

(test (let ((lst '(1 2 3))) (fill! lst (cons 1 2)) (set! (car (car lst)) 3) (caadr lst)) 3)


;; generic for-each/map
(test (let ((sum 0)) (for-each (lambda (n) (set! sum (+ sum n))) (vector 1 2 3)) sum) 6)      
(test (map (lambda (n) (+ n 1)) (vector 1 2 3)) '(2 3 4))
(test (map (lambda (a b) (/ a b)) (list 1 2 3) (list 4 5 6)) '(1/4 2/5 1/2))

;; try some applicable stuff
(test (let ((lst (list 1 2 3)))
	(set! (lst 1) 32)
	(list (lst 0) (lst 1)))
      (list 1 32))

(test (let ((hash (make-hash-table)))
	(set! (hash 'hi) 32)
	(hash 'hi))
      32)

(test (let ((str (string #\1 #\2 #\3)))
	(set! (str 1) #\a)
	(str 1))
      #\a)

(test (let ((v (vector 1 2 3)))
	(set! (v 1) 0)
	(v 1))
      0)

(let ()
  (define (hiho a) __func__)
  (test (or (equal? (hiho 1) 'hiho)
	    (equal? (car (hiho 1)) 'hiho))
	#t))


(test (stacktrace #(23)) 'error)
(for-each
 (lambda (arg)
   (test (stacktrace arg) 'error))
 (list "hi" '(1 2 3) 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i 1 '() "" (if #f #f) #<eof> (lambda (a) (+ a 1))))
(test (stacktrace *error-info* 1) 'error)

(let ((val (catch #t (lambda () (/ 1 0.0)) (lambda args args))))
  (let* ((tag (car val))
	 (descr (cadr val))
	 (cur-info *error-info*))
    (test tag 'division-by-zero)
    (test descr '("~A: division by zero, ~A" "/" 0.0)) ; this changes...
    (test (vector? cur-info) #t)
    (test (> (length cur-info) 5) #t)
    (test tag (cur-info 0))
    (test descr (cur-info 1))
    (test (equal? (cur-info 2) '(/ 1 0.0)) #t)
    (test (or (not (cur-info 3)) (integer? (cur-info 3))) #t) ; line-number
    (test (or (not (cur-info 4)) (string? (cur-info 4))) #t) ; file name
    ))


(for-each
 (lambda (arg)
   (test (gc arg) 'error))
 (list "hi" '(1 2 3) #() 'a-symbol abs _ht_ quasiquote macroexpand make-type hook-functions 
       3.14 3/4 1.0+1.0i 1 '() "" (if #f #f) (lambda (a) (+ a 1))))

(test (gc #f #t) 'error)
;(test (catch #t (lambda () (gc) #f) (lambda args 'error)) #f)
;(test (dynamic-wind (lambda () (gc)) (lambda () (gc) #f) (lambda () (gc))) #f)


;;; -------- tail recursion tests

(let ((max-stack 0))
  (define (tc-1 a c) 
    (let ((b (+ a 1))) 
      (if (> (-s7-stack-size) max-stack)
	  (set! max-stack (-s7-stack-size)))
      (if (< b c) 
	  (tc-1 b c))))
  (tc-1 0 32)
  (if (> max-stack 10) (format #t "tc-1 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-1 a c) 
    (if (> (-s7-stack-size) max-stack)
	(set! max-stack (-s7-stack-size)))
    (if (< a c) 
	(tc-2 (+ a 1) c)))
  (define (tc-2 a c) 
    (if (> (-s7-stack-size) max-stack)
	(set! max-stack (-s7-stack-size)))
    (if (< a c) 
	(tc-1 (+ a 1) c)))
  (tc-1 0 32)
  (if (> max-stack 10) (format #t "tc-1-1 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-2 a c) 
    (let ((b (+ a 1))) 
      (if (> (-s7-stack-size) max-stack)
	  (set! max-stack (-s7-stack-size)))
      (if (= b c)
	  #f
	  (tc-2 b c))))
  (tc-2 0 32)
  (if (> max-stack 10) (format #t "tc-2 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-2 a c) 
    (let ((b (+ a 1))) 
      (if (> (-s7-stack-size) max-stack)
	  (set! max-stack (-s7-stack-size)))
      (if (< b c)
	  (tc-2 b c)
	  #f)))
  (tc-2 0 32)
  (if (> max-stack 10) (format #t "tc-2-1 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-3 a c) 
    (let ((b (+ a 1))) 
      (if (> (-s7-stack-size) max-stack)
	  (set! max-stack (-s7-stack-size)))
      (cond ((= b c) #f)
	    ((< b c)
	     (tc-3 b c)))))
  (tc-3 0 32)
  (if (> max-stack 10) (format #t "tc-3 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-4 a c) 
    (let ((b (+ a 1))) 
      (if (> (-s7-stack-size) max-stack)
	  (set! max-stack (-s7-stack-size)))
      (cond ((= b c) #f)
	    (else (tc-4 b c)))))
  (tc-4 0 32)
  (if (> max-stack 10) (format #t "tc-4 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-5 a c) 
    (let ((b (+ a 1))) 
      (if (> (-s7-stack-size) max-stack)
	  (set! max-stack (-s7-stack-size)))
      (case b
	((32) #f)
	(else (tc-5 b c)))))
  (tc-5 0 32)
  (if (> max-stack 10) (format #t "tc-5 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-6 a c) 
    (let ((b (+ a 1))) 
      (if (> (-s7-stack-size) max-stack)
	  (set! max-stack (-s7-stack-size)))
      (case b
	((17) #f)
	((0 1 2 3 4 5 6 7 8) (tc-6 b c))
	((9 10 11 12 13 14 15 16) (tc-6 b c)))))
  (tc-6 0 32)
  (if (> max-stack 10) (format #t "tc-6 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-7 a c) 
    (let ((b (+ a 1))) 
      (if (> (-s7-stack-size) max-stack)
	  (set! max-stack (-s7-stack-size)))
      (or (>= b c)
	  (tc-7 b c))))
  (tc-7 0 32)
  (if (> max-stack 10) (format #t "tc-7 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-8 a c) 
    (let ((b (+ a 1))) 
      (if (> (-s7-stack-size) max-stack)
	  (set! max-stack (-s7-stack-size)))
      (and (< b c)
	   (tc-8 b c))))
  (tc-8 0 32)
  (if (> max-stack 10) (format #t "tc-8 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-9 a c) 
    (let tc-9a ((b a))
      (if (> (-s7-stack-size) max-stack)
	  (set! max-stack (-s7-stack-size)))
      (if (< b c)
	  (tc-9a (+ b 1)))))
  (tc-9 0 32)
  (if (> max-stack 10) (format #t "tc-9 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-10 a c) 
    (let* ((b (+ a 1))) 
      (if (> (-s7-stack-size) max-stack)
	  (set! max-stack (-s7-stack-size)))
      (and (< b c)
	   (tc-10 b c))))
  (tc-10 0 32)
  (if (> max-stack 10) (format #t "tc-10 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-11 a c) 
    (letrec ((b (+ a 1))) 
      (if (> (-s7-stack-size) max-stack)
	  (set! max-stack (-s7-stack-size)))
      (and (< b c)
	   (tc-11 b c))))
  (tc-11 0 32)
  (if (> max-stack 10) (format #t "tc-11 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-12 a c) 
    (if (< a c)
	(begin
	  (if (> (-s7-stack-size) max-stack)
	      (set! max-stack (-s7-stack-size)))
	  (tc-12 (+ a 1) c))))
  (tc-12 0 32)
  (if (> max-stack 10) (format #t "tc-12 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-13 a c) 
    (if (> (-s7-stack-size) max-stack)
	(set! max-stack (-s7-stack-size)))
    (cond ((= a c) #f)
	  ((< a c)
	   (if (> a c) (display "oops"))
	   (tc-13 (+ a 1) c))))
  (tc-13 0 32)
  (if (> max-stack 10) (format #t "tc-13 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-14 a c) 
    (if (> (-s7-stack-size) max-stack)
	(set! max-stack (-s7-stack-size)))
    (cond ((>= a c) #f)
	  ((values (+ a 1) c) => tc-14)))
  (tc-14 0 32)
  (if (> max-stack 10) (format #t "tc-14 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-15 a c) 
    (if (> (-s7-stack-size) max-stack)
	(set! max-stack (-s7-stack-size)))
    (or (>= a c)
	(apply tc-15 (list (+ a 1) c))))
  (tc-15 0 32)
  (if (> max-stack 10) (format #t "tc-15 max: ~D~%" max-stack)))

(let ((max-stack 0))
  (define (tc-17 a c) 
    (if (> (-s7-stack-size) max-stack)
	(set! max-stack (-s7-stack-size)))
    (or (and (>= a c) a)
	(eval `(tc-17 (+ ,a 1) ,c))))
  (let ((val (tc-17 0 32)))
    (test (and (= val 32) (< max-stack 28)) #t)))

#|
(let ((max-stack 0))
  (define (tc-19 a c) 
    (if (> (-s7-stack-size) max-stack)
	(set! max-stack (-s7-stack-size)))
    (call/cc
     (lambda (r)
       (if (>= a c) (r a))
       (tc-19 (+ a 1) c))))
  (let ((val (tc-19 0 16)))
    (test (and (= val 16) (< max-stack 8)) #t)))
|#

(let ((max-stack 0))
  (define (tc-21 a) 
    (if (< a 32)
	(do ((i (- a 1) (+ i 1)))
	    ((= i a) 
	     (tc-21 (+ a 1)))
	  (if (> (-s7-stack-size) max-stack)
	      (set! max-stack (-s7-stack-size))))
	a))
  (let ((val (tc-21 0)))
    (if (> max-stack 10) (format #t "tc-21 max: ~D~%" max-stack))
    (if (not (= val 32)) (format #t "tc-21 returned: ~A~%" val))))

(let ((max-stack 0))
  (define (tc-env a c) 
    (with-environment (augment-environment (current-environment) (cons 'b (+ a 1)))
      (if (> (-s7-stack-size) max-stack)
	  (set! max-stack (-s7-stack-size)))
      (if (< b c) 
	  (tc-env b c))))
  (tc-env 0 32)
  (if (> max-stack 10) (format #t "tc-env max: ~D~%" max-stack)))


;;; make sure for-each and map aren't messed up

(let ((max-stack 0))
  (for-each
   (lambda (a)
     (if (> (-s7-stack-size) max-stack)
	 (set! max-stack (-s7-stack-size)))
     (if (not (= a 1))
	 (error 'wrong-type-arg ";for-each arg is ~A" a)))
   (make-list 100 1))
  (test (< max-stack 20) #t)) ; 10 is not snd-test (and below)

(let ((max-stack 0))
  (map
   (lambda (a)
     (if (> (-s7-stack-size) max-stack)
	 (set! max-stack (-s7-stack-size)))
     (if (not (= a 1))
	 (error 'wrong-type-arg ";map arg is ~A" a)))
   (make-list 100 1))
  (test (< max-stack 20) #t))


;;; the next 3 are not tail-recursive
;;;
;;;   eval-string pushes stack markers to catch multiple statements
;;;   OP_DEACTIVATE_GOTO in call-with-exit
;;;   OP_DYNAMIC_WIND in the dynamic-wind case

(let ((max-stack 0))
  (define (tc-17 a c) 
    (if (> (-s7-stack-size) max-stack)
	(set! max-stack (-s7-stack-size)))
    (or (and (>= a c) a)
	(eval-string (format #f "(tc-17 (+ ~A 1) ~A)" a c))))
  (let ((val (tc-17 0 32)))
    (test (and (= val 32) (< max-stack 28)) #f)))

(let ((max-stack 0))
  (define (tc-16 a c) 
    (if (> (-s7-stack-size) max-stack)
	(set! max-stack (-s7-stack-size)))
    (call-with-exit
     (lambda (r)
       (if (>= a c) (r a))
       (tc-16 (+ a 1) c))))
  (let ((val (tc-16 0 32)))
    (test (and (= val 32) (> max-stack 28)) #t)))

(let ((max-stack 0))
  (define (tc-18 a c) 
    (dynamic-wind
	(lambda ()
	  (if (> (-s7-stack-size) max-stack)
	      (set! max-stack (-s7-stack-size))))
	(lambda ()
	  (or (and (>= a c) a)
	      (tc-18 (+ a 1) c)))
	(lambda ()
	  #f)))
  (let ((val (tc-18 0 32)))
    (test (and (= val 32) (> max-stack 28)) #t)))




;;; -------- miscellaneous amusements

(test ((number->string -1) 0) #\-)
(test ((reverse '(1 2)) 0) 2)
(test ((append begin) list) list)
(test ((begin object->string) car) "car")
(test ((and abs) -1) 1)
(test (((values begin) object->string) car) "car")
(test (((values (begin begin)) object->string) car) "car")
(test ((((values append) begin) object->string) car) "car")
(test ((((((values and) or) append) begin) object->string) car) "car")
(test (((((((values values) and) or) append) begin) object->string) car) "car")
(test ((((eval lambda) lcm gcd))) 0)
(test (((append s7-version)) 0) #\s)
(test ((values (lambda hi #()))) #())
(test (((((lambda () (lambda () (lambda () (lambda () 1)))))))) 1)
(test (((cond (cond => cond)) (cond)) ((cond (#t #t)))) #t)
(test ((object->string #f) (ceiling 3/4)) #\f)
(test (((lambda* ((a :optional) (b :key)) (apply lambda* (list (list a b 'c) 'c)))) 1) 1) ; (lambda* (:optional :key c) c)
(test (procedure? ((((((lambda* ((x (lambda () x))) x))))))) #t)
(test (procedure? ((((((letrec ((x (lambda () x))) x))))))) #t)
(test (procedure? ((((((letrec ((x (lambda () y)) (y (lambda () x))) x))))))) #t)
(test (procedure? ((((((let x () x))))))) #t)
(test (procedure? ((((((lambda (x) (set! x (lambda () x))) (lambda () x))))))) #t)
(test ((do ((i 0 (+ i 1))) ((= i 1) (lambda () 3)))) 3)
(test (dynamic-wind s7-version s7-version s7-version) (s7-version))
(test ((((lambda - -) -) 0) 1) -1)
(num-test ((list .(log 0)) 1) 0)
(num-test (((cons .(log 0)) 0) 1) 0.0)

(test (+ (+) (*)) 1)
(test (modulo (lcm) (gcd)) 1)
(test (max (+) (*)) 1)
(test (min (gcd) (lcm)) 0)
(test (symbol->value (gensym) (global-environment)) #<undefined>)
(test (string-ref (s7-version) (*)) #\7)
(test (string>=? (string-append) (string)) #t)
(test (substring (string-append) (+)) "")
(test (ash (*) (+)) 1)
(test (> (*) (+)) #t)
(test ((or #f list)) ())
(test ((or #f lcm)) 1)
(test ((or begin symbol?)) ())
(test ((or begin make-polar)) ())
(test ((and #t begin)) ())
(test (begin) ())
(test ((or #f lcm) 2 3) 6)
(test ((or and) #f #t) #f)
(test ((and or) #f #t) #t)
(test (or (or) (and)) #t)
(test ((car '((1 2) (3 4))) 0) 1)
(test ((or apply) lcm) 1)
(test (- ((or *))) -1)
(test ((car (list lcm))) 1)
(test ((or (cond (lcm)))) 1)
(test ((cond (asin floor *))) 1)
(test (logior (#(1 #\a (3)) 0) (truncate 1.5)) 1)
(test (real? (*)) #t)
(test (- (lcm)) -1)
(test (* (*)) 1)
(test (+ (+) (+ (+)) (+ (+ (+)))) 0)
(test (+(*(+))(*)(+(+)(+)(*))) 2)
(num-test (+(-(*).(+1))(*(+).(-1))(*(+).(-10))(*(-(+)0)1.)(-(+)(*).01)(*(-(+)).01)(-(+)(*)1.0)(-(*(+))1.0)(*(-(+))1.0)(-(+(*)1).0))-2.01)
(num-test (+(-(*).(+1.0))(*(+).(-1.0))(-(+)1.(*)0.)(-(*(+)0.)1.)(-(+(*)1.)0.)(+(-(*)0.)1.))1.0)
;; float comparison so can't use direct '=' here

(test (nan? (asinh (cos (real-part (log 0.0))))) #t)
(num-test(cos(sin(log(tan(*))))) 0.90951841537482)
(num-test (asinh (- 9223372036854775807)) -44.361419555836)
(num-test (imag-part (asin -9223372036854775808)) 44.361419555836)
(if (provided? 'dfls-exponents)
    (begin
      ;; proof that these exponents should be disallowed
      (num-test (string->number "1l1") 10.0)
      (num-test (string->number "1l1+1l1i") 10+10i)
      (num-test (string->number "1l11+11l1i") 100000000000+110i)
      (num-test (string->number "#d1d1") 10.0)
      (num-test (string->number "#d0001d0001") 10.0)))
(test (#|#<|# = #|#f#|# #o#e0 #|#>|# #e#o0 #|#t#|#) #t)
(num-test (apply * (map (lambda (r) (sin (* pi (/ r 130)))) (list 1 67 69 73 81 97))) (/ 1.0 64))
(num-test (max 0(+)(-(*))1) 1)

(test ((call-with-exit object->string) 0) #\#) ; #<goto>
(test ((begin begin) 1) 1)
(test ((values begin) 1) 1)
(test ((provide or) 3/4) 'error)
(test (string? cond) #f)
(test (list? or) #f)
(test (pair? define) #f)
(test (number? lambda*) #f)
(test ((s7-version) (rationalize 0)) #\s)
(test (cond (((values '(1 2) '(3 4)) 0 0))) 'error)
(test (cond (((#2d((1 2) (3 4)) 0) 0) 32)) 32)
(test (cond ((apply < '(1 2)))) #t)
(test (dynamic-wind lcm gcd *) 'error)
(test (case 0 ((> 0 1) 32)) 32)
(test (char-downcase (char-downcase #\newline)) #\newline)
(test (and (and) (and (and)) (and (and (and (or))))) #f)
(test ((values begin #\a 1)) 1)
(test ((values and 1 3)) 3)
(test ((((lambda () begin)) (values begin 1))) 1)
(test (+ (((lambda* () values)) 1 2 3)) 6)
(test ((values ('((1 2) (3 4)) 1) (abs -1))) 4)
(test ((apply lambda '() '(() ()))) '())
(test ((lambda* ((symbol "#(1 #\\a (3))")) #t)) #t)
(test (apply if ('((1 2) (3 4)) 1)) 4)

(test (((lambda #\newline gcd))) 'error)
(test (symbol? (let () (define (hi) (+ 1 2)))) #t)
(test (symbol? (begin (define (x y) y) (x (define (x y) y)))) #t)
(test (symbol? (do () ((define (x) 1) (define (y) 2)))) #t)
(test (cond (0 (define (x) 3) (x))) 3)
(test (let ((x (lambda () 3))) (if (define (x) 4) (x) 0)) 4)
(test (and (define (x) 4) (+ (x) 1)) 5)
(test (do ((x (lambda () 3) (lambda () 4))) ((= (x) 4) (define (x) 5) (x))) 5)
(test (begin (if (define (x) 3) (define (x) 4) (define (x) 5)) (x)) 4)

(test (let ((1,1 3) (1'1 4) (1|1 5) (1#1 6) (1\1 7) (1?1 8)) (+ 1,1 1'1 1|1 1#1 1\1 1?1)) 33)
(test (let ((,a 3)) ,a) 'error)
(test (let ((@a 3)) @a) 3)
(test (let (("a" 3)) "a") 'error)
(test (let ((`a 3)) `a) 'error)
(test (let (('a 3)) 'a) 'error)
(test (let ((a`!@#$%^&*~.,<>?/'{}[]\|+=_-a 3)) a`!@#$%^&*~.,<>?/'{}[]\|+=_-a) 3)

(test (set! ((quote (1 2)) 0) #t) #t)
(test (set! (((lambda () (list 1 2))) 0) 2) 2)
(test (let ((x (list 1 2))) (set! (((lambda () x)) 0) 3) x) '(3 2))
(test (let ((x (list 1 2))) (set! (((vector x) 0) 1) 32) x) '(1 32))
(test (let ((x (list 1 2))) (set! ((((lambda () (vector x))) 0) 0) 3) x) '(3 2))
(test (let ((x (list 1 2))) (set! ((((lambda () (list x))) 0) 0) 3) x) '(3 2))
(test (let ((x (list 1 2))) (set! ((set! x (list 4 3)) 0) 32) x) '(32 3))
(test (let ((x (list 1 2))) (list-set! (set! x (list 4 3)) 0 32) x) '(32 3))
(test (let ((x (list 1 2))) (set! ((list-set! x 0 (list 4 3)) 0) 32) x) '((32 3) 2))
(test (let ((x (list 1 2))) (list-set! (list-set! x 0 (list 4 3)) 0 32) x) '((32 3) 2))
(test (set! (('((0 2) (3 4)) 0) 0) 0) 0)
(test (set! ((map abs '(1 2)) 1) 0) 0)
(test (let () (set! ((define x #(1 2)) 0) 12) x) #(12 2))
(test (let ((x (list 1 2))) (set! ((call-with-exit (lambda (k) (k x))) 0) 12) x) '(12 2))
(test (let ((x #2d((1 2) (3 4)))) (set! (((values x) 0) 1) 12) x) #2D((1 12) (3 4)))
(test (let ((x 0)) (set! ((make-procedure-with-setter (lambda () x) (lambda (y) (set! x y)))) 12) x) 12)
(test (let ((x 0) (str "hiho")) (string-set! (let () (set! x 32) str) 0 #\x) (list x str)) '(32 "xiho"))
(test (let ((x "hi") (y "ho")) (set! ((set! x y) 0) #\x) (list x y)) '("xo" "xo"))
(test (let ((x "hi") (y "ho")) (set! x y) (set! (y 0) #\x) (list x y)) '("xo" "xo")) ; Guile gets the same result
(test (let ((x (lambda (a) (a z 1) z))) (x define)) 1) ; !
(test (let ((x (lambda (a) (a z (lambda (b) (+ b 1))) (z 2)))) (x define)) 3)
(test (let ((x (lambda (a b c) (apply a (list b c))))) (x let '() 3)) 3)
(test (let ((x (lambda (a b c) (apply a (list b c))))) (x let '((y 2)) '(+ y 1))) 3)

(let () (test ((values let '((x 1))) '(+ x 1)) 2)) ; !
(let () (test ((values begin '(define x 32)) '(+ x 1)) 33))
(let () (test (((values lambda '(x) '(+ x 1))) 32) 33))
(let () (test (let ((arg '(x)) (body '(+ x 1))) (((values lambda arg body)) 32)) 33))
(let () (test (let ((arg '(x)) (body '(+ x 1))) ((apply lambda arg (list body)) 32)) 33))
(let () (test (let ((x 12)) ((apply lambda '(x) (list (list '+ 1 x 'x))) 3)) 16))
(let () (test (let* ((x 3) (arg '(x)) (body `((+ ,x x 1)))) ((apply lambda arg body) 12)) 16))

(let ()
  (define (bcase start end func)
    (let ((body '()))
      (do ((i start (+ i 1)))
	  ((= i end))
	  (set! body (cons `((,i) ,(func i)) body)))
      (lambda (i) (apply case i body))))
  (test ((bcase 0 3 abs) 1) 1))






;;; ------ CLisms ------------------------------------------------------------------------


(let ()

      ;; **********************************************************************
      ;; 
      ;; Copyright (C) 2002 Heinrich Taube (taube@uiuc.edu) 
      ;; 
      ;; This program is free software; you can redistribute it and/or
      ;; modify it under the terms of the GNU General Public License
      ;; as published by the Free Software Foundation; either version 2
      ;; of the License, or (at your option) any later version.
      ;; 
      ;; This program is distributed in the hope that it will be useful,
      ;; but WITHOUT ANY WARRANTY; without even the implied warranty of
      ;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      ;; GNU General Public License for more details.
      ;; 
      ;; **********************************************************************
      
      ;; $Name:  $
      ;; $Revision: 1.6 $
      ;; $Date: 2005/11/17 13:29:37 $
      
      ;;
      ;; Implementation of the CLTL2 loop macro. The following 
      ;; non Rev 5 definitions need to be in effect before the file
      ;; is loaded:
      ;;
      ;; (define-macro (name . args) ...)
      ;; (error string)
      ;; (gensym string)
      ;;
      
      (defmacro when (test . forms)
	`(if ,test (begin ,@forms)))
	  
      (define-macro (loop . args)
	(let ()
	  
	  (define-macro (push val sym)
	    `(begin (set! ,sym (cons ,val ,sym)) ,sym))
	  
	  (define-macro (pop sym)
	    (let ((v (gensym "v")))
	      `(let ((,v (car ,sym)))
		 (set! ,sym (cdr ,sym))
		 ,v)))
	  
	  ;; this next one is a no-op but i need it as a marker for my cltl2
	  ;; translator.
	  
	  (define-macro (function sym) sym)     
	  
	  ;; getters and setters for the loop-clause "struct"
	  
	  (define (loop-operator c)          (vector-ref  c 0))
	  (define (loop-operator-set! c x)   (vector-set! c 0 x))
	  (define (loop-bindings c)          (vector-ref  c 1))
	  (define (loop-bindings-set! c x)   (vector-set! c 1 x))
	  (define (loop-collectors c)        (vector-ref  c 2))
	  (define (loop-collectors-set! c x) (vector-set! c 2 x))
	  (define (loop-initially c)         (vector-ref  c 3))
	  (define (loop-initially-set! c x)  (vector-set! c 3 x))
	  (define (loop-end-tests c)         (vector-ref  c 4))
	  (define (loop-end-tests-set! c x)  (vector-set! c 4 x))
	  (define (loop-looping c)           (vector-ref  c 5))
	  (define (loop-looping-set! c x)    (vector-set! c 5 x))
	  (define (loop-stepping c)          (vector-ref  c 6))
	  (define (loop-stepping-set! c x)   (vector-set! c 6 x))
	  (define (loop-finally c)           (vector-ref  c 7))
	  (define (loop-finally-set! c x)    (vector-set! c 7 x))
	  (define (loop-returning c)         (vector-ref  c 8))
	  (define (loop-returning-set! c x)  (vector-set! c 8 x))
	  
	  (define (make-loop-clause . args)
	    (let ((v (vector #f '() '() '() '() '() '() '() '())))
	      (if (null? args) v
		  (do ((a args (cddr a)))
		      ((null? a) v)
		    (case (car a)
		      ((operator) (loop-operator-set! v (cadr a)))
		      ((bindings) (loop-bindings-set! v (cadr a)))
		      ((collectors) (loop-collectors-set! v (cadr a)))
		      ((initially) (loop-initially-set! v (cadr a)))
		      ((end-tests) (loop-end-tests-set! v (cadr a)))
		      ((looping) (loop-looping-set! v (cadr a)))
		      ((stepping) (loop-stepping-set! v (cadr a)))
		      ((finally) (loop-finally-set! v (cadr a)))
		      ((returning) (loop-returning-set! v (cadr a))))))))
	  
	  (define (gather-clauses caller clauses)
	    ;; nconc all clausal expressions into one structure
	    (let ((gather-clause 
		   (lambda (clauses accessor)
		     ;; append data from clauses
		     (do ((l '()))
			 ((null? clauses) l)
		       (set! l (append l (accessor (car clauses))))
		       (set! clauses (cdr clauses))))))
	      (make-loop-clause 'operator caller
				'bindings
				(gather-clause clauses 
					       (function loop-bindings))
				'collectors
				(gather-clause clauses 
					       (function loop-collectors))
				'initially 
				(gather-clause clauses 
					       (function loop-initially))
				'end-tests 
				(gather-clause clauses 
					       (function loop-end-tests))
				'looping 
				(gather-clause clauses 
					       (function loop-looping))
				'stepping 
				(gather-clause clauses 
					       (function loop-stepping))
				'finally 
				(gather-clause clauses
					       (function loop-finally))
				'returning 
				(gather-clause clauses
					       (function loop-returning)))))
	  
	  (define (loop-op? x ops)
	    (assoc x ops))
	  
	  (define (loop-variable? x)
	    (symbol? x))
	  
	  (define (make-binding var val)
	    (list var val))
	  
	  (define (loop-error ops forms . args)
	    ;; all error messages include error context.
	    (let ((loop-context
		   (lambda (lst ops)
		     ;; return tail of expr up to next op in cdr of tail
		     (do ((h lst)
			  (l '()))
			 ((or (null? lst)
			      ;; ignore op if in front.
			      (and (not (eq? h lst))
				   (loop-op? (car lst) ops)))
			  (reverse l))
		       (set! l (cons (car lst) l))
		       (set! lst (cdr lst))))))
	      (let ((forms (loop-context forms ops)))
		(newline)
		(display "LOOP ERROR: ")
		(do ((tail args (cdr tail)))
		    ((null? tail) #f)
		  (display (car tail)))
		(newline)
		(display "clause context: ")
		(if (null? forms) 
		    (display "()")
		    (do ((tail forms (cdr tail)))
			((null? tail) #f)
		      (if (eq? tail forms) (display "'"))
		      (display (car tail))
		      (display (if (null? (cdr tail)) "'" " "))))
		(newline)
		(error "illegal loop syntax"))))
	  
	  (define (parse-for forms clauses ops)
	    ;; forms is (FOR ...)
	    (let ((op (loop-op? (car forms) ops)))
	      (if (null? (cdr forms))
		  (loop-error ops forms "Variable expected but source code ran out." )
		  (let ((var (cadr forms)))
		    (if (loop-variable? var)
			(if (null? (cddr forms))
			    (loop-error ops forms
					"'for' clause expected but source code ran out.")
			    ;; find the iteration path in the op
			    (let ((path (assoc (caddr forms) (cdddr op))))
			      ;; path is (<pathop> <parser>)
			      (if (not path)
				  (loop-error ops forms "'" (caddr forms) "'"
					      " is not valid with 'for'.")
				  ( (cadr path) forms clauses ops))))
			(loop-error ops forms "Found '" (cadr forms)
				    "' where a variable expected."))))))
	  
	  (define (parse-numerical-for forms clauses ops)
	    ;; forms is (FOR <var> <OP> ...)
	    ;; where <OP> is guaranteed to be one of: FROM TO BELOW ABOVE DOWNTO
	    clauses
	    (let ((var (cadr forms))
		  (tail (cddr forms))
		  (bind '())
		  (from #f)
		  (head #f)
		  (last #f)
		  (stop #f)
		  (step #f)
		  (test #f)
		  (incr #f))
	      
	      (do ((next #f))
		  ((or (null? tail) (loop-op? (car tail) ops)))
		(set! next (pop tail))
		(if (null? tail)
		    (loop-error ops forms
				"Expected expression but source code ran out."))
		(case next
		  ((from downfrom)
		   (if head (loop-error ops forms "Found '" next "' when '"
					head "' in effect."))
		   (set! head next)
		   (set! from (pop tail)))
		  ((below)
		   (if last (loop-error ops forms "Found '" next "' when '"
					last "' in effect."))
		   (set! stop (pop tail))
		   (set! last next))
		  ((to)
		   (if last (loop-error ops forms "Found '" next "' when '"
					last "' in effect."))
		   (set! stop (pop tail) )
		   (set! last next))
		  ((above )
		   (if last (loop-error ops forms "Found '" next "' when '"
					last "' in effect."))
		   (set! stop (pop tail))
		   (set! last next))
		  ((downto )
		   (if last (loop-error ops forms "Found '" next "' when '"
					last "' in effect."))
		   (set! stop (pop tail))
		   (set! last next))
		  ((by)
		   (if step (loop-error ops forms "Found duplicate 'by'."))
		   (set! step (pop tail)))
		  (else
		   (loop-error ops forms 
			       "'" next "' is not valid with 'for'."))))
	      (if (not head)
		  (set! head 'from))
	      (if (or (eq? head 'downfrom)
		      (eq? last 'downto)
		      (eq? last 'above))
		  (begin
		    (set! incr '-)
		    (if (eq? last 'above)
			(set! test '<=)
			(set! test '<)))   ; allow to for downto
		  (begin
		    (set! incr '+)
		    (if (eq? last 'below)
			(set! test '>=)
			(set! test '>))))
	      
	      ;; add binding for initial value
	      (push (make-binding var (or from 0)) bind)
	      ;; add binding for non-constant stepping values.
	      (if (not step)
		  (set! step 1)
		  (if (not (number? step))
		      (let ((var (gensym "v")))
			(push (make-binding var step) bind)
			(set! step var))))
	      (set! step `(set! ,var (,incr ,var ,step)))
	      (if stop
		  (let ((end (gensym "v")))
		    (push (make-binding end stop) bind)
		    (set! stop (list test var end))))
	      (values (make-loop-clause 'operator 'for
					'bindings (reverse bind)
					'stepping (list step)
					'end-tests (if (not stop)
						       '() (list stop)))
		      tail)))
	  
	  (define (parse-repeat forms clauses ops)
	    ;; forms is (REPEAT <FORM> ...)
	    (if (null? (cdr forms))
		(loop-error ops forms 
			    "'repeat' clause expected but source code ran out." )
		(call-with-values (lambda ()
				    (parse-numerical-for 
				     (list 'for (gensym "v") 'below (cadr forms))
				     clauses ops))
		  (lambda (clause ignore)
		    ignore
		    (values clause (cddr forms))))))
	  
	  (define (parse-sequence-iteration forms clauses ops)
	    ;; tail is (FOR <var> <OP> ...)
	    ;; <OP> is guaranteed to be one of: IN ON ACROSS
	    clauses
	    (let ((head forms)
		  (var (cadr forms))
		  (seq (gensym "v"))
		  (tail (cddr forms))
		  (bind '())
		  (data #f) 
		  (init '()) 
		  (loop '()) 
		  (incr #f)
		  (stop '()) 
		  (step '()) 
		  (type #f))
	      
	      (do ((next #f))
		  ((or (null? tail) (loop-op? (car tail) ops)))
		(set! next (pop tail))
		(when (null? tail)
		      (loop-error ops head
				  "Expression expected but source code ran out." ))
		(case next
		  ((in on across)
		   (if type (loop-error ops head 
					"Extraneous '" next "' when '"
					type "' in effect."))
		   (set! type next)
		   (set! data (pop tail)))
		  ((by )
		   (if incr 
		       (loop-error ops head "Duplicate 'by'." )
		       (if (eq? type 'across)
			   (loop-error ops head "'by' is invalid with 'across'." )
			   (set! incr (pop tail)))))
		  (else
		   (loop-error ops head "'" next "' is not valid with 'for'."))))
					; add bindings for stepping var and source
	      (push (make-binding var #f) bind)
	      (push (make-binding seq data) bind)
	      (if (eq? type 'across)
		  (let ((pos (gensym "v"))
			(max (gensym "v")))
		    (push (make-binding pos 0) bind)
		    (push (make-binding max #f) bind)
		    (push `(set! ,max (vector-length ,seq)) init)
		    (push `(set! ,pos (+ 1 ,pos)) step)
		    (push `(set! ,var (vector-ref ,seq ,pos)) loop)
		    (push `(>= ,pos ,max) stop))
		  (begin
		    (if incr
			(if (and (list? incr) (eq? (car incr) 'quote))
			    (push `(set! ,seq (,(cadr incr) ,seq)) step)
			    (push `(set! ,seq (,incr ,seq)) step))
			(push `(set! ,seq (cdr ,seq)) step))
		    (push (if (eq? type 'in)
			      `(set! ,var (car ,seq))
			      `(set! ,var ,seq))
			  loop)
		    (push `(null? ,seq) stop)))
	      
	      (values (make-loop-clause 'operator 'for
					'bindings (reverse bind)
					'end-tests stop
					'initially init
					'looping loop
					'stepping step)
		      tail)))
	  
	  (define (parse-general-iteration forms clauses ops)
	    ;; forms is (FOR <var> = ...)
	    clauses
	    (let ((head forms)
		  (var (cadr forms))
		  (tail (cddr forms))
		  (init #f)
		  (type #f)
		  (loop #f)
		  (step #f))
	      (do ((next #f))
		  ((or (null? tail) (loop-op? (car tail) ops)))
		(set! next (pop tail))
		(if (null? tail)
		    (loop-error ops head 
				"Expression expected but source code ran out."))
		(case next
		  ((= )
		   (if type (loop-error ops head "Duplicate '='."))
		   (set! loop `(set! ,var ,(pop tail)))
		   (set! type next))
		  ((then )
		   (if init (loop-error ops head "Duplicate 'then'."))
		   (set! init loop)
		   (set! loop #f)
		   (set! step `(set! ,var ,(pop tail)))
		   (set! type next))
		  (else
		   (loop-error ops head "'" next "' is not valid with 'for'."))))
	      
	      (values (make-loop-clause 'operator 'for
					'bindings (list (make-binding var #f))
					'initially (if init (list init) '())
					'looping (if loop (list loop) '())
					'stepping (if step (list step) '()))
		      tail)))
	  
	  (define (parse-with forms clauses ops)
	    ;; forms is (WITH <var> = ...)
	    clauses
	    (let ((head forms)
		  (tail (cdr forms))
		  (var #f)
		  (expr #f)
		  (and? #f)
		  (bind '())
		  (init '()))
	      (do ((need #t) 
		   (next #f))
		  ((or (null? tail) (loop-op? (car tail) ops)))
		(set! next (pop tail))
		(cond ((and (loop-variable? next) need)
		       (if var
			   (loop-error ops head
				       "Found '" next "' where 'and' expected."))
		       (if expr
			   (loop-error ops head
				       "Found '" next "' where 'and' expected."))
		       (set! var next)
		       (set! expr #f)
		       (set! and? #f)
		       (set! need #f))
		      ((eq? next 'and)
		       (if and?
			   (loop-error ops head "Duplicate 'and'.")
			   (if var 
			       (if expr
				   (begin
				     (push (make-binding var #f) bind)
				     (push `(set! ,var ,expr) init))
				   (push (make-binding var #f) bind))
			       (loop-error ops head "Extraneous 'and'.")))
		       (set! var #f)
		       (set! expr #f)
		       (set! and? #t)
		       (set! need #t))
		      ((eq? next '=)
		       (if expr
			   (loop-error ops head 
				       "Found '=' where 'and' expected.")
			   (set! expr (pop tail))))
		      (else
		       (if need
			   (loop-error ops head
				       "Found '" next "' where variable expected.")
			   (loop-error ops head "Found '" next
				       "' where '=' or 'and' expected.")))))
	      (if and? 
		  (loop-error ops head "Extraneous 'and'.")
		  (if var 
		      (if expr
			  (begin (push (make-binding var #f) bind)
				 (push `(set! ,var ,expr) init))
			  (push (make-binding var #f) bind))))
	      
	      (values (make-loop-clause 'operator 'with
					'bindings (reverse bind)
					'initially (reverse init))
		      tail)))
	  
	  (define (parse-do forms clauses ops)
	    clauses
	    (let ((head forms)
		  (oper (pop forms))
		  (body '()))
	      (do ()
		  ((or (null? forms)
		       (loop-op? (car forms) ops))
		   (if (null? body)
		       (loop-error ops head "Missing '" oper "' expression.")
		       (set! body (reverse body))))
		(push (car forms) body)
		(set! forms (cdr forms)))
	      (values
	       (make-loop-clause 'operator oper 'looping body)
	       forms)))
	  
	  (define (parse-finally forms clauses ops)
	    clauses
	    (let ((oper (pop forms))
		  (expr #f))
	      (if (null? forms)
		  (loop-error ops forms "Missing '" oper "' expression."))
	      (set! expr (pop forms))
	      (values (make-loop-clause 'operator oper 'finally (list expr))
		      forms)))
	  
	  (define (parse-initially forms clauses ops)
	    clauses
	    (let ((oper (pop forms))
		  (expr #f))
	      (if (null? forms)
		  (loop-error ops forms "Missing '" oper "' expression."))
	      (set! expr (pop forms))
	      (values (make-loop-clause 'operator oper 'initially (list expr))
		      forms)))
	  
	  (define (lookup-collector var clauses)
	    ;; collector is list: (<var> <type> <acc> <head>)
	    ;; returns the clause where the collect variable VAR is
	    ;; actually bound or nil if var hasn't already been bound
	    ;; if var is nil only the single system allocated collecter
	    ;; is possibly returned.
	    (let ((checkthem (lambda (var lis)
			       (do ((a #f)) 
				   ((or (null? lis) a) a)
				 (if (eq? var (car (car lis))) ;collector-var
				     (set! a (car lis)))
				 (set! lis (cdr lis))))))
	      (do ((c #f))
		  ((or (null? clauses) c) c)
		(set! c (checkthem var (loop-collectors (car clauses))))
		(set! clauses (cdr clauses)))))
	  
	  (define (compatible-accumulation? typ1 typ2)
	    (let ((l1 '(collect append nconc))
		  (l2 '(never always))
		  (l3 '(minimize maximize)))
	      (or (eq? typ1 typ2)
		  (and (member typ1 l1) (member typ2 l1))
		  (and (member typ1 l2) (member typ2 l2))
		  (and (member typ1 l3) (member typ2 l3)))))
	  
	  (define (parse-accumulation forms clauses ops)
	    ;; forms is (<op> form ...)
	    ;; where <op> is collect append nconc
	    (let ((save forms)
		  (oper (pop forms))
		  (make-collector (lambda (var type acc head)
				    (list var type acc head)))
		  ;; removed because noop
		  ;;(collector-var (lambda (col) (car col)))
		  (collector-type (lambda (col) (cadr col)))
		  (collector-acc (lambda (col) (caddr col)))
		  (collector-head (lambda (col) (cadddr col)))
		  (expr #f)
		  (coll #f)
		  (new? #f)
		  (into #f)
		  (loop '())
		  (bind '())
		  (init '())
		  (tests '())
		  (return '()))
	      
	      (if (null? forms)
		  (loop-error ops forms "Missing '" oper "' expression."))
	      (set! expr (pop forms))
	      (if (not (null? forms))
		  (if (eq? (car forms) 'into)
		      (begin
			(if (null? (cdr forms))
			    (loop-error ops save "Missing 'into' variable."))
			(if (loop-variable? (cadr forms))
			    (begin (set! into (cadr forms))
				   (set! forms (cddr forms)))
			    (loop-error ops save "Found '" (car forms)
					"' where 'into' variable expected.")))))
	      
	      ;; search for a clause that already binds either the user specified
	      ;; accumulator (into) or a system allocated one if no into.
	      ;; system collectors
	      ;;   o only one  allowed, all accumuations must be compatible
	      ;;   o returns value
	      ;;   value collector: (nil <op> <#:acc>)
	      ;;   list collector:  (nil <op> <#:tail> <#:head>)
	      ;; into collectors
	      ;;   o any number allowed
	      ;;   o returns nothing.
	      ;;   value collector: (<into> <op> <into> )
	      ;;   list collector:  (<into> <op> <#:tail> <#:head>)
	      (set! coll (lookup-collector into clauses))
	      (if (not coll)
		  (set! new? #t)
		  ;; accumulator already established by earlier clause
		  ;; check to make sure clauses are compatible.
		  (if (not (compatible-accumulation? oper (collector-type coll)))
		      (loop-error ops save "'" (collector-type coll)
				  "' and '" oper "' are incompatible accumulators.")))
	      (case oper 
		((sum count)
		 (let ((acc #f))
		   (if new?
		       (begin 
			 (set! acc (or into (gensym "v")))
			 (push (make-binding acc 0) bind)
			 ;; coll= (nil <op> <#:acc>) or (<into> <op> <into>)
			 (set! coll (make-collector into oper acc #f))
			 ;; only add a return value if new collector isnt into
			 (if (not into) (push acc return)))
		       (set! acc (collector-acc coll)))
		   (if (eq? oper 'sum)
		       (push `(set! ,acc (+ ,acc ,expr)) loop)
		       (push `(if ,expr (set! ,acc (+ ,acc 1))) loop))))
		((minimize maximize)
		 (let ((var (gensym "v"))
		       (opr (if (eq? oper 'minimize) '< '>))
		       (acc #f))
		   (if new?
		       (begin
			 (set! acc (or into (gensym "v")))
			 (push (make-binding acc #f) bind)
			 ;; coll= (nil <op> <#:acc>) or (<into> <op> <into>)
			 (set! coll (make-collector into oper acc #f))
			 ;; only add a return value if new collector isnt into
			 (if (not into) (push `(or ,acc 0) return)))
		       (set! acc (collector-acc coll)))
		   (push (make-binding var #f) bind)
		   (push `(begin (set! ,var ,expr)
				 (if (or (not ,acc) 
					 (,opr ,var ,acc))
				     (set! ,acc ,var)))
			 loop)))
		((append collect nconc)
		 ;; for list accumulation a pointer to the tail of the list
		 ;; is updated and the head of the list is returned. any
		 ;; into variable is set to the head inside the loop.
		 (let ((head #f)
		       (tail #f))
		   (if (not new?)
		       (begin (set! tail (collector-acc coll))
			      (set! head (collector-head coll)))
		       (begin
			 (if into (push (make-binding into '(list)) bind))
			 (set! tail (gensym "v"))
			 ;; allocate a pointer to the head of list
			 (set! head (gensym "v"))
			 (push (make-binding head '(list #f)) bind)
			 (push (make-binding tail #f) bind)
			 ;; initialize tail to head
			 (push `(set! ,tail ,head) init)
			 (set! coll (make-collector into oper tail head))
			 ;; only add a return value if new collector isnt into
			 (if (not into)
			     (push `(cdr ,head) return))))
		   ;; add loop accumulation forms
		   (if (eq? oper 'append)
		       (begin
			 (push `(set-cdr! ,tail (append ,expr (list))) loop)
			 (push `(set! ,tail (last-pair ,tail)) loop))
		       (if (eq? oper 'collect)
			   (begin
			     (push `(set-cdr! ,tail (list ,expr)) loop)
			     (push `(set! ,tail (cdr ,tail)) loop))
			   (begin 
			     (push `(set-cdr! ,tail ,expr) loop)
			     (push `(set! ,tail (last-pair ,tail)) loop))))
		   ;; update user into variable inside the main loop
		   ;; regardless of whether its a new collector or not
		   (if into
		       (push `(set! ,into (cdr ,head)) loop)))))
	      
	      (values (make-loop-clause 'operator oper
					'bindings (reverse bind)
					'initially (reverse init)
					'looping (reverse loop)
					'returning (reverse return)
					'collectors (if new? (list coll) '())
					'end-tests (reverse tests))
		      forms)))
	  
					;(define (loop-stop expr)
					;  `(%done% ,expr))
	  
	  (define (loop-return expr)
	    `(return ,expr))
	  
	  (define (parse-while-until forms clauses ops)
	    clauses
	    (let ((head forms)
		  (oper (pop forms))
		  (test #f)
		  (stop '(go #t))) ; :done
	      (if (null? forms)
		  (loop-error ops head "Missing '" oper "' expression."))
	      
	      (case oper
		((until ) (set! test (pop forms)))
		((while ) (set! test `(not ,(pop forms)))))
	      ;; calls the DONE continuation.
	      (values (make-loop-clause 'operator oper
					'looping (list `(if ,test ,stop)))
		      forms)))
	  
	  (define (parse-thereis forms clauses ops)
	    clauses
	    (let ((oper (car forms))
		  (expr #f)
		  (bool #f)
		  (func #f))
	      (if (null? (cdr forms))
		  (loop-error ops forms "Missing '" (car forms) "' expression." ))
	      (set! expr (cadr forms))
	      ;; fourth element of operator definition must be
	      ;; a function that returns the stop expression.
	      (set! func (cadddr (loop-op? oper ops) ))
	      
	      (case oper
		((thereis ) 
		 ;; return true as soon as expr is true or false at end
		 (set! bool #f))
		((always )
		 ;; return false as soon as expr is false, or true at end
		 (set! expr `(not ,expr))
		 (set! bool #t))
		((never )
		 ;; return false as soon as expr is true, or true at end
		 (set! bool #t)))
	      (set! forms (cddr forms))
	      ;; this calls the RETURN continuation
	      (values (make-loop-clause 'operator 'thereis
					'looping 
					(list `(if ,expr ,(func (not bool))))
					'returning 
					(list bool))
		      forms)))
	  
	  (define (parse-return forms clauses ops)
	    clauses
	    (let ((oper (car forms))
		  (expr #f)
		  (func #f))
	      (if (null? (cdr forms))
		  (loop-error ops forms "Missing '" (car forms) "' expression."))
	      (set! expr (cadr forms))
	      (set! forms (cddr forms))
	      ;; fourth element of operator definition must be
	      ;; a function that returns the stop expression.
	      (set! func (cadddr (loop-op? oper ops) ))
	      ;; this calls the RETURN continuation
	      (values (make-loop-clause 'operator 'return
					'looping `(,(func expr)))
		      forms)))
	  
	  (define (legal-in-conditional? x ops)
	    ;; FIXED (member (loop-operator...))
	    (let ((op (loop-op? x ops)))
	      (if (and op 
		       (not (null? (cddr op)))
		       (eq? (caddr op) 'task)
		       (not (member (car op) '(thereis never always))))
		  op #f)))
	  
	  (define (parse-then-else-dependents forms clauses ops)
	    (let ((previous forms)
		  (stop? #f)
		  (parsed '()))
	      
	      (do ((op #f)
		   (clause #f)
		   (remains #f))
		  ((or (null? forms) stop?))
		(set! op (legal-in-conditional? (car forms) ops))
		(if (not op)
		    (loop-error ops previous "'" (car forms)
				"' is not conditional operator."))
					;(multiple-value-setq 
					; (clause remains)
					; ( (cadr op) forms (append clauses parsed) ops))
		(call-with-values
		    (lambda () ( (cadr op) forms (append clauses parsed) ops))
		  (lambda (a b) (set! clause a) (set! remains b)))
		
					;(format #t "~%after call clause=~s forms=~S" clause forms)      
		
		(set! parsed (append parsed (list clause)))
		(set! previous forms)
		(set! forms remains)
		
		(if (not (null? forms))
		    (if (eq? (car forms) 'and)
			(begin
			  (set! forms (cdr forms))
			  (if (null? forms)
			      (loop-error ops previous "Missing 'and' clause.")))
			(if (eq? (car forms) 'else)
			    (set! stop? #t)
			    (if (loop-op? (car forms) ops)
				(set! stop? #t))))))
	      (values parsed forms)))
	  
	  (define (parse-conditional forms clauses ops)
	    (let ((ops (cons '(else ) ops))
		  (save forms)
		  (oper (car forms))
		  (loop (list))  ; avoid '() because of acl bug
		  (expr (list))
		  (then (list))
		  (else (list)))
	      (if (null? (cdr forms))
		  (loop-error ops save "Missing '" oper "' expression."))
	      (set! forms (cdr forms))
	      (set! expr (pop forms))
	      (if (null? forms)
		  (loop-error ops forms "Missing conditional clause."))
	      (if (eq? oper 'unless)
		  (set! expr (list 'not expr)))
	      (call-with-values
		  (lambda () (parse-then-else-dependents forms clauses ops))
		(lambda (a b)
		  (set! then a)
		  (set! forms b)))
	      
	      ;; combine dependant clauses if more than one
	      (if (not (null? (cdr then)))
		  (set! then (gather-clauses (list) then))
		  (set! then (car then)))
	      (loop-operator-set! then 'if)
	      
	      ;; this (if ...) is hacked so that it is a newly
	      ;; allocated list. otherwise acl and clisp have a
	      ;; nasty structure sharing problem.
	      (set! loop (list 'if expr 
			       (append `(begin ,@(loop-looping then)) (list))
			       #f))
	      (if (and (not (null? forms))
		       (eq? (car forms) 'else))
		  (begin
		    (set! forms (cdr forms))
		    (when (null? forms)
			  (loop-error ops save "Missing 'else' clause."))
		    (call-with-values 
			(lambda ()
			  (parse-then-else-dependents 
			   forms (append clauses (list then))
			   ops))
		      (lambda (a b) (set! else a) (set! forms b)))
		    (if (not (null? (cdr else)))
			(set! else (gather-clauses '() else))
			(set! else (car else)))
		    (set-car! (cdddr loop) `(begin ,@(loop-looping else)))
		    ;; flush loop forms so we dont gather actions.
		    (loop-looping-set! then '())
		    (loop-looping-set! else '())
		    (set! then (gather-clauses 'if (list then else)))))
	      (loop-looping-set! then (list loop))
	      (values then forms)))
	  
	  (define (parse-clauses forms cond? ops)
	    (if (or (null? forms)
		    (not (symbol? (car forms))))
		(list (make-loop-clause 'operator 'do 'looping forms))
		(let ((op-type? (lambda (op type)
				  (and (not (null? (cddr op)))
				       (eq? (caddr op) type)))))
		  (let ((previous forms)
			(clauses '()))
		    (do ((op #f)
			 (clause #f)
			 (remains '())
			 (body '()) )
			((null? forms))
		      (if (and cond? (eq? (car forms) 'and))
			  (pop forms))
		      (set! op (loop-op? (car forms) ops))
		      (if (not op)
			  (loop-error ops previous "Found '" (car forms)
				      "' where operator expected."))
					;(multiple-value-setq (clause remains)
					;                     ((cadr op) forms clauses ops))
		      (call-with-values
			  (lambda () ( (cadr op) forms clauses ops))
			(lambda (a b)
			  (set! clause a)
			  (set! remains b)))
		      (if (op-type? op 'task)
			  (set! body op)
			  (if (op-type? op 'iter)
			      (if (not (null? body))
				  (loop-error ops previous "'" (car op)
					      "' clause cannot follow '"
					      (car body) "'."))))
		      (set! previous forms)
		      (set! forms remains)
		      (set! clauses (append clauses (list clause))))
		    clauses))))
	  
	  (define (parse-iteration caller forms ops)
	    (gather-clauses caller (parse-clauses forms '() ops)))
	  
	  ;;
	  ;; loop implementation
	  ;;
	  
	  (define *loop-operators*
	    ;; each clause is (<op> <parser> <tag> . <whatever>)
	    (list (list 'with (function parse-with) #f)
		  (list 'initially (function parse-initially) #f)
		  (list 'repeat (function parse-repeat) 'iter)
		  (list 'for (function parse-for) 'iter
			(list 'from (function parse-numerical-for))
			(list 'downfrom (function parse-numerical-for))
			(list 'below (function parse-numerical-for))
			(list 'to (function parse-numerical-for))
			(list 'above (function parse-numerical-for))
			(list 'downto (function parse-numerical-for))
			(list 'in (function parse-sequence-iteration))
			(list 'on (function parse-sequence-iteration))
			(list 'across (function parse-sequence-iteration))
			(list '= (function parse-general-iteration)))
		  (list 'as (function parse-for) 'iter)
		  (list 'do (function parse-do) 'task)
		  (list 'collect (function parse-accumulation) 'task)
		  (list 'append (function parse-accumulation) 'task)
		  (list 'nconc (function parse-accumulation) 'task)
		  (list 'sum (function parse-accumulation) 'task)
		  (list 'count (function parse-accumulation) 'task)
		  (list 'minimize (function parse-accumulation) 'task)
		  (list 'maximize (function parse-accumulation) 'task)
		  (list 'thereis (function parse-thereis) 'task
			(function loop-return))
		  (list 'always (function parse-thereis) 'task
			(function loop-return))
		  (list 'never (function parse-thereis) 'task 
			(function loop-return))
		  (list 'return (function parse-return) 'task 
			(function loop-return))
		  (list 'while (function parse-while-until) #f )
		  (list 'until (function parse-while-until) #f )
		  (list 'when (function parse-conditional) 'task)
		  (list 'unless (function parse-conditional) 'task)
		  (list 'if (function parse-conditional) 'task)
		  (list 'finally (function parse-finally) #f)))
	  
	  ;;
	  ;; loop expansions for scheme and cltl2
	  ;;
	  
	  (define (scheme-loop forms)
	    (let ((name (gensym "v"))
		  (parsed (parse-iteration 'loop forms *loop-operators*))
		  (end-test '())
		  (done '(go #t))  ; :done
		  (return #f))
					;(write (list :parsed-> parsed))
	      ;; cltl2's loop needs a way to stop iteration from with the run
	      ;; block (the done form) and/or immediately return a value
	      ;; (the return form).  scheme doesnt have a block return or a
	      ;; go/tagbody mechanism these conditions are implemented using
	      ;; continuations.  The forms that done and return expand to are
	      ;; not hardwired into the code because this utility is also used
	      ;; by CM's 'process' macro. Instead, the done and return forms
	      ;; are returned by functions assocated with the relevant operator
	      ;; data. For example, the function that returns the return form
	      ;; is stored as the fourth element in the return operator data.
	      ;; and the done function is stored in the while and until op data.
	      
	      ;; the cadddr of the RETURN operator is a function that
	      ;; provides the form for immediately returning a value
	      ;; from the iteration.
	      
	      (let ((returnfn (cadddr (assoc 'return *loop-operators*))))
		(set! return (returnfn
			      (if (null? (loop-returning parsed))
				  #f
				  (car (loop-returning parsed))))))
	      
	      ;; combine any end-tests into a single IF expression
	      ;; that calls the (done) continuation if true. multiple
	      ;; tests are OR'ed togther
	      
	      (set! end-test
		    (let ((ends (loop-end-tests parsed)))
		      (if (null? ends)
			  '()
			  (list
			   `(if ,(if (null? (cdr ends))
				     (car ends)
				     (cons 'or ends))
				;;  calls the done continuation
				,done 
				#f)))))
	      `(let (,@ (loop-bindings parsed))
		 ,@(loop-initially parsed)
		 (call-with-exit
		  (lambda (return)     ; <- (return) returns from this lambda
		    (call-with-exit
		     (lambda (go)  ; <- (go #t) returns from this lambda
		       ;; a named let provides the actual looping mechanism.
		       ;; the various tests and actions may exit via the
		       ;; (done) or (return) continuations.
		       (let ,name () 
			    ,@end-test
			    ,@(loop-looping parsed)
			    ,@(loop-stepping parsed)
			    (,name))))
		    ;; this is the lexical point for (go #t) continuation.
		    ,@(loop-finally parsed)
		    ;; invoke the RETURN continuation with loop value or #f
		    ,return)))))
	  
	  
	  (scheme-loop args)))
      
      ;;
      ;; loop tests.
      ;;
      
      (test (loop for i below 10 collect i) '(0 1 2 3 4 5 6 7 8 9))
      (test (loop for i to 10 sum i) 55)
      (test (loop for i downto -10 count (even? i)) 6)
      (test (loop for x in '(0 1 2 3 4 5 6 7 8 9) thereis (= x 4)) #t)
      (test (loop for x in '(0 1 2 3 4 5 6 7 8 9) by 'cddr collect x) '(0 2 4 6 8))
      (test (loop for x on '(0 1 2 3) by 'cddr collect x) '((0 1 2 3) (2 3)))
      (test (loop for x in '(0 1 2 3 4 5 6 7 8 9) thereis (= x 4)) #t)
      (test (loop for x in '(0 1 2 3 4 5 6 7 8 9) never (= x 4)) #f)
      (test (loop for x in '(0 1 2 3 4 5 6 7 8 9) never (= x 40)) #t)
      (test (loop for x in '(0 2 3 4 5 6 7 8 9) always (< x 40)) #t)
      (test (loop repeat 10 with x = 0 collect x do (set! x (+ x 1))) '(0 1 2 3 4 5 6 7 8 9))
      (test (loop repeat 10 for x = #t then (not x) collect x) '(#t #f #t #f #t #f #t #f #t #f))
      (test (loop repeat 10 count #t) 10)
      (test (loop repeat 10 count #f) 0)
      (test (loop for i to 10 collect i collect (* 2 i)) '(0 0 1 2 2 4 3 6 4 8 5 10 6 12 7 14 8 16 9 18 10 20))
      (test (loop for i from -10 to 10 by 2 nconc (list i (- i))) '(-10 10 -8 8 -6 6 -4 4 -2 2 0 0 2 -2 4 -4 6 -6 8 -8 10 -10))
      (test (loop for i from -10 downto 10 by -1 collect i) '())
      (test (loop for i downfrom 10 downto -10 by 2 collect i) '(10 8 6 4 2 0 -2 -4 -6 -8 -10))
      (test (loop for i from 10 to -10 by 1 collect i) '())
      (test (loop for i to 10 for j downfrom 10 collect i collect j) '(0 10 1 9 2 8 3 7 4 6 5 5 6 4 7 3 8 2 9 1 10 0))
      (test (loop for i below 0 collect i into foo finally (return foo)) '())
      (test (loop for i below 0 sum i into foo finally (return foo)) 0)
      (test (loop for i below 0 maximize i into foo finally (return foo)) #f)
      (test (loop with a and b = 'x and c = 2 repeat 10 for x = 1 then 'fred collect (list x a b c))
	    '((1 #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2)))
      (test (loop for i across #(0 1 2 3) append (list i (expt 2 i))) '(0 1 1 2 2 4 3 8))
      (test (loop with a = 0 and b = -1 while (< a 10) sum a into foo do (set! a (+ a 1)) finally (return (list foo b))) '(45 -1))
      (test (loop for i from 0 until (> i 9) collect i) '(0 1 2 3 4 5 6 7 8 9))
      (test (loop for i from 0 while (< i 9) when (even? i) collect i) '(0 2 4 6 8))
      (test (loop with l = (list 0) for s in spec for k = s then (+ k s) do (push k l) finally (return l)) 'error)
      (test (loop with l = (list (encode-interval 'p 1)) for s in spec for k = (interval s) then (transpose k (interval s)) do (push k l) finally (return l)) 'error)
      ;; end loop

      ;; more macros from Rick's stuff

      (defmacro dolist (spec . body)
	;; spec = (var list . return)
	(let ((v (gensym)))
	  `(do ((,v ,(cadr spec) (cdr ,v))
		(,(car spec) #f))
	       ((null? ,v) ,@(cddr spec))
	     (set! ,(car spec) (car ,v))
	     ,@body)))

      (test (let ((sum 0)) (dolist (v (list 1 2 3) sum) (set! sum (+ sum v)))) 6)
      
      (defmacro dotimes (spec . body)
	;; spec = (var end . return)
	(let ((e (gensym))
	      (n (car spec)))
	  `(do ((,e ,(cadr spec))
		(,n 0))
	       ((>= ,n ,e) ,@(cddr spec))
	     ,@body
	     (set! ,n (+ ,n 1)))))

      (test (let ((sum 0)) (dotimes (i 3 sum) (set! sum (+ sum i)))) 3)
      
      (defmacro do* (spec end . body)
	`(let* (,@(map (lambda (var) (list (car var) (cadr var))) spec))
	   (do () ,end
	     ,@body
	     ,@(map (lambda (var) (list 'set! (car var) (caddr var))) spec))))

      (test (let ((sum 0)) (do* ((i 0 (+ i 1)) (j i (+ i 1))) ((= i 3) sum) (set! sum (+ sum j)))) 5)

      (define-macro (fluid-let xexe . body)
	;; taken with changes from Teach Yourself Scheme
	(let ((xx (map car xexe))
	      (ee (map cadr xexe))
	      (old-xx (map (lambda (ig) (gensym)) xexe)))
	  `(let ,(map (lambda (old-x x) `(,old-x ,x)) 
		      old-xx xx)
	     (dynamic-wind
		 (lambda () #f)
		 (lambda ()
		   ,@(map (lambda (x e)
			    `(set! ,x ,e)) 
			  xx ee)
		   (let ()
		     ,@body))
		 (lambda ()
		   ,@(map (lambda (x old-x)
			    `(set! ,x ,old-x)) 
			  xx old-xx))))))
      
      (test (let ((x 32)
		  (y 0))
	      (define (gx) x)
	      (fluid-let ((x 12))
		(set! y (gx)))
	      (list x y))
	    '(32 12))
      
      (test (let ((x "hi")
		  (y 0)
		  (z '(1 2 3)))
	      (define (gx) (+ x z))
	      (fluid-let 
		  ((x 32) (z (+ 123 (car z))))
		(set! y (gx)))
	      (list x y z))
            '("hi" 156 (1 2 3)))
      
      (test (let ((x 32)
		  (y 0))
	      (define (gx) x)
	      (call-with-exit
	       (lambda (return)
		 (fluid-let ((x 12))
		   (set! y (gx))
		   (return))))
	      (list x y))
	    '(32 12))

      (test (let ((x 32)
		  (y 0))
	      (define (gx) x)
	      (let ((x 100))
		(fluid-let ((x 12))
		  (set! y (gx))))
	      (list x y))
	    '(32 32))
      ;; oops! fluid-let doesn't actually work!

      ;; in CL: (defvar x 32) (let ((y 0)) (defun gx () x) (let ((x 12)) (setf y (gx))) (list x y)) -> '(32 12)
      ;;                      (let ((y 0)) (defun gx () x) (let ((x 100)) (let ((x 12)) (setf y (gx)))) (list x y)) -> '(32 12)
      ;;                      (let ((y 0)) (defun gx () x) (let ((x 100)) (let ((x 12)) (setf y (gx)) (setf x 123)) (list x y))) -> '(100 12) !
      ;; (the defvar makes x dynamic)

      
      ;; define** treats args before :optional as required args
      (define-macro (define** declarations . forms)
	(let ((name (car declarations))
	      (args (cdr declarations)))
	  (define (position thing lst count)
	    (if (or (null? lst)
		    (not (pair? (cdr lst))))
		#f
		(if (eq? thing (car lst))
		    count
		    (position thing (cdr lst) (+ count 1)))))
	  (let ((required-args (position :optional args 0)))
	    (if required-args
		`(define* (,name . func-args)
		   (if (< (length func-args) ,required-args)
		       (error "~A requires ~D argument~A: ~A" 
			      ',name ,required-args (if (> ,required-args 1) "s" "") func-args)
		       (apply (lambda* ,args ,@forms) func-args)))
		`(define* ,declarations ,@forms)))))

      ;; Rick's with-optkeys

      (define-macro (with-optkeys spec . body)
	(
	 (lambda (user rawspec body)
	   
	   (define (string->keyword str) (symbol->keyword (string->symbol str)))
	   
	   (define (key-parse-clause info mode args argn user)
	     ;; return a cond clause that parses one keyword. info for each
	     ;; var is: (<got> <var> <val>)
	     (let* ((got (car info))
		    (var (cadr info))
		    (key (string->keyword (symbol->string var))))
	       `((eq? (car ,args) ,key )
		 (if ,got (error "duplicate keyword: ~S" , key))
		 (set! ,var (if (null? (cdr ,args))
				(error "missing value for keyword: ~S" 
				       , user)
				(cadr ,args)))
		 (set! ,got #t) ; mark that we have a value for this param
		 (set! ,mode #t) ; mark that we are now parsing keywords
		 (set! ,argn (+ ,argn 1))
		 (set! ,args (cddr ,args)))))
	   
	   (define (pos-parse-clause info mode args argn I)
	     ;; return a cond clause that parses one positional. info for
	     ;; each var is: (<got> <var> <val>)
	     (let ((got (car info))
		   (var (cadr info)))
	       `((= ,argn ,I)
		 (set! ,var (car ,args))
		 (set! ,got #t) ; mark that we have a value for this param
		 (set! ,argn (+ ,argn 1))
		 (set! ,args (cdr ,args)))))
	   
	   (let* ((otherkeys? (member '&allow-other-keys rawspec))
		  ;; remove &allow-other-keys from spec
		  (spec (if otherkeys? (reverse (cdr (reverse rawspec))) rawspec))
		  (data (map (lambda (v)
			       ;; for each optkey variable v return a list
			       ;; (<got> <var> <val>) where the <got>
			       ;; variable indicates that <var> has been
			       ;; set, <var> is the optkey variable itself
			       ;; and <val> is its default value
			       (if (pair? v)
				   (cons (gensym (symbol->string (car v))) v)
				   (list (gensym (symbol->string v)) v #f)))
			     spec))
		  (args (gensym "args")) ; holds arg data as its parsed
		  (argn (gensym "argn"))
		  (SIZE (length data))
		  (mode (gensym "keyp")) ; true if parsing keywords
		  ;; keyc are cond clauses that parse valid keyword
		  (keyc (map (lambda (d) (key-parse-clause d mode args argn user))
			     data))
		  (posc (let lup ((tail data) (I 0))
			  (if (null? tail) (list)
			      (cons (pos-parse-clause (car tail) mode args argn I)
				    (lup (cdr tail) (+ I 1))))))
		  (bindings (map cdr data)) ; optkey variable bindings
		  )
	     
	     (if otherkeys?
		 (set! bindings (cons '(&allow-other-keys (list)) bindings)))
	     
	     `(let* ,bindings ; bind all the optkey variables with default values
		;; bind status and parsing vars
		(let ,(append (map (lambda (i) (list (car i) #f)) data)
			      `((,args ,user)
				(,argn 0)
				(,mode #f)))
		  ;; iterate arglist and set opt/key values
		  (do ()
		      ((null? ,args) #f)
		    (cond 
		     ;; add valid keyword clauses first
		     ,@ keyc
			;; a keyword in (car args) is now either added to
			;; &allow-other-keys or an error
			, (if otherkeys?
			      `((keyword? (car ,args))
				(if (not (pair? (cdr ,args)))
				    (error "missing value for keyword ~S" (car ,args)))
				(set! &allow-other-keys (append &allow-other-keys
								(list (car ,args)
								      (cadr ,args))))
				(set! ,mode #t) ; parsing keys now...
				(set! ,args (cddr ,args)) )
			      `((keyword? (car ,args)) ;(and ,mode (keyword? (car ,args)))
				(error "invalid keyword: ~S" (car ,args)) )
			      )
			  ;; positional clauses illegal if keywords have happened
			  (,mode (error "positional after keywords: ~S" (car ,args)))
			  ;; too many value specified
			  ((not (< ,argn ,SIZE)) (error "too many args: ~S" , args))
			  ;; add the valid positional clauses
			  ,@ posc
			     ))
		  ,@ body))
	     ))
	 (car spec)
	 (cdr spec)
	 body
	 ))
      
      (test (let ((args '(1 2 3)))  (with-optkeys (args a b c) (list a b c))) '(1 2 3))
      (test (let ((args '(1 2 3 4)))  (with-optkeys (args a b c) (list a b c))) 'error)
      (test (let ((args '(1 2))) (with-optkeys (args a b (c 33)) (list a b c))) '(1 2 33))
      (test (let ((args '())) (with-optkeys (args a b (c 33)) (list a b c))) '(#f #f 33))
      (test (let ((args '(:b 22))) (with-optkeys (args a b (c 33)) (list a b c))) '(#f 22 33))
      (test (let ((args '(-1 :z 22))) (with-optkeys (args a b (c 33)) (list a b c))) 'error)
      (test (let ((args '(:b 99 :z 22))) (with-optkeys (args a b (c 33)) (list a b c))) 'error)
      (test (let ((args '(:z 22))) (with-optkeys (args a b (c 33) &allow-other-keys) (list a b c &allow-other-keys))) '(#f #f 33 (:z 22)))
      (test (let ((args '(:id "0" :inst "flute" :name "Flute"))) (with-optkeys (args id inst &allow-other-keys) (list id inst &allow-other-keys))) '("0" "flute" (:name "Flute")))
      (test (let ((args '(:inst "flute" :id "0" :name "Flute"))) (with-optkeys (args id inst &allow-other-keys) (list id inst &allow-other-keys))) '("0" "flute" (:name "Flute")))
      (test (let ((args '(:id "0" :name "Flute" :inst "flute"))) (with-optkeys (args id inst &allow-other-keys) (list id inst &allow-other-keys))) '("0" "flute" (:name "Flute")))
      (test (let ((args '(:name "Flute" :inst "flute" :id "0"))) (with-optkeys (args id inst &allow-other-keys) (list id inst &allow-other-keys))) '("0" "flute" (:name "Flute")))
      
      
      (let ()
	
	;; some common lispisms
	;;   where names are the same, but functions are different (abs for example), 
	;;   I'll prepend "cl-" to the CL version; otherwise we end up redefining
	;;   map and member, for example, which can only cause confusion.
	;;
	;; also I'm omitting the test-if-not and test-not args which strike me as ridiculous.
	;; If CLtL2 says something is deprecated, it's not included.
	;; Series and generators are ignored.
	;;
	;;  ... later ... I've run out of gas.
	
					;(define-macro (progn . body) `(let () ,@body))
	(define progn begin)
	(define-macro (prog1 first . body) (let ((result (gensym))) `(let ((,result ,first)) ,@body ,result)))
	(define-macro (prog2 first second . body) `(prog1 (progn ,first ,second) ,@body))

	(defmacro the (type form) form)
	(define-macro (defvar var . args) `(define ,var (or ,(and (not (null? args)) (car args)) #f)))

	(defmacro incf (sym . val) `(let () (set! ,sym (+ ,sym ,(if (null? val) 1 (car val)))) ,sym))
	(defmacro decf (sym . val) `(let () (set! ,sym (- ,sym ,(if (null? val) 1 (car val)))) ,sym))

	(defmacro push (val sym) 
	  `(let () 
	     (setf ,sym (cons ,val ,sym)) 
	     ,sym))

	(defmacro pop (sym) 
	  (let ((v (gensym))) 
	    `(let ((,v (car ,sym))) 
	       (setf ,sym (cdr ,sym)) 
	       ,v)))

	(defmacro* pushnew (val sym (test equal?) (key identity))
	  (let ((g (gensym))
		(k (if (procedure? key) key identity))) ; can be explicit nil!
	    `(let ((,g ,val))
	       (if (null? (cl-member (,k ,g) ,sym ,test ,k))
		   (push ,g ,sym))
	       ,sym)))

	(defmacro unless (test . forms) `(if (not ,test) (begin ,@forms)))
	(define-macro (declare . args) #f)
	(defmacro set (a b) `(set! ,(symbol->value a) ,b))

	(define-macro (setf . pairs)
	  (if (not (even? (length pairs)))
	      (error "setf has odd number of args"))
	  `(let () ,@(let ((var #f)) 
		       (map (lambda (p) 
			      (if var
				  (let ((val (if (pair? var)
						 (if (member (car var) '(aref svref elt char schar))
						     (list 'set! (cdr var) p)
						     (if (eq? (car var) 'car)
							 (list 'set-car! (cadr var) p)
							 (if (eq? (car var) 'cdr)
							     (list 'set-cdr! (cadr var) p)
							     (if (eq? (car var) 'nth)
								 (list 'set! (list (caddr var) (cadr var)) p)
								 (list 'set! var p)
								 ))))
						 (list 'set! var p))))
				    (set! var #f)
				    val)
				  (begin
				    (set! var p)
				    '())))
			    pairs))))

	(define-macro (setq . pairs)
	  (if (not (even? (length pairs)))
	      (error "setq has odd number of args"))
	  `(let () ,@(let ((var #f)) 
		       (map (lambda (p) 
			      (if var
				  (let ((val (list 'set! var p)))
				    (set! var #f)
				    val)
				  (begin
				    (set! var p)
				    '())))
			    pairs))))

	(define-macro (psetq . pairs)
	  (let ((vals '())
		(vars '()))
	    (do ((var-val pairs (cddr var-val)))
		((null? var-val))
	      (let ((interval (gensym)))
		(set! vals (cons (list interval (cadr var-val)) vals))
		(set! vars (cons (list 'set! (car var-val) interval) vars))))
	    `(let ,(reverse vals)
	       ,@vars)))

	(define (mapcar func . lists)
	  ;; not scheme's map because lists can be different lengths
	  ;; and args can be any sequence type (all mixed together)
	  (define (mapcar-seqs func seqs)
	    (if (null? seqs)
		'()
		(cons (func (car seqs))
		      (mapcar-seqs func (cdr seqs)))))

	  (define (mapcar-1 index lens func seqs)
	    (if (member index lens)
		'()
		(cons (apply func (mapcar-seqs (lambda (obj) (obj index)) seqs))
		      (mapcar-1 (+ index 1) lens func seqs))))

	  (let ((lens (map length lists)))
	    (mapcar-1 0 lens func lists)))
#|
(define (mapcar func . lists)
  ;; not scheme's map because lists can be different lengths
  (if (member '() lists)
      '()
      (cons (apply func (map car lists))
	    (apply mapcar func (map cdr lists)))))
|#

	(define (maplist function . lists)
	  (if (member '() lists)
	      '()
	      (cons (apply function lists)
		    (apply maplist function (map cdr lists)))))

	(define (mapc function . lists)
	  (define (mapc-1 function . lists)
	    (if (not (member '() lists))
		(begin
		  (apply function (map car lists))
		  (apply mapc-1 function (map cdr lists)))))
	  (apply mapc-1 function lists)
	  (car lists))

	(define (mapl function . lists)
	  (define (mapl-1 function . lists)
	    (if (not (member '() lists))
		(begin
		  (apply function lists)
		  (apply mapl-1 function (map cdr lists)))))
	  (apply mapl-1 function lists)
	  (car lists))

	(define (mapcon function . lists)
	  (apply nconc (apply maplist function lists)))

	(define (mapcan function . lists)
	  (apply nconc (apply mapcar function lists)))
	  
	(define* (map-into result-sequence function . sequences)
	  (if (or (null? result-sequence)
		  (null? sequences))
	      result-sequence
	      (let* ((vals (apply mapcar function sequences))
		     (len (min (length vals) (length result-sequence))))
		(do ((i 0 (+ i 1)))
		    ((= i len))
		  (set! (result-sequence i) (vals i)))
		result-sequence)))


	(define input-stream-p input-port?)
	(define output-stream-p output-port?)


	;; -------- lists

	;; in CL (cdr '()) is nil

	(define (first l) (if (not (null? l)) (list-ref l 0) '()))
	(define (second l) (if (> (length l) 1) (list-ref l 1) '()))
	(define (third l) (if (> (length l) 2) (list-ref l 2) '()))
	(define (fourth l) (if (> (length l) 3) (list-ref l 3) '()))
	(define (fifth l) (if (> (length l) 4) (list-ref l 4) '()))
	(define (sixth l) (if (> (length l) 5) (list-ref l 5) '()))
	(define (seventh l) (if (> (length l) 6) (list-ref l 6) '()))
	(define (eighth l) (if (> (length l) 7) (list-ref l 7) '()))
	(define (ninth l) (if (> (length l) 8) (list-ref l 8) '()))
	(define (tenth l) (if (> (length l) 9) (list-ref l 9) '()))
	(define (nth n l) (if (< n (length l)) (list-ref l n) '()))
	(define (endp val) (if (null? val) #t (if (pair? val) #f (error "bad arg to endp"))))
	(define rest cdr)
	(define list-length length)
	(define* (cl-make-list size (initial-element '())) (make-list size initial-element))

	(define (copy-list lis) 
	  (if (not (pair? lis))
	      lis
	      (cons (car lis) (copy-list (cdr lis)))))

	(define (rplaca x y) (set-car! x y) x)
	(define (rplacd x y) (set-cdr! x y) x)

	(define (copy-tree lis)
	  (if (pair? lis)
	      (cons (copy-tree (car lis))
		    (copy-tree (cdr lis)))
	      lis))

	(define* (butlast lis (n 1))
	  (let ((len (length lis)))
	    (if (<= len n)
		'()
		(let ((result '()))
		  (do ((i 0 (+ i 1))
		       (lst lis (cdr lst)))
		      ((= i (- len n)) (reverse result))
		    (set! result (cons (car lst) result)))))))

	(define* (last lst (n 1))
	  (let ((len (length lst)))
	    (do ((i 0 (+ i 1))
		 (l lst (cdr l)))
		((or (null? l)
		     (>= i (- len n)))
		 l))))

	(define (nthcdr n lst) 
	  (do ((i n (- i 1)) 
	       (result lst (cdr result))) 
	      ((or (null? result) (zero? i)) result)))

	(define* (tree-equal a b (test eql)) 
	  (define (teq a b)
	    (if (not (pair? a))
		(and (not (pair? b))
		     (test a b))
		(and (pair? b)
		     (teq (car a) (car b))
		     (teq (cdr a) (cdr b)))))
	  (teq a b))

	(define (acons key datum alist) (cons (cons key datum) alist))

	(define* (subst-if new test tree (key identity))
	  (if (test (key tree))
	      new
	      (if (not (pair? tree))
		  tree
		  (cons (subst-if new test (car tree) key)
			(subst-if new test (cdr tree) key)))))

	(define* (subst-if-not new test tree (key identity))
	  (subst-if new (lambda (obj) (not (test obj))) tree key))

	(define* (subst new old tree (test eql) (key identity))
	  (subst-if new (lambda (obj) (test old obj)) tree key))

	(define (list* obj1 . objs)
	  (define (list-1 obj)
	    (if (null? (cdr obj))
		(car obj)
		(cons (car obj) (list-1 (cdr obj)))))
	  (if (null? objs)
	      obj1
	      (cons obj1 (list-1 objs))))

	(define* (assoc-if predicate alist (key car))
	  (if (null? alist)
	      '()
	      (if (and (not (null? (car alist)))
		       (predicate (key (car alist))))
		  (car alist)
		  (assoc-if predicate (cdr alist) key))))
	
	(define* (assoc-if-not predicate alist (key car))
	  (assoc-if (lambda (obj) (not (predicate obj))) alist key))

	(define* (cl-assoc item alist (test eql) (key car))
	  (assoc-if (lambda (obj) (test item obj)) alist key))
	
	(define* (rassoc-if predicate alist (key cdr))
	  (if (null? alist)
	      '()
	      (if (and (not (null? (car alist)))
		       (predicate (key (car alist))))
		  (car alist)
		  (rassoc-if predicate (cdr alist) key))))
	
	(define* (rassoc-if-not predicate alist (key cdr))
	  (rassoc-if (lambda (obj) (not (predicate obj))) alist key))

	(define* (rassoc item alist (test eql) (key cdr))
	  (rassoc-if (lambda (obj) (test item obj)) alist key))

	(define (copy-alist alist)
	  (if (null? alist)
	      '()
	      (cons (if (pair? (car alist))
			(cons (caar alist) (cdar alist))
			(car alist))
		    (copy-alist (cdr alist)))))

	(define (revappend x y) (append (reverse x) y))

	
	(define* (pairlis keys data alist)
	  (if (not (= (length keys) (length data)))
	      (error "pairlis keys and data lists should have the same length"))
	  (let ((lst (or alist '())))
	    (if (null? keys)
		lst
		(do ((key keys (cdr key))
		     (datum data (cdr datum)))
		    ((null? key) lst)
		  (set! lst (cons (cons (car key) (car datum)) lst))))))

	(define* (sublis alist tree (test eql) (key car))
	  (let ((val (cl-assoc tree alist test key)))
	    (if (not (null? val))
		(cdr val)
		(if (not (pair? tree))
		    tree
		    (cons (sublis alist (car tree) test key)
			  (sublis alist (cdr tree) test key))))))

	(define* (nsublis alist tree (test eql) (key car)) ; sacla
	  (define (sub subtree)
	    (let ((ac (cl-assoc subtree alist test key)))
	      (if (not (null? ac))
		  (cdr ac)
		  (if (not (pair? subtree))
		      subtree
		      (let ()
			(set-car! subtree (sub (car subtree)))
			(set-cdr! subtree (sub (cdr subtree)))
			subtree)))))
	  (sub tree))

	(define* (nsubst-if new predicate tree (key identity)) ; sacla
	  (define (sub subtree)
	    (if (predicate (key subtree))
		new
		(if (not (pair? subtree))
		    subtree
		    (let ()
		      (set-car! subtree (sub (car subtree)))
		      (set-cdr! subtree (sub (cdr subtree)))
		      subtree))))
	  (sub tree))

	(define* (nsubst-if-not new predicate tree (key identity))
	  (nsubst-if new (lambda (obj) (not (predicate obj))) tree key))
    
	(define* (nsubst new old tree (test eql) (key identity))
	  (nsubst-if new (lambda (obj) (test old obj)) tree key))

	(define (ldiff lst object) ; sacla
	  (if (not (eqv? lst object))
	      (let* ((result (list (car lst)))
		     (splice result))
		(call-with-exit
		 (lambda (return)
		   (do ((l (cdr lst) (cdr l)))
		       ((not (pair? l))
			(if (eql l object) 
			    (set-cdr! splice '()))
			result)
		     (if (eqv? l object)
			 (return result)
			 (set! splice (cdr (rplacd splice (list (car l))))))))))
	      '()))

	(define* (member-if predicate list (key identity))
	  (if (null? list)
	      '()
	      (if (predicate (key (car list)))
		  list
		  (member-if predicate (cdr list) key))))

	(define* (member-if-not predicate list (key identity))
	  (member-if (lambda (obj) (not (predicate obj))) list key))

	(define* (cl-member item list (test eql) (key identity))
	  (if (null? list)
	      '()
	      (if (test item (key (car list)))
		  list
		  (cl-member item (cdr list) test key))))

	(define* (adjoin item list (test eql) (key identity))
	  (if (not (null? (cl-member (key item) list test key)))
	      list
	      (cons item list)))

	(define (tailp sublist list)
	  (or (eq? sublist list)
	      (and (not (null? list))
		   (tailp sublist (cdr list)))))

	(define* (union list1 list2 (test eql) (key identity))
	  (let ((new-list (copy list1)))
	    (do ((obj list2 (cdr obj)))
		((null? obj) new-list)
	      (set! new-list (adjoin (car obj) new-list test key)))))

	(define nunion union) ; this is not required to be destructive

	(define* (intersection list1 list2 (test eql) (key identity))
	  (let ((new-list '()))
	    (do ((obj list1 (cdr obj)))
		((null? obj) new-list)
	      (if (not (null? (cl-member (key (car obj)) list2 test key)))
		  (set! new-list (adjoin (car obj) new-list test key))))))

	(define nintersection intersection)
	    
	(define* (set-difference list1 list2 (test eql) (key identity))
	  (let ((new-list '()))
	    (do ((obj list1 (cdr obj)))
		((null? obj) new-list)
	      (if (null? (cl-member (key (car obj)) list2 test key))
		  (set! new-list (adjoin (car obj) new-list test key))))))

	(define nset-difference set-difference)

	(define* (set-exclusive-or list1 list2 (test eql) (key identity))
	  (let ((new-list '()))
	    (do ((obj list1 (cdr obj)))
		((null? obj))
	      (if (null? (cl-member (key (car obj)) list2 test key))
		  (set! new-list (adjoin (car obj) new-list test key))))
	    (do ((obj list2 (cdr obj)))
		((null? obj) new-list)
	      (if (null? (cl-member (key (car obj)) list1 test key))
		  (set! new-list (adjoin (car obj) new-list test key))))))

	(define nset-exclusive-or set-exclusive-or)

	(define* (subsetp list1 list2 (test eql) (key identity))
	  (call-with-exit
	   (lambda (return)
	     (do ((obj list1 (cdr obj)))
		 ((null? obj) #t)
	      (if (null? (cl-member (key (car obj)) list2 test key))
		  (return nil))))))

	(define* (nbutlast list (n 1)) ; sacla
	  (if (null? list)
	      '()
	      (let ((length (do ((p (cdr list) (cdr p))
				 (i 1 (+ i 1)))
				((not (pair? p)) i))))
		(if (> length n)
		    (do ((1st (cdr list) (cdr 1st))
			 (2nd list 1st)
			 (count (- length n 1) (- count 1)))
			((zero? count) 
			 (set-cdr! 2nd '())
			 list))
		    '()))))

	(define (nconc . lists) ; sacla sort of
	  (let ((ls (let ()
		      (define (strip-nulls lst)
			(if (null? lst)
			    '()
			    (if (null? (car lst))
				(strip-nulls (cdr lst))
				lst)))
		      (strip-nulls lists))))
	    (if (null? ls)
		'()
	      (let* ((top (car ls))
		     (splice top))
		(do ((here (cdr ls) (cdr here)))
		    ((null? here) top)
		  (set-cdr! (last splice) (car here))
		  (if (not (null? (car here)))
		      (set! splice (car here))))))))

	(define (nreconc x y) (nconc (nreverse x) y))



	;; -------- numbers

	(define (conjugate z) (make-rectangular (real-part z) (- (imag-part z))))
	(define zerop zero?)
	(define oddp odd?)
	(define evenp even?)
	(define plusp positive?)
	(define minusp negative?)
	(define realpart real-part)
	(define imagpart imag-part)
	(define* (float x ignore) (* 1.0 x))
	(define rational rationalize)
	(define mod modulo)
	(define rem remainder)

	(define (logtest i1 i2) (not (zero? (logand i1 i2))))
	(define (logbitp index integer) (logtest (expt 2 index) integer))
	(define (lognand n1 n2) (lognot (logand n1 n2)))
	(define (lognor n1 n2) (lognot (logior n1 n2)))
	(define (logandc1 n1 n2) (logand (lognot n1) n2))
	(define (logandc2 n1 n2) (logand n1 (lognot n2)))
	(define (logorc1 n1 n2) (logior (lognot n1) n2))
	(define (logorc2 n1 n2) (logior n1 (logior n2)))
	(define (logeqv . ints) (lognot (apply logxor ints)))

	;; from slib
	(define (logcount n)
	  (define bitwise-bit-count
	    (letrec ((logcnt (lambda (n tot)
			       (if (zero? n)
				   tot
				   (logcnt (quotient n 16)
					   (+ (vector-ref
					       '#(0 1 1 2 1 2 2 3 1 2 2 3 2 3 3 4)
					       (modulo n 16))
					      tot))))))
	      (lambda (n)
		(cond ((negative? n) (lognot (logcnt (lognot n) 0)))
		      ((positive? n) (logcnt n 0))
		      (else 0)))))
	  (cond ((negative? n) (bitwise-bit-count (lognot n)))
		(else (bitwise-bit-count n))))

	(define-constant boole-clr 0)
	(define-constant boole-set 1)
	(define-constant boole-1 2)
	(define-constant boole-2 3)
	(define-constant boole-c1 4)
	(define-constant boole-c2 5)
	(define-constant boole-and 6)
	(define-constant boole-ior 7)
	(define-constant boole-xor 8)
	(define-constant boole-eqv 9)
	(define-constant boole-nand 10)
	(define-constant boole-nor 11)
	(define-constant boole-andc1 12)
	(define-constant boole-andc2 13)
	(define-constant boole-orc1 14)
	(define-constant boole-orc2 15)

	(define (boole op int1 int2)
	  (cond
	    ((= op boole-clr)   0)
	    ((= op boole-set)   -1) ;; all ones -- "always 1" is misleading
	    ((= op boole-1)     int1)
	    ((= op boole-2)     int2)
	    ((= op boole-c1)    (lognot int1))
	    ((= op boole-c2)    (lognot int2))
	    ((= op boole-and)   (logand int1 int2))
	    ((= op boole-ior)   (logior int1 int2))
	    ((= op boole-xor)   (logxor int1 int2))
	    ((= op boole-eqv)   (logeqv int1 int2))
	    ((= op boole-nand)  (lognot (logand int1 int2)))
	    ((= op boole-nor)   (lognot (logior int1 int2)))
	    ((= op boole-andc1) (logand (lognot int1) int2))
	    ((= op boole-andc2) (logand int1 (lognot int2)))
	    ((= op boole-orc1)  (logior (lognot int1) int2))
	    ((= op boole-orc2)  (logior int1 (lognot int2)))))

	;; from Rick
	(define (byte siz pos)
	  ;; cache size, position and mask.
	  (list siz pos (ash (- (ash 1 siz) 1) pos)))

	(define (byte-size bytespec) (car bytespec))
	(define (byte-position bytespec) (cadr bytespec))
	(define (byte-mask bytespec) (caddr bytespec))

	(define (ldb bytespec integer)
	  (ash (logand integer (byte-mask bytespec))
	       (- (byte-position bytespec))))

	(define (dpb integer bytespec into)
	  (logior (ash (logand integer (- (ash 1 (byte-size bytespec)) 1)) (byte-position bytespec))
		  (logand into (lognot (byte-mask bytespec)))))

	(define (ldb-test byte int) (not (zero? (ldb byte int))))
	(define (mask-field byte int) (logand int (dpb -1 byte 0)))
	(define (deposit-field byte spec int) (logior (logand byte (byte-mask spec)) (logand int (lognot (byte-mask spec)))))
	(define (scale-float x k) (* x (expt 2.0 k)))
	
	;; from clisp -- can't see any point to most of these
	(define-constant double-float-epsilon 1.1102230246251568e-16)
	(define-constant double-float-negative-epsilon 5.551115123125784e-17)
	(define-constant least-negative-double-float -2.2250738585072014e-308)
	(define-constant least-negative-long-float -5.676615526003731344e-646456994)
	(define-constant least-negative-normalized-double-float -2.2250738585072014e-308)
	(define-constant least-negative-normalized-long-float -5.676615526003731344e-646456994)
	(define-constant least-negative-normalized-short-float -1.1755e-38)
	(define-constant least-negative-normalized-single-float -1.1754944e-38)
	(define-constant least-negative-short-float -1.1755e-38)
	(define-constant least-negative-single-float -1.1754944e-38)
	(define-constant least-positive-double-float 2.2250738585072014e-308)
	(define-constant least-positive-long-float 5.676615526003731344e-646456994)
	(define-constant least-positive-normalized-double-float 2.2250738585072014e-308)
	(define-constant least-positive-normalized-long-float 5.676615526003731344e-646456994)
	(define-constant least-positive-normalized-short-float 1.1755e-38)
	(define-constant least-positive-normalized-single-float 1.1754944e-38)
	(define-constant least-positive-short-float 1.1755e-38)
	(define-constant least-positive-single-float 1.1754944e-38)
	(define-constant long-float-epsilon 5.4210108624275221706e-20)
	(define-constant long-float-negative-epsilon 2.7105054312137610853e-20)
	(define-constant most-negative-double-float -1.7976931348623157e308)
	;; most-negative-fixnum 
	(define-constant most-negative-long-float -8.8080652584198167656e646456992) 
	(define-constant most-negative-short-float -3.4028e38)
	(define-constant most-negative-single-float -3.4028235e38)
	(define-constant most-positive-double-float 1.7976931348623157e308)
	;; most-positive-fixnum 
	(define-constant most-positive-long-float 8.8080652584198167656e646456992)
	(define-constant most-positive-short-float 3.4028e38)
	(define-constant most-positive-single-float 3.4028235e38)
	(define-constant short-float-epsilon 7.6295e-6)
	(define-constant short-float-negative-epsilon 3.81476e-6)
	(define-constant single-float-epsilon 5.960465e-8)
	(define-constant single-float-negative-epsilon 2.9802326e-8)

	(define (lisp-implementation-type) "s7")
	(define (lisp-implementation-version) (s7-version))
	(define (software-type) "s7")
	(define (software-version) (s7-version))

	(define (machine-version)
	  (if (and (defined? 'file-exists?)
                   (file-exists? "/proc/cpuinfo"))
	      (call-with-input-file "/proc/cpuinfo"
		(lambda (cpufile)
		  (do ((line (read-line cpufile) (read-line cpufile)))
		      ((or (eof-object? line)
			   (string=? (substring line 0 10) "model name"))
		       (if (string? line)
			   (string-trim " " (substring line (+ 1 (position #\: line))))
			   "unknown")))))
	      "unknown"))
	
	;; = < <= > >= are the same, also min max + - * / lcm gcd exp expt log sqrt
	;; sin cos tan acos asin atan pi sinh cosh tanh asinh acosh atanh
	;; numerator denominator logior logxor logand ash integer-length random

	;; slightly different: floor ceiling truncate round and the ff cases thereof
	;; abs of complex -> magnitude
	(define (cl-abs x) (if (not (zero? (imag-part x))) (magnitude x) (abs x)))

	;; these actually return multiple values
	(define* (cl-floor x (divisor 1)) (floor (/ x divisor)))
	(define* (cl-ceiling x (divisor 1)) (ceiling (/ x divisor)))
	(define* (cl-truncate x (divisor 1)) (truncate (/ x divisor)))
	(define* (cl-round x (divisor 1)) (round (/ x divisor)))
	(define* (ffloor x divisor) (* 1.0 (cl-floor x divisor)))
	(define* (fceling x divisor) (* 1.0 (cl-ceiling x divisor)))
	(define* (ftruncate x divisor) (* 1.0 (cl-truncate x divisor)))
	(define* (fround x divisor) (* 1.0 (cl-round x divisor)))
       
	(define (/= . args) 
	  (if (null? (cdr args))
	      #t 
	      (if (member (car args) (cdr args))
		  #f
		  (apply /= (cdr args)))))

	(define (1+ x) (+ x 1))
	(define (1- x) (- x 1))
	(define (isqrt x) (floor (sqrt x)))
	(define phase angle)
	(define* (complex rl (im 0.0)) (make-rectangular rl im))
	(define (signum x) (if (zerop x) x (/ x (abs x))))
	(define (cis x) (exp (make-rectangular 0.0 x)))


	;; -------- characters

	(define char-code-limit 256)
	(define alpha-char-p char-alphabetic?)
	(define upper-case-p char-upper-case?)
	(define lower-case-p char-lower-case?)
	(define* (digit-char-p c (radix 10)) (string->number (string c) radix))
	(define (alphanumericp c) (or (char-alphabetic? c) (char-numeric? c)))

	(define* (char= . args) (or (< (length args) 2) (apply char=? args)))
	(define* (char< . args) (or (< (length args) 2) (apply char<? args)))
	(define* (char<= . args) (or (< (length args) 2) (apply char<=? args)))
	(define* (char> . args) (or (< (length args) 2) (apply char>? args)))
	(define* (char>= . args) (or (< (length args) 2) (apply char>=? args)))
	(define* (char-equal . args) (or (< (length args) 2) (apply char-ci=? args)))
	(define* (char-lessp . args) (or (< (length args) 2) (apply char-ci<? args)))
	(define* (char-greaterp . args) (or (< (length args) 2) (apply char-ci>? args)))
	(define* (char-not-lessp . args) (or (< (length args) 2) (apply char-ci>=? args)))
	(define* (char-not-greaterp . args) (or (< (length args) 2) (apply char-ci<=? args)))

	(define (char/= . args) 
	  (if (null? (cdr args))
	      #t 
	      (if (member (car args) (cdr args))
		  #f
		  (apply char/= (cdr args)))))

	(define (char-not-equal . args) 
	  (if (null? (cdr args))
	      #t 
	      (if (or (member (char-upcase (car args)) (cdr args))
		      (member (char-downcase (car args)) (cdr args)))
		  #f
		  (apply char-not-equal (cdr args)))))

	(define char-code char->integer)
	(define code-char integer->char)

	(define (character c) 
	  (if (char? c) 
	      c 
	      (if (integer? c)
		  (integer->char c)
		  (if (string? c)
		      (c 0)
		      (if (symbol? c)
			  ((symbol->string c) 0))))))

	;; char-upcase and char-downcase are ok
	(define char-int char->integer)
	(define int-char integer->char)

	(define* (digit-char w (radix 10))
	  (let ((str (number->string w radix)))
	    (and str (= (length str) 1) (str 0))))

	(define (both-case-p c) "unimplemented")
	(define (standard-char-p c) "unimplemented")
	(define (char-name c) "unimplemented")
	(define (name-char s) "unimplemented")

	;; --------

	(define terpri newline)


	;; -------- types

	(define vectorp vector?)
	(define simple-vector-p vector?)
	(define symbolp symbol?)
	(define (atom obj) (not (pair? obj)))
	(define consp pair?)
	(define (null obj) (or (not obj) (null? obj)))
	(define (listp obj) (or (null? obj) (pair? obj)))
	(define numberp number?)
	(define integerp integer?)
	(define rationalp rational?)
	(define (floatp l) (and (number? l) (not (rational? l)) (zero? (imag-part l)))) ; clisp
	(define (complexp l) (and (complex? l) (not (real? l))))
	(define realp real?)
	(define characterp char?)
	(define stringp string?)
	(define simple-string-p string?)
	(define arrayp vector?)
	(define simple-bit-vector-p vector?)
	(define keywordp keyword?)
	(define functionp procedure?)

	(define-constant t #t)
	(define-constant nil '())

	(define eq eq?)
	(define eql eqv?)
	(define equal equal?)

	(define (equalp x y)
	  (or (equal x y)
	      (and (char? x) (char? y) (char-ci=? x y))
	      (and (number? x) (number? y) (= x y))
	      (and (string? x) (string? y) (string-ci=? x y))))

	(define symbol-value symbol->value)
	(define symbol-function symbol->value)
	(define fdefinition symbol->value)
	(define boundp defined?)
	(define fboundp defined?)
	(define (funcall fn . arguments) (apply fn arguments))
	(define-constant call-arguments-limit 65536)

	(define (identity x) x)


	;; -------- sequences

	(define* (count-if predicate sequence from-end (start 0) end (key identity))
	  (let* ((counts 0)
		 (len (length sequence))
		 (nd (or (and (number? end) end) len))) ; up to but not including end
	    (if (< nd start)
		(error "count-if :start ~A is greater than ~A ~A" start (if end ":end" "length") nd))
	    (if (not from-end)
		(do ((i start (+ i 1)))
		    ((= i nd))
		  (if (predicate (key (sequence i)))
		      (set! counts (+ counts 1))))
		(do ((i (- nd 1) (- i 1)))
		    ((< i start))
		  (if (predicate (key (sequence i)))
		      (set! counts (+ counts 1)))))
		counts))

	(define* (count-if-not predicate sequence from-end (start 0) end (key identity))
	  (count-if (lambda (obj) (not (predicate obj))) sequence from-end start end key))

	(define* (count item sequence from-end (test eql) (start 0) end (key identity))
	  (count-if (lambda (arg) (test item arg)) sequence from-end start end key))

	(define* (find-if predicate sequence from-end (start 0) end (key identity))
	  (let* ((len (length sequence))
		 (nd (or (and (number? end) end) len))) ; up to but not including end
	    (if (< nd start)
		(error "~A :start ~A is greater than ~A ~A" __func__ start (if end ":end" "length") nd))
	    (call-with-exit
	     (lambda (return)
	       (if (not from-end)
		   (do ((i start (+ i 1)))
		       ((= i nd) #f)
		     (if (predicate (key (sequence i)))
			 (return (sequence i))))
		   (do ((i (- nd 1) (- i 1)))
		       ((< i start) #f)
		     (if (predicate (key (sequence i)))
			 (return (sequence i)))))))))

	(define* (find-if-not predicate sequence from-end (start 0) end (key identity))
	  (find-if (lambda (obj) (not (predicate obj))) sequence from-end start end key))

	(define* (find item sequence from-end (test eql) (start 0) end (key identity))
	  (find-if (lambda (arg) (test item arg)) sequence from-end start end key))
	     
	(define* (position-if predicate sequence from-end (start 0) end (key identity))
	  (let* ((len (length sequence))
		 (nd (or (and (number? end) end) len))) ; up to but not including end
	    (if (< nd start)
		(error "~A :start ~A is greater than ~A ~A" __func__ start (if end ":end" "length") nd))
	    (call-with-exit
	     (lambda (return)
	       (if (not from-end)
		   (do ((i start (+ i 1)))
		       ((= i nd) #f)
		     (if (predicate (key (sequence i)))
			 (return i)))
		   (do ((i (- nd 1) (- i 1)))
		       ((< i start) #f)
		     (if (predicate (key (sequence i)))
			 (return i))))))))

	(define* (position-if-not predicate sequence from-end (start 0) end (key identity))
	  (position-if (lambda (obj) (not (predicate obj))) sequence from-end start end key))

	(define* (position item sequence from-end (test eql) (start 0) end (key identity))
	  (position-if (lambda (arg) (test item arg)) sequence from-end start end key))


	(define* (nsubstitute-if new-item test sequence from-end (start 0) end count (key identity))
	  (if (and (number? count)
		   (not (positive? count)))
	      sequence
	      (let* ((len (length sequence))
		     (nd (or (and (number? end) end) len))) ; up to but not including end
		(if (< nd start)
		    (error "~A :start ~A is greater than ~A ~A" __func__ start (if end ":end" "length") nd))
		(let ((cur-count 0))
		  (if (not (number? count))
		      (set! count len))
		  (if (not from-end)
		      (do ((i start (+ i 1)))
			  ((or (= cur-count count)
			       (= i nd))
			   sequence)
			(if (test (key (sequence i)))
			    (begin
			      (set! cur-count (+ cur-count 1))
			      (set! (sequence i) new-item))))
		      (do ((i (- nd 1) (- i 1)))
			  ((or (= cur-count count)
			       (< i start))
			   sequence)
			(if (test (key (sequence i)))
			    (begin
			      (set! cur-count (+ cur-count 1))
			      (set! (sequence i) new-item)))))))))

	(define* (nsubstitute-if-not new-item test sequence from-end (start 0) end count (key identity))
	  (nsubstitute-if new-item (lambda (obj) (not (test obj))) sequence from-end start end count key))

	(define* (nsubstitute new-item old-item sequence from-end (test eql) (start 0) end count (key identity))
	  (nsubstitute-if new-item (lambda (arg) (test old-item arg)) sequence from-end start end count key))

	(define* (substitute-if new-item test sequence from-end (start 0) end count (key identity))
	  (nsubstitute-if new-item test (copy sequence) from-end start end count key))

	(define* (substitute-if-not new-item test sequence from-end (start 0) end count (key identity))
	  (substitute-if new-item (lambda (obj) (not (test obj))) (copy sequence) from-end start end count key))

	(define* (substitute new-item old-item sequence from-end (test eql) (start 0) end count (key identity))
	  (nsubstitute new-item old-item (copy sequence) from-end test start end count key))

	(define* (reduce function sequence from-end (start 0) end initial-value (key identity))
	  (let* ((slen (length sequence))
		 (nd (or (and (number? end) end) slen))
		 (len (min slen (- nd start))))
	    (if (< nd start)
		(error "~A :start ~A is greater than ~A ~A" __func__ start (if end ":end" "length") nd))
	    (if (not (positive? len))
		(or initial-value
		    (function))
		(if (and (= len 1)
			 (not initial-value))
		    (sequence start)
		    (if (and (not from-end) (not (null? from-end)))
			(let* ((first-arg (or initial-value (key (sequence start))))
			       (second-arg (if initial-value (key (sequence start)) (key (sequence (+ start 1)))))
			       (val (function first-arg second-arg)))
			  (do ((i (if initial-value (+ start 1) (+ start 2)) (+ i 1)))
			      ((= i nd) val)
			    (set! val (function val (key (sequence i))))))
			(let* ((second-arg (or initial-value (key (sequence (- nd 1)))))
			       (first-arg (if initial-value (key (sequence (- nd 1))) (key (sequence (- nd 2)))))
			       (val (function first-arg second-arg)))
			  (do ((i (if initial-value (- nd 2) (- nd 3)) (- i 1)))
			      ((< i start) val)
			    (set! val (function (key (sequence i)) val)))))))))

	(define (nreverse sequence)
	  (let ((len (length sequence)))
	    (do ((i 0 (+ i 1))
		 (j (- len 1) (- j 1)))
		((>= i j) sequence)
	      (let ((tmp (sequence i)))
		(set! (sequence i) (sequence j))
		(set! (sequence j) tmp)))))

	(define (cl-reverse sequence)
	  (nreverse (copy sequence)))
	
	(define copy-seq copy)
	(define (complement fn) (lambda args (not (apply fn args))))
	(define (elt sequence index) (sequence index))
	;; length is ok

	(define* (some predicate . sequences)
	  (call-with-exit
	   (lambda (return)
		     (apply for-each 
		      (lambda args
			(let ((val (apply predicate args)))
			  (if val (return val))))
		      sequences)
		     #f)))

	(define* (notany predicate . sequences)
	  (not (apply some predicate sequences)))

	(define* (every predicate . sequences)
	  (call-with-exit
	   (lambda (return)
		     (apply for-each 
		      (lambda args
			(if (not (apply predicate args))
			    (return #f)))
		      sequences)
		     #t)))

	(define* (notevery predicate . sequences)
	  (not (apply every predicate sequences)))

	(define* (cl-fill sequence item (start 0) end) ; actuall "fill" doesn't collide, but it's confusing
	  (let ((nd (or (and (not (null? end)) end)
			(length sequence))))
	    (if (and (= start 0)
		     (= nd (length sequence)))
		(fill! sequence item)
		(do ((i start (+ i 1)))
		    ((= i nd))
		  (set! (sequence i) item)))
	    sequence))

	;; many of the sequence functions return a different length sequence, but
	;;   for user-defined sequence types, we can't use the 'type kludge (or
	;;   at least it's ugly), so we need either (make obj size initial-value)
	;;   where obj is a representative of the desired type, or another
	;;   arg to copy giving the new object's size.  For now, I'll cobble up
	;;   something explicit.
	;;
	;; perhaps the extended type could give its type symbol as well as the make function?
	;; 'vct and make-vct etc

	(define (make obj size)
	  (cond ((vector? obj)     (make-vector size))
		((list? obj)       (make-list size))
		((string? obj)     (make-string size))
		((hash-table? obj) (make-hash-table size)))) ; does this make any sense?

	(define* (make-sequence type size initial-element)
	  (case type 
	    ((vector bit-vector simple-vector) (make-vector size initial-element))
	    ((hash-table) (make-hash-table size))
	    ((string) (cl-make-string size (or initial-element #\null))) ; not #f!
	    ((list) (cl-make-list size initial-element))
            (else '())))

	(define (cl-map type func . lists)
	  (let* ((vals (apply mapcar func lists))
		 (len (length vals)))
	    (let ((obj (make-sequence (or type 'list) len)))
	      (if (> (length obj) 0)
		  (do ((i 0 (+ i 1)))
		      ((= i len))
		    (set! (obj i) (vals i))))
	      obj)))

	(define* (subseq sequence start end)
	  (let* ((len (length sequence))
		 (nd (or (and (number? end) end) len))
		 (size (- nd start))
		 (obj (make sequence size)))
	    (do ((i start (+ i 1))
		 (j 0 (+ j 1)))
		((= i nd) obj)
	      (set! (obj j) (sequence i)))))
	
	(define (concatenate type . sequences)
	  (let* ((len (apply + (map length sequences)))
		 (new-obj (make-sequence type len))
		 (ctr 0))
	    (for-each
	     (lambda (sequence)
	       (for-each
		(lambda (obj)
		  (set! (new-obj ctr) obj)
		  (set! ctr (+ ctr 1)))
		sequence))
	     sequences)
	    new-obj))

	;; :(concatenate 'list "hiho" '#(1 2)) -> (#\h #\i #\h #\o 1 2)

	(define* (replace seq1 seq2 (start1 0) end1 (start2 0) end2)
	  (let* ((len1 (length seq1))
		 (len2 (length seq2))
		 (nd1 (or (and (number? end1) end1) len1))
		 (nd2 (or (and (number? end2) end2) len2)))
	    (if (and (eq? seq1 seq2)
		     (> start1 start2))
		(let ((offset (- start1 start2)))
		  (do ((i (- nd1 1) (- i 1)))
		      ((< i start1) seq1)
		    (set! (seq1 i) (seq1 (- i offset)))))
		(do ((i start1 (+ i 1))
		     (j start2 (+ j 1)))
		    ((or (= i nd1)
			 (= j nd2))
		     seq1)
		  (set! (seq1 i) (seq2 j))))))
	
	(define* (remove-if predicate sequence from-end (start 0) end count (key identity))
	  (let* ((len (length sequence))
		 (nd (or (and (number? end) end) len))
		 (num (if (number? count) count len))
		 (changed 0))
	    (if (not (positive? num))
		sequence
		(let ((result '()))
		  (if (null from-end)
		      (do ((i 0 (+ i 1)))
			  ((= i len))
			(if (or (< i start)
				(>= i nd)
				(>= changed num)
				(not (predicate (key (sequence i)))))
			    (set! result (cons (sequence i) result))
			    (set! changed (+ changed 1))))
		      (do ((i (- len 1) (- i 1)))
			  ((< i 0))
			(if (or (< i start)
				(>= i nd)
				(>= changed num)
				(not (predicate (key (sequence i)))))
			    (set! result (cons (sequence i) result))
			    (set! changed (+ changed 1)))))		    
		  (let* ((len (length result))
			 (obj (make sequence len))
			 (vals (if (null from-end) (reverse result) result)))
		    (do ((i 0 (+ i 1)))
			((= i len))
		      (set! (obj i) (vals i)))
		    obj)))))
	
	(define* (remove-if-not predicate sequence from-end (start 0) end count (key identity))
	  (remove-if (lambda (obj) (not (predicate obj))) sequence from-end start end count key))
	
	(define* (remove item sequence from-end (test eql) (start 0) end count (key identity))
	  (remove-if (lambda (arg) (test item arg)) sequence from-end start end count key))

	(define-macro* (delete-if predicate sequence from-end (start 0) end count (key identity))
	  `(let ((obj (remove-if ,predicate ,sequence ,from-end ,start ,end ,count ,key)))
	     (if (symbol? ',sequence)
		 (set! ,sequence obj))
	     obj))
	
	(define-macro* (delete-if-not predicate sequence from-end (start 0) end count (key identity))
	  `(let ((obj (remove-if-not ,predicate ,sequence ,from-end ,start ,end ,count ,key)))
	     (if (symbol? ',sequence)
		 (set! ,sequence obj))
	     obj))
	
	(define-macro* (delete item sequence from-end (test eql) (start 0) end count (key identity))
	  `(let ((obj (remove ,item ,sequence ,from-end ,test ,start ,end ,count ,key)))
	     (if (symbol? ',sequence)
		 (set! ,sequence obj))
	     obj))
	
	(define* (remove-duplicates sequence from-end (test eql) (start 0) end (key identity))
	  (let* ((result '())
		 (start-seq (+ start 1))
		 (len (length sequence))
		 (nd (if (number? end) end len)))
	    (do ((i start (+ i 1)))
		((= i nd))
	      (let* ((orig-obj (sequence i))
		     (obj (key orig-obj)))
		(if (null from-end)
		    (begin
		      (if (not (find obj sequence :start start-seq :end nd :test test :key key))
			  (set! result (cons orig-obj result)))
		      (set! start-seq (+ start-seq 1)))
		    (if (not (find obj result :test test :key key))
			(set! result (cons orig-obj result))))))
	    (let* ((res (reverse result))
		   (new-len (+ (length result) start (- len nd)))
		   (new-seq (make sequence new-len)))
	      (let ((n 0))
		(do ((i 0 (+ i 1)))
		    ((= i len) new-seq)
		  (if (or (< i start)
			  (>= i nd))
		      (begin
			(set! (new-seq n) (sequence i))
			(set! n (+ n 1)))
		      (if (not (null? res))
			  (begin
			    (set! (new-seq n) (car res))
			    (set! res (cdr res))
			    (set! n (+ n 1))))))))))
	
	(define-macro* (delete-duplicates sequence from-end (test eql) (start 0) end (key identity))
	  `(let ((obj (remove-duplicates ,sequence ,from-end ,test ,start ,end ,key)))
	     (if (symbol? ,sequence)
		 (set! ,sequence obj))
	     obj))
	
	(define* (merge result-type seq1 seq2 predicate (key identity))
	  (let* ((len1 (length seq1))
		 (len2 (length seq2))
		 (size (+ len1 len2))
		 (obj (make-sequence result-type size))
		 (i 0)
		 (j 0))
	    (do ((n 0 (+ n 1)))
		((or (= i len1)
		     (= j len2))
		 (if (< i len1)
		     (do ((k i (+ k 1)))
			 ((= k len1) obj)
		       (set! (obj n) (seq1 k))
		       (set! n (+ n 1)))
		     (if (< j len2)
			 (do ((k j (+ k 1)))
			     ((= k len2) obj)
			   (set! (obj n) (seq2 k))
			   (set! n (+ n 1)))
			 obj)))
	      (if (null (predicate (key (seq1 i)) (key (seq2 j))))
		  (begin
		    (set! (obj n) (seq2 j))
		    (set! j (+ j 1)))
		  (begin
		    (set! (obj n) (seq1 i))
		    (set! i (+ i 1)))))))
	
	(define* (search seq1 seq2 from-end (test eql) (key identity) (start1 0) (start2 0) end1 end2)
	  (let* ((len1 (length seq1))
		 (len2 (length seq2))
		 (nd1 (or (and (number? end1) end1) len1))
		 (nd2 (or (and (number? end2) end2) len2)))
	    (set! len1 (min len1 (- nd1 start1)))
	    (set! len2 (min len2 (- nd2 start2)))
	    (if (or (= len2 0)
		    (> len1 len2))
		'()
		(call-with-exit
		 (lambda (return)
		   (if (or (not from-end) (null? from-end))
		       (do ((i start2 (+ i 1)))
			   ((> i (- nd2 len1)) '())
			 (do ((j start1 (+ j 1))
			      (k i (+ k 1)))
			     ((or (= j nd1)
				  (not (test (key (seq1 j)) (key (seq2 k)))))
			      (if (= j nd1)
				  (return i)))))
		       (do ((i (- nd2 len1) (- i 1)))
			   ((< i start2) '())
			 (do ((j start1 (+ j 1))
			      (k i (+ k 1)))
			     ((or (= j nd1)
				  (not (test (key (seq1 j)) (key (seq2 k)))))
			      (if (= j nd1)
				  (return i)))))))))))
	
	(define* (mismatch seq1 seq2 from-end (test eql) (key identity) (start1 0) (start2 0) end1 end2)
	  (let* ((nd1 (or (and (number? end1) end1) (length seq1)))
		 (nd2 (or (and (number? end2) end2) (length seq2))))
	    (if (not from-end)
		(do ((i start1 (+ i 1))
		     (j start2 (+ j 1)))
		    ((or (= i nd1)
			 (= j nd2)
			 (not (test (key (seq1 i)) (key (seq2 j)))))
		     (if (and (= i nd1) (= j nd2))
			 '()
			 i)))
		(do ((i (- nd1 1) (- i 1))
		     (j (- nd2 1) (- j 1)))
		    ((or (< i start1)
			 (< j start2)
			 (not (test (key (seq1 i)) (key (seq2 j)))))
		     (if (and (< i start1) (< j start2))
			 '()
			 (+ i 1)))))))
	
	
	;; -------- strings
	
	(define char string-ref)
	(define schar string-ref)
	(define* (cl-make-string size (initial-element #\null)) (make-string size initial-element))
	
	(define (cl-string x)
	  (if (string? x) x
	      (if (char? x)
		  (string x)
		  (if (symbol? x) (symbol->string x)
		      (error "string ~A?" x)))))
	
	(define* (string= str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2))))
	    (if (and (not end1) (not end2) (= start1 0) (= start2 0))
		(string=? str1 str2)
		(string=? (subseq str1 start1 nd1)
			  (subseq str2 start2 nd2)))))
	
	(define* (string-equal str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2))))
	    (if (and (not end1) (not end2) (= start1 0) (= start2 0))
		(string-ci=? str1 str2)
		(string-ci=? (subseq str1 start1 nd1)
			     (subseq str2 start2 nd2)))))
	
	(define (string-prefixes-equal str1 str2 start1 nd1 start2 nd2)
	  (do ((i start1 (+ i 1))
	       (j start2 (+ j 1)))
	      ((or (= i nd1)
		   (= j nd2)
		   (not (char=? (str1 i) (str2 j))))
	       i)))
	
	(define (string-prefixes-equal-ci str1 str2 start1 nd1 start2 nd2)
	  (do ((i start1 (+ i 1))
	       (j start2 (+ j 1)))
	      ((or (= i nd1)
		   (= j nd2)
		   (not (char-ci=? (str1 i) (str2 j))))
	       i)))
	
	(define* (string< str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string<? str1 str2)
			  (string<? (subseq str1 start1 nd1)
				    (subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal str1 str2 start1 nd1 start2 nd2))))

	(define* (string-lessp str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string-ci<? str1 str2)
			  (string-ci<? (subseq str1 start1 nd1)
				       (subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal-ci str1 str2 start1 nd1 start2 nd2))))

	(define* (string<= str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string<=? str1 str2)
			  (string<=? (subseq str1 start1 nd1)
				     (subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal str1 str2 start1 nd1 start2 nd2))))

	(define* (string-not-greaterp str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string-ci<=? str1 str2)
			  (string-ci<=? (subseq str1 start1 nd1)
					(subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal-ci str1 str2 start1 nd1 start2 nd2))))

	(define* (string> str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string>? str1 str2)
			  (string>? (subseq str1 start1 nd1)
				    (subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal str1 str2 start1 nd1 start2 nd2))))

	(define* (string-greaterp str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string-ci>? str1 str2)
			  (string-ci>? (subseq str1 start1 nd1)
				       (subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal-ci str1 str2 start1 nd1 start2 nd2))))

	(define* (string>= str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string>=? str1 str2)
			  (string>=? (subseq str1 start1 nd1)
				     (subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal str1 str2 start1 nd1 start2 nd2))))

	(define* (string-not-lessp str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string-ci>=? str1 str2)
			  (string-ci>=? (subseq str1 start1 nd1)
					(subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal-ci str1 str2 start1 nd1 start2 nd2))))

	(define* (string/= str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (not (string=? str1 str2))
			  (not (string=? (subseq str1 start1 nd1)
					 (subseq str2 start2 nd2))))))
	    (and val (string-prefixes-equal str1 str2 start1 nd1 start2 nd2))))
	
	(define* (string-not-equal str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (not (string-ci=? str1 str2))
			  (not (string-ci=? (subseq str1 start1 nd1)
					    (subseq str2 start2 nd2))))))
	    (and val (string-prefixes-equal-ci str1 str2 start1 nd1 start2 nd2))))

	(define (string-left-trim bag str-1)
	  (let ((str (cl-string str-1)))
	    (if (string? bag) (set! bag (string->list bag)))
	    (let ((len (length str)))
	      (do ((i 0 (+ i 1)))
		  ((or (= i len)
		       (not (member (str i) bag)))
		   (if (= i 0)
		       str
		       (subseq str i)))))))
		 
	(define (string-right-trim bag str-1)
	  (let ((str (cl-string str-1)))
	    (if (string? bag) (set! bag (string->list bag)))
	    (let ((len (length str)))
	      (do ((i (- len 1) (- i 1)))
		  ((or (< i 0)
		       (not (member (str i) bag)))
		   (if (= i (- len 1))
		       str
		       (subseq str 0 (+ i 1))))))))
		 
	(define (string-trim bag str)
	  (string-right-trim bag (string-left-trim bag str)))

	(define* (nstring-upcase str (start 0) end)
	  (let ((nd (if (number? end) end (length str))))
	    (do ((i start (+ i 1)))
		((= i nd) str)
	      (set! (str i) (char-upcase (str i))))))

	(define* (string-upcase str-1 (start 0) end)
	  (let ((str (cl-string str-1)))
	    (nstring-upcase (copy str) start end)))

	(define* (nstring-downcase str (start 0) end)
	  (let ((nd (if (number? end) end (length str))))
	    (do ((i start (+ i 1)))
		((= i nd) str)
	      (set! (str i) (char-downcase (str i))))))

	(define* (string-downcase str-1 (start 0) end)
	  (let ((str (cl-string str-1)))
	    (nstring-downcase (copy str) start end)))

	(define* (nstring-capitalize str-1 (start 0) end)
	  (define (alpha? c) 
	    (or (char-alphabetic? c) 
		(char-numeric? c)))
	  (let ((str (cl-string str-1)))
	    (let ((nd (if (number? end) end (length str))))
	      (do ((i start (+ i 1)))
		  ((= i nd) str)
		(if (alpha? (str i))
		    (if (or (= i 0)
			    (not (alpha? (str (- i 1)))))
			(set! (str i) (char-upcase (str i)))
			(set! (str i) (char-downcase (str i)))))))))

	(define* (string-capitalize str-1 (start 0) end)
	  (let ((str (cl-string str-1)))
	    (nstring-capitalize (copy str) start end)))


	;; -------- vectors

	;; vector is ok

	(define svref vector-ref)
	(define aref vector-ref)
        (define array-dimensions vector-dimensions) 
	(define array-total-size vector-length)
        (define (array-dimension array num) (list-ref (vector-dimensions array) num))

	(define-constant array-dimension-limit 16777215)
	(define-constant array-rank-limit 4096)
	(define-constant array-total-size-limit 16777215)

	(define* (make-array dimensions element-type initial-element initial-contents adjustable fill-pointer displaced-to displaced-index-offset)
	  (if (eq? element-type 'character)
	      (or (and initial-contents
		       (string-copy initial-contents))
		  (cl-make-string dimensions initial-element))
	      (make-vector (or dimensions 1) initial-element)))

	(define (array-in-bounds-p array . subscripts)
	  (define (in-bounds dims subs)
	    (or (null? subs)
		(null? dims)
		(and (< (car subs) (car dims))
		     (in-bounds (cdr dims) (cdr subs)))))
	  (in-bounds (vector-dimensions array) subscripts))

	(define (row-major-index array . subscripts) 
	  (apply + (maplist (lambda (x y)
			      (* (car x) (apply * (cdr y))))
			    subscripts
			    (vector-dimensions array))))


	;; -------- defstruct

	(defmacro defstruct (struct-name . fields)
	  (let* ((name (if (list? struct-name) (car struct-name) struct-name))
		 (sname (if (string? name) name (symbol->string name)))
		 
		 (fsname (if (list? struct-name)
			     (let ((cname (assoc :conc-name (cdr struct-name))))
			       (if cname 
				   (symbol->string (cadr cname))
				   sname))
			     sname))
		 
		 (make-name (if (list? struct-name)
				(let ((cname (assoc :constructor (cdr struct-name))))
				  (if cname 
				      (cadr cname)
				      (string->symbol (string-append "make-" sname))))
				(string->symbol (string-append "make-" sname))))
		 
		 (copy-name (if (list? struct-name)
				(let ((cname (assoc :copier (cdr struct-name))))
				  (if cname 
				      (cadr cname)
				      (string->symbol (string-append "copy-" sname))))
				(string->symbol (string-append "copy-" sname))))
		 
		 (field-names (map (lambda (n)
				     (symbol->string (if (list? n) (car n) n)))
				   fields))
		 
		 (field-types (map (lambda (field)
				     (if (list? field)
					 (apply (lambda* (val type read-only) type) (cdr field))
					 #f))
				   fields))
		 
		 (field-read-onlys (map (lambda (field)
					  (if (list? field)
					      (apply (lambda* (val type read-only) read-only) (cdr field))
					      #f))
					fields)))
	    `(begin
	       
	       (define ,(string->symbol (string-append sname "?"))
		 (lambda (obj)
		   (and (vector? obj)
			(eq? (obj 0) ',(string->symbol sname)))))
	       
	       (define* (,make-name
			 ,@(map (lambda (n)
				  (if (and (list? n)
					   (>= (length n) 2))
				      (list (car n) (cadr n))
				      (list n #f)))
				fields))
		 (vector ',(string->symbol sname) ,@(map string->symbol field-names)))
	       
	       (define ,copy-name copy)
	       
	       ,@(map (let ((ctr 1))
			(lambda (n type read-only)
			  (let ((val (if read-only
					 `(define ,(string->symbol (string-append fsname "-" n))
					    (lambda (arg) (arg ,ctr)))
					 `(define ,(string->symbol (string-append fsname "-" n))
					    (make-procedure-with-setter 
					     (lambda (arg) (arg ,ctr)) 
					     (lambda (arg val) (set! (arg ,ctr) val)))))))
			    (set! ctr (+ 1 ctr))
			    val)))
		      field-names field-types field-read-onlys))))
	
	;; not yet implemented: :print-function :include :named :type :initial-offset
	;;   also the explicit constructor business

	(define-macro (enum . args) ; (enum zero one two)
	  `(begin
	     ,@(let ((names '()))
		 (do ((arg args (cdr arg))
		      (i 0 (+ i 1)))
		     ((null? arg) names)
		   (set! names (cons
				`(define ,(car arg) ,i)
				names))))))

	(define-macro (let*-values vals . body)
	  (let ((args '())
		(exprs '()))
	    (for-each
	     (lambda (arg+expr)
	       (set! args (cons (car arg+expr) args))
	       (set! exprs (cons (cadr arg+expr) exprs)))
	     vals)
	    (let ((form `((lambda ,(car args) ,@body) ,(car exprs))))
	      (if (not (null? (cdr args)))
		  (for-each
		   (lambda (arg expr)
		     (set! form `((lambda ,arg ,form) ,expr)))
		   (cdr args)
		   (cdr exprs)))
	      form)))

        (let ()

          ;; this is the nbody computer shootout benchmark taken from mzscheme
          ;; if we were serious about benchmarks, this could use run.

	  (define +days-per-year+ 365.24)
	  (define +solar-mass+ (* 4 pi pi))
	  (defstruct body x y z vx vy vz mass)
	  
	  (define *sun*
	    (make-body 0.0 0.0 0.0 0.0 0.0 0.0 +solar-mass+))
	  
	  (define *jupiter*
	    (make-body 4.84143144246472090
		       -1.16032004402742839
		       -1.03622044471123109e-1
		       (* 1.66007664274403694e-3 +days-per-year+)
		       (* 7.69901118419740425e-3 +days-per-year+)
		       (* -6.90460016972063023e-5 +days-per-year+)
		       (* 9.54791938424326609e-4 +solar-mass+)))
	  
	  (define *saturn*
	    (make-body 8.34336671824457987
		       4.12479856412430479
		       -4.03523417114321381e-1
		       (* -2.76742510726862411e-3 +days-per-year+)
		       (* 4.99852801234917238e-3 +days-per-year+)
		       (* 2.30417297573763929e-5 +days-per-year+)
		       (* 2.85885980666130812e-4 +solar-mass+)))
	  
	  (define *uranus*
	    (make-body 1.28943695621391310e1
		       -1.51111514016986312e1
		       -2.23307578892655734e-1
		       (* 2.96460137564761618e-03 +days-per-year+)
		       (* 2.37847173959480950e-03 +days-per-year+)
		       (* -2.96589568540237556e-05 +days-per-year+)
		       (*  4.36624404335156298e-05 +solar-mass+)))
	  
	  (define *neptune*
	    (make-body 1.53796971148509165e+01
		       -2.59193146099879641e+01
		       1.79258772950371181e-01
		       (* 2.68067772490389322e-03 +days-per-year+)
		       (* 1.62824170038242295e-03 +days-per-year+)
		       (* -9.51592254519715870e-05 +days-per-year+)
		       (* 5.15138902046611451e-05 +solar-mass+)))
	  
	  (define (offset-momentum system)
	    (let loop-i ((i system) (px 0.0) (py 0.0) (pz 0.0))
	      (if (null? i)
		  (begin
		    (set! (body-vx (car system)) (/ (- px) +solar-mass+))
		    (set! (body-vy (car system)) (/ (- py) +solar-mass+))
		    (set! (body-vz (car system)) (/ (- pz) +solar-mass+)))
		  (loop-i (cdr i)
			  (+ px (* (body-vx (car i)) (body-mass (car i))))
			  (+ py (* (body-vy (car i)) (body-mass (car i))))
			  (+ pz (* (body-vz (car i)) (body-mass (car i))))))))
	  
	  (define (energy system)
	    (let loop-o ((o system) (e 0.0))
	      (if (null? o)
		  e
		  (let ((e (+ e (* 0.5 (body-mass (car o))
				   (+ (* (body-vx (car o)) (body-vx (car o)))
				      (* (body-vy (car o)) (body-vy (car o)))
				      (* (body-vz (car o)) (body-vz (car o))))))))
		    (let loop-i ((i (cdr o)) (e e))
		      (if (null? i)
			  (loop-o (cdr o) e)
			  (let* ((dx (- (body-x (car o)) (body-x (car i))))
				 (dy (- (body-y (car o)) (body-y (car i))))
				 (dz (- (body-z (car o)) (body-z (car i))))
				 (distance (sqrt (+ (* dx dx) (* dy dy) (* dz dz)))))
			    (let ((e  (- e (/ (* (body-mass (car o)) (body-mass (car i))) distance))))
			      (loop-i (cdr i) e)))))))))
	  
	  (define (advance system dt)
	    (let loop-o ((o system))
	      (unless (null? o)
		      (let loop-i ((i (cdr o)))
			(unless (null? i)
				(let* ((o1 (car o))
				       (i1 (car i))
				       (dx (- (body-x o1) (body-x i1)))
				       (dy (- (body-y o1) (body-y i1)))
				       (dz (- (body-z o1) (body-z i1)))
				       (distance (sqrt (+ (* dx dx) (* dy dy) (* dz dz))))
				       (mag (/ dt (* distance distance distance)))
				       (dxmag (* dx mag))
				       (dymag (* dy mag))
				       (dzmag (* dz mag))
				       (om (body-mass o1))
				       (im (body-mass i1)))
				  (set! (body-vx o1) (- (body-vx o1) (* dxmag im)))
				  (set! (body-vy o1) (- (body-vy o1) (* dymag im)))
				  (set! (body-vz o1) (- (body-vz o1) (* dzmag im)))
				  (set! (body-vx i1) (+ (body-vx i1) (* dxmag om)))
				  (set! (body-vy i1) (+ (body-vy i1) (* dymag om)))
				  (set! (body-vz i1) (+ (body-vz i1) (* dzmag om)))
				  (loop-i (cdr i)))))
		      (loop-o (cdr o))))
	    (let loop-o ((o system))
	      (unless (null? o)
		      (let ((o1 (car o)))
			(set! (body-x o1) (+ (body-x o1) (* dt (body-vx o1))))
			(set! (body-y o1) (+ (body-y o1) (* dt (body-vy o1))))
			(set! (body-z o1) (+ (body-z o1) (* dt (body-vz o1))))
			(loop-o (cdr o))))))
	  
	  ;; (define (nbody-test)
	  
	  (let ((n 10) ;(n 1000) ; (command-line #:args (n) (string->number n)))
		(system (list *sun* *jupiter* *saturn* *uranus* *neptune*)))
	    (offset-momentum system)
	    (let ((initial (energy system)))
	      (do ((i 1 (+ i 1)))
		  ((< n i))
		(advance system 0.01))
	      (let ((final (energy system)))
		(num-test initial -0.16907516382852)
		(num-test final -0.16908760523461)
					;(list initial final))))) ; (-0.16907516382852 -0.16908760523461)
		))))
	

	;;; ----------------
	;;; some of these tests are taken (with modifications) from sacla which has 
	;;;  the following copyright notice:
	;;;
	;; Copyright (C) 2002-2004, Yuji Minejima <ggb01164@nifty.ne.jp>
	;; ALL RIGHTS RESERVED.
	;;
	;; Redistribution and use in source and binary forms, with or without
	;; modification, are permitted provided that the following conditions
	;; are met:
	;; 
	;;  * Redistributions of source code must retain the above copyright
	;;    notice, this list of conditions and the following disclaimer.
	;;  * Redistributions in binary form must reproduce the above copyright
	;;    notice, this list of conditions and the following disclaimer in
	;;    the documentation and/or other materials provided with the
	;;    distribution.
	;; 
	;; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
	;; "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
	;; LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
	;; A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
	;; OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
	;; SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
	;; LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
	;; DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
	;; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
	;; (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
	;; OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

	(test-t (equal 'a 'a))
	(test-t (not (equal 'a 'b)))
	(test-t (equal 'abc 'abc))
	(test-t (equal 1 1))
	(test-t (equal 2 2))
	(test-t (equal 0.1 0.1))
	(test-t (equal 1/3 1/3))
	(test-t (not (equal 0 1)))
	(test-t (not (equal 1 1.0)))
	(test-t (not (equal 1/3 1/4)))
	(test-t (equal #\a #\a))
	(test-t (equal #\b #\b))
	(test-t (not (equal #\b #\B)))
	(test-t (not (equal #\C #\c)))
	(test-t (equal '(0) '(0)))
	(test-t (equal '(0 #\a) '(0 #\a)))
	(test-t (equal '(0 #\a x) '(0 #\a x)))
	(test-t (equal '(0 #\a x (0)) '(0 #\a x (0))))

	(test-t (eql (identity 101) 101))
	(test-t (eq (identity 'x) 'x))

	;; chars
	(test-t (char= #\d #\d))
	(test-t (not (char= #\A #\a)))
	(test-t (not (char= #\d #\x)))
	(test-t (not (char= #\d #\D)))
	(test-t (not (char/= #\d #\d)))
	(test-t (char/= #\d #\x))
	(test-t (char/= #\d #\D))
	(test-t (char= #\d #\d #\d #\d))
	(test-t (not (char/= #\d #\d #\d #\d)))
	(test-t (not (char= #\d #\d #\x #\d)))
	(test-t (not (char/= #\d #\d #\x #\d)))
	(test-t (not (char= #\d #\y #\x #\c)))
	(test-t (char/= #\d #\y #\x #\c))
	(test-t (not (char= #\d #\c #\d)))
	(test-t (not (char/= #\d #\c #\d)))
	(test-t (char< #\d #\x))
	(test-t (char<= #\d #\x))
	(test-t (not (char< #\d #\d)))
	(test-t (char<= #\d #\d))
	(test-t (char< #\a #\e #\y #\z))
	(test-t (char<= #\a #\e #\y #\z))
	(test-t (not (char< #\a #\e #\e #\y)))
	(test-t (char<= #\a #\e #\e #\y))
	(test-t (char> #\e #\d))
	(test-t (char>= #\e #\d))
	(test-t (char> #\d #\c #\b #\a))
	(test-t (char>= #\d #\c #\b #\a))
	(test-t (not (char> #\d #\d #\c #\a)))
	(test-t (char>= #\d #\d #\c #\a))
	(test-t (not (char> #\e #\d #\b #\c #\a)))
	(test-t (not (char>= #\e #\d #\b #\c #\a)))
	(test-t (char-equal #\A #\a))
	(test-t (char= #\a))
	(test-t (char= #\a #\a))
	(test-t (char= #\a #\a #\a))
	(test-t (char= #\a #\a #\a #\a))
	(test-t (char= #\a #\a #\a #\a #\a))
	(test-t (char= #\a #\a #\a #\a #\a #\a))
	(test-t (let ((c #\z))  (and (eq c c)       (char= c c))))
	(test-t (not (char= #\Z #\z)))
	(test-t (not (char= #\z #\z #\z #\a)))
	(test-t (not (char= #\a #\z #\z #\z #\a)))
	(test-t (not (char= #\z #\i #\z #\z)))
	(test-t (not (char= #\z #\z #\Z #\z)))
	(test-t (char/= #\a))
	(test-t (char/= #\a #\b))
	(test-t (char/= #\a #\b #\c))
	(test-t (char/= #\a #\b #\c #\d))
	(test-t (char/= #\a #\b #\c #\d #\e))
	(test-t (char/= #\a #\b #\c #\d #\e #\f))
	(test-t (let ((c #\z))  (and (eq c c)       (not (char/= c c)))))
	(test-t (char/= #\Z #\z))
	(test-t (not (char/= #\z #\z #\z #\a)))
	(test-t (not (char= #\a #\z #\z #\z #\a)))
	(test-t (not (char= #\z #\i #\z #\z)))
	(test-t (not (char= #\z #\z #\Z #\z)))
	(test-t (not (char/= #\a #\a #\b #\c)))
	(test-t (not (char/= #\a #\b #\a #\c)))
	(test-t (not (char/= #\a #\b #\c #\a)))
	(test-t (char< #\a))
	(test-t (char< #\a #\z))
	(test-t (char< #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m       #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z))
	(test-t (not (char< #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n	    #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a)))
	(test-t (char< #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M       #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z))
	(test-t (not (char< #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N	    #\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A)))
	(test-t (char< #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9))
	(test-t (not (char< #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0)))
	(test-t (or (char< #\9 #\A)    (char< #\Z #\0)))
	(test-t (or (char< #\9 #\a)    (char< #\z #\0)))
	(test-t (not (char< #\a #\a #\b #\c)))
	(test-t (not (char< #\a #\b #\a #\c)))
	(test-t (not (char< #\a #\b #\c #\a)))
	(test-t (not (char< #\9 #\0)))
	(test-t (char> #\a))
	(test-t (not (char> #\a #\z)))
	(test-t (char> #\z #\a))
	(test-t (not (char> #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m	    #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z)))
	(test-t (char> #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n       #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a))
	(test-t (not (char> #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M	    #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z)))
	(test-t (char> #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N      #\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A))
	(test-t (not (char> #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
	(test-t (char> #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0))
	(test-t (or (char> #\A #\9)    (char> #\0 #\Z)))
	(test-t (or (char> #\a #\9)    (char> #\0 #\z)))
	(test-t (not (char> #\a #\a #\b #\c)))
	(test-t (not (char> #\a #\b #\a #\c)))
	(test-t (not (char> #\a #\b #\c #\a)))
	(test-t (char> #\9 #\0))
	(test-t (char<= #\a))
	(test-t (char<= #\a #\z))
	(test-t (char<= #\a #\a))
	(test-t (char<= #\Z #\Z))
	(test-t (char<= #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m	#\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z))
	(test-t (char<= #\a #\a #\b #\b #\c #\c #\d #\d #\e #\e #\f #\f #\g #\g #\h #\h 
			#\i #\i #\j #\j #\k #\k #\l #\l #\m #\m	#\n #\n #\o #\o #\p #\p #\q #\q #\r #\r #\s #\s	
			#\t #\t #\u #\u #\v #\v #\w #\w #\x #\x #\y #\y #\z #\z))
	(test-t (not (char<= #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n     #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a)))
	(test-t (char<= #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M	#\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z))
	(test-t (char<= #\A #\B #\B #\C #\D #\E #\E #\F #\G #\H #\I #\I #\J #\K #\L #\M	#\N #\N #\O #\P #\Q #\R #\S #\T #\T #\U #\V #\W #\X #\Y #\Z))
	(test-t (not (char<= #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N     #\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A)))
	(test-t (char<= #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9))
	(test-t (char<= #\0 #\1 #\2 #\2 #\3 #\3 #\3 #\3 #\4 #\5 #\6 #\7 #\8 #\9 #\9))
	(test-t (not (char<= #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0)))
	(test-t (or (char<= #\9 #\A)    (char<= #\Z #\0)))
	(test-t (or (char<= #\9 #\a)    (char<= #\z #\0)))
	(test-t (char<= #\a #\a #\b #\c))
	(test-t (not (char<= #\a #\b #\a #\c)))
	(test-t (not (char<= #\a #\b #\c #\a)))
	(test-t (not (char<= #\9 #\0)))
	(test-t (char>= #\a))
	(test-t (not (char>= #\a #\z)))
	(test-t (char>= #\z #\a))
	(test-t (char>= #\a #\a))
	(test-t (char>= #\Z #\Z))
	(test-t (not (char>= #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m     #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z)))
	(test-t (char>= #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n	#\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a))
	(test-t (char>= #\z #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n #\n	#\m #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a #\a))
	(test-t (not (char>= #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M     #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z)))
	(test-t (char>= #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N	#\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A))
	(test-t (char>= #\Z #\Y #\X #\W #\V #\U #\U #\T #\T #\S #\S #\R #\Q #\P #\O #\N	#\M #\L #\K #\J #\I #\H #\H #\G #\G #\F #\F #\E #\D #\C #\B #\A))
	(test-t (not (char>= #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
	(test-t (char>= #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0))
	(test-t (char>= #\9 #\8 #\8 #\8 #\7 #\6 #\5 #\4 #\3 #\3 #\3 #\2 #\1 #\0))
	(test-t (or (char>= #\A #\9)    (char>= #\0 #\Z)))
	(test-t (or (char>= #\a #\9)    (char>= #\0 #\z)))
	(test-t (char>= #\c #\b #\a #\a))
	(test-t (not (char>= #\c #\b #\a #\a #\b #\c)))
	(test-t (not (char>= #\c #\b #\a #\c)))
	(test-t (not (char>= #\c #\b #\c #\a)))
	(test-t (char>= #\9 #\0))
	(test-t (not (char>= #\0 #\9)))
	(test-t (char-equal #\a))
	(test-t (char-equal #\a #\a))
	(test-t (char-equal #\a #\a #\a))
	(test-t (char-equal #\a #\a #\a #\a))
	(test-t (char-equal #\a #\a #\a #\a #\a))
	(test-t (char-equal #\a #\a #\a #\a #\a #\a))
	(test-t (char-equal #\a #\A))
	(test-t (char-equal #\a #\A #\a))
	(test-t (char-equal #\a #\a #\A #\a))
	(test-t (char-equal #\a #\a #\a #\A #\a))
	(test-t (char-equal #\a #\a #\a #\a #\A #\a))
	(test-t (let ((c #\z))  (and (eq c c)       (char-equal c c))))
	(test-t (char-equal #\Z #\z))
	(test-t (not (char-equal #\z #\z #\z #\a)))
	(test-t (not (char-equal #\a #\z #\z #\z #\a)))
	(test-t (not (char-equal #\z #\i #\z #\z)))
	(test-t (char-equal #\z #\z #\Z #\z))
	(test-t (char-equal #\a #\A #\a #\A #\a #\A #\a #\A #\a #\A))
	(test-t (char-not-equal #\a))
	(test-t (char-not-equal #\a #\b))
	(test-t (char-not-equal #\a #\b #\c))
	(test-t (char-not-equal #\a #\b #\c #\d))
	(test-t (char-not-equal #\a #\b #\c #\d #\e))
	(test-t (char-not-equal #\a #\b #\c #\d #\e #\f))
	(test-t (let ((c #\z))  (and (eq c c)       (not (char-not-equal c c)))))
	(test-t (not (char-not-equal #\Z #\z)))
	(test-t (not (char-not-equal #\z #\z #\z #\a)))
	(test-t (not (char= #\a #\z #\z #\z #\a)))
	(test-t (not (char= #\z #\i #\z #\z)))
	(test-t (not (char= #\z #\z #\Z #\z)))
	(test-t (not (char-not-equal #\a #\a #\b #\c)))
	(test-t (not (char-not-equal #\a #\b #\a #\c)))
	(test-t (not (char-not-equal #\a #\b #\c #\a)))
	(test-t (not (char-not-equal #\a #\A #\a #\A)))
	(test-t (char-lessp #\a))
	(test-t (char-lessp #\a #\z))
	(test-t (char-lessp #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m       #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z))
	(test-t (not (char-lessp #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n	    #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a)))
	(test-t (char-lessp #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M       #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z))
	(test-t (not (char-lessp #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N	    #\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A)))
	(test-t (char-lessp #\a #\B #\c #\D #\e #\F #\g #\H #\i #\J #\k #\L #\m       #\N #\o #\P #\q #\R #\s #\T #\u #\V #\w #\X #\y #\Z))
	(test-t (char-lessp #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9))
	(test-t (not (char-lessp #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0)))
	(test-t (or (char-lessp #\9 #\A)    (char-lessp #\Z #\0)))
	(test-t (or (char-lessp #\9 #\a)    (char-lessp #\z #\0)))
	(test-t (not (char-lessp #\a #\a #\b #\c)))
	(test-t (not (char-lessp #\a #\b #\a #\c)))
	(test-t (not (char-lessp #\a #\b #\c #\a)))
	(test-t (not (char-lessp #\9 #\0)))
	(test-t (and (char-lessp #\a #\Z)     (char-lessp #\A #\z)))
	(test-t (char-greaterp #\a))
	(test-t (not (char-greaterp #\a #\z)))
	(test-t (char-greaterp #\z #\a))
	(test-t (not (char-greaterp #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m	    #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z)))
	(test-t (char-greaterp #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n       #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a))
	(test-t (not (char-greaterp #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M	    #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z)))
	(test-t (char-greaterp #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N   #\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A))
	(test-t (char-greaterp #\z #\Y #\x #\W #\v #\U #\t #\S #\r #\Q #\p #\O #\n   #\M #\l #\K #\j #\I #\h #\G #\f #\E #\d #\C #\b #\A))
	(test-t (not (char-greaterp #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
	(test-t (char-greaterp #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0))
	(test-t (or (char-greaterp #\A #\9)    (char-greaterp #\0 #\Z)))
	(test-t (or (char-greaterp #\a #\9)    (char-greaterp #\0 #\z)))
	(test-t (not (char-greaterp #\a #\a #\b #\c)))
	(test-t (not (char-greaterp #\a #\b #\a #\c)))
	(test-t (not (char-greaterp #\a #\b #\c #\a)))
	(test-t (char-greaterp #\9 #\0))
	(test-t (and (char-greaterp #\z #\A)     (char-greaterp #\Z #\a)))
	(test-t (char-not-greaterp #\a))
	(test-t (char-not-greaterp #\a #\z))
	(test-t (char-not-greaterp #\a #\a))
	(test-t (char-not-greaterp #\Z #\Z))
	(test-t (char-not-greaterp #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z))
	(test-t (char-not-greaterp #\a #\a #\b #\b #\c #\c #\d #\d #\e #\e #\f #\f #\g #\g #\h #\h #\i #\i 
				   #\j #\j #\k #\k #\l #\l #\m #\m #\n #\n #\o #\o #\p #\p #\q #\q #\r #\r #\s #\s #\t 
				   #\t #\u #\u #\v #\v #\w #\w #\x #\x #\y #\y #\z #\z))
	(test-t (char-not-greaterp #\a #\A #\b #\B #\c #\C #\d #\D #\e #\E #\f #\F #\g #\G #\h #\H #\i #\I #\j #\J 
				   #\k #\K #\l #\L #\m #\M #\n #\N #\o #\O #\p #\P #\q #\Q #\r #\R #\s #\S #\t #\T 
				   #\u #\U #\v #\V #\w #\W #\x #\X #\y #\Y #\z #\z))
	(test-t (not (char-not-greaterp      #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n      #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a)))
	(test-t (char-not-greaterp #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z))
	(test-t (char-not-greaterp #\A #\B #\B #\C #\D #\E #\E #\F #\G #\H #\I #\I #\J #\K #\L #\M #\N #\N #\O #\P #\Q #\R #\S #\T #\T #\U #\V #\W #\X #\Y #\Z))
	(test-t (not (char-not-greaterp      #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N      #\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A)))
	(test-t (char-not-greaterp #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9))
	(test-t (char-not-greaterp #\0 #\1 #\2 #\2 #\3 #\3 #\3 #\3 #\4 #\5 #\6 #\7 #\8 #\9 #\9))
	(test-t (not (char-not-greaterp #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0)))
	(test-t (or (char-not-greaterp #\9 #\A)    (char-not-greaterp #\Z #\0)))
	(test-t (or (char-not-greaterp #\9 #\a)    (char-not-greaterp #\z #\0)))
	(test-t (char-not-greaterp #\a #\a #\b #\c))
	(test-t (not (char-not-greaterp #\a #\b #\a #\c)))
	(test-t (not (char-not-greaterp #\a #\b #\c #\a)))
	(test-t (not (char-not-greaterp #\9 #\0)))
	(test-t (and (char-not-greaterp #\A #\z)     (char-not-greaterp #\a #\Z)))
	(test-t (char-not-lessp #\a))
	(test-t (not (char-not-lessp #\a #\z)))
	(test-t (char-not-lessp #\z #\a))
	(test-t (char-not-lessp #\a #\a))
	(test-t (char-not-lessp #\Z #\Z))
	(test-t (not (char-not-lessp #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m  #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z)))
	(test-t (char-not-lessp #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n	#\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a))
	(test-t (char-not-lessp #\z #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n #\n	#\m #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a #\a))
	(test-t (not (char-not-lessp #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\m     #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z)))
	(test-t (char-not-lessp #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N	#\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A))
	(test-t (char-not-lessp #\Z #\Y #\X #\W #\V #\U #\U #\T #\T #\S #\S #\R #\Q #\P #\O #\N	#\M #\L #\K #\J #\I #\H #\H #\G #\G #\F #\F #\E #\D #\C #\B #\A))
	(test-t (char-not-lessp #\z #\Z #\y #\x #\w #\V #\v #\u #\t #\s #\r #\q #\p #\o #\n #\n	#\m #\M #\l #\k #\K #\j #\i #\h #\g #\f #\e #\d #\c #\b #\A #\a))
	(test-t (not (char-not-lessp #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
	(test-t (char-not-lessp #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0))
	(test-t (char-not-lessp #\9 #\8 #\8 #\8 #\7 #\6 #\5 #\4 #\3 #\3 #\3 #\2 #\1 #\0))
	(test-t (or (char-not-lessp #\A #\9)    (char-not-lessp #\0 #\Z)))
	(test-t (or (char-not-lessp #\a #\9)    (char-not-lessp #\0 #\z)))
	(test-t (char-not-lessp #\c #\b #\a #\a))
	(test-t (not (char-not-lessp #\c #\b #\a #\a #\b #\c)))
	(test-t (not (char-not-lessp #\c #\b #\a #\c)))
	(test-t (not (char-not-lessp #\c #\b #\c #\a)))
	(test-t (char-not-lessp #\9 #\0))
	(test-t (not (char-not-lessp #\0 #\9)))
	(test-t (and (char-not-lessp #\z #\A)     (char-not-lessp #\Z #\a)))
	(test-t (char= (character #\a) #\a))
	(test-t (char= (character #\b) #\b))
;	(test-t (char= (character #\Space) #\Space))
	(test-t (char= (character "a") #\a))
	(test-t (char= (character "X") #\X))
	(test-t (char= (character "z") #\z))
	(test-t (char= (character 'a) #\a))
;	(test-t (char= (character '\a) #\a))
	(test-t (alpha-char-p #\a))
	(test-t (every alpha-char-p '(#\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z)))
	(test-t (every alpha-char-p '(#\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z)))
	(test-t (notany alpha-char-p '(#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
;	(test-t (not (alpha-char-p #\Newline)))
	(test-t (alphanumericp #\Z))
	(test-t (alphanumericp #\9))
	(test-t (every alphanumericp '(#\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m  #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z)))
	(test-t (every alphanumericp '(#\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M  #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z)))
	(test-t (every alphanumericp '(#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
;	(test-t (not (alphanumericp #\Newline)))
	(test-t (not (alphanumericp #\#)))
	(test-t (char= (digit-char 0) #\0))
;	(test-t (char= (digit-char 10 11) #\A))
	(test-t (null (digit-char 10 10)))
	(test-t (char= (digit-char 7) #\7))
	(test-t (null (digit-char 12)))
;	(test-t (char= (digit-char 12 16) #\C))
	(test-t (null (digit-char 6 2))) 
	(test-t (char= (digit-char 1 2) #\1))
;	(test-t (char= (digit-char 35 36) #\Z))
	(test-t (= (digit-char-p #\0) 0))
	(test-t (= (digit-char-p #\5) 5))
	(test-t (not (digit-char-p #\5 2)))
	(test-t (not (digit-char-p #\A)))
	(test-t (not (digit-char-p #\a)))
;	(test-t (= (digit-char-p #\A 11) 10))
	(test-t (= (digit-char-p #\a 11) 10))
;	(test-t (standard-char-p #\a))
;	(test-t (standard-char-p #\z))
;	(test-t (standard-char-p #\Newline))
;	(test-t (every standard-char-p " !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_'abcdefghijklmnopqrstuvwxyz{|}~"))
	(test-t (char= (char-upcase #\a) #\A))
	(test-t (char= (char-upcase #\A) #\A))
	(test-t (char= (char-upcase #\-) #\-))
	(test-t (char= (char-downcase #\A) #\a))
	(test-t (char= (char-downcase #\a) #\a))
	(test-t (char= (char-downcase #\-) #\-))
	(test-t (not (upper-case-p #\a)))
	(test-t (upper-case-p #\A))
	(test-t (not (upper-case-p #\-)))
	(test-t (not (lower-case-p #\A)))
	(test-t (lower-case-p #\a))
	(test-t (not (lower-case-p #\-)))
;	(test-t (char= #\ (name-char (char-name #\ ))))
;	(test-t (char= #\Space (name-char (char-name #\Space))))
;	(test-t (char= #\Newline (name-char (char-name #\Newline))))


	(test-t (simple-string-p ""))
	(test-t (simple-string-p "abc"))
	(test-t (not (simple-string-p 'not-a-string)))
	(test-t (char= (char "abc" 0) #\a))
	(test-t (char= (char "abc" 1) #\b))
	(test-t (char= (char "abc" 2) #\c))
	(test-t (char= (schar "abc" 0) #\a))
	(test-t (char= (schar "abc" 1) #\b))
	(test-t (char= (schar "abc" 2) #\c))
	(test-t (string= (cl-string "") ""))
	(test-t (string= (cl-string "abc") "abc"))
	(test-t (string= (cl-string "a") "a"))
	(test-t (string= (cl-string 'abc) "abc"))
	(test-t (string= (cl-string 'a) "a"))
	(test-t (string= (cl-string #\a) "a"))
	(test-t (string= (string-upcase "abcde") "ABCDE"))
	(test-t (string= (string-upcase "Dr. Livingston, I presume?")	 "DR. LIVINGSTON, I PRESUME?"))
	(test-t (string= (string-upcase "Dr. Livingston, I presume?" :start 6 :end 10)	 "Dr. LiVINGston, I presume?"))
	(test-t (string= (string-upcase 'Kludgy-HASH-Search) "KLUDGY-HASH-SEARCH"))
	(test-t (string= (string-upcase "abcde" :start 2 :end nil) "abCDE"))
	(test-t (string= (string-downcase "Dr. Livingston, I presume?")	 "dr. livingston, i presume?"))
	(test-t (string= (string-downcase 'Kludgy-HASH-Search) "kludgy-hash-search"))
	(test-t (string= (string-downcase "A FOOL" :start 2 :end nil) "A fool"))
	(test-t (string= (string-capitalize "elm 13c arthur;fig don't")	 "Elm 13c Arthur;Fig Don'T"))
	(test-t (string= (string-capitalize " hello ") " Hello "))
	(test-t (string= (string-capitalize  "occlUDeD cASEmenTs FOreSTAll iNADVertent DEFenestraTION") "Occluded Casements Forestall Inadvertent Defenestration"))
	(test-t (string= (string-capitalize 'kludgy-hash-search) "Kludgy-Hash-Search"))
	(test-t (string= (string-capitalize "DON'T!") "Don'T!"))
	(test-t (string= (string-capitalize "pipe 13a, foo16c") "Pipe 13a, Foo16c"))
	(test-t (string= (string-capitalize "a fool" :start 2 :end nil) "a Fool"))
	(test-t (let ((str (copy-seq "0123ABCD890a")))  (and (string= (nstring-downcase str :start 5 :end 7) "0123AbcD890a")       (string= str "0123AbcD890a"))))
	(test-t (let* ((str0 (copy-seq "abcde"))  (str  (nstring-upcase str0)))  (and (eq str0 str)       (string= str "ABCDE"))))
	(test-t (let* ((str0 (copy-seq "Dr. Livingston, I presume?")) (str  (nstring-upcase str0))) (and (eq str0 str) (string= str "DR. LIVINGSTON, I PRESUME?"))))
	(test-t (let* ((str0 (copy-seq "Dr. Livingston, I presume?")) (str  (nstring-upcase str0 :start 6 :end 10))) (and (eq str0 str) (string= str "Dr. LiVINGston, I presume?"))))
	(test-t (let* ((str0 (copy-seq "abcde")) (str (nstring-upcase str0 :start 2 :end nil)))  (string= str "abCDE")))
	(test-t (let* ((str0 (copy-seq "Dr. Livingston, I presume?")) (str  (nstring-downcase str0))) (and (eq str0 str) (string= str "dr. livingston, i presume?"))))
	(test-t (let* ((str0 (copy-seq "ABCDE")) (str (nstring-downcase str0 :start 2 :end nil)))  (string= str "ABcde")))
	(test-t (let* ((str0 (copy-seq "elm 13c arthur;fig don't")) (str  (nstring-capitalize str0))) (and (eq str0 str) (string= str "Elm 13c Arthur;Fig Don'T"))))
	(test-t (let* ((str0 (copy-seq " hello ")) (str  (nstring-capitalize str0)))  (and (eq str0 str) (string= str " Hello "))))
	(test-t (let* ((str0 (copy-seq "occlUDeD cASEmenTs FOreSTAll iNADVertent DEFenestraTION")) (str (nstring-capitalize str0))) (and (eq str0 str) (string= str		"Occluded Casements Forestall Inadvertent Defenestration"))))
	(test-t (let* ((str0 (copy-seq "DON'T!"))  (str  (nstring-capitalize str0))) (and (eq str0 str)  (string= str "Don'T!"))))
	(test-t (let* ((str0 (copy-seq "pipe 13a, foo16c"))       (str  (nstring-capitalize str0)))  (and (eq str0 str)       (string= str "Pipe 13a, Foo16c"))))
	(test-t (let* ((str0 (copy-seq "a fool"))       (str (nstring-capitalize str0 :start 2 :end nil)))  (string= str "a Fool")))
	(test-t (string= (string-trim "abc" "abcaakaaakabcaaa") "kaaak"))
	(test-t (string= (string-trim '(#\space) " garbanzo beans        ") "garbanzo beans"))
	(test-t (string= (string-trim " (*)" " ( *three (silly) words* ) ")	 "three (silly) words"))
	(test-t (string= (string-left-trim "abc" "labcabcabc") "labcabcabc"))
	(test-t (string= (string-left-trim " (*)" " ( *three (silly) words* ) ")	 "three (silly) words* ) "))
	(test-t (string= (string-right-trim " (*)" " ( *three (silly) words* ) ") 	 " ( *three (silly) words"))
	(test-t (string= (string-trim "ABC" "abc") "abc"))
	(test-t (string= (string-trim "AABBCC" "abc") "abc"))
	(test-t (string= (string-trim "" "abc") "abc"))
	(test-t (string= (string-trim "ABC" "") ""))
	(test-t (string= (string-trim "cba" "abc") ""))
	(test-t (string= (string-trim "cba" "abccba") ""))
	(test-t (string= (string-trim "ccbbba" "abccba") ""))
	(test-t (string= (string-trim "cba" "abcxabc") "x"))
	(test-t (string= (string-trim "xyz" "xxyabcxyyz") "abc"))
	(test-t (string= (string-trim "a" #\a) ""))
	(test-t (string= (string-left-trim "ABC" "abc") "abc"))
	(test-t (string= (string-left-trim "" "abc") "abc"))
	(test-t (string= (string-left-trim "ABC" "") ""))
	(test-t (string= (string-left-trim "cba" "abc") ""))
	(test-t (string= (string-left-trim "cba" "abccba") ""))
	(test-t (string= (string-left-trim "cba" "abcxabc") "xabc"))
	(test-t (string= (string-left-trim "xyz" "xxyabcxyz") "abcxyz"))
	(test-t (string= (string-left-trim "a" #\a) ""))
	(test-t (string= (string-right-trim "ABC" "abc") "abc"))
	(test-t (string= (string-right-trim "" "abc") "abc"))
	(test-t (string= (string-right-trim "ABC" "") ""))
	(test-t (string= (string-right-trim "cba" "abc") ""))
	(test-t (string= (string-right-trim "cba" "abccba") ""))
	(test-t (string= (string-right-trim "cba" "abcxabc") "abcx"))
	(test-t (string= (string-right-trim "xyz" "xxyabcxyz") "xxyabc"))
	(test-t (string= (string-right-trim "a" #\a) ""))
	(test-t (string= (cl-string "already a string") "already a string"))
	(test-t (string=  (cl-string #\c) "c"))
	(test-t (string= "foo" "foo"))
	(test-t (not (string= "foo" "Foo")))
	(test-t (not (string= "foo" "bar")))
	(test-t (string= "together" "frog" :start1 1 :end1 3 :start2 2))
	(test-t (string-equal "foo" "Foo"))
	(test-t (string= "abcd" "01234abcd9012" :start2 5 :end2 9))
	(test-t (eql (string< "aaaa" "aaab") 3))
	(test-t (eql (string>= "aaaaa" "aaaa") 4))
	(test-t (eql (string-not-greaterp "Abcde" "abcdE") 5))
	(test-t (eql (string-lessp "012AAAA789" "01aaab6"   :start1 3 :end1 7   :start2 2 :end2 6) 6))
	(test-t (not (string-not-equal "AAAA" "aaaA")))
	(test-t (string= "" ""))
	(test-t (not (string= "abc" "")))
	(test-t (not (string= "" "abc")))
	(test-t (not (string= "A" "a")))
	(test-t (string= "abc" "xyz" :start1 3 :start2 3))
	(test-t (string= "abc" "xyz" :start1 1 :end1 1 :start2 0 :end2 0))
	(test-t (string= "axyza" "xyz" :start1 1 :end1 4))
	(test-t (string= "axyza" "xyz" :start1 1 :end1 4 :start2 0 :end2 nil))
	(test-t (string= "abxyz" "xyabz" :end1 2 :start2 2 :end2 4))
	(test-t (not (string= "love" "hate")))
	(test-t (string= 'love 'love))
	(test-t (not (string= 'love "hate")))
	(test-t (string= #\a #\a))
	(test-t (not (string/= "" "")))
	(test-t (eql (string/= "abc" "") 0))
	(test-t (eql (string/= "" "abc") 0))
	(test-t (eql (string/= "A" "a") 0))
	(test-t (not (string/= "abc" "xyz" :start1 3 :start2 3)))
	(test-t (not (string/= "abc" "xyz" :start1 1 :end1 1 :start2 0 :end2 0)))
	(test-t (not (string/= "axyza" "xyz" :start1 1 :end1 4)))
	(test-t (not (string/= "axyza" "xyz" :start1 1 :end1 4 :start2 0 :end2 nil)))
	(test-t (not (string/= "abxyz" "xyabz" :end1 2 :start2 2 :end2 4)))
	(test-t (eql (string/= "love" "hate") 0))
	(test-t (eql (string/= "love" "loVe") 2))
	(test-t (not (string/= "life" "death" :start1 3 :start2 1 :end2 2)))
	(test-t (eql (string/= "abcxyz" "ABCxyZ" :start1 3 :start2 3) 5))
	(test-t (eql (string/= "abcxyz" "ABCxyZ" :start1 3 :end1 nil :start2 3 :end2 nil) 5))
	(test-t (eql (string/= "abcxyz" "ABCxyZ" :end1 nil :start2 3 :end2 3) 0))
	(test-t (eql (string/= "abc" "abcxyz") 3))
	(test-t (eql (string/= "abcxyz" "abc") 3))
	(test-t (eql (string/= "abcxyz" "") 0))
	(test-t (eql (string/= "AbcDef" "cdef" :start1 2) 3))
	(test-t (eql (string/= "cdef" "AbcDef" :start2 2) 1))
	(test-t (= (string/= 'love "hate") 0))
	(test-t (not (string/= 'love 'love)))
	(test-t (not (string/= #\a #\a)))
	(test-t (= (string/= #\a #\b) 0))
	(test-t (not (string< "" "")))
	(test-t (not (string< "dog" "dog")))
	(test-t (not (string< " " " ")))
	(test-t (not (string< "abc" "")))
	(test-t (eql (string< "" "abc") 0))
	(test-t (eql (string< "ab" "abc") 2))
	(test-t (not (string< "abc" "ab")))
	(test-t (eql (string< "aaa" "aba") 1))
	(test-t (not (string< "aba" "aaa")))
	(test-t (not (string< "my cat food" "your dog food" :start1 6 :start2 8)))
	(test-t (not (string< "cat food 2 dollars" "dog food 3 dollars"	      :start1 3 :end1 9 :start2 3 :end2 9)))
	(test-t (eql (string< "xyzabc" "abcd" :start1 3) 6))
	(test-t (eql (string< "abc" "abc" :end1 1) 1))
	(test-t (eql (string< "xyzabc" "abc" :start1 3 :end1 5) 5))
	(test-t (eql (string< "xyz" "abcxyzXYZ" :start2 3) 3))
	(test-t (not (string< "abc" "abcxyz" :end2 3)))
	(test-t (eql (string< "xyz" "abcxyz" :end1 2 :start2 3) 2))
	(test-t (not (string< "xyzabc" "abcdef" :start1 3 :end2 3)))
	(test-t (eql (string< "aaaa" "z") 0))
	(test-t (eql (string< "pppTTTaTTTqqq" "pTTTxTTT" :start1 3 :start2 1) 6))
	(test-t (eql (string< "pppTTTaTTTqqq" "pTTTxTTT"     :start1 6 :end1 7     :start2 4 :end2 5) 6))
	(test-t (not (string< 'love 'hate)))
	(test-t (= (string< 'peace 'war) 0))
	(test-t (not (string< 'love 'love)))
	(test-t (not (string< #\a #\a)))
	(test-t (= (string< #\a #\b) 0))
	(test-t (not (string> "" "")))
	(test-t (not (string> "dog" "dog")))
	(test-t (not (string> " " " ")))
	(test-t (eql (string> "abc" "") 0))
	(test-t (not (string> "" "abc")))
	(test-t (not (string> "ab" "abc")))
	(test-t (eql (string> "abc" "ab") 2))
	(test-t (eql (string> "aba" "aaa") 1))
	(test-t (not (string> "aaa" "aba")))
	(test-t (not (string> "my cat food" "your dog food" :start1 6 :start2 8)))
	(test-t (not (string> "cat food 2 dollars" "dog food 3 dollars"	     :start1 3 :end1 9 :start2 3 :end2 9)))
	(test-t (eql (string> "xyzabcde" "abcd" :start1 3) 7))
	(test-t (not (string> "abc" "abc" :end1 1)))
	(test-t (eql (string> "xyzabc" "a" :start1 3 :end1 5) 4))
	(test-t (eql (string> "xyzXYZ" "abcxyz" :start2 3) 3))
	(test-t (eql (string> "abcxyz" "abcxyz" :end2 3) 3))
	(test-t (not (string> "xyzXYZ" "abcxyz" :end1 2 :start2 3)))
	(test-t (not (string> "xyzabc" "abcdef" :start1 3 :end2 3)))
	(test-t (eql (string> "z" "aaaa") 0))
	(test-t (eql (string> "pTTTxTTTqqq" "pppTTTaTTT" :start1 1 :start2 3) 4))
	(test-t (eql (string> "pppTTTxTTTqqq" "pTTTaTTT"     :start1 6 :end1 7      :start2 4 :end2 5) 6))
	(test-t (= (string> 'love 'hate) 0))
	(test-t (not (string> 'peace 'war)))
	(test-t (not (string> 'love 'love)))
	(test-t (not (string> #\a #\a)))
	(test-t (not (string> #\a #\b)))
	(test-t (= (string> #\z #\a) 0))
	(test-t (eql (string<= "" "") 0))
	(test-t (eql (string<= "dog" "dog") 3))
	(test-t (eql (string<= " " " ") 1))
	(test-t (not (string<= "abc" "")))
	(test-t (eql (string<= "ab" "abc") 2))
	(test-t (eql (string<= "aaa" "aba") 1))
	(test-t (not (string<= "aba" "aaa")))
	(test-t (eql (string<= "my cat food" "your dog food" :start1 6 :start2 8) 11))
	(test-t (eql (string<= "cat food 2 dollars" "dog food 3 dollars"      :start1 3 :end1 9 :start2 3 :end2 9) 9))
	(test-t (eql (string<= "xyzabc" "abcd" :start1 3) 6))
	(test-t (eql (string<= "abc" "abc" :end1 1) 1))
	(test-t (eql (string<= "xyzabc" "abc" :start1 3 :end1 5) 5))
	(test-t (eql (string<= "xyz" "abcxyzXYZ" :start2 3) 3))
	(test-t (eql (string<= "abc" "abcxyz" :end2 3) 3))
	(test-t (eql (string<= "xyz" "abcxyz" :end1 2 :start2 3) 2))
	(test-t (eql (string<= "xyzabc" "abcdef" :start1 3 :end2 3) 6))
	(test-t (eql (string<= "aaaa" "z") 0))
	(test-t (eql (string<= "pppTTTaTTTqqq" "pTTTxTTT" :start1 3 :start2 1) 6))
	(test-t (eql (string<= "pppTTTaTTTqqq" "pTTTxTTT"      :start1 6 :end1 7     :start2 4 :end2 5) 6))
	(test-t (not (string<= 'love 'hate)))
	(test-t (= (string<= 'peace 'war) 0))
	(test-t (= (string<= 'love 'love) 4))
	(test-t (= (string<= #\a #\a) 1))
	(test-t (= (string<= #\a #\b) 0))
	(test-t (not (string<= #\z #\a)))
	(test-t (eql (string>= "" "") 0))
	(test-t (eql (string>= "dog" "dog") 3))
	(test-t (eql (string>= " " " ") 1))
	(test-t (eql (string>= "abc" "") 0))
	(test-t (not (string>= "" "abc")))
	(test-t (not (string>= "ab" "abc")))
	(test-t (eql (string>= "abc" "ab") 2))
	(test-t (eql (string>= "aba" "aaa") 1))
	(test-t (not (string>= "aaa" "aba")))
	(test-t (eql (string>= "my cat food" "your dog food" :start1 6 :start2 8) 11))
	(test-t (eql (string>= "cat food 2 dollars" "dog food 3 dollars"      :start1 3 :end1 9 :start2 3 :end2 9) 9))
	(test-t (eql (string>= "xyzabcde" "abcd" :start1 3) 7))
	(test-t (not (string>= "abc" "abc" :end1 1)))
	(test-t (eql (string>= "xyzabc" "a" :start1 3 :end1 5) 4))
	(test-t (eql (string>= "xyzXYZ" "abcxyz" :start2 3) 3))
	(test-t (eql (string>= "abcxyz" "abcxyz" :end2 3) 3))
	(test-t (not (string>= "xyzXYZ" "abcxyz" :end1 2 :start2 3)))
	(test-t (eql (string>= "xyzabc" "abcdef" :start1 3 :end2 3) 6))
	(test-t (eql (string>= "z" "aaaa") 0))
	(test-t (eql (string>= "pTTTxTTTqqq" "pppTTTaTTT" :start1 1 :start2 3) 4))
	(test-t (eql (string>= "pppTTTxTTTqqq" "pTTTaTTT"     :start1 6 :end1 7      :start2 4 :end2 5) 6))
	(test-t (= (string>= 'love 'hate) 0))
	(test-t (not (string>= 'peace 'war)))
	(test-t (= (string>= 'love 'love) 4))
	(test-t (= (string>= #\a #\a) 1))
	(test-t (not (string>= #\a #\b)))
	(test-t (= (string>= #\z #\a) 0))
	(test-t (string-equal "" ""))
	(test-t (not (string-equal "abc" "")))
	(test-t (not (string-equal "" "abc")))
	(test-t (string-equal "A" "a"))
	(test-t (string-equal "abc" "xyz" :start1 3 :start2 3))
	(test-t (string-equal "abc" "xyz" :start1 1 :end1 1 :start2 0 :end2 0))
	(test-t (string-equal "axyza" "xyz" :start1 1 :end1 4))
	(test-t (string-equal "axyza" "xyz" :start1 1 :end1 4 :start2 0 :end2 nil))
	(test-t (string-equal "abxyz" "xyabz" :end1 2 :start2 2 :end2 4))
	(test-t (not (string-equal "love" "hate")))
	(test-t (string-equal "xyz" "XYZ"))
	(test-t (not (string-equal 'love 'hate)))
	(test-t (not (string-equal 'peace 'war)))
	(test-t (string-equal 'love 'love))
	(test-t (string-equal #\a #\a))
	(test-t (not (string-equal #\a #\b)))
	(test-t (not (string-equal #\z #\a)))
	(test-t (not (string-not-equal "" "")))
	(test-t (eql (string-not-equal "abc" "") 0))
	(test-t (eql (string-not-equal "" "abc") 0))
	(test-t (not (string-not-equal "A" "a")))
	(test-t (not (string-not-equal "abc" "xyz" :start1 3 :start2 3)))
	(test-t (not (string-not-equal "abc" "xyz" :start1 1 :end1 1 :start2 0 :end2 0)))
	(test-t (not (string-not-equal "axyza" "xyz" :start1 1 :end1 4)))
	(test-t (not (string-not-equal "axyza" "xyz" :start1 1 :end1 4 :start2 0 :end2 nil)))
	(test-t (not (string-not-equal "abxyz" "xyabz" :end1 2 :start2 2 :end2 4)))
	(test-t (eql (string-not-equal "love" "hate") 0))
	(test-t (not (string-not-equal "love" "loVe")))
	(test-t (not (string-not-equal "life" "death" :start1 3 :start2 1 :end2 2)))
	(test-t (not (string-not-equal "abcxyz" "ABCxyZ" :start1 3 :start2 3)))
	(test-t (not (string-not-equal "abcxyz" "ABCxyZ" :start1 3 :end1 nil :start2 3 :end2 nil)))
	(test-t (eql (string-not-equal "abcxyz" "ABCxyZ" :end1 nil :start2 3 :end2 3) 0))
	(test-t (eql (string-not-equal "abc" "abcxyz") 3))
	(test-t (eql (string-not-equal "abcxyz" "abc") 3))
	(test-t (eql (string-not-equal "abcxyz" "") 0))
	(test-t (not (string-not-equal "AbcDef" "cdef" :start1 2)))
	(test-t (not (string-not-equal "cdef" "AbcDef" :start2 2)))
	(test-t (not (string-not-equal "ABC" "abc")))
	(test-t (= (string-not-equal 'love 'hate) 0))
	(test-t (= (string-not-equal 'peace 'war) 0))
	(test-t (not (string-not-equal 'love 'love)))
	(test-t (not (string-not-equal #\a #\a)))
	(test-t (= (string-not-equal #\a #\b) 0))
	(test-t (= (string-not-equal #\z #\a) 0))
	(test-t (not (string-lessp "" "")))
	(test-t (not (string-lessp "dog" "dog")))
	(test-t (not (string-lessp " " " ")))
	(test-t (not (string-lessp "abc" "")))
	(test-t (eql (string-lessp "" "abc") 0))
	(test-t (eql (string-lessp "ab" "abc") 2))
	(test-t (not (string-lessp "abc" "ab")))
	(test-t (eql (string-lessp "aaa" "aba") 1))
	(test-t (not (string-lessp "aba" "aaa")))
	(test-t (not (string-lessp "my cat food" "your dog food" :start1 6 :start2 8)))
	(test-t (not (string-lessp "cat food 2 dollars" "dog food 3 dollars"   :start1 3 :end1 9 :start2 3 :end2 9)))
	(test-t (eql (string-lessp "xyzabc" "abcd" :start1 3) 6))
	(test-t (eql (string-lessp "abc" "abc" :end1 1) 1))
	(test-t (eql (string-lessp "xyzabc" "abc" :start1 3 :end1 5) 5))
	(test-t (eql (string-lessp "xyz" "abcxyzXYZ" :start2 3) 3))
	(test-t (not (string-lessp "abc" "abcxyz" :end2 3)))
	(test-t (eql (string-lessp "xyz" "abcxyz" :end1 2 :start2 3) 2))
	(test-t (not (string-lessp "xyzabc" "abcdef" :start1 3 :end2 3)))
	(test-t (eql (string-lessp "aaaa" "z") 0))
	(test-t (eql (string-lessp "pppTTTaTTTqqq" "pTTTxTTT" :start1 3 :start2 1) 6))
	(test-t (eql (string-lessp "pppTTTaTTTqqq" "pTTTxTTT"   :start1 6 :end1 7   :start2 4 :end2 5) 6))
	(test-t (and (not (string-lessp "abc" "ABC"))     (not (string-lessp "ABC" "abc"))))
	(test-t (not (string-lessp 'love 'hate)))
	(test-t (= (string-lessp 'peace 'war) 0))
	(test-t (not (string-lessp 'love 'love)))
	(test-t (not (string-lessp #\a #\a)))
	(test-t (= (string-lessp #\a #\b) 0))
	(test-t (not (string-lessp #\z #\a)))
	(test-t (not (string-greaterp "" "")))
	(test-t (not (string-greaterp "dog" "dog")))
	(test-t (not (string-greaterp " " " ")))
	(test-t (eql (string-greaterp "abc" "") 0))
	(test-t (not (string-greaterp "" "abc")))
	(test-t (not (string-greaterp "ab" "abc")))
	(test-t (eql (string-greaterp "abc" "ab") 2))
	(test-t (eql (string-greaterp "aba" "aaa") 1))
	(test-t (not (string-greaterp "aaa" "aba")))
	(test-t (not (string-greaterp "my cat food" "your dog food" :start1 6 :start2 8)))
	(test-t (not (string-greaterp "cat food 2 dollars" "dog food 3 dollars"      :start1 3 :end1 9 :start2 3 :end2 9)))
	(test-t (eql (string-greaterp "xyzabcde" "abcd" :start1 3) 7))
	(test-t (not (string-greaterp "abc" "abc" :end1 1)))
	(test-t (eql (string-greaterp "xyzabc" "a" :start1 3 :end1 5) 4))
	(test-t (eql (string-greaterp "xyzXYZ" "abcxyz" :start2 3) 3))
	(test-t (eql (string-greaterp "abcxyz" "abcxyz" :end2 3) 3))
	(test-t (not (string-greaterp "xyzXYZ" "abcxyz" :end1 2 :start2 3)))
	(test-t (not (string-greaterp "xyzabc" "abcdef" :start1 3 :end2 3)))
	(test-t (eql (string-greaterp "z" "aaaa") 0))
	(test-t (eql (string-greaterp "pTTTxTTTqqq" "pppTTTaTTT" :start1 1 :start2 3) 4))
	(test-t (eql (string-greaterp "pppTTTxTTTqqq" "pTTTaTTT"     :start1 6 :end1 7	     :start2 4 :end2 5) 6))
	(test-t (and (not (string-greaterp "abc" "ABC"))     (not (string-greaterp "ABC" "abc"))))
	(test-t (= (string-greaterp 'love 'hate) 0))
	(test-t (not (string-greaterp 'peace 'war)))
	(test-t (not (string-greaterp 'love 'love)))
	(test-t (not (string-greaterp #\a #\a)))
	(test-t (not (string-greaterp #\a #\b)))
	(test-t (= (string-greaterp #\z #\a) 0))
	(test-t (eql (string-not-greaterp "" "") 0))
	(test-t (eql (string-not-greaterp "dog" "dog") 3))
	(test-t (eql (string-not-greaterp " " " ") 1))
	(test-t (not (string-not-greaterp "abc" "")))
	(test-t (eql (string-not-greaterp "ab" "abc") 2))
	(test-t (eql (string-not-greaterp "aaa" "aba") 1))
	(test-t (not (string-not-greaterp "aba" "aaa")))
	(test-t (eql (string-not-greaterp "my cat food" "your dog food" :start1 6 :start2 8) 11))
	(test-t (eql (string-not-greaterp "cat food 2 dollars" "dog food 3 dollars"  :start1 3 :end1 9 :start2 3 :end2 9) 9))
	(test-t (eql (string-not-greaterp "xyzabc" "abcd" :start1 3) 6))
	(test-t (eql (string-not-greaterp "abc" "abc" :end1 1) 1))
	(test-t (eql (string-not-greaterp "xyzabc" "abc" :start1 3 :end1 5) 5))
	(test-t (eql (string-not-greaterp "xyz" "abcxyzXYZ" :start2 3) 3))
	(test-t (eql (string-not-greaterp "abc" "abcxyz" :end2 3) 3))
	(test-t (eql (string-not-greaterp "xyz" "abcxyz" :end1 2 :start2 3) 2))
	(test-t (eql (string-not-greaterp "xyzabc" "abcdef" :start1 3 :end2 3) 6))
	(test-t (eql (string-not-greaterp "aaaa" "z") 0))
	(test-t (eql (string-not-greaterp "pppTTTaTTTqqq" "pTTTxTTT" :start1 3 :start2 1) 6))
	(test-t (eql (string-not-greaterp "pppTTTaTTTqqq" "pTTTxTTT"  :start1 6 :end1 7	  :start2 4 :end2 5) 6))
	(test-t (and (eql (string-not-greaterp "abc" "ABC") 3)    (eql (string-not-greaterp "ABC" "abc") 3)))
	(test-t (not (string-not-greaterp 'love 'hate)))
	(test-t (= (string-not-greaterp 'peace 'war) 0))
	(test-t (= (string-not-greaterp 'love 'love) 4))
	(test-t (= (string-not-greaterp #\a #\a) 1))
	(test-t (= (string-not-greaterp #\a #\b) 0))
	(test-t (not (string-not-greaterp #\z #\a)))
	(test-t (eql (string-not-lessp "" "") 0))
	(test-t (eql (string-not-lessp "dog" "dog") 3))
	(test-t (eql (string-not-lessp " " " ") 1))
	(test-t (eql (string-not-lessp "abc" "") 0))
	(test-t (not (string-not-lessp "" "abc")))
	(test-t (not (string-not-lessp "ab" "abc")))
	(test-t (eql (string-not-lessp "abc" "ab") 2))
	(test-t (eql (string-not-lessp "aba" "aaa") 1))
	(test-t (not (string-not-lessp "aaa" "aba")))
	(test-t (eql (string-not-lessp "my cat food" "your dog food" :start1 6 :start2 8) 11))
	(test-t (eql (string-not-lessp "cat food 2 dollars" "dog food 3 dollars"  :start1 3 :end1 9 :start2 3 :end2 9) 9))
	(test-t (eql (string-not-lessp "xyzabcde" "abcd" :start1 3) 7))
	(test-t (not (string-not-lessp "abc" "abc" :end1 1)))
	(test-t (eql (string-not-lessp "xyzabc" "a" :start1 3 :end1 5) 4))
	(test-t (eql (string-not-lessp "xyzXYZ" "abcxyz" :start2 3) 3))
	(test-t (eql (string-not-lessp "abcxyz" "abcxyz" :end2 3) 3))
	(test-t (not (string-not-lessp "xyzXYZ" "abcxyz" :end1 2 :start2 3)))
	(test-t (eql (string-not-lessp "xyzabc" "abcdef" :start1 3 :end2 3) 6))
	(test-t (eql (string-not-lessp "z" "aaaa") 0))
	(test-t (eql (string-not-lessp "pTTTxTTTqqq" "pppTTTaTTT" :start1 1 :start2 3) 4))
	(test-t (eql (string-not-lessp "pppTTTxTTTqqq" "pTTTaTTT"       :start1 6 :end1 7       :start2 4 :end2 5) 6))
	(test-t (and (eql (string-not-lessp "abc" "ABC") 3)    (eql (string-not-lessp "ABC" "abc") 3)))
	(test-t (= (string-not-lessp 'love 'hate) 0))
	(test-t (not (string-not-lessp 'peace 'war)))
	(test-t (= (string-not-lessp 'love 'love) 4))
	(test-t (= (string-not-lessp #\a #\a) 1))
	(test-t (not (string-not-lessp #\a #\b)))
	(test-t (= (string-not-lessp #\z #\a) 0))
	(test-t (stringp "aaaaaa"))
	(test-t (not (stringp #\a)))
	(test-t (not (stringp 'a)))
	(test-t (not (stringp '(string))))
	(test-t (string= (cl-make-string 3 :initial-element #\a) "aaa"))
	(test-t (string= (cl-make-string 1 :initial-element #\space) " "))
	(test-t (string= (cl-make-string 0) ""))
	
	(test-t (null (dotimes (i 10))))
	(test-t (= (dotimes (temp-one 10 temp-one)) 10))
	(test-t (let ((temp-two 0)) (and (eq t (dotimes (temp-one 10 t) (incf temp-two)))  (eql temp-two 10))))
	(test-t (let ((count 0))  (eql (dotimes (i 5 count) (incf count)) 5)))
	(test-t (let ((count 0))  (eql (dotimes (i 1 count) (incf count)) 1)))
	(test-t (let ((count 0))  (zerop (dotimes (i 0 count) (incf count)))))
	(test-t (let ((count 0))  (zerop (dotimes (i -1 count) (incf count)))))
	(test-t (let ((count 0))  (zerop (dotimes (i -100 count) (incf count)))))
	(test-t (eql (dotimes (i 3 i)) 3))
	(test-t (eql (dotimes (i 2 i)) 2))
	(test-t (eql (dotimes (i 1 i)) 1))
	(test-t (eql (dotimes (i 0 i)) 0))
	(test-t (eql (dotimes (i -1 i)) 0))
	(test-t (eql (dotimes (i -2 i)) 0))
	(test-t (eql (dotimes (i -10 i)) 0))
	(test-t (let ((list nil))  (and (eq (dotimes (i 10 t) (push i list)) t)       (equal list '(9 8 7 6 5 4 3 2 1 0)))))
	(test-t (let ((list nil))  (equal (dotimes (i 10 (push i list)) (push i list))	 '(10 9 8 7 6 5 4 3 2 1 0))))
	(test-t (let ((list nil))  (equal (dotimes (i '10 (push i list)) (push i list))	 '(10 9 8 7 6 5 4 3 2 1 0))))
	(test-t (let ((list nil))  (equal (dotimes (i (/ 100 10) (push i list)) (push i list))	 '(10 9 8 7 6 5 4 3 2 1 0))))
	(test-t (= 3 (let ((i 3)) (dotimes (i i i) ))))
	(test-t (= 3 (let ((x 0)) (dotimes (i 3 x) (incf x)))))
	(test-t (= 3 (dotimes (i 3 i) )))
	(test-t (= 3 (let ((x 0)) (dotimes (i 3 x) (declare (fixnum i)) (incf x)))))
	(test-t (null (dolist (x '()))))
	(test-t (null (dolist (x '(a)))))
	(test-t (eq t (dolist (x nil t))))
	(test-t (= 6 (let ((sum 0))       (dolist (x '(0 1 2 3) sum)	 (incf sum x)))))
	(test-t (let ((temp-two '()))  (equal (dolist (temp-one '(1 2 3 4) temp-two) (push temp-one temp-two))	 '(4 3 2 1))))
	(test-t (let ((temp-two 0))  (and (null (dolist (temp-one '(1 2 3 4)) (incf temp-two)))       (eql temp-two 4))))
	(test-t (null (dolist (var nil var))))
	(test-t (let ((list nil))  (equal (dolist (var '(0 1 2 3) list)	   (push var list))	 '(3 2 1 0))))
	(test-t (null (dolist (var '(0 1 2 3)))))
	(test-t (eql (do ((temp-one 1 (1+ temp-one))	  (temp-two 0 (1- temp-two)))	 ((> (- temp-one temp-two) 5) temp-one))     4))
	(test-t (eql (do ((temp-one 1 (1+ temp-one))	  (temp-two 0 (1+ temp-one)))     	 ((= 3 temp-two) temp-one))     3))
	(test-t (eql (do* ((temp-one 1 (1+ temp-one))	   (temp-two 0 (1+ temp-one)))	 ((= 3 temp-two) temp-one))     2))
	
	(test-t (let ((a-vector (vector 1 nil 3 nil)))
		  (do ((i 0 (+ i 1))
		       (n (array-dimension a-vector 0)))
		      ((= i n))
		    (when (null (aref a-vector i))
			  (setf (aref a-vector i) 0)))
		  (equalp a-vector #(1 0 3 0))))
	
	(test-t (let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
		  (equalp (do ((i 0 (1+ i))
			       (n #f)
			       (j 9 (1- j)))
			      ((>= i j) vec)
			    (setq n (aref vec i))
			    (setf (aref vec i) (aref vec j))
			    (setf (aref vec j) n))
			  #(9 8 7 6 5 4 3 2 1 0))))
	
	(test-t (let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
		  (and (null (do ((i 0 (1+ i))
				  (n #f)
				  (j 9 (1- j)))
				 ((>= i j))
			       (setq n (aref vec i))
			       (setf (aref vec i) (aref vec j))
			       (setf (aref vec j) n)))
		       (equalp vec #(9 8 7 6 5 4 3 2 1 0)))))
	
	(test-t (let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
		  (and (null (do ((i 0 (1+ i))
				  (n #f)
				  (j 9 (1- j)))
				 ((>= i j))
			       (setq n (aref vec i))
			       (setf (aref vec i) (aref vec j))
			       (setf (aref vec j) n)))
		       (equalp vec #(9 8 7 6 5 4 3 2 1 0)))))
	
	(test-t (let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
		  (and (null (do ((i 0 (1+ i))
				  (n #f)
				  (j 9 (1- j)))
				 ((>= i j))
			       (setq n (aref vec i))
			       (setf (aref vec i) (aref vec j))
			       (setf (aref vec j) n)))
		       (equalp vec #(9 8 7 6 5 4 3 2 1 0)))))
	
	(test-t (let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
		  (and (null (do ((n #f)
				  (i 0 (1+ i))
				  (j 9 (1- j)))
				 ((>= i j))
			       (setq n (aref vec i))
			       (setf (aref vec i) (aref vec j))
			       (setf (aref vec j) n)))
		       (equalp vec #(9 8 7 6 5 4 3 2 1 0)))))
	
	(test-t (let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
		  (and (null (do ((i 0 (1+ i))
				  (j 9 (1- j))
				  (n #f))
				 ((>= i j))
			       (setq n (aref vec i))
			       (setf (aref vec i) (aref vec j))
			       (setf (aref vec j) n)))
		       (equalp vec #(9 8 7 6 5 4 3 2 1 0)))))
	
	(test-t (= (funcall (lambda (x) (+ x 3)) 4) 7))
	(test-t (= (funcall (lambda args (apply + args)) 1 2 3 4) 10))
	(test-t (functionp (lambda args (apply + args))))
	
	(test-t (consp (cons 'a 'b)))
	(test-t (consp '(1 . 2)))
	(test-t (consp (list nil)))
	(test-t (not (consp 'a)))
	(test-t (not (consp nil)))
	(test-t (not (consp 1)))
	(test-t (not (consp #\a)))
	(test-t (let ((a (cons 1 2))) (and (eql (car a) 1) (eql (cdr a) 2))))
	(test-t (equal (cons 1 nil) '(1)))
	(test-t (equal (cons nil nil) '(())))
	(test-t (equal (cons 'a (cons 'b (cons 'c '()))) '(a b c)))
	(test-t (atom 'a))
	(test-t (atom nil))
	(test-t (atom 1))
	(test-t (atom #\a))
	(test-t (not (atom (cons 1 2))))
	(test-t (not (atom '(a . b))))
	(test-t (not (atom (list nil))))
	(test-t (listp nil))
	(test-t (listp '(a b c)))
	(test-t (listp '(a . b)))
	(test-t (listp (cons 'a 'b)))
	(test-t (not (listp 1)))
	(test-t (not (listp 't)))
	(test-t (null '()))
	(test-t (null nil))
	(test-t (not (null t)))
	(test-t (null (cdr '(a))))
	(test-t (not (null (cdr '(1 . 2)))))
	(test-t (not (null 'a)))
	(test-t (endp '()))
	(test-t (not (endp '(1))))
	(test-t (not (endp '(1 2))))
	(test-t (not (endp '(1 2 3))))
	(test-t (not (endp (cons 1 2))))
	(test-t (endp (cddr '(1 2))))
	(test-t (let ((a (cons 1 2))) (and (eq (rplaca a 0) a) (equal a '(0 . 2)))))
	(test-t (let ((a (list 1 2 3))) (and (eq (rplaca a 0) a) (equal a '(0 2 3)))))
	(test-t (let ((a (cons 1 2))) (and (eq (rplacd a 0) a) (equal a '(1 . 0)))))
	(test-t (let ((a (list 1 2 3))) (and (eq (rplacd a 0) a) (equal a '(1 . 0)))))
	(test-t (eq (car '(a . b)) 'a))
	(test-t (let ((a (cons 1 2))) (eq (car (list a)) a)))
	(test-t (eq (cdr '(a . b)) 'b))
	(test-t (eq (rest '(a . b)) 'b))
	(test-t (let ((a (cons 1 2))) (eq (cdr (cons 1 a)) a)))
	(test-t (let ((a (cons 1 2))) (eq (rest (cons 1 a)) a)))
	(test-t (eq (caar '((a) b c)) 'a))
	(test-t (eq (cadr '(a b c)) 'b))
	(test-t (eq (cdar '((a . aa) b c)) 'aa))
	(test-t (eq (cddr '(a b . c)) 'c))
	(test-t (eq (caaar '(((a)) b c)) 'a))
	(test-t (eq (caadr '(a (b) c)) 'b))
	(test-t (eq (cadar '((a aa) b c)) 'aa))
	(test-t (eq (caddr '(a b c)) 'c))
	(test-t (eq (cdaar '(((a . aa)) b c)) 'aa))
	(test-t (eq (cdadr '(a (b . bb) c)) 'bb))
	(test-t (eq (cddar '((a aa . aaa) b c)) 'aaa))
	(test-t (eq (cdddr '(a b c . d)) 'd))
	(test-t (eq (caaaar '((((a))) b c)) 'a))
	(test-t (eq (caaadr '(a ((b)) c)) 'b))
	(test-t (eq (caadar '((a (aa)) b c)) 'aa))
	(test-t (eq (caaddr '(a b (c))) 'c))
	(test-t (eq (cadaar '(((a aa)) b c)) 'aa))
	(test-t (eq (cadadr '(a (b bb) c)) 'bb))
	(test-t (eq (caddar '((a aa aaa) b c)) 'aaa))
	(test-t (eq (cadddr '(a b c d)) 'd))
	(test-t (eq (cdaaar '((((a . aa))) b c)) 'aa))
	(test-t (eq (cdaadr '(a ((b . bb)) c)) 'bb))
	(test-t (eq (cdadar '((a (aa . aaa)) b c)) 'aaa))
	(test-t (eq (cdaddr '(a b (c . cc))) 'cc))
	(test-t (eq (cddaar '(((a aa . aaa)) b c)) 'aaa))
	(test-t (eq (cddadr '(a (b bb . bbb) c)) 'bbb))
	(test-t (eq (cdddar '((a aa aaa . aaaa) b c)) 'aaaa))
	(test-t (eq (cddddr '(a b c d . e)) 'e))
	(test-t (eq (copy-tree nil) nil))
	(test-t (let* ((a (list 'a))
		       (b (list 'b))
		       (c (list 'c))
		       (x3 (cons c nil))
		       (x2 (cons b x3))
		       (x (cons a x2))
		       (y (copy-tree x)))
		  (and (not (eq x y))
		       (not (eq (car x) (car y)))
		       (not (eq (cdr x) (cdr y)))
		       (not (eq (cadr x) (cadr y)))
		       (not (eq (cddr x) (cddr y)))
		       (not (eq (caddr x) (caddr y)))
		       (eq (cdddr x) (cdddr y))
		       (equal x y)
		       (eq (car x) a) (eq (car a) 'a) (eq (cdr a) nil)
		       (eq (cdr x) x2)
		       (eq (car x2) b) (eq (car b) 'b) (eq (cdr b) nil)
		       (eq (cdr x2) x3)
		       (eq (car x3) c) (eq (car c) 'c) (eq (cdr c) nil)
		       (eq (cdr x3) nil))))
	(test-t (let* ((x (list (list 'a 1) (list 'b 2) (list 'c 3)))
		       (y (copy-tree x)))
		  (and (not (eq (car x) (car y)))
		       (not (eq (cadr x) (cadr y)))
		       (not (eq (caddr x) (caddr y))))))
	(test-t (let* ((x (list (list (list 1))))
		       (y (copy-tree x)))
		  (and (not (eq x y))
		       (not (eq (car x) (car y)))
		       (not (eq (caar x) (caar y))))))
	(test-t (let ((x (list 'a 'b 'c 'd)))
		  (and (equal (sublis '((a . 1) (b . 2) (c . 3)) x)
			      '(1 2 3 d))
		       (equal x '(a b c d)))))
	(test-t (eq (sublis '() '()) '()))
	(test-t (equal (sublis '() '(1 2 3)) '(1 2 3)))
	(test-t (eq (sublis '((a . 1) (b . 2)) '()) nil))
	(test-t (equal (sublis '((a . 1) (b . 2) (c . 3)) '(((a)) (b) c)) '(((1)) (2) 3)))
	(test-t (equal (sublis '(((a) . 1) ((b) . 2) ((c) . 3)) '((((a))) ((b)) (c))) '((((a))) ((b)) (c))))
	(test-t (equal (sublis '(((a) . 1) ((b) . 2) ((c) . 3)) '((((a))) ((b)) (c)) :test equal) '(((1)) (2) 3)))
	(test-t (equal (nsublis '((a . 1) (b . 2) (c . 3)) (list 'a 'b 'c 'd)) '(1 2 3 d)))
	(test-t (let* ((x (list 'a 'b 'c 'd)) (y (nsublis '((a . 1) (b . 2) (c . 3)) x))) (and (eq x y) (equal x '(1 2 3 d)))))
	(test-t (let ((x (list 'l 'm 'n))) (and (eq (nsublis '((a . 1) (b . 2) (c . 3)) x) x) (equal x '(l m n)))))
	(test-t (let* ((n (cons 'n nil))
		       (m (cons 'm n))
		       (l (cons 'l m))
		       (x (nsublis '((a . 1) (b . 2) (c . 3)) l)))
		  (and (eq x l)
		       (eq (car l) 'l)
		       (eq (cdr l) m)
		       (eq (car m) 'm)
		       (eq (cdr m) n)
		       (eq (car n) 'n)
		       (eq (cdr n) nil))))
	(test-t (eq (nsublis '() '()) '()))
	(test-t (equal (nsublis '() '(1 2 3)) '(1 2 3)))
	(test-t (eq (nsublis '((a . 1) (b . 2)) '()) nil))
	(test-t (equal (nsublis '((a b c) (b c d) (c d e)) (list 'a 'b 'c)) '((b c) (c d) (d e))))
	(test-t (equal (nsublis '((a . 1) (b . 2) (c . 3)) (copy-tree '(((a)) (b) c))) '(((1)) (2) 3)))
	(test-t (equal (nsublis '(((a) . 1) ((b) . 2) ((c) . 3)) (copy-tree '((((a))) ((b)) (c)))) '((((a))) ((b)) (c))))
	(test-t (equal (nsublis '(((a) . 1) ((b) . 2) ((c) . 3)) (copy-tree '((((a))) ((b)) (c))) :test equal) '(((1)) (2) 3)))
	(test-t (let ((tree '(old (old) ((old))))) (equal (subst 'new 'old tree) '(new (new) ((new))))))
	(test-t (eq (subst 'new 'old 'old) 'new))
	(test-t (eq (subst 'new 'old 'not-old) 'not-old))
	(test-t (equal (subst 'new '(b) '(a ((b))) :test equal) '(a (new))))
	(test-t (equal (subst 'x 3 '(1 (1 2) (1 2 3) (1 2 3 4)) :key (lambda (y) (and (listp y) (third y)))) '(1 (1 2) x x)))
	(test-t (equal (subst 'x "D" '("a" ("a" "b") ("a" 