/*
   [Digi] A new config module for ices.
   Reads configuration from the XML tree.
*/

#include <libxml/parser.h>
#include <libxml/xmlmemory.h>
#include <lame.h>
#include <util.h>
#include <xpl.h>
#include "definitions.h"
#include <debug.h>

#define _MAX_INT       0x7FFFFFFF

// Returns node's text.
static PSZ _xmlGetNodeValue(xmlNodePtr xmlNode)
{
  return ( ( xmlNode->children == NULL || xmlNode->children->content == NULL ||
           xmlNode->children->content[0] == '\0' )
           ? NULL
           : xmlNode->children->content );
}

// Converts a string pszVal to a flag *piDest:
//   (in) pszVal is Y, YES, ON or 1 - (out) *piDest = 1,
//   (in) pszVal is N, NO, OFF or 0 - (out) *piDest = 0.
// Returns FALSE if string cannot be converted.
static BOOL _strToFlag(int *piDest, PSZ pszVal)
{
  if ( ices_util_word_index( "Y YES ON 1", pszVal ) != -1 )
    *piDest = 1;
  else if ( ices_util_word_index( "N NO OFF 0", pszVal ) != -1 )
    *piDest = 0;
  else
    return FALSE;

  return TRUE;
}

// Converts a string pszVal to an integer *piDest. Returns FALSE if string
// cannot be converted or result not in range iMin..iMax.
static BOOL _strToInt(int *piDest, PSZ pszVal, int iMin, int iMax)
{
  int        iVal;
  PCHAR      pcEnd;

  if ( pszVal == NULL )
    return FALSE;

  ICES_UTIL_SKIP_SPACES( pszVal );

  iVal = (int)strtol( pszVal, &pcEnd, 10 );
  if ( ( pcEnd == pszVal ) || ( iVal < iMin ) || ( iVal > iMax ) )
    return FALSE;

  *piDest = iVal;
  return TRUE;
}

// Removes leading and trailing spaces from pszVal, frees allocated memory
// *ppszDest, writes pointer to the new value to *ppszDest.
static VOID _setConfigString(PSZ *ppszDest, PSZ pszVal)
{
  PCHAR      pcEnd;
  ULONG      cbVal;
  PSZ        pszNew;

  if ( pszVal == NULL )
    return;

  ICES_UTIL_SKIP_SPACES( pszVal );

  pcEnd = strchr( pszVal, '\0' );
  while( ( pcEnd > pszVal ) && isspace( *(pcEnd - 1) ) )
    pcEnd--;

  if ( pcEnd == pszVal )
    // Empty value.
    return;

  cbVal = pcEnd - pszVal;
  pszNew = debugMAlloc( cbVal + 1 );
  if ( pszNew == NULL )
  {
    debug( "Not enough memory" );
    return;
  }

  memcpy( pszNew, pszVal, cbVal );
  pszNew[cbVal] = '\0';

  if ( *ppszDest != NULL )
    debugFree( *ppszDest );

  *ppszDest = pszNew;
}


// Parse XML tree "Playlist". When an error has occurred, returns xml node with
// invalid value. Returns NULL if the operation succeeds.
static xmlNodePtr _parsePlaylistTree(ices_config_t *ices_config,
                                     xmlNodePtr xmlNode)
{
  PSZ        pszVal, pszAttr;
  int        iVal;
  int        iNodeIdx;
  BOOL       fError = FALSE;

  for( xmlNode = xmlNode->xmlChildrenNode; ( xmlNode != NULL ) && !fError;
       xmlNode = xmlNode->next )
  {
    if ( xmlIsBlankNode( xmlNode ) || ( xmlNode->type == XML_COMMENT_NODE ) )
      continue;

    pszVal = _xmlGetNodeValue( xmlNode );
    iNodeIdx = ices_util_word_index( "Randomize Type File Module Crossfade "
                                     "MinCrossfade CrossMix id3v1Enc",
                                     (char *)xmlNode->name );
    switch( iNodeIdx )
    {
      case 0: // Randomize
        fError = !_strToFlag( &ices_config->pm.randomize, pszVal );
        break;

      case 1: // Type
        switch( ices_util_word_index( "script program builtin built-in",
                                      pszVal ) )
        {
          case 0: // script
            ices_config->pm.playlist_type = ices_playlist_script_e;
            break;

          case 1: // program
            ices_config->pm.playlist_type = ices_playlist_program_e;
            break;

          case 2: // builtin
          case 3: // built-in
            ices_config->pm.playlist_type = ices_playlist_builtin_e;
            break;

          default:
            fError = TRUE;
        }
        break;

      case 2: // File
        _setConfigString( &ices_config->pm.playlist_file, pszVal );
        _setConfigString( &ices_config->pm.ext,
                          xmlGetNoNsProp( xmlNode, "ext" ) );

        pszAttr = xmlGetNoNsProp( xmlNode, "scan-per" );
        if ( pszAttr != NULL )
        {
          ULONG        ulVal;
          PCHAR        pcEnd = NULL;

          ices_config->pm.scan_per = 0;
          do
          {
            ICES_UTIL_SKIP_SPACES( pszAttr );
            if ( *pszAttr == '\0' )
              break;

            ulVal = strtoul( pszAttr, &pcEnd, 10 );
            if ( pcEnd == pszVal )
            {
              fError = TRUE;
              break;
            }

            pszAttr = pcEnd;
            ICES_UTIL_SKIP_SPACES( pszAttr );
            if ( *pszAttr == '\0' )
            {
              // Last numeric value without EU - get it as minutes.
              ices_config->pm.scan_per += ulVal;
              break;
            }

            switch( ices_util_word_index( "m min minute minutes "
                                          "h hour hours d day days",
                                          pszAttr ) )
            {
              case 0: // m
              case 1: // min
              case 2: // minute
              case 3: // minutes
                ices_config->pm.scan_per += ulVal;
                break;

              case 4: // h
              case 5: // hour
              case 6: // hours
                ices_config->pm.scan_per += ( ulVal * 60 );
                break;

              case 7: // d
              case 8: // day
              case 9: // days
                ices_config->pm.scan_per += ( ulVal * 24 * 60 );
                break;

              default:
                fError = TRUE;
                break;
            }

            ICES_UTIL_SKIP_NOT_SPACES( pszAttr );
          }
          while( !fError );

          fError |= ices_config->pm.scan_per == 0;
          break;
        }

        break;

      case 3: // Module
        _setConfigString( &ices_config->pm.module, pszVal );
        break;

      case 4: // Crossfade
        if ( _strToInt( &iVal, pszVal, 1, _MAX_INT ) )
          ices_config->plugins = crossfade_plugin( iVal );
        else
          fError = TRUE;
        break;

      case 5: // MinCrossfade
      case 6: // CrossMix
        if ( !_strToInt( &iVal, pszVal, 1, 180 ) )
          fError = TRUE;
        else
        {
          ices_plugin_t  *pPlugin = ices_config->plugins;

          while( pPlugin != NULL )
          {
            if ( strcmp( pPlugin->name, "crossfade" ) == 0 )
            {
              if ( pPlugin->options( iNodeIdx == 5 ? // Node "MinCrossfade"?
                                       CFOPT_FADEMINLEN : CFOPT_CROSSMIX,
                                     &iVal ) < 0 )
                fError = TRUE;

              break;
            }
            pPlugin = pPlugin->next;
          }

          if ( pPlugin == NULL )
            ices_log( "Option specified for non-registered plugin: %s",
                      xmlNode->name );
        }
        break;

      case 7: // id3v1Enc
        _setConfigString( &ices_config->pm.id3v1_enc, pszVal );
        break;

      default:
        ices_log( "Unknown configuration node: %s", xmlNode->name );
    }

    if ( fError )
      return xmlNode;
  } // for( xmlNode ...

  return NULL; 
}

// Parse XML tree "Execution". When an error has occurred, returns xml node
// with invalid value. Returns NULL if the operation succeeds.
static xmlNodePtr _parseExecutionTree(ices_config_t *ices_config,
                                      xmlNodePtr xmlNode)
{
  PSZ        pszVal;
  int        iNodeIdx;
  BOOL       fError = FALSE;

  for( xmlNode = xmlNode->xmlChildrenNode; ( xmlNode != NULL ) && !fError;
       xmlNode = xmlNode->next )
  {
    if ( xmlIsBlankNode( xmlNode ) || ( xmlNode->type == XML_COMMENT_NODE ) )
      continue;

    pszVal = _xmlGetNodeValue( xmlNode );
    iNodeIdx = ices_util_word_index( "Background Verbose BaseDirectory "
                                     "LogMaxSize LogRotate InstanceId NoCUE "
                                     "MaxErrors", (char *)xmlNode->name );
    switch( iNodeIdx )
    {
      case 0: // Background (not used)
        break;

      case 1: // Verbose
        fError = !_strToFlag( &ices_config->verbose, pszVal );
        break;

      case 2: // BaseDirectory
        _setConfigString( &ices_config->base_directory, pszVal );
        break;

      case 3: // LogMaxSize
        fError = !_strToInt( &ices_config->log_max_size, pszVal, 0, _MAX_INT );
        break;

      case 4: // LogRotate
        fError = !_strToInt( &ices_config->log_rotate, pszVal, 0, _MAX_INT );
        break;

      case 5: // InstanceId
        _setConfigString( &ices_config->instance_id, pszVal );
        break;

      case 6: // NoCUE
        fError = !_strToFlag( &ices_config->no_cue, pszVal );
        break;

      case 7: // MaxErrors
        fError = !_strToInt( &ices_config->max_stream_errors, pszVal, 0,
                            _MAX_INT );
        break;

      default:
        ices_log( "Unknown configuration node: %s", xmlNode->name );
    }

    if ( fError )
      return xmlNode;
  } // for( xmlNode ...

  return NULL; 
}

// Parse XML tree "Server". When an error has occurred, returns xml node with
// invalid value. Returns NULL if the operation succeeds.
static xmlNodePtr _parseServerTree(ices_stream_t *pStream, xmlNodePtr xmlNode)
{
  PSZ            pszVal;
  int            iNodeIdx;
  BOOL           fError = FALSE;

  for( xmlNode = xmlNode->xmlChildrenNode; ( xmlNode != NULL ) && !fError;
       xmlNode = xmlNode->next )
  {
    if ( xmlIsBlankNode( xmlNode ) || ( xmlNode->type == XML_COMMENT_NODE ) )
      continue;

    pszVal = _xmlGetNodeValue( xmlNode );
    iNodeIdx = ices_util_word_index( "Port Hostname Password Protocol",
                                     (char *)xmlNode->name );
    switch( iNodeIdx )
    {
      case 0: // Port
        fError = !_strToInt( &pStream->port, pszVal, 1, 0xFFFF );
        break;

      case 1: // Hostname
        _setConfigString( &pStream->host, pszVal );
        break;

      case 2: // Password
        _setConfigString( &pStream->password, pszVal );
        break;

      case 3: // Protocol
        switch( ices_util_word_index( "icy xaudiocast http", pszVal ) )
        {
          case 0: // icy
            pStream->protocol = icy_protocol_e;
            break;

          case 1: // xaudiocast
            pStream->protocol = xaudiocast_protocol_e;
            break;

          case 2: // http
            pStream->protocol = http_protocol_e;
            break;

          default:
            fError = TRUE;
        }
        break;

      default:
        ices_log( "Unknown configuration node: %s", xmlNode->name );
    }

    if ( fError )
      return xmlNode;
  } // for( xmlNode ...

  return NULL; 
}

// Parse XML tree "Stream". When an error has occurred, returns xml node
// with invalid value. Returns NULL if the operation succeeds.
static xmlNodePtr _parseStreamTree(ices_stream_t *pStream,
                                   xmlNodePtr xmlNode)
{
  PSZ            pszVal, pszAttr;
  int            iNodeIdx;
  BOOL           fError = FALSE;

  for( xmlNode = xmlNode->xmlChildrenNode; ( xmlNode != NULL ) && !fError;
       xmlNode = xmlNode->next )
  {
    if ( xmlIsBlankNode( xmlNode ) || ( xmlNode->type == XML_COMMENT_NODE ) )
      continue;

    pszVal = _xmlGetNodeValue( xmlNode );
    iNodeIdx = ices_util_word_index( "Server Name Genre Description URL "
                                     "Mountpoint Dumpfile Bitrate Public "
                                     "MetadataEnc Reencode Samplerate Channels",
                                     (char *)xmlNode->name );
    switch( iNodeIdx )
    {
      case 0: // Server
        {
          xmlNodePtr  xmlInvalidNode = _parseServerTree( pStream, xmlNode );

          if ( xmlInvalidNode != NULL )
            return xmlInvalidNode;
        }
        break;

      case 1: // Name
        _setConfigString( &pStream->name, pszVal );
        break;

      case 2: // Genre
        _setConfigString( &pStream->genre, pszVal );
        break;

      case 3: // Description
        _setConfigString( &pStream->description, pszVal );
        break;

      case 4: // URL
        _setConfigString( &pStream->url, pszVal );
        break;

      case 5: // Mountpoint
        _setConfigString( &pStream->mount, pszVal );
        break;

      case 6: // Dumpfile
        _setConfigString( &pStream->dumpfile, pszVal );
        break;

      case 7: // Bitrate
        fError = !_strToInt( &pStream->bitrate, pszVal, 32, 320 );
        break;

      case 8: // Public
        fError = !_strToFlag( &pStream->ispublic, pszVal );
        break;

      case 9: // MetadataEnc
        _setConfigString( &pStream->metadata_enc, pszVal );
        break;

      case 10: // Reencode
        iNodeIdx = ices_util_word_index( "N NO OFF 0 Y YES ON 1 CBR VBR ABR",
                                         pszVal );

        if ( iNodeIdx == -1 )
        {
          fError = TRUE;
          break;
        }

        if ( iNodeIdx <= 3 ) // N NO OFF 0
          pStream->reencode = 0;
        else
        {
          pStream->reencode = 1;

          if ( iNodeIdx <= 8 ) // Y YES ON 1 CBR
            pStream->reenc_mode = vbr_off;
          else if ( iNodeIdx == 9 )
            pStream->reenc_mode = vbr_mtrh;
          else
            pStream->reenc_mode = vbr_abr;
        }

        pszAttr = xmlGetNoNsProp( xmlNode, "quality" );
        if ( ( pszAttr != NULL ) &&
             !_strToInt( &pStream->quality, pszAttr, 0, 9 ) )
          ices_log( "Invalid attribute \"quality\" for the node \"%s\". "
                    "The default value %d used.",
                    xmlNode->name, pStream->quality );

        pszAttr = xmlGetNoNsProp( xmlNode, "min-br" );
        if ( ( pszAttr != NULL ) &&
             !_strToInt( &pStream->reenc_min_bitrate, pszAttr, 8, 320 ) )
          ices_log( "Invalid attribute \"min-br\" for the node \"%s\". "
                    "The default value %d used.",
                    xmlNode->name, pStream->reenc_min_bitrate );

        pszAttr = xmlGetNoNsProp( xmlNode, "max-br" );
        if ( ( pszAttr != NULL ) &&
             !_strToInt( &pStream->reenc_max_bitrate, pszAttr, 8, 320 ) )
          ices_log( "Invalid attribute \"max-br\" for the node \"%s\". "
                    "The default value %d used.",
                    xmlNode->name, pStream->reenc_max_bitrate );

        pszAttr = xmlGetNoNsProp( xmlNode, "vbr-quality" );
        if ( ( pszAttr != NULL ) &&
             !_strToInt( &pStream->reenc_vbr_quality, pszAttr, 0, 9 ) )
          ices_log( "Invalid attribute \"vbr-quality\" for the node \"%s\". "
                    "The default value %d used.",
                    xmlNode->name, pStream->reenc_vbr_quality );

        break;

      case 11: // Samplerate
        fError = !_strToInt( &pStream->out_samplerate, pszVal, 8000, 48000 );
        break;

      case 12: // Channels
        fError = !_strToInt( &pStream->out_numchannels, pszVal, 1, 2 );
        break;

      default:
        ices_log( "Unknown configuration node: %s", xmlNode->name );
    }

    if ( fError )
      return xmlNode;
  } // for( xmlNode ...

  return NULL; 
}


/* Top level XML configfile parser */
int ices_xml_parse_config_file(ices_config_t *ices_config,
                               const char *configfile)
{
  xmlDocPtr      xmlDoc = xmlParseFile( configfile );
  xmlNodePtr     xmlNode;
  xmlNodePtr     xmlInvalidNode = NULL;
  ices_stream_t  *pStream = ices_config->streams;
  ULONG          cStreams = 0;

  if ( xmlDoc == NULL ) {
    ices_log_error( "Error while parsing %s", configfile );
    return 0;
  }

  xmlNode = xmlDocGetRootElement( xmlDoc );
  if ( xmlNode == NULL )
  {
    ices_log_error( "No root node at %s", configfile );
    xmlFreeDoc( xmlDoc );
    return 0;
  }

  if ( ( xmlNode->name == NULL ) ||
       ( stricmp( xmlNode->name, "Configuration" ) != 0 ) )
  {
    ices_log_error( "Root node in %s is not 'Configuration'", configfile );
    xmlFreeDoc( xmlDoc );
    return 0;
  }

  // Parse xml-trees: Playlist, Execution and Stream.
  for( xmlNode = xmlNode->xmlChildrenNode;
       ( xmlNode != NULL ) && ( xmlInvalidNode == NULL );
       xmlNode = xmlNode->next )
  {
    if ( xmlIsBlankNode( xmlNode ) || ( xmlNode->type == XML_COMMENT_NODE ) )
      continue;

    switch( ices_util_word_index( "Playlist Execution Stream Server",
                                  (char *)xmlNode->name ) )
    {
      case 0: // Playlist
        xmlInvalidNode = _parsePlaylistTree( ices_config, xmlNode );
        break;

      case 1: // Execution
        xmlInvalidNode = _parseExecutionTree( ices_config, xmlNode );
        break;

      case 2: // Stream
        // Allocate memory for a new stream configuration. First stream is
        // preallocated.
        if ( cStreams != 0 )
        {
          pStream->next = (ices_stream_t *)debugMAlloc( sizeof(ices_stream_t) );
          if ( pStream->next == NULL )
          {
            debug( "Not enough memory" );
            break;
          }
          pStream = pStream->next;
          ices_setup_parse_stream_defaults( pStream );
        }
        cStreams++;

        xmlInvalidNode = _parseStreamTree( pStream, xmlNode );
        break;

      default:
        ices_log( "Unknown configuration node: %s", xmlNode->name );
    }
  }

  if ( xmlInvalidNode != NULL )
  {
    PSZ      pszVal = _xmlGetNodeValue( xmlInvalidNode );

    ices_log_error_output( "Invalid value \"%s\" for the node: %s",
                           ICES_UTIL_NULL_TO_EMPTY_STRING( pszVal ),
                           xmlInvalidNode->name );
    xmlFreeDoc( xmlDoc );
    return 0;
  }

  xmlFreeDoc( xmlDoc );
  xmlCleanupParser();

  return 1;
}
