#include <windows.h>
#include "definitions.h"
#include "sysctrl.h"
#include <debug.h>

#define SEM_COUNT      ( sizeof(apszCommands) / sizeof(apszCommands[0]) )

extern ices_config_t ices_config;

// Commands for ices_sysctrl_command()
static PSZ             apszCommands[] =
{
  "STOP",       // SEMID_STOP
  "NEXT",       // SEMID_NEXT
  "RELOAD",     // SEMID_RELOAD
  "ROTATE"      // SEMID_ROTATE
#ifdef DEBUG_FILE
#  define SEMID_DBGCNT     4
  ,"DBGCNT"      // SEMID_DBGCNT
#endif
};

static HANDLE aEvSem[SEM_COUNT] = { NULL };

static LONG _makeInstanceBaseName(PCHAR pcBuf)
{
  return ices_config.instance_id != NULL
           ? _snprintf( pcBuf, MAX_PATH - 8, "Global\\IceS_%s-",
                        ices_config.instance_id )
           : _snprintf( pcBuf, MAX_PATH - 8, "Global\\IceS" );
}


void ices_sysctrl_initialize(void)
{
  ULONG      ulIdx;
  CHAR       szBuf[MAX_PATH];
  BOOL       fError = FALSE;
  LONG       cbBase = _makeInstanceBaseName( &szBuf );

  if ( cbBase <= 0 )
    return;

  for( ulIdx = 0; ulIdx < SEM_COUNT; ulIdx++ )
  {
    strcpy( &szBuf[cbBase], apszCommands[ulIdx] );
    aEvSem[ulIdx] = CreateEvent( NULL, ulIdx == SEMID_STOP, FALSE, szBuf );

    if ( aEvSem[ulIdx] == NULL )
    {
      ices_log_error_output( "Cannot create an event semaphore %s, error %u",
                             szBuf, GetLastError() );
      fError = TRUE;
      break;
    }

    if ( GetLastError() == ERROR_ALREADY_EXISTS )
    {
      ices_log_error_output( "Duplicate name for an event semaphore %s "
                             "Does instance %s already launched?",
                             szBuf,
                             ices_config.instance_id == NULL
                               ? "(default)" : ices_config.instance_id );
      fError = TRUE;
      break;
    }
  }

  if ( fError )
  {
    // Error - close created event semaphores and shutdown.

    for( ulIdx = 0; ulIdx < SEM_COUNT; ulIdx++ )
      if ( aEvSem[ulIdx] != NULL )
      {
        CloseHandle( aEvSem[ulIdx] );
        aEvSem[ulIdx] = NULL;
      }

    ices_setup_shutdown();
  }
}

void ices_sysctrl_shutdown(void)
{
  ULONG      ulIdx;

  for( ulIdx = 0; ulIdx < SEM_COUNT; ulIdx++ )
    if ( aEvSem[ulIdx] != NULL )
      CloseHandle( aEvSem[ulIdx] );
}

unsigned long ices_sysctrl_check(unsigned long timeout, input_stream_t* source)
{
  DWORD      dwRC;

  dwRC = WaitForMultipleObjects( SEM_COUNT, &aEvSem, FALSE, timeout );

  if ( dwRC > (WAIT_OBJECT_0 + SEM_COUNT - 1) )
  {
    if ( dwRC != WAIT_TIMEOUT)
      debug( "WaitForMultipleObjects(), rc = 0x%X", dwRC );
    return (unsigned int)(-1);
  }

  switch( dwRC )
  {
    case SEMID_STOP:
      debug( "An event semaphore STOP posted" );
      break;

    case SEMID_NEXT:
      ices_log_debug( "An event semaphore NEXT posted, "
                      "skipping to next track...");
      ices_stream_next();
      break;

    case SEMID_RELOAD:
      ices_log_debug( "An event semaphore RELOAD posted, "
                      "Reloading playlist..." );
      ices_playlist_reload();
      break;

    case SEMID_ROTATE:
      ices_log_debug( "An event semaphore ROTATE posted, "
                      "Rotate log files..." );
      ices_log_reopen_logfile();
      break;
#ifdef DEBUG_FILE
    case SEMID_DBGCNT:
      debugState();
      break;
#endif

    default:
      debug( "Unknown semaphore posted: %d", dwRC );
      break;
  }

  return (unsigned long)dwRC;
}

void ices_sysctrl_command(char *cmd)
{
  ULONG      ulIdx;
  BOOL       fFound = FALSE;
  CHAR       szBuf[MAX_PATH];
  LONG       cbBase;
  HANDLE     hEv;

  for( ulIdx = 0; ulIdx < sizeof(apszCommands) / sizeof(PSZ); ulIdx++ )
  {
    if ( stricmp( apszCommands[ulIdx], cmd ) == 0 )
    {
      fFound = TRUE;
      break;
    }
  }

  if ( !fFound )
  {
    ices_log_error_output( "Unknown command \"%s\"", cmd );
    return;
  }

  // Make "base" name (path) for semaphore.
  cbBase = _makeInstanceBaseName( &szBuf );
  if ( cbBase <= 0 )
    return;

  // Add name of semaphore for given command.
  strcpy( &szBuf[cbBase], apszCommands[ulIdx] );

  hEv = OpenEvent( EVENT_ALL_ACCESS, FALSE, szBuf );
  if ( hEv == NULL )
  {
    DWORD    dwRC = GetLastError();

    switch( dwRC )
    {
      case ERROR_ACCESS_DENIED:
        ices_log_error_output( "Access is denied." );
        break;

      case ERROR_FILE_NOT_FOUND:
        ices_log_error_output( "Semaphore does not exist. Does instance %s "
                               "launched?",
                               ices_config.instance_id == NULL ?
                                 "(default)" : ices_config.instance_id );
        break;

      default:
        ices_log_error_output( "OpenEvent(), error: %d", GetLastError() );
    }
  }
  else
  {
    SetEvent( hEv );
    CloseHandle( hEv );
    ices_log_error_output( "Ok." );
  }
}

void ices_sysctrl_post(unsigned long ulSemaphoreNumber)
{
  if ( ulSemaphoreNumber < SEM_COUNT )
    SetEvent( aEvSem[ulSemaphoreNumber] );
}
