/******************************************************************************
    GNU General Public License: WebCamApplet

    A Java applet suite for the display of images served from a web cam.

    Copyright (C) 2002 Paul Sidnell

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

package org.webcamapplet;

import java.awt.*;
import java.awt.event.*;
import java.applet.*;
import java.net.*;
import java.util.*;
import java.text.*;
import java.io.*;

/**
 * @author $Author: psidnell $
 * @version $Revision: 1.7 $
 * @since $Date: 2002/12/13 12:19:07 $
 *
 * <P>A "simple" Web Cam applet. Well, it was simple when I started.
 *
 * <P>If you have an image that's 160x120 and want a label
 * of 20 pixels then you want width=160, height=140, labelHeight=20.
 *
 * <P> No resizing is done so all the images referred to must be the same
 * size.
 *
 * <P> When you're developing, use appletviewer most of the time
 * rather than a browser since many browsers cache the class file
 * forever as far as I can tell and reloading the page doesn't
 * reload the class. What I do is diddle with it in appletviewer
 * until I'm happy then RENAME the class (hence it's daft name)
 * before testing in in Netscape/Explorer.
 *
 * <P>REMEMBER the image URLS must appear to come from the SAME base url
 * as the applet itself or you'll get security exceptions. If this isn't
 * possible (it isn't for me) than consider investigating Apache. It can
 * do clever stuff like proxy a jpg from another machine as if it were
 * a jpg in the applets directory.
 *
 * <P>It supports the following parameters:
 * <BLOCKQUOTE>
 *    <P>The TOTAL applet width (same as the applet width value)
 *    <br>name=width value="160"
 *
 *    <P>The TOTAL applet height (same as the applet height value)
 *    <br>name=height value="140"
 *
 *    <P>The height of the label at the bottom of the screen.
 *    <br>name=labelHeight value="20"
 *
 *    <P>Don't load images more frequently than this:
 *    <br>name=delayMillis value="500"
 *
 *    <P>If the camera appears to be off, check again every:
 *    <br>name=camRetrySecs value="20"
 *
 *    <P>Don't let the viewer watch for more than:
 *    <br>name=camviewLimitMins value="20"
 *
 *    <P>Loop over label every:
 *    <br>name="labelReloadSecs" value="3"
 *
 *    <P>Add a custom title in the label:
 *    <br>name="title" value="Paul's Office"
 *
 *    <P>URL of live image:
 *    <br>name=onImage value="http://sidnell.dynip.com/ps/webcam/live.jpg"
 *
 *    <P>URL of image served when camera is off:
 *    <br>name=offImage value="http://sidnell.dynip.com/ps/webcam/offimage.jpg"
 *
 *    <P>URL of image served when view limit exceeded:
 *    <br>name=limitImage value="http://sidnell.dynip.com/ps/webcam/limitimage.jpg"
 *
 *    <P>URL to be launched in separate browser when applet clicked on
 *    <br>name=click value="http://www.sidnell.co.uk"
 *
 *    <P>Turn logging on/off in the console
 *    <br>name=logging value="true"
 *  </BLOCKQUOTE>
 */
public abstract class WebCamApplet extends Applet implements Runnable, MouseListener
{
  protected static final int MIN_POLL_DELAY = 50;
  protected static final int MAX_POLL_DELAY = 1000;
  protected static final DateFormat DATE_FORMAT = new SimpleDateFormat ("h':'mm':'ss' 'a");
  
  
  // Colors used
  public static final Color LIGHT_GREY = new Color (220, 220, 220);
  public static final Color DARK_GREY = new Color (100, 100, 100);
  
  // Applets state
  protected static final int OK = 0;
  protected static final int TIME_UP = 1;
  protected static final int CAMERA_OFF = 2;
  protected int state = OK;
  
  // Applet params
  protected boolean LOGGING = false;
  protected int GOOD_DELAY;
  protected int BAD_DELAY;
  protected int VIEW_LIMIT;
  protected int LABEL_HEIGHT;
  protected int WIDTH;
  protected int HEIGHT;
  protected String ON_IMAGE;
  protected String OFF_IMAGE;
  protected String LIMIT_IMAGE;
  protected String CLICK_URL;
  protected long DEACTIVATION_TIME;
  protected String TITLE;
  protected int LABEL_RELOAD_TIME;
  protected long START_TIME;
  
  // Other cruft
  protected Image currentImage;
  protected boolean animateFlag = true;
  protected long imageReloadTime = 0;
  protected long messageReloadTime = 0;
  protected Label label = new Label ("");
  protected long frameCount = 0;
  
  // Ugly - these ought to be local to method where used
  // but I'd be repeatedly re/de-allocating which is pointless
  protected static byte buffer[] = new byte[4096];
  protected ByteArrayOutputStream out = new ByteArrayOutputStream (40000);

  /** 
   * Return the version number. Provided by the auto generated subclass.
   * @return the version number.
   */  
  public abstract String getVersion ();
  
  
  /** 
   * Return the compilation date. Provided by the auto generated subclass.
   * @return the compilation date.
   */  
  public abstract String getCompilationDate ();
  
  /**
   * A trivial "logging" mechanism. Prints messages in the Java console
   * window when the logging parameter is set to true.
   * @param s String to be logged
   */
  protected void log (String s)
  {
    if (LOGGING)
    {
      synchronized (this)
      {
        System.out.println (s);
      }
    }
  }
  
  /**
   * A wrapper for the standard Applet method getParameter. It has a
   * default value for when none is supplied.
   * @param name Parameter name
   * @param defltVal The default value
   * @return the parameter value (if any) or default value.
   */
  public String getParameterDefault (String name, String defltVal)
  {
    String result = getParameter (name);
    if (result == null)
      result = defltVal;
    return result;
  }
  
  /**
   * Load in all the applet parameters.
   */
  public void loadParams ()
  {
    log ("loadParams");
    
    // params from the HTML
    WIDTH = Integer.parseInt (getParameterDefault ("width", "100"));
    HEIGHT = Integer.parseInt (getParameterDefault ("height", "100"));
    LABEL_HEIGHT = Integer.parseInt (getParameterDefault ("labelHeight", "20"));
    GOOD_DELAY = Integer.parseInt (getParameterDefault ("delayMillis", "500"));
    BAD_DELAY = 1000 * Integer.parseInt (getParameterDefault ("camRetrySecs", "5"));
    LABEL_RELOAD_TIME = 1000 * Integer.parseInt (getParameterDefault ("labelReloadSecs", "3"));
    VIEW_LIMIT = 60 * 1000 * Integer.parseInt (getParameterDefault ("camViewLimitMins", "10"));
    CLICK_URL = getParameterDefault ("click", "http://www.sidnell.co.uk");
    TITLE = getParameterDefault ("title", "Hello There!");
    ON_IMAGE = getParameterDefault ("onImage", "webcam.jpg");
    OFF_IMAGE = getParameterDefault ("offImage", "off.jpg");
    LIMIT_IMAGE = getParameterDefault ("limitImage", "limit.jpg");
    LOGGING = getParameterDefault ("logging", "false").equals ("true");
    
    //derived params
    DEACTIVATION_TIME = System.currentTimeMillis () + VIEW_LIMIT;
    START_TIME = System.currentTimeMillis ();
  }
  
  /**
   * Standard applet init method. Lays out the UI and
   * initialises everything.
   */
  public void init ()
  {
    log ("init");
    
    loadParams ();
    
    setLayout (null);
    add (label, BorderLayout.SOUTH);
    Rectangle bounds = new Rectangle (0, HEIGHT - LABEL_HEIGHT, WIDTH, LABEL_HEIGHT);
    label.setBounds (bounds);
    label.setBackground (LIGHT_GREY);
    label.setForeground (DARK_GREY);
    
    addMouseListener (this);
    
    currentImage = getImageFailSafe ();
  }
  
  /**
   * Standard applet start method. Kicks off the two threads, one
   * updating the image, the other the label.
   */
  public void start ()
  {
    log ("start");
    
    (new Thread (this, "Image Thread")).start ();
    (new Thread (this, "Message Thread")).start ();
  }
  
  /**
   * Standard applet stop method.
   */
  public void stop ()
  {
    log ("stop");
    
    animateFlag = false;
  }
  
  public void mouseEntered (MouseEvent e)
  {}
  public void mouseExited (MouseEvent e)
  {}
  public void mousePressed (MouseEvent e)
  {}
  public void mouseReleased (MouseEvent e)
  {}
  
  /**
   * Launch a URL in a new browser when the applet is clicked on.
   * @param e The event.
   */
  public void mouseClicked (MouseEvent e)
  {
    try
    {
      log ("Launching " + CLICK_URL);
      getAppletContext ().showDocument (new URL (CLICK_URL), "new");
    }
    catch (Exception ex)
    {
      System.out.println (ex);
    }
  }
  
  /**
   * Load an input stream into a byte array.
   * @param in The input stream
   * @throws IOException if theres an error
   * @return the bytes from the stream
   */
  protected byte[] loadInputStream (InputStream in)
  throws IOException
  {
    try
    {
      out.reset ();
      int bytesRead = in.read (buffer);
      while (bytesRead != -1)
      {
        out.write (buffer, 0, bytesRead);
        bytesRead = in.read (buffer);
      }
      return out.toByteArray ();
    }
    finally
    {
      out.reset ();
    }
  }
  
  /**
   * Check to see if the image is really there.
   * Annoyingly, I can't find a way to do this from
   * the image itself so I create the image the hard way.
   * This also seems to solve some cacheing problems where
   * the image never changed.
   * @param imageUrl the image location
   * @throws IOException if there's a problem
   * @return the loaded image
   */
  protected Image forcedImageLoad (String imageUrl)
  throws IOException
  {
    log ("forcedImageLoad URL = " + imageUrl);
    
    // The url may or may or may not be absolute
    if (!imageUrl.startsWith("http:") && !imageUrl.startsWith("file:"))
    {
        String codeBase = getCodeBase ().toString ();
        if (!codeBase.endsWith ("/")) codeBase += "/";
        imageUrl = codeBase + imageUrl;
        log ("reconstructed URL: " + imageUrl);    
    }
    
    URL url = new URL (imageUrl);
    
    // This throws an exception if the camera is off
    InputStream in = new BufferedInputStream (url.openStream ());
    try
    {
      byte buffer[] = loadInputStream (in);
      Image i = Toolkit.getDefaultToolkit ().createImage (buffer);
      log ("forcedImageLoad - loaded (" + buffer.length + ")");
      frameCount ++;
      return i;
    }
    finally
    {
      in.close ();
    }
  }
  
  /**
   * Try and load the live image. If that doesn't work then load
   * one of the alternative images. This updates the applets "state"
   * depending on success.
   * @return the image to be displayed or null if no new image is to be shown
   */
  protected Image getImageFailSafe ()
  {
    Image image = null;
    
    long now = System.currentTimeMillis ();
    
    try
    {
      if (now > DEACTIVATION_TIME)
      {
        log ("DEACTIVATION_TIME reached");
        // Has the viewer been on too long?
        // OI! GET OFF MY BANDWIDTH!
        
        image = getImage (getCodeBase (), LIMIT_IMAGE);
        state = TIME_UP;
        return image;
      }
      else if (now > imageReloadTime)
      {
        log ("imageReloadTime reached");
        imageReloadTime = now + GOOD_DELAY;
        image = forcedImageLoad (ON_IMAGE); // avoid cacheing
        if (state != OK)
        {
          START_TIME = now;
          frameCount = 0;
        }
        state = OK;
      }
    }
    catch (Exception e)
    {
      log ("Load failure: " + e);
      // We only get here (usually) if the url check fails so the
      // camera is probably off.
      image = getImage (getCodeBase (), OFF_IMAGE);
      state = CAMERA_OFF;
      imageReloadTime = now + BAD_DELAY;
    }
    return image;
  }
  
  /**
   * Set the Applets message based on state. The message alternates
   * between several different strings with a configurable delay.
   */
  protected void setMessage ()
  {
    long now = System.currentTimeMillis ();
    if (now < messageReloadTime)
      return;
    
    String message = "";
    messageReloadTime = now + LABEL_RELOAD_TIME;
    long whatMessage = (System.currentTimeMillis () / LABEL_RELOAD_TIME) % 3;
    switch ((int) whatMessage)
    {
      case 0: message = "Applet Version: " + getVersion (); break;
      case 1: message = TITLE; break;
      default:
        float fps = 0;
        if (frameCount > 0)
          fps = frameCount / ((now - START_TIME) / 1000f);
        DecimalFormat df = new DecimalFormat ("##0.0");
        message = "Frames Per Second: " + df.format (fps);
        break;
    }
    
    if (!label.getText ().equals (message))
      label.setText (message);
  }
  
  /**
   * The standard Thread.run method
   */
  public void run ()
  {
    String threadName = Thread.currentThread ().getName ();
    if (threadName.equals ("Image Thread"))
      imageUpdaterLoop ();
    else if (threadName.equals ("Message Thread"))
      messageUpdaterLoop ();
  }
  
  /**
   * The standard paint method.
   * @param g the applets Graphics object
   */
  public void paint (Graphics g)
  {
    if (currentImage != null)
      g.drawImage (currentImage,0,0,this);
  }
  
  /**
   * The standard update method.
   * @param g The applets Graphics object.
   */
  public void update (Graphics g)
  {
    paint (g);
  }
  
  /**
   * Given the start time of some operation and a total time,
   * sleep for the remainder of the total time accounting for
   * how much has elapsed.
   *
   * For example, if we want to be showing 1 frame a second but
   * the act of displaying that frame took 300ms, this method will
   * sleep for the remaining 700ms.
   * @param started The time in ms that the operation started
   * @param total The total time we want the operation to "take".
   */
  protected void sleepForRemainder (long started, long total)
  {
    String threadName = Thread.currentThread ().getName ();
    long duration = System.currentTimeMillis () - started;
    long remainder = total - duration;
    if (remainder > 0)
    {
      log (threadName + " WAITING " + remainder + " TOOK " + duration);
      try
      {Thread.sleep (remainder);}
      catch(InterruptedException e)
      {}
    }
  }
  
  /**
   * Update the image.
   */
  protected void imageUpdaterLoop ()
  {
    while(animateFlag && state != TIME_UP)
    {
      long start = System.currentTimeMillis ();
      
      Image i = getImageFailSafe ();
      if (i != null)
      {
        if (currentImage != null)
          currentImage.flush ();
        currentImage = i;
        repaint ();
      }
      
      int min_delay = MAX_POLL_DELAY;
      if (GOOD_DELAY < min_delay)
        min_delay = GOOD_DELAY;
      if (min_delay < MIN_POLL_DELAY)
        min_delay = MIN_POLL_DELAY;
      
      sleepForRemainder (start, min_delay);
    }
  }
  
  /**
   * Update the label.
   */
  protected void messageUpdaterLoop ()
  {
    while (animateFlag && state != TIME_UP)
    {
      long start = System.currentTimeMillis ();
      
      setMessage ();
      
      sleepForRemainder (start, MAX_POLL_DELAY);
    }
    
    label.setText ("Stopped at: " + DATE_FORMAT.format (new Date ()));
  }
}
