/*****************************************************************************
 *                                                                          
 * This software module was originally developed by 
 *
 * J. Ignacio Ronda (UPM-GTI / ACTS-MoMuSys)
 *                                                                      
 * and edited by                                                        
 *
 * Martina Eckert (UPM-GTI / ACTS-MoMuSys)
 *
 *
 * Copyright (c) 1997
 *
 *****************************************************************************/

/***********************************************************HeaderBegin*******
 *                                                                         
 * File:        rc_util2.c
 *
 * Author:      J. Ignacio Ronda, UPM-GTI
 * Created:     09-12-97
 *                                                                         
 * Description: Utility functions, calling to NUMERICAL RECIPIES!
 *
 * Flags:       -D_RC_DEBUG_  -  RC debugging   
 *
 * Modified:
 ***********************************************************HeaderEnd*********/

/************************    INCLUDE FILES    ********************************/
/*#ifdef _RC_UPM2_*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#if !defined(WIN32)
#  include <unistd.h>
#endif

#include <ctype.h>

#include "momusys.h"
#include "mom_vop.h"
#include "vm_config.h"
#include "mom_vol.h"
#include "mom_structs.h"
#include "vm_enc_defs.h"

#include "rc.h"

#define NRANSI
#define MAX_NUM_DATA 100
#define NMAX 5
#define MAX_NUM_COEFS 5
#define NR_END 1
#define FREE_ARG char*


double **nr_dmatrix(long nrl, long nrh, long ncl, long nch); 
void dsvdcmp(double **a, int m, int n, double w[], double **v);
double dpythag(double a, double b);
Void genXYmatrix(Double **X, Double **y2, Int n, Double *x1, Double *y);

static double dsqrarg;
#define DSQR(a) ((dsqrarg=(a)) == 0.0 ? 0.0 : dsqrarg*dsqrarg)

static double dmaxarg1,dmaxarg2;
#define DMAX(a,b) (dmaxarg1=(a),dmaxarg2=(b),(dmaxarg1) > (dmaxarg2) ?\
        (dmaxarg1) : (dmaxarg2))

static int iminarg1,iminarg2;
#define IMIN(a,b) (iminarg1=(a),iminarg2=(b),(iminarg1) < (iminarg2) ?\
        (iminarg1) : (iminarg2))

#define NRSIGN(a,b) ((b) >= 0.0 ? fabs(a) : -fabs(a))
#define SIGN2(a,b) ((b) >= 0.0 ? fabs(a) : -fabs(a))


/* ------------------------------------------------------------- */
/* Functions from former rc_matrix.c                             */
/* ------------------------------------------------------------- */


/***************************************************************************
  nrerror();
  ***************************************************************************/

Void nrerror(Char *error_text)
{
    fprintf(stderr, "Numerical Recipes run-time error...\n");
    fprintf(stderr, "%s\n", error_text);
    fprintf(stderr, "...now exiting to system...\n");
    exit(1);
}

/***************************************************************************
  dvector();
  ***************************************************************************/

Double *dvector(int nl, Int nh)
{
	Double *v;

	v=(Double *)malloc((U_Int) (nh-nl+1)*sizeof(Double));
	if (!v) nrerror("allocation failure in dvector()");
	return v-nl;
}


/***************************************************************************
  free_matrix();
  ***************************************************************************/

Void free_matrix(Double **m, Int nrl, Int nrh, Int ncl, Int nch)
{
    Int i;

    if((m[nrl-1][0] != (Double)(nrh - nrl + 1)) ||
       (m[nrl-1][1] != (Double)(nch - ncl + 1)))
      {
	fprintf(stderr,"original h=%d w=%d release h=%d w=%d\n",
	       (Int)m[nrl-1][0],(Int)m[ncl-1][1], 
	       nrh - nrl + 1, nch - ncl + 1);
	perror("Inconsistency");
      }
    m[nrl-1][1]=m[nrl-1][0]=0;
    free((Double *) (m[nrl-1]));

    for (i = nrh; i >= nrl; i--)
	free((Double *) (m[i] + ncl)), m[i]=NULL;
    free((Double *) (m + nrl-1)), m=NULL;
  }

/***************************************************************************
  free_dvector();
  ***************************************************************************/

Void free_dvector(Double *v, Int nl)
{
	free((Char*) (v+nl));
}


/***************************************************************************
  matrix();
  ***************************************************************************/

Double **matrix(Int nrl, Int nrh, Int ncl, Int nch)
{
    Int i, j;
    Double **m;
    
    m = (Double **) malloc((U_Int) (nrh - nrl + 2) * sizeof(Double *));
    if (!m)
	nrerror("allocation failure 1 in matrix()");
    m -= nrl;
    m++;

    m[nrl-1]=(Double *) malloc((2 * sizeof(Double)));
    m[nrl-1][0]= (Double) (nrh - nrl + 1);
    m[nrl-1][1]= (Double) (nch - ncl + 1);

    for (i = nrl; i <= nrh; i++) {
	m[i] = (Double *) malloc((U_Int) (nch - ncl + 1) * sizeof(Double));
	if (!m[i])
	    nrerror("allocation failure 2 in matrix()");
	m[i] -= ncl;
    }

    for (i = nrl; i <= nrh; i++)
	for (j = ncl; j <= nch; j++)
	    m[i][j] = 0.0;

   return(m);
}



/***********************************************************CommentBegin******
 *  
 * -- rc_2svd_comp --
 *
 * Author : 
 *      J. Ignacio Ronda, UPM-GTI     
 *
 * Created :            
 *      09-12-97
 *
 * Purpose :
 *
 *      SVD adjustment of a function of two variables:
 *             y[i] = p1*x1[i]+p2*x2[i]
 * 
 ***********************************************************CommentEnd********/

void rc_2svd_comp(
		  Double *y,
		  Double *x1,
		  Double *x2,
		  Int     n,  /* Number of data */
		  Double *p1,
		  Double *p2
		  )
{
  Double **X, **y2, w[3], **V;
  Double s1,s2,perr;
  Int i,j;


  /* Calculation of Prediction Error */
  perr=CUAD((*p1)*x1[0]+(*p2)*x2[0]-y[0]);

  fprintf(stdout,"PREDICTION ERROR:%f\n",perr);
  
  /* Default Model Parameters */
  *p1 = 0; s1 = 0;
  *p2 = 0; s2 = 0;

  X = (Double**)matrix(0, n, 0, 2);
  y2 = (Double**)matrix(0, n, 0, 1);
  V = (Double**)matrix(0, 2, 0, 2);
  genXYmatrix(X, y2, n, x1, y);


  /* SVD */
  dsvdcmp(X,n,2,w,V);

  for (i=1;i<=2;i++)
      if (w[i]<1e-6)
         w[i]=0;
      else
         w[i]=1/w[i];
            
  for (i=1;i<=2;i++)
      for (j=1;j<=2;j++)
          V[i][j]*=w[j];

  for (i=1;i<=n;i++)
      {
      s1+=X[i][1]*y2[i][1];
      s2+=X[i][2]*y2[i][1];
      }	
  *p1 = V[1][1]*s1+V[1][2]*s2;
  *p2 = V[2][1]*s1+V[2][2]*s2;   

#ifdef _RC_DEBUG_
  fprintf(stdout, "RC: rc_2svd_comp: n= %d  X1= %.3f  X2= %.3f\n", n, *p1, *p2);
#endif /* _RC_DEBUG_ */
  free_matrix(V, 0, 2, 0, 2);
  free_matrix(X, 0, n, 0, 2);
  free_matrix(y2, 0, n, 0, 1);
}

/***********************************************************CommentBegin******
 *  
 * -- rc_nsvd_comp --
 *
 * Author : 
 *      J. Ignacio Ronda, UPM-GTI     
 *
 * Created :            
 *      09-12-97
 *
 * Purpose :
 *      SVD adjustment of a function of n variables:
 *             y[i] = p[1]*x[1][i] + ... + p[n]*x[n][i]
 *             i starting with 1
 *
 ***********************************************************CommentEnd********/

void rc_nsvd_comp(Double  *y,  /* Data */
                  Double **x,  /* Vectors x[i] */
                  Int      m,  /* Number of data */
		            Int      n,  /* Number of coefficients */
                  Double  *p   /* Adjustment parameters */
                 )
{
  static Double **X, *y2, *y3, *y4, *w, **V;
  Double perr, average;
  Int i, j;
  /* FILE *fp;*/

  static int init=0;

  if (!init)
    {
    w=(Double *) malloc((unsigned) (n+1)*sizeof(Double));
    /***
    X = matrix(0, m, 0, n);
    ***/
    X = nr_dmatrix(1, MAX_NUM_DATA, 1, MAX_NUM_DATA);
    V = nr_dmatrix(1, NMAX, 1, NMAX);
    y2 = (Double *) malloc((unsigned) (NMAX+1)*sizeof(Double));
    y3 = (Double *) malloc((unsigned) (NMAX+1)*sizeof(Double));
    y4 = (Double *) malloc((unsigned) (NMAX+1)*sizeof(Double));
    init=1;
    }

  if (n>NMAX || m>MAX_NUM_DATA)
    {
    printf("rc_nsvd_comp: Error: n o m\n");
    exit(1);
    }
  
  for(i=1;i<=m;i++)
	  for(j=1;j<=n;j++)
        {
		  X[i][j]=x[j-1][i-1]; /* Atention: Change of index range */
        /*** printf("Input data: X[%d][%d]= %f\n", i, j, X[i][j]); fflush(stdout);***/
        }
  
  /*JIR*//* fprintf(stdout, "*** rc_nsvd_comp: 7 ***\n"); fflush(stdout); */

  /* SVD */
  dsvdcmp(X,m,n,w,V);
  
  for (i=1;i<=n;i++)
     {
     /*** printf("Result: w[%d]= %f\n", i, w[i]); ***/
     if (fabs(w[i])<1E-9)
        w[i]=0.;
     }

  /*JIR*//* fprintf(stdout, "*** rc_nsvd_comp: 7 ***\n"); fflush(stdout); */
            
  /***
  fp=fopen("X","w");
  out_matrix(fp,X,1,1,m,n);
  fclose(fp);
  fp=fopen("w","w");
  out_vector(fp,w,1,n);
  fclose(fp);
  fp=fopen("V","w");
  out_matrix(fp,V,1,1,n,n);
  fclose(fp);
  ***/

  /*JIR*//*  fprintf(stdout, "*** rc_nsvd_comp: 7 ***\n"); fflush(stdout); */
  /* y2 = Ut y */
  for (i=1;i<=n;i++)
     {
     y2[i] = 0;
     for (j=1;j<=m;j++)
         {
         /*** printf("Result: u[%d][%d]= %f\n", i, j, X[i][j]); ***/
         y2[i] += X[j][i]*y[j-1];  /* Atention: Change of index range */
         }
     /*** JIR: printf("Result: y2[%d]= %f\n", i, y2[i]); ***/
     }

  /*JIR fprintf(stdout, "*** rc_nsvd_comp: 8 ***\n"); fflush(stdout); */
  /* y3 = W* y2 */
  for (i=1;i<=n;i++)
     {
     if (w[i] != 0.)
        y3[i]=y2[i]/w[i];
     else
        y3[i]=0.;
     /*** printf("Result: y3[%d]= %f\n", i, y3[i]); ***/
     }
  
  /* y4 = V y3 */
  for (i=1;i<=n;i++)
     {
     y4[i] = 0;
     for (j=1;j<=n;j++)
        {
        /*** printf("Result: V[%d][%d]= %f\n", i, j, V[i][j]); ***/
        y4[i] += V[i][j]*y3[j];
        }
     /***printf("Result: y4[%d]= %f\n", i, y4[i]); ***/
     p[i-1] = y4[i];       /* Atention: Change of index range */
     }

  perr = 0;
  average = 0;
  for (i=0;i<m;i++)
     {
     Double pred=0;
     for (j=0;j<n;j++)
        pred += p[j]*x[j][i];
     perr += CUAD(y[i]-pred);
     average += y[i];
     /*** fprintf(stdout, "JIR: y[%d]= %f  pred= %f, RE= %f\n", i, y[i],
                   pred, (pred-y[i])/y[i]); ***/
     }
  perr /= m;
  average /= m;
  /*JIR*/fprintf(stdout, "JIR: rel. perr (1) = %E\n", sqrt(perr)/average);

  /*JIR fprintf(stdout, "*** rc_nsvd_comp: 9 ***\n"); fflush(stdout); */
  /***
  fp=fopen("p","w");
  out_vector(fp,p,1,n);
  fclose(fp);
  free(w);
  free(s);
  free_matrix(V, 0, n, 0, n);
  free_matrix(X, 0, m, 0, n);
  free_matrix(y2, 0, m, 0, 1);
  ***/
}

/***********************************************************CommentBegin******
 *  
 * -- nr_dmatrix --
 *
 * Author : 
 *      J. Ignacio Ronda, UPM-GTI     
 *
 * Created :            
 *      09-12-97
 *
 * Purpose :
 *      Allocate a double matrix with subscript range m[nrl..nrh][ncl..nch] 
 *
 ***********************************************************CommentEnd********/

double **nr_dmatrix(long nrl, long nrh, long ncl, long nch)
{
	long i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
	double **m;

	/* allocate pointers to rows */
	m=(double **) malloc((size_t)((nrow+NR_END)*sizeof(double*)));
	if (!m) nrerror("allocation failure 1 in matrix()");
	m += NR_END;
	m -= nrl;

	/* allocate rows and set pointers to them */
	m[nrl]=(double *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(double)));
	if (!m[nrl]) nrerror("allocation failure 2 in matrix()");
	m[nrl] += NR_END;
	m[nrl] -= ncl;

	for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

	/* return pointer to array of pointers to rows */
	return m;
}


/***********************************************************CommentBegin******
 * 
 * -- dsvdcmp -- 
 *
 * (C) Copr. 1986-92 Numerical Recipes Software 'w50-30!V.
 *
 ***********************************************************CommentEnd********/

void dsvdcmp(double **a, int m, int n, double w[], double **v)
{
	double dpythag(double a, double b);
	int flag,i,its,j,jj,k,l=0,nm=0;
	double anorm,c,f,g,h,s,scale,x,y,z,*rv1;

	rv1=dvector(1,n);
	g=scale=anorm=0.0;
	for (i=1;i<=n;i++) {
		l=i+1;
		rv1[i]=scale*g;
		g=s=scale=0.0;
		if (i <= m) {
			for (k=i;k<=m;k++) scale += fabs(a[k][i]);
			if (scale) {
				for (k=i;k<=m;k++) {
					a[k][i] /= scale;
					s += a[k][i]*a[k][i];
				}
				f=a[i][i];
				g = -SIGN2(sqrt(s),f);
				h=f*g-s;
				a[i][i]=f-g;
				for (j=l;j<=n;j++) {
					for (s=0.0,k=i;k<=m;k++) s += a[k][i]*a[k][j];
					f=s/h;
					for (k=i;k<=m;k++) a[k][j] += f*a[k][i];
				}
				for (k=i;k<=m;k++) a[k][i] *= scale;
			}
		}
		w[i]=scale *g;
		g=s=scale=0.0;
		if (i <= m && i != n) {
			for (k=l;k<=n;k++) scale += fabs(a[i][k]);
			if (scale) {
				for (k=l;k<=n;k++) {
					a[i][k] /= scale;
					s += a[i][k]*a[i][k];
				}
				f=a[i][l];
				g = -SIGN2(sqrt(s),f);
				h=f*g-s;
				a[i][l]=f-g;
				for (k=l;k<=n;k++) rv1[k]=a[i][k]/h;
				for (j=l;j<=m;j++) {
					for (s=0.0,k=l;k<=n;k++) s += a[j][k]*a[i][k];
					for (k=l;k<=n;k++) a[j][k] += s*rv1[k];
				}
				for (k=l;k<=n;k++) a[i][k] *= scale;
			}
		}
		anorm=DMAX(anorm,(fabs(w[i])+fabs(rv1[i])));
	}

   /*** printf("JIR: dsvdcmp: 1\n"); fflush(stdout); ***/

	for (i=n;i>=1;i--) {
		if (i < n) {
			if (g) {
				for (j=l;j<=n;j++) v[j][i]=(a[i][j]/a[i][l])/g;
				for (j=l;j<=n;j++) {
   /*** printf("JIR: dsvdcmp: i= %d  j= %d\n", i, j); fflush(stdout); ***/
					for (s=0.0,k=l;k<=n;k++) s += a[i][k]*v[k][j];
					for (k=l;k<=n;k++) v[k][j] += s*v[k][i];
				}
			}
			for (j=l;j<=n;j++) v[i][j]=v[j][i]=0.0;
		}
		v[i][i]=1.0;
		g=rv1[i];
		l=i;
	}

   /*** printf("JIR: dsvdcmp: 2\n"); fflush(stdout); ***/

	for (i=IMIN(m,n);i>=1;i--) {
		l=i+1;
		g=w[i];
		for (j=l;j<=n;j++) a[i][j]=0.0;
		if (g) {
			g=1.0/g;
			for (j=l;j<=n;j++) {
				for (s=0.0,k=l;k<=m;k++) s += a[k][i]*a[k][j];
				f=(s/a[i][i])*g;
				for (k=i;k<=m;k++) a[k][j] += f*a[k][i];
			}
			for (j=i;j<=m;j++) a[j][i] *= g;
		} else for (j=i;j<=m;j++) a[j][i]=0.0;
		++a[i][i];
	}
	for (k=n;k>=1;k--) {
		for (its=1;its<=30;its++) {
			flag=1;
			for (l=k;l>=1;l--) {
				nm=l-1;
				if ((double)(fabs(rv1[l])+anorm) == anorm) {
					flag=0;
					break;
				}
				if ((double)(fabs(w[nm])+anorm) == anorm) break;
			}
			if (flag) {
				c=0.0;
				s=1.0;
				for (i=l;i<=k;i++) {
					f=s*rv1[i];
					rv1[i]=c*rv1[i];
					if ((double)(fabs(f)+anorm) == anorm) break;
					g=w[i];
					h=dpythag(f,g);
					w[i]=h;
					h=1.0/h;
					c=g*h;
					s = -f*h;
					for (j=1;j<=m;j++) {
						y=a[j][nm];
						z=a[j][i];
						a[j][nm]=y*c+z*s;
						a[j][i]=z*c-y*s;
					}
				}
			}
			z=w[k];
			if (l == k) {
				if (z < 0.0) {
					w[k] = -z;
					for (j=1;j<=n;j++) v[j][k] = -v[j][k];
				}
				break;
			}
			if (its == 30) nrerror("no convergence in 30 dsvdcmp iterations");
			x=w[l];
			nm=k-1;
			y=w[nm];
			g=rv1[nm];
			h=rv1[k];
			f=((y-z)*(y+z)+(g-h)*(g+h))/(2.0*h*y);
			g=dpythag(f,1.0);
			f=((x-z)*(x+z)+h*((y/(f+SIGN2(g,f)))-h))/x;
			c=s=1.0;
			for (j=l;j<=nm;j++) {
				i=j+1;
				g=rv1[i];
				y=w[i];
				h=s*g;
				g=c*g;
				z=dpythag(f,h);
				rv1[j]=z;
				c=f/z;
				s=h/z;
				f=x*c+g*s;
				g = g*c-x*s;
				h=y*s;
				y *= c;
				for (jj=1;jj<=n;jj++) {
					x=v[jj][j];
					z=v[jj][i];
					v[jj][j]=x*c+z*s;
					v[jj][i]=z*c-x*s;
				}
				z=dpythag(f,h);
				w[j]=z;
				if (z) {
					z=1.0/z;
					c=f*z;
					s=h*z;
				}
				f=c*g+s*y;
				x=c*y-s*g;
				for (jj=1;jj<=m;jj++) {
					y=a[jj][j];
					z=a[jj][i];
					a[jj][j]=y*c+z*s;
					a[jj][i]=z*c-y*s;
				}
			}
			rv1[l]=0.0;
			rv1[k]=f;
			w[k]=x;
		}
	}
   /*** printf("JIR: dsvdcmp: 3\n"); fflush(stdout); ***/
	free_dvector(rv1,1);
   /*** printf("JIR: dsvdcmp: 4\n"); fflush(stdout); ***/
}

double dpythag(double a, double b)
{
	double absa,absb;
	absa=fabs(a);
	absb=fabs(b);
	if (absa > absb) return absa*sqrt(1.0+DSQR(absb/absa));
	else return (absb == 0.0 ? 0.0 : absb*sqrt(1.0+DSQR(absa/absb)));
}
#undef NRANSI
/* (C) Copr. 1986-92 Numerical Recipes Software 'w50-30!V. */


/* ------------------------------------------------------------- */
/* Functions from original Sarnoff Q2 implementation.            */
/* ------------------------------------------------------------- */

/***************************************************************************
  genXYmatrix();
  ***************************************************************************/

Void genXYmatrix(Double **X, Double **y2, Int n, Double *x1, Double *y)
{
   Int k;

    for (k=1; k<=n; k++)
       {
       X[k][1] = 1;
       X[k][2] = x1[k-1];
       y2[k][1] = y[k-1]/x1[k-1];
       }
}

/*#endif*/

