/*  10 March 2003: Daniel Hellerstein (danielh@crosslink.net)

      S_REFINANCE.CMD -- an SREhttp/2 addon
                        compute financial implications of different loan plans.

   This program lets you compare 3 "home loan refinancing" scenarios. 
   One of them (PlanARM) can be an ARM, or a balloon, loan.

   Output consists of a month-by-month description of the "paid off principal" and
   the "money in your bank" for all three loans; from month 1 to month 180.

   In addition to setting the terms (interest rate, closing costs, etc.) of
   each of these loans, a number of other features are supported, including:

     * You can also take out a "home equity loan" at some point in time (say,
       after 8 years), and pay it off over several years.

     * You can model the future trends in "bond rate" (the interest you recieve
       in your 'tax-free' bank account), using a quadratic equation.

     * You can set your marginal tax rate

   The following section of "user changeable" parameters should be adjusted to 
   describe the choice you have avaiable, your personal situation, and your
   best guess as to the future.

***/   


/********** BEGIN USER CHANGEABLE PARAMETERS   **************/
/**** These are used as default, in case the <form> does not supply a parameter ***/

old_principal=0     /* principal of house already owned */

principal=100000    /* principal you need to pay off */

strategy=1           /* Overall investment strategy:  
                       1 to "build bank account", 0 to "pay off loan" */


willpay=850      /* amount to pay per month,
                   If > a mortgage payment, remainder goes to principal or bank 
                   account (depending on the value of STRATEGY).
                   If 0, use the largest of the three monthly payments. */

willsave=50       /* extra to save to bank, per month. This NOT effected by STRATEGY */
willsave2=300     /* amount to use (per month) to pay off home equity loan 
                     Set to 0 to use max of the three loans.
                     If this value is < a required payment, money will be withdrawn
                     from your bank account (your bank account CAN go negative) */

startinbank=2000     /* how much do you start with in your bank (closing costs
                        may be deducted from this). A value of 0 is allowable */


/* PlanA: typically, short term loan with lower interest rate */
interest15=4.875           /* interest rate.  Example: 5.25 means "5.25 percent" */
months15=180            /* term of loan, in months */
costs15=0               /* closing costs: paid from bank */
costs15_prin=0          /* closing costs, rolled into loan */


/* PlanB: typically, long term loan with higher interest rate */
interest30=5.5      
months30=360
costs30=0       
costs30_prin=0

/* PlanARM:  an ARM loan
   Rates are fixed for a while, and then are adjusted every 12 months. 
   Length of full loan is months_arm
   Loan rate is fixed for first months_arm1 months (at interest_arm1)
   After that, it floats, depending on that year's BOND rate, and the value
   of using the '+x.x' or '*x.x' syntax

   Or, to do a normal loan, see months_arm1=100000 

   Or, to do a balloon loan, set interest_adj=x.x
*/

months_arm=180           /* term of ARM, in months */
interest_arm1=3.625        /*  interest during first period */
months_arm1=36           /* # months of first rate (set to 10000 for a fixed-rate loan) */
interest_adj='+2'        /* compute ARM rate using ..
                             x.x   = x.x is the actual rate (use this for a balloon loan)
                           '+x.x'  = add x.x to each year's bond rate
                           '*x.x'  = multiply x.x by each year's bond rate */
max_rise=2              /* maximum yearly increase in interest rate */
max_rise_all=6          /* maximum increase over lifetime of loan (over starting value) */
costsarm=0
costsarm_prin=0

/* 2nd (home-equity) loan info: 
     After WHENEQUITY months, get the "2nd loan" amount of NEEDBORROW
     Note: if you won't need to withdraw money, set WHENEQUITY=10000

     NEEDBORROW is obtained from the bank account.
     If more money is needed, the shortfall (what your bank account can not cover)
     is obtained via a 2nd (home equity loan).
   
    If a home-equity loan is needed, then the maximum (of all three plans) home-equity
    payment is paid. If a plan's required home-equity payment (possibly 0, if the
    NEEDBORROW is less then money in the bank) is less then this maximum, 
    the difference is either  put in the bank or applied to the principal (depending
    on the value of STRATEGY).
*/

needborrow=40000           /* size of second loan. -1 means "use PlanB's bank balance" */     
interest_equity='+2'       /* loan rate of 2nd loan,
                               x.x   = x.x is the actual rate
                             '+x.x'  = add x.x to month_whenequity bond rate
                             '*x.x'  = multiply x.x by month_whenequity bond rate */
whenequity=72             /* month number that you take out 2nd loan */
equity_months=84          /* months required to pay back 2nd loan */


/* other parameters */

taxrate=0.4    /* marginal tax rate (you get this fraction of interest payments back) */

/* determine path of bank account interest rate, or bond interest rate (depending on
   the value of BANK_BOND) using a quadratic:
     rate = bond_b0 + bond_b1 * month +  bond_b2*month*month 
  Examples:
    For constant value of 3%
        b0=3 ; b1=0 ; b2=0
    For a 0.5% rise per year, starting at 3%
        b0=3 ; b1=0.5/12  ; b2=0
    For a 0.5% rise per year, that flattens out toward the end 
        b0=3 ; b1=0.5/12  ; b2=-0.0001
    For a slow, but increasing rise:
        b0=3  ; b1=0.1   ; b2=0.0001  

   Actually, this is better thought of as the interest you'ld get investing your money
   in a "tax-free bond fund" -- interest earned is NOT taxed, and the
   possibly-ever-changing interest rate applies to your entire bank balance.
*/

bond_b0=3
bond_b1=0.3/12
bond_b2=-0.00006


BANK_BOND=1     /* The type of savings:
                        0= normal savings account -- the entire balance is subject to
                           the current interest rate
                        1= a sequence of bonds -- a seperate, long-term 
                           fixed-interest-rate, bond is bought each month
                        In both cases, the rate used is "after tax" */


/*******************     end of user changeable parameters    **************/

call init 


parse arg list,servername,verb,tempfile,,
          prog_file,reqnum,verbose,client_ip,privset,,
          uri,host_nickname,id_info,aiter,attribs,seluse2

if verb=" " then do
   say "This SREhttp/2 procedure is not meant to be run in standalone mode "
   exit
end  


/* Now get the parameters */
call get_params
res1=result
if res1<>0 then return res1


call lineout tempfile,'<html><head><title>Refinance your home calculations</title></head>'
call lineout tempfile,'<h3>Refinance your home: results</h3>'
call lineout tempfile,'<pre>'

minterest15=(interest15/100)/12
minterest30=(interest30/100)/12
minterestarm=(interest_arm1/100)/12

p30=old_principal-costs30_prin ; p15=old_principal-costs15_prin 
parm=old_principal-costsarm_prin 

if datatype(bond_b0)<>'NUM' | datatype(bond_b1)<>'NUM' | datatype(bond_b2)<>'NUM' then do
    return 'string bad values for bond equation: 'bond_b0 bond_b1 bond_b2
end

interest_arm2=interest_arm1

full_principal=principal+old_principal


if needborrow<1 & strategy=0 then whenequity=10000  /* do NOT do a 2nd loan */

/* compute monthly mortgage payment (and schedules, which aren't currently used) */

mpay15=fig_mortgage(principal+costs15_prin,interest15/100,months15)
call copy_payments("PAY15")

if willpay=0 then willpay=mpay15

mpay30=fig_mortgage(principal+costs30_prin,interest30/100,months30)
call copy_payments("PAY30")

mpayarm=fig_mortgage(principal+costsarm_prin,interest_arm1/100,months_arm)
call copy_payments("PAYARM")

/* compute the "extra" payments. These are used to ensure that total
   "reductions from consumption" are the same in all 3 plans
   It's possible for these to be negative! */

diff_15=willpay-mpay15

diff_30=willpay-mpay30
                   
diff_arm=willpay-mpayarm

call lineout tempfile,'</pre><table>'
call lineout tempfile,'<tr><th>Size of loan <tt>(of total house value)</tt></th>'
call lineout tempfile,'<td>'||addcomma(principal)||'  (<tt>'||addcomma(full_principal)||'</tt>) </td>'
call lineout tempfile,'<tr><th>Strategy:</th><td><em>'||astrategy.strategy||'</em></td>'
call lineout tempfile,'<tr><th>Savings Type:</th><td><em>'bankbond.bank_bond||'</em></td>'

call lineout tempfile,'<tr><th>Scenarios:</th><td><table bgcolor="lightgrey">'
call lineout tempfile,'<tr><td colspan=4>&nbsp;</td><th colspan=4>First payment summary</th>'

call lineout tempfile,'<tr><th>Plan</th><th>%Interest</th><th>Mortgage Payment</th>'
call lineout tempfile,'<th>Costs<br>(loan<br>bank)</th>'
call lineout tempfile,'<th>Extra Principal</th><th>principal</th>'
call lineout tempfile,'<th>tax deductions <br>(on interest)</th>'
call lineout tempfile,'<th>asset accumulation</th>'

goob=trunc(costs15_prin)||'<br>'||trunc(costs15)  
call lineout tempfile,'<tr><th> Loan A</th>'
call lineout tempfile,'<td>'||interest15||'</td>'
call lineout tempfile,'<td>'||trunc(mpay15)||'</td>'
call lineout tempfile,'<td>'||goob||'</td>'
call lineout tempfile,'<td>'||diff_15||'</td>'
call lineout tempfile,'<td>'||trunc(pay15.1.!PRINCIPAL)||'</td>'
call lineout tempfile,'<td>'||trunc(pay15.1.!interest*taxrate)||'</td>'
call lineout tempfile,'<td>'||trunc(diff_15+pay15.1.!PRINCIPAL+pay15.1.!interest*taxrate)||'</td>'   

goob=trunc(costs30_prin)||'<br>'||trunc(costs30)  
call lineout tempfile,'<tr><th>Loan B</th>'
call lineout tempfile,'<td>'||interest30||'</td>'
call lineout tempfile,'<td>'||trunc(mpay30)||'</td>'
call lineout tempfile,'<td>'||goob||'</td>'
call lineout tempfile,'<td>'||trunc(diff_30)||'</td>'
call lineout tempfile,'<td>'||trunc(pay30.1.!PRINCIPAL)||'</td>'
call lineout tempfile,'<td>'||trunc(pay30.1.!interest*taxrate)||'</td>'
call lineout tempfile,'<td>'||trunc(diff_30+pay30.1.!PRINCIPAL+pay30.1.!interest*taxrate)||'</td>'   

goob=trunc(costsarm_prin)||'<br>'||trunc(costsARM)  
call lineout tempfile,'<tr><th rowspan=2>Loan ARM</th>'
call lineout tempfile,'<td>'||interestarm||'</td>'
call lineout tempfile,'<td>'||trunc(mpayarm)||'</td>'
call lineout tempfile,'<td>'||goob||'</td>'
call lineout tempfile,'<td>'||trunc(diff_arm)||'</td>'
call lineout tempfile,'<td>'||trunc(payarm.1.!PRINCIPAL)||'</td>'
call lineout tempfile,'<td>'||trunc(payarm.1.!interest*taxrate)||'</td>'
call lineout tempfile,'<td>'||trunc(diff_arm+payarm.1.!PRINCIPAL+payarm.1.!interest*taxrate)||'</td>'   


call lineout tempfile, '<tr><td colspan=7>adjust ARM using: 'interest_adj' after '||months_arm1' months'
call lineout tempfile,' --max yearly-rise='||max_rise||', max total-rise='||max_rise_all||'</td>'


call lineout tempfile,'</table></td>'


/******
call lineout tempfile,'</table>'
call asay left('  LoanA : 'interest15||' ('||goob||') = '||trunc(mpay15),29,' ')||,
        ' ; '||diff_15||' '||,
        trunc(pay15.1.!PRINCIPAL)||' '||trunc(pay15.1.!interest*taxrate)||,
        ' (= '||trunc(diff_15+pay15.1.!PRINCIPAL+pay15.1.!interest*taxrate)||')'   

goob=trunc(costs30_prin)||', '||trunc(costs30)  
call asay left('  LoanB : 'interest30||' ('||goob||') = '||trunc(mpay30),29,' ')||,
        ' ; '||diff_30||' '||,
        trunc(pay30.1.!PRINCIPAL)||' '||trunc(pay30.1.!interest*taxrate) ||,
        ' (= '||trunc(diff_30+pay30.1.!PRINCIPAL+pay30.1.!interest*taxrate)||')'         

goob=trunc(costsarm_prin)||', '||trunc(costsarm)  
call asay left('  LoanARM: 'interest_ARM1||' ('||goob||') = '||trunc(mpayARM),29,' ')||,
        ' ; '||diff_ARM||' '||,
        trunc(payARM.1.!PRINCIPAL)||' '||trunc(payARM.1.!interest*taxrate)||,
        ' (= '||trunc(diff_arm+payarm.1.!PRINCIPAL+payarm.1.!interest*taxrate)||')'          

****/


call lineout tempfile, "<tr><th>Monthly outlay:</th> <td>"||trunc(willpay)|| ' + exta-savings of 'willsave||'</td>'
call lineout tempfile,'</table><pre>'

call asay "Month. Rate  |  prinA+bankA           :  prinB+bankB       : prinARM+bankARM "
call asay ' -----------------------------------------------------------------------'

/*  ----- Compute and Display disposition of mortgage and other monthly payments */




if BANK_BOND=1 then do
   bank30s.0=startinbank-costs30 ; bank15s.0=startinbank-costs15
   bankarms.0=startinbank-costsarm
   mbrates.0=fig_brate(0)
end
else do
   bank30=startinbank-costs30 ; bank15=startinbank-costs15
   bankarm=startinbank-costsarm
end


/* assume you sell the house after months15 (i.e.; in 15 years) */
/* say "payments "mpay15 ',' mpay30', diff15 diff30 'diff_15' 'diff_30 */

do mmx=1 to months15                    


  if (mmx//12)=1 & (mmx>months_arm1) & (ispaidarm=0) then do  /* switch the ARM rate  ? */
     oldb=interest_arm2
     interest_arm2=fig_brate(mmx,interest_adj)
     interest_arm2=min(interest_arm2,oldb+max_rise)  /* limit 1 year jump */
     interest_arm2=max(0.5,interest_arm2)           /* min of .5% */
     interest_arm2=min(interest_arm2,interest_arm1+max_rise_all)  /* overall max */

     mpayarm=fig_mortgage(full_principal-parm,interest_arm2/100,1+months_arm-mmx)
     call copy_payments("PAYARM")
     diff_arm=trunc(willpay-mpayarm)
     tt=trunc(payarm.1.!PRINCIPAL+(payarm.1.!INTEREST*taxrate)+diff_arm)
     call asay '  <em>Computing new ARM rate: '||format(interest_ARM2,,2) '%, w/payment= '||trunc(mpayARM)||' ; '||diff_ARM||' '||,
        trunc(payARM.1.!PRINCIPAL)||' '||trunc(payARM.1.!interest*taxrate)||,
        '(= '||tt||')</em>'

     minterestarm=(interest_arm2/100)/12
  end 

/* compute principal and interest components (which depend on extra payments
   to principal you may have made.
   Adjust remaining principal (if any remains) */

/* say "p15 p30 " format(p15,,2)||' '||format(p30,,2) */

  foo=fig_payment_comp(minterest15,p15,mpay15,full_principal)
  parse var  foo comp_prin_15 comp_int_15

  foo=fig_payment_comp(minterest30,p30,mpay30,full_principal)
  parse var foo comp_prin_30 comp_int_30

/* say "prin15 prin30 "||format(comp_prin_15,,2)||' '||format(comp_prin_30,,2)||" ; "||,
    "int15 int30 "||format(comp_int_15,,2)||' '||format(comp_int_30,,2)  */

  foo=fig_payment_comp(minterestarm,parm,mpayarm,full_principal)
  parse var  foo comp_prin_arm comp_int_arm


  if p15<(full_principal) then do
       p15=p15+comp_prin_15
  end 
  if p30<(full_principal) then p30=p30+comp_prin_30
  if parm<(full_principal) then  parm=parm+comp_prin_arm

  /* say "new p15 p30 "||format(p15,,2)||' '||format(p30,,2) */

/*  bank account grows (or, if negative, opportunity costs grow) */

  mbrate=fig_brate(mmx)         /* bond rate */
  mbrates.mmx=mbrate
  if BANK_BOND=1 then do   /* fixed rate savings bonds purchased each month */
     do jj=0 to mmx
        bank15s.jj=bank15s.jj*mbrates.jj
        bank30s.jj=bank30s.jj*mbrates.jj 
        bankarms.jj=bankarms.jj*mbrates.jj 
     end
     bank15=willsave
     bank30=willsave
     bankarm=willsave
  end 
  else do               /* dynamic savings (one big account effected by single rate */
     bank15=bank15*mbrate+willsave
     bank30=bank30*mbrate+willsave
     bankarm=bankarm*mbrate+willsave
  end

  /* say "Bankint15 bankInt30 "||format(bank15-foo1,,2)||' '||format(bank30-foo2,,2) */


/* check for principal-paid-off */
  if p15>=(full_principal) & ispaid15=0 then do
      ispaid15=1
      arem=full_principal-(p15-comp_prin_15)
      aextra=willpay-(arem+comp_int_15)
      bank15=bank15+aextra
      p15=full_principal
  end      
  if p30>=(full_principal) & ispaid30=0 then do
      ispaid30=1
      arem=full_principal-(p30-comp_prin_30)
      aextra=willpay-(arem+comp_int_30)
      bank30=bank30+aextra
      p30=full_principal
  end      
  if parm>=(full_principal) & ispaidarm=0 then do
      ispaidarm=1
      arem=full_principal-(parm-comp_prin_arm)
      aextra=willpay-(arem+comp_int_arm)
      bankarm=bankarm+aextra
      parm=full_principal
  end      

/* dispose of interest component, and difference between willpay */

   foo=figpay1(p15,bank15,diff_15,comp_int_15,ispaid15)
   parse var foo p15 bank15 int_lost
   totint15=totint15+int_lost

   /* say "new p15 bank15 "||format(p15,,2)||' '||format(bank15,,2) */

   foo=figpay1(p30,bank30,diff_30,comp_int_30,ispaid30)
   parse var foo p30 bank30 int_lost
   totint30=totint30+int_lost

   /* say "new p30 bank30 "||format(p30,,2)||' '||format(bank30,,2) */

   foo=figpay1(parm,bankarm,diff_arm,comp_int_arm,ispaidarm)
   parse var foo parm bankarm int_lost
   totintarm=totintarm+int_lost

   if ispaid15<>2 then do
      if p15>=(full_principal) then do 
         ispaid15=2
         bank15=bank15+(p15-(full_principal))
         p15=full_principal
      end
   end
   if ispaid30<>2 then do
      if p30>=(full_principal) then do 
         ispaid30=2
         bank30=bank30+(p30-(full_principal))
         p30=full_principal
      end
   end
   if ispaidarm<>2 then do
      if parm>=(full_principal) then do 
         ispaidarm=2
         bankarm=bankarm+(parm-(full_principal))
         parm=full_principal
      end
   end


/* time to take out that 2nd loan (say, to pay tuition?) So borrow, either
   by drawing down your bank account or by using a home-equity loan.. */ 

  if mmx=whenequity  then do  
     CALL START_LOAN2
     call asay "Monthly outlay = "||trunc(willpay) ' + ' ||trunc(eqmpay)|| ' = '||trunc(willpay+eqmpay)|| ' + exta-savings of 'willsave
  END

/* period during which homequity loan is being paid? */
  if  (mmx>whenequity) & (mmx<=(whenequity+equity_months)) then do 
     mm2=mm2+1
     call make_eqmpay mm2
  end           /* home equity payment */

  if   ( mmx=(1+whenequity+equity_months)) then do 
    call lineout tempfile,"<b>Home equity payments stop. </b>Monthly outlay reverts to  "willpay|| ' + exta-savings of 'willsave
  end


/* accumulate value of "bonds" */
  if bank_bond=1 then do
      bank15s.mmx=bank15
      bank30s.mmx=bank30
      bankarms.mmx=bankarm
      do jj2=0 to mmx-1   /* sum up value of older bonds */
         bank15=bank15+bank15s.jj2
         bank30=bank30+bank30s.jj2
         bankarm=bankarm+bankarms.jj2
      end
  end

/* write some results for this month */
  call sayline


end 
call lineout tempfile,'</pre><hr><em>After 180 months...</em>'
call lineout tempfile,'<table border=1><tr bgcolor="lightgrey">'
call lineout tempfile,'<th>Loan Plan</th>'
call lineout tempfile,'<th>Total Assets</th>'
call lineout tempfile,'<th>= Principal +  Saved  </th>'
call lineout tempfile,'<th>(Nominal-interest-paid)</th>'

t15=p15+bank15 ; t30=p30+bank30 ; tarm=parm+bankarm
call lineout tempfile,'<tr><td>LoanA </td>'
call lineout tempfile,'<th>'||addcomma(t15)||'</td>'
call lineout tempfile,'<td>'||addcomma(p15)||' + '||addcomma(bank15)||'</td>'
call lineout tempfile,'<td>'||' <tt>('||addcomma(totint15)||')</tt></td>'


call lineout tempfile,'<tr><td>LoanB </td>'
call lineout tempfile,'<th>'||addcomma(t30)||'</td>'
call lineout tempfile,'<td>'||addcomma(p30)||' + '||addcomma(bank30)||'</td>'
call lineout tempfile,'<td>'||' <tt>('||addcomma(totint30)||')</tt></td>'


call lineout tempfile,'<tr><td>LoanARM </td>'
call lineout tempfile,'<th>'||addcomma(tarm)||'</td>'
call lineout tempfile,'<td>'||addcomma(parm)||' + '||addcomma(bankarm)||'</td>'
call lineout tempfile,'<td>'||' <tt>('||addcomma(totintarm)||')</tt></td>'
call lineout tempfile,'</table>'

call lineout tempfile,'</body></html>'
call lineout tempfile

return 'File erase type text/html name ' tempfile




/**********/
/* read parameters */
get_params:

params.=''
do until list=''
   parse var list a1 '&' list
   parse var a1 avar '=' avalue
   vv='!'||strip(translate(avar))
   aval=strip(translate(avalue,' ','+'))
   if vv='!BOND_B1'  then do
      interpret 'aval='||avalue
call pmprintf(' xxx 'aval)
   end
   if vv='!WILLSAVE2'  then do
       if aval='' then aval=0
   end 
   if vv='!WILLPAY'  then do
       if aval='' then aval=0
   end 
   if vv='!OLD_PRINCIPAL'  then do
        if aval='' then aval=0
   end 
   if vv='!WHENEQUITY'  then do
       if aval='' then aval=100000
   end 
   if vv='!INTEREST_EQUITY'  then do
       if aval<>'' & abbrev(aval,'+')=0 then aval='+'||strip(aval)
   end 
   if vv='!INTEREST_ADJ'  then do
       if aval<>'' & abbrev(aval,'+')=0 then aval='+'||strip(aval)
   end 

   params.vv=strip(params.vv||' '||aval)

   params.0=params.0||' '||vv  
end

do until params.0=''
   parse var params.0 a1 params.0
   va1=substr(a1,2)
   foo=value(va1,params.a1)
end



return 0

/*
do until params.0=''
   parse var params.0 a1 params.0
   call lineout tempfile, " a1 " a1 " = " params.a1
end
call lineout tempfile
return 'file erase type text/plain name 'tempfile
*/
/**********/
/* prediction of this months bond rate */
fig_brate:procedure expose bond_b0 bond_b1 bond_b2  tempfile 
parse arg mth,asis
asis=strip(asis)

thisrate=bond_b0 + (mth*bond_b1) + (mth*mth*bond_b2)
if asis='' then do              /* return monthly rate */
  brate=1+(thisrate/1200)
  return brate
end

/* else, adjust and return yearly rate as whole number (i.e.; 3 = 3%) */
tt=asis         /* default is to use "asis" as is ! */
if abbrev(asis,'+')=1 then do
  addto=substr(asis,2)
  tt=thisrate+addto
end 
if abbrev(asis,'*')=1 then do
  mult=substr(asis,2)
  tt=thisrate*mult
end 
return tt

/*
delta=(bondrate_100-bondrate)/100
thisrate=bondrate+(delta*mth) */


/*******/
/* write results for this month */
sayline:

  eqowed15=trunc(max(0,toborrow15-eqp15))
  eqowed30=trunc(max(0,toborrow30-eqp30))
  eqowedarm=trunc(max(0,toborrowarm-eqparm))
  t15=trunc(p15+bank15-eqowed15)
  t30=trunc(p30+bank30-eqowed30)
  tarm=trunc(parm+bankarm-eqowedarm)

  yrrate=(pow(mbrate,12)-1)*100

  aa='+'
  if bank15<0 then aa=''
  if eqowed15>0 then
      w1=trunc(p15)||aa||trunc(bank15)||"-"||eqowed15||'= '||t15
  else
      w1=trunc(p15)||aa||trunc(bank15)||'= '||addcomma(t15)

  aa='+'
  if bank30<0 then aa=''
  if eqowed30>0 then
      w2=trunc(p30)||aa||trunc(bank30)||"-"||eqowed30||'= '||t30
  else
      w2=trunc(p30)||aa||trunc(bank30)||'= '||addcomma(t30)

  aa='+'
  if bankarm<0 then aa=''
  if eqowedarm>0 then
      w3=trunc(parm)||aa||trunc(bankarm)||"-"||eqowedarm||'= '||tarm
  else
      w3=trunc(parm)||aa||trunc(bankarm)||'= '||addcomma(tarm)

  call asay left(mmx||'.',4,' ')||format(yrrate,3,1)||' | '||left(w1,25,' ')||' : '||left(w2,25,' ')||' : '||left(w3,25,' ')

return 1

/*****************/
/* write line to tempfile */
asay:procedure expose tempfile
parse arg a1
call lineout tempfile,a1
return 1

/*****************/
/* Given a monthly mortgage payment and a monthly interest rate,
   and given the currently paid principal and the size of the original loan,
   determine the principal and interest components 
   (these two components MUST sum to the monthly mortgage payment)

parse arg Monthly interest,currently paid principal, monthly payment, original loan size

*/
fig_payment_comp:procedure  expose tempfile
parse arg mint,theprin,apay,princ0
remprin=princ0-theprin
if remprin<=0 then return '0 0'

aint=mint*remprin   /* interest on remaining principal */
aprin=apay-aint     /* mortgate payement, minus interest, is added to principal */
return aprin||' '||aint


/**************/
/* deal with an equity payment */
make_eqmpay:
parse arg mm22

foo=fig_payment_comp(minterest_equity,eqp15,eqmpay15,toborrow15)
parse var foo  ap_15 ai_15
foo=make_eqmpay2(p15,eqp15,bank15,eqmpay15,ap_15,ai_15,toborrow15,costs15_prin)
parse var foo p15 eqp15 bank15


foo=fig_payment_comp(minterest_equity,eqp30,eqmpay30,toborrow30)
parse var foo  ap_30 ai_30
foo=make_eqmpay2(p30,eqp30,bank30,eqmpay30,ap_30,ai_30,toborrow30,costs30_prin)
parse var foo p30 eqp30 bank30
            

foo=fig_payment_comp(minterest_equity,eqparm,eqmpayarm,toborrowarm)
parse var foo  ap_arm ai_arm
foo=make_eqmpay2(parm,eqparm,bankarm,eqmpayarm,ap_arm,ai_arm,toborrowarm,costsarm_prin)
parse var foo parm eqparm bankarm

return 1

/********/
/* equity payment, working proc */
make_eqmpay2:procedure expose  strategy taxrate eqmpay principal tempfile 
parse arg prin_main,aprin,abank,ampay,prin_part,int_part,origprin,costs_prin

if  ampay=0 | aprin>=origprin  then do       /* easy case: nothing owed on home equity loan */
  if strategy=1 then do
     abank=abank+eqmpay    
  end        
  else do
     prin_main=prin_main+eqmpay
  end
  return prin_main||' '||aprin||' '||abank
end

aprin=aprin+prin_part
toadd=eqmpay-ampay
if aprin>origprin then do
   toadd=toadd+(aprin-origprin)
   aprin=origprin
end 

toadd=toadd+ (INT_part*taxrate)
if strategy=1 | aprin>(principal+costs_prin) | toadd<0 then do
   abank=abank + toadd
end
else do
  prin_main=prin_main+ toadd
end
return prin_main||' '||aprin||' '||abank
        

/****************/
/* Start the second loan */
start_loan2:
mm2=0

call lineout tempfile,'<table><tr><th>Home-equity loan taken:</th><td>'
interest_equity=fig_brate(mmx,interest_equity)

minterest_equity=(interest_equity/100)/12

if needborrow=-1 then needborrow=bank30

if needborrow<=bank15 then do
         call asay " <b>Plan A:</b> w/d "||trunc(needborrow)
         bank15=bank15-needborrow
 end
 else do
   if bank15>0 then
      toborrow15=needborrow-bank15
   else
      toborrow15=needborrow
   eqmpay15=fig_mortgage(toborrow15,interest_equity/100,equity_months)
        call copy_payments("PAYEQUITY15")
        call lineout tempfile,'<b>Plan A:</b>'
        call asay "  w/d "||max(0,trunc(bank15))||" & borrow (@ "||format(interest_equity,,2)||'%) '||,
                trunc(toborrow15)|| " w/home-equity payments of "||trunc(eqmpay15)
    bank15=min(0,bank15)
end


if needborrow<=bank30 then do
    call asay "<br><b>Plan B:</b> w/d "||trunc(needborrow)
    bank30=bank30-needborrow
end
else do
   if bank30>0 then
      toborrow30=needborrow-bank30
   else
      toborrow30=needborrow
   eqmpay30=fig_mortgage(toborrow30,interest_equity/100,equity_months)
   call copy_payments("PAYEQUITY30")
    call lineout tempfile,'<br><b>Plan B:</b>'
   call asay " w/d "||max(0,trunc(bank30))||" & borrow (@ "||format(interest_equity,,2)||'%) '||,
        trunc(toborrow30)|| " w/home-equity payments of "||trunc(eqmpay30)
   bank30=min(0,bank30)
end


if needborrow<=bankarm then do
         call asay "<br><b> PlanARM:</b> w/d "||trunc(needborrow)
         bankarm=bankarm-needborrow
 end
 else do
        if bankarm>0 then
           toborrowarm=needborrow-bankarm
        else
           toborrowarm=needborrow
       
        eqmpayarm=fig_mortgage(toborrowARM,interest_equity/100,equity_months)
        call copy_payments("PAYEQUITYARM")
        call lineout tempfile,'<br><b>Plan ARM:</b>'
        call asay "w/d "||max(0,trunc(bankarm))||" & borrow (@ "||format(interest_equity,,2)||'%) '||,
                trunc(toborrowarm)|| " w/home-equity payments of "||trunc(eqmpayarm)

        bankarm=min(bankarm,0)
end

call lineout tempfile,'</td></table>'

eqmpay=max(eqmpay15,eqmpay30,eqmpayarm)
if willsave2<>0 then eqmpay=willsave2

return 1


/******/
/* figure "extra" changes in principal and bank account; return new values of prin and bankacct */
figpay1:procedure expose taxrate principal willpay strategy  tempfile 
parse arg aprin,abank,adiff,ainterest,ispaid

tt=ainterest*taxrate
int_lost=0
if ispaid=1 then do             /* just went over principal */
    abank=abank+(ainterest*taxrate)  /* but not deduction on interest payment */
    int_lost=ainterest-tt
    return aprin||' '||abank||' '||int_lost
end

if ispaid=2 then do             /* well past end of loan, save all the willpay */
    abank=abank+willpay             
    return aprin||' '||abank||' '||int_lost
end

int_lost=ainterest-tt

/* else, still need to pay principal */
if strategy=0 then do               /*pay off loan */
/*  say " adiff " adiff ',' ainterest '=' ainterest*taxrate   '( '||ainterest*(1-taxrate)   */

   if adiff>0 then do
       aprin=aprin+adiff+tt
   end 
   else do              /* monthly payment exceeds cash allocated */
       tt2=adiff+tt
       if tt2>0 then          /* shortfall covered by tax deduction */
           aprin=aprin+ tt2
        else                  /* shortfall must be deducted from bank */
           abank=abank+tt2
   end 
end
else do                 /* build bank account */
   abank=abank+ adiff + tt
end
return aprin||' '||abank||' '||int_lost



/*******/
/* copy payments. to "newname" */
     PAYMENTS.n.!x, x= interest or principal 
             payments.0 = # months
             payments.0.!interest : total interest paid
             payments.0.!principal : total principal paid (should = amt)
*/

copy_payments:
parse arg newname

foo=value(newname||".0",payments.0)
foo=value(newname||".0.!INTEREST",payments.0.!INTEREST)
foo=value(newname||".0.!PRINCIPAL",payments.0.!PRINCIPAL)
do mm=1 to payments.0
  foo=value(newname||"."||mm||".!INTEREST",payments.mm.!INTEREST)
  foo=value(newname||"."||mm||".!PRINCIPAL",payments.mm.!PRINCIPAL)
end
return 0

end /* do */

/*******/
init:

astrategy.0='Quickly pay down principal'
astrategy.1='Build bank account'

ispaid30=0 ; ispaid15=0 ; ispaidarm=0
eqp15=0 ; eqp30=0 ; eqparm=0

toborrowarm=0 ; eqmpayarm=0 
toborrow30=0 ; eqmpay30=0 ; 
toborrow15=0 ; eqmpay15=0 ; 
bank15s.=0 ; bank30s.=0 ; bankarms.=0

bankbond.0='Normal savings account'
bankbond.1='Monthly bond purchase '

totint15=0 ; totint30=0 ; totintarm=0

return 0





/************/
/* ADD COMMAS TO A NUMBER */
addcomma:
parse arg aval,ndec
aval=strip(aval)
parse var aval p1 '.' p2

if aval<0 & aval > -1000 then return aval

if ndec='' then do
   p2=''
end
else do
   p2='.'||left(p2,ndec,'0')
end /* do */

plen=length(p1)
p1new=''
do i=1 to 10000 while plen>3
   p1new=','right(p1,3)||p1new
   p1=delstr(p1,plen-2)
   plen=plen-3
end /* do */

return p1||p1new||p2


/****************************************************************************/
/* Compute payment schedule for a mortgage.
   Return results monthly mortgage payment
   Also sets exposed PAYMENTS.
             PAYMENTS.n.!x, x= interest or principal 
             payments.0 = # months
             payments.0.!interest : total interest paid
             payments.0.!principal : total principal paid (should = amt)

 Arguments:             
    amt: amount of mortgage
    aintr:  interest rate (yearly, but not apr)
    mmonths: length of loan 
*/

fig_mortgage:procedure expose payments.  tempfile 
payments.=''
parse arg amt,aintr,mmonths
mintr=aintr/12


M_P=0.5*AMT/MMONTHS     /* guess at starting monthly principal */

goo=fig_pays(amt,mintr,mmonths,M_P)

step=M_P/100
do forever
  if cum_prin>amt then do
      m_P=m_P-step
      wasmore=1
  end
  else do
      m_P=m_P+step
      wasmore=0
  end
  MPAY =fig_pays(amt,mintr,mmonths,M_P)
  if abs(AMT-CUM_PRIN)/AMT < 0.0000005 then leave

  if CUM_PRIN>AMT then do
      if wasmore=1 then iterate /* same step size */
      step=step/2       /* switched, cut step size in half */
  end
  else do
      if wasmore=0 then iterate /* same step size */
      step=step/2       /* switched, cut step size in half */
  end 
end /* do */
payments.0.!interest=cum_int
payments.0.!principal=cum_prin
payments.0=mmonths
return mpay


/**********/
/* compute mortgage schedule, given 1st month principal */
fig_pays:procedure expose cum_prin cum_int payments.  tempfile 
parse arg newamt,mintr,jmon,m_p
cum_prin=0
cum_int=0
m_i=newamt*mintr
mpay=m_i+m_p
do mm=1 to jmon
    cum_prin=cum_prin+m_p
    cum_int=cum_int+m_i
    newamt=newamt-m_p

    m_i=newamt*mintr    
    m_p=mpay-m_i
    payments.mm.!interest=m_i
    payments.mm.!principal=m_p
end
return MPAY


exit

/******/
/* compute monthly interest rate given yearly rate
   We don't use this, since most calculators assume rate is "monthly" */
fig_mint:procedure
parse arg interest
interest=interest+1
mint=1+((interest-1)/12)
ji=do_pow(mint,12)
step=0.001/12
do forever
  if ji>interest then do
      mint=mint-step
      wasmore=1
  end
  else do
      mint=mint+step
      wasmore=0
  end
  ji=do_pow(mint,12)
  if abs(ji-interest)/interest < 0.000001 then leave

  if ji>interest then do
      if wasmore=1 then iterate /* same step size */
      step=step/2       /* switched, cut step size in half */
  end
  else do
      if wasmore=0 then iterate /* same step size */
      step=step/2       /* switched, cut step size in half */
  end 
end /* do */
return mint-1


