/* OS/2 16bit realtime DART Audio support for Rsynth								 */
/* 													  Samuel Audet <guardia@cam.org>  */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <float.h>
#define INCL_OS2MM
#define INCL_DOS
#include <os2.h>
#include <os2me.h>
#include "proto.h"
#include "getargs.h"
#include "hplay.h"
#include "limits.h"

long samp_rate  = 8000;
long bits = 16;
int quiet = FALSE;
double ampfactor = 1.0f;
ULONG volume = 100;
ULONG deviceID = 0;

/* new stuff */
#define MIX_BUFFER_EOS	1

#define NUM_BUFFERS 8		  /* Number of audio buffer usable */
static ULONG ulMCIBuffers;   /* warning, minimum 2 */

static MCI_AMP_OPEN_PARMS	maop;
static MCI_MIXSETUP_PARMS	mmp;
static MCI_BUFFER_PARMS 	mbp;
static MCI_GENERIC_PARMS	mgp = {0};
static MCI_MIX_BUFFER		MixBuffers[NUM_BUFFERS];
static MCI_SET_PARMS       msp = {0};
static unsigned char *wave, *startwave = NULL, *endwave;
static ULONG sizewave;

static HEV dataplayed;

static void MciError(ULONG ulError)
{
	unsigned char buffer[128];
	ULONG rc;

	rc = mciGetErrorString(ulError, buffer, sizeof(buffer));

	if (rc == MCIERR_SUCCESS)
		winprintf("MCI Error %d: %s",ULONG_LOWD(ulError),buffer);
	else
		winprintf("MCI Error %d: Cannot query error message.",ULONG_LOWD(rc));

	winprintf("  Switching to quiet mode.\n");

	quiet = TRUE;
}

void set_volume(void)
{
  ULONG rc;
  if (maop.usDeviceID)
  {
     if(volume > 100) volume = 100;

     memset(&msp,0,sizeof(msp));
     msp.ulAudio = MCI_SET_AUDIO_ALL;
     msp.ulLevel = volume;

     rc = mciSendCommand(maop.usDeviceID, MCI_SET,
                    MCI_WAIT | MCI_SET_AUDIO | MCI_SET_VOLUME,
                    &msp, 0);

     if ( rc != MCIERR_SUCCESS )
	  {
		  MciError(rc);
		  return(-1);
	  }

  }
}

static LONG APIENTRY MyEvent(ULONG ulStatus, PMCI_MIX_BUFFER PlayedBuffer, ULONG ulFlags)
{
	if(PlayedBuffer->ulFlags == MIX_BUFFER_EOS)
      DosPostEventSem(dataplayed);
	else
	switch(ulFlags)
	{
	case MIX_STREAM_ERROR | MIX_WRITE_COMPLETE:	/* error occur in device */

		if ( ulStatus == ERROR_DEVICE_UNDERRUN)
			/* Write buffers to rekick off the amp mixer. */
			mmp.pmixWrite( mmp.ulMixHandle,
								MixBuffers,
								ulMCIBuffers );
		break;

	case MIX_WRITE_COMPLETE:							/* for playback  */

		if(wave < endwave)
		{
         /* copy new data in the played buffer */
         PlayedBuffer->ulFlags = 0;
         PlayedBuffer->ulBufferLength = mbp.ulBufferSize;
			if(wave + PlayedBuffer->ulBufferLength >= endwave)
			{
				PlayedBuffer->ulFlags = MIX_BUFFER_EOS; /* set end of stream */
				PlayedBuffer->ulBufferLength = (ULONG) (endwave - wave);
			}
         memcpy(PlayedBuffer->pBuffer, wave, PlayedBuffer->ulBufferLength);
			wave += PlayedBuffer->ulBufferLength;

			mmp.pmixWrite( mmp.ulMixHandle,
								PlayedBuffer /* contains new data */,
								1 );
		}

		break;

	} /* end switch */

	return( TRUE );

} /* end MyEvent */

void stop_audio(void)
{
  ULONG rc;
  if (maop.usDeviceID)
  {

	  rc = mciSendCommand( maop.usDeviceID,
								  MCI_STOP,
								  MCI_WAIT,
								  &mgp,
								  0 );

	  if ( rc != MCIERR_SUCCESS )
	  {
		  MciError(rc);
		  return(-1);
	  }

	  DosCloseEventSem(dataplayed);

	  rc = mciSendCommand( maop.usDeviceID,
								  MCI_BUFFER,
								  MCI_WAIT | MCI_DEALLOCATE_MEMORY,
								  &mbp,
								  0 );

	  if ( rc != MCIERR_SUCCESS )
	  {
		  MciError(rc);
		  return(-1);
	  }

	  rc = mciSendCommand( maop.usDeviceID,
								  MCI_CLOSE,
								  MCI_WAIT ,
								  &mgp,
								  0 );

     maop.usDeviceID = 0;

	  if ( rc != MCIERR_SUCCESS )
	  {
		  MciError(rc);
		  return(-1);
	  }
  }
}

int audio_init(int argc, char **argv)
{
 int rate = 8;

 argc = getargs("Audio Initialization", argc, argv,
					 "r", "%d", &rate,   "Sample Rate in kHz - 8, 11, 22, or 44",
					 "Q", NULL, &quiet,   "Quiet - No sound ouput",
					 "b", "%d", &bits, "8 or 16 bit sample playback",
					 "V", "%u", &volume, "Volume in %",
					 "a", "%lg", &ampfactor, "Amplification factor",
					 "D", "%u", &deviceID, "Sound Card Device ID (0 = default device)",
					 NULL);
 if (help_only)
  return (argc);

 switch(rate)
	 {
	  case  8 : samp_rate =  8000;
					break;
	  case 11 : samp_rate = 11025;
					break;
	  case 22 : samp_rate = 22050;
					break;
	  case 44 : samp_rate = 44100;
					break;
	  default : samp_rate =  8000;
	 }

  if (bits != 8)
	  bits = 16;

  return (argc);
}

void start_audio(void)
{
  ULONG rc;

  if(!quiet)
  {
	  /* open the mixer device */
	  memset (&maop, 0, sizeof(maop));
	  maop.usDeviceID = 0;
     maop.pszDeviceType = (PSZ) MAKEULONG(MCI_DEVTYPE_AUDIO_AMPMIX, deviceID);

	  rc = mciSendCommand(0,
								 MCI_OPEN,
								 MCI_WAIT | MCI_OPEN_TYPE_ID | MCI_OPEN_SHAREABLE,
								 &maop,
								 0);

	  if (rc != MCIERR_SUCCESS)
	  {
		  MciError(rc);
		  return(-1);
	  }

	  /* Set the MCI_MIXSETUP_PARMS data structure to match the audio stream. */

	  memset(&mmp, 0, sizeof(mmp));

	  mmp.ulBitsPerSample = bits;
	  mmp.ulFormatTag = MCI_WAVE_FORMAT_PCM;
	  mmp.ulSamplesPerSec = samp_rate;
	  mmp.ulChannels = 1;

	  /* Setup the mixer for playback of wave data	*/
	  mmp.ulFormatMode = MCI_PLAY;
	  mmp.ulDeviceType = MCI_DEVTYPE_WAVEFORM_AUDIO;
	  mmp.pmixEvent	 = MyEvent;

	  rc = mciSendCommand( maop.usDeviceID,
								  MCI_MIXSETUP,
								  MCI_WAIT | MCI_MIXSETUP_INIT,
								  &mmp,
								  0 );

	  if ( rc != MCIERR_SUCCESS )
	  {
		  MciError(rc);
		  return(-1);
	  }

	  /* Set up the BufferParms data structure and allocate
		* device buffers from the Amp-Mixer  */

	  ulMCIBuffers = NUM_BUFFERS;
	  mbp.ulNumBuffers = ulMCIBuffers;
	  mbp.ulBufferSize = mmp.ulBufferSize;
	  memset(MixBuffers, 0, sizeof(MixBuffers[0])*NUM_BUFFERS);
	  mbp.pBufList = MixBuffers;

	  rc = mciSendCommand( maop.usDeviceID,
								  MCI_BUFFER,
								  MCI_WAIT | MCI_ALLOCATE_MEMORY,
								  (PVOID) &mbp,
								  0 );

	  if ( ULONG_LOWD(rc) != MCIERR_SUCCESS )
	  {
		  MciError(rc);
		  return(-1);
	  }

	  ulMCIBuffers = mbp.ulNumBuffers; /* never know! */

	  set_volume();

	  /* Create a semaphore to know when data has been played by the DART thread */
     DosCreateEventSem(NULL,&dataplayed,0,TRUE);
  }
 _control87(EM_INVALID | EM_DENORMAL | EM_ZERODIVIDE | EM_OVERFLOW | EM_UNDERFLOW | EM_INEXACT, MCW_EM);
}

void end_audio(void)
{
  ULONG rc;

  if(maop.usDeviceID)
  {
	  if(DosWaitEventSem(dataplayed, -1)) return;
	  DosCloseEventSem(dataplayed);

	  rc = mciSendCommand( maop.usDeviceID,
								  MCI_BUFFER,
								  MCI_WAIT | MCI_DEALLOCATE_MEMORY,
								  &mbp,
								  0 );

	  if ( rc != MCIERR_SUCCESS )
	  {
		  MciError(rc);
		  return(-1);
	  }

	  rc = mciSendCommand( maop.usDeviceID,
								  MCI_CLOSE,
								  MCI_WAIT ,
								  &mgp,
								  0 );

     maop.usDeviceID = 0;

	  if ( rc != MCIERR_SUCCESS )
	  {
		  MciError(rc);
		  return(-1);
	  }

  }
}

void audio_play(int n, short *data, char *sentence, char *phone)
{
	extern int verbose, text;
	char temp[256];
	ULONG resetcount;
	ULONG rc;
    int i;

	if (quiet)
	{
		if(text) winprintf("%s",sentence);
		if(verbose) winprintf("\n%s\n",phone);
		free(sentence);
		free(phone);
		return;
	}

	if(ampfactor != 1.0f)
	{
		short *current = data,
				*end = data + n;

		while(current < end)
		{
			if (*current < 0)
			{
				if ((*current * ampfactor) > SHRT_MIN)
					*current = (short) (*current * ampfactor);
				else
					*current = SHRT_MIN;
			}
			else if (*current > 0)
			{
				if ((*current * ampfactor) < SHRT_MAX)
					*current = (short) (*current * ampfactor);
				else
					*current = SHRT_MAX;
			}
			current++;
		}
	}

	if(DosWaitEventSem(dataplayed, -1)) return;
	DosResetEventSem(dataplayed,&resetcount);

	if(text) winprintf("%s",sentence);
	if(verbose) winprintf("\n%s\n",phone);
	free(sentence);
	free(phone);

	free(startwave);

   if (bits == 8)
	{
      sizewave = n * sizeof(char);
      startwave = (char *) malloc(sizewave);
      if (startwave)
      {
         char *p = startwave;
         char *e = p + n;
         while (p < e)
            *p++ = (char) (*data++ / 256 + 128);
      }
      else
      {
         winprintf("Insufficient memory for Play Buffer\n\n");
         return;
      }
	}
	else /* if bits == 16 */
	{
		sizewave = n * sizeof(short);
		startwave = (char *) malloc(sizewave);
        if (startwave)
		{
            short *p = startwave;
            short *e = p + n;
			while (p < e)
            *p++ = *data++;
		}
		else
		{
			winprintf("Insufficient memory for Play Buffer\n\n");
			return;
		}
	}

	endwave = startwave + sizewave;
	wave = startwave;

	/* Fill all device buffers with data from the wave stream.	*/

	for(i = 0; i < ulMCIBuffers; i++)
	{
		if(wave < endwave)
		{
			MixBuffers[i].ulFlags = 0;
			MixBuffers[i].ulBufferLength = mbp.ulBufferSize;
			if(wave + MixBuffers[i].ulBufferLength >= endwave)
			{
				MixBuffers[i].ulFlags = MIX_BUFFER_EOS; /* set end of stream */
				MixBuffers[i].ulBufferLength = (ULONG) (endwave - wave);
			}
			memcpy(MixBuffers[i].pBuffer, wave, MixBuffers[i].ulBufferLength);
			wave += MixBuffers[i].ulBufferLength;
		}
      else
         break;
	}

	/* Write buffers to kick off the amp mixer. */
	rc = mmp.pmixWrite( mmp.ulMixHandle,
							  MixBuffers,
                       i );

	if ( rc != MCIERR_SUCCESS )
	{
		MciError(rc);
		return(-1);
	}

}
