//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// Generic value prompt implementation text/windowed
//
// Author: J. van Wijk
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license
// JvW  19-08-2005 Initial version, split off from TXCON.C

#include <txlib.h>
#include <txwpriv.h>                            // private window interface


#if defined (USEWINDOWING)
static  char txMsgPromTitle[] = " Request for additional input ";
#endif

/*****************************************************************************/
// Prompt for user input with a message and read the input
/*****************************************************************************/
BOOL TxPrompt                                   // RET   value not empty
(
   ULONG               helpid,                  // IN    helpid confirmation
   short               length,                  // IN    max length of value
   char               *value,                   // OUT   entry field value
   char               *fmt,                     // IN    format string (printf)
   ...                                          // IN    variable arguments
)
{
   char                text[512];
   va_list             varargs;
   char              **mText;                   // formatted text for display
   int                 ll;                      // real max line length
   int                 lines;                   // nr of lines
   DEVICE_STATE        sa = TxScreenState(DEVICE_TEST);

   if ((ll = TxScreenCols() -12) < 20)          // ScreenCols will be 0 if
   {                                            // output is redirected!
      ll = 20;
   }
   if ((txwa->desktop != NULL) && (!txwa->pedantic))
   {
      TxScreenState(DEVICE_OFF);                // no regular text if windowed
   }                                            // and not 'pedantic mode'
   else
   {
      TxScreenState(DEVICE_ON);                 // force screen on for input
   }
   va_start(varargs, fmt);
   vsprintf( text, fmt, varargs);               // format the message

   mText = txString2Text( text, &ll, &lines);   // convert to multi-line
   TxPrint( "\n");                              // force empty line before
   TxShowTxt(  mText);                          // display in scrollbuffer
   txFreeText( mText);                          // free the text memory
   TxPrint("\n");

   #if defined (USEWINDOWING)
   if (txwa->desktop != NULL)                   // there is a desktop
   {
      TxStripAnsiCodes( text);                  // remove colors for popup
      if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                        text, txMsgPromTitle, helpid,
                        TXPB_MOVEABLE | TXPB_HCENTER,
                        length, value)
                        == TXDID_CANCEL)
      {
         strcpy( value, "");                    // wipe value, signal cancel
         TxCancelAbort();                       // avoid aborting on Escape ...
      }
   }
   else                                         // use non-windowed interface
   #endif                                       // USEWINDOWING
   {
      TXLN          defval;

      memset(  defval, 0, TXMAXLN);
      strncpy( defval, value, length);
      TxPrint( "\nDefault on <Enter>: '%s'\n", defval);

      TxPrint(   "or different value: ");
      memset(  value, 0, length);
      fgets(   value, length, stdin);           // get the value
      TxRepl(  value, '\n', 0);
      TxPrint("\n");
      if (strlen( value) == 0)                  // empty ==> use default
      {
         strcpy( value, defval);
      }
   }
   TxScreenState(sa);                           // restore screen state

   return(strlen(value) != 0);
}                                               // end 'TxPrompt'
/*---------------------------------------------------------------------------*/

