//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// CRC32-C implementation, using on-the-fly created lookup-table
//         Compatible with the Apple APFS  filename-hash
//
// Author: J. van Wijk
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license

#include <txlib.h>                              // TxLib interface

#define TXCRC32C_TSIZE    256                   // table size 2^8

static ULONG   txcr32c_table[ TXCRC32C_TSIZE];
static ULONG   txcr32c_current_poly = 0;        // Polynome value for current table


/*****************************************************************************/
// Calculate the crc32c over given byte array (APFS name-hash compatible)
/*****************************************************************************/
ULONG TxCalcCrc32C
(
   ULONG               poly,                    // IN    Polynome to use
   ULONG               initial,                 // IN    Initial value
   ULONG               length,                  // IN    Length of the data
   BYTE               *data                     // IN    Data to calculate CRC for
)
{
   ULONG               rc = initial;            // function return
   ULONG               i;

   ENTER();
   TRACES(("poly:0x%8.8x initial:0x%8.8x length:%u data: %p\n", poly, initial, length, data));

   if (poly != txcr32c_current_poly)            // table not for this polynome
   {
      ULONG            r;
      ULONG            b;

      txcr32c_current_poly = poly;

      poly = ((poly << 16) & 0xFFFF0000) | ((poly >> 16) & 0x0000FFFF);
      poly = ((poly <<  8) & 0xFF00FF00) | ((poly >>  8) & 0x00FF00FF);
      poly = ((poly <<  4) & 0xF0F0F0F0) | ((poly >>  4) & 0x0F0F0F0F);
      poly = ((poly <<  2) & 0xCCCCCCCC) | ((poly >>  2) & 0x33333333);
      poly = ((poly <<  1) & 0xAAAAAAAA) | ((poly >>  1) & 0x55555555);

      TRACES(("Create CRC lookup-table. morphed poly: 0x%8.8x\n", poly));

      for (i = 0; i < TXCRC32C_TSIZE; i++)
      {
         r = i;
         for (b = 0; b < 8; b++)
         {
            if (r & 1)
            {
               r = (r >> 1) ^ poly;
            }
            else
            {
               r = (r >> 1);
            }
         }
         txcr32c_table[ i] = r;
      }
      #if defined (DUMP)
         if (TxTrLevel > 100)
         {
            DEVICE_STATE tss = TxScreenState(DEVICE_TEST);
            if (TxTrLogOnly)   TxScreenState(DEVICE_OFF);
            for (i = 0; i < 16; i++)
            {
               for (b = 0; b < 16; b++)
               {
                  TxPrint( " %8.8x", txcr32c_table[ (16*i) + b]);
               }
               TxPrint( "\n");
            }
            if (TxTrLogOnly)   TxScreenState(tss);
         }
      #endif
   }

   for (i = 0; i < length; i++)                   // calculate the CRC
   {
      rc = txcr32c_table[ data[ i] ^ (rc & 0xFF)] ^ (rc >> 8);
   }
   RETURN (rc);
}                                               // end 'TxCalcCrc32C'
/*---------------------------------------------------------------------------*/

