//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TX formatting function for mixed ASCII/HEX strings
// file-logging facilities
//
// Author: J. van Wijk
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license
// JvW  24-07-2005 Initial version, split off from TXCON.C

#include <txlib.h>


#define               ST_SKIPW   0              // skip whitespace
#define               ST_HEX_0   1              // first HEX digit
#define               ST_HEX_1   2              // second HEX digit
#define               ST_ASCII   3              // copy plain ASCII
#define               ST_UNICO   4              // generate UNICODE from ASCII
/*****************************************************************************/
// Translate Mixed Ascii / Hex string specification to length + buffer
/*****************************************************************************/
USHORT TxFormatMixedStr                         // RET   length of buffer
(
   char               *data,                    // IN    mixed string
   char               *buf                      // OUT   formatted buffer
)
{
   char               *s;
   int                 c;
   USHORT              i;
   char                hex[3];
   int                 state = ST_SKIPW;        // parsing state, ascii/hex

   hex[2] = '\0';
   for (s=data, i=0; *s; s++)
   {
//    TxPrint("\nParse state %u, char: '%c' for buffer pos: %u", state, *s, i);
      switch (state)
      {
         case ST_SKIPW:
            switch (*s)
            {
               case TXk_SPACE:
                  break;

               default:
                  s--;                          // evaluate again in new state
                  state = ST_ASCII;
                  break;
            }
            break;

         case ST_HEX_0:
            switch (*s)
            {
               case '\'':
                  state = ST_ASCII;
                  break;

               case '0': case '1': case '2': case '3': case '4':
               case '5': case '6': case '7': case '8': case '9':
               case 'a': case 'b': case 'c': case 'd': case 'e': case 'f':
               case 'A': case 'B': case 'C': case 'D': case 'E': case 'F':
                  hex[0] = *s;
                  state = ST_HEX_1;
                  break;

               case TXk_SPACE:
               case '-':
               case '}':
                  break;

               default:
                  TxPrint("\nInvalid hex char in mixed string at char: %c", *s);
                  break;
            }
            break;

         case ST_HEX_1:
            switch (*s)
            {
               case '\'':
                  TxPrint("\nSingle hex char in mixed string at char: %c", *s);
                  state = ST_ASCII;             // error, single HEX char
                  break;

               case '0': case '1': case '2': case '3': case '4':
               case '5': case '6': case '7': case '8': case '9':
               case 'a': case 'b': case 'c': case 'd': case 'e': case 'f':
               case 'A': case 'B': case 'C': case 'D': case 'E': case 'F':
                  hex[1] = *s;
                  if (sscanf( hex, "%X", &c) != 0)
                  {
                     buf[i++] = (char) c;
                  }
                  state = ST_HEX_0;
                  break;

               case TXk_SPACE:
               case '-':
               case '}':
                  break;

               default:
                  TxPrint("\nInvalid hex char in mixed string at char: %c", *s);
                  break;
            }
            break;

         case ST_UNICO:
            switch (*s)
            {
               case '"':
                  state = ST_ASCII;
                  break;

               default:
                  buf[i++] = *s;
                  buf[i++] = 0;
                  break;
            }
            break;

         default:
            switch (*s)
            {
               case '\'':                       // to HEX in mixed string
               case '{':                        // to HEX for UUID format
                  state = ST_HEX_0;
                  break;

               case '"':                        // to UNICODE
                  state = ST_UNICO;
                  break;

               default:                         // just copy ASCII
                  buf[i++] = *s;
                  break;
            }
            break;
      }
   }
   return (i);
}                                               // end 'TxFormatMixedStr'
/*---------------------------------------------------------------------------*/

