//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TX library anchor and init/terminate functions
// file-logging facilities
//
// Author: J. van Wijk
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license
// JvW  24-07-2005 Initial version, split off from TXCON.C

#include <txlib.h>
#include <txwpriv.h>                            // private window interface

// Needs to be in a non-windowing component for the OEM versions
TXW_ANCHOR txwanchor =
{
   #if   defined (DEV32)
      NULL,                                     // Thread Info Block
      NULL,                                     // Process Info Block
      NULL,                                     // dynamic entrypoints
      PT_WINDOWABLEVIO,                         // sessiontype default (VIO/FS)
      TX_CHECKED,                               // startup MousAction,  assume ON
      TX_CHECKED,                               // startup DragTextMark assume ON
      0,                                        // sysmouse setting 0 = AUTO (both)
      FALSE,                                    // KBD reader thread kill
   #elif defined (WIN32)
      0,                                        // sysmouse setting 0 = disable QE
   #elif defined (LINUX)
      FALSE,                                    // X selection CLIPBOARD (not PRIMARY)
   #elif defined (DOS32)
      NULL,                                     // DPMI parameter  ptr,  512
      NULL,                                     // DPMI parameter  ptr,  512
      0,
      0,
   #endif
   0,                                           // set by -c switch, or Init
   76,                                          // EF autoInsert on large fields
   FALSE,                                       // HIDE hidden files/dir in FileDialogs
   TRUE,                                        // global insert-mode
   TRUE,                                        // typeahead enabled
   FALSE,                                       // reset terminal OFF
   TXW_HIST_PLAIN,                              // no filter, no unique history
   TXW_ARROW_STD,                               // stdandard arrow handling
   0,                                           // first auto assigned id
   0,                                           // std radix bits, all decimal
   0, 0,                                        // saved cursor position
   {0,0,79,24},                                 // default 80x25 screen
   0,                                           // no automenu header ID
   0, 0, 0,                                     // start pos and flags for drag
   TRUE,                                        // mouse enabled
   TRUE,                                        // RMB click simulates <F10> for menu
   FALSE,                                       // pedantic mode
   FALSE,                                       // Menu style MOVE to next dropdown
   TXHWND_NULL,                                 // no mouse capture
   TXHWND_DESKTOP,                              // default modality
   NULL,                                        // focus window
   NULL,                                        // first window in chain
   NULL,                                        // last window in chain
   0,                                           // actual window count
   NULL,                                        // SBview for quick status
   TXHWND_NULL,                                 // Entryfield marked as cmdline
   TXHWND_NULL,                                 // Canvas marked as main menu
   TRUE,                                        // show default status
   0,                                           // codepage, std NOT 437 compatible
   NULL,                                        // system accelerator table
   ""                                           // list match string value
};

TXW_ANCHOR *txwa = &txwanchor;                  // private anchor (HAB)


static char *TxLibSwitchhelp[] =
{
   "",
   "TX library / user-interface specific switches:",
   "==============================================",
   "",
#if defined (DUMP)
   " -123[t][s][l][r][dxx] = trace level 123, [t]stamp; [s]creen, No [l]ogging",
   "                         [r]e-open, [d]elay program xx ms per trace line",
#endif
   "",
   " -7            = use 7-bit ASCII only (no 'graphic' or drawing characters)",
   " -7-           = use 8-bit ASCII incl graphic/drawing characters (default)",
   " -$            = do variable/expression substitution in commands (default)",
   " -$-           = no variable/expression substitution   (see 'SET var off')",
   " -a            = switch off usage of ANSI escape characters for color",
#if defined (USEWINDOWING)
   " -color:value  = output colors, 0..7 add any: 0=std 1=invert 2=bright 4=blue",
   " -C:[2|88|256] = Force number of colors in palette to 1, 2, 8, 16, 88 or 256",
#endif
   " -dir:path     = Default data directory for File Open/Save and related cmds",
   " -E:[c|i|q]    = default error strategy Confirm, Ignore or Quit the command",
   " -f            = Use a border frame on output-window and desktop    (default)",
   " -f-           = Suppress border frames where possible, allowing more lines",
#if defined (DOS32)
   " -F            = Do auto-Fail critical errors, no popup   (default, DOS only)",
   " -F-           = No auto-Fail critical errors, use Retry pop-up    (DOS only)",
#endif
   " -insert:n     = INSERT mode: 0=OFF, 1=ON, 2..N = ON for entryfields size > N",
#if defined (DOS32)
   " -keyb:cc      = foreign keyboard support using 'key\cc.kl' file   (DOS only)",
#endif
   " -l:logfile    = start logging immediately to 7bit ascii file 'logfile.log'",
   "                 (supporting ~1..9 date/time substitution, see cmd 'log -?')",
   " -l            = present file-save dialog at startup, start logging to file",
   " -l-           = do NOT present dialog at startup to start logging   (default)",
   " -logauto      = automatically number logfiles when started from dialog",
   " -logdate      = automatically postfix logfilename with a 'YYYYMMDD' date",
#if defined (USEWINDOWING)
   " -menu         = startup with the menu active, default menu selected (default)",
   " -menu-        = do not start the menu at startup, start on the commandline",
   " -menu:letter  = startup with menu dropdown indicated by 'letter' activated",
   " -M:[0 .. 5]   = menu-style: 0=AutoDrop, none of the values added up (default)",
   "                             1=no-submenu-open-on-right-arrow, open next menu",
   "                             2=No-auto-drop on open, or on move to next menu",
   "                             4=Delayed-drop-on-F10-open when auto-drop is ON",
#endif
   " -mouse        = Program specific mouse-handling, click, drag, mark   (default)",
   " -mouse-       = No DFSee specific mouse-handling, disable mouse completely",
   "                 (may allow the OS environment to use native mouse operations)",
   " -O:s|q|n|v|m  = default output verbosity Silent, Quiet, Normal, Verbose, Max",
   " -refresh      = Refresh whole screen at every update of the statusline",
   " -reset        = resets screen/terminal on exit (UNIX: reset, others: CLS)",
#if defined (USEWINDOWING)
   " -scheme:name  = window scheme: grey|m3d|nobl|cmdr|half|full|white|black|dfsee",
#endif
#if defined (UNIX)
   " -screen:WCxR  = explicit size; Cols x Rows, like W80x25 or W120x39 (Linux/Mac)",
#endif
   " -sigint-      = Use Ctrl-C for the 'Copy to Clipboard' function      (default)",
   "                 (abort running operations with the <Esc> key instead)",
#if defined   (WIN32)
   " -sigint       = Let Ctrl-C exit program  (may require 2 quick Ctrl-C presses!)",
#else
   " -sigint       = Let Ctrl-C exit the program immediately   (aka emergency exit)",
#endif
#if defined (USEWINDOWING)
   " -style:value  = line style, 0..3: 0=std-double 1=3d 2=halfblock 3=fullblock",
#endif
#if defined (DEV32)
   " -sysmouse:n   = Manage OS/2 'Mouse Actions' and 'DragText->Mouse Marking' menus",
   "           0   = Disable std OS/2 + Dragtext items until program exit (default)",
   "           1   = Disable only std OS/2 item, leaving DragText fully operational",
   "           2   = Leave std OS/2 and DragText menu items under user control",
#endif
#if defined (WIN32)
   " -sysmouse:n   = Manage automatic Windows console 'Quick Edit Mode' handling",
   "           0   = Disable Windows 'Quick Edit Mode' for this session   (default)",
   "           1   = Leave Windows 'Quick Edit Mode' under user control",
#endif
#if defined (USEWINDOWING)
 #if defined (WIN32)
   " -W            = Force scrollbar/window-resize check, even on Win10 and later",
   " -W-           = NO startup scrollbar/window-resize check; default on Win10",
 #endif
#endif
   "",
   NULL
};



/*****************************************************************************/
// Initialize TX-libary, after parsing EXE parameters and switches
/*****************************************************************************/
void TxLibraryInit
(
   BOOL                screenOff                // IN    start with screen off
)
{
   ENTER();

   #if defined (DEV32)
      if (DosGetInfoBlocks( &txwa->tib, &txwa->pib) == NO_ERROR)
      {
         TxInitDllApiOS2( &txwa->api);          // Init dynamic APIs and morph to PM

         #if defined (USEWINDOWING)
            txwa->sysmouse = TxaExeSwitchNum( TXA_O_SMOUSE, NULL, 0); // desired sysmouse handling

            if (txwa->sysmouse < 2)             // manage 'Mouse Actions' ?
            {
               txwa->smMouseAction  = TxSysMenuMouseActions(  TX_UNCHECKED);
            }
            else                                // just remember startup-state
            {
               txwa->smMouseAction  = TxSysMenuMouseActions(  TX_QUERY);
            }

            if (txwa->sysmouse < 1)             // manage 'DragText Marking' ?
            {
               txwa->smDragTextMark = TxDragTextMouseMarking( TX_UNCHECKED);
            }
            else                                // just remember startup-state
            {
               txwa->smDragTextMark = TxDragTextMouseMarking( TX_QUERY);
            }
         #endif
      }
      else
      {
         TRACES(("DosGetInfoBlocks failed!\n"));
      }
   #endif

   TxTmrInitializeBase();                       // set nanosecond timer base to 0

   if (screenOff)
   {
      TxScreenState(DEVICE_OFF);
      TxSetAnsiMode( A_OFF);
   }
   TxFsSetCritErrHandler( TRUE);
   if (TxaExeSwitchSet( TXA_O_INSERT))          // (auto) insert mode set
   {
      txwa->autoInsert = TxaExeSwitchNum( TXA_O_INSERT, NULL, 76); // Set, default 76 (all except cmdline)
   }
   if (TxaExeSwitchUnSet( TXA_O_MOUSE))         // don't use the mouse
   {
      txwa->useMouse = FALSE;
   }
   if (TxaExeSwitch( 'p'))                      // pedantic warnings/errors
   {
      txwa->pedantic = TRUE;
   }
   if (TxaExeSwitch(TXA_O_RESET))
   {
      txwa->resetterm = TRUE;
   }
   #if defined (DOS32)
      if (TxaExeSwitchSet( TXA_O_KEYB))         // keyb switch
      {
         TxSetNlsKeyboard( TxaExeSwitchStr( TXA_O_KEYB, "", "us"), "");
      }
      txwa->dpmi1 = txDpmiAlloc( 1, 512, &(txwa->dpmi1Selector));
      txwa->dpmi2 = txDpmiAlloc( 1, 512, &(txwa->dpmi2Selector));
   #endif
   TxPrintInit( TxaExeSwitch('a'), TxaExeSwitch('7')); // Use full 8-bit on all, by default

   #if defined (USEWINDOWING)
      TxInputInitialize();                      // Init windowed KBD/MOU input
   #endif

   TxsScriptInit();                             // Init Script (help)

   VRETURN ();
}                                               // end 'TxLibraryInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Terminate TX-libary
/*****************************************************************************/
void TxLibraryExit
(
   void
)
{
   #if defined (DOS32)
      txDpmiFree( txwa->dpmi1Selector);
      txDpmiFree( txwa->dpmi2Selector);
   #endif

   #if defined (DEV32)
      #if defined (USEWINDOWING)
         if (txwa->sysmouse < 2)                // manage 'Mouse Actions' ?
         {
            (void) TxSysMenuMouseActions(  txwa->smMouseAction);
         }
         if (txwa->sysmouse < 1)                // manage 'DragText Marking' ?
         {
            (void) TxDragTextMouseMarking( txwa->smDragTextMark);
         }
      #endif
      TxTermDllApiOS2();                        // terminate Morph-to-PM
   #endif

   TxAppendToLogFile("", TRUE);                 // stop logging/tracing
   TxPrintTerminate();                          // stop printf redirection
   TxFsSetCritErrHandler( FALSE);

   #if defined (USEWINDOWING)
     TxInputTerminate();                        // stop windowed KBD/MOU input
   #endif

   TxScreenTerminate();

   if (txwa->resetterm)                         // Reset of terminal/screen at end
   {
      #if defined (UNIX)
         (void) system( "reset");
      #else
         (void) system( "cls");
      #endif
   }
   else                                         // OSX and Linux, re-enable echo
   {
      #if defined (UNIX)
         (void) system( "stty echo");
      #endif
   }
}                                               // end 'TxLibraryExit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get array of strings containing the TxLib specific startup switches help
/*****************************************************************************/
char **TxGetSwitchhelp
(
   void
)
{
   return (TxLibSwitchhelp);
}                                               // end 'TxGetSwitchhelp'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get setting for the FileDialog ShowHidden feature
/*****************************************************************************/
BOOL TxGetFdlgShowHidden                        // RET   FileDialog SHOW hidden
(
   void
)
{
   return( txwa->dialogShowHidden);
}                                               // end 'TxGetFdlgShowHidden'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get setting for the RMB2MENU feature
/*****************************************************************************/
BOOL TxGetRmb2MenuKey                           // RET   Translate RMB to menu key
(
   void
)
{
   return( txwa->rmb2Menu);
}                                               // end 'TxGetRmb2MenuKey'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Terminate TxPrint environment
/*****************************************************************************/
void TxPrintTerminate
(
   void
)
{
   //- nothing to do yet ...
}                                               // end 'TxPrintTerminate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialise Ansi environment
/*****************************************************************************/
void TxPrintInit
(
   BOOL                ansiOff,                 // IN    don't use ANSI colors
   BOOL                ascii7                   // IN    Convert to 7-bit ASCII
)
{
   #if defined (DOS32)
      static BOOL      force_ansi_probe = FALSE;
   #endif

   if (ansiOff)
   {
      TxSetAnsiMode( A_OFF);                    // so do not use it
   }
   else
   {
      TxSetAnsiMode( A_ON);                     // Allow ANSI to start with

      #if   defined (DOS32)                     // but conditionally for DOS
      if ((TxaExeSwitchUnSet('w')) ||           // when non-windowed to avoid
          (force_ansi_probe))                   // on-screen garbage strings
      {                                         // if ANSI.SYS not loaded
         short         col = TxCursorCol();

         printf("%s", CNN);                     // Reset color attribute
         fflush( stdout);                       // make sure it is done!
         if (col != TxCursorCol())              // cursor has moved!
         {                                      // ANSI not working
            TxSetAnsiMode( A_OFF);              // so do not use it
            printf( "    ");            // try to undo the probe
         }
      }
      else
      {
         force_ansi_probe = TRUE;               // always probe after first
      }                                         // time (non EXE startup)
      #endif
   }
   TRACES(( "ASCII7 set 2: %s\n", (ascii7) ? "TRUE" : "FALSE"));
   TxSetAscii7Mode( ascii7);

   TxClaimKeyboard();

   TxSetBrightBgMode( TRUE);                    // no blinking, use bright BG
}                                               // end 'TxPrintInit'
/*---------------------------------------------------------------------------*/

#if defined (TXALLOC_MACROS)
#else

/*****************************************************************************/
// Allocate and clear memory, compatible with standard 'calloc'
/*****************************************************************************/
void *TxAllocMemory
(
   size_t              number,                  // IN    number of elements
   size_t              size                     // IN    size of one element
)
{
   void               *rc = calloc( number, size);

   TRHEAP( 500);                             // heapcheck on high trace levels
   TRACES(("TxAllocMem:%6u items of size:%6u bytes, to: %p\n", number, size, rc));

   return (rc);
}                                               // end 'TxAllocMemory'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Free memory previously allocated with TxAllocMem (or malloc/calloc)
/*****************************************************************************/
void TxFreeMemory
(
   void               *ptr                      // IN    ptr to allocated memory
)
{
   if (ptr != NULL)
   {
      TRHEAP( 700);                             // heapcheck on high trace levels
      TRACES(("TxFreeMem: %p\n", ptr));
      free( ptr);
   }
   else
   {
      TRLEVX(700,("TxFreeMem: NULL-POINTER!\n"));
   }
}                                               // end 'TxFreeMemory'
/*---------------------------------------------------------------------------*/
#endif


#if defined (USEWINDOWING)
 #if defined (LINUX)
/*****************************************************************************/
// Get value of 'xclip uses X-selection CLIPBOARD' setting
/*****************************************************************************/
BOOL TxGetXselClipboard                         // RET    xSelClip setting
(
   void
)
{
   return (txwa->xSelClip);
}                                               // end 'TxGetXselClipboard'
/*---------------------------------------------------------------------------*/
 #endif
#endif
