//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TxLib filesystem functions, make std and 8.3 paths
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license

#include <txlib.h>                              // TxLib interface

#include <sys/stat.h>                           // for low level stuff


/*****************************************************************************/
// Create path in a recursive fashion
/*****************************************************************************/
ULONG TxMakePath                                // RET   result
(
   char               *filename                 // IN    relative/absolute path
)
{
   ULONG               rc = NO_ERROR;           // return value
   #if defined (WIN32)
      ULONG            ulAttr;
   #elif defined (DOS32)
      unsigned         usAttr;                  // attributes (dos.h)
   #elif defined (UNIX)
      int              stat_rc;
      USHORT           st_mode = 0;
      struct stat      f;
      #if !defined (DARWIN)
         struct stat64 f64;
      #endif
   #else
      FILESTATUS3      fstat;
   #endif
   char               *p = NULL;

   ENTER();
   TRACES(("filename: '%s'\n", filename));

   #if defined (WIN32)
      ulAttr = GetFileAttributes(filename);
      if (ulAttr != TXINVALID)
      {
         if ((ulAttr & FILE_ATTRIBUTE_DIRECTORY) == 0)
         {
            TxPrint("\nCannot create directory, a file '%s' exists  ", filename);
            rc = ERROR_FILE_EXISTS;             // file with same name exists
         }
      }
      else                                      // not existing
      {
         rc = NO_ERROR;
         if (strcmp(filename+1, ":"))           // no bare drive specification
         {
            if ((p = TxStripBaseName(filename)) != NULL) // last component
            {
               rc = TxMakePath(filename);
               *(--p) = FS_PATH_SEP;            // join last part again
            }
            if (rc == NO_ERROR)
            {
               if (!CreateDirectory(filename, NULL)) // New, no security
               {
                  rc = TX_INVALID_PATH;
               }
            }
         }
      }
   #elif defined (DOS32)
      rc = (ULONG) _dos_getfileattr(filename, &usAttr);
      TRACES(("get attr, RC %u, attr %hx on: '%s'\n", rc, usAttr, filename));
      if (rc == NO_ERROR)
      {
         if ((usAttr & _A_SUBDIR) == 0)         // it is an existing file
         {
            TxPrint("\nCannot create directory, a file '%s' exists  ", filename);
            rc = ERROR_FILE_EXISTS;             // file with same name exists
         }
      }
      else                                      // not existing
      {
         rc = NO_ERROR;
         if (strcmp(filename+1, ":"))           // no bare drive specification
         {
            if ((p = TxStripBaseName(filename)) != NULL) // find last component
            {
               rc = TxMakePath(filename);
               *(--p) = FS_PATH_SEP;            // join last part again
            }
            if (rc == NO_ERROR)
            {
               if (mkdir( filename) != 0)
               {
                  rc = TX_INVALID_PATH;
               }
            }
         }
      }
   #elif defined (UNIX)
      if ((stat_rc = stat( filename, &f)) != -1)
      {
         st_mode = (USHORT) f.st_mode;
      }
    #if !defined (DARWIN)
      else if (errno == EOVERFLOW)
      {
         //- retry with 64-bit stat (latest Linux kernels fail on regular)
         if ((stat_rc = stat64( filename, &f64)) != -1)
         {
            st_mode   = (USHORT) f64.st_mode;
         }
      }
    #endif

      TRACES(("STAT rc:%d  mode:0x%4.4x  errno:%d\n", stat_rc, st_mode, errno));

      if (stat_rc != -1)                        // name exists
      {
         if (S_ISDIR( st_mode) == 0)            // but not a directory ?
         {
            TxPrint("\nCannot create directory, a file '%s' exists  ", filename);
            rc = ERROR_FILE_EXISTS;             // file with same name exists
         }
      }
      else if ((errno == EFBIG) ||              // must be a large-file
               (errno == EOVERFLOW))
      {
         TxPrint("\nCannot create directory, a file '%s' exists  ", filename);
         rc = ERROR_FILE_EXISTS;                // large file with same name exists
      }
      else                                      // Can and must create this level
      {
         if ((p = TxStripBaseName(filename)) != NULL) // find last component
         {
            rc = TxMakePath(filename);
            *(--p) = FS_PATH_SEP;               // join last part again
         }
         if (rc == NO_ERROR)
         {
            TRACES(("Attempt mkdir for '%s'\n", filename));
            if (mkdir( filename, S_IRWXU | S_IRWXG | S_IRWXO) != 0)
            {
               rc = TX_INVALID_PATH;
            }
         }
      }
   #else
      rc = DosQPathInfo(filename,               // Path string
                        FIL_STANDARD,           // Path data required
                        &fstat,                 // Path data buffer
                        sizeof(fstat));         // Path data buffer size
      TRACES(("DosQPathInfo, RC %u on: '%s'\n", rc, filename));
      if (rc == NO_ERROR)
      {
         if ((fstat.attrFile & FILE_DIRECTORY) == 0)
         {
            TxPrint("\nCannot create directory, a file '%s' exists  ", filename);
            rc = ERROR_FILE_EXISTS;             // file with same name exists
         }
      }
      else
      {
         rc = NO_ERROR;
         if (strcmp(filename+1, ":"))           // no bare drive specification
         {
            if ((p = TxStripBaseName(filename)) != NULL) // find last component
            {
               rc = TxMakePath(filename);
               *(--p) = FS_PATH_SEP;            // join last part again
            }
            if (rc == NO_ERROR)
            {
               TRACES(("Attempt mkdir for '%s'\n", filename));
               rc = DosMkDir(filename, NULL);   // Create New, without EA's
            }
         }
      }
   #endif
   RETURN (rc);
}                                               // end 'TxMakePath'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make an 8-dot-3 compatible copy of supplied path/filename string (no spaces)
/*****************************************************************************/
ULONG TxMake8dot3                               // RET   length converted name
(
   char               *filename,                // IN    relative/absolute path
   char               *fname8d3                 // OUT   8dot3 compatible
)
{
   ULONG               rc = 0;                  // return value
   char               *p  = filename;           // ptr in source name
   char               *d  = fname8d3;           // ptr in converted name
   ULONG               cs = 0;                  // checksum for segment
   ULONG               i  = 0;                  // length converted part

   ENTER();

   TRACES(("filename : '%s'\n", filename));
   do
   {
      if ((*p == '\\') || (*p == '/') ||        // end of any segment
          (*p == ':')  || (*p == 0))
      {
         if (*p == ':')                         // end of drive part
         {
            cs = 0;                             // no cs on drive part
         }
         else if ((cs != 0) && (i > 8))         // add cs to converted ?
         {                                      // . plus 3 upcase hex digits
            cs = ((cs ^ (cs >> 24)) &0xfff);    // XOR and shuffle a bit
            sprintf( d, ".%3.3X", cs);
            d += strlen(d);                     // advance to end of string
         }
         *d++ = *p;                             // copy character, convert space
         i    = 0;                              // reset count on next segment
         cs   = 0;                              // reset checksum
      }
      else                                      // other characters
      {
         if (i < 8)
         {
            if (*p == '.')                      // dot in dir/file name
            {
               if (strcspn( p+1, "\\/") <= 3)   // will be valid if total
               {                                // stays 12 or less (8+1+3)
                  *d = '.';                     // so keep the original dot
                  i  = 4;                       // avoid conversion on next 3
               }
               else                             // translate to '_' because
               {                                // .XXX will be added for cs
                  *d = '_';
                  i++;
               }
            }
            else
            {
               *d = *p;                         // copy character
               i++;
            }
            d++;                                // advance counter & dest
         }
         else                                   // conversion will happen
         {
            if ((i == 8) && (*p == '.'))        // dot in dir/file name
            {
               if (strcspn( p+1, "\\/") <= 3)   // will be valid if total
               {                                // stays 12 or less (8+1+3)
                  *d++ = '.';                   // so keep the original dot
                  i    = 4;                     // avoid conversion on next 3
               }
               else                             // translate to '_' because
               {                                // .XXX will be added for cs
                  *(d-3) = *(d-2);              // shift last 2 chars back
                  *(d-2) = *(d-1);
                  if (*p == '.')
                  {
                     *(d-1) = '_';
                  }
                  else
                  {
                     *(d-1) = *p;
                  }

                  i++;
                  rc = 1;                       // signal conversion made
               }
            }
            else
            {
               *(d-3) = *(d-2);                 // shift last 2 chars back
               *(d-2) = *(d-1);
               if (*p == '.')
               {
                  *(d-1) = '_';
               }
               else
               {
                  *(d-1) = *p;
               }

               i++;
               rc = 1;                          // signal conversion made
            }
         }
         cs *= 2;                               // multiply by 2
         cs += *p;                              // add char to checksum
      }
      if (*p != 0)
      {
         p++;
      }
   } while ((*p != 0) || (i >= 8));             // until end and cs added
   *d = 0;                                      // always terminated

   TxRepl( fname8d3, ' ', '_');                 // translate spaces
   TxRepl( fname8d3, '*', '-');                 // translate wildcard '*'
   TxRepl( fname8d3, '?', '-');                 // translate wildcard '?'
   TxRepl( fname8d3, '&', '!');                 // translate illegal  '&'
   TxRepl( fname8d3, '=', '!');                 // translate illegal  '='
   TxRepl( fname8d3, '+', '!');                 // translate illegal  '+'
   if (rc != 0)                                 // conversion made ?
   {
      rc = strlen( fname8d3);
   }
   TRACES(("fname out: '%s'\n", fname8d3));
   RETURN (rc);
}                                               // end 'TxMake8dot3'
/*---------------------------------------------------------------------------*/

