//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// Generic TX utility functions
//
// Author: J. van Wijk
//
// JvW  13-06-2002 Initial version, split off from TXCON
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license

#include <txlib.h>

static  char  txc_ascii7_t[] =                  // translate table 128..255

        "CueaaaaceeeiiiAAEaAooouuyOUcLYPfaiounNao?[]24i<>-=#|++++++|+++++"
// 128  ""

        "++++-++++++++=+++++++++++++*=||=abcpE_ut08OSmfeU=+GL[]:~o..vn2. ";
// 192  ""



/*****************************************************************************/
// Test if area consists completely of the empty byte/character ==> is empty
/*****************************************************************************/
BOOL TxAreaEmpty                                // RET   area is empty
(
   void               *area,                    // IN    area to check
   int                 size,                    // IN    size of area, bytes
   int                 empty                    // IN    the empty char
)
{
   BOOL                rc = TRUE;
   char               *ch = area;

   while (size--)
   {
      if (*ch != empty)
      {
         rc = FALSE;
         break;
      }
      ch++;
   }
   return (rc);
}                                               // end 'TxAreaEmpty'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Filter 8-bit ASCII, strip to 7-bit, translate, or replace by specified char
// Can NOT have any internal tracing! Used in TxPrint to scrollbuffer stream!
/*****************************************************************************/
char *TxAscii827                                // RET   resulting string
(
   char               *str,                     // INOUT string to convert
   unsigned char       new                      // IN    new char, or 0x00
)
{
   char                ch;

   while ((ch = *str) != 0)
   {
      if ((ch & 0x80) != 0)                     // 8th bit is set
      {
         switch (new)
         {
            case TXASCII827_STRIP:
               *str = (char)(ch & 0x7f);        // simple strip off
               break;

            case TXASCII827_TRANS:
               *str = txc_ascii7_t[ch & 0x7f]; // translate using table
               break;

            default:
               *str = new;                      // replace by specified
               break;
         }
      }
      str++;
   }
   return (str);
}                                               // end 'TxAscii827'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Copy string to destination, replacing non printables, clip and terminate it
/*****************************************************************************/
char *TxCopy                                    // RET   resulting string
(
   char               *dest,                    // OUT   destination string
   char               *source,                  // IN    source string
   int                 len                      // IN    max length incl \0
)
{
   char               *cc = dest;               // copy character pointer

   if (len > 0)
   {
      while ((*source) && (--len))
      {
         *cc++ = *source++;
      }
      *cc = 0;
   }
   return (dest);
}                                               // end 'TxCopy'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Replace non-printable characters in a string by a '.' DOT character
/*****************************************************************************/
char *TxStrPrintable                            // RET   resulting string
(
   char               *source                   // INOUT source string
)
{
   while (*source)
   {
      *source = TxPrintable(*source);
      source++;
   }
   return (source);
}                                               // end 'TxStrPrintable'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Count number of '1' bits in a 32-bit value
/*****************************************************************************/
ULONG TxUlBitCount                              // RET   Number of '1' bits
(
   ULONG               value                    // IN    32 bit value
)
{
   ULONG               rc = 0;                  // function return

   ENTER();
   TRACES(("value:%8.8lx\n", value));

   do
   {
      rc += (value & 1);
   } while (value >>= 1);
   RETURN (rc);
}                                               // end 'TxUlBitCount'
/*---------------------------------------------------------------------------*/

