//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TX keyboard (TXWM_CHAR) processing for window-classes
//
// Author: J. van Wijk
//
// JvW  06-07-1995 Initial version
// JvW  25-05-2014 Moved iKeySbView to txwsbuf.c (renamed from txwprint.c)
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license

#include <memory.h>

#include <txlib.h>                              // public interface
#include <txwpriv.h>                            // private window interface

static TXTM  statusMsg;                         // short-lived status message

/*****************************************************************************/
// Input key handling for the TEXTVIEW window class
/*****************************************************************************/
ULONG txwIkeyTextview                           // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   TXWMPARAM           mp1,                     // IN    param 1
   TXWMPARAM           mp2                      // IN    param 2
)
{
   ULONG               rc  = NO_ERROR;
   TXWINBASE          *wnd = (TXWINBASE *) hwnd;
   TXWINDOW           *win = wnd->window;
   short               sx  = win->client.right  - win->client.left +1;
   short               sy  = win->client.bottom - win->client.top  +1;
   BOOL                upd = FALSE;             // window draw required
   BOOL                skp = FALSE;
   short               length;                  // temporary value
   short               longest = 0;             // longest line in text
   short               nr;                      // nr of lines in buf
   TXTEXTVIEW         *dat;
   char              **s;

   ENTER();
   if ((wnd != NULL) && (win != NULL))
   {
      dat = &win->tv;
      dat->maxtop = 0;
      dat->maxcol = 0;
      for (nr = 0, s = dat->buf; s && *s; s++)  // count entries
      {
         length  = txSlen( *s);
         longest = max( longest, length);
         nr++;
      }
      if (nr > sy)                              // text longer than window
      {
         dat->maxtop = nr - sy +1;              // allow one more than size
      }
      if (longest > sx)                         // text wider as window
      {
         dat->maxcol = longest - sx +1;         // allow one more than size
      }
      else
      {
         dat->maxcol = 0;
      }
      TRACES(("top:%4lu, left:%4hd, max:%4lu, lines:%4lu\n",
               dat->topline, dat->leftcol, dat->maxtop, nr));
      switch ((ULONG) mp2)                      // key value
      {
         case TXk_UP:
            if (dat->topline != 0)
            {
               dat->topline--;
               upd = TRUE;                      // full update (status)
            }
            break;

         case TXk_DOWN:
            if (dat->topline < dat->maxtop)
            {
               dat->topline++;
               upd = TRUE;                      // full update (status)
            }
            break;

         case TXk_PGUP:
            if (dat->topline > (ULONG) sy)
            {
               dat->topline -= sy;

               //- keep scrolling up until last line non-empty, or until the
               //- TOP line contains a # at start of line (#help files only)
               while ((strlen(win->tv.buf[win->tv.topline +sy -1]) ==  0 ) &&
                      (      (win->tv.buf[win->tv.topline][0])     != '#') &&
                      (      (win->tv.buf[              0][0])     == '#') &&
                      (dat->topline > (ULONG) sy))
               {
                  dat->topline--;
               }
               upd = TRUE;
            }
            else if (dat->topline != 0)
            {
               dat->topline = 0;
               upd = TRUE;
            }
            break;

         case TXk_PGDN:
            if (dat->topline + sy <= dat->maxtop)
            {
               dat->topline += sy;

               //- keep scrolling down until first line non-empty
               while ((strlen(win->tv.buf[win->tv.topline]) == 0) &&
                      (dat->topline + 1 <= dat->maxtop))
               {
                  dat->topline++;
               }
               upd = TRUE;
            }
            else if (dat->topline != dat->maxtop)
            {
               dat->topline = dat->maxtop;
               upd = TRUE;
            }
            break;

         case TXc_HOME:
            if (dat->topline != 0)
            {
               dat->topline = 0;
               upd = TRUE;
            }
            break;

         case TXc_END:
            if (dat->topline != dat->maxtop)
            {
               dat->topline = dat->maxtop;
               upd = TRUE;
            }
            break;

         case '<':
         case ',':
         case TXa_COMMA:
         case TXk_LEFT:
            if (dat->leftcol != 0)
            {
               dat->leftcol--;
               upd = TRUE;
            }
            break;

         case '>':
         case '.':
         case TXa_DOT:
         case TXk_RIGHT:
            if (dat->leftcol < dat->maxcol)
            {
               dat->leftcol++;
               upd = TRUE;
            }
            break;

         case TXk_HOME:
            if (dat->leftcol != 0)
            {
               dat->leftcol = 0;
               upd = TRUE;
            }
            break;

         case TXk_END:
            if (dat->leftcol != dat->maxcol)
            {
               dat->leftcol = dat->maxcol;
               upd = TRUE;
            }
            break;

         case TXa_F2:
         case TXs_F2:                           // print full text to SBUF
            for (s = dat->buf; s && *s; s++)    // all lines in text
            {
               length = txSlen( *s);
               if ((length != 0) || (skp == FALSE))
               {
                  TxPrint( "%s\n", *s);
               }
               skp = (length == 0);             // skip next line if empty too
            }
            break;

         case TXc_A:                            // mark ALL text, or WHOLE item
            if (nr < 24)                        // smaller text, likely a message
            {
               dat->markLine  = 0;              // from first line in text
               dat->markLines = nr;             // all lines
            }
            else                                // large text, probably HELP
            {
               ULONG   count = 0;               // #lines to mark
               ULONG   empty = 0;               // #consequtive empty lines

               for (longest = 1, s = &dat->buf[ dat->topline]; s && *s; s++, count++)
               {
                  length = txSlen( *s);
                  if (length == 0)
                  {
                     empty++;
                     if (count > 160)
                     {
                        break;                  // EMPTY line after 160 ...
                     }
                  }
                  else
                  {
                     if (length > longest)      // determine longest in this range
                     {
                        longest = length;
                     }
                     empty = 0;
                  }
                  if ((empty == 3) || (count > 200))
                  {
                     count -= 2;                // don't include the last (empty) lines
                     break;                     // at absolute maximum, or 3 empty lines
                  }
               }
               dat->markLines = count;          // whole item, maximum 200 lines
               dat->markLine  = dat->topline;   // from first line in window
            }
            dat->markSize  = longest;           // longest line
            dat->markCol   = 0;                 // from first column
            upd = TRUE;
            break;

         case TXa_U:                            // unmark TV
            dat->markSize  = 0;                 // reset the mark
            upd = TRUE;
            break;

         case TXa_C:                            // COPY to clipboard, TEXT up to 160 lines
         case TXc_C:                            // or until 3 EMPTY lines encountered
         case TXc_INSERT:                       // then stop at 1st EMPTY, or 200 lines
            {
               ULONG    count = 0;
               ULONG    size  = 1;               // minimum size to alloc (status message)
               char    *text  = NULL;

               if (dat->markSize != 0)          // there is a marked area
               {
                  char *mark = NULL;

                  if ((mark = TxAlloc( dat->markSize + 1, 1)) != NULL)
                  {
                     size += (((ULONG) dat->markSize + 1) *  ((ULONG) dat->markLines)) + 1;
                     if ((text = TxAlloc( 1, size)) != NULL)
                     {
                        //- build the clipboard string
                        count = dat->markLines;
                        nr    = (short) count;  // keep around
                        for ( s = &dat->buf[ dat->markLine]; count--; s++) // count down over #lines
                        {
                           TxCopy( mark,  *s + dat->markCol, dat->markSize + 1);
                           strcat( text, mark);
                           strcat( text, "\n");
                        }
                     }
                     TxFreeMem( mark);
                  }
               }
               else                             // no mark, use default (one help item)
               {
                  ULONG   empty = 0;

                  //- determine size of clipboard string, from first visible line
                  for ( s = &dat->buf[ dat->topline]; s && *s; s++, count++)
                  {
                     length = txSlen( *s);
                     size  += length + 1;       // line + EOL

                     if (length == 0)
                     {
                        empty++;
                        if (count > 160)
                        {
                           break;               // EMPTY line after 160 ...
                        }
                     }
                     else
                     {
                        empty = 0;
                     }
                     if ((empty == 3) || (count > 200))
                     {
                        count -= 2;             // don't include the last (empty) lines
                        break;                  // at absolute maximum, or 3 empty lines
                     }
                  }
                  if ((text = TxAlloc( 1, size)) != NULL)
                  {
                     //- build the clipboard string
                     nr = (short) count;
                     for ( s = &dat->buf[ dat->topline]; count--; s++) // count down over #lines
                     {
                        strcat( text, *s);
                        strcat( text, "\n");
                     }
                  }
               }
               if (text != NULL)                // copy prepared text to clipboard
               {
                  if (TxCopyClipBoardText( text) == NO_ERROR) // string to clipboard
                  {
                     if ((dat->markSize != 0) && (dat->markLines == 1))
                     {
                        sprintf( statusMsg, "Copied %hd Marked-area bytes to the clipboard", dat->markSize);
                     }
                     else                       // multiple lines
                     {
                        sprintf( statusMsg, "Copied %hd lines of %s to the clipboard",
                                 nr, (dat->markSize != 0) ? "Marked-area" : "text");
                     }
                     txwSetSbviewStatus( statusMsg, cSchemeColor);
                     TxSleep( 900);             // allow brief reading time :)
                  }
                  TxFreeMem( text);
               }
            }
            break;

         default:
            txwPostMsg((TXWHANDLE) wnd->owner, TXWM_CHAR, mp1, mp2);
            break;
      }
      if (upd)
      {
         rc = txwInvalidateWindow( hwnd, TRUE, TRUE);
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'txwIkeyTextview'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Input key handling for the ENTRYFIELD window class
/*****************************************************************************/
ULONG txwIkeyEntryfield                         // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   TXWMPARAM           mp1,                     // IN    param 1
   TXWMPARAM           mp2                      // IN    param 2
)
{
   ULONG               rc  = NO_ERROR;
   TXWINBASE          *wnd = (TXWINBASE *) hwnd;
   TXWINDOW           *win;
   BOOL                upd = FALSE;             // window draw required
   short               length;
   short               index;
   TXENTRYFIELD       *dat;
   TXTM                completion;              // completion text
   char               *cur;
   BOOL                inRange;                 // character in-range for field
   ULONG               key = (ULONG) mp2;
   ULONG               ccSize = 0;              // clip-copy size

   ENTER();
   if ((wnd != NULL) && ((win = wnd->window) != NULL))
   {
      short            sx    = win->client.right  - win->client.left +1;
      USHORT           wid   = txwQueryWindowUShort( hwnd, TXQWS_ID);

      dat    = &win->ef;
      length = txSlen( dat->buf);
      index  = dat->leftcol + dat->curpos;      // index in string value buffer
      cur    = &dat->buf[ index];               // address of current char
      if (length > sx)                          // text wider as window
      {
         dat->maxcol = length - sx +1;
      }
      else
      {
         dat->maxcol = 0;
      }
      TRACES(("left:%4lu, max:%4lu, curpos: %4lu\n", dat->leftcol, dat->maxcol, dat->curpos));

      switch (key)                      // key value
      {
         case TXk_INSERT:
            txwa->insert = (txwa->insert) ? FALSE : TRUE;
            txwSetCursorStyle( hwnd, txwa->insert);
            break;

         case TXk_ENTER:
            txwAdd2History( hwnd, dat->buf);
            if (win->style & TXES_DLGE_FIELD)   // field in dialog
            {
               txwPostMsg((TXWHANDLE) wnd->owner, TXWM_CHAR, (TXWMPARAM) wid, mp2);
            }
            else                                // stand-alone, clear and upd
            {
               dat->curpos  = 0;
               dat->leftcol = 0;
               strcpy( dat->buf, "");
               upd = TRUE;                      // repaint
            }
            break;

         case TXc_K:                            // Store in history, clear
            txwAdd2History( hwnd, dat->buf);
            strcpy( dat->buf, "");
            dat->curpos  = 0;
            dat->leftcol = 0;
            upd = TRUE;
            break;

         case TXk_F11:                          // list history buffer
         case TXc_O:
            if (dat->history != NULL)
            {
               TXRECT     where = win->border;  // Close to the field's ULC
               TXSELIST  *list;
               ULONG      mcode;                // listbox result

               memset( completion, 0, TXMAXTM);
               if ((txwa->historyMode & TXW_HIST_FILTER) &&
                   ((dat->leftcol + dat->curpos) > 0)) // filter prefix there ...
               {
                  memcpy( completion, dat->buf, min( dat->leftcol + dat->curpos, TXMAXTM - 1));
               }
               if (TxSelistHistory( dat->history, completion, &list) == NO_ERROR)
               {
                  sprintf( completion, "History (%u)", list->count);
                  list->flags |= TXSL_ITEM_DELETE; //- allow item delete using
                  mcode = txwListBox(              //- Ctrl-D, as ListBox RC
                           TXHWND_DESKTOP,
                           hwnd,                // current is owner of popup
                           &where,              // popup position
                           completion, "",      // title in the popup
                           win->helpid,         // and same global help
                           TXLB_MOVEABLE,
                           cMenuTextStand,      // Use Menu color scheme
                           cMenuBorder_top,     // for client & border
                           list);

                  switch (mcode)
                  {
                     case TXDID_CANCEL:
                        break;

                     case TXc_D:                // delete current item
                        if (dat->history != NULL)
                        {
                           dat->history->current = list->selected;
                           txwDelCurrentHistory( dat->history);
                        }
                        break;

                     default:
                        dat->history->current = mcode - TXDID_MAX -1;
                        strcpy( dat->buf, txwGetHistory( dat->history, TXH_THIS));
                        break;
                  }
                  txSelDestroy( &list);

                  if (mcode == TXc_D)           // deleted item, display popup
                  {
                     txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXk_F11);
                  }
                  upd = TRUE;
               }
               else
               {
                  TxNamedMessage( TRUE, 0, " HISTORY: selection error ",
                                  "There are no history items that match the selection criteria");
               }
            }
            break;

         case TXc_D:                            // Delete from history, clear
            if (dat->history != NULL)
            {
               if ((dat->leftcol + dat->curpos) > 0) // prefix there ...
               {
                  memset( completion, 0, TXMAXTM);
                  memcpy( completion, dat->buf, min( dat->leftcol + dat->curpos, TXMAXTM));

                  txwDelCurrentHistory( dat->history);

                  if ((cur = txwRfndHistory( dat->history, completion, TRUE)) != NULL)
                  {
                     strcpy( dat->buf, cur);
                  }
               }
               else                             // no prefix, take previous
               {
                  txwDelCurrentHistory( dat->history);
                  strcpy( dat->buf, txwGetHistory( dat->history, TXH_PREV));
               }
            }
            upd = TRUE;
            break;

         case TXk_UP:
            if (dat->history != NULL)
            {
               if ((txwa->historyMode & TXW_HIST_FILTER) &&
                   ((dat->leftcol + dat->curpos) > 0)) // filter prefix there ...
               {
                  memset( completion, 0, TXMAXTM);
                  memcpy( completion, dat->buf, min( dat->leftcol + dat->curpos, TXMAXTM));
                  if ((cur = txwRfndHistory( dat->history, completion, TRUE)) != NULL)
                  {
                     strcpy( dat->buf, cur);
                  }
               }
               else                             // no prefix, take previous
               {
                  strcpy( dat->buf, txwGetHistory( dat->history, TXH_PREV));
               }
               upd = TRUE;                      // repaint
            }
            else
            {
               txwSetFocus( txwFindPrevFocus( hwnd, FALSE));
            }
            break;

         case TXk_DOWN:
            if (dat->history != NULL)
            {
               if ((txwa->historyMode & TXW_HIST_FILTER) &&
                   ((dat->leftcol + dat->curpos) > 0)) // filter prefix there ...
               {
                  memset( completion, 0, TXMAXTM);
                  memcpy( completion, dat->buf, min( dat->leftcol + dat->curpos, TXMAXTM));
                  if ((cur = txwFindHistory( dat->history, completion, TRUE)) != NULL)
                  {
                     strcpy( dat->buf, cur);
                     upd = TRUE;                // repaint
                  }
               }
               else                             // no prefix, take next
               {
                  strcpy( dat->buf, txwGetHistory( dat->history, TXH_NEXT));
               }
               upd = TRUE;                      // repaint
            }
            else
            {
               txwSetFocus( txwFindNextFocus( hwnd, FALSE));
            }
            break;

         case TXk_LEFT:
            if      (dat->curpos  != 0)         // not at start of field
            {
               dat->curpos--;
               txwSetCursorPos( hwnd, 0, dat->curpos);
               upd = TRUE;                      // repaint
            }
            else if (dat->leftcol != 0)         // scrolled field
            {
               dat->leftcol--;
               upd = TRUE;                      // repaint
            }
            break;

         case TXc_LEFT:                         // one word to left
            if (index > 0)
            {
               index--;
               cur--;
               while ((index > 0) && (*cur == ' '))
               {
                  index--;
                  cur--;
               }
               while ((index > 0) && (*cur != ' '))
               {
                  index--;
                  cur--;
               }
               if (index > 0)                   // stopped at space BEFORE word
               {
                  index++;
                  cur++;
               }
               if (index < dat->leftcol)
               {
                  dat->leftcol = index;         // curpos at start field
               }
               upd = TRUE;                      // repaint (cursorpos indicator)
               dat->curpos = index - dat->leftcol;
               txwSetCursorPos( hwnd, 0, dat->curpos);
            }
            else                               // related stuff (scroll sbview)
            {
               txwPostMsg((TXWHANDLE) wnd->owner, TXWM_CHAR, mp1, mp2);
            }
            break;

         case TXk_RIGHT:
            if (index < dat->rsize -1)          // room to move/grow
            {
               if (index >= length)             // beyond current string-value
               {
                  strcat( dat->buf, " ");       // extend string by one space
               }
               if (dat->curpos  < (sx -1))      // not at end of field
               {
                  dat->curpos++;
                  txwSetCursorPos( hwnd, 0, dat->curpos);
               }
               else if (dat->rsize > sx)        // scroll field
               {
                  dat->leftcol++;
               }
               upd = TRUE;                      // repaint
            }
            else if (dat->rsize == sx)          // no-scrolling field ?
            {
               if (dat->buf[0] == ' ')          // string starts with spaces
               {
                  memmove( dat->buf, dat->buf +1, txSlen( dat->buf));
               }
               upd = TRUE;
            }
            break;

         case TXc_RIGHT:                        // one word to right
            if (index < length -1)              // not at end of string
            {
               while ((index < length -1) && (*cur != ' '))
               {
                  index++;                      // skip the word itself
                  cur++;
               }
               while ((index < length -1) && (*cur == ' '))
               {
                  index++;                      // skip next whitespace
                  cur++;
               }
               if (index > dat->leftcol + sx)
               {
                  dat->leftcol = index - sx;
                  upd = TRUE;
               }
               upd = TRUE;                      // repaint (cursorpos indicator)
               dat->curpos  = index - dat->leftcol;
               txwSetCursorPos( hwnd, 0, dat->curpos);
            }
            else                                // related stuff (scroll sbview)
            {
               txwPostMsg((TXWHANDLE) wnd->owner, TXWM_CHAR, mp1, mp2);
            }
            break;

         case TXc_E:                            // delete to end of field
            *cur = 0;
            upd = TRUE;                         // repaint
            break;

         case TXc_A:                            // mark ALL text in entryfield
            dat->markSize  = strlen( dat->buf);
            dat->markCol   = 0;                 // from first column
            upd = TRUE;
            break;

         case TXa_U:                            // unmark EF
            dat->markSize  = 0;                 // reset the EF mark
            upd = TRUE;
            break;

         case TXa_X:                            // CUT  to clipboard, as EF text
         case TXc_X:                            // CUT  to clipboard, as EF text
         case TXa_C:                            // Copy to clipboard, as EF text
         case TXc_C:                            // Copy to clipboard, as EF text
         case TXc_INSERT:                       // remove leading/trailing spaces
            if (dat->markSize != 0)             // there is a marked area
            {
               ccSize = dat->markSize;          // always a fixed size to the clipboard!
            }
            else if ((key != TXa_X) && (key != TXc_X)) // no CUT if no mark
            {
               if (strlen( cur))                // no mark, from curpos or whole field
               {
                  ccSize = strlen( cur);
               }
               else                             // no mark, cursor at end, use whole field
               {
                  ccSize = strlen( dat->buf);
               }

            }
            if (ccSize != 0)                    // anything to copy ?
            {
               char   *text = TxAlloc( 1, ccSize + 1); // field copy / status

               if (text)
               {
                  if (dat->markSize != 0)       // copy contents of mark, fill with spaces
                  {
                     if (dat->markCol < strlen( dat->buf))
                     {
                        TxCopy( text, dat->buf + dat->markCol, ccSize + 1); // text inside the mark
                        if ((key == TXa_X) || (key == TXc_X))
                        {
                           cur = dat->buf + dat->markCol; // cursor to start MARK
                           if ((dat->markCol + dat->markSize) < strlen( dat->buf))
                           {
                              memmove( cur,  cur + dat->markSize, strlen( cur + dat->markSize) + 1);
                           }
                           else                 // mark extends beyond text, just clip it
                           {
                              *cur = 0;
                           }
                        }
                     }
                     while (strlen( text) < ccSize)
                     {
                        strcat( text, " ");     // fill out with spaces, to whole marked length
                     }
                     if ((key == TXa_X) || (key == TXc_X))
                     {
                        dat->markSize = 0;      // unmark
                        upd = TRUE;
                     }
                  }
                  else                          // no mark, copy from cursor or whole field
                  {
                     char   *from = (strlen( cur) ? cur : dat->buf); // when at end, whole field

                     strcpy(  text, from);
                     TxStrip( text, text, ' ', ' '); // strip leading/trainling spaces
                  }

                  if (TxCopyClipBoardText( text) == NO_ERROR) // string to clipboard
                  {
                     sprintf( statusMsg, "Copied %u entryfield bytes to the clipboard", txLlen( text));
                     txwSetSbviewStatus( statusMsg, cSchemeColor);
                     // TxSleep( 900);             // allow brief reading time (delays unmarking!)
                  }
                  TxFreeMem( text);
               }
            }
            else
            {
               if ((key != TXa_X) && (key != TXc_X)) // no CUT if no mark
               {
                  TxNamedMessage( TRUE, 0, " ERROR: No text to copy ", "You can not copy an empty field or marked area to the clipboard");
               }
               else
               {
                  TxNamedMessage( TRUE, 0, " ERROR: No mark to CUT ", "You can not CUT to the clipboard if there is no marked area");
               }
            }
            break;

         case TXa_V:
         case TXc_V:                            // Paste from clipboard, if text (insert / replace mark)
         case TXa_INSERT:
            {
               char   *pasteBuf = TxAlloc( 1, dat->rsize * 2); // enough room for doubled size

               if (pasteBuf)
               {
                  if (TxPasteClipBoardLines( dat->rsize, TRUE, pasteBuf) != 0) // single line
                  {
                     short   length = (short) strlen( pasteBuf);

                     if ((dat->markSize != 0) && (cur >= (dat->buf + dat->markCol))
                                              && (cur <  (dat->buf + dat->markCol + dat->markSize)))
                     {
                        //- delete existing marked text, when cursor is INSIDE the marked area
                        cur = dat->buf + dat->markCol; // cursor to start MARK
                        if ((dat->markCol + dat->markSize) < strlen( dat->buf))
                        {
                           memmove( cur,  cur + dat->markSize, strlen( cur + dat->markSize) + 1);
                        }
                        else                    // mark extends beyond text, just clip it
                        {
                           *cur = 0;
                        }
                        dat->markSize = 0;      // unmark
                     }
                     strcat( pasteBuf, cur);    // append tail from cursor-position
                     *cur = 0;                  // terminate at cursor position
                     pasteBuf[ dat->rsize - strlen( dat->buf) - 1] = 0; // Clip at max length
                     strcpy( cur, pasteBuf);    // add pasted text + tail end

                     index += length;           // skip pasted new text
                     cur   += length;
                     if (index > dat->leftcol + sx)
                     {
                        dat->leftcol = index - sx;
                     }
                     dat->curpos  = index - dat->leftcol;
                     txwSetCursorPos( hwnd, 0, dat->curpos);
                     upd = TRUE;
                  }
                  TxFreeMem( pasteBuf);
               }
            }
            break;

         case TXk_ESCAPE:
            if (win->style & TXES_DLGE_FIELD)   // field in dialog
            {
               txwPostMsg((TXWHANDLE) wnd->owner, TXWM_CHAR, mp1, mp2);
               break;
            }
         case TXc_BACKSP:                       // delete whole field
            *cur = 0;
            TRACES(("Esc-BACKSP, cur:%8.8lx dat:%8.8lx  dat->buf:%8.8lx\n", cur, dat, dat->buf));
         case TXc_B:                            // delete to begin field
            memmove( dat->buf, cur, strlen(cur) +1);
            upd = TRUE;                         // repaint
         case TXk_HOME:                         // cursor to start field
            TRACES(("Esc-HOME\n"));
            if (dat->leftcol != 0)
            {
               dat->leftcol = 0;
               upd = TRUE;
            }
            dat->curpos = 0;
            TRACES(("Esc-CursorPos\n"));
            txwSetCursorPos( hwnd, 0, dat->curpos);
            break;

         case TXk_END:
            if (dat->leftcol != dat->maxcol)
            {
               dat->leftcol = dat->maxcol;
               upd = TRUE;
            }
            dat->curpos = min((sx -1), length);
            txwSetCursorPos( hwnd, 0, dat->curpos);
            break;

         case TXk_BACKSPACE:
            if (index != 0)
            {
               if      (dat->curpos  != 0)      // not at start of field
               {
                  dat->curpos--;
                  txwSetCursorPos( hwnd, 0, dat->curpos);
               }
               else if (dat->leftcol != 0)      // scrolled field
               {
                  dat->leftcol--;
                  upd = TRUE;                   // repaint
               }
               memmove( cur -1, cur, strlen(cur) +1);
               upd = TRUE;                      // repaint
            }
            break;

         case TXk_DELETE:
            if ((dat->markSize != 0) && (cur >= (dat->buf + dat->markCol))
                                     && (cur <  (dat->buf + dat->markCol + dat->markSize)))
            {                                   // we are INSIDE an existing mark
               if (dat->markCol < strlen( dat->buf))
               {
                  cur = dat->buf + dat->markCol; // cursor to start MARK
                  if ((dat->markCol + dat->markSize) < strlen( dat->buf))
                  {
                     memmove( cur,  cur + dat->markSize, strlen( cur + dat->markSize) + 1);
                  }
                  else                          // mark extends beyond text, just clip it
                  {
                     *cur = 0;
                  }
               }
               dat->markSize = 0;               // unmark
               upd = TRUE;
            }
            else if (index < length)            // no mark, delete char at cursor position
            {
               memmove( cur, cur +1, strlen(cur) +1);
               upd = TRUE;                      // repaint
            }
            break;

         default:
            if (win->style & TXES_DECNUM_ONLY)       //- decimal numbers only
            {
               inRange = ((key >= '0') && (key <= '9'));
            }
            else if (win->style & TXES_HEXNUM_ONLY)  //- hexadecimal numbers only
            {
               inRange = (((key >= '0') && (key <= '9')) ||
                          ((key >= 'A') && (key <= 'F')) ||
                          ((key >= 'a') && (key <= 'f')) );
            }
            else
            {
               inRange = txwPossibleAsciiKey(key);
            }
            TRACES(("EF-style: 0x%8.8lx  key: 0x%8.8lx  inRange: %s\n", win->style, key, (inRange) ? "Yes" : "No"));
            if (inRange)
            {
               //- first delete existing marked-area, when we are INSIDE the mark
               if ((dat->markSize != 0) && (cur >= (dat->buf + dat->markCol))
                                        && (cur <  (dat->buf + dat->markCol + dat->markSize)))
               {                                // we are INSIDE an existing mark
                  if (dat->markCol < strlen( dat->buf))
                  {
                     cur = dat->buf + dat->markCol; // cursor to start MARK
                     if ((dat->markCol + dat->markSize) < strlen( dat->buf))
                     {
                        memmove( cur,  cur + dat->markSize, strlen( cur + dat->markSize) + 1);
                     }
                     else                       // mark extends beyond text, just clip it
                     {
                        *cur = 0;
                     }
                  }
                  dat->markSize = 0;            // unmark

                  length = txSlen( dat->buf);   // re-calculate field metrics
                  index  = (short) (cur - dat->buf);
                  dat->curpos   = index - dat->leftcol;

                  if (length > sx)              // text wider as window
                  {
                     dat->maxcol = length - sx +1;
                  }
                  else
                  {
                     dat->maxcol = 0;
                  }
                  upd = TRUE;
               }

               if ((length < dat->rsize) ||     // room to grow
                   ((index < dat->rsize) &&     // or no room needed
                    (txwa->insert == FALSE)))
               {
                  if (txwa->insert)
                  {
                     memmove( cur +1, cur, strlen(cur) +1);
                  }
                  else if (index >= length)     // beyond current string
                  {
                     strcat( dat->buf, " ");    // extend string by one space
                  }
                  *cur = (char) (key & 0xff);
                  if (dat->curpos < (sx -1))    // not at end of field
                  {
                     dat->curpos++;
                     txwSetCursorPos( hwnd, 0, dat->curpos);
                  }
                  else if (dat->rsize > sx)     // scroll field
                  {
                     dat->leftcol++;
                  }
                  else                          // at end-of-field, no-scroll
                  {
                     if (dat->buf[0] == ' ')    // string starts with spaces
                     {
                        memmove( dat->buf, dat->buf +1, txSlen( dat->buf));
                     }
                  }
                  upd = TRUE;                   // repaint
               }
            }
            else
            {
               txwPostMsg((TXWHANDLE) wnd->owner, TXWM_CHAR, mp1, mp2);
            }
            break;
      }
      wnd->cursor.y = 0;
      wnd->cursor.x = dat->curpos;              // record actual cursor position
      if (upd)
      {
         rc = txwInvalidateWindow( hwnd, FALSE, TRUE);
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'txwIkeyEntryfield'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Input key handling for the BUTTON window class
/*****************************************************************************/
ULONG txwIkeyButton                             // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   TXWMPARAM           mp1,                     // IN    param 1
   TXWMPARAM           mp2                      // IN    param 2
)
{
   ULONG               rc  = NO_ERROR;
   TXWINBASE          *wnd = (TXWINBASE *) hwnd;
   TXWINDOW           *win;

   ENTER();
   if ((wnd != NULL) && ((win = wnd->window) != NULL))
   {
      TXBUTTON        *dat   = &win->bu;
      TXWHANDLE        owner = (TXWHANDLE) wnd->owner;
      USHORT           wid   = txwQueryWindowUShort( hwnd, TXQWS_ID);

      switch ((ULONG) mp2)                      // key value
      {
         case 'x':                              // same as check-character 'x'
         case TXk_SPACE:
            switch (win->style & TXBS_PRIMARYSTYLES)
            {
               case TXBS_PUSHBUTTON:
                  if (win->style & TXBS_HELP)
                  {
                     txwPostMsg( hwnd,  TXWM_HELP,    (TXWMPARAM) wid, (TXWMPARAM) TXCMDSRC_PUSHBUTTON);
                  }
                  else
                  {
                     txwPostMsg( owner, TXWM_COMMAND, (TXWMPARAM) wid, (TXWMPARAM) TXCMDSRC_PUSHBUTTON);
                  }
                  break;

               case TXBS_AUTORADIO:
                  if (*(dat->checked) == FALSE) // going to SET this one ?
                  {
                     txwResetAutoRadioGroup( owner, // reset others in same group
                       txwQueryWindowUShort( hwnd, TXQWS_GROUP));
                     *(dat->checked) = TRUE;
                     txwInvalidateWindow( hwnd, FALSE, FALSE);
                  }
               case TXBS_RADIOBUTTON:           // just signal owner (dialog)
                  txwPostMsg( owner, TXWM_COMMAND, (TXWMPARAM) wid, (TXWMPARAM) TXCMDSRC_RADIOBUTTON);
                  break;

               case TXBS_AUTOCHECK:
                  *(dat->checked) = !(*(dat->checked));
                  txwInvalidateWindow( hwnd, FALSE, FALSE);
               case TXBS_CHECKBOX:              // signal owner (dialog)
                  txwPostMsg( owner, TXWM_COMMAND, (TXWMPARAM) wid, (TXWMPARAM) TXCMDSRC_CHECKBOX);
                  break;

               default:
                  txwPostMsg( owner, TXWM_CONTROL, TXMPFROM2SH(wid,0), 0);
                  break;
            }
            break;

         case TXk_ENTER:
            if (((win->style & TXBS_PRIMARYSTYLES) == TXBS_PUSHBUTTON) &&
                ((win->style & TXBS_HELP)          != 0              )  )
            {
               txwPostMsg( hwnd, TXWM_HELP, 0, 0);
            }
            else if (win->style & TXBS_DLGE_BUTTON)
            {
               txwPostMsg( owner, TXWM_COMMAND, (TXWMPARAM) wid, (TXWMPARAM) TXCMDSRC_PUSHBUTTON);
            }
            else                                // pass to owner (dismiss!)
            {
               txwPostMsg( owner, TXWM_CHAR, (TXWMPARAM) wid, mp2);
            }
            break;

         case TXk_UP:
         case TXk_LEFT:
            txwSetFocus( txwFindPrevFocus( hwnd, FALSE));
            break;

         case TXk_DOWN:
         case TXk_RIGHT:
            txwSetFocus( txwFindNextFocus( hwnd, FALSE));
            break;

         case TXc_A:                            // Mark ALL
         case TXa_C:                            // Copy to clipboard, as BT text
         case TXc_C:                            // Copy to clipboard, as BT text
         case TXc_INSERT:                       // Copy to clipboard, as BT text
            if ((win->style & TXBS_PRIMARYSTYLES) == TXBS_PUSHBUTTON)
            {
               //- allow owner to copy to clipboard (like MsgBox, whole message text)
               txwPostMsg( owner, TXWM_CHAR, mp1, mp2);
            }
            else if ((ULONG) mp2 != TXc_A)
            {
               char   *text = TxAlloc( 1, strlen( dat->text) + 60); // button txt / status

               if (text)
               {
                  switch (win->style & TXBS_PRIMARYSTYLES)
                  {
                     case TXBS_RADIOBUTTON:
                     case TXBS_AUTORADIO:
                        sprintf( text, "(%c) %s", (*(dat->checked)) ? '*' : ' ', dat->text);
                        break;

                     case TXBS_CHECKBOX:
                     case TXBS_AUTOCHECK:
                        sprintf( text, "[%c] %s", (*(dat->checked)) ? 'x' : ' ', dat->text);
                        break;

                     default:
                        sprintf( text, "[ %s ]",  dat->text);
                        break;
                  }
                  if (TxCopyClipBoardText( text) == NO_ERROR) // string to clipboard
                  {
                     sprintf( statusMsg, "Copied %u bytes button description to the clipboard", txLlen( text));
                     txwSetSbviewStatus( statusMsg, cSchemeColor);
                     TxSleep( 900);             // allow brief reading time :)
                  }
                  TxFreeMem( text);
               }
            }
            break;

         default:
            txwPostMsg( owner, TXWM_CHAR, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'txwIkeyButton'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Input key handling for the LISTBOX window class
/*****************************************************************************/
ULONG txwIkeyListBox                            // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   TXWMPARAM           mp1,                     // IN    param 1
   TXWMPARAM           mp2                      // IN    param 2
)
{
   ULONG               rc  = NO_ERROR;
   TXWINBASE          *wnd = (TXWINBASE *) hwnd;
   TXWINDOW           *win;

   ENTER();
   if ((wnd != NULL) && ((win = wnd->window) != NULL))
   {
      USHORT           wid = txwQueryWindowUShort( hwnd, TXQWS_ID);
      BOOL             upd = FALSE;             // window draw required
      TXLISTBOX       *dat;
      ULONG            key = (ULONG) mp2;

      if ((dat = &win->lb) != NULL)
      {
         TXWHANDLE     owner = (TXWHANDLE) wnd->owner;
         TXRECT        where = win->border;
         TXSELIST     *list  = dat->list;
         TXS_ITEM     *item  = NULL;

         TRECTA( "Where:  ", (&where));
         TRACES(("cpos:%4hu, ccol:%2hx, icol:%2hx, style:%8.8lx\n", dat->cpos, dat->ccol, dat->icol, win->style));
         TRACES(("Drop-value: %s\n", (win->style & TXLS_DROP_VALUE) ? "YES" : "NO"));
         TRACES(("Drop-enter: %s\n", (win->style & TXLS_DROP_ENTER) ? "YES" : "NO"));
         TRACES(("Drop-menu:  %s\n", (win->style & TXLS_DROP_MENU ) ? "YES" : "NO"));
         TRACES(("Spin-wrap:  %s\n", (win->style & TXLS_SPIN_WRAP ) ? "YES" : "NO"));

         TRACES(("List at:%8.8lx, WM_CHAR mp1:0x%8.8x mp2:0x%8.8x\n", list, mp1, mp2));

         if ((list != NULL) && (list->count != 0)) // we do have a real list
         {
            ULONG this = list->selected;

            TRACES(("astatus:%8.8lx   miscInfo:%p\n",  list->astatus, list->miscInfo));
            TRACES(("top:%6u  select:%6u count:%6u\n", list->top,     list->selected, list->count));
            TRACES(("asize:%6u vsize:%6u tsize:%6u\n", list->asize,   list->vsize,    list->tsize));

            if      (win->style & TXLS_DROP_VALUE) // value drop-down, like a selection list
            {
               TRACES(("Handle key value: 0x0%x  for DROP_VALUE (like a menu)\n", key));
               switch (key)
               {
                  case TXk_ENTER:               // ENTER 'selects' a menu item (executes, or open submenu)
                  case TXk_SPACE:               // SPACE, more intiutive to toggle mark menu-item (reopen)
                     if ((win->style & TXLS_DROP_ENTER) == 0) // it is a NOT a submenu, so 'execute' item
                     {
                        txwPostMsg( owner, TXWM_CONTROL,
                                    TXMPFROM2SH(wid,TXLN_ENTER),
                                    (TXWMPARAM) list);
                        break;
                     }                          // else treat ENTER as a_ENTER
                  case TXa_ENTER:               // drop down
                  case TXk_MENU:
                     if (where.top > 0)
                     {                          // let popup fall on-top of
                        where.top--;            // current title/value-field
                     }
                     if (txwListBox( TXHWND_DESKTOP,
                            hwnd,               // current is owner of popup
                            &where,             // popup position
                            win->title, "",     // repeat title in popup
                            (owner != 0) ? wnd->owner->window->helpid
                                         : win->helpid, // try to get owner dlg help
                            TXLB_MOVEABLE,
                            win->clientclear.at, // inherit color scheme
                            win->borderclear.at,
                            list) != TXDID_CANCEL) // ENTER on popup item
                     {
                        //- notify owner of the original (drop_value) list so
                        //- some action could be taken by its WinProc (prompt)

                        txwPostMsg( owner, TXWM_CONTROL,
                                    TXMPFROM2SH(wid,TXLN_ENTER),
                                    (TXWMPARAM) list);
                     }
                     upd = TRUE;
                     break;

                  case TXk_DOWN:
                     do
                     {
                        if ((list->selected +1) < list->count)
                        {
                           list->selected++;    // make next 'current'
                        }
                        else if (win->style & TXLS_SPIN_WRAP)
                        {
                           list->selected = 0;  // wrap arround
                        }
                        else                    // wrap not allowed
                        {
                           list->selected = this; // reset to startpoint
                        }
                        item = list->items[list->selected];
                     } while ((item->flags & TXSF_DISABLED) &&
                              (list->selected != this));

                     if (list->selected != this)
                     {
                        upd = TRUE;
                     }
                     break;

                  case TXk_UP:
                     do
                     {
                        if (list->selected > 0)
                        {
                           list->selected--;    // make prev 'current'
                        }
                        else if (win->style & TXLS_SPIN_WRAP)
                        {
                           list->selected = list->count -1; // wrap arround
                        }
                        else                    // wrap not allowed
                        {
                           list->selected = this; // reset to startpoint
                        }
                        item = list->items[list->selected];
                     } while ((item->flags & TXSF_DISABLED) &&
                              (list->selected != this));

                     if (list->selected != this)
                     {
                        upd = TRUE;
                     }
                     break;

                  case TXk_HOME:
                  case TXk_PGUP:
                     if (list->selected > 0)
                     {
                        list->selected = 0;     // start-of-list 'current'
                        upd = TRUE;
                     }
                     break;

                  case TXk_END:
                  case TXk_PGDN:
                     if ((list->selected +1) < list->count)
                     {
                        list->selected = list->count -1; // list-end 'current'
                        upd = TRUE;
                     }
                     break;

                  case TXa_C:                   // COPY to clipboard, item text only
                  case TXc_C:
                  case TXc_INSERT:
                     //- if (win->title)        // should not be needed
                     {
                        item  = list->items[list->selected];

                        if (TxCopyClipBoardText( item->text) == NO_ERROR) // string to clipboard
                        {
                           sprintf( statusMsg, "Copied %u drop-down list bytes to the clipboard", txLlen( item->text));
                           txwSetSbviewStatus( statusMsg, cSchemeColor);
                           TxSleep( 900);       // allow brief reading time :)
                        }
                     }
                     break;

                  default:
                     if (txwPossibleAsciiKey(key))
                     {
                        ULONG   selnow = list->selected;

                        if (TxSelCharSelect( list, toupper(key)) != selnow)
                        {
                           upd = TRUE;
                        }
                     }
                     else
                     {
                        txwPostMsg( owner, TXWM_CHAR, mp1, mp2);
                     }
                     break;
               }
            }
            else if (win->style & TXLS_DROP_MENU) // like the DFSee menu-bar itself
            {
               ULONG        mcode;              // listbox result
               TXS_MENUBAR *mbar = (TXS_MENUBAR *) list->miscInfo;

               TRACES(("Handle key value: 0x0%x  for DROP_MENU (like a menubar)\n", key));
               switch (key)
               {
                  case TXa_ENTER:               // drop down
                  case TXk_ENTER:
                  case TXk_DOWN:
                  case TXk_UP:
                     mcode = txwListBox(
                              TXHWND_DESKTOP,
                              hwnd,             // current is owner of popup
                              &where,           // popup position
                              "", "",           // no title in submenu
                              win->helpid,      // and same global help
                              TXLB_MOVEABLE |   // move allowed
                              TXLB_P_FIX_ROW,   // keep row on create
                              cMenuTextStand,   // Use Menu color scheme
                              cMenuBorder_top,  // for client & border
                              list);

                     if (mcode != TXDID_CANCEL) // notify owner of selection
                     {
                        txwPostMsg( owner, TXWM_MENUSELECT, (TXWMPARAM) mcode, TXMPFROM2SH(wid,0));
                     }
                     if (mbar != NULL)          // make default for next time
                     {
                        int          m;
                        TXS_MENU    *menu;

                        for (m = 0; m < mbar->count; m++)
                        {
                           if ((menu = mbar->menu[m]) != NULL)
                           {
                              if (menu->list == list) // this is us ...
                              {
                                 mbar->defopen = m;
                                 break;
                              }
                           }
                        }
                     }
                     break;

                  case TXk_LEFT:                // to prev menu heading
                     txwSetFocus( txwFindPrevFocus( hwnd, FALSE));
                     break;

                  case TXk_RIGHT:               // to next menu heading
                     txwSetFocus( txwFindNextFocus( hwnd, FALSE));
                     break;

                  case TXs_F10:                 // Menu alternative (Gnome)
                  case TXk_F10:                 // Close Owner (MenuBar)
                  case TXk_MENU:                // behave like Esc/Cancel
                     txwPostMsg( owner, TXWM_MENUEND, (TXWMPARAM) TXDID_CANCEL, 0);
                     break;

                  case TXa_U:                   // UNMARK, may want to send to SB
                  case TXa_C:                   // COPY to clipboard, title
                  case TXc_C:
                  case TXc_INSERT:
                     if ((txwa->sbview) && (txwa->sbview->window->sb.sbdata->markSize))
                     {
                        //- there is a mark in Output-Window, has prority over menu-heading!
                        txwPostMsg( txwa->sbview, TXWM_CHAR, mp1, mp2); // send to SB
                     }
                     else if (key != TXa_U)     // copy menu-heading to clipboard
                     {
                        if (win->title)         // is the menu heading, copy that
                        {
                           if (TxCopyClipBoardText( win->title) == NO_ERROR) // string to clipboard
                           {
                              sprintf( statusMsg, "Copied %u menu-heading bytes to the clipboard", txLlen( win->title));
                              txwSetSbviewStatus( statusMsg, cSchemeColor);
                              TxSleep( 900);    // allow brief reading time :)
                           }
                        }
                     }
                     break;

                  default:
                     txwPostMsg( owner, TXWM_CHAR, mp1, mp2);
                     break;
               }
            }
            else                                // multi-line lists (like the DFSee browser)
            {
               ULONG      dr = NO_ERROR;
               ULONG      li;
               char       select;               // selection char
               BOOL       disabled = FALSE;
               TX1K       textbuf;

               TRACES(("Handle key value: 0x0%x  for MULTI_LINE (like a menu, or BROWSE)\n", key));
               switch (key)
               {
                  case TXk_F8:                  // allow list-toggle (grep)
                  case TXk_RIGHT:               // open submenu if present
                  case TXk_ENTER:               // open or 'execute' unless disabled
                     li = (ULONG) (list->top + dat->cpos);

                     item     = list->items[li];
                     disabled = ((item->flags & TXSF_DISABLED) != 0);

                     if ((key == TXk_RIGHT) &&                    //- This is right-arrow-key and
                         (txwa->rightmove || disabled ||          //- want RIGHT_MOVE or disabled
                         ((item->flags & TXSF_P_LISTBOX) == 0)))  //- not a submenu
                     {
                        disabled = TRUE;                          //- don't execute and go
                        txwPostMsg( owner, TXWM_CHAR, mp1, mp2);  //- to next main-menu header
                     }
                     if (!disabled)             // 'execute' the menu item using CONTROL msg
                     {
                        txwPostMsg( owner, TXWM_CONTROL, TXMPFROM2SH(wid,TXLN_ENTER), (TXWMPARAM) list);
                     }
                     else if (key == TXk_ENTER)
                     {
                        sprintf( textbuf,
                                "Menu item '%s' is currently DISABLED!\n\n"
                                "You may need to perform other actions or use different "
                                "menu items first to enable the functionality behind "
                                "the one now selected.%s%s\n",
                                 item->text,
                                (item->info) ? "\n\nLikely reason(s) to disable this menu item:\n\n" : "",
                                (item->info) ?  item->info : "");

                        TxNamedMessage( TRUE, TXWD_DISABLEDITEM, " MENU: Disabled menu-item ", textbuf);
                     }
                     strcpy( txwa->listmatch, "");  //- reset string
                     break;

                  case TXa_C:                   // COPY to clipboard, item text
                  case TXc_C:
                  case TXc_INSERT:
                     {
                        char   *text = NULL;
                        ULONG   nr   = 0;       // default 1 line (current item)
                        ULONG   size = 1;       // (minimum) size to alloc/copy

                        if ((list->markinfo) && (list->markinfo->marked != 0)) // multi-select
                        {
                           ULONG     i;

                           //- First iteration, count and determine total size
                           for (i = 1; i < list->count; i++)
                           {
                              if (list->markinfo->markings[i] & TXSF_MARK_STD)
                              {
                                 item  = list->items[i];
                                 size += (strlen( item->text) + 1); // text + EOL
                                 nr++;
                              }
                           }
                           TRACES(("Alloc : %u bytes for %u items\n", size, list->markinfo->marked));

                           if ((text = TxAlloc( 1, size)) != NULL)
                           {
                              //- Second iteration, build clipboard string
                              for (i = 1; i < list->count; i++)
                              {
                                 if (list->markinfo->markings[i] & TXSF_MARK_STD)
                                 {
                                    item = list->items[i];
                                    strcat( text, item->text);
                                    strcat( text, "\n"); // end of line
                                 }
                              }
                           }
                        }
                        else                    // copy current item only, 1 line
                        {
                           li    = (ULONG) (list->top + dat->cpos);
                           item  = list->items[li];

                           size += strlen( item->text);
                           if ((text = TxAlloc( 1, size)) != NULL)
                           {
                              strcpy(  text, item->text);
                              nr = 1;           // just this one line
                           }
                        }

                        if (text != NULL)
                        {
                           if (TxCopyClipBoardText( text) == NO_ERROR) // string to clipboard
                           {
                              if (nr > 1)
                              {
                                 sprintf( statusMsg, "Copied %d listitems in %u bytes to the clipboard", nr, txLlen( text));
                              }
                              else
                              {
                                 sprintf( statusMsg, "Copied one listitem of %u bytes to the clipboard", txLlen( text));
                              }
                              txwSetSbviewStatus( statusMsg, cSchemeColor);
                              TxSleep( 900);    // allow brief reading time :)
                           }
                           TxFreeMem( text);
                        }
                     }
                     break;

                  case TXa_LBRACKET:            // mark all 1st text-char EQ uservalue
                  case TXa_RBRACKET:            // mark all 1st text-char NE uservalue
                     if (list->markinfo)        // in multiple mark lists
                     {
                        for (li = 0; li < list->count; li++)
                        {
                           if (list->items[li]->text[0] == list->uservalue)
                           {
                              TxSelSetSelected( list, li, (key == TXa_LBRACKET) ? TX_SET : TX_RESET, TXSF_MARK_STD);
                           }
                           else
                           {
                              TxSelSetSelected( list, li, (key == TXa_LBRACKET) ? TX_RESET : TX_SET, TXSF_MARK_STD);
                           }
                        }
                        txwListMark2trhText( hwnd); //- update count in title
                        upd = TRUE;                 //- fall-through, auto-DOWN
                     }
                     else
                     {
                        txwPostMsg( owner, TXWM_CHAR, mp1, mp2);
                     }
                     break;

               #if !defined (UNIX)
                  case TXc_Z:                   // c-Z is SIGSTP, suspend program! (resume with 'fg')
               #endif
                  case TXa_Z:                   // Unmark ALL
                  case '=':
                  case '*':
                  case TXc_A:                   // Mark ALL
                  case TXc_I:                   // Mark Invert
                  case '-':
                     if (list->markinfo)        // in multiple mark lists
                     {
                        for (li = 0; li < list->count; li++)
                        {
                           TxSelSetSelected( list, li,
                              ((key == '-'  ) || (key == TXc_I)) ? TX_TOGGLE :
                              ((key == TXa_Z) || (key == TXc_Z)) ? TX_RESET   :
                                                                   TX_SET, TXSF_MARK_STD);
                        }
                        txwListMark2trhText( hwnd); //- update count in title
                        upd = TRUE;
                     }
                     break;

                  case '+':
                  case TXk_INSERT:              // for FC/2 style, INSERT,
                  case TXk_SPACE:               // + or SPACE, toggle mark
                     if (list->markinfo)        // in multiple mark lists
                     {
                        li = (ULONG) (list->top + dat->cpos);

                        TxSelSetSelected( list, li, TX_TOGGLE, TXSF_MARK_STD);
                        txwListMark2trhText( hwnd); //- update count in title
                        upd = TRUE;                 //- fall-through, auto-DOWN
                     }
                     else                       // probably a normal menu
                     {
                        if (!disabled)          // 'execute' the menu item, just like with ENTER
                        {                       // more intiutive to toggle menu-item-marks (with reopen)
                           txwPostMsg( owner, TXWM_CONTROL, TXMPFROM2SH(wid,TXLN_ENTER), (TXWMPARAM) list);
                        }
                        break;                  // do not 'auto-scroll-down' when marking not enabled
                     }
                  case TXk_DOWN:
                     do
                     {
                        if (((list->count ) >  0         ) &&
                            ((dat->cpos +1) < list->vsize) &&
                            ((dat->cpos +1) < list->count)  )
                        {
                           dat->cpos++;         // make next 'current'
                           upd = TRUE;
                           dr  = NO_ERROR;
                        }
                        else if ((dr = TxSelScrollDown( list)) == NO_ERROR)
                        {
                           upd = TRUE;
                        }
                        else                    // at last item, wrap ?
                        {
                           TxSelSetPosition( list, TXSEL_TOP);
                           dat->cpos = 0;
                           upd = TRUE;
                        }
                        if (list->count != 0)
                        {
                           if ((li = (ULONG) (list->top + dat->cpos)) < list->count)
                           {
                              item     = list->items[li];

                              TRACES(("list-item for disable-check: %lu\n", li));
                              disabled = (((item->flags & TXSF_AUTOSKIP) ||
                                           (item->flags & TXSF_SEPARATOR) ) &&
                                          ( item->flags & TXSF_DISABLED   )  );
                           }
                           else
                           {
                              disabled = FALSE;
                           }
                        }
                     } while ((disabled) && (dr == NO_ERROR));
                     strcpy( txwa->listmatch, ""); // reset string
                     TRACES(("k_DOWN processed\n"));
                     break;

                  case TXk_UP:
                     do
                     {
                        if ((dat->cpos) > 0)
                        {
                           dat->cpos--;         // make prev 'current'
                           upd = TRUE;
                           dr  = NO_ERROR;
                        }
                        else if ((dr = TxSelScrollUp( list)) == NO_ERROR)
                        {
                           upd = TRUE;
                        }
                        else                    // at top item, wrap ?
                        {
                           TxSelSetPosition( list, TXSEL_END);
                           dat->cpos = min( list->vsize, list->count) -1;
                           upd = TRUE;
                        }
                        if (list->count != 0)
                        {
                           if ((li = (ULONG) (list->top + dat->cpos)) < list->count)
                           {
                              item = list->items[li];

                              TRACES(("list-item for disable-check: %lu\n", li));
                              disabled = (((item->flags & TXSF_AUTOSKIP) ||
                                           (item->flags & TXSF_SEPARATOR) ) &&
                                          ( item->flags & TXSF_DISABLED   )  );
                           }
                           else
                           {
                              disabled = FALSE;
                           }
                        }
                     } while ((disabled) && (dr == NO_ERROR));
                     strcpy( txwa->listmatch, ""); // reset string
                     TRACES(("k_UP processed\n"));
                     break;

                  case TXk_HOME:                // to start of list
                     TxSelSetPosition( list, TXSEL_TOP);
                     dat->cpos = 0;
                     upd = TRUE;
                     strcpy( txwa->listmatch, ""); // reset string
                     break;

                  case TXk_END:
                     TxSelSetPosition( list, TXSEL_END);
                     dat->cpos = min( list->vsize, list->count) -1;
                     upd = TRUE;
                     strcpy( txwa->listmatch, ""); // reset string
                     break;

                  case TXk_PGUP:
                     if (dat->cpos != 0)
                     {
                        dat->cpos = 0;
                        upd = TRUE;
                     }
                     else if (list->top != 0)
                     {
                        TxSelSetPosition( list, (list->top > list->vsize)
                                          ?     (list->top - list->vsize)
                                          :      TXSEL_TOP);
                        upd = TRUE;
                     }
                     strcpy( txwa->listmatch, ""); // reset string
                     break;

                  case TXk_PGDN:
                     if (dat->cpos < list->vsize -1)
                     {
                        dat->cpos = min( list->vsize, list->count) -1;
                        upd = TRUE;
                     }
                     else if ((list->top + list->vsize) < list->count)
                     {
                        TxSelSetPosition( list, list->top + list->vsize);
                        upd = TRUE;
                     }
                     strcpy( txwa->listmatch, ""); // reset string
                     break;

               #if defined (UNIX)
                  case TXc_P:          // Ctrl-Y is delayed-stop in most Unixes
               #else
                  case TXc_Y:          // Ctrl-P is PRINT in many non-Unixes
               #endif
                  case TXc_B:  case TXc_D:  case TXc_E:  case TXc_N:  case TXc_R:
                  case TXc_S:  case TXc_T:  case TXc_U:  case TXc_X:
                  case TXc_F1: case TXc_F2: case TXc_F3: case TXc_F4: case TXc_F5:
                  case TXc_F6: case TXc_F7: case TXc_F8: case TXc_F9:
                     if (list && (list->sortinfo))    //- Allow use as accelerator
                     {                                //- when no sorting specified (menu)
                        switch (key)
                        {
                           case TXc_U:
                           case TXc_F7:
                              if (TxSelSortUserData( list) == TRUE) // list changed
                              {
                                 txwListSort2frhText( hwnd); // update footer indicator
                                 txwPositionListBox(  dat ); // fixup positions
                                 upd = TRUE;
                              }
                              break;

                           case TXc_R:          // Reverse List sort
                           case TXc_F8:
                              if (TxSelReverseOrder( list) == TRUE) // list changed
                              {
                                 txwListSort2frhText( hwnd); // update footer indicator
                                 txwPositionListBox(  dat ); // fixup positions
                                 upd = TRUE;
                              }
                              break;

                           case TXc_X:
                           case TXc_F2:
                              if (TxSelSortString( list, TXS_SORT_1) == TRUE) // list changed
                              {
                                 txwListSort2frhText( hwnd); // update footer indicator
                                 txwPositionListBox(  dat ); // fixup positions
                                 upd = TRUE;
                              }
                              break;

                           case TXc_D:
                           case TXc_F1:
                              if (TxSelSortString( list, TXS_SORT_2) == TRUE) // list changed
                              {
                                 txwListSort2frhText( hwnd); // update footer indicator
                                 txwPositionListBox(  dat ); // fixup positions
                                 upd = TRUE;
                              }
                              break;

                        #if defined (UNIX)
                           case TXc_P:          // Ctrl-Y is delayed-stop in most Unixes
                        #else
                           case TXc_Y:          // Ctrl-P is PRINT in many non-Unixes
                        #endif
                           case TXc_F9:
                              if (TxSelSortString( list, TXS_SORT_3) == TRUE) // list changed
                              {
                                 txwListSort2frhText( hwnd); // update footer indicator
                                 txwPositionListBox(  dat ); // fixup positions
                                 upd = TRUE;
                              }
                              break;

                           case TXc_T:
                           case TXc_F5:
                              if (TxSelSortString( list, TXS_SORT_4) == TRUE) // list changed
                              {
                                 txwListSort2frhText( hwnd); // update footer indicator
                                 txwPositionListBox(  dat ); // fixup positions
                                 upd = TRUE;
                              }
                              break;

                           case TXc_B:
                           case TXc_F6:
                              if (TxSelSortString( list, TXS_SORT_5) == TRUE) // list changed
                              {
                                 txwListSort2frhText( hwnd); // update footer indicator
                                 txwPositionListBox(  dat ); // fixup positions
                                 upd = TRUE;
                              }
                              break;

                           case TXc_N:
                           case TXc_F3:
                              if (TxSelSortString( list, TXS_SORT_6) == TRUE) // list changed
                              {
                                 txwListSort2frhText( hwnd); // update footer indicator
                                 txwPositionListBox(  dat ); // fixup positions
                                 upd = TRUE;
                              }
                              break;

                           case TXc_E:
                           case TXc_F4:
                              if (TxSelSortString( list, TXS_SORT_7) == TRUE) // list changed
                              {
                                 txwListSort2frhText( hwnd); // update footer indicator
                                 txwPositionListBox(  dat ); // fixup positions
                                 upd = TRUE;
                              }
                              break;

                           default:
                              break;
                        }
                     }
                     else
                     {
                        txwPostMsg( owner, TXWM_CHAR, mp1, mp2);
                     }
                     break;

                  case TXk_BACKSPACE:
                     if ((li = strlen(txwa->listmatch)) != 0)
                     {
                        txwa->listmatch[li -1] = '\0';
                     }
                     break;

                  case TXc_BACKSP:
                     strcpy( txwa->listmatch, ""); // reset string
                     break;

                  case TXa_A: case TXa_B:             case TXa_D: case TXa_E:
                  case TXa_F: case TXa_G: case TXa_H: case TXa_I: case TXa_J:
                  case TXa_K: case TXa_L: case TXa_M: case TXa_N: case TXa_O:
                  case TXa_P: case TXa_Q: case TXa_R: case TXa_S: case TXa_T:
                  case TXa_U:             case TXa_W: case TXa_X: case TXa_Y:               // most <Alt> + ASCII
                     if (list && (list->sortinfo))    //- Allow use as accelerator
                     {                                //- when no sorting specified (menu)
                        strcpy( txwa->listmatch, ""); // reset string
                        switch (key)
                        {
                           case TXa_A: key = (ULONG) 'a'; break;
                           case TXa_B: key = (ULONG) 'b'; break;
                           case TXa_C: key = (ULONG) 'c'; break;
                           case TXa_D: key = (ULONG) 'd'; break;
                           case TXa_E: key = (ULONG) 'e'; break;
                           case TXa_F: key = (ULONG) 'f'; break;
                           case TXa_G: key = (ULONG) 'g'; break;
                           case TXa_H: key = (ULONG) 'h'; break;
                           case TXa_I: key = (ULONG) 'i'; break;
                           case TXa_J: key = (ULONG) 'j'; break;
                           case TXa_K: key = (ULONG) 'k'; break;
                           case TXa_L: key = (ULONG) 'l'; break;
                           case TXa_M: key = (ULONG) 'm'; break;
                           case TXa_N: key = (ULONG) 'n'; break;
                           case TXa_O: key = (ULONG) 'o'; break;
                           case TXa_P: key = (ULONG) 'p'; break;
                           case TXa_Q: key = (ULONG) 'q'; break;
                           case TXa_R: key = (ULONG) 'r'; break;
                           case TXa_S: key = (ULONG) 's'; break;
                           case TXa_T: key = (ULONG) 't'; break;
                           case TXa_U: key = (ULONG) 'u'; break;
                           case TXa_V: key = (ULONG) 'v'; break;
                           case TXa_W: key = (ULONG) 'w'; break;
                           case TXa_X: key = (ULONG) 'x'; break;
                           case TXa_Y: key = (ULONG) 'y'; break;
                           case TXa_Z: key = (ULONG) 'z'; break;
                        }                       // <Alt> + ASCII fall through!
                     }                          // with modified mp2 == ASCII
                  default:                      // search ASCII select-characters

                     if (txwPossibleAsciiKey(key)     &&
                         (key != '<') && (key != '>') && // and no accelerator
                         (list->renderNewItem == NULL))
                     {
                        if (list->flags & TXSL_MULTI_QUICK)
                        {
                           strcat( txwa->listmatch, " ");
                           txwa->listmatch[ strlen( txwa->listmatch) -1] = (char) key;

                           for ( li = 0;
                                (li < list->count) && (upd == FALSE);
                                 li++)
                           {
                              item = list->items[li];

                              TRACES(("listmatch: '%s'  item: '%s'\n", txwa->listmatch, item->text));

                              if (strncasecmp( txwa->listmatch, item->text + list->multiQuickPos,
                                       strlen( txwa->listmatch)) == 0)
                              {
                                 TRACES(("Match for '%s' at item %lu = '%s'\n", txwa->listmatch, li, item->text));
                                 if (li < list->vsize)
                                 {
                                    dat->cpos = li;
                                    TxSelSetPosition( list, 0);
                                 }
                                 else
                                 {
                                    dat->cpos = list->vsize -1;
                                    TxSelSetPosition( list, li - dat->cpos);
                                 }
                                 upd = TRUE;    // force list-update
                              }
                           }
                           if (upd == FALSE)    // not found, remove char
                           {
                              txwa->listmatch[ strlen( txwa->listmatch) -1] = 0;
                           }
                        }
                        else                    // select on single char only
                        {
                           for ( li = 0;
                                (li < list->count) && (upd == FALSE);
                                 li++)
                           {
                              item = list->items[li];
                              if (item->index != 0) // explicit single quick-char
                              {
                                 select = item->text[ item->index -1];
                                 if ((tolower( select)) == (tolower((char) key)))
                                 {
                                    if (li < list->vsize)
                                    {
                                       dat->cpos = li;
                                       TxSelSetPosition( list, 0);
                                    }
                                    else
                                    {
                                       dat->cpos = list->vsize -1;
                                       TxSelSetPosition( list, li - dat->cpos);
                                    }
                                    upd = TRUE; // force list-update

                                    if (( win->style  & TXLS_CHAR_ENTER)  &&
                                        ((item->flags & TXSF_DISABLED) == 0))
                                    {
                                       txwPostMsg( owner, TXWM_CONTROL,
                                                   TXMPFROM2SH(wid,TXLN_ENTER),
                                                   (TXWMPARAM) list);
                                    }
                                 }
                              }
                           }
                        }
                     }
                     else
                     {
                        txwPostMsg( owner, TXWM_CHAR, mp1, (TXWMPARAM) key);
                     }
                     break;
               }
               if (upd)
               {
                  list->selected = (ULONG) (list->top + dat->cpos);
                  TRACES(( "dat->cpos:%hu top:%lu  selected now: %lu\n",
                            dat->cpos, list->top, list->selected));
               }
               if (list->flags & TXSL_MULTI_QUICK)
               {
                  if (strlen(txwa->listmatch) != 0)
                  {
                     sprintf( textbuf, "             Match: %s ", txwa->listmatch);
                     txwSetSbviewStatus( textbuf, cSchemeColor);
                  }
                  else
                  {
                     txwSetSbviewStatus( "", cSchemeColor);
                  }
               }
            }
            if (upd)
            {
               txwPostMsg( owner, TXWM_CONTROL,
                           TXMPFROM2SH(wid,TXLN_SELECT),
                           (TXWMPARAM) (list));
               rc = txwInvalidateWindow( hwnd, TRUE, TRUE);
            }
         }
         else                                   // no list available
         {
            TRACES(("NO list or EMPTY list attached to the window!\n"));

            txwPostMsg( owner, TXWM_CHAR, mp1, mp2); // allow things like Esc
         }
      }
      else
      {
         rc = TX_INVALID_DATA;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'txwIkeyListBox'
/*---------------------------------------------------------------------------*/

