#ifndef TXTWRIV_H
#define TXTWRIV_H
//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
//
// TX window library private interfaces
//
// Author: J. van Wijk
//
// Developed for LPTool/DFSee utilities
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license
// JvW  29-08-2001 Added dialog windows
// JvW  12-07-1998 Initial version

typedef struct txwinbase                        // private window and link info
{
   struct txwinbase   *parent;                  // parent window
   struct txwinbase   *owner;                   // owner  window
   TXWINDOW           *window;                  // actual window data
   ULONG              *acceltable;              // window accelerator table
   TXWINPROC           winproc;                 // window procedure
   TXPOINT             cursor;                  // relative cursor position
   BOOL                curvisible;              // cursor visibility
   USHORT              us[TXQWS_SIZE];          // window USHORT values
   ULONG               ul[TXQWL_SIZE];          // window ULONG  values
   void               *up[TXQWP_SIZE];          // window PTR    values
   TXRECT              restore;                 // size/pos for restore (a_F5)
   TXWHANDLE           oldFocus;                // previous focus window
   TXCELL             *oldContent;              // screen area save/restore
   char               *oldFooter;               // original footer text
   struct txwinbase   *next;                    // next window (visiting order)
   struct txwinbase   *prev;                    // previous window
} TXWINBASE;                                    // end of struct "txwinbase"

#define TXW_ARROW_STD              0x0000       // no special arrow handling
#define TXW_ARROW_MOVE             0x0001       // move a window
#define TXW_ARROW_SIZE             0x0002       // size a window
#define TXW_ARROW_COLOR            0x0004       // select color scheme

#define TXW_HIST_PLAIN             0x0000       // no completion, no unique
#define TXW_HIST_FILTER            0x0001       // completion filtering
#define TXW_HIST_UNIQUE            0x0002       // unique, duplicate removal
#define TXW_HIST_UNIFIL            0x0003       // unique, duplicate removal

typedef struct txw_anchor
{
   #if defined   (DEV32)
      TIB             *tib;                     // Thread Info Block
      PIB             *pib;                     // Process Info Block
      TXF_OS2DLLAPI   *api;                     // dynamic entrypoints
      ULONG            session;                 // startup sessiontype VIO/FS/PM
      TXSETVALUE       smMouseAction;           // startup MousAction   menu state
      TXSETVALUE       smDragTextMark;          // startup DragTextMark menu state
      ULONG            sysmouse;                // sysmouse setting 0..2
      BOOL             KbdKill;                 // KBD reader thread kill
   #elif defined (WIN32)
      ULONG            sysmouse;                // sysmouse setting 0..2
   #elif defined (LINUX)
      BOOL             xSelClip;                // X selection CLIPBOARD (not PRIMARY)
   #elif defined (DOS32)
      char            *dpmi1;                   // DPMI parameter  ptr,  512
      char            *dpmi2;                   // DPMI parameter  ptr,  512
      short            dpmi1Selector;
      short            dpmi2Selector;
   #endif
   int                 colors;                  // nr of colors in palette
   int                 autoInsert;              // Entryfield auto-insert length
   BOOL                dialogShowHidden;        // Show hidden files/dir in FileDialogs
   BOOL                insert;                  // global insert-mode
   BOOL                typeahead;               // typeahead enabled
   BOOL                resetterm;               // reset terminal on exit
   ULONG               arrowMode;               // in special arrow mode
   USHORT              historyMode;             // history filter/unique
   USHORT              autoid;                  // auto assigned window id
   ULONG               radixclass;              // radix class bits, 1=HEX
   short               cursorRow;               // saved cursor row
   short               cursorCol;               // saved cursor col
   TXRECT              screen;                  // screen rectangle
   USHORT              reopenMenu;              // menu-header ID to re-open
   short               mDragRow;                // absolute row and col start
   short               mDragCol;                // position on window drag
   ULONG               mDragFlags;              // mouse dragging flags
   BOOL                useMouse;                // mouse usage enabled
   BOOL                rmb2Menu;                // RMB click simulates <F10> for menu
   BOOL                pedantic;                // pedantic warnings/errors
   BOOL                rightmove;               // Menu style MOVE to next dropdown
   TXWHANDLE           capture;                 // mouse capture window
   TXWHANDLE           modality;                // modality window
   TXWINBASE          *focus;                   // focus window
   TXWINBASE          *desktop;                 // first window in chain
   TXWINBASE          *last;                    // last window in chain
   long                windows;                 // actual window count
   TXWINBASE          *sbview;                  // SBview for quick status
   TXWHANDLE           maincmd;                 // Entryfield marked as cmdline
   TXWHANDLE           mainmenu;                // Canvas marked as main menu
   BOOL                defaultStatus;           // show default status text
   ULONG               codepage;                // codepage, std 437 compatible
   ULONG              *acceltable;              // system accelerator table
   TXLN                listmatch;               // list match string value
} TXW_ANCHOR;                                   // end of struct "txw_anchor"

extern TXW_ANCHOR *txwa;                        // private anchor (HAB)

extern BYTE *txVbuf;                            // one line video buffer

// Validate window-handle, return its TXWBASE pointer, NULL is invalid
TXWINBASE *txwValidateHandle                    // RET   window base pointer
(
   TXWHANDLE           hwnd,                    // IN    handle to validate
   TXWINDOW          **win                      // OUT   attached window ptr
);

// Validate handle and calculate clip rectangle from all (parent) client areas
TXWINBASE *txwValidateAndClip                   // RET   window base pointer
(
   TXWHANDLE           hwnd,                    // IN    handle to validate
   TXWINDOW          **window,                  // OUT   attached window
   BOOL                border,                  // IN    start clip with border
   TXRECT             *pclip                    // OUT   combined clip rect
);

// Calculate visible region(s) for a 1-line string area
void txwVisibleLineRegions
(
   TXWINBASE          *wnd,                     // IN    window with string
   TXRECT_ELEM        *clr                      // INOUT list of clip rectangles
);                                              //       on input the full line

// Fast ScrollBuff paint, SBVIEW only; supports paint arround pop-up windows
void txwPaintSbView
(
   TXWINBASE          *wnd                      // IN    current window
);

// Default paint border-shadow for window, no tracing (for txwPaintSbView)
void txwPaintShadow
(
   TXWINBASE          *wnd                      // IN    current window
);

// Save window contents to be restored at destroy time (incl borders)
ULONG txwSaveWindowContent
(
   TXWINBASE          *wnd                      // IN    window
);

// Restore window contents saved at creation time (incl borders)
ULONG txwRestoreWindowContent
(
   TXWINBASE          *wnd,                     // IN    window
   TXRECT             *clip1,                   // IN    partial restore area 1
   TXRECT             *clip2                    // IN    partial restore area 2
);

// Default paint window border and focus indication, for any window-class
void txwPaintBorder
(
   TXWINBASE          *wnd,                     // IN    current window
   BOOL                focus                    // IN    focus indicator
);

// Default paint, for standard window-classes; draw client window contents
ULONG txwPaintWindow
(
   TXWINBASE          *wnd,                     // IN    current window
   BOOL                border                   // IN    paint border too
);

// Default paint status-indicator parts for a window-class
ULONG txwPaintWinStatus
(
   TXWINBASE          *wnd,                     // IN    current window
   char               *text,                    // IN    optional status text
   BYTE                color                    // IN    optional status color
);

// Find enabled and visible window that is 'next' to origin window
TXWHANDLE txwFindNextFocus
(
   TXWHANDLE           hwnd,                    // IN    origin window
   BOOL                tabgroup                 // IN    next GROUP, not window
);

// Find enabled and visible window that is 'previous' to origin window
TXWHANDLE txwFindPrevFocus
(
   TXWHANDLE           hwnd,                    // IN    origin window
   BOOL                tabgroup                 // IN    prev GROUP, not window
);

/*======== Generic Widgets,                       TXWIDGET.C ================*/

// Initialize generic-widget data structures
BOOL txwInitGenericWidgets
(
   void
);

// Terminate all generic-widget data structures
void txwTermGenericWidgets
(
   void
);

/*======== Standard dialog handling,              TXWSTDLG.C ================*/

// Initialize all standard dialog data structures
BOOL txwInitStandardDialogs
(
   void
);

// Terminate all standard dialog data structures
void txwTermStandardDialogs
(
   void
);

/*======== Keyboard processing for window classes, TXWIKEY.C ================*/

// Input key handling for the TEXTVIEW window class
ULONG txwIkeyTextview                           // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   TXWMPARAM           mp1,                     // IN    param 1
   TXWMPARAM           mp2                      // IN    param 2
);

// Input key and user-msg handling for the SBVIEW window class
ULONG txwSbViewWinProc                          // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    param 1
   TXWMPARAM           mp2                      // IN    param 2
);

// Input key handling for the ENTRYFIELD window class
ULONG txwIkeyEntryfield                         // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   TXWMPARAM           mp1,                     // IN    param 1
   TXWMPARAM           mp2                      // IN    param 2
);

// Input key handling for the BUTTON window class
ULONG txwIkeyButton                             // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   TXWMPARAM           mp1,                     // IN    param 1
   TXWMPARAM           mp2                      // IN    param 2
);

// Input key handling for the LISTBOX window class
ULONG txwIkeyListBox                            // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   TXWMPARAM           mp1,                     // IN    param 1
   TXWMPARAM           mp2                      // IN    param 2
);


/*======== Hex Editor dialog and class handling TXWHEXED.C ==================*/

// Hex Editor class, paint window
void txwPaintHexEdit
(
   TXWINBASE          *wnd                      // IN    current window
);

// Window procedure for the HEXED window class, handling mouse and keyboard
// Called AFTER generic default processing, as class specific default handling
ULONG txwHexEditWinProc                         // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    param 1
   TXWMPARAM           mp2                      // IN    param 2
);


// Initialize the Hex-editor dialog/control data structures
BOOL txwInitHexeditDialogs
(
   void
);

/*======== HELP related functions, TXWHELP.C ===============================*/


// Initialize the HELP dialog/control data structures and register its HELP
BOOL txwInitHelpDialogs
(
   void
);

/*======== Screen related functions, TXWIOP.C ===============================*/

#if defined (UNIX)
   #define TXSCREEN_BEGIN_UPDATE()  TxUnixUpdateScreen(FALSE)
   #define TXSCREEN_ENDOF_UPDATE()  TxUnixUpdateScreen(TRUE)
   #define TXSCREEN_FLUSH_CACHED()  TxUnixClearTerminal()


// refresh whole screen for UNIX terminal
void TxUnixUpdateScreen
(
   BOOL                finished                 // IN    Virtual updates done
);                                              //       will cause real screen
                                                //       update if outer level


// Check if screen-size in use matches desired size (as set, or terminal size)
BOOL TxUnixScreenSizeInValid                    // RET   Screen needs a resize
(
   void
);

// Clear terminal screen buffer, forcing complete redraw on next Update
void TxUnixClearTerminal
(
   void
);

#else
   #define TXSCREEN_FLUSH_CACHED()
   #define TXSCREEN_BEGIN_UPDATE()
   #define TXSCREEN_ENDOF_UPDATE()
#endif

#if defined (WIN32)
// Specific Win-NT console definitions and functions

// display a string on NT console, with full ANSI support
void txNtConsoleDisplay
(
   char *text
);
#endif


/*======== Message related functions, TXWMSG.C ==============================*/
// Initialize low level input handling
ULONG TxInputInitialize
(
   void
);

// Initialize input for windowed Desktop environment
ULONG TxInputDesktopInit
(
   void
);

// Terminate low level input handling
ULONG TxInputTerminate
(
   void
);

#endif
