#define INCL_DOSQUEUES
#define INCL_DOSMEMMGR
#define INCL_DOSPROCESS

#include <os2.h>
#include <stdio.h>

#include "mglserver_internal.h"
#include "pmapi.h"

static MGLDC *mglhdc = NULL;
static HQUEUE VideoSwitcherNotificationQueue = 0;
static char allowMGLAPI = 0, needPaletteRefresh = 0;

static FILE *dbfp;
static HMTX mode_switching_mux;

int VideoSwitcher( MGLDC *dc, int flags )
{
  int rc;

  allowMGLAPI = (flags == PM_REACTIVATE);

  rc = DosWriteQueue( VideoSwitcherNotificationQueue, 
    MGLC_VIDEO_SWITCH_NOTIFICATION, sizeof(int),
    (void *) (flags == PM_REACTIVATE), 0 );

  if ( rc )
  {
    fprintf( dbfp, "Error %d writing to client notification queue.\n", rc );
    fflush( dbfp );
  }

  fprintf( dbfp, "Video switch notification (%d) sent.\n", flags == PM_REACTIVATE );
  fflush( dbfp );

  if ( flags == PM_REACTIVATE )
  {
    DosReleaseMutexSem( mode_switching_mux );
    return PM_NO_SUSPEND_APP;
  } else {
    DosRequestMutexSem( mode_switching_mux, SEM_INDEFINITE_WAIT );
    return PM_SUSPEND_APP;
  }
}

int main( void ) {
  unsigned long rc, clientcount = 0, nextclient = 0, client_pid;
  HQUEUE command_queue;
  REQUESTDATA request;
  ULONG reqlen, numelem; char priority;
  PVOID data;
  char serverloop = 1;
  event_t evt;
  MGL_PALETTE_ENTRY shadow_pal[256] = {0,0,0};
  
  dbfp = fopen( "mglserver.log", "wa" );

  rc = DosCreateQueue( &command_queue, QUE_FIFO, MGLS_COMMAND_QUEUE_NAME );

  if ( rc == 332 ) { // Queue is a duplicate
    fprintf( dbfp, "This service is already running!?!\n" );
    fflush( dbfp );
    return 1;
  } else if ( rc != 0 ) {
    fprintf( dbfp, "Error %ld creating command queue.\n", rc );
    fflush( dbfp );
    return 1;
  }

  if ( rc )
  {
    fprintf( dbfp, "Error %d creating event semaphore for server queue.\n", rc );
  }

  while ( serverloop ) {
    rc = DosReadQueue( command_queue, &request, &reqlen, &data, 0,
      0, &priority, 0 );
    if ( rc )
    {
      fprintf( dbfp, "DosReadQueue returned %ld.\n", rc );
      fflush( dbfp );
      break;
    }

    switch ( request.ulData ) {
      case MGLS_INIT:
      {
        MGL_SERVER_INIT_PACKET *ip = data;

        mglhdc = NULL;

        fprintf( dbfp, "Init received!  Shared data at %lx\n", data );
        fflush( dbfp );

        rc = DosGetSharedMem( data, PAG_READ | PAG_WRITE );

        if ( rc ) {
          fprintf( dbfp, "Could not get the shared memory block.  RC=%ld.", rc );
          fflush( dbfp );
          ip->client_queue = 0;
          ip->input_queue_name[0] = 0;
          break;
        }

        rc = DosOpenEventSem( ip->client_semaphore_name, 
          &ip->client_wakeup );

        if ( rc )
        {
          fprintf( dbfp, "Server could not open the client's event semaphore!  Error %ld.\n", rc );
          fflush( dbfp );
          ip->client_queue = 0;
          ip->input_queue_name[0] = 0;
          DosFreeMem( data );
          break;
        }

        fprintf( dbfp, "Client wakeup semaphore %s was opened successfully.\n", ip->client_semaphore_name );
        fflush( dbfp );

        rc = DosOpenQueue( &client_pid, &(ip->client_queue), 
          ip->input_queue_name );

        if ( rc )
        {
          fprintf( dbfp, "Server could not open the client's listener queue!  Error %ld.\n", rc );
          fflush( dbfp );
          ip->client_queue = 0;
          ip->input_queue_name[0] = 0;
          DosFreeMem( data );
          DosPostEventSem( ip->client_wakeup );
          // Wake the client up because it's SOL
          DosCloseEventSem( ip->client_wakeup );
          break;
        }


        fprintf( dbfp, "Client queue %s has been opened.\n", ip->input_queue_name );
        fflush( dbfp );

        VideoSwitcherNotificationQueue = ip->client_queue;

        DosSetPriority( PRTYS_PROCESS, PRTYC_TIMECRITICAL, 31, 0 );

        if ( MGL_init( ".", NULL ) == 0 ) {
          MGL_fatalError( MGL_errorMsg(MGL_result()) );
          ip->client_queue = 0;
          ip->input_queue_name[0] = 0;
          DosPostEventSem( ip->client_wakeup );
          // Wake the client up because it's SOL
          DosCloseEventSem( ip->client_wakeup );
          DosFreeMem( data );
          DosSetPriority( PRTYS_PROCESS, PRTYC_REGULAR, 0, 0 );
          break;
        }

        DosSetPriority( PRTYS_PROCESS, PRTYC_REGULAR, 0, 0 );

        fprintf( dbfp, "MGL_init succeeded.\n" );
        fflush( dbfp );

        MGL_checkIdentityPalette(FALSE);

        clientcount++;
        nextclient++;

        allowMGLAPI = 1;

        rc = DosWriteQueue( VideoSwitcherNotificationQueue, 
          MGLC_VIDEO_SWITCH_NOTIFICATION, sizeof(int), (void *)1, 0 );

        if ( rc ) sprintf( ip->input_queue_name, "rc = %d", rc );

        rc = DosCreateMutexSem( NULL, &mode_switching_mux, 0, FALSE );

        if ( rc )
        {
          fprintf( dbfp, "Couldn't create mode switch mutex sem.  (rc=%d)\n",
            rc );
          fflush( dbfp );
        }

        DosPostEventSem( ip->client_wakeup );
        // Tell the client that the two-way communication tunnel
        // has been opened

        break;
      }
      case MGLS_SHUTDOWN:
      {
        MGL_SERVER_SHUTDOWN_PACKET *sp = data;

        DosRequestMutexSem( mode_switching_mux, SEM_INDEFINITE_WAIT );

        if ( mglhdc ) MGL_destroyDC( mglhdc );
        mglhdc = NULL;

        DosSetPriority( PRTYS_PROCESS, PRTYC_TIMECRITICAL, 31, 0 );
        MGL_exit();
        DosSetPriority( PRTYS_PROCESS, PRTYC_REGULAR, 0, 0 );

        DosReleaseMutexSem( mode_switching_mux );

        fprintf( dbfp, "Shutdown received!\n" );
        fflush( dbfp );

        DosCloseQueue( sp->client_queue );
        DosPostEventSem( sp->client_wakeup );
        DosCloseEventSem( sp->client_wakeup );
        DosCloseMutexSem( mode_switching_mux );

        --clientcount;
        if ( clientcount == 0 )
        {
          serverloop = 0;
        }

        DosFreeMem( data );
        // Remove access to this shared memory
        break;
      }
      case MGLS_INIT_VIDMODE:
      {
        MGL_SERVER_INIT_VIDMODE_PACKET *iv = data;
        int mode = -1;

        fprintf( dbfp, "Received request to init video mode (%dx%d %dbpp).\n",
          iv->width, iv->height, iv->depth );
        fflush( dbfp );

        DosRequestMutexSem( mode_switching_mux, SEM_INDEFINITE_WAIT );

        if ( mglhdc )
        {
          fprintf( dbfp, "Destroying old video mode context.\n" );
          fflush( dbfp );
          MGL_destroyDC( mglhdc );
        }

        if ( iv->flags & MGLS_VMIFLAG_USE_CUSTOM )
        {
          MGL_addCustomMode( iv->width, iv->height, iv->depth );
          fprintf( dbfp, "Custom mode creation attempted.\n" );
          fflush( dbfp );
        }

        fprintf( dbfp, "Attempting to find the mode.\n" );
        fflush( dbfp );

        iv->flags = 0;
        mode = MGL_findMode( iv->width, iv->height, iv->depth );

        mglhdc = NULL;

        if ( mode == -1 )
        {
          fprintf( dbfp, "Couldn't find mode %d x %d %dbpp.\n", 
            iv->width, iv->height, iv->depth );
          fflush( dbfp );
          DosPostEventSem( iv->client_wakeup );
          DosReleaseMutexSem( mode_switching_mux );
          break;
        }

        fprintf( dbfp, "Mode was found.  Attempting to create display.\n" );
        fflush( dbfp );

        mglhdc = MGL_createDisplayDC( mode, 1, MGL_DEFAULT_REFRESH );

        if ( mglhdc )
        {
          iv->flags |= MGLS_VMOFLAG_SUCCESS;
          fprintf( dbfp, "Display created.\n" );
          fprintf( dbfp, "Setting video mode switch callback and waking up client.\n" );
          fflush( dbfp );

          MGL_setSuspendAppCallback( VideoSwitcher );
        } else {
          fprintf( dbfp, "Display creation failed.  Reason reported by MGL:\n==> %s\nWaking up client.\n",
            MGL_errorMsg(MGL_result()) );
          fflush( dbfp );
        }

        needPaletteRefresh = 1;

        DosReleaseMutexSem( mode_switching_mux );
        DosPostEventSem( iv->client_wakeup );

        fprintf( dbfp, "Done with video mode init.\n" );
        fflush( dbfp );

        break;
      }
      case MGLS_SHUTDOWN_VIDMODE:
      {
        MGL_SERVER_SHUTDOWN_VIDMODE_PACKET *sv = data;

        DosRequestMutexSem( mode_switching_mux, SEM_INDEFINITE_WAIT );

        if ( mglhdc )
        {
          MGL_destroyDC( mglhdc );
          mglhdc = NULL;
        }

        DosReleaseMutexSem( mode_switching_mux );
        DosPostEventSem( sv->client_wakeup );

        break;
      }
      case MGLS_INIT_BUFFER:
      {
        MGL_SERVER_INIT_BUFFER_PACKET *ib = data;
        pixel_format_t tPixel, *tp;
        int mf = 1;

        rc = DosGetSharedMem( ib->buffer, PAG_READ | PAG_WRITE );

        if ( rc ) {
          fprintf( dbfp, "Could not get the shared memory block for video buffer.  RC=%ld.", rc );
          fflush( dbfp );
          ib->mdc = NULL;
          DosPostEventSem( ib->client_wakeup );
          break;
        }

        switch ( ib->depth )
        {
          case 8:
            tp = NULL;
          break;
          case 16:
            mf = 2;
            tp = &tPixel;

            tPixel.greenMask = 0x3f;
            tPixel.redMask = 0x1f;
            tPixel.blueMask = 0x1f;
            tPixel.rsvdMask = 0x00;

            tPixel.greenPos = 5;
            tPixel.bluePos = 0;
            tPixel.redPos = 11;
            tPixel.rsvdPos = 0;

            tPixel.greenAdjust = 0;
            tPixel.blueAdjust = 0;
            tPixel.redAdjust = 0;
            tPixel.rsvdAdjust = 0;
          break;
          case 24:
            mf = 3;
            tp = &tPixel;

            tPixel.greenMask = 0xff;
            tPixel.redMask = 0xff;
            tPixel.blueMask = 0xff;
            tPixel.rsvdMask = 0x00;

            tPixel.greenPos = 8;
            tPixel.bluePos = 16;
            tPixel.redPos = 0;
            tPixel.rsvdPos = 0;

            tPixel.greenAdjust = 0;
            tPixel.blueAdjust = 0;
            tPixel.redAdjust = 0;
            tPixel.rsvdAdjust = 0;
          break;
          case 32:
            mf = 4;
            tp = &tPixel;

            tPixel.greenMask = 0xff;
            tPixel.redMask = 0xff;
            tPixel.blueMask = 0xff;
            tPixel.rsvdMask = 0xff;

            tPixel.greenPos = 8;
            tPixel.bluePos = 16;
            tPixel.redPos = 0;
            tPixel.rsvdPos = 24;

            tPixel.greenAdjust = 0;
            tPixel.blueAdjust = 0;
            tPixel.redAdjust = 0;
            tPixel.rsvdAdjust = 0;
          break;
        }
        ib->mdc = MGL_createCustomDC( ib->width, ib->height,
          ib->depth, tp, ib->width * mf, ib->buffer, NULLHANDLE );
        
        DosPostEventSem( ib->client_wakeup );

        break;
      }
      case MGLS_FREE_BUFFER:
      {
        MGL_SERVER_FREE_BUFFER_PACKET *fb = data;
        void *tempdata = fb->mdc->surface;

        MGL_destroyDC( fb->mdc );
        DosFreeMem( tempdata );

        DosPostEventSem( fb->client_wakeup );

        break;
      }
      case MGLS_BLIT_BUFFER:
      {
        MGL_SERVER_BLIT_BUFFER_PACKET *bb = data;

        MGL_bitBltCoord( mglhdc, bb->mdc, bb->left, bb->top,
          bb->right, bb->bottom, bb->destx, bb->desty,
          MGL_REPLACE_MODE );

        DosPostEventSem( bb->client_wakeup );

        break;
      }
      case MGLS_STRETCHBLIT_BUFFER:
      {
        MGL_SERVER_STRETCHBLIT_BUFFER_PACKET *sb = data;

        MGL_stretchBltCoord( mglhdc, sb->mdc, sb->left, sb->top,
          sb->right, sb->bottom, sb->destleft, sb->desttop,
          sb->destright, sb->destbottom, MGL_REPLACE_MODE );

        DosPostEventSem( sb->client_wakeup );

        break;
      }
      case MGLS_EVENT_FLUSH:
      {
        // User should send one EVENT_FLUSH per frame
        static event_t evt;
        static int button = 0;

        if ( allowMGLAPI )
        {
          while ( EVT_getNext( &evt, EVT_EVERYEVT ) )
          {
            rc = 0;
            // Handle MGL input here.
            switch ( evt.what )
            {
              case EVT_KEYDOWN:
              case EVT_KEYREPEAT:
                rc = DosWriteQueue( VideoSwitcherNotificationQueue, 
                 MGLC_KEYDOWN_NOTIFICATION, sizeof(int),
                 (void *)((int)EVT_scanCode(evt.message)), 0 );
              break;
              case EVT_KEYUP:
                rc = DosWriteQueue( VideoSwitcherNotificationQueue, 
                 MGLC_KEYUP_NOTIFICATION, sizeof(int),
                 (void *)((int)EVT_scanCode(evt.message)), 0 );
              break;
              case EVT_MOUSEMOVE:
                rc = DosWriteQueue( VideoSwitcherNotificationQueue,
                 MGLC_MOUSEMOVE_NOTIFICATION, sizeof(int),
                 (void *)((evt.where_x & 0xffff) | ((evt.where_y & 0xffff)<<16)), 0 );
                // X position is encoded into low 16 bits, y is in the high 16
              break;
              case EVT_MOUSEDOWN:
                if ( evt.message & EVT_LEFTBMASK )   button |= 1;
                if ( evt.message & EVT_RIGHTBMASK )  button |= 2;
                if ( evt.message & EVT_MIDDLEBMASK ) button |= 4;
                rc = DosWriteQueue( VideoSwitcherNotificationQueue,
                 MGLC_MOUSEBUTTON_NOTIFICATION, sizeof(int),
                 (void *)(button), 0 );
              break;
              case EVT_MOUSEUP:
                if ( evt.message & EVT_LEFTBMASK )   button ^= 1;
                if ( evt.message & EVT_RIGHTBMASK )  button ^= 2;
                if ( evt.message & EVT_MIDDLEBMASK ) button ^= 4;
                rc = DosWriteQueue( VideoSwitcherNotificationQueue,
                 MGLC_MOUSEBUTTON_NOTIFICATION, sizeof(int),
                 (void *)(button), 0 );
              break;
            }
            if ( rc )
            {
              fprintf( dbfp, "Client listener queue write error #%d.\n", rc );
            }
          }
        }
        break;
      }
      case MGLS_SET_COLORS:
      {
        int i, firstchangedcol = 256, lastchangedcol = 0;
        MGL_SERVER_COLORS_SET_PACKET *cs = data;

        for ( i=0; i<256; ++i )
        {
          if ( ((unsigned long *)shadow_pal)[i] != 
               ((unsigned long *)cs->colors)[i] || needPaletteRefresh )
          {
            // Only set the entry if the color changed
            MGL_setPaletteEntry( mglhdc, i, cs->colors[i].red,
              cs->colors[i].green, cs->colors[i].blue );
            if ( i < firstchangedcol ) firstchangedcol = i;
            if ( i > lastchangedcol ) lastchangedcol = i;
          }
        }

        if ( firstchangedcol < 256 )
        {
          MGL_realizePalette( mglhdc, 1 + (lastchangedcol - firstchangedcol),
            firstchangedcol, MGL_dontWait );
        }

        needPaletteRefresh = 0;

        memcpy( shadow_pal, cs->colors, 1024 );
        // Update the shadow palette

        DosPostEventSem( cs->client_wakeup );

        break;
      }
      case MGLS_SET_MOUSE_POS:
      {
        MGL_SERVER_MOUSE_POS_PACKET *mp = data;

        EVT_setMousePos( mp->newx, mp->newy );
        DosPostEventSem( mp->client_wakeup );

        break;
      }
      default:
        fprintf( dbfp, "UNKNOWN QUEUE COMMAND!\n");
        fflush( dbfp );
        serverloop = 0;
    }
  }

  DosCloseQueue( command_queue );

  fprintf( dbfp, "Last client has exitted.  Shutting down server.\n" );
  fclose( dbfp );

  return 0;
}

