/* -*-C-*-

$Id$

Copyright (c) 1995 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. */

#define INCL_DOS
#define INCL_DOSERRORS
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ealib.h"

static ea_list *    read_eas (ULONG, PVOID);
static ea_binding * read_ea_by_index (ULONG, PVOID, ULONG);
static ea_binding * read_ea_by_name (ULONG, PVOID, PSZ);
static PDENA2       read_dena_by_index (ULONG, PVOID, ULONG);
static ea_binding * get_ea_value (ULONG, PVOID, PDENA2);
static void         setup_query_ea_info (PDENA2, PEAOP2);
static ea_binding * convert_feal_to_binding (PFEA2LIST);
static void         write_eas (ULONG, PVOID, ea_list *);
static void         write_ea (ULONG, PVOID, ea_binding *);
static PFEA2LIST    convert_binding_to_feal (ea_binding *);
static void         fatal_api_call_error (APIRET, const char *);
static void *       xmalloc (unsigned int);

#define API_CALL(proc, args) do						\
{									\
  APIRET API_CALL_rc = (proc args);					\
  if (API_CALL_rc != NO_ERROR)						\
    fatal_api_call_error (API_CALL_rc, #proc);				\
} while (0)

#define MALLOC xmalloc
#define FREE(ptr) free ((void *) (ptr))

ea_list *
read_file_eas (HFILE file_handle)
{
  return (read_eas (ENUMEA_REFTYPE_FHANDLE, (& file_handle)));
}

ea_list *
read_path_eas (PSZ path)
{
  return (read_eas (ENUMEA_REFTYPE_PATH, path));
}

ea_binding *
read_file_ea_by_index (HFILE file_handle, ULONG index)
{
  return (read_ea_by_index (ENUMEA_REFTYPE_FHANDLE, (& file_handle), index));
}

ea_binding *
read_path_ea_by_index (PSZ path, ULONG index)
{
  return (read_ea_by_index (ENUMEA_REFTYPE_PATH, path, index));
}

ea_binding *
read_file_ea_by_name (HFILE file_handle, PSZ name)
{
  return (read_ea_by_name (ENUMEA_REFTYPE_FHANDLE, (& file_handle), name));
}

ea_binding *
read_path_ea_by_name (PSZ path, PSZ name)
{
  return (read_ea_by_name (ENUMEA_REFTYPE_PATH, path, name));
}

void
write_file_eas (HFILE file_handle, ea_list * list)
{
  write_eas (ENUMEA_REFTYPE_FHANDLE, (& file_handle), list);
}

void
write_path_eas (PSZ path, ea_list * list)
{
  write_eas (ENUMEA_REFTYPE_PATH, path, list);
}

void
write_file_ea (HFILE file_handle, ea_binding * binding)
{
  write_ea (ENUMEA_REFTYPE_FHANDLE, (& file_handle), binding);
}

void
write_path_ea (PSZ path, ea_binding * binding)
{
  write_ea (ENUMEA_REFTYPE_PATH, path, binding);
}

void
free_ea_list (ea_list * list)
{
  while (list != 0)
    {
      ea_list * next = (EA_LIST_NEXT (list));
      free_ea_binding (EA_LIST_BINDING (list));
      FREE (list);
      list = next;
    }
}

void
free_ea_binding (ea_binding * binding)
{
  FREE (EA_BINDING_NAME (binding));
  if ((EA_BINDING_VALUE (binding)) != 0)
    FREE (EA_BINDING_VALUE (binding));
  FREE (binding);
}

static ea_list *
read_eas (ULONG type, PVOID pfile)
{
  ULONG index = 1;
  ea_list * head = 0;
  ea_list * tail = 0;
  while (1)
    {
      ea_binding * binding = (read_ea_by_index (type, pfile, index));
      if (binding == 0)
	break;
      {
	ea_list * list = (MALLOC (sizeof (ea_list)));
	(EA_LIST_BINDING (list)) = binding;
	(EA_LIST_NEXT (list)) = 0;
	if (head == 0)
	  head = list;
	else
	  (EA_LIST_NEXT (tail)) = list;
	tail = list;
      }
      index += 1;
    }
  return (head);
}

static ea_binding *
read_ea_by_index (ULONG type, PVOID pfile, ULONG index)
{
  PDENA2 dena = (read_dena_by_index (type, pfile, index));
  return ((dena == 0) ? 0 : (get_ea_value (type, pfile, dena)));
}

static ea_binding *
read_ea_by_name (ULONG type, PVOID pfile, PSZ name)
{
  ULONG index = 1;
  while (1)
    {
      PDENA2 dena = (read_dena_by_index (type, pfile, index));
      if (dena == 0)
	return (0);
      if ((strcmp (name, (dena -> szName))) == 0)
	return (get_ea_value (type, pfile, dena));
      FREE (dena);
      index += 1;
    }
}

static PDENA2
read_dena_by_index (ULONG type, PVOID pfile, ULONG index)
{
  ULONG count = 1;
  PDENA2 dena = (MALLOC (500));	/* 500 is magic -- IBM doesn't explain. */
  API_CALL (DosEnumAttribute, (type, pfile, index, dena, 500, (& count),
			       ENUMEA_LEVEL_NO_VALUE));
  if (count == 0)
    {
      FREE (dena);
      return (0);
    }
  else
    return (dena);
}

static ea_binding *
get_ea_value (ULONG type, PVOID pfile, PDENA2 dena)
{
  ULONG level = FIL_QUERYEASFROMLIST;
  EAOP2 eaop;
  ULONG size = (sizeof (eaop));
  setup_query_ea_info (dena, (& eaop));
  if (type == ENUMEA_REFTYPE_FHANDLE)
    API_CALL (DosQueryFileInfo, ((* ((PHFILE) pfile)), level, (& eaop), size));
  else
    API_CALL (DosQueryPathInfo, (pfile, level, (& eaop), size));
  FREE (eaop . fpGEA2List);
  return (convert_feal_to_binding (eaop . fpFEA2List));
}

static void
setup_query_ea_info (PDENA2 dena, PEAOP2 eaop)
{
  unsigned int geal_size = ((sizeof (GEA2LIST)) + (dena -> cbName));
  unsigned int feal_size
    = ((sizeof (FEA2LIST)) + (dena -> cbName) + (dena -> cbValue));
  (eaop -> fpGEA2List) = (MALLOC (geal_size));
  ((eaop -> fpGEA2List) -> cbList) = geal_size;
  (eaop -> fpFEA2List) = (MALLOC (feal_size));
  ((eaop -> fpFEA2List) -> cbList) = feal_size;
  (eaop -> oError) = 0;
  {
    PGEA2 gea = (& (((eaop -> fpGEA2List) -> list) [0]));
    (gea -> oNextEntryOffset) = 0;
    (gea -> cbName) = (dena -> cbName);
    strcpy ((gea -> szName), (dena -> szName));
  }
  FREE (dena);
}

static ea_binding *
convert_feal_to_binding (PFEA2LIST feal)
{
  PFEA2 fea = (& ((feal -> list) [0]));
  ea_binding * binding = (MALLOC (sizeof (ea_binding)));
  (EA_BINDING_FLAGS (binding)) = (fea -> fEA);
  (EA_BINDING_NAME_LENGTH (binding)) = (fea -> cbName);
  (EA_BINDING_VALUE_LENGTH (binding)) = (fea -> cbValue);
  (EA_BINDING_NAME (binding)) = (MALLOC ((fea -> cbName) + 1));
  strcpy ((EA_BINDING_NAME (binding)), (fea -> szName));
  (EA_BINDING_VALUE (binding)) = (MALLOC (fea -> cbValue));
  memcpy ((EA_BINDING_VALUE (binding)),
	  (& ((fea -> szName) [(fea -> cbName) + 1])),
	  (fea -> cbValue));
  FREE (feal);
  return (binding);
}

static void
write_eas (ULONG type, PVOID pfile, ea_list * list)
{
  while (list != 0)
    {
      write_ea (type, pfile, (EA_LIST_BINDING (list)));
      list = (EA_LIST_NEXT (list));
    }
}

static void
write_ea (ULONG type, PVOID pfile, ea_binding * binding)
{
  ULONG level = FIL_QUERYEASIZE;
  EAOP2 eaop;
  ULONG size = (sizeof (eaop));
  (eaop . fpGEA2List) = 0;
  (eaop . fpFEA2List) = (convert_binding_to_feal (binding));
  (eaop . oError) = 0;
  if (type == ENUMEA_REFTYPE_FHANDLE)
    API_CALL (DosSetFileInfo, ((* ((PHFILE) pfile)), level, (& eaop), size));
  else
    API_CALL (DosSetPathInfo, (pfile, level, (& eaop), size, DSPI_WRTTHRU));
  FREE (eaop . fpFEA2List);
}

static PFEA2LIST
convert_binding_to_feal (ea_binding * binding)
{
  unsigned int feal_size
    = ((sizeof (FEA2LIST))
       + (EA_BINDING_NAME_LENGTH (binding))
       + (EA_BINDING_VALUE_LENGTH (binding)));
  PFEA2LIST feal = (MALLOC (feal_size));
  PFEA2 fea = (& ((feal -> list) [0]));
  (feal -> cbList) = feal_size;
  (fea -> oNextEntryOffset) = 0;
  (fea -> fEA) = (EA_BINDING_FLAGS (binding));
  (fea -> cbName) = (EA_BINDING_NAME_LENGTH (binding));
  (fea -> cbValue) = (EA_BINDING_VALUE_LENGTH (binding));
  strcpy ((fea -> szName), (EA_BINDING_NAME (binding)));
  if ((EA_BINDING_VALUE (binding)) != 0)
    memcpy ((& ((fea -> szName) [(fea -> cbName) + 1])),
	    (EA_BINDING_VALUE (binding)),
	    (fea -> cbValue));
  return (feal);
}

void (* ea_error_hook) (APIRET, const char *);

static void
fatal_api_call_error (APIRET rc, const char * name)
{
  if (ea_error_hook != 0)
    (* ea_error_hook) (rc, name);
  else
    {
      fprintf (stderr, "Fatal error %d in procedure %s.\n", rc, name);
      fflush (stderr);
    }
  exit (EXIT_FAILURE);
}

static void *
xmalloc (unsigned int nbytes)
{
  void * result = (malloc (nbytes));
  if (result == 0)
    {
      fprintf (stderr, "Unable to allocate %d bytes of memory.\n", nbytes);
      fflush (stderr);
      exit (EXIT_FAILURE);
    }
  return (result);
}
