(**************************************************************************)
(*                                                                        *)
(*  Program to work out module dependencies in a Modula-2 program.        *)
(*  Copyright (C) 2019   Peter Moylan                                     *)
(*                                                                        *)
(*  This program is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation, either version 3 of the License, or     *)
(*  (at your option) any later version.                                   *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful,       *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU General Public License for more details.                          *)
(*                                                                        *)
(*  You should have received a copy of the GNU General Public License     *)
(*  along with this program.  If not, see <http://www.gnu.org/licenses/>. *)
(*                                                                        *)
(*  To contact author:   http://www.pmoylan.org   peter@pmoylan.org       *)
(*                                                                        *)
(**************************************************************************)

IMPLEMENTATION MODULE Scanner;

        (********************************************************)
        (*                                                      *)
        (*              Simplified lexical analyser             *)
        (*                                                      *)
        (*  New experimental version, not yet in production     *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            14 January 2000                 *)
        (*  Last edited:        25 September 2019               *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (*    Remark: For checking imports we need to parse     *)
        (*    only a small subset of the language, and one      *)
        (*    consequence of this is that we can get away       *)
        (*    with a very crude lexical analyser.  We need      *)
        (*    only detect whitespace, comments, alphanumeric    *)
        (*    strings starting with a letter; everything else   *)
        (*    is interpreted as a one-character token.          *)
        (*                                                      *)
        (********************************************************)

IMPORT TextIO, STextIO, Strings, IOChan, IOConsts, ChanConsts, SeqFile;

(************************************************************************)

CONST
    CharsPerLine = 1024;
    Nul = CHR(0);  Tab = CHR(9);  CtrlZ = CHR(26);
    OpenPragma = CHR(1);
    ClosePragma = CHR(2);
    MaxLevel = 15;
    MaxSyms = 32767;
    MaxSymLength = 256;

TYPE
    LineIndex = [0..CharsPerLine-1];
    CharSet = SET OF CHAR;
    LineBufferType = ARRAY LineIndex OF CHAR;
    TokenType = LineBufferType;
    SymbolType = ARRAY [0..MaxSymLength-1] OF CHAR;

    (* In XDS pragmas and project files, a variable can be a Boolean    *)
    (* or a string, and some variables can also be synonyms.  To deal   *)
    (* with this, we carry along the type as part of the value.         *)

    ValueKind = (false, true, syn, str);
    ValueType = RECORD
                    kind: ValueKind;
                    strval: SymbolType;
                END (*RECORD*);

VAR
    (* No multithreading, so we can afford to use a global "current line". *)

    LineBuffer: LineBufferType;
    NextPos: CARDINAL;
    fileid: IOChan.ChanId;

    (* Variable level is the current level of nesting of IF pragmas     *)
    (* (with level = 0 if we are not currently within any IF), and      *)
    (* condval[level] is what the IF condition evaluated to.              *)

    level: CARDINAL;
    condval: ARRAY [0..MaxLevel] OF BOOLEAN;

    (* The symbol table.  *)

    SymTbl: ARRAY [0..MaxSyms-1] OF
        RECORD
            symbol: SymbolType;
            val2: ValueType;
        END (*RECORD*);

CONST
    Letters = CharSet{'A'..'Z', 'a'..'z', '_'};
    IdChars = Letters + CharSet {'0'..'9', '$'};
    Digits = CharSet{'0'..'9'};
    NumChars = Digits + CharSet {'A'..'F', 'a'..'f', 'H'};
    FalseVal = ValueType {false, ''};

(************************************************************************)
(*                            MISCELLANEOUS                             *)
(************************************************************************)

PROCEDURE ErrMes(text: ARRAY OF CHAR);

    BEGIN
        STextIO.WriteString (text);  STextIO.WriteLn;
    END ErrMes;

(************************************************************************)

PROCEDURE StringMatch (a, b: ARRAY OF CHAR): BOOLEAN;

    (* Compares a and b in a case-independent way. *)

    VAR j: CARDINAL;

    BEGIN
        IF LENGTH(a) <> LENGTH(b) THEN
            RETURN FALSE;
        END (*IF*);
        j := 0;
        LOOP
            IF a[j] = Nul THEN RETURN TRUE;
            ELSIF CAP(a[j]) <> CAP(b[j]) THEN RETURN FALSE;
            ELSE INC(j);
            END (*IF*);
        END (*LOOP*);
    END StringMatch;

(********************************************************************************)

PROCEDURE StripSpaces (VAR (*INOUT*) string: ARRAY OF CHAR);

    (* Removes leading and trailing spaces from string. *)

    VAR k: CARDINAL;

    BEGIN
        k := Strings.Length (string);
        WHILE (k > 0) AND ((string[k-1] = ' ') OR (string[k-1] = Tab)) DO
            DEC (k);
        END (*WHILE*);
        string[k] := Nul;
        k := 0;
        WHILE (string[k] = ' ') OR (string[k] = Tab) DO
            INC (k);
        END (*WHILE*);
        IF k > 0 THEN
            Strings.Delete (string, 0, k);
        END (*IF*);
    END StripSpaces;

(************************************************************************)
(*                          THE SYMBOL TABLE                            *)
(************************************************************************)

PROCEDURE PosInSymTbl (VAR (*IN*) name: ARRAY OF CHAR): CARDINAL;

    (* Returns the position in the symbol table of 'name'. If not       *)
    (* found, returns the position of the first unused entry.           *)

    VAR pos: CARDINAL;

    BEGIN
        pos := 0;
        LOOP
            IF SymTbl[pos].symbol[0] = Nul THEN
                RETURN pos;
            ELSIF Strings.Equal (name, SymTbl[pos].symbol) THEN
                RETURN pos;
            ELSE
                INC (pos);
            END (*IF*);
        END (*LOOP*);
    END PosInSymTbl;

(************************************************************************)

PROCEDURE Search (VAR (*IN*) token: ARRAY OF CHAR;  expectstr: BOOLEAN;
                                        VAR (*OUT*) tokenval: ValueType);

    (* Looks up the identifier token in the symbol table, returns its   *)
    (* position in the table and its value.                             *)

    (* The expectstr parameter in this and several later procedures     *)
    (* says what to do if token is not found in the symbol table.  If   *)
    (* expectstr is TRUE, we expect to find a string value, so we       *)
    (* return token as a literal string value.  Otherwise, we expect to *)
    (* find a Boolean result, so we return Boolean FALSE as the result. *)

    VAR pos: CARDINAL;

    BEGIN
        pos := PosInSymTbl (token);
        IF SymTbl[pos].symbol[0] <> Nul THEN
            tokenval := SymTbl[pos].val2;
        ELSIF expectstr THEN
            tokenval.kind := str;
            Strings.Assign (token, tokenval.strval);
        ELSE
            tokenval := FalseVal;
        END (*IF*);
    END Search;

(************************************************************************)

PROCEDURE IsDefined (VAR (*IN*) token: ARRAY OF CHAR): BOOLEAN;

    (* Returns TRUE iff token is in the symbol table.   *)

    VAR pos: CARDINAL;

    BEGIN
        pos := PosInSymTbl (token);
        RETURN SymTbl[pos].symbol[0] <> Nul;
    END IsDefined;

(************************************************************************)

PROCEDURE SetValue (name: ARRAY OF CHAR;  vkind: ValueKind;
                                        strval: ARRAY OF CHAR);

    (* Stores a value in the symbol table. *)

    VAR pos: CARDINAL;
        v: ValueType;

    BEGIN
        pos := PosInSymTbl (name);
        IF SymTbl[pos].symbol[0] = Nul THEN
            (* New entry. *)
            Strings.Assign (name, SymTbl[pos].symbol);
            IF pos < MaxSyms THEN
                SymTbl[pos+1].symbol[0] := Nul;
            END (*IF*);
        END (*IF*);
        v.kind := vkind;
        Strings.Assign (strval, v.strval);
        SymTbl[pos].val2 := v;
    END SetValue;

(************************************************************************)

PROCEDURE SetBoolValue (name: ARRAY OF CHAR;  value: BOOLEAN);

    (* Stores a Boolean value in the symbol table. *)

    VAR kind: ValueKind;

    BEGIN
        IF value THEN kind := true;
        ELSE kind := false;
        END (*IF*);
        SetValue (name, kind, '');
    END SetBoolValue;

(************************************************************************)

PROCEDURE SetStringValue (name, value: ARRAY OF CHAR);

    (* Stores a string value in the symbol table. *)

    BEGIN
        SetValue (name, str, value);
    END SetStringValue;

(************************************************************************)

PROCEDURE SetSynonym (name1, name2: ARRAY OF CHAR);

    (* Stores name1 as a synonym of name2. *)

    BEGIN
        SetValue (name1, syn, name2);
    END SetSynonym;

(************************************************************************)

PROCEDURE BooleanToVT (x: BOOLEAN): ValueType;

    (* Returns a result equivalent to Boolean value x.  *)

    VAR result: ValueType;

    BEGIN
        result.strval := '';
        IF x THEN
            result.kind := true;
        ELSE
            result.kind := false;
        END (*IF*);
        RETURN result;
    END BooleanToVT;

(************************************************************************)

PROCEDURE ForceBoolean (x: ValueType): ValueType;

    (* Ensures that the result is not a string.  *)

    VAR result: ValueType;
        message: ARRAY [0..511] OF CHAR;

    BEGIN
        result.strval := '';
        result.kind := x.kind;
        IF x.kind = str THEN
            IF Strings.Equal (x.strval, "TRUE") THEN
                result.kind := true;
            ELSIF Strings.Equal (x.strval, "FALSE") THEN
                result.kind := true;
            ELSE
                Strings.Assign (x.strval, message);
                Strings.Append (" is not a Boolean value.", message);
                ErrMes (message);
                IF x.strval[0] = Nul THEN
                    result.kind := false;
                ELSE
                    result.kind := true;
                END (*IF*);
            END (*IF*);
        END (*IF*);
        result.strval := '';
        RETURN result;
    END ForceBoolean;

(************************************************************************)

PROCEDURE VTtoBoolean (x: ValueType): BOOLEAN;

    (* Converts ValueType to BOOLEAN. *)

    VAR y: ValueType;

    BEGIN
        y := ForceBoolean(x);
        IF y.kind = true THEN RETURN TRUE;
        ELSE RETURN FALSE;
        END (*IF*);
    END VTtoBoolean;

(************************************************************************)

PROCEDURE IsEqual (x, y: ValueType): BOOLEAN;

    (* Returns TRUE iff x = y.  String comparisons are case-independent. *)
    (* If a comparison between a Boolean value and a string value is    *)
    (* attempted, we force the string value to Boolean first.           *)

    VAR z: ValueType;

    BEGIN
        (* Beware of mismatched data types. *)

        IF x.kind = str THEN
            IF y.kind = str THEN
                RETURN StringMatch (x.strval, y.strval);
            ELSE
                z := ForceBoolean(x);
                RETURN z.kind = y.kind;
            END (*IF*);
        ELSE
            IF y.kind = str THEN
                z := ForceBoolean(y);
                RETURN z.kind = x.kind;
            ELSE
                RETURN x.kind = y.kind;
            END (*IF*);
        END (*IF*);

    END IsEqual;

(************************************************************************)
(*                       READING FROM INPUT FILE                        *)
(************************************************************************)

PROCEDURE StartScan (cid: IOChan.ChanId);

    (* Resets the scanner to work with a new file. *)

    BEGIN
        fileid := cid;
        LineBuffer := "";  NextPos := 0;
        level := 0;
    END StartScan;

(************************************************************************)

PROCEDURE ReadLine (cid: IOChan.ChanId;  VAR (*OUT*) line: ARRAY OF CHAR);

    (* Reads a new line into LineBuffer, resets NextPos. *)

    VAR status: IOConsts.ReadResults;

    BEGIN
        TextIO.ReadRestLine (cid, line);
        NextPos := 0;
        status := IOChan.ReadResult(cid);
        IF (status <> IOConsts.allRight) AND (status <> IOConsts.endOfLine) THEN
            line[0] := CtrlZ;
        END (*IF*);
        TextIO.SkipLine (cid);
    END ReadLine;

(************************************************************************)

PROCEDURE GetNextLine;

    (* Reads a new line into LineBuffer, resets NextPos. *)

    BEGIN
        ReadLine (fileid, LineBuffer);
    END GetNextLine;

(************************************************************************)
(*                      SKIPPING SPACES AND COMMENTS                    *)
(************************************************************************)

PROCEDURE SkipComments;

    (* Skips over comments, including nested comments.  On entry we     *)
    (* have passed the '(' and are looking at the '*'.                  *)

    VAR ch: CHAR;

    BEGIN
        INC (NextPos);
        LOOP
            ch := LineBuffer[NextPos];  INC(NextPos);
            IF ch = Nul THEN
                GetNextLine;
            ELSIF ch = CtrlZ THEN
                EXIT (*LOOP*);
            ELSIF (ch = '(') AND (LineBuffer[NextPos] = '*') THEN
                SkipComments;
            ELSIF (ch = '*') AND (LineBuffer[NextPos] = ')') THEN
                INC (NextPos);
                EXIT (*LOOP*);
            END (*IF*);
        END (*LOOP*);
    END SkipComments;

(************************************************************************)

PROCEDURE SpaceSkip (VAR (*IN*) line: ARRAY OF CHAR;  VAR (*INOUT*) pos: CARDINAL);

    (* Skips over whitespace in a single line. *)

    BEGIN
        WHILE (line[pos] = ' ') OR (line[pos] = Tab) DO
            INC (pos);
        END (*WHILE*);
    END SpaceSkip;

(************************************************************************)

PROCEDURE SkipBlanks;

    (* Skips over whitespace and comments. *)

    VAR ch: CHAR;

    BEGIN
        LOOP
            SpaceSkip (LineBuffer, NextPos);
            ch := LineBuffer[NextPos];
            WHILE ch = Nul DO
                GetNextLine;
                SpaceSkip (LineBuffer, NextPos);
                ch := LineBuffer[NextPos];
            END (*WHILE*);
            IF ch = CtrlZ THEN
                EXIT (*LOOP*);
            END (*IF*);
            IF (ch = '(') AND (LineBuffer[NextPos+1] = '*') THEN
                SkipComments;
            ELSE
                EXIT (*LOOP*);
            END (*IF*);
        END (*LOOP*);
    END SkipBlanks;

(************************************************************************)

PROCEDURE GetTokenFromString (VAR (*IN*) str: ARRAY OF CHAR;
                        VAR (*INOUT*) pos: CARDINAL;
                        VAR (*OUT*) token: ARRAY OF CHAR);

    (* Loads token from str[pos], updates pos.  We assume that leading  *)
    (* spaces have already been skipped.                                *)

    VAR k: CARDINAL;

    BEGIN
        IF str[pos] IN Letters THEN
            (* Alphanumeric *)
            k := 0;
            REPEAT
                token[k] := str[pos];
                INC (k);  INC (pos);
            UNTIL (k > HIGH(token)) OR NOT (str[pos] IN IdChars);
            IF k <= HIGH(token) THEN
                token[k] := Nul;
            END (*IF*);
        ELSIF str[pos] IN Digits THEN
            (* Alphanumeric *)
            k := 0;
            REPEAT
                token[k] := str[pos];
                INC (k);  INC (pos);
            UNTIL (k > HIGH(token)) OR NOT (str[pos] IN NumChars);
            IF k <= HIGH(token) THEN
                token[k] := Nul;
            END (*IF*);
        ELSE
            token[0] := str[pos];
            IF token[0] = CtrlZ THEN
                token[0] := Nul;
            ELSE
                INC (pos);
            END (*IF*);

            (* Handle the special cases of two-character tokens. *)

            IF (token[0] = '<') AND (str[pos] = '*') THEN
                token[0] := OpenPragma;
                INC (pos);
            ELSIF (token[0] = '*') AND (str[pos] = '>') THEN
                token[0] := ClosePragma;
                INC (pos);
            END (*IF*);

            token[1] := Nul;
        END (*IF*);
    END GetTokenFromString;

(************************************************************************)

PROCEDURE GetToken (VAR (*OUT*) token: ARRAY OF CHAR);

    (* Returns the next input token.  *)

    BEGIN
        SkipBlanks;
        GetTokenFromString (LineBuffer, NextPos, token);
    END GetToken;

(************************************************************************)
(*                        OPTIONS AND EQUATIONS                         *)
(************************************************************************)

PROCEDURE InlineOption (VAR (*IN*) token: ARRAY OF CHAR;
                          VAR (*INOUT*) text: ARRAY OF CHAR;
                            VAR (*INOUT*) pos: CARDINAL);

    (* This is the same as HandleOption, see below, except that on      *)
    (* entry the first token has already been scanned.                  *)

    VAR token2: SymbolType;

    BEGIN
        IF token[0] = '+' THEN

            (* Oldstyle option setting. *)

            SpaceSkip (text, pos);
            GetTokenFromString (text, pos, token);
            SetBoolValue (token, TRUE);

        ELSE
            IF token[0] = '-' THEN

                (* Oldstyle option clearing, or new style directive in  *)
                (* a project file.  Inline pragmas don't have the       *)
                (* leading '-', because the XDS design never got to the *)
                (* point of making these various files more consistent  *)
                (* with one another, but we can get away with just      *)
                (* skipping it.                                         *)

                SpaceSkip (text, pos);
                GetTokenFromString (text, pos, token);

            END (*IF*);

            IF Strings.Equal(token, "PUSH") OR Strings.Equal(token, "POP") THEN

                (* Currently we don't handle the PUSH and POP directives. *)

                RETURN;

            END (*IF*);

            IF Strings.Equal (token, "NEW") THEN

                (* The NEW keyword declares a new option or equation,   *)
                (* as distinct from altering an old value, but for our  *)
                (* purposes we don't need to distinguish between old    *)
                (* and new values.                                      *)

                INC (pos);
                SpaceSkip (text, pos);
                GetTokenFromString (text, pos, token);

            END (*IF*);

            SpaceSkip (text, pos);
            IF text[pos] = ':' THEN

                (* In a project file the colon has the same function as *)
                (* the NEW keyword above, and again we can ignore it.   *)

                INC (pos);
                SpaceSkip (text, pos);
            END (*IF*);

            IF text[pos] = Nul THEN
                (* Oldstyle setting. *)

                SetBoolValue (token, FALSE);
            ELSIF text[pos] = '-' THEN
                INC (pos);
                SetBoolValue (token, FALSE);
            ELSIF text[pos] = '+' THEN
                INC (pos);
                SetBoolValue (token, TRUE);
            ELSIF text[pos] = '=' THEN

                (* Assigning a string value to token. *)

                INC (pos);
                SpaceSkip (text, pos);
                GetTokenFromString (text, pos, token2);
                SetStringValue (token, token2);

            ELSIF text[pos] = ':' THEN

                (* A second colon; this declares a synonym. *)

                INC (pos);
                SpaceSkip (text, pos);
                GetTokenFromString (text, pos, token2);
                SetSynonym (token, token2);

            ELSE
                (* Ignore all other cases. *)
            END (*IF*);
        END (*IF*);
    END InlineOption;

(************************************************************************)

PROCEDURE HandleOption (VAR (*IN*) text: ARRAY OF CHAR);

    (* Handles a line in a project file or the pragma body in an inline *)
    (* pragma.  (Except for conditional compilation pragmas, which are  *)
    (* handled elsewhere in this module.)  The syntax is different for  *)
    (* the two cases, but there is enough overlap that we can combine   *)
    (* the code.  Our only interest is in storing option values in our  *)
    (* symbol table, so we just ignore features we don't need.  We      *)
    (* don't check for syntax errors because the compiler does that.    *)

    VAR token: SymbolType;
        pos: CARDINAL;

    BEGIN
        pos := 0;
        SpaceSkip (text, pos);
        GetTokenFromString (text, pos, token);
        InlineOption (token, text, pos);
    END HandleOption;

(************************************************************************)

PROCEDURE ExtractOptions (cid: IOChan.ChanId);

    VAR buffer: ARRAY [0..255] OF CHAR;

    BEGIN
        LOOP
            ReadLine (cid, buffer);
            IF buffer[0] = CtrlZ THEN
                EXIT (*LOOP*);
            END (*IF*);
            HandleOption (buffer);
        END (*LOOP*);
    END ExtractOptions;

(************************************************************************)

PROCEDURE SetProjectFilename (projfile: ARRAY OF CHAR);

    (* If this procedure is called, projfile is examined to extract     *)
    (* values for some compiler options and equations.                  *)

    VAR cid: IOChan.ChanId;
        res: ChanConsts.OpenResults;

    BEGIN
        SeqFile.OpenRead (cid, projfile, SeqFile.text, res);
        IF res = ChanConsts.opened THEN
            ExtractOptions (cid);
            SeqFile.Close (cid);
        END (*IF*);
    END SetProjectFilename;

(************************************************************************)

PROCEDURE TokenVal(token: ARRAY OF CHAR;  expectstr: BOOLEAN): ValueType;

    (* Looks up the value of this token. *)

    (* Looks up the identifier token in the symbol table.  If not   *)
    (* found, we return Boolean FALSE.                              *)

    VAR result: ValueType;

    BEGIN
        Search (token, expectstr, result);
        WHILE result.kind = syn DO
            Search (result.strval, expectstr, result);
        END (*WHILE*);
        RETURN result;
    END TokenVal;

(************************************************************************)
(*                     EVALUATING AN EXPRESSION                         *)
(*                                                                      *)
(*  In these procedures, an entry assumption is that comments and       *)
(*  blanks have already been skipped.  An expression can have type      *)
(*  string or Boolean.  The expectstr parameter in these procedures     *)
(*  is TRUE if we are expecting a string result, FALSE for Boolean.     *)
(*                                                                      *)
(************************************************************************)

PROCEDURE Expression(expectstr: BOOLEAN): ValueType;   FORWARD;

(************************************************************************)

PROCEDURE Ident(token: ARRAY OF CHAR;  expectstr: BOOLEAN): ValueType;

    (* Looks up the value of this token. *)

    BEGIN
        RETURN TokenVal(token, expectstr);
    END Ident;

(************************************************************************)

PROCEDURE String(token: ARRAY OF CHAR): ValueType;

    (* Looks up the value of this token. *)

    BEGIN
        RETURN TokenVal(token, TRUE);
    END String;

(************************************************************************)

PROCEDURE Factor(expectstr: BOOLEAN): ValueType;

    (* Factor = Ident | string | "DEFINED" "(" Ident ")"                *)
    (*              | "(" Expression ")" | "~" Factor | "NOT" Factor    *)
    (* Ident = option | equation                                        *)

    VAR result: ValueType;
        token: TokenType;

    BEGIN
        result.strval := '';
        GetToken (token);
        IF token[0] IN Letters THEN
            IF Strings.Equal (token, "DEFINED") THEN
                GetToken (token);
                IF token[0] <> '(' THEN
                    ErrMes ("'(' expected after DEFINED");
                    RETURN BooleanToVT (FALSE);
                ELSE
                    GetToken (token);
                    IF IsDefined(token) THEN
                        result.kind := true;
                    ELSE
                        result.kind := false;
                    END (*IF*);
                    GetToken (token);
                    IF token[0] <> ')' THEN
                        ErrMes ("')' expected after DEFINED");
                    END (*IF*);
                    RETURN result;
                END (*IF*);
            ELSIF Strings.Equal (token, "NOT") THEN
                result := ForceBoolean (Factor(FALSE));
                IF result.kind = true THEN
                    result.kind := false;
                ELSE
                    result.kind := true;
                END (*IF*);
                RETURN result;
            ELSE
                RETURN Ident(token, expectstr);
            END (*IF*);
        ELSIF token[0] = '~' THEN
            result := ForceBoolean(Factor(FALSE));
            IF result.kind = true THEN
                result.kind := false;
            ELSE
                result.kind := true;
            END (*IF*);
            RETURN result;
        ELSIF token[0] = '(' THEN
            result := Expression(expectstr);
            GetToken (token);
            IF token[0] <> ')' THEN
                ErrMes ("Mismatched parentheses");
            END (*IF*);
            RETURN result;
        ELSE
            RETURN String(token);
        END (*IF*);
    END Factor;

(************************************************************************)

PROCEDURE Term(expectstr: BOOLEAN): ValueType;

    (* Term = Factor { "&" Factor )     *)

    VAR result: ValueType;

    BEGIN
        result := Factor(expectstr);
        SkipBlanks;
        WHILE LineBuffer[NextPos] = "&" DO
            INC (NextPos);
            result := BooleanToVT(VTtoBoolean(result) AND VTtoBoolean(Factor(FALSE)));
            SkipBlanks;
        END (*WHILE*);
        RETURN result;
    END Term;

(************************************************************************)

PROCEDURE SimpleExpr(expectstr: BOOLEAN): ValueType;

    (* Evaluates an expression using the grammar specified for IF       *)
    (* arguments in a pragma.                                           *)

    VAR result: ValueType;

    BEGIN
        result := Term(expectstr);
        SkipBlanks;
        WHILE (LineBuffer[NextPos] = "O") AND (LineBuffer[NextPos+1] = "R") DO
            INC (NextPos, 2);
            result := BooleanToVT(VTtoBoolean(result) OR VTtoBoolean(Term(FALSE)));
            SkipBlanks;
        END (*WHILE*);
        RETURN result;
    END SimpleExpr;

(************************************************************************)

PROCEDURE Expression(expectstr: BOOLEAN): ValueType;

    (* Evaluates an expression using the grammar specified for IF       *)
    (* arguments in a pragma.                                           *)
    (* Expression = SimpleExpression                                    *)
    (*               [ ("=" | "#") SimpleExpression ]                   *)

    VAR result: ValueType;
        ch: CHAR;  cmp: BOOLEAN;

    BEGIN
        result := SimpleExpr(expectstr);
        SkipBlanks;
        ch := LineBuffer[NextPos];
        IF (ch = '=') OR (ch = '#') THEN
            INC (NextPos);
            cmp := IsEqual (result, SimpleExpr(TRUE));
            IF ch = '#' THEN
                cmp := NOT cmp;
            END (*IF*);
            result := BooleanToVT (cmp);
        END (*IF*);
        RETURN result;
    END Expression;

(************************************************************************)

PROCEDURE BoolExpr(): BOOLEAN;

    (* Evaluates an expression using the grammar specified for IF       *)
    (* arguments in a pragma.                                           *)

    BEGIN
        RETURN VTtoBoolean(Expression(FALSE));
    END BoolExpr;

(************************************************************************)
(*                    SKIPPING OVER UNWANTED SECTIONS                   *)
(************************************************************************)

PROCEDURE SkipPast (str: ARRAY OF CHAR);

    (* Skips to beyond the specified string.  More precisely, beyond    *)
    (* the first occurrence of the string that is not inside a comment. *)

    VAR pos1, pos2: CARDINAL;
        found1, found2: BOOLEAN;

    BEGIN
        REPEAT
            Strings.FindNext ('(*', LineBuffer, NextPos, found1, pos1);
            Strings.FindNext (str, LineBuffer, NextPos, found2, pos2);
            IF found1 AND found2 THEN
                IF pos1 < pos2 THEN
                    found2 := FALSE;
                ELSE
                    found1 := FALSE;
                END (*IF*);
            END (*IF*);

            (* Now at most one of found1 and found2 is true. *)

            IF found1 THEN
                NextPos := pos1 + 1;
                SkipComments;
            ELSIF found2 THEN
                NextPos := pos2 + LENGTH(str);
            ELSE
                GetNextLine;
                IF LineBuffer[0] = CtrlZ THEN
                    (* End of file, so abort the search. *)
                    found2 := TRUE;
                END (*IF*);
            END (*IF*);

        UNTIL found2;

    END SkipPast;

(************************************************************************)

PROCEDURE SkipPastEnd();

    (* Advances the file pointer to just after the next END *)
    (* pragma.  We also have to skip any nested IF clauses. *)

    VAR token: ARRAY [0..63] OF CHAR;

    BEGIN
        LOOP
            SkipPast ('<*');
            GetToken (token);
            IF token[0] IN Letters THEN
                IF Strings.Equal (token, "IF") THEN
                    SkipPastEnd();
                ELSIF Strings.Equal (token, "END") THEN
                    SkipPast ("*>");
                    RETURN;
                END (*IF*);
            END (*IF*);
        END (*LOOP*);
    END SkipPastEnd;

(************************************************************************)

PROCEDURE HandlePragma (VAR (*IN*) token: ARRAY OF CHAR);   FORWARD;

(************************************************************************)

PROCEDURE SkipToNext();

    (* Advances the file pointer to just after the next ELSIF or ELSE   *)
    (* or END pragma.  We also have to skip any nested IF clauses.      *)

    VAR token: ARRAY [0..63] OF CHAR;

    BEGIN
        LOOP
            SkipPast ('<*');
            GetToken (token);
            IF Strings.Equal (token, "IF") THEN
                SkipPastEnd();
            ELSIF Strings.Equal (token, "ELSIF")
                     OR Strings.Equal (token, "ELSE")
                     OR Strings.Equal (token, "END") THEN
                HandlePragma (token);
                EXIT (*LOOP*);
            ELSE
                SkipPast ("*>");
            END (*IF*);
        END (*LOOP*);
    END SkipToNext;

(************************************************************************)
(*                          TOP-LEVEL SCANNING                          *)
(************************************************************************)

PROCEDURE HandlePragma (VAR (*IN*) token: ARRAY OF CHAR);

    (* Deals with <* ... *> comments, and in particular skips sections  *)
    (* as required by IF/THEN/ELSE conditions. On entry the '<*' and    *)
    (* following blanks have already been passed, and parameter         *)
    (* 'token' is the first token after that.                           *)

    BEGIN
        IF Strings.Equal (token, "IF") THEN
            INC (level);
            condval[level] := BoolExpr();
            SkipPast ("*>");
            IF NOT condval[level] THEN
                SkipToNext();
            END (*IF*);
        ELSIF Strings.Equal (token, "ELSIF") THEN
            IF condval[level] THEN
                (* We have already processed the "IF" part, so we   *)
                (* can ignore the rest of this "IF" clause.         *)
                SkipPast ("*>");
                SkipPastEnd();
            ELSE
                condval[level] := BoolExpr();
                SkipPast ("*>");
                IF NOT condval[level] THEN
                    SkipToNext();
                END (*IF*);
            END (*IF*);
        ELSIF Strings.Equal (token, "ELSE") THEN
            SkipPast ("*>");
            IF condval[level] THEN
                (* We have already processed the "IF" part, so we   *)
                (* can ignore the rest of this "IF" clause.         *)
                SkipPastEnd();
            END (*IF*);
        ELSIF Strings.Equal (token, "END") THEN
            SkipPast ("*>");
            DEC (level);
        ELSIF token[0] <> ClosePragma THEN

            (* This might introduce or change an option value. *)

            InlineOption (token, LineBuffer, NextPos);
            SkipPast ("*>");

        END (*IF*);
    END HandlePragma;

(************************************************************************)

PROCEDURE Scan (VAR (*OUT*) token: ARRAY OF CHAR);

    (* Returns the next input token.  *)

    BEGIN
        GetToken (token);
        WHILE token[0] = OpenPragma DO
            GetToken (token);
            SkipBlanks;
            HandlePragma (token);
            GetToken (token);
        END (*WHILE*);
    END Scan;

(************************************************************************)

VAR j: CARDINAL;

BEGIN
    level := 0;
    condval[0] := TRUE;
    FOR j := 0 TO MaxSyms-1 DO
        SymTbl[j].symbol[0] := Nul;
    END (*FOR*);
END Scanner.

