IMPLEMENTATION MODULE MatExtra;

        (********************************************************)
        (*                                                      *)
        (*          Some additional matrix operations           *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Last edited:        23 March 20155                  *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (* Note that many of the functions in this module occur *)
        (*   in pairs, one for real and one for complex data.   *)
        (*                                                      *)
        (********************************************************)


FROM LongMath IMPORT
    (* proc *)  sqrt;

FROM LongComplexMath IMPORT
    (* proc *)  abs, conj, scalarMult;

FROM MiscM2 IMPORT
    (* proc *)  WriteString, WriteLn;

FROM Vec IMPORT
    (* type *)  VectorPtr, CxVectorPtr,
    (* proc *)  NewVector, DisposeVector, NewCxVector, DisposeCxVector;

FROM Mat IMPORT
    (* type *)  ArrayPtr, CxArrayPtr,
    (* proc *)  NewArray, DisposeArray, NewCxArray, DisposeCxArray,
                ReCopy, CxCopy, Transpose, Adjoint;

(************************************************************************)

CONST
    small = 1.0E-15;
    CxZero = CMPLX (0.0, 0.0);

(************************************************************************)
(*                          QR FACTORISATION                            *)
(************************************************************************)
(*                                                                      *)
(* Background: the QR factorisation of a real or complex matrix A is    *)
(* a factorisation A = QR, where Q has the property that Q*Q = I and    *)
(* R is upper triangular.  Here, Q* means the adjoint of Q: the complex *)
(* conjugate transpose in the complex case, or just the transpose in    *)
(* the real case. Thus, the columns of Q have unit norm and are         *)
(* orthogonal to one another. The triangular shape of R can simplify a  *)
(* lot of calculations, especially when A has many more rows than       *)
(* columns.                                                             *)
(*                                                                      *)
(* Let A be an rxc matrix. In the traditional "thick" factorisation     *)
(* Q is a square matrix, i.e. Q is rxr and R is rxc. In this case Q     *)
(* is an orthonormal matrix, a slightly stronger property than Q*Q=I.   *)
(* Whether this matters depends on what you are going to do with Q and  *)
(* R once you have calculated them. In many situations it does not      *)
(* matter. If r>c, then the last (r-c) rows of R are zero, because of   *)
(* the uppper triangular property. In this case it is obvious that we   *)
(* can delete those zero rows, and delete the last (r-c) columns of Q,  *)
(* to obtain the so-called "thin" QR factorisation, where Q is rxc and  *)
(* R is cxc.                                                            *)
(*                                                                      *)
(* Remark: if A has more columns than rows the above argument does not  *)
(* apply, and there is no thin factorisation. In the practical          *)
(* applications I have worked on, however, the case r>>c is precisely   *)
(* the case where a QR factorisation is useful.                         *)
(*                                                                      *)
(* The code below gives an "even thinner" QR factorisation, where Q is  *)
(* rxq and R is qxc, where q=rank(A). It works in both cases r>c and    *)
(* r<=c. I have not found this case described in the literature, but    *)
(* I have worked out the theory (details on request) to justify the     *)
(* algorithm. In my opinion this is the most useful QR factorisation    *)
(* of all, because it guarantees that the rows of R are linearly        *)
(* independent. One consequence of this is a new and (I believe)        *)
(* computationally efficient way to calculate a pseudo-inverse -- see   *)
(* the code later in this module.                                       *)
(*                                                                      *)
(************************************************************************)

PROCEDURE ReQRFactor (A: ARRAY OF ARRAY OF LONGREAL;  r, c: CARDINAL;
                         VAR (*OUT*) Q, R: ARRAY OF ARRAY OF LONGREAL): CARDINAL;

    (* QR factorisation of an rxc matrix A, not necessarily square.     *)
    (* The result is an rxq matrix Q such that Q*Q=I, and an upper      *)
    (* triangular qxc matrix R.  If A<>0 the integer q, which is        *)
    (* returned as the function result, is equal to the rank of A, and  *)
    (* the rows of R are linearly independent.                          *)
    (* If A=0 then we return q=0, and R is a single row of zeros.       *)
    (* Assumption: the caller has declared Q and R to be large enough   *)
    (* to hold the result. Note that q<=min(r,c), which tells the       *)
    (* caller how much space to reserve even though q is not known in   *)
    (* advance.  Unused rows and columns are left unaltered.            *)

    (* This is the "even thinner" version of QR factorisation. If the   *)
    (* "thick" version is needed, see ReFullQRFactor below.             *)

    (* This version uses the modified Gram-Schmidt algorithm, with      *)
    (* provision for the possibility that A is rank-deficient.          *)

    VAR
        j, col, row, rank: CARDINAL;
        temp, norm, val: LONGREAL;
        W: ArrayPtr;
        allzero: BOOLEAN;

    BEGIN
        (* The case A = 0 needs separate handling. *)

        allzero := TRUE;  row := 0;
        WHILE allzero AND (row < r) DO
            col := 0;
            WHILE allzero AND (col < c) DO
                IF ABS(A[row, col]) >= small THEN
                     allzero := FALSE;
                END (*IF*);
                INC (col);
            END (*WHILE*);
            INC (row);
        END (*WHILE*);
        IF allzero THEN
            Q[0,0] := 1.0;
            FOR row := 1 TO r-1 DO
                Q[row, 0] := 0.0;
            END (*FOR*);
            FOR col := 0 TO c-1 DO
                R[0, col] := 0.0;
            END (*FOR*);
            RETURN 0;
        END (*IF*);

        (* From now on we can assume that A <> 0. *)

        (* Set subdiagonal elements of R to zero. Here rank is being    *)
        (* used as a temporary variable.                                *)

        IF r >= c THEN rank := c ELSE rank := r END(*IF*);
        FOR row := 1 TO rank-1 DO
            FOR col := 0 TO row-1 DO
                R[row, col] := 0.0;
            END (*FOR*);
        END (*FOR*);

        (* To avoid destroying the original A, copy A to a work matrix  *)
        (* W^.  As the calculation works through column 'col', the      *)
        (* final Q will be stored in columns 0 to rank-1 of the work    *)
        (* matrix, while columns col+1 upwards will hold a modified A.  *)
        (* In the case r >= c we could have used Q as the work matrix,  *)
        (* but Q might not have enough columns in the general case.     *)

        (* Note that col is the current column of A we are working      *)
        (* with, rank is both the current column of Q and the current   *)
        (* row of R.  In the full-rank case, rank=col, but if A is      *)
        (* rank-deficient we will sometimes step to the next column     *)
        (* of W^ without incrementing rank.                             *)

        rank := 0;
        W := NewArray (r, c);
        ReCopy (A, r, c, W^);
        col := 0;
        WHILE col < c DO
            norm := 0.0;

            (* Set norm equal to the norm of the current column. *)

            FOR row := 0 TO r-1 DO
                val := W^[row, col];
                norm := norm + val*val;
            END (*FOR*);
            norm := sqrt(norm);

            IF norm < small THEN

                (* Special case: first column is zero. *)

                R[rank, col] := 0.0;

            ELSE

                (* Now current column of W^ must be set equal to a copy,    *)
                (* normalised, of the current column, and the diagonal      *)
                (* element of R is the normalisation factor.  Subdiagonal   *)
                (* elements of R have already been set to zero.             *)

                R[rank, col] := norm;
                temp := 1.0/norm;
                allzero := (col+1 < c);     (* FALSE if in last column *)
                FOR row := 0 TO r-1 DO
                    W^[row, rank] := temp * W^[row, col];
                END (*FOR*);

                (* Set the rest of the current row of R. *)

                IF allzero THEN            (* i.e. if not in last column *)
                    FOR j := col+1 TO c-1 DO
                        val := 0.0;
                        FOR row := 0 TO r-1 DO
                            val := val + W^[row, rank] * W^[row, j];
                        END (*FOR*);
                        R[rank, j] := val;

                        (* Adjust the remaining columns of W^ in preparation *)
                        (* for the next step in the calculation.             *)

                        FOR row := 0 TO r-1 DO
                            W^[row, j] := W^[row, j] - val * W^[row, rank];
                            IF ABS(W^[row, j]) >= small THEN
                                 allzero := FALSE;
                            END (*IF*);
                        END (*FOR*);
                    END (*FOR "adjust remaining columns" loop*);
                END (*IF*);

                INC (rank);

                (* There is still one special case to handle, and that   *)
                (* is where the adjusted remaining part of W^ turns out  *)
                (* to be zero. In that case we have finished the         *)
                (* calculations and don't need to look at the rest of W^.*)

                IF allzero THEN col := c END (*IF*);

            END (* IF norm *);
            INC (col);

        END; (* loop over columns *)

        ReCopy (W^, r, rank, Q);
        DisposeArray (W, r, c);
        RETURN rank;

    END ReQRFactor;

(************************************************************************)

PROCEDURE ReFullQRFactor (A: ARRAY OF ARRAY OF LONGREAL;  r, c: CARDINAL;
                                VAR (*OUT*) Q, R: ARRAY OF ARRAY OF LONGREAL);

    (* QR factorisation of an rxc matrix A, with r>=c.  This is the     *)
    (* same as ReQRFactor, except that we guarantee to return a square  *)
    (* matrix Q. That is, Q is rxr and R is rxc, with extra zero rows   *)
    (* added to R as needed.  We still have the property Qadj*Q=I.      *)

    VAR col, onepos, i, k: CARDINAL;
        scale: LONGREAL;
        u: VectorPtr;

    BEGIN
        u := NewVector (r);
        col := ReQRFactor (A, r, c, Q, R);
        onepos := 0;
        WHILE col < r DO

            (* The first col columns of Q have been filled in.  We will now *)
            (* use the Gram-Schmidt procedure to generate the next column.  *)

            scale := 0.0;
            WHILE scale < small DO

                (* Create the vector u.  *)

                FOR i := 0 TO r-1 DO
                    u^[i] := 0.0;
                END (*FOR*);
                u^[onepos] := 1.0;
                IF col > 0 THEN
                    FOR k := 0 TO col-1 DO
                        scale := Q[onepos,k];
                        FOR i := 0 TO r-1 DO
                            u^[i] := u^[i] - scale*Q[i,k];
                        END (*FOR*);
                    END (*FOR*);
                END (*IF*);
                scale := 0.0;
                FOR i := 0 TO r-1 DO
                    scale := scale + u^[i]*u^[i];
                END (*FOR*);
                scale := sqrt (scale);
                INC (onepos);

                (* If scale is too small, we just try again. *)

            END (*WHILE*);

            (* Fill in the new column of Q, and zero a new row of R.  *)

            scale := 1.0/scale;
            FOR i := 0 TO r-1 DO
                Q[i,col] := scale * u^[i];
            END (*FOR*);
            FOR k := 0 TO c-1 DO
                R[col,k] := 0.0;
            END (*FOR*);
            INC (col);
        END (*WHILE*);

        DisposeVector (u, r);

    END ReFullQRFactor;

(************************************************************************)

PROCEDURE CxQRFactor (A: ARRAY OF ARRAY OF LONGCOMPLEX;  r, c: CARDINAL;
                         VAR (*OUT*) Q, R: ARRAY OF ARRAY OF LONGCOMPLEX): CARDINAL;

    (* QR factorisation of an rxc matrix A, not necessarily square.     *)
    (* The result is an rxq matrix Q such that Q*Q=I, and an upper      *)
    (* triangular qxc matrix R.  If A<>0 the integer q, which is        *)
    (* returned as the function result, is equal to the rank of A, and  *)
    (* the rows of R are linearly independent.                          *)
    (* If A=0 then we return q=0, and R is a single row of zeros.       *)
    (* Assumption: the caller has declared Q and R to be large enough   *)
    (* to hold the result. Note that q<=min(r,c), which tells the       *)
    (* caller how much space to reserve even though q is not known in   *)
    (* advance.  Unused rows and columns are left unaltered.            *)

    (* This is the "even thinner" version of QR factorisation. If the   *)
    (* "thick" version is needed, see CxFullQRFactor below.             *)

    (* This version uses the modified Gram-Schmidt algorithm, with      *)
    (* provision for the possibility that A is rank-deficient.          *)

    VAR
        j, col, row, rank: CARDINAL;
        norm: LONGREAL;
        val: LONGCOMPLEX;
        W: CxArrayPtr;
        allzero: BOOLEAN;

    BEGIN

        (* The case A = 0 needs separate handling. *)

        allzero := TRUE;  row := 0;
        WHILE allzero AND (row < r) DO
            col := 0;
            WHILE allzero AND (col < c) DO
                IF abs(A[row, col]) >= small THEN
                     allzero := FALSE;
                END (*IF*);
                INC (col);
            END (*WHILE*);
            INC (row);
        END (*WHILE*);
        IF allzero THEN
            Q[0,0] := CMPLX (1.0, 0.0);
            FOR row := 1 TO r-1 DO
                Q[row, 0] := CxZero;
            END (*FOR*);
            FOR col := 0 TO c-1 DO
                R[0, col] := CxZero;
            END (*FOR*);
            RETURN 0;
        END (*IF*);

        (* From now on we can assume that A <> 0.  *)

        (* Set subdiagonal elements of R to zero. Here rank is being    *)
        (* used as a temporary variable.                                *)

        IF r >= c THEN rank := c ELSE rank := r END(*IF*);
        FOR row := 1 TO rank-1 DO
            FOR col := 0 TO row-1 DO
                R[row, col] := CxZero;
            END (*FOR*);
        END (*FOR*);

        (* To avoid destroying the original A, copy A to a work matrix  *)
        (* W^.  As the calculation works through column 'col', the      *)
        (* final Q will be stored in columns 0 to rank-1 of the work    *)
        (* matrix, while columns col+1 upwards will hold a modified A.  *)
        (* In the case r >= c we could have used Q as the work matrix,  *)
        (* but Q might not have enough columns in the general case.     *)

        (* Note that col is the current column of W^ we are working     *)
        (* with, rank is both the current column of Q and the current   *)
        (* row of R.  In the full-rank case, rank=col, but if A is      *)
        (* rank-deficient we will sometimes step to the next column     *)
        (* of A without incrementing rank.                              *)

        rank := 0;
        W := NewCxArray (r, c);
        CxCopy (A, r, c, W^);
        col := 0;
        WHILE col < c DO
            norm := 0.0;

            (* Set norm equal to the norm of the current column. *)

            FOR row := 0 TO r-1 DO
                val := W^[row, col];
                norm := norm + RE(val)*RE(val) + IM(val)*IM(val);
            END (*FOR*);
            norm := sqrt(norm);

            IF norm < small THEN

                (* Special case: first column is zero. *)

                R[rank, col] := CxZero;

            ELSE

                (* Now current column of W^ must be normalised, and the     *)
                (* diagonal element of R is the normalisation factor.       *)
                (* Subdiagonal elements of R have already been set to zero. *)

                R[rank, col] := CMPLX (norm, 0.0);
                norm := 1.0/norm;
                allzero := (col+1 < c);     (* FALSE if in last column *)
                FOR row := 0 TO r-1 DO
                    W^[row, rank] := scalarMult (norm, W^[row, col]);
                END (*FOR*);

                (* Set the rest of the current row of R. *)

                IF allzero THEN            (* i.e. if not in last column *)
                    FOR j := col+1 TO c-1 DO
                        val := CxZero;
                        FOR row := 0 TO r-1 DO
                            val := val + conj(W^[row, rank]) * W^[row, j];
                        END (*FOR*);
                        R[rank, j] := val;

                        (* Adjust the remaining columns of our copy of W^ in    *)
                        (* preparation for the next step in the calculation.    *)

                        FOR row := 0 TO r-1 DO
                            W^[row, j] := W^[row, j] - val * W^[row, rank];
                            IF abs(W^[row, j]) >= small THEN
                                 allzero := FALSE;
                            END (*IF*);
                        END (*FOR*);
                    END (*FOR "adjust remaining columns" loop*);
                END (*IF*);

                INC (rank);

                (* There is still one special case to handle, and that  *)
                (* is where the adjusted remaining part of W^ turns out *)
                (* to be zero. In that case we have finished the        *)
                (* calculations and don't need to process more columns. *)

                IF allzero THEN col := c END (*IF*);

            END (* IF norm *);
            INC (col);

        END; (* loop over columns *)

        CxCopy (W^, r, rank, Q);
        DisposeCxArray (W, r, c);
        RETURN rank;

    END CxQRFactor;

(************************************************************************)

PROCEDURE CxFullQRFactor (A: ARRAY OF ARRAY OF LONGCOMPLEX;  r, c: CARDINAL;
                                VAR (*OUT*) Q, R: ARRAY OF ARRAY OF LONGCOMPLEX);

    (* QR factorisation of an rxc matrix A, with r>=c.  This is the     *)
    (* same as CxQRFactor, except that we guarantee to return a square  *)
    (* matrix Q. That is, Q is rxr and R is rxc, with extra zero rows   *)
    (* added to R as needed.  We still have the property Qadj*Q=I.      *)

    VAR col, onepos, i, k: CARDINAL;
        norm: LONGREAL;
        cxscale: LONGCOMPLEX;
        u: CxVectorPtr;

    BEGIN
        u := NewCxVector (r);
        col := CxQRFactor (A, r, c, Q, R);
        onepos := 0;
        WHILE col < r DO

            (* The first col columns of Q have been filled in.  We will now *)
            (* use the Gram-Schmidt procedure to generate the next column.  *)

            norm := 0.0;
            WHILE norm < small DO

                (* Create the vector u.  *)

                FOR i := 0 TO r-1 DO
                    u^[i] := CxZero;
                END (*FOR*);
                u^[onepos] := CMPLX (1.0, 0.0);
                IF col > 0 THEN
                    FOR k := 0 TO col-1 DO
                        cxscale := conj (Q[onepos,k]);
                        FOR i := 0 TO r-1 DO
                            u^[i] := u^[i] - cxscale*Q[i,k];
                        END (*FOR*);
                    END (*FOR*);
                END (*IF*);
                norm := 0.0;
                FOR i := 0 TO r-1 DO
                    norm := norm + RE(u^[i])*RE(u^[i]) + IM(u^[i])*IM(u^[i]);
                END (*FOR*);
                norm := sqrt (norm);
                INC (onepos);

                (* If norm is too small, we just try again. *)

            END (*WHILE*);

            (* Fill in the new column of Q, and zero a new row of R.  *)

            norm := 1.0/norm;
            FOR i := 0 TO r-1 DO
                Q[i,col] := scalarMult (norm, u^[i]);
            END (*FOR*);
            FOR k := 0 TO c-1 DO
                R[col,k] := CxZero;
            END (*FOR*);
            INC (col);
        END (*WHILE*);

        DisposeCxVector (u, r);

    END CxFullQRFactor;

(************************************************************************)
(*                       UPPER TRIANGULAR MATRICES                      *)
(************************************************************************)

PROCEDURE ReUTinverse (S: ARRAY OF ARRAY OF LONGREAL;  r: CARDINAL;
                                VAR (*OUT*) T: ARRAY OF ARRAY OF LONGREAL);

    (* Returns T = inverse(S), for the special case where S is an       *)
    (* upper triangular rxr matrix.  The caller must guarantee that     *)
    (* the matrix is nonsingular.                                       *)

    VAR i, j, k: CARDINAL;
        temp, Tii: LONGREAL;

    BEGIN
        FOR i := r-1 TO 0 BY -1 DO
            IF i > 0 THEN
                FOR j := 0 TO i-1 DO
                    T[i,j] := 0.0;
                END (*FOR*);
            END (*IF*);
            Tii := 1.0/S[i,i];
            T[i,i] := Tii;
            IF i < r-1 THEN
                FOR j := i+1 TO r-1 DO
                    temp := 0.0;
                    FOR k := i+1 TO j DO
                        temp := temp + S[i,k] * T[k,j];
                    END (*FOR*);
                    T[i,j] := -temp*Tii;
                END (*FOR*);
             END (*IF*);
        END (*FOR*);
    END ReUTinverse;

(************************************************************************)

PROCEDURE CxUTinverse (S: ARRAY OF ARRAY OF LONGCOMPLEX;  r: CARDINAL;
                                VAR (*OUT*) T: ARRAY OF ARRAY OF LONGCOMPLEX);

    (* Returns T = inverse(S), for the special case where S is an       *)
    (* upper triangular rxr matrix.  The caller must guarantee that     *)
    (* the matrix is nonsingular.                                       *)

    VAR i, j, k: CARDINAL;
        temp, Tii: LONGCOMPLEX;

    BEGIN
        FOR i := r-1 TO 0 BY -1 DO
            IF i > 0 THEN
                FOR j := 0 TO i-1 DO
                    T[i,j] := CxZero;
                END (*FOR*);
            END (*IF*);
            Tii := CMPLX (1.0, 0.0) / S[i,i];
            T[i,i] := Tii;
            IF i < r-1 THEN
                FOR j := i+1 TO r-1 DO
                    temp := CxZero;
                    FOR k := i+1 TO j DO
                        temp := temp + S[i,k] * T[k,j];
                    END (*FOR*);
                    T[i,j] := -temp*Tii;
                END (*FOR*);
             END (*IF*);
        END (*FOR*);
    END CxUTinverse;

(************************************************************************)
(*                            PSEUDOINVERSE                             *)
(************************************************************************)

PROCEDURE ReUTpsinverse (S: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL;
                           VAR (*OUT*) T: ARRAY OF ARRAY OF LONGREAL);

    (* Returns the Moore-Penrose pseudoinverse T = psinv(S), where S is *)
    (* an upper triangular rows x cols matrix, with full row rank.      *)
    (* (The caller is responsible for ensuring that S has these         *)
    (* properties.) The result is a cols x rows matrix.                 *)

    VAR Q, R, Rinv, Str: ArrayPtr;
        temp: LONGREAL;
        i, j, k, rank: CARDINAL;

    BEGIN
        IF rows = cols THEN
            ReUTinverse (S, rows, T);
        ELSE
            (* rows < cols, so S transpose has a QR decomposition for       *)
            (* which R is a square (rows x rows) nonsingular upper          *)
            (* triangular matrix.                                           *)

            Rinv := NewArray (rows, rows);
            Q := NewArray (cols, rows);
            R := NewArray (rows, rows);
            Str := NewArray (cols, rows);
            Transpose (S, rows, cols, Str^);

            rank := ReQRFactor (Str^, cols, rows, Q^, R^);
            IF rank <> rows THEN
                WriteString ("ERROR: rank condition failed in ReUTpsinverse");
                WriteLn;
            END (*IF*);
            DisposeArray (Str, cols, rows);

            ReUTinverse (R^, rows, Rinv^);
            DisposeArray (R, rows, rows);

            (* The result is Q times the adjoint of Rinv. *)

            FOR i := 0 TO cols-1 DO
                FOR j := 0 TO rows-1 DO
                    temp := 0.0;
                    FOR k := j TO rows-1 DO
                        temp := temp + Q^[i,k] * Rinv^[j,k];
                    END (*FOR*);
                    T[i,j] := temp;
                END (*FOR*);
            END (*FOR*);
            DisposeArray (Q, cols, rows);
            DisposeArray (Rinv, rows, rows);
        END (*IF*);
    END ReUTpsinverse;

(************************************************************************)

PROCEDURE CxUTpsinverse (S: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL;
                           VAR (*OUT*) T: ARRAY OF ARRAY OF LONGCOMPLEX);

    (* Returns the Moore-Penrose pseudoinverse T = psinv(S), where S is *)
    (* an upper triangular rows x cols matrix, with full row rank.      *)
    (* (The caller is responsible for ensuring that S has these         *)
    (* properties.) The result is a cols x rows matrix.                 *)

    VAR Q, R, Rinv, Sadj: CxArrayPtr;
        temp: LONGCOMPLEX;
        i, j, k, rank: CARDINAL;

    BEGIN
        IF rows = cols THEN
            CxUTinverse (S, rows, T);
        ELSE
            (* rows < cols, so S* has a QR decomposition for which R is a   *)
            (* square (rows x rows) nonsingular upper triangular matrix.    *)

            Rinv := NewCxArray (rows, rows);
            Q := NewCxArray (cols, rows);
            R := NewCxArray (rows, rows);
            Sadj := NewCxArray (cols, rows);
            Adjoint (S, rows, cols, Sadj^);

            rank := CxQRFactor (Sadj^, cols, rows, Q^, R^);
            IF rank <> rows THEN
                WriteString ("ERROR: rank condition failed in CxUTpsinverse");
                WriteLn;
            END (*IF*);
            DisposeCxArray (Sadj, cols, rows);

            CxUTinverse (R^, rows, Rinv^);
            DisposeCxArray (R, rows, rows);

            (* The result is Q times the adjoint of Rinv. *)

            FOR i := 0 TO cols-1 DO
                FOR j := 0 TO rows-1 DO
                    temp := CxZero;
                    FOR k := j TO rows-1 DO
                        temp := temp + Q^[i,k] * conj(Rinv^[j,k]);
                    END (*FOR*);
                    T[i,j] := temp;
                END (*FOR*);
            END (*FOR*);
            DisposeCxArray (Q, cols, rows);
            DisposeCxArray (Rinv, rows, rows);
        END (*IF*);
    END CxUTpsinverse;

(************************************************************************)

PROCEDURE RePseudoInverse (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL;
                             VAR (*OUT*) PSI: ARRAY OF ARRAY OF LONGREAL): CARDINAL;

    (* Returns the Moore-Penrose pseudoinverse PSI of an arbitrary      *)
    (* rowsxcols matrix A, and also returns the rank of A.              *)
    (* The result is a colsxrows matrix.                                *)

    VAR Q, R, Rpsinv: ArrayPtr;
        val: LONGREAL;
        rank, i, j, k: CARDINAL;

    BEGIN
        (* Perform a QR factorisation of A. *)

        Q := NewArray (rows, rows);
        R := NewArray (rows, cols);

        rank := ReQRFactor (A, rows, cols, Q^, R^);

        IF rank = 0 THEN

            (* Special case: a zero matrix. *)

            FOR i := 0 TO cols-1 DO
                FOR j := 0 TO rows-1 DO
                    PSI[i,j] := 0.0;
                END (*FOR*);
            END (*FOR*);

        ELSE
            (* Calculate the pseudoinverse of R. *)

            Rpsinv := NewArray (cols, rank);
            ReUTpsinverse (R^, rank, cols, Rpsinv^);

            (* Postmultiply by Q adjoint. *)

            FOR i := 0 TO cols-1 DO
                FOR j := 0 TO rows-1 DO
                    val := 0.0;
                    FOR k := 0 TO rank-1 DO
                        val := val + Rpsinv^[i,k] * Q^[j,k];
                    END (*FOR*);
                    PSI[i,j] := val;
                END (*FOR*);
            END (*FOR*);

            DisposeArray (Rpsinv, cols, rank);

        END (*IF*);

        DisposeArray (R, rows, cols);
        DisposeArray (Q, rows, rows);

        RETURN rank;

    END RePseudoInverse;

(************************************************************************)

PROCEDURE CxPseudoInverse (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL;
                             VAR (*OUT*) PSI: ARRAY OF ARRAY OF LONGCOMPLEX): CARDINAL;

    (* Returns the Moore-Penrose pseudoinverse PSI of an arbitrary      *)
    (* rowsxcols matrix A, and also returns the rank of A.              *)
    (* The result is a colsxrows matrix.                                *)

    VAR Q, R, Rpsinv: CxArrayPtr;
        val: LONGCOMPLEX;
        rank, i, j, k: CARDINAL;

    BEGIN
        (* Perform a QR factorisation of A. *)

        Q := NewCxArray (rows, rows);
        R := NewCxArray (rows, cols);

        rank := CxQRFactor (A, rows, cols, Q^, R^);

        IF rank = 0 THEN

            (* Special case: a zero matrix. *)

            FOR i := 0 TO cols-1 DO
                FOR j := 0 TO rows-1 DO
                    PSI[i,j] := CxZero;
                END (*FOR*);
            END (*FOR*);

        ELSE
            (* Calculate the pseudoinverse of R. *)

            Rpsinv := NewCxArray (cols, rank);
            CxUTpsinverse (R^, rank, cols, Rpsinv^);

            (* Postmultiply by Q adjoint. *)

            FOR i := 0 TO cols-1 DO
                FOR j := 0 TO rows-1 DO
                    val := CxZero;
                    FOR k := 0 TO rank-1 DO
                        val := val + Rpsinv^[i,k] * conj(Q^[j,k]);
                    END (*FOR*);
                    PSI[i,j] := val;
                END (*FOR*);
            END (*FOR*);

            DisposeCxArray (Rpsinv, cols, rank);

        END (*IF*);

        DisposeCxArray (R, rows, cols);
        DisposeCxArray (Q, rows, rows);

        RETURN rank;

    END CxPseudoInverse;

(************************************************************************)
(*                       CHOLESKY FACTORISATION                         *)
(************************************************************************)

PROCEDURE ReCholesky (R: ARRAY OF ARRAY OF LONGREAL;  N: CARDINAL;
                   VAR (*OUT*) S: ARRAY OF ARRAY OF LONGREAL): BOOLEAN;

    (* Cholesky decomposition of a positive definite symmetric NxN      *)
    (* matrix R as S*S, where S is upper triangular, and its diagonal   *)
    (* entries are real and positive. If R is not positive definite and *)
    (* symmetric the calculation will fail.                             *)

    (* We could have done an in-place calculation, but the way we call  *)
    (* this function it turns out to be more convenient to have         *)
    (* separate input and output arrays.                                *)

    (* Returns FALSE if matrix was not positive definite.               *)

    VAR
        i, j, k: CARDINAL;
        sum, Sii, Ski: LONGREAL;

    BEGIN
        FOR i := 0 TO N-1 DO

            (* Put zero in the below-diagonal entries of the result. *)

            IF i > 0 THEN
                FOR j := 0 TO i-1 DO
                    S[i,j] := 0.0;
                END (*FOR*);
            END (*IF*);

            (* Calculate the diagonal entry. *)
    
            sum := R[i,i];
            IF i > 0 THEN
                FOR k := 0 TO i-1 DO
                    Ski := S[k,i];
                    sum := sum - Ski * Ski;
                END (*FOR*);
            END (*IF*);

            IF sum <= 0.0 THEN RETURN FALSE END(*IF*);
            Sii := sqrt(sum);
    
            S[i,i] := Sii;

            (* Now the rest of row i. *)
    
            FOR j := i+1 TO N-1 DO
                sum := R[i,j];
                IF i > 0 THEN
                    FOR k := 0 TO i-1 DO
                        sum := sum - S[k,i] * S[k,j];
                    END (*FOR*);
                END (*IF*);
                S[i,j] := sum / Sii;
            END (*FOR*);
        END (*FOR*);
        RETURN TRUE;
    END ReCholesky;

(************************************************************************)

PROCEDURE CxCholesky (R: ARRAY OF ARRAY OF LONGCOMPLEX;  N: CARDINAL;
                   VAR (*OUT*) S: ARRAY OF ARRAY OF LONGCOMPLEX): BOOLEAN;

    (* Cholesky decomposition of a positive definite Hermitian NxN      *)
    (* matrix R as S*S, where S is upper triangular, and its diagonal   *)
    (* entries are real and positive. If R is not positive definite and *)
    (* Hermitian the calculation will fail.                             *)

    (* We could have done an in-place calculation, but the way we call  *)
    (* this function it turns out to be more convenient to have         *)
    (* separate input and output arrays.                                *)

    (* Returns FALSE if matrix was not positive definite.               *)

    VAR
        i, j, k: CARDINAL;
        Sii, resum: LONGREAL;
        Ski, cxsum: LONGCOMPLEX;

    BEGIN
        FOR i := 0 TO N-1 DO

            (* Put zero in the below-diagonal entries of the result. *)

            IF i > 0 THEN
                FOR j := 0 TO i-1 DO
                    S[i,j] := CxZero;
                END (*FOR*);
            END (*IF*);

            (* Calculate the diagonal entry. *)

            resum := RE(R[i,i]);
            IF i > 0 THEN
                FOR k := 0 TO i-1 DO
                    Ski := S[k,i];
                    resum := resum - RE(Ski) * RE(Ski) - IM(Ski) * IM(Ski);
                END (*FOR*);
            END (*IF*);

            IF resum <= 0.0 THEN RETURN FALSE END(*IF*);
            Sii := sqrt(resum);

            S[i,i] := CMPLX (Sii, 0.0);

            (* Now the rest of row i. *)

            FOR j := i+1 TO N-1 DO
                cxsum := R[i,j];
                IF i > 0 THEN
                    FOR k := 0 TO i-1 DO
                        cxsum := cxsum - conj(S[k,i]) * S[k,j];
                    END (*FOR*);
                END (*IF*);
                S[i,j] := scalarMult (1.0/Sii, cxsum);
            END (*FOR*);
        END (*FOR*);
        RETURN TRUE;
    END CxCholesky;

(************************************************************************)

END MatExtra.

