MODULE MatExtraTest;

        (********************************************************)
        (*                                                      *)
        (*              Test of MatExtra module                 *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Last edited:        23 March 2015                   *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)


FROM MatExtra IMPORT
    (* proc *)  ReQRFactor, ReFullQRFactor,
                CxQRFactor, CxFullQRFactor,
                RePseudoInverse, CxPseudoInverse,
                ReCholesky, CxCholesky;

FROM MiscM2 IMPORT
    (* proc *)  SelectWindow, WriteString, WriteCard, WriteLn, PressAnyKey;

FROM Mat IMPORT
    (* type *)  ArrayPtr, CxArrayPtr,
    (* proc *)  Write, CxWrite, NewArray, DisposeArray, NewCxArray,
                DisposeCxArray, Transpose, Adjoint,
                Sub, Mul, CxSub, CxMul, Random, CxRandom;

(************************************************************************)

TYPE
    Re1x4 = ARRAY [1..1], [1..4] OF LONGREAL;
    Re2x2 = ARRAY [1..2], [1..2] OF LONGREAL;
    Re3x3 = ARRAY [1..3], [1..3] OF LONGREAL;
    Cx1x2 = ARRAY [1..1], [1..2] OF LONGCOMPLEX;

(************************************************************************)
(*                          QR FACTORISATION                            *)
(************************************************************************)

PROCEDURE ReQRtest (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL);

    (* Test of QR factorisation. *)

    CONST
        fieldsize = 10;

    VAR Q, R, QR: ArrayPtr;
        rank: CARDINAL;

    BEGIN
        Q := NewArray (rows, rows);
        R := NewArray (rows, cols);
        WriteString ("----------------");  WriteLn;
        WriteString ("Factoring matrix");  WriteLn;
        Write (A, rows, cols, fieldsize);
        rank := ReQRFactor (A, rows, cols, Q^, R^);
        WriteString ("rank(A)=");  WriteCard (rank);  WriteLn;
        WriteString ("Q =");  WriteLn;
        Write (Q^, rows, rank, fieldsize);
        WriteString ("R =");  WriteLn;
        Write (R^, rank, cols, fieldsize);

        QR := NewArray (rows, cols);
        Mul (Q^, R^, rows, rank, cols, QR^);
        WriteString ("QR =");  WriteLn;
        Write (QR^, rows, cols, fieldsize);

        PressAnyKey;

        IF rank < rows THEN

            (* Do a full QR factorisation as well. *)

            WriteString ("Full QR factorisation:");  WriteLn;
            ReFullQRFactor (A, rows, cols, Q^, R^);
            WriteString ("Q =");  WriteLn;
            Write (Q^, rows, rows, fieldsize);
            WriteString ("R =");  WriteLn;
            Write (R^, rows, cols, fieldsize);

            QR := NewArray (rows, cols);
            Mul (Q^, R^, rows, rank, cols, QR^);
            WriteString ("QR =");  WriteLn;
            Write (QR^, rows, cols, fieldsize);
            PressAnyKey;
        END (*IF*);

        DisposeArray (QR, rows, cols);
        DisposeArray (R, rows, cols);
        DisposeArray (Q, rows, rows);

    END ReQRtest;

(************************************************************************)

PROCEDURE CxQRtest (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL);

    (* Test of QR factorisation. *)

    CONST
        fieldsize = 8;

    VAR Q, R, QR: CxArrayPtr;
        rank: CARDINAL;

    BEGIN
        Q := NewCxArray (rows, rows);
        R := NewCxArray (rows, cols);
        WriteString ("----------------");  WriteLn;
        WriteString ("Factoring matrix");  WriteLn;
        CxWrite (A, rows, cols, fieldsize);
        rank := CxQRFactor (A, rows, cols, Q^, R^);
        WriteString ("rank(A)=");  WriteCard (rank);  WriteLn;
        WriteString ("Q =");  WriteLn;
        CxWrite (Q^, rows, rank, fieldsize);
        WriteString ("R =");  WriteLn;
        CxWrite (R^, rank, cols, fieldsize);

        QR := NewCxArray (rows, cols);
        CxMul (Q^, R^, rows, rank, cols, QR^);
        WriteString ("QR =");  WriteLn;
        CxWrite (QR^, rows, cols, fieldsize);

        PressAnyKey;

        IF rank < rows THEN

            (* Do a full QR factorisation as well. *)

            WriteString ("Full QR factorisation:");  WriteLn;
            CxFullQRFactor (A, rows, cols, Q^, R^);
            WriteString ("Q =");  WriteLn;
            CxWrite (Q^, rows, rows, fieldsize);
            WriteString ("R =");  WriteLn;
            CxWrite (R^, rows, cols, fieldsize);

            QR := NewCxArray (rows, cols);
            CxMul (Q^, R^, rows, rank, cols, QR^);
            WriteString ("QR =");  WriteLn;
            CxWrite (QR^, rows, cols, fieldsize);
            PressAnyKey;
        END (*IF*);

        DisposeCxArray (QR, rows, cols);
        DisposeCxArray (R, rows, cols);
        DisposeCxArray (Q, rows, rows);

    END CxQRtest;

(************************************************************************)

PROCEDURE DoQRtests;

    (* Tests of QR factorisation. *)

    VAR Rand: ArrayPtr;
        B: CxArrayPtr;
        r, c: CARDINAL;

    BEGIN
        SelectWindow (0);
        (**)
        ReQRtest (Re1x4 {{1.0, 2.0, 3.0, 4.0}}, 1, 4);
        ReQRtest (Re2x2 {{1.0, 0.0},
                         {0.0, 1.0}}, 2, 2);
        ReQRtest (Re2x2 {{1.0, 1.0},
                         {1.0, 1.0}}, 2, 2);
        ReQRtest (Re3x3 {{0.0, 1.0, 1.0},
                         {0.0, 1.0, 1.0},
                         {0.0, 1.0, 1.0}}, 3, 3);
        Rand := NewArray (5, 3);
        Random (Rand^, 5, 3);
        ReQRtest (Rand^, 5, 3);
        DisposeArray (Rand, 5, 3);
        (**)

        (* The following example was giving wrong results,  *)
        (* but now seems OK.                                *)

        r := 2;  c := 2;
        B := NewCxArray (r, c);
        B^[0,0] := CMPLX(0.9,0.4);  B^[0,1] := CMPLX(0.1,0.9);
        B^[1,0] := CMPLX(0.4,0.1);  B^[1,1] := CMPLX(0.9,0.0);
        CxQRtest (B^, r, c);
        DisposeCxArray (B, r, c);
        (**)

    END DoQRtests;

(************************************************************************)
(*                              PSEUDOINVERSE                           *)
(************************************************************************)

PROCEDURE RePStest (A: ARRAY OF ARRAY OF LONGREAL;  rows, cols: CARDINAL);

    (* Test of QR factorisation. *)

    CONST
        fieldsize = 10;

    VAR B, Product: ArrayPtr;
        rank: CARDINAL;

    BEGIN
        B := NewArray (cols, rows);
        WriteString ("----------------");  WriteLn;
        WriteString ("Matrix A =");  WriteLn;
        Write (A, rows, cols, fieldsize);
        rank := RePseudoInverse (A, rows, cols, B^);
        WriteString ("rank(A)=");  WriteCard (rank);  WriteLn;
        WriteString ("Its pseudoinverse B =");  WriteLn;
        Write (B^, cols, rows, fieldsize);

        Product := NewArray (rows, rows);
        Mul (A, B^, rows, cols, rows, Product^);
        WriteString ("AB =");  WriteLn;
        Write (Product^, rows, rows, fieldsize);
        DisposeArray (Product, rows, rows);

        Product := NewArray (cols, cols);
        Mul (B^, A, cols, rows, cols, Product^);
        WriteString ("BA =");  WriteLn;
        Write (Product^, cols, cols, fieldsize);
        DisposeArray (Product, cols, cols);

        DisposeArray (B, rows, cols);

        PressAnyKey;

    END RePStest;

(************************************************************************)

PROCEDURE CxPStest (A: ARRAY OF ARRAY OF LONGCOMPLEX;  rows, cols: CARDINAL);

    (* Test of QR factorisation. *)

    CONST
        fieldsize = 8;

    VAR B, Product: CxArrayPtr;
        rank: CARDINAL;

    BEGIN
        B := NewCxArray (cols, rows);
        WriteString ("----------------");  WriteLn;
        WriteString ("Matrix A =");  WriteLn;
        CxWrite (A, rows, cols, fieldsize);
        rank := CxPseudoInverse (A, rows, cols, B^);
        WriteString ("rank(A)=");  WriteCard (rank);  WriteLn;
        WriteString ("Its pseudoinverse B =");  WriteLn;
        CxWrite (B^, cols, rows, fieldsize);

        Product := NewCxArray (rows, rows);
        CxMul (A, B^, rows, cols, rows, Product^);
        WriteString ("AB =");  WriteLn;
        CxWrite (Product^, rows, rows, fieldsize);
        DisposeCxArray (Product, rows, rows);

        Product := NewCxArray (cols, cols);
        CxMul (B^, A, cols, rows, cols, Product^);
        WriteString ("BA =");  WriteLn;
        CxWrite (Product^, cols, cols, fieldsize);
        DisposeCxArray (Product, cols, cols);

        DisposeCxArray (B, rows, cols);

        PressAnyKey;

    END CxPStest;

(************************************************************************)

PROCEDURE psinvtest;

    (* Test of pseudo-inverse. *)

    VAR A: ArrayPtr;  B: CxArrayPtr;
        r, c: CARDINAL;

    BEGIN
        SelectWindow (0);
        (**)
        A := NewArray (1, 4);
        A^[0,0] := 1.0;  A^[0,1] := 2.0;
        A^[0,2] := 3.0;  A^[0,3] := 4.0;
        RePStest (A^, 1, 4);
        DisposeArray (A, 1, 4);
        (**)
        A := NewArray (5, 3);
        Random (A^, 5, 3);
        RePStest (A^, 5, 3);
        DisposeArray (A, 5, 3);
        (**)
        r := 2;  c := 3;
        A := NewArray (r, c);
        Random (A^, r, c);
        RePStest (A^, r, c);
        DisposeArray (A, r, c);
        (**)
        B := NewCxArray (r, c);
        CxRandom (B^, r, c);
        CxPStest (B^, r, c);
        DisposeCxArray (B, r, c);
        (**)

        (* The following example was giving wrong results,  *)
        (* but now seems OK.                                *)

        r := 2;  c := 2;
        B := NewCxArray (r, c);
        B^[0,0] := CMPLX(0.9,0.4);  B^[0,1] := CMPLX(0.1,0.9);
        B^[1,0] := CMPLX(0.4,0.1);  B^[1,1] := CMPLX(0.9,0.0);
        CxPStest (B^, r, c);
        DisposeCxArray (B, r, c);
        (**)
    END psinvtest;

(************************************************************************)
(*                          CHOLESKY FACTORISATION                      *)
(************************************************************************)

PROCEDURE CholeskyTest;

    (* Test of Cholesky factorisation of a positive definite matrix. *)

    CONST
        fieldsize = 8;
        rows = 3;  cols = 3;

    VAR T, Ttr, R, S, Str, Product, Diff: ArrayPtr;
        cT, cTtr, cR, cS, cStr, cProduct, cDiff: CxArrayPtr;
        k: CARDINAL;  success: BOOLEAN;

    BEGIN
        (* One easy way to generate a positive definite matrix is to    *)
        (* compute S*S for an arbitrary S, and then add some diagonal   *)
        (* entries to ensure that the result is not singular.           *)

        T := NewArray (rows, cols);
        Ttr := NewArray (cols, rows);
        R := NewArray (rows, rows);
        S := NewArray (rows, rows);
        Str := NewArray (rows, rows);
        Product := NewArray (rows, rows);
        Diff := NewArray (rows, rows);

        Random (T^, rows, cols);
        Transpose (T^, rows, cols, Ttr^);
        Mul (T^, Ttr^, rows, cols, rows, R^);
        FOR k := 0 TO rows-1 DO
            R^[k,k] := R^[k,k] + 0.1;
        END (*FOR*);

        WriteString ("----------------");  WriteLn;
        WriteString ("Matrix R =");  WriteLn;
        Write (R^, rows, rows, fieldsize);
        success := ReCholesky (R^, rows, S^);
        WriteString ("Its Cholesky factor S =");  WriteLn;
        Write (S^, cols, rows, fieldsize);
        Transpose (S^, rows, rows, Str^);
        Mul (Str^, S^, rows, rows, rows, Product^);
        Sub (Product^, R^, rows, cols, Diff^);
        WriteString ("S*S - R =");  WriteLn;
        Write (Diff^, rows, rows, fieldsize);

        DisposeArray (Diff, rows, rows);
        DisposeArray (Product, rows, rows);
        DisposeArray (Str, rows, rows);
        DisposeArray (S, rows, rows);
        DisposeArray (R, rows, rows);
        DisposeArray (Ttr, cols, rows);
        DisposeArray (T, rows, cols);
        PressAnyKey;

        (* Now repeat the same test for the complex case. *)

        cT := NewCxArray (rows, cols);
        cTtr := NewCxArray (cols, rows);
        cR := NewCxArray (rows, rows);
        cS := NewCxArray (rows, rows);
        cStr := NewCxArray (rows, rows);
        cProduct := NewCxArray (rows, rows);
        cDiff := NewCxArray (rows, rows);

        CxRandom (cT^, rows, cols);
        Adjoint (cT^, rows, cols, cTtr^);
        CxMul (cT^, cTtr^, rows, cols, rows, cR^);
        FOR k := 0 TO rows-1 DO
            cR^[k,k] := cR^[k,k] + CMPLX(0.1,0.0);
        END (*FOR*);

        WriteString ("----------------");  WriteLn;
        WriteString ("Matrix R =");  WriteLn;
        CxWrite (cR^, rows, rows, fieldsize);
        success := CxCholesky (cR^, rows, cS^);
        WriteString ("Its Cholesky factor S =");  WriteLn;
        CxWrite (cS^, cols, rows, fieldsize);
        Adjoint (cS^, rows, rows, cStr^);
        CxMul (cStr^, cS^, rows, rows, rows, cProduct^);
        WriteString ("S*S =");  WriteLn;
        CxWrite (cProduct^, rows, rows, fieldsize);
        CxSub (cProduct^, cR^, rows, cols, cDiff^);
        WriteString ("S*S - R =");  WriteLn;
        CxWrite (cDiff^, rows, rows, fieldsize);

        DisposeCxArray (cDiff, rows, rows);
        DisposeCxArray (cProduct, rows, rows);
        DisposeCxArray (cStr, rows, rows);
        DisposeCxArray (cS, rows, rows);
        DisposeCxArray (cR, rows, rows);
        DisposeCxArray (cTtr, cols, rows);
        DisposeCxArray (cT, rows, cols);
        PressAnyKey;

    END CholeskyTest;

(************************************************************************)
(*                              MAIN PROGRAM                            *)
(************************************************************************)

BEGIN
    DoQRtests;
    psinvtest;
    CholeskyTest;
END MatExtraTest.

