/* 
 * tclOS2Pipe.c -- This file implements the OS/2-specific pipeline exec
 *                 functions.
 *      
 *
 * Copyright (c) 1996 Sun Microsystems, Inc.
 * Copyright (c) 1996-1998 Illya Vaes
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */


#include "tclOS2Int.h"

#define HF_STDIN  0
#define HF_STDOUT 1
#define HF_STDERR 2

/*
 * The following defines identify the various types of applications that
 * run under OS/2.  There is special case code for the various types.
 */

#define APPL_NONE       0
#define APPL_DOS        1
#define APPL_WIN3X      2
#define APPL_WIN32      3
#define APPL_OS2WIN     4
#define APPL_OS2FS      5
#define APPL_OS2PM      6
#define APPL_OS2CMD     7

/*
 * Declarations for functions used only in this file.
 */

static int              ApplicationType(Tcl_Interp *interp,
                            const char *fileName, char *fullName);

/*
 *----------------------------------------------------------------------
 *
 * Tcl_WaitPid --
 *
 *      Emulates the waitpid system call.
 *
 * Results:
 *      Returns 0 if the process is still alive, -1 on an error, or
 *      the pid on a clean close.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_WaitPid(pid, statPtr, options)
    pid_t pid;
    int *statPtr;
    int options;
{
    ULONG flags;

    if (options & WNOHANG) {
        flags = DCWW_NOWAIT;
    } else {
        flags = DCWW_WAIT;
    }

    rc = waitpid((int)pid, statPtr, options);
    return rc;
}

/*
 *----------------------------------------------------------------------
 *
 * TclpCreateProcess --
 *
 *      Create a child process that has the specified files as its
 *      standard input, output, and error.  The child process runs
 *      asynchronously and runs with the same environment variables
 *      as the creating process.
 *
 *      The complete OS/2 search path is searched to find the specified
 *      executable.  If an executable by the given name is not found,
 *      automatically tries appending ".com", ".exe", and ".bat" to the
 *      executable name.
 *
 * Results:
 *      The return value is TCL_ERROR and an error message is left in
 *      interp->result if there was a problem creating the child
 *      process.  Otherwise, the return value is TCL_OK and *pidPtr is
 *      filled with the process id of the child process.
 *
 * Side effects:
 *      A process is created.
 *
 *----------------------------------------------------------------------
 */

int
TclpCreateProcess(interp, argc, argv, inputFile, outputFile, errorFile,
        inputFileName, outputFileName, errorFileName, pidPtr)
    Tcl_Interp *interp;         /* Interpreter in which to leave errors that
                                 * occurred when creating the child process.
                                 * Error messages from the child process
                                 * itself are sent to errorFile. */
    int argc;                   /* Number of arguments in following array. */
    char **argv;                /* Array of argument strings.  argv[0]
                                 * contains the name of the executable
                                 * converted to native format (using the
                                 * Tcl_TranslateFileName call).  Additional
                                 * arguments have not been converted. */
    Tcl_File inputFile;         /* If non-NULL, gives the file to use as
                                 * input for the child process.  If inputFile
                                 * file is not readable or is NULL, the child
                                 * will receive no standard input. */
    Tcl_File outputFile;        /* If non-NULL, gives the file that
                                 * receives output from the child process.  If
                                 * outputFile file is not writeable or is
                                 * NULL, output from the child will be
                                 * discarded. */
    Tcl_File errorFile;         /* If non-NULL, gives the file that
                                 * receives errors from the child process.  If
                                 * errorFile file is not writeable or is NULL,
                                 * errors from the child will be discarded.
                                 * errorFile may be the same as outputFile. */
    char *inputFileName;        /* If non-NULL, gives the name of the disk
                                 * file that corresponds to inputFile.  If
                                 * NULL, then the name was not available
                                 * because inputFile corresponds to a channel,
                                 * pipe, socket, etc. */
    char *outputFileName;       /* If non-NULL, gives the name of the disk
                                 * file that corresponds to outputFile.  If
                                 * NULL, then the name was not available
                                 * because outputFile corresponds to a
                                 * channel, pipe, socket, etc. */
    char *errorFileName;        /* If non-NULL, gives the name of the disk
                                 * file that corresponds to errorFile.  If
                                 * NULL, then the name was not available
                                 * because errorFile corresponds to a channel,
                                 * pipe, socket, etc. */
    int *pidPtr;                /* If this procedure is successful, pidPtr
                                 * is filled with the process id of the child
                                 * process. */
{
    int result, type, applType, nextArg, count;
    HFILE h, inputHandle, outputHandle, errorHandle;
    HFILE stdIn = HF_STDIN, stdOut = HF_STDOUT, stdErr = HF_STDERR;
    HFILE orgIn = -1, orgOut = -1, orgErr = -1;
    char execPath[MAX_PATH];
    char *originalName;
    ULONG action;
    char *arguments[256];

    applType = ApplicationType(interp, argv[0], execPath);
    if (applType == APPL_NONE) {
        return TCL_ERROR;
    }
    originalName = argv[0];
    argv[0] = execPath;

    /* Backup original stdin, stdout, stderr by Dup-ing to new handle */
    rc = DosDupHandle(stdIn, &orgIn);
    rc = DosDupHandle(stdOut, &orgOut);
    rc = DosDupHandle(stdErr, &orgErr);

    result = TCL_ERROR;

    /*
     * We have to check the type of each file, since we cannot duplicate
     * some file types.
     */

    inputHandle = -1;
    if (inputFile != NULL) {
        h = (HFILE) Tcl_GetFileInfo(inputFile, &type);
        if ((type >= TCL_OS2_PIPE) && (type <= TCL_OS2_CONSOLE)) {
            inputHandle = h;
        }
    }
    outputHandle = -1;
    if (outputFile != NULL) {
        h = (HFILE) Tcl_GetFileInfo(outputFile, &type);
        if ((type >= TCL_OS2_PIPE) && (type <= TCL_OS2_CONSOLE)) {
            outputHandle = h;
        }
    }
    errorHandle = -1;
    if (errorFile != NULL) {
        h = (HFILE) Tcl_GetFileInfo(errorFile, &type);
        if ((type >= TCL_OS2_PIPE) && (type <= TCL_OS2_CONSOLE)) {
            errorHandle = h;
        }
    }

    /*
     * Duplicate all the handles which will be passed off as stdin, stdout
     * and stderr of the child process. The duplicate handles are set to
     * be inheritable, so the child process can use them.
     */

    if (inputHandle == -1) {
        /*
         * If handle was not set, open NUL as input.
         */
        rc = DosOpen((PSZ)"NUL", &inputHandle, &action, 0, FILE_NORMAL,
                     OPEN_ACTION_CREATE_IF_NEW,
                     OPEN_SHARE_DENYNONE | OPEN_ACCESS_READONLY, NULL);
        if (rc != NO_ERROR) {
            TclOS2ConvertError(rc);
            Tcl_AppendResult(interp, "couldn't open NUL as input handle: ",
                    Tcl_PosixError(interp), (char *) NULL);
            goto end;
        }
    }
    if (inputHandle != stdIn) {
        /* Duplicate standard input handle */
        rc = DosDupHandle(inputHandle, &stdIn);
        if (rc != NO_ERROR) {
            TclOS2ConvertError(rc);
            Tcl_AppendResult(interp, "couldn't duplicate input handle: ",
                    Tcl_PosixError(interp), (char *) NULL);
            goto end;
        }
    }

    if (outputHandle == -1) {
        /*
         * If handle was not set, open NUL as output.
         */
        rc = DosOpen((PSZ)"NUL", &outputHandle, &action, 0, FILE_NORMAL,
                     OPEN_ACTION_CREATE_IF_NEW,
                     OPEN_SHARE_DENYNONE | OPEN_ACCESS_READONLY, NULL);
        if (rc != NO_ERROR) {
            TclOS2ConvertError(rc);
            Tcl_AppendResult(interp, "couldn't open NUL as output handle: ",
                    Tcl_PosixError(interp), (char *) NULL);
            goto end;
        }
    }
    if (outputHandle != stdOut) {
        /* Duplicate standard input handle */
        rc = DosDupHandle(outputHandle, &stdOut);
        if (rc != NO_ERROR) {
            TclOS2ConvertError(rc);
            Tcl_AppendResult(interp, "couldn't duplicate output handle: ",
                    Tcl_PosixError(interp), (char *) NULL);
            goto end;
        }
    }

    if (errorHandle == -1) {
        /*
         * If handle was not set, open NUL as output.
         */
        rc = DosOpen((PSZ)"NUL", &errorHandle, &action, 0, FILE_NORMAL,
                     OPEN_ACTION_CREATE_IF_NEW,
                     OPEN_SHARE_DENYNONE | OPEN_ACCESS_READONLY, NULL);
        if (rc != NO_ERROR) {
            TclOS2ConvertError(rc);
            Tcl_AppendResult(interp, "couldn't open NUL as error handle: ",
                    Tcl_PosixError(interp), (char *) NULL);
            goto end;
        }
    }
    if (errorHandle != stdErr) {
        /* Duplicate standard input handle */
        rc = DosDupHandle(errorHandle, &stdErr);
        if (rc != NO_ERROR) {
            TclOS2ConvertError(rc);
            Tcl_AppendResult(interp, "couldn't duplicate error handle: ",
                    Tcl_PosixError(interp), (char *) NULL);
            goto end;
        }
    }

    /*
     * EMX's spawnv handles all the nitty-gritty DosStartSession stuff (like
     * session/no-session, building environment and arguments with/without
     * quoting etc.) for us, so we'll just use that and keep ourselves to
     * it's arguments like P_WAIT, P_PM, ....
     */

    /*
     * Run DOS (incl. .bat) and .cmd files via cmd.exe.
     */

    if (applType == APPL_DOS || applType == APPL_OS2CMD) {
        /*
         * Run the program as a normal process inside of a hidden console
         * application.
         */

        arguments[0] = "cmd.exe";
        arguments[1] = "/c";
        nextArg = 2;
    } else {
        nextArg = 0;
    }
    for (count = 0; count < argc && nextArg < 256; count++) {
        arguments[nextArg] = argv[count];
        nextArg++;
    }
    arguments[nextArg] = '\0';

    *pidPtr = spawnv(P_SESSION | P_DEFAULT | P_MINIMIZE | P_BACKGROUND,
                     arguments[0], arguments);
    if (*pidPtr == -1) {
        TclOS2ConvertError(rc);
        Tcl_AppendResult(interp, "couldn't execute \"", originalName,
                "\": ", Tcl_PosixError(interp), (char *) NULL);
        goto end;
    }

    result = TCL_OK;

    end:
    /* Restore original stdin, stdout, stderr by Dup-ing from new handle */
    stdIn = HF_STDIN; stdOut = HF_STDOUT; stdErr = HF_STDERR;
    if (inputHandle != -1) {
        rc = DosDupHandle(orgIn, &stdIn);
        rc = DosClose(orgIn);
    }
    if (outputHandle != -1) {
        rc = DosDupHandle(orgOut, &stdOut);
        rc = DosClose(orgOut);
    }
    if (errorHandle != -1) {
        rc = DosDupHandle(orgErr, &stdErr);
        rc = DosClose(orgErr);
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * TclCreatePipe --
 *
 *      Creates an anonymous pipe.
 *
 * Results:
 *      Returns 1 on success, 0 on failure. 
 *
 * Side effects:
 *      Creates a pipe.
 *
 *----------------------------------------------------------------------
 */

int
TclCreatePipe(readPipe, writePipe)
    Tcl_File *readPipe; /* Location to store file handle for
                                 * read side of pipe. */
    Tcl_File *writePipe;        /* Location to store file handle for
                                 * write side of pipe. */
{
    HFILE readHandle, writeHandle;

    rc = DosCreatePipe(&readHandle, &writeHandle, 1024);

    *readPipe = Tcl_GetFile((ClientData)readHandle, TCL_OS2_PIPE);
    *writePipe = Tcl_GetFile((ClientData)writeHandle, TCL_OS2_PIPE);
    return 1;
}

/*
 *--------------------------------------------------------------------
 *
 * ApplicationType --
 *
 *      Search for the specified program and identify if it refers to a DOS,
 *      Windows 3.x, OS/2 Windowable, OS/2 Full-Screen, OS/2 PM program.
 *      Used to determine how to invoke a program (if it can even be invoked).
 * Results:
 *      The return value is one of APPL_DOS, APPL_WIN3X, or APPL_WIN32
 *      if the filename referred to the corresponding application type.
 *      If the file name could not be found or did not refer to any known
 *      application type, APPL_NONE is returned and an error message is
 *      left in interp.  .bat files are identified as APPL_DOS.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
ApplicationType(interp, originalName, fullPath)
    Tcl_Interp *interp;         /* Interp, for error message. */
    const char *originalName;   /* Name of the application to find. */
    char fullPath[MAX_PATH];    /* Filled with complete path to
                                 * application. */
{
    int applType, i;
    char *ext;
    static char extensions[][5] = {"", ".cmd", ".exe", ".bat", ".com"};
    char tmpPath[MAX_PATH];

    applType = APPL_NONE;
    for (i = 0; i < (int) (sizeof(extensions) / sizeof(extensions[0])); i++) {
        FILESTATUS3 filestat;
        ULONG flags;

        strncpy((char *)tmpPath, originalName, MAX_PATH - 5);
        strcat(tmpPath, extensions[i]);

        if (tmpPath[1]!=':' && tmpPath[0]!='\\') {
            rc = DosSearchPath(SEARCH_ENVIRONMENT | SEARCH_CUR_DIRECTORY |
                               SEARCH_IGNORENETERRS, "PATH", tmpPath,
                               (PBYTE)fullPath, MAXPATH);
            if (rc != NO_ERROR) {
                continue;
            }
        } else {
            strcpy(fullPath, tmpPath);
        }

        /*
         * Ignore matches on directories or data files, return if identified
         * a known type.
         */

        rc = DosQueryPathInfo(fullPath, FIL_STANDARD, &filestat,
                              sizeof(FILESTATUS3));
        if (rc != NO_ERROR || filestat.attrFile & FILE_DIRECTORY) {
            continue;
        }

        rc = DosQueryAppType(fullPath, &flags);
        if (rc != NO_ERROR) {
            continue;
        } 

        if (flags & FAPPTYP_DLL || flags & FAPPTYP_PHYSDRV ||
            flags & FAPPTYP_VIRTDRV || flags & FAPPTYP_PROTDLL) {
            /* No executable */
            continue;
        } 

        if (flags & FAPPTYP_NOTSPEC) {

            /* Not a recognized type, try to see if it's a .cmd or .bat */
            ext = strrchr(fullPath, '.');
            if ((ext != NULL) && (stricmp(ext, ".cmd") == 0)) {
                applType = APPL_OS2CMD;
                break;
            }
            if ((ext != NULL) && (stricmp(ext, ".bat") == 0)) {
                applType = APPL_DOS;
                break;
            }
            /* Still not recognized, might be a Win32 PE-executable */
            applType = APPL_NONE;
            break;
        }

        if (flags & FAPPTYP_NOTWINDOWCOMPAT) {
            applType = APPL_OS2FS;
        }
        if (flags & FAPPTYP_WINDOWCOMPAT) {
            applType = APPL_OS2WIN;
        }
        if (flags & FAPPTYP_WINDOWAPI) {
            applType = APPL_OS2PM;
        }
        if (flags & FAPPTYP_DOS) {
            applType = APPL_DOS;
        }

        break;
    }

    if (applType == APPL_NONE) {
        TclOS2ConvertError(rc);
        Tcl_AppendResult(interp, "couldn't execute \"", originalName,
                "\": ", Tcl_PosixError(interp), (char *) NULL);
        return APPL_NONE;
    }
    return applType;
}
