/* Asuactl.dll
 * Pasek narzdzi - object manager
 * (c) 1999 Wojciech Gazda
 *
 * objmgr.c
 *
 * $Author: Wojciech_Gazda $
 * $Date: 1999/06/27 12:36:00 $
 * $Name:  $
 * $RCSfile: objmgr.c $
 * $Revision: 1.1 $
 *
 */

/*  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


//#define  INCL_DOSDEV
#define  INCL_DOSPROCESS


// Deklaracje OS/2
#define  INCL_GPILOGCOLORTABLE
#define  INCL_WINRECTANGLES
#define  INCL_WINMESSAGEMGR
#define  INCL_WINWINDOWMGR
#define  INCL_WINFRAMEMGR
#define  INCL_WINPOINTERS
#define  INCL_WINBUTTONS
#define  INCL_WINTIMER
#define  INCL_WININPUT
#define  INCL_WINATOM
#define  INCL_WINSYS
#include <os2.h>

// Funkcje biblioteczne
#include <malloc.h>
#include <stdlib.h>
#include <string.h>

// Deklaracje lokalne
#define  __INTERNAL_USE__
#include "asuintl.h"
#include "toolbar.h"
#include "tooldefs.h"
#include "objmgr.h"


// Deklaracje staych
#define TMR_TIPREMOVE  1      // Czasomierz sterujcy wyaczaniem podpowiedzi
#define TMR_TIPDELAY   2      // czasomierz odmierzajcy czas opnienia podpowidzi


// Prototypy funkcji
MRESULT EXPENTRY ObjectPadProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);
MRESULT EXPENTRY ObjectPassThru(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);
LONG ObjDeleteObject(HWND hwnd, ULONG objId);
LONG ObjIdFromPosition(HWND hwnd, ULONG pos);
LONG ObjInitialize(TOOLCTL *tcl);
LONG ObjInsertObject(HWND hwnd, TOOLOBJ *tobj, TOOLWIN *twin);
LONG ObjInsertTooltip(HWND hwnd, ULONG objId, PSZ tip);
LONG ObjMoveObject(HWND hwnd, ULONG objId, ULONG newpos, ULONG flAttrs);
HWND ObjQueryHandle(HWND hwnd, ULONG id);
LONG ObjQueryHeight(TOOLCTL *tcl, ULONG flNewState);
LONG ObjQueryObject(HWND hwnd, ULONG objId, TOOLOBJ *tobj);
LONG ObjQueryObjectCount(HWND hwnd);
LONG ObjQueryPosition(HWND hwnd, ULONG objId);
LONG ObjQueryTip(HWND hwnd, ULONG objId, ULONG ulBufSize, PSZ TipText);
LONG ObjQueryWidth(TOOLCTL *tcl, ULONG flNewState);
VOID ObjReleaseResources(TOOLCTL *tcl);
LONG ObjSetObject(HWND hwnd, ULONG objId, ULONG ulOptions, TOOLOBJ *tobj);
LONG ObjShowObject(HWND hwnd, ULONG id, ULONG bNewState);


// Prototypy funkcji lokalnych
static LONG ObjControlObjects(HWND hwnd, ULONG id, ULONG command);
static VOID ObjControlTip(TOOLCTL *tcl, ULONG timer, ULONG id);
static VOID ObjDisplayTip(TOOLCTL *tcl, ULONG id);
static LONG ObjHittest(HWND hwnd, POINTS *cpos);
static VOID ObjPlaceObjects(HWND hwnd);
static VOID ObjRedrawObjects(HWND hwnd);


// Dodatkowe funkcje usugowe
static VOID    CalcObjectSize(TOOLCTL *tcl);
static int     CompareObjects(const void *object1, const void *object2);
static LONG    DeletePositionIndex(TOOLCTL *tcl, ULONG objidx);
static LONG    ExpandObjectTable(TOOLCTL *tcl);
static OBJCTL *FindObjectID(TOOLCTL *tcl, ULONG id);
static LONG    InsertPositionIndex(TOOLCTL *tcl, ULONG objidx, ULONG iPosition);
static VOID    PlaceEndcut(TOOLCTL *tcl, RECTL *area);
static VOID    PlaceStandard(TOOLCTL *tcl, RECTL *area);
static VOID    ShrinkObjectTable(TOOLCTL *tcl, ULONG objidx);
static VOID    UpdateAlignment(TOOLCTL *tcl, LONG objidx, ULONG flNewAttrs);
static VOID    UpdateGroupStyle(TOOLCTL *tcl, ULONG ulGroup, ULONG flNewAttrs);
static VOID    DrawSeparator(HPS hps, OBJCTL *obj, RECTL *area, ULONG flState);




// Procedura obsugujca okno przechowujce obiekty
//
MRESULT EXPENTRY ObjectPadProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
  switch(msg)
  { case TM_OBJECTCONTROL:      // Sterowanie zachowaniem przyciskw i innych obiektw
      return(MRFROMLONG(ObjControlObjects(hwnd, LONGFROMMP(mp1), LONGFROMMP(mp2))));

    case WM_CALCVALIDRECTS:
      WinInvalidateRect(hwnd, NULL, TRUE);
      // DosBeep(1000, 10);
      break;

    case WM_HITTEST:     // Przezroczysto dla komunikatw myszy
      return(MRFROMLONG(ObjHittest(hwnd, (POINTS *)&mp1)));

    case WM_PAINT:       // Rysowanie separatorw i ta
      ObjRedrawObjects(hwnd);
      break;

    case WM_SIZE:        // Pozycjonowanie obiektw
      ObjPlaceObjects(hwnd);
      break;

    case WM_TIMER:       // Sterowanie procesem wywietlania podpowiedzi
      { TOOLCTL *tcl;    // Wskanik do gwnej struktury kontrolnej paska narzdzi

        // Odczyt adresu struktir kontrolnych
        tcl = ToolLoadData(hwnd, "ObjectPadProc");
        if(tcl == NULL) break;
        // Sterowanie wywietlaniem podpowiedzi
        ObjControlTip(tcl, SHORT1FROMMP(mp1), 0);
      }
      break;
  }

  // Powrt do standardowej procedury okna
  return(WinDefWindowProc(hwnd, msg, mp1, mp2));
}





// Procedura dokonuje wstpnej obrbki komunikatw napywajcych do
// obiektu umieszczonego na pasku narzdzi, i generuje komunikaty
// sterujce wywietlaniem podpowiedzi i wspomagajce
// sterowaniem przyciskami.
//
MRESULT EXPENTRY ObjectPassThru(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{ TOOLCTL *tcl;     // Wskanik do gwnej struktury kontrolnej paska narzdzi
  OBJCTL  *obj;     // Adresu struktury kontrolnej obiektu
  ULONG    id;      // Identyfikator okna
  HWND     parent;  // Uchwyt waciciela okna obiektu

  // Odczyt uchwytu okna rodzicielskiego i sprawdzenie jego poprawnoci
  { UCHAR classname[OBJ_CLASSIZE];      // Bufor zawierajcy nazw klasy

    parent = WinQueryWindow(hwnd, QW_PARENT);
    WinQueryClassName(parent, OBJ_CLASSIZE, classname);
    if(strcmp(classname, INTC_OBJECTPAD))
    { // Niewaciwa klasa okna rodzicielskiego, powrt
      return(WinDefWindowProc(hwnd, msg, mp1, mp2));
    }
  }
  // Odczyt adresu struktury kontrolnej paska narzdzi
  tcl = ToolLoadData(parent, "ObjectPassThru");
  if(tcl == NULL) return(WinDefWindowProc(hwnd, msg, mp1, mp2));
  // Odczyt wskanika do obiektu
  id = WinQueryWindowUShort(hwnd, QWS_ID);
  obj = FindObjectID(tcl, id);
  if(obj == NULL) return(WinDefWindowProc(hwnd, msg, mp1, mp2));

  // Analiza komunikatw przeznaczonych dla obiektw
  switch(obj->flAttribute & TBO_BASICTYPES)
  { case TBO_BUTTON:          // Przyciski WC_WINBUTTON
    case TBO_STDBUTTON:       // Standardowe przyciski systemowe
      // Uwzgldniane s tylko przyciski w grupach TBO_CHECKBOX i TBO_RADIOBUTTON
      if(WinIsWindowEnabled(obj->hwndObject) == TRUE)
        switch(obj->flAttribute & TBO_RADIOBUTTON)
        { case TBO_RADIOBUTTON:
            if(msg == WM_BUTTON1DOWN)
            { // Zwolnienie pozostaych przyciskw w grupie
              WinPostMsg(parent, TM_OBJECTCONTROL, MPFROMLONG(id), MPFROMLONG(TNC_RELEASEGROUP));
            }

          case TBO_CHECKBOX:
            if(msg == WM_BUTTON1DOWN)
            { // Odczyt aktualnego stanu przycisku i zapamitanie go
              if((LONG)obj->pfnObjProc(hwnd, BM_QUERYHILITE, 0, 0) == TRUE)
                obj->flAttribute |= TBO_BUTTONPRESS;
              else obj->flAttribute &= ~TBO_BUTTONPRESS;
            }
            if(msg == WM_BUTTON1UP)
            { // Blokada przerysowywania przycisku
              WinEnableWindowUpdate(hwnd, FALSE);
              // Wstawienie do kolejki komunikatw okna rodzicielskiego potwierdzenia
              WinPostMsg(parent, TM_OBJECTCONTROL, MPFROMLONG(id), MPFROMLONG(TNC_REDRAWBUTTON));
            }
        }

    case TBO_STDWINDOW:       // Pozostae obiekty + przyciski
      if(msg == WM_MOUSEMOVE)
      { // Wysanie komunikatu rozpoczynajcego wywietlanie podpowiedzi
        if(tcl->lActiveTip != id)
          WinPostMsg(parent, TM_OBJECTCONTROL, MPFROMLONG(id), MPFROMLONG(TNC_STARTTIP));
      }
      break;
  }

//// Korekta bdw powstajcych w stanie floating podczas zmiany ogniska
//if(msg == WM_QUERYFOCUSCHAIN)
//{ switch(SHORT1FROMMP(mp1))
//  { case QFC_PARTOFCHAIN:
//      return(MRFROMLONG(FALSE));
//
//  }
//}

  // Powrt do oryginalnej procedury okna obiektu
  return(obj->pfnObjProc(hwnd, msg, mp1, mp2));
}





// Usunicie obiektu z paska narzdzi
//
// Parametry:
//   hwnd      - [parametr] uchwyt paska narzdzi
//   objId     - [parametr] identyfikator obiektu
//
// Powrt:
//   FALSE - bad wykonania
//   TRUE  - obiekt usunito
//
LONG ObjDeleteObject(HWND hwnd, ULONG objId)
{ TOOLCTL *tcl;     // Wskanik do struktur kontrolnych okna
  OBJCTL  *obj;     // Wskanik do znalezionego obiektu
  LONG     objidx;  // Indeks usuwanego obiektu

  tcl = ToolLoadData(hwnd, "ObjDeleteObject");
  if(tcl == NULL) return(FALSE);

  // Odczyt wskanika do obiektu
  obj = FindObjectID(tcl, objId);
  if(obj == NULL) return(FALSE);

  // Usunicie okna
  if(obj->hwndObject != NULLHANDLE)
    WinDestroyWindow(obj->hwndObject);
  // Usunicie wpisu z tablicy atomw (usunicie podpowiedzi)
  if(obj->aToolTip != 0)
    WinDeleteAtom(tcl->hatomTips, obj->aToolTip);

  // Reorganizacja tablicy obiektw
  objidx = DeletePositionIndex(tcl, obj->iPosition);
  // Usunicie wpisu z tablicy obiektw
  ShrinkObjectTable(tcl, objidx);

  // Aktualizacja rozmiarw paska narzdzi
  if(!(tcl->flToolState & TST_MINIMIZED))
    if((LONG)WinSendMsg(tcl->hwndToolBar, TM_AUTOSIZE, 0, 0) == FALSE)
    { // Rozmieszczenie obiektw
      ObjPlaceObjects(tcl->hwndObjectPad);
      // Wymuszenie przerysowania separatorw
      WinInvalidateRect(tcl->hwndObjectPad, NULL, FALSE);
    }
  return(FALSE);
}





// Funkcja odczytuje identyfikator obieku na podstawie przekazanej pozycji
//
// Parametry:
//   hwnd      - [parametr] uchwyt paska narzdzi
//   pos       - [parametr] pozycja obiektu
//
// Powrt:
//   0 - bd odczytu
//   Identyfikator obiektu.
//
LONG ObjIdFromPosition(HWND hwnd, ULONG pos)
{ TOOLCTL *tcl;     // Wskanik do struktur kontrolnych paska narzdzi
  OBJCTL  *objtab;  // Wskanik do tablicy definicji obiektw

  tcl = ToolLoadData(hwnd, "ObjIdFromPosition");
  if(tcl == NULL) return(0);
  objtab = tcl->objtab;

  if(tcl->nObjects == 0) return(0);
  if(pos >= tcl->nObjects) pos = tcl->nObjects - 1;

  // Odczyt identyfikatora
  return(objtab[pos].id);
}





// Funkcja inicjuje podstawowe struktury danych kontrolujce obiekty
// umieszczane na pasku narzdzi
//
// Parametry:
//   tcl       - [parametr] adres gwnej struktury kontrolnej paska narzdzi
//
// Powrt:
//   FALSE     - poprawne wykonanie funkcji
//   TRUE      - bd wykonania
//
LONG ObjInitialize(TOOLCTL *tcl)
{
  // Tworzenie lokalnej tablicy atomw dla podpowiedzi
  tcl->hatomTips = WinCreateAtomTable(0, 0);
  if(tcl->hatomTips == NULLHANDLE) return(TRUE);

  // Tworzenie okna przechowujcego obiekty
  tcl->hwndObjectPad = WinCreateWindow(tcl->hwndToolBar, INTC_OBJECTPAD, "", 0,
                         0, 0, 0, 0,
                         tcl->hwndToolBar, HWND_TOP,
                         TCID_OBJECTPAD,
                         NULL, NULL);
  if(tcl->hwndObjectPad == NULLHANDLE) return(TRUE);

  // Zapamitanie wskanika do struktur kontrolnych
  WinSetWindowPtr(tcl->hwndObjectPad, 0L, tcl);
  return(FALSE);
}





// Funkcja dodaje okno potomne do paka narzdzi, ustawiajc jednoczenie
// odpowiednie atrybuty.
//
// Parametry:
//   hwnd      - [parametr] uchwyt paska narzdzi
//   tobj      - [parametr] wskanik do struktury TOOLOBJ zawierajcej atrybuty obiektu
//   twin      - [parametr] wskanik do struktury TOOLWIN identyfikujcej typ okna
//
// Powrt:
//   TRUE  - poprawne wykonanie funkcji
//   FALSE - bd wykonania
//
LONG ObjInsertObject(HWND hwnd, TOOLOBJ *tobj, TOOLWIN *twin)
{ TOOLCTL *tcl;          // Wskanik do gwnych struktur kontrolnych okna
  OBJCTL  *obj;          // Wskanik do struktury zawierajcej dodawany obiekt
  LONG     winstyle;     // Skorygowany styl okna
  LONG     objidx;       // Indeks nowo utworzonego obiektu
  HWND     owner;        // Waciciel noweho okna potomnego

  // Odczyt adresu struktur kontrolnych
  tcl = ToolLoadData(hwnd, "ObjInsertObject");
  if(tcl == NULL) return(FALSE);

  // Sprawdzenie poprawnoci danych przekazywanych za porednictwem struktur
  if(tobj == NULL) return(FALSE);
  // Sprawdzenie czy identyfikator nie powtarza si
  if(FindObjectID(tcl, tobj->id) != NULL) return(FALSE);
  // Jeli obiekt nie jest separatorem, wymaga dodatkowych struktur
  if((tobj->flAttribute & 0x0F) != TBO_SEPARATOR)
  { // Sprawdzenie czy przekazano wskanik do tych struktur
    if(twin == NULL) return(FALSE);
  }

  // Alokacja pamici dla dodatkowej struktury kontrolnej
  objidx = ExpandObjectTable(tcl);
  if(objidx < 0) return(FALSE);
  // Okrelenie pooenia nowego obiektu
  objidx = InsertPositionIndex(tcl, objidx, tobj->iPosition);
  obj = tcl->objtab + objidx;

  // Wypenianie struktury kontrolnej obiektu
  obj->ulGroup     = tobj->ulGroup;          // Grupa przyciskw
  obj->flAttribute = tobj->flAttribute;      // Zapamitanie atrybutw
  obj->cx          = tobj->cx;               // Pocztkowe wymiary okna
  obj->cy          = tobj->cy;
  obj->id          = tobj->id;               // Identyfikator okna
  obj->ulUser      = tobj->ulUser;           // Dane uytkowika

  // Aktualizacja parametrw grupy
  UpdateGroupStyle(tcl, obj->ulGroup, tobj->flAttribute);
  // Aktualizacja sposobu wyrwnania
  UpdateAlignment(tcl, objidx, tobj->flAttribute);

  // Tworzenie okna potomnego
  if((tobj->flAttribute & TBO_BASICTYPES) != TBO_SEPARATOR)
  { // Okrelenie waciciela nowego obiektu
    owner = WinQueryWindow(tcl->hwndToolBar, QW_OWNER);

    // Korekta stylu okna
    winstyle  = twin->flStyle | WS_VISIBLE | WS_CLIPSIBLINGS;
    winstyle &= ~WS_DISABLED;
    if(tobj->flAttribute & TBO_HIDE)    winstyle &= ~WS_VISIBLE;
    if(tobj->flAttribute & TBO_DISABLE) winstyle |=  WS_DISABLED;

    // Tworzenie okna potomnego
    obj->hwndObject = WinCreateWindow(tcl->hwndObjectPad, twin->pszClass, twin->pszName,
                       winstyle,
                       0, 0, 0, 0,
                       owner, HWND_TOP, obj->id,
                       twin->pCtlData, twin->pPresParams);

    // Obsuga bdu tworzenia okna
    if(obj->hwndObject == NULLHANDLE)
    { // Zmiana pooenia pozostaych obiektw
      objidx = DeletePositionIndex(tcl, objidx);
      // Usunicie struktur kontrolnych
      ShrinkObjectTable(tcl, objidx);
      return(FALSE);
    }

    // Zamiana procedury okna
    obj->pfnObjProc = WinSubclassWindow(obj->hwndObject, ObjectPassThru);
  }
  else obj->hwndObject = NULLHANDLE;

  // Aktualizacja rozmiarw okna
  if(!(tcl->flToolState & TST_MINIMIZED))
    if((LONG)WinSendMsg(tcl->hwndToolBar, TM_AUTOSIZE, 0, 0) == FALSE)
    { // Rozmieszczenie obiektw
      ObjPlaceObjects(tcl->hwndObjectPad);
      // Wymuszenie przerysowania separatorw
      WinInvalidateRect(tcl->hwndObjectPad, NULL, FALSE);
    }
  return(TRUE);
}




// Funkcja dodaje lub usuwa podpowied do obiektu o podanym identyfikatorze,
// umieszczonym na pasku narzdzi.
//
// Parametry:
//   hwnd      - [parametr] uchwyt paska narzdzi
//   objId     - [parametr] identyfikator obiektu
//   tip       - [parametr] dodawany cig znakw
//
// Powrt:
//   TRUE  - poprawne wykonanie
//   FALSE - bad wykonanie
//
LONG ObjInsertTooltip(HWND hwnd, ULONG objId, PSZ tip)
{ TOOLCTL *tcl;     // Wskanik do gwnej struktury kontrolnej okna
  OBJCTL  *obj;     // Wskanik do struktury kontrolnej obiektu

  // Inicjacja zmiennych
  tcl = ToolLoadData(hwnd, "ObjInsertToolTip");
  if(tcl == NULL) return(FALSE);
  obj = FindObjectID(tcl, objId);
  if(obj == NULL) return(FALSE);

  // Sprawdzneie czy jest to separator
  if((obj->flAttribute & TBO_BASICTYPES) == TBO_SEPARATOR) return(FALSE);

  // Usunicie tekstu podpowiedzi
  if(obj->aToolTip != 0)
  { WinDeleteAtom(tcl->hatomTips, obj->aToolTip);
    obj->aToolTip = 0;
  }

  // Dodanie nowego tekstu podpowiedzi
  if(tip != NULL)
    if(*tip != 0)
      obj->aToolTip = WinAddAtom(tcl->hatomTips, tip);
  return(TRUE);
}





// Funkcja odczytuje uchwyt okna nalecy do obiektu o podanym identyfikatorze.
//
// Paramery:
//   hwnd      - [parametr] uchwyt paska narzdzi
//   id        - [parametr] identyfikator obiektu
//
// Powrt:
//   Uchwyt okna
//   NULLHANDLE - obiekt jest separatorem lub nie istnieje
//
HWND ObjQueryHandle(HWND hwnd, ULONG id)
{ TOOLCTL *tcl;     // Wskanik do gwnej struktury kontrolnej okna
  OBJCTL  *obj;     // Wskanik do definicji ukrywanego obiektu

  tcl = ToolLoadData(hwnd, "ObjQueryHandle");
  if(tcl == NULL) return(NULLHANDLE);
  obj = FindObjectID(tcl, id);
  if(obj == NULL) return(NULLHANDLE);
  return(obj->hwndObject);
}





// Funkcja przenosi obiekt z dotychczasowej pozycji na now,
// umoliwiajc jednoczenie zmian typu wyrwnania. Atrybuty flAttrs
// mog przyjmowa wartoci: TBO_BEGINALIGN, TBO_ENDALIGN lub TBO_CENTERALIGN.
//
// Parametry:
//   hwnd      - [parametr] uchwyt paska narzdzi
//   objId     - [parametr] identyfikator obiektu
//   newpos    - [parametr] nowa pozycja obiektu
//   flAttrs   - [parametr] nowe parametry sterujce wyrwnaniem
//
// Powrt:
//   TRUE  - poprawne wykonanie
//   FALSE - bd
//
LONG ObjMoveObject(HWND hwnd, ULONG objId, ULONG newpos, ULONG flAttrs)
{ TOOLCTL *tcl;     // Adres struktur kontrolnych paska narzdzi
  OBJCTL  *obj;     // Adres struktury kontrolnej przemieszczanego obiektu
  OBJCTL  *objtab;  // Adres pocztku tablicy definicji obiektw
  LONG     objidx;  // Nowy indeks obiektu powstay w wyniku jego przemieszczenia

  // Inicjacja zmiennych
  tcl = ToolLoadData(hwnd, "ObjMoveObject");
  if(tcl == NULL) return(FALSE);
  obj = FindObjectID(tcl, objId);
  if(obj == NULL) return(FALSE);
  objtab = tcl->objtab;

  // Sprawdzenie czy w tabeli s obiekty
  if(tcl->nObjects == 0) return(FALSE);
  // Korekta pozycji docelowej
  if(newpos >= tcl->nObjects) newpos = tcl->nObjects - 1;

  // Sprawdzenie czy obiekt nie jest ju w pozycji docelowej
  if(newpos != obj->iPosition)
  { // Ustalenie nieprawidowej wartoci pozycji w tabeli dla przemieszczanego obiektu
    objidx = DeletePositionIndex(tcl, obj->iPosition);
    // Przesunicie obiektu na pozycj docelow
    objidx = InsertPositionIndex(tcl, objidx, newpos);
  }

  // Korekta justowania obiektu
  UpdateAlignment(tcl, objidx, flAttrs);

  // Rozmieszczenie obiektw
  if(!(tcl->flToolState & TST_MINIMIZED))
  { ObjPlaceObjects(tcl->hwndObjectPad);
    // Wymuszenie przerysowania separatorw
    WinInvalidateRect(tcl->hwndObjectPad, NULL, FALSE);
  }
  return(TRUE);
}





// Funkcja odczytuje wysoko obszaru w ktrym znajduj si obiekty,
// niezbdn do ich umieszczenia.
//
// Parametry:
//   tcl        - [parametr] wskanik do gwnej struktury kontrolnej paska anrzdzi
//   flNewState - [parametr] stan paska o ktry pytamy
//
// Powrt:
//   Szeroko obszaru.
//
LONG ObjQueryHeight(TOOLCTL *tcl, ULONG flNewState)
{ OBJCTL  *objtab;  // Tablica definicji obiektw
  LONG     height;  // Obliczona wysoko okna
  LONG i, tmp;      // Zmienne pomocnicze

  // Inicjacja zmiennych
  objtab = tcl->objtab;
  // Aktualizacja szerokoci obiektw
  CalcObjectSize(tcl);

  for(i = 0, height = 0; i < tcl->nObjects; ++i)
  { if((objtab[i].flAttribute & TBO_BASICTYPES) == TBO_SEPARATOR)
    { // Separator - szeroko zaley od pooenia paska narzdzi
      tmp = flNewState & TST_ROTATED ? objtab[i].cy : 0;
    }
    else
    { // Pozostae obiekty na pasku narzdzi
      tmp = objtab[i].cy;
    }
    // Sprawdzenie czy obiekt jest widoczny, i jeli nie - blokada odczytu
    if(objtab[i].flAttribute & TBO_HIDE) tmp = 0;

    if(flNewState & TST_ROTATED)
    { // Pasek umieszczony pionowo - odczyt sumy wysokoci wszystkich obiektw
      height += tmp;
      if((i < (tcl->nObjects - 1)) && !(objtab[i].flAttribute & TBO_HIDE))
        height += CXY_OBJECTSPACE;
    }
    else
    { // Pasek umieszczony poziomo - odczyt maksymalnej wysokoci
      if(height < tmp) height = tmp;
    }
  }

  // Ustawienie domylnej wysokoci okna (gdy nie ma na nim obiektw)
  if(!height) height = 25;
  // Powrt - odczytana wysoko okna
  return(height);
}





// Odczyt parametrw obuiektu
//
// Parametry:
//   hwnd      - [parametr] uchwyt paska narzdzi,
//   objId     - [parametr] identyfikator obiektu,
//   tobj      - [rezultat] wskanik do struktury informacyjnej obiektu
//
// Powrt:
//   FALSE - bd odczytu,
//   TRUE  - poprawny odczyt danych
//
LONG ObjQueryObject(HWND hwnd, ULONG objId, TOOLOBJ *tobj)
{ TOOLCTL *tcl;     // Adres struktury kontrolnej paska narzdzi
  OBJCTL  *obj;     // Wskanik do struktury kontrolnej obiketu

  // Inicjacja zmiennych
  tcl = ToolLoadData(hwnd, "ObjQueryObject");
  if(tcl == NULL) return(FALSE);
  obj = FindObjectID(tcl, objId);
  if(obj == NULL) return(FALSE);

  // Kopiowanie pl struktury
  tobj->iPosition   = obj->iPosition;
  tobj->ulGroup     = obj->ulGroup;

  // Tylko atrybuty przeznaczone dla uytkownika
  tobj->flAttribute = obj->flAttribute & 0xFFFF;
  // Aktualizacja znacznika blokady okna
  tobj->flAttribute &= ~TBO_DISABLE;
  if(obj->hwndObject != NULLHANDLE)
    if(WinIsWindowEnabled(obj->hwndObject) == FALSE)
       tobj->flAttribute |= TBO_DISABLE;

  // Pozostae pola struktury
  tobj->cx          = obj->cx;
  tobj->cy          = obj->cy;
  tobj->id          = obj->id;
  tobj->ulUser      = obj->ulUser;
  return(TRUE);
}





// Funkcja odczytuje liczn obiektw umieszczonych na pasku narzdzi.
//
// Parametry:
//   hwnd      - [parametr] uchwyt paska narzdz
//
// Powrt:
//   Liczba obiektw umieszczonych na pasku
//
LONG ObjQueryObjectCount(HWND hwnd)
{ TOOLCTL *tcl;     // Adres gwnej struktury kontrolnej paska narzdzi

  tcl = ToolLoadData(hwnd, "ObjQueryObjectCout");
  if(tcl == NULL) return(0);
  return(tcl->nObjects);
}





// Funkcja odczytuje pooenie obiektu na podstawie identyfikatora
//
// Parametry:
//   hwnd      - [parametr] uchwyt paska narzdzi
//   objId     - [parametr] identyfikator obiektu
//
// Powrt:
//   >0 - pooenie obiektu
//   -1 - bd
//
LONG ObjQueryPosition(HWND hwnd, ULONG objId)
{ TOOLCTL *tcl;     // Wskanik do gwnej struktury kontrolnej paska narzdzi
  OBJCTL  *obj;     // Wskanik do obiektu

  // Inicjacja zmiennych
  tcl = ToolLoadData(hwnd, "ObjQueryPosition");
  if(tcl == NULL) return(-1);
  obj = FindObjectID(tcl, objId);
  if(obj == NULL) return(-1);

  // Odczytana pozycja
  return(obj->iPosition);
}





// Odczyt tekstu podpowiedzi do obiektu
//
// Parametry:
//   hwnd      - [parametr] uchwyt paska narzdzi
//   objId     - [parametr] identyfikator obiektu
//   ulBufSize - [parametr] rozmiar bufora TipText
//   TipText   - [rezultat] odczytany tekst pdpowiedzi
//
// Rezultaty:
//   Dugo tekstu podpowiedzi bez uzgldnienia kocowego znaku 0.
//
LONG ObjQueryTip(HWND hwnd, ULONG objId, ULONG ulBufSize, PSZ TipText)
{ TOOLCTL *tcl;     // Wskanik do gwnej struktury kontrolnej paska narzdzi
  OBJCTL  *obj;     // Adres struktury kontrolnej obiektu
  LONG tip;         // Dugo tekstu podpowiedzi

  // Inicjacja zmiennych
  tcl = ToolLoadData(hwnd, "ObjQueryTip");
  if(tcl == NULL) return(0);
  obj = FindObjectID(tcl, objId);
  if(obj == NULL) return(0);

  // Sprawdzenie czy jest tekstu podpowiedzi
  if(obj->aToolTip == 0) return(0);
  // Odczyt dugoci tekstu
  tip = WinQueryAtomLength(tcl->hatomTips, obj->aToolTip);
  if(tip && (ulBufSize != 0) && (TipText != NULL))
  { // Odczyt tekstu
    WinQueryAtomName(tcl->hatomTips, obj->aToolTip, TipText, ulBufSize);
  }

  // Odczytana dugoc tekstu
  return(tip);
}





// Funkcja odczytuje szeroko obszaru w ktrym znajduj si obiekty,
// niezbdn do ich umieszczenia.
//
// Parametry:
//   tcl        - [parametr] wskanik do gwnej struktury kontrolnej paska narzdzi
//   flNewState - [parametr] stan paska o ktry pytamy
//
// Powrt:
//   Szeroko obszaru.
//
LONG ObjQueryWidth(TOOLCTL *tcl, ULONG flNewState)
{ OBJCTL  *objtab;  // Tablica definicji obiektw
  LONG     width;   // Obliczona szeroko okna
  LONG i, tmp;      // Zmienne pomocnicze

  // Inicjacja zmiennych
  objtab = tcl->objtab;
  // Obliczenie rozmiarw obiektw
  CalcObjectSize(tcl);

  for(i = 0, width = 0; i < tcl->nObjects; ++i)
  { if((objtab[i].flAttribute & TBO_BASICTYPES) == TBO_SEPARATOR)
    { // Separator - szeroko zaley od pooenia paska narzdzi
      tmp = flNewState & TST_ROTATED ? 0 : objtab[i].cx;
    }
    else
    { // Pozostae obiekty na pasku narzdzi
      tmp = objtab[i].cx;
    }
    // Sprawdzenie czy obiekt jest widoczny, i jeli nie - blokada odczytu
    if(objtab[i].flAttribute & TBO_HIDE) tmp = 0;

    if(flNewState & TST_ROTATED)
    { // Pasek umieszczony pionowo - odczyt maksymalnej szerokoci
      if(width < tmp) width = tmp;
    }
    else
    { // Pasek umieszczony poziomo - odczyt sumy szerokoci wszystkich obiektw
      width += tmp;
      if((i < (tcl->nObjects - 1)) && !(objtab[i].flAttribute & TBO_HIDE))
        width += CXY_OBJECTSPACE;
    }
  }

  // Ustawienie domylnej szerokoci okna (gdy nie ma na nim obiektw)
  if(!width) width = 25;
  // Powrt - odczytana szeroko okna
  return(width);
}





// Funkcja oczyszczajca struktury danych paska narzdzi podczas obsugi WM_DESTROY.
// Zostaje zwolniona pami, tablice atomw, oraz zostaj zniszczone okna potomne.
//
// Parametry:
//   tcl       - [parametr] adres gwnej struktury kontrolnej paska narzdzi
//
VOID ObjReleaseResources(TOOLCTL *tcl)
{
  // Usunicie tablicy atomw przechowujcej podpowiedzi
  if(tcl->hatomTips != NULLHANDLE)
    WinDestroyAtomTable(tcl->hatomTips);

  // Usunicie okna przechowujcego obiekty
  if(tcl->hwndObjectPad != NULLHANDLE)
    WinDestroyWindow(tcl->hwndObjectPad);

  // Zwolnienie pamici zajmowanej przez tablic obiektw
  if(tcl->objtab != NULL) free(tcl->objtab);
}





// Funkcja zmienia parametry wybranego obiektu na podstawie danych przekazanych
// za porednictwem struktury TOOLOBJ, oraz zmiennej ulOptions.
//
// Parametry:
//   hwnd      - [parametr] uchwyt paska narzdzi
//   objId     - [parametr] identyfikator obiektu
//   ulOptions - [parametr] opcje umoliwiajce wybr wanych pl struktury tobj
//   tobj      - [parametr] struktura zawierajca moyfikowane parametry
//
// Powrt:
//   FALSE - bd wykonania
//   TRUE  - zmiana parametrw zostaa wykonana poprawnie
//
LONG ObjSetObject(HWND hwnd, ULONG objId, ULONG ulOptions, TOOLOBJ *tobj)
{ TOOLCTL *tcl;     // Wskanik do gwnej struktury kontrolnej paska narzdzi
  OBJCTL  *obj;     // Wskanik do struktury kontrolnej wybranego obiektu

  // Inicjacja zmiennych
  tcl = ToolLoadData(hwnd, "ObjSetObject");
  if(tcl == NULL) return(FALSE);
  obj = FindObjectID(tcl, objId);
  if(obj == NULL) return(FALSE);

  // Zmiana przynalenoci do grupy
  if((ulOptions & TSO_GROUP) && (obj->ulGroup != tobj->ulGroup))
  { OBJCTL *objtab;      // Adres pocztku tabeli zawierajcej definicje obiektw
    LONG i;

    // Inicjacja zmiennych
    objtab = tcl->objtab;
    obj->flAttribute &= ~TBO_RADIOBUTTON;
    // Odczyt parametrw grupy (jeli s)
    for(i = 0; i < tcl->nObjects; ++i)
      if(objtab[i].ulGroup == tobj->ulGroup)
      { // Nadanie parametrw obiektowi
        obj->flAttribute |= (objtab[i].ulGroup & TBO_RADIOBUTTON);
        break;
      }
    // Zmiana numeru grupy
    obj->ulGroup = tobj->ulGroup;
  }

  // Zmiana wymiarw obiektu
  if(ulOptions & TSO_SIZE)
  { obj->cx = tobj->cx;
    obj->cy = tobj->cy;
    // Korekta wymiarw okna
    if(!(tcl->flToolState & TST_MINIMIZED))
      if((LONG)WinSendMsg(tcl->hwndToolBar, TM_AUTOSIZE, 0, 0) == FALSE)
      { // Pozycjonowanie obiektw w przypadku gdy wymiary okna nie zmieniy si
        ObjPlaceObjects(tcl->hwndObjectPad);
        // Wymuszenie przerysowania separatorw
        WinInvalidateRect(tcl->hwndObjectPad, NULL, FALSE);
      }
  }

  // Zmiana pola uytkownika
  if(ulOptions & TSO_USER)
  { obj->ulUser = tobj->ulUser;
  }
  return(TRUE);
}





// Funkcja ukrywa lub wywietla wybrany obiekt umieszczony na pasku narzdzi.
// Gdy bNewState = TRUE obiekt jest wywietlany, gdy FALSE - jest ukrywany.
//
// Parametry:
//   hwnd      - [parametr] uchwyt paska narzdzi
//   id        - [parametr] identyfikator obiektu
//   bNewState - [parametr] nowy stan obiektu
//
// Powrt:
//   TRUE  - poprawne wykonanie funkcji
//   FALSE - bd wykonania
//
LONG ObjShowObject(HWND hwnd, ULONG id, ULONG bNewState)
{ TOOLCTL *tcl;     // Wskanik do gwnej struktury kontrolnej okna
  OBJCTL  *obj;     // Wskanik do definicji ukrywanego obiektu

  tcl = ToolLoadData(hwnd, "ObjShowObject");
  if(tcl == NULL) return(FALSE);
  obj = FindObjectID(tcl, id);
  if(obj == NULL) return(FALSE);

  if(bNewState)
  { // Wywietlenie obiektu
    obj->flAttribute &= ~TBO_HIDE;
    // Aktywacja okna, jeli poprzednio byo ukryte
    if((obj->flAttribute & TBO_BASICTYPES) != TBO_SEPARATOR)
      if(WinIsWindowVisible(obj->hwndObject) == FALSE)
        WinShowWindow(obj->hwndObject, TRUE);
  }
  else
  { // Ukrycie obiektu
    obj->flAttribute |=  TBO_HIDE;
    // Dezaktywacja okna jeli poprzednio byo widoczne
    if((obj->flAttribute & TBO_BASICTYPES) != TBO_SEPARATOR)
      if(WinIsWindowVisible(obj->hwndObject) == TRUE)
        WinShowWindow(obj->hwndObject, FALSE);
  }

  // Aktualizacja rozmiarw okna
  if(!(tcl->flToolState & TST_MINIMIZED))
    if((LONG)WinSendMsg(tcl->hwndToolBar, TM_AUTOSIZE, 0, 0) == FALSE)
    { // Rozmieszczenie obiektw
      ObjPlaceObjects(tcl->hwndObjectPad);
      // Wymuszenie przerysowania separatorw
      WinInvalidateRect(tcl->hwndObjectPad, NULL, FALSE);
    }
  return(TRUE);
}





//*******************/
//* Funkcje lokalne */
//*******************/

// Funkcja steruje zachowaniem przyciskw umieszczonych na pasku narzdzi,
// oraz wywietlaniem tekstw podpowiedzi do obiektw
//
// Parametry:
//   hwnd      - [parametr] uchwyt okna przechowujcego obiekty
//   id        - [parametr] identyfikator obiektu
//   command   - [parametr] czynno do wykonania
//
// Rezultaty:
//   0    - stanadardowe zakoczenie wykonania
//
static LONG ObjControlObjects(HWND hwnd, ULONG id, ULONG command)
{ TOOLCTL *tcl;     // Wskanik do gwnej struktury kontrolnej paska narzdzi
  OBJCTL  *obj;     // Wskanik do struktury kontrolnej obiektu
  OBJCTL  *objtab;  // Wskanik do tablicy definicji obiektw
  LONG i;           // Zmienne pomocnicze

  // Inicjacja zmiennych
  tcl = ToolLoadData(hwnd, "ObjControlObjects");
  if(tcl == NULL) return(0);
  obj = FindObjectID(tcl, id);
  if(obj == NULL) return(0);
  objtab = tcl->objtab;

  // Realizacja czynnoci
  switch(command)
  { case TNC_REDRAWBUTTON:         // Sterowanie przyciskami
      { BOOL newstate;        // Nowy stan przycisku

        // Zmiana wygldu przycisku
        if(obj->flAttribute & TBO_BUTTONPRESS)
        { if((obj->flAttribute & TBO_RADIOBUTTON) == TBO_CHECKBOX)
            newstate = FALSE;
        }
        else newstate = TRUE;

        // Zmiana stanu okna
        WinSendMsg(obj->hwndObject, BM_SETHILITE, MPFROMLONG(newstate), 0);
        // Wymuszenie przerysowania przycisku
        WinEnableWindowUpdate(obj->hwndObject, TRUE);
      }
      break;

    case TNC_RELEASEGROUP:    // Zwolnienie wszystkich przyciskw w grupie TBO_RADIOBUTTON
      for(i = 0; i < tcl->nObjects; ++i)
      { if((objtab[i].ulGroup == obj->ulGroup) && (objtab[i].id != obj->id))
          switch(objtab[i].flAttribute & TBO_BASICTYPES)
          { case TBO_STDBUTTON:
            case TBO_BUTTON:
              // Sprawdzenie stanu przycisku
              if((LONG)WinSendMsg(objtab[i].hwndObject, BM_QUERYHILITE, 0, 0) == TRUE)
                // Wyczenie przycisku
                WinSendMsg(objtab[i].hwndObject, BM_SETHILITE, MPFROMLONG(FALSE), 0);
          }
      }
      break;

    case TNC_STARTTIP:        // Rozpoczcie procedury wywietlania podpowiedzi
      ObjControlTip(tcl, 0, id);
      break;
  }
  return(0);
}





// Funkcja steruje zalenociami czasowymi podczas wywietlania tekstu
// podpowiedzi do obiektw.
//
// Parametry:
//   tcl       - [parametr] wskanik do gwnej struktury kontrolnej paska narzdzi
//   timer     - [parametr] identyfikator czasomierza
//   id        - [parametr] identyfikator obiektu dajcego podpowiedzi
//
static VOID ObjControlTip(TOOLCTL *tcl, ULONG timer, ULONG id)
{ POINTL cpos;      // Pozycja kursora
  LONG   delay;     // Czas opnienia przed wywietleniem podpowiedzi
  HWND   hwndc;     // Uchwyt okna znajdujcego si pod kursorem
  HAB    hab;       // Uchwyt anchor block PM

  // Inicjacja zmiennych
  hab = WinQueryAnchorBlock(tcl->hwndToolBar);

  // Zgoszenie dania wywietlenia podpowiedzi
  if(id)
  { // Zapamitanie identyfikatora
    tcl->lActiveTip = id;
    // Wyzwolenie czasomierza umoliwiajcego wykrycie zejcia kursora z paska narzdzi
    WinStartTimer(hab, tcl->hwndObjectPad, TMR_TIPREMOVE, TIME_TIPREMOVE);

    // Odczyt czasu opnienia przed wczeniem podpowiedzi
    if(!WinQueryPresParam(tcl->hwndToolBar, PP_TOOLTIPDELAY, 0L, NULL, 4, &delay, 0L))
      delay = TIME_TIPDELAY;
    // Wyzwolenie czasomierza odmierzajcego czas opnienia przed wywietleniem podpowiedzi
    WinStartTimer(hab, tcl->hwndObjectPad, TMR_TIPDELAY, delay);

    // Sprawdzenie czy podpowied mona wywietli natychmiast
    if(tcl->flToolState & TST_TIPACTIVATED)
    { // Wywietlenie tekstu podpowiedzi
      ObjDisplayTip(tcl, tcl->lActiveTip);
    }
  }
  else if(timer)
  { // Odczyt pozycji kursora
    WinQueryPointerPos(HWND_DESKTOP, &cpos);
    WinMapWindowPoints(HWND_DESKTOP, tcl->hwndObjectPad, &cpos, 1);

    // Blokada pbsugi komunikatu WM_HITTEST
    tcl->flToolState |= TST_HITTEST;
    // Odczyt uchwytu okna znajdujcego si pod kursorem
    hwndc = WinWindowFromPoint(tcl->hwndObjectPad, &cpos, FALSE);
    // Przywrcenie normalnej obsugi komunikatu WM_HITTEST
    tcl->flToolState &= ~TST_HITTEST;

    if(hwndc == NULLHANDLE)
    { // Wyczenie podpowiedzi - kursor znajduje si poza paskiem narzdzi
      WinStopTimer(hab, tcl->hwndObjectPad, TMR_TIPREMOVE);
      WinStopTimer(hab, tcl->hwndObjectPad, TMR_TIPDELAY);
      tcl->flToolState &= ~TST_TIPACTIVATED;
      if(tcl->lActiveTip != 0)
      { tcl->lActiveTip   = 0;
        // Wyczenie podpowiedzi
        WinShowWindow(tcl->hwndToolTip, FALSE);
      }
    }
    else if(hwndc == tcl->hwndObjectPad)
    { // Kursor znajduje si na pasku pomidzy obiektami
      WinStopTimer(hab, tcl->hwndObjectPad, TMR_TIPDELAY);
      if(tcl->lActiveTip != 0)
      { tcl->lActiveTip = 0;
        // Wyczenie podpowiedzi
        WinShowWindow(tcl->hwndToolTip, FALSE);
      }
    }
    else
    { // Kursor znajduje si nad jednym z obiektw
      if((timer == TMR_TIPDELAY) && !(tcl->flToolState & TST_TIPACTIVATED))
      { tcl->flToolState |= TST_TIPACTIVATED;
        // Wywietlenie tekstu podpowiedzi
        ObjDisplayTip(tcl, tcl->lActiveTip);
      }
    }
  }
}





// Funkcja wywietla podpowied do obiektu
//
// Parametry:
//   tcl       - [parametr] adres gwnej struktury kontrolnej paska narzdzi
//   id        - [parametr] identyfikator obiektu
//
// Powrt:
//   NULL      - nie ma tekstu podpowiedzi
//   <>NULL    - wskanik do cigu ASCIIZ zawierajcego podpowied
//
static VOID ObjDisplayTip(TOOLCTL *tcl, ULONG id)
{ OBJCTL *obj;      // Adres struktury kontrolnej obiketu
  LONG    txtl;     // Dugo odczytywanego tekstu
  PSZ     tiptext;  // Wskanik do obszaru danych zawierajcego tekst podpowiedzi

  // Odczyt adresu obiektu
  obj = FindObjectID(tcl, id);
  if(obj != NULL)
    // Sprawdzenie czy jest tekst podpowiedzi
    if(obj->aToolTip != 0)
    {
      // Odczyt dugoci tekstu podpowiedzi
      txtl = WinQueryAtomLength(tcl->hatomTips, obj->aToolTip);
      if(txtl)
      { // Alokacja pamici dla tekstu
        tiptext = malloc(txtl + 1);
        if(tiptext != NULL)
        { // Odczyt tekstu podpowiedzi
          WinQueryAtomName(tcl->hatomTips, obj->aToolTip, tiptext, txtl + 1);
          tiptext[txtl] = 0;

          // Wywietlenie podpowiedzi
          ToolDisplayTip(tcl->hwndToolBar, obj->hwndObject, tiptext);
          // Zwolnienie pamici
          free(tiptext);
          return;
        }
      }
    }
  // Usunicie podpowiedzi w przypadku niepowodzenia
  WinShowWindow(tcl->hwndToolTip, FALSE);
}





// Funkcja nadaje przezroczystoc dla komunikatw myszy dla okna TCID_OBJECTPAD.
//
// Parametry:
//   hwnd      - [parametr] uchwyt okna TCID_OBJECTPAD
//   cpos      - [parametr] pozycja kursora myszy
//
// Powrt:
//   HT_TRANSPARENT - kursor jest nad oknem TCID_OBJECTPAD
//   HT_NORMAL      - kursor jest nad niezablokowanym oknem
//   HT_ERROR       - kursor jest nad zablokownym obiektem
//
static LONG ObjHittest(HWND hwnd, POINTS *cpos)
{ TOOLCTL *tcl;     // Wskanik do gwnej struktury sterujcej paska narzdzi
  POINTL   pos;     // Pozycja kursora
  HWND     hobj;    // Uchwyt okna nad ktrym jest kursor

  // Odczyt adresu struktury kontrolnej
  tcl = ToolLoadData(hwnd, "ObjHittest");
  if(tcl == NULL) return(HT_NORMAL);

  if(!(tcl->flToolState & TST_HITTEST))
  { // Inicjacja zmiennych
    pos.x = (LONG)((SHORT)cpos->x);
    pos.y = (LONG)((SHORT)cpos->y);

    // Zaznaczenie, e trwa przetwarzanie WinWindowFromPoint
    tcl->flToolState |= TST_HITTEST;
    // Odczyt uchwytu okna nad ktrym jest kursor
    hobj = WinWindowFromPoint(tcl->hwndObjectPad, &pos, FALSE);
    // Usunicie znacznika
    tcl->flToolState &= ~TST_HITTEST;

    // Kursor jest nad oknem TCID_OBJECTPAD
    if(hobj == hwnd) return(HT_TRANSPARENT);
    // Kursor jest nad jednym z okien potomnych
    if(hobj != NULLHANDLE)
    { // Sprawdzenie czy obiekt jest zablokowany
      if(WinIsWindowEnabled(hobj) == FALSE) return(HT_ERROR);
      else                                  return(HT_NORMAL);
    }
  }
  // Sytuacje nieprzewidziane
  return(HT_NORMAL);
}





// Pozycjonowanie okien potomnych
//
// Parametry:
//   hwnd      - [parametr] uchwyt okna przechowujcego obiekty
//
static VOID ObjPlaceObjects(HWND hwnd)
{ TOOLCTL *tcl;     // Wskanik do gwnej struktury kontrolnej paska narzdzi
  OBJCTL  *objtab;  // Wskznik do pocztku tablicy obiektw
  RECTL    area;    // Obszar okna
  LONG     length;  // Minimalna dugo czci uytkowej paska narzdzi
  LONG i;           // Zmienne pomocnicze

  tcl = ToolLoadData(hwnd, "ObjPlaceObjects");
  if(tcl == NULL) return;
  // Inicjacja struktury TOOLADJ
  objtab = tcl->objtab;
  // Odczyt wymiarw okna
  WinQueryWindowRect(hwnd, &area);

  // Odczyt minimalnej dugoci uytkowej czci paska narzdzi
  // Funkcje automatycznie uaktualniaj te wymiary obiektw w tablicach
  if(tcl->flToolState & TST_ROTATED)
  { length = ObjQueryHeight(tcl, tcl->flToolState);
    i = area.yTop - area.yBottom;
  }
  else
  { length = ObjQueryWidth(tcl, tcl->flToolState);
    i = area.xRight - area.xLeft;
  }

  // Sprawdzenie czy obiekty mieszcz si w obszarze *area
  if(length > i)
  { // Standardowe pozycjonowanie niemieszczcych si obiektw - obcicie
    PlaceEndcut(tcl, &area);
  }
  else
    // Standardowe rozmieszczenie elementw (wszystko si mieci)
    PlaceStandard(tcl, &area);

  // Sprawdzenie czy pasek nie jest zminimalizowany
  if(!(tcl->flToolState & TST_MINIMIZED))
  { // Pozycjonowanie okien na podstawie danych zapisanych w tablicach
    for(i = 0; i < tcl->nObjects; ++i)
      // PD: Hack for WC_SPINBUTTON
      if((objtab[i].flAttribute & TBO_BASICTYPES) != TBO_SEPARATOR)
      { CHAR achClass[64];
        ULONG lRet;

        memset(&achClass,0,sizeof(achClass));
        // Zapytanie jak klas jest obiekt
        lRet=WinQueryClassName(objtab[i].hwndObject, sizeof(achClass), &achClass);
#if 0
        if (!strcmp(achClass,"#32"))  // SPINBUTTON
        {  ULONG ulTemp=(area.yTop - area.yBottom)/2 - 10;
           WinSetWindowPos(objtab[i].hwndObject, HWND_TOP,
                           objtab[i].x,  ulTemp,
                           tcl->flToolState & TST_ROTATED ? area.xRight - area.xLeft : objtab[i].cx,
                           tcl->flToolState & TST_ROTATED ? objtab[i].cy : area.yTop - area.yBottom,
                           SWP_MOVE | SWP_SIZE | SWP_ZORDER | SWP_NOADJUST);
        
        }
        else
#endif
        {

          WinSetWindowPos(objtab[i].hwndObject, HWND_TOP,
                          objtab[i].x,  objtab[i].y,
                          tcl->flToolState & TST_ROTATED ? area.xRight - area.xLeft : objtab[i].cx,
                          tcl->flToolState & TST_ROTATED ? objtab[i].cy : area.yTop - area.yBottom,
                          SWP_MOVE | SWP_SIZE | SWP_ZORDER | SWP_NOADJUST);
       }
     }
  }
}





// Funkcja przerysowuje obiekty (separatory) umieszczone na pasku narzdzi,
// oraz przerwy midzy obiektami zdefiniowane jako CXY_OBJECTSPACE
//
// Parametry:
//   hwnd      - [parametr] uchwyt okna przechowujcego obiekty paska narzdzi
//
static VOID ObjRedrawObjects(HWND hwnd)
{ TOOLCTL *tcl;     // Wskanik do gwnej struktury kontrolnej paska narzdzi
  OBJCTL  *objtab;  // Wskanik do tablicy definicji obiektw
  RECTL    area;    // Wymiary okna
  HPS      hps;     // Uchwyt presentation space
  LONG i;           // Zmienne pomocnicze

  tcl = ToolLoadData(hwnd, "ObjRedrawObjects");
  if(tcl == NULL) return;
  // Obiekty nie s rysowane na zminimalizowanym pasku narzdzi
  if(tcl->flToolState & TST_MINIMIZED) return;

  // Rozpoczcie rysowania
  hps = WinBeginPaint(hwnd, NULL, NULLHANDLE);
  // adowanie tablicy kolorw
  PpmQueryPresColors(tcl->hwndToolBar, TOOL_MAXCOLOR, PPmColor, tcl->colors);
  // Tworzenie logicznej tablicy kolorw
  GpiCreateLogColorTable(hps, 0L, LCOLF_CONSECRGB, 0L, TOOL_MAXCOLOR, tcl->colors);
  // Odczyt wymiarw paska narzdzi
  WinQueryWindowRect(hwnd, &area);

  // Kasowanie ta
  GpiSetColor(hps, TOOL_BACKGROUND /*TOOL_TIPBACKGROUND*/);
  GpiMove(hps, (POINTL *)&area);
  GpiBox(hps, DRO_FILL, ((POINTL *)&area) + 1, 0, 0);

  // Inicjacja zmiennych
  objtab = tcl->objtab;
  // Rysowanie separatorw
  for(i = 0; i < tcl->nObjects; ++i)
    if(!(objtab[i].flAttribute & TBO_HIDE))
      if((objtab[i].flAttribute & TBO_BASICTYPES) == TBO_SEPARATOR)
        // Rysowanie separatora
        DrawSeparator(hps, objtab + i, &area, tcl->flToolState);

  // Koniec rysowania
  WinEndPaint(hps);
}





/******************************/
/* Dodatkowe funkcje usugowe */
/******************************/

// Funkcja oblicza pocztkowe rozmiary przyciskw WC_WINBUTTON
// oraz separatorw i umieszcza je w polach cx i cy tablicy struktur OBJCTL.
//
// Parametry:
//   tcl       - [parametr/rezultat] wskanik do struktur kontrolnych paska narzdzi
//
static VOID CalcObjectSize(TOOLCTL *tcl)
{ OBJCTL *objtab;   // Wskanik do pocztku tablicy definicji obiketw
  LONG    cx, cy;   // Obliczone wymiary przycisku klasy WC_WINBUTTON
  LONG i, tmp;      // Zmienne pomocnicze

  // Inicjacja zmiennych
  objtab = tcl->objtab;
  cx = 0; cy = 0;

  // Obliczanie wymiarw
  for(i = 0; i < tcl->nObjects; ++i)
    switch(objtab[i].flAttribute & TBO_BASICTYPES)
    { case TBO_BUTTON:        // Obiekt klasy WC_WINBUTTON
        //tmp = (LONG)WinSendMsg(objtab[i].hwndObject, BM_QUERYWIDTH,  0, 0);
        tmp = 25;
        if(cx < tmp) cx = tmp;
        //tmp = (LONG)WinSendMsg(objtab[i].hwndObject, BM_QUERYHEIGHT, 0, 0);
        tmp = 25;
        if(cy < tmp) cy = tmp;
        break;

      case TBO_SEPARATOR:     // Separator
        objtab[i].cx = CXY_SEPARATOR;
        objtab[i].cy = CXY_SEPARATOR;
        break;
    }

  // Aktualizacja wymiarw okien klasy WC_WINBUTTON
  for(i = 0; i < tcl->nObjects; ++i)
    if((objtab[i].flAttribute & TBO_BASICTYPES) == TBO_BUTTON)
    { objtab[i].cx = cx;
      objtab[i].cy = cy;
    }
}





// Funkcja porwnuje dwa obiekty (jest uywana przez funkcj qsort)
// za kryterium przyjmujc pozycj okna (pole iPosition struktur kontrolnych)
//
// Parametry:
//   object1   - [parametr] wskanik do pierwszego obiektu
//   object2   - [parametr] wskanik do drugiego obiektu
//
static int CompareObjects(const void *object1, const void *object2)
{
   if(((OBJCTL *)object1)->iPosition > ((OBJCTL *)object2)->iPosition) return(1);
   if(((OBJCTL *)object1)->iPosition < ((OBJCTL *)object2)->iPosition) return(-1);
   return(0);
}





// Funkcja zastpuje pole iPosition struktury OBJCTL liczb TB_END.
// Pola iPosition pozostaych struktur s tak modyfikowane, aby
// nie powstaa "dziura" po usunitym obiekcie.
// Po odjciu indeksu, tablica struktur kontrolnych jest sortowana.
//
// Parametry:
//   tcl       - [parametr/rezultat] - wskanik do struktur kontrolnych paska narzdzi
//   objidx    - [parametr] - indeks obiektu w tablicy struktur
//
// Powrt:
//   Nowe pooenie obiektu w tabeli struktur
//
static LONG DeletePositionIndex(TOOLCTL *tcl, ULONG objidx)
{ OBJCTL *tmp;
  LONG   i;

  // Odczyt adresu obiektu
  if(objidx >= tcl->nObjects) return(-1);
  tmp = tcl->objtab + objidx;
  // Zapisanie nieprawidowej pozycji TB_END
  tmp->iPosition = TB_END;

  // Aktualizacja pozostaych struktur
  for(i = 0, tmp = tcl->objtab; i < tcl->nObjects; ++i)
    if((tmp[i].iPosition > objidx) && (tmp[i].iPosition != TB_END))
    { // Przesunicie obiektu znajdujcego si za usuwanym o jeden w ty
      tmp[i].iPosition --;
    }

  // Sortowanie tablicy obiektw ze wzgldu na pooenie
  qsort(tcl->objtab, tcl->nObjects, sizeof(OBJCTL), CompareObjects);
  // Nowe pooenie obiektu
  return(tcl->nObjects - 1);
}





// Funkcja oblicza pozycj obieku na podstawie iPosition, aktualizujc jednoczenie
// pooenia pozostaych obiektw. Obliczona pozycja jest zapamitywana
// w strukturze kontrolnej obiektu obj.
// Po obliczeniu nowej pozycji, tablica jest sortowana, a funkcja zwraca
// nowy indeks struktury kontrolnej obiektu
//
// Parametry:
//   tcl       - [parametr/rezultat] wskanik do struktur kontrolnych paska narzdzi
//   objidx    - [parametr/rezultat] indeks dodawanego obiektu
//   iPosition - [parametr] proponowana pozycja nowego obiektu (moe te by TB_END)
//
// Powrt:
//   Nowy indeks obiektu w tablicy struktur, lub -1 w przypadku bdu
//
static LONG InsertPositionIndex(TOOLCTL *tcl, ULONG objidx, ULONG iPosition)
{ OBJCTL *tmp;      // Pomocniczy wskanik do obiektu
  OBJCTL *objtab;   // Wskanik do tablicy obiektw
  LONG   i;         // Zmienne pomocznicze

  // Odczyt adresu obiektu
  if(objidx >= tcl->nObjects) return(-1);
  tmp = tcl->objtab + objidx;
  // Zapisanie nieprawidowej pozycji TB_END
  tmp->iPosition = TB_END;

  // Korekta pooenia iPosition
  if(iPosition >= tcl->nObjects) iPosition = tcl->nObjects - 1;

  // Aktualizacja pooenie pozostaych obiektw
  for(i = 0, objtab = tcl->objtab; i < tcl->nObjects; ++i)
    if((objtab[i].iPosition >= iPosition) && (objtab[i].iPosition != TB_END))
    { // Przesunicie obiektu znajdujcego si za dodawanym o jeden w przd
      objtab[i].iPosition ++;
    }

  // Zapamitanie pozycji nowego obiektu
  tmp->iPosition = iPosition;
  // Sortowanie tablicy obiektw ze wzgldu na pooenie
  qsort(objtab, tcl->nObjects, sizeof(OBJCTL), CompareObjects);
  // Przekazanie nowej pozycji
  return(iPosition);
}





// Funkcja dodaje struktur informacyjn do tablicy definicji obiektw.
// W wyniku wykonania funkcji modyfikowana jest zmienna nObjects zawierajca
// liczb obiektw obecnych na pasku narzdzi.
//
// Parametry:
//   tcl       - [parametr/rezultat] wskanik do gwnej struktury kontrolnej paska narzdzi
//
// Powrt:
//   >= 0 - indeks w tabeli definicji obiektw
//     -1 - bad alokacji pamici
//
static LONG ExpandObjectTable(TOOLCTL *tcl)
{ OBJCTL *tmp;

  if(tcl->objtab != NULL)
  { // Zmiana rozmiaru tablicy
    tmp = realloc(tcl->objtab, (tcl->nObjects + 1) * sizeof(OBJCTL));
    if(tmp != NULL)
    { // Zapamitanie nowego wskanika
      tcl->objtab = tmp;
      // Obliczenie adresu nowozaalokowanej struktury
      tmp = tcl->objtab + tcl->nObjects;
      // Zerowanie pamici
      memset(tmp, 0, sizeof(OBJCTL));
      // Zapamitanie nowej liczby struktur
      tcl->nObjects ++;
      // Przekazanie indeksu obiektu
      return(tcl->nObjects - 1);
    }
  }
  else
  { // Pocztkowo jest tylko jeden element
    tcl->nObjects = 1;
    // alokacja tablicy
    tcl->objtab = malloc(sizeof(OBJCTL));
    // Zerowanie zaalokowanego obszaru
    memset(tcl->objtab, 0, sizeof(OBJCTL));
    // Przekazanie indeksu
    if(tcl->objtab != NULL) return(0);
  }

  // Bd wykonania
  return(-1);
}





// Funkcja poszukuje obiektu o podanym identyfikatorze.
//
// Parametry:
//   tcl       - [parametr] wskanik do gwnych struktur kontrolnych paska narzdzi
//   id        - [parametr] identyfikator poszukiwanego obiektu
//
// Powrt:
//   Wskanik do struktury kontrolnej odnalezionego obiektu
//   NULL - nie znaleziono obiektu
//
static OBJCTL *FindObjectID(TOOLCTL *tcl, ULONG id)
{ OBJCTL *objtab;   // Wskanik do tablicy definicji obiektw
  LONG i;           // Zmienne pomocnicze

  // Odczyt adresu pocztku tablicy
  objtab = tcl->objtab;

  // Przeszukiwanie tablicy obiektw
  for(i = 0; i < tcl->nObjects; ++i)
  { if(objtab[i].id == id)
      return(objtab + i);
  }
  return(NULL);
}





// Funkcja rozmieszcza obiekty na pasku narzdzi w sytuacji gdy si one tam nie mieszcz.
// Wszystkie obiekty s dosuwane do pocztku paska anrzdzi, a wystajce zostaj obcite.
// Jest uwzgldniana zawarto pola lScrollOffset uywanego przez pasek
// z atrybutem TBS_SCROLLABLE
//
// Parametry:
//   tcl       - [parametr] adres gwnej struktury kontrolnej paska narzdzi
//   area      - [parametr] obszar uytkowy
//
static VOID  PlaceEndcut(TOOLCTL *tcl, RECTL *area)
{ OBJCTL *objtab;   // Wskanik do pocztku tablicy definicji obiektw
  LONG    bpos;     // Pozycja liczona od pocztku paska narzdzi
  LONG    i;        // Zmienne pomocznicze

  // Obliczenie wymiarw przyciskw
  CalcObjectSize(tcl);
  // Inicjacja pozostaych zmiennych
  objtab = tcl->objtab;

  // Pozycja pocztkowa z uwzgldnieniem przesunicia
  if(tcl->flToolState & TST_ROTATED)
    bpos = area->yTop + tcl->lScrollOffset;
  else bpos = area->xLeft - tcl->lScrollOffset;

  // Rozmieszczanie obiektw justowanych od lewej
  for(i = 0; i < tcl->nObjects; ++i)
    if(!(objtab[i].flAttribute & TBO_HIDE))
      if(tcl->flToolState & TST_ROTATED)
      { // Pozycjonowanie obiektw na pionowym pasku narzdzi
        objtab[i].x = area->xLeft;
        objtab[i].y = bpos - objtab[i].cy;
        bpos -= (objtab[i].cy + CXY_OBJECTSPACE);
      }
      else
      { // Pozycjonowanie obiektw na poziomym pasku narzdzi
        objtab[i].x = bpos;
        objtab[i].y = area->yBottom;
        bpos += (objtab[i].cx + CXY_OBJECTSPACE);
      }
}





// Funkcja rozmieszcza obiekty na pasku narzdzi w sytuacji gdy wszystkie si
// tam mieszcz. Rozpoznawane s atrybuty sterujce procesem justowania
// obiektw.
//
// Parametry:
//   tcl       - [parametr] adres gwnej struktury kontrolnej paska narzdzi
//   area      - [parametr] obszar uytkowy
//
static VOID PlaceStandard(TOOLCTL *tcl, RECTL *area)
{ OBJCTL *objtab;   // Wskanik do pocztku tablicy definicji obiektw
  LONG    bpos;     // Pozycja liczona od pocztku paska narzdzi
  LONG    bidx;     // Indeks liczony od pocztku paska narzdzi
  LONG    epos;     // Pozycja liczona od koca paska narzdzi
  LONG    eidx;     // Indeks liczony od koca paska narzdzi
  LONG    i, tmp;   // Zmienne pomocznicze

  // Obliczenie wymiarw przyciskw
  CalcObjectSize(tcl);
  // Inicjacja pozostaych zmiennych
  objtab = tcl->objtab;
  // Pozycja pocztkowa
  bpos = tcl->flToolState & TST_ROTATED ? area->yTop    : area->xLeft;
  bidx = 0;
  epos = tcl->flToolState & TST_ROTATED ? area->yBottom : area->xRight;
  eidx = tcl->nObjects - 1;

  // Rozmieszczanie obiektw justowanych od lewej
  for(i = 0; (i < tcl->nObjects) && ((objtab[i].flAttribute & TBO_JUSTTYPES) == TBO_BEGINALIGN); ++i)
  { if(!(objtab[i].flAttribute & TBO_HIDE))
    { if(tcl->flToolState & TST_ROTATED)
      { // Pozycjonowanie obiektw na pionowym pasku narzdzi
        objtab[i].x = area->xLeft;
        objtab[i].y = bpos - objtab[i].cy;
        bpos -= (objtab[i].cy + CXY_OBJECTSPACE);
      }
      else
      { // Pozycjonowanie obiektw na poziomym pasku narzdzi
        objtab[i].x = bpos;
        objtab[i].y = area->yBottom;
        bpos += (objtab[i].cx + CXY_OBJECTSPACE);
      }
    }
    // Zwikszenie licznika okien umieszczanych od lewej strony
    bidx ++;
  }

  // Rozmieszczenie obiektw justowanych od prawej
  for(i = eidx; (i >= 0) && ((objtab[i].flAttribute & TBO_JUSTTYPES) == TBO_ENDALIGN); --i)
  { if(!(objtab[i].flAttribute & TBO_HIDE))
      if(tcl->flToolState & TST_ROTATED)
      { // Dodanie marginesu lub odstpu midzy obiektami
        if(i != (tcl->nObjects - 1)) epos += CXY_OBJECTSPACE;
        // Pozycjonowanie obiektw na pionowym pasku narzdzi
        objtab[i].x = area->xLeft;
        objtab[i].y = epos;
        epos += objtab[i].cy;
      }
      else
      { // Dodanie marginesu lub odstpu midzy obiektami
        if(i != (tcl->nObjects - 1)) epos -= CXY_OBJECTSPACE;
        // Pozycjonowanie obiektw na poziomym pasku narzdzi
        objtab[i].x = epos - objtab[i].cx;
        objtab[i].y = area->yBottom;
        epos -= objtab[i].cx;
      }
    // Zmniejszenie licznika obiektw umieszczanych od prawej strony
    eidx --;
  }

  // Rozmieszczenie obiektw wycentrowanych
  // Obliczenie cakowitej szerokoci obiektw
  for(i = bidx, tmp = 0; i <= eidx; ++i)
    if(!(objtab[i].flAttribute & TBO_HIDE))
    { tmp += (tcl->flToolState & TST_ROTATED ? objtab[i].cy : objtab[i].cx);
      if(i < eidx) tmp += CXY_OBJECTSPACE;
    }

  // Obliczenie pozycji pocztkowej
  if(tcl->flToolState & TST_ROTATED)
    bpos -= ((bpos - epos - tmp) / 2);
  else bpos += ((epos - bpos - tmp) / 2);

  // Pozycjonowanie obiektw
  for(i = bidx; i <= eidx; ++i)
    if(!(objtab[i].flAttribute & TBO_HIDE))
      if(tcl->flToolState & TST_ROTATED)
      { // Pozycjonowanie obiektw na pionowym pasku narzdzi
        objtab[i].x = area->xLeft;
        objtab[i].y = bpos - objtab[i].cy;
        bpos -= (objtab[i].cy + CXY_OBJECTSPACE);
      }
      else
      { // Pozycjonowanie obiektw na poziomym pasku narzdzi
        objtab[i].x = bpos;
        objtab[i].y = area->yBottom;
        bpos += (objtab[i].cx + CXY_OBJECTSPACE);
      }
}





// Funkcja usuwa obiekt z tablicy danych.
//
// Parametry:
//   tcl       - [parametr/rezultat] wskanik do gwnej struktury kontrolnej okna
//   objidx    - [parametr] numer usuwanego obiektu
//
static VOID ShrinkObjectTable(TOOLCTL *tcl, ULONG objidx)
{ OBJCTL *tmp;

  // Sprawdzenie czy indeks obiektu jest poprawny
  if(tcl->nObjects == 0) return;
  if(objidx >= tcl->nObjects) return;

  // Przesunicie obiektw znajdujcych si za usuwanym
  if(objidx < (tcl->nObjects - 1))
    memmove(tcl->objtab + objidx, tcl->objtab + objidx + 1, tcl->nObjects - objidx - 1);
  // Zmniejszenie liczby obiektw
  tcl->nObjects --;

  // Ponowna alokacja pamici
  tmp = realloc(tcl->objtab, tcl->nObjects * sizeof(OBJCTL));
  if(tmp != NULL) tcl->objtab = tmp;
}





// Funkcja uaktualnia sposb wyrwnywania obiektw ssiadujcych z obiektem
// dodawanym do paska narzdzi.
// Jeeli nowy obiekt posiada styl TBO_BEGINALIGN, to wszystkie obiekty po lewej (w gr)
// uzyskuj styl TBO_BEGINALIGN.
// Jeeli nowy obiekt posiada styl TBO_ENDALIGN, to wszystkie obiekty po prawej (w d)
// uzyskuj styl TBO_ENDALIGN.
// Jeeli obiekt posiada styl TBO_CENTERALIGN, to wszystkie obiekty po lewej (u gry)
// posiadajce styl TBO_ENDALIGN uzyskuj TBO_CENTERALIGN, oraz wszystkie obiekty po
// prawej (u dou) posiadajce styl TB_BEGINALIGN uzyskuj TBO_CENTERALIGN.
//
// Parametry:
//   tcl        - [parametr/rezultat] adres gwnej struktury kontrolnej paska narzdzi
//   objidx     - [parametr] indeks dodawanego obiektu
//   flNewAttrs - [parametr] atrybuty nowego obiektu
//
static VOID UpdateAlignment(TOOLCTL *tcl, LONG objidx, ULONG flNewAttrs)
{ OBJCTL *objtab;   // Adres pocztku tablicy definicji obiektw
  LONG i;           // Zmienne pomocnicze

  // Inicjacja zmiennych
  objtab = tcl->objtab;
  // Usunicie pozostaych trybw justowania
  flNewAttrs &= TBO_JUSTTYPES;

  switch(flNewAttrs)
  { case TBO_BEGINALIGN:      // Korekta po dodaniu obiektu justowanego od lewej
      for(i = objidx; i >= 0; --i)
      { objtab[i].flAttribute &= ~TBO_JUSTTYPES;
        objtab[i].flAttribute |=  TBO_BEGINALIGN;
      }
      break;

    case TBO_ENDALIGN:        // Korekta po dodaniu obiektu justowanego do prawej
      for(i = objidx; i < tcl->nObjects; ++i)
      { objtab[i].flAttribute &= ~TBO_JUSTTYPES;
        objtab[i].flAttribute |=  TBO_ENDALIGN;
      }
      break;

    case TBO_CENTERALIGN:     // Korekta po dodaniu obiektu centrowanego
      // Sprawdzenie obiektw po lewej
      for(i = objidx; i >= 0; --i)
        if((objtab[i].flAttribute & TBO_JUSTTYPES) != TBO_BEGINALIGN)
        { objtab[i].flAttribute &= ~TBO_JUSTTYPES;
          objtab[i].flAttribute |=  TBO_CENTERALIGN;
        }
      // Sprawdzenie obiektw po prawej
      for(i = objidx; i < tcl->nObjects; ++i)
        if((objtab[i].flAttribute & TBO_JUSTTYPES) != TBO_ENDALIGN)
        { objtab[i].flAttribute &= ~TBO_JUSTTYPES;
          objtab[i].flAttribute |=  TBO_CENTERALIGN;
        }
      break;
  }
}





// Funkcja uaktualnia atrybuty w strukturach wszystkich obiektw nalecych do
// podanej grupy. Uaktualniane jest pola flAttribute struktur OBJCTL,
// ale tylko wtedy gdy zosta uyty ktry z trybw:
// TBO_PUSHBUTTON, TBO_RADIOBUTTON lub TBO_CHECKBOX.
// W przeciwnym wypadku style pozostaj bez zmian.
//
// Parametry:
//   tcl        - [parametr/rezultat] wskanik do gwnej struktury kontrolnej paska narzdzi
//   ulGroup    - [parametr] grupa, dla ktrej s modyfikowane parametry
//   flNewAttrs - [parametr] nowy zestaw atrybutw
//
static VOID UpdateGroupStyle(TOOLCTL *tcl, ULONG ulGroup, ULONG flNewAttrs)
{ OBJCTL *objtab;   // Adres pocztku tablicy definicji obiektw
  LONG i;           // Zmienne pomocnicze

  // Inicjacja zmiennych
  objtab = tcl->objtab;

  // Sprawdzenie czy parametry grupy s modyfikowane
  if(!(flNewAttrs & TBO_RADIOBUTTON))
  {
    // Jeli nie, to poszukiwanie domylnego parametru grupy
    for(i = 0; i < tcl->nObjects; ++i)
      if(objtab[i].ulGroup == ulGroup)
        if(objtab[i].flAttribute & TBO_RADIOBUTTON)
        { // Przyjcie parametrw domylnych
          flNewAttrs = objtab[i].flAttribute & TBO_RADIOBUTTON;
          break;
        }
  }

  // Przeszukiwanie tablicy obiektw i korekta parametrw grupy
  for(i = 0; i < tcl->nObjects; ++i)
    if(objtab[i].ulGroup == ulGroup)
    { // Aktualizacja parametrw grupy
      // Kasowanie zastanego stlu grupy
      objtab[i].flAttribute &= ~TBO_RADIOBUTTON;
      // Dodanie nowego stylu
      objtab[i].flAttribute |= (flNewAttrs & TBO_RADIOBUTTON);
    }
}





// Funkcja rysuje na pasku narzdzi separator.
// Zmienna flState okrela pooenie paska narzdzi, bit: TST_ROTATED.
// Zmienna objAttrs okrea wygld separatora, stae SPS_*.
//
// Parametry:
//   hps       - [parametr] uchwyt presentation space okna
//   obj       - [parametr] wskanik do struktury definiujcej obiekt
//   area      - [parametr] wskanik do obszaru przeznaczonego na obiekty
//   flState   - [parametr] stan paska narzdzi
//
static VOID DrawSeparator(HPS hps, OBJCTL *obj, RECTL *area, ULONG flState)
{ POINTL pos;       // Wsprzdne oglnego przeznaczenia
  RECTL  line;      // Wsprzdne rysowanej lini
  ULONG  objAttr;   // Atrybuty separatora

  // Okrelenie pozycji i wymiarw separatora
  if(flState & TST_ROTATED)
  { line.xLeft   = area->xLeft;
    line.yBottom = obj->y;
    line.xRight  = area->xRight - 1;
    line.yTop    = obj->y + obj->cy - 1;
  }
  else
  { line.xLeft   = obj->x;
    line.yBottom = area->yBottom;
    line.xRight  = obj->x + obj->cx - 1;
    line.yTop    = area->yTop - 1;
  }

  // Rysowanie ta pod separatorem
  GpiSetColor(hps, TOOL_BACKGROUND);
  GpiMove(hps, (POINTL *)&line);
  GpiBox(hps, DRO_FILL, ((POINTL *)&line) + 1, 0, 0);

  // Obliczenie wsprzdnych pocztkowych i kocowych rysowanych linii
  if(flState & TST_ROTATED)
  { line.yBottom += ((obj->cy - 2) / 2);
    line.yTop     = line.yBottom + 1;
  }
  else
  { line.xLeft   += ((obj->cx - 2) / 2);
    line.xRight   = line.xLeft + 1;
  }

  // Okrelenie atrybutw separatora
  objAttr = obj->flAttribute & SPS_SEPARATORS;

  // Rysowanie separatorw rnych typw
  switch(objAttr)
  { case SPS_LINE:       // Paska linia w kolorze PP_BORDERDARKCOLOR
      GpiSetColor(hps, TOOL_FRAMEDARK);
      GpiMove(hps, (POINTL *)&line);
      GpiBox(hps, DRO_FILL, ((POINTL *)&line) + 1, 0, 0);
      break;

    case SPS_CONCAVE:    // Wklsa linia 3D
    case SPS_CONVEX:     // Wypuka linia 3D
      GpiSetColor(hps, objAttr == SPS_CONCAVE ? TOOL_FRAMEDARK : TOOL_FRAMEHILITE );
      pos.x = line.xLeft;  pos.y = line.yBottom;
      GpiMove(hps, &pos);
      pos.y = line.yTop;   GpiLine(hps, &pos);
      pos.x = line.xRight; GpiLine(hps, &pos);

      GpiSetColor(hps, objAttr == SPS_CONCAVE ? TOOL_FRAMEHILITE : TOOL_FRAMEDARK );
      pos.y = flState & TST_ROTATED ? pos.y : pos.y - 1;
      GpiMove(hps, &pos);
      pos.y = line.yBottom; GpiLine(hps, &pos);
      pos.x = flState & TST_ROTATED ? line.xLeft + 1 : line.xLeft;
      GpiLine(hps, &pos);
      break;
  }
}

/*
 * $Log: objmgr.c $
 * Revision 1.1  1999/06/27 12:36:00  Wojciech_Gazda
 * Initial revision
 *
 */
