//----------------------------------------------------------------------------------------------------//
/*
   PHYSICS FOR GAME DEVELOPERS

   CHAPTER 4 EXAMPLE PROGRAM

   NAME:    Particle Explosion
   PURPOSE: To demonstrate 2D particle kinematics
   BY:         David Bourg
   DATE:    04/28/00
   COPYRIGHT:  Copyright 2000 by David Bourg
*/
//----------------------------------------------------------------------------------------------------//
#ifdef WINDOZE
// Windows Header Files:
#include <windows.h>
#include <windef.h>
#include <commctrl.h>
#include <commdlg.h>
#include <wingdi.h>
#else
#define  INCL_BASE
#define  INCL_GPI
#define INCL_WINSWITCHLIST
#define INCL_WINPROGRAMLIST
#define INCL_WINWINDOWMGR
#define INCL_WINDIALOGS
#include <os2.h>
#endif


// C RunTime Header Files:
#include <stdlib.h>
#include <malloc.h>
#include <memory.h>
#include <stdio.h>
#include <math.h>

// Local Header Files:
#include "resource.h"
#include "cannon.h"

// Defines:
#define     MYSIDEVIEW  2000

#ifdef WINDOZE
// Forward declarations for window related functions
LRESULT CALLBACK DemoDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam);
LRESULT CALLBACK DefaultWndProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam);

// Window related global variables
HINSTANCE   hinst;
HWND     hMainWindow;
HDC         hBufferDC;
HBITMAP     hBufferBitmap;


// Forward declarations for non-window related functions
void  InitializeVariables(void);
void  DrawLine(HDC hdc, int h1, int v1, int h2, int v2, int thk, COLORREF clr);
void  DrawRectangle(HDC hdc, RECT *r, int thk, COLORREF clr);
void  DrawCircle(HDC hdc, int cx, int cy, int r, COLORREF clr);
void  DrawString(HDC hdc, int x, int y, LPCSTR lpszString, int size, int ptsz);
#else
HBITMAP     hBufferBitmap;
HPS         hBufferDC;

void  InitializeVariables(void);

static MRESULT EXPENTRY WinMain( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 );
static MRESULT EXPENTRY DemoDlgProc( HWND hDlg, ULONG message, MPARAM wParam, MPARAM lParam );
static void DisplayMessage( HAB, ULONG );
static void InitTitle( HAB hab, HWND hwnd, char *szTitle );
static MRESULT EXPENTRY DefaultWndProc(HWND hwnd, ULONG message, MPARAM wParam, MPARAM lParam);
static char   szAppName[ 255 ];    /* application name string        */
#define MSG_BOX_ID     256                  /* error message box id           */
void  DrawRectangle(HPS hdc, RECTL *r, int thk, RGB clr);
void  DrawSideView(HPS hdc, RECTL *r);
void  DrawLine(HPS hdc, int h1, int v1, int h2, int v2, int thk, RGB clr);
void DrawString(HPS hdc, int x1, int y1, PSZ lpszString, int size, int ptsz);

HAB   hab = NULLHANDLE;                  /* PM anchor block handle         */
HMQ   hmq = NULLHANDLE;                  /* message queue handle           */
HWND  hwndFrame = NULLHANDLE;            /* frame window handle            */
static HWND   hwndClient = NULLHANDLE;      /* client window handle           */
RECTL RectDlgItem;

#endif
   RGB       red = {0,0,255};
   RGB       black = {0,0,0};
   RGB       blue = {255,0, 0};
   RGB       white = {255, 255, 255};
   RGB       green = {0, 255, 0};
#define RGB1(r, g, b) (r << 16 + g << 8 + b)

int      DoSimulation(void);

BOOL  UpdateParticleState(TParticle* p, int dtime);
void     CreateParticleExplosion(int x, int y, int force, int life, float gravity, float angle);
BOOL  DrawParticleExplosion(HDC hdc);
BOOL  DrawParticleExplosion1(HDC hdc);
int tb_Rnd(int min, int max);
int tb_Round(double x);


int main( void )
{
   ULONG flCreate = 0UL;                    /* window creation control flags  */
   QMSG  qmsg;                              /* message from message queue     */
   int   rc = 1;

   do
   {
      /* Initialize PM and create a message queue of default size.            */

      if ( ( hab = WinInitialize( 0UL ) ) == NULLHANDLE )
         break;

      if ( ( hmq = WinCreateMsgQueue( hab, 0UL ) ) == NULLHANDLE )
         break;

      /* Register client window class.                                        */

      if ( !WinRegisterClass( hab,          /* PM anchor block handle         */
                              WINDOW_CLASS, /* window class name              */
                              WinMain, /* address of window procedure    */
                              CS_SIZEREDRAW,/* size changes cause redrawing   */
                              0UL ) )       /* window data                    */
      {
         DisplayMessage( hab, IDS_NOREGISTER );
         break;
      }

      /* Create the standard windows but do not add the application to the    */
      /* task manager list.                                                   */

      flCreate = FCF_STANDARD & ~FCF_TASKLIST & ~FCF_ICON;

      hwndFrame =
         WinCreateStdWindow( HWND_DESKTOP,  /* make desktop window the parent */
                             0L,    /* frame window class style       */
                             &flCreate,     /* frame control flags            */
                             WINDOW_CLASS,  /* client window class name       */
                             "",            /* title bar text                 */
                             CS_SIZEREDRAW, /* client window class style      */
                             0UL,           /* resource file handle - in EXE  */
                             ID_WINDOW,     /* resources identifier           */
                             &hwndClient ); /* client window handle           */
      if ( hwndFrame == NULLHANDLE )
      {
         DisplayMessage( hab, IDS_NOSTDWINDOWS );
         break;
      }

      /* Initialize the window title and task switch list.                    */

      InitTitle( hab, hwndFrame, szAppName );

      /* Create the thread that will draw the lines.                          */
      /* NOTE: _beginthread MUST be used if the thread contains CRT calls     */


      /* While the WM_QUIT message is not received, dispatch the message.     */
      /* When the WM_QUIT message is received, WinGetMsg will return FALSE.   */

      while( WinGetMsg( hab, &qmsg, 0UL, 0UL, 0UL ) )
         WinDispatchMsg( hab,               /* PM anchor block handle         */
                         &qmsg );           /* pointer to message             */
      rc = 0;

   }  while ( FALSE );

   /* Destroy the standard windows if they were created.                      */

   if ( hwndFrame != NULLHANDLE )
      WinDestroyWindow( hwndFrame );        /* frame window handle            */

   /* Destroy the message queue and release the anchor block.                 */

   if ( hmq != NULLHANDLE )
      WinDestroyMsgQueue( hmq );

   if ( hab != NULLHANDLE )
      WinTerminate( hab );

   return rc;
}

/*+--------------------------------------------------------------------------+*/
/*| DisplayMessage - display an error message in a message box.              |*/
/*+--------------------------------------------------------------------------+*/

static void DisplayMessage( HAB hab, ULONG ulStringNum )
{
   char szTemp[ 255 ];

   WinLoadString( hab, 0UL, ulStringNum, 255, szTemp );

   WinAlarm( HWND_DESKTOP,                  /* desktop window handle          */
             WA_ERROR );                    /* type of alarm                  */

   WinMessageBox( HWND_DESKTOP,             /* parent window handle           */
                  HWND_DESKTOP,             /* owner window handle            */
                  szTemp,                   /* pointer to message text        */
                  szAppName,                /* pointer to title text          */
                  MSG_BOX_ID,               /* message box identifier         */
                  MB_OK | MB_ERROR |        /* message box style              */
                  MB_SYSTEMMODAL );

   return;
}

/*+--------------------------------------------------------------------------+*/
/*| InitTitle - Initializes window title and task switch list.               |*/
/*+--------------------------------------------------------------------------+*/

static void InitTitle( HAB hab, HWND hwnd, char *szTitle )
{
   SWCNTRL tSwcntrl;                        /* task switch structure          */

   /* Load the application name from the resources in the EXE.  Set the title */
   /* bar text to this name.  Then add this application to the task manager   */
   /* list with the loaded application name.                                  */

   WinLoadString( hab, 0UL, IDS_APPNAME, MAXNAMEL, szTitle );
   WinSetWindowText( hwnd, szTitle );

   tSwcntrl.hwnd = hwnd;
   tSwcntrl.hwndIcon = NULLHANDLE;
   tSwcntrl.hprog = NULLHANDLE;
   tSwcntrl.idProcess = 0;
   tSwcntrl.idSession = 0;
   tSwcntrl.uchVisibility = SWL_VISIBLE;
   tSwcntrl.fbJump = SWL_JUMPABLE;
   strcpy( tSwcntrl.szSwtitle, szTitle );
   tSwcntrl.bProgType = PROG_PM;
   WinAddSwitchEntry( &tSwcntrl );

   return;
}

//----------------------------------------------------------------------------------------------------//
// This is the applications "main" function. Note that I'm not using a message loop here
// since there is no main window. All I do is display a dialog box immediately upon startup
// and let the dialog handler take care of the messages.
//----------------------------------------------------------------------------------------------------//
static MRESULT EXPENTRY WinMain( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
//----------------------------------------------------------------------------------------------------//
{
   HPS   hps = NULLHANDLE;                  /* presentation space handle      */


   switch( msg )
   {
      case WM_CREATE:
         /* The client window has been created but is not visible yet.        */
         /* Initialize the window here.                                       */
         WinDlgBox(HWND_DESKTOP, hwnd, DemoDlgProc, 0UL, IDD_DEMODIALOG, NULL);
         WinPostMsg( hwnd, WM_QUIT, NULL, NULL );
         break;

      case WM_CLOSE:
         /* Tell the main procedure to terminate the message loop.            */

         WinPostMsg( hwnd, WM_QUIT, NULL, NULL );
         break;

      case WM_PAINT:
      {
         RECTL rClient;
         RECTL rcl;

         hps = WinBeginPaint( hwnd, NULLHANDLE, &rcl );

         WinFillRect(hps, &rcl, CLR_WHITE);

         WinEndPaint( hps );
         break;
      }

      default:
         /* For all other messages, let the default window procedure          */
         /* process them.                                                     */

         return ( WinDefWindowProc( hwnd, msg, mp1, mp2 ) );
   }
   return NULL;

}

//----------------------------------------------------------------------------------------------------//
// This is the message handler function for the main dialog box.  It is responsible for handling
// any interaction with it's controls and for updating the trajectory views that we'll be displaying.
//----------------------------------------------------------------------------------------------------//
static MRESULT EXPENTRY DemoDlgProc( HWND hDlg, ULONG message, MPARAM wParam, MPARAM lParam )
//----------------------------------------------------------------------------------------------------//
{
   static   HWND  hTopView;
   static   HWND  hSideView;
/* WNDCLASSEX     wclass;  */
   char        str[16];
   BOOL           status;
   RECTL        r;
   APIRET      ret;
/* HDC            hdc;  */
   HPS         hps;
   BITMAPINFOHEADER2 bmih;  /* bit map info structure               */


   switch (message) {
      // Initialize the dialog box here:
      case WM_INITDLG:
      {
         RECTL RectDlg;
         SWP swpPos;
         SWP swpDlg;
         HENUM  henum;           /* enumeration handle                   */
         HWND hwndChild;
         HWND hwndItem;
         BOOL fSuccess;
         CHAR szClass[50];
         double dOffset;



         // setup a child window class so we can create a couple
         // of windows to draw the two trajectory views on

         WinQueryWindowPos(hDlg, &swpDlg);
         WinQueryWindowRect(hDlg, &RectDlg);

         /* #9 = Title Bar
            #6 = Entry Fields
            #5 = LText
            #3 = Pushbuttons */


         /* move dialog items relative to bottom of dialog instead of
            dumb windoze relative to the top of the dialog this way
            you and use the windoze generated dialog boxes with out
            modification! */
         henum = WinBeginEnumWindows(hDlg);
         while ((hwndChild = WinGetNextWindow(henum)) != NULL)
         {
            WinQueryClassName(hwndChild, sizeof(szClass), szClass);
            WinQueryWindowPos(hwndChild, &swpPos);

            /* subtract the Title Bar */
            if (strcmp(szClass,"#9") == 0)
               swpDlg.cy -= swpPos.cy;

            swpPos.y = swpDlg.cy - (swpPos.y + swpPos.cy);

            if (strcmp(szClass,"#9") != 0) /* Don't move the Title Bar */
               WinSetWindowPos(hwndChild, swpPos.hwndInsertBehind, swpPos.x, swpPos.y,
                              swpPos.cx, swpPos.cy, SWP_MOVE | SWP_SHOW | SWP_SIZE);
         }

         fSuccess = WinEndEnumWindows(henum);

         WinRegisterClass( hab, "ViewClass", DefaultWndProc, CS_SIZEREDRAW, 16UL );
         RectDlgItem.xLeft = (LONG)((double)swpDlg.cx * 0.225);
         RectDlgItem.xRight = (swpDlg.cx - 10) - (RectDlgItem.xLeft);
         RectDlgItem.yBottom = 20;
         RectDlgItem.yTop = swpDlg.cy - 40;
         // Now go ahead and create a child window for the side view
         hSideView = WinCreateWindow(hDlg, "ViewClass","Status", WS_VISIBLE,
                     RectDlgItem.xLeft, RectDlgItem.yBottom, RectDlgItem.xRight, RectDlgItem.yTop, hDlg, HWND_TOP,150, 0, 0 );
/*                     RectDlgItem.xLeft, 10+10, 400, 400, hDlg, HWND_TOP,150, 0, 0 );  */
         // Tag the window with our identifier, MYSIDEVIEW, so that we
         // can distinguish it from the top view window when it's time
         // to draw in the window.
         ret = WinSetWindowULong(hSideView, QWL_USER, MYSIDEVIEW);


         // Now initialize all of the edit controls on the dialog box with our
         // default values for each variable.

         // Set default values for all variables
         InitializeVariables();

         // Now convert each variable value to a string and
         // set the appropriate edit control
         sprintf( str, "%d", xc );
         WinSetDlgItemText(hDlg, IDC_X, str);

         sprintf( str, "%d", yc );
         WinSetDlgItemText(hDlg, IDC_Y, str);

         sprintf( str, "%d", V0 );
         WinSetDlgItemText(hDlg, IDC_V0, str);

         sprintf( str, "%d", Duration );
         WinSetDlgItemText(hDlg, IDC_DURATION, str);

         sprintf( str, "%f", Gravity );
         WinSetDlgItemText(hDlg, IDC_GRAVITY, str);

         sprintf( str, "%f", Angle );
         WinSetDlgItemText(hDlg, IDC_ANGLE, str);

         // create a buffer to draw on:
/*       hdc = GetDC(hSideView);
         hBufferDC = CreateCompatibleDC(hdc);
         hBufferBitmap = CreateCompatibleBitmap(hdc, 400, 400);
         ReleaseDC(hSideView, hdc);  */

         //hBufferBitmap = CreateBitmap(500, 200, 1, 24, NULL);


/*         SelectObject(hBufferDC, hBufferBitmap);  */


         hBufferDC = WinGetPS(hSideView);
         memset (&bmih,0, sizeof(BITMAPINFOHEADER2));
         bmih.cbFix        = sizeof(BITMAPINFOHEADER2);
         bmih.cx           = 500;
         bmih.cy           = 200;
         bmih.cPlanes      = 1;
         bmih.cBitCount = 1;
         hBufferBitmap = GpiCreateBitmap(hBufferDC, &bmih, 0L, NULL, NULL);
         WinReleasePS(hSideView);

         WinPostMsg(hDlg, WM_USER, 0L, 0L); /* post a message to clear the initial window */
         break;
      }

      case WM_USER:
      {

         r.xLeft = 0;
         r.xRight = RectDlgItem.xRight;
         r.yTop = 0;
         r.yBottom = RectDlgItem.yTop;
         DrawRectangle(hBufferDC, &r, 1, black);

         break;
      }


      // handle the dialog controls here:
      case WM_COMMAND:
         switch( SHORT1FROMMP( wParam) )
         {
            case IDC_RANDOM:
            case IDC_FIRE:
               // update the variables with
               // the values shown in the edit controls

               if(SHORT1FROMMP(wParam) == IDC_FIRE)
               {
                  WinQueryDlgItemText(hDlg, IDC_X, 15, str);
                  xc = atoi(str);

                  WinQueryDlgItemText(hDlg, IDC_Y, 15, str);
                  yc = atoi(str);

                  WinQueryDlgItemText(hDlg, IDC_V0, 15, str);
                  V0 = atoi(str);

                  WinQueryDlgItemText(hDlg, IDC_DURATION, 15, str);
                  Duration = atoi(str);

                  WinQueryDlgItemText(hDlg, IDC_GRAVITY, 15, str);
                  Gravity = atof(str);

                  WinQueryDlgItemText(hDlg, IDC_ANGLE, 15, str);
                  Angle = atof(str);
               } else {
/*                  xc = tb_Rnd(100, 300);  */
                  xc = tb_Rnd(100, RectDlgItem.xRight - 100);
/*                  yc = tb_Rnd(100, 300);  */
                  yc = tb_Rnd(100, RectDlgItem.yTop - 100);
                  V0 = tb_Rnd(10, 400);
                  Duration = tb_Rnd(2000, 6000);
                  Gravity = tb_Rnd(0, 100);
                  Angle = 999;

                  sprintf( str, "%d", xc );
                  WinSetDlgItemText(hDlg, IDC_X, str);

                  sprintf( str, "%d", yc );
                  WinSetDlgItemText(hDlg, IDC_Y, str);

                  sprintf( str, "%d", V0 );
                  WinSetDlgItemText(hDlg, IDC_V0, str);

                  sprintf( str, "%d", Duration );
                  WinSetDlgItemText(hDlg, IDC_DURATION, str);

                  sprintf( str, "%f", Gravity );
                  WinSetDlgItemText(hDlg, IDC_GRAVITY, str);

                  sprintf( str, "%f", Angle );
                  WinSetDlgItemText(hDlg, IDC_ANGLE, str);
               }

               CreateParticleExplosion(xc, yc, V0, Duration, Gravity, Angle);
               r.xLeft = 0;
               r.xRight = 400;
               r.xRight = RectDlgItem.xRight;
               r.yTop = 0;
               r.yBottom = 400;
               r.yBottom = RectDlgItem.yTop;

               // initialize the time and status variables
               status = TRUE;

               // start stepping through time for the sim.
               // until the target is hit, the shell hits
               // the ground, or the sim. times out.
               DrawRectangle(hBufferDC, &r, 1, black);
               status = DrawParticleExplosion(hBufferDC);
               while(status)
               {
                  status = DrawParticleExplosion1(hBufferDC);


/*                hdc = GetDC(hSideView);
                  //status = DrawParticleExplosion(hdc);
                  if(!BitBlt( hdc,
                        0,
                        0,
                        r.right,
                        r.bottom,
                        hBufferDC,
                        0,
                        0,
                        SRCCOPY))
                  {
                        MessageBox(NULL, "BitBlt failed", "Error", MB_OK);
                        status = FALSE;
                  }

                  ReleaseDC(hSideView, hdc);  */
               }
               break;

            case ID_CLOSE:
               // clean up the child windows and close the dialog box
/*             DeleteObject(hBufferBitmap);
               DeleteDC(hBufferDC);
               DestroyWindow(hSideView);
               EndDialog(hDlg, 1);  */
               WinDestroyWindow( hSideView );
               WinDismissDlg( hDlg, FALSE );
               break;

            case DID_CANCEL:
               // clean up the child windows and close the dialog box
/*             DeleteObject(hBufferBitmap);
               DeleteDC(hBufferDC);
               DestroyWindow(hSideView);
               EndDialog(hDlg, 0);  */
               WinDestroyWindow( hSideView );
               WinDismissDlg( hDlg, FALSE );
               break;
         }
         break;

      default:
         return WinDefDlgProc( hDlg, message, wParam, lParam );
   }
    return ( TRUE );
}

//----------------------------------------------------------------------------------------------------//
// This is the default message processing function for the two child windows that are created
// so to draw the trajectory views.
//----------------------------------------------------------------------------------------------------//
static MRESULT EXPENTRY DefaultWndProc(HWND hwnd, ULONG message, MPARAM wParam, MPARAM lParam)
//----------------------------------------------------------------------------------------------------//
{
/* PAINTSTRUCT       ps;
   HDC               hdc;
   RECT           r;
   LONG           tag;
*/
   switch (message) {
/*    case WM_PAINT:
         hdc = BeginPaint(hwnd, &ps);

         // Get the tag that we set when we created the child windows
         // where we plan to draw the shell trajectory
         tag = GetWindowLong(hwnd, GWL_USERDATA);

         if(tag == MYSIDEVIEW)  // We need to draw the side view
         {
            GetClientRect(hwnd, &r);
            DrawSideView(hdc, &r);
         }

         EndPaint(hwnd, &ps);
         break;*/
   }
    return WinDefWindowProc(hwnd, message, wParam, lParam);
}

//----------------------------------------------------------------------------------------------------//
// Initialize the global variables required for the simulation.
//----------------------------------------------------------------------------------------------------//
void  InitializeVariables(void)
//----------------------------------------------------------------------------------------------------//
{
   xc = 300;
   yc = 500;
   V0 = 175;
   Duration = 4000;
   Gravity = 30.0f;
   Angle = 999;
}


//----------------------------------------------------------------------------------------------------//
// This function simply draws a solid line to the given device context, given the line
// start and end point, its thickness and its color.
//----------------------------------------------------------------------------------------------------//
void DrawLine(HPS hdc, int h1, int v1, int h2, int v2, int thk, RGB clr)
//----------------------------------------------------------------------------------------------------//
{
   POINTL ptl;
   LINEBUNDLE lb;
   ULONG *ulp;
   ULONG lp;
   APIRET rc;
   ULONG *ulClr;
   ULONG lClr;

   GpiCreateLogColorTable ( hdc, LCOL_RESET, LCOLF_RGB, 0, 0, 0 ) ;
   ulClr = (ULONG *)&clr;
   lClr = *ulClr & 0xFFFFFF;
   GpiSetColor(hdc, lClr);
   GpiSetLineWidth(hdc, thk);
   GpiSetLineType(hdc, LINETYPE_SOLID);

   ptl.x = h1;
   ptl.y = v1;
   GpiMove(hdc, &ptl);

   ptl.x = h2;
   ptl.y = v2;
   GpiLine(hdc, &ptl);

}

//----------------------------------------------------------------------------------------------------//
// This function simply draws a filled rectangle to the given device context, given the
// rectangle dimensions, its border thickness and its border color (the rectangle is filled
// in black).
//----------------------------------------------------------------------------------------------------//
void DrawRectangle(HPS hps, RECTL *r, int thk, RGB clr)
{
   POINTL pt;
   ULONG *ulClr;
   ULONG lClr;
   RGB FColor = clr;
   RGB BColor = {0, 0, 0};

   pt.x = r->xLeft;
   pt.y = r->yBottom;
   GpiMove(hps, &pt);
   GpiSetColor(hps, CLR_BLACK);
   pt.x = r->xRight;
   pt.y = r->yTop;
   GpiBox(hps, DRO_OUTLINEFILL, &pt , 0 , 0);

   pt.x = r->xLeft;
   pt.y = r->yBottom;
   GpiMove(hps, &pt);
   GpiCreateLogColorTable ( hps, LCOL_RESET, LCOLF_RGB, 0, 0, 0 ) ;
   ulClr = (ULONG *)&clr;
   lClr = *ulClr & 0xFFFFFF;
   GpiSetColor(hps, lClr);
   pt.x = r->xRight;
   pt.y = r->yTop;
   GpiBox(hps, DRO_OUTLINE, &pt, 0, 0);

}

//----------------------------------------------------------------------------------------------------//
// This function simply draws a filled circle to the given device context
//----------------------------------------------------------------------------------------------------//
void DrawCircle(HDC hdc, int cx, int cy, int r, RGB clr)
{
   ARCPARAMS arcp;
   /* = { 1, 1, 0, 0 };  /* Arc parameters.          */
   POINTL ptl;
   /* = {100, 100}; */
   ULONG *ulClr;
   ULONG lClr;
/* HBRUSH      CurrentBrush;
   HBRUSH      OldBrush;
   HPEN     CurrentPen;
   HPEN     OldPen;
   COLORREF FColor = clr;
   COLORREF BColor = clr;

   CurrentBrush = CreateSolidBrush(BColor);
   OldBrush = (HBRUSH) SelectObject( hdc, CurrentBrush);
   CurrentPen = CreatePen(PS_SOLID, 1, FColor);
   OldPen = (HPEN) SelectObject(hdc, CurrentPen); */

   arcp.lP = r;
   arcp.lQ = r;
   arcp.lR = 0;
   arcp.lS = 0;
   ptl.x = cx;
   ptl.y = cy;

   ulClr = (ULONG *)&clr;
   lClr = *ulClr & 0xFFFFFF;
   GpiSetColor(hdc, lClr);
   GpiSetArcParams(hdc, &arcp);
   GpiMove(hdc, &ptl);
   if (GpiFullArc(hdc, DRO_OUTLINEFILL, MAKEFIXED(1, 0)) == GPI_OK)
      ptl.x = 0;



/*   Ellipse(hdc, cx - r, cy - r, cx + r, cy + r);

   SelectObject(hdc, OldBrush);
   SelectObject(hdc, OldPen);
   DeleteObject(CurrentBrush);
   DeleteObject(CurrentPen);  */
}


//----------------------------------------------------------------------------------------------------//
// This function simply draws text to the given device context, given the text string
// and the x,y coordinates of its lower left corner, the number of characters in the string,
// and the desired point size.
//----------------------------------------------------------------------------------------------------//
void DrawString(HPS hdc, int x1, int y1, PSZ lpszString, int size, int ptsz)
{

#define NONE 0
#define RAISED 1
#define INDENTED 2
#define GRAYEDOUT 3
#define USE_OVERPAINT 1
#define USE_DEFAULT     0
   RECTL rClient;
   SHORT sShift;
   SHORT x;
   SHORT sFeature = RAISED;
   SHORT sMix = USE_OVERPAINT;
   ULONG ulAttrs = DT_LEFT | DT_BOTTOM | DT_TEXTATTRS;

   rClient.xLeft = x1;
   rClient.yBottom = y1;
   rClient.xRight = rClient.xLeft + 100;
   rClient.yTop = rClient.yBottom + 50;

   GpiCreateLogColorTable ( hdc, LCOL_RESET, LCOLF_INDRGB, 0, 0, 0 ) ;
   if (sFeature != NONE)
   {

      switch (sFeature)
      {
         case RAISED:
            GpiSetColor(hdc,CLR_WHITE);
            sShift = 1;
            break;
         case INDENTED:
            GpiSetColor(hdc,CLR_DARKGRAY);
            sShift = 1;
            break;
         case GRAYEDOUT:
            GpiSetColor(hdc,CLR_PALEGRAY);
            sShift = 1;
            break;
      }
      GpiSetBackColor(hdc,CLR_BLACK);
      if (sMix == USE_OVERPAINT)
         GpiSetBackMix(hdc,BM_OVERPAINT);

   /*   rClient.yTop = 50; */
      rClient.xLeft -= sShift;
      rClient.yBottom += sShift;
      WinDrawText(hdc,strlen(lpszString),lpszString,&rClient,0L,0L,ulAttrs );

      if (sMix == USE_OVERPAINT)
         GpiSetBackMix(hdc,BM_DEFAULT);

      for (x = 0;x > sShift ; x++)
      {

         rClient.xLeft = x1;
         rClient.yBottom = y1;
         rClient.xLeft -= x;
         WinDrawText(hdc,strlen(lpszString),lpszString,&rClient,0L,0L,ulAttrs );

         rClient.yBottom += x;
         WinDrawText(hdc,strlen(lpszString),lpszString,&rClient,0L,0L,ulAttrs );

         rClient.xLeft += x;
         WinDrawText(hdc,strlen(lpszString),lpszString,&rClient,0L,0L,ulAttrs );
      }

      switch (sFeature)
      {
         case RAISED:
            GpiSetColor(hdc,CLR_DARKGRAY);
            break;
         case INDENTED:
            GpiSetColor(hdc,CLR_WHITE);
            break;
         case GRAYEDOUT:
            GpiSetColor(hdc,CLR_BLACK);
            break;
      }



      for (x = 1; x <= sShift; x++)
      {

         rClient.xLeft = x1;
         rClient.yBottom = y1;
         rClient.yBottom -= x;
         WinDrawText(hdc,strlen(lpszString),lpszString,&rClient,0L,0L,ulAttrs );

      }

      for (x = 1; x <= sShift; x++)
      {

         rClient.xLeft = x1;
         rClient.yBottom = y1;
         rClient.yBottom -= sShift;
         rClient.xLeft += x;
         WinDrawText(hdc,strlen(lpszString),lpszString,&rClient,0L,0L,ulAttrs );

      }

      for (x = 1; x <= sShift; x++)
      {

         rClient.xLeft = x1;
         rClient.yBottom = y1;
         rClient.yBottom -= sShift;
         rClient.yBottom += x;
         rClient.xLeft += sShift;
         WinDrawText(hdc,strlen(lpszString),lpszString,&rClient,0L,0L,ulAttrs );

      }


   }

   rClient.xLeft = x1;
   rClient.yBottom = y1;
   switch (sFeature)
   {
      case RAISED:
      case NONE:
         GpiSetColor(hdc,CLR_WHITE);
         break;
      case INDENTED:
         GpiSetColor(hdc,CLR_BLACK);
         break;
      case GRAYEDOUT:
         GpiSetColor(hdc,CLR_PALEGRAY);
         break;
   }
   WinDrawText(hdc,strlen(lpszString),lpszString,&rClient,0L,0L,ulAttrs );
}

//----------------------------------------------------------------------------------------------------//
// This function steps the simulation ahead in time. This is where the kinematic properties
// are calculated.  The function will return 1 when the target is hit, and 2 when the shell
// hits the ground (x-z plane) before hitting the target, otherwise the function returns 0.
//----------------------------------------------------------------------------------------------------//
int   DoSimulation(void)
//----------------------------------------------------------------------------------------------------//
{

   return 0;
}

///////////////////////////////////////////////////////////////////
/*  This is generic function to update the state of a given particle.
   There's no need to call this function directly, it's used by the
   particle explosion and hose functions.

   p:    pointer to a particle structure
   dtime:   time increment in milliseconds to advance the state of the particle

   If the total elapsed time for this particle has exceeded the particle's
   set life, then this function returns FALSE indicating that the particle
   should expire.
*/
BOOL  UpdateParticleState(TParticle* p, int dtime)
{
   BOOL retval;
   float t;
   ULONG ulCurrentMSCount;
   APIRET rc;

   rc = DosQuerySysInfo(QSV_MS_COUNT, QSV_MS_COUNT,
                  (PVOID)&p->ultimeLast,
                  sizeof(p->ultimeLast));
   p->time+=dtime;
   t = (float)p->time/1000.0f;
   t = (float)(p->ultimeLast - p->ultime)/1000.0f;
   p->x = p->vi * cos(p->angle*PI/180.0f) * t;
/*   p->y = p->vi * sin(p->angle*PI/180.0f) * t + (p->gravity*t*t)/2.0f; For windoze add gravity */
   p->y = p->vi * sin(p->angle*PI/180.0f) * t + (((p->gravity*t*t)/2.0f) * -1.0);  /* OS/2 subtract gravity */

/*   if (p->time >= p->life)  */
   if ((p->ultimeLast - p->ultime) >= p->life)
      retval = FALSE;
   else
      retval = TRUE;

   return retval;
}

/////////////////////////////////////////////////////////////////////
/* This function creates a new particle explosion effect.

   x,y:  starting point of the effect
   Vinit:   a measure of how fast the particles will be sent flying (it's actually the initial
         velocity of the particles)
   life: life of the particles in milliseconds; particles will fade and die out as they approach
         their specified life
   gravity: the acceleration due to gravity which controls the rate at which particles will fall
            as they fly
   angle:   initial trajectory angle of the particles, specify 999 to create a particle explosion
         that emits particles in all directions; otherwise, 0 right, 90 up, 180 left, etc...

   clr:  use one of the defines in FX.h to indicate a color; kfxRandom randomly selects one of
         kfxRed, kfxBlue or kfxGreen.

   Returns an index to the new effect.
*/
void     CreateParticleExplosion(int x, int y, int Vinit, int life, float gravity, float angle)
{
   int   i;
   int   m;
   float f;
   APIRET rc;
   int iRed;
   int iGreen;
   int iBlue;

   Explosion.Active = TRUE;
   Explosion.x = x;
   Explosion.y = y;
   Explosion.V0 = Vinit;
   Explosion.Particles = tb_Rnd(5, _MAXPARTICLES);
   Explosion.fFlash = (float)tb_Rnd(250, 500);
   iRed = tb_Rnd(5, 255);
   iGreen = tb_Rnd(5, 255);
   iBlue = tb_Rnd(5, 255);

/*   for(i=0; i<_MAXPARTICLES; i++)  */
   for(i=0; i<Explosion.Particles; i++)
   {
      Explosion.p[i].x = 0;
      Explosion.p[i].y = 0;
      Explosion.p[i].vi = tb_Rnd(Vinit/2, Vinit);

      if(angle < 999)
      {
         if(tb_Rnd(0,1) == 0)
            m = -1;
         else
            m = 1;
         Explosion.p[i].angle = -angle + m * tb_Rnd(0,10);
      } else
         Explosion.p[i].angle = tb_Rnd(0,360);

      f = (float) tb_Rnd(80, 100) / 100.0f;
      Explosion.p[i].life = tb_Round(life * f);
      Explosion.p[i].r = 255;//tb_Rnd(225, 255);
      Explosion.p[i].r = tb_Rnd(225, 255);
      Explosion.p[i].r = iRed;
      Explosion.p[i].g = 255;//tb_Rnd(85, 115);
      Explosion.p[i].g = tb_Rnd(45, 175);
      Explosion.p[i].g = iGreen;
      Explosion.p[i].b = 255;//tb_Rnd(15, 45);
      Explosion.p[i].b = tb_Rnd(15, 45);
      Explosion.p[i].b = iBlue;

      Explosion.p[i].time = 0;
      rc = DosQuerySysInfo(QSV_MS_COUNT, QSV_MS_COUNT,
                     (PVOID)&Explosion.p[i].ultime,
                     sizeof(Explosion.p[i].ultime));

      Explosion.p[i].Active = TRUE;
      Explosion.p[i].gravity = gravity;
   }

}


///////////////////////////////////////////////////////////////////
// Draws the particle system and updates the state of each particle.
// Returns false when all of the particles have died out.
BOOL  DrawParticleExplosion(HDC hdc)
{
   int   i;
   BOOL  finished = TRUE;
   float       r;
/*   COLORREF clr;  */
   RGB clr;

   ULONG ulCurrentMSCount;
   APIRET rc;

   rc = DosQuerySysInfo(QSV_MS_COUNT, QSV_MS_COUNT,
                  (PVOID)&ulCurrentMSCount,
                  sizeof(ulCurrentMSCount));

   if(Explosion.Active)
/*    for(i=0; i<_MAXPARTICLES; i++)  */
     for(i=0; i<Explosion.Particles; i++)
     {
      if(Explosion.p[i].Active)
      {
         finished = FALSE;
         r =   ((float)(Explosion.p[i].life-(ulCurrentMSCount - Explosion.p[i].ultime))/(float)(Explosion.p[i].life));
/*         clr = RGB1(tb_Round(r*Explosion.p[i].r), tb_Round(r*Explosion.p[i].g), tb_Round(r*Explosion.p[i].b));*/
         clr.bRed = tb_Round(r*Explosion.p[i].r);
         clr.bGreen = tb_Round(r*Explosion.p[i].g);
         clr.bBlue =  tb_Round(r*Explosion.p[i].b);
         DrawCircle( hdc,
                  Explosion.x+tb_Round(Explosion.p[i].x),
                  Explosion.y+tb_Round(Explosion.p[i].y),
                  2,
                  clr);
         Explosion.p[i].Active = UpdateParticleState(&(Explosion.p[i]), 10);
      }
     }

   if(finished)
      Explosion.Active = FALSE;

   return !finished;
}

BOOL  DrawParticleExplosion1(HDC hdc)
{
   int   i;
   BOOL  finished = TRUE;
   float       r;
/*   COLORREF clr;  */
   RGB clr;
   ULONG ulCurrentMSCount;
   APIRET rc;

   rc = DosQuerySysInfo(QSV_MS_COUNT, QSV_MS_COUNT,
                  (PVOID)&ulCurrentMSCount,
                  sizeof(ulCurrentMSCount));

   if(Explosion.Active)
/*    for(i=0; i<_MAXPARTICLES; i++)  */
     for(i=0; i<Explosion.Particles; i++)
     {
      if(Explosion.p[i].Active)
      {
         finished = FALSE;
         r =   ((float)(Explosion.p[i].life-(Explosion.p[i].ultimeLast - Explosion.p[i].ultime))/(float)(Explosion.p[i].life));
/*         clr = RGB1(tb_Round(r*Explosion.p[i].r), tb_Round(r*Explosion.p[i].g), tb_Round(r*Explosion.p[i].b));*/
         clr.bRed = 0;
         clr.bGreen = 0;
         clr.bBlue =  0;
         DrawCircle( hdc,
                  Explosion.x+tb_Round(Explosion.p[i].x),
                  Explosion.y+tb_Round(Explosion.p[i].y),
                  2,
                  clr);
         Explosion.p[i].Active = UpdateParticleState(&(Explosion.p[i]), 10);

         if(Explosion.p[i].Active)
         {
            r =   ((float)(Explosion.p[i].life-(Explosion.p[i].ultimeLast - Explosion.p[i].ultime))/(float)(Explosion.p[i].life));
   /*         clr = RGB1(tb_Round(r*Explosion.p[i].r), tb_Round(r*Explosion.p[i].g), tb_Round(r*Explosion.p[i].b));*/
            if (Explosion.p[i].ultimeLast - Explosion.p[i].ultime < tb_Round(Explosion.fFlash))
            {
               clr.bRed = tb_Round(r*Explosion.p[i].r + ((255.0 - (r*Explosion.p[i].r)) * ((Explosion.fFlash - (Explosion.p[i].ultimeLast - Explosion.p[i].ultime))/ Explosion.fFlash)));
               clr.bGreen = tb_Round(r*Explosion.p[i].g + ((255.0 - (r*Explosion.p[i].g)) * ((Explosion.fFlash - (Explosion.p[i].ultimeLast - Explosion.p[i].ultime))/ Explosion.fFlash)));
               clr.bBlue = tb_Round(r*Explosion.p[i].b + ((255.0 - (r*Explosion.p[i].b)) * ((Explosion.fFlash - (Explosion.p[i].ultimeLast - Explosion.p[i].ultime))/ Explosion.fFlash)));

            }
            else
            {
               clr.bRed = tb_Round(r*Explosion.p[i].r);
               clr.bGreen = tb_Round(r*Explosion.p[i].g);
               clr.bBlue =  tb_Round(r*Explosion.p[i].b);
            }
            DrawCircle( hdc,
                     Explosion.x+tb_Round(Explosion.p[i].x),
                     Explosion.y+tb_Round(Explosion.p[i].y),
                     2,
                     clr);
         }

      }
     }

   if(finished)
      Explosion.Active = FALSE;

   return !finished;
}

///////////////////////////////////////////////////////////////////
//  This function finds a random number
int tb_Rnd(int min, int max) {
   int   number;

    //number = min + rand()%max;
    //number = min +tb_Trunc(max*rand());
    number = (((abs(rand())%(max-min+1))+min));

    if(number>max) {
      number = max;
      //tb_Beep();
    }

    if(number<min) {
      number = min;
      //tb_Beep();
    }

   return number;
}

///////////////////////////////////////////////////////////////////
//  This function rounds a number
int tb_Round(double x) {
   int   result;
   double   remainder;

   result = (int) (x);

   if (result == 0) return 0;

   remainder = fmod(x, (double) result);
   if (remainder>=0.5)
      result++;

   return result;
}


