/*:VRX         
*/

/*
 * writevrw.vrx -- REXX routines needed to write out VX-REXX .VRW files.
 *
 *    The following routines can be used to write out .VRW files that
 *    can then be loaded by VX-REXX programs from disk.  See the
 *    accompanying documentation for details.  To use this file, either
 *    paste it in as a section in one of your files, or (with VX-REXX 2.1)
 *    add it as a shared section.
 *
 *    Version 1.0, February 14, 1995.
 */

/*:VRX         VRCloseVRW
*/

/*
 * VRCloseVRW -- Close a .VRW file.
 *
 *   ok = VRCloseVRW( filename )
 */

VRCloseVRW: procedure
    call stream arg(1), 'c', 'close'
return 1

/*:VRX         VRCreateVRW
*/

/* 
 * VRCreateVRW -- Opens a new .VRW file for writing.
 *
 *    ok = VRCreateVRW( filename, [version] )
 *
 * where
 * 
 *    filename = name of file to open, i.e. 'd:\foo.vrw'
 *    version = VX-REXX version to use, must be of the
 *              form 'A.BB' (defaults to '2.10')
 *    ok = 1 if file was successfully created
 */

VRCreateVRW: procedure

    filename = arg(1)
    version  = arg(2)

    if( version = "" )then version = "2.10"

    if( length( version ) \= 4 )then return 0

    if( VRFileExists( filename ) )then do
        call VRDeleteFile filename
    end

    call charout filename, "VRX   OS/2 v" || version
    call charout filename, "1a"x || "B3"

return 1
/*:VRX         VRLoadFromVRW
*/

/* 
 * VRLoadFromVRW -- Load a VRW file, returning the
 *                  handle of the window, or null if
 *                  nothing was loaded.
 *
 *    ok = VRLoadFromVRW( filename, [parent] )
 */

VRLoadFromVRW: procedure
    return 'VRLoad'( arg(2), arg(1) )
/*:VRX         VRStringToByte
*/

/* 
 * VRStringToByte -- Convert a numeric string into a single
 *                   byte.
 */

VRStringToByte: procedure expose error
    error = 0
    value = arg(1)
    if( value = "" | datatype( value, 'w' ) \= 1 )then do
        error = 1
        return d2c( 0 )
    end
    if( value < 0 | value > 255 )then do
        error = 1
        return d2c( 0 )
    end
return d2c( value )

/*:VRX         VRStringToLong
*/

/* 
 * VRStringToLong -- Convert a numeric string into an Intel
 *                   4 byte sequence.
 */

VRStringToLong: procedure expose error
    error = 0
    value = arg(1)
    if( value = "" | datatype( value, 'w' ) \= 1 )then do
        error = 1
        return '00000000'x
    end
    if( value < 0 )then do
        error = 1
        return '00000000'x
    end

    hiw = trunc( value / 65536 )
    low = value - hiw * 65536

    hihi = trunc( hiw / 256 )
    hilo = hiw - hihi * 256

    lohi = trunc( low / 256 )
    lolo = low - lohi * 256

return d2c( lolo ) || d2c( lohi ) || d2c( hilo ) ,
       || d2c( hihi )
/*:VRX         VRStringToShort
*/

/* 
 * VRStringToShort -- Convert a numeric string into an Intel
 *                    2 byte sequence.
 */

VRStringToShort: procedure expose error
    error = 0
    value = arg(1)
    if( value = "" | datatype( value, 'w' ) \= 1 )then do
        error = 1
        return '0000'x
    end
    if( value < 0 | value > 65535 )then do
        error = 1
        return '0000'x
    end
    hi = trunc( value / 256 )
    lo = value - hi * 256
return d2c( lo ) || d2c( hi )

/*:VRX         VRWriteByte
*/

/*
 * VRWriteByte
 *
 *    ok = VRWriteByte( filename, value )
 */

VRWriteByte: procedure

    filename = arg(1)
    value = arg(2)

    if( filename = "" | value = "" )then return 0
    if( datatype( value, 'w' ) \= 1 )then return 0
    if( value < 0 | value > 255 )then return 0
 
    call charout filename, d2c( value )
return

/*:VRX         VRWriteClassName
*/
VRWriteClassName: procedure
 
    filename = arg(1)
    classname = arg(2)

    call VRWriteByte filename, length( classname )
    call charout filename, classname

return 1

/*:VRX         VRWriteProperty
*/

/*
 * VRWriteProperty -- Write a property value out
 *
 *    ok = VRWriteProperty( filename, propname, value )
 */

VRWriteProperty: procedure

    filename = arg(1)
    propname = arg(2)
    value = arg(3)

    if( filename = '' | propname = '' )then return 0

    call VRWriteByte filename, length( propname )
    call charout filename, propname
    call VRWriteShort filename, length( value )

    if( length( value ) > 0 )then do
        call charout filename, value
    end

return 1

/*:VRX         VRWriteShort
*/

/* 
 * VRWriteShort -- Write a 2-byte value to a file.
 *
 *    ok = VRWriteShort( filename, value )
 */

VRWriteShort: procedure

    filename = arg(1)
    value    = arg(2)

    if( filename = "" | value = "" )then return 0
    if( datatype( value, 'w' ) \= 1 )then return 0
    if( value < 0 | value > 65535 )then return 0

    hi = trunc( value / 256 )
    lo = value - hi * 256

    call charout filename, d2c( lo ) || d2c( hi )
return
