/****************************************************************************/
/*                                                                          */
/*                              EYEBALLS                                    */
/*                                                                          */
/****************************************************************************/
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "guild.h"      /* Guild Definitions file. */
#include "guildpro.h"   /* Guild Prototype file. */

/****************************************************************************/
/*                     LOCAL PROTOTYPES                                     */
/****************************************************************************/

void RegEyeBallCFuncs(void);
Pointer EyeBallRegisterCallbacks( ArgsList args );
RtnStatus EyeBallPaintCallback(HandleType, RectLRec, BOOL);
RtnStatus EyeBallKeyCallback(HandleType, KeyFlags, VKeyType, short, USHORT );
RtnStatus EyeBallButtonCallback(HandleType, KeyFlags, BtnMsgType, PointLRec );
CursorType EyeBallMouseCallback(HandleType, KeyFlags, PointLRec, BOOL);
Pointer ZoomFunc( ArgsList args );
void ClearEyes(HandleType wndHandle);
void DrawEyeBrows(HandleType wndHandle);
void DrawEyes(HandleType wndHandle);
void DrawEyeballs(HandleType wndHandle);
void CalcEyeballPosition(long, PointLRec, long, PointLRecPtr);

/****************************************************************************/
/*                     LOCAL DATA DECLARATIONS                              */
/****************************************************************************/
PointLRec       cursorPos;
BOOL            leftEyeBlink = FALSE, rightEyeBlink = FALSE;
PointLRec       prevLeftEyeballPos, prevRightEyeballPos;
RGBColorType    bkgColor;

/****************************************************************************/
/* RegEyeBallCFuncs()                                                       */
/*      Register the window's callback functions.                           */
/****************************************************************************/
void RegEyeBallCFuncs( void )
{
    ActRegisterCFunction("EyeBallRegisterCallbacks", EyeBallRegisterCallbacks, DataTnone);
    ActRegisterCFunction("ZoomFunc", ZoomFunc, DataTnone);
}
/****************************************************************************/
/* EyeBallRegisterCallbacks()                                             */
/*      Register the window's callback functions.                           */
/****************************************************************************/
Pointer EyeBallRegisterCallbacks( ArgsList args )
{
    RectLRec        world;
    HandleType      wndHandle;

    if (!args)
        return(NULL);

    /* The window handle was passed in as the first (and only) arguement. */
    wndHandle = (HandleType) atoi(args[0]);

    GuiRegisterCallbacks( wndHandle,
                          EyeBallPaintCallback,
                          EyeBallMouseCallback,
                          NULL,
                          EyeBallButtonCallback,
                          NULL,
                          NULL );

    /* Set world dimensions for the window to be (0,0)-(1000,1000). This is
       done to get a fairly detailed drawing space. */
    world.x0 = world.y0 = 0L;
    world.x1 = world.y1 = 1000L;
    GuiSetWorld(wndHandle, &world);

    VarRetrievePropertyVariable( NULL, NULL, ObjPropTbackgroundcolor, 0, NULL, DataTlong, &bkgColor, 0);

    return(NULL);
}
/****************************************************************************/
/* EyeBallPaintCallback()                                                    */
/****************************************************************************/
RtnStatus EyeBallPaintCallback( HandleType wndHandle, RectLRec rectl, BOOL iconic)
{
    DrawEyeBrows(wndHandle);
    DrawEyes(wndHandle);
    return(RTN_OK);
}

/****************************************************************************/
/* EyeBallMouseCallback()                                                    */
/****************************************************************************/
CursorType EyeBallMouseCallback( HandleType wndHandle, KeyFlags keyFlags, PointLRec pointl, BOOL onWindow )
{
static  int     count = 0;

    if (++count == 3)
    {
        count = 0;
        cursorPos.x = pointl.x;
        cursorPos.y = pointl.y;
        DrawEyeballs(wndHandle);
    }
    return((CursorType)"Fly");
}

/****************************************************************************/
/* EyeBallButtonCallback()                                                   */
/****************************************************************************/
RtnStatus EyeBallButtonCallback( HandleType wndHandle, KeyFlags keyFlags, BtnMsgType event, PointLRec pointl )
{
    switch (event)
    {
        case BtnMsgTbuttonLclick:
            /* Toggle status of left eye. */
            leftEyeBlink = (short)(1 - leftEyeBlink);
            ClearEyes(wndHandle);
            DrawEyes(wndHandle);
            break;

        case BtnMsgTbuttonRclick:
            /* Toggle status of right eye. */
            rightEyeBlink = (short)(1 - rightEyeBlink);
            ClearEyes(wndHandle);
            DrawEyes(wndHandle);
            break;

        default:
            break;
    }
    return(RTN_OK);
}

/****************************************************************************/
/* ZoomFunc()                                                               */
/****************************************************************************/
Pointer ZoomFunc( ArgsList args )
{
static double   zoom = 1.0;
    RectLRec    viewport;
    PointLRec   center;
    HandleType  wndHandle;

    if (args && MemListCount(args) >= 2)
    {
        wndHandle = (HandleType)atoi(args[0]);

        if (strcmp(args[1], "IN") == 0)
            zoom = zoom * 1.5;
        else if (strcmp(args[1], "OUT") == 0)
            zoom = zoom / 1.5;
        else
            return (NULL);

        if (zoom < 1.0)
            zoom = 1.0;
        else if (zoom > 3.375)
            zoom = 3.375;

        GuiQueryViewport(wndHandle, &viewport);

        center.x = (viewport.x1 + viewport.x0) / 2;
        center.y = (viewport.y1 + viewport.y0) / 2;

        /* Modify the viewport by the zoom factor. */
        viewport.x0 = center.x - (long)(500 / zoom);
        if (viewport.x0 < 0)
            viewport.x0 = 0;
        viewport.y0 = center.y - (long)(500 / zoom);
        if (viewport.y0 < 0)
            viewport.y0 = 0;
        viewport.x1 = center.x + (long)(500 / zoom);
        if (viewport.x1 > 1000)
            viewport.x1 > 1000;
        viewport.y1 = center.y + (long)(500 / zoom);
        if (viewport.y1 > 1000)
            viewport.y1 > 1000;

        GuiSetViewport(wndHandle, &viewport);
        GuiInvalidateRegion(wndHandle, &viewport, FALSE);
    }
    return(NULL);
}

/****************************************************************************/
/* ClearEyes()                                                              */
/****************************************************************************/
void ClearEyes(HandleType wndHandle)
{
    PointLRec   pt1;
    long        width, height;

    GuiSetMixMode(wndHandle, MixModeTcopy);
    GuiSetColors(wndHandle, bkgColor, ColorTdefault);
    GuiSetFillContext(wndHandle, FillStyleTsolid);
    GuiSetLineContext(wndHandle, LineStyleTsolid, LineWidthTthin);
    width = 80;
    height = 80;
    pt1.y = 500;

    /* clear left eye */
    pt1.x = 400;
    GuiDrawEllipse(wndHandle, &pt1, &width, &height);

    /* clear right eye */
    pt1.x = 600;
    GuiDrawEllipse(wndHandle, &pt1, &width, &height);
}

/****************************************************************************/
/* DrawEyeBrows()                                                           */
/****************************************************************************/
void DrawEyeBrows(HandleType wndHandle)
{
    PointLRec   pt1;
    long        width, height, angle1, angle2;

    GuiSetMixMode(wndHandle, MixModeTcopy);
    GuiSetColors(wndHandle, ColorTblack, ColorTdefault);
    GuiSetFillContext(wndHandle, FillStyleTnone);

    /* Draw eye brows */
    GuiSetLineContext(wndHandle, LineStyleTsolid, LineWidthTthick);
    width = 80;
    height = 80;
    angle1 = 30 * 60;
    angle2 = 150 * 60;
    pt1.y = 600;
    pt1.x = 400;
    GuiDrawArc(wndHandle, &pt1, &width, &height, &angle1, &angle2);
    pt1.x = 600;
    GuiDrawArc(wndHandle, &pt1, &width, &height, &angle1, &angle2);
}

/****************************************************************************/
/* DrawEyes()                                                               */
/****************************************************************************/
void DrawEyes(HandleType wndHandle)
{
    PointLRec   pt1;
    long        width, height, angle1, angle2;

    GuiSetMixMode(wndHandle, MixModeTcopy);
    GuiSetColors(wndHandle, ColorTblack, ColorTdefault);
    GuiSetFillContext(wndHandle, FillStyleTnone);

    /* Draw eye sockets */
    GuiSetLineContext(wndHandle, LineStyleTsolid, LineWidthTthin);
    width = 80;
    height = 80;
    pt1.y = 500;

    /* Draw open left eye */
    if (!leftEyeBlink)
    {
        pt1.x = 400;
        GuiDrawEllipse(wndHandle, &pt1, &width, &height);
    }

    /* Draw open right eye */
    if (!rightEyeBlink)
    {
        pt1.x = 600;
        GuiDrawEllipse(wndHandle, &pt1, &width, &height);
    }

    if (!leftEyeBlink || !rightEyeBlink)
        DrawEyeballs(wndHandle);

    /* Draw closed left eye */
    if (leftEyeBlink)
    {
        GuiSetColors(wndHandle, ColorTblack, ColorTdefault);
        GuiSetFillContext(wndHandle, FillStyleTsolid);
        angle1 = 0 * 60;
        angle2 = 180 * 60;
        pt1.x = 400;
        GuiDrawChord(wndHandle, &pt1, &width, &height, &angle1, &angle2);
    }

    /* Draw closed right eye */
    if (rightEyeBlink)
    {
        GuiSetColors(wndHandle, ColorTblack, ColorTdefault);
        GuiSetFillContext(wndHandle, FillStyleTsolid);
        angle1 = 0 * 60;
        angle2 = 180 * 60;
        pt1.x = 600;
        GuiDrawChord(wndHandle, &pt1, &width, &height, &angle1, &angle2);
    }
}

/****************************************************************************/
/* DrawEyeballs()                                                           */
/****************************************************************************/
void DrawEyeballs(HandleType wndHandle)
{
    PointLRec   pt1,pt2;
    long        width, height;

    GuiSetMixMode(wndHandle, MixModeTcopy);
    GuiSetColors(wndHandle, bkgColor, ColorTdefault);
    GuiSetFillContext(wndHandle, FillStyleTsolid);
    GuiSetLineContext(wndHandle, LineStyleTsolid, LineWidthTthin);

    width = 30;
    height = 30;

    /* Draw left eyeball at old location to erase it */
    if (prevLeftEyeballPos.y != -1 && !leftEyeBlink)
    {
        pt1.y = prevLeftEyeballPos.y;
        pt1.x = prevLeftEyeballPos.x;
        GuiDrawEllipse(wndHandle, &pt1, &width, &height);
    }

    /* Draw right eyeball at old location to erase it */
    if (prevRightEyeballPos.y != -1 && !rightEyeBlink)
    {
        pt1.y = prevRightEyeballPos.y;
        pt1.x = prevRightEyeballPos.x;
        GuiDrawEllipse(wndHandle, &pt1, &width, &height);
    }

    GuiSetColors(wndHandle, ColorTblue, ColorTdefault);

    /* Draw left eyeball at new location */
    if (!leftEyeBlink)
    {
        pt2.y = 500;
        pt2.x = 400;
        CalcEyeballPosition(80, pt2, 30, &pt1);
        GuiDrawEllipse(wndHandle, &pt1, &width, &height);
        prevLeftEyeballPos.x = pt1.x;
        prevLeftEyeballPos.y = pt1.y;
    }

    /* Draw right eyeball at new location */
    if (!rightEyeBlink)
    {
        pt2.y = 500;
        pt2.x = 600;
        CalcEyeballPosition(80, pt2, 30, &pt1);
        GuiDrawEllipse(wndHandle, &pt1, &width, &height);
        prevRightEyeballPos.x = pt1.x;
        prevRightEyeballPos.y = pt1.y;
    }
}

/****************************************************************************/
/* CalcEyeballPosition()                                                    */
/*      Find the center of the eyeball given the radius of the eyeball and  */
/*      the radius and center of the eye socket.                            */
/****************************************************************************/
void CalcEyeballPosition(
    long            socketRadius,
    PointLRec       socketCenter,
    long            eyeballRadius,
    PointLRecPtr    eyeballCenter
)
{
    double      angle;

    if (socketCenter.y == cursorPos.y)
        angle = 1;
    else
        angle = atan(((double)(socketCenter.x - cursorPos.x)) / ((double)(socketCenter.y - cursorPos.y)) );

    if (cursorPos.y < socketCenter.y)
    {

        eyeballCenter->x = socketCenter.x +
                           (long)((double)(socketRadius - eyeballRadius - 5) * sin(-angle));

        eyeballCenter->y = socketCenter.y +
                           (long)((double)(socketRadius - eyeballRadius - 5) * -cos(-angle));
    }
    else
    {
        eyeballCenter->x = socketCenter.x +
                           (long)((double)(socketRadius - eyeballRadius - 5) * sin(angle));

        eyeballCenter->y = socketCenter.y +
                           (long)((double)(socketRadius - eyeballRadius - 5) * cos(angle));
    }
}

