/* MapSymTool - mapsym helper
   Supports editing working map file to fit mapsym size limits
   Can estimate sym file size to assist with editing
   Can run mapsym on demand
   Supports converting OpenWatcom maps to IBM map format

   FIXME to support applying c++filt to C++ symbols
   FIXME to support import symbols from reference map file

   Copyright (c) 2019-2023 Steven Levine and Associates, Inc.
   All rights reserved.

   2019-04-08 SHL Baseline - clone from elsewhere
   2019-05-23 SHL Correct case
   2020-01-04 SHL Tweak status output
   2022-06-27 SHL Sync with templates
   2022-06-27 SHL DoRunMapSym - clear queue on error
   2023-02-21 SHL Sync with templates
*/

signal on Error
signal on Failure name Error
signal on Halt
signal on NotReady name Error
signal on NoValue name Error
signal on Syntax name Error

gVersion = '0.2 2023-02-21'

Globals = 'gBkupMapFile gCmdName gDbgLvl gEditor gEnv',
	  'gLastAction gMapFile gRefMapFile',
	  'gSymFile gSymFileState gVerbose gVersion'

call Initialize

Main:

  /* Ensure running under 4OS2 */
  call Chk4OS2

  parse arg cmdLine
  call ScanArgs cmdLine
  drop cmdLine

  call DoInKeyLoop

  exit

/* end main */

/*====================*/
/*=== Menu options ===*/
/*====================*/

/*=== DoInKeyLoop() Do inkey loop template ===*/

DoInKeyLoop: procedure expose (Globals)

  /* Requires InKey */
  /* Requires ToLower */

  escKey = x2c('1b')
  enterKey = x2c('0d')
  gLastAction = ''
  if IsFile(gSymFile) then
    gSymFileState = 'sym file status unknown'
  else
    gSymFileState = ''

  skipStatus = 0

  do forever

    if \ skipStatus then do
      skipStatus = 0
      say
      'dir /kmt' gMapFile
      if IsFile(gSymFile) then do
      say
      'dir /kmt' gSymFile
      end
      if gSymFileState \== '' then do
	say
	say gSymFileState
	gSymFileState = ''

      end
    end

    say
    keys = 'bcemrsvyHhq!?[Esc][Enter]'
    key = InKey(keys, 'B)ackup C)heck E)dit I)mport M)apsym R)estore S)tatus s(Y)nc H)elp Q)uit')
    key = ToLower(key)
    select
    when key == 'q' then do
      retcode = 1
      leave
    end
    when key == escKey then do
      retcode = 2
      leave
    end
    when key == enterKey then do
      say
      if gLastAction \== '' then do
	say
	say 'Last action was' gLastAction
      end
      iterate
    end
    when key == 'b' then do
      call DoBkupMapFile
      iterate
    end
    when key == 'c' then do
      call DoEstimateSize
      iterate
    end
    when key == 'e' then do
      call DoEditMapFile
      iterate
    end
    when key == 'f' then do
      call DoFilterSymbols
      iterate
    end
    when key == 'i' then do
      call DoImportSymbols
      iterate
    end
    when key == 'm' then do
      call DoRunMapSym
      iterate
    end
    when key == 'r' then do
      call DoRestoreBkupMapFile
      iterate
    end
    when key == 's' then do
      call DoShowStatus
      skipStatus = 1
      iterate
    end
    when key == 'y' then do
      call DoSyncAddresses
      iterate
    end
    when key == '!' then do
      /* Shell */
      say
      shell = value('COMSPEC',, gEnv)
      signal off Error
      shell
      signal on Error
    end
    when key == 'h' | key == '?' then
      call DoShowHelp
    otherwise
      say
      say 'Unexpected key'
      'pause'
      exit 1
    end /* select */
  end /* forever */

  return retcode

/* end DoInKeyLoop */

/*=== DoOne() Do one thing ===*/

DoOne: procedure expose (Globals)

  return

/* end DoOne */

/*=== DoOne() Do one thing ===*/

DoOne: procedure expose (Globals)

  return

/* end DoOne */

/*=== DoOne() Do one thing ===*/

DoOne: procedure expose (Globals)

  return

/* end DoOne */

/*=== DoBkupMapFile() Offer to backup working map file ===*/

DoBkupMapFile: procedure expose (Globals)

  'dir /kmt' gMapFile
  signal off Error
  say
  'dir /kmt' gBkupMapFile
  signal on Error

  say
  signal off Error
  'copy /p' gMapFile gBkupMapFile
  say
  'dir /kmt' gMapFile gBkupMapFile
  signal on Error

  return

/* end DoBkupMapFile */

/*=== DoFilterSymbols() Offer to filter C++ symbols ===*/

DoFilterSymbols: procedure expose (Globals)

  say 'FIXME to be done'
  return

/* end DoFilterSymbols */

/*=== DoEditMapFile() edit map file ===*/

DoEditMapFile: procedure expose (Globals)

  if symbol('gEditor') \== 'VAR' then do
    call FindGUIEditor
  end

  if \ IsFile(gBkupMapFile) then do
    say 'Saving' gMapFile 'as' gBkupMapFile
    'copy' gMapFile gBkupMapFile
  end

  cmd = gEditor gMapFile

  signal off Error
  cmd
  signal on Error

  if RC \ = 0 then
    say cmd 'completed with RC' RC

  return

/* end DoEditMapFile */

/*=== DoEstimateSize() Estimate sym file size ===*/

DoEstimateSize: procedure expose (Globals)

  call stream gMapFile, 'C', 'OPEN READ'

  /* Assume sorted */
  objects = 0
  do while lines(gMapFile) > 0

    line = linein(gMapFile)
    if left(line, 1) \== ' ' then
      iterate

    /* 0001:000009f0       svn_cl_blame */

    line = strip(line)
    parse var line object':'offset sym
    if object < 1 | object > 9999 then
      iterate
    if \ datatype(offset, 'X') then
      iterate
    sym = strip(sym)
    if sym == '' then
      iterate
    if object > objects then do
      do objectNum = objects + 1 to object
	objectSize.objectNum = 0
      end
      objects = object
    end

    /* Accumulate */
    object = 0 + object
    objectSize.object = objectSize.object + length(sym) + 6

  end /* do */

  call stream gMapFile, 'C', 'CLOSE'

  say
  totalSize = 0
  do object = 1 to objects
    size = objectSize.object
    say 'object' object 'estimated size is' size
    totalSize = totalSize + size
  end
  say 'Estimated total size is' totalSize

  return

/* end DoEstimateSize */

/*=== DoImportSymbols() Import symbols from reference map file ===*/

DoImportSymbols: procedure expose (Globals)

  say 'FIXME to be done'
  return

/* end DoImportSymbols */

/*=== DoRestoreBkupMapFile() reset map file to original ===*/

DoRestoreBkupMapFile: procedure expose (Globals)

  if \ IsFile(gBkupMapFile) then do
    say 'Cannot access' gBkupMapFile '- cannot restore' gMapFile
    return
  end

  'dir /kmt' gBkupMapFile
  'dir /kmt' gMapFile
  say
  cmd = 'copy /p' gBkupMapFile gMapFile

  cmd
  signal off Error
  signal on Error

  if RC \= 0 then
    call WarnMsg cmd 'failed with error' RC

  return

/* end DoRestoreBkupMapFile */

/*=== DoRunMapSym() Import archive file to git repo ===*/

DoRunMapSym: procedure expose (Globals)

  call EmptyQueue

  cmd = 'mapsym' gMapFile '|& rxqueue'

  signal off Error
  cmd
  signal on Error

  if RC \= 0 then do
    do while queued() \= 0
      parse pull line
    end
    gSymFileState = 'mapsym failed with error' RC
    say cmd 'failed with error' RC
    return
  end

  gSymFileState = 'mapsym ran without errors'
  do while queued() \= 0
    parse pull line
    line = strip(line)
    if line == '' then
      iterate
    say line
    if pos('exceed 64K', line) > 0 then
      gSymFileState = line
  end

  return

/* end DoRunMapSym */

/*=== DoShowStatus(arg) show current state ===*/

DoShowStatus: procedure expose (Globals)

  say
  say 'working map file is' gMapFile
  'dir /kmt' gMapFile

  say
  say 'sym file is' gSymFile
  if IsFile(gSymFile) then
    'dir /kmt' gSymFile

  say
  say 'reference map file is' gRefMapFile
  if IsFile(gRefMapFile) then
    'dir /kmt' gRefMapFile

  say
  say 'backup map file is' gBkupMapFile
  if IsFile(gBkupMapFile) then
    'dir /kmt' gBkupMapFile

  return

/* end DoShowStatus */

/*=== DoShowHelp() Show help ===*/

DoShowHelp: procedure expose (Globals)
  say
  say 'B - Backup working map file to backup map file'
  say 'C - Check estimated sym file size'
  say 'E - Edit working map file'
  say 'F - Filter C++ symbols with c++filt (FIXME)'
  say 'H - Display this screen'
  say 'I - Import symbols from reference map file (FIXME)'
  say 'M - Run mapsym'
  say 'Q - Quit'
  say 'R - Restore working map file from backup map file'
  say 'S - Show current state'
  say 'Y - Sync working map file addresses with reference map file'
  say 'Escape - Quit'
  say 'Enter - Show current state'
  say '? - Display this screen'
  say '! - Shell'
  return

/* end DoShowHelp */

/*=== DoSyncAddresses() Sync working map file addresses with ref map file ===*/

DoSyncAddresses: procedure expose (Globals)

  if \ IsFIle(gRefMapFile) then
    say 'Cannot access' gRefMapFile '- cannot sync addresses'

  /* Read working map file to stem */
  call stream gMapFile, 'C', 'OPEN READ'
  mapLines.0 = 0
  do while lines(gMapFile) > 0
    line = linein(gMapFile)
    lineNum = mapLines.0 + 1
    mapLines.lineNum = line
    mapLines.0 = lineNum
  end
  call stream gMapFile, 'C', 'CLOSE'


  /* Read reference map file to stem */
  call stream gRefMapFile, 'C', 'OPEN READ'
  refMapLines.0 = 0
  do while lines(gRefMapFile) > 0
    line = linein(gRefMapFile)
    lineNum = refMapLines.0 + 1
    refMapLines.lineNum = line
    refMapLines.0 = lineNum
  end
  call stream gRefMapFile, 'C', 'CLOSE'


  /* Update working map file addresses from reference map file
     We assume that reference map file is from a newer build and
     that the working map file already contains symbols of interest
     FIXME to attempt to update manually added symbols maybe
  */

  /* Skip header */
  do mapLineNum = 1 to mapLines.0
    s = mapLines.mapLineNum
    if s == '  Address         Publics by Value' then
      leave
  end /* mapLineNum */
  if mapLineNum > mapLines.0 then
    call Die 'Cannot locate Publics by Value in' mapFile
  mapLineNum = mapLineNum + 1

  /* Skip header */
  do refMapLineNum = 1 to refMapLines.0
    s = refMapLines.refMapLineNum
    if s == '  Address         Publics by Value' then
      leave
  end /* refMapLineNum */
  if refMapLineNum > refMapLines.0 then
    call Die 'Cannot locate Publics by Value in' refMapFile
  refMapLineNum = refMapLineNum + 1

  changes = 0
  symbols = 0
  missing = 0
  debug = 0			/* Enable for trace debugging FIXME */

  do mapLineNum = mapLineNum to mapLines.0
    mapLine = mapLines.mapLineNum
    if mapLine = '' then
      iterate				/* Skip blank */

    symbols = symbols + 1

    /* Preserve tail in case have manual comments */
    /* 0001:00000040       _new_array */
    parse var mapLine mapAddr mapSymbol mapLineTail

    /* Allow for whitespace variations */
    mapSymbol = strip(mapSymbol)

    /* Symbols are usually, but not always in same order so we may need 2nd pass */
    if refMapLineNum <= refMapLines.0 then
      pass = 1
    else do
      pass = 2
      refMapLineNum = 1
    end

    /* Search */
    do refMapLineNum = refMapLineNum to refMapLines.0
      refMapLine = refMapLines.refMapLineNum
      parse var refMapLine refMapAddr refMapSymbol

      /* Allow for whitespace variations */
      refMapSymbol = strip(refMapSymbol)

      if 0 & trace() <> '?A' then trace '?A' ; nop	/* FIXME to be gone debug */

      if mapSymbol == refMapSymbol then
	leave				/* Found it */

      if pass = 1 & refMapLineNum = refMapLines.0 then do
	pass = 2
	refMapLineNum = 0
	if 0 & trace() <> '?A' then trace '?A' ; nop	/* FIXME to be gone debug */
      end
    end /* refMapLines */

    if refMapLineNum > refMapLines.0 then do
      say
      say mapLine
      call AskYNQ 'Cannot locate' strip(mapSymbol) 'in reference map file - continue'
      if RESULT >= 2 then exit
      if RESULT \= 0 then return
      missing = missing + 1
    end
    else do
      if mapAddr \== refMapAddr then do
	if debug then do
	  say '"' || mapSymbol || '"'
	  say '"' || refMapSymbol || '"'
	  say '"' || mapAddr || '"'
	  say '"' || refMapAddr || '"'
	  if debug & trace() <> '?A' then trace '?A' ; nop	/* FIXME to be gone debug */
	end
	/* Rewrite line with standard spacing */
	mapLine = '' refMapAddr || copies(' ', 7) || mapSymbol || mapLineTail
	mapLines.mapLineNum = mapLine
	changes = changes + 1
      end /* if changed */
    end /* if found */
  end /* mapLines */

  say
  say 'Processed' Plural(symbols, 'symbol') 'in map file'
  if 0 & trace() <> '?A' then trace '?A' ; nop	/* FIXME to be gone debug */
  say 'Modified' Plural(changes, 'address') 'in map file'
  say 'Detected' Plural(missing, 'missing symbol') 'in reference map file'

  if missing > 0 then do
    call AskYNQ 'Map file contains symbols not found in reference map file - continue'
    if RESULT >= 2 then exit
    if RESULT \= 0 then return
  end

  if changes = 0 then do
    say
    say 'No addresses changed - map file not modified'
  end
  else do
    call AskYNQ 'Rewrite' gMapFile
    if RESULT >= 2 then exit
    if RESULT = 0 then do
      /* FIXME to preserve EAs maybe */
      call SysFileDelete gMapFile
      call stream gMapFile, 'C', 'OPEN WRITE'
      do mapLineNum = 1 to mapLines.0
	mapLine = mapLines.mapLineNum
	call lineout gMapFIle, mapLine
      end
      call stream gMapFile, 'C', 'CLOSE'
    end /* if OK */
  end /* changes */
  return

/* end DoSyncAddresses */

/*=========================*/
/*=== Support functions ===*/
/*=========================*/

/*=== ConvertWatcomMapFileIf(mapFile) Convert Watcom map file if allowed or die ===*/

/**
 * @param mapFile is map file to offer to convert
 * @returns working map file name
 */

ConvertWatcomMapFileIf: procedure expose (Globals)

  parse arg mapFile

  call stream mapFile, 'C', 'OPEN READ'
  line = linein(mapFile)
  call stream mapFile, 'C', 'CLOSE'
  /* Good enough simple check */
  if pos('Open Watcom Linker Version', line) = 0 then
    return mapFile

  say
  say mapFile 'is OpenWatcom format'
  call AskYNQ 'Convert to IBM format - N dies'
  if RESULT \= 0 then
    exit

  ibmMapFile = mapFile || '-ibm'
  if IsFile(ibmMapFile) then do
    'dir /kmt' ibmMapFile
    say
    call AskYNQ 'Overwrite' ibmMapFile
    if RESULT \= 0 then
      exit
  end

  cmd = 'mapsymw.pl' mapFile
  signal off Error
  cmd
  signal on Error

  if RC \= 0 then do
    say cmd 'failed with error' RC
    exit RC
  end

  return ibmMapFile

/* end ConvertWatcomMapFileIf */

/*=== Initialize() Initialize globals ===*/

Initialize: procedure expose (Globals)
  call SetCmdName
  call LoadRexxUtil
  gEnv = 'OS2ENVIRONMENT'
  return

/* end Initialize */

/*=== ScanArgsInit() ScanArgs initialization exit routine ===*/

ScanArgsInit: procedure expose (Globals) cmdTail swCtl keepQuoted

  /* Preset defaults */
  gDbgLvl = 0				/* Display debug messages */
  gVerbose = 0				/* Verbose messages */
  gMapFile = ''				/* Map file */
  gRefMapFile = ''			/* Refernce map file */

  return

/* end ScanArgsInit */

/*=== ScanArgsSwitch() ScanArgs switch option exit routine ===*/

ScanArgsSwitch: procedure expose (Globals) curSw curSwArg

  select
  when curSw == 'd' then
    gDbgLvl = gDbgLvl + 1
  when curSw == 'h' | curSw == '?' then
    call ScanArgsHelp
  when curSw == 'v' then
    gVerbose = gVerbose + 1
  when curSw == 'V' then do
    say gCmdName gVersion
    exit
  end
  otherwise
    call ScanArgsUsage 'Switch "-' || curSw || '" unexpected'
  end /* select */

  return

/* end ScanArgsSwitch */

/*=== ScanArgsArg() ScanArgs argument option exit routine ===*/

ScanArgsArg: procedure expose (Globals) curArg

  if gRefMapFile \== '' then
    call ScanArgsUsage 'Already have map file and reference map file'

  call SysFileTree curArg, 'fileList', 'FO'
  if fileList.0 = 0 then
    call ScanArgsUsage 'Cannot access' curArg
  if fileList.0 \= 1 then
    call ScanArgsUsage curArg 'must match only one file'

  if gMapFile == '' then
    gMapFile = fileList.1
  else
    gRefMapFile = fileList.1

  return

/* end ScanArgsArg */

/*=== ScanArgsTerm() ScanArgs scan end exit routine ===*/

ScanArgsTerm: procedure expose (Globals)

  if gMapFile == '' then
    call ScanArgsUsage 'Map file name required'

  iSlash = lastpos('\', gMapFile)
  iDot = lastpos('.', gMapFile)
  if iDot <= iSlash then
    call ScanArgsUsage gMapFile 'must have a file extension'
  gSymFile = left(gMapFile, iDot) || 'sym'

  gMapFile = ConvertWatcomMapFileIf(gMapFile)

  gBkupMapFile = gMapFile || '-bkup'

  if IsFile(gBkupMapFile) then do
    say
    'dir /kmt' gMapFile
    say
    'dir /kmt' gBkupMapFile
    call AskYNQ 'OK to use' gBkupMapFile 'as' gMapFile 'backup'
    if RESULT >= 2 then exit
    if RESULT \= 0 then exit
  end

  if gRefMapFile == '' then
    gRefMapFIle = gBkupMapFile

  gRefMapFile = ConvertWatcomMapFileIf(gRefMapFile)

  return

/* end ScanArgsTerm */

/*=== ScanArgsHelp() Display ScanArgs usage help exit routine ===*/

ScanArgsHelp:
  say
  say 'git import helper.'
  say
  say 'Usage:' gCmdName '[-d] [-h] [-v] [-V] [-?] mapfile'
  say
  say '  -d           Enable debug logic, repeat for more verbosity'
  say '  -h -?        Display this message'
  say '  -v           Enable verbose output, repeat for more verbosity'
  say '  -V           Display version number and quit'
  say
  say '  mapfile      map file name'
  exit 255

/* end ScanArgsHelp */

/*=== ScanArgsUsage(message) Report Scanargs usage error exit routine ===*/

ScanArgsUsage:
  parse arg msg
  say
  if msg \== '' then
    say msg
  say 'Usage:' gCmdName '[-d] [-h] [-v] [-V] [-?] [repo] mapfile'
  exit 255

/* end ScanArgsUsage */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*=== AskYNQ([prompt][, noskip[, nofocus]])) returns 0=Yes, 1=No, 2=Quit, skips line unless noskip ===*/

AskYNQ: procedure
  parse arg msg, noskip, nofocus

  /* Take focus with 4OS2 or fail if cannot match window title */
  /* If script designed for CMD too, use nofocus arg to avoid error noise */
  signal off Error
  /* Map 1st left bracket to wild card - [pid] seems to confuse activate */
  if nofocus = '' | nofocus \= 1 then
    '@if defined _WINTITLE activate "%@replace[[,*,%_WINTITLE]"'
  signal on Error

  /* Skip line unless suppressed by noskip arg - any non-zero value requests noskip */
  if noskip = '' | noskip = 0 then
    call lineout 'STDERR', ''

  if msg == '' then
    msg = 'Continue'
  call charout 'STDERR', msg '(y/n/q) ? '
  do forever
    key = translate(SysGetKey('NOECHO'))
    if key == 'Y' | key == 'N' then do
      call lineout 'STDERR', key
      if key == 'Y' then
	ynq = 0
      else
	ynq = 1
      leave
    end
    if key == 'Q' | c2x(key) == '1B' then do
      call lineout 'STDERR', ''
      ynq = 2
      leave
    end
  end /* forever */
  return ynq

/* end AskYNQ */

/*=== Chk4OS2() Return if running under 4OS2 or die ===*/

Chk4OS2: procedure expose (Globals)
  /* Keep Is4OS2 and Chk4OS2 in sync */
  old = value('_X',, gEnv)		/* In case in use */
  /* 4OS2 sets _X to 0, cmd.exe sets x to @eval[0], rxd leaves X unchanged */
  '@set _X=%@eval[0]'
  new = value('_X',, gEnv)
  '@set _X=' || old			/* Restore */
  yes = new = 0 | old == new		/* Assume 4OS2 if running under rxd */
  if \ yes then
    call Die gCmdName 'must run under 4OS2'
  return				/* if running under 4OS2 */

/* end Chk4OS2 */

/*=== DbgMsg([minLvl, ]message,...) Optionally write multi-line message to STDERR ===*/

/**
 * Write message if gDbgLvl >= minLvl
 * @param minLvl defaults to 1 if omitted
 * @returns true if message written
 */

DbgMsg: procedure expose (Globals)
  minLvl = arg(1)
  if datatype(minLvl, 'W') then
    start = 2
  else do
    minLvl = 1
    start = 1
  end
  if gDbgLvl >= minLvl then do
    do i = start to arg()
      msg = arg(i)
      if msg \== '' then
	msg = ' *' msg
      call lineout 'STDERR', msg
    end
  end
  return gDbgLvl >= minLvl

/* end DbgMsg */

/*=== EmptyQueue() Empty default queue ===*/

EmptyQueue: procedure
  do while queued() \= 0
    parse pull .
  end
  return

/* end EmptyQueue */

/*=== FindGUIEditor() Find GUI editor ===*/

FindGUIEditor: procedure expose (Globals)
  /* Requires GetEnv */
  /* Requires IsExeInPath */
  /* Uses gEditor from Globals */
  /* Uses EDITOR from environment */
  /* Uses CPE from environment */
  do while symbol('gEditor') \== 'VAR'
    gEditor = GetEnv('EDITOR')
    if gEditor \== '' then leave
    gEditor = GetEnv('CPE')
    if gEditor \== '' then do
      gEditor = gEditor || '\bin\cpe'
      leave
    end
    gEditor = 'vim'
    if IsExeInPath(gEditor) then leave
    gEditor = 'vimx.cmd'
    if IsExeInPath(gEditor) then do
      gEditor = '4os2 /c vimx'
      leave
    end
    gEditor = 'tedit'
    if IsExeInPath(gEditor) then leave
    call Die 'EDITOR not defined and cannot guess usable GUI editor'
  end
  return

/* end FindGUIEditor */

/*=== GetEnv(var) Return value for environment variable or empty string ===*/

GetEnv: procedure expose (Globals)
  parse arg var
  if var = '' then
    call Die 'GetEnv requires an argument'
  return value(var,, gEnv)

/* end GetEnv */

/*=== InKey(Keys, Prompt) returns key code ===*/

InKey: procedure
  parse arg keys, msg
  /* Convert key names to characters */
  i = pos('[Enter]', keys)
  if i > 0 then
    keys = substr(keys, 1, i - 1) || x2c('0d') || substr(keys, i + 7)
  i = pos('[Esc]', keys)
  if i > 0 then
    keys = substr(keys, 1, i - 1) || x2c('1b') || substr(keys, i + 5)
  call charout 'STDERR', msg '? '
  do forever
    key = SysGetKey('NOECHO')
    i = pos(key, keys)
    if i > 0 then do
      i = pos(key, xrange('20'x, '7e'x))
      if i > 0 then
	call lineout 'STDERR', key
      leave
    end
  end /* forever */
  return key

/* end InKey */

/*=== IsDir(dirName[, full]) return true if directory is valid, accessible directory ===*/

IsDir: procedure
  /* If arg(2) not omitted, return full directory name or empty string */
  parse arg dir, full
  newdir = ''

  do 1
    if dir == '' then do
      cwd = ''				/* No restore needed */
      leave
    end
    dir = translate(dir, '\', '/')	/* Convert to OS/2 slashes */
    s = strip(dir, 'T', '\')		/* Chop trailing slashes unless root */
    if s \== '' & right(s, 1) \== ":" then
      dir = s				/* Chop */
    drv = filespec('D', dir)
    cwd = directory()			/* Remember */
    /* If have drive letter and requested directory on some other drive */
    if drv \== '' & translate(drv) \== translate(left(cwd, 2)) then do
      /* Avoid slow failures and unwanted directory changes */
      drvs = SysDriveMap('A:')
      if pos(translate(drv), drvs) = 0 then
	leave				/* Unknown drive */
      if SysDriveInfo(drv) == '' then
	leave				/* Drive not ready */
      cwd2 = directory(drv)		/* Remember current directory on other drive */
      newdir = directory(dir)		/* Try to change and get full path name */
      call directory cwd2		/* Restore current directory on other drive */
      leave
    end

    /* If no drive letter or same drive and not UNC name */
    if left(dir, 2) \== '\\' then do
      newdir = directory(dir)		/* Try to change and get full path name */
      leave
    end

    /* UNC name - hopefully server is accessible or this will be slow
       Accept
	 \\server
	 \\server\
	 \\server\dir\
	 \\server\dir
     */
    cwd = ''				/* No restore needed */
    wc = dir
    if right(wc, 1) \== '\' then
      wc = wc || '\'
    i = lastpos('\', wc)
    if substr(wc, 3, 1) == '\' then
      leave				/* Malformed UNC - no server name */
    if pos('*', wc) > 0 | pos('?', wc) > 0 then
      leave				/* No wildcards allowed */
    call SysFileTree wc, 'files', 'O'
    if files.0 > 0 then do
      s = files.1			/* Exists and is not empty */
      i = lastpos('\', s)
      newdir = left(s, i - 1)		/* Extract directory name from full path name */
      leave
    end
    /* Try wildcarded directory name */
    wc = strip(wc, 'T', '\')
    i = lastpos('\', wc)
    base = substr(wc, i + 1)
    if base == '' then
      leave				/* Should have matched above */
    wc = substr(wc, 1, i) || '*' || base || '*'
    call SysFileTree wc, 'files', 'DO'
    do fileNum = 1 to files.0
      /* Find directory name is list */
      s = files.fileNum
      i = lastpos('\', s)
      s2 = substr(s, i + 1)
      if translate(base) == translate(s2) then do
	newdir = left(s, i - 1)
	leave
      end
    end /* i */
  end /* 1 */

  if cwd \== '' then
    call directory cwd			/* Restore original directory and drive */

  if full \== '' then
    ret = newdir			/* Return full directory name or empty string */
  else
    ret = newdir \== ''			/* Return true if valid and accessible */
  return ret

/* end IsDir */

/*=== IsFile(file) return true if arg is file and file exists ===*/

IsFile: procedure expose (Globals)
  parse arg file
  if file == '' then
    yes = 0
  else do
    /* '.' and '..' returns files in '.' or '..' - so avoid false positives */
    call SysFileTree file, 'files', 'F'
    if RESULT \= 0 then
      call Die 'SysFileTree' file 'failed'
    /* Assume caller knows if arg contains wildcards */
    yes = file \== '.' & file \== '..' & files.0 \= 0
  end
  return yes

/* end IsFile */

/*=== MakePath(pathparts,...) Make path name from parts ===*/

MakePath: procedure

  /* All parts optional - code guesses what caller means.
     If last arg begins with a dot and is not .. and does not
     contain a slash, it is assumed to be a file extension.
     To avoid this behavior, pass empty arg as last arg.
     Empty args are ignored.
     Automatically converts unix slashes to dos slashes.
     If 1st arg is drive letter, it must have trailing colon.
   */

  argCnt = arg()

  path = ''

  do argNum = 1 to argCnt
    s = arg(argNum)
    s = translate(s, '\', '/')		/* Ensure DOS */
    if s == '' & argNum = argCnt then
      iterate				/* Ignore nul last arg */
    if argNum = 1 then
      path = s
    else do
      lead = left(s, 1)
      tail = right(path, 1)
      if tail == ':' & argNum = 2 then
	path = path || s		/* Append path part to drive spec */
      else if lead == '.' & argNum = argCnt & s \== '..' & pos('\', s) = 0  then
	path = path || s		/* Assume extension unless .. or contains \ */
      else if tail == '\' & lead == '\' then
	path = path || substr(s, 2)	/* Drop extra backslash */
      else if path \== '' & tail \== '\' & lead \== '\' then
	path = path || '\' || s		/* Ensure have backslash */
      else
	path = path || s
    end
  end /* for */

  return path

/* end MakePath */

/*=== Plural(cnt, units, suffix, suffix1, no) Return formatted cnt and units ===*/

/**
 * @param cnt is non-negative quantity
 * @param units is unit of measure for quantity 1
 * @param suffix is optional suffix for quantities other than 1, defaults to es or s
 * @param suffix1 is optional suffix for quantity 1, defaults to empty string
 * @param no is optional override for 0 cnt display, defaults to No
 * @notes tries to guess right suffix if suffix omitted
 */

Plural: procedure expose (Globals)

  parse arg cnt, units, suffix, suffix1, no

  if cnt = 1 then do
    if suffix1 \== '' then
      units = units || suffix1
  end
  else do
    /* Not 1 */
    select
    when suffix \== '' then
      units = units || suffix
    when right(units, 1) == 's' then
      units = units || 'es'
    when right(units, 2) == 'ch' then
      units = units || 'es'
    otherwise
      units = units || 's'
    end
  end

  if cnt = 0 then do
    if no == '' then
      no = 'No'
    s = no units
  end
  else
    s = cnt units

  return s

/* end Plural */

/*=== ToLower(s) Convert to lower case ===*/

ToLower: procedure
  parse arg s
  return translate(s, xrange('a', 'z'), xrange('A', 'Z'))

/* end ToLower */

/*=== VerboseMsg([level, ]message,...) Write multi-line message to STDERR if verbose ===*/

VerboseMsg: procedure expose (Globals)
  /* Requires gVerbose */
  level = arg(1)
  if datatype(level, 'W') then
    start = 2
  else do
    level = 1
    start = 1
  end
   if level <= gVerbose then do
    do i = start to arg()
      call lineout 'STDERR', arg(i)
    end
  end
  return

/* end VerboseMsg */

/*=== WarnMsg(message,...) Write multi-line warning message to STDERR ===*/

WarnMsg: procedure
  do i = 1 to arg()
    msg = arg(i)
    call lineout 'STDERR', msg
  end
  return

/* end WarnMsg */

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== Die([message,...]) Write multi-line message to STDERR and die ===*/

Die:
  call lineout 'STDERR', ''
  do i = 1 to arg()
    call lineout 'STDERR', arg(i)
  end
  call lineout 'STDERR', gCmdName 'aborting at line' SIGL || '.'
  call beep 200, 300
  call SysSleep 2
  exit 254

/* end Die */

/*=== Error() Set gErrCondition; report to STDOUT; trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  gErrCondition = condition('C')
  say gErrCondition 'signaled at line' SIGL 'of' cmd || '.'
  if condition('D') \== '' then
    say 'REXX reason =' condition('D') || '.'
  if gErrCondition == 'SYNTAX' & symbol('RC') == 'VAR' then
    say 'REXX error =' RC '-' errortext(RC) || '.'
  else if symbol('RC') == 'VAR' then
    say 'RC =' RC || '.'
  say 'Source =' sourceline(SIGL)

  if condition('I') \== 'CALL' | gErrCondition == 'NOVALUE' | gErrCondition == 'SYNTAX' then do
    trace '?A'
    say 'Enter REXX commands to debug failure.  Press enter to exit script.'
    nop
    if symbol('RC') \== 'VAR' then
      RC = 255
    exit RC
  end

  return

/* end Error */

/*=== Halt() Report HALT condition to STDOUT and exit ===*/

Halt:
  say
  parse source . . cmd
  say condition('C') 'signaled at' cmd 'line' SIGL || '.'
  say 'Source =' sourceline(SIGL)
  say 'Sleeping for 2 seconds...'
  call SysSleep 2
  exit 253

/* end Halt */

/*=== LoadRexxUtil() Load RexxUtil functions ===*/

LoadRexxUtil:
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then
      call Die 'Cannot load SysLoadFuncs.'
    call SysLoadFuncs
  end
  return

/* end LoadRexxUtil */

/*=== ScanArgs(cmdLine) Scan command line ===*/

ScanArgs: procedure expose (Globals)

  /* Calls user exits to process arguments and switches */

  parse arg cmdTail
  cmdTail = strip(cmdTail)

  call ScanArgsInit

  /* Ensure optional settings initialized */
  if symbol('SWCTL') \== 'VAR' then
    swCtl = ''				/* Switches that take args, append ? if optional */
  if symbol('KEEPQUOTED') \== 'VAR' then
    keepQuoted = 0			/* Set to 1 to keep arguments quoted */

  /* Scan */
  curArg = ''				/* Current arg string */
  curSwList = ''			/* Current switch list */
  /* curSwArg = '' */			/* Current switch argument, if needed */
  noMoreSw = 0				/* End of switches */

  do while cmdTail \== '' | curArg \== '' | curSwList \== ''

    /* If arg buffer empty, refill */
    if curArg == '' then do
      qChar = left(cmdTail, 1)		/* Remember quote */
      if \ verify(qChar,'''"', 'M') then
	parse var cmdTail curArg cmdTail	/* Not quoted */
      else do
	/* Arg is quoted */
	curArg = ''
	do forever
	  /* Parse dropping quotes */
	  parse var cmdTail (qChar)quotedPart(qChar) cmdTail
	  curArg = curArg || quotedPart
	  /* Check for escaped quote within quoted string (i.e. "" or '') */
	  if left(cmdTail, 1) \== qChar then do
	    cmdTail = strip(cmdTail)	/* Strip leading whitespace */
	    leave			/* Done with this quoted arg */
	  end
	  curArg = curArg || qChar	/* Append quote */
	  if keepQuoted then
	    curArg = curArg || qChar	/* Append escaped quote */
	  parse var cmdTail (qChar) cmdTail	/* Strip quote */
	end /* do forever */
	if keepQuoted then
	  curArg = qChar || curArg || qChar	/* requote */
      end /* if quoted */
    end /* if curArg empty */

    /* If switch buffer empty, refill */
    if curSwList == '' & \ noMoreSw then do
      if left(curArg, 1) == '-' & curArg \== '-' then do
	if curArg == '--' then
	  noMoreSw = 1
	else
	  curSwList = substr(curArg, 2)	/* Remember switch string */
	curArg = ''			/* Mark empty */
	iterate				/* Refill arg buffer */
      end /* if switch */
    end /* if curSwList empty */

    /* If switch in progress */
    if curSwList \== '' then do
      curSw = left(curSwList, 1)	/* Next switch */
      curSwList = substr(curSwList, 2)	/* Drop from pending */
      /* Check switch allows argument, avoid matching ? */
      if pos(curSw, translate(swCtl,,'?')) \= 0 then do
	if curSwList \== '' then do
	  curSwArg = curSwList		/* Use rest of switch string for switch argument */
	  curSwList = ''
	end
	else if curArg \== '' & left(curArg, 1) \== '-' then do
	  curSwArg = curArg		/* Arg string is switch argument */
	  curArg = ''			/* Mark arg string empty */
	end
	else if pos(curSw'?', swCtl) = 0 then
	  call ScanArgsUsage 'Switch "-' || curSw || '" requires an argument'
	else
	  curSwArg = ''			/* Optional arg omitted */
      end

      call ScanArgsSwitch		/* Passing curSw and curSwArg */
      drop curSwArg			/* Must be used by now */
    end /* if switch */

    /* If arg */
    else if curArg \== '' then do
      noMoreSw = 1
      call ScanArgsArg			/* Passing curArg */
      curArg = ''
    end

  end /* while not done */

  call ScanArgsTerm

  return

/* end ScanArgs */

/*=== SetCmdName() Set gCmdName to short script name ===*/

SetCmdName: procedure expose (Globals)
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  gCmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end SetCmdName */

/* eof */
