/* bounds-l.h -- changed for emx by Eberhard Mattes -- Nov 1998 */
/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rjones@orchestream.com>.	*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	lib/bounds-lib.h
 * Summary:
 *	Header file for the bounds checking library.
 * Other notes:
 *	
 * Author      	Date		Notes
 * RWMJ		4/12/94		Initial implementation.
 * RWMJ		24/1/94		Added inline `is_.._pointer' functions.
 *----------------------------------------------------------------------*/

#ifndef _BOUNDS_LIB_H_
#define _BOUNDS_LIB_H_

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include "objects.h"
#include "alloca.h"
#include "functions.h"
#include "version.h"

#ifdef __vxworks
typedef unsigned size_t;
#endif

typedef int bool;			/* Return type of <, >, etc. */

/* These macros should be used to cast between pointers & unsigned integers.
 * Notice that on some machines, sizeof (int) != sizeof (pointer), so we
 * should take care of that here too.
 */
typedef unsigned long ptr_as_unsigned_t;
#define PTR_TO_UNSIGNED(p) ((ptr_as_unsigned_t)(p))
#define UNSIGNED_TO_PTR(u) ((void *)(u))

/* Constants.
 */
#define ILLEGAL		((void *)-2)	/* Bad ptr. arithmetic returns this. */
#define COLLECT_STATS	0		/* Collect statistics on call freq. */
#define USE_CACHE	0		/* Cache object lookups (don't use). */
#define DEBUG_FEATURES	1		/* Various extra features. */

/* Global variables defined by the compiler when bounds checking has been
 * enabled.
 */
extern int __bounds_array_index_check;	/* Check array indexes. */
extern int __bounds_never_free_heap;	/* Heap allocation mode. */
extern int __bounds_age_limit;		/* Age limit before freeing. */
extern int __bounds_debug_print_calls;	/* Print calls to bounds library. */
extern int __bounds_debug_print_heap;	/* Print heap. */
extern int __bounds_debug_no_checking;	/* Disable checking of pointers. */
extern int __bounds_checking_on;	/* Bounds checking is on (in ___main)*/
extern int __bounds_warn_unchecked_statics; /* Warn if unchecked statics used*/
extern int __bounds_warn_unchecked_stack; /* Warn if unchecked stack used*/
extern int __bounds_warn_free_null;	/* Warn if free(0) is called. */
extern int __bounds_warn_misc_strings;  /* Warn for miscellaneous string
					 * nonsense from X11 and GNU libc. */
extern int __bounds_warn_illegal;	/* Warn if illegal pointers are made.*/
extern int __bounds_warn_unaligned;	/* Warn if pointers used unaligned. */
extern int __bounds_never_fatal;	/* Don't abort on error. */

#if COLLECT_STATS
/* Global variables for collecting call frequency statistics.
 */
extern int  __bounds_stats_ptr_plus_int;
extern int  __bounds_stats_array_reference;
extern int  __bounds_stats_ptr_diff;
extern int  __bounds_stats_reference;
extern int  __bounds_stats_component_reference;
extern int  __bounds_stats_ptr_le_ptr;
extern int  __bounds_stats_ptr_lt_ptr;
extern int  __bounds_stats_ptr_ge_ptr;
extern int  __bounds_stats_ptr_gt_ptr;
extern int  __bounds_stats_ptr_ne_ptr;
extern int  __bounds_stats_ptr_eq_ptr;
extern int  __bounds_stats_ptr_postinc;
extern int  __bounds_stats_ptr_preinc;
extern int  __bounds_stats_ptr_postdec;
extern int  __bounds_stats_ptr_predec;
extern int  __bounds_stats_ptr_true;
extern int  __bounds_stats_ptr_false;
extern int  __bounds_stats_push_function;
extern int  __bounds_stats_pop_function;
extern int  __bounds_stats_param_function;
extern int  __bounds_stats_environment;
extern int  __bounds_stats_add_stack;
extern int  __bounds_stats_delete_stack;
extern int  __bounds_stats_add_heap;
extern int  __bounds_stats_delete_heap;
extern int  __bounds_stats_unchecked_stack;
extern int  __bounds_stats_unchecked_static;

#endif /* COLLECT_STATS */

/* `etext' is the first byte of the data segment. `edata' is the first byte of
 * the bss (uninitialized data). `end' is the first byte of the heap.
 */
extern char etext, edata, end;

/* Format of the private table passed by GCC listing all unnamed static
 * objects in a source file.
 */
typedef struct {
  void *ptr;				/* Pointer to object. */
  void *size;				/* Size (bytes) of object. */
  char *name;				/* Name of the object (NULL=unnamed) */
} private_table;

/* Format of the table passed by grab-statics utility to the library, listing
 * various named statics in external libraries and object files.
 * N.B.: This table is shared with `tools/find-objects.c'
 */
typedef struct {
  void *address;			/* Object address */
  size_t size;				/* Size (bytes) */
  size_t align;				/* Alignment */
  char *filename;			/* Filename where found */
  int line;				/* Line where found */
  char *name;				/* Name of object */
} external_statics_table;

/* Prototypes of functions in this library.
 */

/* Initialize the library. This is called as the first constructor function. */
/* (init.c) */

void  __bounds_initialize_library (void);

/* Initialize the cache. This only exists if USE_CACHE is set (above). */
/* (objects.c) */

void  __bounds_initialize_cache (void);

/* Allocate/deallocate memory dynamically, but without checking. */
/* (malloc.c) */

void* __bounds_memalign (size_t, size_t);
void* __bounds_malloc (size_t);
void  __bounds_free (void *);
void* __bounds_realloc (void *, size_t);
void* __bounds_calloc (size_t, size_t);
void  __bounds_cfree (void *);
void* __bounds_valloc (size_t);

/* Notify, check and delete heap objects. Used by new malloc/free. */
/* (objects.c) */

void  __bounds_add_heap_object (void *base, size_t count, size_t align,
				char *name, char *filename, int line);
int   __bounds_is_heap_object (void *base);
void  __bounds_delete_heap_object (void *base);

/* Notify, check and delete stack objects. Used on entry/exit to functions. */
/* (objects.c) */

void  __bounds_add_stack_object (void *base, size_t count, size_t align,
				 char *filename, int line, char *name);
void  __bounds_delete_stack_object (void *base);

/* Internal versions of the above, which work with/return object pointers. */
/* (objects.c) */

object*__bounds_internal_add_stack_object (void *base, size_t count,
					   size_t align, char *filename,
					   int line, char *name,
					   int no_padding);
void  __bounds_internal_delete_stack_object (object *obj);

/* Find the object referred to by a pointer. */
/* (objects.c) */

object* __bounds_find_object         (void *pointer);
object* __bounds_find_object_by_base (void *base);

/* Notify of a constructed static object in the text or data segment. */
/* (objects.c) */

void  __bounds_note_constructed_object (void *pointer, int size,
					size_t align,
					char *filename, int line,
					char *name);
void  __bounds_note_constructed_private_table (private_table *,
					       char *filename, int line);

/* This is the new form of `__bounds_note_constructed_private_table'.
 * Eventually all calls to the above will be replaced with this, which
 * makes more stringent checks.
 * (exstatics.c)
 */
void  __bounds_add_static_objects_table (external_statics_table *);

/* Notify the library of the arguments to main (). */
/* (objects.c) */

void  __bounds_note_main_args (int argc, char **argv);

/* Push and pop in function contexts. */
/* (functions.c) */

void  __bounds_push_function (char *name, int in_main, int nr_args,
			      char *filename, int line);
void  __bounds_pop_function (char *name);

/* Add a function parameter for the current function context. */
/* (functions.c) */

void  __bounds_add_param_object (void *ptr, size_t size, size_t align,
				 char *filename, int line, char *name);

/* Add an alloca allocation to the current function context. */
/* (functions.c) */

char *__bounds_add_alloca_chain (alloca_item *);

/* Remove all the alloca allocations for this function context. */
/* (alloca.c) */

void  __bounds_alloca_free (alloca_item *alloca_stack);

/* Check pointer arithmetic. These never return if they fail. */
/* (check.c) */

void* __bounds_check_ptr_plus_int (void *pointer, int offset, size_t size,
				   int is_plus, char *filename, int line);
int   __bounds_check_ptr_diff (void *pointer1, void *pointer2, size_t size,
			       char *filename, int line);

/* Check array reference. This is almost the same as ptr_plus_int. */
/* (check.c) */

int __bounds_check_array_reference (void *pointer, int offset, size_t size,
				      size_t array_size, char *filename, int line);

/* Check component references ('->'). */
/* (check.c) */

void* __bounds_check_component_reference (void *pointer, int offset,
					  int size, char *filename,
					  int line);

/* Check pointer references. These never return if they fail. */
/* (check.c) */

void* __bounds_check_reference (void *pointer, size_t size,
				char *filename, int line);

/* Check pointer truthvalue or falsity. These never return if they fail. */
/* (check.c) */

bool  __bounds_check_ptr_true (void *pointer, char *filename, int line);
bool  __bounds_check_ptr_false (void *pointer, char *filename, int line);

/* Check pointer equality and inequality operations. These never return if
 * they fail. */
/* (check.c) */

bool  __bounds_check_ptr_le_ptr (void *pointer1, void *pointer2,
				 char *filename, int line);
bool  __bounds_check_ptr_lt_ptr (void *pointer1, void *pointer2,
				 char *filename, int line);
bool  __bounds_check_ptr_ge_ptr (void *pointer1, void *pointer2,
				 char *filename, int line);
bool  __bounds_check_ptr_gt_ptr (void *pointer1, void *pointer2,
				 char *filename, int line);
bool  __bounds_check_ptr_ne_ptr (void *pointer1, void *pointer2,
				 char *filename, int line);
bool  __bounds_check_ptr_eq_ptr (void *pointer1, void *pointer2,
				 char *filename, int line);

/* Pre and post increment and decrement functions. */
/* (check.c) */

void* __bounds_check_ptr_postinc (void **ptr_to_ptr, int inc, char *filename,
				  int line);
void* __bounds_check_ptr_preinc  (void **ptr_to_ptr, int inc, char *filename,
				  int line);
void* __bounds_check_ptr_postdec (void **ptr_to_ptr, int inc, char *filename,
				  int line);
void* __bounds_check_ptr_predec  (void **ptr_to_ptr, int inc, char *filename,
				  int line);

/* Error reporting functions. */
/* (error.c) */

void  __bounds_error (char *message,
		      char *filename, int line,
		      void *pointer,
		      object *obj)
      /* __attribute__ ((noreturn)) */;
void  __bounds_error2(char *message,
		      char *filename, int line,
		      void *pointer1, void *pointer2,
		      object *obj1, object *obj2)
      /* __attribute__ ((noreturn)) */;
void  __bounds_errorf(char *filename, int line,
		      void *pointer, object *obj,
		      char *message, ...)
      /* __attribute__ ((noreturn)) */;
void  __bounds_internal_error (char *message, char *filename, int line)
  __attribute__ ((noreturn));

/* Warning functions. */
/* (error.c) */

void  __bounds_warning (char *filename, int line, char *function, char *message, ...);
void  __bounds_warn1 (char *message,
		      char *filename, int line,
		      void *pointer,
		      object *obj);
void  __bounds_warnf(char *filename, int line,
		      void *pointer, object *obj,
		      char *message, ...);

/* Breakpoint for external debuggers. */
/* (error.c) */

void  __bounds_breakpoint (void);

/* Debugging functions.
 */

void  __bounds_debug_memory (void*,void*);/* (objects.c) Produce memory map. */
void  __bounds_debug_stack_trace (void);  /* (functions.c) Stack trace. */
void  __bounds_no_checking_in_this_function (void);	/* (functions.c) */
void  __bounds_tree_check (tree*);	  /* (objects.c) Check tree is OK. */
void  __bound_print_statistics (void);	  /* (objects.c) print statistics */
void  __bound_print_heap (void);	  /* (objects.c) print heap */


/* Message printing functions (normal ones may call malloc). */
/* (print.c) */

#if !defined(__osf__)
void  __bounds_printf (char *format, ...);
void  __bounds_vprintf (char *format, va_list list);

#define printf __bounds_printf
#define vprintf __bounds_vprintf
#endif /* ! OSF */

/* Unchecked versions of the string functions. */
/* (string.c) */

void* __bounds_memcpy (void *, const void *, size_t);
void* __bounds_memmove (void *, const void *, size_t);
void* __bounds_memccpy (void *, const void *, int, size_t);
void* __bounds_memset (void *, int, size_t);
int   __bounds_memcmp (const void *, const void *, size_t);
void* __bounds_memchr (const void *, int, size_t);
char* __bounds_strcpy (char *, const char *);
size_t  __bounds_strlen (const char *);
int   __bounds_strcmp (const char *, const char *);
int   __bounds_strncmp (const char *, const char *, size_t);

/* I always call `abort' in the case of a bounds error. Sometimes `exit' tries
 * to allocate memory for some reason, so `abort' is safer.
 */
#define ABORT() abort()

#endif /* _BOUNDS_LIB_H_ */
