/* pow.c (emx+gcc) */

#include <stdio.h>
#include <math.h>
#include <float.h>
#include <emx/float.h>

static double nan = 0.0 / 0.0;
static double inf = 1e9999999;
static int term;

static int fxam (double x)
{
  int f;
  f = _fxam (x);
  if (f == FX_N_NAN)
    f = FX_P_NAN;
  return f;
}


static void test (const char *txt, double x, double y, double exp,
                  unsigned ex_exp)
{
  double res;
  int fxam_exp, fxam_res;
  unsigned ex_res;

  _clear87 ();
  res = pow (x, y);
  ex_res = _status87 () & (EM_INVALID | EM_ZERODIVIDE);
  fxam_res = fxam (res);
  fxam_exp = fxam (exp);

  if (fxam_res != fxam_exp || (fxam_res != FX_P_NAN && res != exp)
      || ex_res != ex_exp)
    {
      printf ("Error in test %s: x=%g[%d], y=%g[%d]\n"
              "  res=%g[%d], exp=%g[%d], ex_res=%#x, ex_exp=%#x\n",
              txt, x, fxam (x), y, fxam (y), res, fxam_res, exp, fxam_exp,
              ex_res, ex_exp);
      term = 1;
    }
}


static int fxaml (long double x)
{
  int f;
  f = _fxaml (x);
  if (f == FX_N_NAN)
    f = FX_P_NAN;
  return f;
}


static void testl (const char *txt, long double x, long double y,
                   long double exp, unsigned ex_exp)
{
  long double res;
  int fxam_exp, fxam_res;
  unsigned ex_res;

  __asm__ ("fclex");
  res = _powl (x, y);
  ex_res = _status87 () & (EM_INVALID | EM_ZERODIVIDE);
  fxam_res = fxaml (res);
  fxam_exp = fxaml (exp);

  if (fxam_res != fxam_exp || (fxam_res != FX_P_NAN && res != exp)
      || ex_res != ex_exp)
    {
      printf ("Error in test %s: x=%Lg[%d], y=%Lg[%d]\n"
              "  res=%Lg[%d], exp=%Lg[%d], ex_res=%#x, ex_exp=%#x\n",
              txt, x, fxam (x), y, fxam (y), res, fxam_res, exp, fxam_exp,
              ex_res, ex_exp);
      term = 1;
    }
}


static void test_all (void)
{
  test ("0a",  2.0,      3.0,  8.0,   0);
  test ("0b",  2.0,     -3.0,  0.125, 0);
  test ("0c", -2.0,      3.0, -8.0,   0);
  test ("0d", -2.0,      4.0, 16.0,   0);
  test ("0e",  2.1,    1e300,  inf,   0);
  test ("0f",  2.1,  DBL_MAX,  inf,   0);
  test ("0g", -2.1,    1e300,  inf,   0);
  test ("0h", -2.1,  DBL_MAX,  inf,   0);
  test ("0i",  0.1,    1e300,  0.0,   0);
  test ("0j",  0.1,  DBL_MAX,  0.0,   0);
  test ("0k", -0.1,    1e300,  0.0,   0);
  test ("0l", -0.1,  DBL_MAX,  0.0,   0);
  test ("0m",  2.1,   -1e300,  0.0,   0);
  test ("0n",  2.1, -DBL_MAX,  0.0,   0);
  test ("0o", -2.1,   -1e300,  0.0,   0);
  test ("0p", -2.1, -DBL_MAX,  0.0,   0);
  test ("0q",  0.1,   -1e300,  inf,   0);
  test ("0r",  0.1, -DBL_MAX,  inf,   0);
  test ("0s", -0.1,   -1e300,  inf,   0);
  test ("0t", -0.1, -DBL_MAX,  inf,   0);
  test ("0u",  0.5,      3.0,  0.125, 0);
  test ("0v",  0.5,     -3.0,  8.0,   0);
  test ("0w", -0.5,      3.0, -0.125, 0);
  test ("0x", -0.5,     -3.0, -8.0,   0);

  test ("1a",  0.0, 0.0, 1.0, 0);
  test ("1b",  1.7, 0.0, 1.0, 0);
  test ("1c", -1.7, 0.0, 1.0, 0);
  test ("1d", -inf, 0.0, 1.0, 0);
  test ("1e",  inf, 0.0, 1.0, 0);
  test ("1f",  nan, 0.0, 1.0, 0);

  test ("2a",  1.000001, inf, inf, 0);
  test ("2b", -1.000001, inf, inf, 0);
  test ("2c",     42.17, inf, inf, 0);
  test ("2d",    -17.42, inf, inf, 0);
  test ("2e",       inf, inf, inf, 0);
  test ("2f",      -inf, inf, inf, 0);

  test ("3a",  0.0,    inf, 0.0, 0);
  test ("3b",  0.9999, inf, 0.0, 0);
  test ("3c", -0.9999, inf, 0.0, 0);

  test ("4a",  1.000001, -inf, 0.0, 0);
  test ("4b", -1.000001, -inf, 0.0, 0);
  test ("4c",     42.17, -inf, 0.0, 0);
  test ("4d",    -17.42, -inf, 0.0, 0);
  test ("4e",       inf, -inf, 0.0, 0);
  test ("4f",      -inf, -inf, 0.0, 0);

  test ("5a",  0.0,    -inf, inf, 0);
  test ("5b",  0.9999, -inf, inf, 0);
  test ("5c", -0.9999, -inf, inf, 0);

  test ("6a", inf, 0.00001, inf, 0);
  test ("6b", inf,   42.17, inf, 0);
  test ("6c", inf,     inf, inf, 0);

  test ("7a", inf, -0.00001, 0.0, 0);
  test ("7b", inf,   -42.17, 0.0, 0);
  test ("7c", inf,     -inf, 0.0, 0);

  test ("8a", -inf,      1.0, -inf, 0);
  test ("8b", -inf,     17.0, -inf, 0);
  test ("8c", -inf, 777777.0, -inf, 0);

  test ("9a", -inf,      1.1,      inf, 0);
  test ("9b", -inf, 777778.0,      inf, 0);
  test ("9c", -inf, 777777.000001, inf, 0);
  test ("9d", -inf,         1e300, inf, 0);
  test ("9e", -inf,       DBL_MAX, inf, 0); /* DBL_MAX is always even */
  test ("9f", -inf,           inf, inf, 0);

  test ("10a", -inf,      -1.0, -0.0, 0);
  test ("10b", -inf, -777777.0, -0.0, 0);

  test ("11a", -inf,      -1.1, 0.0, 0);
  test ("11b", -inf, -777778.0, 0.0, 0);
  test ("11c", -inf,    -1e300, 0.0, 0);
  test ("11d", -inf,  -DBL_MAX, 0.0, 0); /* DBL_MAX is always even */
  test ("11e", -inf,      -inf, 0.0, 0);

  test ("12a",  0.0, nan, nan, 0);
  test ("12b",  1.0, nan, nan, 0);
  test ("12c",  inf, nan, nan, 0);
  test ("12d", -inf, nan, nan, 0);
  test ("12e", nan,  2.0, nan, 0);
  test ("12f", nan,  inf, nan, 0);
  test ("12g", nan, -inf, nan, 0);
  test ("12h", nan,  nan, nan, 0);

  test ("13a",  1.0,  inf, nan, EM_INVALID);
  test ("13b", -1.0,  inf, nan, EM_INVALID);
  test ("13c",  1.0, -inf, nan, EM_INVALID);
  test ("13d", -1.0, -inf, nan, EM_INVALID);

  test ("14a", -3.0,     2.5,    nan, EM_INVALID);
  test ("14b", -0.0001, -1.0001, nan, EM_INVALID);

  test ("15a",  0.0, -77777.0, inf,  EM_ZERODIVIDE);
  test ("15b", -0.0, -77777.0, -inf, EM_ZERODIVIDE);

  test ("16a", 0.0, -77777.1, inf, EM_ZERODIVIDE);
  test ("16b", 0.0, -77778.0, inf, EM_ZERODIVIDE);

  test ("17a",  0.0, 77777.0,  0.0, 0);
  test ("17b", -0.0, 77777.0, -0.0, 0);

  test ("18a",  0.0,    77.1, 0.0, 0);
  test ("18b", -0.0, 77778.0, 0.0, 0);

  /* TODO: Check overflow/underflow exception */
  testl ("19a", LDBL_MIN, LDBL_MAX, 0.0, 0);
  testl ("19b", -LDBL_MIN, LDBL_MAX, 0.0, 0); /* LDBL_MAX is always even */
  testl ("19c", LDBL_MAX, LDBL_MAX, inf, 0);
  testl ("19d", -LDBL_MAX, LDBL_MAX, inf, 0); /* LDBL_MAX is always even */
  testl ("19e", LDBL_MIN, LDBL_MIN, 1.0, 0);
  testl ("19f", LDBL_MAX, LDBL_MIN, 1.0, 0);
  testl ("19g", LDBL_MAX, 3.0, inf, 0);
  testl ("19h", -LDBL_MAX, 3.0, -inf, 0);
  testl ("19i", LDBL_MAX, 4.0, inf, 0);
  testl ("19j", -LDBL_MAX, 4.0, inf, 0);
  testl ("19k", 3.0, LDBL_MAX, inf, 0);
  testl ("19l", -3.0, LDBL_MAX, inf, 0); /* LDBL_MAX is always even */
}


int main (void)
{
  int i;

  /* Run the tests multiple times to detect FP stack overflow. */

  term = 0;
  for (i = 0; i < 20 && term == 0; ++i)
    test_all ();
  return term;
}
