/*
  ln4exe - generate forwarder for OS/2 executables

  This program is public domain.
  You can use/modify/redistribute it freely but NO WARRANTY.
*/


#define INCL_DOS
#define INCL_DOSERRORS
#include <os2.h>

#include <sys/types.h>
#include <fcntl.h>
#include <io.h>
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <process.h>
#include <unistd.h>

#ifndef PATH_MAX
#define PATH_MAX 260
#endif

#define PATHFIELD_LENGTH (PATH_MAX+32)

#define FORWARDING_LENGTH (6+18+1)
static char szForwardingPathname[PATHFIELD_LENGTH] = \
  { PATHFIELD_LENGTH & 0xff,
    (PATHFIELD_LENGTH >> 8) & 0xff,
    (PATHFIELD_LENGTH >> 16) & 0xff,
    (PATHFIELD_LENGTH >> 24) & 0xff,
     'l','n','4','e','x','e',
     'F','O','R','W','A','R','D','I','N','G','P','A','T','H','N','A','M','E',
     '=',
     0
  };
#if 0
#define BASE_LENGTH (6+12+1)
static char szBasePathname[PATHFIELD_LENGTH] = \
  { PATHFIELD_LENGTH & 0xff,
    (PATHFIELD_LENGTH >> 8) & 0xff,
    (PATHFIELD_LENGTH >> 16) & 0xff,
    (PATHFIELD_LENGTH >> 24) & 0xff,
     'l','n','4','e','x','e',
     'B','A','S','E','P','A','T','H','N','A','M','E',
     '=',
     0
  };
#endif

static char *progname;
static unsigned char *progimg;
static size_t progimgsize;
static size_t fwdpn_len;
static char *fwdpn;

static int optHelp;
static int opt_v;
static int opt_s;
static int opt_f;

static char *srcfn;
static char *dstfn;

#if !defined(__EMX__)
static void _fnslashify(char *s) {
  char c;
  while((c = *s) != '\0') {
    if (c == '\\') *s = '/';
    s++; /* todo: treat DBCS */
  }
}
#endif
static void _fnbackslashify(char *s)
{
  char c;
  while((c = *s) != '\0') {
    if (c == '/') *s = '\\';
    s++;
  }
}

int
mygetopt(int argc, char **argv)
{
  char c, *s;
  
  while(argc > 0) {
    s = *argv;
    if (*s == '-') {
      c = s[1];
      if (c == 'v') opt_v = 1;
      else if (c == 'f') opt_f = 1;
      else if (c == 's') opt_s = 1;
      else if (c == 'h' || c == 'H' || c == '?')
        optHelp = 1;
    }
    else if (!srcfn) srcfn = s;
    else if (!dstfn) dstfn = s;
    --argc;
    ++argv;
  }
  
  return ((srcfn && dstfn) || optHelp) ? 0 : -1;
}

static
unsigned long
peekdw_le(const void *p)
{
  unsigned char *b = (unsigned char *)p;
  
  return *b | 
         ((unsigned long)b[1] << 8) |
         ((unsigned long)b[2] << 16) |
         ((unsigned long)b[3] << 24);
}

static
void *
mymemmem(const void *img, size_t imglen, const void *key, size_t keylen)
{
  char *p;
  if (!img || !key || imglen < keylen)
    return NULL;
  if (keylen == 0)
    return (void *)img;
  
  for(p=(char *)img; imglen>=keylen; imglen++, p++) {
    if (memcmp(p, key, keylen) == 0)
      return (void *)p;
  }
  
  return NULL;
}

static
char *
diff_path(const char *src, const char *dst, int *same_dir)
{
  APIRET rc;
  char *sful;
  char *dful;
  char c, cprev;
  size_t dirbase, dirtb, i;
  char *rp;
  
  sful = alloca(PATH_MAX + 1);
  dful = alloca(PATH_MAX + 1);
  rc = DosQueryPathInfo((PSZ)src, FIL_QUERYFULLNAME, sful, PATH_MAX);
  if (rc == 0)
    rc = DosQueryPathInfo((PSZ)dst, FIL_QUERYFULLNAME, dful, PATH_MAX);
  if (rc) {
    fprintf(stderr, "ln4exe: incorrect pathname.\n");
    return NULL;
  }
  _fnslashify(sful);
  _fnslashify(dful);
  
  for(i=0, dirbase=0; sful[i] && dful[i]; i++) {
    if (sful[i] != dful[i]) break;
    if (sful[i] == '/') dirbase = i + 1;
  }
  
#if 0
  if (dirbase == 0) {
    fprintf(stderr, "ln4exe: destination drive is not same as source drive.\n");
    return NULL;
  }
#endif

  for(dirtb=0, i=dirbase, cprev='\0' ;(c = sful[i]) != '\0' ; i++) {
    if (c == '/' && c != cprev)
      dirtb++;
    cprev = c;
  }

  rp = malloc(PATH_MAX * 2 + 1);
  if (rp) {
     /* todo: rewrite with strlcat() */
    *rp = '\0';
    while(dirtb--) {
      strcat(rp, "../");
    }
    strcat(rp, dful + dirbase);
  }
  
  if (same_dir) {
    *same_dir = strchr(rp, '/') == NULL;
  }

  return rp;
}

char *
gen_forwarding_pathname(const char *progfullpath, const char *relpathname)
{
  APIRET rc;
  size_t n;
  char *fn, *fn_fp, *p;
  
  fn = alloca(PATH_MAX * 2 + 1);
  fn_fp = malloc(PATH_MAX + 1);
  if (!fn_fp) return NULL;
  
  strcpy(fn, progfullpath);
  if ((p = strrchr(fn, '/')) != NULL) p[1] = '\0';
  n = strlen(fn);
  if (n > 0 && fn[n-1] != '/') {
    fn[n++] = '/';
    fn[n] = '\0';
  }
  strcat(fn, relpathname);
  rc = DosQueryPathInfo((PSZ)fn, FIL_QUERYFULLNAME, fn_fp, PATH_MAX);
  if (rc) {
    free(fn_fp);
    fn_fp = NULL;
  }
  _fnslashify(fn_fp);
  
  return fn_fp;
}


#define FILE_RWCNT 0x4000

static
void *
load_file_to_mem(const char *filename, size_t *prog_length)
{
  FILE *fi;
  off_t len, n, n_once;
  char *p;
  fi = fopen(filename, "rb");
  if (!fi) return NULL;
  if (fseek(fi, 0, SEEK_END) < 0) return NULL;
  len = ftell(fi);
  if (len == (off_t)-1) return NULL;
  fseek(fi, 0, SEEK_SET);
  p = malloc(len);
  if (p) {
    for(n=0; n<len;) {
      n_once = len - n;
      if (n_once > FILE_RWCNT) n_once = FILE_RWCNT;
      n_once = fread((char *)p + n, 1, n_once, fi);
      if (n_once <= 0) break;
      n += n_once;
    }
    if (prog_length) *prog_length = n;
  }
  fclose(fi);
  return (void *)p;
}

static
int
save_mem_to_file(const char *filename, const void *img, size_t img_length)
{
  int rc;
  FILE *fo;
  off_t n, n_once;
  fo = fopen(filename, "wb");
  if (!fo) return -1;
  rc = 0;
  for(n=0; n<img_length;) {
    n_once = img_length - n;
    if (n_once > FILE_RWCNT) n_once = FILE_RWCNT;
    n_once = fwrite((char *)img + n, 1, n_once, fo);
    if (n_once <= 0) {
      rc = -1;
      break;
    }
    n += n_once;
  }
  fclose(fo);
  return rc;
}


static
int
do_forward(int argc, char **argv)
{
  int rc;
  char *fn;
  
  fn = gen_forwarding_pathname(progname, fwdpn + FORWARDING_LENGTH);
  if (!fn) {
    fprintf(stderr, "ln4exe: incorrect forwarding_pathname '%s'.\n", fwdpn + FORWARDING_LENGTH
);
    exit(127);
  }
  rc = spawnv(P_WAIT, fn, argv);
  
  if (rc == -1) {
    fprintf(stderr, "ln4exe: can't exec/spawn '%s'.\n", fn);
  }
  
  return rc;
}

static
int do_link(const char *srcfn, const char *dstfn, int always_symlink)
{
  APIRET rc;
  char *sful, *dful;
  char *dp;
  int same_dir;
  
  sful = alloca(PATH_MAX + 1);
  dful = alloca(PATH_MAX + 1);
  rc = DosQueryPathInfo((PSZ)srcfn, FIL_QUERYFULLNAME, sful, PATH_MAX);
  if (rc == 0)
    rc = DosQueryPathInfo((PSZ)dstfn, FIL_QUERYFULLNAME, dful, PATH_MAX);
  if (rc) {
    fprintf(stderr, "ln4exe: incorrect pathname.\n");
    return -1;
  }
  _fnslashify(sful);
  _fnslashify(dful);
  
  dp = diff_path(dful, sful, &same_dir);
  if (!dp)
    return -1;
  if (always_symlink || same_dir) {
    /* generate forwarder */
    if (fwdpn_len <= FORWARDING_LENGTH || strlen(dp) >= fwdpn_len - FORWARDING_LENGTH) {
      fprintf(stderr, "ln4exe: forwarding_pathname too long.\n");
      return -1;
    }
    strcpy(fwdpn + FORWARDING_LENGTH, dp);
    if (save_mem_to_file(dful, progimg, progimgsize) < 0) {
      fprintf(stderr, "ln4exe: can't write '%s'.\n", dstfn);
      return -1;
    }
  }
  else {
    /* just copy */
    _fnbackslashify(sful);
    _fnbackslashify(dful);
    rc = DosCopy(sful, dful, DCPY_EXISTING);
    if (rc) {
      fprintf(stderr, "ln4exe: DosCopy error (rc=%d).\n", rc);
    }
    return rc;
  }
  return 0;
}


int
main(int argc, char **argv)
{
  int rc;
  char *dp;
  
#if defined(__EMX__) && 0
  _response(&argc, &argv);
#endif
  
  progname = malloc(PATHFIELD_LENGTH);
  if (progname) {
    APIRET apirc;
    PPIB ppib;
    PTIB ptib;
    
    apirc = DosGetInfoBlocks(&ptib, &ppib);
    if (apirc == 0)
      apirc = DosQueryModuleName((HMODULE)(ppib->pib_hmte), PATHFIELD_LENGTH - 1, progname);
    if (apirc != 0) {
      fprintf(stderr, "ln4exe: can't get real progname.\n");
      exit(127);
    }
    _fnslashify(progname);
  }

  progimg = load_file_to_mem(progname, &progimgsize);
  if (!progimg) {
    fprintf(stderr, "ln4exe: can't load '%s'.\n", progname);
    exit(127);
  }
  fwdpn = mymemmem(progimg, progimgsize, szForwardingPathname + 4, FORWARDING_LENGTH);
  if (fwdpn) fwdpn_len = peekdw_le((char *)fwdpn - 4);
  if (!fwdpn) {
    fprintf(stderr, "ln4exe: broken dataarea.\n");
    exit(127);
  }
  
  if (fwdpn[FORWARDING_LENGTH])
    return do_forward(argc, argv);
  
  rc = mygetopt(argc - 1, argv + 1);

  if (rc < 0 || optHelp) {
    fprintf(rc < 0 ? stderr : stdout, "usage: ln4exe [-s] srcfile destfile\n");
    return rc < 0 ? 127 : 0;
  }
  
#if 1
  return do_link(srcfn, dstfn, opt_s);
#else
  dp = diff_path(dstfn, srcfn);
  if (!dp) {
    exit(127);
  }
  if (fwdpn_len <= FORWARDING_LENGTH || strlen(dp) >= fwdpn_len - FORWARDING_LENGTH) {
    fprintf(stderr, "ln4exe: forwarding_pathname too long.\n");
    exit(127);
  }
  strcpy(fwdpn + FORWARDING_LENGTH, dp);
  
  if (save_mem_to_file(dstfn, progimg, progimgsize) < 0) {
    fprintf(stderr, "ln4exe: can't write '%s'.\n", dstfn);
    exit(127);
  }
#endif
  
  return 0;
}

