#include <stdio.h>
#include <string.h>

#include <ithread.hpp>

#include "procssta.hpp"
#include "statline.rch"

//********************************************************************************
// AProcessStatus :: AProcessStatus - Constructor for status-line                *
//********************************************************************************
AProcessStatus :: AProcessStatus (IWindow * parentWindow) 
               : IStaticText (WND_STATUS_MAIN, parentWindow, parentWindow)
{
  resetAllVariables ();                     // reset all member variables fist

  IPaintHandler :: handleEventsFor(this);   // set self as paint handler

  AStatusHandler :: handleEventsFor(this);  // set self as status message handler

  setAutoDeleteObject();                    // delete C++ object, if PM object is deleted
}

//********************************************************************************
// AProcessStatus :: ~AProcessStatus - Destructor for status-line                *
//********************************************************************************
AProcessStatus :: ~AProcessStatus (void) 
{
  delete m_pFont;  // delete the allocated font object
}

//********************************************************************************
// AProcessStatus :: resetAllVariables - resets all member variables             *
//********************************************************************************
void AProcessStatus :: resetAllVariables (void) 
{
  m_pFont = 0L;

  m_ulTotalTicks =
  m_ulActualTick =
  m_ulPercentReady =
  m_ulPercent =
  m_ulNewSliderSize =
  m_ulOldSliderSize =
  m_ulY = 0L;

  m_lStartSliderX =
  m_lStartPercentNumberX =
  m_lEndPercentNumberX =
  m_lWidth =
  m_lResourceId = 0L;

  memset (m_szBuffer, 0, sizeof (m_szBuffer));

  m_dSizePerTick = 0.0;
  
  m_nPaintWhat = ePaintStd;

  m_bProcessIsRunning = false;

  m_bPaintSentByMe = false;
}

//********************************************************************************
// AProcessStatus :: paintWindow - control member for paint events               *
//********************************************************************************
Boolean AProcessStatus :: paintWindow (IPaintEvent& evt) 
{
  if(m_bPaintSentByMe)          // did I send the paint event?
     {                          // yes                     
      m_bPaintSentByMe = false; // reset the boolean
      if(!m_bProcessIsRunning)  // is the process to indicate still running
         return (true);         // no: return
      switch(m_nPaintWhat)      // the process is running, what is there to do? 
         {
             case ePaintStartProcess:               // do the painting for process's start
                  return (paintStartProcess (evt));
             case ePaintPercentNumber:              // paint the percent number
                  return (paintPercentNumber (evt));
             case ePaintSlider:                     // paint the slider
                  return (paintSlider (evt));
             default:                               // no special painting
                  return (paintStd (evt));
            } 
     }

  // the paint event was sent by the system

  paintStd (evt);

  if(m_bProcessIsRunning)                             // is a process to indicate running?
     {                                                // yes
      paintStartProcess (evt);                        // do the painting as if the process starts
      m_ulOldSliderSize = 0L;                         // reset the slider size
      m_ulActualTick--;                               // because of m_ulActualTick++ in paintStartProcess
      WinPostMsg (handle(), MYM_STATUS_PROCEED, 0, 0);// will call proceed to do the rest of the painting
     }

  return (true);
}

//********************************************************************************
// AProcessStatus :: paintStd - Std-Paint, sets the background color             *
//********************************************************************************
Boolean AProcessStatus :: paintStd (IPaintEvent& evt) 
{
  IPresSpaceHandle hps ;

  hps = evt.presSpaceHandle() ;

  GpiSetColor (hps, CLR_PALEGRAY) ; // set the actual colour to pale gray

  // move to the lower left corner

  m_Point.x = 0 ;
  m_Point.y = 0 ;

  GpiMove (hps, &m_Point) ;

  // set the point to the upper right corner

  m_Point.x = size().width();
  m_Point.y = size().height();

  // fill the windows's rectangle with pale gray

  GpiBox (hps, DRO_OUTLINEFILL, &m_Point, 0, 0) ;

  return (true);
}

//********************************************************************************
// AProcessStatus :: paintStartProcess - painting when a process starts          *
//********************************************************************************
Boolean AProcessStatus :: paintStartProcess (IPaintEvent& evt) 
{
  m_nPaintWhat = ePaintStd; // reset m_nPaintWhat

  HAB hab = (HAB) IThread :: current().anchorBlock();  // get the anchor block

  IPresSpaceHandle hps ;

  hps = evt.presSpaceHandle() ; // get the handle to the presentation space

  if(!m_pFont) // no font was set
     {
      m_pFont = new IFont (hps); // set the font from this presentation space to get the actual font of the window
      m_ulY = (ULONG) 2 * m_pFont -> maxDescender (); // calculate the y-coordinate of the text
     }
  
  if(!WinLoadString (hab, 0, m_lResourceId, 80, (PSZ) m_szBuffer)) // load the process description from the string table
     strcpy(m_szBuffer, "Process started"); // set the text to a fixed value if WinLoadString failed

  GpiSetColor (hps, CLR_BLACK) ; // set the actual colour to black

  m_Point.x = 5;      // set the coordinate where to paint the text
  m_Point.y = m_ulY;

  GpiCharStringAt(hps, &m_Point, strlen (m_szBuffer), (PCH) m_szBuffer); // paint the text

  GpiQueryCurrentPosition (hps, &m_Point); // query the current position

  m_lStartSliderX = m_Point.x + 4; // increase it by 4 and set the starting x-coordinate of the slider

  sprintf (m_szBuffer, "%lu%%", m_ulPercentReady); // get the percent actually proceeded as a string

  m_lWidth = (LONG) m_pFont -> textWidth (m_szBuffer); // calculate the width of this string

  // calculate the starting x-coordinate of the percent number

  m_lStartPercentNumberX = 
  m_Point.x = (((LONG) size (). width() - m_lStartSliderX - m_lWidth) / 2L) 
              + m_lStartSliderX;

  GpiCharStringAt(hps, &m_Point, strlen (m_szBuffer), (PCH) m_szBuffer); // paint the percent number

  GpiQueryCurrentPosition (hps, &m_Point); // query the current position

  m_lEndPercentNumberX = m_Point.x;        // keep the x-value for later use

  // calculate the position and size of the percent number at its hightest value

  m_lPercentNumberMaxXRight = ((LONG) size (). width() - m_lStartSliderX - displaySize ("100%"). width () / 2L) 
                               + m_lStartSliderX + displaySize ("100%"). width ();

  return (true);
}

//********************************************************************************
// AProcessStatus :: paintPercentNumber - refresh the percent number             *
//********************************************************************************
Boolean AProcessStatus :: paintPercentNumber (IPaintEvent& evt) 
{
  m_nPaintWhat = ePaintStd; // reset m_nPaintWhat

  IPresSpaceHandle hps;

  hps = evt.presSpaceHandle();  // get the handle to the presentation space

  // clean up the old percent number

  GpiSetColor (hps, CLR_PALEGRAY); // set the actual colour to pale gray, the background colour

  // set the point to the upper right corner of the rectangle around the percent number

  m_Point.x = m_lEndPercentNumberX;
  m_Point.y = size().height();

  GpiMove (hps, &m_Point) ;

  // set the point to the lower left corner of the rectangle around the percent number

  m_Point.x = m_lStartPercentNumberX;
  m_Point.y = 0L;

  GpiBox (hps, DRO_OUTLINEFILL, &m_Point, 0, 0) ; // fill the rectangle

  // paint the new percent value

  sprintf (m_szBuffer, "%lu%%", m_ulPercentReady); // get the percent as a string

  GpiSetColor (hps, CLR_BLACK) ; // set the actual colour to black

  m_lWidth = (LONG) m_pFont -> textWidth (m_szBuffer); // calculate the string's size in dots

  // calculate the starting x-coordinate of the new percent number

  m_lStartPercentNumberX = 
  m_Point.x = (((LONG) size (). width() - m_lStartSliderX - m_lWidth) / 2L) 
              + m_lStartSliderX;

  m_Point.y = m_ulY;

  GpiCharStringAt(hps, &m_Point, strlen (m_szBuffer), (PCH) m_szBuffer); // display it

  GpiQueryCurrentPosition (hps, &m_Point); // query the current position

  m_lEndPercentNumberX = m_Point.x; // keep the x-value for later use

  return (true);
}

//********************************************************************************************
// AProcessStatus :: paintSlider - paints the slider and the percent number inside of it     *
//********************************************************************************************
Boolean AProcessStatus :: paintSlider (IPaintEvent& evt) 
{
  m_nPaintWhat = ePaintStd;  // reset m_nPaintWhat

  IPresSpaceHandle hps ;

  hps = evt.presSpaceHandle() ;  // get the handle to the presentation space

  GpiSetColor (hps, CLR_BLUE) ;  // set the actual colour to blue

  // move to upper right corner of the slider

  m_Point.x = m_lStartSliderX;
  m_Point.y = 2L;

  GpiMove (hps, &m_Point);

  // set the point to the lower left end of the slider

  m_Point.x = m_lStartSliderX + m_ulNewSliderSize;
  m_Point.y = size (). height () - 2L;

  GpiBox (hps, DRO_OUTLINEFILL, &m_Point, 0, 0) ; // paint the slider

  if(evt. rect(). right() < m_lStartPercentNumberX) // does the slider overlay (a part of) the percent number?
     return (true); // no ==> return

  // set the point to the lower left corner of the percent number

  m_Point.x = m_lStartPercentNumberX;
  m_Point.y = m_ulY;

  sprintf (m_szBuffer, "%lu%%", m_ulPercentReady); // get the percent number as a string

  GpiSetColor (hps, CLR_WHITE) ; // set the actual colour to white

  GpiCharStringAt(hps, &m_Point, strlen (m_szBuffer), (PCH) m_szBuffer); // display the overlaid (part of the) percent number

  return (true);
}

//********************************************************************************
// AProcessStatus :: startProcess - MYM_STATUS_START_PROCESS messages received   *
//********************************************************************************
Boolean AProcessStatus :: startProcess (IEvent& evt) 
{
  m_bProcessIsRunning = true;  // the process is running

  m_ulPercent =
  m_ulPercentReady = 0L;

  // get the 1. event parameter

  PROCESS_START_PARAMS * pStartProcess = ((PROCESS_START_PARAMS *) ((char * ) evt. parameter1 ())); 

  if(!pStartProcess -> ulTickSize)    // must be a minimum of 1
     pStartProcess -> ulTickSize = 1L;

  m_ulTotalTicks = pStartProcess -> ulProcessSize / pStartProcess -> ulTickSize; // get the total number of ticks

  if(!m_ulTotalTicks)       // must also be a minimum of 1
     m_ulTotalTicks = 1L;

  m_lResourceId = pStartProcess -> lResourceId; // keep the resource id of the process description

  m_ulActualTick = 0L;  // reset the number of ticks actually proceeded

  m_nPaintWhat = ePaintStartProcess;  // we want to paint the process's name

  m_bPaintSentByMe = true;  // paint event is "selfmade"

  // calculate the rect to invalidate

  m_Rectl. xLeft = 0L;
  m_Rectl. xRight = size (). width ();
  m_Rectl. yBottom = 0;
  m_Rectl. yTop = size (). height ();

  WinInvalidateRect (handle(), &m_Rectl, FALSE);  // send a paint event

  m_bPaintSentByMe = false;  // reset the flag

  return (true);
}

//********************************************************************************
// AProcessStatus :: proceed - MYM_STATUS_PROCEED message received               *
//********************************************************************************
Boolean AProcessStatus :: proceed (IEvent& evt) 
{
  m_ulActualTick++;  // one more tick proceeded

  m_dSizePerTick = (double) (size (). width () - m_lStartSliderX) / (double) m_ulTotalTicks;  // calculate the slider size per tick

  m_ulNewSliderSize = (ULONG) ((double) m_ulActualTick * m_dSizePerTick);  // get the new slider size

  if(m_ulNewSliderSize == m_ulOldSliderSize)  // did the size change?
     return (true);                           // no ==> nothing to do ==> return

  m_ulPercent = (m_ulActualTick * (ULONG) 100) / m_ulTotalTicks;  // calculate the percentage proceeded

  if(m_ulPercent != m_ulPercentReady)                         // did the percentage change?
     {                                                        // yes
      m_ulPercentReady = m_ulPercent;                         // keep the new percentage proceeded
      m_Rectl. xLeft = m_lStartSliderX + m_ulNewSliderSize;   // set the left end of the rect to invalidate
      if(m_Rectl. xLeft < m_lEndPercentNumberX)               // does the slider overlay the percent number total
        {                                                     // no
         m_nPaintWhat = ePaintPercentNumber;    	           // set what we want to paint
         m_bPaintSentByMe = true;                             // paint event is "selfmade"
         m_Rectl. yBottom = 0L;                 	           // set the rect to invalidate
         m_Rectl. yTop = size (). height ();
         m_Rectl. xRight = m_lPercentNumberMaxXRight;         // largest right-end-x-coordinate of the percent number
         WinInvalidateRect (handle(), &m_Rectl, FALSE);       // send a paint event
         m_bPaintSentByMe = false;                        	  // reset the flag
       }
     }

  m_nPaintWhat = ePaintSlider;                           // set what we want to paint

  m_Rectl. xLeft = m_lStartSliderX;                      // set the rect to invalidate
  m_Rectl. xRight = m_lStartSliderX + m_ulNewSliderSize;
  m_Rectl. yBottom = 0L;
  m_Rectl. yTop = size (). height ();

  m_ulOldSliderSize = m_ulNewSliderSize;                 // keep the new slider size

  m_bPaintSentByMe = true;                               // paint event is "selfmade"

  WinInvalidateRect (handle(), &m_Rectl, FALSE);         // send a paint event

  m_bPaintSentByMe = false;                              // reset the flag

  return (true);
}

//********************************************************************************
// AProcessStatus :: endProcess - MYM_STATUS_END_PROCESS message received        *
//********************************************************************************
Boolean AProcessStatus :: endProcess (IEvent& evt) 
{
  m_nPaintWhat = ePaintStd;             // set what we want to paint (clean up the status line)

  m_bProcessIsRunning = false;          // the process stopped

  m_Rectl. xLeft = 0L;                  // get the window's rect
  m_Rectl. xRight = size (). width ();
  m_Rectl. yBottom = 0;
  m_Rectl. yTop = size (). height ();

  WinInvalidateRect (handle(), &m_Rectl, FALSE);  // send a paint event

  return (true);
}
