typedef struct FCIINSTALLINFO {
   ULONG ulSzStruct;
   CHAR achVolLabel[12];
   ULONG ulVolSerial;
   ULONG ulNumLicenses;
} FCIINSTALLINFO, *PFCIINSTALLINFO;

ULONG EXPENTRY CmnFilInstallDisk(PCHAR pchFile,
                                 PCHAR pchVendor,
                                 PCHAR pchApp,
                                 PCHAR pchName,
                                 PULONG pulNumLicenses,
                                 ULONG ulRequest)
//-------------------------------------------------------------------------
// This function performs one of the available actions on a diskette
// which is intended to be an install diskette for an application.  The
// action to be performed is specified in ulRequest:
//
//    CFID_REQ_CREATE - initialize the install diskette with a specified
//                      number of licenses
//    CFID_REQ_DECREMENT - decrement the number of licenses
//    CFID_REQ_INCREMENT - increment the number of licenses
//    CFID_REQ_QUERY - query the number of licenses
//
// On all actions but create, the diskette is checked to insure that
// it is the original diskette initialized with the create action and
// an error code is returned if not.
//
// Input:  pchFile - points to the name of a file to contain the
//                   installation information.
//         pchVendor - points to the name of the company producing the
//                     product.
//         pchApp - points to the name of the application.
//         pchName - points to an arbitrary name.
//         pulNumLicenses - if ulRequest=CFID_REQ_CREATE, this points
//                          to a variable specifying the initial number
//                          of licenses for this installation diskette.
//                          Otherwise, it points to a variable to receive
//                          the updated number of available licenses or
//                          is NULL meaning that the information isn't
//                          needed.
//         ulRequest - one of the CFID_REQ_* constants
// Output:  pulNumLicenses - if non NULL, this contains the updated number
//                           of licenses after the request was performed.
// Returns:  CFID_ERR_NOERROR if successful, a CFID_ERR_* constant otherwise.
//-------------------------------------------------------------------------
{
   APIRET arRc;
   CHAR achFullFile[CCHMAXPATH];
   CHAR chDrive;
   FSINFO fsiInfo;
   BYTE bNumDrives;
   ULONG ulVolSerial;
   USHORT usAttr;
   USHORT usSzData;
   FCIINSTALLINFO fiiInstall;
   BOOL bReturn;

   //----------------------------------------------------------------------
   // Get the fully qualified name of the file so that we can get the
   // drive letter.  FIL_QUERYFULLNAME does require the diskette to be in
   // the drive so that OS/2 can query the current directory on the drive.
   //----------------------------------------------------------------------
   DosError(FERR_DISABLEHARDERR);

   arRc=DosQueryPathInfo(pchFile,
                         FIL_QUERYFULLNAME,
                         achFullFile,
                         sizeof(achFullFile));
   if (arRc!=0) {
      DosError(FERR_ENABLEHARDERR);
      return CFID_ERR_DRIVENOTREADY;
   } /* endif */

   DosError(FERR_ENABLEHARDERR);

   chDrive=toupper(achFullFile[0])-'A'+1;

   //----------------------------------------------------------------------
   // Verify that the drive specified is a diskette drive.
   //----------------------------------------------------------------------
   DosDevConfig(&bNumDrives,DEVINFO_FLOPPY);

   if (chDrive>bNumDrives) {
      return CFID_ERR_NOTFLOPPY;
   } /* endif */

   //----------------------------------------------------------------------
   // Query the file system information on the diskette.
   //----------------------------------------------------------------------
   if (DosQueryFSInfo(chDrive,
                      FSIL_VOLSER,
                      &fsiInfo,
                      sizeof(fsiInfo))!=0) {
      return CFID_ERR_READFAILED;
   } /* endif */

   //----------------------------------------------------------------------
   // Get the volume serial number of the diskette.
   //----------------------------------------------------------------------
   ulVolSerial=*((PULONG)&fsiInfo.fdateCreation);

   usAttr=EAT_BINARY;
   usSzData=sizeof(fiiInstall);

   switch (ulRequest) {
   case CFID_REQ_CREATE:
      //-------------------------------------------------------------------
      // pulNumLicenses cannot be NULL if we are initializing the diskette.
      //-------------------------------------------------------------------
      if (pulNumLicenses==NULL) {
         return CFID_ERR_BADPARM;
      } /* endif */

      //-------------------------------------------------------------------
      // Setup the intended EA data and write it to the file.
      //-------------------------------------------------------------------
      fiiInstall.ulSzStruct=sizeof(fiiInstall);
      strcpy(fiiInstall.achVolLabel,fsiInfo.vol.szVolLabel);
      fiiInstall.ulVolSerial=ulVolSerial;
      fiiInstall.ulNumLicenses=*pulNumLicenses;

      bReturn=CmnFilSetExtAttribute(achFullFile,
                                    usAttr,
                                    pchVendor,
                                    pchApp,
                                    pchName,
                                    (PCHAR)&fiiInstall,
                                    usSzData);
      if (!bReturn) {
         return CFID_ERR_WRITEFAILED;
      } /* endif */
      break;
   case CFID_REQ_DECREMENT:
      //-------------------------------------------------------------------
      // Get the current information from the file.
      //-------------------------------------------------------------------
      bReturn=CmnFilQueryExtAttribute(achFullFile,
                                      pchVendor,
                                      pchApp,
                                      pchName,
                                      &usAttr,
                                      (PCHAR)&fiiInstall,
                                      &usSzData);
      if (!bReturn) {
         return CFID_ERR_READFAILED;
      } /* endif */

      //-------------------------------------------------------------------
      // If the data is incorrect or the volume serial numbers do not
      // match, then this isn't the original install disk.
      //-------------------------------------------------------------------
      if ((fiiInstall.ulSzStruct!=sizeof(fiiInstall)) ||
          (strcmp(fiiInstall.achVolLabel,fsiInfo.vol.szVolLabel)!=0) ||
          (fiiInstall.ulVolSerial!=ulVolSerial)) {
         return CFID_ERR_NOTINSTALLDISK;
      } /* endif */

      //-------------------------------------------------------------------
      // Insure that we have at least one more license left.
      //-------------------------------------------------------------------
      if (fiiInstall.ulNumLicenses==0) {
         return CFID_ERR_NOMORELICENSES;
      } /* endif */

      //-------------------------------------------------------------------
      // Update and write the new information back to the diskette.
      //-------------------------------------------------------------------
      fiiInstall.ulNumLicenses--;

      if (pulNumLicenses!=NULL) {
         *pulNumLicenses=fiiInstall.ulNumLicenses;
      } /* endif */

      bReturn=CmnFilSetExtAttribute(achFullFile,
                                    usAttr,
                                    pchVendor,
                                    pchApp,
                                    pchName,
                                    (PCHAR)&fiiInstall,
                                    usSzData);
      if (!bReturn) {
         return CFID_ERR_WRITEFAILED;
      } /* endif */
      break;
   case CFID_REQ_INCREMENT:
      //-------------------------------------------------------------------
      // Get the current information from the file.
      //-------------------------------------------------------------------
      bReturn=CmnFilQueryExtAttribute(achFullFile,
                                      pchVendor,
                                      pchApp,
                                      pchName,
                                      &usAttr,
                                      (PCHAR)&fiiInstall,
                                      &usSzData);
      if (!bReturn) {
         return CFID_ERR_READFAILED;
      } /* endif */

      //-------------------------------------------------------------------
      // If the data is incorrect or the volume serial numbers do not
      // match, then this isn't the original install disk.
      //-------------------------------------------------------------------
      if ((fiiInstall.ulSzStruct!=sizeof(fiiInstall)) ||
          (strcmp(fiiInstall.achVolLabel,fsiInfo.vol.szVolLabel)!=0) ||
          (fiiInstall.ulVolSerial!=ulVolSerial)) {
         return CFID_ERR_NOTINSTALLDISK;
      } /* endif */

      //-------------------------------------------------------------------
      // Update and write the new information back to the diskette.
      //-------------------------------------------------------------------
      fiiInstall.ulNumLicenses++;

      if (pulNumLicenses!=NULL) {
         *pulNumLicenses=fiiInstall.ulNumLicenses;
      } /* endif */

      bReturn=CmnFilSetExtAttribute(achFullFile,
                                    usAttr,
                                    pchVendor,
                                    pchApp,
                                    pchName,
                                    (PCHAR)&fiiInstall,
                                    usSzData);
      if (!bReturn) {
         return CFID_ERR_WRITEFAILED;
      } /* endif */
      break;
   case CFID_REQ_QUERY:
      //-------------------------------------------------------------------
      // Get the current information from the file.
      //-------------------------------------------------------------------
      bReturn=CmnFilQueryExtAttribute(achFullFile,
                                      pchVendor,
                                      pchApp,
                                      pchName,
                                      &usAttr,
                                      (PCHAR)&fiiInstall,
                                      &usSzData);
      if (!bReturn) {
         return CFID_ERR_READFAILED;
      } /* endif */

      //-------------------------------------------------------------------
      // If the data is incorrect or the volume serial numbers do not
      // match, then this isn't the original install disk.
      //-------------------------------------------------------------------
      if ((fiiInstall.ulSzStruct!=sizeof(fiiInstall)) ||
          (strcmp(fiiInstall.achVolLabel,fsiInfo.vol.szVolLabel)!=0) ||
          (fiiInstall.ulVolSerial!=ulVolSerial)) {
         return CFID_ERR_NOTINSTALLDISK;
      } /* endif */

      if (pulNumLicenses!=NULL) {
         *pulNumLicenses=fiiInstall.ulNumLicenses;
      } /* endif */
      break;
   default:
      return CFID_ERR_BADPARM;
   } /* endswitch */

   return CFID_ERR_NOERROR;
}
