#include <ctype.h>
#include "jftpgw.h"

#define SOURCEPART 0
#define USERPART   1
#define TARGETPART 2
#define BROADCAST "255.255.255.255"

struct ipinfo {
	unsigned long int ip;
	unsigned long int netmask;
	struct ipinfo *next;
};

struct nameinfo {
	char* name;
	struct nameinfo *next;
};

struct userinfo {
	char* user;
	struct userinfo *next;
};

struct allowed_lst {
	struct ipinfo* ip;
	struct nameinfo* name;
	struct userinfo* user;
	struct ipinfo* target_ip;
	struct nameinfo* target_name;
	struct allowed_lst *next;
};

struct allowed_lst* allowedbase;
struct hostent* host;

char* read_line(FILE*);
int read_access(const char*);
char* get_line_values(char*, int);
struct ipinfo* get_ips(char*);
int get_allowed_list(FILE*);
void init_allowed(struct allowed_lst*);

int is_user_allowed(const char*, struct allowed_lst*);
struct ipinfo* is_ip_in_list(const char*, struct ipinfo*);
struct nameinfo* is_name_in_list(const char*, struct nameinfo*);
struct allowed_lst* is_ip_in_sources(const char*, struct allowed_lst*);
struct allowed_lst* is_name_in_sources(const char*, struct allowed_lst*);


void init_allowed(struct allowed_lst* allowed) {
        allowed->ip =0;
        allowed->name =0;
	allowed->user =0;
        allowed->target_ip =0;
        allowed->target_name =0;
        allowed->next =0;
}


int read_access(const char* filename) {
	FILE *conf = fopen(filename, "r");
	
	if (!conf) {
		perror("Error reading access table");
		log(1, "Could not read access table (%s): %s",
				filename, strerror(errno));
		return -1;
	}
	get_allowed_list(conf);
	fclose(conf);
	return 0;
}

char* read_line(FILE* file) {
	const int LINESIZE = 255;
	char buffer[LINESIZE];
	char* line, *line2;

	line = (char*) malloc(1);
	enough_mem(line);
	line[0] = '\0';
	while (fgets(buffer, LINESIZE, file) == buffer
		&& buffer[0] != '\0') {
		
		line2 = line;
		line = (char*) malloc(LINESIZE + strlen(line2) + 1);
		enough_mem(line);
		strcpy(line, line2);
		strcat(line, buffer);
		free(line2);
		line2 = 0;
		if (buffer[strlen(buffer) - 1] == '\n') {
		       	break;
		}
	}
	return line;
}

char* get_line_values(char* fullline, int whichpart) {
	char *line;
	char *delim1 = strchr(fullline, ':');
	char *delim2;
	if (!delim1) {
		return 0;
	}
	delim2 = strchr(delim1 + 1, ':');
	if (!delim2 || strchr(delim2 + 1, ':')) {
		return 0;
	}
	line = (char*) malloc(strlen(fullline)+1);
	enough_mem(line);
	switch (whichpart) {
		case SOURCEPART:
			strncpy(line, fullline, delim1 - fullline);
			line[delim1 - fullline] = '\0';
			break;
		case USERPART:
			strncpy(line, delim1+1, delim2 - delim1-1);
			line[delim2 - delim1-1] = '\0';
			break;
		case TARGETPART:
			strcpy(line, delim2+1);
			break;
		default:
			log(5, "Error: Wrong argument in get_line_values");
	}
	return line;
}

struct ipinfo* get_ips(char* line) {
	struct ipinfo* ips =0, *pcurip =0, *tmp =0;
	char* p = line;
	int invalid = 0;
	char ipbuf[16];
	int i =0;
	
	for (;;) {
		while (*p) {
			/* set p to the next part */
			if (isdigit((int)*p)) {
				if (p == line) {
					break;
				}
				if (isspace((int)*(p-1))) {
					break;
				}
			}
			p++;
		}
		if (!*p) {
			break;
		}
		/* p theoretically points to an ip */
		if (isdigit((int)*p)) {
			if (pcurip) {
				tmp = pcurip;
			}
			pcurip = (struct ipinfo*) malloc(sizeof(struct ipinfo));
			enough_mem(pcurip);
			pcurip->next =0;
			if (tmp) {
				tmp->next = pcurip;
			}
			if (!ips) {
				ips = pcurip;
			}
			i =0;
			while ((isdigit((int)*p) || *p == '.') && *p != '/') {
				if (i < sizeof(ipbuf)-1) {
					ipbuf[i++] = *p;
				}
				p++;
			}
			ipbuf[i] = '\0';
			log(8, "Found ip %s", ipbuf);
			pcurip->ip = inet_addr(ipbuf);
			if (pcurip->ip == (unsigned long int) UINT_MAX
					&& strcmp(ipbuf, BROADCAST)) {
				log(4, "Invalid IP: %s", ipbuf);
				free(pcurip);
/*				return (struct ipinfo *) 0;*/
				continue;
			}
			if (*p == '/') {
				p++; i =0;
				while (!isspace((int)*p) && *p) {
					if (i < sizeof(ipbuf)-1) {
						ipbuf[i++] = *p;
					}
					p++;
				}
				ipbuf[i] = '\0';
				log(8, "With netmask %s", ipbuf);
				invalid = 0;
				
				if (strlen(ipbuf) < 3 && !strchr(ipbuf, '.')) {
					pcurip->netmask = atoi(ipbuf);
					if (pcurip->netmask < 0 
						|| pcurip->netmask > 32) {
						invalid = 1;
					}
					else {
						pcurip->netmask = 
							setlastbits(pcurip->netmask);
					}
				} else {
					pcurip->netmask = inet_addr(ipbuf);
					if (pcurip->netmask == (unsigned long int) UINT_MAX
						&& strcmp(ipbuf, BROADCAST)) {
						invalid = 1;
					}
				}
				if (invalid) {
					log(4, "Invalid netmask: %s",
							ipbuf);
					
					/* Set the netmask to
					 * 255.255.255.255 */
					
					pcurip->netmask = -1;
				}
			} else {
				/* no netmask specified */
				pcurip->netmask = -1;
			}
		}
	}
	return ips;
}

struct nameinfo* get_names(char* line) {
	const int NAMESIZE = 3;
	struct nameinfo* names =0, *ptmp =0, *pcurname =0;
	char* p = line;
	char* buf, *tmp;
	char n[NAMESIZE];
	int i;
	
	for (;;) {
		while(*p && !isalpha((int)*p)) {
			/* see if it is a domain like .ibm.com */
			if (*p == '.') {
				if (p == line) {
					/* at the beginning of the line */
					break;
				}
				if (isspace((int)*(p-1))) {
					/* the dot is following a space chr */
					break;
				}
			}
			p++;
		}
		if (!*p) {
			break;
		}
		buf = (char*) malloc(1);
		buf[0] = '\0';
		while(*p && !isspace((int)*p)) {
			i =0;
			while (i < NAMESIZE - 1 && !isspace((int)*p)) {
				n[i++] = *p++;
			}
			n[i] = '\0';
			tmp = buf;
			buf = (char*) malloc(strlen(buf)+NAMESIZE+1);
			strcpy(buf, tmp);
			strcat(buf, n);
			free(tmp);
			tmp = 0;
		}
		log(9, "Read name %s", buf);
		if (pcurname) {
			ptmp = pcurname;
		}
		else {
			ptmp = 0;
		}
		pcurname = malloc(sizeof(struct nameinfo));
		pcurname->name = buf;
		pcurname->next = 0;
		if (ptmp) {
			ptmp->next = pcurname;
		}
		if (!names) {
			/* first structure in list */
			names = pcurname;
		}
	}
	return names;
}


struct userinfo* get_users(char* line) {
	char *p = line;
	char* ustart, *uend;
	struct userinfo* users, *pcuruser, *tmp;
	users = pcuruser = tmp = 0;
	for(;;) {
		while (*p && isspace((int)*p)) {
			p++;
		}
		if (!*p) {
			break;
		}
		ustart = p;
		uend = 0;
		/* user `user "joe@localhost,3944"@foo.de' should work */
		if (*p == '"') {
			uend = strchr(ustart+1, '"');
		}
		/* if uend == 0, it was an invalid "" */
		if (0 == uend) {
			/* beginning of username */
			do {
				p++;
			} while (*p && isalnum((int)*p));
			/* end of username */
			uend = p;
		}
		else {
			/* skip over the `"' */
			ustart++;
			p = uend + 1;
		}
		pcuruser = (struct userinfo*) malloc(sizeof(struct userinfo));
		if (tmp) {
			tmp->next = pcuruser;
		}
		tmp = pcuruser;
		enough_mem(pcuruser);
		if (!users) {
			/* initialize base */
			users = pcuruser;
		}
		pcuruser->user = (char*) malloc((uend - ustart) + 1);
		strncpy(pcuruser->user, ustart, uend - ustart);
		pcuruser->user[uend - ustart] = '\0';
		pcuruser->next = 0;
	}
	return users;
}
		


int get_allowed_list(FILE* file) {
	struct allowed_lst* pcurallowed =0, *tmp =0;

	char* line;
	char* source, *target, *user;
	allowedbase = (struct allowed_lst*) 0;

	while (!feof(file) && (line = read_line(file))) {
		if (line[0] == '\0' || line[0] == '#' || line[0] == '\n') {
			free(line);
			line = (char*) 0;
			continue;
		}
		source = get_line_values(line, SOURCEPART);
		user = get_line_values(line, USERPART);
		target = get_line_values(line, TARGETPART);
		if (!source || !user || !target) {
			if (strlen(line)) {
				/* chop the newline character */
				line[strlen(line)-1] = '\0';
			}
			fprintf(stderr, "Ignoring invalid line: `%s\' (empty or wrong number of colons?)\n", line);
			log(5, "Ignored invalid line: `%s\'", line);
			free(line);
			continue;
		}
		if (pcurallowed) {
			tmp = pcurallowed;
		}
		pcurallowed = malloc(sizeof(struct allowed_lst));
		init_allowed(pcurallowed);
		if (tmp) {
			tmp->next = pcurallowed;
		}

		pcurallowed->ip = get_ips(source);
		pcurallowed->target_ip = get_ips(target);
		pcurallowed->name = get_names(source);
		pcurallowed->user = get_users(user);
		pcurallowed->target_name = get_names(target);
		if (!allowedbase) {
			allowedbase = pcurallowed;
		}
		free(line);
		free(source);
		free(user);
		free(target);		
	}
	return 0;
}

/* ===================== Matching ================== */
/* Return values:
 * 
 *  0 = Not allowed
 *  1 = Allowed
 * -1 = Could not lookup
 * 
 */

struct allowed_lst* is_ip_allowed(const char* ip, struct allowed_lst* allowed) {
	struct allowed_lst* ret;
	char* nip = 0;
	int i =0;
	unsigned long int iaddr = inet_addr(ip);
	if (!allowed) {
		allowed = allowedbase;
	}
	host = gethostbyaddr((char*) &iaddr, sizeof(iaddr), AF_INET);
	if (!host) {
		log(4, "Could not look up %s (is_ip_allowed)", ip);
		if((ret = is_ip_in_sources(ip, allowed))) {
			return ret;
		}
		else { 
			return (struct allowed_lst*) 0;
		}
	}
	
	
	/* check the IPs of the host */
	log(8, "Checking if %s is allowed.", ip);
	do {
		nip = gethostentip(host->h_addr_list[i]);
		
		if((ret = is_ip_in_sources(nip, allowed))) {
			return ret;
		}
		i++;
	} while (host->h_addr_list[i]);
	/* and check the name */
	
	if (host->h_name) {
		log(8, "Checking if %s is allowed.", host->h_name);
		if ((ret = is_name_in_sources(host->h_name, allowed))) {
			return ret;
		}
	}
	/* and the aliases */
	i =0;
	while (host->h_aliases[i]) {
		if((ret = is_name_in_sources(host->h_aliases[i], allowed))) {
			return ret;
		}
		i++;
	}
	
	return (struct allowed_lst*) 0;
}

int is_ip_target(const char* source_ip,
		 const char* target_ip, 
		 const char* user,        /* the original username passed
					   * to jftpgw */
		 const char* destuser) {  /* the username jftpgw passes to
					     the destination machine. This
					     variable may be a NULL pointer */
	struct allowed_lst* allowed;
	char* nip, *du;
	char ipbuf[16];
	struct hostent* target_host;
	unsigned long int target_iaddr;
	int i;

	/* target_ip points to a static char[] and is returned by
	 * gethostentip. We must save its contents */
	strcpy(ipbuf, target_ip);
	target_ip = ipbuf;
	
	allowed = is_ip_allowed(source_ip, 0);
	
	if (!allowed) {
		log(1, "Error: Host that was not allowed asked for a valid target!");
		return 0;
	}
	
	target_iaddr = inet_addr(target_ip);
	log(9, "Target ip was %s, inet_addr returned %ld",
			target_ip, target_iaddr);
	if (target_iaddr == (unsigned long int) UINT_MAX) {
		log(3, "Got invalid ip in is_ip_target: %s",
					target_ip);
			return 0;
	}
	target_host = gethostbyaddr((char*) &target_iaddr, sizeof(target_iaddr),
					AF_INET);
	if (!target_host) {
		log(5, "Could not lookup %s (is_ip_target)", target_ip);
		do {
			if (is_ip_in_list(target_ip,
						   allowed->target_ip)) {
				if (is_user_allowed(user, allowed)) {
					return 1;
				}
					if (destuser) {
					du = (char*) malloc(strlen(destuser) 
							+ 2);
					du[0] = '>';
					strcpy(&du[1], destuser);
					if (is_user_allowed(du, allowed)) {
						free(du);
						return 1;
					}
					free (du);
				}
			}
		} while (allowed->next
			&& (allowed = is_ip_allowed(source_ip, allowed->next)));
		return 0;
	}
	
	/* successful lookup */
	
	for (;;) {
		target_host = gethostbyaddr((char*) &target_iaddr,
				sizeof(target_iaddr), AF_INET);
		if (!target_host) {
			log(1, "A previously successful DNS lookup failed: %s",
					strerror(errno));
			log(1, "Bailing out");
			return 0;
		}
		i =0;
		do {
			nip = gethostentip(target_host->h_addr_list[i]);
			if (is_ip_in_list(nip, allowed->target_ip)) {
				if (is_user_allowed(user, allowed)) {
					return 1;
				}
				if (destuser) {
					du = (char*) malloc(strlen(destuser) 
							+ 2);
					du[0] = '>';
					strcpy(&du[1], destuser);
					if (is_user_allowed(du, allowed)) {
						free(du);
						return 1;
					}
					free (du);
				}
			}
			i++;
		} while (target_host->h_addr_list[i]);
	
		if (target_host->h_name) {
			if (is_name_in_list(target_host->h_name,
						allowed->target_name)) {
				if (is_user_allowed(user, allowed)) {
					return 1;
				}
			}
		}
		/* and the aliases */
		i =0;
		while (target_host->h_aliases[i]) {
			if(is_name_in_list(target_host->h_aliases[i],
						allowed->target_name)) {
				if(is_user_allowed(user, allowed)) {
					return 1;
				}
			}
			i++;
		}
		if (allowed->next) {
			allowed = is_ip_allowed(source_ip, allowed->next);
		}
		if (!allowed || !allowed->next) {
			break;
		}
	}
	return 0;
}

struct ipinfo* is_ip_in_list(const char* ip, struct ipinfo* list) {
	unsigned long int ipn = inet_addr(ip);
	if (!list) {
		return (struct ipinfo*) 0;
	}
	do {
		if ((ipn & list->netmask) == (list->ip & list->netmask)) {
			log(8, "%s is allowed - found in list (ips)", ip);
			return list;
		}
		list = list->next;
	} while (list);
	return (struct ipinfo*) 0;
}

struct nameinfo* is_name_in_list(const char* name, struct nameinfo* list) {
	if (!list || !list->name) {
		return (struct nameinfo*) 0;
	}
	do {
		if (cmp_domains(name, list->name)) {
			log(8, "%s is allowed to connect - found in list (domains)", name);
			return list;
		}
	} while ((list = list->next));
	return (struct nameinfo*) 0;
}

int is_user_allowed(const char* user, struct allowed_lst* allowed) {
	struct userinfo* ui;
	if (!allowed) {
		return 0;
	}
	log(8, "Checking whether user %s is allowed to connect", user);
	ui = allowed->user;
	do {
		if (0 == strcmp(ui->user, user) || 
		    0 == strcmp(ui->user, "*")) {
			log(8, "Allowed, found in %s", ui->user);
			return 1;
		}
	} while ((ui = ui->next));

	return 0;
}

struct allowed_lst* is_ip_in_sources(const char* ip, struct allowed_lst* allowed) {
	if (!allowed) {
		return 0;
	}
	do {
		if (!allowed->ip) {
			continue;
		}
		if (is_ip_in_list(ip, allowed->ip)) {
			return allowed;
		}
	} while ((allowed = allowed->next));
	return 0;
}

struct allowed_lst* is_name_in_sources(const char* name, struct allowed_lst* allowed) {
	if (!allowed) {
		return 0;
	}
	do {
		if (!allowed->name) {
			continue;
		}
		if (is_name_in_list(name, allowed->name)) {
			return allowed;
		}
	} while ((allowed = allowed->next));
	return 0;
}


void clean_ipinfo(struct ipinfo *ipi, int n) {
	if (!ipi) {
		return;
	}
	if (ipi->next) {
		clean_ipinfo(ipi->next,n+1);
		ipi->next =0;
	}
	free(ipi);
}

void clean_userinfo(struct userinfo *ui) {
	if (!ui) {
		return;
	}
	if (ui->next) {
		clean_userinfo(ui->next);
	}
	free(ui->user);
	free(ui);
}

void clean_nameinfo(struct nameinfo *ni) {
	if (!ni) {
		return;
	}
	if (ni->next) {
		clean_nameinfo(ni->next);
	}
	free(ni->name);
	free(ni);
}

void clean_allowed(struct allowed_lst *a) {
	if (!a) {
		return;
	}
	if (a->next) {
		clean_allowed(a->next);
	}
	clean_ipinfo(a->ip,1);
	clean_ipinfo(a->target_ip,1);
	clean_nameinfo(a->name);
	clean_nameinfo(a->target_name);
	clean_userinfo(a->user);
	free(a);
}

void clean_allowedlist() {
	clean_allowed(allowedbase);
	allowedbase = 0;
}


/*
 * Takes a socket descriptor and returns the string contain the peer's
 * IP address.  -- from tinyproxy
 */
char *getpeer_ip(int fd)
{
	char* ipaddr = (char*) malloc(16);
	struct sockaddr_in name;
#ifdef HAVE_SOCKLEN_T
	socklen_t namelen;
#else
	int namelen;
#endif
	namelen = sizeof(name);
	
	if (getpeername(fd, (struct sockaddr *) &name, &namelen) != 0) {
		log(2, "Could not get peername");
		strcpy(ipaddr, "255.255.255.255");
	} else{
		strcpy(ipaddr, inet_ntoa(*(struct in_addr *) &name.sin_addr.s_addr));
	}

	return ipaddr;
}

unsigned int get_usesport(int fd) {
	struct sockaddr_in sin;
#ifdef HAVE_SOCKLEN_T
        socklen_t slen;
#else
        int slen;
#endif

	slen = sizeof(struct sockaddr);
	if (getsockname(fd, (struct sockaddr*) &sin, &slen) < 0) {
		log(2, "getsockname failed trying to get the port number: %s",
				strerror(errno));
		return DEFAULTBINDPORT;
	}
	return ntohs(sin.sin_port);
}
