#include "os2port.h"

#include <stdio.h>
#include <strings.h>
#include <stdlib.h>
#include <unistd.h>

#include <freeswan.h>

#include "constants.h"
#include "defs.h"
#include "state.h"
#include "connections.h"
#include "packet.h"
#include "demux.h"  /* needs packet.h */
#include "log.h"
#include "sha1.h"
#include "md5.h"
#include "crypto.h" /* requires sha1.h and md5.h */
#include "ipsec_doi.h"

#include "conf.h"

/*
     Initiator                        Responder
    -----------                      -----------
     HDR*, HASH, ATTR      -->
                           <--        HDR*, HASH, ATTR

     HASH = prf( SKEYID_a, M-ID | ATTR )

*/

stf_status
conf_mode_set (struct state *st) {

  u_char hash_val[MAX_DIGEST_LEN];
  size_t hash_len;
  char attrib[1024];
  pb_stream reply;
  pb_stream rbody;
  struct isakmp_hdr hdr;
  u_char space[8192];
  msgid_t msgid;
  struct connection *c = st->st_connection;

  init_pbs(&reply, space, sizeof(space), "reply packet");
  memset(&hdr, '\0', sizeof(hdr));
  st->st_msgid = generate_msgid(c->that.host);
  hdr.isa_version = ISAKMP_MAJOR_VERSION << ISA_MAJ_SHIFT | ISAKMP_MINOR_VERSION;
  hdr.isa_np = ISAKMP_NEXT_HASH;
  hdr.isa_xchg = ISAKMP_XCHG_CONF;
  hdr.isa_msgid = st->st_msgid;
  hdr.isa_flags = ISAKMP_FLAG_ENCRYPTION;
  memcpy(hdr.isa_icookie, st->st_icookie, COOKIE_SIZE);
  memcpy(hdr.isa_rcookie, st->st_rcookie, COOKIE_SIZE);
  msgid = st->st_msgid;
  if (!out_struct(&hdr, &isakmp_hdr_desc, &reply, &rbody))
    return STF_INTERNAL_ERROR;

  // set attribute
  //00 00 00 10  03 00 00 00  00 01 00 04

  bzero (attrib, 16);
  attrib[3] = 0x10;	
  attrib[4] = 0x03;	
  attrib[9] = 0x01;	
  attrib[11] = 0x04;	
  attrib[12] = 0xac;	
  attrib[13] = 0x12;	
  attrib[14] = 0x02;	
  attrib[15] = 0x02;	

  bzero (hash_val, MAX_DIGEST_LEN);
  mode_config_hash(st, hash_val, &hash_len, msgid, attrib, 16);
  if (!out_generic_raw(ISAKMP_NEXT_ATTR, &isakmp_hash_desc, &rbody
      , hash_val, hash_len, "HASH"))
    return STF_INTERNAL_ERROR;

  // send attribute - 0 len - ack 
  if (!out_generic_raw(ISAKMP_NEXT_NONE, &isakmp_attribute_desc, &rbody
      , attrib+4, 12, "ATTR"))
    return STF_INTERNAL_ERROR;
  // encrypt message, except for fixed part of header 

  if (!encrypt_message(&rbody, st))
    return STF_INTERNAL_ERROR;  // ??? we may be partly committed 
  clonetochunk(st->st_tpacket, reply.start, pbs_offset(&reply)
                  , "reply packet from quick_outI1");

  send_packet(st, "conf_outI1");

  // Advance state 
  st->st_state = STATE_CONF_I1;
//  st->st_pending_quick = TRUE; // kick quick mode again
  return STF_NO_REPLY;
//    return STF_REPLY;

}

stf_status
conf_mode_ack (struct msg_digest *md) {

    struct state *st = md->st;
    u_char hash_val[MAX_DIGEST_LEN];
    size_t hash_len;
    pb_stream *const hash_pbs = &md->chain[ISAKMP_NEXT_HASH]->pbs; 
    pb_stream *const attr_pbs = &md->chain[ISAKMP_NEXT_ATTR]->pbs; 
    struct isakmp_generic *const attr_h = (struct isakmp_generic *) &md->chain[ISAKMP_NEXT_ATTR]->payload;
    struct connection *c = st->st_connection;
    char attrib[1024];
    struct in_addr inip;
    msgid_t msgid;
    int result = STF_INTERNAL_ERROR;
    int alen = 0;
    u_int8_t atype;
    u_int16_t aident;
    int ca;
    int len;
    unsigned int af, start, need_name, need_pass, type; 
    int newstate = st->st_state;
    int flag = 0;

    msgid = md->hdr.isa_msgid;
    need_name = 0;
    need_pass = 0;

    // get attribute
    attr_h->isag_length = htons (attr_h->isag_length); // restore back
    memcpy (attrib, attr_h, 4); //header
    memcpy (&attrib[4], attr_pbs->cur, pbs_left (attr_pbs)); // attr itself
    alen = pbs_left (attr_pbs)+4;
    DBG_dump ("received ATTR payload :", attrib, alen); 
    if (alen < 8) {
	log ("Malformed attribute payload (length < 8)");
        return STF_INTERNAL_ERROR;
    }

    // HASH in 
    mode_config_hash(st, hash_val, &hash_len, msgid, attrib, alen);


    if (pbs_left(hash_pbs) != hash_len || memcmp(hash_pbs->cur, hash_val, hash_len) != 0) { 
      DBG_cond_dump (DBG_CRYPT, "received HASH :", hash_pbs->cur, pbs_left (hash_pbs)); 
      DBG_cond_dump (DBG_CRYPT, "calculated HASH:", hash_val, pbs_left(hash_pbs)); 
      log("received HASH does not match computed value in Config Mode Message");
      return STF_FAIL + INVALID_HASH_INFORMATION; 
    }

    // HDR out 

    {
      struct isakmp_hdr r_hdr = md->hdr;

      memcpy(r_hdr.isa_rcookie, st->st_rcookie, COOKIE_SIZE);
      r_hdr.isa_np = ISAKMP_NEXT_HASH;
      if (!out_struct(&r_hdr, &isakmp_hdr_desc, &md->reply, &md->rbody))
        return STF_INTERNAL_ERROR;
    }


    atype = attrib[4];
    aident = make2b (attrib[6], attrib[7]);

    if (alen < 12) {
	log ("zero attribute payload");
        return STF_NO_REPLY;
    }
    ca = 8;
    switch (atype) {
      case ISAKMP_CFG_REQUEST:
        while (ca < alen) {
	  type = make2b (attrib[ca], attrib[ca+1]);
          af = type & 0x8000;
	  if (af != 0)	
            type -= 0x8000;
          len = 0; 
          if (af == 0)
            len = make2b (attrib[ca+2], attrib[ca+3]);
          if ((ca+4+len) > alen) {
            log ("Attribute ends below attribute payload bound");
            return STF_INTERNAL_ERROR;
          }
          start = ca+4;
          switch (type) {
            case XAUTH_TYPE:
            case 0x0d:		// cisco style
              DBG_dump ("XAUTH_TYPE :", attrib+start, len); 
              break;
            case XAUTH_USER_NAME:
            case 0x0e:		// cisco style
              DBG_dump ("XAUTH_USER_NAME request :", attrib + start, len); 
	      need_name = 1;
              break;
            case XAUTH_USER_PASSWORD:
            case 0x0f:		// cisco style
              DBG_dump ("XAUTH_USER_PASSWORD request :", attrib + start, len); 
              need_pass = 1;
              break;
            default:
              log ("Unknown attrubute skipped (type :%d)", type);
              break;
           
          }
          ca += (4+len);
        }
        if (need_name && need_pass) {
	  // reply username, password
	  // 00 00 00 18  02 00 00 00  00 0e 00 04  6A 6F 68 6E  00 0f 00 04  6A 6F 68 6E
	    char username[64];
	    char userpass[64];
	    int unlen, uplen, rlen;

            if (c->this.xaname[0] != '\0')
	      strcpy (username, c->this.xaname);
            else
   	      strcpy (username, "none");
            if (c->this.xapass[0] != '\0')
	      strcpy (userpass, c->this.xapass);
            else
   	      strcpy (userpass, "none");

            unlen = strlen (username);
            uplen = strlen (userpass);
            rlen = 16 + unlen + uplen;
 	    bzero (attrib, rlen);
            attrib[3] = rlen;
	    attrib[4] = ISAKMP_CFG_REPLY;
            attrib[6] = (aident & 0xFF00) >> 8;
	    attrib[7] = (aident & 0x00FF);
            type = 0x0e;
	    attrib[8] = (type & 0xFF00) >> 8;
	    attrib[9] = (type & 0x00FF);
	    attrib[10] = (unlen & 0xFF00) >> 8;
	    attrib[11] = (unlen & 0x00FF);
            memcpy (&attrib[12], username, unlen);
            type = 0x0f;
            attrib[12+unlen] = (type & 0xFF00) >> 8;
            attrib[12+unlen+1] = (type & 0x00FF);
            attrib[12+unlen+2] = (uplen & 0xFF00) >> 8;
            attrib[12+unlen+3] = (uplen & 0x00FF);
            memcpy (&attrib[12+unlen+4], userpass, uplen);

	    bzero (hash_val, MAX_DIGEST_LEN);
	    mode_config_hash(st, hash_val, &hash_len, msgid, attrib, rlen);
	    if (!out_generic_raw(ISAKMP_NEXT_ATTR, &isakmp_hash_desc, &md->rbody
        	, hash_val, hash_len, "HASH"))
	      return STF_INTERNAL_ERROR;
            // send attribute - 0 len - ack 
            if (!out_generic_raw(ISAKMP_NEXT_NONE, &isakmp_attribute_desc, &md->rbody
                , attrib+4, rlen-4, "ATTR"))
              return STF_INTERNAL_ERROR;
            result = STF_REPLY;
	    newstate = STATE_CONF_I1;
        }  
        break;
      case ISAKMP_CFG_SET:
        type = make2b (attrib[ca], attrib[ca+1]);
        af = type & 0x8000;
        if (af != 0)	
          type -= 0x8000;
        len = 0; 
        if (af == 0)
          len = make2b (attrib[ca+2], attrib[ca+3]);
        if ((ca+4+len) > alen) {
          log ("Attribute ends below attribute payload bound ca %d len %d alen %d", ca, len, alen);
          return STF_INTERNAL_ERROR;
        }
        switch (type) {
          case XAUTH_STATUS:
          case 0x14:		// cisco style
            DBG_dump ("XAUTH STATUS :", attrib+start, len); 
            result = STF_REPLY;
            flag = 1;
	    newstate = STATE_AGGR_I2;
            break;
	  case INTERNAL_IP4_ADDRESS:
            memcpy (&st->st_connection->this.innerip, attrib+12, 4);
            inip.s_addr = st->st_connection->this.innerip;
            log ("Inner IP received: %s", inet_ntoa (inip));
            // Advance state 
            st->st_pending_quick = TRUE; // kick quick mode again
            result = STF_UNPEND_CONF;
	    newstate = STATE_CONF_I1;
          break;
          default:
            log ("Unsupported attribute type :%d", type);
            return STF_NO_REPLY;
            break;
        }
	bzero (attrib, 12);
    	attrib[3] = 0x0C;	
    	attrib[4] = ISAKMP_CFG_ACK;
	attrib[6] = (aident & 0xFF00) >> 8;
	attrib[7] = (aident & 0x00FF);
	attrib[8] = (type & 0xFF00) >> 8;
	attrib[9] = (type & 0x00FF);

    	bzero (hash_val, MAX_DIGEST_LEN);
    	mode_config_hash(st, hash_val, &hash_len, msgid, attrib, 12);
    	if (!out_generic_raw(ISAKMP_NEXT_ATTR, &isakmp_hash_desc, &md->rbody
            , hash_val, hash_len, "HASH"))
          return STF_INTERNAL_ERROR;
        // send attribute - 0 len - ack 
        if (!out_generic_raw(ISAKMP_NEXT_NONE, &isakmp_attribute_desc, &md->rbody
            , attrib+4, 8, "ATTR"))
          return STF_INTERNAL_ERROR;
        
        break;
      default:
        log ("Unsupported attribyte payload type :%d", atype);
        return STF_NO_REPLY;
        break;
    }
    // encrypt message, except for fixed part of header 

    if (!encrypt_message(&md->rbody, st))
      return STF_INTERNAL_ERROR;  // ??? we may be partly committed 
    st->st_state = newstate;	
    if (flag)
      reinit_phase2_iv (st);	
	
    return result;
}



static void
mode_config_hash(struct state *st, u_char *hash_val, size_t *hash_len, msgid_t msgid, char *attr, size_t attr_size)
{
    struct hmac_ctx ctx;
    hmac_init_chunk(&ctx, st->st_oakley.hasher, st->st_skeyid_a);

    *hash_len = ctx.hmac_digest_len;

    DBG_cond_dump(DBG_CRYPT, "MSG_ID", (const u_char *) &msgid, sizeof (msgid_t));
    hmac_update(&ctx, (const u_char *) &msgid, sizeof(msgid_t));

    // ATTR inner ip

    DBG_cond_dump(DBG_CRYPT, "attribute", attr, attr_size);
    hmac_update(&ctx,  attr, attr_size);

    hmac_final(hash_val, &ctx);
    DBG_dump("hash computed ", hash_val, ctx.hmac_digest_len);
}

