/*
Copyright (C) 1999 Igor Khasilev, igor@paco.net

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include	"../oops.h"
#include	"../modules.h"

#define	MODULE_NAME	"err"
#define	MODULE_INFO	"Error reporting module"

#if	defined(MODULES)
char		module_type   = MODULE_ERR  ;
char		module_name[] = MODULE_NAME ;
char		module_info[] = MODULE_INFO ;
int		mod_load(void);
int		mod_unload(void);
int		mod_config_beg(int), mod_config_end(int), mod_config(char*,int), mod_run(void);
int		err(int so, char *msg, char *reason, int code, struct request* rq, int *flags);
#else
static	char	module_type   = MODULE_ERR  ;
static	char	module_name[] = MODULE_NAME ;
static	char	module_info[] = MODULE_INFO ;
static	int	mod_load(void);
static	int	mod_unload(void);
static	int	mod_config_beg(int), mod_config_end(int), mod_config(char*, int), mod_run(void);
static	int	err(int so, char *msg, char *reason, int code, struct request* rq, int *flags);
#endif

struct	err_module err_mod = {
	{
	NULL, NULL,
	MODULE_NAME,
	mod_load,
	mod_unload,
	mod_config_beg,
	mod_config_end,
	mod_config,
	NULL,
	MODULE_ERR,
	MODULE_INFO,
	mod_run
	},
	err
};


#define	LANG_EN	0
#define	LANG_RU	1

static	char	 	err_lang[16];
static	char	 	err_template[MAXPATHLEN];
static	int		curr_lang;
static	char		*template;
static	int		template_size;
static	time_t		template_mtime;
static	time_t		template_check_time;
static	pthread_rwlock_t	err_config_lock;
static	void			check_template_age(void);
static	void			reload_template(void);

#define	WRLOCK_ERR_CONFIG	pthread_rwlock_wrlock(&err_config_lock)
#define	RDLOCK_ERR_CONFIG	pthread_rwlock_rdlock(&err_config_lock)
#define	UNLOCK_ERR_CONFIG	pthread_rwlock_unlock(&err_config_lock)

char	*messages[2][8] = {

	/*-- LANG_EN --*/
	{ "Bad formed request or url",
	  "Bad port",
	  "Access denied to this domain",
	  "DNS error, can't resolve",
	  "Internal error",
	  "Access denied",
	  "Data transfer error",
	  "Denied by ACL"
	},

	/*-- LANG RU --*/
	{"  URL",
	 " ",
	 "   ",
	 " DNS,    ",
	 " ",
	 " ",
	 "  ",
	 "  ACL"}
};

int
mod_run(void)
{
    return(MOD_CODE_OK);
}

int
mod_load(void)
{
    err_lang[0]	    = 0;
    err_template[0] = 0;
    curr_lang = LANG_EN;
    template = NULL;
    template_size = 0;
    template_mtime = 0;
    template_check_time = 0;
    pthread_rwlock_init(&err_config_lock, NULL);

    printf("Err_report started\n");

    return(MOD_CODE_OK);
}

int
mod_unload(void)
{
    WRLOCK_ERR_CONFIG ;
    printf("Err_report stopped\n");
    return(MOD_CODE_OK);
}

int
mod_config_beg(int i)
{
    WRLOCK_ERR_CONFIG ;
    err_lang[0]	    = 0;
    err_template[0] = 0;
    curr_lang = LANG_EN;
    IF_FREE(template);
    template_size = 0;
    template_mtime = 0;
    template_check_time = 0;
    UNLOCK_ERR_CONFIG ;
    return(MOD_CODE_OK);
}

int
mod_config_end(int i)
{

    WRLOCK_ERR_CONFIG ;
    if ( !strcasecmp(err_lang, "ru") ) {
	printf("Setting Language to `ru'\n");
	curr_lang = LANG_RU;
    }
    if ( err_template[0] ) {
	reload_template();
    }
    UNLOCK_ERR_CONFIG ;
    return(MOD_CODE_OK);
}

int
mod_config(char *config, int i)
{
char	*p = config;

    WRLOCK_ERR_CONFIG ;
    SKIP_SPACES(p);
    if ( !strncasecmp(p, "lang", 4) ) {
	p += 4;
	SKIP_SPACES(p);
	strncpy(err_lang, p, sizeof(err_lang) -1 );
    } else
    if ( !strncasecmp(p, "template", 8) ) {
	p += 8;
	SKIP_SPACES(p);
	strncpy(err_template, p, sizeof(err_template) -1 );
    }
    UNLOCK_ERR_CONFIG ;
    return(MOD_CODE_OK);
}

int
err(int so, char *msg, char *reason, int code, struct request* rq, int *flags) {
char	*hdr = "<html><body>\
		<i><h2>Invalid request:</h2></i><p><pre>";
char	*rf= "</pre><b>";
char	*trailer="\
		</b><p>Please, check URL.<p>\
		<hr>\
		Generated by Oops.\
		</body>\
		</html>";
struct	output_object	*obj;
struct	buff		*body;


    if ( !(obj = calloc(1, sizeof(*obj))) )
	return(0);

    if( code==3 || code==6 || code==8 ) {
	put_av_pair(&obj->headers,"HTTP/1.0", "403 Forbidden");
    }
    else {
	put_av_pair(&obj->headers,"HTTP/1.0", "400 Bad Request");
    }
    put_av_pair(&obj->headers,"Cache-Control:", "no-cache");
    put_av_pair(&obj->headers,"Pragma:", "no-cache");
    put_av_pair(&obj->headers,"Expires:", "Thu, 01 Jan 1970 00:00:01 GMT");
    put_av_pair(&obj->headers,"Content-Type:", "text/html");

    check_template_age();

    RDLOCK_ERR_CONFIG ;


    if ( template ) {
	char 	*tptr, *tptrend, *proc;

	body = alloc_buff(template_size);
	if ( !body )
	    goto failed;
	obj->body = body;

	tptr = template;
	tptrend = tptr+template_size;

	/* send template loop */
	while( tptr < tptrend ) {
	    proc = strchr(tptr, '%');
	    if ( !proc ) {
		int rc;
		rc = attach_data(tptr, tptrend-tptr, body);
		if ( rc ) goto failed;
		UNLOCK_ERR_CONFIG ;
		process_output_object(so, obj, rq);
		if ( obj ) free_output_obj(obj);
		SET(*flags, MOD_AFLAG_OUT);
		return(0);
	    }
	    attach_data(tptr, proc-tptr, body);
	    switch ( *(proc+1) ) {
		case '%':
			attach_data("%", 1, body);
			tptr = proc+2;
			break;
		case 'l':
			attach_data(messages[LANG_EN][code-1],
				strlen(messages[LANG_EN][code-1]),
				body);
			tptr = proc+2;
			break;
		case 'd':
			{   /* %d - print code */
			    char	numtmp[6];
			    int		numlen;

			    numlen = snprintf( numtmp, 5, "%d", code );
			    if( numlen > 0 )
				attach_data(numtmp, numlen, body);
			    else
				attach_data("??", 2, body);
			}
			tptr = proc+2;
			break;
		case 'u':   /* %u - print htmlized url path */
			if ( rq && rq->url.path != NULL ) {
			    char	*hpath = NULL;
			    if ( rq->url.proto )
				attach_data(rq->url.proto, strlen(rq->url.proto), body);
			    else
				attach_data("null", 4, body);
			    attach_data("://", 3, body);
			    if ( rq->url.host )
				attach_data(rq->url.host, strlen(rq->url.host), body);
			    else
				attach_data("null", 4, body);
			    hpath = html_escaping(rq->url.path);
			    if ( hpath ) {
				attach_data(hpath, strlen(hpath), body);
				free(hpath);
			    } else
				attach_data("/null", 5, body);
			} else
			    attach_data("NULL", 4, body);
			tptr = proc+2;
			break;
		case 'm':
			attach_data(messages[LANG_EN][code-1],
				strlen(messages[LANG_EN][code-1]),
				body);
			if ( code == ERR_DNS_ERR ) {
			    attach_data(": ", 2, body);
			    attach_data(reason, strlen(reason), body);
			}
			if ( code == ERR_TRANSFER ) {
			    attach_data(": ", 2, body);
			    attach_data(reason, strlen(reason), body);
			}
			if ( code == ERR_ACL_DENIED ) {
			    attach_data(": ", 2, body);
			    attach_data(reason, strlen(reason), body);
			}
			if ( code == ERR_BAD_URL ) {
			    /* show what we was able to parse from request */
			    if ( rq && rq->url.proto != NULL ) {
				attach_data("\n<br>Protocol:",14, body);
				attach_data(rq->url.proto, strlen(rq->url.proto), body);
			    } else
				attach_data("\n<br>Protocol: NULL",19, body);
			    if ( rq && rq->url.host != NULL ) {
				attach_data("\n<br>Host:", 10, body);
				attach_data(rq->url.host, strlen(rq->url.host), body);
			    } else
				attach_data("\n<br>Host: NULL", 15, body);
			    if ( rq && (rq->url.port != 0) ) {
				char buf[10];
				snprintf(buf, sizeof(buf)-1, "%d", rq->url.port);
				attach_data("\n<br>Port:", 10, body);
				attach_data(buf, strlen(buf), body);
			    }
			    if ( rq && rq->url.path != NULL ) {
				char	*htmlized_path = NULL;
				htmlized_path = html_escaping(rq->url.path);
				if ( htmlized_path ) {
				    attach_data("\n<br>Path:", 10, body);
				    attach_data(htmlized_path, strlen(htmlized_path), body);
				    free(htmlized_path);
				}
			    } else
				attach_data("\n<br>Path: NULL", 15, body);
			}
			tptr = proc+2;
			break;
		case 'M':
			attach_data(messages[curr_lang][code-1],
				strlen(messages[curr_lang][code-1]),
				body);
			if ( code == ERR_DNS_ERR ) {
			    attach_data(": ", 2, body);
			    attach_data(reason, strlen(reason), body);
			}
			if ( code == ERR_TRANSFER ) {
			    attach_data(": ", 2, body);
			    attach_data(reason, strlen(reason), body);
			}
			if ( code == ERR_ACL_DENIED ) {
			    attach_data(": ", 2, body);
			    attach_data(reason, strlen(reason), body);
			}
			if ( code == ERR_BAD_URL ) {
			    /* show what we was able to parse from request */
			    if ( rq && rq->url.proto != NULL ) {
				attach_data("\n<br>Protocol:",14, body);
				attach_data(rq->url.proto, strlen(rq->url.proto), body);
			    } else
				attach_data("\n<br>Protocol: NULL",19, body);
			    if ( rq && rq->url.host != NULL ) {
				attach_data("\n<br>Host:", 10, body);
				attach_data(rq->url.host, strlen(rq->url.host), body);
			    } else
				attach_data("\n<br>Host: NULL", 15, body);
			    if ( rq && (rq->url.port != 0) ) {
				char buf[10];
				snprintf(buf, sizeof(buf)-1, "%d", rq->url.port);
				attach_data("\n<br>Port:", 10, body);
				attach_data(buf, strlen(buf), body);
			    }
			    if ( rq && rq->url.path != NULL ) {
				char	*htmlized_path = NULL;
				htmlized_path = html_escaping(rq->url.path);
				if ( htmlized_path ) {
				    attach_data("\n<br>Path:", 10, body);
				    attach_data(htmlized_path, strlen(htmlized_path), body);
				    free(htmlized_path);
				}
			    } else
				attach_data("\n<br>Path: NULL", 15, body);
			}
			tptr = proc+2;
			break;
		case 'H':
		case 'h':
			{
			char	*buf = malloc(strlen(host_name)+10);
			    if ( buf ) {
				sprintf(buf, "%s:%d", host_name, http_port);
				attach_data(buf, strlen(buf), body);
				free(buf);
			    }
			}
			tptr = proc+2;
			break;
		default:
			attach_data("%", 1, body);
			tptr = proc+1;
			break;
	    }
	}
    }

    UNLOCK_ERR_CONFIG ;
    body = alloc_buff(128);
    if ( body ) {
	obj->body = body;
	attach_data(hdr, strlen(hdr), body);
	attach_data(msg, strlen(msg), body);
	attach_data(rf, strlen(rf), body);
	attach_data(reason, strlen(reason), body);
	attach_data(trailer, strlen(trailer), body);
	process_output_object(so, obj, rq);
    }
    SET(*flags, MOD_AFLAG_OUT);
    if ( obj ) free_output_obj(obj);
    return(MOD_CODE_OK);

  failed:
    UNLOCK_ERR_CONFIG ;
    if ( obj ) free_output_obj(obj);
    return(MOD_CODE_OK);
}

static void
check_template_age(void)
{
    if ( global_sec_timer - template_check_time < 5 )
	return;
    WRLOCK_ERR_CONFIG ;
    reload_template();
    UNLOCK_ERR_CONFIG ;
}

static void
reload_template(void)
{
struct stat sb;
int	rc, size;
char	*in_mem;

    /* must be called under locked err_config_lock */
    rc = stat(err_template, &sb);
    if ( rc != -1 ) {
	if ( sb.st_mtime <= template_mtime )
	    return;
	if ( !err_template[0] )
	    return;
	my_xlog(OOPS_LOG_NOTICE|OOPS_LOG_DBG|OOPS_LOG_INFORM, "reload_template(): Loading template from `%s'.\n", err_template);

	size   = (int)sb.st_size;
	IF_FREE(template);
	
	in_mem = malloc(size+1);
	if ( in_mem ) {
	    int fd = open(err_template, O_RDONLY | O_BINARY);
	    if ( fd != -1 ) {
		if ( read(fd, in_mem, size) == size ) {
		    template	= in_mem;
		    template_size	= size;
		    template_mtime	= sb.st_mtime;
		    template_check_time = global_sec_timer;
		    template[size]	= 0; /* so we can use str... functions */
		} else {
		    verb_printf("reload_template(): Read failed: %m\n");
		    free(in_mem);
		}
		close(fd);
	    } /* fd != -1 */ else {
		verb_printf("reload_template(): Open(%s) failed: %m\n", err_template);
		free(in_mem);
	    }
	} /* if in_mem */
    } /* stat() != -1 */
}
