#include "oops.h"

typedef char bool;
#define true 1
#define false 0

/*
 ************************ Memory Debugging ************************
 * (must be at the end of source file so #undef's will not conflict
 * with debug #defines.
 */

#ifdef _MEMDEBUG

#undef calloc

static bool        memdebug_active = 1;

static bool memdebug_mutex_inited;
static pthread_mutex_t  memdebug_mutex;


#define MEMDEBUG_MAX_THREADS  1024

static struct MemDebug_ThreadData
{
	const char *memdebug_lastpointfile;
	int         memdebug_lastpointline;
	int         memdebug_level;

} TLS_Data[MEMDEBUG_MAX_THREADS];

static struct MemDebug_Tree
{
	void       *left, *right;

	void       *value;
	const char *file;
	int         line;
	bool        allocated;
	const char *freefile;
	int         freeline;

	signed char balance;

} HEAD;

static struct MemDebug_ThreadData *memdebug_tls(void)
{
	int tid = *_threadid;

	if (tid >= MEMDEBUG_MAX_THREADS)
	{
		memdebug_active = 0;
		my_xlog(OOPS_LOG_SEVERE, "MEMDEBUG_FATAL: thread %d out of range, max %d\n", tid, MEMDEBUG_MAX_THREADS);
		tid = 0;
	}
	return &TLS_Data[tid];
}

static void memdebug_fatal(const char *s, struct MemDebug_Tree *node,
						   struct MemDebug_ThreadData *tls)
{
	memdebug_active = 0;

	my_xlog(OOPS_LOG_SEVERE, "MEMDEBUG_FATAL: line %d of `%s': %s\n", tls->memdebug_lastpointline, tls->memdebug_lastpointfile, s);
	if (node)
	{
		my_xlog(OOPS_LOG_SEVERE, "MEMDEBUG_FATAL:   [block %p allocated on line %d of `%s']\n", node->value, node->line, node->file);
		if (node->freefile)
			my_xlog(OOPS_LOG_SEVERE, "MEMDEBUG_FATAL:   [block freeded on line %d of `%s']\n", node->freeline, node->freefile);
	}

	memdebug_active = 1;
}

void _export memdebug_setpoint(const char *file, int line)
{
	if (!memdebug_mutex_inited)
	{
		pthread_mutex_init(&memdebug_mutex, NULL);
		memdebug_mutex_inited = 1;
	}

	if (memdebug_active)
	{
		struct MemDebug_ThreadData *tls = memdebug_tls();
		if (tls->memdebug_level == 0)
		{
			tls->memdebug_lastpointfile = file;
			tls->memdebug_lastpointline = line;
		}
		tls->memdebug_level++;
	}
}

void memdebug_declevel(void)
{
	if (memdebug_active)
		memdebug_tls()->memdebug_level--;
}

static struct MemDebug_Tree *memdebug_initnode(struct MemDebug_Tree *node, void *K,
											   struct MemDebug_ThreadData *tls)
{
	if (node == NULL)
		node = calloc(sizeof(*node), 1);

	if (node == NULL)
		memdebug_fatal("no memory for debug node", NULL, tls);
	else
	{
		if (node->allocated)
			memdebug_fatal("node already allocated???", node, tls);

		node->value = K;
		node->file  = tls->memdebug_lastpointfile;
		node->line  = tls->memdebug_lastpointline;
		node->allocated = true;
		node->freefile  = NULL;
		node->freeline  = 0;
	}
	return node;
}

void *memdebug_check_alloc(void *K)
{
	struct MemDebug_Tree *P, *S, *T, *Q, *R;
	bool useLeftTree;
	signed char a;
	struct MemDebug_ThreadData *tls;

	if (memdebug_active == false)
		return K;

	tls = memdebug_tls();

	if (--(tls->memdebug_level) != 0)
		return K;

	if (K == NULL)
		return K;

	pthread_mutex_lock(&memdebug_mutex);

	if (HEAD.right == NULL) /* trivial case */
	{
		HEAD.right = memdebug_initnode(NULL, K, tls);
		pthread_mutex_unlock(&memdebug_mutex);
		return K;
	}

	T = &HEAD;
	S = P = HEAD.right;

	for (;;)
	{
		if (K == P->value) /* A2 */
		{
			memdebug_initnode(P, K, tls);
			break;
		}
		useLeftTree = (K < P->value);
		if (useLeftTree)  /* A3 */
			Q = P->left;
		else
			Q = P->right; /* A4 */

		if (Q)
		{
			if (Q->balance != 0)
			{
				T = P;
				S = Q;
			}
			P = Q;
			continue;
		}

		Q = memdebug_initnode(NULL, K, tls); /* A5 */
		if (Q == NULL)
			break;

		if (useLeftTree)
			P->left = Q;  /* A3 */
		else
			P->right = Q; /* A4 */

		a = (K < S->value) ? -1 : 1;          /* A6 */
		R = P = (a < 0 ? S->left: S->right);

		while (P != Q)
		{
			if (K < P->value) { P->balance = -1; P = P->left;  }
			else              { P->balance =  1; P = P->right; }
		}

		/* A7 */

		if (S->balance == 0)
		{
			S->balance = a;
			/* Can increment tree depth here */
			break;
		}
		if (S->balance != a)
		{
			S->balance = 0;
			break;
		}

		if (R->balance == a) /* A8 */
		{
			P = R;
			if (a < 0) { S->left  = R->right; R->right = S; }
			else       { S->right = R->left;  R->left  = S; }
			S->balance = R->balance = 0;
		}
		else /* A9 */
		{
			if (a < 0)
			{
				P        = R->right;
				R->right = P->left;
				P->left  = R;
				S->left  = P->right;
				P->right = S;
			}
			else
			{
				P        = R->left;
				R->left  = P->right;
				P->right = R;
				S->right = P->left;
				P->left  = S;
			}
			if      (P->balance == a) { S->balance = -a; R->balance = 0; }
			else if (P->balance == 0) { S->balance =     R->balance = 0; }
			else                      { S->balance = 0;  R->balance = a; }
			P->balance = 0;
		}

		if (S == T->right)  T->right = P;
		else                T->left  = P;

		break;
	}

	pthread_mutex_unlock(&memdebug_mutex);

	return K;
}

void *memdebug_check_free(void *p)
{
	struct MemDebug_ThreadData *tls;

	if (memdebug_active == false)
		return p;

	tls = memdebug_tls();

	if (tls->memdebug_level != 1)
		return p;

	if (p == NULL)
		memdebug_fatal("free of NULL pointer", NULL, tls);
	else
	{
		struct MemDebug_Tree *node;

		pthread_mutex_lock(&memdebug_mutex);
		for (node = HEAD.right; node; )
		{
			if (p == node->value)
				break;
			node = (p < node->value) ? node->left : node->right;
		}
		if (node == NULL)
		{
			memdebug_fatal("free of non-allocated pointer", node, tls);
			memdebug_active = 0;
			my_xlog(OOPS_LOG_SEVERE, "MEMDEBUG_FATAL:   [pointer value is %p]\n", p);
			memdebug_active = 1;
		}
		else
		{
			if (!node->allocated)
				memdebug_fatal("double free of pointer", node, tls);
			node->allocated = false;
			node->freefile  = tls->memdebug_lastpointfile;
			node->freeline  = tls->memdebug_lastpointline;
		}
		pthread_mutex_unlock(&memdebug_mutex);
	}

	return p;
}

void *memdebug_check_free0(void *p)
{
	if (memdebug_active && p)
		memdebug_check_free(p);

	return p;
}

#undef free

void memdebug_free(void *p, const char *file, int line)
{
	memdebug_setpoint(file, line);
	free(memdebug_check_free(p));
	memdebug_declevel();
}

#endif
